"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.voiceConnectorValidator = void 0;
const voiceConnector_1 = require("./voiceConnector");
var ISO_3166_ALPHA_2 = /^A[^ABCHJKNPVY]|B[^CKPUX]|C[^BEJPQST]|D[EJKMOZ]|E[CEGHRST]|F[IJKMOR]|G[^CJKOVXZ]|H[KMNRTU]|I[DEL-OQ-T]|J[EMOP]|K[EGHIMNPRWYZ]|L[ABCIKR-VY]|M[^BIJ]|N[ACEFGILOPRUZ]|OM|P[AE-HK-NRSTWY]|QA|R[EOSUW]|S[^FPQUW]|T[^ABEIPQSUXY]|U[AGMSYZ]|V[ACEGINU]|WF|WS|YE|YT|Z[AMW]$/;
var RFC_1918 = /(^192\.168\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])$)|(^172\.([1][6-9]|[2][0-9]|[3][0-1])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])$)|(^10\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])$)/;
var IP_ADDRESS = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/;
var VALID_CIDR = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\/(2[7-9])|(3[0-2])$/;
var RFC_1918_CIDR = /(^192\.168\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])(\/((2[7-9])|(3[0-2]))$))|(^172\.([1][6-9]|[2][0-9]|[3][0-1])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])(\/((2[7-9])|(3[0-2]))$)$)|(^10\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])\.([0-9]|[0-9][0-9]|[0-2][0-5][0-5])(\/((2[7-9])|(3[0-2]))$))/;
var FQDN = /^(?!.*?_.*?)(?!(?:[\w]+?\.)?\-[\w\.\-]*?)(?![\w]+?\-\.(?:[\w\.\-]+?))(?=[\w])(?=[\w\.\-]*?\.+[\w\.\-]*?)(?![\w\.\-]{254})(?!(?:\.?[\w\-\.]*?[\w\-]{64,}\.)+?)[\w\.\-]+?(?<![\w\-\.]*?\.[\d]+?)(?<=[\w\-]{2,})(?<![\w\-]{25})$/;
function voiceConnectorValidator(props) {
    if (props.region) {
        if (!(props.region.includes('Token[AWS.Region.') ||
            props.region === 'us-east-1' ||
            props.region === 'us-west-2' ||
            props.region === 'ap-northeast-1' ||
            props.region === 'ap-northeast-2' ||
            props.region === 'ap-southeast-1' ||
            props.region === 'ap-southeast-2' ||
            props.region === 'ca-central-1' ||
            props.region === 'eu-central-1' ||
            props.region === 'eu-west-1' ||
            props.region === 'eu-west-2')) {
            throw new Error('Region must be a valid region: https://docs.aws.amazon.com/chime-sdk/latest/dg/sdk-available-regions.html#sdk-pstn-regions');
        }
    }
    if (props.termination && props.termination.cps !== undefined) {
        if (props.termination.cps < 1) {
            throw new Error('CPS must be between 1 and 256');
        }
        if (props.termination.cps > 256) {
            throw new Error('CPS must be between 1 and 256');
        }
    }
    if (props.termination && props.termination.callingRegions) {
        for (var country of props.termination.callingRegions) {
            if (!ISO_3166_ALPHA_2.test(country)) {
                throw new Error(`Invalid Country: ${country}`);
            }
        }
    }
    if (props.termination && props.termination.terminationCidrs) {
        for (var terminationCidr of props.termination.terminationCidrs) {
            if (terminationCidr.includes('Token')) {
            }
            else if (!VALID_CIDR.test(terminationCidr)) {
                throw new Error('Termination CIDR must be a valid non-RFC1918 IPv4 CIDR block (/27-/32)');
            }
        }
    }
    if (props.termination && props.termination.terminationCidrs) {
        for (var terminationCidr of props.termination.terminationCidrs) {
            if (RFC_1918_CIDR.test(terminationCidr)) {
                throw new Error('Termination CIDR must not be RFC1918 CIDR block (/27-/32)');
            }
        }
    }
    if (props.origination) {
        for (var route of props.origination) {
            if (route.host.includes('Token')) {
            }
            else if (IP_ADDRESS.test(route.host)) {
                if (RFC_1918.test(route.host)) {
                    throw new Error('Origination IP must not be RFC1918 IP Address');
                }
            }
            else if (!FQDN.test(route.host)) {
                throw new Error('Origination IP must be valid, non-RFC1918 IPv4 Address or FQDN');
            }
        }
    }
    if (props.origination) {
        for (var route of props.origination) {
            if (route.port < 1 || route.port > 65535) {
                throw new Error('Port range must be valid: 1-65535');
            }
        }
    }
    if (props.origination && props.encryption) {
        for (var route of props.origination) {
            if (route.protocol === voiceConnector_1.Protocol.UDP) {
                throw new Error('TCP must be used with Encryption enabled');
            }
        }
    }
    return true;
}
exports.voiceConnectorValidator = voiceConnectorValidator;
//# sourceMappingURL=data:application/json;base64,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