"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EfsFargateTask = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
/**
 * Represents the AWS Fargate task with EFS and secret manager support.
 *
 * @stability stable
 */
class EfsFargateTask extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id);
        const stack = cdk.Stack.of(scope);
        const task = new ecs.FargateTaskDefinition(stack, `TaskDefinition${id}`, {
            cpu: 256,
            memoryLimitMiB: 512,
        });
        this.task = task;
        task.addVolume({
            name: 'efs-data',
            efsVolumeConfiguration: {
                fileSystemId: props.accessPoint.fileSystem.fileSystemId,
                authorizationConfig: {
                    accessPointId: props.accessPoint.accessPointId,
                },
                transitEncryption: 'ENABLED',
            },
        });
        let secret;
        if ((_b = props.secret) === null || _b === void 0 ? void 0 : _b.id) {
            // format the arn e.g. 'arn:aws:secretsmanager:eu-west-1:111111111111:secret:MySecret';
            const secretPartialArn = stack.formatArn({
                service: 'secretsmanager',
                resource: 'secret',
                resourceName: (_c = props.secret) === null || _c === void 0 ? void 0 : _c.id,
                sep: ':',
            });
            secret = secretsmanager.Secret.fromSecretAttributes(stack, 'GithubSecret', {
                secretPartialArn,
            });
            // allow task to read the secret
            secret.grantRead(task.taskRole);
        }
        const logGroup = new aws_logs_1.LogGroup(stack, `LogGroup${id}`, {
            retention: aws_logs_1.RetentionDays.ONE_DAY,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        new cdk.CfnOutput(stack, `LogGroup${id}Output`, { value: logGroup.logGroupName });
        const syncWorker = task.addContainer('SyncWorker', {
            logging: new ecs.AwsLogDriver({
                streamPrefix: 'SyncWorker',
                logGroup,
            }),
            ...props.syncContainer,
        });
        syncWorker.addMountPoints({
            containerPath: (_d = props.efsMountTarget) !== null && _d !== void 0 ? _d : '/mnt/efsmount',
            sourceVolume: 'efs-data',
            readOnly: false,
        });
        task.addToExecutionRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'elasticfilesystem:ClientMount',
                'elasticfilesystem:ClientWrite',
            ],
            resources: [
                stack.formatArn({
                    service: 'elasticfilesystem',
                    resource: 'file-system',
                    sep: '/',
                    resourceName: props.accessPoint.fileSystem.fileSystemId,
                }),
            ],
        }));
        // create a default security group for the fargate task
        this.securityGroup = new ec2.SecurityGroup(stack, `FargateSecurityGroup${id}`, { vpc: props.vpc });
        // allow fargate ingress to the efs filesystem
        props.accessPoint.fileSystem.connections.allowFrom(this.securityGroup, ec2.Port.tcp(2049));
    }
}
exports.EfsFargateTask = EfsFargateTask;
_a = JSII_RTTI_SYMBOL_1;
EfsFargateTask[_a] = { fqn: "cdk-efs-assets.EfsFargateTask", version: "0.3.93" };
//# sourceMappingURL=data:application/json;base64,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