import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecs from '@aws-cdk/aws-ecs';
import * as efs from '@aws-cdk/aws-efs';
import * as lambda from '@aws-cdk/aws-lambda';
import * as s3 from '@aws-cdk/aws-s3';
import * as cdk from '@aws-cdk/core';
/**
 * @stability stable
 */
export interface SyncSourceProps {
    /**
     * The VPC of the Amazon EFS Filesystem.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
    /**
     * Where to place the network interfaces within the VPC.
     *
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * Timeout duration for sync Lambda function.
     *
     * (optional, default: Duration.minutes(3))
     *
     * @stability stable
     */
    readonly timeout?: cdk.Duration;
    /**
     * The (absolute) directory path inside the EFS AccessPoint to sync files to.
     *
     * Specify '/' to restore synced files to the root
     * directory. (optional, default: a source-specific directory path. For example, for the GitHub source, the default
     * behavior is to restore to a directory matching the name of the repository)
     *
     * @stability stable
     */
    readonly syncDirectoryPath?: string;
}
/**
 * @stability stable
 */
export interface GithubSecret {
    /**
     * The secret ID from AWS Secrets Manager.
     *
     * @stability stable
     */
    readonly id: string;
    /**
     * The key of the secret.
     *
     * @stability stable
     */
    readonly key: string;
}
/**
 * @stability stable
 */
export interface GithubSourceProps extends SyncSourceProps {
    /**
     * The github repository HTTP URI.
     *
     * @stability stable
     */
    readonly repository: string;
    /**
     * The github secret for the private repository.
     *
     * @stability stable
     */
    readonly secret?: GithubSecret;
}
/**
 * @stability stable
 */
export interface S3ArchiveSourceProps extends SyncSourceProps {
    /**
     * The S3 bucket containing the archive file.
     *
     * @stability stable
     */
    readonly bucket: s3.IBucket;
    /**
     * The path of the zip file to extract in the S3 bucket.
     *
     * @stability stable
     */
    readonly zipFilePath: string;
    /**
     * If this is set to true, then whenever a new object is uploaded to the specified path, an EFS sync will be triggered.
     *
     * Currently, this functionality depends on at least one CloudTrail Trail
     * existing in your account that captures the S3 event.
     *
     * The option is only available with the `LAMBDA` sync engine.
     *
     * @default true
     * @stability stable
     */
    readonly syncOnUpdate?: boolean;
}
/**
 * @stability stable
 */
export interface FargateTaskConfig {
    /**
     * @stability stable
     */
    readonly task: ecs.TaskDefinition;
    /**
     * The security group of the fargate task.
     *
     * @stability stable
     */
    readonly securityGroup: ec2.ISecurityGroup;
}
/**
 * @stability stable
 */
export declare abstract class SyncSource {
    /**
     * @stability stable
     */
    static github(props: GithubSourceProps): SyncSource;
    /**
     * @stability stable
     */
    static s3Archive(props: S3ArchiveSourceProps): SyncSource;
    /** @internal */
    abstract _createHandler(accessPoint: efs.AccessPoint): lambda.Function;
    /** @internal */
    abstract _createFargateTask(id: string, accessPoint: efs.AccessPoint): FargateTaskConfig;
}
/**
 * @stability stable
 */
export declare class GithubSyncSource extends SyncSource {
    private readonly props;
    /**
     * @stability stable
     */
    constructor(props: GithubSourceProps);
    /**
     * @internal
     * @param accessPoint The EFS Access Point
     */
    _createHandler(accessPoint: efs.AccessPoint): lambda.Function;
    /**
     * @internal
     * @param id The task ID.
     * @param accessPoint The EFS access point.
     */
    _createFargateTask(id: string, accessPoint: efs.AccessPoint): FargateTaskConfig;
}
/**
 * @stability stable
 */
export declare class S3ArchiveSyncSource extends SyncSource {
    private readonly props;
    /**
     * @stability stable
     */
    constructor(props: S3ArchiveSourceProps);
    /**
     * @internal
     * @param accessPoint The EFS access point.
     */
    _createHandler(accessPoint: efs.AccessPoint): lambda.Function;
    /**
     * @internal
     * @param id The Fargate task ID.
     * @param accessPoint The EFS access point.
     */
    _createFargateTask(id: string, accessPoint: efs.AccessPoint): FargateTaskConfig;
}
/**
 * @stability stable
 */
export declare enum SyncEngine {
    /**
     * @stability stable
     */
    FARGATE = 0,
    /**
     * @stability stable
     */
    LAMBDA = 1
}
/**
 * @stability stable
 */
export interface SyncedAccessPointProps extends efs.AccessPointProps {
    /**
     * @stability stable
     */
    readonly syncSource: SyncSource;
    /**
     * The VPC to run the sync job.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
    /**
     * Trigger the sync with AWS Lambda or AWS Fargate.
     *
     * @stability stable
     */
    readonly engine?: SyncEngine;
}
/**
 * @stability stable
 */
export declare class SyncedAccessPoint extends efs.AccessPoint implements efs.IAccessPoint {
    /**
     * @stability stable
     */
    constructor(scope: cdk.Construct, id: string, props: SyncedAccessPointProps);
}
