"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SyncedAccessPoint = exports.SyncEngine = exports.S3ArchiveSyncSource = exports.GithubSyncSource = exports.SyncSource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const url_1 = require("url");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const efs = require("@aws-cdk/aws-efs");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const cdk_fargate_run_task_1 = require("cdk-fargate-run-task");
const efs_fargate_task_1 = require("./efs-fargate-task");
/**
 * @stability stable
 */
class SyncSource {
    /**
     * @stability stable
     */
    static github(props) {
        return new GithubSyncSource(props);
    }
    /**
     * @stability stable
     */
    static s3Archive(props) {
        return new S3ArchiveSyncSource(props);
    }
}
exports.SyncSource = SyncSource;
_a = JSII_RTTI_SYMBOL_1;
SyncSource[_a] = { fqn: "cdk-efs-assets.SyncSource", version: "0.3.93" };
/**
 * @stability stable
 */
class GithubSyncSource extends SyncSource {
    /**
     * @stability stable
     */
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * @internal
     * @param accessPoint The EFS Access Point
     */
    _createHandler(accessPoint) {
        var _e, _f, _g, _h;
        const stack = cdk.Stack.of(accessPoint);
        const region = stack.region;
        const vpcSubnets = (_e = this.props.vpcSubnets) !== null && _e !== void 0 ? _e : { subnetType: ec2.SubnetType.PRIVATE };
        const timeout = (_f = this.props.timeout) !== null && _f !== void 0 ? _f : cdk.Duration.minutes(3);
        let syncDirectoryPath;
        if (this.props.syncDirectoryPath === undefined) {
            // if property is unspecified, use repository name as output directory
            const parsed = new url_1.URL(this.props.repository);
            syncDirectoryPath = '/' + path.basename(parsed.pathname, '.git');
        }
        else {
            syncDirectoryPath = this.props.syncDirectoryPath;
        }
        const lambdaEnv = {
            REPOSITORY_URI: this.props.repository,
            MOUNT_TARGET: '/mnt/efsmount',
            SYNC_PATH: syncDirectoryPath,
        };
        if (this.props.secret) {
            lambdaEnv.GITHUB_SECRET_ID = this.props.secret.id;
            lambdaEnv.GITHUB_SECRET_KEY = this.props.secret.key;
        }
        const handler = new lambda.Function(accessPoint, 'GithubHandler', {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda-handler', 'github-sync')),
            handler: 'index.on_event',
            layers: [
                lambda.LayerVersion.fromLayerVersionArn(accessPoint, 'GitLayer', `arn:aws:lambda:${region}:553035198032:layer:git-lambda2:7`),
            ],
            filesystem: lambda.FileSystem.fromEfsAccessPoint(accessPoint, '/mnt/efsmount'),
            vpcSubnets: vpcSubnets,
            vpc: this.props.vpc,
            memorySize: 512,
            timeout: timeout,
            environment: lambdaEnv,
            currentVersionOptions: {
                provisionedConcurrentExecutions: 1,
            },
        });
        if ((_g = this.props.secret) === null || _g === void 0 ? void 0 : _g.id) {
            // format the arn e.g. 'arn:aws:secretsmanager:eu-west-1:111111111111:secret:MySecret';
            const secretPartialArn = stack.formatArn({
                service: 'secretsmanager',
                resource: 'secret',
                resourceName: (_h = this.props.secret) === null || _h === void 0 ? void 0 : _h.id,
                sep: ':',
            });
            const secret = secretsmanager.Secret.fromSecretAttributes(stack, 'GithubSecret', {
                secretPartialArn,
            });
            // allow lambda to read the secret
            secret.grantRead(handler);
        }
        return handler;
    }
    /**
     * @internal
     * @param id The task ID.
     * @param accessPoint The EFS access point.
     */
    _createFargateTask(id, accessPoint) {
        var _e, _f;
        const stack = cdk.Stack.of(accessPoint);
        const mountTarget = '/mnt/efsmount';
        let syncDirectoryPath;
        if (this.props.syncDirectoryPath === undefined) {
            // if property is unspecified, use repository name as output directory
            const parsed = new url_1.URL(this.props.repository);
            syncDirectoryPath = '/' + path.basename(parsed.pathname, '.git');
        }
        else {
            syncDirectoryPath = this.props.syncDirectoryPath;
        }
        const environment = {
            REPOSITORY_URI: this.props.repository,
            MOUNT_TARGET: mountTarget,
            SYNC_PATH: syncDirectoryPath,
        };
        if (this.props.secret) {
            environment.GITHUB_SECRET_ID = this.props.secret.id;
            environment.GITHUB_SECRET_KEY = this.props.secret.key;
        }
        let secret;
        if ((_e = this.props.secret) === null || _e === void 0 ? void 0 : _e.id) {
            // format the arn e.g. 'arn:aws:secretsmanager:eu-west-1:111111111111:secret:MySecret';
            const secretPartialArn = stack.formatArn({
                service: 'secretsmanager',
                resource: 'secret',
                resourceName: (_f = this.props.secret) === null || _f === void 0 ? void 0 : _f.id,
                sep: ':',
            });
            secret = secretsmanager.Secret.fromSecretAttributes(stack, 'GithubSecret', {
                secretPartialArn,
            });
        }
        const fargateSyncTask = new efs_fargate_task_1.EfsFargateTask(stack, id, {
            vpc: this.props.vpc,
            accessPoint,
            efsMountTarget: mountTarget,
            syncContainer: {
                image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../docker.d')),
                command: ['/root/githubsync.sh'],
                environment,
                secrets: secret ? {
                    OAUTH_TOKEN: ecs.Secret.fromSecretsManager(secret, 'oauth_token'),
                } : undefined,
            },
        });
        // allow task to read the secret
        secret === null || secret === void 0 ? void 0 : secret.grantRead(fargateSyncTask.task.taskRole);
        return {
            task: fargateSyncTask.task,
            securityGroup: fargateSyncTask.securityGroup,
        };
    }
    ;
}
exports.GithubSyncSource = GithubSyncSource;
_b = JSII_RTTI_SYMBOL_1;
GithubSyncSource[_b] = { fqn: "cdk-efs-assets.GithubSyncSource", version: "0.3.93" };
/**
 * @stability stable
 */
class S3ArchiveSyncSource extends SyncSource {
    /**
     * @stability stable
     */
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * @internal
     * @param accessPoint The EFS access point.
     */
    _createHandler(accessPoint) {
        var _e, _f, _g;
        const vpcSubnets = (_e = this.props.vpcSubnets) !== null && _e !== void 0 ? _e : { subnetType: ec2.SubnetType.PRIVATE };
        const syncOnUpdate = (_f = this.props.syncOnUpdate) !== null && _f !== void 0 ? _f : true;
        const timeout = (_g = this.props.timeout) !== null && _g !== void 0 ? _g : cdk.Duration.minutes(3);
        const filename = path.basename(this.props.zipFilePath, '.zip');
        let syncDirectoryPath;
        if (this.props.syncDirectoryPath === undefined) {
            // if property is unspecified, use zip file name as output directory
            syncDirectoryPath = '/' + filename;
        }
        else {
            syncDirectoryPath = this.props.syncDirectoryPath;
        }
        const handler = new lambda.Function(accessPoint, 'SyncHandler', {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda-handler', 's3-archive-sync')),
            handler: 'index.on_event',
            filesystem: lambda.FileSystem.fromEfsAccessPoint(accessPoint, '/mnt/efsmount'),
            vpcSubnets: vpcSubnets,
            vpc: this.props.vpc,
            memorySize: 512,
            timeout: timeout,
            environment: {
                MOUNT_TARGET: '/mnt/efsmount',
                BUCKET_NAME: this.props.bucket.bucketName,
                ZIPPED_KEY: this.props.zipFilePath,
                SYNC_PATH: syncDirectoryPath,
            },
            currentVersionOptions: {
                provisionedConcurrentExecutions: 1,
            },
            initialPolicy: [
                new aws_iam_1.PolicyStatement({
                    actions: ['s3:GetObject*'],
                    resources: ['arn:aws:s3:::' + this.props.bucket.bucketName + '/' + this.props.zipFilePath],
                }),
            ],
        });
        if (syncOnUpdate) {
            // In order to support bucket notifications for imported IBucket objects, onCloudTrailWriteObject is used.
            // TODO: When https://github.com/aws/aws-cdk/issues/2004 is closed, can use handler.addEventSource instead.
            /*
            handler.addEventSource(
              new S3EventSource(props.bucket, {
                events: [s3.EventType.OBJECT_CREATED],
                filters: [{ prefix: props.zipFilePath }]
              })
            );
             */
            this.props.bucket.onCloudTrailWriteObject('S3FileListener-' + filename, {
                paths: [this.props.zipFilePath],
                target: new aws_events_targets_1.LambdaFunction(handler),
            });
        }
        return handler;
    }
    /**
     * @internal
     * @param id The Fargate task ID.
     * @param accessPoint The EFS access point.
     */
    _createFargateTask(id, accessPoint) {
        const stack = cdk.Stack.of(accessPoint);
        const mountTarget = '/mnt/efsmount';
        const filename = path.basename(this.props.zipFilePath, '.zip');
        const syncDirectoryPath = this.props.syncDirectoryPath === undefined ?
            '/' + filename : this.props.syncDirectoryPath;
        const environment = {
            MOUNT_TARGET: '/mnt/efsmount',
            BUCKET_NAME: this.props.bucket.bucketName,
            ZIPPED_KEY: this.props.zipFilePath,
            SYNC_PATH: syncDirectoryPath,
        };
        const fargateSyncTask = new efs_fargate_task_1.EfsFargateTask(stack, id, {
            vpc: this.props.vpc,
            accessPoint,
            efsMountTarget: mountTarget,
            syncContainer: {
                image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../docker.d')),
                command: ['/root/s3sync.sh'],
                environment,
            },
        });
        // allow ecs task to get the s3 object
        fargateSyncTask.task.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:GetObject*'],
            resources: [
                stack.formatArn({
                    service: 's3',
                    resource: this.props.bucket.bucketName,
                    region: '',
                    account: '',
                    sep: '/',
                    resourceName: this.props.zipFilePath,
                }),
            ],
        }));
        return {
            task: fargateSyncTask.task,
            securityGroup: fargateSyncTask.securityGroup,
        };
    }
    ;
}
exports.S3ArchiveSyncSource = S3ArchiveSyncSource;
_c = JSII_RTTI_SYMBOL_1;
S3ArchiveSyncSource[_c] = { fqn: "cdk-efs-assets.S3ArchiveSyncSource", version: "0.3.93" };
/**
 * @stability stable
 */
var SyncEngine;
(function (SyncEngine) {
    SyncEngine[SyncEngine["FARGATE"] = 0] = "FARGATE";
    SyncEngine[SyncEngine["LAMBDA"] = 1] = "LAMBDA";
})(SyncEngine = exports.SyncEngine || (exports.SyncEngine = {}));
/**
 * @stability stable
 */
class SyncedAccessPoint extends efs.AccessPoint {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.engine === SyncEngine.LAMBDA) {
            const handler = props.syncSource._createHandler(this);
            // create a custom resource to trigger the sync
            const myProvider = new cr.Provider(this, 'Provider', {
                onEventHandler: handler,
            });
            new cdk.CustomResource(this, 'SyncTrigger', { serviceToken: myProvider.serviceToken });
            // ensure the mount targets are available as dependency for the sync function
            handler.node.addDependency(props.fileSystem.mountTargetsAvailable);
        }
        else {
            const taskConfig = props.syncSource._createFargateTask(`${id}FargateTask`, this);
            const cluster = new ecs.Cluster(this, 'Cluster', { vpc: props.vpc });
            const runTask = new cdk_fargate_run_task_1.RunTask(this, 'SyncTrigger', {
                task: taskConfig.task,
                securityGroup: taskConfig.securityGroup,
                cluster,
                fargatePlatformVersion: cdk_fargate_run_task_1.PlatformVersion.V1_4_0,
            });
            runTask.node.addDependency(props.fileSystem.mountTargetsAvailable);
        }
    }
}
exports.SyncedAccessPoint = SyncedAccessPoint;
_d = JSII_RTTI_SYMBOL_1;
SyncedAccessPoint[_d] = { fqn: "cdk-efs-assets.SyncedAccessPoint", version: "0.3.93" };
//# sourceMappingURL=data:application/json;base64,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