import os
from typing import List, Union
from typing_extensions import Literal  # for py3.7 support
import inspect
import subprocess
import re
import argparse
import datetime
import json
import time
import functools
import requests
from fabric import Connection
from paramiko.ssh_exception import PasswordRequiredException
from invoke import UnexpectedExit
from jinja2 import Environment, FileSystemLoader, Template
from colorama import Style, Fore
import yaml
import secrets
import configparser

try:
    # ipydex is used for debugging only
    # noinspection PyUnresolvedReferences
    from ipydex import IPS
except ImportError:
    def IPS(*args, **kwargs):
        pass


class Container(object):
    def __init__(self, **kwargs):
        self.__dict__.update(**kwargs)


class EContainer(Container):
    def __init__(self, **kwargs):
        self.exited = None
        self.stdout = ""
        self.stderr = ""
        super().__init__(**kwargs)


# it is useful for deployment scripts to handle cli arguments
# the following reduces the boilerplate
argparser = argparse.ArgumentParser()
argparser.add_argument(
    "target", help="deployment target: `local` or `remote`.", choices=["local", "remote"], default="remote"
)
argparser.add_argument("-u", "--unsafe", help="omit security confirmation", action="store_true")
argparser.add_argument("-i", "--initial", help="flag for initial deployment", action="store_true")
argparser.add_argument(
    "-l",
    "--symlink",
    help="use symlinking instead of copying (local deployment only)",
    action="store_true",
)


def parse_args(*args, **kwargs):
    args = argparser.parse_args(*args, **kwargs)
    if args.target != "local" and args.symlink:
        raise ValueError(f"incompatible options: target: {args.target} and --symlink: True")
    return args


def render_template(tmpl_path, context, target_path=None):
    """
    Render a jinja2 template and save it to target_path. If target_path ist `None` (default),
    autogenerate it by dropping the then mandatory `template_` substring of the templates filename.

    :param tmpl_path:
    :param context:     dict with context data for rendering
    :param target_path: None or string
    :return:
    """

    path, fname = os.path.split(tmpl_path)
    assert path != ""

    jin_env = Environment(loader=FileSystemLoader(path))

    if target_path is None:
        special_str = "template_"
        assert (
            fname.startswith(special_str)
            and (fname.count(special_str) == 1)
            and len(fname) > len(special_str)
        )
        res_fname = fname.replace(special_str, "")
        target_path = os.path.join(path, res_fname)

    template = jin_env.get_template(fname)
    if "warning" not in context:
        context["warning"] = "This file was autogenerated from the template: {}".format(fname)
    result = template.render(context=context)

    target_dir = os.path.dirname(target_path)
    os.makedirs(target_dir, exist_ok=True)

    with open(target_path, "w") as resfile:
        resfile.write(result)

    # also return the result (useful for testing)
    return result


def merge_dicts(a, b, path=None):
    """
    merges dict b into dict a. In case of conflict: choose value from b
    source: https://stackoverflow.com/a/7205107/333403 (adapted)

    :param a:
    :param b:
    :param path:
    :return:
    """
    if path is None:
        path = []
    for key in b:
        if key in a:
            if isinstance(a[key], dict) and isinstance(b[key], dict):
                merge_dicts(a[key], b[key], path + [str(key)])
            else:
                # overwrite a[key]
                a[key] = b[key]
        else:
            # add a[key]
            a[key] = b[key]
    return a


def render_json_template(base_data_path, new_data, target_path, data_format=None):
    """
    Load data from a json file, update the dict with new_data and save it under target_path

    :param base_data_path:
    :param new_data:
    :param target_path:
    :param data_format:          explicitly specify format or None (guess)
    :return:
    """

    if (data_format is None and base_data_path.endswith(".json")) or data_format == "json":
        load_func = json.load
    elif (data_format is None and base_data_path.endswith(".yml")) or data_format == "yaml":
        load_func = yaml.safe_load
    else:
        raise ValueError(f"Unknown format for {base_data_path}")

    with open(base_data_path) as jsonfile:
        payload_data = load_func(jsonfile)
    assert isinstance(payload_data, dict)
    merge_dicts(payload_data, new_data)
    template = Template("""{{ data | tojson(indent=2) }}""")
    result = template.render(data=payload_data)

    target_dir = os.path.dirname(target_path)
    os.makedirs(target_dir, exist_ok=True)

    with open(target_path, "w") as resfile:
        resfile.write(result)

    return result


class StateConnection(object):
    """
    Wrapper class for fabric connection which remembers the working directory. Also has a target attribute to
    distinguish between remote and local operation.
    """

    def __init__(self, remote, user, target="remote"):
        self.dir = None
        self.cwd = None
        self.venv_path = None
        self.venv_target = None
        self.last_result = None
        self.last_command = None
        self.remote = remote
        self.user = user
        self.env_variables = {}

        assert target in ("remote", "local")
        self.target = target
        if target == "remote":
            self._c = Connection(remote, user)
            res = self.run('echo "Connection successful!"', hide=True)
            if res.exited != 0:
                msg = "Could not connect via ssh. Ensure that ssh-agent is activated."
                raise SystemExit(msg)
        else:
            self._c = None

    def cprint(self, txt, target_spec="both"):
        """
        Colored print-function. Color (bright or gray) depends on `target_spec` and `self.target`.

        :param txt:           the string to print
        :param target_spec:   one of `both` (default), `remote` or `local`
        :return:
        """

        if target_spec in (self.target, "both"):
            print(bright(txt))
        else:
            msg = f"Omit: (target_spec is not {self.target} "
            print(dim(f"{msg}{txt}"))

    def activate_venv(self, venv_path, venv_target: Literal["remote", "both"] = "remote"):
        """
        Store the virtual environment which should be activated for all commands (until deactivation).
        Also execute a test command.

        :param venv_path:    path to the activate script
        :param venv_target:  target platform
        """

        self.venv_path = venv_path
        self.venv_target = venv_target

        # !! this assumes to run remote
        return self.run('python -c "import sys; print(sys.path)"')

    def deactivate_venv(self):

        self.venv_path = None
        self.venv_target = None

    def chdir(
        self, path, target_spec: Literal["remote", "local", "both"] = "both", tolerate_error=False
    ):
        """
        The following works on uberspace:

        c.chdir("etc")
        c.chdir("~")
        c.chdir("$HOME")

        :param path:
        :param target_spec:
        :param tolerate_error:
        :return:
        """

        if path is None:
            self.dir = None
            return

        assert len(path) > 0

        # handle relative paths

        if path[0] not in ("/", "~", "$"):
            # path is a relative directory
            if self.dir is None:
                # this should prevent too hazardous invocations
                msg = "Relative path cannot be the first path specification"
                raise ValueError(msg)

            pwd_res = self.run("pwd", hide=True, warn=True, target_spec=target_spec)
            assert pwd_res.exited == 0
            abs_path = f"{pwd_res.stdout.strip()}/{path}"
        else:
            # !! handle the cases of $RELATIVE_PATH and $UNDEFINED (however, not so important)
            abs_path = path

        old_path = self.dir
        self.dir = abs_path

        cmd = "pwd"
        res = self.run(cmd, hide=True, warn=True, target_spec=target_spec)
        pwd_txt = res.stdout.strip()

        if res.exited != 0:
            print(bred(f"Could not change directory. Error message: {res.stderr}"))
            self.dir = old_path

        # assure they have the last component in common
        # the rest might differ due to symlinks and relative paths
        elif not pwd_txt.endswith(os.path.split(path)[1]):
            if not tolerate_error and not path.startswith("~") and not path.startswith("$"):
                print(bred(f"Could not change directory. `pwd`-result: {res.stdout}"))
            self.dir = old_path
            res = EContainer(exited=1, old_res=res)

        return res

    def set_env(self, name: str, value: str):
        self.env_variables[name] = value

    def run(
        self,
        cmd,
        use_dir: bool = True,
        hide: bool = False,
        warn: Union[bool, str] = "smart",
        printonly=False,
        target_spec: Literal["remote", "local", "both"] = "remote",
        use_venv: bool = True,
    ):
        """

        :param cmd:             the command to execute, preferably as a list like it is expected by subprocess
        :param use_dir:         boolean flag whether or not to use self.dir
        :param use_venv:        boolean flag whether or not to use self.venv_path
        :param hide:            see docs of invoke {"out", "err", True, False}
        :param warn:            see docs of invoke and handling of "smart" below
        :param printonly:       flag for debugging
        :param target_spec:     str; default: "remote"
        :return:
        """

        # full_command_list will be a list of lists
        if isinstance(cmd, list):
            full_command_list = [cmd]
        else:
            full_command_list = [cmd.split(" ")]

        cmd_txt = " ".join(full_command_list[-1])

        self.cwd = None  # reset possible residuals from last call
        if use_dir and self.dir is not None:
            if self.target == "remote":
                full_command_list.insert(0, ["cd", self.dir])
            else:
                self.cwd = self.dir

        assert target_spec in ("remote", "local", "both")
        assert self.venv_target in (None, "remote", "both")

        venv_target_condition = self.venv_target == "both" or (
            target_spec != "local" and self.venv_target is not None
        )

        if use_venv and self.venv_path is not None and venv_target_condition:
            full_command_list.insert(0, ["source", self.venv_path])

        for env_var, value in self.env_variables.items():
            full_command_list.insert(0, ["export", f'{env_var}="{value}"'])

        self.last_command = full_command_list

        if warn == "smart":
            # -> get a result object (which would not be the case for warn=False)
            warn = True

            # safe the result self.last_result
            smart_error_handling = True
        else:
            smart_error_handling = False

        if not hide:
            print(dim("-> "), cmd_txt)

        if not printonly:
            # noinspection PyUnusedLocal
            try:
                if not hide:
                    print(dim("<- "), end="")
                res = self.run_target_command(
                    full_command_list, hide=hide, warn=warn, target_spec=target_spec
                )
            except UnexpectedExit as ex:

                if warn:
                    # fabric/invoke raises this error on "normal failure"

                    msg = (
                        f"The command {cmd} failed. You can run it again with `warn=smart`"
                        f"(recommended) or `warn=True` and inspect `result.stderr` to "
                        f"get more information.\nOriginal exception follows:\n"
                    )

                    raise ValueError(msg)
                else:
                    res = EContainer(exited=1, exception=ex)

            except PasswordRequiredException as ex:
                print(bred("Could not connect via ssh. Ensure that ssh-agent is activated."))
                print(dim("hint: use something like `eval $(ssh-agent); ssh-add -t 1m`\n"))
                res = EContainer(exited=1, exception=ex)
            else:
                self.last_result = res
                if smart_error_handling and res.exited != 0:
                    msg = (
                        f"The command `{cmd}` failed with code {res.exited}. This is res.stderr:\n\n"
                        f"{res.stderr}\n\n"
                        "You can also investigate c.last_result and c.last_command"
                    )
                    raise ValueError(msg)
        else:
            # printonly mode
            res = EContainer(exited=0)

        return res

    def string_to_file(self, txt: str, fpath, mode=">"):
        import base64
        """
        Convert string to base64 and on remote site use `echo` and `| base64 -d >` to write it to a file.
        Motivation: circumvent bug-prone escaping of special chars in combination of bash, ssh and python

        :return: the result of `cat fpath`
        """

        txt_b64 = base64.b64encode(txt.encode("utf8")).decode("utf8")
        self.run(f"echo {txt_b64} | base64 -d {mode} {fpath}")
        res = self.run(f"cat {fpath}")
        return res.stdout

    def edit_file(self, fpath: str, old: str, new, delete_aux_files=True):
        """
        upload `old` and `new` to remote site and then run `search_and_replace.py`

        :param fpath:   path to file to edit
        :param old:     old string
        :param new:     new string
        :param delete_aux_files:
                        flag to simplify debugging
        """
        s_and_r_fname = "search_and_replace.py"
        s_and_r_fpath = f"{get_dir_of_this_file()}/{s_and_r_fname}"
        assert os.path.isfile(s_and_r_fpath)

        import tempfile
        ts = time.strftime("%Y-%m-%d__%H-%M-%S")

        old_string_fpath = tempfile.mktemp(prefix=f"du_{ts}_old_", suffix=".txt")
        new_string_fpath = tempfile.mktemp(prefix=f"du_{ts}_new_", suffix=".txt")
        with open(old_string_fpath, 'w', encoding='utf-8') as fp: fp.write(old)
        with open(new_string_fpath, 'w', encoding='utf-8') as fp: fp.write(new)

        target_spec = "remote"

        old_string_target_path = f"~/tmp/{os.path.split(old_string_fpath)[1]}"
        self.rsync_upload(old_string_fpath, old_string_target_path, target_spec)

        # same for new
        new_string_target_path = f"~/tmp/{os.path.split(new_string_fpath)[1]}"
        self.rsync_upload(new_string_fpath, new_string_target_path, target_spec)


        s_and_r_target_fpath = f"~/tmp/{s_and_r_fname}"
        self.rsync_upload(s_and_r_fpath, s_and_r_target_fpath, target_spec)
        self.run(f"python3 {s_and_r_target_fpath} {fpath} {old_string_target_path} {new_string_target_path}")

        if delete_aux_files:
            os.remove(old_string_fpath)
            os.remove(new_string_fpath)
            self.run(f"rm {old_string_target_path}")
            self.run(f"rm {new_string_target_path}")

        return old, new

    def multi_edit_file(self, fpath: str, replacements: list[tuple[str]], delete_aux_files=True):
        """
        convert `replacements` to json, upload to remote site and then run `multi_search_and_replace.py`

        :param fpath:   path to file to edit
        :param replacements: list of tuples (old_string, new_string)
        :param delete_aux_files:
                        flag to simplify debugging
        """

        assert isinstance(replacements, list)
        for tup in replacements:
            assert isinstance(tup, (tuple, list))
            assert len(tup) == 2
            assert isinstance(tup[0], str) and isinstance(tup[1], str)

        import tempfile
        ts = time.strftime("%Y-%m-%d__%H-%M-%S")

        rplmt_data = {
            "target_file": fpath,
            "replacements": replacements,
        }

        replacements_fpath = tempfile.mktemp(prefix=f"du_{ts}_replacements_", suffix=".json")
        with open(replacements_fpath, 'w', encoding='utf-8') as fp: json.dump(rplmt_data, fp, indent=2)

        target_spec = "remote"

        replacements_target_path = f"~/tmp/{os.path.split(replacements_fpath)[1]}"
        self.rsync_upload(replacements_fpath, replacements_target_path, target_spec)

        m_s_and_r_fname = "multi_search_and_replace.py"
        m_s_and_r_fpath = f"{get_dir_of_this_file()}/{m_s_and_r_fname}"
        assert os.path.isfile(m_s_and_r_fpath)

        m_s_and_r_target_fpath = f"~/tmp/{m_s_and_r_fname}"
        self.rsync_upload(m_s_and_r_fpath, m_s_and_r_target_fpath, target_spec)
        self.run(f"python3 {m_s_and_r_target_fpath} {replacements_target_path}")

        if delete_aux_files:
            os.remove(replacements_fpath)
            self.run(f"rm {replacements_target_path}")

    def run_target_command(
        self, full_command_lists: List[list], hide: bool, warn: bool, target_spec: str
    ) -> Union[EContainer, subprocess.CompletedProcess]:
        """
        Actually run the command (or not), depending on self.target and target_spec.

        :param full_command_lists:  nested list of commands like: [["cd", "/path"], ["echo", "$(pwd)"]]
        :param hide:
        :param warn:
        :param target_spec:
        :return:
        """

        assert isinstance(full_command_lists, list) and isinstance(full_command_lists[0], list)

        full_command_txt = "; ".join([" ".join(cmd_list) for cmd_list in full_command_lists])

        # this is only for status messages
        last_command = " ".join(full_command_lists[-1])
        omit_message = dim(
            f"> Omitting command `{last_command}`\n> due to target_spec: {target_spec}."
        )

        assert self.target in ("local", "remote"), f"Invalid target: {self.target}"
        if self.target == "remote":

            if target_spec in ("remote", "both"):
                res = self._c.run(full_command_txt, hide=hide, warn=warn)
            else:
                print(omit_message)
                res = EContainer(exited=0, command_omitted=True)
        else:
            # -> self.target != "remote"
            # TODO : handle warn flag
            if target_spec in ("local", "both"):
                orig_dir = os.getcwd()

                if self.cwd:
                    # necessary because subprocess.run does not work with "cd my/path; my_command"
                    os.chdir(self.cwd)

                res = subprocess.run(
                    full_command_txt, capture_output=True, shell=True, executable="/bin/bash"
                )
                res.exited = res.returncode
                res.stdout = res.stdout.decode("utf8")
                res.stderr = res.stderr.decode("utf8")

                os.chdir(orig_dir)
                if res is not None and res.stdout and hide not in (True, "out"):
                    print(res.stdout)

            else:
                # -> self.target != "remote" but target_spec == "remote"
                print(omit_message)
                res = EContainer(exited=0, command_omitted=True)

        return res

    def rsync_upload(
        self,
        source,
        dest,
        target_spec,
        filters="",
        printonly=False,
        tol_nonzero_exit=False,
        delete=False,
        additional_flags="",
    ):
        """
        Perform the appropriate rsync command (or not), depending on self.target and target_spec.

        :param source:
        :param dest:
        :param target_spec:
        :param filters:
        :param printonly:
        :param tol_nonzero_exit:    boolean; tolerate nonzero exit code
        :param delete:              insert the --delete flag
        :param additional_flags:    possibility to add more flags
        :return:
        """

        # construct the destination
        if self.target == "remote":
            full_dest = f"{self.user}@{self.remote}:{dest}"
            self.check_rsync()
        else:
            full_dest = dest

        return self._rsync_call(
            source,
            full_dest,
            target_spec,
            filters,
            printonly=printonly,
            tol_nonzero_exit=tol_nonzero_exit,
            delete=delete,
            additional_flags=additional_flags,
        )

    def rsync_download(
        self,
        source,
        dest,
        target_spec,
        filters="",
        printonly=False,
        tol_nonzero_exit=False,
        delete=False,
        additional_flags="",
    ):
        """
        Perform the appropriate rsync command (or not), depending on self.target and target_spec.

        :param source:
        :param dest:
        :param target_spec:
        :param filters:
        :param printonly:
        :param tol_nonzero_exit:    boolean; tolerate nonzero exit code
        :param delete:              insert the --delete flag
        :param additional_flags:    possibility to add more flags
        :return:
        """

        # construct the source
        if self.target == "remote":
            full_source = f"{self.user}@{self.remote}:{source}"
            self.check_rsync()
        else:
            full_source = source

        return self._rsync_call(
            full_source,
            dest,
            target_spec,
            filters,
            printonly=printonly,
            tol_nonzero_exit=tol_nonzero_exit,
            delete=delete,
            additional_flags=additional_flags,
        )

    def check_rsync(self):
        res = self.run("rsync --version", warn=False)
        if not res.exited == 0:
            msg = "rsync must be installed on the remote machine but `rsync --version` failed."
            raise ValueError(msg)

    def _rsync_call(
        self,
        source,
        dest,
        target_spec,
        filters,
        printonly=False,
        tol_nonzero_exit=False,
        delete=False,
        additional_flags="",
    ):

        if delete is True:
            d = " --delete"
        else:
            d = ""

        if additional_flags:
            additional_flags = f" {additional_flags.lstrip()}"

        if self.target == "remote":
            cnctn = " --rsh='ssh  -p 22'"
        else:
            cnctn = ""

        cmd_start = f"rsync -pthrvz{d}{additional_flags}{cnctn}"
        cmd = f"{cmd_start} {filters} {source} {dest}"

        if printonly:
            print("->:", cmd)
            res = EContainer(exited=0)
        elif target_spec != "both" and self.target != target_spec:
            print(dim(f"> Omitting rsync command `{cmd}`\n> due to target_spec: {target_spec}."))
            res = EContainer(exited=0)
        else:
            # TODO: instead of locally calling rsync, find a more elegant (platform-independent) way to do this
            exitcode = os.system(cmd)
            res = EContainer(exited=exitcode)

            if not tol_nonzero_exit and exitcode != 0:
                msg = "rsync failed. See error message above."
                raise ValueError(msg)
        return res

    def deploy_this_package(self, pip_command="pip"):
        """
        Deploy the current version of this package to the remote host. This is a convenience function,
        to prevent to publish too much development versions of this package to pypi or git repo.

        :return:     None
        """

        assert self.target == "remote"

        project_main_dir = get_dir_of_this_file(
            upcount_dir=2
        )  # this is where setup.py lives (top level)
        assert os.path.isfile(f"{project_main_dir}/setup.py")

        package_dir = project_main_dir
        package_dir_name = os.path.split(package_dir)[1]
        package_name = os.path.split(get_dir_of_this_file())[1]

        filters = (
            f"--exclude='.git/' "
            f"--exclude='.idea/' "
            f"--exclude='*/__pycache__/*' "
            f"--exclude='__pycache__/' "
        )

        self.rsync_upload(package_dir, "~/tmp", filters=filters, target_spec="remote")

        self.run(f"{pip_command} uninstall -y {package_name}", warn=False)

        self.run(f"{pip_command} install ~/tmp/{package_dir_name}")

    def deploy_local_package(
        self, local_path, target_path=None, pip_command="pip", pip_flags="", package_name=None
    ):
        """
        Upload and deploy a package from the local machine to a remote machine. This is useful to easily deploy
        local development versions. "Deployment" means installation with pip.

        :param local_path:      the directory where setup.py lies
        :param target_path:     target_path for rsync (parent dir of package_dir )
        :param pip_command:
        :param pip_flags:       obsolete, only for backward compatibility
        :param package_name:
        :return:
        """

        filters = (
            f"--exclude='.git/' "
            f"--exclude='.aider/' "
            f"--exclude='.idea/' "
            f"--exclude='*/__pycache__/*' "
            f"--exclude='__pycache__/' "
        )

        if target_path is None:
            target_path = "~/tmp"
        package_dir_name = os.path.split(local_path)[1]

        # remove old version
        self.run(f"rm -rf {target_path}/{package_dir_name}")

        self.rsync_upload(local_path, "~/tmp", filters=filters, target_spec="remote")

        if package_name:
            self.run(f"{pip_command} uninstall -y {package_name}", warn=False)

        self.run(f"{pip_command} install {target_path}/{package_dir_name}")

    def check_existence(self, path, target_spec="remote", operator_flag="-e"):
        """
        Check the existence of a remote file or directory.

        :param path:
        :param target_spec:
        :param operator_flag:   "-e" (default, both directory and file), "-d" (directory), "-f" (file)
        :return:
        """

        if not target_spec == "both":
            assert target_spec == self.target

        cmd = f"test {operator_flag} {path}"
        res = self.run(cmd, warn=False)
        return res.exited == 0


def warn_user(app_name, target, unsafe_flag, deployment_path, user=None, host=None):

    user_at_host = f"{user}@{host}"
    print(
        f"\n  You are running the deployment for {bright(app_name)} with target {bright(target)} "
        f"→ {bright(user_at_host)},\n"
        f"\n  deployment path: `{deployment_path}`.\n"
        f"\n  {yellow('Caution:')} All existing user data of the app and any other changes in the\n"
        f"  deployment directory will probably be be replaced by predefined data and fixtures.\n\n"
    )

    if not unsafe_flag:
        res = input("Continue (N/y)? ")
        if res.lower() != "y":
            print(bred("Aborted."))
            exit()


def get_dir_of_this_file(upcount: int = 1, upcount_dir: int = 0):
    """
    Assumes that this function is called from a script. Return the path of that script (excluding the script itself).

    :param upcount:     specifies how many frames to go back/up. default: 1 (means that the caller-frame is processed)
    :param upcount_dir: specifies how many directories to go up (default: 0)
    """

    frame = inspect.currentframe()
    for i in range(upcount):
        frame = frame.f_back

    dn = os.path.dirname(os.path.abspath(inspect.getfile(frame)))

    # if specified, go upwards some additional levels
    for i in range(upcount_dir):
        dn = os.path.dirname(dn)

    return dn


def preserve_cwd(function):
    """
    This is a decorator that ensures that the current working directory is unchanged during the function call.
    """

    @functools.wraps(function)
    def decorator(*args, **kwargs):
        cwd = os.getcwd()
        try:
            return function(*args, **kwargs)
        finally:
            os.chdir(cwd)

    return decorator


@preserve_cwd
def get_nearest_config(
    fname: str = "config.ini",
    limit: int = None,
    devmode: bool = False,
    start_dir: Union[str, None] = None,
):
    """
    Look for a file `fname` in the directory of the calling file and then up the tree (up to `limit`-steps).

    Advantage over directly using `from decouple import config` the full filename can be defined explicitly.

    :param fname:       filename or absolute path
    :param limit:       How much steps to go up at maximum (default: 4, if fname is only a filename)
    :param devmode:     Flag that triggers development mode (default: False).
                        If True variables which end with "__DEVMODE" will replace variables without such appendix

    :param start_dir:   (optional) start directory

    :return:    config object from decouple module
    """
    assert fname.endswith(".ini") or fname.endswith(".toml")

    path0, fname = os.path.split(fname)

    if path0 != "":
        assert start_dir is None
        assert limit is None
        path0 = os.path.abspath(path0)
        limit = 0
    elif limit is None:
        limit = 4  # set the default value if fname was only a filename

    old_dir = os.getcwd()

    if start_dir is None:
        if path0 == "":
            start_dir = get_dir_of_this_file(upcount=3)
        else:
            start_dir = path0
    else:
        assert os.path.isdir(start_dir)
    os.chdir(start_dir)

    path_list = [fname]
    for i in range(limit + 1):
        path = os.path.join(*path_list)
        if os.path.isfile(path):
            break
        path_list.insert(0, "..")
    else:
        msg = f"Could not find {fname} in current directory nor in {limit} parent dirs."
        raise FileNotFoundError(msg)

    # this is kept local to keep the dependency optional

    if fname.endswith(".ini"):
        from decouple import Config, RepositoryIni, Csv
        config = Config(RepositoryIni(path))
        config.settings_dict = config.repository.parser.__dict__["_sections"]["settings"]
        # enable convenient access to Csv parser
        config.Csv = Csv
    elif fname.endswith(".toml"):
        config = TOMLConfig(path)

    if devmode:
        for key, value in config.settings_dict.items():
            # it seems that keys are converted to lowercase automatically
            if "__" in key:
                suffix = key.split("__")[-1]
            else:
                suffix = None
            if suffix in ("devmode", "DEVMODE"):

                # use the value specified for the development-mode for the actual variable (if it exists)
                main_key = key.replace(f"__{suffix}", "")
                if main_key in config.settings_dict:
                    config.settings_dict[main_key] = value

    # enable convenient access to the actual path of the file and the containing directory
    config.path = os.path.abspath(path)
    config.dirpath = os.path.dirname(os.path.abspath(path))

    os.chdir(old_dir)
    return config


# based on decouple.Config
class TOMLConfig(object):
    """
    Handle .toml file format used by Foreman.
    """

    def __init__(self, fpath):
        try:
            # this will be part of standard library for python >= 3.11
            import tomllib
        except ModuleNotFoundError:
            import tomli as tomllib

        # variable substitution regex
        self.vsre = re.compile(r"%\((.+?)\)s")

        with open(fpath, "rb") as fp:
            complete_dict = tomllib.load(fp)
            self.settings_dict = complete_dict

        self._perform_replacements(self.settings_dict)

    @staticmethod
    def _cast_do_nothing(value):
        return value

    def _get_table_content(self, key):
        """
        assume key = "table1::entry_name" -> resolve the "::"-separated parts
        """
        assert key not in self.settings_dict

        current_dict = self.settings_dict
        parts = key.split("::")
        len_parts = len(parts)

        used_parts = []

        for i, part_key in enumerate(parts, start=1):
            res = current_dict[part_key]

            if i == len_parts:
                return res

            used_parts.append(part_key)
            if not isinstance(res, dict):
                msg = (
                    f"the partial key {'::'.join(used_parts)} does not address a table "
                    f"but is of type {type(res)} instead"
                )
                raise TypeError(msg)

            current_dict = res

    def get(self, key, ignore_undefined=False, default=None):
        """
        Return the value for option or default if allowed.
        """

        if key in self.settings_dict:
            value = self.settings_dict[key]
        elif "::" in key:
            # Note: we use "::" instead of "." as table separator because some table name are urls (containing ".")
            try:
                value = self._get_table_content(key)
            except KeyError:
                if not ignore_undefined:
                    raise
                value = default
        else:
            if not ignore_undefined:
                raise KeyError(key)

            value = default

        if not isinstance(value, str):
            return value

        return value

    def _perform_replacements(self, datadict: dict):
        """
        This methods performs all variable replacements once and for all
        """

        for key, value in datadict.items():
            if isinstance(value, dict):
                self._perform_replacements(value)

            elif isinstance(value, list):
                new_value = []
                for elt in value:
                    if isinstance(elt, str):
                        res = self._replace_str(elt)
                    else:
                        res = elt
                    new_value.append(res)
                datadict[key] = new_value

            elif isinstance(value, str):
                # perform the substitution
                datadict[key] = self._replace_str(value)

            else:
                # different datatype -> nothing todo
                pass

    def _replace_str(self, value: str) -> str:
        matches = list(self.vsre.finditer(value))
        if matches:
            start_idx = 0
            new_value_parts = []
            for match in matches:
                new_value_parts.append(value[start_idx:match.start()])

                var_name = match.groups(1)[0]
                var_value = self.get(var_name)
                new_value_parts.append(var_value)
                start_idx = match.end()

            # add the part after the last match
            new_value_parts.append(value[start_idx:])
            res = "".join(new_value_parts)
        else:
            res = value
        return res


    def __call__(self, *args, **kwargs):
        """
        Convenient shortcut to get.
        """
        return self.get(*args, **kwargs)


def set_repo_tag(
    ref_path: str = None, message: str = None, repo_path: str = None, ask=True
) -> None:
    """
    Set a git tag to the current or specified repo (default: `deploy/<datetime>`)

    :param ref_path:    name of the tag; default: `deploy/<datetime>`
    :param message:     message, optional
    :param ask:         flag whether to ask before tagging
    :param repo_path:   path to repository (optional); if not provided take the parent dir of the calling script

    :return:
    """

    try:
        from git import Repo, InvalidGitRepositoryError
    except ImportError:
        err_msg = "Could not import `git`-package. Omit tagging."
        print(yellow(err_msg))
        return None

    if ask:
        res = input("\n should a new tag be created for the git repo (y/N)? ")

        if res.lower().strip() != "y":
            return

    if repo_path is None:
        # assume that this function is called from a deployment script which lives in repo_root/subdir/deploy.py
        repo_path = get_dir_of_this_file(upcount=2, upcount_dir=0)

    repo_path = os.path.abspath(repo_path)
    assert os.path.isdir(repo_path)

    try:
        repo = Repo(repo_path)
    except InvalidGitRepositoryError:
        err_msg = "Could not find git repository. Omit tagging."
        print(yellow(err_msg))
        return None

    if ref_path is None:
        now = time.strftime("%Y-%m-%d__%H-%M-%S") + f"_{os.environ['TZ']}"

        ref_path = f"deploy/{now}"

    if repo.is_dirty():
        repo.git.commit("-a", "-m", "autocommit during deployment")
    repo.create_tag(ref_path, message)

    print(f"Created tag for repo: `{ref_path}`.")


def ensure_http_response(url, expected_status_code=200, sleep=0):

    assert float(sleep) == sleep and sleep >= 0, f"invalid value for sleep: {sleep}"

    time.sleep(sleep)
    try:
        r = requests.get(url)
    except requests.exceptions.SSLError as err:
        print(bred(f"{url}: There was an SSLError (see below)"))
        print(err)
        return 1

    if r.status_code == expected_status_code:
        print(bgreen(f"{url}: expected status code received: {expected_status_code}."))
        return 0
    else:
        print(bred(f"{url}: unexpected status code: {r.status_code}."))
        return 2


def get_example_values(data: dict) -> dict:
    res = {}
    for k, v in data.items():
        if "__EXAMPLE" in k:
            res[k] = v
    return res


def toml_quote(obj):

    res = repr(obj)

    if obj in [True, False]:
        res = res.lower()
    return res



def remove_secrets_from_config(path, new_path=None):
    """
    Parse the ini/toml file at `path` and create a copy where every non-comment line containing `pass` or `key`
    has a dummy value.

    NOTE: This function has no support for multiline assignments (allowed by toml)

    Use case: When developing deployment software with deployment tools, one often wants to share the
    general configuration but not the secrets. This function serves to automate this process.

    The dummy value is either generated or read from an actual variable name with the pattern
    `<secret-varname>-example`.  E.g. `my_pass-example = toFoh9pu`

    :param path:        path of the original file
    :param new_path:    (optional) path of new file to create

    :return:            `new_path`
    """
    example_values = None
    if path.endswith(".ini"):
        fname_suffix = ".ini"

        # for ini-format: strings should not be quoted
        quote_func = lambda x: x

        config = configparser.ConfigParser()
        config.optionxform = str  # preserve case when parsing the keys (non-default)
        config.read(path)
        keys = config["settings"].keys()
        config.settings_dict = config["settings"]
        example_values = get_example_values(config.settings_dict)

    elif path.endswith(".toml"):
        fname_suffix = ".toml"
        # for toml-format: strings should be quoted
        quote_func = toml_quote
        config = TOMLConfig(path)
        keys = []

        # first: get level 0 example values
        example_values = get_example_values(config.settings_dict)

        # now get level 1 example values and also add direct keys
        for key, value in config.settings_dict.items():
            if isinstance(value, dict):
                keys.extend(value.keys())
                new_exvals = get_example_values(value)
                for ex_key in new_exvals.keys():
                    if ex_key in example_values:
                        msg = (
                            f"The key '{ex_key}' occurs in the dict '{key}' and at least one other dict. "
                            "This is not supported by the script."
                        )
                        raise ValueError(msg)
                example_values.update(new_exvals)
            else:
                keys.append(key)
    else:
        msg = f"Unexpected file type (neither .ini nor .toml): of {path}"
        raise TypeError(msg)

    with open(path) as fp:
        full_text_lines = fp.readlines()

    critical_keys = []
    for k in keys:
        if contains_critical_token(k) and not k.endswith("__EXAMPLE"):
            critical_keys.append(k)

    keys_with_example_values = [k.replace("__EXAMPLE", "") for k in keys if k.endswith("__EXAMPLE")]

    more_critical_keys = process_nested_dicts(config.settings_dict, level=0)

    # critical_keys_with_example_values = set(critical_keys).intersection(keys_with_example_values)
    critical_keys = list(set(critical_keys + more_critical_keys) - set(keys_with_example_values))

    action_keys = critical_keys + keys_with_example_values

    result_lines = []

    for line in full_text_lines:

        line = line.lstrip(" ")
        line_parts = line.split("=")
        for ak in action_keys:
            key_str = line_parts[0].strip()

            if ak == key_str or key_str.replace(ak, "").startswith("__"):
                # action-key found, no need to search further in this line
                break
        else:
            # this else-branch is triggered if the inner for loop got no break
            # no critical key in this line
            # -> use this line and proceed to next one
            result_lines.append(line)
            continue

        assert ak in key_str
        if line.startswith("#"):
            # ignore this line (this might omit useful comments, but safety first!)
            continue

        if ak in critical_keys:
            n = 10
            xx = secrets.token_urlsafe(2 * n)
            new_line = f'{ak} = "{xx[:n]}--example-secret--{xx[n:]}"\n'
            result_lines.append(new_line)
        else:
            assert ak in keys_with_example_values
            if line.startswith(f"{ak}__EXAMPLE"):
                # do not add the example-key-value-line to the output
                # but only the real key with the example value
                continue
            example_value = example_values[f"{ak}__EXAMPLE"]
            new_line = f"{ak} = {quote_func(example_value)}\n"
            result_lines.append(new_line)
            # replacement_lines.append((ak, new_line))

    if new_path is None:
        if "production" in path:
            new_path = path.replace("production", "example")
        else:
            new_path = path.replace(fname_suffix, f"-example{fname_suffix}")
    else:
        if new_path == path:
            raise ValueError("new_path must be different from original path")

    with open(new_path, "w") as fp:
        fp.writelines(result_lines)

    print("The values for the following keys were replaced: ", ", ".join(action_keys))
    print("File written", new_path)
    return new_path

def contains_critical_token(key_str):
    kl = key_str.lower()
    return (("pass" in kl) or ("key" in kl) or ("secret" in kl))

def process_nested_dicts(some_dict, level) -> list:
    """
    iterate over nested tables
    return a list of critical keys
    """
    result = []
    for key, value in some_dict.items():
        if isinstance(value, dict):
            if contains_critical_token(key):
                msg = f"{key} is not allowed as name of a table (cannot remove secrets)"
                raise ValueError(msg)
            else:
                result.extend(process_nested_dicts(value, level+1))
        elif level > 0 and contains_critical_token(key):
            result.append(key)
    return result
# noinspection PyShadowingBuiltins
def get_deployment_date(fpath: str, format="%Y-%m-%d %H:%M:%S") -> str:
    """
    Find out the modification date of a file and return it

    :param fpath:
    :param format:
    :return:
    """

    try:
        timestamp = os.path.getmtime(fpath)
    except FileNotFoundError:
        return "<not available>"

    if format:
        res = datetime.datetime.fromtimestamp(timestamp).strftime(format)
    else:
        res = str(timestamp)

    return res




def dim(txt):
    return f"{Fore.LIGHTBLACK_EX}{txt}{Fore.RESET}"
    # original solution (seems not to work everywhere)
    # return f"{Style.DIM}{txt}{Style.RESET_ALL}"


def bright(txt):
    return f"{Style.BRIGHT}{txt}{Style.RESET_ALL}"


def bgreen(txt):
    return f"{Fore.GREEN}{Style.BRIGHT}{txt}{Style.RESET_ALL}"


def bred(txt):
    return f"{Fore.RED}{Style.BRIGHT}{txt}{Style.RESET_ALL}"


def yellow(txt):
    return f"{Fore.YELLOW}{txt}{Style.RESET_ALL}"
