# -*- coding: utf-8 -*-
"""
Set of HTML utilities.
"""
# ----------------------------------------------------------------------------------------------------------------------
#  Copyright (c) 2023. Antonio Bulgheroni.
#  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
#  documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
#  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
#  permit persons to whom the Software is furnished to do so, subject to the following conditions:
#  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the
#  Software.
#  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
#  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
#  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
#  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
# ----------------------------------------------------------------------------------------------------------------------
from __future__ import annotations

import re
from pathlib import Path

from autologbook import autoconfig, autoerror

__author__ = 'Antonio Bulgheroni'
__email__ = 'antonio.bulgheroni@gmail.com'


class HTMLHelperMixin:
    """Mixin class containing helpful method for HTML generation."""

    @staticmethod
    def convert_path_to_uri(path: str | Path) -> str:
        """
        Convert a path to a URI.

        This utility takes an input path and it manipulates it in order to obtain
        a valid URL.

        First it removes the IMAGE_SERVER_BASE_PATH and replaces it with the
        IMAGE_SERVER_ROOT_URL (both values are taken from the autoconfig constants)
        and then it replaces all back-slashes with normal slashes.

        Parameters
        ----------
        path : Path-like object or string
            The input path to be converted.

        Returns
        -------
        url : URL
            The URL generated by the conversion.

        """
        if isinstance(path, Path):
            path = str(path)

        return path.replace(str(autoconfig.IMAGE_SERVER_BASE_PATH), str(
            autoconfig.IMAGE_SERVER_ROOT_URL)).replace('\\', '/')

    @staticmethod
    def make_url_from_parameters(base_url: str, logbook: str, msg_id: int) -> str:
        """
        Construct an elog URL starting from its elements.

        This method will return a string like this:

        baser_url/logbook/msg_id

        Parameters
        ----------
        base_url: str
            The base url
        logbook: str
            The name of the logbook
        msg_id: int
            The message identification number

        Returns
        -------

        """
        return f'{base_url.rstrip("/")}/{logbook.rstrip("/")}/{msg_id}'

    @staticmethod
    def get_parameters_from_url(url: str) -> (str, str, int):
        """
        Split a ELOG URL in its basic elements.

        Parameters
        ----------
        url: str
            An ELOG URL

        Returns
        -------
        A 3-tuple containing the base url, the logbook name and the message identification number

        """
        pattern = r'(http[s]?://.*)/(.*)/([0-9]+)'
        match = re.match(pattern, url)
        if match:
            return match[1], match[2], int(match[3])
        else:
            raise autoerror.InvalidConnectionParameters
