"""
Observables Core - Advanced API for extending the library

⚠️ DEVELOPMENT STATUS: NOT PRODUCTION READY
This library is under active development. API may change without notice.
Use for experimental and development purposes only.

This module contains the core components and base classes for building on top of the 
Observables library. These are lower-level abstractions meant for users who want to 
create custom observable types or extend the library's functionality.

Core Components:
- BaseXObject: Base class for all observable types
- Hook/HookProtocol: Core hook implementations and protocols
- OwnedHook/HookWithOwnerProtocol: Owned hook implementations and protocols
- FloatingHook: Advanced hook with validation and reaction capabilities
- Nexus: Central storage for actual data values
- BaseCarriesHooks/CarriesSomeHooksProtocol: Base classes for hook carriers
- HookWithIsolatedValidationProtocol: Protocol for hooks with custom validation
- HookWithReactionProtocol: Protocol for hooks that react to changes
- BaseListening/BaseListeningProtocol: Base classes for listener management

Example Usage with New Protocol-Based Architecture:
    >>> from observables.core import BaseXObject, OwnedHook, HookWithOwnerProtocol
    >>> 
    >>> # Create a custom observable type using the new architecture
    >>> class MyCustomObservable(BaseXObject):
    ...     def __init__(self, initial_value):
    ...         # Create owned hook
    ...         self._value_hook = OwnedHook(owner=self, initial_value=initial_value)
    ...         super().__init__({"value": self._value_hook})
    ...     
    ...     @property
    ...     def value(self):
    ...         return self._value_hook.value
    ...     
    ...     @value.setter
    ...     def value(self, new_value):
    ...         self._value_hook.submit_value(new_value)
    ...     
    ...     @property
    ...     def value_hook(self) -> HookWithOwnerProtocol[Any]:
    ...         return self._value_hook

Advanced Usage with FloatingHook:
    >>> from observables.core import FloatingHook
    >>> 
    >>> def validate_value(value):
    ...     return value >= 0, "Value must be non-negative"
    >>> 
    >>> def on_change():
    ...     print("Value changed!")
    ...     return True, "Reaction completed"
    >>> 
    >>> # Create floating hook with validation and reaction
    >>> hook = FloatingHook(
    ...     value=42,
    ...     isolated_validation_callback=validate_value,
    ...     reaction_callback=on_change
    ... )

Configuring Float Tolerance:
    >>> from nexpy import default
    >>> # Adjust tolerance for your use case
    >>> default.FLOAT_ACCURACY = 1e-6  # More lenient for UI
    >>> # This must be done before creating observables

For normal usage of the library, import from the main package:
    >>> from observables import ObservableSingleValue, ObservableList
"""

from .nexus_system.nexus import Nexus
from .auxiliary.listening_base import ListeningBase
from .auxiliary.listening_protocol import ListeningProtocol
from .nexus_system.nexus_manager import NexusManager
from .publisher_subscriber.subscriber import Subscriber
from .nexus_system.submission_error import SubmissionError

# Re-export the module for advanced configuration
# Note: For user-facing configuration, use: from nexpy import default

__all__ = [
    'Nexus',
    'ListeningBase',
    'ListeningProtocol',
    'NexusManager',
    'Subscriber',
    'SubmissionError',
]

