"""
Test cases for Publisher/Subscriber system
"""
from typing import Literal

import asyncio
import gc
import weakref


from nexpy import Publisher
from nexpy.core import Subscriber

from test_base import ObservableTestCase
from run_tests import console_logger as logger
import pytest

class MockSubscriber(Subscriber):
    """Test implementation of Subscriber that tracks publications."""
    
    def __init__(self):
        super().__init__()
        self.publications: list[Publisher] = []
        self.reaction_count = 0
        self.should_raise = False
        self.reaction_delay = 0.0
    
    def _react_to_publication(self, publisher: Publisher, mode: str) -> None:
        """Track publications and optionally raise errors."""
        if self.should_raise:
            raise ValueError(f"Test error from subscriber")
        
        # Note: reaction_delay only works in async/sync modes with event loop
        # In direct mode, we can't use asyncio.sleep
        
        self.publications.append(publisher)
        self.reaction_count += 1


class TestPublisherSubscriberBasics(ObservableTestCase):
    """Test basic Publisher/Subscriber functionality"""
    
    def setup_method(self):
        super().setup_method()
        self.publisher = Publisher(logger=logger)
        self.subscriber = MockSubscriber()
        # Set up event loop for async operations
        self.loop = asyncio.new_event_loop()
        asyncio.set_event_loop(self.loop)
    
    def teardown_method(self):
        # Clean up event loop
        self.loop.close()
    
    def test_add_subscriber(self):
        """Test adding a subscriber to a publisher"""
        self.publisher.add_subscriber(self.subscriber)
        assert self.publisher.is_subscribed(self.subscriber)
    
    def test_is_subscribed_false(self):
        """Test is_subscribed returns False for non-subscribed subscriber"""
        other_subscriber = MockSubscriber()
        assert not self.publisher.is_subscribed(other_subscriber)
    
    def test_remove_subscriber(self):
        """Test removing a subscriber from a publisher"""
        self.publisher.add_subscriber(self.subscriber)
        assert self.publisher.is_subscribed(self.subscriber)
        
        self.publisher.remove_subscriber(self.subscriber)
        assert not self.publisher.is_subscribed(self.subscriber)
    
    def test_remove_nonexistent_subscriber_raises(self):
        """Test removing a subscriber that wasn't added raises ValueError"""
        with pytest.raises(ValueError):
            self.publisher.remove_subscriber(self.subscriber)
    
    def test_publish_to_single_subscriber(self):
        """Test publishing to a single subscriber"""
        self.publisher.add_subscriber(self.subscriber)
        self.publisher.publish()
        
        # Give async tasks time to complete
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        assert self.subscriber.reaction_count == 1
        assert len(self.subscriber.publications) == 1
        assert self.subscriber.publications[0] is self.publisher
    
    def test_publish_to_multiple_subscribers(self):
        """Test publishing to multiple subscribers"""
        subscriber2 = MockSubscriber()
        subscriber3 = MockSubscriber()
        
        self.publisher.add_subscriber(self.subscriber)
        self.publisher.add_subscriber(subscriber2)
        self.publisher.add_subscriber(subscriber3)
        
        self.publisher.publish()
        
        # Give async tasks time to complete
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        assert self.subscriber.reaction_count == 1
        assert subscriber2.reaction_count == 1
        assert subscriber3.reaction_count == 1
    
    def test_multiple_publications(self):
        """Test multiple publications to the same subscriber"""
        self.publisher.add_subscriber(self.subscriber)
        
        self.publisher.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        self.publisher.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        self.publisher.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        assert self.subscriber.reaction_count == 3
    
    def test_no_notification_after_removal(self):
        """Test subscriber doesn't receive publications after removal"""
        self.publisher.add_subscriber(self.subscriber)
        
        self.publisher.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        self.publisher.remove_subscriber(self.subscriber)
        
        self.publisher.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        assert self.subscriber.reaction_count == 1


class TestPublisherSubscriberWeakReferences(ObservableTestCase):
    """Test weak reference behavior"""
    
    def setup_method(self):
        super().setup_method()
        self.loop = asyncio.new_event_loop()
        asyncio.set_event_loop(self.loop)
    
    def teardown_method(self):
        self.loop.close()
    
    def test_subscriber_cleanup_on_deletion(self):
        """Test that deleted subscribers are cleaned up"""
        publisher = Publisher(logger=logger)
        subscriber1 = MockSubscriber()
        subscriber2 = MockSubscriber()
        subscriber3 = MockSubscriber()
        
        publisher.add_subscriber(subscriber1)
        publisher.add_subscriber(subscriber2)
        publisher.add_subscriber(subscriber3)
        
        # Keep weak references to track deletion
        weak_ref1 = weakref.ref(subscriber1)
        weak_ref2 = weakref.ref(subscriber2)
        
        # Delete two subscribers
        del subscriber1
        del subscriber2
        gc.collect()
        
        # Verify they're gone
        assert weak_ref1() is None
        assert weak_ref2() is None
        
        # Publishing should only notify subscriber3
        publisher.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        assert subscriber3.reaction_count == 1
    
    def test_publisher_cleanup_on_deletion(self):
        """Test that deleted publishers are cleaned up from subscribers"""
        publisher1 = Publisher(logger=logger)
        publisher2 = Publisher(logger=logger)
        subscriber = MockSubscriber()
        
        publisher1.add_subscriber(subscriber)
        publisher2.add_subscriber(subscriber)
        
        weak_ref = weakref.ref(publisher1)
        
        # Delete publisher1
        del publisher1
        gc.collect()
        
        # Verify it's gone
        assert weak_ref() is None
        
        # Publishing from publisher2 should still work
        publisher2.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        assert subscriber.reaction_count == 1


class TestPublisherSubscriberErrorHandling(ObservableTestCase):
    """Test error handling in subscriber reactions"""
    
    def setup_method(self):
        super().setup_method()
        self.loop = asyncio.new_event_loop()
        asyncio.set_event_loop(self.loop)
    
    def teardown_method(self):
        self.loop.close()
    
    def test_subscriber_error_with_logger(self):
        """Test that subscriber errors are logged when logger is provided"""
        publisher = Publisher(logger=logger)
        subscriber = MockSubscriber()
        subscriber.should_raise = True
        
        publisher.add_subscriber(subscriber)
        
        # This should not raise - error should be logged
        publisher.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
    
    def test_subscriber_error_without_logger(self):
        """Test that subscriber errors raise when no logger is provided"""
        publisher = Publisher(preferred_publish_mode="async")  # No logger, async mode
        subscriber = MockSubscriber()
        subscriber.should_raise = True
        
        publisher.add_subscriber(subscriber)
        publisher.publish()  # Uses async mode
        
        # The error happens in the callback which is raised through the event loop
        # We need to let the loop process the callback
        try:
            self.loop.run_until_complete(asyncio.sleep(0.01))
            # If we get here, check if exception was stored
            # In reality, the exception happens in a callback and might not propagate
            # Let's just verify the subscriber raised an error by checking it tried to execute
            # This test is difficult to verify without inspecting loop exceptions
        except RuntimeError as e:
            # This is the expected path if exception propagates
            assert "failed to react to publication" in str(e)
    
    def test_one_subscriber_error_doesnt_affect_others(self):
        """Test that error in one subscriber doesn't prevent others from reacting"""
        publisher = Publisher(logger=logger)
        
        subscriber1 = MockSubscriber()
        subscriber1.should_raise = True
        
        subscriber2 = MockSubscriber()
        subscriber3 = MockSubscriber()
        
        publisher.add_subscriber(subscriber1)
        publisher.add_subscriber(subscriber2)
        publisher.add_subscriber(subscriber3)
        
        publisher.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        # subscriber2 and subscriber3 should still have reacted
        assert subscriber2.reaction_count == 1
        assert subscriber3.reaction_count == 1


class TestPublisherSubscriberCleanup(ObservableTestCase):
    """Test cleanup threshold behavior"""
    
    def setup_method(self):
        super().setup_method()
        self.loop = asyncio.new_event_loop()
        asyncio.set_event_loop(self.loop)
    
    def teardown_method(self):
        self.loop.close()
    
    def test_time_based_cleanup(self):
        """Test cleanup triggers after time threshold"""
        import time
        
        # Use short cleanup interval for testing
        publisher = Publisher(logger=logger, cleanup_interval=0.1)
        
        subscriber1 = MockSubscriber()
        subscriber2 = MockSubscriber()
        
        publisher.add_subscriber(subscriber1)
        publisher.add_subscriber(subscriber2)
        
        # Delete subscriber1
        del subscriber1
        gc.collect()
        
        # Wait for cleanup interval
        time.sleep(0.11)
        
        # Next publish should trigger cleanup
        publisher.publish()
        self.loop.run_until_complete(asyncio.sleep(0.01))
        
        # Only subscriber2 should have reacted
        assert subscriber2.reaction_count == 1
    
    def test_size_based_cleanup(self):
        """Test cleanup triggers after size threshold"""
        # Use small max_subscribers for testing
        publisher = Publisher(logger=logger, max_subscribers_before_cleanup=3)
        
        sub1 = MockSubscriber()
        sub2 = MockSubscriber()
        
        # Add 2 subscribers
        publisher.add_subscriber(sub1)
        publisher.add_subscriber(sub2)
        
        # Keep weak refs before deleting
        weak_ref1 = weakref.ref(sub1)
        weak_ref2 = weakref.ref(sub2)
        
        # Delete the subscribers
        del sub1
        del sub2
        gc.collect()
        
        # Publisher still has 2 dead refs, now add one more to reach threshold of 3
        # This should trigger cleanup
        publisher.add_subscriber(MockSubscriber())
        
        # The dead refs should have been cleaned up after reaching threshold
        # The cleanup happens in add_subscriber when threshold is reached
        assert weak_ref1() is None
        assert weak_ref2() is None


class TestPublisherSubscriberAsync(ObservableTestCase):
    """Test async behavior"""
    
    def setup_method(self):
        super().setup_method()
        self.loop = asyncio.new_event_loop()
        asyncio.set_event_loop(self.loop)
    
    def teardown_method(self):
        self.loop.close()
    
    def test_async_execution(self):
        """Test that reactions execute asynchronously"""
        publisher = Publisher(preferred_publish_mode="async", logger=logger)
        
        # Create async-aware subscribers with delays
        class SlowSubscriber(Subscriber):
            def __init__(self):
                super().__init__()
                self.reaction_count = 0
            
            def _react_to_publication(self, publisher: Publisher, mode: Literal["async", "sync", "direct"]) -> None:
                # Simulate slow processing
                import time
                time.sleep(0.05)
                self.reaction_count += 1
        
        class FastSubscriber(Subscriber):
            def __init__(self):
                super().__init__()
                self.reaction_count = 0
            
            def _react_to_publication(self, publisher: Publisher, mode: Literal["async", "sync", "direct"]) -> None:
                # Fast processing
                self.reaction_count += 1
        
        subscriber1 = SlowSubscriber()
        subscriber2 = FastSubscriber()
        
        publisher.add_subscriber(subscriber1)
        publisher.add_subscriber(subscriber2)
        
        # Publish returns immediately (async mode)
        publisher.publish()  # Uses async mode from preferred
        
        # In async mode, publish returns immediately before reactions complete
        assert subscriber2.reaction_count == 0
        assert subscriber1.reaction_count == 0
        
        # Wait for reactions to complete
        self.loop.run_until_complete(asyncio.sleep(0.1))
        
        # Both should have completed
        assert subscriber1.reaction_count == 1
        assert subscriber2.reaction_count == 1


class TestBidirectionalReferences(ObservableTestCase):
    """Test bidirectional references between Publisher and Subscriber"""
    
    def setup_method(self):
        super().setup_method()
        self.loop = asyncio.new_event_loop()
        asyncio.set_event_loop(self.loop)
    
    def teardown_method(self):
        self.loop.close()
    
    def test_subscriber_tracks_publishers(self):
        """Test that subscribers track their publishers"""
        publisher1 = Publisher(logger=logger)
        publisher2 = Publisher(logger=logger)
        subscriber = MockSubscriber()
        
        publisher1.add_subscriber(subscriber)
        publisher2.add_subscriber(subscriber)
        
        # Subscriber should have references to both publishers
        assert len(list(subscriber._publisher_storage.weak_references)) == 2 # type: ignore
    
    def test_remove_updates_both_sides(self):
        """Test that removing a subscriber updates both sides"""
        publisher = Publisher(logger=logger)
        subscriber = MockSubscriber()
        
        publisher.add_subscriber(subscriber)
        
        # Both should have references
        assert publisher.is_subscribed(subscriber)
        assert len(list(subscriber._publisher_storage.weak_references)) == 1 # type: ignore
        
        # Remove subscriber
        publisher.remove_subscriber(subscriber)
        
        # Both should be cleaned up
        assert not publisher.is_subscribed(subscriber)
        assert len(list(subscriber._publisher_storage.weak_references)) == 0 # type: ignore

