#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
MoleditPy — A Python-based molecular editing software

Author: Hiromichi Yokoyama
License: Apache-2.0 license
Repo: https://github.com/HiroYokoyama/python_molecular_editor
DOI 10.5281/zenodo.17268532
"""

#Version
VERSION = '1.6.0'

print("-----------------------------------------------------")
print("MoleditPy — A Python-based molecular editing software")
print("-----------------------------------------------------\n")

import sys
import numpy as np
import pickle
import copy
import math
import io
import os
import ctypes
import itertools
import json 
import vtk

from collections import deque

# PyQt6 Modules
from PyQt6.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,
    QPushButton, QSplitter, QGraphicsView, QGraphicsScene, QGraphicsItem,
    QToolBar, QStatusBar, QGraphicsTextItem, QGraphicsLineItem, QDialog, QGridLayout,
    QFileDialog, QSizePolicy, QLabel, QLineEdit, QToolButton, QMenu, QMessageBox, QInputDialog,
    QColorDialog, QCheckBox, QSlider, QFormLayout
)

from PyQt6.QtGui import (
    QPen, QBrush, QColor, QPainter, QAction, QActionGroup, QFont, QPolygonF,
    QPainterPath, QPainterPathStroker, QFontMetrics, QFontMetricsF, QKeySequence, QTransform, QCursor, QPixmap, QIcon, QShortcut, QDesktopServices, QImage
)


from PyQt6.QtCore import Qt, QPointF, QRectF, QLineF, QObject, QThread, pyqtSignal, QEvent, QMimeData, QByteArray, QUrl, QTimer

from vtkmodules.vtkInteractionStyle import vtkInteractorStyleTrackballCamera

# RDKit
from rdkit import Chem
from rdkit.Chem import AllChem
from rdkit.Chem import Descriptors
from rdkit.Chem import rdMolDescriptors


# Open Babel Python binding (optional; required for fallback)
#from openbabel import pybel

# PyVista
import pyvista as pv
from pyvistaqt import QtInteractor

# --- Constants ---
ATOM_RADIUS = 18
BOND_OFFSET = 3.5
DEFAULT_BOND_LENGTH = 75 # テンプレートで使用する標準結合長
CLIPBOARD_MIME_TYPE = "application/x-moleditpy-fragment"

# Physical bond length (approximate) used to convert scene pixels to angstroms.
# DEFAULT_BOND_LENGTH is the length in pixels used in the editor UI for a typical bond.
# Many molecular file formats expect coordinates in angstroms; use ~1.5 Å as a typical single-bond length.
DEFAULT_BOND_LENGTH_ANGSTROM = 1.5
# Multiply pixel coordinates by this to get angstroms: ANGSTROM_PER_PIXEL = 1.5Å / DEFAULT_BOND_LENGTH(px)
ANGSTROM_PER_PIXEL = DEFAULT_BOND_LENGTH_ANGSTROM / DEFAULT_BOND_LENGTH

# UI / drawing / behavior constants (centralized for maintainability)
FONT_FAMILY = "Arial"
FONT_SIZE_LARGE = 20
FONT_SIZE_SMALL = 12
FONT_WEIGHT_BOLD = QFont.Weight.Bold

# Hit / visual sizes (in pixels at scale=1)
DESIRED_ATOM_PIXEL_RADIUS = 15.0
DESIRED_BOND_PIXEL_WIDTH = 18.0

# Bond/EZ label
EZ_LABEL_TEXT_OUTLINE = 2.5
EZ_LABEL_MARGIN = 16
EZ_LABEL_BOX_SIZE = 28

# Interaction thresholds
SNAP_DISTANCE = 14.0
SUM_TOLERANCE = 5.0

# Misc drawing
NUM_DASHES = 8
HOVER_PEN_WIDTH = 8

CPK_COLORS = {
    'H': QColor('#FFFFFF'), 'C': QColor('#222222'), 'N': QColor('#3377FF'), 'O': QColor('#FF3333'), 'F': QColor('#99E6E6'),
    'Cl': QColor('#33FF33'), 'Br': QColor('#A52A2A'), 'I': QColor('#9400D3'), 'S': QColor('#FFC000'), 'P': QColor('#FF8000'),
    'Si': QColor('#DAA520'), 'B': QColor('#FA8072'), 'He': QColor('#D9FFFF'), 'Ne': QColor('#B3E3F5'), 'Ar': QColor('#80D1E3'),
    'Kr': QColor('#5CACC8'), 'Xe': QColor('#429EB0'), 'Rn': QColor('#298FA2'), 'Li': QColor('#CC80FF'), 'Na': QColor('#AB5CF2'),
    'K': QColor('#8F44D7'), 'Rb': QColor('#702EBC'), 'Cs': QColor('#561B9E'), 'Fr': QColor('#421384'), 'Be': QColor('#C2FF00'),
    'Mg': QColor('#8AFF00'), 'Ca': QColor('#3DFF00'), 'Sr': QColor('#00FF00'), 'Ba': QColor('#00E600'), 'Ra': QColor('#00B800'),
    'Sc': QColor('#E6E6E6'), 'Ti': QColor('#BFC2C7'), 'V': QColor('#A6A6AB'), 'Cr': QColor('#8A99C7'), 'Mn': QColor('#9C7AC7'),
    'Fe': QColor('#E06633'), 'Co': QColor('#F090A0'), 'Ni': QColor('#50D050'), 'Cu': QColor('#C88033'), 'Zn': QColor('#7D80B0'),
    'Ga': QColor('#C28F8F'), 'Ge': QColor('#668F8F'), 'As': QColor('#BD80E3'), 'Se': QColor('#FFA100'), 'Tc': QColor('#3B9E9E'),
    'Ru': QColor('#248F8F'), 'Rh': QColor('#0A7D8F'), 'Pd': QColor('#006985'), 'Ag': QColor('#C0C0C0'), 'Cd': QColor('#FFD700'),
    'In': QColor('#A67573'), 'Sn': QColor('#668080'), 'Sb': QColor('#9E63B5'), 'Te': QColor('#D47A00'), 'La': QColor('#70D4FF'),
    'Ce': QColor('#FFFFC7'), 'Pr': QColor('#D9FFC7'), 'Nd': QColor('#C7FFC7'), 'Pm': QColor('#A3FFC7'), 'Sm': QColor('#8FFFC7'),
    'Eu': QColor('#61FFC7'), 'Gd': QColor('#45FFC7'), 'Tb': QColor('#30FFC7'), 'Dy': QColor('#1FFFC7'), 'Ho': QColor('#00FF9C'),
    'Er': QColor('#00E675'), 'Tm': QColor('#00D452'), 'Yb': QColor('#00BF38'), 'Lu': QColor('#00AB24'), 'Hf': QColor('#4DC2FF'),
    'Ta': QColor('#4DA6FF'), 'W': QColor('#2194D6'), 'Re': QColor('#267DAB'), 'Os': QColor('#266696'), 'Ir': QColor('#175487'),
    'Pt': QColor('#D0D0E0'), 'Au': QColor('#FFD123'), 'Hg': QColor('#B8B8D0'), 'Tl': QColor('#A6544D'), 'Pb': QColor('#575961'),
    'Bi': QColor('#9E4FB5'), 'Po': QColor('#AB5C00'), 'At': QColor('#754F45'), 'Ac': QColor('#70ABFA'), 'Th': QColor('#00BAFF'),
    'Pa': QColor('#00A1FF'), 'U': QColor('#008FFF'), 'Np': QColor('#0080FF'), 'Pu': QColor('#006BFF'), 'Am': QColor('#545CF2'),
    'Cm': QColor('#785CE3'), 'Bk': QColor('#8A4FE3'), 'Cf': QColor('#A136D4'), 'Es': QColor('#B31FD4'), 'Fm': QColor('#B31FBA'),
    'Md': QColor('#B30DA6'), 'No': QColor('#BD0D87'), 'Lr': QColor('#C70066'), 'Al': QColor('#B3A68F'), 'Y': QColor('#99FFFF'), 
    'Zr': QColor('#7EE7E7'), 'Nb': QColor('#68CFCE'), 'Mo': QColor('#52B7B7'), 'DEFAULT': QColor('#FF1493') # Pink fallback
}
CPK_COLORS_PV = {
    k: [c.redF(), c.greenF(), c.blueF()] for k, c in CPK_COLORS.items()
}

pt = Chem.GetPeriodicTable()
VDW_RADII = {pt.GetElementSymbol(i): pt.GetRvdw(i) * 0.3 for i in range(1, 119)}

def main():
    # --- Windows タスクバーアイコンのための追加処理 ---
    if sys.platform == 'win32':
        myappid = 'hyoko.moleditpy.1.0' # アプリケーション固有のID（任意）
        ctypes.windll.shell32.SetCurrentProcessExplicitAppUserModelID(myappid)

    app = QApplication(sys.argv)
    file_path = sys.argv[1] if len(sys.argv) > 1 else None
    window = MainWindow(initial_file=file_path)
    window.show()
    sys.exit(app.exec())


# --- Data Model ---
class MolecularData:
    def __init__(self):
        self.atoms = {}
        self.bonds = {}
        self._next_atom_id = 0
        self.adjacency_list = {} 

    def add_atom(self, symbol, pos, charge=0, radical=0):
        atom_id = self._next_atom_id
        self.atoms[atom_id] = {'symbol': symbol, 'pos': pos, 'item': None, 'charge': charge, 'radical': radical}
        self.adjacency_list[atom_id] = [] 
        self._next_atom_id += 1
        return atom_id

    def add_bond(self, id1, id2, order=1, stereo=0):
        # 立体結合の場合、IDの順序は方向性を意味するため、ソートしない。
        # 非立体結合の場合は、キーを正規化するためにソートする。
        if stereo == 0:
            if id1 > id2: id1, id2 = id2, id1

        bond_data = {'order': order, 'stereo': stereo, 'item': None}
        
        # 逆方向のキーも考慮して、新規結合かどうかをチェック
        is_new_bond = (id1, id2) not in self.bonds and (id2, id1) not in self.bonds
        if is_new_bond:
            if id1 in self.adjacency_list and id2 in self.adjacency_list:
                self.adjacency_list[id1].append(id2)
                self.adjacency_list[id2].append(id1)

        if (id1, id2) in self.bonds:
            self.bonds[(id1, id2)].update(bond_data)
            return (id1, id2), 'updated'
        else:
            self.bonds[(id1, id2)] = bond_data
            return (id1, id2), 'created'

    def remove_atom(self, atom_id):
        if atom_id in self.atoms:
            try:
                # Safely get neighbors before deleting the atom's own entry
                neighbors = self.adjacency_list.get(atom_id, [])
                for neighbor_id in neighbors:
                    if neighbor_id in self.adjacency_list and atom_id in self.adjacency_list[neighbor_id]:
                        self.adjacency_list[neighbor_id].remove(atom_id)

                # Now, safely delete the atom's own entry from the adjacency list
                if atom_id in self.adjacency_list:
                    del self.adjacency_list[atom_id]

                del self.atoms[atom_id]
                
                # Remove bonds involving this atom
                bonds_to_remove = [key for key in self.bonds if atom_id in key]
                for key in bonds_to_remove:
                    del self.bonds[key]
                    
            except Exception as e:
                print(f"Error removing atom {atom_id}: {e}")
                import traceback
                traceback.print_exc()

    def remove_bond(self, id1, id2):
        try:
            # 方向性のある立体結合(順方向/逆方向)と、正規化された非立体結合のキーを探す
            key_to_remove = None
            if (id1, id2) in self.bonds:
                key_to_remove = (id1, id2)
            elif (id2, id1) in self.bonds:
                key_to_remove = (id2, id1)

            if key_to_remove:
                if id1 in self.adjacency_list and id2 in self.adjacency_list[id1]:
                    self.adjacency_list[id1].remove(id2)
                if id2 in self.adjacency_list and id1 in self.adjacency_list[id2]:
                    self.adjacency_list[id2].remove(id1)
                del self.bonds[key_to_remove]
                
        except Exception as e:
            print(f"Error removing bond {id1}-{id2}: {e}")
            import traceback
            traceback.print_exc()


    def to_rdkit_mol(self, use_2d_stereo=True):
        """
        use_2d_stereo: Trueなら2D座標からE/Zを推定（従来通り）。FalseならE/Zラベル優先、ラベルがない場合のみ2D座標推定。
        3D変換時はuse_2d_stereo=Falseで呼び出すこと。
        """
        if not self.atoms:
            return None
        mol = Chem.RWMol()

        # --- Step 1: atoms ---
        atom_id_to_idx_map = {}
        for atom_id, data in self.atoms.items():
            atom = Chem.Atom(data['symbol'])
            atom.SetFormalCharge(data.get('charge', 0))
            atom.SetNumRadicalElectrons(data.get('radical', 0))
            atom.SetIntProp("_original_atom_id", atom_id)
            idx = mol.AddAtom(atom)
            atom_id_to_idx_map[atom_id] = idx

        # --- Step 2: bonds & stereo info保存（ラベル情報はここで保持） ---
        bond_stereo_info = {}  # bond_idx -> {'type': int, 'atom_ids': (id1,id2), 'bond_data': bond_data}
        for (id1, id2), bond_data in self.bonds.items():
            if id1 not in atom_id_to_idx_map or id2 not in atom_id_to_idx_map:
                continue
            idx1, idx2 = atom_id_to_idx_map[id1], atom_id_to_idx_map[id2]

            order_val = float(bond_data['order'])
            order = {1.0: Chem.BondType.SINGLE, 1.5: Chem.BondType.AROMATIC,
                     2.0: Chem.BondType.DOUBLE, 3.0: Chem.BondType.TRIPLE}.get(order_val, Chem.BondType.SINGLE)

            bond_idx = mol.AddBond(idx1, idx2, order) - 1

            # stereoラベルがあれば、bond_idxに対して詳細を保持（あとで使う）
            if 'stereo' in bond_data and bond_data['stereo'] in [1, 2, 3, 4]:
                bond_stereo_info[bond_idx] = {
                    'type': int(bond_data['stereo']),
                    'atom_ids': (id1, id2),
                    'bond_data': bond_data
                }

        # --- Step 3: sanitize ---
        final_mol = mol.GetMol()
        try:
            Chem.SanitizeMol(final_mol)
        except Exception as e:
            return None

        # --- Step 4: add 2D conformer ---
        # Convert from scene pixels to angstroms when creating RDKit conformer.
        conf = Chem.Conformer(final_mol.GetNumAtoms())
        conf.Set3D(False)
        for atom_id, data in self.atoms.items():
            if atom_id in atom_id_to_idx_map:
                idx = atom_id_to_idx_map[atom_id]
                pos = data.get('pos')
                if pos:
                    ax = pos.x() * ANGSTROM_PER_PIXEL
                    ay = pos.y() * ANGSTROM_PER_PIXEL
                    conf.SetAtomPosition(idx, (ax, ay, 0.0))
        final_mol.AddConformer(conf)

        # --- Step 5: E/Zラベル優先の立体設定 ---
        # まず、E/Zラベルがあるbondを記録
        ez_labeled_bonds = set()
        for bond_idx, info in bond_stereo_info.items():
            if info['type'] in [3, 4]:
                ez_labeled_bonds.add(bond_idx)

        # 2D座標からE/Zを推定するのは、use_2d_stereo=True かつE/Zラベルがないbondのみ
        if use_2d_stereo:
            Chem.SetDoubleBondNeighborDirections(final_mol, final_mol.GetConformer(0))
        else:
            # 3D変換時: E/Zラベルがある場合は座標ベースの推定を完全に無効化
            if ez_labeled_bonds:
                # E/Zラベルがある場合は、すべての結合のBondDirをクリアして座標ベースの推定を無効化
                for b in final_mol.GetBonds():
                    b.SetBondDir(Chem.BondDir.NONE)
            else:
                # E/Zラベルがない場合のみ座標ベースの推定を実行
                Chem.SetDoubleBondNeighborDirections(final_mol, final_mol.GetConformer(0))

        # ヘルパー: 重原子優先で近傍を選ぶ
        def pick_preferred_neighbor(atom, exclude_idx):
            for nbr in atom.GetNeighbors():
                if nbr.GetIdx() == exclude_idx:
                    continue
                if nbr.GetAtomicNum() > 1:
                    return nbr.GetIdx()
            for nbr in atom.GetNeighbors():
                if nbr.GetIdx() != exclude_idx:
                    return nbr.GetIdx()
            return None

        # --- Step 6: ラベルベースで上書き（E/Z を最優先） ---
        for bond_idx, info in bond_stereo_info.items():
            stereo_type = info['type']
            bond = final_mol.GetBondWithIdx(bond_idx)

            # 単結合の wedge/dash ラベル（1/2）がある場合
            if stereo_type in [1, 2]:
                if stereo_type == 1:
                    bond.SetBondDir(Chem.BondDir.BEGINWEDGE)
                elif stereo_type == 2:
                    bond.SetBondDir(Chem.BondDir.BEGINDASH)
                continue

            # 二重結合の E/Z ラベル（3/4）
            if stereo_type in [3, 4]:
                if bond.GetBondType() != Chem.BondType.DOUBLE:
                    continue

                begin_atom_idx = bond.GetBeginAtomIdx()
                end_atom_idx = bond.GetEndAtomIdx()

                bond_data = info.get('bond_data', {}) or {}
                stereo_atoms_specified = bond_data.get('stereo_atoms')

                if stereo_atoms_specified:
                    try:
                        a1_id, a2_id = stereo_atoms_specified
                        neigh1_idx = atom_id_to_idx_map.get(a1_id)
                        neigh2_idx = atom_id_to_idx_map.get(a2_id)
                    except Exception:
                        neigh1_idx = None
                        neigh2_idx = None
                else:
                    neigh1_idx = pick_preferred_neighbor(final_mol.GetAtomWithIdx(begin_atom_idx), end_atom_idx)
                    neigh2_idx = pick_preferred_neighbor(final_mol.GetAtomWithIdx(end_atom_idx), begin_atom_idx)

                if neigh1_idx is None or neigh2_idx is None:
                    continue

                bond.SetStereoAtoms(neigh1_idx, neigh2_idx)
                if stereo_type == 3:
                    bond.SetStereo(Chem.BondStereo.STEREOZ)
                elif stereo_type == 4:
                    bond.SetStereo(Chem.BondStereo.STEREOE)

                # 座標ベースでつけられた隣接単結合の BondDir（wedge/dash）がラベルと矛盾する可能性があるので消す
                b1 = final_mol.GetBondBetweenAtoms(begin_atom_idx, neigh1_idx)
                b2 = final_mol.GetBondBetweenAtoms(end_atom_idx, neigh2_idx)
                if b1 is not None:
                    b1.SetBondDir(Chem.BondDir.NONE)
                if b2 is not None:
                    b2.SetBondDir(Chem.BondDir.NONE)

        # Step 7: 最終化（キャッシュ更新 + 立体割当の再実行）
        final_mol.UpdatePropertyCache(strict=False)
        
        # 3D変換時（use_2d_stereo=False）でE/Zラベルがある場合は、force=Trueで強制適用
        if not use_2d_stereo and ez_labeled_bonds:
            Chem.AssignStereochemistry(final_mol, cleanIt=False, force=True)
        else:
            Chem.AssignStereochemistry(final_mol, cleanIt=False, force=False)
        return final_mol

    def to_mol_block(self):
        try:
            mol = self.to_rdkit_mol()
            if mol:
                return Chem.MolToMolBlock(mol, includeStereo=True)
        except Exception:
            pass
        if not self.atoms: return None
        atom_map = {old_id: new_id for new_id, old_id in enumerate(self.atoms.keys())}
        num_atoms, num_bonds = len(self.atoms), len(self.bonds)
        mol_block = "\n  MoleditPy\n\n"
        mol_block += f"{num_atoms:3d}{num_bonds:3d}  0  0  0  0  0  0  0  0999 V2000\n"
        for old_id, atom in self.atoms.items():
            # Convert scene pixel coordinates to angstroms when emitting MOL block
            x_px = atom['item'].pos().x()
            y_px = -atom['item'].pos().y()
            x, y = x_px * ANGSTROM_PER_PIXEL, y_px * ANGSTROM_PER_PIXEL
            z, symbol = 0.0, atom['symbol']
            charge = atom.get('charge', 0)

            chg_code = 0
            if charge == 3: chg_code = 1
            elif charge == 2: chg_code = 2
            elif charge == 1: chg_code = 3
            elif charge == -1: chg_code = 5
            elif charge == -2: chg_code = 6
            elif charge == -3: chg_code = 7

            mol_block += f"{x:10.4f}{y:10.4f}{z:10.4f} {symbol:<3} 0  0  0{chg_code:3d}  0  0  0  0  0  0  0\n"

        for (id1, id2), bond in self.bonds.items():
            idx1, idx2, order = atom_map[id1] + 1, atom_map[id2] + 1, bond['order']
            stereo_code = 0
            bond_stereo = bond.get('stereo', 0)
            if bond_stereo == 1:
                stereo_code = 1
            elif bond_stereo == 2:
                stereo_code = 6

            mol_block += f"{idx1:3d}{idx2:3d}{order:3d}{stereo_code:3d}  0  0  0\n"
            
        mol_block += "M  END\n"
        return mol_block


class AtomItem(QGraphicsItem):
    def __init__(self, atom_id, symbol, pos, charge=0, radical=0):
        super().__init__()
        self.atom_id, self.symbol, self.charge, self.radical, self.bonds, self.chiral_label = atom_id, symbol, charge, radical, [], None
        self.setPos(pos)
        self.implicit_h_count = 0 
        self.setFlags(QGraphicsItem.GraphicsItemFlag.ItemIsMovable | QGraphicsItem.GraphicsItemFlag.ItemIsSelectable)
        self.setZValue(1); self.font = QFont(FONT_FAMILY, FONT_SIZE_LARGE, FONT_WEIGHT_BOLD); self.update_style()
        self.setAcceptHoverEvents(True)
        self.hovered = False
        self.has_problem = False 

    def boundingRect(self):
        # --- paint()メソッドと完全に同じロジックでテキストの位置とサイズを計算 ---
        font = QFont(FONT_FAMILY, FONT_SIZE_LARGE, FONT_WEIGHT_BOLD)
        fm = QFontMetricsF(font)

        hydrogen_part = ""
        if self.implicit_h_count > 0:
            is_skeletal_carbon = (self.symbol == 'C' and 
                                      self.charge == 0 and 
                                      self.radical == 0 and 
                                      len(self.bonds) > 0)
            if not is_skeletal_carbon:
                hydrogen_part = "H"
                if self.implicit_h_count > 1:
                    subscript_map = str.maketrans("0123456789", "₀₁₂₃₄₅₆₇₈₉")
                    hydrogen_part += str(self.implicit_h_count).translate(subscript_map)

        flip_text = False
        if hydrogen_part and self.bonds:
            my_pos_x = self.pos().x()
            total_dx = sum((b.atom2.pos().x() if b.atom1 is self else b.atom1.pos().x()) - my_pos_x for b in self.bonds)
            if total_dx > 0:
                flip_text = True
        
        if flip_text:
            display_text = hydrogen_part + self.symbol
        else:
            display_text = self.symbol + hydrogen_part

        text_rect = fm.boundingRect(display_text)
        text_rect.adjust(-2, -2, 2, 2)
        if hydrogen_part:
            symbol_rect = fm.boundingRect(self.symbol)
            if flip_text:
                offset_x = symbol_rect.width() // 2
                text_rect.moveTo(offset_x - text_rect.width(), -text_rect.height() / 2)
            else:
                offset_x = -symbol_rect.width() // 2
                text_rect.moveTo(offset_x, -text_rect.height() / 2)
        else:
            text_rect.moveCenter(QPointF(0, 0))

        # 1. paint()で描画される背景の矩形(bg_rect)を計算する
        bg_rect = text_rect.adjusted(-5, -8, 5, 8)
        
        # 2. このbg_rectを基準として全体の描画領域を構築する
        full_visual_rect = QRectF(bg_rect)

        # 電荷記号の領域を計算に含める
        if self.charge != 0:
            if self.charge == 1: charge_str = "+"
            elif self.charge == -1: charge_str = "-"
            else: charge_str = f"{self.charge:+}"
            charge_font = QFont("Arial", 12, QFont.Weight.Bold)
            charge_fm = QFontMetricsF(charge_font)
            charge_rect = charge_fm.boundingRect(charge_str)
            
            if flip_text:
                charge_pos = QPointF(text_rect.left() - charge_rect.width() - 2, text_rect.top())
            else:
                charge_pos = QPointF(text_rect.right() + 2, text_rect.top())
            charge_rect.moveTopLeft(charge_pos)
            full_visual_rect = full_visual_rect.united(charge_rect)

        # ラジカル記号の領域を計算に含める
        if self.radical > 0:
            radical_area = QRectF(text_rect.center().x() - 8, text_rect.top() - 8, 16, 8)
            full_visual_rect = full_visual_rect.united(radical_area)

        # 3. 選択ハイライト等のための最終的なマージンを追加する
        return full_visual_rect.adjusted(-3, -3, 3, 3)

    def shape(self):
        scene = self.scene()
        if not scene or not scene.views():
            path = QPainterPath()
            hit_r = max(4.0, ATOM_RADIUS - 6.0) * 2
            path.addEllipse(QRectF(-hit_r, -hit_r, hit_r * 2.0, hit_r * 2.0))
            return path

        view = scene.views()[0]
        scale = view.transform().m11()

        scene_radius = DESIRED_ATOM_PIXEL_RADIUS / scale

        path = QPainterPath()
        path.addEllipse(QPointF(0, 0), scene_radius, scene_radius)
        return path

    def paint(self, painter, option, widget):
        color = CPK_COLORS.get(self.symbol, CPK_COLORS['DEFAULT'])
        if self.is_visible:
            # 1. 描画の準備
            painter.setFont(self.font)
            fm = painter.fontMetrics()

            # --- 水素部分のテキストを作成 ---
            hydrogen_part = ""
            if self.implicit_h_count > 0:
                is_skeletal_carbon = (self.symbol == 'C' and 
                                      self.charge == 0 and 
                                      self.radical == 0 and 
                                      len(self.bonds) > 0)
                if not is_skeletal_carbon:
                    hydrogen_part = "H"
                    if self.implicit_h_count > 1:
                        subscript_map = str.maketrans("0123456789", "₀₁₂₃₄₅₆₇₈₉")
                        hydrogen_part += str(self.implicit_h_count).translate(subscript_map)

            # --- テキストを反転させるか決定 ---
            flip_text = False
            # 水素ラベルがあり、結合が1本以上ある場合のみ反転を考慮
            if hydrogen_part and self.bonds:

                # 相対的なX座標で、結合が左右どちらに偏っているか判定
                my_pos_x = self.pos().x()
                total_dx = 0
                for bond in self.bonds:
                    other_atom = bond.atom1 if bond.atom2 is self else bond.atom2
                    total_dx += (other_atom.pos().x() - my_pos_x)

                # 結合が主に右側にある場合はテキストを反転させる
                if total_dx > 0:
                    flip_text = True

            # --- 表示テキストとアライメントを最終決定 ---
            if flip_text:
                display_text = hydrogen_part + self.symbol
                alignment_flag = Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignVCenter
            else:
                display_text = self.symbol + hydrogen_part
                alignment_flag = Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignVCenter

            text_rect = fm.boundingRect(display_text)
            text_rect.adjust(-2, -2, 2, 2)
            symbol_rect = fm.boundingRect(self.symbol) # 主元素のみの幅を計算

            # --- テキストの描画位置を決定 ---
            # 水素ラベルがない場合 (従来通り中央揃え)
            if not hydrogen_part:
                alignment_flag = Qt.AlignmentFlag.AlignCenter
                text_rect.moveCenter(QPointF(0, 0).toPoint())
            # 水素ラベルがあり、反転する場合 (右揃え)
            elif flip_text:
                # 主元素の中心が原子の中心に来るように、矩形の右端を調整
                offset_x = symbol_rect.width() // 2
                text_rect.moveTo(offset_x - text_rect.width(), -text_rect.height() // 2)
            # 水素ラベルがあり、反転しない場合 (左揃え)
            else:
                # 主元素の中心が原子の中心に来るように、矩形の左端を調整
                offset_x = -symbol_rect.width() // 2
                text_rect.moveTo(offset_x, -text_rect.height() // 2)

            # 2. 原子記号の背景を白で塗りつぶす
            if self.scene():
                bg_brush = self.scene().backgroundBrush()
                bg_rect = text_rect.adjusted(-5, -8, 5, 8)
                painter.setBrush(bg_brush)
                painter.setPen(Qt.PenStyle.NoPen)
                painter.drawEllipse(bg_rect)
            
            # 3. 原子記号自体を描画
            if self.symbol == 'H':
                painter.setPen(QPen(Qt.GlobalColor.black))
            else:
                painter.setPen(QPen(color))
            painter.drawText(text_rect, int(alignment_flag), display_text)
            
            # --- 電荷とラジカルの描画  ---
            if self.charge != 0:
                if self.charge == 1: charge_str = "+"
                elif self.charge == -1: charge_str = "-"
                else: charge_str = f"{self.charge:+}"
                charge_font = QFont("Arial", 12, QFont.Weight.Bold)
                painter.setFont(charge_font)
                charge_rect = painter.fontMetrics().boundingRect(charge_str)
                # 電荷の位置も反転に対応
                if flip_text:
                    charge_pos = QPointF(text_rect.left() - charge_rect.width() -2, text_rect.top() + charge_rect.height() - 2)
                else:
                    charge_pos = QPointF(text_rect.right() + 2, text_rect.top() + charge_rect.height() - 2)
                painter.setPen(Qt.GlobalColor.black)
                painter.drawText(charge_pos, charge_str)
            
            if self.radical > 0:
                painter.setBrush(QBrush(Qt.GlobalColor.black))
                painter.setPen(Qt.PenStyle.NoPen)
                radical_pos_y = text_rect.top() - 5
                if self.radical == 1:
                    painter.drawEllipse(QPointF(text_rect.center().x(), radical_pos_y), 3, 3)
                elif self.radical == 2:
                    painter.drawEllipse(QPointF(text_rect.center().x() - 5, radical_pos_y), 3, 3)
                    painter.drawEllipse(QPointF(text_rect.center().x() + 5, radical_pos_y), 3, 3)


        # --- 選択時のハイライトなど ---
        if self.has_problem:
            painter.setBrush(Qt.BrushStyle.NoBrush)
            painter.setPen(QPen(QColor(255, 0, 0, 200), 4))
            painter.drawRect(self.boundingRect())
        elif self.isSelected():
            painter.setBrush(Qt.BrushStyle.NoBrush)
            painter.setPen(QPen(QColor(0, 100, 255), 3))
            painter.drawRect(self.boundingRect())
        if (not self.isSelected()) and getattr(self, 'hovered', False):
            pen = QPen(QColor(144, 238, 144, 200), 3)
            pen.setJoinStyle(Qt.PenJoinStyle.RoundJoin)
            painter.setBrush(Qt.BrushStyle.NoBrush)
            painter.setPen(pen)
            painter.drawRect(self.boundingRect())

    def update_style(self):
        self.is_visible = not (self.symbol == 'C' and len(self.bonds) > 0 and self.charge == 0 and self.radical == 0)
        self.update()

    # 約203行目 AtomItem クラス内

    def itemChange(self, change, value):
        res = super().itemChange(change, value)
        if change == QGraphicsItem.GraphicsItemChange.ItemPositionHasChanged:
            if self.flags() & QGraphicsItem.GraphicsItemFlag.ItemIsMovable:
                # Prevent cascading updates during batch operations
                if not getattr(self, '_updating_position', False):
                    for bond in self.bonds: 
                        if bond.scene():  # Only update if bond is still in scene
                            bond.update_position()
            
        return res

    def hoverEnterEvent(self, event):
        # シーンのモードにかかわらず、ホバー時にハイライトを有効にする
        self.hovered = True
        self.update()
        super().hoverEnterEvent(event)

    def hoverLeaveEvent(self, event):
        if self.hovered:
            self.hovered = False
            self.update()
        super().hoverLeaveEvent(event)

class BondItem(QGraphicsItem):

    def get_ez_label_rect(self):
        """E/Zラベルの描画範囲（シーン座標）を返す。ラベルが無い場合はNone。"""
        if self.order != 2 or self.stereo not in [3, 4]:
            return None
        line = self.get_line_in_local_coords()
        center = line.center()
        label_width = EZ_LABEL_BOX_SIZE
        label_height = EZ_LABEL_BOX_SIZE
        label_rect = QRectF(center.x() - label_width/2, center.y() - label_height/2, label_width, label_height)
        # シーン座標に変換
        return self.mapToScene(label_rect).boundingRect()
    def set_stereo(self, new_stereo):
        try:
            # ラベルを消す場合は、消す前のboundingRectをscene().invalidateで強制的に無効化
            if new_stereo == 0 and self.stereo in [3, 4] and self.scene():
                from PyQt6.QtWidgets import QGraphicsScene
                rect = self.mapToScene(self.boundingRect()).boundingRect()
                self.scene().invalidate(rect, QGraphicsScene.SceneLayer.BackgroundLayer | QGraphicsScene.SceneLayer.ForegroundLayer)
            
            self.prepareGeometryChange()
            self.stereo = new_stereo
            self.update()
            
            if self.scene() and self.scene().views():
                try:
                    self.scene().views()[0].viewport().update()
                except (IndexError, RuntimeError):
                    # Handle case where views are being destroyed
                    pass
                    
        except Exception as e:
            print(f"Error in BondItem.set_stereo: {e}")
            # Continue without crashing
            self.stereo = new_stereo

    def set_order(self, new_order):
        self.prepareGeometryChange()
        self.order = new_order
        self.update()
        if self.scene() and self.scene().views():
            self.scene().views()[0].viewport().update()
    def __init__(self, atom1_item, atom2_item, order=1, stereo=0):
        super().__init__()
        # Validate input parameters
        if atom1_item is None or atom2_item is None:
            raise ValueError("BondItem requires non-None atom items")
        self.atom1, self.atom2, self.order, self.stereo = atom1_item, atom2_item, order, stereo
        self.setFlag(QGraphicsItem.GraphicsItemFlag.ItemIsSelectable)
        self.pen = QPen(Qt.GlobalColor.black, 2)
        self.setZValue(0)
        self.update_position()
        self.setAcceptHoverEvents(True)
        self.hovered = False


    def get_line_in_local_coords(self):
        if self.atom1 is None or self.atom2 is None:
            return QLineF(0, 0, 0, 0)
        try:
            p2 = self.mapFromItem(self.atom2, 0, 0)
            return QLineF(QPointF(0, 0), p2)
        except (RuntimeError, TypeError):
            # Handle case where atoms are deleted from scene
            return QLineF(0, 0, 0, 0)

    def boundingRect(self):
        try:
            line = self.get_line_in_local_coords()
        except Exception:
            line = QLineF(0, 0, 0, 0)
        bond_offset = globals().get('BOND_OFFSET', 2)
        extra = (getattr(self, 'order', 1) - 1) * bond_offset + 20
        rect = QRectF(line.p1(), line.p2()).normalized().adjusted(-extra, -extra, extra, extra)

        # E/Zラベルの描画範囲も考慮して拡張（QFontMetricsFで正確に）
        if self.order == 2 and self.stereo in [3, 4]:
            font = QFont(FONT_FAMILY, FONT_SIZE_LARGE, FONT_WEIGHT_BOLD)
            font.setItalic(True)
            text = "Z" if self.stereo == 3 else "E"
            fm = QFontMetricsF(font)
            text_rect = fm.boundingRect(text)
            outline = EZ_LABEL_TEXT_OUTLINE  # 輪郭の太さ分
            margin = EZ_LABEL_MARGIN   # 追加余白
            center = line.center()
            label_rect = QRectF(center.x() - text_rect.width()/2 - outline - margin,
                                center.y() - text_rect.height()/2 - outline - margin,
                                text_rect.width() + 2*outline + 2*margin,
                                text_rect.height() + 2*outline + 2*margin)
            rect = rect.united(label_rect)
        return rect

    def shape(self):
        path = QPainterPath()
        try:
            line = self.get_line_in_local_coords()
        except Exception:
            return path 
        if line.length() == 0:
            return path

        scene = self.scene()
        if not scene or not scene.views():
            return super().shape()

        view = scene.views()[0]
        scale = view.transform().m11()

        scene_width = DESIRED_BOND_PIXEL_WIDTH / scale

        stroker = QPainterPathStroker()
        stroker.setWidth(scene_width)
        stroker.setCapStyle(Qt.PenCapStyle.RoundCap)  
        stroker.setJoinStyle(Qt.PenJoinStyle.RoundJoin) 

        center_line_path = QPainterPath(line.p1())
        center_line_path.lineTo(line.p2())
        
        return stroker.createStroke(center_line_path)

    def paint(self, painter, option, widget):
        if self.atom1 is None or self.atom2 is None:
            return
        line = self.get_line_in_local_coords()
        if line.length() == 0: return

        # --- 1. 選択状態に応じてペンとブラシを準備 ---
        if self.isSelected():
            selection_color = QColor("blue")
            painter.setPen(QPen(selection_color, 3))
            painter.setBrush(QBrush(selection_color))
        else:
            painter.setPen(self.pen)
            painter.setBrush(QBrush(Qt.GlobalColor.black))

        # --- 立体化学 (Wedge/Dash) の描画 ---
        if self.order == 1 and self.stereo in [1, 2]:
            vec = line.unitVector()
            normal = vec.normalVector()
            p1 = line.p1() + vec.p2() * 5
            p2 = line.p2() - vec.p2() * 5

            if self.stereo == 1: # Wedge (くさび形)
                offset = QPointF(normal.dx(), normal.dy()) * 6.0
                poly = QPolygonF([p1, p2 + offset, p2 - offset])
                painter.drawPolygon(poly)
            
            elif self.stereo == 2: # Dash (破線)
                painter.save()
                if not self.isSelected():
                    pen = painter.pen()
                    pen.setWidthF(2.5) 
                    painter.setPen(pen)
                
                num_dashes = 8
                for i in range(num_dashes + 1):
                    t = i / num_dashes
                    start_pt = p1 * (1 - t) + p2 * t
                    width = 12.0 * t
                    offset = QPointF(normal.dx(), normal.dy()) * width / 2.0
                    painter.drawLine(start_pt - offset, start_pt + offset)
                painter.restore()
        
        # --- 通常の結合 (単/二重/三重) の描画 ---
        else:
            if self.order == 1:
                painter.drawLine(line)
            else:
                v = line.unitVector().normalVector()
                offset = QPointF(v.dx(), v.dy()) * BOND_OFFSET

                if self.order == 2:
                    # -------------------- ここから差し替え --------------------)
                    line1 = line.translated(offset)
                    line2 = line.translated(-offset)
                    painter.drawLine(line1)
                    painter.drawLine(line2)

                    # E/Z ラベルの描画処理
                    if self.stereo in [3, 4]:
                        painter.save() # 現在の描画設定を保存

                        # --- ラベルの設定 ---
                        font = QFont(FONT_FAMILY, FONT_SIZE_LARGE, FONT_WEIGHT_BOLD)
                        font.setItalic(True)
                        text_color = QColor("gray")
                        # 輪郭の色を背景色と同じにする（scene()がNoneのときは安全なフォールバックを使う）
                        outline_color = None
                        try:
                            sc = self.scene()
                            if sc is not None:
                                outline_color = sc.backgroundBrush().color()
                        except Exception:
                            outline_color = None
                        if outline_color is None:
                            # デフォルトでは白背景を想定して黒系の輪郭が見やすい
                            outline_color = QColor(255, 255, 255)

                        # --- 描画パスの作成 ---
                        text = "Z" if self.stereo == 3 else "E"
                        path = QPainterPath()
                        
                        # テキストが正確に中央に来るように位置を計算
                        fm = QFontMetricsF(font)
                        text_rect = fm.boundingRect(text)
                        text_rect.moveCenter(line.center())
                        path.addText(text_rect.topLeft(), font, text)

                        # --- 輪郭の描画 ---
                        stroker = QPainterPathStroker()
                        stroker.setWidth(EZ_LABEL_TEXT_OUTLINE) # 輪郭の太さ
                        outline_path = stroker.createStroke(path)
                        
                        painter.setBrush(outline_color)
                        painter.setPen(Qt.PenStyle.NoPen)
                        painter.drawPath(outline_path)

                        # --- 文字本体の描画 ---
                        painter.setBrush(text_color)
                        painter.setPen(text_color)
                        painter.drawPath(path)

                        painter.restore() # 描画設定を元に戻す

                elif self.order == 3:
                    painter.drawLine(line)
                    painter.drawLine(line.translated(offset))
                    painter.drawLine(line.translated(-offset))

        # --- 2. ホバー時のエフェクトを上から重ねて描画 ---
        if (not self.isSelected()) and getattr(self, 'hovered', False):
            try:
                # ホバー時のハイライトを太めの半透明な線で描画
                hover_pen = QPen(QColor(144, 238, 144, 180), HOVER_PEN_WIDTH) # LightGreen, 半透明
                hover_pen.setCapStyle(Qt.PenCapStyle.RoundCap)
                painter.setPen(hover_pen)
                painter.drawLine(line) 
            except Exception:
                pass



    def update_position(self):
        try:
            self.prepareGeometryChange()
            if self.atom1:
                self.setPos(self.atom1.pos())
            self.update()
        except Exception as e:
            print(f"Error updating bond position: {e}")
            # Continue without crashing


    def hoverEnterEvent(self, event):
        scene = self.scene()
        mode = getattr(scene, 'mode', '')
        self.hovered = True
        self.update()
        if self.scene():
            self.scene().set_hovered_item(self)
        super().hoverEnterEvent(event)

    def hoverLeaveEvent(self, event):
        if self.hovered:
            self.hovered = False
            self.update()
        if self.scene():
            self.scene().set_hovered_item(None)
        super().hoverLeaveEvent(event)


class TemplatePreviewItem(QGraphicsItem):
    def __init__(self):
        super().__init__()
        self.setZValue(2)
        self.pen = QPen(QColor(80, 80, 80, 180), 2)
        self.polygon = QPolygonF()
        self.is_aromatic = False

    def set_geometry(self, points, is_aromatic=False):
        self.prepareGeometryChange()
        self.polygon = QPolygonF(points)
        self.is_aromatic = is_aromatic
        self.update()

    def boundingRect(self):
        return self.polygon.boundingRect().adjusted(-5, -5, 5, 5)

    def paint(self, painter, option, widget):
            
        painter.setPen(self.pen)
        painter.setBrush(Qt.BrushStyle.NoBrush)
        if not self.polygon.isEmpty():
            painter.drawPolygon(self.polygon)
            if self.is_aromatic:
                center = self.polygon.boundingRect().center()
                radius = QLineF(center, self.polygon.first()).length() * 0.6
                painter.drawEllipse(center, radius, radius)

class MoleculeScene(QGraphicsScene):
    def __init__(self, data, window):
        super().__init__()
        self.data, self.window = data, window
        self.mode, self.current_atom_symbol = 'select', 'C'
        self.bond_order, self.bond_stereo = 1, 0
        self.start_atom, self.temp_line, self.start_pos = None, None, None; self.press_pos = None
        self.mouse_moved_since_press = False
        self.data_changed_in_event = False
        self.hovered_item = None
        
        self.key_to_symbol_map = {
            Qt.Key.Key_C: 'C', Qt.Key.Key_N: 'N', Qt.Key.Key_O: 'O', Qt.Key.Key_S: 'S',
            Qt.Key.Key_F: 'F', Qt.Key.Key_B: 'B', Qt.Key.Key_I: 'I', Qt.Key.Key_H: 'H',
            Qt.Key.Key_P: 'P',
        }
        self.key_to_symbol_map_shift = { Qt.Key.Key_C: 'Cl', Qt.Key.Key_B: 'Br', Qt.Key.Key_S: 'Si',}

        self.key_to_bond_mode_map = {
            Qt.Key.Key_1: 'bond_1_0',
            Qt.Key.Key_2: 'bond_2_0',
            Qt.Key.Key_3: 'bond_3_0',
            Qt.Key.Key_W: 'bond_1_1',
            Qt.Key.Key_D: 'bond_1_2',
        }
        self.reinitialize_items()

    def reinitialize_items(self):
        self.template_preview = TemplatePreviewItem(); self.addItem(self.template_preview)
        self.template_preview.hide(); self.template_preview_points = []; self.template_context = {}

    def clear_all_problem_flags(self):
        """全ての AtomItem の has_problem フラグをリセットし、再描画する"""
        needs_update = False
        for atom_data in self.data.atoms.values():
            item = atom_data.get('item')
            # hasattr は安全性のためのチェック
            if item and hasattr(item, 'has_problem') and item.has_problem: 
                item.has_problem = False
                item.update()
                needs_update = True
        return needs_update

    def mousePressEvent(self, event):
        self.clear_all_problem_flags()
        self.press_pos = event.scenePos()
        self.mouse_moved_since_press = False
        self.data_changed_in_event = False
        self.initial_positions_in_event = {item: item.pos() for item in self.items() if isinstance(item, AtomItem)}

        if not self.window.is_2d_editable:
            return

        if event.button() == Qt.MouseButton.RightButton:
            item = self.itemAt(event.scenePos(), self.views()[0].transform())
            if not isinstance(item, (AtomItem, BondItem)):
                return # 対象外のものをクリックした場合は何もしない

            data_changed = False
            # --- E/Zモード専用処理 ---
            if self.mode == 'bond_2_5':
                if isinstance(item, BondItem):
                    try:
                        # E/Zラベルを消す（ノーマルに戻す）
                        if item.stereo in [3, 4]:
                            item.set_stereo(0)
                            # データモデルも更新
                            for (id1, id2), bdata in self.data.bonds.items():
                                if bdata.get('item') is item:
                                    bdata['stereo'] = 0
                                    break
                            self.window.push_undo_state()
                            data_changed = False  # ここでundo済みなので以降で積まない
                    except Exception as e:
                        print(f"Error clearing E/Z label: {e}")
                        import traceback
                        traceback.print_exc()
                        if hasattr(self.window, 'statusBar'):
                            self.window.statusBar().showMessage(f"Error clearing E/Z label: {e}", 5000)
                # AtomItemは何もしない
            # --- 通常の処理 ---
            elif isinstance(item, AtomItem):
                # ラジカルモードの場合、ラジカルを0にする
                if self.mode == 'radical' and item.radical != 0:
                    item.prepareGeometryChange()
                    item.radical = 0
                    self.data.atoms[item.atom_id]['radical'] = 0
                    item.update_style()
                    data_changed = True
                # 電荷モードの場合、電荷を0にする
                elif self.mode in ['charge_plus', 'charge_minus'] and item.charge != 0:
                    item.prepareGeometryChange()
                    item.charge = 0
                    self.data.atoms[item.atom_id]['charge'] = 0
                    item.update_style()
                    data_changed = True
                # 上記以外のモード（テンプレート、電荷、ラジカルを除く）では原子を削除
                elif not self.mode.startswith(('template', 'charge', 'radical')):
                    data_changed = self.delete_items({item})
            elif isinstance(item, BondItem):
                # テンプレート、電荷、ラジカルモード以外で結合を削除
                if not self.mode.startswith(('template', 'charge', 'radical')):
                    data_changed = self.delete_items({item})

            if data_changed:
                self.window.push_undo_state()
            self.press_pos = None
            event.accept()
            return # 右クリック処理を完了し、左クリックの処理へ進ませない

        if self.mode.startswith('template'):
            self.clearSelection() # テンプレートモードでは選択処理を一切行わず、クリック位置の記録のみ行う
            return

        # Z,Eモードの時は選択処理を行わないようにする
        if self.mode in ['bond_2_5']:
            self.clearSelection()
            event.accept()
            return

        if getattr(self, "mode", "") != "select":
            self.clearSelection()
            event.accept()

        item = self.itemAt(self.press_pos, self.views()[0].transform())

        if isinstance(item, AtomItem):
            self.start_atom = item
            if self.mode != 'select':
                self.clearSelection()
                self.temp_line = QGraphicsLineItem(QLineF(self.start_atom.pos(), self.press_pos))
                self.temp_line.setPen(QPen(Qt.GlobalColor.red, 2, Qt.PenStyle.DotLine))
                self.addItem(self.temp_line)
            else:
                super().mousePressEvent(event)
        elif item is None and (self.mode.startswith('atom') or self.mode.startswith('bond')):
            self.start_pos = self.press_pos
            self.temp_line = QGraphicsLineItem(QLineF(self.start_pos, self.press_pos)); self.temp_line.setPen(QPen(Qt.GlobalColor.red, 2, Qt.PenStyle.DotLine)); self.addItem(self.temp_line)
        else:
            super().mousePressEvent(event)

    def mouseMoveEvent(self, event):
        if not self.window.is_2d_editable:
            return 

        if self.mode.startswith('template'):
            self.update_template_preview(event.scenePos())
        
        if not self.mouse_moved_since_press and self.press_pos:
            if (event.scenePos() - self.press_pos).manhattanLength() > QApplication.startDragDistance():
                self.mouse_moved_since_press = True
        
        if self.temp_line and not self.mode.startswith('template'):
            start_point = self.start_atom.pos() if self.start_atom else self.start_pos
            if not start_point:
                super().mouseMoveEvent(event)
                return

            current_pos = event.scenePos()
            end_point = current_pos

            target_atom = None
            for item in self.items(current_pos):
                if isinstance(item, AtomItem):
                    target_atom = item
                    break
            
            is_valid_snap_target = (
                target_atom is not None and
                (self.start_atom is None or target_atom is not self.start_atom)
            )

            if is_valid_snap_target:
                end_point = target_atom.pos()
            
            self.temp_line.setLine(QLineF(start_point, end_point))
        else: 
            # テンプレートモードであっても、ホバーイベントはここで伝播する
            super().mouseMoveEvent(event)

    def mouseReleaseEvent(self, event):
        
        if not self.window.is_2d_editable:
            return 

        end_pos = event.scenePos()
        is_click = self.press_pos and (end_pos - self.press_pos).manhattanLength() < QApplication.startDragDistance()

        if self.temp_line:
            self.removeItem(self.temp_line)
            self.temp_line = None

        if self.mode.startswith('template') and is_click:
            if self.template_context and self.template_context.get('points'):
                context = self.template_context
                self.add_molecule_fragment(context['points'], context['bonds_info'], existing_items=context.get('items', []))
                self.data_changed_in_event = True
                
                # イベント処理をここで完了させ、下のアイテムが選択されるのを防ぐ
                self.start_atom=None; self.start_pos = None; self.press_pos = None
                if self.data_changed_in_event: self.window.push_undo_state()
                return

        released_item = self.itemAt(end_pos, self.views()[0].transform())
        
        # 1. 特殊モード（ラジカル/電荷）の処理
        if (self.mode == 'radical') and is_click and isinstance(released_item, AtomItem):
            atom = released_item
            atom.prepareGeometryChange()
            # ラジカルの状態をトグル (0 -> 1 -> 2 -> 0)
            atom.radical = (atom.radical + 1) % 3 
            self.data.atoms[atom.atom_id]['radical'] = atom.radical
            atom.update_style()
            self.data_changed_in_event = True
            self.start_atom=None; self.start_pos = None; self.press_pos = None
            if self.data_changed_in_event: self.window.push_undo_state()
            return
        elif (self.mode == 'charge_plus' or self.mode == 'charge_minus') and is_click and isinstance(released_item, AtomItem):
            atom = released_item
            atom.prepareGeometryChange()
            delta = 1 if self.mode == 'charge_plus' else -1
            atom.charge += delta
            self.data.atoms[atom.atom_id]['charge'] = atom.charge
            atom.update_style()
            self.data_changed_in_event = True
            self.start_atom=None; self.start_pos = None; self.press_pos = None
            if self.data_changed_in_event: self.window.push_undo_state()
            return

        elif self.mode.startswith('bond') and is_click and isinstance(released_item, BondItem):
            b = released_item 
            
            if self.mode == 'bond_2_5':
                try:
                    if b.order == 2:
                        current_stereo = b.stereo
                        if current_stereo not in [3, 4]:
                            new_stereo = 3  # None -> Z
                        elif current_stereo == 3:
                            new_stereo = 4  # Z -> E
                        else:  # current_stereo == 4
                            new_stereo = 0  # E -> None
                        self.update_bond_stereo(b, new_stereo)
                        self.window.push_undo_state()  # ここでUndo stackに積む
                except Exception as e:
                    print(f"Error in E/Z stereo toggle: {e}")
                    import traceback
                    traceback.print_exc()
                    if hasattr(self.window, 'statusBar'):
                        self.window.statusBar().showMessage(f"Error changing E/Z stereochemistry: {e}", 5000)
                return # この後の処理は行わない
            
            elif self.bond_stereo != 0 and b.order == self.bond_order and b.stereo == self.bond_stereo:
                # 方向性を反転させる
                old_id1, old_id2 = b.atom1.atom_id, b.atom2.atom_id
                
                # 1. 古い方向の結合をデータから削除
                self.data.remove_bond(old_id1, old_id2)
                
                # 2. 逆方向で結合をデータに再追加
                new_key, _ = self.data.add_bond(old_id2, old_id1, self.bond_order, self.bond_stereo)
                
                # 3. BondItemの原子参照を入れ替え、新しいデータと関連付ける
                b.atom1, b.atom2 = b.atom2, b.atom1
                self.data.bonds[new_key]['item'] = b
                
                # 4. 見た目を更新
                b.update_position()

            else:
                # 既存の結合を一度削除
                self.data.remove_bond(b.atom1.atom_id, b.atom2.atom_id)

                # BondItemが記憶している方向(b.atom1 -> b.atom2)で、新しい結合様式を再作成
                # これにより、修正済みのadd_bondが呼ばれ、正しい方向で保存される
                new_key, _ = self.data.add_bond(b.atom1.atom_id, b.atom2.atom_id, self.bond_order, self.bond_stereo)

                # BondItemの見た目とデータ参照を更新
                b.prepareGeometryChange()
                b.order = self.bond_order
                b.stereo = self.bond_stereo
                self.data.bonds[new_key]['item'] = b
                b.update()

            self.clearSelection()
            self.data_changed_in_event = True

        # 3. 新規原子・結合の作成処理 (atom_* モード および すべての bond_* モードで許可)
        elif self.start_atom and (self.mode.startswith('atom') or self.mode.startswith('bond')):
            line = QLineF(self.start_atom.pos(), end_pos); end_item = self.itemAt(end_pos, self.views()[0].transform())

            # 使用する結合様式を決定
            # atomモードの場合は bond_order/stereo を None にして create_bond にデフォルト値(1, 0)を適用
            # bond_* モードの場合は現在の設定 (self.bond_order/stereo) を使用
            order_to_use = self.bond_order if self.mode.startswith('bond') else None
            stereo_to_use = self.bond_stereo if self.mode.startswith('bond') else None
    
            
            if is_click:
                # 短いクリック: 既存原子のシンボル更新 (atomモードのみ)
                if self.mode.startswith('atom') and self.start_atom.symbol != self.current_atom_symbol:
                    self.start_atom.symbol=self.current_atom_symbol; self.data.atoms[self.start_atom.atom_id]['symbol']=self.current_atom_symbol; self.start_atom.update_style()
                    self.data_changed_in_event = True
            else:
                # ドラッグ: 新規結合または既存原子への結合
                if isinstance(end_item, AtomItem) and self.start_atom!=end_item: 
                    self.create_bond(self.start_atom, end_item, bond_order=order_to_use, bond_stereo=stereo_to_use)
                else:
                    new_id = self.create_atom(self.current_atom_symbol, end_pos); new_item = self.data.atoms[new_id]['item']
                    self.create_bond(self.start_atom, new_item, bond_order=order_to_use, bond_stereo=stereo_to_use)
                self.data_changed_in_event = True
                
        # 4. 空白領域からの新規作成処理 (atom_* モード および すべての bond_* モードで許可)
        elif self.start_pos and (self.mode.startswith('atom') or self.mode.startswith('bond')):
            line = QLineF(self.start_pos, end_pos)

            # 使用する結合様式を決定
            order_to_use = self.bond_order if self.mode.startswith('bond') else None
            stereo_to_use = self.bond_stereo if self.mode.startswith('bond') else None
    
            if line.length() < 10:
                self.create_atom(self.current_atom_symbol, end_pos); self.data_changed_in_event = True
            else:
                end_item = self.itemAt(end_pos, self.views()[0].transform())

                if isinstance(end_item, AtomItem):
                    start_id = self.create_atom(self.current_atom_symbol, self.start_pos)
                    start_item = self.data.atoms[start_id]['item']
                    self.create_bond(start_item, end_item, bond_order=order_to_use, bond_stereo=stereo_to_use)
                
                else:
                    start_id = self.create_atom(self.current_atom_symbol, self.start_pos)
                    end_id = self.create_atom(self.current_atom_symbol, end_pos)
                    self.create_bond(
                        self.data.atoms[start_id]['item'], 
                        self.data.atoms[end_id]['item'], 
                        bond_order=order_to_use, 
                        bond_stereo=stereo_to_use
                    )
                self.data_changed_in_event = True 
        
        # 5. それ以外の処理 (Selectモードなど)
        else: super().mouseReleaseEvent(event)


        moved_atoms = [item for item, old_pos in self.initial_positions_in_event.items() if item.scene() and item.pos() != old_pos]
        if moved_atoms:
            self.data_changed_in_event = True
            bonds_to_update = set()
            for atom in moved_atoms:
                self.data.atoms[atom.atom_id]['pos'] = atom.pos()
                bonds_to_update.update(atom.bonds)
            for bond in bonds_to_update: bond.update_position()
            if self.views(): self.views()[0].viewport().update()
        
        self.start_atom=None; self.start_pos = None; self.press_pos = None; self.temp_line = None
        self.template_context = {}
        if self.data_changed_in_event: self.window.push_undo_state()

    def mouseDoubleClickEvent(self, event):
        """ダブルクリックイベントを処理する"""
        item = self.itemAt(event.scenePos(), self.views()[0].transform())

        if self.mode in ['charge_plus', 'charge_minus', 'radical'] and isinstance(item, AtomItem):
            if self.mode == 'radical':
                item.prepareGeometryChange()
                item.radical = (item.radical + 1) % 3
                self.data.atoms[item.atom_id]['radical'] = item.radical
                item.update_style()
            else:
                item.prepareGeometryChange()
                delta = 1 if self.mode == 'charge_plus' else -1
                item.charge += delta
                self.data.atoms[item.atom_id]['charge'] = item.charge
                item.update_style()

            self.window.push_undo_state()

            event.accept()
            return
        
        elif self.mode in ['bond_2_5']:
                event.accept()
                return

        super().mouseDoubleClickEvent(event)

    def create_atom(self, symbol, pos, charge=0, radical=0):
        atom_id = self.data.add_atom(symbol, pos, charge=charge, radical=radical)
        atom_item = AtomItem(atom_id, symbol, pos, charge=charge, radical=radical)
        self.data.atoms[atom_id]['item'] = atom_item; self.addItem(atom_item); return atom_id


    def create_bond(self, start_atom, end_atom, bond_order=None, bond_stereo=None):
        try:
            if start_atom is None or end_atom is None:
                print("Error: Cannot create bond with None atoms")
                return
                
            exist_b = self.find_bond_between(start_atom, end_atom)
            if exist_b:
                return

            # 引数で次数が指定されていればそれを使用し、なければ現在のモードの値を使用する
            order_to_use = self.bond_order if bond_order is None else bond_order
            stereo_to_use = self.bond_stereo if bond_stereo is None else bond_stereo

            key, status = self.data.add_bond(start_atom.atom_id, end_atom.atom_id, order_to_use, stereo_to_use)
            if status == 'created':
                bond_item = BondItem(start_atom, end_atom, order_to_use, stereo_to_use)
                self.data.bonds[key]['item'] = bond_item
                if hasattr(start_atom, 'bonds'):
                    start_atom.bonds.append(bond_item)
                if hasattr(end_atom, 'bonds'):
                    end_atom.bonds.append(bond_item)
                self.addItem(bond_item)
            
            if hasattr(start_atom, 'update_style'):
                start_atom.update_style()
            if hasattr(end_atom, 'update_style'):
                end_atom.update_style()
                
        except Exception as e:
            print(f"Error creating bond: {e}")
            import traceback
            traceback.print_exc()

    def add_molecule_fragment(self, points, bonds_info, existing_items=None, symbol='C'):
        """
        add_molecule_fragment の最終確定版。
        - 既存の結合次数を変更しないポリシーを徹底（最重要）。
        - ベンゼン環テンプレートは、フューズされる既存結合の次数に基づき、
          「新規に作られる二重結合が2本になるように」回転を決定するロジックを適用（条件分岐あり）。
        """
    
        num_points = len(points)
        atom_items = [None] * num_points

        is_benzene_template = (num_points == 6 and any(o == 2 for _, _, o in bonds_info))

    
        def coords(p):
            if hasattr(p, 'x') and hasattr(p, 'y'):
                return (p.x(), p.y())
            try:
                return (p[0], p[1])
            except Exception:
                raise ValueError("point has no x/y")
    
        def dist_pts(a, b):
            ax, ay = coords(a); bx, by = coords(b)
            return math.hypot(ax - bx, ay - by)
    
        # --- 1) 既にクリックされた existing_items をテンプレート頂点にマップ ---
        existing_items = existing_items or []
        used_indices = set()
        ref_lengths = [dist_pts(points[i], points[j]) for i, j, _ in bonds_info if i < num_points and j < num_points]
        avg_len = (sum(ref_lengths) / len(ref_lengths)) if ref_lengths else 20.0
        map_threshold = max(0.5 * avg_len, 8.0)
    
        for ex_item in existing_items:
            try:
                ex_pos = ex_item.pos()
                best_idx, best_d = -1, float('inf')
                for i, p in enumerate(points):
                    if i in used_indices: continue
                    d = dist_pts(p, ex_pos)
                    if best_d is None or d < best_d:
                        best_d, best_idx = d, i
                if best_idx != -1 and best_d <= max(map_threshold, 1.5 * avg_len):
                    atom_items[best_idx] = ex_item
                    used_indices.add(best_idx)
            except Exception:
                pass
    
        # --- 2) シーン内既存原子を self.data.atoms から列挙してマップ ---
        mapped_atoms = {it for it in atom_items if it is not None}
        for i, p in enumerate(points):
            if atom_items[i] is not None: continue
            
            nearby = None
            best_d = float('inf')
            
            for atom_data in self.data.atoms.values():
                a_item = atom_data.get('item')
                if not a_item or a_item in mapped_atoms: continue
                try:
                    d = dist_pts(p, a_item.pos())
                except Exception:
                    continue
                if d < best_d:
                    best_d, nearby = d, a_item

            if nearby and best_d <= map_threshold:
                atom_items[i] = nearby
                mapped_atoms.add(nearby)
    
        # --- 3) 足りない頂点は新規作成　---
        for i, p in enumerate(points):
            if atom_items[i] is None:
                atom_id = self.create_atom(symbol, p)
                atom_items[i] = self.data.atoms[atom_id]['item']
    
        # --- 4) テンプレートのボンド配列を決定（ベンゼン回転合わせの処理） ---
        template_bonds_to_use = list(bonds_info)
        is_6ring = (num_points == 6 and len(bonds_info) == 6)
        template_has_double = any(o == 2 for (_, _, o) in bonds_info)
    
        if is_6ring and template_has_double:
            existing_orders = {} # key: bonds_infoのインデックス, value: 既存の結合次数
            for k, (i_idx, j_idx, _) in enumerate(bonds_info):
                if i_idx < len(atom_items) and j_idx < len(atom_items):
                    a, b = atom_items[i_idx], atom_items[j_idx]
                    if a is None or b is None: continue
                    eb = self.find_bond_between(a, b)
                    if eb:
                        existing_orders[k] = getattr(eb, 'order', 1) 

            if existing_orders:
                orig_orders = [o for (_, _, o) in bonds_info]
                best_rot = 0
                max_score = -999 # スコアは「適合度」を意味する

                # --- フューズされた辺の数による条件分岐 ---
                if len(existing_orders) >= 2:
                    # 2辺以上フューズ: 単純に既存の辺の次数とテンプレートの辺の次数が一致するものを最優先する
                    # (この場合、新しい環を交互配置にするのは難しく、単に既存の構造を壊さないことを優先)
                    for rot in range(num_points):
                        current_score = sum(100 for k, exist_order in existing_orders.items() 
                                            if orig_orders[(k + rot) % num_points] == exist_order)
                        if current_score > max_score:
                            max_score = current_score
                            best_rot = rot

                elif len(existing_orders) == 1:
                    # 1辺フューズ: 既存の辺を維持しつつ、その両隣で「反転一致」を達成し、新しい環を交互配置にする
                    
                    # フューズされた辺のインデックスと次数を取得
                    k_fuse = next(iter(existing_orders.keys()))
                    exist_order = existing_orders[k_fuse]
                    
                    # 目標: フューズされた辺の両隣（k-1とk+1）に来るテンプレートの次数が、既存の辺の次数と逆であること
                    # k_adj_1 -> (k_fuse - 1) % 6
                    # k_adj_2 -> (k_fuse + 1) % 6
                    
                    for rot in range(num_points):
                        current_score = 0
                        rotated_template_order = orig_orders[(k_fuse + rot) % num_points]

                        # 1. まず、フューズされた辺自体が次数を反転させられる位置にあるかチェック（必須ではないが、回転を絞る）
                        if (exist_order == 1 and rotated_template_order == 2) or \
                           (exist_order == 2 and rotated_template_order == 1):
                            current_score += 100 # 大幅ボーナス: 理想的な回転

                        # 2. 次に、両隣の辺の次数をチェック（交互配置維持の主目的）
                        # 既存辺の両隣は、新規に作成されるため、テンプレートの次数でボンドが作成されます。
                        # ここで、テンプレートの次数が既存辺の次数と逆になる回転を選ぶ必要があります。
                        
                        # テンプレートの辺は、回転後のk_fuseの両隣（m_adj1, m_adj2）
                        m_adj1 = (k_fuse - 1 + rot) % num_points 
                        m_adj2 = (k_fuse + 1 + rot) % num_points
                        
                        neighbor_order_1 = orig_orders[m_adj1]
                        neighbor_order_2 = orig_orders[m_adj2]

                        # 既存が単結合(1)の場合、両隣は二重結合(2)であってほしい
                        if exist_order == 1:
                            if neighbor_order_1 == 2: current_score += 50
                            if neighbor_order_2 == 2: current_score += 50
                        
                        # 既存が二重結合(2)の場合、両隣は単結合(1)であってほしい
                        elif exist_order == 2:
                            if neighbor_order_1 == 1: current_score += 50
                            if neighbor_order_2 == 1: current_score += 50
                            
                        # 3. タイブレーク: その他の既存結合（フューズ辺ではない）との次数一致度も加味
                        for k, e_order in existing_orders.items():
                             if k != k_fuse:
                                r_t_order = orig_orders[(k + rot) % num_points]
                                if r_t_order == e_order: current_score += 10 # 既存構造維持のボーナス
                        
                        if current_score > max_score:
                            max_score = current_score
                            best_rot = rot
                
                # 最終的な回転を反映
                new_tb = []
                for m in range(num_points):
                    i_idx, j_idx, _ = bonds_info[m]
                    new_order = orig_orders[(m + best_rot) % num_points]
                    new_tb.append((i_idx, j_idx, new_order))
                template_bonds_to_use = new_tb
    
        # --- 5) ボンド作成／更新---
        for id1_idx, id2_idx, order in template_bonds_to_use:
            if id1_idx < len(atom_items) and id2_idx < len(atom_items):
                a_item, b_item = atom_items[id1_idx], atom_items[id2_idx]
                if not a_item or not b_item or a_item is b_item: continue

                id1, id2 = a_item.atom_id, b_item.atom_id
                if id1 > id2: id1, id2 = id2, id1

                exist_b = self.find_bond_between(a_item, b_item)

                if exist_b:
                    # デフォルトでは既存の結合を維持する
                    should_overwrite = False

                    # 条件1: ベンゼン環テンプレートであること
                    # 条件2: 接続先が単結合であること
                    if is_benzene_template and exist_b.order == 1:

                        # 条件3: 接続先の単結合が共役系の一部ではないこと
                        # (つまり、両端の原子が他に二重結合を持たないこと)
                        atom1 = exist_b.atom1
                        atom2 = exist_b.atom2

                        # atom1が他に二重結合を持つかチェック
                        atom1_has_other_double_bond = any(b.order == 2 for b in atom1.bonds if b is not exist_b)

                        # atom2が他に二重結合を持つかチェック
                        atom2_has_other_double_bond = any(b.order == 2 for b in atom2.bonds if b is not exist_b)

                        # 両方の原子が他に二重結合を持たない「孤立した単結合」の場合のみ上書きフラグを立てる
                        if not atom1_has_other_double_bond and not atom2_has_other_double_bond:
                            should_overwrite = True

                    if should_overwrite:
                        # 上書き条件が全て満たされた場合にのみ、結合次数を更新
                        exist_b.order = order
                        exist_b.stereo = 0
                        self.data.bonds[(id1, id2)]['order'] = order
                        self.data.bonds[(id1, id2)]['stereo'] = 0
                        exist_b.update()
                    else:
                        # 上書き条件を満たさない場合は、既存の結合を維持する
                        continue
                else:
                    # 新規ボンド作成
                    self.create_bond(a_item, b_item, bond_order=order, bond_stereo=0)
        
        # --- 6) 表示更新　---
        for at in atom_items:
            try:
                if at: at.update_style() 
            except Exception:
                pass
    
        return atom_items


    def update_template_preview(self, pos):
        mode_parts = self.mode.split('_')
        is_aromatic = False
        if mode_parts[1] == 'benzene':
            n = 6
            is_aromatic = True
        else:
            try: n = int(mode_parts[1])
            except ValueError: return

        items_under = self.items(pos)  # top-most first
        item = None
        for it in items_under:
            if isinstance(it, (AtomItem, BondItem)):
                item = it
                break

        points, bonds_info = [], []
        l = DEFAULT_BOND_LENGTH
        self.template_context = {}


        if isinstance(item, AtomItem):
            p0 = item.pos()
            continuous_angle = math.atan2(pos.y() - p0.y(), pos.x() - p0.x())
            snap_angle_rad = math.radians(15)
            snapped_angle = round(continuous_angle / snap_angle_rad) * snap_angle_rad
            p1 = p0 + QPointF(l * math.cos(snapped_angle), l * math.sin(snapped_angle))
            points = self._calculate_polygon_from_edge(p0, p1, n)
            self.template_context['items'] = [item]

        elif isinstance(item, BondItem):
            # 結合にスナップ
            p0, p1 = item.atom1.pos(), item.atom2.pos()
            points = self._calculate_polygon_from_edge(p0, p1, n, cursor_pos=pos, use_existing_length=True)
            self.template_context['items'] = [item.atom1, item.atom2]

        else:
            angle_step = 2 * math.pi / n
            start_angle = -math.pi / 2 if n % 2 != 0 else -math.pi / 2 - angle_step / 2
            points = [
                pos + QPointF(l * math.cos(start_angle + i * angle_step), l * math.sin(start_angle + i * angle_step))
                for i in range(n)
            ]

        if points:
            if is_aromatic:
                bonds_info = [(i, (i + 1) % n, 2 if i % 2 == 0 else 1) for i in range(n)]
            else:
                bonds_info = [(i, (i + 1) % n, 1) for i in range(n)]

            self.template_context['points'] = points
            self.template_context['bonds_info'] = bonds_info

            self.template_preview.set_geometry(points, is_aromatic)

            self.template_preview.show()
            if self.views():
                self.views()[0].viewport().update()
        else:
            self.template_preview.hide()
            if self.views():
                self.views()[0].viewport().update()

    def _calculate_polygon_from_edge(self, p0, p1, n, cursor_pos=None, use_existing_length=False):
        if n < 3: return []
        v_edge = p1 - p0
        edge_length = math.sqrt(v_edge.x()**2 + v_edge.y()**2)
        if edge_length == 0: return []
        
        target_length = edge_length if use_existing_length else DEFAULT_BOND_LENGTH
        
        v_edge = (v_edge / edge_length) * target_length
        
        if not use_existing_length:
             p1 = p0 + v_edge

        points = [p0, p1]
        
        interior_angle = (n - 2) * math.pi / n
        rotation_angle = math.pi - interior_angle
        
        if cursor_pos:
            # Note: v_edgeは正規化済みだが、方向は同じなので判定には問題ない
            v_cursor = cursor_pos - p0
            cross_product_z = (p1 - p0).x() * v_cursor.y() - (p1 - p0).y() * v_cursor.x()
            if cross_product_z < 0:
                rotation_angle = -rotation_angle

        cos_a, sin_a = math.cos(rotation_angle), math.sin(rotation_angle)
        
        current_p, current_v = p1, v_edge
        for _ in range(n - 2):
            new_vx = current_v.x() * cos_a - current_v.y() * sin_a
            new_vy = current_v.x() * sin_a + current_v.y() * cos_a
            current_v = QPointF(new_vx, new_vy)
            current_p = current_p + current_v
            points.append(current_p)
        return points

    def delete_items(self, items_to_delete):
        """指定されたアイテムセット（原子・結合）を安全に削除する共通メソッド"""
        try:
            if not items_to_delete:
                return False

            atoms_to_delete = {item for item in items_to_delete if isinstance(item, AtomItem)}
            bonds_to_delete = {item for item in items_to_delete if isinstance(item, BondItem)}

            # 削除対象の原子に接続している結合も、すべて削除対象に加える
            for atom in atoms_to_delete:
                if hasattr(atom, 'bonds'):
                    bonds_to_delete.update(atom.bonds)

            # 影響を受ける（が削除はされない）原子を特定する
            atoms_to_update = set()
            for bond in bonds_to_delete:
                if bond.atom1 and bond.atom1 not in atoms_to_delete:
                    atoms_to_update.add(bond.atom1)
                if bond.atom2 and bond.atom2 not in atoms_to_delete:
                    atoms_to_update.add(bond.atom2)

            # --- データモデルからの削除 ---
            # 最初に原子をデータモデルから削除（関連する結合も内部で削除される）
            for atom in atoms_to_delete:
                if hasattr(atom, 'atom_id'):
                    self.data.remove_atom(atom.atom_id)
                    
            # 次に、明示的に選択された結合（まだ残っているもの）を削除
            for bond in bonds_to_delete:
                if bond.atom1 and bond.atom2 and hasattr(bond.atom1, 'atom_id') and hasattr(bond.atom2, 'atom_id'):
                    self.data.remove_bond(bond.atom1.atom_id, bond.atom2.atom_id)
            
            # --- シーンからのグラフィックアイテム削除（必ず結合を先に）---
            # まずremoveItem
            for bond in bonds_to_delete:
                if bond.scene(): 
                    self.removeItem(bond)
                    
            for atom in atoms_to_delete:
                if atom.scene(): 
                    self.removeItem(atom)
            
            # その後BondItem/AtomItemの参照をクリア
            for bond in bonds_to_delete:
                # atom1/atom2のbondsリストからこのbondを除去
                try:
                    if bond.atom1 and hasattr(bond.atom1, 'bonds') and bond in bond.atom1.bonds:
                        bond.atom1.bonds = [b for b in bond.atom1.bonds if b is not bond]
                    if bond.atom2 and hasattr(bond.atom2, 'bonds') and bond in bond.atom2.bonds:
                        bond.atom2.bonds = [b for b in bond.atom2.bonds if b is not bond]
                    # Clear bond references to prevent dangling pointers
                    bond.atom1 = None
                    bond.atom2 = None
                except Exception as bond_cleanup_error:
                    print(f"Error cleaning up bond references: {bond_cleanup_error}")
            
            for atom in atoms_to_delete:
                # Clear all bond references from deleted atoms
                try:
                    if hasattr(atom, 'bonds'):
                        atom.bonds.clear()
                except Exception as atom_cleanup_error:
                    print(f"Error cleaning up atom references: {atom_cleanup_error}")

            # --- 生き残った原子の内部参照とスタイルを更新 ---
            for atom in atoms_to_update:
                try:
                    if hasattr(atom, 'bonds'):
                        atom.bonds = [b for b in atom.bonds if b not in bonds_to_delete]
                    if hasattr(atom, 'update_style'):
                        atom.update_style()
                except Exception as update_error:
                    print(f"Error updating atom style: {update_error}")
                    
            return True
            
        except Exception as e:
            print(f"Error during delete_items operation: {e}")
            import traceback
            traceback.print_exc()
            return False

    def leaveEvent(self, event):
        self.template_preview.hide(); super().leaveEvent(event)

    def set_hovered_item(self, item):
        """BondItemから呼ばれ、ホバー中のアイテムを記録する"""
        self.hovered_item = item

    def keyPressEvent(self, event):
        view = self.views()[0]
        cursor_pos = view.mapToScene(view.mapFromGlobal(QCursor.pos()))
        item_at_cursor = self.itemAt(cursor_pos, view.transform())
        key = event.key()
        modifiers = event.modifiers()
        
        if not self.window.is_2d_editable:
            return    


        if key == Qt.Key.Key_4:
            # --- 動作1: カーソルが原子/結合上にある場合 (ワンショットでテンプレート配置) ---
            if isinstance(item_at_cursor, (AtomItem, BondItem)):
                
                # ベンゼンテンプレートのパラメータを設定
                n, is_aromatic = 6, True
                points, bonds_info, existing_items = [], [], []
                
                # update_template_preview と同様のロジックで配置情報を計算
                if isinstance(item_at_cursor, AtomItem):
                    p0 = item_at_cursor.pos()
                    l = DEFAULT_BOND_LENGTH
                    direction = QLineF(p0, cursor_pos).unitVector()
                    p1 = p0 + direction.p2() * l if direction.length() > 0 else p0 + QPointF(l, 0)
                    points = self._calculate_polygon_from_edge(p0, p1, n)
                    existing_items = [item_at_cursor]

                elif isinstance(item_at_cursor, BondItem):
                    p0, p1 = item_at_cursor.atom1.pos(), item_at_cursor.atom2.pos()
                    points = self._calculate_polygon_from_edge(p0, p1, n, cursor_pos=cursor_pos, use_existing_length=True)
                    existing_items = [item_at_cursor.atom1, item_at_cursor.atom2]
                
                if points:
                    bonds_info = [(i, (i + 1) % n, 2 if i % 2 == 0 else 1) for i in range(n)]
                    
                    # 計算した情報を使って、その場にフラグメントを追加
                    self.add_molecule_fragment(points, bonds_info, existing_items=existing_items)
                    self.window.push_undo_state()

            # --- 動作2: カーソルが空白領域にある場合 (モード切替) ---
            else:
                self.window.set_mode_and_update_toolbar('template_benzene')

            event.accept()
            return

        # --- 0a. ラジカルの変更 (.) ---
        if key == Qt.Key.Key_Period:
            target_atoms = []
            selected = self.selectedItems()
            if selected:
                target_atoms = [item for item in selected if isinstance(item, AtomItem)]
            elif isinstance(item_at_cursor, AtomItem):
                target_atoms = [item_at_cursor]

            if target_atoms:
                for atom in target_atoms:
                    # ラジカルの状態をトグル (0 -> 1 -> 2 -> 0)
                    atom.prepareGeometryChange()
                    atom.radical = (atom.radical + 1) % 3
                    self.data.atoms[atom.atom_id]['radical'] = atom.radical
                    atom.update_style()
                self.window.push_undo_state()
                event.accept()
                return

        # --- 0b. 電荷の変更 (+/-キー) ---
        if key == Qt.Key.Key_Plus or key == Qt.Key.Key_Minus:
            target_atoms = []
            selected = self.selectedItems()
            if selected:
                target_atoms = [item for item in selected if isinstance(item, AtomItem)]
            elif isinstance(item_at_cursor, AtomItem):
                target_atoms = [item_at_cursor]

            if target_atoms:
                delta = 1 if key == Qt.Key.Key_Plus else -1
                for atom in target_atoms:
                    atom.prepareGeometryChange()
                    atom.charge += delta
                    self.data.atoms[atom.atom_id]['charge'] = atom.charge
                    atom.update_style()
                self.window.push_undo_state()
                event.accept()
                return

        # --- 1. Atomに対する操作 (元素記号の変更) ---
        if isinstance(item_at_cursor, AtomItem):
            new_symbol = None
            if modifiers == Qt.KeyboardModifier.NoModifier and key in self.key_to_symbol_map:
                new_symbol = self.key_to_symbol_map[key]
            elif modifiers == Qt.KeyboardModifier.ShiftModifier and key in self.key_to_symbol_map_shift:
                new_symbol = self.key_to_symbol_map_shift[key]

            if new_symbol and item_at_cursor.symbol != new_symbol:
                item_at_cursor.prepareGeometryChange()
                
                item_at_cursor.symbol = new_symbol
                self.data.atoms[item_at_cursor.atom_id]['symbol'] = new_symbol
                item_at_cursor.update_style()


                atoms_to_update = {item_at_cursor}
                for bond in item_at_cursor.bonds:
                    bond.update()
                    other_atom = bond.atom1 if bond.atom2 is item_at_cursor else bond.atom2
                    atoms_to_update.add(other_atom)

                for atom in atoms_to_update:
                    atom.update_style()

                self.window.push_undo_state()
                event.accept()
                return

        # --- 2. Bondに対する操作 (次数・立体化学の変更) ---
        target_bonds = []
        if isinstance(item_at_cursor, BondItem):
            target_bonds = [item_at_cursor]
        else:
            target_bonds = [it for it in self.selectedItems() if isinstance(it, BondItem)]

        if target_bonds:
            any_bond_changed = False
            for bond in target_bonds:
                # 1. 結合の向きを考慮して、データ辞書内の現在のキーを正しく特定する
                id1, id2 = bond.atom1.atom_id, bond.atom2.atom_id
                current_key = None
                if (id1, id2) in self.data.bonds:
                    current_key = (id1, id2)
                elif (id2, id1) in self.data.bonds:
                    current_key = (id2, id1)
                
                if not current_key: continue

                # 2. 変更前の状態を保存
                old_order, old_stereo = bond.order, bond.stereo

                # 3. キー入力に応じてBondItemのプロパティを変更
                if key == Qt.Key.Key_W:
                    if bond.stereo == 1:
                        bond_data = self.data.bonds.pop(current_key)
                        new_key = (current_key[1], current_key[0])
                        self.data.bonds[new_key] = bond_data
                        bond.atom1, bond.atom2 = bond.atom2, bond.atom1
                        bond.update_position()
                        was_reversed = True
                    else:
                        bond.order = 1; bond.stereo = 1

                elif key == Qt.Key.Key_D:
                    if bond.stereo == 2:
                        bond_data = self.data.bonds.pop(current_key)
                        new_key = (current_key[1], current_key[0])
                        self.data.bonds[new_key] = bond_data
                        bond.atom1, bond.atom2 = bond.atom2, bond.atom1
                        bond.update_position()
                        was_reversed = True
                    else:
                        bond.order = 1; bond.stereo = 2

                elif key == Qt.Key.Key_1 and (bond.order != 1 or bond.stereo != 0):
                    bond.order = 1; bond.stereo = 0
                elif key == Qt.Key.Key_2 and (bond.order != 2 or bond.stereo != 0):
                    bond.order = 2; bond.stereo = 0; needs_update = True
                elif key == Qt.Key.Key_3 and bond.order != 3:
                    bond.order = 3; bond.stereo = 0; needs_update = True

                # 4. 実際に変更があった場合のみデータモデルを更新
                if old_order != bond.order or old_stereo != bond.stereo:
                    any_bond_changed = True
                    
                    # 5. 古いキーでデータを辞書から一度削除
                    bond_data = self.data.bonds.pop(current_key)
                    bond_data['order'] = bond.order
                    bond_data['stereo'] = bond.stereo

                    # 6. 変更後の種類に応じて新しいキーを決定し、再登録する
                    new_key_id1, new_key_id2 = bond.atom1.atom_id, bond.atom2.atom_id
                    if bond.stereo == 0:
                        if new_key_id1 > new_key_id2:
                            new_key_id1, new_key_id2 = new_key_id2, new_key_id1
                    
                    new_key = (new_key_id1, new_key_id2)
                    self.data.bonds[new_key] = bond_data
                    
                    bond.update()

            if any_bond_changed:
                self.window.push_undo_state()
            
            if key in [Qt.Key.Key_1, Qt.Key.Key_2, Qt.Key.Key_3, Qt.Key.Key_W, Qt.Key.Key_D]:
                event.accept()
                return

        if isinstance(self.hovered_item, BondItem) and self.hovered_item.order == 2:
            if event.key() == Qt.Key.Key_Z:
                self.update_bond_stereo(self.hovered_item, 3)  # Z-isomer
                self.window.push_undo_state()
                event.accept()
                return
            elif event.key() == Qt.Key.Key_E:
                self.update_bond_stereo(self.hovered_item, 4)  # E-isomer
                self.window.push_undo_state()
                event.accept()
                return
                    
        # --- 3. Atomに対する操作 (原子の追加 - マージされた機能) ---
        if key == Qt.Key.Key_1:
            start_atom = None
            if isinstance(item_at_cursor, AtomItem):
                start_atom = item_at_cursor
            else:
                selected_atoms = [item for item in self.selectedItems() if isinstance(item, AtomItem)]
                if len(selected_atoms) == 1:
                    start_atom = selected_atoms[0]

            if start_atom:
                start_pos = start_atom.pos()
                l = DEFAULT_BOND_LENGTH
                new_pos_offset = QPointF(0, -l) # デフォルトのオフセット (上)

                # 接続している原子のリストを取得 (H原子以外)
                neighbor_positions = []
                for bond in start_atom.bonds:
                    other_atom = bond.atom1 if bond.atom2 is start_atom else bond.atom2
                    if other_atom.symbol != 'H': # 水素原子を無視 (四面体構造の考慮のため)
                        neighbor_positions.append(other_atom.pos())

                num_non_H_neighbors = len(neighbor_positions)
                
                if num_non_H_neighbors == 0:
                    # 結合ゼロ: デフォルト方向
                    new_pos_offset = QPointF(0, -l)
                
                elif num_non_H_neighbors == 1:
                    # 結合1本: 既存結合と約120度（または60度）の角度
                    bond = start_atom.bonds[0]
                    other_atom = bond.atom1 if bond.atom2 is start_atom else bond.atom2
                    existing_bond_vector = start_pos - other_atom.pos()
                    
                    # 既存の結合から時計回り60度回転 (ベンゼン環のような構造にしやすい)
                    angle_rad = math.radians(60) 
                    cos_a, sin_a = math.cos(angle_rad), math.sin(angle_rad)
                    vx, vy = existing_bond_vector.x(), existing_bond_vector.y()
                    new_vx, new_vy = vx * cos_a - vy * sin_a, vx * sin_a + vy * cos_a
                    rotated_vector = QPointF(new_vx, new_vy)
                    line = QLineF(QPointF(0, 0), rotated_vector)
                    line.setLength(l)
                    new_pos_offset = line.p2()

                elif num_non_H_neighbors == 3:

                    bond_vectors_sum = QPointF(0, 0)
                    for pos in neighbor_positions:
                        # start_pos から neighbor_pos へのベクトル
                        vec = pos - start_pos 
                        # 単位ベクトルに変換
                        line_to_other = QLineF(QPointF(0,0), vec)
                        if line_to_other.length() > 0:
                            line_to_other.setLength(1.0)
                            bond_vectors_sum += line_to_other.p2()
                    
                    # SUM_TOLERANCE is now a module-level constant
                    if bond_vectors_sum.manhattanLength() > SUM_TOLERANCE:
                        new_direction_line = QLineF(QPointF(0,0), -bond_vectors_sum)
                        new_direction_line.setLength(l)
                        new_pos_offset = new_direction_line.p2()
                    else:
                        new_pos_offset = QPointF(l * 0.7071, -l * 0.7071) 


                else: # 2本または4本以上の場合 (一般的な骨格の継続、または過結合)
                    bond_vectors_sum = QPointF(0, 0)
                    for bond in start_atom.bonds:
                        other_atom = bond.atom1 if bond.atom2 is start_atom else bond.atom2
                        line_to_other = QLineF(start_pos, other_atom.pos())
                        if line_to_other.length() > 0:
                            line_to_other.setLength(1.0)
                            bond_vectors_sum += line_to_other.p2() - line_to_other.p1()
                    
                    if bond_vectors_sum.manhattanLength() > 0.01:
                        new_direction_line = QLineF(QPointF(0,0), -bond_vectors_sum)
                        new_direction_line.setLength(l)
                        new_pos_offset = new_direction_line.p2()
                    else:
                        # 総和がゼロの場合は、デフォルト（上）
                         new_pos_offset = QPointF(0, -l)


                # SNAP_DISTANCE is a module-level constant
                target_pos = start_pos + new_pos_offset
                
                # 近くに原子を探す
                near_atom = self.find_atom_near(target_pos, tol=SNAP_DISTANCE)
                
                if near_atom and near_atom is not start_atom:
                    # 近くに既存原子があれば結合
                    self.create_bond(start_atom, near_atom)
                else:
                    # 新規原子を作成し結合
                    new_atom_id = self.create_atom('C', target_pos)
                    new_atom_item = self.data.atoms[new_atom_id]['item']
                    self.create_bond(start_atom, new_atom_item)

                self.clearSelection()
                self.window.push_undo_state()
                event.accept()
                return

        # --- 4. 全体に対する操作 (削除、モード切替など) ---
        if key == Qt.Key.Key_Delete or key == Qt.Key.Key_Backspace:
            if self.temp_line:
                self.removeItem(self.temp_line)
                self.temp_line = None; self.start_atom = None; self.start_pos = None
                self.initial_positions_in_event = {}
                event.accept()
                return

            items_to_process = set(self.selectedItems()) 
            # カーソル下のアイテムも削除対象に加える
            if item_at_cursor and isinstance(item_at_cursor, (AtomItem, BondItem)):
                items_to_process.add(item_at_cursor)

            if self.delete_items(items_to_process):
                self.window.push_undo_state()
                self.window.statusBar().showMessage("Deleted selected items.")

            # もしデータモデル内の原子が全て無くなっていたら、シーンをクリアして初期状態に戻す
            if not self.data.atoms:
                # 1. シーン上の全グラフィックアイテムを削除する
                self.clear() 

                # 2. テンプレートプレビューなど、初期状態で必要なアイテムを再生成する
                self.reinitialize_items()
                
                # 3. 結合描画中などの一時的な状態も完全にリセットする
                self.temp_line = None
                self.start_atom = None
                self.start_pos = None
                self.initial_positions_in_event = {}
                
                # このイベントはここで処理完了とする
                event.accept()
                return
    
            # 描画の強制更新
            if self.views():
                self.views()[0].viewport().update() 
                QApplication.processEvents()
    
                event.accept()
                return
        

        if key == Qt.Key.Key_Space:
            if self.mode != 'select':
                self.window.activate_select_mode()
            else:
                self.window.select_all()
            event.accept()
            return

        # グローバルな描画モード切替
        mode_to_set = None

        # 1. 原子描画モードへの切り替え
        symbol_for_mode_change = None
        if modifiers == Qt.KeyboardModifier.NoModifier and key in self.key_to_symbol_map:
            symbol_for_mode_change = self.key_to_symbol_map[key]
        elif modifiers == Qt.KeyboardModifier.ShiftModifier and key in self.key_to_symbol_map_shift:
            symbol_for_mode_change = self.key_to_symbol_map_shift[key]
        
        if symbol_for_mode_change:
            mode_to_set = f'atom_{symbol_for_mode_change}'

        # 2. 結合描画モードへの切り替え
        elif modifiers == Qt.KeyboardModifier.NoModifier and key in self.key_to_bond_mode_map:
            mode_to_set = self.key_to_bond_mode_map[key]

        # モードが決定されていれば、モード変更を実行
        if mode_to_set:
            if hasattr(self.window, 'set_mode_and_update_toolbar'):
                 self.window.set_mode_and_update_toolbar(mode_to_set)
                 event.accept()
                 return
        
        # --- どの操作にも当てはまらない場合 ---
        super().keyPressEvent(event)
        
    def find_atom_near(self, pos, tol=14.0):
        # Create a small search rectangle around the position
        search_rect = QRectF(pos.x() - tol, pos.y() - tol, 2 * tol, 2 * tol)
        nearby_items = self.items(search_rect)

        for it in nearby_items:
            if isinstance(it, AtomItem):
                # Check the precise distance only for candidate items
                if QLineF(it.pos(), pos).length() <= tol:
                    return it
        return None

    def find_bond_between(self, atom1, atom2):
        for b in atom1.bonds:
            if (b.atom1 is atom1 and b.atom2 is atom2) or \
               (b.atom1 is atom2 and b.atom2 is atom1):
                return b
        return None

    def update_bond_stereo(self, bond_item, new_stereo):
        """結合の立体化学を更新する共通メソッド"""
        try:
            if bond_item is None:
                print("Error: bond_item is None in update_bond_stereo")
                return
                
            if bond_item.order != 2 or bond_item.stereo == new_stereo:
                return

            if not hasattr(bond_item, 'atom1') or not hasattr(bond_item, 'atom2'):
                print("Error: bond_item missing atom references")
                return
                
            if bond_item.atom1 is None or bond_item.atom2 is None:
                print("Error: bond_item has None atom references")
                return
                
            if not hasattr(bond_item.atom1, 'atom_id') or not hasattr(bond_item.atom2, 'atom_id'):
                print("Error: bond atoms missing atom_id")
                return

            id1, id2 = bond_item.atom1.atom_id, bond_item.atom2.atom_id

            # E/Z結合は方向性を持つため、キーは(id1, id2)のまま探す
            key_to_update = (id1, id2)
            if key_to_update not in self.data.bonds:
                # Wedge/Dashなど、逆順で登録されている可能性も考慮
                key_to_update = (id2, id1)
                if key_to_update not in self.data.bonds:
                    # Log error instead of printing to console
                    if hasattr(self.window, 'statusBar'):
                        self.window.statusBar().showMessage(f"Warning: Bond between atoms {id1} and {id2} not found in data model.", 3000)
                    print(f"Error: Bond key not found: {id1}-{id2} or {id2}-{id1}")
                    return
                    
            # Update data model
            self.data.bonds[key_to_update]['stereo'] = new_stereo
            
            # Update visual representation
            bond_item.set_stereo(new_stereo)
            
            self.data_changed_in_event = True
            
        except Exception as e:
            print(f"Error in update_bond_stereo: {e}")
            import traceback
            traceback.print_exc()
            if hasattr(self.window, 'statusBar'):
                self.window.statusBar().showMessage(f"Error updating bond stereochemistry: {e}", 5000)

class ZoomableView(QGraphicsView):
    """ マウスホイールでのズームと、中ボタン or Shift+左ドラッグでのパン機能を追加したQGraphicsView """
    def __init__(self, scene, parent=None):
        super().__init__(scene, parent)
        self.setTransformationAnchor(QGraphicsView.ViewportAnchor.AnchorUnderMouse)
        self.setResizeAnchor(QGraphicsView.ViewportAnchor.AnchorViewCenter)
        self.setVerticalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOn)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOn)
        self.setDragMode(QGraphicsView.DragMode.NoDrag)

        self.main_window = parent
        self.setAcceptDrops(False)

        self._is_panning = False
        self._pan_start_pos = QPointF()
        self._pan_start_scroll_h = 0
        self._pan_start_scroll_v = 0

    def wheelEvent(self, event):
        """ マウスホイールを回した際のイベント """
        if event.modifiers() & Qt.KeyboardModifier.ControlModifier:
            zoom_in_factor = 1.1
            zoom_out_factor = 1 / zoom_in_factor

            transform = self.transform()
            current_scale = transform.m11()
            min_scale, max_scale = 0.05, 20.0

            if event.angleDelta().y() > 0:
                if max_scale > current_scale:
                    self.scale(zoom_in_factor, zoom_in_factor)
            else:
                if min_scale < current_scale:
                    self.scale(zoom_out_factor, zoom_out_factor)
            
            event.accept() 
        else:
            super().wheelEvent(event)

    def mousePressEvent(self, event):
        """ 中ボタン or Shift+左ボタンが押されたらパン（視点移動）モードを開始 """
        is_middle_button = event.button() == Qt.MouseButton.MiddleButton
        is_shift_left_button = (event.button() == Qt.MouseButton.LeftButton and
                                event.modifiers() & Qt.KeyboardModifier.ShiftModifier)

        if is_middle_button or is_shift_left_button:
            self._is_panning = True
            self._pan_start_pos = event.pos() # ビューポート座標で開始点を記録
            # 現在のスクロールバーの位置を記録
            self._pan_start_scroll_h = self.horizontalScrollBar().value()
            self._pan_start_scroll_v = self.verticalScrollBar().value()
            self.setCursor(Qt.CursorShape.ClosedHandCursor)
            event.accept()
        else:
            super().mousePressEvent(event)

    def mouseMoveEvent(self, event):
        """ パンモード中にマウスを動かしたら、スクロールバーを操作して視点を移動させる """
        if self._is_panning:
            delta = event.pos() - self._pan_start_pos # マウスの移動量を計算
            # 開始時のスクロール位置から移動量を引いた値を新しいスクロール位置に設定
            self.horizontalScrollBar().setValue(self._pan_start_scroll_h - delta.x())
            self.verticalScrollBar().setValue(self._pan_start_scroll_v - delta.y())
            event.accept()
        else:
            super().mouseMoveEvent(event)

    def mouseReleaseEvent(self, event):
        """ パンに使用したボタンが離されたらパンモードを終了 """
        # パンを開始したボタン（中 or 左）のどちらかが離されたかをチェック
        is_middle_button_release = event.button() == Qt.MouseButton.MiddleButton
        is_left_button_release = event.button() == Qt.MouseButton.LeftButton

        if self._is_panning and (is_middle_button_release or is_left_button_release):
            self._is_panning = False
            # 現在の描画モードに応じたカーソルに戻す
            current_mode = self.scene().mode if self.scene() else 'select'
            if current_mode == 'select':
                self.setCursor(Qt.CursorShape.ArrowCursor)
            elif current_mode.startswith(('atom', 'bond', 'template')):
                self.setCursor(Qt.CursorShape.CrossCursor)
            elif current_mode.startswith(('charge', 'radical')):
                self.setCursor(Qt.CursorShape.CrossCursor)
            else:
                self.setCursor(Qt.CursorShape.ArrowCursor)
            event.accept()
        else:
            super().mouseReleaseEvent(event)



class CalculationWorker(QObject):
    status_update = pyqtSignal(str) 
    finished=pyqtSignal(object); error=pyqtSignal(str)
    
    def run_calculation(self, mol_block):
        try:
            if not mol_block:
                raise ValueError("No atoms to convert.")
            
            self.status_update.emit("Creating 3D structure...")
            mol = Chem.MolFromMolBlock(mol_block, removeHs=False)
            if mol is None:
                raise ValueError("Failed to create molecule from MOL block.")

            # CRITICAL FIX: Extract and restore explicit E/Z labels from MOL block
            # Parse M CFG lines to get explicit stereo labels
            explicit_stereo = {}
            mol_lines = mol_block.split('\n')
            for line in mol_lines:
                if line.startswith('M  CFG'):
                    parts = line.split()
                    if len(parts) >= 4:
                        try:
                            bond_idx = int(parts[3]) - 1  # MOL format is 1-indexed
                            cfg_value = int(parts[4])
                            # cfg_value: 1=Z, 2=E in MOL format
                            if cfg_value == 1:
                                explicit_stereo[bond_idx] = Chem.BondStereo.STEREOZ
                            elif cfg_value == 2:
                                explicit_stereo[bond_idx] = Chem.BondStereo.STEREOE
                        except (ValueError, IndexError):
                            continue

            # Force explicit stereo labels regardless of coordinates
            for bond_idx, stereo_type in explicit_stereo.items():
                if bond_idx < mol.GetNumBonds():
                    bond = mol.GetBondWithIdx(bond_idx)
                    if bond.GetBondType() == Chem.BondType.DOUBLE:
                        # Find suitable stereo atoms
                        begin_atom = bond.GetBeginAtom()
                        end_atom = bond.GetEndAtom()
                        
                        # Pick heavy atom neighbors preferentially
                        begin_neighbors = [nbr for nbr in begin_atom.GetNeighbors() if nbr.GetIdx() != end_atom.GetIdx()]
                        end_neighbors = [nbr for nbr in end_atom.GetNeighbors() if nbr.GetIdx() != begin_atom.GetIdx()]
                        
                        if begin_neighbors and end_neighbors:
                            # Prefer heavy atoms
                            begin_heavy = [n for n in begin_neighbors if n.GetAtomicNum() > 1]
                            end_heavy = [n for n in end_neighbors if n.GetAtomicNum() > 1]
                            
                            stereo_atom1 = (begin_heavy[0] if begin_heavy else begin_neighbors[0]).GetIdx()
                            stereo_atom2 = (end_heavy[0] if end_heavy else end_neighbors[0]).GetIdx()
                            
                            bond.SetStereoAtoms(stereo_atom1, stereo_atom2)
                            bond.SetStereo(stereo_type)

            # Do NOT call AssignStereochemistry here as it overrides our explicit labels

            mol = Chem.AddHs(mol)
            
            # CRITICAL: Re-apply explicit stereo after AddHs which may renumber atoms
            for bond_idx, stereo_type in explicit_stereo.items():
                if bond_idx < mol.GetNumBonds():
                    bond = mol.GetBondWithIdx(bond_idx)
                    if bond.GetBondType() == Chem.BondType.DOUBLE:
                        # Re-find suitable stereo atoms after hydrogen addition
                        begin_atom = bond.GetBeginAtom()
                        end_atom = bond.GetEndAtom()
                        
                        # Pick heavy atom neighbors preferentially
                        begin_neighbors = [nbr for nbr in begin_atom.GetNeighbors() if nbr.GetIdx() != end_atom.GetIdx()]
                        end_neighbors = [nbr for nbr in end_atom.GetNeighbors() if nbr.GetIdx() != begin_atom.GetIdx()]
                        
                        if begin_neighbors and end_neighbors:
                            # Prefer heavy atoms
                            begin_heavy = [n for n in begin_neighbors if n.GetAtomicNum() > 1]
                            end_heavy = [n for n in end_neighbors if n.GetAtomicNum() > 1]
                            
                            stereo_atom1 = (begin_heavy[0] if begin_heavy else begin_neighbors[0]).GetIdx()
                            stereo_atom2 = (end_heavy[0] if end_heavy else end_neighbors[0]).GetIdx()
                            
                            bond.SetStereoAtoms(stereo_atom1, stereo_atom2)
                            bond.SetStereo(stereo_type)

            params = AllChem.ETKDGv2()
            params.randomSeed = 42
            # CRITICAL: Force ETKDG to respect the existing stereochemistry
            params.useExpTorsionAnglePrefs = True
            params.useBasicKnowledge = True
            params.enforceChirality = True  # This is critical for stereo preservation
            
            # Store original stereochemistry before embedding (prioritizing explicit labels)
            original_stereo_info = []
            for bond_idx, stereo_type in explicit_stereo.items():
                if bond_idx < mol.GetNumBonds():
                    bond = mol.GetBondWithIdx(bond_idx)
                    if bond.GetBondType() == Chem.BondType.DOUBLE:
                        stereo_atoms = bond.GetStereoAtoms()
                        original_stereo_info.append((bond.GetIdx(), stereo_type, stereo_atoms))
            
            # Also store any other stereo bonds not in explicit_stereo
            for bond in mol.GetBonds():
                if (bond.GetBondType() == Chem.BondType.DOUBLE and 
                    bond.GetStereo() != Chem.BondStereo.STEREONONE and
                    bond.GetIdx() not in explicit_stereo):
                    stereo_atoms = bond.GetStereoAtoms()
                    original_stereo_info.append((bond.GetIdx(), bond.GetStereo(), stereo_atoms))
            
            self.status_update.emit("Embedding 3D coordinates...")
            
            # Try multiple times with different approaches if needed
            conf_id = -1
            
            # First attempt: Standard ETKDG with stereo enforcement
            try:
                conf_id = AllChem.EmbedMolecule(mol, params)
            except Exception as e:
                self.status_update.emit(f"Standard embedding failed: {e}")
            
            # Second attempt: Use constraint embedding if available
            if conf_id == -1:
                try:
                    # Create distance constraints for double bonds to enforce E/Z geometry
                    from rdkit.DistanceGeometry import DoTriangleSmoothing
                    bounds_matrix = AllChem.GetMoleculeBoundsMatrix(mol)
                    
                    # Add constraints for E/Z bonds
                    for bond_idx, stereo, stereo_atoms in original_stereo_info:
                        bond = mol.GetBondWithIdx(bond_idx)
                        if len(stereo_atoms) == 2:
                            atom1_idx = bond.GetBeginAtomIdx()
                            atom2_idx = bond.GetEndAtomIdx()
                            neighbor1_idx = stereo_atoms[0]
                            neighbor2_idx = stereo_atoms[1]
                            
                            # For Z (cis): neighbors should be closer
                            # For E (trans): neighbors should be farther
                            if stereo == Chem.BondStereo.STEREOZ:
                                # Z configuration: set shorter distance constraint
                                target_dist = 3.0  # Angstroms
                                bounds_matrix[neighbor1_idx][neighbor2_idx] = min(bounds_matrix[neighbor1_idx][neighbor2_idx], target_dist)
                                bounds_matrix[neighbor2_idx][neighbor1_idx] = min(bounds_matrix[neighbor2_idx][neighbor1_idx], target_dist)
                            elif stereo == Chem.BondStereo.STEREOE:
                                # E configuration: set longer distance constraint  
                                target_dist = 5.0  # Angstroms
                                bounds_matrix[neighbor1_idx][neighbor2_idx] = max(bounds_matrix[neighbor1_idx][neighbor2_idx], target_dist)
                                bounds_matrix[neighbor2_idx][neighbor1_idx] = max(bounds_matrix[neighbor2_idx][neighbor1_idx], target_dist)
                    
                    DoTriangleSmoothing(bounds_matrix)
                    conf_id = AllChem.EmbedMolecule(mol, bounds_matrix, params)
                    self.status_update.emit("Constraint-based embedding succeeded")
                except Exception as e:
                    self.status_update.emit(f"Constraint embedding failed: {e}")
                    
            # Fallback: Try basic embedding
            if conf_id == -1:
                try:
                    basic_params = AllChem.ETKDGv2()
                    basic_params.randomSeed = 42
                    conf_id = AllChem.EmbedMolecule(mol, basic_params)
                except Exception:
                    pass
            '''
            if conf_id == -1:
                self.status_update.emit("Initial embedding failed, retrying with ignoreSmoothingFailures=True...")
                # Try again with ignoreSmoothingFailures instead of random-seed retries
                params.ignoreSmoothingFailures = True
                # Use a deterministic seed to avoid random-coordinate behavior here
                params.randomSeed = 0
                conf_id = AllChem.EmbedMolecule(mol, params)

            if conf_id == -1:
                self.status_update.emit("Random-seed retry failed, attempting with random coordinates...")
                try:
                    conf_id = AllChem.EmbedMolecule(mol, useRandomCoords=True, ignoreSmoothingFailures=True)
                except TypeError:
                    # Some RDKit versions expect useRandomCoords in params
                    params.useRandomCoords = True
                    conf_id = AllChem.EmbedMolecule(mol, params)
            '''

            if conf_id != -1:
                # Success with RDKit: optimize and finish
                # CRITICAL: Restore original stereochemistry after embedding (explicit labels first)
                for bond_idx, stereo, stereo_atoms in original_stereo_info:
                    bond = mol.GetBondWithIdx(bond_idx)
                    if len(stereo_atoms) == 2:
                        bond.SetStereoAtoms(stereo_atoms[0], stereo_atoms[1])
                    bond.SetStereo(stereo)
                
                try:
                    AllChem.MMFFOptimizeMolecule(mol)
                except Exception:
                    # fallback to UFF if MMFF fails
                    try:
                        AllChem.UFFOptimizeMolecule(mol)
                    except Exception:
                        pass
                
                # CRITICAL: Restore stereochemistry again after optimization (explicit labels priority)
                for bond_idx, stereo, stereo_atoms in original_stereo_info:
                    bond = mol.GetBondWithIdx(bond_idx)
                    if len(stereo_atoms) == 2:
                        bond.SetStereoAtoms(stereo_atoms[0], stereo_atoms[1])
                    bond.SetStereo(stereo)
                
                # Do NOT call AssignStereochemistry here as it would override our explicit labels
                self.finished.emit(mol)
                self.status_update.emit("RDKit 3D conversion succeeded.")
                return

            # ---------- RDKit failed: try Open Babel via pybel only (no CLI fallback) ----------
            '''
            self.status_update.emit("RDKit embedding failed. Attempting Open Babel fallback...")

            try:
                # pybel expects an input format; provide mol block
                # pybel.readstring accepts format strings like "mol" or "smi"
                ob_mol = pybel.readstring("mol", mol_block)
                # ensure hydrogens
                try:
                    ob_mol.addh()
                except Exception:
                    pass
                # build 3D coordinates
                ob_mol.make3D()
                try:
                    # まず第一候補であるMMFF94で最適化を試みる
                    self.status_update.emit("Optimizing with Open Babel (MMFF94)...")
                    ob_mol.localopt(forcefield='mmff94', steps=500)
                except Exception:
                    # MMFF94が失敗した場合、UFFにフォールバックして再試行
                    try:
                        self.status_update.emit("MMFF94 failed, falling back to UFF...")
                        ob_mol.localopt(forcefield='uff', steps=500)
                    except Exception:
                        # UFFも失敗した場合はスキップ
                        self.status_update.emit("UFF optimization also failed.")
                        pass
                # get molblock and convert to RDKit
                molblock_ob = ob_mol.write("mol")
                rd_mol = Chem.MolFromMolBlock(molblock_ob, removeHs=False)
                if rd_mol is None:
                    raise ValueError("Open Babel produced invalid MOL block.")
                # optimize in RDKit as a final step if possible
                rd_mol = Chem.AddHs(rd_mol)
                # Do NOT call AssignStereochemistry here to preserve original stereo info
                try:
                    AllChem.MMFFOptimizeMolecule(rd_mol)
                except Exception:
                    try:
                        AllChem.UFFOptimizeMolecule(rd_mol)
                    except Exception:
                        pass
                # Do NOT call AssignStereochemistry after optimization either
                self.status_update.emit("Open Babel embedding succeeded. Warning: Conformation accuracy may be limited.")
                self.finished.emit(rd_mol)
                return
            except Exception as ob_err:
                # pybel was available but failed
                raise RuntimeError(f"Open Babel 3D conversion failed: {ob_err}")
            '''

        except Exception as e:
            self.error.emit(str(e))


class PeriodicTableDialog(QDialog):
    element_selected = pyqtSignal(str)
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Select an Element")
        layout = QGridLayout(self)
        self.setLayout(layout)

        elements = [
            ('H',1,1), ('He',1,18),
            ('Li',2,1), ('Be',2,2), ('B',2,13), ('C',2,14), ('N',2,15), ('O',2,16), ('F',2,17), ('Ne',2,18),
            ('Na',3,1), ('Mg',3,2), ('Al',3,13), ('Si',3,14), ('P',3,15), ('S',3,16), ('Cl',3,17), ('Ar',3,18),
            ('K',4,1), ('Ca',4,2), ('Sc',4,3), ('Ti',4,4), ('V',4,5), ('Cr',4,6), ('Mn',4,7), ('Fe',4,8),
            ('Co',4,9), ('Ni',4,10), ('Cu',4,11), ('Zn',4,12), ('Ga',4,13), ('Ge',4,14), ('As',4,15), ('Se',4,16),
            ('Br',4,17), ('Kr',4,18),
            ('Rb',5,1), ('Sr',5,2), ('Y',5,3), ('Zr',5,4), ('Nb',5,5), ('Mo',5,6), ('Tc',5,7), ('Ru',5,8),
            ('Rh',5,9), ('Pd',5,10), ('Ag',5,11), ('Cd',5,12), ('In',5,13), ('Sn',5,14), ('Sb',5,15), ('Te',5,16),
            ('I',5,17), ('Xe',5,18),
            ('Cs',6,1), ('Ba',6,2), ('La',6,3), ('Hf',6,4), ('Ta',6,5), ('W',6,6), ('Re',6,7), ('Os',6,8),
            ('Ir',6,9), ('Pt',6,10), ('Au',6,11), ('Hg',6,12), ('Tl',6,13), ('Pb',6,14), ('Bi',6,15), ('Po',6,16),
            ('At',6,17), ('Rn',6,18),
            ('Fr',7,1), ('Ra',7,2), ('Ac',7,3), ('Rf',7,4), ('Db',7,5), ('Sg',7,6), ('Bh',7,7), ('Hs',7,8),
            ('Mt',7,9), ('Ds',7,10), ('Rg',7,11), ('Cn',7,12), ('Nh',7,13), ('Fl',7,14), ('Mc',7,15), ('Lv',7,16),
            ('Ts',7,17), ('Og',7,18),
            # Lanthanides (placed on a separate row)
            ('La',8,3), ('Ce',8,4), ('Pr',8,5), ('Nd',8,6), ('Pm',8,7), ('Sm',8,8), ('Eu',8,9), ('Gd',8,10), ('Tb',8,11),
            ('Dy',8,12), ('Ho',8,13), ('Er',8,14), ('Tm',8,15), ('Yb',8,16), ('Lu',8,17),
            # Actinides (separate row)
            ('Ac',9,3), ('Th',9,4), ('Pa',9,5), ('U',9,6), ('Np',9,7), ('Pu',9,8), ('Am',9,9), ('Cm',9,10), ('Bk',9,11),
            ('Cf',9,12), ('Es',9,13), ('Fm',9,14), ('Md',9,15), ('No',9,16), ('Lr',9,17),
        ]
        for symbol, row, col in elements:
            b = QPushButton(symbol)
            b.setFixedSize(40,40)

            # CPK_COLORSから色を取得。見つからない場合はデフォルト色を使用
            q_color = CPK_COLORS.get(symbol, CPK_COLORS['DEFAULT'])

            # 背景色の輝度を計算して、文字色を黒か白に決定
            # 輝度 = (R*299 + G*587 + B*114) / 1000
            brightness = (q_color.red() * 299 + q_color.green() * 587 + q_color.blue() * 114) / 1000
            text_color = "white" if brightness < 128 else "black"

            # ボタンのスタイルシートを設定
            b.setStyleSheet(
                f"background-color: {q_color.name()};"
                f"color: {text_color};"
                "border: 1px solid #555;"
                "font-weight: bold;"
            )

            b.clicked.connect(self.on_button_clicked)
            layout.addWidget(b, row, col)

    def on_button_clicked(self):
        b=self.sender()
        self.element_selected.emit(b.text())
        self.accept()

# --- 最終版 AnalysisWindow クラス ---
class AnalysisWindow(QDialog):
    def __init__(self, mol, parent=None):
        super().__init__(parent)
        self.mol = mol
        self.setWindowTitle("Molecule Analysis")
        self.setMinimumWidth(400)
        self.init_ui()

    def init_ui(self):
        main_layout = QVBoxLayout(self)
        grid_layout = QGridLayout()
        
        # --- 分子特性を計算 ---
        try:
            # RDKitのモジュールをインポート
            from rdkit import Chem
            from rdkit.Chem import Descriptors, rdMolDescriptors

            # SMILES生成用に、一時的に水素原子を取り除いた分子オブジェクトを作成
            mol_for_smiles = Chem.RemoveHs(self.mol)
            # 水素を取り除いた分子からSMILESを生成（常に簡潔な表記になる）
            smiles = Chem.MolToSmiles(mol_for_smiles, isomericSmiles=True)

            # 各種プロパティを計算
            mol_formula = rdMolDescriptors.CalcMolFormula(self.mol)
            mol_wt = Descriptors.MolWt(self.mol)
            exact_mw = Descriptors.ExactMolWt(self.mol)
            num_heavy_atoms = self.mol.GetNumHeavyAtoms()
            num_rings = rdMolDescriptors.CalcNumRings(self.mol)
            log_p = Descriptors.MolLogP(self.mol)
            tpsa = Descriptors.TPSA(self.mol)
            num_h_donors = rdMolDescriptors.CalcNumHBD(self.mol)
            num_h_acceptors = rdMolDescriptors.CalcNumHBA(self.mol)
            
            # 表示するプロパティを辞書にまとめる
            properties = {
                "SMILES:": smiles,
                "Molecular Formula:": mol_formula,
                "Molecular Weight:": f"{mol_wt:.4f}",
                "Exact Mass:": f"{exact_mw:.4f}",
                "Heavy Atoms:": str(num_heavy_atoms),
                "Ring Count:": str(num_rings),
                "LogP (o/w):": f"{log_p:.3f}",
                "TPSA (Å²):": f"{tpsa:.2f}",
                "H-Bond Donors:": str(num_h_donors),
                "H-Bond Acceptors:": str(num_h_acceptors),
            }
        except Exception as e:
            main_layout.addWidget(QLabel(f"Error calculating properties: {e}"))
            return

        # --- 計算結果をUIに表示 ---
        row = 0
        for label_text, value_text in properties.items():
            label = QLabel(f"<b>{label_text}</b>")
            value = QLineEdit(value_text)
            value.setReadOnly(True)
            
            copy_btn = QPushButton("Copy")
            copy_btn.clicked.connect(lambda _, v=value: self.copy_to_clipboard(v.text()))

            grid_layout.addWidget(label, row, 0)
            grid_layout.addWidget(value, row, 1)
            grid_layout.addWidget(copy_btn, row, 2)
            row += 1
            
        main_layout.addLayout(grid_layout)
        
        # --- OKボタン ---
        ok_button = QPushButton("OK")
        ok_button.clicked.connect(self.accept)
        main_layout.addWidget(ok_button, 0, Qt.AlignmentFlag.AlignCenter)
        
        self.setLayout(main_layout)

    def copy_to_clipboard(self, text):
        clipboard = QApplication.clipboard()
        clipboard.setText(text)
        if self.parent() and hasattr(self.parent(), 'statusBar'):
            self.parent().statusBar().showMessage(f"Copied '{text}' to clipboard.", 2000)


class SettingsDialog(QDialog):
    def __init__(self, current_settings, parent=None):
        super().__init__(parent)
        self.setWindowTitle("3D View Settings")
        
        # デフォルト設定をクラス内で定義
        self.default_settings = {
            'background_color': '#919191',
            'lighting_enabled': True,
            'specular': 0.20,
            'specular_power': 20,
            'light_intensity': 1.0,
            'show_3d_axes': True,
        }
        
        # --- 選択された色を管理する専用のインスタンス変数 ---
        self.current_bg_color = None

        # --- UI要素の作成 ---
        layout = QVBoxLayout(self)
        form_layout = QFormLayout()

        # 1. 背景色
        self.bg_button = QPushButton()
        self.bg_button.setToolTip("Click to select a color")
        self.bg_button.clicked.connect(self.select_color)
        form_layout.addRow("Background Color:", self.bg_button)

        # 1a. 軸の表示/非表示
        self.axes_checkbox = QCheckBox()
        form_layout.addRow("Show 3D Axes:", self.axes_checkbox)

        # 2. ライトの有効/無効
        self.light_checkbox = QCheckBox()
        form_layout.addRow("Enable Lighting:", self.light_checkbox)

        # 光の強さスライダーを追加
        self.intensity_slider = QSlider(Qt.Orientation.Horizontal)
        self.intensity_slider.setRange(0, 200) # 0.0 ~ 2.0 の範囲
        form_layout.addRow("Light Intensity:", self.intensity_slider)

        # 3. 光沢 (Specular)
        self.specular_slider = QSlider(Qt.Orientation.Horizontal)
        self.specular_slider.setRange(0, 100)
        form_layout.addRow("Shininess (Specular):", self.specular_slider)
        
        # 4. 光沢の強さ (Specular Power)
        self.spec_power_slider = QSlider(Qt.Orientation.Horizontal)
        self.spec_power_slider.setRange(0, 100)
        form_layout.addRow("Shininess Power:", self.spec_power_slider)

        # 渡された設定でUIと内部変数を初期化
        self.update_ui_from_settings(current_settings)

        layout.addLayout(form_layout)

        # --- ボタンの配置 ---
        buttons = QHBoxLayout()
        reset_button = QPushButton("Reset to Defaults")
        reset_button.clicked.connect(self.reset_to_defaults)
        
        ok_button = QPushButton("OK")
        cancel_button = QPushButton("Cancel")
        ok_button.clicked.connect(self.accept)
        cancel_button.clicked.connect(self.reject)

        buttons.addWidget(reset_button)
        buttons.addStretch(1)
        buttons.addWidget(ok_button)
        buttons.addWidget(cancel_button)
        layout.addLayout(buttons)

    def reset_to_defaults(self):
        """UIをデフォルト設定に戻す"""
        self.update_ui_from_settings(self.default_settings)

    def update_ui_from_settings(self, settings_dict):
        self.current_bg_color = settings_dict.get('background_color', self.default_settings['background_color'])
        self.update_color_button(self.current_bg_color)
        self.axes_checkbox.setChecked(settings_dict.get('show_3d_axes', self.default_settings['show_3d_axes']))
        self.light_checkbox.setChecked(settings_dict.get('lighting_enabled', self.default_settings['lighting_enabled']))
        self.intensity_slider.setValue(int(settings_dict.get('light_intensity', self.default_settings['light_intensity']) * 100))
        self.specular_slider.setValue(int(settings_dict.get('specular', self.default_settings['specular']) * 100))
        self.spec_power_slider.setValue(settings_dict.get('specular_power', self.default_settings['specular_power']))
      
    def select_color(self):
        """カラーピッカーを開き、選択された色を内部変数とUIに反映させる"""
        # 内部変数から現在の色を取得してカラーピッカーを初期化
        color = QColorDialog.getColor(QColor(self.current_bg_color), self)
        if color.isValid():
            # 内部変数を更新
            self.current_bg_color = color.name()
            # UIの見た目を更新
            self.update_color_button(self.current_bg_color)

    def update_color_button(self, color_hex):
        """ボタンの背景色と境界線を設定する"""
        self.bg_button.setStyleSheet(f"background-color: {color_hex}; border: 1px solid #888;")

    def get_settings(self):
        return {
            'background_color': self.current_bg_color,
            'show_3d_axes': self.axes_checkbox.isChecked(),
            'lighting_enabled': self.light_checkbox.isChecked(),
            'light_intensity': self.intensity_slider.value() / 100.0,
            'specular': self.specular_slider.value() / 100.0,
            'specular_power': self.spec_power_slider.value()
        }


class CustomQtInteractor(QtInteractor):
    def __init__(self, parent=None, main_window=None, **kwargs):
        super().__init__(parent, **kwargs)
        self.main_window = main_window

    def wheelEvent(self, event):
        """
        マウスホイールイベントをオーバーライドする。
        """
        # 最初に親クラスのイベントを呼び、通常のズーム処理を実行させる
        super().wheelEvent(event)
        
        # ズーム処理の完了後、2Dビューにフォーカスを強制的に戻す
        if self.main_window and hasattr(self.main_window, 'view_2d'):
            self.main_window.view_2d.setFocus()


    def mouseReleaseEvent(self, event):
        """
        Qtのマウスリリースイベントをオーバーライドし、
        3Dビューでの全ての操作完了後に2Dビューへフォーカスを戻す。
        """
        super().mouseReleaseEvent(event) # 親クラスのイベントを先に処理
        if self.main_window and hasattr(self.main_window, 'view_2d'):
            self.main_window.view_2d.setFocus()

# --- 3Dインタラクションを管理する専用クラス ---
class CustomInteractorStyle(vtkInteractorStyleTrackballCamera):
    def __init__(self, main_window):
        super().__init__()
        self.main_window = main_window
        # カスタム状態を管理するフラグを一つに絞ります
        self._is_dragging_atom = False
        # undoスタックのためのフラグ
        self.is_dragging = False

        self.AddObserver("LeftButtonPressEvent", self.on_left_button_down)
        self.AddObserver("MouseMoveEvent", self.on_mouse_move)
        self.AddObserver("LeftButtonReleaseEvent", self.on_left_button_up)

    def on_left_button_down(self, obj, event):
        """
        クリック時の処理を振り分けます。
        原子を掴めた場合のみカスタム動作に入り、それ以外は親クラス（カメラ回転）に任せます。
        """
        mw = self.main_window
        is_temp_mode = bool(QApplication.keyboardModifiers() & Qt.KeyboardModifier.AltModifier)
        is_edit_active = mw.is_3d_edit_mode or is_temp_mode

        # 測定モードが有効な場合の処理
        if mw.measurement_mode and mw.current_mol:
            click_pos = self.GetInteractor().GetEventPosition()
            picker = mw.plotter.picker
            picker.Pick(click_pos[0], click_pos[1], 0, mw.plotter.renderer)

            # 原子がクリックされた場合のみ特別処理
            if picker.GetActor() is mw.atom_actor:
                picked_position = np.array(picker.GetPickPosition())
                distances = np.linalg.norm(mw.atom_positions_3d - picked_position, axis=1)
                closest_atom_idx = np.argmin(distances)

                # クリック閾値チェック
                atom = mw.current_mol.GetAtomWithIdx(int(closest_atom_idx))
                if atom:
                    atomic_num = atom.GetAtomicNum()
                    vdw_radius = pt.GetRvdw(atomic_num)
                    click_threshold = vdw_radius * 1.5

                    if distances[closest_atom_idx] < click_threshold:
                        mw.handle_measurement_atom_selection(int(closest_atom_idx))
                        return  # 原子選択処理完了、カメラ回転は無効
            
            # 空白領域をクリックした場合、測定をクリアしてから通常の回転処理を継続
            if picker.GetActor() != mw.atom_actor:
                mw.clear_measurement_selection()
            
            # 測定モードでも通常のカメラ回転を許可
            self._is_dragging_atom = False
            super().OnLeftButtonDown()
            return

        # 3D分子(mw.current_mol)が存在する場合のみ、原子の選択処理を実行
        if is_edit_active and mw.current_mol:
            click_pos = self.GetInteractor().GetEventPosition()
            picker = mw.plotter.picker
            picker.Pick(click_pos[0], click_pos[1], 0, mw.plotter.renderer)

            if picker.GetActor() is mw.atom_actor:
                picked_position = np.array(picker.GetPickPosition())
                distances = np.linalg.norm(mw.atom_positions_3d - picked_position, axis=1)
                closest_atom_idx = np.argmin(distances)

                # RDKitのMolオブジェクトから原子を安全に取得
                atom = mw.current_mol.GetAtomWithIdx(int(closest_atom_idx))
                if atom:
                    atomic_num = atom.GetAtomicNum()
                    vdw_radius = pt.GetRvdw(atomic_num)
                    click_threshold = vdw_radius * 1.5

                    if distances[closest_atom_idx] < click_threshold:
                        # 原子を掴むことに成功した場合
                        self._is_dragging_atom = True
                        self.is_dragging = False 
                        mw.dragged_atom_info = {'id': int(closest_atom_idx)}
                        mw.plotter.setCursor(Qt.CursorShape.ClosedHandCursor)
                        return  # 親クラスのカメラ回転を呼ばない

        self._is_dragging_atom = False
        super().OnLeftButtonDown()

    def on_mouse_move(self, obj, event):
        """
        マウス移動時の処理。原子ドラッグ中か、それ以外（カメラ回転＋ホバー）かをハンドリングします。
        """
        mw = self.main_window
        interactor = self.GetInteractor()

        if self._is_dragging_atom:
            # カスタムの原子ドラッグ処理
            self.is_dragging = True
            atom_id = mw.dragged_atom_info['id']
            conf = mw.current_mol.GetConformer()
            renderer = mw.plotter.renderer
            current_display_pos = interactor.GetEventPosition()
            pos_3d = conf.GetAtomPosition(atom_id)
            renderer.SetWorldPoint(pos_3d.x, pos_3d.y, pos_3d.z, 1.0)
            renderer.WorldToDisplay()
            display_coords = renderer.GetDisplayPoint()
            new_display_pos = (current_display_pos[0], current_display_pos[1], display_coords[2])
            renderer.SetDisplayPoint(new_display_pos[0], new_display_pos[1], new_display_pos[2])
            renderer.DisplayToWorld()
            new_world_coords_tuple = renderer.GetWorldPoint()
            new_world_coords = list(new_world_coords_tuple)[:3]
            mw.atom_positions_3d[atom_id] = new_world_coords
            mw.glyph_source.points = mw.atom_positions_3d
            mw.glyph_source.Modified()
            conf.SetAtomPosition(atom_id, new_world_coords)
            interactor.Render()
        else:
            # カメラ回転処理を親クラスに任せます
            super().OnMouseMove()

            # その後、カーソルの表示を更新します
            is_edit_active = mw.is_3d_edit_mode or interactor.GetAltKey()
            if is_edit_active:
                # 編集がアクティブな場合のみ、原子のホバーチェックを行う
                atom_under_cursor = False
                click_pos = interactor.GetEventPosition()
                picker = mw.plotter.picker
                picker.Pick(click_pos[0], click_pos[1], 0, mw.plotter.renderer)
                if picker.GetActor() is mw.atom_actor:
                    atom_under_cursor = True

                if atom_under_cursor:
                    mw.plotter.setCursor(Qt.CursorShape.OpenHandCursor)
                else:
                    mw.plotter.setCursor(Qt.CursorShape.ArrowCursor)
            else:
                mw.plotter.setCursor(Qt.CursorShape.ArrowCursor)

    def on_left_button_up(self, obj, event):
        """
        クリック終了時の処理。状態をリセットします。
        """
        mw = self.main_window

        if self._is_dragging_atom:
            # カスタムドラッグの後始末
            if self.is_dragging:
                if mw.current_mol:
                    mw.draw_molecule_3d(mw.current_mol)
                mw.push_undo_state()
            mw.dragged_atom_info = None
        else:
            # カメラ回転の後始末を親クラスに任せます
            super().OnLeftButtonUp()

        # 状態をリセット
        self._is_dragging_atom = False
        self.is_dragging = False # is_draggingもリセット
        
        # ボタンを離した後のカーソル表示を最新の状態に更新
        self.on_mouse_move(obj, event)

        # 2Dビューにフォーカスを戻し、ショートカットキーなどが使えるようにする
        if mw and mw.view_2d:
            mw.view_2d.setFocus()

class MainWindow(QMainWindow):

    start_calculation = pyqtSignal(str)
    def __init__(self, initial_file=None):
        super().__init__()
        self.setAcceptDrops(True)
        self.settings_dir = os.path.join(os.path.expanduser('~'), '.moleditpy')
        self.settings_file = os.path.join(self.settings_dir, 'settings.json')
        self.settings = {}
        self.load_settings()
        self.initial_settings = self.settings.copy()
        self.setWindowTitle("MoleditPy -- Python Molecular Editor  Ver. " + VERSION); self.setGeometry(100, 100, 1400, 800)
        self.data = MolecularData(); self.current_mol = None
        self.current_3d_style = 'ball_and_stick'
        self.show_chiral_labels = False
        self.is_3d_edit_mode = False
        self.dragged_atom_info = None
        self.atom_actor = None 
        self.is_2d_editable = True
        self.axes_actor = None
        self.axes_widget = None
        self.undo_stack = []
        self.redo_stack = []
        self.mode_actions = {} 
        
        # 測定機能用の変数
        self.measurement_mode = False
        self.selected_atoms_for_measurement = []
        self.measurement_labels = []  # (atom_idx, label_text) のタプルのリスト
        self.measurement_text_actor = None
        
        self.init_ui()
        self.init_worker_thread()
        self._setup_3d_picker() 

        # --- RDKit初回実行コストの事前読み込み（ウォームアップ）---
        try:
            # Create a molecule with a variety of common atoms to ensure
            # the valence/H-count machinery is fully initialized.
            warmup_smiles = "OC(N)C(S)P"
            warmup_mol = Chem.MolFromSmiles(warmup_smiles)
            if warmup_mol:
                for atom in warmup_mol.GetAtoms():
                    atom.GetNumImplicitHs()
        except Exception as e:
            print(f"RDKit warm-up failed: {e}")

        self.reset_undo_stack()
        self.scene.selectionChanged.connect(self.update_edit_menu_actions)
        QApplication.clipboard().dataChanged.connect(self.update_edit_menu_actions)

        self.update_edit_menu_actions()

        if initial_file:
            self.load_command_line_file(initial_file)
        
        QTimer.singleShot(0, self.apply_initial_settings)
        # カメラ初期化フラグ（初回描画時のみリセットを許可する）
        self._camera_initialized = False

    def init_ui(self):
        # 1. 現在のスクリプトがあるディレクトリのパスを取得
        script_dir = os.path.dirname(os.path.abspath(__file__))
        
        # 2. 'assets'フォルダ内のアイコンファイルへのフルパスを構築
        icon_path = os.path.join(script_dir, 'assets', 'icon.png')
        
        # 3. ファイルパスから直接QIconオブジェクトを作成
        if os.path.exists(icon_path): # ファイルが存在するか確認
            app_icon = QIcon(icon_path)
            
            # 4. ウィンドウにアイコンを設定
            self.setWindowIcon(app_icon)
        else:
            print(f"警告: アイコンファイルが見つかりません: {icon_path}")

        self.init_menu_bar()

        self.splitter = QSplitter(Qt.Orientation.Horizontal)
        self.setCentralWidget(self.splitter)

        left_pane=QWidget()
        left_pane.setAcceptDrops(True)
        left_layout=QVBoxLayout(left_pane)

        self.scene=MoleculeScene(self.data,self)
        self.scene.setSceneRect(-4000,-4000,4000,4000)
        self.scene.setBackgroundBrush(QColor("#FFFFFF"))

        self.view_2d=ZoomableView(self.scene, self)
        self.view_2d.setRenderHint(QPainter.RenderHint.Antialiasing)
        self.view_2d.setSizePolicy(
            QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding
        )
        left_layout.addWidget(self.view_2d, 1)

        self.view_2d.scale(0.75, 0.75)

        # --- 左パネルのボタンレイアウト ---
        left_buttons_layout = QHBoxLayout()
        self.cleanup_button = QPushButton("Optimize 2D")
        self.cleanup_button.clicked.connect(self.clean_up_2d_structure)
        left_buttons_layout.addWidget(self.cleanup_button)

        self.convert_button = QPushButton("Convert 2D to 3D")
        self.convert_button.clicked.connect(self.trigger_conversion)
        left_buttons_layout.addWidget(self.convert_button)
        
        left_layout.addLayout(left_buttons_layout)
        self.splitter.addWidget(left_pane)

        # --- 右パネルとボタンレイアウト ---
        right_pane = QWidget()
        # 1. 右パネル全体は「垂直」レイアウトにする
        right_layout = QVBoxLayout(right_pane)
        self.plotter = CustomQtInteractor(right_pane, main_window=self, lighting='none')
        self.plotter.setAcceptDrops(False)
        self.plotter.setSizePolicy(
            QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding
        )
        # 2. 垂直レイアウトに3Dビューを追加
        right_layout.addWidget(self.plotter, 1)
        #self.plotter.installEventFilter(self)

        # 3. ボタンをまとめるための「水平」レイアウトを作成
        right_buttons_layout = QHBoxLayout()

        # 3D最適化ボタン
        self.optimize_3d_button = QPushButton("Optimize 3D")
        self.optimize_3d_button.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed)
        self.optimize_3d_button.clicked.connect(self.optimize_3d_structure)
        self.optimize_3d_button.setEnabled(False) # 初期状態は無効
        right_buttons_layout.addWidget(self.optimize_3d_button)

        # エクスポートボタン (メニュー付き)
        self.export_button = QToolButton()
        self.export_button.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed)
        self.export_button.setText("Export 3D")
        self.export_button.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
        self.export_button.setEnabled(False) # 初期状態は無効

        export_menu = QMenu(self)
        export_mol_action = QAction("Export as MOL...", self)
        export_mol_action.triggered.connect(self.save_3d_as_mol)
        export_menu.addAction(export_mol_action)

        export_xyz_action = QAction("Export as XYZ...", self)
        export_xyz_action.triggered.connect(self.save_as_xyz)
        export_menu.addAction(export_xyz_action)

        self.export_button.setMenu(export_menu)
        right_buttons_layout.addWidget(self.export_button)

        # 4. 水平のボタンレイアウトを、全体の垂直レイアウトに追加
        right_layout.addLayout(right_buttons_layout)
        self.splitter.addWidget(right_pane)
        
        self.splitter.setSizes([600, 600])

        # ステータスバーを左右に分離するための設定
        self.status_bar = self.statusBar()
        self.formula_label = QLabel("")  # 右側に表示するラベルを作成
        # 右端に余白を追加して見栄えを調整
        self.formula_label.setStyleSheet("padding-right: 8px;")
        # ラベルを右側に常時表示ウィジェットとして追加
        self.status_bar.addPermanentWidget(self.formula_label)

        #self.view_2d.fitInView(self.scene.sceneRect(), Qt.AspectRatioMode.KeepAspectRatio)

        toolbar = QToolBar("Main Toolbar")
        self.addToolBar(toolbar)
        self.tool_group = QActionGroup(self)
        self.tool_group.setExclusive(True)

        actions_data = [
            ("Select", 'select', 'Space'), ("C", 'atom_C', 'c'), ("H", 'atom_H', 'h'), ("B", 'atom_B', 'b'),
            ("N", 'atom_N', 'n'), ("O", 'atom_O', 'o'), ("S", 'atom_S', 's'), ("Si", 'atom_Si', 'Shift+S'), ("P", 'atom_P', 'p'), 
            ("F", 'atom_F', 'f'), ("Cl", 'atom_Cl', 'Shift+C'), ("Br", 'atom_Br', 'Shift+B'), ("I", 'atom_I', 'i'), 
            ("Other...", 'atom_other', '')
        ]

        for text, mode, shortcut_text in actions_data:
            if text == "C": toolbar.addSeparator()
            
            action = QAction(text, self, checkable=(mode != 'atom_other'))
            if shortcut_text: action.setToolTip(f"{text} ({shortcut_text})")

            if mode == 'atom_other':
                action.triggered.connect(self.open_periodic_table_dialog)
                self.other_atom_action = action
            else:
                action.triggered.connect(lambda c, m=mode: self.set_mode(m))
                self.mode_actions[mode] = action

            toolbar.addAction(action)
            if mode != 'atom_other': self.tool_group.addAction(action)
            
            if text == "Select":
                select_action = action
        
        toolbar.addSeparator()

        # --- 結合ボタンのアイコンを生成するヘルパー関数 ---
        def create_bond_icon(bond_type, size=32):
            pixmap = QPixmap(size, size)
            pixmap.fill(Qt.GlobalColor.transparent)
            painter = QPainter(pixmap)
            painter.setRenderHint(QPainter.RenderHint.Antialiasing)

            p1 = QPointF(6, size / 2)
            p2 = QPointF(size - 6, size / 2)
            line = QLineF(p1, p2)

            painter.setPen(QPen(Qt.GlobalColor.black, 2))
            painter.setBrush(QBrush(Qt.GlobalColor.black))

            if bond_type == 'single':
                painter.drawLine(line)
            elif bond_type == 'double':
                v = line.unitVector().normalVector()
                offset = QPointF(v.dx(), v.dy()) * 2.5
                painter.drawLine(line.translated(offset))
                painter.drawLine(line.translated(-offset))
            elif bond_type == 'triple':
                v = line.unitVector().normalVector()
                offset = QPointF(v.dx(), v.dy()) * 3.0
                painter.drawLine(line)
                painter.drawLine(line.translated(offset))
                painter.drawLine(line.translated(-offset))
            elif bond_type == 'wedge':
                vec = line.unitVector()
                normal = vec.normalVector()
                offset = QPointF(normal.dx(), normal.dy()) * 5.0
                poly = QPolygonF([p1, p2 + offset, p2 - offset])
                painter.drawPolygon(poly)
            elif bond_type == 'dash':
                vec = line.unitVector()
                normal = vec.normalVector()

                num_dashes = NUM_DASHES
                for i in range(num_dashes + 1):
                    t = i / num_dashes
                    start_pt = p1 * (1 - t) + p2 * t
                    width = 10 * t
                    offset = QPointF(normal.dx(), normal.dy()) * width / 2.0
                    painter.setPen(QPen(Qt.GlobalColor.black, 1.5))
                    painter.drawLine(start_pt - offset, start_pt + offset)

            elif bond_type == 'ez_toggle':
                # アイコン下部に二重結合を描画
                p1 = QPointF(6, size * 0.75)
                p2 = QPointF(size - 6, size * 0.75)
                line = QLineF(p1, p2)
                v = line.unitVector().normalVector()
                offset = QPointF(v.dx(), v.dy()) * 2.0
                painter.setPen(QPen(Qt.GlobalColor.black, 2))
                painter.drawLine(line.translated(offset))
                painter.drawLine(line.translated(-offset))
                # 上部に "Z⇌E" のテキストを描画
                painter.setPen(QPen(Qt.GlobalColor.black, 1))
                font = painter.font()
                font.setPointSize(10)
                font.setBold(True)
                painter.setFont(font)
                text_rect = QRectF(0, 0, size, size * 0.6)
                # U+21CC は右向きと左向きのハープーンが重なった記号 (⇌)
                painter.drawText(text_rect, Qt.AlignmentFlag.AlignCenter, "Z⇌E")

            painter.end()
            return QIcon(pixmap)

        # --- 結合ボタンをツールバーに追加 ---
        bond_actions_data = [
            ("Single Bond", 'bond_1_0', '1', 'single'),
            ("Double Bond", 'bond_2_0', '2', 'double'),
            ("Triple Bond", 'bond_3_0', '3', 'triple'),
            ("Wedge Bond", 'bond_1_1', 'W', 'wedge'),
            ("Dash Bond", 'bond_1_2', 'D', 'dash'),
            ("Toggle E/Z", 'bond_2_5', 'E/Z', 'ez_toggle'),
        ]

        for text, mode, shortcut_text, icon_type in bond_actions_data:
            action = QAction(self)
            action.setIcon(create_bond_icon(icon_type))
            action.setToolTip(f"{text} ({shortcut_text})")
            action.setCheckable(True)
            action.triggered.connect(lambda checked, m=mode: self.set_mode(m))
            self.mode_actions[mode] = action
            toolbar.addAction(action)
            self.tool_group.addAction(action)
        
        toolbar.addSeparator()

        charge_plus_action = QAction("+ Charge", self, checkable=True)
        charge_plus_action.setToolTip("Increase Atom Charge (+)")
        charge_plus_action.triggered.connect(lambda c, m='charge_plus': self.set_mode(m))
        self.mode_actions['charge_plus'] = charge_plus_action
        toolbar.addAction(charge_plus_action)
        self.tool_group.addAction(charge_plus_action)

        charge_minus_action = QAction("- Charge", self, checkable=True)
        charge_minus_action.setToolTip("Decrease Atom Charge (-)")
        charge_minus_action.triggered.connect(lambda c, m='charge_minus': self.set_mode(m))
        self.mode_actions['charge_minus'] = charge_minus_action
        toolbar.addAction(charge_minus_action)
        self.tool_group.addAction(charge_minus_action)

        radical_action = QAction("Radical", self, checkable=True)
        radical_action.setToolTip("Toggle Radical (0/1/2) (.)")
        radical_action.triggered.connect(lambda c, m='radical': self.set_mode(m))
        self.mode_actions['radical'] = radical_action
        toolbar.addAction(radical_action)
        self.tool_group.addAction(radical_action)

        toolbar.addSeparator()
        toolbar.addWidget(QLabel(" Templates:"))
        
        # --- アイコンを生成するヘルパー関数 ---
        def create_template_icon(n, is_benzene=False):
            size = 32
            pixmap = QPixmap(size, size)
            pixmap.fill(Qt.GlobalColor.transparent)
            painter = QPainter(pixmap)
            painter.setRenderHint(QPainter.RenderHint.Antialiasing)
            painter.setPen(QPen(Qt.GlobalColor.black, 2))

            center = QPointF(size / 2, size / 2)
            radius = size / 2 - 4 # アイコンの余白

            points = []
            angle_step = 2 * math.pi / n
            # ポリゴンが直立するように開始角度を調整
            start_angle = -math.pi / 2 if n % 2 != 0 else -math.pi / 2 - angle_step / 2

            for i in range(n):
                angle = start_angle + i * angle_step
                x = center.x() + radius * math.cos(angle)
                y = center.y() + radius * math.sin(angle)
                points.append(QPointF(x, y))

            painter.drawPolygon(QPolygonF(points))

            if is_benzene:
                painter.drawEllipse(center, radius * 0.6, radius * 0.6)

            if n in [7, 8, 9]:
                font = QFont("Arial", 10, QFont.Weight.Bold)
                painter.setFont(font)
                painter.drawText(QRectF(0, 0, size, size), Qt.AlignmentFlag.AlignCenter, str(n))

            painter.end()
            return QIcon(pixmap)

        # --- ヘルパー関数を使ってアイコン付きボタンを作成 ---
        templates = [("Benzene", "template_benzene", 6)] + [(f"{i}-Ring", f"template_{i}", i) for i in range(3, 10)]
        for text, mode, n in templates:
            action = QAction(self) # テキストなしでアクションを作成
            action.setCheckable(True)

            is_benzene = (text == "Benzene")
            icon = create_template_icon(n, is_benzene=is_benzene)
            action.setIcon(icon) # アイコンを設定

            if text == "Benzene":
                action.setToolTip(f"{text} Template (4)")
            else:
                action.setToolTip(f"{text} Template")

            action.triggered.connect(lambda c, m=mode: self.set_mode(m))
            self.mode_actions[mode] = action
            toolbar.addAction(action)
            self.tool_group.addAction(action)

        # 初期モードを'select'から'atom_C'（炭素原子描画モード）に変更
        self.set_mode('atom_C')
        # 対応するツールバーの'C'ボタンを選択状態にする
        if 'atom_C' in self.mode_actions:
            self.mode_actions['atom_C'].setChecked(True)

        # スペーサーを追加して、次のウィジェットを右端に配置する
        spacer = QWidget()
        spacer.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        toolbar.addWidget(spacer)

        # 測定機能ボタンを追加
        self.measurement_action = QAction("Measure", self, checkable=True)
        self.measurement_action.setToolTip("Enable distance, angle, and dihedral measurement in 3D view")
        self.measurement_action.setEnabled(False)  # 初期状態では無効
        self.measurement_action.triggered.connect(self.toggle_measurement_mode)
        toolbar.addAction(self.measurement_action)

        self.edit_3d_action = QAction("3D Edit", self, checkable=True)
        self.edit_3d_action.setToolTip("Toggle 3D atom editing mode (Hold Alt for temporary mode)")
        self.edit_3d_action.setEnabled(False)
        self.edit_3d_action.toggled.connect(self.toggle_3d_edit_mode)
        toolbar.addAction(self.edit_3d_action)

        # 3Dスタイル変更ボタンとメニューを作成

        self.style_button = QToolButton()
        self.style_button.setText("3D Style")
        self.style_button.setPopupMode(QToolButton.ToolButtonPopupMode.InstantPopup)
        toolbar.addWidget(self.style_button)

        style_menu = QMenu(self)
        self.style_button.setMenu(style_menu)

        style_group = QActionGroup(self)
        style_group.setExclusive(True)

        # Ball & Stick アクション
        bs_action = QAction("Ball & Stick", self, checkable=True)
        bs_action.setChecked(True)
        bs_action.triggered.connect(lambda: self.set_3d_style('ball_and_stick'))
        style_menu.addAction(bs_action)
        style_group.addAction(bs_action)

        # CPK アクション
        cpk_action = QAction("CPK (Space-filling)", self, checkable=True)
        cpk_action.triggered.connect(lambda: self.set_3d_style('cpk'))
        style_menu.addAction(cpk_action)
        style_group.addAction(cpk_action)

        quit_shortcut = QShortcut(QKeySequence("Ctrl+Q"), self)
        quit_shortcut.activated.connect(self.close)

        self.view_2d.setFocus()

    def init_menu_bar(self):
        menu_bar = self.menuBar()
        
        file_menu = menu_bar.addMenu("&File")
        load_mol_action = QAction("Import MOL/SDF...", self); load_mol_action.triggered.connect(self.load_mol_file)
        file_menu.addAction(load_mol_action)
        import_smiles_action = QAction("Import SMILES...", self)
        import_smiles_action.triggered.connect(self.import_smiles_dialog)
        file_menu.addAction(import_smiles_action)
        import_inchi_action = QAction("Import InChI...", self)
        import_inchi_action.triggered.connect(self.import_inchi_dialog)
        file_menu.addAction(import_inchi_action)

        file_menu.addSeparator()
        load_3d_mol_action = QAction("Load 3D MOL/SDF (3D Only)...", self)
        load_3d_mol_action.triggered.connect(self.load_mol_file_for_3d_viewing)
        file_menu.addAction(load_3d_mol_action)
        
        load_3d_xyz_action = QAction("Load 3D XYZ (3D Only)...", self)
        load_3d_xyz_action.triggered.connect(self.load_xyz_for_3d_viewing)
        file_menu.addAction(load_3d_xyz_action)

        file_menu.addSeparator()
        save_mol_action = QAction("Save 2D as MOL...", self); save_mol_action.triggered.connect(self.save_as_mol)
        file_menu.addAction(save_mol_action)
        
        save_3d_mol_action = QAction("Save 3D as MOL...", self); save_3d_mol_action.triggered.connect(self.save_3d_as_mol)
        file_menu.addAction(save_3d_mol_action)
        
        save_xyz_action = QAction("Save 3D as XYZ...", self); save_xyz_action.triggered.connect(self.save_as_xyz)
        file_menu.addAction(save_xyz_action)
        file_menu.addSeparator()
        save_raw_action = QAction("Save Project...", self); save_raw_action.triggered.connect(self.save_raw_data)
        save_raw_action.setShortcut(QKeySequence.StandardKey.Save) 
        file_menu.addAction(save_raw_action)
        load_raw_action = QAction("Open Project...", self); load_raw_action.triggered.connect(self.load_raw_data)
        load_raw_action.setShortcut(QKeySequence.StandardKey.Open) 
        file_menu.addAction(load_raw_action)
        
        file_menu.addSeparator()
        
        export_2d_png_action = QAction("Export 2D as PNG...", self)
        export_2d_png_action.triggered.connect(self.export_2d_png)
        file_menu.addAction(export_2d_png_action)

        export_3d_png_action = QAction("Export 3D as PNG...", self)
        export_3d_png_action.triggered.connect(self.export_3d_png)
        file_menu.addAction(export_3d_png_action)
        
        file_menu.addSeparator()
        quit_action = QAction("Quit", self)
        quit_action.triggered.connect(self.close)
        file_menu.addAction(quit_action)
        
        edit_menu = menu_bar.addMenu("&Edit")
        self.undo_action = QAction("Undo", self); self.undo_action.setShortcut(QKeySequence.StandardKey.Undo)
        self.undo_action.triggered.connect(self.undo); edit_menu.addAction(self.undo_action)
        
        self.redo_action = QAction("Redo", self); self.redo_action.setShortcut(QKeySequence.StandardKey.Redo)
        self.redo_action.triggered.connect(self.redo); edit_menu.addAction(self.redo_action)
        
        edit_menu.addSeparator()

        self.cut_action = QAction("Cut", self)
        self.cut_action.setShortcut(QKeySequence.StandardKey.Cut)
        self.cut_action.triggered.connect(self.cut_selection)
        edit_menu.addAction(self.cut_action)

        self.copy_action = QAction("Copy", self)
        self.copy_action.setShortcut(QKeySequence.StandardKey.Copy)
        self.copy_action.triggered.connect(self.copy_selection)
        edit_menu.addAction(self.copy_action)
        
        self.paste_action = QAction("Paste", self)
        self.paste_action.setShortcut(QKeySequence.StandardKey.Paste)
        self.paste_action.triggered.connect(self.paste_from_clipboard)
        edit_menu.addAction(self.paste_action)

        edit_menu.addSeparator()

        optimize_2d_action = QAction("Optimize 2D", self)
        optimize_2d_action.setShortcut(QKeySequence("Ctrl+J"))
        optimize_2d_action.triggered.connect(self.clean_up_2d_structure)
        edit_menu.addAction(optimize_2d_action)
        
        convert_3d_action = QAction("Convert 2D to 3D", self)
        convert_3d_action.setShortcut(QKeySequence("Ctrl+K"))
        convert_3d_action.triggered.connect(self.trigger_conversion)
        edit_menu.addAction(convert_3d_action)

        optimize_3d_action = QAction("Optimize 3D", self)
        optimize_3d_action.setShortcut(QKeySequence("Ctrl+L")) 
        optimize_3d_action.triggered.connect(self.optimize_3d_structure)
        edit_menu.addAction(optimize_3d_action)

        edit_menu.addSeparator()
        
        select_all_action = QAction("Select All", self); select_all_action.setShortcut(QKeySequence.StandardKey.SelectAll)
        select_all_action.triggered.connect(self.select_all); edit_menu.addAction(select_all_action)
        
        clear_all_action = QAction("Clear All", self)
        clear_all_action.setShortcut(QKeySequence("Ctrl+Shift+C"))
        clear_all_action.triggered.connect(self.clear_all); edit_menu.addAction(clear_all_action)

        view_menu = menu_bar.addMenu("&View")

        zoom_in_action = QAction("Zoom In", self)
        zoom_in_action.setShortcut(QKeySequence.StandardKey.ZoomIn) # Ctrl +
        zoom_in_action.triggered.connect(self.zoom_in)
        view_menu.addAction(zoom_in_action)

        zoom_out_action = QAction("Zoom Out", self)
        zoom_out_action.setShortcut(QKeySequence.StandardKey.ZoomOut) # Ctrl -
        zoom_out_action.triggered.connect(self.zoom_out)
        view_menu.addAction(zoom_out_action)

        reset_zoom_action = QAction("Reset Zoom", self)
        reset_zoom_action.setShortcut(QKeySequence("Ctrl+0"))
        reset_zoom_action.triggered.connect(self.reset_zoom)
        view_menu.addAction(reset_zoom_action)
        
        fit_action = QAction("Fit to View", self)
        fit_action.setShortcut(QKeySequence("Ctrl+9"))
        fit_action.triggered.connect(self.fit_to_view)
        view_menu.addAction(fit_action)

        view_menu.addSeparator()

        reset_3d_view_action = QAction("Reset 3D View", self)
        reset_3d_view_action.triggered.connect(lambda: self.plotter.reset_camera() if hasattr(self, 'plotter') else None)
        reset_3d_view_action.setShortcut(QKeySequence("Ctrl+R"))
        view_menu.addAction(reset_3d_view_action)
        
        view_menu.addSeparator()

        self.toggle_chiral_action = QAction("Show Chiral Labels", self, checkable=True)
        self.toggle_chiral_action.setChecked(self.show_chiral_labels)
        self.toggle_chiral_action.triggered.connect(self.toggle_chiral_labels_display)
        view_menu.addAction(self.toggle_chiral_action)

        analysis_menu = menu_bar.addMenu("&Analysis")
        self.analysis_action = QAction("Show Analysis...", self)
        self.analysis_action.triggered.connect(self.open_analysis_window)
        self.analysis_action.setEnabled(False)
        analysis_menu.addAction(self.analysis_action)

        settings_menu = menu_bar.addMenu("&Settings")
        view_settings_action = QAction("3D View Settings...", self)
        view_settings_action.triggered.connect(self.open_settings_dialog)
        settings_menu.addAction(view_settings_action)

        help_menu = menu_bar.addMenu("&Help")
        about_action = QAction("About", self)
        about_action.triggered.connect(lambda: QMessageBox.about(
            self,
            "About MoleditPy",
            f"MoleditPy for Linux Ver. {VERSION}\nAuthor: Hiromichi Yokoyama\nLicense: Apache-2.0"
        ))
        help_menu.addAction(about_action)

        github_action = QAction("GitHub", self)
        github_action.triggered.connect(
            lambda: QDesktopServices.openUrl(QUrl("https://github.com/HiroYokoyama/python_molecular_editor"))
        )
        help_menu.addAction(github_action)
        
    def init_worker_thread(self):
        self.thread=QThread();self.worker=CalculationWorker();self.worker.moveToThread(self.thread)
        self.start_calculation.connect(self.worker.run_calculation)
        self.worker.finished.connect(self.on_calculation_finished); self.worker.error.connect(self.on_calculation_error)
        self.worker.status_update.connect(self.update_status_bar)
        self.thread.start()

    def update_status_bar(self, message):
        """ワーカースレッドからのメッセージでステータスバーを更新するスロット"""
        self.statusBar().showMessage(message)

    def set_mode(self, mode_str):
        self.scene.mode = mode_str
        self.view_2d.setMouseTracking(True) 
        if not mode_str.startswith('template'):
            self.scene.template_preview.hide()

        # カーソル形状の設定
        if mode_str == 'select':
            self.view_2d.setCursor(Qt.CursorShape.ArrowCursor)
        elif mode_str.startswith(('atom', 'bond', 'template')):
            self.view_2d.setCursor(Qt.CursorShape.CrossCursor)
        elif mode_str.startswith(('charge', 'radical')):
            self.view_2d.setCursor(Qt.CursorShape.CrossCursor)
        else:
            self.view_2d.setCursor(Qt.CursorShape.ArrowCursor)

        if mode_str.startswith('atom'): 
            self.scene.current_atom_symbol = mode_str.split('_')[1]
            self.statusBar().showMessage(f"Mode: Draw Atom ({self.scene.current_atom_symbol})")
            self.view_2d.setDragMode(QGraphicsView.DragMode.NoDrag)
            self.view_2d.setMouseTracking(True) 
            self.scene.bond_order = 1
            self.scene.bond_stereo = 0
        elif mode_str.startswith('bond'):
            self.scene.current_atom_symbol = 'C'
            parts = mode_str.split('_')
            self.scene.bond_order = int(parts[1])
            self.scene.bond_stereo = int(parts[2]) if len(parts) > 2 else 0
            stereo_text = {0: "", 1: " (Wedge)", 2: " (Dash)"}.get(self.scene.bond_stereo, "")
            self.statusBar().showMessage(f"Mode: Draw Bond (Order: {self.scene.bond_order}{stereo_text})")
            self.view_2d.setDragMode(QGraphicsView.DragMode.NoDrag)
            self.view_2d.setMouseTracking(True)
        elif mode_str.startswith('template'):
            self.statusBar().showMessage(f"Mode: {mode_str.split('_')[1].capitalize()} Template")
            self.view_2d.setDragMode(QGraphicsView.DragMode.NoDrag)
        elif mode_str == 'charge_plus':
            self.statusBar().showMessage("Mode: Increase Charge (Click on Atom)")
            self.view_2d.setDragMode(QGraphicsView.DragMode.NoDrag)
        elif mode_str == 'charge_minus':
            self.statusBar().showMessage("Mode: Decrease Charge (Click on Atom)")
            self.view_2d.setDragMode(QGraphicsView.DragMode.NoDrag)
        elif mode_str == 'radical':
            self.statusBar().showMessage("Mode: Toggle Radical (Click on Atom)")
            self.view_2d.setDragMode(QGraphicsView.DragMode.NoDrag)

        else: # Select mode
            self.statusBar().showMessage("Mode: Select")
            self.view_2d.setDragMode(QGraphicsView.DragMode.RubberBandDrag)
            self.scene.bond_order = 1
            self.scene.bond_stereo = 0

    def set_mode_and_update_toolbar(self, mode_str):
        self.set_mode(mode_str)
        if mode_str in self.mode_actions:
            self.mode_actions[mode_str].setChecked(True)

    def set_3d_style(self, style_name):
        """3D表示スタイルを設定し、ビューを更新する"""
        if self.current_3d_style == style_name:
            return

        self.current_3d_style = style_name
        self.statusBar().showMessage(f"3D style set to: {style_name}")
        
        # 現在表示中の分子があれば、新しいスタイルで再描画する
        if self.current_mol:
            self.draw_molecule_3d(self.current_mol)

    def copy_selection(self):
        """選択された原子と結合をクリップボードにコピーする"""
        try:
            selected_atoms = [item for item in self.scene.selectedItems() if isinstance(item, AtomItem)]
            if not selected_atoms:
                return

            # 選択された原子のIDセットを作成
            selected_atom_ids = {atom.atom_id for atom in selected_atoms}
            
            # 選択された原子の幾何学的中心を計算
            center = QPointF(
                sum(atom.pos().x() for atom in selected_atoms) / len(selected_atoms),
                sum(atom.pos().y() for atom in selected_atoms) / len(selected_atoms)
            )
            
            # コピー対象の原子データをリストに格納（位置は中心からの相対座標）
            # 同時に、元のatom_idから新しいインデックス(0, 1, 2...)へのマッピングを作成
            atom_id_to_idx_map = {}
            fragment_atoms = []
            for i, atom in enumerate(selected_atoms):
                atom_id_to_idx_map[atom.atom_id] = i
                fragment_atoms.append({
                    'symbol': atom.symbol,
                    'rel_pos': atom.pos() - center,
                    'charge': atom.charge,
                    'radical': atom.radical,
                })
                
            # 選択された原子同士を結ぶ結合のみをリストに格納
            fragment_bonds = []
            for (id1, id2), bond_data in self.data.bonds.items():
                if id1 in selected_atom_ids and id2 in selected_atom_ids:
                    fragment_bonds.append({
                        'idx1': atom_id_to_idx_map[id1],
                        'idx2': atom_id_to_idx_map[id2],
                        'order': bond_data['order'],
                        'stereo': bond_data.get('stereo', 0),  # E/Z立体化学情報も保存
                    })

            # pickleを使ってデータをバイト配列にシリアライズ
            data_to_pickle = {'atoms': fragment_atoms, 'bonds': fragment_bonds}
            byte_array = QByteArray()
            buffer = io.BytesIO()
            pickle.dump(data_to_pickle, buffer)
            byte_array.append(buffer.getvalue())

            # カスタムMIMEタイプでクリップボードに設定
            mime_data = QMimeData()
            mime_data.setData(CLIPBOARD_MIME_TYPE, byte_array)
            QApplication.clipboard().setMimeData(mime_data)
            self.statusBar().showMessage(f"Copied {len(fragment_atoms)} atoms and {len(fragment_bonds)} bonds.", 2000)
            
        except Exception as e:
            print(f"Error during copy operation: {e}")
            import traceback
            traceback.print_exc()
            self.statusBar().showMessage(f"Error during copy operation: {e}", 5000)

    def cut_selection(self):
        """選択されたアイテムを切り取り（コピーしてから削除）"""
        try:
            selected_items = self.scene.selectedItems()
            if not selected_items:
                return
            
            # 最初にコピー処理を実行
            self.copy_selection()
            
            if self.scene.delete_items(set(selected_items)):
                self.push_undo_state()
                self.statusBar().showMessage("Cut selection.", 2000)
                
        except Exception as e:
            print(f"Error during cut operation: {e}")
            import traceback
            traceback.print_exc()
            self.statusBar().showMessage(f"Error during cut operation: {e}", 5000)

    def paste_from_clipboard(self):
        """クリップボードから分子フラグメントを貼り付け"""
        try:
            clipboard = QApplication.clipboard()
            mime_data = clipboard.mimeData()
            if not mime_data.hasFormat(CLIPBOARD_MIME_TYPE):
                return

            byte_array = mime_data.data(CLIPBOARD_MIME_TYPE)
            buffer = io.BytesIO(byte_array)
            try:
                fragment_data = pickle.load(buffer)
            except pickle.UnpicklingError:
                self.statusBar().showMessage("Error: Invalid clipboard data format", 3000)
                return
            
            paste_center_pos = self.view_2d.mapToScene(self.view_2d.mapFromGlobal(QCursor.pos()))
            self.scene.clearSelection()

            new_atoms = []
            for atom_data in fragment_data['atoms']:
                pos = paste_center_pos + atom_data['rel_pos']
                new_id = self.scene.create_atom(
                    atom_data['symbol'], pos,
                    charge=atom_data.get('charge', 0),
                    radical=atom_data.get('radical', 0)
                )
                new_item = self.data.atoms[new_id]['item']
                new_atoms.append(new_item)
                new_item.setSelected(True)

            for bond_data in fragment_data['bonds']:
                atom1 = new_atoms[bond_data['idx1']]
                atom2 = new_atoms[bond_data['idx2']]
                self.scene.create_bond(
                    atom1, atom2,
                    bond_order=bond_data.get('order', 1),
                    bond_stereo=bond_data.get('stereo', 0)  # E/Z立体化学情報も復元
                )
            
            self.push_undo_state()
            self.statusBar().showMessage(f"Pasted {len(fragment_data['atoms'])} atoms and {len(fragment_data['bonds'])} bonds.", 2000)
            
        except Exception as e:
            print(f"Error during paste operation: {e}")
            import traceback
            traceback.print_exc()
            self.statusBar().showMessage(f"Error during paste operation: {e}", 5000)
        self.statusBar().showMessage(f"Pasted {len(new_atoms)} atoms.", 2000)
        self.activate_select_mode()

    def update_edit_menu_actions(self):
        """選択状態やクリップボードの状態に応じて編集メニューを更新"""
        try:
            has_selection = len(self.scene.selectedItems()) > 0
            self.cut_action.setEnabled(has_selection)
            self.copy_action.setEnabled(has_selection)
            
            clipboard = QApplication.clipboard()
            mime_data = clipboard.mimeData()
            self.paste_action.setEnabled(mime_data is not None and mime_data.hasFormat(CLIPBOARD_MIME_TYPE))
        except RuntimeError:
            pass


    def activate_select_mode(self):
        self.set_mode('select')
        if 'select' in self.mode_actions:
            self.mode_actions['select'].setChecked(True)

    def trigger_conversion(self):
        self.scene.clear_all_problem_flags()
        mol = self.data.to_rdkit_mol(use_2d_stereo=False)
        if not mol or mol.GetNumAtoms() == 0:
            # 3Dビューと関連データをクリア
            self.plotter.clear()
            self.current_mol = None
            self.analysis_action.setEnabled(False)
            self.statusBar().showMessage("3D view cleared.")
            self.view_2d.setFocus() 
            return

        problems = Chem.DetectChemistryProblems(mol)
        if problems:
            self.statusBar().showMessage(f"Error: {len(problems)} chemistry problem(s) found.")
            # 既存の選択状態をクリア
            self.scene.clearSelection() 
            
            # 問題のある原子に赤枠フラグを立てる
            for prob in problems:
                atom_idx = prob.GetAtomIdx()
                rdkit_atom = mol.GetAtomWithIdx(atom_idx)
                # エディタ側での原子IDの取得と存在確認
                if rdkit_atom.HasProp("_original_atom_id"):
                    original_id = rdkit_atom.GetIntProp("_original_atom_id")
                    if original_id in self.data.atoms and self.data.atoms[original_id]['item']:
                        item = self.data.atoms[original_id]['item']
                        item.has_problem = True 
                        item.update()
                
            self.view_2d.setFocus()
            return

        try:
            Chem.SanitizeMol(mol)
        except Exception:
            self.statusBar().showMessage("Error: Invalid chemical structure.")
            self.view_2d.setFocus() 
            return

        if len(Chem.GetMolFrags(mol)) > 1:
            self.statusBar().showMessage("Error: 3D conversion not supported for multiple molecules.")
            self.view_2d.setFocus() 
            return
            
        mol_block = Chem.MolToMolBlock(mol, includeStereo=True)
        
        # CRITICAL FIX: Manually add stereo information to MOL block for E/Z bonds
        # This ensures the stereo info survives the MOL block round-trip
        mol_lines = mol_block.split('\n')
        
        # Find bonds with explicit E/Z labels from our data and map to RDKit bond indices
        ez_bond_info = {}
        for (id1, id2), bond_data in self.data.bonds.items():
            if bond_data.get('stereo') in [3, 4]:  # E/Z labels
                # Find corresponding atoms in RDKit molecule by _original_atom_id property
                rdkit_idx1 = None
                rdkit_idx2 = None
                for atom in mol.GetAtoms():
                    if atom.HasProp("_original_atom_id"):
                        orig_id = atom.GetIntProp("_original_atom_id")
                        if orig_id == id1:
                            rdkit_idx1 = atom.GetIdx()
                        elif orig_id == id2:
                            rdkit_idx2 = atom.GetIdx()
                
                if rdkit_idx1 is not None and rdkit_idx2 is not None:
                    rdkit_bond = mol.GetBondBetweenAtoms(rdkit_idx1, rdkit_idx2)
                    if rdkit_bond and rdkit_bond.GetBondType() == Chem.BondType.DOUBLE:
                        ez_bond_info[rdkit_bond.GetIdx()] = bond_data['stereo']
        
        # Add M  CFG lines for E/Z stereo if needed
        if ez_bond_info:
            insert_idx = len(mol_lines) - 1  # Before M  END
            for bond_idx, stereo_type in ez_bond_info.items():
                cfg_value = 1 if stereo_type == 3 else 2  # 1=Z, 2=E in MOL format
                cfg_line = f"M  CFG  1 {bond_idx + 1:3d}   {cfg_value}"
                mol_lines.insert(insert_idx, cfg_line)
                insert_idx += 1
            mol_block = '\n'.join(mol_lines)
        
        self.convert_button.setEnabled(False)
        self.cleanup_button.setEnabled(False)
        self.optimize_3d_button.setEnabled(False)
        self.export_button.setEnabled(False)
        self.analysis_action.setEnabled(False)
        self.edit_3d_action.setEnabled(False)
        self.statusBar().showMessage("Calculating 3D structure...")
        self.plotter.clear() 
        bg_color_hex = self.settings.get('background_color', '#919191')
        bg_qcolor = QColor(bg_color_hex)
        
        if bg_qcolor.isValid():
            luminance = bg_qcolor.toHsl().lightness()
            text_color = 'black' if luminance > 128 else 'white'
        else:
            text_color = 'white'
        
        text_actor = self.plotter.add_text(
            "Calculating...",
            position='lower_right',
            font_size=15,
            color=text_color,
            name='calculating_text'
        )
        text_actor.GetTextProperty().SetOpacity(1)
        self.plotter.render()
        self.start_calculation.emit(mol_block)
        
        self.view_2d.setFocus()

    def optimize_3d_structure(self):
        """現在の3D分子構造を力場で最適化する"""
        if not self.current_mol:
            self.statusBar().showMessage("No 3D molecule to optimize.")
            return

        self.statusBar().showMessage("Optimizing 3D structure...")
        QApplication.processEvents() # UIの更新を確実に行う

        try:
            # MMFF力場での最適化を試みる
            AllChem.MMFFOptimizeMolecule(self.current_mol)
        except Exception:
            # MMFFが失敗した場合、UFF力場でフォールバック
            try:
                AllChem.UFFOptimizeMolecule(self.current_mol)
            except Exception as e:
                self.statusBar().showMessage(f"3D optimization failed: {e}")
                return
        
        # 最適化後の構造で3Dビューを再描画
        try:
            # キラル中心を3D座標から再計算（R/Sのみ）
            if self.current_mol.GetNumConformers() > 0:
                Chem.AssignAtomChiralTagsFromStructure(self.current_mol, confId=0)
            
            self.update_chiral_labels() # キラル中心のラベルも更新
        except Exception:
            pass
            
        self.draw_molecule_3d(self.current_mol)
        
        self.statusBar().showMessage("3D structure optimization successful.")
        self.push_undo_state() # Undo履歴に保存
        self.view_2d.setFocus()

    def on_calculation_finished(self, mol):
        self.dragged_atom_info = None
        self.current_mol = mol
        
        # キラル中心を3D座標から再計算（R/Sのみ）
        try:
            if mol.GetNumConformers() > 0:
                Chem.AssignAtomChiralTagsFromStructure(mol, confId=0)
            
            self.update_chiral_labels()
        except Exception:
            # 念のためエラーを握り潰して UI を壊さない
            pass

        self.draw_molecule_3d(mol)

        #self.statusBar().showMessage("3D conversion successful.")
        self.convert_button.setEnabled(True)
        self.analysis_action.setEnabled(True)
        self.measurement_action.setEnabled(True) 
        self.push_undo_state()
        self.view_2d.setFocus()
        self.cleanup_button.setEnabled(True)
        self.optimize_3d_button.setEnabled(True)
        self.export_button.setEnabled(True)
        self.edit_3d_action.setEnabled(True)
        self.plotter.reset_camera()
        
    def on_calculation_error(self, error_message):
        self.plotter.clear()
        self.dragged_atom_info = None
        self.statusBar().showMessage(f"Error: {error_message}")
        self.cleanup_button.setEnabled(True)
        self.convert_button.setEnabled(True)
        self.analysis_action.setEnabled(False)
        self.edit_3d_action.setEnabled(False) 
        self.view_2d.setFocus() 

    def eventFilter(self, obj, event):
        if obj is self.plotter and event.type() == QEvent.Type.MouseButtonPress:
            self.view_2d.setFocus()
        return super().eventFilter(obj, event)

    def get_current_state(self):
        atoms = {atom_id: {'symbol': data['symbol'],
                           'pos': (data['item'].pos().x(), data['item'].pos().y()),
                           'charge': data.get('charge', 0),
                           'radical': data.get('radical', 0)} 
                 for atom_id, data in self.data.atoms.items()}
        bonds = {key: {'order': data['order'], 'stereo': data.get('stereo', 0)} for key, data in self.data.bonds.items()}
        state = {'atoms': atoms, 'bonds': bonds, '_next_atom_id': self.data._next_atom_id}

        state['version'] = VERSION 
        
        if self.current_mol: state['mol_3d'] = self.current_mol.ToBinary()

        state['is_3d_viewer_mode'] = not self.is_2d_editable
            
        return state

    def set_state_from_data(self, state_data):
        self.dragged_atom_info = None
        self.clear_2d_editor(push_to_undo=False)
        
        loaded_data = copy.deepcopy(state_data)

        # ファイルのバージョンを取得（存在しない場合は '0.0.0' とする）
        file_version_str = loaded_data.get('version', '0.0.0')

        try:
            app_version_parts = tuple(map(int, VERSION.split('.')))
            file_version_parts = tuple(map(int, file_version_str.split('.')))

            # ファイルのバージョンがアプリケーションのバージョンより新しい場合に警告
            if file_version_parts > app_version_parts:
                QMessageBox.warning(
                    self,
                    "Version Mismatch",
                    f"The file you are opening was saved with a newer version of MoleditPy (ver. {file_version_str}).\n\n"
                    f"Your current version is {VERSION}.\n\n"
                    "Some features may not load or work correctly."
                )
        except (ValueError, AttributeError):
            pass

        raw_atoms = loaded_data.get('atoms', {})
        raw_bonds = loaded_data.get('bonds', {})

        for atom_id, data in raw_atoms.items():
            pos = QPointF(data['pos'][0], data['pos'][1])
            charge = data.get('charge', 0)
            radical = data.get('radical', 0)  # <-- ラジカル情報を取得
            # AtomItem生成時にradicalを渡す
            atom_item = AtomItem(atom_id, data['symbol'], pos, charge=charge, radical=radical)
            # self.data.atomsにもradical情報を格納する
            self.data.atoms[atom_id] = {'symbol': data['symbol'], 'pos': pos, 'item': atom_item, 'charge': charge, 'radical': radical}
            self.scene.addItem(atom_item)
        
        self.data._next_atom_id = loaded_data.get('_next_atom_id', max(self.data.atoms.keys()) + 1 if self.data.atoms else 0)

        for key_tuple, data in raw_bonds.items():
            id1, id2 = key_tuple
            if id1 in self.data.atoms and id2 in self.data.atoms:
                atom1_item = self.data.atoms[id1]['item']; atom2_item = self.data.atoms[id2]['item']
                bond_item = BondItem(atom1_item, atom2_item, data.get('order', 1), data.get('stereo', 0))
                self.data.bonds[key_tuple] = {'order': data.get('order', 1), 'stereo': data.get('stereo', 0), 'item': bond_item}
                atom1_item.bonds.append(bond_item); atom2_item.bonds.append(bond_item)
                self.scene.addItem(bond_item)

        for atom_data in self.data.atoms.values():
            if atom_data['item']: atom_data['item'].update_style()
        self.scene.update()

        if 'mol_3d' in loaded_data:
            try:
                self.current_mol = Chem.Mol(loaded_data['mol_3d'])
                self.draw_molecule_3d(self.current_mol)
                self.plotter.reset_camera()
                self.analysis_action.setEnabled(True)
                self.optimize_3d_button.setEnabled(True)
                self.export_button.setEnabled(True)
                self.edit_3d_action.setEnabled(True)
            except Exception as e:
                self.statusBar().showMessage(f"Could not load 3D model from project: {e}")
                self.current_mol = None; self.analysis_action.setEnabled(False)
        else:
            self.current_mol = None; self.plotter.clear(); self.analysis_action.setEnabled(False)
            self.optimize_3d_button.setEnabled(False)
            self.export_button.setEnabled(False) 
            self.edit_3d_action.setEnabled(False)

        self.update_implicit_hydrogens()
        self.update_chiral_labels()

        if loaded_data.get('is_3d_viewer_mode', False):
            self._enter_3d_viewer_ui_mode()
            self.statusBar().showMessage("Project loaded in 3D Viewer Mode.")
        else:
            self.restore_ui_for_editing()
        

    def push_undo_state(self):
        current_state_for_comparison = {
            'atoms': {k: (v['symbol'], v['item'].pos().x(), v['item'].pos().y(), v.get('charge', 0), v.get('radical', 0)) for k, v in self.data.atoms.items()},
            'bonds': {k: (v['order'], v.get('stereo', 0)) for k, v in self.data.bonds.items()},
            '_next_atom_id': self.data._next_atom_id,
            'mol_3d': self.current_mol.ToBinary() if self.current_mol else None
        }
        
        last_state_for_comparison = None
        if self.undo_stack:
            last_state = self.undo_stack[-1]
            last_atoms = last_state.get('atoms', {})
            last_bonds = last_state.get('bonds', {})
            last_state_for_comparison = {
                'atoms': {k: (v['symbol'], v['pos'][0], v['pos'][1], v.get('charge', 0), v.get('radical', 0)) for k, v in last_atoms.items()},
                'bonds': {k: (v['order'], v.get('stereo', 0)) for k, v in last_bonds.items()},
                '_next_atom_id': last_state.get('_next_atom_id'),
                'mol_3d': last_state.get('mol_3d', None)
            }

        if not last_state_for_comparison or current_state_for_comparison != last_state_for_comparison:
            state = self.get_current_state()
            self.undo_stack.append(state)
            self.redo_stack.clear()
        
        self.update_implicit_hydrogens()
        self.update_realtime_info()
        self.update_undo_redo_actions()

    def reset_undo_stack(self):
        self.undo_stack.clear()
        self.redo_stack.clear()
        self.push_undo_state()

    def undo(self):
        if len(self.undo_stack) > 1:
            self.redo_stack.append(self.undo_stack.pop())
            state = self.undo_stack[-1]
            self.set_state_from_data(state)
        self.update_undo_redo_actions()
        self.update_realtime_info()
        self.view_2d.setFocus() 

    def redo(self):
        if self.redo_stack:
            state = self.redo_stack.pop()
            self.undo_stack.append(state)
            self.set_state_from_data(state)
        self.update_undo_redo_actions()
        self.update_realtime_info()
        self.view_2d.setFocus() 
        
    def update_undo_redo_actions(self):
        self.undo_action.setEnabled(len(self.undo_stack) > 1)
        self.redo_action.setEnabled(len(self.redo_stack) > 0)

    def update_realtime_info(self):
        """ステータスバーの右側に現在の分子情報を表示する"""
        if not self.data.atoms:
            self.formula_label.setText("")  # 原子がなければ右側のラベルをクリア
            return

        try:
            mol = self.data.to_rdkit_mol()
            if mol:
                # 水素原子を明示的に追加した分子オブジェクトを生成
                mol_with_hs = Chem.AddHs(mol)
                mol_formula = rdMolDescriptors.CalcMolFormula(mol)
                # 水素を含む分子オブジェクトから原子数を取得
                num_atoms = mol_with_hs.GetNumAtoms()
                # 右側のラベルのテキストを更新
                self.formula_label.setText(f"Formula: {mol_formula}   |   Atoms: {num_atoms}")
        except Exception:
            # 計算に失敗してもアプリは継続
            self.formula_label.setText("Invalid structure")

    def select_all(self):
        for item in self.scene.items():
            if isinstance(item, (AtomItem, BondItem)):
                item.setSelected(True)

    def clear_all(self):

        self.restore_ui_for_editing()

        # データが存在しない場合は何もしない
        if not self.data.atoms and self.current_mol is None:
            return
        
        self.dragged_atom_info = None
            
        # 2Dエディタをクリアする（Undoスタックにはプッシュしない）
        self.clear_2d_editor(push_to_undo=False)
        
        # 3Dモデルをクリアする
        self.current_mol = None
        self.plotter.clear()
        
        # 解析メニューを無効化する
        self.analysis_action.setEnabled(False)
        self.measurement_action.setEnabled(False)  # 測定機能も無効化

        self.optimize_3d_button.setEnabled(False) 
        self.export_button.setEnabled(False)
        
        # Undo/Redoスタックをリセットする
        self.reset_undo_stack()
        
        # シーンとビューの明示的な更新
        self.scene.update()
        if self.view_2d:
            self.view_2d.viewport().update()

        self.optimize_3d_button.setEnabled(False) 
        self.export_button.setEnabled(False)
        self.edit_3d_action.setEnabled(False)
        
        # 3Dプロッターの再描画
        self.plotter.render()
        
        # アプリケーションのイベントループを強制的に処理し、画面の再描画を確実に行う
        QApplication.processEvents()
        
        self.statusBar().showMessage("Cleared all data.")
        
    def clear_2d_editor(self, push_to_undo=True):
        self.data = MolecularData()
        self.scene.data = self.data
        self.scene.clear()
        self.scene.reinitialize_items()
        if push_to_undo:
            self.push_undo_state()

    def update_implicit_hydrogens(self):
        """現在の2D構造に基づいて各原子の暗黙の水素数を計算し、AtomItemに反映する"""
        if not self.data.atoms:
            return

        try:
            mol = self.data.to_rdkit_mol()
            if mol is None:
                # 構造が不正な場合、全原子の水素カウントを0に戻して再描画
                for atom_data in self.data.atoms.values():
                    if atom_data.get('item') and atom_data['item'].implicit_h_count != 0:
                        atom_data['item'].implicit_h_count = 0
                        atom_data['item'].update()
                return
            
            items_to_update = []
            for atom in mol.GetAtoms():
                if atom.HasProp("_original_atom_id"):
                    original_id = atom.GetIntProp("_original_atom_id")
                    if original_id in self.data.atoms:
                        item = self.data.atoms[original_id].get('item')
                        if item:
                            h_count = atom.GetNumImplicitHs()
                            if item.implicit_h_count != h_count:
                                item.prepareGeometryChange()
                                item.implicit_h_count = h_count
                                items_to_update.append(item)
            
            # カウントが変更されたアイテムのみ再描画をトリガー
            for item in items_to_update:
                item.update()
        except Exception:
            # 編集中に一時的に発生するエラーなどで計算が失敗してもアプリは継続
            pass


    def import_smiles_dialog(self):
        """ユーザーにSMILES文字列の入力を促すダイアログを表示する"""
        smiles, ok = QInputDialog.getText(self, "Import SMILES", "Enter SMILES string:")
        if ok and smiles:
            self.load_from_smiles(smiles)

    def import_inchi_dialog(self):
        """ユーザーにInChI文字列の入力を促すダイアログを表示する"""
        inchi, ok = QInputDialog.getText(self, "Import InChI", "Enter InChI string:")
        if ok and inchi:
            self.load_from_inchi(inchi)

    def load_from_smiles(self, smiles_string):
        """SMILES文字列から分子を読み込み、2Dエディタに表示する"""
        try:
            cleaned_smiles = smiles_string.strip()
            
            mol = Chem.MolFromSmiles(cleaned_smiles)
            if mol is None:
                if not cleaned_smiles:
                    raise ValueError("SMILES string was empty.")
                raise ValueError("Invalid SMILES string.")

            AllChem.Compute2DCoords(mol)
            Chem.Kekulize(mol)

            AllChem.AssignStereochemistry(mol, cleanIt=True, force=True)
            conf = mol.GetConformer()
            AllChem.WedgeMolBonds(mol, conf)

            self.restore_ui_for_editing()
            self.clear_2d_editor(push_to_undo=False)
            self.current_mol = None
            self.plotter.clear()
            self.analysis_action.setEnabled(False)

            conf = mol.GetConformer()
            SCALE_FACTOR = 50.0
            
            view_center = self.view_2d.mapToScene(self.view_2d.viewport().rect().center())
            positions = [conf.GetAtomPosition(i) for i in range(mol.GetNumAtoms())]
            mol_center_x = sum(p.x for p in positions) / len(positions) if positions else 0.0
            mol_center_y = sum(p.y for p in positions) / len(positions) if positions else 0.0

            rdkit_idx_to_my_id = {}
            for i in range(mol.GetNumAtoms()):
                atom = mol.GetAtomWithIdx(i)
                pos = conf.GetAtomPosition(i)
                charge = atom.GetFormalCharge()
                
                relative_x = pos.x - mol_center_x
                relative_y = pos.y - mol_center_y
                
                scene_x = (relative_x * SCALE_FACTOR) + view_center.x()
                scene_y = (-relative_y * SCALE_FACTOR) + view_center.y()
                
                atom_id = self.scene.create_atom(atom.GetSymbol(), QPointF(scene_x, scene_y), charge=charge)
                rdkit_idx_to_my_id[i] = atom_id
            

            for bond in mol.GetBonds():
                b_idx, e_idx = bond.GetBeginAtomIdx(), bond.GetEndAtomIdx()
                b_type = bond.GetBondTypeAsDouble()
                b_dir = bond.GetBondDir()
                stereo = 0
                # 単結合の立体
                if b_dir == Chem.BondDir.BEGINWEDGE:
                    stereo = 1 # Wedge
                elif b_dir == Chem.BondDir.BEGINDASH:
                    stereo = 2 # Dash
                # 二重結合のE/Z
                if bond.GetBondType() == Chem.BondType.DOUBLE:
                    if bond.GetStereo() == Chem.BondStereo.STEREOZ:
                        stereo = 3 # Z
                    elif bond.GetStereo() == Chem.BondStereo.STEREOE:
                        stereo = 4 # E

                if b_idx in rdkit_idx_to_my_id and e_idx in rdkit_idx_to_my_id:
                    a1_id, a2_id = rdkit_idx_to_my_id[b_idx], rdkit_idx_to_my_id[e_idx]
                    a1_item = self.data.atoms[a1_id]['item']
                    a2_item = self.data.atoms[a2_id]['item']
                    self.scene.create_bond(a1_item, a2_item, bond_order=int(b_type), bond_stereo=stereo)

            self.statusBar().showMessage(f"Successfully loaded from SMILES.")
            self.reset_undo_stack()
            QTimer.singleShot(0, self.fit_to_view)
            
        except ValueError as e:
            self.statusBar().showMessage(f"Invalid SMILES: {e}")
        except Exception as e:
            self.statusBar().showMessage(f"Error loading from SMILES: {e}")
            import traceback
            traceback.print_exc()

    def load_from_inchi(self, inchi_string):
        """InChI文字列から分子を読み込み、2Dエディタに表示する"""
        try:
            cleaned_inchi = inchi_string.strip()
            
            mol = Chem.MolFromInchi(cleaned_inchi)
            if mol is None:
                if not cleaned_inchi:
                    raise ValueError("InChI string was empty.")
                raise ValueError("Invalid InChI string.")

            AllChem.Compute2DCoords(mol)
            Chem.Kekulize(mol)

            AllChem.AssignStereochemistry(mol, cleanIt=True, force=True)
            conf = mol.GetConformer()
            AllChem.WedgeMolBonds(mol, conf)

            self.restore_ui_for_editing()
            self.clear_2d_editor(push_to_undo=False)
            self.current_mol = None
            self.plotter.clear()
            self.analysis_action.setEnabled(False)

            conf = mol.GetConformer()
            SCALE_FACTOR = 50.0
            
            view_center = self.view_2d.mapToScene(self.view_2d.viewport().rect().center())
            positions = [conf.GetAtomPosition(i) for i in range(mol.GetNumAtoms())]
            mol_center_x = sum(p.x for p in positions) / len(positions) if positions else 0.0
            mol_center_y = sum(p.y for p in positions) / len(positions) if positions else 0.0

            rdkit_idx_to_my_id = {}
            for i in range(mol.GetNumAtoms()):
                atom = mol.GetAtomWithIdx(i)
                pos = conf.GetAtomPosition(i)
                charge = atom.GetFormalCharge()
                
                relative_x = pos.x - mol_center_x
                relative_y = pos.y - mol_center_y
                
                scene_x = (relative_x * SCALE_FACTOR) + view_center.x()
                scene_y = (-relative_y * SCALE_FACTOR) + view_center.y()
                
                atom_id = self.scene.create_atom(atom.GetSymbol(), QPointF(scene_x, scene_y), charge=charge)
                rdkit_idx_to_my_id[i] = atom_id
            
            for bond in mol.GetBonds():
                b_idx, e_idx = bond.GetBeginAtomIdx(), bond.GetEndAtomIdx()
                b_type = bond.GetBondTypeAsDouble()
                b_dir = bond.GetBondDir()
                stereo = 0
                # 単結合の立体
                if b_dir == Chem.BondDir.BEGINWEDGE:
                    stereo = 1 # Wedge
                elif b_dir == Chem.BondDir.BEGINDASH:
                    stereo = 2 # Dash
                # 二重結合のE/Z
                if bond.GetBondType() == Chem.BondType.DOUBLE:
                    if bond.GetStereo() == Chem.BondStereo.STEREOZ:
                        stereo = 3 # Z
                    elif bond.GetStereo() == Chem.BondStereo.STEREOE:
                        stereo = 4 # E

                if b_idx in rdkit_idx_to_my_id and e_idx in rdkit_idx_to_my_id:
                    a1_id, a2_id = rdkit_idx_to_my_id[b_idx], rdkit_idx_to_my_id[e_idx]
                    a1_item = self.data.atoms[a1_id]['item']
                    a2_item = self.data.atoms[a2_id]['item']
                    self.scene.create_bond(a1_item, a2_item, bond_order=int(b_type), bond_stereo=stereo)

            self.statusBar().showMessage(f"Successfully loaded from InChI.")
            self.reset_undo_stack()
            QTimer.singleShot(0, self.fit_to_view)
            
        except ValueError as e:
            self.statusBar().showMessage(f"Invalid InChI: {e}")
        except Exception as e:
            self.statusBar().showMessage(f"Error loading from InChI: {e}")
            import traceback
            traceback.print_exc()

    def load_mol_file(self, file_path=None):
        if not file_path:
            options = QFileDialog.Option.DontUseNativeDialog
            file_path, _ = QFileDialog.getOpenFileName(self, "Import MOL File", "", "Chemical Files (*.mol *.sdf);;All Files (*)", options=options)
            if not file_path: 
                return

        try:
            self.dragged_atom_info = None
            suppl = Chem.SDMolSupplier(file_path, removeHs=False)
            mol = next(suppl, None)
            if mol is None: raise ValueError("Failed to read molecule from file.")

            Chem.Kekulize(mol)

            self.restore_ui_for_editing()
            self.clear_2d_editor(push_to_undo=False)
            self.current_mol = None
            self.plotter.clear()
            self.analysis_action.setEnabled(False)
            
            # 1. 座標がなければ2D座標を生成する
            if mol.GetNumConformers() == 0: 
                AllChem.Compute2DCoords(mol)
            
            # 2. 座標の有無にかかわらず、常に立体化学を割り当て、2D表示用にくさび結合を設定する
            # これにより、3D座標を持つMOLファイルからでも正しく2Dの立体表現が生成される
            AllChem.AssignStereochemistry(mol, cleanIt=True, force=True)
            conf = mol.GetConformer()
            AllChem.WedgeMolBonds(mol, conf)

            conf = mol.GetConformer()

            SCALE_FACTOR = 50.0
            
            view_center = self.view_2d.mapToScene(self.view_2d.viewport().rect().center())

            positions = [conf.GetAtomPosition(i) for i in range(mol.GetNumAtoms())]
            if positions:
                mol_center_x = sum(p.x for p in positions) / len(positions)
                mol_center_y = sum(p.y for p in positions) / len(positions)
            else:
                mol_center_x, mol_center_y = 0.0, 0.0

            rdkit_idx_to_my_id = {}
            for i in range(mol.GetNumAtoms()):
                atom = mol.GetAtomWithIdx(i)
                pos = conf.GetAtomPosition(i)
                charge = atom.GetFormalCharge()
                
                relative_x = pos.x - mol_center_x
                relative_y = pos.y - mol_center_y
                
                scene_x = (relative_x * SCALE_FACTOR) + view_center.x()
                scene_y = (-relative_y * SCALE_FACTOR) + view_center.y()
                
                atom_id = self.scene.create_atom(atom.GetSymbol(), QPointF(scene_x, scene_y), charge=charge)
                rdkit_idx_to_my_id[i] = atom_id
                        
            for bond in mol.GetBonds():
                b_idx,e_idx=bond.GetBeginAtomIdx(),bond.GetEndAtomIdx()
                b_type = bond.GetBondTypeAsDouble(); b_dir = bond.GetBondDir()
                stereo = 0
                # Check for single bond Wedge/Dash
                if b_dir == Chem.BondDir.BEGINWEDGE:
                    stereo = 1
                elif b_dir == Chem.BondDir.BEGINDASH:
                    stereo = 2
                # ADDED: Check for double bond E/Z stereochemistry
                if bond.GetBondType() == Chem.BondType.DOUBLE:
                    if bond.GetStereo() == Chem.BondStereo.STEREOZ:
                        stereo = 3 # Z
                    elif bond.GetStereo() == Chem.BondStereo.STEREOE:
                        stereo = 4 # E

                a1_id, a2_id = rdkit_idx_to_my_id[b_idx], rdkit_idx_to_my_id[e_idx]
                a1_item,a2_item=self.data.atoms[a1_id]['item'],self.data.atoms[a2_id]['item']

                self.scene.create_bond(a1_item, a2_item, bond_order=int(b_type), bond_stereo=stereo)

            self.statusBar().showMessage(f"Successfully loaded {file_path}")
            self.reset_undo_stack()
            QTimer.singleShot(0, self.fit_to_view)
            
        except FileNotFoundError:
            self.statusBar().showMessage(f"File not found: {file_path}")
        except ValueError as e:
            self.statusBar().showMessage(f"Invalid MOL file format: {e}")
        except Exception as e: 
            self.statusBar().showMessage(f"Error loading file: {e}")
            import traceback
            traceback.print_exc()
    
    def load_mol_for_3d_viewing(self):
        options = QFileDialog.Option.DontUseNativeDialog
        file_path, _ = QFileDialog.getOpenFileName(self, "Load 3D MOL (View Only)", "", "Chemical Files (*.mol *.sdf);;All Files (*)", options=options)
        if not file_path:
            return

        try:
            suppl = Chem.SDMolSupplier(file_path, removeHs=False)
            mol = next(suppl, None)
            if mol is None:
                raise ValueError("Failed to read molecule.")
            if mol.GetNumConformers() == 0:
                raise ValueError("MOL file has no 3D coordinates.")

            # 2Dエディタをクリア
            self.clear_2d_editor(push_to_undo=False)
            
            # 3D構造をセットして描画
            self.current_mol = mol
            self.draw_molecule_3d(self.current_mol)
            self.plotter.reset_camera()

            # UIを3Dビューアモードに設定
            self._enter_3d_viewer_ui_mode()
            
            # 測定機能を有効化
            self.measurement_action.setEnabled(True)
            
            self.statusBar().showMessage(f"3D Viewer Mode: Loaded {os.path.basename(file_path)}")
            self.reset_undo_stack()

        except FileNotFoundError:
            self.statusBar().showMessage(f"File not found: {file_path}", 5000)
            self.restore_ui_for_editing()
        except ValueError as e:
            self.statusBar().showMessage(f"Invalid 3D MOL file: {e}", 5000)
            self.restore_ui_for_editing()
        except Exception as e:
            self.statusBar().showMessage(f"Error loading 3D file: {e}", 5000)
            self.restore_ui_for_editing()
            import traceback
            traceback.print_exc()

    def load_xyz_for_3d_viewing(self, file_path=None):
        """XYZファイルを読み込んで3Dビューアで表示する"""
        if not file_path:
            options = QFileDialog.Option.DontUseNativeDialog
            file_path, _ = QFileDialog.getOpenFileName(self, "Load 3D XYZ (View Only)", "", "XYZ Files (*.xyz);;All Files (*)", options=options)
            if not file_path:
                return

        try:
            mol = self.load_xyz_file(file_path)
            if mol is None:
                raise ValueError("Failed to create molecule from XYZ file.")
            if mol.GetNumConformers() == 0:
                raise ValueError("XYZ file has no 3D coordinates.")

            # 2Dエディタをクリア
            self.clear_2d_editor(push_to_undo=False)
            
            # 3D構造をセットして描画
            self.current_mol = mol
            self.draw_molecule_3d(self.current_mol)
            self.plotter.reset_camera()

            # UIを3Dビューアモードに設定
            self._enter_3d_viewer_ui_mode()
            
            # 測定機能を有効化
            self.measurement_action.setEnabled(True)
            
            self.statusBar().showMessage(f"3D Viewer Mode: Loaded {os.path.basename(file_path)}")
            self.reset_undo_stack()

        except FileNotFoundError:
            self.statusBar().showMessage(f"File not found: {file_path}", 5000)
            self.restore_ui_for_editing()
        except ValueError as e:
            self.statusBar().showMessage(f"Invalid XYZ file: {e}", 5000)
            self.restore_ui_for_editing()
        except Exception as e:
            self.statusBar().showMessage(f"Error loading XYZ file: {e}", 5000)
            self.restore_ui_for_editing()
            import traceback
            traceback.print_exc()

    def load_xyz_file(self, file_path):
        """XYZファイルを読み込んでRDKitのMolオブジェクトを作成する"""
        import math
        from rdkit.Chem import rdGeometry
        
        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                lines = f.readlines()
            
            # 空行とコメント行を除去（但し、先頭2行は保持）
            non_empty_lines = []
            for i, line in enumerate(lines):
                stripped = line.strip()
                if i < 2:  # 最初の2行は原子数とコメント行なので保持
                    non_empty_lines.append(stripped)
                elif stripped and not stripped.startswith('#'):  # 空行とコメント行をスキップ
                    non_empty_lines.append(stripped)
            
            if len(non_empty_lines) < 2:
                raise ValueError("XYZ file format error: too few lines")
            
            # 原子数を読み取り
            try:
                num_atoms = int(non_empty_lines[0])
            except ValueError:
                raise ValueError("XYZ file format error: invalid atom count")
            
            if num_atoms <= 0:
                raise ValueError("XYZ file format error: atom count must be positive")
            
            # コメント行（2行目）
            comment = non_empty_lines[1] if len(non_empty_lines) > 1 else ""
            
            # 原子データを読み取り
            atoms_data = []
            data_lines = non_empty_lines[2:]
            
            if len(data_lines) < num_atoms:
                raise ValueError(f"XYZ file format error: expected {num_atoms} atom lines, found {len(data_lines)}")
            
            for i, line in enumerate(data_lines[:num_atoms]):
                parts = line.split()
                if len(parts) < 4:
                    raise ValueError(f"XYZ file format error: invalid atom data at line {i+3}")
                
                symbol = parts[0].strip()
                
                # 元素記号の妥当性をチェック
                try:
                    # RDKitで認識される元素かどうかをチェック
                    test_atom = Chem.Atom(symbol)
                except:
                    # 認識されない場合、最初の文字を大文字にして再試行
                    symbol = symbol.capitalize()
                    try:
                        test_atom = Chem.Atom(symbol)
                    except:
                        raise ValueError(f"Unrecognized element symbol: {parts[0]} at line {i+3}")
                
                try:
                    x, y, z = float(parts[1]), float(parts[2]), float(parts[3])
                except ValueError:
                    raise ValueError(f"XYZ file format error: invalid coordinates at line {i+3}")
                
                atoms_data.append((symbol, x, y, z))
            
            if len(atoms_data) == 0:
                raise ValueError("XYZ file format error: no atoms found")
            
            # RDKitのMolオブジェクトを作成
            mol = Chem.RWMol()
            
            # 原子を追加
            for symbol, x, y, z in atoms_data:
                atom = Chem.Atom(symbol)
                mol.AddAtom(atom)
            
            # 3D座標を設定
            conf = Chem.Conformer(len(atoms_data))
            for i, (symbol, x, y, z) in enumerate(atoms_data):
                conf.SetAtomPosition(i, rdGeometry.Point3D(x, y, z))
            mol.AddConformer(conf)
            
            # 結合を推定（距離ベース）
            self.estimate_bonds_from_distances(mol)
            
            # 分子を最終化
            try:
                mol = mol.GetMol()
                # 基本的な妥当性チェック
                if mol is None:
                    raise ValueError("Failed to create valid molecule object")
                Chem.SanitizeMol(mol)
            except Exception as e:
                # 化学的に不正な構造でも表示は可能にする
                mol = mol.GetMol()
                if mol is None:
                    raise ValueError("Failed to create molecule object")
            
            return mol
            
        except (OSError, IOError) as e:
            raise ValueError(f"File I/O error: {e}")
        except Exception as e:
            if "XYZ file format error" in str(e) or "Unrecognized element" in str(e):
                raise e
            else:
                raise ValueError(f"Error parsing XYZ file: {e}")

    def estimate_bonds_from_distances(self, mol):
        """原子間距離に基づいて結合を推定する"""
        from rdkit.Chem import rdMolTransforms
        
        # 一般的な共有結合半径（Ångström）- より正確な値
        covalent_radii = {
            'H': 0.31, 'He': 0.28, 'Li': 1.28, 'Be': 0.96, 'B': 0.84, 'C': 0.76,
            'N': 0.71, 'O': 0.66, 'F': 0.57, 'Ne': 0.58, 'Na': 1.66, 'Mg': 1.41,
            'Al': 1.21, 'Si': 1.11, 'P': 1.07, 'S': 1.05, 'Cl': 1.02, 'Ar': 1.06,
            'K': 2.03, 'Ca': 1.76, 'Sc': 1.70, 'Ti': 1.60, 'V': 1.53, 'Cr': 1.39,
            'Mn': 1.39, 'Fe': 1.32, 'Co': 1.26, 'Ni': 1.24, 'Cu': 1.32, 'Zn': 1.22,
            'Ga': 1.22, 'Ge': 1.20, 'As': 1.19, 'Se': 1.20, 'Br': 1.20, 'Kr': 1.16,
            'Rb': 2.20, 'Sr': 1.95, 'Y': 1.90, 'Zr': 1.75, 'Nb': 1.64, 'Mo': 1.54,
            'Tc': 1.47, 'Ru': 1.46, 'Rh': 1.42, 'Pd': 1.39, 'Ag': 1.45, 'Cd': 1.44,
            'In': 1.42, 'Sn': 1.39, 'Sb': 1.39, 'Te': 1.38, 'I': 1.39, 'Xe': 1.40
        }
        
        conf = mol.GetConformer()
        num_atoms = mol.GetNumAtoms()
        
        # 追加された結合をトラッキング
        bonds_added = []
        
        for i in range(num_atoms):
            for j in range(i + 1, num_atoms):
                atom_i = mol.GetAtomWithIdx(i)
                atom_j = mol.GetAtomWithIdx(j)
                
                # 原子間距離を計算
                distance = rdMolTransforms.GetBondLength(conf, i, j)
                
                # 期待される結合距離を計算
                symbol_i = atom_i.GetSymbol()
                symbol_j = atom_j.GetSymbol()
                
                radius_i = covalent_radii.get(symbol_i, 1.0)  # デフォルト半径
                radius_j = covalent_radii.get(symbol_j, 1.0)
                
                expected_bond_length = radius_i + radius_j
                
                # 結合タイプによる許容範囲を調整
                # 水素結合は通常の共有結合より短い
                if symbol_i == 'H' or symbol_j == 'H':
                    tolerance_factor = 1.2  # 水素は結合が短くなりがち
                else:
                    tolerance_factor = 1.3  # 他の原子は少し余裕を持たせる
                
                max_bond_length = expected_bond_length * tolerance_factor
                min_bond_length = expected_bond_length * 0.5  # 最小距離も設定
                
                # 距離が期待値の範囲内なら結合を追加
                if min_bond_length <= distance <= max_bond_length:
                    try:
                        mol.AddBond(i, j, Chem.BondType.SINGLE)
                        bonds_added.append((i, j, distance))
                    except:
                        # 既に結合が存在する場合はスキップ
                        pass
        
        # デバッグ情報（オプション）
        # print(f"Added {len(bonds_added)} bonds based on distance analysis")
        
        return len(bonds_added)


    def save_raw_data(self):
        if not self.data.atoms and not self.current_mol: 
            self.statusBar().showMessage("Error: Nothing to save.")
            return
            
        try:
            save_data = self.get_current_state()
            options = QFileDialog.Option.DontUseNativeDialog
            file_path, _ = QFileDialog.getSaveFileName(self, "Save Project File", "", "Project Files (*.pmeraw);;All Files (*)", options=options)
            if not file_path:
                return
                
            if not file_path.lower().endswith('.pmeraw'): 
                file_path += '.pmeraw'
                
            with open(file_path, 'wb') as f: 
                pickle.dump(save_data, f)
            self.statusBar().showMessage(f"Project saved to {file_path}")
            
        except (OSError, IOError) as e:
            self.statusBar().showMessage(f"File I/O error: {e}")
        except pickle.PicklingError as e:
            self.statusBar().showMessage(f"Data serialization error: {e}")
        except Exception as e: 
            self.statusBar().showMessage(f"Error saving project file: {e}")
            import traceback
            traceback.print_exc()


    def load_raw_data(self, file_path=None):
        if not file_path:
            options = QFileDialog.Option.DontUseNativeDialog
            file_path, _ = QFileDialog.getOpenFileName(self, "Open Project File", "", "Project Files (*.pmeraw);;All Files (*)", options=options)
            if not file_path: 
                return
        
        try:
            with open(file_path, 'rb') as f: 
                loaded_data = pickle.load(f)
            self.restore_ui_for_editing()
            self.set_state_from_data(loaded_data)
            self.statusBar().showMessage(f"Project loaded from {file_path}")
            self.reset_undo_stack()
            QTimer.singleShot(0, self.fit_to_view)
            
        except FileNotFoundError:
            self.statusBar().showMessage(f"File not found: {file_path}")
        except (OSError, IOError) as e:
            self.statusBar().showMessage(f"File I/O error: {e}")
        except pickle.UnpicklingError as e:
            self.statusBar().showMessage(f"Invalid project file format: {e}")
        except Exception as e: 
            self.statusBar().showMessage(f"Error loading project file: {e}")
            import traceback
            traceback.print_exc()

    def save_as_mol(self):
        try:
            mol_block = self.data.to_mol_block()
            if not mol_block: 
                self.statusBar().showMessage("Error: No 2D data to save."); 
                return
                
            lines = mol_block.split('\n')
            if len(lines) > 1 and 'RDKit' in lines[1]:
                lines[1] = '  MoleditPy Ver. ' + VERSION + '  2D'
            modified_mol_block = '\n'.join(lines)
            
            options = QFileDialog.Option.DontUseNativeDialog
            file_path, _ = QFileDialog.getSaveFileName(self, "Save 2D MOL File", "", "MOL Files (*.mol);;All Files (*)", options=options)
            if not file_path:
                return
                
            if not file_path.lower().endswith('.mol'): 
                file_path += '.mol'
                
            with open(file_path, 'w', encoding='utf-8') as f: 
                f.write(modified_mol_block)
            self.statusBar().showMessage(f"2D data saved to {file_path}")
            
        except (OSError, IOError) as e:
            self.statusBar().showMessage(f"File I/O error: {e}")
        except UnicodeEncodeError as e:
            self.statusBar().showMessage(f"Text encoding error: {e}")
        except Exception as e: 
            self.statusBar().showMessage(f"Error saving file: {e}")
            import traceback
            traceback.print_exc()
            
    def save_3d_as_mol(self):
        if not self.current_mol:
            self.statusBar().showMessage("Error: Please generate a 3D structure first.")
            return
            
        try:
            options = QFileDialog.Option.DontUseNativeDialog
            file_path, _ = QFileDialog.getSaveFileName(self, "Save 3D MOL File", "", "MOL Files (*.mol);;All Files (*)", options=options)
            if not file_path:
                return
                
            if not file_path.lower().endswith('.mol'):
                file_path += '.mol'

            mol_to_save = Chem.Mol(self.current_mol)

            if mol_to_save.HasProp("_2D"):
                mol_to_save.ClearProp("_2D")

            mol_block = Chem.MolToMolBlock(mol_to_save, includeStereo=True)
            lines = mol_block.split('\n')
            if len(lines) > 1 and 'RDKit' in lines[1]:
                lines[1] = '  MoleditPy Ver. ' + VERSION + '  3D'
            modified_mol_block = '\n'.join(lines)
            
            with open(file_path, 'w', encoding='utf-8') as f:
                f.write(modified_mol_block)
            self.statusBar().showMessage(f"3D data saved to {file_path}")
            
        except (OSError, IOError) as e:
            self.statusBar().showMessage(f"File I/O error: {e}")
        except UnicodeEncodeError as e:
            self.statusBar().showMessage(f"Text encoding error: {e}")
        except Exception as e: 
            self.statusBar().showMessage(f"Error saving 3D MOL file: {e}")
            import traceback
            traceback.print_exc()

    def save_as_xyz(self):
        if not self.current_mol: self.statusBar().showMessage("Error: Please generate a 3D structure first."); return
        options=QFileDialog.Option.DontUseNativeDialog
        file_path,_=QFileDialog.getSaveFileName(self,"Save 3D XYZ File","","XYZ Files (*.xyz);;All Files (*)",options=options)
        if file_path:
            if not file_path.lower().endswith('.xyz'): file_path += '.xyz'
            try:
                conf=self.current_mol.GetConformer(); num_atoms=self.current_mol.GetNumAtoms()
                xyz_lines=[str(num_atoms)]; smiles=Chem.MolToSmiles(Chem.RemoveHs(self.current_mol))
                xyz_lines.append(f"Generated by MoleditPy Ver. {VERSION}")
                for i in range(num_atoms):
                    pos=conf.GetAtomPosition(i); symbol=self.current_mol.GetAtomWithIdx(i).GetSymbol()
                    xyz_lines.append(f"{symbol} {pos.x:.6f} {pos.y:.6f} {pos.z:.6f}")
                with open(file_path,'w') as f: f.write("\n".join(xyz_lines) + "\n")
                self.statusBar().showMessage(f"Successfully saved to {file_path}")
            except Exception as e: self.statusBar().showMessage(f"Error saving file: {e}")

    def export_2d_png(self):
        if not self.data.atoms:
            self.statusBar().showMessage("Nothing to export.", 2000)
            return

        options = QFileDialog.Option.DontUseNativeDialog
        filePath, _ = QFileDialog.getSaveFileName(self, "Export 2D as PNG", "", "PNG Files (*.png)", options=options)
        if not filePath:
            return

        if not (filePath.lower().endswith(".png")):
            filePath += ".png"

        reply = QMessageBox.question(self, 'Choose Background',
                                     'Do you want a transparent background?\n(Choose "No" for a white background)',
                                     QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No | QMessageBox.StandardButton.Cancel,
                                     QMessageBox.StandardButton.No)

        if reply == QMessageBox.StandardButton.Cancel:
            self.statusBar().showMessage("Export cancelled.", 2000)
            return

        is_transparent = (reply == QMessageBox.StandardButton.Yes)

        QApplication.processEvents()

        items_to_restore = {}
        original_background = self.scene.backgroundBrush()

        try:
            all_items = list(self.scene.items())
            for item in all_items:
                is_mol_part = isinstance(item, (AtomItem, BondItem))
                if not (is_mol_part and item.isVisible()):
                    items_to_restore[item] = item.isVisible()
                    item.hide()

            molecule_bounds = QRectF()
            for item in self.scene.items():
                if isinstance(item, (AtomItem, BondItem)) and item.isVisible():
                    molecule_bounds = molecule_bounds.united(item.sceneBoundingRect())

            if molecule_bounds.isEmpty() or not molecule_bounds.isValid():
                self.statusBar().showMessage("Error: Could not determine molecule bounds for export.", 5000)
                return

            if is_transparent:
                self.scene.setBackgroundBrush(QBrush(Qt.BrushStyle.NoBrush))
            else:
                self.scene.setBackgroundBrush(QBrush(QColor("#FFFFFF")))

            rect_to_render = molecule_bounds.adjusted(-20, -20, 20, 20)

            w = max(1, int(math.ceil(rect_to_render.width())))
            h = max(1, int(math.ceil(rect_to_render.height())))

            if w <= 0 or h <= 0:
                self.statusBar().showMessage("Error: Invalid image size calculated.", 5000)
                return

            image = QImage(w, h, QImage.Format.Format_ARGB32_Premultiplied)
            if is_transparent:
                image.fill(Qt.GlobalColor.transparent)
            else:
                image.fill(Qt.GlobalColor.white)

            painter = QPainter()
            ok = painter.begin(image)
            if not ok or not painter.isActive():
                self.statusBar().showMessage("Failed to start QPainter for image rendering.", 5000)
                return

            try:
                painter.setRenderHint(QPainter.RenderHint.Antialiasing)
                target_rect = QRectF(0, 0, w, h)
                source_rect = rect_to_render
                self.scene.render(painter, target_rect, source_rect)
            finally:
                painter.end()

            saved = image.save(filePath, "PNG")
            if saved:
                self.statusBar().showMessage(f"2D view exported to {filePath}", 3000)
            else:
                self.statusBar().showMessage(f"Failed to save image. Check file path or permissions.", 5000)

        except Exception as e:
            self.statusBar().showMessage(f"An unexpected error occurred during 2D export: {e}", 5000)

        finally:
            for item, was_visible in items_to_restore.items():
                item.setVisible(was_visible)
            self.scene.setBackgroundBrush(original_background)
            if self.view_2d:
                self.view_2d.viewport().update()

    def export_3d_png(self):
        if not self.current_mol:
            self.statusBar().showMessage("No 3D molecule to export.", 2000)
            return

        options = QFileDialog.Option.DontUseNativeDialog
        filePath, _ = QFileDialog.getSaveFileName(self, "Export 3D as PNG", "", "PNG Files (*.png)", options=options)
        if not filePath:
            return

        if not (filePath.lower().endswith(".png")):
            filePath += ".png"

        reply = QMessageBox.question(self, 'Choose Background',
                                     'Do you want a transparent background?\n(Choose "No" for current background)',
                                     QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No | QMessageBox.StandardButton.Cancel,
                                     QMessageBox.StandardButton.No)

        if reply == QMessageBox.StandardButton.Cancel:
            self.statusBar().showMessage("Export cancelled.", 2000)
            return

        is_transparent = (reply == QMessageBox.StandardButton.Yes)

        try:
            self.plotter.screenshot(filePath, transparent_background=is_transparent)
            self.statusBar().showMessage(f"3D view exported to {filePath}", 3000)
        except Exception as e:
            self.statusBar().showMessage(f"Error exporting 3D PNG: {e}", 3000)


    def open_periodic_table_dialog(self):
        dialog=PeriodicTableDialog(self); dialog.element_selected.connect(self.set_atom_from_periodic_table)
        checked_action=self.tool_group.checkedAction()
        if checked_action: self.tool_group.setExclusive(False); checked_action.setChecked(False); self.tool_group.setExclusive(True)
        dialog.exec()

    def set_atom_from_periodic_table(self, symbol): 
        self.set_mode(f'atom_{symbol}')

   
    def clean_up_2d_structure(self):
        self.statusBar().showMessage("Optimizing 2D structure...")
        mol = self.data.to_rdkit_mol()
        if mol is None or mol.GetNumAtoms() == 0:
            self.statusBar().showMessage("Error: No atoms to optimize."); return

        try:
            # 安定版：原子IDとRDKit座標の確実なマッピング
            view_center = self.view_2d.mapToScene(self.view_2d.viewport().rect().center())
            new_positions_map = {}
            AllChem.Compute2DCoords(mol)
            conf = mol.GetConformer()
            for rdkit_atom in mol.GetAtoms():
                original_id = rdkit_atom.GetIntProp("_original_atom_id")
                new_positions_map[original_id] = conf.GetAtomPosition(rdkit_atom.GetIdx())

            if not new_positions_map:
                self.statusBar().showMessage("Optimization failed to generate coordinates."); return

            target_atom_items = [self.data.atoms[atom_id]['item'] for atom_id in new_positions_map.keys() if atom_id in self.data.atoms and 'item' in self.data.atoms[atom_id]]
            if not target_atom_items:
                self.statusBar().showMessage("Error: Atom items not found for optimized atoms."); return

            # 元の図形の中心を維持
            #original_center_x = sum(item.pos().x() for item in target_atom_items) / len(target_atom_items)
            #original_center_y = sum(item.pos().y() for item in target_atom_items) / len(target_atom_items)

            positions = list(new_positions_map.values())
            rdkit_cx = sum(p.x for p in positions) / len(positions)
            rdkit_cy = sum(p.y for p in positions) / len(positions)

            SCALE = 50.0

            # 新しい座標を適用
            for atom_id, rdkit_pos in new_positions_map.items():
                if atom_id in self.data.atoms:
                    item = self.data.atoms[atom_id]['item']
                    sx = ((rdkit_pos.x - rdkit_cx) * SCALE) + view_center.x()
                    sy = (-(rdkit_pos.y - rdkit_cy) * SCALE) + view_center.y()
                    new_scene_pos = QPointF(sx, sy)
                    item.setPos(new_scene_pos)
                    self.data.atoms[atom_id]['pos'] = new_scene_pos

            # 最終的な座標に基づき、全ての結合表示を一度に更新
            for bond_data in self.data.bonds.values():
                if bond_data.get('item'):
                    bond_data['item'].update_position()

            # 重なり解消ロジックを実行
            self. resolve_overlapping_groups()
            
            # シーン全体の再描画を要求
            self.scene.update()

            self.statusBar().showMessage("2D structure optimization successful.")
            self.push_undo_state()

        except Exception as e:
            self.statusBar().showMessage(f"Error during 2D optimization: {e}")
        finally:
            self.view_2d.setFocus()

    def resolve_overlapping_groups(self):
        """
        誤差範囲で完全に重なっている原子のグループを検出し、
        IDが大きい方のフラグメントを左下に平行移動して解消する。
        """

        # --- パラメータ設定 ---
        # 重なっているとみなす距離の閾値。構造に合わせて調整してください。
        OVERLAP_THRESHOLD = 0.5  
        # 左下へ移動させる距離。
        MOVE_DISTANCE = 20

        # self.data.atoms.values() から item を安全に取得
        all_atom_items = [
            data['item'] for data in self.data.atoms.values() 
            if data and 'item' in data
        ]

        if len(all_atom_items) < 2:
            return

        # --- ステップ1: 重なっている原子ペアを全てリストアップ ---
        overlapping_pairs = []
        for item1, item2 in itertools.combinations(all_atom_items, 2):
            # 結合で直接結ばれているペアは重なりと見なさない
            if self.scene.find_bond_between(item1, item2):
                continue

            dist = QLineF(item1.pos(), item2.pos()).length()
            if dist < OVERLAP_THRESHOLD:
                overlapping_pairs.append((item1, item2))

        if not overlapping_pairs:
            self.statusBar().showMessage("No overlapping atoms found.", 2000)
            return

        # --- ステップ2: Union-Findアルゴリズムで重なりグループを構築 ---
        # 各原子がどのグループに属するかを管理する
        parent = {item.atom_id: item.atom_id for item in all_atom_items}

        def find_set(atom_id):
            # atom_idが属するグループの代表（ルート）を見つける
            if parent[atom_id] == atom_id:
                return atom_id
            parent[atom_id] = find_set(parent[atom_id])  # 経路圧縮による最適化
            return parent[atom_id]

        def unite_sets(id1, id2):
            # 2つの原子が属するグループを統合する
            root1 = find_set(id1)
            root2 = find_set(id2)
            if root1 != root2:
                parent[root2] = root1

        for item1, item2 in overlapping_pairs:
            unite_sets(item1.atom_id, item2.atom_id)

        # --- ステップ3: グループごとに移動計画を立てる ---
        # 同じ代表を持つ原子でグループを辞書にまとめる
        groups_by_root = {}
        for item in all_atom_items:
            root_id = find_set(item.atom_id)
            if root_id not in groups_by_root:
                groups_by_root[root_id] = []
            groups_by_root[root_id].append(item.atom_id)

        move_operations = []
        processed_roots = set()

        for root_id, group_atom_ids in groups_by_root.items():
            # 処理済みのグループや、メンバーが1つしかないグループはスキップ
            if root_id in processed_roots or len(group_atom_ids) < 2:
                continue
            processed_roots.add(root_id)

            # 3a: グループを、結合に基づいたフラグメントに分割する (BFSを使用)
            fragments = []
            visited_in_group = set()
            group_atom_ids_set = set(group_atom_ids)

            for atom_id in group_atom_ids:
                if atom_id not in visited_in_group:
                    current_fragment = set()
                    q = deque([atom_id])
                    visited_in_group.add(atom_id)
                    current_fragment.add(atom_id)

                    while q:
                        current_id = q.popleft()
                        # 隣接リスト self.adjacency_list があれば、ここでの探索が高速になります
                        for neighbor_id in self.data.adjacency_list.get(current_id, []):
                            if neighbor_id in group_atom_ids_set and neighbor_id not in visited_in_group:
                                visited_in_group.add(neighbor_id)
                                current_fragment.add(neighbor_id)
                                q.append(neighbor_id)
                    fragments.append(current_fragment)

            if len(fragments) < 2:
                continue  # 複数のフラグメントが重なっていない場合

            # 3b: 移動するフラグメントを決定する
            # このグループの重なりの原因となった代表ペアを一つ探す
            rep_item1, rep_item2 = None, None
            for i1, i2 in overlapping_pairs:
                if find_set(i1.atom_id) == root_id:
                    rep_item1, rep_item2 = i1, i2
                    break

            if not rep_item1: continue

            # 代表ペアがそれぞれどのフラグメントに属するかを見つける
            frag1 = next((f for f in fragments if rep_item1.atom_id in f), None)
            frag2 = next((f for f in fragments if rep_item2.atom_id in f), None)

            # 同一フラグメント内の重なりなどはスキップ
            if not frag1 or not frag2 or frag1 == frag2:
                continue

            # 仕様: IDが大きい方の原子が含まれるフラグメントを動かす
            if rep_item1.atom_id > rep_item2.atom_id:
                ids_to_move = frag1
            else:
                ids_to_move = frag2

            # 3c: 移動計画を作成
            translation_vector = QPointF(-MOVE_DISTANCE, MOVE_DISTANCE)  # 左下方向へのベクトル
            move_operations.append((ids_to_move, translation_vector))

        # --- ステップ4: 計画された移動を一度に実行 ---
        if not move_operations:
            self.statusBar().showMessage("No actionable overlaps found.", 2000)
            return

        for group_ids, vector in move_operations:
            for atom_id in group_ids:
                item = self.data.atoms[atom_id]['item']
                new_pos = item.pos() + vector
                item.setPos(new_pos)
                self.data.atoms[atom_id]['pos'] = new_pos

        # --- ステップ5: 表示と状態を更新 ---
        for bond_data in self.data.bonds.values():
            if bond_data and 'item' in bond_data:
                bond_data['item'].update_position()
        self.scene.update()
        self.push_undo_state()
        self.statusBar().showMessage("Resolved overlapping groups.", 2000)



    def draw_molecule_3d(self, mol):
        """3D 分子を描画し、軸アクターの参照をクリアする（軸の再制御は apply_3d_settings に任せる）"""
        
        # 測定選択をクリア（分子が変更されたため）
        if hasattr(self, 'measurement_mode'):
            self.clear_measurement_selection()
        
        # 1. カメラ状態とクリア
        camera_state = self.plotter.camera.copy()

        # **残留防止のための強制削除**
        if self.axes_actor is not None:
            try:
                self.plotter.remove_actor(self.axes_actor)
            except Exception:
                pass 
            self.axes_actor = None

        self.plotter.clear()
            
        # 2. 背景色の設定
        self.plotter.set_background(self.settings.get('background_color', '#4f4f4f'))

        # 3. mol が None または原子数ゼロの場合は、背景と軸のみで終了
        if mol is None or mol.GetNumAtoms() == 0:
            self.atom_actor = None
            self.current_mol = None
            self.plotter.render()
            return
            
        # 4. ライティングの設定
        is_lighting_enabled = self.settings.get('lighting_enabled', True)

        if is_lighting_enabled:
            light = pv.Light(
                position=(1, 1, 2),
                light_type='cameralight',
                intensity=self.settings.get('light_intensity', 1.2)
            )
            self.plotter.add_light(light)
            
        # 5. 分子描画ロジック
        conf = mol.GetConformer()

        self.atom_positions_3d = np.array([list(conf.GetAtomPosition(i)) for i in range(mol.GetNumAtoms())])

        sym = [a.GetSymbol() for a in mol.GetAtoms()]
        col = np.array([CPK_COLORS_PV.get(s, [0.5, 0.5, 0.5]) for s in sym])

        if self.current_3d_style == 'cpk':
            rad = np.array([pt.GetRvdw(pt.GetAtomicNumber(s)) * 1.0 for s in sym])
        else:
            rad = np.array([VDW_RADII.get(s, 0.4) for s in sym])

        self.glyph_source = pv.PolyData(self.atom_positions_3d)
        self.glyph_source['colors'] = col
        self.glyph_source['radii'] = rad

        glyphs = self.glyph_source.glyph(scale='radii', geom=pv.Sphere(radius=1.0, theta_resolution=32, phi_resolution=32), orient=False)

        mesh_props = dict(
            smooth_shading=True,
            specular=self.settings.get('specular', 0.2),
            specular_power=self.settings.get('specular_power', 20),
            lighting=is_lighting_enabled,
        )

        if is_lighting_enabled:
            self.atom_actor = self.plotter.add_mesh(glyphs, scalars='colors', rgb=True, **mesh_props)
        else:
            self.atom_actor = self.plotter.add_mesh(
                glyphs, scalars='colors', rgb=True, 
                style='surface', show_edges=True, edge_color='grey',
                **mesh_props
            )
            self.atom_actor.GetProperty().SetEdgeOpacity(0.3)


        if self.current_3d_style == 'ball_and_stick':
            bond_meshes = []
            for bond in mol.GetBonds():
                sp = np.array(conf.GetAtomPosition(bond.GetBeginAtomIdx()))
                ep = np.array(conf.GetAtomPosition(bond.GetEndAtomIdx()))
                bt = bond.GetBondType()
                c = (sp + ep) / 2
                d = ep - sp
                h = np.linalg.norm(d)
                if h == 0: continue

                cyl_radius = 0.1
                if bt == Chem.rdchem.BondType.SINGLE or bt == Chem.rdchem.BondType.AROMATIC:
                    cyl = pv.Cylinder(center=c, direction=d, radius=cyl_radius, height=h, resolution=16)
                    bond_meshes.append(cyl)
                else:
                    v1 = d / h
                    v_arb = np.array([0, 0, 1])
                    if np.allclose(np.abs(np.dot(v1, v_arb)), 1.0): v_arb = np.array([0, 1, 0])
                    off_dir = np.cross(v1, v_arb)
                    off_dir /= np.linalg.norm(off_dir)
                    r, s = cyl_radius * 0.8, cyl_radius * 2.0
                    if bt == Chem.rdchem.BondType.DOUBLE:
                        c1, c2 = c + off_dir * (s / 2), c - off_dir * (s / 2)
                        bond_meshes.append(pv.Cylinder(center=c1, direction=d, radius=r, height=h, resolution=16))
                        bond_meshes.append(pv.Cylinder(center=c2, direction=d, radius=r, height=h, resolution=16))
                    elif bt == Chem.rdchem.BondType.TRIPLE:
                        bond_meshes.append(pv.Cylinder(center=c, direction=d, radius=r, height=h, resolution=16))
                        bond_meshes.append(pv.Cylinder(center=c + off_dir * s, direction=d, radius=r, height=h, resolution=16))
                        bond_meshes.append(pv.Cylinder(center=c - off_dir * s, direction=d, radius=r, height=h, resolution=16))

            if bond_meshes:
                combined_bonds = pv.merge(bond_meshes)
                self.plotter.add_mesh(combined_bonds, color='grey', **mesh_props)

        if getattr(self, 'show_chiral_labels', False):
            try:
                # 3D座標からキラル中心を計算
                chiral_centers = Chem.FindMolChiralCenters(mol, includeUnassigned=True)
                if chiral_centers:
                    pts, labels = [], []
                    z_off = 0
                    for idx, lbl in chiral_centers:
                        coord = self.atom_positions_3d[idx].copy(); coord[2] += z_off
                        pts.append(coord); labels.append(lbl if lbl is not None else '?')
                    try: self.plotter.remove_actor('chiral_labels')
                    except Exception: pass
                    self.plotter.add_point_labels(np.array(pts), labels, font_size=20, point_size=0, text_color='blue', name='chiral_labels', always_visible=True, tolerance=0.01, show_points=False)
            except Exception as e: self.statusBar().showMessage(f"3D chiral label drawing error: {e}")

        # E/Zラベルも表示
        if getattr(self, 'show_chiral_labels', False):
            try:
                self.show_ez_labels_3d(mol)
            except Exception as e: 
                self.statusBar().showMessage(f"3D E/Z label drawing error: {e}")

        self.plotter.camera = camera_state

    def show_ez_labels_3d(self, mol):
        """3DビューでE/Zラベルを表示する（RDKitのステレオ化学判定を使用）"""
        if not mol:
            return
        
        try:
            # 既存のE/Zラベルを削除
            self.plotter.remove_actor('ez_labels')
        except:
            pass
        
        pts, labels = [], []
        
        # 3D座標が存在するかチェック
        if mol.GetNumConformers() == 0:
            return
            
        conf = mol.GetConformer()
        
        # RDKitに3D座標からステレオ化学を計算させる
        try:
            # 3D座標からステレオ化学を再計算
            Chem.AssignStereochemistry(mol, cleanIt=True, force=True, flagPossibleStereoCenters=True)
        except:
            pass
        
        # 二重結合でRDKitが判定したE/Z立体化学を表示
        for bond in mol.GetBonds():
            if bond.GetBondType() == Chem.BondType.DOUBLE:
                stereo = bond.GetStereo()
                if stereo in [Chem.BondStereo.STEREOE, Chem.BondStereo.STEREOZ]:
                    # 結合の中心座標を計算
                    begin_pos = np.array(conf.GetAtomPosition(bond.GetBeginAtomIdx()))
                    end_pos = np.array(conf.GetAtomPosition(bond.GetEndAtomIdx()))
                    center_pos = (begin_pos + end_pos) / 2
                    
                    # RDKitの判定結果を使用
                    label = 'E' if stereo == Chem.BondStereo.STEREOE else 'Z'
                    pts.append(center_pos)
                    labels.append(label)
        
        if pts and labels:
            self.plotter.add_point_labels(
                np.array(pts), 
                labels, 
                font_size=18,
                point_size=0,
                text_color='darkgreen',  # 暗い緑色
                name='ez_labels',
                always_visible=True,
                tolerance=0.01,
                show_points=False
            )


    def toggle_chiral_labels_display(self, checked):
        """Viewメニューのアクションに応じてキラルラベル表示を切り替える"""
        self.show_chiral_labels = checked
        
        if self.current_mol:
            self.draw_molecule_3d(self.current_mol) 
        
        if checked:
            self.statusBar().showMessage("Chiral labels: will be (re)computed after Convert→3D.")
        else:
            self.statusBar().showMessage("Chiral labels disabled.")


    def update_chiral_labels(self):
        """分子のキラル中心を計算し、2Dビューの原子アイテムにR/Sラベルを設定/解除する
        ※ 可能なら 3D（self.current_mol）を優先して計算し、なければ 2D から作った RDKit 分子を使う。
        """
        # まず全てのアイテムからラベルをクリア
        for atom_data in self.data.atoms.values():
            if atom_data.get('item'):
                atom_data['item'].chiral_label = None

        if not self.show_chiral_labels:
            self.scene.update()
            return

        # 3D の RDKit Mol（コンフォマーを持つもの）を使う
        mol_for_chirality = None
        if getattr(self, 'current_mol', None) is not None:
            mol_for_chirality = self.current_mol
        else:
            return

        if mol_for_chirality is None or mol_for_chirality.GetNumAtoms() == 0:
            self.scene.update()
            return

        try:
            # --- 重要：3D コンフォマーがあるなら、それを使って原子のキラルタグを割り当てる ---
            if mol_for_chirality.GetNumConformers() > 0:
                # confId=0（最初のコンフォマー）を指定して、原子のキラリティータグを3D座標由来で設定
                try:
                    Chem.AssignAtomChiralTagsFromStructure(mol_for_chirality, confId=0)
                except Exception:
                    # 古い RDKit では関数が無い場合があるので（念のため保護）
                    pass

            # RDKit の通常の stereochemistry 割当（念のため）
            #Chem.AssignStereochemistry(mol_for_chirality, cleanIt=True, force=True, flagPossibleStereoCenters=True)

            # キラル中心の取得（(idx, 'R'/'S'/'?') のリスト）
            chiral_centers = Chem.FindMolChiralCenters(mol_for_chirality, includeUnassigned=True)

            # RDKit atom index -> エディタ側 atom_id へのマッピング
            rdkit_idx_to_my_id = {}
            for atom in mol_for_chirality.GetAtoms():
                if atom.HasProp("_original_atom_id"):
                    rdkit_idx_to_my_id[atom.GetIdx()] = atom.GetIntProp("_original_atom_id")

            # 見つかったキラル中心を対応する AtomItem に設定
            for idx, label in chiral_centers:
                if idx in rdkit_idx_to_my_id:
                    atom_id = rdkit_idx_to_my_id[idx]
                    if atom_id in self.data.atoms and self.data.atoms[atom_id].get('item'):
                        # 'R' / 'S' / '?'
                        self.data.atoms[atom_id]['item'].chiral_label = label

        except Exception as e:
            self.statusBar().showMessage(f"Update chiral labels error: {e}")

        # 最後に 2D シーンを再描画
        self.scene.update()


    def open_analysis_window(self):
        if self.current_mol:
            dialog = AnalysisWindow(self.current_mol, self)
            dialog.exec()
        else:
            self.statusBar().showMessage("Please generate a 3D structure first to show analysis.")

    def closeEvent(self, event):
        if self.settings != self.initial_settings:
            self.save_settings()
        reply = QMessageBox.question(self, 'Confirm Exit', 
                                     "Are you sure you want to exit?", 
                                     QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No, 
                                     QMessageBox.StandardButton.No)

        if reply == QMessageBox.StandardButton.Yes:
            if self.scene and self.scene.template_preview:
                self.scene.template_preview.hide()

            self.thread.quit()
            self.thread.wait()
            
            event.accept()
        else:
            event.ignore()

    def zoom_in(self):
        """ ビューを 20% 拡大する """
        self.view_2d.scale(1.2, 1.2)

    def zoom_out(self):
        """ ビューを 20% 縮小する """
        self.view_2d.scale(1/1.2, 1/1.2)
        
    def reset_zoom(self):
        """ ビューの拡大率をデフォルト (75%) にリセットする """
        transform = QTransform()
        transform.scale(0.75, 0.75)
        self.view_2d.setTransform(transform)

    def fit_to_view(self):
        """ シーン上のすべてのアイテムがビューに収まるように調整する """
        if not self.scene.items():
            self.reset_zoom()
            return
            
        bounds = self.scene.itemsBoundingRect()
        visible_items_rect = QRectF()
        for item in self.scene.items():
            if item.isVisible() and not isinstance(item, TemplatePreviewItem):
                 if visible_items_rect.isEmpty():
                     visible_items_rect = item.sceneBoundingRect()
                 else:
                     visible_items_rect = visible_items_rect.united(item.sceneBoundingRect())
        
        if not visible_items_rect.isEmpty():
             self.view_2d.fitInView(visible_items_rect, Qt.AspectRatioMode.KeepAspectRatio)
             self.view_2d.scale(0.6, 0.6)
        else:
             self.reset_zoom()

    def toggle_3d_edit_mode(self, checked):
        """「3D Edit」ボタンの状態に応じて編集モードを切り替える"""
        self.is_3d_edit_mode = checked
        if checked:
            self.statusBar().showMessage("3D Edit Mode: ON.")
        else:
            self.statusBar().showMessage("3D Edit Mode: OFF.")
        self.view_2d.setFocus()

    def _setup_3d_picker(self):
        self.plotter.picker = vtk.vtkCellPicker()
        self.plotter.picker.SetTolerance(0.025)

        # 新しいカスタムスタイル（原子移動用）のインスタンスを作成
        style = CustomInteractorStyle(self)
        
        # 調査の結果、'style' プロパティへの代入が正しい設定方法と判明
        self.plotter.interactor.SetInteractorStyle(style)
        self.plotter.interactor.Initialize()
        
    def load_mol_file_for_3d_viewing(self, file_path=None):
        """MOL/SDFファイルを3Dビューアーで開く"""
        if not file_path:
            file_path, _ = QFileDialog.getOpenFileName(
                self, "Open MOL/SDF File", "", 
                "MOL/SDF Files (*.mol *.sdf);;All Files (*)"
            )
            if not file_path:
                return
        
        try:
            # RDKitでMOL/SDFファイルを読み込み
            if file_path.lower().endswith('.sdf'):
                suppl = Chem.SDMolSupplier(file_path)
                mol = next(suppl, None)
            else:
                mol = Chem.MolFromMolFile(file_path)
            
            if mol is None:
                self.statusBar().showMessage(f"Failed to load molecule from {file_path}")
                return
            
            # 3D座標がない場合は2Dから3D変換（最適化なし）
            if mol.GetNumConformers() == 0:
                self.statusBar().showMessage("No 3D coordinates found. Converting to 3D...")
                try:
                    AllChem.EmbedMolecule(mol)
                    # 最適化は実行しない
                except:
                    self.statusBar().showMessage("Failed to generate 3D coordinates")
                    return
            
            # 3Dビューアーに表示
            self.current_mol = mol
            self.draw_molecule_3d(mol)
            
            # カメラをリセット
            self.plotter.reset_camera()
            
            # UIを3Dビューアーモードに設定
            self._enter_3d_viewer_ui_mode()
            
            self.statusBar().showMessage(f"Loaded {file_path} in 3D viewer")
            
        except Exception as e:
            self.statusBar().showMessage(f"Error loading MOL/SDF file: {e}")
    
    def load_command_line_file(self, file_path):
        """コマンドライン引数で指定されたファイルを開く"""
        if not file_path or not os.path.exists(file_path):
            return
        
        file_ext = file_path.lower().split('.')[-1]
        
        if file_ext in ['mol', 'sdf']:
            self.load_mol_file_for_3d_viewing(file_path)
        elif file_ext == 'xyz':
            self.load_xyz_for_3d_viewing(file_path)
        elif file_ext == 'pmeraw':
            self.load_raw_data(file_path=file_path)
        else:
            self.statusBar().showMessage(f"Unsupported file type: {file_ext}")
        
    def dragEnterEvent(self, event):
        """ウィンドウ全体で .pmeraw、.mol、.sdf、.xyz ファイルのドラッグを受け入れる"""
        if event.mimeData().hasUrls():
            urls = event.mimeData().urls()
            if urls and urls[0].isLocalFile():
                file_path = urls[0].toLocalFile()
                if file_path.lower().endswith(('.pmeraw', '.mol', '.sdf', '.xyz')):
                    event.acceptProposedAction()
                    return
        event.ignore()

    def dropEvent(self, event):
        """ファイルがウィンドウ上でドロップされたときに呼び出される"""
        urls = event.mimeData().urls()
        if urls and urls[0].isLocalFile():
            file_path = urls[0].toLocalFile()
            
            # ドロップ位置を取得
            drop_pos = event.position().toPoint()
            
            # 拡張子に応じて適切な読み込みメソッドを呼び出す
            if file_path.lower().endswith('.pmeraw'):
                self.load_raw_data(file_path=file_path)
                event.acceptProposedAction()
            elif file_path.lower().endswith(('.mol', '.sdf')):
                # MOL/SDFファイルは3Dビューアー領域にドロップされたかを判定
                plotter_widget = self.splitter.widget(1)  # 3Dビューアーウィジェット
                plotter_rect = plotter_widget.geometry()
                
                if plotter_rect.contains(drop_pos):
                    # 3Dビューアー領域にドロップ → 3Dビューアーで開く
                    self.load_mol_file_for_3d_viewing(file_path=file_path)
                else:
                    # 2D領域にドロップ → 従来の2D読み込み
                    if hasattr(self, 'load_mol_file'):
                        self.load_mol_file(file_path=file_path)
                    else:
                        self.statusBar().showMessage("MOL file import not implemented for 2D editor.")
                event.acceptProposedAction()
            elif file_path.lower().endswith('.xyz'):
                # XYZファイルは常に3Dビューアーで開く
                self.load_xyz_for_3d_viewing(file_path=file_path)
                event.acceptProposedAction()
            else:
                self.statusBar().showMessage(f"Unsupported file type: {file_path}")
                event.ignore()
        else:
            event.ignore()

    def _enter_3d_viewer_ui_mode(self):
        """3DビューアモードのUI状態に設定する"""
        self.is_2d_editable = False
        self.cleanup_button.setEnabled(False)
        self.convert_button.setEnabled(False)
        for action in self.tool_group.actions():
            action.setEnabled(False)
        if hasattr(self, 'other_atom_action'):
            self.other_atom_action.setEnabled(False)
        
        self.minimize_2d_panel()

        self.optimize_3d_button.setEnabled(True)
        self.export_button.setEnabled(True)
        self.edit_3d_action.setEnabled(True)
        self.analysis_action.setEnabled(True)
        self.measurement_action.setEnabled(True)  # 測定機能を有効化

    def restore_ui_for_editing(self):
        """Enables all 2D editing UI elements."""
        self.is_2d_editable = True
        self.restore_2d_panel()
        self.cleanup_button.setEnabled(True)
        self.convert_button.setEnabled(True)

        for action in self.tool_group.actions():
            action.setEnabled(True)
        
        if hasattr(self, 'other_atom_action'):
            self.other_atom_action.setEnabled(True)

    def minimize_2d_panel(self):
        """2Dパネルを最小化（非表示に）する"""
        sizes = self.splitter.sizes()
        # すでに最小化されていなければ実行
        if sizes[0] > 0:
            total_width = sum(sizes)
            self.splitter.setSizes([0, total_width])

    def restore_2d_panel(self):
        """最小化された2Dパネルを元のサイズに戻す"""
        sizes = self.splitter.sizes()
        
        # sizesリストが空でないことを確認してからアクセスする
        if sizes and sizes[0] == 0:
            self.splitter.setSizes([600, 600])

            
    def apply_initial_settings(self):
        """UIの初期化が完了した後に、保存された設定を3Dビューに適用する"""
        if self.plotter and self.plotter.renderer:
            bg_color = self.settings.get('background_color', '#919191')
            self.plotter.set_background(bg_color)
            self.apply_3d_settings()


    def apply_3d_settings(self):
        """3Dビューの視覚設定を適用する"""
        if not hasattr(self, 'plotter'):
            return  
        
        # レンダラーのレイヤー設定を有効化（テキストオーバーレイ用）
        renderer = self.plotter.renderer
        if renderer and hasattr(renderer, 'SetNumberOfLayers'):
            try:
                renderer.SetNumberOfLayers(2)  # レイヤー0:3Dオブジェクト、レイヤー1:2Dオーバーレイ
            except:
                pass  # PyVistaのバージョンによってはサポートされていない場合がある  

        # --- 3D軸ウィジェットの設定 ---
        show_axes = self.settings.get('show_3d_axes', True) 

        # ウィジェットがまだ作成されていない場合は作成する
        if self.axes_widget is None and hasattr(self.plotter, 'interactor'):
            axes = vtk.vtkAxesActor()
            self.axes_widget = vtk.vtkOrientationMarkerWidget()
            self.axes_widget.SetOrientationMarker(axes)
            self.axes_widget.SetInteractor(self.plotter.interactor)
            # 左下隅に設定 (幅・高さ20%)
            self.axes_widget.SetViewport(0.0, 0.0, 0.2, 0.2)

        # 設定に応じてウィジェットを有効化/無効化
        if self.axes_widget:
            if show_axes:
                self.axes_widget.On()
                self.axes_widget.SetInteractive(False)  
            else:
                self.axes_widget.Off()  

        self.draw_molecule_3d(self.current_mol)
        # 設定変更時にカメラ位置をリセットしない（初回のみリセット）
        if not getattr(self, '_camera_initialized', False):
            try:
                self.plotter.reset_camera()
            except Exception:
                pass
            self._camera_initialized = True



    def open_settings_dialog(self):
        dialog = SettingsDialog(self.settings, self)
        if dialog.exec():
            self.settings = dialog.get_settings()
            self.save_settings()
            self.apply_3d_settings()

    def load_settings(self):
        default_settings = {
            'background_color': '#919191',
            'lighting_enabled': True,
            'specular': 0.2,
            'specular_power': 20,
            'light_intensity': 1.0,
            'show_3d_axes': True,
        }

        try:
            if os.path.exists(self.settings_file):
                with open(self.settings_file, 'r') as f:
                    loaded_settings = json.load(f)
                
                for key, value in default_settings.items():
                    loaded_settings.setdefault(key, value)
                self.settings = loaded_settings
            
            else:
                self.settings = default_settings
        
        except Exception:
            self.settings = default_settings

    def save_settings(self):
        try:
            if not os.path.exists(self.settings_dir):
                os.makedirs(self.settings_dir)
            with open(self.settings_file, 'w') as f:
                json.dump(self.settings, f, indent=4)
        except Exception as e:
            print(f"Error saving settings: {e}")

    def toggle_measurement_mode(self, checked):
        """測定モードのオン/オフを切り替える"""
        self.measurement_mode = checked
        if not checked:
            self.clear_measurement_selection()
        
        # ボタンのテキストとステータスメッセージを更新
        if checked:
            self.statusBar().showMessage("Measurement mode enabled. Click atoms to measure distances/angles/dihedrals.")
        else:
            self.statusBar().showMessage("Measurement mode disabled.")

    def handle_measurement_atom_selection(self, atom_idx):
        """測定用の原子選択を処理する"""
        # 既に選択されている原子の場合は除外
        if atom_idx in self.selected_atoms_for_measurement:
            return
        
        self.selected_atoms_for_measurement.append(atom_idx)
        
        # 4つ以上選択された場合はクリア
        if len(self.selected_atoms_for_measurement) > 4:
            self.clear_measurement_selection()
            self.selected_atoms_for_measurement.append(atom_idx)
        
        # 原子にラベルを追加
        self.add_measurement_label(atom_idx, len(self.selected_atoms_for_measurement))
        
        # 測定値を計算して表示
        self.calculate_and_display_measurements()

    def add_measurement_label(self, atom_idx, label_number):
        """原子に数字ラベルを追加する"""
        if not self.current_mol or atom_idx >= self.current_mol.GetNumAtoms():
            return
        
        # 測定ラベルリストを更新
        self.measurement_labels.append((atom_idx, str(label_number)))
        
        # すべての測定ラベルを再描画
        self.update_measurement_labels_display()

    def update_measurement_labels_display(self):
        """測定ラベルを3D表示に描画する（キラルラベルと同じ方法）"""
        try:
            # 既存の測定ラベルを削除
            self.plotter.remove_actor('measurement_labels')
        except:
            pass
        
        if not self.measurement_labels or not self.current_mol:
            return
        
        # ラベル位置とテキストを準備
        pts, labels = [], []
        for atom_idx, label_text in self.measurement_labels:
            if atom_idx < len(self.atom_positions_3d):
                coord = self.atom_positions_3d[atom_idx].copy()
                coord[2] += 0.3  # 少し上にオフセット
                pts.append(coord)
                labels.append(label_text)
        
        if pts and labels:
            # PyVistaのpoint_labelsを使用（小さなフォント、等幅、左上寄せ）
            self.plotter.add_point_labels(
                np.array(pts), 
                labels, 
                font_size=16,  # より小さく
                point_size=0,
                text_color='red',  # 測定値は赤色で
                name='measurement_labels',
                always_visible=True,
                tolerance=0.01,
                show_points=False
            )

    def clear_measurement_selection(self):
        """測定選択をクリアする"""
        self.selected_atoms_for_measurement.clear()
        
        # ラベルを削除
        self.measurement_labels.clear()
        try:
            self.plotter.remove_actor('measurement_labels')
        except:
            pass
        
        # 測定結果のテキストを削除
        if self.measurement_text_actor:
            try:
                self.plotter.remove_actor(self.measurement_text_actor)
                self.measurement_text_actor = None
            except:
                pass
        
        self.plotter.render()

    def calculate_and_display_measurements(self):
        """選択された原子に基づいて測定値を計算し表示する"""
        num_selected = len(self.selected_atoms_for_measurement)
        if num_selected < 2:
            return
        
        measurement_text = []
        
        if num_selected >= 2:
            # 距離の計算
            atom1_idx = self.selected_atoms_for_measurement[0]
            atom2_idx = self.selected_atoms_for_measurement[1]
            distance = self.calculate_distance(atom1_idx, atom2_idx)
            measurement_text.append(f"Distance 1-2: {distance:.3f} Å")
        
        if num_selected >= 3:
            # 角度の計算
            atom1_idx = self.selected_atoms_for_measurement[0]
            atom2_idx = self.selected_atoms_for_measurement[1] 
            atom3_idx = self.selected_atoms_for_measurement[2]
            angle = self.calculate_angle(atom1_idx, atom2_idx, atom3_idx)
            measurement_text.append(f"Angle 1-2-3: {angle:.2f}°")
        
        if num_selected >= 4:
            # 二面角の計算
            atom1_idx = self.selected_atoms_for_measurement[0]
            atom2_idx = self.selected_atoms_for_measurement[1]
            atom3_idx = self.selected_atoms_for_measurement[2]
            atom4_idx = self.selected_atoms_for_measurement[3]
            dihedral = self.calculate_dihedral(atom1_idx, atom2_idx, atom3_idx, atom4_idx)
            measurement_text.append(f"Dihedral 1-2-3-4: {dihedral:.2f}°")
        
        # 測定結果を3D画面の右上に表示
        self.display_measurement_text(measurement_text)

    def calculate_distance(self, atom1_idx, atom2_idx):
        """2原子間の距離を計算する"""
        pos1 = np.array(self.atom_positions_3d[atom1_idx])
        pos2 = np.array(self.atom_positions_3d[atom2_idx])
        return np.linalg.norm(pos2 - pos1)

    def calculate_angle(self, atom1_idx, atom2_idx, atom3_idx):
        """3原子の角度を計算する（中央が頂点）"""
        pos1 = np.array(self.atom_positions_3d[atom1_idx])
        pos2 = np.array(self.atom_positions_3d[atom2_idx])  # 頂点
        pos3 = np.array(self.atom_positions_3d[atom3_idx])
        
        # ベクトルを計算
        vec1 = pos1 - pos2
        vec2 = pos3 - pos2
        
        # 角度を計算（ラジアンから度に変換）
        cos_angle = np.dot(vec1, vec2) / (np.linalg.norm(vec1) * np.linalg.norm(vec2))
        # 数値誤差による範囲外の値をクリップ
        cos_angle = np.clip(cos_angle, -1.0, 1.0)
        angle_rad = np.arccos(cos_angle)
        return np.degrees(angle_rad)

    def calculate_dihedral(self, atom1_idx, atom2_idx, atom3_idx, atom4_idx):
        """4原子の二面角を計算する"""
        pos1 = np.array(self.atom_positions_3d[atom1_idx])
        pos2 = np.array(self.atom_positions_3d[atom2_idx])
        pos3 = np.array(self.atom_positions_3d[atom3_idx])
        pos4 = np.array(self.atom_positions_3d[atom4_idx])
        
        # ベクトルを計算
        b1 = pos2 - pos1
        b2 = pos3 - pos2
        b3 = pos4 - pos3
        
        # 法線ベクトルを計算
        n1 = np.cross(b1, b2)
        n2 = np.cross(b2, b3)
        
        # 二面角を計算
        cos_angle = np.dot(n1, n2) / (np.linalg.norm(n1) * np.linalg.norm(n2))
        cos_angle = np.clip(cos_angle, -1.0, 1.0)
        
        # 符号を決定するための計算
        m1 = np.cross(n1, b2 / np.linalg.norm(b2))
        sign = np.sign(np.dot(m1, n2))
        
        angle_rad = np.arccos(cos_angle) * sign
        return np.degrees(angle_rad)

    def display_measurement_text(self, measurement_lines):
        """測定結果のテキストを3D画面の左上に表示する（小さな等幅フォント）"""
        # 既存のテキストを削除
        if self.measurement_text_actor:
            try:
                self.plotter.remove_actor(self.measurement_text_actor)
            except:
                pass
        
        if not measurement_lines:
            self.measurement_text_actor = None
            return
        
        # テキストを結合
        text = '\n'.join(measurement_lines)
        
        # 背景色から適切なテキスト色を決定
        try:
            bg_color_hex = self.settings.get('background_color', '#919191')
            bg_qcolor = QColor(bg_color_hex)
            if bg_qcolor.isValid():
                luminance = bg_qcolor.toHsl().lightness()
                text_color = 'black' if luminance > 128 else 'white'
            else:
                text_color = 'white'
        except:
            text_color = 'white'
        
        # 左上に表示（小さな等幅フォント）
        self.measurement_text_actor = self.plotter.add_text(
            text,
            position='upper_left',
            font_size=10,  # より小さく
            color=text_color,  # 背景に合わせた色
            font='courier',  # 等幅フォント
            name='measurement_display'
        )
        
        self.plotter.render()


# --- Application Execution ---
if __name__ == '__main__':
    main()

