"""
Fix Result Checker
"""
from __future__ import annotations

import json
import logging
from typing import Dict, Any, Union, Optional, TYPE_CHECKING

from openai.types.chat import ChatCompletionMessageParam

if TYPE_CHECKING:
    from siada.agent_hub.coder.tracing.bug_fix_trace_collector import BugFixTraceCollector

from siada.provider.client_factory import get_client_with_kwargs
from siada.foundation.setting import settings

logger = logging.getLogger(__name__)

class FixResultChecker:

    async def check(self, issue_desc: str, fix_code: str, context: Any, trace_collector: Optional["BugFixTraceCollector"] = None) -> Dict[str, Any]:
        """
        
        Args:
            issue_desc: Problems description, describing the issue to be fixed
            fix_code: fix code, the code changes made to fix the issue
            context: includes provider and other necessary information for the model to analyze the fix
            trace_collector: Optional trace collector for recording check process
            
        Returns:
            Dict[str, Any]:
            {
                "is_fixed": bool,    
                "check_summary": str, 
                "analysis": str    
            }
        """
        try:
            analysis_result = await self._call_model_for_analysis(issue_desc, fix_code, context)
            check_result = self._parse_analysis_result(analysis_result)
            
            self._record_checker_trace(trace_collector, issue_desc, fix_code, check_result)
            
            return check_result
        except Exception as e:
            logger.error(f"Fix result check failed: {e}", exc_info=True)
            error_result = {
                "is_fixed": False,
                "check_summary": f"error in analysis process: {str(e)}",
                "analysis": f"error detail: {str(e)}"
            }
            
            self._record_checker_trace(trace_collector, issue_desc, fix_code, error_result, error=str(e))
            
            return error_result
    
    async def _call_model_for_analysis(self, issue_desc: str, fix_code: str, context: Any) -> str:

        user_task = self.build_prompt(issue_desc, fix_code)
        print("fix result check prompt:", user_task)
        model_messages: list[ChatCompletionMessageParam] = [
            {"role": "user", "content": user_task},
        ]
        

        # Call the model
        default_kwargs = {
            "model": settings.Claude_4_0_SONNET,
            "messages": model_messages,
            "stream": False,
            "temperature": 0.2,  # Lower temperature for accuracy and consistency
        }

        # Use get_client_with_kwargs to support context parameter overrides
        client, complete_kwargs = get_client_with_kwargs(context, default_kwargs)
        response = await client.completion(**complete_kwargs)
        
        if response and response.choices and response.choices[0].message:
            analysis = response.choices[0].message.content
            if analysis:
                return analysis.strip()
        
        raise Exception("cannot get analysis result from model response")
    
    def build_prompt(self, issue_desc: str, fix_code: str) -> str:
        return f"""
        You are a **ZERO TOLERANCE CODE REVIEW Agent (Agent 2)**, responsible for working in the **second stage** of the issue resolution workflow. The current process has three steps:
        1. **Initial Resolution**: Agent 1 attempts to resolve the issue based on the problem description, generating a solution trace and code patch.
        2. **Review & Verification**: Agent 2 evaluates whether the issue has been resolved by reviewing the code patch and execution trace generated by Agent 1; if not, it produces a review conclusion.
        4. **Iterative Improvement**: Agent 1 attempts to resolve the issue again based on the review conclusions from Step 2.

        
        **Please systematically analyze whether the code modification truly resolves the issue by following the steps below and return your analysis in JSON format:**

---
Please systematically analyze whether the code modifications truly fix the problem by following these steps:

## Step 1: Deep Root Cause Analysis
1. **Core Problem Identification**: Extract the fundamental cause of the problem from the issue description, distinguishing between symptoms and true root causes
2. **Problem Impact Scope**: List all affected code paths, usage scenarios, and boundary conditions
3. **Problem Trigger Conditions**: Clarify under what conditions this problem will be triggered, with special attention to edge cases
4. **Expected Behavior Definition**: Based on the problem description, clearly define the specific behavior that should be achieved after the fix
5. **Reverse Logic Check**: Confirm whether the fix direction is correct, avoiding going in the opposite direction of expectations

## Step 2: Fix Strategy Rationality Assessment
1. **Fix Type Classification**:
   - Fundamental fix: Directly addresses the root cause
   - Symptomatic fix: Only masks or bypasses the error phenomenon
   - Compensatory fix: Avoids the problem through other mechanisms
2. **Solution Alignment**: Whether the fix solution directly targets the root cause
3. **Complexity Rationality**: Assess whether there is over-complication or over-engineering
4. **Minimal Intrusion Principle**: Whether it follows the principle of minimal changes, avoiding unnecessary modifications

## Step 3: Fix Code Implementation Quality Analysis
### 3.1 Coverage Assessment
1. **Modification Point Mapping**: Map each code modification point to specific problem scenarios
2. **Coverage Range Check**: Verify whether modifications cover all problem scenarios
3. **Missing Scenario Identification**: Identify uncovered scenarios that may have the same problem

### 3.2 Implementation Detail Analysis
1. **API Usage Appropriateness**: Verify whether the APIs used are the most direct and standard methods
2. **Code Execution Path**: Analyze whether there are unnecessary intermediate steps or roundabout implementations
3. **Error Handling Completeness**: Check whether all possible exception situations are correctly handled
4. **Performance Impact Assessment**: Analyze whether the fix introduces unnecessary performance overhead

## Step 4: Data Security and System Stability Check
1. **Data Security Risk**: Whether modifications may lead to data loss or inconsistency
2. **State Consistency**: Whether system state remains consistent after modifications
3. **Side Effect Assessment**: Evaluate whether modifications may introduce new problems
4. **Backward Compatibility**: Whether modifications maintain backward compatibility
5. **Rollback Safety**: Whether modifications support safe rollback

## Step 5: Design Principles and Architecture Consistency
1. **Architecture Alignment**: Whether modifications align with existing architecture and design patterns
2. **Framework Best Practices**: Whether they conform to the design philosophy and best practices of relevant frameworks
3. **Code Simplicity**: Whether the solution is concise, clear, easy to understand and maintain
4. **Maintainability Assessment**: Analyze the long-term maintainability and extensibility of the fix code

## Step 6: Test Verification Completeness
1. **Test Scenario Coverage**: Whether test cases cover all problem scenarios and boundary conditions
2. **Failed Case Analysis**: If there are test failures, analyze whether they indicate incomplete fixes
3. **Regression Test Verification**: Whether it's verified that modifications don't break existing functionality
4. **Performance Test Consideration**: Assess whether performance-related tests are needed to verify fix quality

## Step 7: Comprehensive Judgment and Recommendations

Based on the above analysis, provide clear conclusions:

### Required Output Fields:
1. **is_fixed**: true/false (partial fixes count as false)
2. **check_summary**: Detailed analysis summary, must include:
   - Specific basis for fix status judgment
   - If not fixed, clearly explain reasons for non-fix
   - If fixed, assess implementation quality and potential risks
   - Specific improvement suggestions or alternative solutions

## Key Analysis Focus:
- Whether the fundamental problem is truly solved rather than just making errors disappear
- Whether the fix direction is correct, avoiding directional errors
- Whether there's a tendency toward over-engineering
- Whether API usage is appropriate, avoiding roundabout or inefficient implementations
- Whether data security and system stability are ensured
- Long-term maintainability and extensibility of the code
---

## **Required JSON Output Format**

You must return your analysis in the following JSON format：

```json
{{
  "analysis": "The analysis results of each step",
  "result": {{
    "is_fixed": True,
    "check_summary": "Summary of each step of the analysis"
  }}
}}
```
---

**Problem Description & Solution Process Trace:**
{issue_desc}


**Code Change:**
{fix_code}
"""
    
    def _parse_analysis_result(self, analysis_result: str) -> Dict[str, Any]:
        try:
            json_content = analysis_result.strip()
            
            if json_content.startswith('```json') or json_content.startswith('```'):
                lines = json_content.split('\n')
                json_lines = []
                in_json_block = False
                for line in lines:
                    line_stripped = line.strip()
                    if line_stripped in ['```json', '```']:
                        if not in_json_block:
                            in_json_block = True
                        else:
                            break
                        continue
                    elif in_json_block:
                        json_lines.append(line)
                json_content = '\n'.join(json_lines)
            
            parsed_json = json.loads(json_content)
            
            if not isinstance(parsed_json, dict):
                raise ValueError("the result is not a valid JSON object")
            
            result = parsed_json.get("result", {})
            analysis = parsed_json.get("analysis", "")
            
            if isinstance(analysis, dict):
                analysis_text = self._build_analysis_text(analysis)
            else:
                analysis_text = str(analysis)
            
            return {
                "is_fixed": result.get("is_fixed", False),
                "check_summary": result.get("check_summary", "no summary provided"),
                "analysis": analysis_text
            }
            
        except (json.JSONDecodeError, ValueError, KeyError) as e:
            return self._fallback_text_parsing(analysis_result, str(e))
    
    def _build_analysis_text(self, analysis: Dict[str, Any]) -> str:

        if isinstance(analysis, str):
            return analysis
            
        if isinstance(analysis, dict):
            step_mappings = [
                ("step1_problem_scope", "Step 1: Deep Root Cause Analysis"),
                ("step2_fix_coverage", "Step 2: Fix Strategy Rationality Assessment"),
                ("step3_test_validation", "Step 3: Fix Code Implementation Quality Analysis"),
                ("step4_logical_consistency", "Step 4: Data Security and System Stability Check"),
                ("step5_final_assessment", "Step 5: Design Principles and Architecture Consistency"),
                ("step6_test_verification", "Step 6: Test Verification Completeness"),
                ("step7_comprehensive_judgment", "Step 7: Comprehensive Judgment and Recommendations")
            ]
            
            formatted_sections = []
            
            for key, title in step_mappings:
                content = analysis.get(key, "")
                if content:
                    formatted_sections.append(f"## {title}\n{content}")
            
            if not formatted_sections:
                for key, value in analysis.items():
                    if value:  
                        title = key.replace("_", " ").title()
                        formatted_sections.append(f"## {title}\n{str(value)}")
            
            return "\n\n".join(formatted_sections) if formatted_sections else str(analysis)
        
        return str(analysis)
    
    def _fallback_text_parsing(self, analysis_result: str, error_msg: str) -> Dict[str, Any]:
        is_fixed = self._extract_fix_status(analysis_result)
        check_summary = self._extract_check_summary(analysis_result, is_fixed)
        
        analysis_with_error = f"[Fail to resolved JSON: {error_msg}]\n\n{analysis_result}"
        
        return {
            "is_fixed": is_fixed,
            "check_summary": check_summary,
            "analysis": analysis_with_error
        }
    
    def _extract_fix_status(self, analysis: str) -> bool:
        analysis_lower = analysis.lower()
        
        if "fully fixed" in analysis_lower:
            return True
        elif "partially fixed" in analysis_lower or "not fixed" in analysis_lower:
            return False
        
        if "is the issue fixed: yes" in analysis_lower:
            return True
        elif "is the issue fixed: no" in analysis_lower:
            return False
        
        positive_indicators = ["resolved", "addressed", "fixed", "solved"]
        negative_indicators = ["not resolved", "not addressed", "not fixed", "incomplete", "missing"]
        
        positive_count = sum(1 for indicator in positive_indicators if indicator in analysis_lower)
        negative_count = sum(1 for indicator in negative_indicators if indicator in analysis_lower)
        
        if negative_count > positive_count:
            return False
        
        return positive_count > 0
    
    def _extract_check_summary(self, analysis: str, is_fixed: bool) -> str:
        if is_fixed:
            return "the issue has been fully resolved"
        
        lines = analysis.split('\n')
        check_summary_lines = []
        
        for line in lines:
            line_lower = line.lower().strip()
            if any(keyword in line_lower for keyword in [
                "check_summary", "because", "however", "but", "missing",
                "not covered", "incomplete", "still exists"
            ]):
                check_summary_lines.append(line.strip())
        
        if check_summary_lines:
            return " ".join(check_summary_lines)
        
        return "analysis does not provide a clear reason for the issue not being fixed"
    
    def _record_checker_trace(
        self, 
        trace_collector: Optional["BugFixTraceCollector"], 
        issue_desc: str, 
        fix_code: str, 
        check_result: Dict[str, Any], 
        error: Optional[str] = None
    ) -> None:
        """
        Record checker trace information
        
        Args:
            trace_collector: Optional trace collector
            issue_desc: Issue description
            fix_code: Fix code
            check_result: Check result
            error: Optional error message for error cases
        """
        if not trace_collector:
            return
            
        checker_prompt = self.build_prompt(issue_desc, fix_code)
        
        if error:
            # Error case
            feedback_message = {
                "role": "user",
                "content": f"Check process failed: {error}"
            }
            should_break = False
            check_summary = check_result.get("check_summary", f"Error: {error}")
        else:
            # Normal case
            is_fixed = check_result.get("is_fixed", False)
            feedback_message = {
                "role": "assistant" if is_fixed else "user",
                "content": check_result.get("check_summary", "No summary available")
            }
            should_break = is_fixed
            check_summary = check_result.get("check_summary", "")
        
        trace_collector.record_checker_trace(
            prompt=checker_prompt,
            user_input=issue_desc,
            check_summary=check_summary,
            feedback_message=feedback_message,
            should_break=should_break
        )
