"""DO NOT EDIT THIS FILE DIRECTLY, INSTEAD RUN `make package-data`"""

import datetime
from decimal import Decimal

from .types import *

__all__ = ('providers',)

providers: list[Provider] = [
    Provider(
        id='anthropic',
        name='Anthropic',
        api_pattern='https://api\\.anthropic\\.com',
        pricing_urls=['https://www.anthropic.com/pricing#api'],
        model_match=ClauseContains(contains='claude'),
        models=[
            ModelInfo(
                id='claude-2',
                match=ClauseOr(or_=[ClauseStartsWith(starts_with='claude-2'), ClauseContains(contains='claude-v2')]),
                name='Claude 2.0 / 2.1',
                description="Claude 2 is Anthropic's previous generation model, offering reliable performance for various tasks. This includes Claude 2.0 and Claude 2.1.\n",
                context_window=200000,
                prices=ModelPrice(input_mtok=Decimal('8'), output_mtok=Decimal('24')),
            ),
            ModelInfo(
                id='claude-3-5-haiku-latest',
                match=ClauseOr(
                    or_=[
                        ClauseStartsWith(starts_with='claude-3-5-haiku'),
                        ClauseStartsWith(starts_with='claude-3.5-haiku'),
                    ]
                ),
                name='Claude Haiku 3.5',
                description='Fastest, most cost-effective model',
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.8'),
                    cache_write_mtok=Decimal('1'),
                    cache_read_mtok=Decimal('0.08'),
                    output_mtok=Decimal('4'),
                ),
            ),
            ModelInfo(
                id='claude-3-5-sonnet',
                match=ClauseOr(
                    or_=[
                        ClauseStartsWith(starts_with='claude-3-5-sonnet'),
                        ClauseStartsWith(starts_with='claude-3.5-sonnet'),
                    ]
                ),
                name='Claude Sonnet 3.5',
                description='Claude 3.5 Sonnet is an ideal balance of intelligence and speed for enterprise workloads. Maximum utility at a lower price, dependable, balanced for scaled deployments.',
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='claude-3-7-sonnet-latest',
                match=ClauseOr(
                    or_=[
                        ClauseStartsWith(starts_with='claude-3-7-sonnet'),
                        ClauseStartsWith(starts_with='claude-3.7-sonnet'),
                    ]
                ),
                name='Claude Sonnet 3.7',
                description='Claude 3.7 Sonnet is an advanced large language model with improved reasoning, coding, and problem-solving capabilities.',
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='claude-3-haiku',
                match=ClauseStartsWith(starts_with='claude-3-haiku'),
                name='Claude Haiku 3',
                description='Fastest, most cost-effective model',
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.25'),
                    cache_write_mtok=Decimal('0.3'),
                    cache_read_mtok=Decimal('0.03'),
                    output_mtok=Decimal('1.25'),
                ),
            ),
            ModelInfo(
                id='claude-3-opus-latest',
                match=ClauseStartsWith(starts_with='claude-3-opus'),
                name='Claude Opus 3',
                description="Claude 3 Opus was Anthropic's most powerful model for highly complex tasks. It boasts top-level performance, intelligence, fluency, and understanding.",
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('15'),
                    cache_write_mtok=Decimal('18.75'),
                    cache_read_mtok=Decimal('1.5'),
                    output_mtok=Decimal('75'),
                ),
            ),
            ModelInfo(
                id='claude-3-sonnet',
                match=ClauseStartsWith(starts_with='claude-3-sonnet'),
                name='Claude 3 Sonnet',
                description='Claude 3 Sonnet is an ideal balance of intelligence and speed for enterprise workloads. Maximum utility at a lower price, dependable, balanced for scaled deployments.',
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='claude-opus-4-0',
                match=ClauseOr(
                    or_=[
                        ClauseStartsWith(starts_with='claude-opus-4'),
                        ClauseStartsWith(starts_with='claude-4-opus-20250522'),
                    ]
                ),
                name='Claude Opus 4',
                description='Most intelligent model for complex tasks',
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('15'),
                    cache_write_mtok=Decimal('18.75'),
                    cache_read_mtok=Decimal('1.5'),
                    output_mtok=Decimal('75'),
                ),
            ),
            ModelInfo(
                id='claude-opus-4-1',
                match=ClauseOr(
                    or_=[
                        ClauseStartsWith(starts_with='claude-opus-4'),
                        ClauseStartsWith(starts_with='claude-4-opus-20250522'),
                    ]
                ),
                name='Claude Opus 4.1',
                description='Most intelligent model for complex tasks',
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('15'),
                    cache_write_mtok=Decimal('18.75'),
                    cache_read_mtok=Decimal('1.5'),
                    output_mtok=Decimal('75'),
                ),
            ),
            ModelInfo(
                id='claude-sonnet-4-0',
                match=ClauseOr(
                    or_=[
                        ClauseStartsWith(starts_with='claude-sonnet-4'),
                        ClauseStartsWith(starts_with='claude-4-sonnet-20250522'),
                    ]
                ),
                name='Claude Sonnet 4',
                description='Optimal balance of intelligence, cost, and speed',
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
        ],
    ),
    Provider(
        id='avian',
        name='Avian',
        api_pattern='https://api\\.avian\\.io',
        pricing_urls=['https://avian.io/pricing/'],
        models=[
            ModelInfo(
                id='Meta-Llama-3.1-405B-Instruct',
                match=ClauseEquals(equals='Meta-Llama-3.1-405B-Instruct'),
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='Meta-Llama-3.1-70B-Instruct',
                match=ClauseEquals(equals='Meta-Llama-3.1-70B-Instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.45'), output_mtok=Decimal('0.45')),
            ),
            ModelInfo(
                id='Meta-Llama-3.1-8B-Instruct',
                match=ClauseEquals(equals='Meta-Llama-3.1-8B-Instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='Meta-Llama-3.3-70B-Instruct',
                match=ClauseEquals(equals='Meta-Llama-3.3-70B-Instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.45'), output_mtok=Decimal('0.45')),
            ),
        ],
    ),
    Provider(
        id='aws',
        name='AWS Bedrock',
        api_pattern='https://bedrock-runtime\\.[a-z0-9-]+\\.amazonaws\\.com/',
        pricing_urls=['https://aws.amazon.com/bedrock/pricing/'],
        models=[
            ModelInfo(
                id='meta.llama3-8b-instruct-v1%3A0',
                match=ClauseEquals(equals='meta.llama3-8b-instruct-v1%3A0'),
                prices=ModelPrice(input_mtok=Decimal('220'), output_mtok=Decimal('720')),
            ),
            ModelInfo(
                id='nova-lite-v1',
                match=ClauseEquals(equals='nova-lite-v1'),
                name='Nova Lite 1.0',
                description='Amazon Nova Lite 1.0 is a very low-cost multimodal model from Amazon that focused on fast processing of image, video, and text inputs to generate text output. Amazon Nova Lite can handle real-time customer interactions, document analysis, and visual question-answering tasks with high accuracy.',
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='nova-micro-v1',
                match=ClauseEquals(equals='nova-micro-v1'),
                name='Nova Micro 1.0',
                description='Amazon Nova Micro 1.0 is a text-only model that delivers the lowest latency responses in the Amazon Nova family of models at a very low cost. With a context length of 128K tokens and optimized for speed and cost, Amazon Nova Micro excels at tasks such as text summarization, translation, content classification, interactive chat, and brainstorming. It has  simple mathematical reasoning and coding abilities.',
                prices=ModelPrice(input_mtok=Decimal('0.035'), output_mtok=Decimal('0.14')),
            ),
            ModelInfo(
                id='nova-pro-v1',
                match=ClauseEquals(equals='nova-pro-v1'),
                name='Nova Pro 1.0',
                description='Amazon Nova Pro 1.0 is a capable multimodal model from Amazon focused on providing a combination of accuracy, speed, and cost for a wide range of tasks. As of December 2024, it achieves state-of-the-art performance on key benchmarks including visual question answering (TextVQA) and video understanding (VATEX).',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('3.2')),
            ),
        ],
    ),
    Provider(
        id='azure',
        name='Microsoft Azure',
        api_pattern='(https?://)?([^.]*\\.)?(?:openai\\.azure\\.com|azure-api\\.net|cognitiveservices\\.azure\\.com)',
        pricing_urls=['https://azure.microsoft.com/en-us/pricing/details/cognitive-services/openai-service/#pricing'],
        price_comments='These are prices for "*-Global" models, prices for "Regional" models are often slightly higher. Retired models are listed at https://learn.microsoft.com/th-th/azure/ai-foundry/openai/concepts/legacy-models',
        models=[
            ModelInfo(
                id='ada',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='ada'),
                        ClauseEquals(equals='text-embedding-ada'),
                        ClauseEquals(equals='text-embedding-ada-002'),
                        ClauseEquals(equals='text-embedding-ada-002-v2'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='babbage',
                match=ClauseOr(or_=[ClauseEquals(equals='babbage'), ClauseEquals(equals='babbage-002')]),
                prices=ModelPrice(input_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='curie',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='curie'),
                        ClauseEquals(equals='text-curie'),
                        ClauseEquals(equals='text-curie-001'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='davinci',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='davinci'),
                        ClauseEquals(equals='davinci-002'),
                        ClauseEquals(equals='text-davinci'),
                        ClauseEquals(equals='text-davinci-002'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-3.5-turbo'),
                        ClauseEquals(equals='gpt-3.5-turbo-0301'),
                        ClauseEquals(equals='gpt-3.5-turbo-0613'),
                        ClauseEquals(equals='gpt-3.5-turbo-instruct'),
                        ClauseEquals(equals='gpt-3.5-turbo-instruct-0914'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo-0125',
                match=ClauseEquals(equals='gpt-3.5-turbo-0125'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo-1106',
                match=ClauseEquals(equals='gpt-3.5-turbo-1106'),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo-16k-0613',
                match=ClauseEquals(equals='gpt-3.5-turbo-16k-0613'),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('4')),
            ),
            ModelInfo(
                id='gpt-35-16k',
                match=ClauseEquals(equals='gpt-35-16k'),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('4')),
            ),
            ModelInfo(
                id='gpt-35-turbo',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-35-turbo'),
                        ClauseEquals(equals='gpt-35-turbo-0613'),
                        ClauseEquals(equals='gpt-35-turbo-1106'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt-35-turbo-16k',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='gpt-35-turbo-16k'), ClauseEquals(equals='gpt-35-turbo-16k-0613')]
                ),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('4')),
            ),
            ModelInfo(
                id='gpt-4',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4'),
                        ClauseEquals(equals='gpt-4-0314'),
                        ClauseEquals(equals='gpt-4-0613'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('30'), output_mtok=Decimal('60')),
            ),
            ModelInfo(
                id='gpt-4-0125-preview',
                match=ClauseEquals(equals='gpt-4-0125-preview'),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4-1106-preview',
                match=ClauseEquals(equals='gpt-4-1106-preview'),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4-1106-vision-preview',
                match=ClauseEquals(equals='gpt-4-1106-vision-preview'),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4-32k',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4-32k'),
                        ClauseEquals(equals='gpt-4-32k-0314'),
                        ClauseEquals(equals='gpt-4-32k-0613'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('60'), output_mtok=Decimal('120')),
            ),
            ModelInfo(
                id='gpt-4-preview-1106',
                match=ClauseEquals(equals='gpt-4-preview-1106'),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4-turbo',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4-turbo'),
                        ClauseEquals(equals='gpt-4-turbo-0125-preview'),
                        ClauseEquals(equals='gpt-4-turbo-2024-04-09'),
                        ClauseEquals(equals='gpt-4-turbo-preview'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4-vision',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-4-vision'), ClauseEquals(equals='gpt-4-vision-preview')]),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4.1',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-4.1'), ClauseEquals(equals='gpt-4.1-2025-04-14')]),
                prices=ModelPrice(input_mtok=Decimal('2'), cache_read_mtok=Decimal('0.5'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='gpt-4.1-mini',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='gpt-4.1-mini'), ClauseEquals(equals='gpt-4.1-mini-2025-04-14')]
                ),
                prices=ModelPrice(
                    input_mtok=Decimal('0.4'), cache_read_mtok=Decimal('0.1'), output_mtok=Decimal('1.6')
                ),
            ),
            ModelInfo(
                id='gpt-4.1-nano',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='gpt-4.1-nano'), ClauseEquals(equals='gpt-4.1-nano-2025-04-14')]
                ),
                prices=ModelPrice(
                    input_mtok=Decimal('0.1'), cache_read_mtok=Decimal('0.03'), output_mtok=Decimal('0.4')
                ),
            ),
            ModelInfo(
                id='gpt-4o-2024-0513',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-4o-2024-05-13'), ClauseEquals(equals='gpt-4o-2024-0513')]),
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='gpt-4o-2024-08-06',
                match=ClauseEquals(equals='gpt-4o-2024-08-06'),
                prices=ModelPrice(
                    input_mtok=Decimal('2.5'), cache_read_mtok=Decimal('1.25'), output_mtok=Decimal('10')
                ),
            ),
            ModelInfo(
                id='gpt-4o-2024-1120',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-4o-2024-1120'), ClauseEquals(equals='gpt-4o-2024-11-20')]),
                prices=ModelPrice(
                    input_mtok=Decimal('2.5'), cache_read_mtok=Decimal('1.25'), output_mtok=Decimal('10')
                ),
            ),
            ModelInfo(
                id='gpt-4o-mini',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4o-mini'),
                        ClauseEquals(equals='gpt-4o-mini-0718'),
                        ClauseStartsWith(starts_with='gpt-4o-mini-audio'),
                    ]
                ),
                context_window=128000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.15'),
                    cache_read_mtok=Decimal('0.075'),
                    output_mtok=Decimal('0.6'),
                    input_audio_mtok=Decimal('10'),
                    output_audio_mtok=Decimal('20'),
                ),
            ),
            ModelInfo(
                id='gpt-4o-mini-realtime',
                match=ClauseStartsWith(starts_with='gpt-4o-mini-realtime'),
                prices=ModelPrice(
                    input_mtok=Decimal('0.6'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('2.4'),
                    input_audio_mtok=Decimal('10'),
                    cache_audio_read_mtok=Decimal('0.3'),
                    output_audio_mtok=Decimal('20'),
                ),
            ),
            ModelInfo(
                id='gpt-4o-realtime',
                match=ClauseStartsWith(starts_with='gpt-4o-realtime'),
                prices=ModelPrice(
                    input_mtok=Decimal('5'),
                    cache_read_mtok=Decimal('2.5'),
                    output_mtok=Decimal('20'),
                    input_audio_mtok=Decimal('40'),
                    cache_audio_read_mtok=Decimal('2.5'),
                    output_audio_mtok=Decimal('80'),
                ),
            ),
            ModelInfo(
                id='gpt-5',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-5'), ClauseEquals(equals='gpt-5-2025-08-07')]),
                name='GPT-5',
                description="GPT-5 is OpenAI's flagship model for coding, reasoning, and agentic tasks across domains.",
                context_window=400000,
                prices=ModelPrice(
                    input_mtok=Decimal('1.25'), cache_read_mtok=Decimal('0.125'), output_mtok=Decimal('10')
                ),
            ),
            ModelInfo(
                id='gpt-5-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-5-mini'), ClauseEquals(equals='gpt-5-mini-2025-08-07')]),
                name='GPT-5 mini',
                description="GPT-5 mini is a faster, more cost-efficient version of GPT-5. It's great for well-defined tasks and precise prompts.",
                context_window=400000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.25'), cache_read_mtok=Decimal('0.025'), output_mtok=Decimal('2')
                ),
            ),
            ModelInfo(
                id='gpt-5-nano',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-5-nano'), ClauseEquals(equals='gpt-5-nano-2025-04-14')]),
                name='GPT-5 nano',
                description="GPT-5 Nano is OpenAI's fastest, cheapest version of GPT-5. It's great for summarization and classification tasks.",
                context_window=400000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.05'), cache_read_mtok=Decimal('0.005'), output_mtok=Decimal('0.4')
                ),
            ),
            ModelInfo(
                id='gpt35',
                match=ClauseEquals(equals='gpt35'),
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt4-turbo-preview',
                match=ClauseEquals(equals='gpt4-turbo-preview'),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='mai-ds-r1:free',
                match=ClauseEquals(equals='mai-ds-r1:free'),
                name='MAI DS R1 (free)',
                description="MAI-DS-R1 is a post-trained variant of DeepSeek-R1 developed by the Microsoft AI team to improve the model's responsiveness on previously blocked topics while enhancing its safety profile. Built on top of DeepSeek-R1's reasoning foundation, it integrates 110k examples from the Tulu-3 SFT dataset and 350k internally curated multilingual safety-alignment samples. The model retains strong reasoning, coding, and problem-solving capabilities, while unblocking a wide range of prompts previously restricted in R1.",
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='o1',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='o1'),
                        ClauseEquals(equals='o1-2024-12-17'),
                        ClauseEquals(equals='o1-preview'),
                        ClauseEquals(equals='o1-preview-2024-09-12'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('15'), cache_read_mtok=Decimal('7.5'), output_mtok=Decimal('60')),
            ),
            ModelInfo(
                id='o1-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='o1-mini'), ClauseEquals(equals='o1-mini-2024-09-12')]),
                prices=ModelPrice(
                    input_mtok=Decimal('1.1'), cache_read_mtok=Decimal('0.55'), output_mtok=Decimal('4.4')
                ),
            ),
            ModelInfo(
                id='o3-2025-04-16',
                match=ClauseOr(or_=[ClauseEquals(equals='o3'), ClauseEquals(equals='o3-2025-04-16')]),
                prices=ModelPrice(input_mtok=Decimal('2'), cache_read_mtok=Decimal('0.5'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='o3-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='o3-mini'), ClauseEquals(equals='o3-mini-2025-01-31')]),
                prices=ModelPrice(
                    input_mtok=Decimal('1.1'), cache_read_mtok=Decimal('0.55'), output_mtok=Decimal('4.4')
                ),
            ),
            ModelInfo(
                id='o4-mini',
                match=ClauseOr(or_=[ClauseContains(contains='o4-mini'), ClauseContains(contains='o4-mini-2025-04-16')]),
                prices=ModelPrice(
                    input_mtok=Decimal('1.1'), cache_read_mtok=Decimal('0.28'), output_mtok=Decimal('4.4')
                ),
            ),
            ModelInfo(
                id='phi-3-medium-128k-instruct',
                match=ClauseEquals(equals='phi-3-medium-128k-instruct'),
                name='Phi-3 Medium 128K Instruct',
                description='Phi-3 128K Medium is a powerful 14-billion parameter model designed for advanced language understanding, reasoning, and instruction following. Optimized through supervised fine-tuning and preference adjustments, it excels in tasks involving common sense, mathematics, logical reasoning, and code processing.',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='phi-3-mini-128k-instruct',
                match=ClauseEquals(equals='phi-3-mini-128k-instruct'),
                name='Phi-3 Mini 128K Instruct',
                description='Phi-3 Mini is a powerful 3.8B parameter model designed for advanced language understanding, reasoning, and instruction following. Optimized through supervised fine-tuning and preference adjustments, it excels in tasks involving common sense, mathematics, logical reasoning, and code processing.',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='phi-3.5-mini-128k-instruct',
                match=ClauseEquals(equals='phi-3.5-mini-128k-instruct'),
                name='Phi-3.5 Mini 128K Instruct',
                description='Phi-3.5 models are lightweight, state-of-the-art open models. These models were trained with Phi-3 datasets that include both synthetic data and the filtered, publicly available websites data, with a focus on high quality and reasoning-dense properties. Phi-3.5 Mini uses 3.8B parameters, and is a dense decoder-only transformer model using the same tokenizer as Phi-3 Mini.',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='phi-4',
                match=ClauseEquals(equals='phi-4'),
                name='Phi 4',
                description='Microsoft Research Phi-4 is designed to perform well in complex reasoning tasks and can operate efficiently in situations with limited memory or where quick responses are needed.',
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.14')),
            ),
            ModelInfo(
                id='phi-4-multimodal-instruct',
                match=ClauseEquals(equals='phi-4-multimodal-instruct'),
                name='Phi 4 Multimodal Instruct',
                description='Phi-4 Multimodal Instruct is a versatile 5.6B parameter foundation model that combines advanced reasoning and instruction-following capabilities across both text and visual inputs, providing accurate text outputs. The unified architecture enables efficient, low-latency inference, suitable for edge and mobile deployments. Phi-4 Multimodal Instruct supports text inputs in multiple languages including Arabic, Chinese, English, French, German, Japanese, Spanish, and more, with visual input optimized primarily for English. It delivers impressive performance on multimodal tasks involving mathematical, scientific, and document reasoning, providing developers and enterprises a powerful yet compact model for sophisticated interactive applications. For more information, see the Phi-4 Multimodal blog post.',
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='phi-4-reasoning-plus',
                match=ClauseEquals(equals='phi-4-reasoning-plus'),
                name='Phi 4 Reasoning Plus',
                description='Phi-4-reasoning-plus is an enhanced 14B parameter model from Microsoft, fine-tuned from Phi-4 with additional reinforcement learning to boost accuracy on math, science, and code reasoning tasks. It uses the same dense decoder-only transformer architecture as Phi-4, but generates longer, more comprehensive outputs structured into a step-by-step reasoning trace and final answer.',
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.35')),
            ),
            ModelInfo(
                id='phi-4-reasoning-plus:free',
                match=ClauseEquals(equals='phi-4-reasoning-plus:free'),
                name='Phi 4 Reasoning Plus (free)',
                description='Phi-4-reasoning-plus is an enhanced 14B parameter model from Microsoft, fine-tuned from Phi-4 with additional reinforcement learning to boost accuracy on math, science, and code reasoning tasks. It uses the same dense decoder-only transformer architecture as Phi-4, but generates longer, more comprehensive outputs structured into a step-by-step reasoning trace and final answer.',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='phi-4-reasoning:free',
                match=ClauseEquals(equals='phi-4-reasoning:free'),
                name='Phi 4 Reasoning (free)',
                description='Phi-4-reasoning is a 14B parameter dense decoder-only transformer developed by Microsoft, fine-tuned from Phi-4 to enhance complex reasoning capabilities. It uses a combination of supervised fine-tuning on chain-of-thought traces and reinforcement learning, targeting math, science, and code reasoning tasks. With a 32k context window and high inference efficiency, it is optimized for structured responses in a two-part format: reasoning trace followed by a final solution.',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='text-embedding-3-large',
                match=ClauseEquals(equals='text-embedding-3-large'),
                prices=ModelPrice(input_mtok=Decimal('0.13')),
            ),
            ModelInfo(
                id='text-embedding-3-small',
                match=ClauseEquals(equals='text-embedding-3-small'),
                prices=ModelPrice(input_mtok=Decimal('0.02')),
            ),
            ModelInfo(
                id='wizardlm-2-8x22b',
                match=ClauseEquals(equals='wizardlm-2-8x22b'),
                name='WizardLM-2 8x22B',
                description="WizardLM-2 8x22B is Microsoft AI's most advanced Wizard model. It demonstrates highly competitive performance compared to leading proprietary models, and it consistently outperforms all existing state-of-the-art opensource models.",
                prices=ModelPrice(input_mtok=Decimal('0.48'), output_mtok=Decimal('0.48')),
            ),
        ],
    ),
    Provider(
        id='cohere',
        name='Cohere',
        api_pattern='https://api\\.cohere\\.ai',
        pricing_urls=['https://cohere.com/pricing'],
        model_match=ClauseStartsWith(starts_with='command-'),
        models=[
            ModelInfo(
                id='command',
                match=ClauseEquals(equals='command'),
                name='Command',
                description='Command is an instruction-following conversational model that performs language tasks with high quality, more reliably and with a longer context than our base generative models.',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='command-a',
                match=ClauseEquals(equals='command-a'),
                name='Command A',
                description='Command A is an open-weights 111B parameter model with a 256k context window focused on delivering great performance across agentic, multilingual, and coding use cases.\nCompared to other leading proprietary and open-weights models Command A delivers maximum performance with minimum hardware costs, excelling on business-critical agentic and multilingual tasks.',
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='command-r',
                match=ClauseOr(or_=[ClauseEquals(equals='command-r'), ClauseEquals(equals='command-r-08-2024')]),
                name='Command R',
                description='Command-R is a 35B parameter model that performs conversational language tasks at a higher quality, more reliably, and with a longer context than previous models. It can be used for complex workflows like code generation, retrieval augmented generation (RAG), tool use, and agents.',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='command-r-plus',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='command-r-plus'), ClauseEquals(equals='command-r-plus-08-2024')]
                ),
                name='Command R+',
                description="Command R+ is a new, 104B-parameter LLM from Cohere. It's useful for roleplay, general consumer usecases, and Retrieval Augmented Generation (RAG).",
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='command-r7b',
                match=ClauseOr(or_=[ClauseEquals(equals='command-r7b'), ClauseEquals(equals='command-r7b-12-2024')]),
                name='Command R7B',
                description='Command R7B (12-2024) is a small, fast update of the Command R+ model, delivered in December 2024. It excels at RAG, tool use, agents, and similar tasks requiring complex reasoning and multiple steps.',
                prices=ModelPrice(input_mtok=Decimal('0.0375'), output_mtok=Decimal('0.15')),
            ),
        ],
    ),
    Provider(
        id='deepseek',
        name='Deepseek',
        api_pattern='https://api\\.deepseek\\.com',
        pricing_urls=['https://api-docs.deepseek.com/quick_start/pricing'],
        price_comments='Deepseek off-peak pricing applies "UTC 16:30-00:30" so we switch it around and use the off-peak pricing as the default (first) price then the second price with a constraint is the "standard" pricing that applies "UTC 00:30-16:30".',
        model_match=ClauseContains(contains='deepseek'),
        models=[
            ModelInfo(
                id='deepseek-chat',
                match=ClauseOr(
                    or_=[ClauseStartsWith(starts_with='deepseek-chat'), ClauseEquals(equals='deepseek-chat-v3-0324')]
                ),
                name='DeepSeek Chat',
                description='DeepSeek-V3 is the latest model from the DeepSeek team, building upon the instruction following and coding abilities of the previous versions. Pre-trained on nearly 15 trillion tokens, the reported evaluations reveal that the model outperforms other open-source models and rivals leading closed-source models.',
                context_window=64000,
                prices=[
                    ConditionalPrice(
                        prices=ModelPrice(
                            input_mtok=Decimal('0.135'), cache_read_mtok=Decimal('0.035'), output_mtok=Decimal('0.55')
                        )
                    ),
                    ConditionalPrice(
                        constraint=TimeOfDateConstraint(
                            start_time=datetime.time(0, 30, tzinfo=datetime.timezone.utc),
                            end_time=datetime.time(16, 30, tzinfo=datetime.timezone.utc),
                        ),
                        prices=ModelPrice(
                            input_mtok=Decimal('0.27'), cache_read_mtok=Decimal('0.07'), output_mtok=Decimal('1.1')
                        ),
                    ),
                ],
            ),
            ModelInfo(
                id='deepseek-reasoner',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='deepseek-reasoner'),
                        ClauseStartsWith(starts_with='deepseek-r1'),
                        ClauseEquals(equals='deepseek-r1-0528'),
                    ]
                ),
                name='Deepseek R1',
                description="DeepSeek R1 is here: Performance on par with OpenAI o1, but open-sourced and with fully open reasoning tokens. It's 671B parameters in size, with 37B active in an inference pass.",
                context_window=64000,
                prices=[
                    ConditionalPrice(
                        prices=ModelPrice(
                            input_mtok=Decimal('0.135'), cache_read_mtok=Decimal('0.035'), output_mtok=Decimal('0.55')
                        )
                    ),
                    ConditionalPrice(
                        constraint=TimeOfDateConstraint(
                            start_time=datetime.time(0, 30, tzinfo=datetime.timezone.utc),
                            end_time=datetime.time(16, 30, tzinfo=datetime.timezone.utc),
                        ),
                        prices=ModelPrice(
                            input_mtok=Decimal('0.55'), cache_read_mtok=Decimal('0.14'), output_mtok=Decimal('2.19')
                        ),
                    ),
                ],
            ),
        ],
    ),
    Provider(
        id='fireworks',
        name='Fireworks',
        api_pattern='https://api\\.fireworks\\.ai',
        pricing_urls=['https://fireworks.ai/pricing'],
        model_match=ClauseStartsWith(starts_with='accounts/fireworks/models/'),
        models=[
            ModelInfo(
                id='deepseek-r1-0528',
                match=ClauseEquals(equals='accounts/fireworks/models/deepseek-r1-0528'),
                name='DeepSeek R1 0528',
                description='The updated DeepSeek-R1-0528 model delivers major improvements in reasoning, inference, and accuracy through enhanced post-training optimization and greater computational resources. It now performs at a level approaching top-tier models like O3 and Gemini 2.5 Pro, with notable gains in complex tasks such as math and programming.',
                context_window=160000,
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='deepseek-v3-0324',
                match=ClauseEquals(equals='accounts/fireworks/models/deepseek-v3-0324'),
                name='Deepseek V3 03-24',
                description='A strong Mixture-of-Experts (MoE) language model with 671B total parameters with 37B activated for each token from Deepseek. Updated checkpoint.',
                context_window=160000,
                price_comments='docs give just one price - "Pricing Per 1M Tokens", we assume that\'s input and output',
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='gemma-3-27b-it',
                match=ClauseEquals(equals='accounts/fireworks/models/gemma-3-27b-it'),
                name='Gemma 3 27B Instruct',
                context_window=131000,
                price_comments='docs give just one price - "Pricing Per 1M Tokens", we assume that\'s input and output',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='llama-v3p1-8b-instruct',
                match=ClauseEquals(equals='accounts/fireworks/models/llama-v3p1-8b-instruct'),
                name='Llama 3.1 8B Instruct',
                description='The Meta Llama 3.1 collection of multilingual large language models (LLMs) is a collection of pretrained and instruction tuned generative models in 8B, 70B and 405B sizes. The Llama 3.1 instruction tuned text only models (8B, 70B, 405B) are optimized for multilingual dialogue use cases and outperform many of the available open source and closed chat models on common industry benchmarks.',
                context_window=131000,
                price_comments='docs give just one price - "Pricing Per 1M Tokens", we assume that\'s input and output',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='llama4-maverick-instruct-basic',
                match=ClauseEquals(equals='accounts/fireworks/models/llama4-maverick-instruct-basic'),
                name='Llama 4 Maverick Instruct (Basic)',
                description='The Meta Llama 3.1 collection of multilingual large language models (LLMs) is a collection of pretrained and instruction tuned generative models in 8B, 70B and 405B sizes. The Llama 3.1 instruction tuned text only models (8B, 70B, 405B) are optimized for multilingual dialogue use cases and outperform many of the available open source and closed chat models on common industry benchmarks.',
                context_window=1000000,
                prices=ModelPrice(input_mtok=Decimal('0.22'), output_mtok=Decimal('0.88')),
            ),
            ModelInfo(
                id='qwen2p5-vl-72b-instruct',
                match=ClauseEquals(equals='accounts/fireworks/models/qwen2p5-vl-72b-instruct'),
                name='Qwen2.5-VL 72B Instruct',
                description="Latest Qwen's VLM model",
                context_window=128000,
                price_comments='docs give just one price - "Pricing Per 1M Tokens", we assume that\'s input and output',
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='qwen3-235b-a22b',
                match=ClauseEquals(equals='accounts/fireworks/models/qwen3-235b-a22b'),
                name='Qwen3 235B-A22B',
                description='Qwen3 is the latest evolution in the Qwen LLM series, featuring both dense and MoE models with major advancements in reasoning, agent capabilities, multilingual support, and instruction following. It uniquely allows seamless switching between "thinking" (for complex logic, math, coding) and "non-thinking" modes (for fast, general dialogue), delivering strong performance across tasks.',
                context_window=128000,
                prices=ModelPrice(input_mtok=Decimal('0.22'), output_mtok=Decimal('0.88')),
            ),
        ],
    ),
    Provider(
        id='google',
        name='Google',
        api_pattern='https://(.*\\.)?googleapis\\.com',
        pricing_urls=[
            'https://ai.google.dev/gemini-api/docs/pricing',
            'https://cloud.google.com/vertex-ai/generative-ai/pricing',
        ],
        model_match=ClauseContains(contains='gemini'),
        provider_match=ClauseOr(or_=[ClauseContains(contains='google'), ClauseEquals(equals='gemini')]),
        models=[
            ModelInfo(
                id='claude-3-5-haiku',
                match=ClauseContains(contains='claude-3-5-haiku'),
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.8'),
                    cache_write_mtok=Decimal('1'),
                    cache_read_mtok=Decimal('0.08'),
                    output_mtok=Decimal('4'),
                ),
            ),
            ModelInfo(
                id='claude-3-5-sonnet',
                match=ClauseContains(contains='claude-3-5-sonnet'),
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='claude-3-7-sonnet',
                match=ClauseContains(contains='claude-3-7-sonnet'),
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='claude-3-haiku',
                match=ClauseContains(contains='claude-3-haiku'),
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.25'),
                    cache_write_mtok=Decimal('0.3'),
                    cache_read_mtok=Decimal('0.03'),
                    output_mtok=Decimal('1.25'),
                ),
            ),
            ModelInfo(
                id='claude-3-opus',
                match=ClauseContains(contains='claude-3-opus'),
                prices=ModelPrice(
                    input_mtok=Decimal('15'),
                    cache_write_mtok=Decimal('18.75'),
                    cache_read_mtok=Decimal('1.5'),
                    output_mtok=Decimal('75'),
                ),
            ),
            ModelInfo(
                id='claude-4-opus',
                match=ClauseOr(
                    or_=[ClauseContains(contains='claude-4-opus'), ClauseContains(contains='claude-opus-4')]
                ),
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('15'),
                    cache_write_mtok=Decimal('18.75'),
                    cache_read_mtok=Decimal('1.5'),
                    output_mtok=Decimal('75'),
                ),
            ),
            ModelInfo(
                id='claude-4-sonnet',
                match=ClauseOr(
                    or_=[ClauseContains(contains='claude-4-sonnet'), ClauseContains(contains='claude-sonnet-4')]
                ),
                context_window=200000,
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='gemini-1.0-pro-vision-001',
                match=ClauseEquals(equals='gemini-1.0-pro-vision-001'),
                name='gemini 1.0 pro vision',
                description="Google's first-generation advanced multimodal model that can understand text, code, and images. It provides strong reasoning capabilities and follows instructions effectively.",
                context_window=32768,
                price_comments="I can't find anything about this model or it's pricing, so trusting the original source",
                prices=ModelPrice(input_mtok=Decimal('0.125'), output_mtok=Decimal('0.375')),
            ),
            ModelInfo(
                id='gemini-1.5-flash',
                match=ClauseContains(contains='gemini-1.5-flash'),
                name='gemini 1.5 flash',
                description='A faster, more cost-effective variant of Gemini 1.5 that maintains strong capabilities while optimizing for performance and cost efficiency. Suitable for production deployments requiring high throughput.',
                context_window=1000000,
                prices=ModelPrice(
                    input_mtok=TieredPrices(base=Decimal('0.075'), tiers=[Tier(start=128000, price=Decimal('0.15'))]),
                    cache_read_mtok=TieredPrices(
                        base=Decimal('0.01875'), tiers=[Tier(start=128000, price=Decimal('0.0375'))]
                    ),
                    output_mtok=TieredPrices(base=Decimal('0.3'), tiers=[Tier(start=128000, price=Decimal('0.6'))]),
                ),
            ),
            ModelInfo(
                id='gemini-1.5-pro',
                match=ClauseContains(contains='gemini-1.5-pro'),
                name='gemini 1.5 Pro',
                description="Google's most capable multimodal model with an extremely long context window of up to 1 million tokens. It excels at complex reasoning, long-form content processing, and multimodal understanding.",
                context_window=1000000,
                prices=ModelPrice(
                    input_mtok=TieredPrices(base=Decimal('1.25'), tiers=[Tier(start=128000, price=Decimal('2.5'))]),
                    output_mtok=TieredPrices(base=Decimal('5'), tiers=[Tier(start=128000, price=Decimal('10'))]),
                ),
            ),
            ModelInfo(
                id='gemini-2.0-flash',
                match=ClauseOr(
                    or_=[
                        ClauseContains(contains='gemini-2.0-flash'),
                        ClauseEquals(equals='gemini-2.0-flash-lite-001'),
                        ClauseEquals(equals='gemini-2.0-flash-001'),
                        ClauseEquals(equals='gemini-2.0-flash-exp:free'),
                    ]
                ),
                name='gemini 2.0 flash',
                description="The newest generation of Google's Gemini models, featuring improved reasoning, instruction following, and factual accuracy, with the Flash variant optimized for cost-efficiency and performance.",
                context_window=1000000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.1'),
                    cache_read_mtok=TieredPrices(
                        base=Decimal('0.025'), tiers=[Tier(start=1000000, price=Decimal('0.175'))]
                    ),
                    output_mtok=Decimal('0.4'),
                    input_audio_mtok=Decimal('0.7'),
                ),
            ),
            ModelInfo(
                id='gemini-2.0-flash-lite',
                match=ClauseContains(contains='gemini-2.0-flash-lite'),
                name='gemini 2.0 flash lite',
                description='A lighter, more cost-effective version of Gemini 2.0 Flash, designed for applications requiring high efficiency while maintaining good performance. Ideal for high-volume, cost-sensitive deployments.',
                context_window=1000000,
                prices=ModelPrice(input_mtok=Decimal('0.075'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='gemini-2.5-flash',
                match=ClauseEquals(equals='gemini-2.5-flash'),
                name='Gemini 2.5 Flash',
                description='Gemini 2.5 Flash is Google\'s state-of-the-art workhorse model, specifically designed for advanced reasoning, coding, mathematics, and scientific tasks. It includes built-in "thinking" capabilities, enabling it to provide responses with greater accuracy and nuanced context handling.',
                prices=ModelPrice(
                    input_mtok=Decimal('0.3'),
                    cache_read_mtok=Decimal('0.075'),
                    output_mtok=Decimal('2.5'),
                    input_audio_mtok=Decimal('1'),
                    cache_audio_read_mtok=Decimal('0.25'),
                ),
            ),
            ModelInfo(
                id='gemini-2.5-flash-lite',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gemini-2.5-flash-lite'),
                        ClauseStartsWith(starts_with='gemini-2.5-flash-lite-preview'),
                    ]
                ),
                name='Gemini 2.5 Flash Lite',
                description='Gemini 2.5 Flash-Lite is a lightweight reasoning model in the Gemini 2.5 family, optimized for ultra-low latency and cost efficiency. It offers improved throughput, faster token generation, and better performance across common benchmarks compared to earlier Flash models. By default, "thinking" (i.e. multi-pass reasoning) is disabled to prioritize speed, but developers can enable it via the Reasoning API parameter to selectively trade off cost for intelligence.',
                context_window=1000000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.1'),
                    cache_read_mtok=Decimal('0.025'),
                    output_mtok=Decimal('0.4'),
                    input_audio_mtok=Decimal('0.5'),
                    cache_audio_read_mtok=Decimal('0.125'),
                ),
            ),
            ModelInfo(
                id='gemini-2.5-flash-preview',
                match=ClauseOr(
                    or_=[
                        ClauseContains(contains='gemini-2.5-flash-preview'),
                        ClauseEquals(equals='gemini-2.5-flash-preview-05-20'),
                        ClauseEquals(equals='gemini-2.5-flash-preview-05-20:thinking'),
                        ClauseEquals(equals='gemini-2.5-flash-preview'),
                        ClauseEquals(equals='gemini-2.5-flash-preview:thinking'),
                    ]
                ),
                name='Gemini 2.5 Flash Preview 05-20',
                description='Gemini 2.5 Flash May 20th Checkpoint is Google\'s state-of-the-art workhorse model, specifically designed for advanced reasoning, coding, mathematics, and scientific tasks. It includes built-in "thinking" capabilities, enabling it to provide responses with greater accuracy and nuanced context handling.',
                price_comments='from https://cloud.google.com/vertex-ai/generative-ai/pricing should be retired 2025-07-15',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='gemini-2.5-pro',
                match=ClauseStartsWith(starts_with='gemini-2.5-pro'),
                name='Gemini 2.5 Pro',
                description='Gemini 2.5 Pro is Google\'s state-of-the-art AI model designed for advanced reasoning, coding, mathematics, and scientific tasks. It employs "thinking" capabilities, enabling it to reason through responses with enhanced accuracy and nuanced context handling. Gemini 2.5 Pro achieves top-tier performance on multiple benchmarks, including first-place positioning on the LMArena leaderboard, reflecting superior human-preference alignment and complex problem-solving abilities.',
                price_comments='See https://ai.google.dev/gemini-api/docs/pricing#gemini-2.5-pro',
                prices=ModelPrice(
                    input_mtok=TieredPrices(base=Decimal('1.25'), tiers=[Tier(start=200000, price=Decimal('2.5'))]),
                    cache_read_mtok=TieredPrices(
                        base=Decimal('0.31'), tiers=[Tier(start=200000, price=Decimal('0.625'))]
                    ),
                    output_mtok=TieredPrices(base=Decimal('10'), tiers=[Tier(start=200000, price=Decimal('15'))]),
                ),
            ),
            ModelInfo(
                id='gemini-flash-1.5',
                match=ClauseEquals(equals='gemini-flash-1.5'),
                name='Gemini 1.5 Flash',
                description="Gemini 1.5 Flash is a foundation model that performs well at a variety of multimodal tasks such as visual understanding, classification, summarization, and creating content from image, audio and video. It's adept at processing visual and text inputs such as photographs, documents, infographics, and screenshots.",
                price_comments='See https://ai.google.dev/gemini-api/docs/pricing#gemini-1.5-flash',
                prices=ModelPrice(
                    input_mtok=TieredPrices(base=Decimal('0.075'), tiers=[Tier(start=128000, price=Decimal('0.15'))]),
                    cache_read_mtok=TieredPrices(
                        base=Decimal('0.01875'), tiers=[Tier(start=128000, price=Decimal('0.0375'))]
                    ),
                    output_mtok=TieredPrices(base=Decimal('0.3'), tiers=[Tier(start=128000, price=Decimal('0.6'))]),
                ),
            ),
            ModelInfo(
                id='gemini-flash-1.5-8b',
                match=ClauseEquals(equals='gemini-flash-1.5-8b'),
                name='gemini 1.5 flash',
                description='A faster, more cost-effective variant of Gemini 1.5 that maintains strong capabilities while optimizing for performance and cost efficiency. Suitable for production deployments requiring high throughput.',
                context_window=1000000,
                price_comments='See https://ai.google.dev/gemini-api/docs/pricing#gemini-1.5-flash-8b',
                prices=ModelPrice(
                    input_mtok=TieredPrices(base=Decimal('0.0375'), tiers=[Tier(start=128000, price=Decimal('0.075'))]),
                    cache_read_mtok=TieredPrices(
                        base=Decimal('0.01'), tiers=[Tier(start=128000, price=Decimal('0.02'))]
                    ),
                    output_mtok=TieredPrices(base=Decimal('0.15'), tiers=[Tier(start=128000, price=Decimal('0.3'))]),
                ),
            ),
            ModelInfo(
                id='gemini-pro',
                match=ClauseOr(or_=[ClauseEquals(equals='gemini-pro'), ClauseEquals(equals='gemini-1.0-pro')]),
                name='gemini 1.0 pro',
                description="Google's first-generation advanced multimodal model that can understand text, code, and images. It provides strong reasoning capabilities and follows instructions effectively.",
                context_window=32768,
                price_comments="I can't find anything so trusting these prices, not sure the model still exists",
                prices=ModelPrice(input_mtok=Decimal('0.125'), output_mtok=Decimal('0.375')),
            ),
            ModelInfo(
                id='gemini-pro-1.5',
                match=ClauseEquals(equals='gemini-pro-1.5'),
                name='Gemini 1.5 Pro',
                description="Google's latest multimodal model, supports image and video[0] in text or chat prompts.",
                context_window=2000000,
                price_comments='See https://ai.google.dev/gemini-api/docs/pricing#gemini-1.5-pro',
                prices=ModelPrice(
                    input_mtok=TieredPrices(base=Decimal('1.25'), tiers=[Tier(start=128000, price=Decimal('2.5'))]),
                    cache_read_mtok=TieredPrices(
                        base=Decimal('0.3125'), tiers=[Tier(start=128000, price=Decimal('0.625'))]
                    ),
                    output_mtok=TieredPrices(base=Decimal('5'), tiers=[Tier(start=128000, price=Decimal('10'))]),
                ),
            ),
            ModelInfo(
                id='gemma-2-27b-it',
                match=ClauseEquals(equals='gemma-2-27b-it'),
                name='Gemma 2 27B',
                description='Gemma 2 27B by Google is an open model built from the same research and technology used to create the Gemini models.',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='gemma-2-9b-it',
                match=ClauseEquals(equals='gemma-2-9b-it'),
                name='Gemma 2 9B',
                description='Gemma 2 9B by Google is an advanced, open-source language model that sets a new standard for efficiency and performance in its size class.',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='gemma-2-9b-it:free',
                match=ClauseEquals(equals='gemma-2-9b-it:free'),
                name='Gemma 2 9B (free)',
                description='Gemma 2 9B by Google is an advanced, open-source language model that sets a new standard for efficiency and performance in its size class.',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemma-3-12b-it',
                match=ClauseEquals(equals='gemma-3-12b-it'),
                name='Gemma 3 12B',
                description='Gemma 3 introduces multimodality, supporting vision-language input and text outputs. It handles context windows up to 128k tokens, understands over 140 languages, and offers improved math, reasoning, and chat capabilities, including structured outputs and function calling. Gemma 3 12B is the second largest in the family of Gemma 3 models after Gemma 3 27B',
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='gemma-3-12b-it:free',
                match=ClauseEquals(equals='gemma-3-12b-it:free'),
                name='Gemma 3 12B (free)',
                description='Gemma 3 introduces multimodality, supporting vision-language input and text outputs. It handles context windows up to 128k tokens, understands over 140 languages, and offers improved math, reasoning, and chat capabilities, including structured outputs and function calling. Gemma 3 12B is the second largest in the family of Gemma 3 models after Gemma 3 27B',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemma-3-27b-it',
                match=ClauseEquals(equals='gemma-3-27b-it'),
                name='Gemma 3 27B',
                description="Gemma 3 introduces multimodality, supporting vision-language input and text outputs. It handles context windows up to 128k tokens, understands over 140 languages, and offers improved math, reasoning, and chat capabilities, including structured outputs and function calling. Gemma 3 27B is Google's latest open source model, successor to Gemma 2",
                price_comments="I can't find anything so trusting these prices, not sure the model still exists on Vertex or GCP",
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='gemma-3-27b-it:free',
                match=ClauseEquals(equals='gemma-3-27b-it:free'),
                name='Gemma 3 27B (free)',
                description="Gemma 3 introduces multimodality, supporting vision-language input and text outputs. It handles context windows up to 128k tokens, understands over 140 languages, and offers improved math, reasoning, and chat capabilities, including structured outputs and function calling. Gemma 3 27B is Google's latest open source model, successor to Gemma 2",
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemma-3-4b-it',
                match=ClauseEquals(equals='gemma-3-4b-it'),
                name='Gemma 3 4B',
                description='Gemma 3 introduces multimodality, supporting vision-language input and text outputs. It handles context windows up to 128k tokens, understands over 140 languages, and offers improved math, reasoning, and chat capabilities, including structured outputs and function calling.',
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='gemma-3-4b-it:free',
                match=ClauseEquals(equals='gemma-3-4b-it:free'),
                name='Gemma 3 4B (free)',
                description='Gemma 3 introduces multimodality, supporting vision-language input and text outputs. It handles context windows up to 128k tokens, understands over 140 languages, and offers improved math, reasoning, and chat capabilities, including structured outputs and function calling.',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemma-3n-e4b-it:free',
                match=ClauseEquals(equals='gemma-3n-e4b-it:free'),
                name='Gemma 3n 4B (free)',
                description='Gemma 3n E4B-it is optimized for efficient execution on mobile and low-resource devices, such as phones, laptops, and tablets. It supports multimodal inputs—including text, visual data, and audio—enabling diverse tasks such as text generation, speech recognition, translation, and image analysis. Leveraging innovations like Per-Layer Embedding (PLE) caching and the MatFormer architecture, Gemma 3n dynamically manages memory usage and computational load by selectively activating model parameters, significantly reducing runtime resource requirements.',
                prices=ModelPrice(),
            ),
        ],
    ),
    Provider(
        id='groq',
        name='Groq',
        api_pattern='https://api\\.groq\\.com',
        pricing_urls=['https://groq.com/pricing/'],
        models=[
            ModelInfo(
                id='gemma-7b-it',
                match=ClauseEquals(equals='gemma-7b-it'),
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.07')),
            ),
            ModelInfo(
                id='gemma2-9b-it',
                match=ClauseEquals(equals='gemma2-9b-it'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='llama2-70b-4096',
                match=ClauseEquals(equals='llama2-70b-4096'),
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='llama3-70b-8192',
                match=ClauseEquals(equals='llama3-70b-8192'),
                prices=ModelPrice(input_mtok=Decimal('0.59'), output_mtok=Decimal('0.79')),
            ),
            ModelInfo(
                id='llama3-8b-8192',
                match=ClauseEquals(equals='llama3-8b-8192'),
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.08')),
            ),
            ModelInfo(
                id='llama3-groq-70b-8192-tool-use-preview',
                match=ClauseEquals(equals='llama3-groq-70b-8192-tool-use-preview'),
                prices=ModelPrice(input_mtok=Decimal('0.89'), output_mtok=Decimal('0.89')),
            ),
            ModelInfo(
                id='llama3-groq-8b-8192-tool-use-preview',
                match=ClauseEquals(equals='llama3-groq-8b-8192-tool-use-preview'),
                prices=ModelPrice(input_mtok=Decimal('0.19'), output_mtok=Decimal('0.19')),
            ),
            ModelInfo(
                id='mixtral-8x7b-32768',
                match=ClauseEquals(equals='mixtral-8x7b-32768'),
                prices=ModelPrice(input_mtok=Decimal('0.24'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='openai/gpt-oss-120b',
                match=ClauseEquals(equals='openai/gpt-oss-120b'),
                description="GPT-OSS 120B is OpenAI's flagship open source model, built on a Mixture-of-Experts (MoE) architecture with\n120 billion parameters and 128 experts.\n",
                context_window=131072,
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.75')),
            ),
            ModelInfo(
                id='openai/gpt-oss-20b',
                match=ClauseEquals(equals='openai/gpt-oss-20b'),
                description="GPT-OSS 20B is OpenAI's flagship open source model, built on a Mixture-of-Experts (MoE) architecture with\n20 billion parameters and 32 experts.\n",
                context_window=131072,
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.5')),
            ),
        ],
    ),
    Provider(
        id='mistral',
        name='Mistral',
        api_pattern='https://api\\.mistral\\.ai',
        pricing_urls=['https://mistral.ai/pricing#api-pricing'],
        model_match=ClauseRegex(regex='(?:mi|code|dev|magi|mini)stral'),
        provider_match=ClauseStartsWith(starts_with='mistral'),
        models=[
            ModelInfo(
                id='codestral',
                match=ClauseOr(or_=[ClauseEquals(equals='codestral-latest'), ClauseEquals(equals='codestral-2501')]),
                name='Codestral',
                description="Mistral's cutting-edge language model for coding. Codestral specializes in low-latency, high-frequency tasks such as fill-in-the-middle (FIM), code correction and test generation.",
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='devstral-small',
                match=ClauseEquals(equals='devstral-small'),
                name='Devstral Small',
                description='Devstral-Small-2505 is a 24B parameter agentic LLM fine-tuned from Mistral-Small-3.1, jointly developed by Mistral AI and All Hands AI for advanced software engineering tasks. It is optimized for codebase exploration, multi-file editing, and integration into coding agents, achieving state-of-the-art results on SWE-Bench Verified (46.8%).',
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.12')),
            ),
            ModelInfo(
                id='devstral-small:free',
                match=ClauseEquals(equals='devstral-small:free'),
                name='Devstral Small (free)',
                description='Devstral-Small-2505 is a 24B parameter agentic LLM fine-tuned from Mistral-Small-3.1, jointly developed by Mistral AI and All Hands AI for advanced software engineering tasks. It is optimized for codebase exploration, multi-file editing, and integration into coding agents, achieving state-of-the-art results on SWE-Bench Verified (46.8%).',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='magistral-medium-2506',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='magistral-medium-2506'),
                        ClauseEquals(equals='magistral-medium-2506:thinking'),
                    ]
                ),
                name='Magistral Medium 2506',
                description="Magistral is Mistral's first reasoning model. It is ideal for general purpose use requiring longer thought processing and better accuracy than with non-reasoning LLMs. From legal research and financial forecasting to software development and creative storytelling — this model solves multi-step challenges where transparency and precision are critical.",
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('5')),
            ),
            ModelInfo(
                id='magistral-small-2506',
                match=ClauseEquals(equals='magistral-small-2506'),
                name='Magistral Small 2506',
                description='Magistral Small is a 24B parameter instruction-tuned model based on Mistral-Small-3.1 (2503), enhanced through supervised fine-tuning on traces from Magistral Medium and further refined via reinforcement learning. It is optimized for reasoning and supports a wide multilingual range, including over 20 languages.',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='ministral-3b',
                match=ClauseEquals(equals='ministral-3b'),
                name='Ministral 3B',
                description="Ministral 3B is a 3B parameter model optimized for on-device and edge computing. It excels in knowledge, commonsense reasoning, and function-calling, outperforming larger models like Mistral 7B on most benchmarks. Supporting up to 128k context length, it's ideal for orchestrating agentic workflows and specialist tasks with efficient inference.",
                prices=ModelPrice(input_mtok=Decimal('0.04'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='ministral-8b',
                match=ClauseEquals(equals='ministral-8b'),
                name='Ministral 8B',
                description='Ministral 8B is an 8B parameter model featuring a unique interleaved sliding-window attention pattern for faster, memory-efficient inference. Designed for edge use cases, it supports up to 128k context length and excels in knowledge and reasoning tasks. It outperforms peers in the sub-10B category, making it perfect for low-latency, privacy-first applications.',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='mistral-7b-instruct',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='mistral-7b-instruct'), ClauseEquals(equals='mistral-7b-instruct-v0.3')]
                ),
                name='Mistral 7B Instruct',
                description='A high-performing, industry-standard 7.3B parameter model, with optimizations for speed and context length.',
                prices=ModelPrice(input_mtok=Decimal('0.028'), output_mtok=Decimal('0.054')),
            ),
            ModelInfo(
                id='mistral-7b-instruct-v0.1',
                match=ClauseEquals(equals='mistral-7b-instruct-v0.1'),
                name='Mistral 7B Instruct v0.1',
                description='A 7.3B parameter model that outperforms Llama 2 13B on all benchmarks, with optimizations for speed and context length.',
                prices=ModelPrice(input_mtok=Decimal('0.11'), output_mtok=Decimal('0.19')),
            ),
            ModelInfo(
                id='mistral-7b-instruct-v0.2',
                match=ClauseEquals(equals='mistral-7b-instruct-v0.2'),
                name='Mistral 7B Instruct v0.2',
                description='A high-performing, industry-standard 7.3B parameter model, with optimizations for speed and context length.',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='mistral-7b-instruct:free',
                match=ClauseEquals(equals='mistral-7b-instruct:free'),
                name='Mistral 7B Instruct (free)',
                description='A high-performing, industry-standard 7.3B parameter model, with optimizations for speed and context length.',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistral-embed',
                match=ClauseEquals(equals='mistral-embed'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='mistral-large',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='mistral-large'),
                        ClauseEquals(equals='mistral-large-latest'),
                        ClauseEquals(equals='mistral-large-2407'),
                        ClauseEquals(equals='mistral-large-2411'),
                    ]
                ),
                name='Mistral Large',
                description="This is Mistral AI's flagship model, Mistral Large 2 (version `mistral-large-2407`). It's a proprietary weights-available model and excels at reasoning, code, JSON, chat, and more. Read the launch announcement here.",
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='mistral-medium-3',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='mistral-medium-3'), ClauseEquals(equals='mistral-medium-latest')]
                ),
                name='Mistral Medium 3',
                description='Mistral Medium 3 is a high-performance enterprise-grade language model designed to deliver frontier-level capabilities at significantly reduced operational cost. It balances state-of-the-art reasoning and multimodal performance with 8× lower cost compared to traditional large models, making it suitable for scalable deployments across professional and industrial use cases.',
                prices=ModelPrice(input_mtok=Decimal('0.4'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='mistral-nemo',
                match=ClauseEquals(equals='mistral-nemo'),
                name='Mistral NeMo',
                description='A 12B parameter model with a 128k token context length built by Mistral in collaboration with NVIDIA.',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='mistral-nemo:free',
                match=ClauseEquals(equals='mistral-nemo:free'),
                name='Mistral Nemo (free)',
                description='A 12B parameter model with a 128k token context length built by Mistral in collaboration with NVIDIA.',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistral-saba',
                match=ClauseOr(or_=[ClauseEquals(equals='mistral-saba'), ClauseEquals(equals='mistral-saba-latest')]),
                name='Mistral Saba',
                description='Mistral Saba is a 24B-parameter language model specifically designed for the Middle East and South Asia, delivering accurate and contextually relevant responses while maintaining efficient performance. Trained on curated regional datasets, it supports multiple Indian-origin languages—including Tamil and Malayalam—alongside Arabic. This makes it a versatile option for a range of regional and multilingual applications. Read more at the blog post here',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='mistral-small',
                match=ClauseEquals(equals='mistral-small'),
                name='Mistral Small',
                description='With 22 billion parameters, Mistral Small v24.09 offers a convenient mid-point between (Mistral NeMo 12B)[/mistralai/mistral-nemo] and (Mistral Large 2)[/mistralai/mistral-large], providing a cost-effective solution that can be deployed across various platforms and environments. It has better reasoning, exhibits more capabilities, can produce and reason about code, and is multiligual, supporting English, French, German, Italian, and Spanish.',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='mistral-small-24b-instruct-2501',
                match=ClauseEquals(equals='mistral-small-24b-instruct-2501'),
                name='Mistral Small 3',
                description='Mistral Small 3 is a 24B-parameter language model optimized for low-latency performance across common AI tasks. Released under the Apache 2.0 license, it features both pre-trained and instruction-tuned versions designed for efficient local deployment.',
                price_comments="Can't find pricing on this model, so just trusting open router",
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.08')),
            ),
            ModelInfo(
                id='mistral-small-24b-instruct-2501:free',
                match=ClauseEquals(equals='mistral-small-24b-instruct-2501:free'),
                name='Mistral Small 3 (free)',
                description='Mistral Small 3 is a 24B-parameter language model optimized for low-latency performance across common AI tasks. Released under the Apache 2.0 license, it features both pre-trained and instruction-tuned versions designed for efficient local deployment.',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistral-small-latest',
                match=ClauseEquals(equals='mistral-small-latest'),
                name='Mistral Small 3.2',
                description='SOTA. Multimodal. Multilingual. Apache 2.0.',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='mistral-tiny',
                match=ClauseEquals(equals='mistral-tiny'),
                name='Mistral Tiny',
                description='Note: This model is being deprecated. Recommended replacement is the newer Ministral 8B',
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('0.25')),
            ),
            ModelInfo(
                id='mixtral-8x22b-instruct',
                match=ClauseEquals(equals='mixtral-8x22b-instruct'),
                name='Mixtral 8x22B Instruct',
                description="Mistral's official instruct fine-tuned version of Mixtral 8x22B. It uses 39B active parameters out of 141B, offering unparalleled cost efficiency for its size. Its strengths include:\n- strong math, coding, and reasoning\n- large context length (64k)\n- fluency in English, French, Italian, German, and Spanish",
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='mixtral-8x7b-instruct',
                match=ClauseEquals(equals='mixtral-8x7b-instruct'),
                name='Mixtral 8x7B Instruct',
                description='Mixtral 8x7B Instruct is a pretrained generative Sparse Mixture of Experts, by Mistral AI, for chat and instruction use. Incorporates 8 experts (feed-forward networks) for a total of 47 billion parameters.',
                prices=ModelPrice(input_mtok=Decimal('0.08'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='open-mistral-7b',
                match=ClauseEquals(equals='open-mistral-7b'),
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('0.25')),
            ),
            ModelInfo(
                id='open-mixtral-8x7b',
                match=ClauseEquals(equals='open-mixtral-8x7b'),
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('0.7')),
            ),
            ModelInfo(
                id='pixtral-12b',
                match=ClauseOr(or_=[ClauseEquals(equals='pixtral-12b'), ClauseEquals(equals='pixtral-12b-latest')]),
                name='Pixtral 12B',
                description='The first multi-modal, text+image-to-text model from Mistral AI. Its weights were launched via torrent: https://x.com/mistralai/status/1833758285167722836.',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='pixtral-large',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='pixtral-large-latest'), ClauseEquals(equals='pixtral-large-2411')]
                ),
                name='Pixtral Large 2411',
                description='Pixtral Large is a 124B parameter, open-weight, multimodal model built on top of Mistral Large 2. The model is able to understand documents, charts and natural images.',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('6')),
            ),
        ],
    ),
    Provider(
        id='novita',
        name='Novita',
        api_pattern='https://api\\.novita\\.ai',
        pricing_urls=['https://novita.ai/pricing'],
        models=[
            ModelInfo(
                id='Sao10K/L3-8B-Stheno-v3.2',
                match=ClauseEquals(equals='Sao10K/L3-8B-Stheno-v3.2'),
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.05')),
            ),
            ModelInfo(
                id='cognitivecomputations/dolphin-mixtral-8x22b',
                match=ClauseEquals(equals='cognitivecomputations/dolphin-mixtral-8x22b'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1',
                match=ClauseEquals(equals='deepseek/deepseek-r1'),
                prices=ModelPrice(input_mtok=Decimal('4'), output_mtok=Decimal('4')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-llama-70b',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-llama-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-llama-8b',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-llama-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.04'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-qwen-14b',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-qwen-14b'),
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-qwen-32b',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-qwen-32b'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='deepseek/deepseek_v3',
                match=ClauseEquals(equals='deepseek/deepseek_v3'),
                prices=ModelPrice(input_mtok=Decimal('0.89'), output_mtok=Decimal('0.89')),
            ),
            ModelInfo(
                id='google/gemma-2-9b-it',
                match=ClauseEquals(equals='google/gemma-2-9b-it'),
                prices=ModelPrice(input_mtok=Decimal('0.08'), output_mtok=Decimal('0.08')),
            ),
            ModelInfo(
                id='gryphe/mythomax-l2-13b',
                match=ClauseEquals(equals='gryphe/mythomax-l2-13b'),
                prices=ModelPrice(input_mtok=Decimal('0.09'), output_mtok=Decimal('0.09')),
            ),
            ModelInfo(
                id='jondurbin/airoboros-l2-70b',
                match=ClauseEquals(equals='jondurbin/airoboros-l2-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.5')),
            ),
            ModelInfo(
                id='meta-llama/llama-3-70b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3-70b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.51'), output_mtok=Decimal('0.74')),
            ),
            ModelInfo(
                id='meta-llama/llama-3-8b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3-8b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.04'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.1-70b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.1-70b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.34'), output_mtok=Decimal('0.39')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.1-8b-instruct',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='meta-llama/llama-3.1-8b-instruct'),
                        ClauseEquals(equals='meta-llama/llama-3.1-8b-instruct-max'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.05')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.1-8b-instruct-bf16',
                match=ClauseEquals(equals='meta-llama/llama-3.1-8b-instruct-bf16'),
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.06')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-11b-vision-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.2-11b-vision-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.06')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-1b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.2-1b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.02')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-3b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.2-3b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.03'), output_mtok=Decimal('0.05')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.3-70b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.3-70b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.39'), output_mtok=Decimal('0.39')),
            ),
            ModelInfo(
                id='microsoft/wizardlm-2-8x22b',
                match=ClauseEquals(equals='microsoft/wizardlm-2-8x22b'),
                prices=ModelPrice(input_mtok=Decimal('0.62'), output_mtok=Decimal('0.62')),
            ),
            ModelInfo(
                id='mistralai/mistral-7b-instruct',
                match=ClauseEquals(equals='mistralai/mistral-7b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.059'), output_mtok=Decimal('0.059')),
            ),
            ModelInfo(
                id='mistralai/mistral-nemo',
                match=ClauseEquals(equals='mistralai/mistral-nemo'),
                prices=ModelPrice(input_mtok=Decimal('0.17'), output_mtok=Decimal('0.17')),
            ),
            ModelInfo(
                id='nousresearch/hermes-2-pro-llama-3-8b',
                match=ClauseEquals(equals='nousresearch/hermes-2-pro-llama-3-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.14'), output_mtok=Decimal('0.14')),
            ),
            ModelInfo(
                id='nousresearch/nous-hermes-llama2-13b',
                match=ClauseEquals(equals='nousresearch/nous-hermes-llama2-13b'),
                prices=ModelPrice(input_mtok=Decimal('0.17'), output_mtok=Decimal('0.17')),
            ),
            ModelInfo(
                id='openchat/openchat-7b',
                match=ClauseEquals(equals='openchat/openchat-7b'),
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.06')),
            ),
            ModelInfo(
                id='qwen/qwen-2-7b-instruct',
                match=ClauseEquals(equals='qwen/qwen-2-7b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.054'), output_mtok=Decimal('0.054')),
            ),
            ModelInfo(
                id='qwen/qwen-2-vl-72b-instruct',
                match=ClauseEquals(equals='qwen/qwen-2-vl-72b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.45'), output_mtok=Decimal('0.45')),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-72b-instruct',
                match=ClauseEquals(equals='qwen/qwen-2.5-72b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.38'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='sao10k/l3-70b-euryale-v2.1',
                match=ClauseEquals(equals='sao10k/l3-70b-euryale-v2.1'),
                prices=ModelPrice(input_mtok=Decimal('1.48'), output_mtok=Decimal('1.48')),
            ),
            ModelInfo(
                id='sao10k/l3-8b-lunaris',
                match=ClauseEquals(equals='sao10k/l3-8b-lunaris'),
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.05')),
            ),
            ModelInfo(
                id='sao10k/l31-70b-euryale-v2.2',
                match=ClauseEquals(equals='sao10k/l31-70b-euryale-v2.2'),
                prices=ModelPrice(input_mtok=Decimal('1.48'), output_mtok=Decimal('1.48')),
            ),
            ModelInfo(
                id='sophosympatheia/midnight-rose-70b',
                match=ClauseEquals(equals='sophosympatheia/midnight-rose-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='teknium/openhermes-2.5-mistral-7b',
                match=ClauseEquals(equals='teknium/openhermes-2.5-mistral-7b'),
                prices=ModelPrice(input_mtok=Decimal('0.17'), output_mtok=Decimal('0.17')),
            ),
        ],
    ),
    Provider(
        id='openai',
        name='OpenAI',
        api_pattern='https://api\\.openai\\.com',
        pricing_urls=[
            'https://platform.openai.com/docs/pricing',
            'https://openai.com/api/pricing/',
            'https://platform.openai.com/docs/models',
            'https://help.openai.com/en/articles/7127956-how-much-does-gpt-4-cost',
        ],
        model_match=ClauseOr(or_=[ClauseStartsWith(starts_with='gpt-'), ClauseRegex(regex='^o[134]')]),
        models=[
            ModelInfo(
                id='ada',
                match=ClauseOr(or_=[ClauseEquals(equals='ada'), ClauseEquals(equals='text-ada-001')]),
                prices=ModelPrice(input_mtok=Decimal('0.4'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='babbage',
                match=ClauseEquals(equals='babbage'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.5')),
            ),
            ModelInfo(
                id='chatgpt-4o-latest',
                match=ClauseEquals(equals='chatgpt-4o-latest'),
                name='ChatGPT-4o',
                description='OpenAI ChatGPT 4o is continually updated by OpenAI to point to the current version of GPT-4o used by ChatGPT. It therefore differs slightly from the API version of GPT-4o in that it has additional RLHF. It is intended for research and evaluation.',
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='codex-mini',
                match=ClauseEquals(equals='codex-mini'),
                name='Codex Mini',
                description='codex-mini-latest is a fine-tuned version of o4-mini specifically for use in Codex CLI. For direct use in the API, we recommend starting with gpt-4.1.',
                prices=ModelPrice(
                    input_mtok=Decimal('1.5'), cache_read_mtok=Decimal('0.375'), output_mtok=Decimal('6')
                ),
            ),
            ModelInfo(
                id='curie',
                match=ClauseOr(or_=[ClauseEquals(equals='curie'), ClauseEquals(equals='text-curie-001')]),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='davinci',
                match=ClauseOr(or_=[ClauseEquals(equals='davinci'), ClauseEquals(equals='text-davinci-001')]),
                prices=ModelPrice(input_mtok=Decimal('20'), output_mtok=Decimal('20')),
            ),
            ModelInfo(
                id='ft:gpt-3.5-turbo-',
                match=ClauseStartsWith(starts_with='ft:gpt-3.5-turbo-'),
                description='GPT-3.5 Turbo fine tuned.',
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='ft:gpt-4o-2024-08-06:',
                match=ClauseStartsWith(starts_with='ft:gpt-4o-2024-08-06:'),
                description='GPT-4o fine tuned.',
                prices=ModelPrice(input_mtok=Decimal('3.75'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='ft:gpt-4o-mini-2024-07-18:',
                match=ClauseStartsWith(starts_with='ft:gpt-4o-mini-2024-07-18:'),
                description='GPT-4o Mini fine tuned.',
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-3.5-turbo'),
                        ClauseEquals(equals='gpt-35-turbo'),
                        ClauseEquals(equals='gpt-3.5-turbo-0125'),
                        ClauseEquals(equals='gpt-3.5-turbo-1106'),
                    ]
                ),
                name='gpt 3.5 turbo',
                description='GPT-3.5 Turbo offers a balance between cost and performance.',
                context_window=16385,
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo-16k',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-3.5-turbo-16k'),
                        ClauseEquals(equals='gpt-3.5-turbo-16k-0613'),
                        ClauseEquals(equals='gpt-35-turbo-16k-0613'),
                    ]
                ),
                name='GPT-3.5 Turbo 16k',
                description='This model offers four times the context length of gpt-3.5-turbo, allowing it to support approximately 20 pages of text in a single request at a higher cost. Training data: up to Sep 2021.',
                context_window=16385,
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('4')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo-instruct',
                match=ClauseOr(
                    or_=[
                        ClauseStartsWith(starts_with='gpt-3.5-turbo-instruct'),
                        ClauseEquals(equals='gpt-3.5-turbo-instruct-0914'),
                    ]
                ),
                name='gpt 3.5 turbo instruct',
                description='GPT-3.5 Turbo offers a balance between cost and performance.',
                context_window=16385,
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt-4',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4'),
                        ClauseEquals(equals='gpt-4-0314'),
                        ClauseEquals(equals='gpt-4-0613'),
                    ]
                ),
                name='gpt 4',
                description='GPT-4 is the latest and most advanced model in the GPT series, demonstrating sophisticated capabilities in complex reasoning, theory of mind, and narrative understanding.',
                context_window=8192,
                prices=ModelPrice(input_mtok=Decimal('30'), output_mtok=Decimal('60')),
            ),
            ModelInfo(
                id='gpt-4-32k',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4-32k'),
                        ClauseEquals(equals='gpt-4-32k-0314'),
                        ClauseEquals(equals='gpt-4-32k-0613'),
                    ]
                ),
                name='gpt 4',
                description='GPT-4 is the latest and most advanced model in the GPT series, demonstrating sophisticated capabilities in complex reasoning, theory of mind, and narrative understanding.',
                context_window=32000,
                price_comments='see https://help.openai.com/en/articles/7127956-how-much-does-gpt-4-cost',
                prices=ModelPrice(input_mtok=Decimal('60'), output_mtok=Decimal('120')),
            ),
            ModelInfo(
                id='gpt-4-turbo',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4-turbo'),
                        ClauseEquals(equals='gpt-4-turbo-2024-04-09'),
                        ClauseEquals(equals='gpt-4-turbo-0125-preview'),
                        ClauseEquals(equals='gpt-4-0125-preview'),
                        ClauseEquals(equals='gpt-4-1106-preview'),
                        ClauseEquals(equals='gpt-4-turbo-preview'),
                    ]
                ),
                name='gpt 4 turbo',
                description='GPT-4 Turbo offers a balance between cost and performance.',
                context_window=128000,
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4-vision-preview',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='gpt-4-vision-preview'), ClauseEquals(equals='gpt-4-1106-vision-preview')]
                ),
                name='gpt 4 vision',
                description='GPT-4 Vision is a model that offers a balance between cost and performance.',
                context_window=128000,
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4.1',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-4.1'), ClauseEquals(equals='gpt-4.1-2025-04-14')]),
                name='gpt 4.1',
                description="GPT-4.1 is OpenAI's latest flagship model, offering major improvements in coding, instruction following, and long context understanding with up to 1 million tokens of context.",
                context_window=1000000,
                prices=ModelPrice(input_mtok=Decimal('2'), cache_read_mtok=Decimal('0.5'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='gpt-4.1-mini',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='gpt-4.1-mini'), ClauseEquals(equals='gpt-4.1-mini-2025-04-14')]
                ),
                name='gpt 4.1 mini',
                description='GPT-4.1 Mini is a significant leap in small model performance, matching or exceeding GPT-4o in many benchmarks while reducing latency by nearly half and cost by 83%.',
                context_window=1000000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.4'), cache_read_mtok=Decimal('0.1'), output_mtok=Decimal('1.6')
                ),
            ),
            ModelInfo(
                id='gpt-4.1-nano',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='gpt-4.1-nano'), ClauseEquals(equals='gpt-4.1-nano-2025-04-14')]
                ),
                name='gpt 4.1 nano',
                description="GPT-4.1 Nano is OpenAI's fastest and cheapest model, delivering exceptional performance for its size with a 1 million token context window, ideal for classification and autocompletion tasks.",
                context_window=1000000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.1'), cache_read_mtok=Decimal('0.025'), output_mtok=Decimal('0.4')
                ),
            ),
            ModelInfo(
                id='gpt-4.5-preview',
                match=ClauseStartsWith(starts_with='gpt-4.5-preview'),
                name='GPT-4.5 (Preview)',
                description="GPT-4.5 (Preview) is a research preview of OpenAI's latest language model, designed to advance capabilities in reasoning, creativity, and multi-turn conversation. It builds on previous iterations with improvements in world knowledge, contextual coherence, and the ability to follow user intent more effectively.",
                prices=ModelPrice(
                    input_mtok=Decimal('75'), cache_read_mtok=Decimal('37.5'), output_mtok=Decimal('150')
                ),
            ),
            ModelInfo(
                id='gpt-4o',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4o'),
                        ClauseEquals(equals='gpt-4o-2024-05-13'),
                        ClauseEquals(equals='gpt-4o-2024-08-06'),
                        ClauseEquals(equals='gpt-4o-2024-11-20'),
                    ]
                ),
                name='gpt 4o',
                description='GPT-4 Optimized (GPT-4o) is designed for high performance in reasoning, creativity, and technical tasks while maintaining consistent output quality.',
                context_window=128000,
                prices=ModelPrice(
                    input_mtok=Decimal('2.5'), cache_read_mtok=Decimal('1.25'), output_mtok=Decimal('10')
                ),
            ),
            ModelInfo(
                id='gpt-4o-audio-preview',
                match=ClauseStartsWith(starts_with='gpt-4o-audio-preview'),
                name='gpt 4o audio preview',
                description='Audio model for gpt-4o',
                context_window=128000,
                prices=ModelPrice(output_mtok=Decimal('10'), input_audio_mtok=Decimal('2.5')),
            ),
            ModelInfo(
                id='gpt-4o-mini',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4o-mini'),
                        ClauseEquals(equals='gpt-4o-mini-2024-07-18'),
                        ClauseEquals(equals='gpt-4o-mini-search-preview'),
                    ]
                ),
                name='gpt 4o mini',
                description='GPT-4o Mini is a cost-optimized variant of GPT-4o, designed for high-efficiency processing while maintaining strong performance. It excels in rapid inference and resource-efficient operations, making it ideal for production deployments requiring a balance of cost and capability.',
                context_window=128000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.15'), cache_read_mtok=Decimal('0.075'), output_mtok=Decimal('0.6')
                ),
            ),
            ModelInfo(
                id='gpt-4o-mini-2024-07-18.ft-',
                match=ClauseStartsWith(starts_with='gpt-4o-mini-2024-07-18.ft-'),
                description='GPT-4o Mini fine tuned.',
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='gpt-4o-mini-audio-preview',
                match=ClauseStartsWith(starts_with='gpt-4o-mini-audio'),
                name='gpt 4o mini audio preview',
                description='Audio model for gpt-4o mini',
                prices=ModelPrice(output_mtok=Decimal('0.6'), input_audio_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='gpt-4o-mini-realtime-preview',
                match=ClauseStartsWith(starts_with='gpt-4o-mini-realtime'),
                prices=ModelPrice(
                    input_mtok=Decimal('0.6'), cache_read_mtok=Decimal('0.3'), output_mtok=Decimal('2.4')
                ),
            ),
            ModelInfo(
                id='gpt-4o-realtime-preview',
                match=ClauseStartsWith(starts_with='gpt-4o-realtime'),
                prices=ModelPrice(
                    input_mtok=Decimal('5'),
                    cache_read_mtok=Decimal('2.5'),
                    output_mtok=Decimal('20'),
                    input_audio_mtok=Decimal('40'),
                    cache_audio_read_mtok=Decimal('2.5'),
                    output_audio_mtok=Decimal('80'),
                ),
            ),
            ModelInfo(
                id='gpt-4o-search-preview',
                match=ClauseEquals(equals='gpt-4o-search-preview'),
                name='GPT-4o Search Preview',
                description='GPT-4o Search Previewis a specialized model for web search in Chat Completions. It is trained to understand and execute web search queries.',
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='gpt-4o:extended',
                match=ClauseEquals(equals='gpt-4o:extended'),
                name='GPT-4o (extended)',
                description='GPT-4o ("o" for "omni") is OpenAI\'s latest AI model, supporting both text and image inputs with text outputs. It maintains the intelligence level of GPT-4 Turbo while being twice as fast and 50% more cost-effective. GPT-4o also offers improved performance in processing non-English languages and enhanced visual capabilities.',
                prices=ModelPrice(input_mtok=Decimal('6'), output_mtok=Decimal('18')),
            ),
            ModelInfo(
                id='gpt-5',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-5'),
                        ClauseEquals(equals='gpt-5-2025-08-07'),
                        ClauseEquals(equals='gpt-5-chat'),
                        ClauseEquals(equals='gpt-5-chat-latest'),
                    ]
                ),
                name='GPT-5',
                description="GPT-5 is OpenAI's flagship model for coding, reasoning, and agentic tasks across domains.",
                context_window=400000,
                prices=ModelPrice(
                    input_mtok=Decimal('1.25'), cache_read_mtok=Decimal('0.125'), output_mtok=Decimal('10')
                ),
            ),
            ModelInfo(
                id='gpt-5-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-5-mini'), ClauseEquals(equals='gpt-5-mini-2025-08-07')]),
                name='GPT-5 mini',
                description="GPT-5 mini is a faster, more cost-efficient version of GPT-5. It's great for well-defined tasks and precise prompts.",
                context_window=400000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.25'), cache_read_mtok=Decimal('0.025'), output_mtok=Decimal('2')
                ),
            ),
            ModelInfo(
                id='gpt-5-nano',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-5-nano'), ClauseEquals(equals='gpt-5-nano-2025-04-14')]),
                name='GPT-5 nano',
                description="GPT-5 Nano is OpenAI's fastest, cheapest version of GPT-5. It's great for summarization and classification tasks.",
                context_window=400000,
                prices=ModelPrice(
                    input_mtok=Decimal('0.05'), cache_read_mtok=Decimal('0.005'), output_mtok=Decimal('0.4')
                ),
            ),
            ModelInfo(
                id='o1',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='o1'),
                        ClauseEquals(equals='o1-2024-12-17'),
                        ClauseEquals(equals='o1-preview'),
                        ClauseEquals(equals='o1-preview-2024-09-12'),
                    ]
                ),
                name='o1',
                description='O1 is a model that offers a balance between cost and performance.',
                context_window=128000,
                prices=ModelPrice(input_mtok=Decimal('15'), cache_read_mtok=Decimal('7.5'), output_mtok=Decimal('60')),
            ),
            ModelInfo(
                id='o1-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='o1-mini'), ClauseEquals(equals='o1-mini-2024-09-12')]),
                name='o1 mini',
                description='O1 Mini is a model that offers a balance between cost and performance.',
                context_window=128000,
                prices=ModelPrice(
                    input_mtok=Decimal('1.1'), cache_read_mtok=Decimal('0.55'), output_mtok=Decimal('4.4')
                ),
            ),
            ModelInfo(
                id='o1-pro',
                match=ClauseEquals(equals='o1-pro'),
                name='o1-pro',
                description='The o1 series of models are trained with reinforcement learning to think before they answer and perform complex reasoning. The o1-pro model uses more compute to think harder and provide consistently better answers.',
                prices=ModelPrice(input_mtok=Decimal('150'), output_mtok=Decimal('600')),
            ),
            ModelInfo(
                id='o3',
                match=ClauseOr(or_=[ClauseEquals(equals='o3'), ClauseEquals(equals='o3-2025-04-16')]),
                name='o3',
                description='o3 is a well-rounded and powerful model across domains. It sets a new standard for math, science, coding, and visual reasoning tasks. It also excels at technical writing and instruction-following. Use it to think through multi-step problems that involve analysis across text, code, and images. Note that BYOK is required for this model. Set up here: https://openrouter.ai/settings/integrations',
                prices=[
                    ConditionalPrice(
                        prices=ModelPrice(
                            input_mtok=Decimal('10'), cache_read_mtok=Decimal('0.5'), output_mtok=Decimal('40')
                        )
                    ),
                    ConditionalPrice(
                        constraint=StartDateConstraint(start_date=datetime.date(2025, 6, 10)),
                        prices=ModelPrice(
                            input_mtok=Decimal('2'), cache_read_mtok=Decimal('0.5'), output_mtok=Decimal('8')
                        ),
                    ),
                ],
            ),
            ModelInfo(
                id='o3-mini',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='o3-mini'),
                        ClauseEquals(equals='o3-mini-2025-01-31'),
                        ClauseEquals(equals='o3-mini-high'),
                    ]
                ),
                name='o3 Mini',
                description='OpenAI o3-mini is a cost-efficient language model optimized for STEM reasoning tasks, particularly excelling in science, mathematics, and coding.',
                prices=ModelPrice(
                    input_mtok=Decimal('1.1'), cache_read_mtok=Decimal('0.55'), output_mtok=Decimal('4.4')
                ),
            ),
            ModelInfo(
                id='o3-pro',
                match=ClauseEquals(equals='o3-pro'),
                name='o3 Pro',
                description='The o-series of models are trained with reinforcement learning to think before they answer and perform complex reasoning. The o3-pro model uses more compute to think harder and provide consistently better answers.',
                prices=ModelPrice(input_mtok=Decimal('20'), output_mtok=Decimal('80')),
            ),
            ModelInfo(
                id='o4-mini',
                match=ClauseOr(
                    or_=[
                        ClauseStartsWith(starts_with='o4-mini'),
                        ClauseEquals(equals='o4-mini-2025-04-16'),
                        ClauseEquals(equals='o4-mini-high'),
                        ClauseEquals(equals='o4-mini'),
                    ]
                ),
                name='o4 Mini High',
                description='OpenAI o4-mini-high is the same model as o4-mini with reasoning_effort set to high.',
                prices=ModelPrice(
                    input_mtok=Decimal('1.1'), cache_read_mtok=Decimal('0.275'), output_mtok=Decimal('4.4')
                ),
            ),
            ModelInfo(
                id='text-davinci-002',
                match=ClauseEquals(equals='text-davinci-002'),
                prices=ModelPrice(input_mtok=Decimal('20'), output_mtok=Decimal('20')),
            ),
            ModelInfo(
                id='text-davinci-003',
                match=ClauseEquals(equals='text-davinci-003'),
                prices=ModelPrice(input_mtok=Decimal('20'), output_mtok=Decimal('20')),
            ),
            ModelInfo(
                id='text-embedding-3-large',
                match=ClauseEquals(equals='text-embedding-3-large'),
                name='text embedding 3',
                description='Text Embedding 3 is a model that offers a balance between cost and performance.',
                context_window=3072,
                prices=ModelPrice(input_mtok=Decimal('0.13')),
            ),
            ModelInfo(
                id='text-embedding-3-small',
                match=ClauseEquals(equals='text-embedding-3-small'),
                name='text embedding 3',
                description='Text Embedding 3 is a model that offers a balance between cost and performance.',
                context_window=3072,
                prices=ModelPrice(input_mtok=Decimal('0.02')),
            ),
            ModelInfo(
                id='text-embedding-ada-002',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='text-embedding-ada'),
                        ClauseEquals(equals='text-embedding-ada-002'),
                        ClauseEquals(equals='text-embedding-ada-002-v2'),
                    ]
                ),
                name='text embedding ada',
                description='Text Embedding Ada is a model that offers a balance between cost and performance.',
                context_window=1536,
                prices=ModelPrice(input_mtok=Decimal('0.1')),
            ),
        ],
    ),
    Provider(
        id='openrouter',
        name='OpenRouter',
        api_pattern='https://(api\\.)?openrouter\\.ai',
        pricing_urls=['https://openrouter.ai/models'],
        models=[
            ModelInfo(
                id='01-ai/yi-large',
                match=ClauseEquals(equals='01-ai/yi-large'),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('3')),
            ),
            ModelInfo(
                id='aetherwiing/mn-starcannon-12b',
                match=ClauseEquals(equals='aetherwiing/mn-starcannon-12b'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='agentica-org/deepcoder-14b-preview:free',
                match=ClauseEquals(equals='agentica-org/deepcoder-14b-preview:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='ai21/jamba-1-5-large',
                match=ClauseEquals(equals='ai21/jamba-1-5-large'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='ai21/jamba-1-5-mini',
                match=ClauseEquals(equals='ai21/jamba-1-5-mini'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='ai21/jamba-1.6-large',
                match=ClauseEquals(equals='ai21/jamba-1.6-large'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='ai21/jamba-1.6-mini',
                match=ClauseEquals(equals='ai21/jamba-1.6-mini'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='ai21/jamba-instruct',
                match=ClauseEquals(equals='ai21/jamba-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.7')),
            ),
            ModelInfo(
                id='aion-1.0',
                match=ClauseEquals(equals='aion-1.0'),
                name='Aion-1.0',
                prices=ModelPrice(input_mtok=Decimal('4'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='aion-1.0-mini',
                match=ClauseEquals(equals='aion-1.0-mini'),
                name='Aion-1.0-Mini',
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('1.4')),
            ),
            ModelInfo(
                id='aion-labs/aion-1.0',
                match=ClauseEquals(equals='aion-labs/aion-1.0'),
                prices=ModelPrice(input_mtok=Decimal('4'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='aion-labs/aion-1.0-mini',
                match=ClauseEquals(equals='aion-labs/aion-1.0-mini'),
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('1.4')),
            ),
            ModelInfo(
                id='aion-labs/aion-rp-llama-3.1-8b',
                match=ClauseEquals(equals='aion-labs/aion-rp-llama-3.1-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='aion-rp-llama-3.1-8b',
                match=ClauseEquals(equals='aion-rp-llama-3.1-8b'),
                name='Aion-RP 1.0 (8B)',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='alfredpros/codellama-7b-instruct-solidity',
                match=ClauseEquals(equals='alfredpros/codellama-7b-instruct-solidity'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='all-hands/openhands-lm-32b-v0.1',
                match=ClauseEquals(equals='all-hands/openhands-lm-32b-v0.1'),
                prices=ModelPrice(input_mtok=Decimal('2.6'), output_mtok=Decimal('3.4')),
            ),
            ModelInfo(
                id='allenai/molmo-7b-d:free', match=ClauseEquals(equals='allenai/molmo-7b-d:free'), prices=ModelPrice()
            ),
            ModelInfo(
                id='alpindale/goliath-120b',
                match=ClauseEquals(equals='alpindale/goliath-120b'),
                prices=ModelPrice(input_mtok=Decimal('6.5625'), output_mtok=Decimal('9.375')),
            ),
            ModelInfo(
                id='alpindale/magnum-72b',
                match=ClauseEquals(equals='alpindale/magnum-72b'),
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('2.25')),
            ),
            ModelInfo(
                id='amazon/nova-lite-v1',
                match=ClauseEquals(equals='amazon/nova-lite-v1'),
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='amazon/nova-micro-v1',
                match=ClauseEquals(equals='amazon/nova-micro-v1'),
                prices=ModelPrice(input_mtok=Decimal('0.035'), output_mtok=Decimal('0.14')),
            ),
            ModelInfo(
                id='amazon/nova-pro-v1',
                match=ClauseEquals(equals='amazon/nova-pro-v1'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('3.2')),
            ),
            ModelInfo(
                id='anthracite-org/magnum-v2-72b',
                match=ClauseEquals(equals='anthracite-org/magnum-v2-72b'),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('3')),
            ),
            ModelInfo(
                id='anthracite-org/magnum-v4-72b',
                match=ClauseEquals(equals='anthracite-org/magnum-v4-72b'),
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('2.25')),
            ),
            ModelInfo(
                id='anthropic/claude-2',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='anthropic/claude-2'),
                        ClauseEquals(equals='anthropic/claude-2.0'),
                        ClauseEquals(equals='anthropic/claude-2.0:beta'),
                        ClauseEquals(equals='anthropic/claude-2.1'),
                        ClauseEquals(equals='anthropic/claude-2.1:beta'),
                        ClauseEquals(equals='anthropic/claude-2:beta'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('8'), output_mtok=Decimal('24')),
            ),
            ModelInfo(
                id='anthropic/claude-3-haiku',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='anthropic/claude-3-haiku'),
                        ClauseEquals(equals='anthropic/claude-3-haiku:beta'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('1.25')),
            ),
            ModelInfo(
                id='anthropic/claude-3-opus',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='anthropic/claude-3-opus'),
                        ClauseEquals(equals='anthropic/claude-3-opus:beta'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('15'), output_mtok=Decimal('75')),
            ),
            ModelInfo(
                id='anthropic/claude-3-sonnet',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='anthropic/claude-3-sonnet'),
                        ClauseEquals(equals='anthropic/claude-3-sonnet:beta'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='anthropic/claude-3.5-haiku',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='anthropic/claude-3.5-haiku'),
                        ClauseEquals(equals='anthropic/claude-3.5-haiku-20241022'),
                        ClauseEquals(equals='anthropic/claude-3.5-haiku-20241022:beta'),
                        ClauseEquals(equals='anthropic/claude-3.5-haiku:beta'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('4')),
            ),
            ModelInfo(
                id='anthropic/claude-3.5-sonnet',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='anthropic/claude-3.5-sonnet'),
                        ClauseEquals(equals='anthropic/claude-3.5-sonnet-20240620'),
                        ClauseEquals(equals='anthropic/claude-3.5-sonnet-20240620:beta'),
                        ClauseEquals(equals='anthropic/claude-3.5-sonnet:beta'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='anthropic/claude-3.7-sonnet',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='anthropic/claude-3.7-sonnet'),
                        ClauseEquals(equals='anthropic/claude-3.7-sonnet:beta'),
                        ClauseEquals(equals='anthropic/claude-3.7-sonnet:thinking'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='anubis-pro-105b-v1',
                match=ClauseEquals(equals='anubis-pro-105b-v1'),
                name='Anubis Pro 105B V1',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='arcee-blitz',
                match=ClauseEquals(equals='arcee-blitz'),
                name='Arcee Blitz',
                prices=ModelPrice(input_mtok=Decimal('0.45'), output_mtok=Decimal('0.75')),
            ),
            ModelInfo(
                id='arliai/qwq-32b-arliai-rpr-v1:free',
                match=ClauseEquals(equals='arliai/qwq-32b-arliai-rpr-v1:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='bytedance-research/ui-tars-72b:free',
                match=ClauseEquals(equals='bytedance-research/ui-tars-72b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='caller-large',
                match=ClauseEquals(equals='caller-large'),
                name='Caller Large',
                prices=ModelPrice(input_mtok=Decimal('0.55'), output_mtok=Decimal('0.85')),
            ),
            ModelInfo(
                id='chatgpt-4o-latest',
                match=ClauseEquals(equals='chatgpt-4o-latest'),
                name='ChatGPT-4o',
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='claude-2',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='claude-2'),
                        ClauseEquals(equals='claude-2.0'),
                        ClauseEquals(equals='claude-2.0:beta'),
                        ClauseEquals(equals='claude-2.1'),
                        ClauseEquals(equals='claude-2.1:beta'),
                        ClauseEquals(equals='claude-2:beta'),
                    ]
                ),
                name='Claude v2',
                prices=ModelPrice(input_mtok=Decimal('8'), output_mtok=Decimal('24')),
            ),
            ModelInfo(
                id='claude-3-haiku',
                match=ClauseOr(or_=[ClauseEquals(equals='claude-3-haiku'), ClauseEquals(equals='claude-3-haiku:beta')]),
                name='Claude 3 Haiku',
                prices=ModelPrice(
                    input_mtok=Decimal('0.25'),
                    cache_write_mtok=Decimal('0.3'),
                    cache_read_mtok=Decimal('0.03'),
                    output_mtok=Decimal('1.25'),
                ),
            ),
            ModelInfo(
                id='claude-3-opus',
                match=ClauseOr(or_=[ClauseEquals(equals='claude-3-opus'), ClauseEquals(equals='claude-3-opus:beta')]),
                name='Claude 3 Opus',
                prices=ModelPrice(
                    input_mtok=Decimal('15'),
                    cache_write_mtok=Decimal('18.75'),
                    cache_read_mtok=Decimal('1.5'),
                    output_mtok=Decimal('75'),
                ),
            ),
            ModelInfo(
                id='claude-3-sonnet',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='claude-3-sonnet'), ClauseEquals(equals='claude-3-sonnet:beta')]
                ),
                name='Claude 3 Sonnet',
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='claude-3.5-haiku',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='claude-3.5-haiku'),
                        ClauseEquals(equals='claude-3.5-haiku-20241022'),
                        ClauseEquals(equals='claude-3.5-haiku-20241022:beta'),
                        ClauseEquals(equals='claude-3.5-haiku:beta'),
                    ]
                ),
                name='Claude 3.5 Haiku',
                prices=ModelPrice(
                    input_mtok=Decimal('0.8'),
                    cache_write_mtok=Decimal('1'),
                    cache_read_mtok=Decimal('0.08'),
                    output_mtok=Decimal('4'),
                ),
            ),
            ModelInfo(
                id='claude-3.5-sonnet',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='claude-3.5-sonnet'),
                        ClauseEquals(equals='claude-3.5-sonnet-20240620'),
                        ClauseEquals(equals='claude-3.5-sonnet-20240620:beta'),
                        ClauseEquals(equals='claude-3.5-sonnet:beta'),
                    ]
                ),
                name='Claude 3.5 Sonnet',
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='claude-3.7-sonnet',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='claude-3.7-sonnet'),
                        ClauseEquals(equals='claude-3.7-sonnet:beta'),
                        ClauseEquals(equals='claude-3.7-sonnet:thinking'),
                    ]
                ),
                name='Claude 3.7 Sonnet',
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='claude-opus-4',
                match=ClauseEquals(equals='claude-opus-4'),
                name='Claude Opus 4',
                prices=ModelPrice(
                    input_mtok=Decimal('15'),
                    cache_write_mtok=Decimal('18.75'),
                    cache_read_mtok=Decimal('1.5'),
                    output_mtok=Decimal('75'),
                ),
            ),
            ModelInfo(
                id='claude-sonnet-4',
                match=ClauseEquals(equals='claude-sonnet-4'),
                name='Claude Sonnet 4',
                prices=ModelPrice(
                    input_mtok=Decimal('3'),
                    cache_write_mtok=Decimal('3.75'),
                    cache_read_mtok=Decimal('0.3'),
                    output_mtok=Decimal('15'),
                ),
            ),
            ModelInfo(
                id='codellama-7b-instruct-solidity',
                match=ClauseEquals(equals='codellama-7b-instruct-solidity'),
                name='CodeLLaMa 7B Instruct Solidity',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='coder-large',
                match=ClauseEquals(equals='coder-large'),
                name='Coder Large',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='codestral-2501',
                match=ClauseEquals(equals='codestral-2501'),
                name='Codestral 2501',
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='codex-mini',
                match=ClauseEquals(equals='codex-mini'),
                name='Codex Mini',
                prices=ModelPrice(
                    input_mtok=Decimal('1.5'), cache_read_mtok=Decimal('0.375'), output_mtok=Decimal('6')
                ),
            ),
            ModelInfo(
                id='cognitivecomputations/dolphin-mixtral-8x22b',
                match=ClauseEquals(equals='cognitivecomputations/dolphin-mixtral-8x22b'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='cognitivecomputations/dolphin-mixtral-8x7b',
                match=ClauseEquals(equals='cognitivecomputations/dolphin-mixtral-8x7b'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.5')),
            ),
            ModelInfo(
                id='cognitivecomputations/dolphin3.0-mistral-24b:free',
                match=ClauseEquals(equals='cognitivecomputations/dolphin3.0-mistral-24b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='cognitivecomputations/dolphin3.0-r1-mistral-24b:free',
                match=ClauseEquals(equals='cognitivecomputations/dolphin3.0-r1-mistral-24b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='cohere/command',
                match=ClauseEquals(equals='cohere/command'),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='cohere/command-a',
                match=ClauseEquals(equals='cohere/command-a'),
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='cohere/command-r',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='cohere/command-r'), ClauseEquals(equals='cohere/command-r-03-2024')]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='cohere/command-r-08-2024',
                match=ClauseEquals(equals='cohere/command-r-08-2024'),
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='cohere/command-r-plus',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='cohere/command-r-plus'),
                        ClauseEquals(equals='cohere/command-r-plus-04-2024'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='cohere/command-r-plus-08-2024',
                match=ClauseEquals(equals='cohere/command-r-plus-08-2024'),
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='cohere/command-r7b-12-2024',
                match=ClauseEquals(equals='cohere/command-r7b-12-2024'),
                prices=ModelPrice(input_mtok=Decimal('0.0375'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='command',
                match=ClauseEquals(equals='command'),
                name='Command',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='command-a',
                match=ClauseEquals(equals='command-a'),
                name='Command A',
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='command-r',
                match=ClauseOr(or_=[ClauseEquals(equals='command-r'), ClauseEquals(equals='command-r-03-2024')]),
                name='Command R',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='command-r-08-2024',
                match=ClauseEquals(equals='command-r-08-2024'),
                name='Command R (08-2024)',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='command-r-plus',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='command-r-plus'), ClauseEquals(equals='command-r-plus-04-2024')]
                ),
                name='Command R+',
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='command-r-plus-08-2024',
                match=ClauseEquals(equals='command-r-plus-08-2024'),
                name='Command R+ (08-2024)',
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='command-r7b-12-2024',
                match=ClauseEquals(equals='command-r7b-12-2024'),
                name='Command R7B (12-2024)',
                prices=ModelPrice(input_mtok=Decimal('0.0375'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='deepcoder-14b-preview:free',
                match=ClauseEquals(equals='deepcoder-14b-preview:free'),
                name='Deepcoder 14B Preview (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deephermes-3-llama-3-8b-preview:free',
                match=ClauseEquals(equals='deephermes-3-llama-3-8b-preview:free'),
                name='DeepHermes 3 Llama 3 8B Preview (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-chat',
                match=ClauseEquals(equals='deepseek-chat'),
                name='DeepSeek V3',
                prices=ModelPrice(input_mtok=Decimal('0.38'), output_mtok=Decimal('0.89')),
            ),
            ModelInfo(
                id='deepseek-chat-v3-0324',
                match=ClauseEquals(equals='deepseek-chat-v3-0324'),
                name='DeepSeek V3 0324',
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.88')),
            ),
            ModelInfo(
                id='deepseek-chat-v3-0324:free',
                match=ClauseEquals(equals='deepseek-chat-v3-0324:free'),
                name='DeepSeek V3 0324 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-chat:free',
                match=ClauseEquals(equals='deepseek-chat:free'),
                name='DeepSeek V3 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-prover-v2',
                match=ClauseEquals(equals='deepseek-prover-v2'),
                name='DeepSeek Prover V2',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('2.18')),
            ),
            ModelInfo(
                id='deepseek-r1',
                match=ClauseEquals(equals='deepseek-r1'),
                name='R1',
                prices=ModelPrice(input_mtok=Decimal('0.45'), output_mtok=Decimal('2.15')),
            ),
            ModelInfo(
                id='deepseek-r1-0528',
                match=ClauseEquals(equals='deepseek-r1-0528'),
                name='R1 0528',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('2.15')),
            ),
            ModelInfo(
                id='deepseek-r1-0528-qwen3-8b',
                match=ClauseEquals(equals='deepseek-r1-0528-qwen3-8b'),
                name='Deepseek R1 0528 Qwen3 8B',
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='deepseek-r1-0528-qwen3-8b:free',
                match=ClauseEquals(equals='deepseek-r1-0528-qwen3-8b:free'),
                name='Deepseek R1 0528 Qwen3 8B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-r1-0528:free',
                match=ClauseEquals(equals='deepseek-r1-0528:free'),
                name='R1 0528 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-r1-distill-llama-70b',
                match=ClauseEquals(equals='deepseek-r1-distill-llama-70b'),
                name='R1 Distill Llama 70B',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='deepseek-r1-distill-llama-70b:free',
                match=ClauseEquals(equals='deepseek-r1-distill-llama-70b:free'),
                name='R1 Distill Llama 70B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-r1-distill-llama-8b',
                match=ClauseEquals(equals='deepseek-r1-distill-llama-8b'),
                name='R1 Distill Llama 8B',
                prices=ModelPrice(input_mtok=Decimal('0.04'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='deepseek-r1-distill-qwen-1.5b',
                match=ClauseEquals(equals='deepseek-r1-distill-qwen-1.5b'),
                name='R1 Distill Qwen 1.5B',
                prices=ModelPrice(input_mtok=Decimal('0.18'), output_mtok=Decimal('0.18')),
            ),
            ModelInfo(
                id='deepseek-r1-distill-qwen-14b',
                match=ClauseEquals(equals='deepseek-r1-distill-qwen-14b'),
                name='R1 Distill Qwen 14B',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='deepseek-r1-distill-qwen-14b:free',
                match=ClauseEquals(equals='deepseek-r1-distill-qwen-14b:free'),
                name='R1 Distill Qwen 14B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-r1-distill-qwen-32b',
                match=ClauseEquals(equals='deepseek-r1-distill-qwen-32b'),
                name='R1 Distill Qwen 32B',
                prices=ModelPrice(input_mtok=Decimal('0.12'), output_mtok=Decimal('0.18')),
            ),
            ModelInfo(
                id='deepseek-r1-distill-qwen-32b:free',
                match=ClauseEquals(equals='deepseek-r1-distill-qwen-32b:free'),
                name='R1 Distill Qwen 32B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-r1-distill-qwen-7b',
                match=ClauseEquals(equals='deepseek-r1-distill-qwen-7b'),
                name='R1 Distill Qwen 7B',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='deepseek-r1:free',
                match=ClauseEquals(equals='deepseek-r1:free'),
                name='R1 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-r1t-chimera:free',
                match=ClauseEquals(equals='deepseek-r1t-chimera:free'),
                name='DeepSeek R1T Chimera (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek-v3-base:free',
                match=ClauseEquals(equals='deepseek-v3-base:free'),
                name='DeepSeek V3 Base (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek/deepseek-chat',
                match=ClauseEquals(equals='deepseek/deepseek-chat'),
                prices=ModelPrice(input_mtok=Decimal('0.38'), output_mtok=Decimal('0.89')),
            ),
            ModelInfo(
                id='deepseek/deepseek-chat-v3-0324',
                match=ClauseEquals(equals='deepseek/deepseek-chat-v3-0324'),
                prices=ModelPrice(input_mtok=Decimal('0.27'), output_mtok=Decimal('1.1')),
            ),
            ModelInfo(
                id='deepseek/deepseek-chat-v3-0324:free',
                match=ClauseEquals(equals='deepseek/deepseek-chat-v3-0324:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek/deepseek-chat:free',
                match=ClauseEquals(equals='deepseek/deepseek-chat:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1',
                match=ClauseEquals(equals='deepseek/deepseek-r1'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('3')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-llama-70b',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-llama-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-llama-70b:free',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-llama-70b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-llama-8b',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-llama-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.04'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-qwen-1.5b',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-qwen-1.5b'),
                prices=ModelPrice(input_mtok=Decimal('0.18'), output_mtok=Decimal('0.18')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-qwen-14b',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-qwen-14b'),
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-qwen-14b:free',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-qwen-14b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-qwen-32b',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-qwen-32b'),
                prices=ModelPrice(input_mtok=Decimal('0.12'), output_mtok=Decimal('0.18')),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-distill-qwen-32b:free',
                match=ClauseEquals(equals='deepseek/deepseek-r1-distill-qwen-32b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1-zero:free',
                match=ClauseEquals(equals='deepseek/deepseek-r1-zero:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek/deepseek-r1:free',
                match=ClauseEquals(equals='deepseek/deepseek-r1:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='deepseek/deepseek-v3-base:free',
                match=ClauseEquals(equals='deepseek/deepseek-v3-base:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='devstral-small',
                match=ClauseEquals(equals='devstral-small'),
                name='Devstral Small',
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.12')),
            ),
            ModelInfo(
                id='devstral-small:free',
                match=ClauseEquals(equals='devstral-small:free'),
                name='Devstral Small (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='dobby-mini-unhinged-plus-llama-3.1-8b',
                match=ClauseEquals(equals='dobby-mini-unhinged-plus-llama-3.1-8b'),
                name='Dobby Mini Plus Llama 3.1 8B',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='dolphin-mixtral-8x22b',
                match=ClauseEquals(equals='dolphin-mixtral-8x22b'),
                name='Dolphin 2.9.2 Mixtral 8x22B 🐬',
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='dolphin3.0-mistral-24b:free',
                match=ClauseEquals(equals='dolphin3.0-mistral-24b:free'),
                name='Dolphin3.0 Mistral 24B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='dolphin3.0-r1-mistral-24b:free',
                match=ClauseEquals(equals='dolphin3.0-r1-mistral-24b:free'),
                name='Dolphin3.0 R1 Mistral 24B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='eleutherai/llemma_7b',
                match=ClauseEquals(equals='eleutherai/llemma_7b'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='eva-llama-3.33-70b',
                match=ClauseEquals(equals='eva-llama-3.33-70b'),
                name='EVA Llama 3.33 70B',
                prices=ModelPrice(input_mtok=Decimal('4'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='eva-qwen-2.5-32b',
                match=ClauseEquals(equals='eva-qwen-2.5-32b'),
                name='EVA Qwen2.5 32B',
                prices=ModelPrice(input_mtok=Decimal('2.6'), output_mtok=Decimal('3.4')),
            ),
            ModelInfo(
                id='eva-qwen-2.5-72b',
                match=ClauseEquals(equals='eva-qwen-2.5-72b'),
                name='EVA Qwen2.5 72B',
                prices=ModelPrice(input_mtok=Decimal('4'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='eva-unit-01/eva-llama-3.33-70b',
                match=ClauseEquals(equals='eva-unit-01/eva-llama-3.33-70b'),
                prices=ModelPrice(input_mtok=Decimal('4'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='eva-unit-01/eva-qwen-2.5-32b',
                match=ClauseEquals(equals='eva-unit-01/eva-qwen-2.5-32b'),
                prices=ModelPrice(input_mtok=Decimal('2.6'), output_mtok=Decimal('3.4')),
            ),
            ModelInfo(
                id='eva-unit-01/eva-qwen-2.5-72b',
                match=ClauseEquals(equals='eva-unit-01/eva-qwen-2.5-72b'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='featherless/qwerky-72b:free',
                match=ClauseEquals(equals='featherless/qwerky-72b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='fimbulvetr-11b-v2',
                match=ClauseEquals(equals='fimbulvetr-11b-v2'),
                name='Fimbulvetr 11B v2',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='gemini-2.0-flash-001',
                match=ClauseEquals(equals='gemini-2.0-flash-001'),
                name='Gemini 2.0 Flash',
                prices=ModelPrice(
                    input_mtok=Decimal('0.1'),
                    cache_write_mtok=Decimal('0.1833'),
                    cache_read_mtok=Decimal('0.025'),
                    output_mtok=Decimal('0.4'),
                ),
            ),
            ModelInfo(
                id='gemini-2.0-flash-exp:free',
                match=ClauseEquals(equals='gemini-2.0-flash-exp:free'),
                name='Gemini 2.0 Flash Experimental (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemini-2.0-flash-lite-001',
                match=ClauseEquals(equals='gemini-2.0-flash-lite-001'),
                name='Gemini 2.0 Flash Lite',
                prices=ModelPrice(input_mtok=Decimal('0.075'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='gemini-2.5-flash',
                match=ClauseEquals(equals='gemini-2.5-flash'),
                name='Gemini 2.5 Flash',
                prices=ModelPrice(
                    input_mtok=Decimal('0.3'),
                    cache_write_mtok=Decimal('0.3833'),
                    cache_read_mtok=Decimal('0.075'),
                    output_mtok=Decimal('2.5'),
                ),
            ),
            ModelInfo(
                id='gemini-2.5-flash-lite-preview-06-17',
                match=ClauseEquals(equals='gemini-2.5-flash-lite-preview-06-17'),
                name='Gemini 2.5 Flash Lite Preview 06-17',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='gemini-2.5-flash-preview',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gemini-2.5-flash-preview'),
                        ClauseEquals(equals='gemini-2.5-flash-preview-05-20'),
                    ]
                ),
                name='Gemini 2.5 Flash Preview 04-17',
                prices=ModelPrice(
                    input_mtok=Decimal('0.15'),
                    cache_write_mtok=Decimal('0.2333'),
                    cache_read_mtok=Decimal('0.0375'),
                    output_mtok=Decimal('0.6'),
                ),
            ),
            ModelInfo(
                id='gemini-2.5-flash-preview-05-20:thinking',
                match=ClauseEquals(equals='gemini-2.5-flash-preview-05-20:thinking'),
                name='Gemini 2.5 Flash Preview 05-20 (thinking)',
                prices=ModelPrice(
                    input_mtok=Decimal('0.15'),
                    cache_write_mtok=Decimal('0.2333'),
                    cache_read_mtok=Decimal('0.0375'),
                    output_mtok=Decimal('3.5'),
                ),
            ),
            ModelInfo(
                id='gemini-2.5-flash-preview:thinking',
                match=ClauseEquals(equals='gemini-2.5-flash-preview:thinking'),
                name='Gemini 2.5 Flash Preview 04-17 (thinking)',
                prices=ModelPrice(
                    input_mtok=Decimal('0.15'),
                    cache_write_mtok=Decimal('0.2333'),
                    cache_read_mtok=Decimal('0.0375'),
                    output_mtok=Decimal('3.5'),
                ),
            ),
            ModelInfo(
                id='gemini-2.5-pro',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gemini-2.5-pro'),
                        ClauseEquals(equals='gemini-2.5-pro-preview'),
                        ClauseEquals(equals='gemini-2.5-pro-preview-05-06'),
                    ]
                ),
                name='Gemini 2.5 Pro',
                prices=ModelPrice(
                    input_mtok=Decimal('1.25'),
                    cache_write_mtok=Decimal('1.625'),
                    cache_read_mtok=Decimal('0.31'),
                    output_mtok=Decimal('10'),
                ),
            ),
            ModelInfo(
                id='gemini-2.5-pro-exp-03-25',
                match=ClauseEquals(equals='gemini-2.5-pro-exp-03-25'),
                name='Gemini 2.5 Pro Experimental',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemini-flash-1.5',
                match=ClauseEquals(equals='gemini-flash-1.5'),
                name='Gemini 1.5 Flash',
                prices=ModelPrice(
                    input_mtok=Decimal('0.075'),
                    cache_write_mtok=Decimal('0.1583'),
                    cache_read_mtok=Decimal('0.01875'),
                    output_mtok=Decimal('0.3'),
                ),
            ),
            ModelInfo(
                id='gemini-flash-1.5-8b',
                match=ClauseEquals(equals='gemini-flash-1.5-8b'),
                name='Gemini 1.5 Flash 8B',
                prices=ModelPrice(
                    input_mtok=Decimal('0.0375'),
                    cache_write_mtok=Decimal('0.0583'),
                    cache_read_mtok=Decimal('0.01'),
                    output_mtok=Decimal('0.15'),
                ),
            ),
            ModelInfo(
                id='gemini-pro-1.5',
                match=ClauseEquals(equals='gemini-pro-1.5'),
                name='Gemini 1.5 Pro',
                prices=ModelPrice(input_mtok=Decimal('1.25'), output_mtok=Decimal('5')),
            ),
            ModelInfo(
                id='gemma-2-27b-it',
                match=ClauseEquals(equals='gemma-2-27b-it'),
                name='Gemma 2 27B',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='gemma-2-9b-it',
                match=ClauseEquals(equals='gemma-2-9b-it'),
                name='Gemma 2 9B',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='gemma-2-9b-it:free',
                match=ClauseEquals(equals='gemma-2-9b-it:free'),
                name='Gemma 2 9B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemma-3-12b-it',
                match=ClauseEquals(equals='gemma-3-12b-it'),
                name='Gemma 3 12B',
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='gemma-3-12b-it:free',
                match=ClauseEquals(equals='gemma-3-12b-it:free'),
                name='Gemma 3 12B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemma-3-27b-it',
                match=ClauseEquals(equals='gemma-3-27b-it'),
                name='Gemma 3 27B',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='gemma-3-27b-it:free',
                match=ClauseEquals(equals='gemma-3-27b-it:free'),
                name='Gemma 3 27B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemma-3-4b-it',
                match=ClauseEquals(equals='gemma-3-4b-it'),
                name='Gemma 3 4B',
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='gemma-3-4b-it:free',
                match=ClauseEquals(equals='gemma-3-4b-it:free'),
                name='Gemma 3 4B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='gemma-3n-e4b-it:free',
                match=ClauseEquals(equals='gemma-3n-e4b-it:free'),
                name='Gemma 3n 4B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='glm-4-32b',
                match=ClauseEquals(equals='glm-4-32b'),
                name='GLM 4 32B',
                prices=ModelPrice(input_mtok=Decimal('0.24'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='glm-4-32b:free',
                match=ClauseEquals(equals='glm-4-32b:free'),
                name='GLM 4 32B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='glm-z1-32b',
                match=ClauseEquals(equals='glm-z1-32b'),
                name='GLM Z1 32B',
                prices=ModelPrice(input_mtok=Decimal('0.24'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='glm-z1-32b:free',
                match=ClauseEquals(equals='glm-z1-32b:free'),
                name='GLM Z1 32B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='glm-z1-rumination-32b',
                match=ClauseEquals(equals='glm-z1-rumination-32b'),
                name='GLM Z1 Rumination 32B',
                prices=ModelPrice(input_mtok=Decimal('0.24'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='goliath-120b',
                match=ClauseEquals(equals='goliath-120b'),
                name='Goliath 120B',
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('12.5')),
            ),
            ModelInfo(
                id='google/gemini-2.0-flash-001',
                match=ClauseEquals(equals='google/gemini-2.0-flash-001'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='google/gemini-2.0-flash-exp:free',
                match=ClauseEquals(equals='google/gemini-2.0-flash-exp:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/gemini-2.0-flash-lite-001',
                match=ClauseEquals(equals='google/gemini-2.0-flash-lite-001'),
                prices=ModelPrice(input_mtok=Decimal('0.075'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='google/gemini-2.0-flash-thinking-exp-1219:free',
                match=ClauseEquals(equals='google/gemini-2.0-flash-thinking-exp-1219:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/gemini-2.0-flash-thinking-exp:free',
                match=ClauseEquals(equals='google/gemini-2.0-flash-thinking-exp:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/gemini-2.5-flash-preview',
                match=ClauseEquals(equals='google/gemini-2.5-flash-preview'),
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='google/gemini-2.5-flash-preview:thinking',
                match=ClauseEquals(equals='google/gemini-2.5-flash-preview:thinking'),
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('3.5')),
            ),
            ModelInfo(
                id='google/gemini-2.5-pro-exp-03-25:free',
                match=ClauseEquals(equals='google/gemini-2.5-pro-exp-03-25:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/gemini-2.5-pro-preview-03-25',
                match=ClauseEquals(equals='google/gemini-2.5-pro-preview-03-25'),
                prices=ModelPrice(input_mtok=Decimal('1.25'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='google/gemini-flash-1.5',
                match=ClauseEquals(equals='google/gemini-flash-1.5'),
                prices=ModelPrice(input_mtok=Decimal('0.075'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='google/gemini-flash-1.5-8b',
                match=ClauseEquals(equals='google/gemini-flash-1.5-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.0375'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='google/gemini-flash-1.5-8b-exp',
                match=ClauseEquals(equals='google/gemini-flash-1.5-8b-exp'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/gemini-pro',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='google/gemini-pro'), ClauseEquals(equals='google/gemini-pro-vision')]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='google/gemini-pro-1.5',
                match=ClauseEquals(equals='google/gemini-pro-1.5'),
                prices=ModelPrice(input_mtok=Decimal('1.25'), output_mtok=Decimal('5')),
            ),
            ModelInfo(
                id='google/gemma-2-27b-it',
                match=ClauseEquals(equals='google/gemma-2-27b-it'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='google/gemma-2-9b-it',
                match=ClauseEquals(equals='google/gemma-2-9b-it'),
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.07')),
            ),
            ModelInfo(
                id='google/gemma-2-9b-it:free',
                match=ClauseEquals(equals='google/gemma-2-9b-it:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/gemma-3-12b-it',
                match=ClauseEquals(equals='google/gemma-3-12b-it'),
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='google/gemma-3-12b-it:free',
                match=ClauseEquals(equals='google/gemma-3-12b-it:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/gemma-3-1b-it:free',
                match=ClauseEquals(equals='google/gemma-3-1b-it:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/gemma-3-27b-it',
                match=ClauseEquals(equals='google/gemma-3-27b-it'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='google/gemma-3-27b-it:free',
                match=ClauseEquals(equals='google/gemma-3-27b-it:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/gemma-3-4b-it',
                match=ClauseEquals(equals='google/gemma-3-4b-it'),
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='google/gemma-3-4b-it:free',
                match=ClauseEquals(equals='google/gemma-3-4b-it:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/learnlm-1.5-pro-experimental:free',
                match=ClauseEquals(equals='google/learnlm-1.5-pro-experimental:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='google/palm-2-chat-bison',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='google/palm-2-chat-bison'),
                        ClauseEquals(equals='google/palm-2-chat-bison-32k'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='google/palm-2-codechat-bison',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='google/palm-2-codechat-bison'),
                        ClauseEquals(equals='google/palm-2-codechat-bison-32k'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-3.5-turbo'), ClauseEquals(equals='gpt-3.5-turbo-0125')]),
                name='GPT-3.5 Turbo',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo-0613',
                match=ClauseEquals(equals='gpt-3.5-turbo-0613'),
                name='GPT-3.5 Turbo (older v0613)',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo-1106',
                match=ClauseEquals(equals='gpt-3.5-turbo-1106'),
                name='GPT-3.5 Turbo 16k (older v1106)',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo-16k',
                match=ClauseEquals(equals='gpt-3.5-turbo-16k'),
                name='GPT-3.5 Turbo 16k',
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('4')),
            ),
            ModelInfo(
                id='gpt-3.5-turbo-instruct',
                match=ClauseEquals(equals='gpt-3.5-turbo-instruct'),
                name='GPT-3.5 Turbo Instruct',
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='gpt-4',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-4'), ClauseEquals(equals='gpt-4-0314')]),
                name='GPT-4',
                prices=ModelPrice(input_mtok=Decimal('30'), output_mtok=Decimal('60')),
            ),
            ModelInfo(
                id='gpt-4-1106-preview',
                match=ClauseEquals(equals='gpt-4-1106-preview'),
                name='GPT-4 Turbo (older v1106)',
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4-turbo',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-4-turbo'), ClauseEquals(equals='gpt-4-turbo-preview')]),
                name='GPT-4 Turbo',
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='gpt-4.1',
                match=ClauseEquals(equals='gpt-4.1'),
                name='GPT-4.1',
                prices=ModelPrice(input_mtok=Decimal('2'), cache_read_mtok=Decimal('0.5'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='gpt-4.1-mini',
                match=ClauseEquals(equals='gpt-4.1-mini'),
                name='GPT-4.1 Mini',
                prices=ModelPrice(
                    input_mtok=Decimal('0.4'), cache_read_mtok=Decimal('0.1'), output_mtok=Decimal('1.6')
                ),
            ),
            ModelInfo(
                id='gpt-4.1-nano',
                match=ClauseEquals(equals='gpt-4.1-nano'),
                name='GPT-4.1 Nano',
                prices=ModelPrice(
                    input_mtok=Decimal('0.1'), cache_read_mtok=Decimal('0.025'), output_mtok=Decimal('0.4')
                ),
            ),
            ModelInfo(
                id='gpt-4.5-preview',
                match=ClauseEquals(equals='gpt-4.5-preview'),
                name='GPT-4.5 (Preview)',
                prices=ModelPrice(
                    input_mtok=Decimal('75'), cache_read_mtok=Decimal('37.5'), output_mtok=Decimal('150')
                ),
            ),
            ModelInfo(
                id='gpt-4o',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='gpt-4o'),
                        ClauseEquals(equals='gpt-4o-2024-08-06'),
                        ClauseEquals(equals='gpt-4o-2024-11-20'),
                    ]
                ),
                name='GPT-4o',
                prices=ModelPrice(
                    input_mtok=Decimal('2.5'), cache_read_mtok=Decimal('1.25'), output_mtok=Decimal('10')
                ),
            ),
            ModelInfo(
                id='gpt-4o-2024-05-13',
                match=ClauseEquals(equals='gpt-4o-2024-05-13'),
                name='GPT-4o (2024-05-13)',
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='gpt-4o-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='gpt-4o-mini'), ClauseEquals(equals='gpt-4o-mini-2024-07-18')]),
                name='GPT-4o-mini',
                prices=ModelPrice(
                    input_mtok=Decimal('0.15'), cache_read_mtok=Decimal('0.075'), output_mtok=Decimal('0.6')
                ),
            ),
            ModelInfo(
                id='gpt-4o-mini-search-preview',
                match=ClauseEquals(equals='gpt-4o-mini-search-preview'),
                name='GPT-4o-mini Search Preview',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='gpt-4o-search-preview',
                match=ClauseEquals(equals='gpt-4o-search-preview'),
                name='GPT-4o Search Preview',
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='gpt-4o:extended',
                match=ClauseEquals(equals='gpt-4o:extended'),
                name='GPT-4o (extended)',
                prices=ModelPrice(input_mtok=Decimal('6'), output_mtok=Decimal('18')),
            ),
            ModelInfo(
                id='grok-2-1212',
                match=ClauseEquals(equals='grok-2-1212'),
                name='Grok 2 1212',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='grok-2-vision-1212',
                match=ClauseEquals(equals='grok-2-vision-1212'),
                name='Grok 2 Vision 1212',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='grok-3',
                match=ClauseOr(or_=[ClauseEquals(equals='grok-3'), ClauseEquals(equals='grok-3-beta')]),
                name='Grok 3',
                prices=ModelPrice(input_mtok=Decimal('3'), cache_read_mtok=Decimal('0.75'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='grok-3-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='grok-3-mini'), ClauseEquals(equals='grok-3-mini-beta')]),
                name='Grok 3 Mini',
                prices=ModelPrice(
                    input_mtok=Decimal('0.3'), cache_read_mtok=Decimal('0.075'), output_mtok=Decimal('0.5')
                ),
            ),
            ModelInfo(
                id='grok-beta',
                match=ClauseEquals(equals='grok-beta'),
                name='Grok Beta',
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='grok-vision-beta',
                match=ClauseEquals(equals='grok-vision-beta'),
                name='Grok Vision Beta',
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='gryphe/mythomax-l2-13b',
                match=ClauseEquals(equals='gryphe/mythomax-l2-13b'),
                prices=ModelPrice(input_mtok=Decimal('0.065'), output_mtok=Decimal('0.065')),
            ),
            ModelInfo(
                id='hermes-2-pro-llama-3-8b',
                match=ClauseEquals(equals='hermes-2-pro-llama-3-8b'),
                name='Hermes 2 Pro - Llama-3 8B',
                prices=ModelPrice(input_mtok=Decimal('0.025'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='hermes-3-llama-3.1-405b',
                match=ClauseEquals(equals='hermes-3-llama-3.1-405b'),
                name='Hermes 3 405B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='hermes-3-llama-3.1-70b',
                match=ClauseEquals(equals='hermes-3-llama-3.1-70b'),
                name='Hermes 3 70B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.12'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='huggingfaceh4/zephyr-7b-beta:free',
                match=ClauseEquals(equals='huggingfaceh4/zephyr-7b-beta:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='infermatic/mn-inferor-12b',
                match=ClauseEquals(equals='infermatic/mn-inferor-12b'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='inflection-3-pi',
                match=ClauseEquals(equals='inflection-3-pi'),
                name='Inflection 3 Pi',
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='inflection-3-productivity',
                match=ClauseEquals(equals='inflection-3-productivity'),
                name='Inflection 3 Productivity',
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='inflection/inflection-3-pi',
                match=ClauseEquals(equals='inflection/inflection-3-pi'),
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='inflection/inflection-3-productivity',
                match=ClauseEquals(equals='inflection/inflection-3-productivity'),
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='internvl3-14b:free',
                match=ClauseEquals(equals='internvl3-14b:free'),
                name='InternVL3 14B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='internvl3-2b:free',
                match=ClauseEquals(equals='internvl3-2b:free'),
                name='InternVL3 2B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='jamba-1.6-large',
                match=ClauseEquals(equals='jamba-1.6-large'),
                name='Jamba 1.6 Large',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='jamba-1.6-mini',
                match=ClauseEquals(equals='jamba-1.6-mini'),
                name='Jamba Mini 1.6',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='jondurbin/airoboros-l2-70b',
                match=ClauseEquals(equals='jondurbin/airoboros-l2-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.5')),
            ),
            ModelInfo(
                id='kimi-dev-72b:free',
                match=ClauseEquals(equals='kimi-dev-72b:free'),
                name='Kimi Dev 72b (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='kimi-vl-a3b-thinking:free',
                match=ClauseEquals(equals='kimi-vl-a3b-thinking:free'),
                name='Kimi VL A3B Thinking (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='l3-euryale-70b',
                match=ClauseEquals(equals='l3-euryale-70b'),
                name='Llama 3 Euryale 70B v2.1',
                prices=ModelPrice(input_mtok=Decimal('1.48'), output_mtok=Decimal('1.48')),
            ),
            ModelInfo(
                id='l3-lunaris-8b',
                match=ClauseEquals(equals='l3-lunaris-8b'),
                name='Llama 3 8B Lunaris',
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.05')),
            ),
            ModelInfo(
                id='l3.1-euryale-70b',
                match=ClauseEquals(equals='l3.1-euryale-70b'),
                name='Llama 3.1 Euryale 70B v2.2',
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='l3.3-euryale-70b',
                match=ClauseEquals(equals='l3.3-euryale-70b'),
                name='Llama 3.3 Euryale 70B',
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='latitudegames/wayfarer-large-70b-llama-3.3',
                match=ClauseEquals(equals='latitudegames/wayfarer-large-70b-llama-3.3'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='lfm-3b',
                match=ClauseEquals(equals='lfm-3b'),
                name='LFM 3B',
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.02')),
            ),
            ModelInfo(
                id='lfm-40b',
                match=ClauseEquals(equals='lfm-40b'),
                name='LFM 40B MoE',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='lfm-7b',
                match=ClauseEquals(equals='lfm-7b'),
                name='LFM 7B',
                prices=ModelPrice(input_mtok=Decimal('0.01'), output_mtok=Decimal('0.01')),
            ),
            ModelInfo(
                id='liquid/lfm-3b',
                match=ClauseEquals(equals='liquid/lfm-3b'),
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.02')),
            ),
            ModelInfo(
                id='liquid/lfm-40b',
                match=ClauseEquals(equals='liquid/lfm-40b'),
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='liquid/lfm-7b',
                match=ClauseEquals(equals='liquid/lfm-7b'),
                prices=ModelPrice(input_mtok=Decimal('0.01'), output_mtok=Decimal('0.01')),
            ),
            ModelInfo(
                id='llama-3-70b-instruct',
                match=ClauseEquals(equals='llama-3-70b-instruct'),
                name='Llama 3 70B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='llama-3-8b-instruct',
                match=ClauseEquals(equals='llama-3-8b-instruct'),
                name='Llama 3 8B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.03'), output_mtok=Decimal('0.06')),
            ),
            ModelInfo(
                id='llama-3-lumimaid-70b',
                match=ClauseEquals(equals='llama-3-lumimaid-70b'),
                name='Llama 3 Lumimaid 70B',
                prices=ModelPrice(input_mtok=Decimal('4'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='llama-3-lumimaid-8b',
                match=ClauseEquals(equals='llama-3-lumimaid-8b'),
                name='Llama 3 Lumimaid 8B',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('1.25')),
            ),
            ModelInfo(
                id='llama-3.1-405b',
                match=ClauseEquals(equals='llama-3.1-405b'),
                name='Llama 3.1 405B (base)',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='llama-3.1-405b-instruct',
                match=ClauseEquals(equals='llama-3.1-405b-instruct'),
                name='Llama 3.1 405B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='llama-3.1-70b-instruct',
                match=ClauseEquals(equals='llama-3.1-70b-instruct'),
                name='Llama 3.1 70B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.28')),
            ),
            ModelInfo(
                id='llama-3.1-8b-instruct',
                match=ClauseEquals(equals='llama-3.1-8b-instruct'),
                name='Llama 3.1 8B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.016'), output_mtok=Decimal('0.029')),
            ),
            ModelInfo(
                id='llama-3.1-8b-instruct:free',
                match=ClauseEquals(equals='llama-3.1-8b-instruct:free'),
                name='Llama 3.1 8B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-3.1-lumimaid-70b',
                match=ClauseEquals(equals='llama-3.1-lumimaid-70b'),
                name='Lumimaid v0.2 70B',
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('3')),
            ),
            ModelInfo(
                id='llama-3.1-lumimaid-8b',
                match=ClauseEquals(equals='llama-3.1-lumimaid-8b'),
                name='Lumimaid v0.2 8B',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('1.25')),
            ),
            ModelInfo(
                id='llama-3.1-nemotron-70b-instruct',
                match=ClauseEquals(equals='llama-3.1-nemotron-70b-instruct'),
                name='Llama 3.1 Nemotron 70B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.12'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='llama-3.1-nemotron-ultra-253b-v1',
                match=ClauseEquals(equals='llama-3.1-nemotron-ultra-253b-v1'),
                name='Llama 3.1 Nemotron Ultra 253B v1',
                prices=ModelPrice(input_mtok=Decimal('0.6'), output_mtok=Decimal('1.8')),
            ),
            ModelInfo(
                id='llama-3.1-nemotron-ultra-253b-v1:free',
                match=ClauseEquals(equals='llama-3.1-nemotron-ultra-253b-v1:free'),
                name='Llama 3.1 Nemotron Ultra 253B v1 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-3.1-sonar-large-128k-online',
                match=ClauseEquals(equals='llama-3.1-sonar-large-128k-online'),
                name='Llama 3.1 Sonar 70B Online',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='llama-3.1-sonar-small-128k-online',
                match=ClauseEquals(equals='llama-3.1-sonar-small-128k-online'),
                name='Llama 3.1 Sonar 8B Online',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='llama-3.2-11b-vision-instruct',
                match=ClauseEquals(equals='llama-3.2-11b-vision-instruct'),
                name='Llama 3.2 11B Vision Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.049'), output_mtok=Decimal('0.049')),
            ),
            ModelInfo(
                id='llama-3.2-11b-vision-instruct:free',
                match=ClauseEquals(equals='llama-3.2-11b-vision-instruct:free'),
                name='Llama 3.2 11B Vision Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-3.2-1b-instruct',
                match=ClauseEquals(equals='llama-3.2-1b-instruct'),
                name='Llama 3.2 1B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.005'), output_mtok=Decimal('0.01')),
            ),
            ModelInfo(
                id='llama-3.2-1b-instruct:free',
                match=ClauseEquals(equals='llama-3.2-1b-instruct:free'),
                name='Llama 3.2 1B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-3.2-3b-instruct',
                match=ClauseEquals(equals='llama-3.2-3b-instruct'),
                name='Llama 3.2 3B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.01'), output_mtok=Decimal('0.02')),
            ),
            ModelInfo(
                id='llama-3.2-3b-instruct:free',
                match=ClauseEquals(equals='llama-3.2-3b-instruct:free'),
                name='Llama 3.2 3B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-3.2-90b-vision-instruct',
                match=ClauseEquals(equals='llama-3.2-90b-vision-instruct'),
                name='Llama 3.2 90B Vision Instruct',
                prices=ModelPrice(input_mtok=Decimal('1.2'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='llama-3.3-70b-instruct',
                match=ClauseEquals(equals='llama-3.3-70b-instruct'),
                name='Llama 3.3 70B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='llama-3.3-70b-instruct:free',
                match=ClauseEquals(equals='llama-3.3-70b-instruct:free'),
                name='Llama 3.3 70B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-3.3-8b-instruct:free',
                match=ClauseEquals(equals='llama-3.3-8b-instruct:free'),
                name='Llama 3.3 8B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-3.3-nemotron-super-49b-v1',
                match=ClauseEquals(equals='llama-3.3-nemotron-super-49b-v1'),
                name='Llama 3.3 Nemotron Super 49B v1',
                prices=ModelPrice(input_mtok=Decimal('0.13'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='llama-3.3-nemotron-super-49b-v1:free',
                match=ClauseEquals(equals='llama-3.3-nemotron-super-49b-v1:free'),
                name='Llama 3.3 Nemotron Super 49B v1 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-4-maverick',
                match=ClauseEquals(equals='llama-4-maverick'),
                name='Llama 4 Maverick',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='llama-4-maverick:free',
                match=ClauseEquals(equals='llama-4-maverick:free'),
                name='Llama 4 Maverick (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-4-scout',
                match=ClauseEquals(equals='llama-4-scout'),
                name='Llama 4 Scout',
                prices=ModelPrice(input_mtok=Decimal('0.08'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='llama-4-scout:free',
                match=ClauseEquals(equals='llama-4-scout:free'),
                name='Llama 4 Scout (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='llama-guard-2-8b',
                match=ClauseEquals(equals='llama-guard-2-8b'),
                name='LlamaGuard 2 8B',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='llama-guard-3-8b',
                match=ClauseEquals(equals='llama-guard-3-8b'),
                name='Llama Guard 3 8B',
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.06')),
            ),
            ModelInfo(
                id='llama-guard-4-12b',
                match=ClauseEquals(equals='llama-guard-4-12b'),
                name='Llama Guard 4 12B',
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.05')),
            ),
            ModelInfo(
                id='llama3.1-typhoon2-70b-instruct',
                match=ClauseEquals(equals='llama3.1-typhoon2-70b-instruct'),
                name='Typhoon2 70B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.88'), output_mtok=Decimal('0.88')),
            ),
            ModelInfo(
                id='llemma_7b',
                match=ClauseEquals(equals='llemma_7b'),
                name='Llemma 7b',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='maestro-reasoning',
                match=ClauseEquals(equals='maestro-reasoning'),
                name='Maestro Reasoning',
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('3.3')),
            ),
            ModelInfo(
                id='magistral-medium-2506',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='magistral-medium-2506'),
                        ClauseEquals(equals='magistral-medium-2506:thinking'),
                    ]
                ),
                name='Magistral Medium 2506',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('5')),
            ),
            ModelInfo(
                id='magistral-small-2506',
                match=ClauseEquals(equals='magistral-small-2506'),
                name='Magistral Small 2506',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='magnum-72b',
                match=ClauseEquals(equals='magnum-72b'),
                name='Magnum 72B',
                prices=ModelPrice(input_mtok=Decimal('4'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='magnum-v2-72b',
                match=ClauseEquals(equals='magnum-v2-72b'),
                name='Magnum v2 72B',
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('3')),
            ),
            ModelInfo(
                id='magnum-v4-72b',
                match=ClauseEquals(equals='magnum-v4-72b'),
                name='Magnum v4 72B',
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('3')),
            ),
            ModelInfo(
                id='mai-ds-r1:free',
                match=ClauseEquals(equals='mai-ds-r1:free'),
                name='MAI DS R1 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mancer/weaver',
                match=ClauseEquals(equals='mancer/weaver'),
                prices=ModelPrice(input_mtok=Decimal('1.125'), output_mtok=Decimal('1.125')),
            ),
            ModelInfo(
                id='mercury-coder-small-beta',
                match=ClauseEquals(equals='mercury-coder-small-beta'),
                name='Mercury Coder Small Beta',
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='meta-llama/llama-2-13b-chat',
                match=ClauseEquals(equals='meta-llama/llama-2-13b-chat'),
                prices=ModelPrice(input_mtok=Decimal('0.22'), output_mtok=Decimal('0.22')),
            ),
            ModelInfo(
                id='meta-llama/llama-2-70b-chat',
                match=ClauseEquals(equals='meta-llama/llama-2-70b-chat'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='meta-llama/llama-3-70b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3-70b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='meta-llama/llama-3-8b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3-8b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.03'), output_mtok=Decimal('0.06')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.1-405b',
                match=ClauseEquals(equals='meta-llama/llama-3.1-405b'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.1-405b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.1-405b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.1-405b:free',
                match=ClauseEquals(equals='meta-llama/llama-3.1-405b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='meta-llama/llama-3.1-70b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.1-70b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.119'), output_mtok=Decimal('0.39')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.1-8b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.1-8b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.03')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.1-8b-instruct:free',
                match=ClauseEquals(equals='meta-llama/llama-3.1-8b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-11b-vision-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.2-11b-vision-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.049'), output_mtok=Decimal('0.049')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-11b-vision-instruct:free',
                match=ClauseEquals(equals='meta-llama/llama-3.2-11b-vision-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-1b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.2-1b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.01'), output_mtok=Decimal('0.01')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-1b-instruct:free',
                match=ClauseEquals(equals='meta-llama/llama-3.2-1b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-3b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.2-3b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.015'), output_mtok=Decimal('0.025')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-3b-instruct:free',
                match=ClauseEquals(equals='meta-llama/llama-3.2-3b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='meta-llama/llama-3.2-90b-vision-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.2-90b-vision-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.3-70b-instruct',
                match=ClauseEquals(equals='meta-llama/llama-3.3-70b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.25')),
            ),
            ModelInfo(
                id='meta-llama/llama-3.3-70b-instruct:free',
                match=ClauseEquals(equals='meta-llama/llama-3.3-70b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='meta-llama/llama-4-maverick',
                match=ClauseEquals(equals='meta-llama/llama-4-maverick'),
                prices=ModelPrice(input_mtok=Decimal('0.17'), output_mtok=Decimal('0.85')),
            ),
            ModelInfo(
                id='meta-llama/llama-4-maverick:free',
                match=ClauseEquals(equals='meta-llama/llama-4-maverick:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='meta-llama/llama-4-scout',
                match=ClauseEquals(equals='meta-llama/llama-4-scout'),
                prices=ModelPrice(input_mtok=Decimal('0.08'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='meta-llama/llama-4-scout:free',
                match=ClauseEquals(equals='meta-llama/llama-4-scout:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='meta-llama/llama-guard-2-8b',
                match=ClauseEquals(equals='meta-llama/llama-guard-2-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='meta-llama/llama-guard-3-8b',
                match=ClauseEquals(equals='meta-llama/llama-guard-3-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='microsoft/phi-3-medium-128k-instruct',
                match=ClauseEquals(equals='microsoft/phi-3-medium-128k-instruct'),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='microsoft/phi-3-mini-128k-instruct',
                match=ClauseEquals(equals='microsoft/phi-3-mini-128k-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='microsoft/phi-3.5-mini-128k-instruct',
                match=ClauseEquals(equals='microsoft/phi-3.5-mini-128k-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='microsoft/phi-4',
                match=ClauseEquals(equals='microsoft/phi-4'),
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.14')),
            ),
            ModelInfo(
                id='microsoft/phi-4-multimodal-instruct',
                match=ClauseEquals(equals='microsoft/phi-4-multimodal-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='microsoft/wizardlm-2-7b',
                match=ClauseEquals(equals='microsoft/wizardlm-2-7b'),
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.07')),
            ),
            ModelInfo(
                id='microsoft/wizardlm-2-8x22b',
                match=ClauseEquals(equals='microsoft/wizardlm-2-8x22b'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.5')),
            ),
            ModelInfo(
                id='midnight-rose-70b',
                match=ClauseEquals(equals='midnight-rose-70b'),
                name='Midnight Rose 70B',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='minimax-01',
                match=ClauseEquals(equals='minimax-01'),
                name='MiniMax-01',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('1.1')),
            ),
            ModelInfo(
                id='minimax-m1',
                match=ClauseEquals(equals='minimax-m1'),
                name='MiniMax M1',
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('1.65')),
            ),
            ModelInfo(
                id='minimax-m1:extended',
                match=ClauseEquals(equals='minimax-m1:extended'),
                name='MiniMax M1 (extended)',
                prices=ModelPrice(input_mtok=Decimal('0.55'), output_mtok=Decimal('2.2')),
            ),
            ModelInfo(
                id='minimax/minimax-01',
                match=ClauseEquals(equals='minimax/minimax-01'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('1.1')),
            ),
            ModelInfo(
                id='ministral-3b',
                match=ClauseEquals(equals='ministral-3b'),
                name='Ministral 3B',
                prices=ModelPrice(input_mtok=Decimal('0.04'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='ministral-8b',
                match=ClauseEquals(equals='ministral-8b'),
                name='Ministral 8B',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='mistral-7b-instruct',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='mistral-7b-instruct'), ClauseEquals(equals='mistral-7b-instruct-v0.3')]
                ),
                name='Mistral 7B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.028'), output_mtok=Decimal('0.054')),
            ),
            ModelInfo(
                id='mistral-7b-instruct-v0.1',
                match=ClauseEquals(equals='mistral-7b-instruct-v0.1'),
                name='Mistral 7B Instruct v0.1',
                prices=ModelPrice(input_mtok=Decimal('0.11'), output_mtok=Decimal('0.19')),
            ),
            ModelInfo(
                id='mistral-7b-instruct-v0.2',
                match=ClauseEquals(equals='mistral-7b-instruct-v0.2'),
                name='Mistral 7B Instruct v0.2',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='mistral-7b-instruct:free',
                match=ClauseEquals(equals='mistral-7b-instruct:free'),
                name='Mistral 7B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistral-large',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='mistral-large'),
                        ClauseEquals(equals='mistral-large-2407'),
                        ClauseEquals(equals='mistral-large-2411'),
                    ]
                ),
                name='Mistral Large',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='mistral-medium',
                match=ClauseEquals(equals='mistral-medium'),
                name='Mistral Medium',
                prices=ModelPrice(input_mtok=Decimal('2.75'), output_mtok=Decimal('8.1')),
            ),
            ModelInfo(
                id='mistral-medium-3',
                match=ClauseEquals(equals='mistral-medium-3'),
                name='Mistral Medium 3',
                prices=ModelPrice(input_mtok=Decimal('0.4'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='mistral-nemo',
                match=ClauseEquals(equals='mistral-nemo'),
                name='Mistral Nemo',
                prices=ModelPrice(input_mtok=Decimal('0.01'), output_mtok=Decimal('0.019')),
            ),
            ModelInfo(
                id='mistral-nemo:free',
                match=ClauseEquals(equals='mistral-nemo:free'),
                name='Mistral Nemo (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistral-saba',
                match=ClauseEquals(equals='mistral-saba'),
                name='Saba',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='mistral-small',
                match=ClauseEquals(equals='mistral-small'),
                name='Mistral Small',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='mistral-small-24b-instruct-2501',
                match=ClauseEquals(equals='mistral-small-24b-instruct-2501'),
                name='Mistral Small 3',
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.09')),
            ),
            ModelInfo(
                id='mistral-small-24b-instruct-2501:free',
                match=ClauseEquals(equals='mistral-small-24b-instruct-2501:free'),
                name='Mistral Small 3 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistral-small-3.1-24b-instruct',
                match=ClauseEquals(equals='mistral-small-3.1-24b-instruct'),
                name='Mistral Small 3.1 24B',
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='mistral-small-3.1-24b-instruct:free',
                match=ClauseEquals(equals='mistral-small-3.1-24b-instruct:free'),
                name='Mistral Small 3.1 24B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistral-small-3.2-24b-instruct:free',
                match=ClauseEquals(equals='mistral-small-3.2-24b-instruct:free'),
                name='Mistral Small 3.2 24B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistral-tiny',
                match=ClauseEquals(equals='mistral-tiny'),
                name='Mistral Tiny',
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('0.25')),
            ),
            ModelInfo(
                id='mistral/ministral-8b',
                match=ClauseEquals(equals='mistral/ministral-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='mistralai/codestral-2501',
                match=ClauseEquals(equals='mistralai/codestral-2501'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='mistralai/codestral-mamba',
                match=ClauseEquals(equals='mistralai/codestral-mamba'),
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('0.25')),
            ),
            ModelInfo(
                id='mistralai/ministral-3b',
                match=ClauseEquals(equals='mistralai/ministral-3b'),
                prices=ModelPrice(input_mtok=Decimal('0.04'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='mistralai/ministral-8b',
                match=ClauseEquals(equals='mistralai/ministral-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='mistralai/mistral-7b-instruct',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='mistralai/mistral-7b-instruct'),
                        ClauseEquals(equals='mistralai/mistral-7b-instruct-v0.3'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.029'), output_mtok=Decimal('0.059')),
            ),
            ModelInfo(
                id='mistralai/mistral-7b-instruct-v0.1',
                match=ClauseEquals(equals='mistralai/mistral-7b-instruct-v0.1'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='mistralai/mistral-7b-instruct-v0.2',
                match=ClauseEquals(equals='mistralai/mistral-7b-instruct-v0.2'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='mistralai/mistral-7b-instruct:free',
                match=ClauseEquals(equals='mistralai/mistral-7b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistralai/mistral-large',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='mistralai/mistral-large'),
                        ClauseEquals(equals='mistralai/mistral-large-2407'),
                        ClauseEquals(equals='mistralai/mistral-large-2411'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='mistralai/mistral-medium',
                match=ClauseEquals(equals='mistralai/mistral-medium'),
                prices=ModelPrice(input_mtok=Decimal('2.75'), output_mtok=Decimal('8.1')),
            ),
            ModelInfo(
                id='mistralai/mistral-nemo',
                match=ClauseEquals(equals='mistralai/mistral-nemo'),
                prices=ModelPrice(input_mtok=Decimal('0.035'), output_mtok=Decimal('0.08')),
            ),
            ModelInfo(
                id='mistralai/mistral-nemo:free',
                match=ClauseEquals(equals='mistralai/mistral-nemo:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistralai/mistral-saba',
                match=ClauseEquals(equals='mistralai/mistral-saba'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='mistralai/mistral-small',
                match=ClauseEquals(equals='mistralai/mistral-small'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='mistralai/mistral-small-24b-instruct-2501',
                match=ClauseEquals(equals='mistralai/mistral-small-24b-instruct-2501'),
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.14')),
            ),
            ModelInfo(
                id='mistralai/mistral-small-24b-instruct-2501:free',
                match=ClauseEquals(equals='mistralai/mistral-small-24b-instruct-2501:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistralai/mistral-small-3.1-24b-instruct',
                match=ClauseEquals(equals='mistralai/mistral-small-3.1-24b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='mistralai/mistral-small-3.1-24b-instruct:free',
                match=ClauseEquals(equals='mistralai/mistral-small-3.1-24b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mistralai/mistral-tiny',
                match=ClauseEquals(equals='mistralai/mistral-tiny'),
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('0.25')),
            ),
            ModelInfo(
                id='mistralai/mixtral-8x22b-instruct',
                match=ClauseEquals(equals='mistralai/mixtral-8x22b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='mistralai/mixtral-8x7b-instruct',
                match=ClauseEquals(equals='mistralai/mixtral-8x7b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.24'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='mistralai/pixtral-12b',
                match=ClauseEquals(equals='mistralai/pixtral-12b'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='mistralai/pixtral-large-2411',
                match=ClauseEquals(equals='mistralai/pixtral-large-2411'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='mixtral-8x22b-instruct',
                match=ClauseEquals(equals='mixtral-8x22b-instruct'),
                name='Mixtral 8x22B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='mixtral-8x7b-instruct',
                match=ClauseEquals(equals='mixtral-8x7b-instruct'),
                name='Mixtral 8x7B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.08'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='mn-celeste-12b',
                match=ClauseEquals(equals='mn-celeste-12b'),
                name='Mistral Nemo 12B Celeste',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='mn-inferor-12b',
                match=ClauseEquals(equals='mn-inferor-12b'),
                name='Mistral Nemo Inferor 12B',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='mn-starcannon-12b',
                match=ClauseEquals(equals='mn-starcannon-12b'),
                name='Starcannon 12B',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='moonshotai/kimi-vl-a3b-thinking:free',
                match=ClauseEquals(equals='moonshotai/kimi-vl-a3b-thinking:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='moonshotai/moonlight-16b-a3b-instruct:free',
                match=ClauseEquals(equals='moonshotai/moonlight-16b-a3b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='mythalion-13b',
                match=ClauseEquals(equals='mythalion-13b'),
                name='Mythalion 13B',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='mythomax-l2-13b',
                match=ClauseEquals(equals='mythomax-l2-13b'),
                name='MythoMax 13B',
                prices=ModelPrice(input_mtok=Decimal('0.065'), output_mtok=Decimal('0.065')),
            ),
            ModelInfo(
                id='neversleep/llama-3-lumimaid-70b',
                match=ClauseEquals(equals='neversleep/llama-3-lumimaid-70b'),
                prices=ModelPrice(input_mtok=Decimal('3.375'), output_mtok=Decimal('4.5')),
            ),
            ModelInfo(
                id='neversleep/llama-3-lumimaid-8b',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='neversleep/llama-3-lumimaid-8b'),
                        ClauseEquals(equals='neversleep/llama-3-lumimaid-8b:extended'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.09375'), output_mtok=Decimal('0.75')),
            ),
            ModelInfo(
                id='neversleep/llama-3.1-lumimaid-70b',
                match=ClauseEquals(equals='neversleep/llama-3.1-lumimaid-70b'),
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('2.25')),
            ),
            ModelInfo(
                id='neversleep/llama-3.1-lumimaid-8b',
                match=ClauseEquals(equals='neversleep/llama-3.1-lumimaid-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.09375'), output_mtok=Decimal('0.75')),
            ),
            ModelInfo(
                id='neversleep/noromaid-20b',
                match=ClauseEquals(equals='neversleep/noromaid-20b'),
                prices=ModelPrice(input_mtok=Decimal('0.75'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='noromaid-20b',
                match=ClauseEquals(equals='noromaid-20b'),
                name='Noromaid 20B',
                prices=ModelPrice(input_mtok=Decimal('1.25'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='nothingiisreal/mn-celeste-12b',
                match=ClauseEquals(equals='nothingiisreal/mn-celeste-12b'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='nous-hermes-2-mixtral-8x7b-dpo',
                match=ClauseEquals(equals='nous-hermes-2-mixtral-8x7b-dpo'),
                name='Hermes 2 Mixtral 8x7B DPO',
                prices=ModelPrice(input_mtok=Decimal('0.6'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='nousresearch/deephermes-3-llama-3-8b-preview:free',
                match=ClauseEquals(equals='nousresearch/deephermes-3-llama-3-8b-preview:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='nousresearch/hermes-2-pro-llama-3-8b',
                match=ClauseEquals(equals='nousresearch/hermes-2-pro-llama-3-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.025'), output_mtok=Decimal('0.04')),
            ),
            ModelInfo(
                id='nousresearch/hermes-3-llama-3.1-405b',
                match=ClauseEquals(equals='nousresearch/hermes-3-llama-3.1-405b'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='nousresearch/hermes-3-llama-3.1-70b',
                match=ClauseEquals(equals='nousresearch/hermes-3-llama-3.1-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.12'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='nousresearch/nous-hermes-2-mixtral-8x7b-dpo',
                match=ClauseEquals(equals='nousresearch/nous-hermes-2-mixtral-8x7b-dpo'),
                prices=ModelPrice(input_mtok=Decimal('0.6'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='nousresearch/nous-hermes-llama2-13b',
                match=ClauseEquals(equals='nousresearch/nous-hermes-llama2-13b'),
                prices=ModelPrice(input_mtok=Decimal('0.18'), output_mtok=Decimal('0.18')),
            ),
            ModelInfo(
                id='nova-lite-v1',
                match=ClauseEquals(equals='nova-lite-v1'),
                name='Nova Lite 1.0',
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='nova-micro-v1',
                match=ClauseEquals(equals='nova-micro-v1'),
                name='Nova Micro 1.0',
                prices=ModelPrice(input_mtok=Decimal('0.035'), output_mtok=Decimal('0.14')),
            ),
            ModelInfo(
                id='nova-pro-v1',
                match=ClauseEquals(equals='nova-pro-v1'),
                name='Nova Pro 1.0',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('3.2')),
            ),
            ModelInfo(
                id='nvidia/llama-3.1-nemotron-70b-instruct',
                match=ClauseEquals(equals='nvidia/llama-3.1-nemotron-70b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.12'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='nvidia/llama-3.1-nemotron-70b-instruct:free',
                match=ClauseEquals(equals='nvidia/llama-3.1-nemotron-70b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='nvidia/llama-3.1-nemotron-nano-8b-v1:free',
                match=ClauseEquals(equals='nvidia/llama-3.1-nemotron-nano-8b-v1:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='nvidia/llama-3.1-nemotron-ultra-253b-v1:free',
                match=ClauseEquals(equals='nvidia/llama-3.1-nemotron-ultra-253b-v1:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='nvidia/llama-3.3-nemotron-super-49b-v1:free',
                match=ClauseEquals(equals='nvidia/llama-3.3-nemotron-super-49b-v1:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='o1',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='o1'),
                        ClauseEquals(equals='o1-preview'),
                        ClauseEquals(equals='o1-preview-2024-09-12'),
                    ]
                ),
                name='o1',
                prices=ModelPrice(input_mtok=Decimal('15'), cache_read_mtok=Decimal('7.5'), output_mtok=Decimal('60')),
            ),
            ModelInfo(
                id='o1-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='o1-mini'), ClauseEquals(equals='o1-mini-2024-09-12')]),
                name='o1-mini',
                prices=ModelPrice(
                    input_mtok=Decimal('1.1'), cache_read_mtok=Decimal('0.55'), output_mtok=Decimal('4.4')
                ),
            ),
            ModelInfo(
                id='o1-pro',
                match=ClauseEquals(equals='o1-pro'),
                name='o1-pro',
                prices=ModelPrice(input_mtok=Decimal('150'), output_mtok=Decimal('600')),
            ),
            ModelInfo(
                id='o3',
                match=ClauseEquals(equals='o3'),
                name='o3',
                prices=ModelPrice(input_mtok=Decimal('2'), cache_read_mtok=Decimal('0.5'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='o3-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='o3-mini'), ClauseEquals(equals='o3-mini-high')]),
                name='o3 Mini',
                prices=ModelPrice(
                    input_mtok=Decimal('1.1'), cache_read_mtok=Decimal('0.55'), output_mtok=Decimal('4.4')
                ),
            ),
            ModelInfo(
                id='o3-pro',
                match=ClauseEquals(equals='o3-pro'),
                name='o3 Pro',
                prices=ModelPrice(input_mtok=Decimal('20'), output_mtok=Decimal('80')),
            ),
            ModelInfo(
                id='o4-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='o4-mini'), ClauseEquals(equals='o4-mini-high')]),
                name='o4 Mini',
                prices=ModelPrice(
                    input_mtok=Decimal('1.1'), cache_read_mtok=Decimal('0.275'), output_mtok=Decimal('4.4')
                ),
            ),
            ModelInfo(
                id='open-r1/olympiccoder-32b:free',
                match=ClauseEquals(equals='open-r1/olympiccoder-32b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='open-r1/olympiccoder-7b:free',
                match=ClauseEquals(equals='open-r1/olympiccoder-7b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='openai/chatgpt-4o-latest',
                match=ClauseEquals(equals='openai/chatgpt-4o-latest'),
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='openai/gpt-3.5-turbo',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='openai/gpt-3.5-turbo'), ClauseEquals(equals='openai/gpt-3.5-turbo-0125')]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='openai/gpt-3.5-turbo-0613',
                match=ClauseEquals(equals='openai/gpt-3.5-turbo-0613'),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='openai/gpt-3.5-turbo-1106',
                match=ClauseEquals(equals='openai/gpt-3.5-turbo-1106'),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='openai/gpt-3.5-turbo-16k',
                match=ClauseEquals(equals='openai/gpt-3.5-turbo-16k'),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('4')),
            ),
            ModelInfo(
                id='openai/gpt-3.5-turbo-instruct',
                match=ClauseEquals(equals='openai/gpt-3.5-turbo-instruct'),
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('2')),
            ),
            ModelInfo(
                id='openai/gpt-4',
                match=ClauseOr(or_=[ClauseEquals(equals='openai/gpt-4'), ClauseEquals(equals='openai/gpt-4-0314')]),
                prices=ModelPrice(input_mtok=Decimal('30'), output_mtok=Decimal('60')),
            ),
            ModelInfo(
                id='openai/gpt-4-1106-preview',
                match=ClauseEquals(equals='openai/gpt-4-1106-preview'),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='openai/gpt-4-32k',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='openai/gpt-4-32k'), ClauseEquals(equals='openai/gpt-4-32k-0314')]
                ),
                prices=ModelPrice(input_mtok=Decimal('60'), output_mtok=Decimal('120')),
            ),
            ModelInfo(
                id='openai/gpt-4-turbo',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='openai/gpt-4-turbo'), ClauseEquals(equals='openai/gpt-4-turbo-preview')]
                ),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('30')),
            ),
            ModelInfo(
                id='openai/gpt-4.1',
                match=ClauseEquals(equals='openai/gpt-4.1'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='openai/gpt-4.1-mini',
                match=ClauseEquals(equals='openai/gpt-4.1-mini'),
                prices=ModelPrice(input_mtok=Decimal('0.4'), output_mtok=Decimal('1.6')),
            ),
            ModelInfo(
                id='openai/gpt-4.1-nano',
                match=ClauseEquals(equals='openai/gpt-4.1-nano'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.4')),
            ),
            ModelInfo(
                id='openai/gpt-4.5-preview',
                match=ClauseEquals(equals='openai/gpt-4.5-preview'),
                prices=ModelPrice(input_mtok=Decimal('75'), output_mtok=Decimal('150')),
            ),
            ModelInfo(
                id='openai/gpt-4o',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='openai/gpt-4o'),
                        ClauseEquals(equals='openai/gpt-4o-2024-08-06'),
                        ClauseEquals(equals='openai/gpt-4o-2024-11-20'),
                        ClauseEquals(equals='openai/gpt-4o-search-preview'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('2.5'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='openai/gpt-4o-2024-05-13',
                match=ClauseEquals(equals='openai/gpt-4o-2024-05-13'),
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='openai/gpt-4o-mini',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='openai/gpt-4o-mini'),
                        ClauseEquals(equals='openai/gpt-4o-mini-2024-07-18'),
                        ClauseEquals(equals='openai/gpt-4o-mini-search-preview'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='openai/gpt-4o:extended',
                match=ClauseEquals(equals='openai/gpt-4o:extended'),
                prices=ModelPrice(input_mtok=Decimal('6'), output_mtok=Decimal('18')),
            ),
            ModelInfo(
                id='openai/o1',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='openai/o1'),
                        ClauseEquals(equals='openai/o1-preview'),
                        ClauseEquals(equals='openai/o1-preview-2024-09-12'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('15'), output_mtok=Decimal('60')),
            ),
            ModelInfo(
                id='openai/o1-mini',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='openai/o1-mini'), ClauseEquals(equals='openai/o1-mini-2024-09-12')]
                ),
                prices=ModelPrice(input_mtok=Decimal('1.1'), output_mtok=Decimal('4.4')),
            ),
            ModelInfo(
                id='openai/o1-pro',
                match=ClauseEquals(equals='openai/o1-pro'),
                prices=ModelPrice(input_mtok=Decimal('150'), output_mtok=Decimal('600')),
            ),
            ModelInfo(
                id='openai/o3',
                match=ClauseEquals(equals='openai/o3'),
                prices=ModelPrice(input_mtok=Decimal('10'), output_mtok=Decimal('40')),
            ),
            ModelInfo(
                id='openai/o3-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='openai/o3-mini'), ClauseEquals(equals='openai/o3-mini-high')]),
                prices=ModelPrice(input_mtok=Decimal('1.1'), output_mtok=Decimal('4.4')),
            ),
            ModelInfo(
                id='openai/o4-mini',
                match=ClauseOr(or_=[ClauseEquals(equals='openai/o4-mini'), ClauseEquals(equals='openai/o4-mini-high')]),
                prices=ModelPrice(input_mtok=Decimal('1.1'), output_mtok=Decimal('4.4')),
            ),
            ModelInfo(
                id='openchat/openchat-7b',
                match=ClauseEquals(equals='openchat/openchat-7b'),
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.07')),
            ),
            ModelInfo(
                id='openhands-lm-32b-v0.1',
                match=ClauseEquals(equals='openhands-lm-32b-v0.1'),
                name='OpenHands LM 32B V0.1',
                prices=ModelPrice(input_mtok=Decimal('2.6'), output_mtok=Decimal('3.4')),
            ),
            ModelInfo(
                id='perplexity/llama-3.1-sonar-large-128k-online',
                match=ClauseEquals(equals='perplexity/llama-3.1-sonar-large-128k-online'),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='perplexity/llama-3.1-sonar-small-128k-online',
                match=ClauseEquals(equals='perplexity/llama-3.1-sonar-small-128k-online'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='perplexity/r1-1776',
                match=ClauseEquals(equals='perplexity/r1-1776'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='perplexity/sonar',
                match=ClauseEquals(equals='perplexity/sonar'),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='perplexity/sonar-deep-research',
                match=ClauseEquals(equals='perplexity/sonar-deep-research'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='perplexity/sonar-pro',
                match=ClauseEquals(equals='perplexity/sonar-pro'),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='perplexity/sonar-reasoning',
                match=ClauseEquals(equals='perplexity/sonar-reasoning'),
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('5')),
            ),
            ModelInfo(
                id='perplexity/sonar-reasoning-pro',
                match=ClauseEquals(equals='perplexity/sonar-reasoning-pro'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='phi-3-medium-128k-instruct',
                match=ClauseEquals(equals='phi-3-medium-128k-instruct'),
                name='Phi-3 Medium 128K Instruct',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='phi-3-mini-128k-instruct',
                match=ClauseEquals(equals='phi-3-mini-128k-instruct'),
                name='Phi-3 Mini 128K Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='phi-3.5-mini-128k-instruct',
                match=ClauseEquals(equals='phi-3.5-mini-128k-instruct'),
                name='Phi-3.5 Mini 128K Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='phi-4',
                match=ClauseEquals(equals='phi-4'),
                name='Phi 4',
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.14')),
            ),
            ModelInfo(
                id='phi-4-multimodal-instruct',
                match=ClauseEquals(equals='phi-4-multimodal-instruct'),
                name='Phi 4 Multimodal Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='phi-4-reasoning-plus',
                match=ClauseEquals(equals='phi-4-reasoning-plus'),
                name='Phi 4 Reasoning Plus',
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.35')),
            ),
            ModelInfo(
                id='phi-4-reasoning-plus:free',
                match=ClauseEquals(equals='phi-4-reasoning-plus:free'),
                name='Phi 4 Reasoning Plus (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='phi-4-reasoning:free',
                match=ClauseEquals(equals='phi-4-reasoning:free'),
                name='Phi 4 Reasoning (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='pixtral-12b',
                match=ClauseEquals(equals='pixtral-12b'),
                name='Pixtral 12B',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='pixtral-large-2411',
                match=ClauseEquals(equals='pixtral-large-2411'),
                name='Pixtral Large 2411',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('6')),
            ),
            ModelInfo(
                id='pygmalionai/mythalion-13b',
                match=ClauseEquals(equals='pygmalionai/mythalion-13b'),
                prices=ModelPrice(input_mtok=Decimal('0.5625'), output_mtok=Decimal('1.125')),
            ),
            ModelInfo(
                id='qwen-2-72b-instruct',
                match=ClauseEquals(equals='qwen-2-72b-instruct'),
                name='Qwen 2 72B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='qwen-2.5-72b-instruct',
                match=ClauseEquals(equals='qwen-2.5-72b-instruct'),
                name='Qwen2.5 72B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.12'), output_mtok=Decimal('0.39')),
            ),
            ModelInfo(
                id='qwen-2.5-72b-instruct:free',
                match=ClauseEquals(equals='qwen-2.5-72b-instruct:free'),
                name='Qwen2.5 72B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen-2.5-7b-instruct',
                match=ClauseEquals(equals='qwen-2.5-7b-instruct'),
                name='Qwen2.5 7B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.04'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='qwen-2.5-coder-32b-instruct',
                match=ClauseEquals(equals='qwen-2.5-coder-32b-instruct'),
                name='Qwen2.5 Coder 32B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='qwen-2.5-coder-32b-instruct:free',
                match=ClauseEquals(equals='qwen-2.5-coder-32b-instruct:free'),
                name='Qwen2.5 Coder 32B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen-2.5-vl-7b-instruct',
                match=ClauseEquals(equals='qwen-2.5-vl-7b-instruct'),
                name='Qwen2.5-VL 7B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='qwen-max',
                match=ClauseEquals(equals='qwen-max'),
                name='Qwen-Max',
                prices=ModelPrice(
                    input_mtok=Decimal('1.6'), cache_read_mtok=Decimal('0.64'), output_mtok=Decimal('6.4')
                ),
            ),
            ModelInfo(
                id='qwen-plus',
                match=ClauseEquals(equals='qwen-plus'),
                name='Qwen-Plus',
                prices=ModelPrice(
                    input_mtok=Decimal('0.4'), cache_read_mtok=Decimal('0.16'), output_mtok=Decimal('1.2')
                ),
            ),
            ModelInfo(
                id='qwen-turbo',
                match=ClauseEquals(equals='qwen-turbo'),
                name='Qwen-Turbo',
                prices=ModelPrice(
                    input_mtok=Decimal('0.05'), cache_read_mtok=Decimal('0.02'), output_mtok=Decimal('0.2')
                ),
            ),
            ModelInfo(
                id='qwen-vl-max',
                match=ClauseEquals(equals='qwen-vl-max'),
                name='Qwen VL Max',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('3.2')),
            ),
            ModelInfo(
                id='qwen-vl-plus',
                match=ClauseEquals(equals='qwen-vl-plus'),
                name='Qwen VL Plus',
                prices=ModelPrice(input_mtok=Decimal('0.21'), output_mtok=Decimal('0.63')),
            ),
            ModelInfo(
                id='qwen/qwen-2-72b-instruct',
                match=ClauseEquals(equals='qwen/qwen-2-72b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-72b-instruct',
                match=ClauseEquals(equals='qwen/qwen-2.5-72b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.12'), output_mtok=Decimal('0.39')),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-72b-instruct:free',
                match=ClauseEquals(equals='qwen/qwen-2.5-72b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-7b-instruct',
                match=ClauseEquals(equals='qwen/qwen-2.5-7b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-7b-instruct:free',
                match=ClauseEquals(equals='qwen/qwen-2.5-7b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-coder-32b-instruct',
                match=ClauseEquals(equals='qwen/qwen-2.5-coder-32b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.15')),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-coder-32b-instruct:free',
                match=ClauseEquals(equals='qwen/qwen-2.5-coder-32b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-vl-72b-instruct',
                match=ClauseEquals(equals='qwen/qwen-2.5-vl-72b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.6'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-vl-7b-instruct',
                match=ClauseEquals(equals='qwen/qwen-2.5-vl-7b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='qwen/qwen-2.5-vl-7b-instruct:free',
                match=ClauseEquals(equals='qwen/qwen-2.5-vl-7b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen/qwen-max',
                match=ClauseEquals(equals='qwen/qwen-max'),
                prices=ModelPrice(input_mtok=Decimal('1.6'), output_mtok=Decimal('6.4')),
            ),
            ModelInfo(
                id='qwen/qwen-plus',
                match=ClauseEquals(equals='qwen/qwen-plus'),
                prices=ModelPrice(input_mtok=Decimal('0.4'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='qwen/qwen-turbo',
                match=ClauseEquals(equals='qwen/qwen-turbo'),
                prices=ModelPrice(input_mtok=Decimal('0.05'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='qwen/qwen-vl-max',
                match=ClauseEquals(equals='qwen/qwen-vl-max'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('3.2')),
            ),
            ModelInfo(
                id='qwen/qwen-vl-plus',
                match=ClauseEquals(equals='qwen/qwen-vl-plus'),
                prices=ModelPrice(input_mtok=Decimal('0.21'), output_mtok=Decimal('0.63')),
            ),
            ModelInfo(
                id='qwen/qwen2.5-coder-7b-instruct',
                match=ClauseEquals(equals='qwen/qwen2.5-coder-7b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='qwen/qwen2.5-vl-32b-instruct',
                match=ClauseEquals(equals='qwen/qwen2.5-vl-32b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='qwen/qwen2.5-vl-32b-instruct:free',
                match=ClauseEquals(equals='qwen/qwen2.5-vl-32b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen/qwen2.5-vl-3b-instruct:free',
                match=ClauseEquals(equals='qwen/qwen2.5-vl-3b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen/qwen2.5-vl-72b-instruct',
                match=ClauseEquals(equals='qwen/qwen2.5-vl-72b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('0.7')),
            ),
            ModelInfo(
                id='qwen/qwen2.5-vl-72b-instruct:free',
                match=ClauseEquals(equals='qwen/qwen2.5-vl-72b-instruct:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen/qwq-32b',
                match=ClauseEquals(equals='qwen/qwq-32b'),
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='qwen/qwq-32b-preview',
                match=ClauseEquals(equals='qwen/qwq-32b-preview'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='qwen/qwq-32b-preview:free',
                match=ClauseEquals(equals='qwen/qwq-32b-preview:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(id='qwen/qwq-32b:free', match=ClauseEquals(equals='qwen/qwq-32b:free'), prices=ModelPrice()),
            ModelInfo(
                id='qwen2.5-vl-32b-instruct',
                match=ClauseEquals(equals='qwen2.5-vl-32b-instruct'),
                name='Qwen2.5 VL 32B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='qwen2.5-vl-32b-instruct:free',
                match=ClauseEquals(equals='qwen2.5-vl-32b-instruct:free'),
                name='Qwen2.5 VL 32B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen2.5-vl-72b-instruct',
                match=ClauseEquals(equals='qwen2.5-vl-72b-instruct'),
                name='Qwen2.5 VL 72B Instruct',
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('0.75')),
            ),
            ModelInfo(
                id='qwen2.5-vl-72b-instruct:free',
                match=ClauseEquals(equals='qwen2.5-vl-72b-instruct:free'),
                name='Qwen2.5 VL 72B Instruct (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen3-14b',
                match=ClauseEquals(equals='qwen3-14b'),
                name='Qwen3 14B',
                prices=ModelPrice(input_mtok=Decimal('0.06'), output_mtok=Decimal('0.24')),
            ),
            ModelInfo(
                id='qwen3-14b:free',
                match=ClauseEquals(equals='qwen3-14b:free'),
                name='Qwen3 14B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen3-235b-a22b',
                match=ClauseEquals(equals='qwen3-235b-a22b'),
                name='Qwen3 235B A22B',
                prices=ModelPrice(input_mtok=Decimal('0.13'), output_mtok=Decimal('0.6')),
            ),
            ModelInfo(
                id='qwen3-235b-a22b:free',
                match=ClauseEquals(equals='qwen3-235b-a22b:free'),
                name='Qwen3 235B A22B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen3-30b-a3b',
                match=ClauseEquals(equals='qwen3-30b-a3b'),
                name='Qwen3 30B A3B',
                prices=ModelPrice(input_mtok=Decimal('0.08'), output_mtok=Decimal('0.29')),
            ),
            ModelInfo(
                id='qwen3-30b-a3b:free',
                match=ClauseEquals(equals='qwen3-30b-a3b:free'),
                name='Qwen3 30B A3B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen3-32b',
                match=ClauseEquals(equals='qwen3-32b'),
                name='Qwen3 32B',
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='qwen3-32b:free',
                match=ClauseEquals(equals='qwen3-32b:free'),
                name='Qwen3 32B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwen3-8b',
                match=ClauseEquals(equals='qwen3-8b'),
                name='Qwen3 8B',
                prices=ModelPrice(input_mtok=Decimal('0.035'), output_mtok=Decimal('0.138')),
            ),
            ModelInfo(
                id='qwen3-8b:free',
                match=ClauseEquals(equals='qwen3-8b:free'),
                name='Qwen3 8B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwerky-72b:free',
                match=ClauseEquals(equals='qwerky-72b:free'),
                name='Qwerky 72B (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwq-32b',
                match=ClauseEquals(equals='qwq-32b'),
                name='QwQ 32B',
                prices=ModelPrice(input_mtok=Decimal('0.15'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='qwq-32b-arliai-rpr-v1:free',
                match=ClauseEquals(equals='qwq-32b-arliai-rpr-v1:free'),
                name='QwQ 32B RpR v1 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='qwq-32b-preview',
                match=ClauseEquals(equals='qwq-32b-preview'),
                name='QwQ 32B Preview',
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='qwq-32b:free', match=ClauseEquals(equals='qwq-32b:free'), name='QwQ 32B (free)', prices=ModelPrice()
            ),
            ModelInfo(
                id='r1-1776',
                match=ClauseEquals(equals='r1-1776'),
                name='R1 1776',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='raifle/sorcererlm-8x22b',
                match=ClauseEquals(equals='raifle/sorcererlm-8x22b'),
                prices=ModelPrice(input_mtok=Decimal('4.5'), output_mtok=Decimal('4.5')),
            ),
            ModelInfo(
                id='reka-flash-3:free',
                match=ClauseEquals(equals='reka-flash-3:free'),
                name='Flash 3 (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='rekaai/reka-flash-3:free',
                match=ClauseEquals(equals='rekaai/reka-flash-3:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='remm-slerp-l2-13b',
                match=ClauseEquals(equals='remm-slerp-l2-13b'),
                name='ReMM SLERP 13B',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='rocinante-12b',
                match=ClauseEquals(equals='rocinante-12b'),
                name='Rocinante 12B',
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('0.5')),
            ),
            ModelInfo(
                id='sao10k/fimbulvetr-11b-v2',
                match=ClauseEquals(equals='sao10k/fimbulvetr-11b-v2'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='sao10k/l3-euryale-70b',
                match=ClauseEquals(equals='sao10k/l3-euryale-70b'),
                prices=ModelPrice(input_mtok=Decimal('1.48'), output_mtok=Decimal('1.48')),
            ),
            ModelInfo(
                id='sao10k/l3-lunaris-8b',
                match=ClauseEquals(equals='sao10k/l3-lunaris-8b'),
                prices=ModelPrice(input_mtok=Decimal('0.02'), output_mtok=Decimal('0.05')),
            ),
            ModelInfo(
                id='sao10k/l3.1-euryale-70b',
                match=ClauseEquals(equals='sao10k/l3.1-euryale-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='sao10k/l3.3-euryale-70b',
                match=ClauseEquals(equals='sao10k/l3.3-euryale-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='sarvam-m:free',
                match=ClauseEquals(equals='sarvam-m:free'),
                name='Sarvam-M (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='scb10x/llama3.1-typhoon2-70b-instruct',
                match=ClauseEquals(equals='scb10x/llama3.1-typhoon2-70b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.88'), output_mtok=Decimal('0.88')),
            ),
            ModelInfo(
                id='scb10x/llama3.1-typhoon2-8b-instruct',
                match=ClauseEquals(equals='scb10x/llama3.1-typhoon2-8b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.18'), output_mtok=Decimal('0.18')),
            ),
            ModelInfo(
                id='shisa-ai/shisa-v2-llama3.3-70b:free',
                match=ClauseEquals(equals='shisa-ai/shisa-v2-llama3.3-70b:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='shisa-v2-llama3.3-70b:free',
                match=ClauseEquals(equals='shisa-v2-llama3.3-70b:free'),
                name='Shisa V2 Llama 3.3 70B  (free)',
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='skyfall-36b-v2',
                match=ClauseEquals(equals='skyfall-36b-v2'),
                name='Skyfall 36B V2',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='sonar',
                match=ClauseEquals(equals='sonar'),
                name='Sonar',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='sonar-deep-research',
                match=ClauseEquals(equals='sonar-deep-research'),
                name='Sonar Deep Research',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='sonar-pro',
                match=ClauseEquals(equals='sonar-pro'),
                name='Sonar Pro',
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='sonar-reasoning',
                match=ClauseEquals(equals='sonar-reasoning'),
                name='Sonar Reasoning',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('5')),
            ),
            ModelInfo(
                id='sonar-reasoning-pro',
                match=ClauseEquals(equals='sonar-reasoning-pro'),
                name='Sonar Reasoning Pro',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='sophosympatheia/midnight-rose-70b',
                match=ClauseEquals(equals='sophosympatheia/midnight-rose-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='sophosympatheia/rogue-rose-103b-v0.2:free',
                match=ClauseEquals(equals='sophosympatheia/rogue-rose-103b-v0.2:free'),
                prices=ModelPrice(),
            ),
            ModelInfo(
                id='sorcererlm-8x22b',
                match=ClauseEquals(equals='sorcererlm-8x22b'),
                name='SorcererLM 8x22B',
                prices=ModelPrice(input_mtok=Decimal('4.5'), output_mtok=Decimal('4.5')),
            ),
            ModelInfo(
                id='spotlight',
                match=ClauseEquals(equals='spotlight'),
                name='Spotlight',
                prices=ModelPrice(input_mtok=Decimal('0.18'), output_mtok=Decimal('0.18')),
            ),
            ModelInfo(
                id='steelskull/l3.3-electra-r1-70b',
                match=ClauseEquals(equals='steelskull/l3.3-electra-r1-70b'),
                prices=ModelPrice(input_mtok=Decimal('0.7'), output_mtok=Decimal('0.95')),
            ),
            ModelInfo(
                id='thedrummer/anubis-pro-105b-v1',
                match=ClauseEquals(equals='thedrummer/anubis-pro-105b-v1'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='thedrummer/rocinante-12b',
                match=ClauseEquals(equals='thedrummer/rocinante-12b'),
                prices=ModelPrice(input_mtok=Decimal('0.25'), output_mtok=Decimal('0.5')),
            ),
            ModelInfo(
                id='thedrummer/skyfall-36b-v2',
                match=ClauseEquals(equals='thedrummer/skyfall-36b-v2'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='thedrummer/unslopnemo-12b',
                match=ClauseEquals(equals='thedrummer/unslopnemo-12b'),
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.5')),
            ),
            ModelInfo(
                id='thudm/glm-4-32b:free', match=ClauseEquals(equals='thudm/glm-4-32b:free'), prices=ModelPrice()
            ),
            ModelInfo(
                id='thudm/glm-z1-32b:free', match=ClauseEquals(equals='thudm/glm-z1-32b:free'), prices=ModelPrice()
            ),
            ModelInfo(
                id='toppy-m-7b',
                match=ClauseEquals(equals='toppy-m-7b'),
                name='Toppy M 7B',
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='undi95/remm-slerp-l2-13b',
                match=ClauseEquals(equals='undi95/remm-slerp-l2-13b'),
                prices=ModelPrice(input_mtok=Decimal('0.5625'), output_mtok=Decimal('1.125')),
            ),
            ModelInfo(
                id='undi95/toppy-m-7b',
                match=ClauseEquals(equals='undi95/toppy-m-7b'),
                prices=ModelPrice(input_mtok=Decimal('0.07'), output_mtok=Decimal('0.07')),
            ),
            ModelInfo(
                id='unslopnemo-12b',
                match=ClauseEquals(equals='unslopnemo-12b'),
                name='UnslopNemo 12B',
                prices=ModelPrice(input_mtok=Decimal('0.45'), output_mtok=Decimal('0.45')),
            ),
            ModelInfo(
                id='valkyrie-49b-v1',
                match=ClauseEquals(equals='valkyrie-49b-v1'),
                name='Valkyrie 49B V1',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='virtuoso-large',
                match=ClauseEquals(equals='virtuoso-large'),
                name='Virtuoso Large',
                prices=ModelPrice(input_mtok=Decimal('0.75'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='virtuoso-medium-v2',
                match=ClauseEquals(equals='virtuoso-medium-v2'),
                name='Virtuoso Medium V2',
                prices=ModelPrice(input_mtok=Decimal('0.5'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='weaver',
                match=ClauseEquals(equals='weaver'),
                name='Weaver (alpha)',
                prices=ModelPrice(input_mtok=Decimal('1.5'), output_mtok=Decimal('1.5')),
            ),
            ModelInfo(
                id='wizardlm-2-8x22b',
                match=ClauseEquals(equals='wizardlm-2-8x22b'),
                name='WizardLM-2 8x22B',
                prices=ModelPrice(input_mtok=Decimal('0.48'), output_mtok=Decimal('0.48')),
            ),
            ModelInfo(
                id='x-ai/grok-2-1212',
                match=ClauseEquals(equals='x-ai/grok-2-1212'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='x-ai/grok-2-vision-1212',
                match=ClauseEquals(equals='x-ai/grok-2-vision-1212'),
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='x-ai/grok-3-beta',
                match=ClauseEquals(equals='x-ai/grok-3-beta'),
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='x-ai/grok-3-mini-beta',
                match=ClauseEquals(equals='x-ai/grok-3-mini-beta'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.5')),
            ),
            ModelInfo(
                id='x-ai/grok-beta',
                match=ClauseEquals(equals='x-ai/grok-beta'),
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='x-ai/grok-vision-beta',
                match=ClauseEquals(equals='x-ai/grok-vision-beta'),
                prices=ModelPrice(input_mtok=Decimal('5'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='xwin-lm/xwin-lm-70b',
                match=ClauseEquals(equals='xwin-lm/xwin-lm-70b'),
                prices=ModelPrice(input_mtok=Decimal('3.75'), output_mtok=Decimal('3.75')),
            ),
            ModelInfo(
                id='yi-large',
                match=ClauseEquals(equals='yi-large'),
                name='Yi Large',
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('3')),
            ),
        ],
    ),
    Provider(
        id='perplexity',
        name='Perplexity',
        api_pattern='https://api\\.perplexity\\.ai',
        pricing_urls=['https://docs.perplexity.ai/guides/pricing'],
        price_comments='Prices per request vary based on usage, this is not represented here, instead we just take the highest price shown for `requests_kcount`.',
        models=[
            ModelInfo(
                id='llama-3.1-sonar-large-128k-online',
                match=ClauseEquals(equals='llama-3.1-sonar-large-128k-online'),
                name='Llama 3.1 Sonar 70B Online',
                description="Llama 3.1 Sonar is Perplexity's latest model family. It surpasses their earlier Sonar models in cost-efficiency, speed, and performance.",
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('1')),
            ),
            ModelInfo(
                id='llama-3.1-sonar-small-128k-online',
                match=ClauseEquals(equals='llama-3.1-sonar-small-128k-online'),
                name='Llama 3.1 Sonar 8B Online',
                description="Llama 3.1 Sonar is Perplexity's latest model family. It surpasses their earlier Sonar models in cost-efficiency, speed, and performance.",
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='r1-1776',
                match=ClauseEquals(equals='r1-1776'),
                name='R1 1776',
                description='R1 1776 is a version of DeepSeek-R1 that has been post-trained to remove censorship constraints related to topics restricted by the Chinese government. The model retains its original reasoning capabilities while providing direct responses to a wider range of queries. R1 1776 is an offline chat model that does not use the perplexity search subsystem.',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='sonar',
                match=ClauseEquals(equals='sonar'),
                name='Sonar',
                description='Sonar is lightweight, affordable, fast, and simple to use — now featuring citations and the ability to customize sources. It is designed for companies seeking to integrate lightweight question-and-answer features optimized for speed.',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('1'), requests_kcount=Decimal('12')),
            ),
            ModelInfo(
                id='sonar-deep-research',
                match=ClauseEquals(equals='sonar-deep-research'),
                name='Sonar Deep Research',
                description='Sonar Deep Research is a research-focused model designed for multi-step retrieval, synthesis, and reasoning across complex topics. It autonomously searches, reads, and evaluates sources, refining its approach as it gathers information. This enables comprehensive report generation across domains like finance, technology, health, and current events.',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8')),
            ),
            ModelInfo(
                id='sonar-pro',
                match=ClauseEquals(equals='sonar-pro'),
                name='Sonar Pro',
                description='Note: Sonar Pro pricing includes Perplexity search pricing. See details here',
                prices=ModelPrice(input_mtok=Decimal('3'), output_mtok=Decimal('15'), requests_kcount=Decimal('14')),
            ),
            ModelInfo(
                id='sonar-reasoning',
                match=ClauseEquals(equals='sonar-reasoning'),
                name='Sonar Reasoning',
                description='Sonar Reasoning is a reasoning model provided by Perplexity based on DeepSeek R1.',
                prices=ModelPrice(input_mtok=Decimal('1'), output_mtok=Decimal('5'), requests_kcount=Decimal('12')),
            ),
            ModelInfo(
                id='sonar-reasoning-pro',
                match=ClauseEquals(equals='sonar-reasoning-pro'),
                name='Sonar Reasoning Pro',
                description='Sonar Pro pricing includes Perplexity search pricing.',
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('8'), requests_kcount=Decimal('14')),
            ),
        ],
    ),
    Provider(
        id='together',
        name='Together AI',
        api_pattern='https://api\\.together\\.xyz',
        pricing_urls=['https://www.together.ai/pricing'],
        models=[
            ModelInfo(
                id='Austism/chronos-hermes-13b',
                match=ClauseEquals(equals='Austism/chronos-hermes-13b'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='Gryphe/MythoMax-L2-13b',
                match=ClauseEquals(equals='Gryphe/MythoMax-L2-13b'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='Nexusflow/NexusRaven-V2-13B',
                match=ClauseEquals(equals='Nexusflow/NexusRaven-V2-13B'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='NousResearch/Nous-Capybara-7B-V1p9',
                match=ClauseEquals(equals='NousResearch/Nous-Capybara-7B-V1p9'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='NousResearch/Nous-Hermes-2-Mixtral-8x7B-DPO',
                match=ClauseEquals(equals='NousResearch/Nous-Hermes-2-Mixtral-8x7B-DPO'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='NousResearch/Nous-Hermes-2-Mixtral-8x7B-SFT',
                match=ClauseEquals(equals='NousResearch/Nous-Hermes-2-Mixtral-8x7B-SFT'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='NousResearch/Nous-Hermes-2-Yi-34B',
                match=ClauseEquals(equals='NousResearch/Nous-Hermes-2-Yi-34B'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='NousResearch/Nous-Hermes-Llama2-13b',
                match=ClauseEquals(equals='NousResearch/Nous-Hermes-Llama2-13b'),
                prices=ModelPrice(input_mtok=Decimal('0.225'), output_mtok=Decimal('0.225')),
            ),
            ModelInfo(
                id='NousResearch/Nous-Hermes-llama-2-7b',
                match=ClauseEquals(equals='NousResearch/Nous-Hermes-llama-2-7b'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='Open-Orca/Mistral-7B-OpenOrca',
                match=ClauseEquals(equals='Open-Orca/Mistral-7B-OpenOrca'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='Qwen/Qwen1.5-0.5B',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='Qwen/Qwen1.5-0.5B'), ClauseEquals(equals='Qwen/Qwen1.5-0.5B-Chat')]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='Qwen/Qwen1.5-1.8B',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='Qwen/Qwen1.5-1.8B'), ClauseEquals(equals='Qwen/Qwen1.5-1.8B-Chat')]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='Qwen/Qwen1.5-14B',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='Qwen/Qwen1.5-14B'), ClauseEquals(equals='Qwen/Qwen1.5-14B-Chat')]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='Qwen/Qwen1.5-4B',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='Qwen/Qwen1.5-4B'), ClauseEquals(equals='Qwen/Qwen1.5-4B-Chat')]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='Qwen/Qwen1.5-72B',
                match=ClauseEquals(equals='Qwen/Qwen1.5-72B'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='Qwen/Qwen1.5-7B',
                match=ClauseOr(
                    or_=[ClauseEquals(equals='Qwen/Qwen1.5-7B'), ClauseEquals(equals='Qwen/Qwen1.5-7B-Chat')]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='Undi95/ReMM-SLERP-L2-13B',
                match=ClauseEquals(equals='Undi95/ReMM-SLERP-L2-13B'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='Undi95/Toppy-M-7B',
                match=ClauseEquals(equals='Undi95/Toppy-M-7B'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='WizardLM/WizardLM-13B-V1.2',
                match=ClauseEquals(equals='WizardLM/WizardLM-13B-V1.2'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='allenai/OLMo-7B',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='allenai/OLMo-7B'),
                        ClauseEquals(equals='allenai/OLMo-7B-Instruct'),
                        ClauseEquals(equals='allenai/OLMo-7B-Twin-2T'),
                    ]
                ),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='codellama/CodeLlama-13b-Instruct-hf',
                match=ClauseEquals(equals='codellama/CodeLlama-13b-Instruct-hf'),
                prices=ModelPrice(input_mtok=Decimal('0.225'), output_mtok=Decimal('0.225')),
            ),
            ModelInfo(
                id='codellama/CodeLlama-34b-Instruct-hf',
                match=ClauseEquals(equals='codellama/CodeLlama-34b-Instruct-hf'),
                prices=ModelPrice(input_mtok=Decimal('0.776'), output_mtok=Decimal('0.776')),
            ),
            ModelInfo(
                id='codellama/CodeLlama-70b-Instruct-hf',
                match=ClauseEquals(equals='codellama/CodeLlama-70b-Instruct-hf'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='codellama/CodeLlama-7b-Instruct-hf',
                match=ClauseEquals(equals='codellama/CodeLlama-7b-Instruct-hf'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='deepseek-ai/deepseek-coder-33b-instruct',
                match=ClauseEquals(equals='deepseek-ai/deepseek-coder-33b-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='garage-bAInd/Platypus2-70B-instruct',
                match=ClauseEquals(equals='garage-bAInd/Platypus2-70B-instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='google/gemma-2b',
                match=ClauseOr(or_=[ClauseEquals(equals='google/gemma-2b'), ClauseEquals(equals='google/gemma-2b-it')]),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='google/gemma-7b',
                match=ClauseOr(or_=[ClauseEquals(equals='google/gemma-7b'), ClauseEquals(equals='google/gemma-7b-it')]),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='lmsys/vicuna-13b-v1.5',
                match=ClauseEquals(equals='lmsys/vicuna-13b-v1.5'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='lmsys/vicuna-7b-v1.5',
                match=ClauseEquals(equals='lmsys/vicuna-7b-v1.5'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='meta-llama/Llama-2-13b-chat-hf',
                match=ClauseEquals(equals='meta-llama/Llama-2-13b-chat-hf'),
                prices=ModelPrice(input_mtok=Decimal('0.225'), output_mtok=Decimal('0.225')),
            ),
            ModelInfo(
                id='meta-llama/Llama-2-70b-chat-hf',
                match=ClauseEquals(equals='meta-llama/Llama-2-70b-chat-hf'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='meta-llama/Llama-2-7b-chat-hf',
                match=ClauseEquals(equals='meta-llama/Llama-2-7b-chat-hf'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='meta-llama/Llama-3-70b-chat-hf',
                match=ClauseEquals(equals='meta-llama/Llama-3-70b-chat-hf'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='meta-llama/Llama-3-8b-chat-hf',
                match=ClauseEquals(equals='meta-llama/Llama-3-8b-chat-hf'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='meta-llama/Llama-3.3-70B-Instruct-Turbo',
                match=ClauseEquals(equals='meta-llama/Llama-3.3-70B-Instruct-Turbo'),
                prices=ModelPrice(input_mtok=Decimal('0.88'), output_mtok=Decimal('0.88')),
            ),
            ModelInfo(
                id='meta-llama/Llama-4-Maverick-17B-128E-Instruct-FP8',
                match=ClauseEquals(equals='meta-llama/Llama-4-Maverick-17B-128E-Instruct-FP8'),
                prices=ModelPrice(input_mtok=Decimal('0.27'), output_mtok=Decimal('0.85')),
            ),
            ModelInfo(
                id='meta-llama/Llama-4-Scout-17B-16E-Instruct',
                match=ClauseEquals(equals='meta-llama/Llama-4-Scout-17B-16E-Instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.18'), output_mtok=Decimal('0.59')),
            ),
            ModelInfo(
                id='meta-llama/Meta-Llama-3-70B-Instruct-Lite',
                match=ClauseEquals(equals='meta-llama/Meta-Llama-3-70B-Instruct-Lite'),
                prices=ModelPrice(input_mtok=Decimal('0.54'), output_mtok=Decimal('0.54')),
            ),
            ModelInfo(
                id='meta-llama/Meta-Llama-3-70B-Instruct-Turbo',
                match=ClauseEquals(equals='meta-llama/Meta-Llama-3-70B-Instruct-Turbo'),
                prices=ModelPrice(input_mtok=Decimal('0.88'), output_mtok=Decimal('0.88')),
            ),
            ModelInfo(
                id='meta-llama/Meta-Llama-3-8B-Instruct-Lite',
                match=ClauseEquals(equals='meta-llama/Meta-Llama-3-8B-Instruct-Lite'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='meta-llama/Meta-Llama-3-8B-Instruct-Turbo',
                match=ClauseEquals(equals='meta-llama/Meta-Llama-3-8B-Instruct-Turbo'),
                prices=ModelPrice(input_mtok=Decimal('0.18'), output_mtok=Decimal('0.18')),
            ),
            ModelInfo(
                id='meta-llama/Meta-Llama-3.1-405B-Instruct-Turbo',
                match=ClauseEquals(equals='meta-llama/Meta-Llama-3.1-405B-Instruct-Turbo'),
                prices=ModelPrice(input_mtok=Decimal('3.5'), output_mtok=Decimal('3.5')),
            ),
            ModelInfo(
                id='meta-llama/Meta-Llama-3.1-70B-Instruct-Turbo',
                match=ClauseEquals(equals='meta-llama/Meta-Llama-3.1-70B-Instruct-Turbo'),
                prices=ModelPrice(input_mtok=Decimal('0.88'), output_mtok=Decimal('0.88')),
            ),
            ModelInfo(
                id='meta-llama/Meta-Llama-3.1-8B-Instruct-Turbo',
                match=ClauseEquals(equals='meta-llama/Meta-Llama-3.1-8B-Instruct-Turbo'),
                prices=ModelPrice(input_mtok=Decimal('0.18'), output_mtok=Decimal('0.18')),
            ),
            ModelInfo(
                id='meta-llama/Meta-Llama-3.3-70B-Instruct-Turbo',
                match=ClauseEquals(equals='meta-llama/Meta-Llama-3.3-70B-Instruct-Turbo'),
                prices=ModelPrice(input_mtok=Decimal('0.88'), output_mtok=Decimal('0.88')),
            ),
            ModelInfo(
                id='microsoft/WizardLM-2-8x22B',
                match=ClauseEquals(equals='microsoft/WizardLM-2-8x22B'),
                prices=ModelPrice(input_mtok=Decimal('1.2'), output_mtok=Decimal('1.2')),
            ),
            ModelInfo(
                id='microsoft/phi-2',
                match=ClauseEquals(equals='microsoft/phi-2'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='mistralai/Mistral-7B-Instruct-v0.1',
                match=ClauseEquals(equals='mistralai/Mistral-7B-Instruct-v0.1'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='mistralai/Mistral-7B-Instruct-v0.2',
                match=ClauseEquals(equals='mistralai/Mistral-7B-Instruct-v0.2'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='mistralai/Mistral-7B-v0.1',
                match=ClauseEquals(equals='mistralai/Mistral-7B-v0.1'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='mistralai/Mixtral-8x22B-Instruct-v0.1',
                match=ClauseEquals(equals='mistralai/Mixtral-8x22B-Instruct-v0.1'),
                prices=ModelPrice(input_mtok=Decimal('2.4'), output_mtok=Decimal('2.4')),
            ),
            ModelInfo(
                id='mistralai/Mixtral-8x7B-Instruct-v0.1',
                match=ClauseEquals(equals='mistralai/Mixtral-8x7B-Instruct-v0.1'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='mistralai/Mixtral-8x7B-v0.1',
                match=ClauseEquals(equals='mistralai/Mixtral-8x7B-v0.1'),
                prices=ModelPrice(input_mtok=Decimal('0.9'), output_mtok=Decimal('0.9')),
            ),
            ModelInfo(
                id='openchat/openchat-3.5-1210',
                match=ClauseEquals(equals='openchat/openchat-3.5-1210'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='snorkelai/Snorkel-Mistral-PairRM-DPO',
                match=ClauseEquals(equals='snorkelai/Snorkel-Mistral-PairRM-DPO'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='teknium/OpenHermes-2-Mistral-7B',
                match=ClauseEquals(equals='teknium/OpenHermes-2-Mistral-7B'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='teknium/OpenHermes-2p5-Mistral-7B',
                match=ClauseEquals(equals='teknium/OpenHermes-2p5-Mistral-7B'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='togethercomputer/GPT-JT-Moderation-6B',
                match=ClauseEquals(equals='togethercomputer/GPT-JT-Moderation-6B'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='togethercomputer/Llama-2-7B-32K-Instruct',
                match=ClauseEquals(equals='togethercomputer/Llama-2-7B-32K-Instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='togethercomputer/RedPajama-INCITE-7B-Base',
                match=ClauseEquals(equals='togethercomputer/RedPajama-INCITE-7B-Base'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='togethercomputer/RedPajama-INCITE-7B-Chat',
                match=ClauseEquals(equals='togethercomputer/RedPajama-INCITE-7B-Chat'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='togethercomputer/RedPajama-INCITE-7B-Instruct',
                match=ClauseEquals(equals='togethercomputer/RedPajama-INCITE-7B-Instruct'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='togethercomputer/RedPajama-INCITE-Base-3B-v1',
                match=ClauseEquals(equals='togethercomputer/RedPajama-INCITE-Base-3B-v1'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='togethercomputer/RedPajama-INCITE-Chat-3B-v1',
                match=ClauseEquals(equals='togethercomputer/RedPajama-INCITE-Chat-3B-v1'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='togethercomputer/RedPajama-INCITE-Instruct-3B-v1',
                match=ClauseEquals(equals='togethercomputer/RedPajama-INCITE-Instruct-3B-v1'),
                prices=ModelPrice(input_mtok=Decimal('0.1'), output_mtok=Decimal('0.1')),
            ),
            ModelInfo(
                id='togethercomputer/StripedHyena-Hessian-7B',
                match=ClauseEquals(equals='togethercomputer/StripedHyena-Hessian-7B'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='togethercomputer/StripedHyena-Nous-7B',
                match=ClauseEquals(equals='togethercomputer/StripedHyena-Nous-7B'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='togethercomputer/alpaca-7b',
                match=ClauseEquals(equals='togethercomputer/alpaca-7b'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
            ModelInfo(
                id='upstage/SOLAR-10.7B-Instruct-v1.0',
                match=ClauseEquals(equals='upstage/SOLAR-10.7B-Instruct-v1.0'),
                prices=ModelPrice(input_mtok=Decimal('0.3'), output_mtok=Decimal('0.3')),
            ),
            ModelInfo(
                id='zero-one-ai/Yi-34B',
                match=ClauseEquals(equals='zero-one-ai/Yi-34B'),
                prices=ModelPrice(input_mtok=Decimal('0.8'), output_mtok=Decimal('0.8')),
            ),
            ModelInfo(
                id='zero-one-ai/Yi-6B',
                match=ClauseEquals(equals='zero-one-ai/Yi-6B'),
                prices=ModelPrice(input_mtok=Decimal('0.2'), output_mtok=Decimal('0.2')),
            ),
        ],
    ),
    Provider(
        id='x-ai',
        name='X AI',
        api_pattern='https://api\\.x\\.ai',
        pricing_urls=['https://docs.x.ai/docs/models'],
        model_match=ClauseContains(contains='grok'),
        models=[
            ModelInfo(
                id='grok-2-1212',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='grok-2-1212'),
                        ClauseEquals(equals='grok-2'),
                        ClauseEquals(equals='grok-2-latest'),
                    ]
                ),
                name='Grok 2 1212',
                description='(deprecated) Grok 2 1212 introduces significant enhancements to accuracy, instruction adherence, and multilingual support, making it a powerful and flexible choice for developers seeking a highly steerable, intelligent model.',
                context_window=32768,
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='grok-2-vision-1212',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='grok-2-vision-1212'),
                        ClauseEquals(equals='grok-2-vision'),
                        ClauseEquals(equals='grok-2-vision-latest'),
                    ]
                ),
                name='Grok 2 Vision 1212',
                description='Our multimodal model that processes documents, diagrams, charts, screenshots, and photographs.',
                context_window=32768,
                prices=ModelPrice(input_mtok=Decimal('2'), output_mtok=Decimal('10')),
            ),
            ModelInfo(
                id='grok-3',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='grok-3'),
                        ClauseEquals(equals='grok-3-latest'),
                        ClauseEquals(equals='grok-3-beta'),
                    ]
                ),
                name='Grok 3',
                description='Flagship model that excels at enterprise use cases like data extraction, coding, and text summarization. Possesses deep domain knowledge in finance, healthcare, law, and science.',
                context_window=131072,
                prices=ModelPrice(input_mtok=Decimal('3'), cache_read_mtok=Decimal('0.75'), output_mtok=Decimal('15')),
            ),
            ModelInfo(
                id='grok-3-fast',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='grok-3-fast'),
                        ClauseEquals(equals='grok-3-fast-latest'),
                        ClauseEquals(equals='grok-3-fast-beta'),
                    ]
                ),
                name='Grok 3 Fast',
                description='Excels at enterprise use cases like data extraction, coding, and text summarization. Possesses deep domain knowledge in finance, healthcare, law, and science.',
                context_window=131072,
                prices=ModelPrice(input_mtok=Decimal('5'), cache_read_mtok=Decimal('1.25'), output_mtok=Decimal('25')),
            ),
            ModelInfo(
                id='grok-3-mini',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='grok-3-mini'),
                        ClauseEquals(equals='grok-3-mini-beta'),
                        ClauseEquals(equals='grok-3-mini-latest'),
                    ]
                ),
                name='Grok 3 Mini',
                description='A lightweight model that thinks before responding. Fast, smart, and great for logic-based tasks that do not require deep domain knowledge. The raw thinking traces are accessible.',
                context_window=131072,
                prices=ModelPrice(
                    input_mtok=Decimal('0.3'), cache_read_mtok=Decimal('0.075'), output_mtok=Decimal('0.5')
                ),
            ),
            ModelInfo(
                id='grok-3-mini-fast',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='grok-3-mini-fast'),
                        ClauseEquals(equals='grok-3-mini-fast-beta'),
                        ClauseEquals(equals='grok-3-mini-fast-latest'),
                    ]
                ),
                name='Grok 3 Mini Fast',
                description='A lightweight model that thinks before responding. Fast, smart, and great for logic-based tasks that do not require deep domain knowledge. The raw thinking traces are accessible.',
                context_window=131072,
                prices=ModelPrice(input_mtok=Decimal('0.6'), cache_read_mtok=Decimal('0.15'), output_mtok=Decimal('4')),
            ),
            ModelInfo(
                id='grok-4-0709',
                match=ClauseOr(
                    or_=[
                        ClauseEquals(equals='grok-4-0709'),
                        ClauseEquals(equals='grok-4'),
                        ClauseEquals(equals='grok-4-latest'),
                    ]
                ),
                name='Grok 4',
                description='A flagship model, offering unparalleled performance in natural language, math and reasoning - the perfect jack of all trades.',
                context_window=256000,
                prices=ModelPrice(input_mtok=Decimal('3'), cache_read_mtok=Decimal('0.75'), output_mtok=Decimal('15')),
            ),
        ],
    ),
]
