import re
from ..autodoc.request import URLParameter
from ..autodoc.schema import String

class SimpleRoutingRoute:
    _di = None
    _handler = None
    _methods = None
    _path = None
    _path_parts = None
    _resource_paths = None

    def __init__(self, di):
        self._di = di

    def configure(
        self,
        handler_class,
        handler_config,
        path=None,
        methods=None,
        authentication=None,
        response_headers=None,
        security_headers=None
    ):
        if authentication is not None and not handler_config.get('authentication'):
            handler_config['authentication'] = authentication
        response_headers = response_headers if response_headers is not None else {}
        if 'response_headers' in handler_config:
            if type(handler_config['response_headers']) != dict:
                raise ValueError("Invalid configuration: 'response_headers' must be a dictionary")
            response_headers = {**response_headers, **handler_config['response_headers']}
        self._path = path
        if handler_config.get('base_url'):
            self._path = path.rstrip('/') + '/' + handler_config.get('base_url').lstrip('/')
        self._path_parts = self._path.strip('/').split('/') if self._path is not None else []
        self._resource_paths = self._extract_resource_paths(self._path_parts)
        if methods is not None:
            self._methods = [methods.upper()] if isinstance(methods, str) else [met.upper() for met in methods]
        sub_handler_config = {
            **handler_config,
            **{
                'base_url': ('/' + path.strip('/')) if path is not None else '/',
            }
        }
        if response_headers:
            sub_handler_config['response_headers'] = response_headers
        security_headers = security_headers if security_headers is not None else []
        if 'security_headers' in handler_config:
            security_headers = [*security_headers, *handler_config['security_headers']]
        sub_handler_config['security_headers'] = security_headers
        self._handler = self._di.build(handler_class, cache=False)
        self._handler.configure(sub_handler_config)

    def _extract_resource_paths(self, path_parts):
        resource_paths = {}
        for (index, part) in enumerate(path_parts):
            if not part:
                continue
            if part[0] != '{':
                continue
            if part[-1] != '}':
                raise ValueError(f"Invalid route configuration for URL '{path}': section '{part}'" + " starts with a '{' but does not end with one")
            match = re.match('{(\w[\w\d_]{0,})\}', part)
            if not match:
                raise ValueError(f"Invalid route configuration for URL '{path}', section '{part}': resource identifiers must start with a letter and contain only letters, numbers, and underscores")
            resource_paths[index] = match.group(1)
        return resource_paths

    def matches(self, full_path, request_method, is_cors=False):
        """ Returns None if the route doesn't match, or a dictionary with route data for a match.

        You can't just match true/false against the return value, because of the route matches
        but has no route data, it returns an empty dictionary.  Check explicitly for None
        to understand if there was no route match at all.
        """
        # If we're routing for CORS then ignore the request method (since it won't match)
        if not is_cors and self._methods is not None and request_method not in self._methods:
            return None
        if self._resource_paths:
            return self._resource_path_match(full_path, self._path_parts, self._resource_paths)
        if self._path is not None:
            full_path = full_path.strip('/')
            my_path = self._path.strip('/')
            my_path_length = len(my_path)
            full_path_length = len(full_path)
            if my_path_length > full_path_length:
                return None
            if full_path[:my_path_length] != my_path:
                return None
            # make sure we don't get confused by partial matches.  `user` should match `user/` and `user/5`,
            # but it shouldn't match `users/`
            if full_path_length > my_path_length and full_path[my_path_length] != '/':
                return None
        return {}

    def _resource_path_match(self, requested_path, path_parts, resource_paths):
        """ Returns None if the route doesn't match, or a dictionary with route data for the match."""
        requested_parts = requested_path.strip('/').split('/')
        route_data = {}
        path_length = len(path_parts)
        # it's okay if the requested path is longer than the configured path, since there may
        # be sub-routes that we don't know about.  However, we won't ever have a match if
        # the requested path is shorter than the configured path.
        if len(requested_parts) < path_length:
            return None
        for index in range(path_length):
            if index in resource_paths:
                route_data[resource_paths[index]] = requested_parts[index]
            else:
                if requested_parts[index] != path_parts[index]:
                    return None
        return route_data

    def __call__(self, input_output):
        # including calling parameters that came from the route matching
        return self._handler(input_output)

    def cors(self, input_output):
        # including calling parameters that came from the route matching
        return self._handler.cors(input_output)

    def documentation(self):
        docs = []
        for doc in self._handler.documentation():
            if self._methods is not None:
                doc.set_request_methods(self._methods)
            docs.append(doc)

            # do we have any resource paths to document?
            for path_name in self._resource_paths.values():
                doc.add_parameter(URLParameter(String(path_name, required=True)))
        return docs

    def documentation_models(self):
        return self._handler.documentation_models()

    def documentation_security_schemes(self):
        return self._handler.documentation_security_schemes()
