import json
import logging
import requests
from how2validate.handler.email_handler import send_email
from how2validate.utility.interface.EmailResponse import EmailResponse
from how2validate.utility.interface.validationResult import ValidationProcess, ValidationResult
from how2validate.utility.tool_utility import handle_active_status, handle_inactive_status, handle_errors, response_validation


def validate_anthropic_api_key(provider: str, service: str, secret: str, response_flag: bool, report: str, is_browser: bool = True) -> ValidationResult:
    """
    Validates the Anthropic API key by making a test API request to the Anthropic API.

    Parameters:
    - provider (str): The provider name (e.g., "Anthropic").
    - service (str): The name of the service being validated.
    - secret (str): The Anthropic API key to validate.
    - response_flag (bool): Flag to indicate whether detailed response data should be returned.
    - report (str): An optional email address to which a validation report should be sent.
    - is_browser (bool): Indicates if the function is called from a browser environment (default is True).

    Returns:
    - ValidationResult: A structured response indicating the validation results.
    """
    # Initialize the response structure as an instance of the ValidationProcess class
    validation_response = ValidationProcess(
        state="",
        message="",
        response=None,
        report="email@how2validate.com"
    )

    # Anthropic API endpoint for sending messages
    url = "https://api.anthropic.com/v1/messages"

    # Headers to authorize the request using the provided API key
    headers = {
        "x-api-key": secret,
        "anthropic-version": "2023-06-01",
        "content-type": "application/json"
    }

    # Payload for the API request
    payload = {
        "model": "claude-3-5-sonnet-20241022",
        "max_tokens": 1024,
        "messages": [{"role": "user", "content": "Hello, world"}]
    }

    try:
        # Send a POST request to the Anthropic API
        response_data = requests.post(url, headers=headers, json=payload)
        # Raise an HTTPError if the response has an unsuccessful status code (4xx or 5xx)
        response_data.raise_for_status()

        # Check if the request was successful (HTTP 200)
        if response_data.status_code == 200:
            # Handle active status
            active_response = handle_active_status(
                provider,
                service,
                response_data,
                response_flag,
                report,
                is_browser
            )
            
            validation_response.state = active_response.data.validate.state
            validation_response.message = active_response.data.validate.message
            validation_response.response = json.dumps(active_response.to_dict(), indent=4)

            return response_validation(active_response, response_flag)

    except requests.HTTPError as error:

        if 400 <= error.response.status_code < 500:
            # Handle inactive token or other statuses
            inactive_response = handle_inactive_status(
                provider,
                service,
                response_flag,
                error,
                report,
                is_browser
            )

            validation_response.state = inactive_response.data.validate.state
            validation_response.message = inactive_response.data.validate.message
            validation_response.response = json.dumps(inactive_response.to_dict(), indent=4)

            return response_validation(inactive_response, response_flag)
        
        elif 500 <= error.response.status_code < 600:
            # Handle server errors
            error_response = handle_errors(
                provider,
                service,
                response_flag,
                report,
                error,
                is_browser
            )

            validation_response.state = error_response.data.validate.state
            validation_response.message = error_response.data.validate.message
            validation_response.response = json.dumps(error_response.to_dict(), indent=4)

            return response_validation(error_response, response_flag)

    finally:
        # If a report email is provided, send the validation result via email
        if report:
            email_response = EmailResponse(
                email=report,
                provider=provider,
                service=service,
                state=validation_response.state,
                message=validation_response.message,
                response=validation_response.response,
            )
            try:
                send_email(email_response)
                logging.info('Validation report sent successfully')
            except Exception as e:
                logging.error('Error sending validation report', exc_info=True)
