"""Indices to the PHI5 Latin corpus."""
from typing import Union

MAP_PHI5_AUTHOR_ID_TO_NAME: dict[str, str] = {
    "LAT0428": "Bellum Alexandrinum [Anonymous]",
    "LAT1276": "Decimus Iunius Iuvenalis, Juvenal",
    "LAT0448": "Gaius Iulius Caesar, Caesar",
    "LAT0490": "Publius Cominius",
    "LAT0514": "Egnatius",
    "LAT9254": "Titius, gram.",
    "LAT2305": "Caelius Aurelianus",
    "LAT0484": "Lucius Cincius",
    "LAT1345": "Silius Italicus",
    "LAT0474": "Marcus Tullius Cicero, Cicero, Tully",
    "LAT0614": "Q. Pompeius Q.f.A.n. Rufus",
    "LAT0510": "Publius Cornelius Dolabella",
    "LAT0413": "Gavius Bassus",
    "LAT9510": "Anonymi Grammatici",
    "LAT0452": "Gaius Iulius Caesar Strabo",
    "LAT0875": "Cn. Cornel. Lentulus Gaetulicus",
    "LAT0648": "Staberius Eros",
    "LAT0990": "Precatio Omnium Herbarum",
    "LAT1672": "Iulius Valerius",
    "LAT0451": "Sinnius Capito",
    "LAT0404": "Lucius Afranius",
    "LAT0143": "Trabea",
    "LAT0528": "Granius Flaccus",
    "LAT0455": "Gaius Calpurnius Piso",
    "LAT0635": "Publius Saturius",
    "LAT1212": "Apuleius Madaurensis",
    "LAT0624": "Quintus Claudius Quadrigarius",
    "LAT0592": "Novius, comoed.",
    "LAT0478": "Quintus Tullius Cicero",
    "LAT0472": "Gaius Valerius Catullus",
    "LAT0568": "Gnaeus Matius",
    "LAT0450": "Lucius Iulius Caesar",
    "LAT2003": "Caelius Apicius",
    "LAT0420": "Publius Aufidius Namusa",
    "LAT0564": "Manius Manilius",
    "LAT0806": "Gaius Ateius Capito",
    "LAT0923": "Aemilius Macer",
    "LAT0905": "Marcus Antistius Labeo",
    "LAT1363": "Aemilius Asper",
    "LAT0469": "Lucius Cassius Longinus",
    "LAT0630": "Sacra Argeorum",
    "LAT1291": "Marianus",
    "LAT0116": "Marcus Pacuvius",
    "LAT0306": "Carmen Devotionis",
    "LAT0987": "Publius Pomponius Secundus",
    "LAT0646": "Lucius Cornelius Sisenna",
    "LAT0552": "Quintus Lutatius Catulus",
    "LAT0730": "Tarquitius Priscus",
    "LAT0405": "Clodius Tuscus",
    "LAT9505": "Anonymi Comici et Tragici",
    "LAT0727": "Cornificius Longus",
    "LAT0082": "Decimus Iunius Silanus",
    "LAT2002": "Albinus, poet.",
    "LAT1221": "C. Iul. Caes. Augustus Octavianus",
    "LAT0466": "Aulus Cascellius",
    "LAT0972": "Petronius",
    "LAT0073": "Gaius Sempronius Gracchus",
    "LAT2301": "Q. Aurelius Memmius Symmachus",
    "LAT1100": "Calpurnius Flaccus",
    "LAT0104": "Gaius Memmius",
    "LAT0929": "Pomponius Mela",
    "LAT1020": "Publius Papinius Statius",
    "LAT1050": "Lucius Verginius Rufus",
    "LAT1294": "Marcus Valerius Martialis",
    "LAT0574": "Gaius Memmius L. f.",
    "LAT0402": "Valerius Aedituus",
    "LAT0530": "Aulus Hirtius",
    "LAT0500": "Lucius Licinius Crassus",
    "LAT0591": "Ninnius Crassus",
    "LAT0486": "Gaius Helvius Cinna",
    "LAT0400": "Lucius Accius",
    "LAT1206": "Lucius Ampelius",
    "LAT0058": "Q. Fabius Maximus Servilianus",
    "LAT0920": "Lucilius iunior",
    "LAT0007": "Atilius",
    "LAT1103": "Priapea",
    "LAT0944": "Imperator Nero",
    "LAT0815": "Bruttedius Niger",
    "LAT0494": "Commentarii Consulares",
    "LAT0409": "Quintus Cornificius",
    "LAT0496": "Commentarius Anquisit. Sergii",
    "LAT0812": "Gaius Caesius Bassus",
    "LAT0085": "Gaius Laelius Sapiens",
    "LAT0690": "Publius Vergilius Maro, Virgil, Vergil",
    "LAT0149": "Carmen Arvale",
    "LAT0887": "Grattius",
    "LAT0966": "Passienus Crispus",
    "LAT1321": "Sextus Pomponius",
    "LAT0140": "Gaius Titius",
    "LAT1263": "Hyginus, myth.",
    "LAT1380": "Philumenus medicus",
    "LAT0587": "Naevius, iunior",
    "LAT0533": "Gaius Iulius Hyginus",
    "LAT0596": "Numitorius",
    "LAT0821": "Bucolica Einsidlensia",
    "LAT0301": "Gnaeus Domitius Ahenobarbus",
    "LAT0456": "Gaius Licinius Macer Calvus",
    "LAT0031": "Cornelia, mater Gracchorum",
    "LAT0959": "Publius Ovidius Naso",
    "LAT1342": "Siculus Flaccus",
    "LAT0309": "Carmen Evocationis",
    "LAT0926": "Marcus Manilius, Manilius",
    "LAT0586": "Mummius",
    "LAT0444": "Marcus Caelius Rufus",
    "LAT0112": "Gnaeus Naevius",
    "LAT0682": "Lucius Varius Rufus",
    "LAT1377": "Fragmenta Bobiensia",
    "LAT0534": "Iuventius, comoed.",
    "LAT0588": "Cornelius Nepos",
    "LAT0425": "Publius Rutilius Lupus",
    "LAT0984": "Pompeius Trogus",
    "LAT0824": "Cn. Arulenus Caelius Sabinus",
    "LAT1236": "Sextus Pompeius Festus",
    "LAT1500": "Altercatio Hadr. et Epicteti",
    "LAT0302": "Marcus Antonius",
    "LAT0076": "Gaius Cassius Hemina",
    "LAT1035": "Gaius Valerius Flaccus",
    "LAT2150": "Zeno of Verona",
    "LAT0536": "Decimus Laberius",
    "LAT9221": "Paulus Quaestor",
    "LAT0863": "Dorcatius",
    "LAT0636": "Quintus Mucius Scaevola",
    "LAT0661": "Ticidas",
    "LAT0576": "M. Valerius Messalla Rufus",
    "LAT0067": "Favorinus",
    "LAT0538": "Laevius",
    "LAT1254": "Aulus Gellius",
    "LAT0556": "Gaius Licinius Macer",
    "LAT0640": "Marcus Aemilius Scaurus",
    "LAT1374": "Velius Longus",
    "LAT2331": "Scriptores Historiae Augustae",
    "LAT0103": "Gnaeus Marcius vates",
    "LAT1279": "Laelius Felix",
    "LAT0010": "Marcus Iunius Brutus [iur.]",
    "LAT0106": "Caecilius Metellus",
    "LAT0584": "Mimi Poetarum Incertorum",
    "LAT1044": "Velleius Paterculus",
    "LAT1056": "Vitruvius",
    "LAT0709": "Domitius Marsus",
    "LAT0134": "Publius Terentius Afer, Terence",
    "LAT0836": "Aulus Cornelius Celsus",
    "LAT0088": "M. Aemilius Lepidus Porcina",
    "LAT1348": "Gaius Suetonius Tranquillus",
    "LAT0013": "Caecilius Statius",
    "LAT0109": "Q. Caecilius Metellus Maced.",
    "LAT0684": "Marcus Terentius Varro, Varro",
    "LAT0638": "Q. Mucius Scaevola [pontifex]",
    "LAT0324": "Saserna",
    "LAT0692": "Appendix Vergiliana",
    "LAT0498": "Gaius Aurelius Cotta",
    "LAT0845": "L. Iunius Moderatus Columella",
    "LAT1047": "Veranius",
    "LAT0899": "Hyginus Astronomus",
    "LAT0878": "Gaius Asinius Gallus",
    "LAT0724": "Cloatius Verus",
    "LAT1038": "Valerius Maximus",
    "LAT0703": "Arbonius Silo",
    "LAT0721": "Antonius Panurgus",
    "LAT0418": "Titus Quinctius Atta",
    "LAT2300": "Aemilius Sura",
    "LAT0662": "Marcus Tullius Tiro",
    "LAT0540": "Tullius Laurea",
    "LAT0432": "Marcus Furius Bibaculus",
    "LAT1242": "Annius Florus",
    "LAT1512": "Pomponius Porphyrio",
    "LAT0416": "Lucius Ateius Praetextatus",
    "LAT0327": "L. Aelius Praeconinus Stilo",
    "LAT0932": "M. Valerius Messalla Corvinus",
    "LAT0097": "Gaius Lucilius",
    "LAT0866": "Fenestella",
    "LAT0458": "Publius Cannutius",
    "LAT0127": "P. Cornel. Scipio Afr. ma.",
    "LAT0034": "Gaius Scribonius Curio avus",
    "LAT0079": "Hostius",
    "LAT0996": "Marcus Valerius Probus",
    "LAT0430": "Bellum Hispaniense [Anonymous]",
    "LAT1011": "Scribonius Largus",
    "LAT0043": "Quintus Ennius",
    "LAT0546": "Gaius Licinius Mucianus",
    "LAT0674": "Valerius, comoed.",
    "LAT1053": "Vibius Crispus",
    "LAT2097": "Sextus Paconianus",
    "LAT0827": "Caesellius Vindex",
    "LAT1029": "Turnus",
    "LAT0061": "Fabius Pictor",
    "LAT0656": "Servius Sulpicius Rufus",
    "LAT0869": "Marcus Verrius Flaccus",
    "LAT1306": "Lucius Neratius Priscus",
    "LAT1005": "Rabirius",
    "LAT1235": "Didascaliae et Per. in Terentium",
    "LAT0975": "Phaedrus",
    "LAT0541": "Cn. Cornel. Lentulus Marcell.",
    "LAT1908": "Gallus Antipater",
    "LAT0694": "Volumnius",
    "LAT0582": "Q. Caecilius Metellus Numid.",
    "LAT0706": "Carmen de Bello Aegyptiaco",
    "LAT1518": "Terentianus Maurus",
    "LAT0650": "Sueius",
    "LAT0446": "Quintus Servilius Caepio",
    "LAT0419": "Lucius Orbilius Pupillus",
    "LAT0644": "Sextilius Ena",
    "LAT0100": "Luscius Lanuvinus",
    "LAT0600": "Gaius Oppius",
    "LAT0442": "Aulus Caecina",
    "LAT0423": "Lucius Herennius Balbus",
    "LAT0408": "Marcus Antonius triumvir",
    "LAT0854": "Cornificius Gallus",
    "LAT1336": "Scaevus Memor",
    "LAT0590": "Publius Nigidius Figulus",
    "LAT0628": "Publius Rutilius Rufus",
    "LAT1351": "Cornelius Tacitus",
    "LAT0028": "Lucius Coelius Antipater",
    "LAT0642": "Sevius Nicanor",
    "LAT0117": "Papinius, epigram.",
    "LAT0615": "Q. Pompeius Q.f.Q.n. Rufus",
    "LAT0914": "Titus Livius, Livy",
    "LAT0460": "Gaius Papirius Carbo Arvina",
    "LAT1014": "Lucius Annaeus Seneca senior",
    "LAT0025": "Marcus Porcius Cato M.f.M.n.",
    "LAT1604": "Iulius Atherianus",
    "LAT0128": "P. Cornel. Scipio Aem. Afr.",
    "LAT2302": "L. Aurel. Avianius Symmachus",
    "LAT0893": "Quintus Horatius Flaccus, Horace",
    "LAT1282": "Lentulus, mimus",
    "LAT2434": "Hilarius Arelatensis",
    "LAT0426": "Bellum Africum [Anonymous]",
    "LAT0016": "Lucius Calpurnius Piso Frugi",
    "LAT0981": "Gaius Asinius Pollio",
    "LAT0445": "Gaius vel Lucius Caepasius",
    "LAT0660": "Albius Tibullus",
    "LAT0631": "Gaius Sallustius Crispus",
    "LAT1318": "C. Plinius Caecilius Secundus, Pliny",
    "LAT0522": "Gaius Aelius Gallus",
    "LAT0488": "Servius Clodius",
    "LAT0678": "Quintus Valerius Soranus",
    "LAT1002": "Marcus Fabius Quintilianus",
    "LAT1339": "Septimius Serenus",
    "LAT1266": "Hyginus Gromaticus",
    "LAT0676": "Valerius Antias",
    "LAT1506": "Anonymi Fragmenta de Iure Fisci",
    "LAT0686": "P. Terentius Varro Atacinus",
    "LAT0064": "Gaius Fannius",
    "LAT0857": "Lucius Annaeus Cornutus",
    "LAT0518": "Aulus Furius Antias",
    "LAT0537": "Titus Labienus",
    "LAT0890": "Homerus Latinus",
    "LAT0550": "Titus Lucretius Carus",
    "LAT0406": "Publius Alfenus Varus",
    "LAT0881": "Claudius Caesar Germanicus",
    "LAT0620": "Sextus Propertius",
    "LAT0137": "Titinius",
    "LAT0512": "Marcus Duronius",
    "LAT0911": "Laus Pisonis",
    "LAT0625": "Lucius Quinctius",
    "LAT0005": "Aquilius, comoed.",
    "LAT0963": "Quintus Remmius Palaemon",
    "LAT0122": "Aulus Postumius Albinus",
    "LAT0473": "Q. Lutatius Catulus iunior",
    "LAT0809": "Aufidius Bassus",
    "LAT0527": "Gannius",
    "LAT0454": "Marcus Calidius",
    "LAT1251": "Gaius, iur., Gaius",
    "LAT1370": "Quintus Terentius Scaurus",
    "LAT0146": "Sextus Turpilius",
    "LAT0842": "Gaius Clodius Licinus",
    "LAT2335": "Anonymi de Differentiis [Fronto]",
    "LAT0532": "Quintus Hortensius Hortalus",
    "LAT0004": "Appius Claudius Caecus",
    "LAT0594": "Lucius Novius",
    "LAT0410": "Aprissius (?)",
    "LAT0526": "Gaius Servilius Glaucia",
    "LAT0800": "Albinovanus Pedo",
    "LAT0884": "Gracchus, trag.",
    "LAT0094": "Lucius Livius Andronicus",
    "LAT0652": "Lucius Cornelius Sulla",
    "LAT0618": "Lucius Pomponius Bononiensis",
    "LAT0412": "Gaius Aquilius Gallus",
    "LAT1248": "Marcus Cornelius Fronto",
    "LAT1234": "Didascaliae et Argum. in Plautum",
    "LAT1203": "Alfius Avitus",
    "LAT0622": "Publilius Syrus",
    "LAT0851": "Cornelius Severus",
    "LAT3211": "Argum. Aen. et Tetrast.",
    "LAT0606": "Lucius Marcius Philippus",
    "LAT1041": "Pseudo-Varro",
    "LAT0830": "Titus Calpurnius Siculus",
    "LAT0002": "Titus Annius Luscus",
    "LAT0664": "Gaius Trebatius Testa",
    "LAT0046": "Cornelius Epicadus",
    "LAT0680": "Gaius Valgius Rufus",
    "LAT0902": "Iulius Africanus",
    "LAT2000": "Ablabius",
    "LAT0993": "Precatio Terrae",
    "LAT9500": "Anonymi Epici et Lyrici",
    "LAT0470": "Marcus Porcius Cato Uticensis",
    "LAT0487": "Publius Clodius Pulcher",
    "LAT1023": "Sulpicia, Caleni uxor",
    "LAT0312": "Fabius Dossennus",
    "LAT0670": "Quintus Aelius Tubero",
    "LAT1285": "Lucius Volusius Maecianus",
    "LAT0022": "Marcus Porcius Cato, Cato",
    "LAT1245": "Sextus Iulius Frontinus",
    "LAT0935": "Iulius Modestus",
    "LAT1032": "Vagellius",
    "LAT0130": "P. Cornel. Scipio Nasica Ser.",
    "LAT1017": "Lucius Annaeus Seneca iunior",
    "LAT0414": "Lucius Arruntius",
    "LAT0330": "Volcacius Sedigitus",
    "LAT0634": "Santra",
    "LAT1227": "Balbus, grom.",
    "LAT1515": "Quintus Serenus (Sammonicus)",
    "LAT0401": "Aufustius",
    "LAT0908": "Attius Labeo",
    "LAT0938": "Iulius Montanus",
    "LAT0091": "Licinius Imbrex",
    "LAT1357": "Imp. Marcus Ulpius Traianus, Trajan",
    "LAT0037": "Gaius Scribonius Curio pater",
    "LAT1224": "Marcus Aurelius",
    "LAT0492": "Commentarii Augurum",
    "LAT0535": "Marcus Iuventius Laterensis",
    "LAT0978": "Gaius Plinius Secundus, Pliny",
    "LAT2468": "Aurelius Augustinus",
    "LAT1327": "Sabidius",
    "LAT0860": "Quintus Curtius Rufus",
    "LAT1260": "Hadrianus",
    "LAT1218": "Sentius Augurinus",
    "LAT0125": "Publius Mucius Scaevola",
    "LAT0119": "Titus Maccius Plautus",
    "LAT0502": "Aulus Cremutius Cordus",
    "LAT1297": "Marullus",
    "LAT0668": "Gnaeus Tremelius Scrofa",
    "LAT1229": "Flavius Caper",
    "LAT0515": "Sextus (vel Spurius) Ennius",
    "LAT0672": "Turranius Niger",
    "LAT0019": "Gaius Papirius Carbo",
    "LAT0917": "Marcus Annaeus Lucanus",
    "LAT9969": "Vita Iuvenalis",
    "LAT0516": "Gaius Erucius",
    "LAT1209": "Annianus",
    "LAT2123": "Publilius Optatianus Porfyrius",
    "LAT2456": "Parthenius Presbyter",
    "LAT2028": "Chalcidius",
    "LAT0303": "Aurelius Opillus",
    "LAT2349": "Maurus Servius Honoratus, Servius",
    "LAT1257": "Granius Licinianus",
    "LAT1000": "Pupius (?)",
    "LAT0070": "Gnaeus Gellius",
    "LAT2806": "Iustinianus, Justinian, Digest",
    "LAT0658": "Tabulae Censoriae",
    "LAT0027": "Lucius Cincius Alimentus",
    "LAT0524": "Gaius Cornelius Gallus, Gallus",
    "LAT0560": "Helvius Mancia",
    "LAT0558": "Gaius Cilnius Maecenas",
    "LAT0562": "Manilius, poet.",
    "LAT0436": "Marcus Iunius Brutus [tyr.]",
    "LAT0803": "Quintus Asconius Pedianus",
    "LAT0118": "L. Aemilius L.f.M.n. Paulus",
    "LAT0321": "Porcius Licinus",
    "LAT0616": "Pompilius",
    "LAT0300": "Sempronius Asellio",
    "LAT0315": "Marcus Iunius Gracchanus",
    "LAT0969": "Aulus Persius Flaccus",
}

MAP_PHI5_AUTHOR_ID_TO_WORKS_AND_NAME: dict[str, dict[str, Union[list[str], str]]] = {
    "LAT0528": {"works": ["001"], "name": "Granius Flaccus"},
    "LAT0650": {"works": ["001"], "name": "Sueius"},
    "LAT1380": {"works": ["001"], "name": "Philumenus medicus"},
    "LAT2002": {"works": ["001", "002"], "name": "Albinus, poet."},
    "LAT0845": {"works": ["001", "002"], "name": "L. Iunius Moderatus Columella"},
    "LAT1245": {
        "works": ["001", "002", "003", "004", "005", "006"],
        "name": "Sextus Iulius Frontinus",
    },
    "LAT0568": {"works": ["001"], "name": "Gnaeus Matius"},
    "LAT0532": {"works": ["001", "002"], "name": "Quintus Hortensius Hortalus"},
    "LAT0672": {"works": ["001"], "name": "Turranius Niger"},
    "LAT0117": {"works": ["001"], "name": "Papinius, epigram."},
    "LAT0824": {"works": ["001"], "name": "Cn. Arulenus Caelius Sabinus"},
    "LAT1363": {"works": ["001", "002"], "name": "Aemilius Asper"},
    "LAT0574": {"works": ["001", "002"], "name": "Gaius Memmius L. f."},
    "LAT0842": {"works": ["001"], "name": "Gaius Clodius Licinus"},
    "LAT0670": {"works": ["001", "002"], "name": "Quintus Aelius Tubero"},
    "LAT0466": {"works": ["001"], "name": "Aulus Cascellius"},
    "LAT2331": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
            "015",
            "016",
            "017",
            "018",
            "019",
            "020",
            "021",
            "022",
            "023",
            "024",
            "025",
            "026",
            "027",
            "028",
            "029",
            "030",
        ],
        "name": "Scriptores Historiae Augustae",
    },
    "LAT1263": {"works": ["001"], "name": "Hyginus, myth."},
    "LAT1260": {"works": ["001", "002"], "name": "Hadrianus"},
    "LAT0442": {"works": ["002"], "name": "Aulus Caecina"},
    "LAT0674": {"works": ["001"], "name": "Valerius, comoed."},
    "LAT0149": {"works": ["001"], "name": "Carmen Arvale"},
    "LAT2806": {"works": ["002"], "name": "Iustinianus, Justinian, Digest"},
    "LAT0588": {"works": ["001", "002"], "name": "Cornelius Nepos"},
    "LAT0515": {"works": ["001"], "name": "Sextus (vel Spurius) Ennius"},
    "LAT0127": {"works": ["001"], "name": "P. Cornel. Scipio Afr. ma."},
    "LAT1306": {"works": ["002"], "name": "Lucius Neratius Priscus"},
    "LAT0470": {"works": ["002"], "name": "Marcus Porcius Cato Uticensis"},
    "LAT0703": {"works": ["001"], "name": "Arbonius Silo"},
    "LAT2305": {"works": ["001"], "name": "Caelius Aurelianus"},
    "LAT0445": {"works": ["001"], "name": "Gaius vel Lucius Caepasius"},
    "LAT0899": {"works": ["001"], "name": "Hyginus Astronomus"},
    "LAT0806": {"works": ["001"], "name": "Gaius Ateius Capito"},
    "LAT1294": {"works": ["001", "002"], "name": "Marcus Valerius Martialis"},
    "LAT0301": {"works": ["001"], "name": "Gnaeus Domitius Ahenobarbus"},
    "LAT0315": {"works": ["001"], "name": "Marcus Iunius Gracchanus"},
    "LAT0088": {"works": ["001"], "name": "M. Aemilius Lepidus Porcina"},
    "LAT0662": {"works": ["001"], "name": "Marcus Tullius Tiro"},
    "LAT1285": {"works": ["001"], "name": "Lucius Volusius Maecianus"},
    "LAT0893": {
        "works": ["001", "002", "003", "004", "005", "006"],
        "name": "Quintus Horatius Flaccus, Horace",
    },
    "LAT0306": {"works": ["001"], "name": "Carmen Devotionis"},
    "LAT0524": {"works": ["001", "002"], "name": "Gaius Cornelius Gallus, Gallus"},
    "LAT0484": {"works": ["001", "002"], "name": "Lucius Cincius"},
    "LAT1206": {"works": ["001"], "name": "Lucius Ampelius"},
    "LAT0975": {"works": ["001", "002"], "name": "Phaedrus"},
    "LAT0587": {"works": ["001"], "name": "Naevius, iunior"},
    "LAT0405": {"works": ["001"], "name": "Clodius Tuscus"},
    "LAT1044": {"works": ["001"], "name": "Velleius Paterculus"},
    "LAT0130": {"works": ["001"], "name": "P. Cornel. Scipio Nasica Ser."},
    "LAT0996": {"works": ["001", "002", "003"], "name": "Marcus Valerius Probus"},
    "LAT1017": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
            "015",
            "016",
            "017",
            "018",
        ],
        "name": "Lucius Annaeus Seneca iunior",
    },
    "LAT0104": {"works": ["001"], "name": "Gaius Memmius"},
    "LAT0516": {"works": ["001"], "name": "Gaius Erucius"},
    "LAT0028": {"works": ["001"], "name": "Lucius Coelius Antipater"},
    "LAT1604": {"works": ["001"], "name": "Iulius Atherianus"},
    "LAT0109": {"works": ["001"], "name": "Q. Caecilius Metellus Maced."},
    "LAT0458": {"works": ["001"], "name": "Publius Cannutius"},
    "LAT0451": {"works": ["001"], "name": "Sinnius Capito"},
    "LAT0512": {"works": ["001"], "name": "Marcus Duronius"},
    "LAT1266": {
        "works": ["001", "002", "003", "004", "005"],
        "name": "Hyginus Gromaticus",
    },
    "LAT1291": {"works": ["001"], "name": "Marianus"},
    "LAT0309": {"works": ["001"], "name": "Carmen Evocationis"},
    "LAT0959": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
            "015",
        ],
        "name": "Publius Ovidius Naso",
    },
    "LAT0984": {"works": ["001", "002"], "name": "Pompeius Trogus"},
    "LAT0416": {"works": ["001"], "name": "Lucius Ateius Praetextatus"},
    "LAT9510": {"works": ["001"], "name": "Anonymi Grammatici"},
    "LAT0730": {"works": ["001"], "name": "Tarquitius Priscus"},
    "LAT0592": {"works": ["001"], "name": "Novius, comoed."},
    "LAT0076": {"works": ["001"], "name": "Gaius Cassius Hemina"},
    "LAT0631": {
        "works": ["001", "002", "003", "004", "005", "006", "007", "008"],
        "name": "Gaius Sallustius Crispus",
    },
    "LAT0146": {"works": ["001"], "name": "Sextus Turpilius"},
    "LAT0537": {"works": ["001"], "name": "Titus Labienus"},
    "LAT0969": {"works": ["001"], "name": "Aulus Persius Flaccus"},
    "LAT0636": {"works": ["001"], "name": "Quintus Mucius Scaevola"},
    "LAT0902": {"works": ["001"], "name": "Iulius Africanus"},
    "LAT2468": {"works": ["001"], "name": "Aurelius Augustinus"},
    "LAT1011": {"works": ["001"], "name": "Scribonius Largus"},
    "LAT0884": {"works": ["001"], "name": "Gracchus, trag."},
    "LAT0067": {"works": ["001"], "name": "Favorinus"},
    "LAT0920": {"works": ["001"], "name": "Lucilius iunior"},
    "LAT0444": {"works": ["002"], "name": "Marcus Caelius Rufus"},
    "LAT0324": {"works": ["001"], "name": "Saserna"},
    "LAT0596": {"works": ["001"], "name": "Numitorius"},
    "LAT0327": {"works": ["001"], "name": "L. Aelius Praeconinus Stilo"},
    "LAT0546": {"works": ["001"], "name": "Gaius Licinius Mucianus"},
    "LAT0881": {"works": ["001", "002", "003"], "name": "Claudius Caesar Germanicus"},
    "LAT9505": {
        "works": ["001", "002", "003", "004"],
        "name": "Anonymi Comici et Tragici",
    },
    "LAT0514": {"works": ["001"], "name": "Egnatius"},
    "LAT0981": {"works": ["001", "003", "004", "005"], "name": "Gaius Asinius Pollio"},
    "LAT0409": {"works": ["001"], "name": "Quintus Cornificius"},
    "LAT0432": {"works": ["001"], "name": "Marcus Furius Bibaculus"},
    "LAT0418": {"works": ["001", "002"], "name": "Titus Quinctius Atta"},
    "LAT0300": {"works": ["001"], "name": "Sempronius Asellio"},
    "LAT0496": {"works": ["001"], "name": "Commentarius Anquisit. Sergii"},
    "LAT0106": {"works": ["001"], "name": "Caecilius Metellus"},
    "LAT0917": {"works": ["001", "002"], "name": "Marcus Annaeus Lucanus"},
    "LAT0656": {"works": ["002", "003"], "name": "Servius Sulpicius Rufus"},
    "LAT0502": {"works": ["001"], "name": "Aulus Cremutius Cordus"},
    "LAT1345": {"works": ["001"], "name": "Silius Italicus"},
    "LAT0869": {"works": ["001", "002"], "name": "Marcus Verrius Flaccus"},
    "LAT0535": {"works": ["001"], "name": "Marcus Iuventius Laterensis"},
    "LAT0682": {"works": ["001", "002"], "name": "Lucius Varius Rufus"},
    "LAT1056": {"works": ["001"], "name": "Vitruvius"},
    "LAT1005": {"works": ["001"], "name": "Rabirius"},
    "LAT0070": {"works": ["001"], "name": "Gnaeus Gellius"},
    "LAT1370": {
        "works": ["001", "002", "003", "004"],
        "name": "Quintus Terentius Scaurus",
    },
    "LAT1374": {"works": ["001"], "name": "Velius Longus"},
    "LAT2150": {"works": ["001"], "name": "Zeno of Verona"},
    "LAT0727": {"works": ["001"], "name": "Cornificius Longus"},
    "LAT0530": {"works": ["001", "002"], "name": "Aulus Hirtius"},
    "LAT0929": {"works": ["001"], "name": "Pomponius Mela"},
    "LAT0082": {"works": ["001"], "name": "Decimus Iunius Silanus"},
    "LAT0634": {"works": ["001", "002"], "name": "Santra"},
    "LAT0863": {"works": ["001"], "name": "Dorcatius"},
    "LAT0302": {"works": ["001"], "name": "Marcus Antonius"},
    "LAT0800": {"works": ["001"], "name": "Albinovanus Pedo"},
    "LAT0116": {"works": ["001", "002"], "name": "Marcus Pacuvius"},
    "LAT0590": {"works": ["001"], "name": "Publius Nigidius Figulus"},
    "LAT1041": {"works": ["001"], "name": "Pseudo-Varro"},
    "LAT1023": {"works": ["001", "002"], "name": "Sulpicia, Caleni uxor"},
    "LAT0560": {"works": ["001"], "name": "Helvius Mancia"},
    "LAT0664": {"works": ["001"], "name": "Gaius Trebatius Testa"},
    "LAT0013": {"works": ["001"], "name": "Caecilius Statius"},
    "LAT0450": {"works": ["001"], "name": "Lucius Iulius Caesar"},
    "LAT0706": {"works": ["001"], "name": "Carmen de Bello Aegyptiaco"},
    "LAT1035": {"works": ["001"], "name": "Gaius Valerius Flaccus"},
    "LAT0140": {"works": ["001"], "name": "Gaius Titius"},
    "LAT2300": {"works": ["001"], "name": "Aemilius Sura"},
    "LAT0404": {"works": ["001"], "name": "Lucius Afranius"},
    "LAT1029": {"works": ["001"], "name": "Turnus"},
    "LAT0676": {"works": ["001"], "name": "Valerius Antias"},
    "LAT0866": {"works": ["001"], "name": "Fenestella"},
    "LAT1053": {"works": ["001"], "name": "Vibius Crispus"},
    "LAT0472": {"works": ["001", "002"], "name": "Gaius Valerius Catullus"},
    "LAT0857": {"works": ["001"], "name": "Lucius Annaeus Cornutus"},
    "LAT1251": {"works": ["001", "002", "004"], "name": "Gaius, iur., Gaius"},
    "LAT0630": {"works": ["001"], "name": "Sacra Argeorum"},
    "LAT0536": {"works": ["001"], "name": "Decimus Laberius"},
    "LAT0638": {"works": ["002"], "name": "Q. Mucius Scaevola [pontifex]"},
    "LAT0914": {"works": ["001", "002", "003", "004"], "name": "Titus Livius, Livy"},
    "LAT0494": {"works": ["001"], "name": "Commentarii Consulares"},
    "LAT0812": {
        "works": ["001", "002", "003", "004", "005", "006", "007"],
        "name": "Gaius Caesius Bassus",
    },
    "LAT0990": {"works": ["001"], "name": "Precatio Omnium Herbarum"},
    "LAT0401": {"works": ["001"], "name": "Aufustius"},
    "LAT0091": {"works": ["001"], "name": "Licinius Imbrex"},
    "LAT0660": {"works": ["001", "002"], "name": "Albius Tibullus"},
    "LAT0085": {"works": ["001"], "name": "Gaius Laelius Sapiens"},
    "LAT0420": {"works": ["001"], "name": "Publius Aufidius Namusa"},
    "LAT0875": {"works": ["001"], "name": "Cn. Cornel. Lentulus Gaetulicus"},
    "LAT0550": {"works": ["001", "002", "003"], "name": "Titus Lucretius Carus"},
    "LAT0425": {"works": ["001"], "name": "Publius Rutilius Lupus"},
    "LAT1038": {"works": ["001"], "name": "Valerius Maximus"},
    "LAT0406": {"works": ["002"], "name": "Publius Alfenus Varus"},
    "LAT1276": {"works": ["001"], "name": "Decimus Iunius Iuvenalis, Juvenal"},
    "LAT0541": {"works": ["001"], "name": "Cn. Cornel. Lentulus Marcell."},
    "LAT0025": {"works": ["001"], "name": "Marcus Porcius Cato M.f.M.n."},
    "LAT0498": {"works": ["001"], "name": "Gaius Aurelius Cotta"},
    "LAT0034": {"works": ["001"], "name": "Gaius Scribonius Curio avus"},
    "LAT0094": {"works": ["001", "002", "003"], "name": "Lucius Livius Andronicus"},
    "LAT0721": {"works": ["001"], "name": "Antonius Panurgus"},
    "LAT0500": {"works": ["001"], "name": "Lucius Licinius Crassus"},
    "LAT0648": {"works": ["001"], "name": "Staberius Eros"},
    "LAT1209": {"works": ["001"], "name": "Annianus"},
    "LAT0487": {"works": ["001"], "name": "Publius Clodius Pulcher"},
    "LAT0686": {"works": ["001"], "name": "P. Terentius Varro Atacinus"},
    "LAT1377": {
        "works": ["001", "002", "003", "004", "005", "006", "007", "008"],
        "name": "Fragmenta Bobiensia",
    },
    "LAT0652": {"works": ["001"], "name": "Lucius Cornelius Sulla"},
    "LAT0510": {"works": ["002"], "name": "Publius Cornelius Dolabella"},
    "LAT0661": {"works": ["001"], "name": "Ticidas"},
    "LAT0488": {"works": ["001"], "name": "Servius Clodius"},
    "LAT0624": {"works": ["001"], "name": "Quintus Claudius Quadrigarius"},
    "LAT2003": {"works": ["001", "002", "003"], "name": "Caelius Apicius"},
    "LAT1357": {"works": ["002"], "name": "Imp. Marcus Ulpius Traianus, Trajan"},
    "LAT1235": {
        "works": ["001", "002", "003", "004", "005", "006"],
        "name": "Didascaliae et Per. in Terentium",
    },
    "LAT0724": {"works": ["001"], "name": "Cloatius Verus"},
    "LAT0938": {"works": ["001"], "name": "Iulius Montanus"},
    "LAT1500": {"works": ["001"], "name": "Altercatio Hadr. et Epicteti"},
    "LAT0582": {"works": ["001"], "name": "Q. Caecilius Metellus Numid."},
    "LAT1327": {"works": ["001"], "name": "Sabidius"},
    "LAT0972": {"works": ["001", "002"], "name": "Petronius"},
    "LAT2335": {"works": ["001"], "name": "Anonymi de Differentiis [Fronto]"},
    "LAT2349": {
        "works": ["001", "002", "003", "004", "005", "006", "007"],
        "name": "Maurus Servius Honoratus, Servius",
    },
    "LAT1515": {"works": ["001", "002"], "name": "Quintus Serenus (Sammonicus)"},
    "LAT0022": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
        ],
        "name": "Marcus Porcius Cato, Cato",
    },
    "LAT0412": {"works": ["001"], "name": "Gaius Aquilius Gallus"},
    "LAT2028": {"works": ["001"], "name": "Chalcidius"},
    "LAT0993": {"works": ["001"], "name": "Precatio Terrae"},
    "LAT0678": {"works": ["001"], "name": "Quintus Valerius Soranus"},
    "LAT1236": {"works": ["001"], "name": "Sextus Pompeius Festus"},
    "LAT0851": {"works": ["001", "002"], "name": "Cornelius Severus"},
    "LAT0908": {"works": ["001"], "name": "Attius Labeo"},
    "LAT0635": {"works": ["001"], "name": "Publius Saturius"},
    "LAT0118": {"works": ["001"], "name": "L. Aemilius L.f.M.n. Paulus"},
    "LAT0518": {"works": ["001"], "name": "Aulus Furius Antias"},
    "LAT0423": {"works": ["001"], "name": "Lucius Herennius Balbus"},
    "LAT0436": {"works": ["002"], "name": "Marcus Iunius Brutus [tyr.]"},
    "LAT1351": {
        "works": ["001", "002", "003", "004", "005"],
        "name": "Cornelius Tacitus",
    },
    "LAT9254": {"works": ["001"], "name": "Titius, gram."},
    "LAT0644": {"works": ["001"], "name": "Sextilius Ena"},
    "LAT0978": {"works": ["001", "002"], "name": "Gaius Plinius Secundus, Pliny"},
    "LAT0430": {"works": ["001"], "name": "Bellum Hispaniense [Anonymous]"},
    "LAT0526": {"works": ["001"], "name": "Gaius Servilius Glaucia"},
    "LAT1339": {"works": ["001"], "name": "Septimius Serenus"},
    "LAT0926": {"works": ["001"], "name": "Marcus Manilius, Manilius"},
    "LAT0815": {"works": ["001"], "name": "Bruttedius Niger"},
    "LAT2456": {"works": ["001"], "name": "Parthenius Presbyter"},
    "LAT0600": {"works": ["001", "002"], "name": "Gaius Oppius"},
    "LAT1218": {"works": ["001"], "name": "Sentius Augurinus"},
    "LAT9500": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
            "015",
            "016",
            "017",
            "018",
            "019",
            "020",
            "021",
            "022",
            "023",
            "024",
            "025",
            "026",
            "027",
            "028",
            "029",
            "030",
            "031",
            "032",
            "033",
            "034",
            "035",
            "036",
            "037",
            "038",
            "039",
            "040",
            "041",
        ],
        "name": "Anonymi Epici et Lyrici",
    },
    "LAT1224": {"works": ["001"], "name": "Marcus Aurelius"},
    "LAT0122": {"works": ["001"], "name": "Aulus Postumius Albinus"},
    "LAT1672": {"works": ["001"], "name": "Iulius Valerius"},
    "LAT0534": {"works": ["001"], "name": "Iuventius, comoed."},
    "LAT0097": {"works": ["001"], "name": "Gaius Lucilius"},
    "LAT0408": {"works": ["002"], "name": "Marcus Antonius triumvir"},
    "LAT1100": {"works": ["001"], "name": "Calpurnius Flaccus"},
    "LAT1234": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
            "015",
            "016",
            "017",
            "018",
            "019",
        ],
        "name": "Didascaliae et Argum. in Plautum",
    },
    "LAT1002": {"works": ["001", "002", "003"], "name": "Marcus Fabius Quintilianus"},
    "LAT0016": {"works": ["001"], "name": "Lucius Calpurnius Piso Frugi"},
    "LAT0944": {"works": ["001"], "name": "Imperator Nero"},
    "LAT0400": {"works": ["001", "002", "003"], "name": "Lucius Accius"},
    "LAT0046": {"works": ["001"], "name": "Cornelius Epicadus"},
    "LAT0486": {"works": ["001"], "name": "Gaius Helvius Cinna"},
    "LAT2301": {"works": ["001"], "name": "Q. Aurelius Memmius Symmachus"},
    "LAT0854": {"works": ["001"], "name": "Cornificius Gallus"},
    "LAT1221": {
        "works": ["001", "002", "003", "004", "005", "006", "007", "008"],
        "name": "C. Iul. Caes. Augustus Octavianus",
    },
    "LAT0414": {"works": ["001"], "name": "Lucius Arruntius"},
    "LAT0538": {"works": ["001", "002"], "name": "Laevius"},
    "LAT0533": {"works": ["001", "002"], "name": "Gaius Iulius Hyginus"},
    "LAT0061": {"works": ["001", "002"], "name": "Fabius Pictor"},
    "LAT3211": {
        "works": ["001", "002", "003", "004"],
        "name": "Argum. Aen. et Tetrast.",
    },
    "LAT0658": {"works": ["001"], "name": "Tabulae Censoriae"},
    "LAT1242": {"works": ["001", "002", "003", "004", "005"], "name": "Annius Florus"},
    "LAT0456": {"works": ["001", "002"], "name": "Gaius Licinius Macer Calvus"},
    "LAT0402": {"works": ["001"], "name": "Valerius Aedituus"},
    "LAT0031": {"works": ["001"], "name": "Cornelia, mater Gracchorum"},
    "LAT0490": {"works": ["001"], "name": "Publius Cominius"},
    "LAT0455": {"works": ["001"], "name": "Gaius Calpurnius Piso"},
    "LAT0019": {"works": ["001"], "name": "Gaius Papirius Carbo"},
    "LAT1908": {"works": ["001"], "name": "Gallus Antipater"},
    "LAT0562": {"works": ["001"], "name": "Manilius, poet."},
    "LAT0836": {"works": ["001", "002", "003"], "name": "Aulus Cornelius Celsus"},
    "LAT0690": {
        "works": ["001", "002", "003"],
        "name": "Publius Vergilius Maro, Virgil, Vergil",
    },
    "LAT0620": {"works": ["001"], "name": "Sextus Propertius"},
    "LAT0473": {"works": ["001"], "name": "Q. Lutatius Catulus iunior"},
    "LAT0932": {"works": ["001", "002"], "name": "M. Valerius Messalla Corvinus"},
    "LAT0576": {"works": ["001", "002"], "name": "M. Valerius Messalla Rufus"},
    "LAT0935": {"works": ["001"], "name": "Iulius Modestus"},
    "LAT0058": {"works": ["001"], "name": "Q. Fabius Maximus Servilianus"},
    "LAT0680": {"works": ["001"], "name": "Gaius Valgius Rufus"},
    "LAT0556": {"works": ["001", "002"], "name": "Gaius Licinius Macer"},
    "LAT0628": {"works": ["001"], "name": "Publius Rutilius Rufus"},
    "LAT0615": {"works": ["001"], "name": "Q. Pompeius Q.f.Q.n. Rufus"},
    "LAT1032": {"works": ["001"], "name": "Vagellius"},
    "LAT0622": {"works": ["001", "002"], "name": "Publilius Syrus"},
    "LAT1050": {"works": ["001"], "name": "Lucius Verginius Rufus"},
    "LAT0004": {"works": ["001"], "name": "Appius Claudius Caecus"},
    "LAT0803": {
        "works": ["001", "002", "003", "004", "005"],
        "name": "Quintus Asconius Pedianus",
    },
    "LAT0552": {"works": ["001", "002"], "name": "Quintus Lutatius Catulus"},
    "LAT0005": {"works": ["001"], "name": "Aquilius, comoed."},
    "LAT1321": {"works": ["002"], "name": "Sextus Pomponius"},
    "LAT0112": {
        "works": ["001", "002", "003", "004", "005", "006", "007"],
        "name": "Gnaeus Naevius",
    },
    "LAT1297": {"works": ["001"], "name": "Marullus"},
    "LAT2302": {"works": ["001"], "name": "L. Aurel. Avianius Symmachus"},
    "LAT0492": {"works": ["001"], "name": "Commentarii Augurum"},
    "LAT0413": {"works": ["001", "002"], "name": "Gavius Bassus"},
    "LAT0586": {"works": ["001"], "name": "Mummius"},
    "LAT0312": {"works": ["001"], "name": "Fabius Dossennus"},
    "LAT0522": {"works": ["001", "002"], "name": "Gaius Aelius Gallus"},
    "LAT0137": {"works": ["001"], "name": "Titinius"},
    "LAT0966": {"works": ["001"], "name": "Passienus Crispus"},
    "LAT0007": {"works": ["001"], "name": "Atilius"},
    "LAT0419": {"works": ["001"], "name": "Lucius Orbilius Pupillus"},
    "LAT0119": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
            "015",
            "016",
            "017",
            "018",
            "019",
            "020",
            "021",
            "022",
        ],
        "name": "Titus Maccius Plautus",
    },
    "LAT0540": {"works": ["001"], "name": "Tullius Laurea"},
    "LAT0594": {"works": ["001"], "name": "Lucius Novius"},
    "LAT1047": {"works": ["001"], "name": "Veranius"},
    "LAT0410": {"works": ["001"], "name": "Aprissius (?)"},
    "LAT0010": {"works": ["001"], "name": "Marcus Iunius Brutus [iur.]"},
    "LAT0428": {"works": ["001"], "name": "Bellum Alexandrinum [Anonymous]"},
    "LAT0128": {"works": ["001"], "name": "P. Cornel. Scipio Aem. Afr."},
    "LAT0890": {"works": ["001"], "name": "Homerus Latinus"},
    "LAT0446": {"works": ["001"], "name": "Quintus Servilius Caepio"},
    "LAT0887": {"works": ["001"], "name": "Grattius"},
    "LAT0905": {"works": ["002"], "name": "Marcus Antistius Labeo"},
    "LAT1257": {"works": ["001"], "name": "Granius Licinianus"},
    "LAT0079": {"works": ["001"], "name": "Hostius"},
    "LAT1020": {
        "works": ["001", "002", "003", "004"],
        "name": "Publius Papinius Statius",
    },
    "LAT1348": {
        "works": ["001", "002", "003", "004", "005", "006"],
        "name": "Gaius Suetonius Tranquillus",
    },
    "LAT0027": {"works": ["001"], "name": "Lucius Cincius Alimentus"},
    "LAT0668": {"works": ["001"], "name": "Gnaeus Tremelius Scrofa"},
    "LAT0103": {"works": ["001"], "name": "Gnaeus Marcius vates"},
    "LAT1203": {"works": ["001"], "name": "Alfius Avitus"},
    "LAT0330": {"works": ["001"], "name": "Volcacius Sedigitus"},
    "LAT9969": {"works": ["001"], "name": "Vita Iuvenalis"},
    "LAT0642": {"works": ["001"], "name": "Sevius Nicanor"},
    "LAT1014": {
        "works": ["001", "002", "003", "004"],
        "name": "Lucius Annaeus Seneca senior",
    },
    "LAT0591": {"works": ["001"], "name": "Ninnius Crassus"},
    "LAT2123": {"works": ["003"], "name": "Publilius Optatianus Porfyrius"},
    "LAT1212": {
        "works": ["001", "002", "003", "004", "005", "006", "007", "008", "009", "010"],
        "name": "Apuleius Madaurensis",
    },
    "LAT0584": {"works": ["001", "002"], "name": "Mimi Poetarum Incertorum"},
    "LAT0923": {"works": ["001", "002"], "name": "Aemilius Macer"},
    "LAT1518": {"works": ["001"], "name": "Terentianus Maurus"},
    "LAT0452": {"works": ["001", "002"], "name": "Gaius Iulius Caesar Strabo"},
    "LAT0860": {"works": ["001"], "name": "Quintus Curtius Rufus"},
    "LAT0625": {"works": ["001"], "name": "Lucius Quinctius"},
    "LAT1336": {"works": ["001"], "name": "Scaevus Memor"},
    "LAT0426": {"works": ["001"], "name": "Bellum Africum [Anonymous]"},
    "LAT0821": {"works": ["001"], "name": "Bucolica Einsidlensia"},
    "LAT0143": {"works": ["001"], "name": "Trabea"},
    "LAT0100": {"works": ["001"], "name": "Luscius Lanuvinus"},
    "LAT2097": {"works": ["001"], "name": "Sextus Paconianus"},
    "LAT9221": {"works": ["001"], "name": "Paulus Quaestor"},
    "LAT0606": {"works": ["001"], "name": "Lucius Marcius Philippus"},
    "LAT0809": {"works": ["001"], "name": "Aufidius Bassus"},
    "LAT0987": {"works": ["001", "002"], "name": "Publius Pomponius Secundus"},
    "LAT0527": {"works": ["001"], "name": "Gannius"},
    "LAT0709": {"works": ["001", "002"], "name": "Domitius Marsus"},
    "LAT1103": {"works": ["001"], "name": "Priapea"},
    "LAT1248": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
            "015",
            "016",
            "017",
        ],
        "name": "Marcus Cornelius Fronto",
    },
    "LAT0614": {"works": ["001"], "name": "Q. Pompeius Q.f.A.n. Rufus"},
    "LAT0478": {"works": ["002", "003"], "name": "Quintus Tullius Cicero"},
    "LAT0125": {"works": ["001"], "name": "Publius Mucius Scaevola"},
    "LAT0911": {"works": ["001"], "name": "Laus Pisonis"},
    "LAT0616": {"works": ["001", "002"], "name": "Pompilius"},
    "LAT0073": {"works": ["001"], "name": "Gaius Sempronius Gracchus"},
    "LAT0618": {"works": ["001"], "name": "Lucius Pomponius Bononiensis"},
    "LAT1279": {"works": ["001"], "name": "Laelius Felix"},
    "LAT0640": {"works": ["001", "002"], "name": "Marcus Aemilius Scaurus"},
    "LAT1512": {
        "works": ["001", "002", "003", "004", "005", "006", "007"],
        "name": "Pomponius Porphyrio",
    },
    "LAT0037": {"works": ["001"], "name": "Gaius Scribonius Curio pater"},
    "LAT1000": {"works": ["001"], "name": "Pupius (?)"},
    "LAT1282": {"works": ["001"], "name": "Lentulus, mimus"},
    "LAT0002": {"works": ["001"], "name": "Titus Annius Luscus"},
    "LAT0830": {"works": ["001"], "name": "Titus Calpurnius Siculus"},
    "LAT2434": {"works": ["001"], "name": "Hilarius Arelatensis"},
    "LAT1318": {
        "works": ["001", "002", "003"],
        "name": "C. Plinius Caecilius Secundus, Pliny",
    },
    "LAT1342": {"works": ["001"], "name": "Siculus Flaccus"},
    "LAT0694": {"works": ["001"], "name": "Volumnius"},
    "LAT0454": {"works": ["001"], "name": "Marcus Calidius"},
    "LAT1227": {"works": ["001"], "name": "Balbus, grom."},
    "LAT0134": {
        "works": ["001", "002", "003", "004", "005", "006"],
        "name": "Publius Terentius Afer, Terence",
    },
    "LAT0564": {"works": ["001"], "name": "Manius Manilius"},
    "LAT0646": {"works": ["001", "002"], "name": "Lucius Cornelius Sisenna"},
    "LAT0303": {"works": ["001"], "name": "Aurelius Opillus"},
    "LAT0474": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
            "015",
            "016",
            "017",
            "018",
            "019",
            "020",
            "021",
            "022",
            "023",
            "024",
            "025",
            "026",
            "027",
            "028",
            "029",
            "030",
            "031",
            "032",
            "033",
            "034",
            "035",
            "036",
            "037",
            "038",
            "039",
            "040",
            "041",
            "042",
            "043",
            "044",
            "045",
            "046",
            "047",
            "048",
            "049",
            "050",
            "051",
            "052",
            "053",
            "054",
            "055",
            "056",
            "057",
            "058",
            "059",
            "060",
            "061",
            "062",
            "063",
            "064",
            "065",
            "066",
            "067",
            "068",
            "069",
            "070",
            "071",
            "072",
            "073",
            "074",
            "075",
        ],
        "name": "Marcus Tullius Cicero, Cicero, Tully",
    },
    "LAT0460": {"works": ["001"], "name": "Gaius Papirius Carbo Arvina"},
    "LAT1506": {"works": ["001"], "name": "Anonymi Fragmenta de Iure Fisci"},
    "LAT0827": {"works": ["001"], "name": "Caesellius Vindex"},
    "LAT0963": {"works": ["001", "002"], "name": "Quintus Remmius Palaemon"},
    "LAT1254": {"works": ["001"], "name": "Aulus Gellius"},
    "LAT0448": {
        "works": ["001", "002", "003", "004", "005", "006", "007", "008"],
        "name": "Gaius Iulius Caesar, Caesar",
    },
    "LAT0878": {"works": ["001", "002"], "name": "Gaius Asinius Gallus"},
    "LAT0043": {
        "works": ["001", "002", "003", "004", "005", "006", "007"],
        "name": "Quintus Ennius",
    },
    "LAT0064": {"works": ["001", "002"], "name": "Gaius Fannius"},
    "LAT1229": {"works": ["001", "002"], "name": "Flavius Caper"},
    "LAT0469": {"works": ["001"], "name": "Lucius Cassius Longinus"},
    "LAT0692": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
        ],
        "name": "Appendix Vergiliana",
    },
    "LAT0684": {
        "works": [
            "001",
            "002",
            "003",
            "004",
            "005",
            "006",
            "007",
            "008",
            "009",
            "010",
            "011",
            "012",
            "013",
            "014",
            "015",
            "016",
            "017",
        ],
        "name": "Marcus Terentius Varro, Varro",
    },
    "LAT0558": {"works": ["001", "002"], "name": "Gaius Cilnius Maecenas"},
    "LAT2000": {"works": ["001"], "name": "Ablabius"},
    "LAT0321": {"works": ["001"], "name": "Porcius Licinus"},
}
