# Agent Email Layer Demo

This demo showcases the "email for agents" layer built on top of A2A protocol.

## Overview

The agent email layer provides:
- **Global agent addressing**: Agents have human-friendly addresses like `cosmos@futurehouse`
- **Agent discovery**: A registry API to resolve addresses to inbox endpoints
- **Standard A2A inbox**: HTTP endpoints for receiving A2A envelopes
- **Cross-org communication**: Demo showing cosmos (FutureHouse) → gemini (Google) message flow

## Architecture

```
Agent Addressing:
- Canonical URI: agent://futurehouse/cosmos
- Email-like: cosmos@futurehouse

Registry API:
- POST /v1/agents: Register an agent
- GET /v1/agents/by-email/{email}: Lookup by email
- GET /v1/agents/by-uri/{uri}: Lookup by canonical URI

Inbox API:
- POST /v1/a2a/inbox: Receive A2A envelopes
```

## Project Structure

```
src/synqed/agent_email/
├── __init__.py
├── addressing.py          # agent id parsing (uri ⟷ email)
├── registry/
│   ├── models.py         # in-memory registry backend
│   ├── db.py             # postgres registry backend
│   └── api.py            # fastapi endpoints
├── inbox/
│   └── api.py            # a2a inbox endpoint
└── main.py               # fastapi app
```

## Running the Demo

### Option 1: In-Process Demo (Recommended for Quick Start)

Run the demo without needing a separate server:

```bash
cd /Users/dorsa/Desktop/PROJECTS/synq_2/synqed-python
python demos/cosmos_gemini_demo.py --in-process
```

This will:
1. Register cosmos and gemini agents in the registry
2. Resolve gemini@google via email address
3. Send an A2A envelope from cosmos to gemini
4. Display gemini's structured response

### Option 2: HTTP API Demo

For a full demonstration using HTTP endpoints:

**Step 1:** Start the demo server (includes pre-registered agent runtimes):

```bash
cd /Users/dorsa/Desktop/PROJECTS/synq_2/synqed-python
python demos/demo_server.py
```

**Step 2:** In another terminal, run the demo:

```bash
cd /Users/dorsa/Desktop/PROJECTS/synq_2/synqed-python
python demos/cosmos_gemini_demo.py
```

Note: The demo server (`demo_server.py`) includes mock cosmos and gemini runtimes pre-registered, so the HTTP demo can work end-to-end.

## What the Demo Shows

The demo demonstrates a complete cross-org message flow:

1. **Agent Registration**: Cosmos (FutureHouse) and Gemini (Google) are registered in the global registry
2. **Address Resolution**: Cosmos only knows `gemini@google` - it resolves this to the full agent URI and inbox URL
3. **Message Sending**: Cosmos constructs an A2A envelope asking for startup idea analysis
4. **Message Processing**: Gemini receives the envelope, processes it, and returns a structured analysis
5. **Response**: The full A2A response envelope is displayed

## Example Output

```
=== COSMOS → GEMINI DEMO ===

=== registering agents in the registry ===
registered cosmos: agent://futurehouse/cosmos
registered gemini: agent://google/gemini

=== cosmos → gemini demo flow ===

[cosmos] wants to send message to: gemini@google
[cosmos] resolving gemini@google via registry...
[cosmos] resolved to agent_id: agent://google/gemini
[cosmos] inbox_url: http://localhost:8000/v1/a2a/inbox
[cosmos] capabilities: ['a2a/1.0', 'general-reasoning']

[cosmos] sending a2a envelope to gemini's inbox...

[gemini] received message from agent://futurehouse/cosmos
[gemini] thread_id: abc123...
[gemini] sending response with analysis

=== inbox response ===
status: accepted
message_id: def456...

=== gemini's response envelope ===
thread_id: abc123...
role: assistant

content:
structured analysis of the proposal:

1. problem statement: analyze the following startup idea...
2. market potential: high demand in academic and biotech sectors
3. technical feasibility: requires robust nlp and reasoning capabilities
4. competitive advantage: integration with a2a protocol for agent collaboration
5. recommendation: proceed with mvp development

this analysis was generated by gemini@google via the agent email layer.

DEMO COMPLETE
```

## API Documentation

Once the server is running, visit:
- OpenAPI docs: http://localhost:8000/docs
- Root endpoint: http://localhost:8000/

The main FastAPI app is now at `synqed/agent_email/main.py` to keep the agent email layer separate from core synqed functionality.

## Testing

Run the test suite:

```bash
cd /Users/dorsa/Desktop/PROJECTS/synq_2/synqed-python
pytest tests/test_addressing.py -v
pytest tests/test_registry.py -v
pytest tests/test_inbox.py -v
```

## Key Concepts

### Agent Addressing

Agents can be identified in two ways:
- **Canonical URI**: `agent://{org}/{name}[/version]` - e.g., `agent://futurehouse/cosmos`
- **Email-like**: `{name}@{org}` - e.g., `cosmos@futurehouse`

### Registry

The registry is the single source of truth for:
- Agent metadata
- Inbox endpoints
- Capabilities

### A2A Inbox

Standard HTTP endpoint that:
- Validates sender/recipient
- Checks registry
- Routes to local agent runtime
- Returns optional response envelope

## Future Enhancements

The code includes placeholders for:
- **Authentication**: Currently accepts all requests (see `validate_auth` in `inbox/api.py`)
- **Public key verification**: `public_key` field in registry entries
- **Database backend**: In-memory registry can be swapped for Postgres/Redis
- **Cross-process routing**: Currently all agents run locally, but designed to support remote inboxes

