"""
Parallel-LLM: Ultra-Fast Parallel Training and Inference for Language Models

A production-ready library for training and inference of language models with
revolutionary parallel token generation using hybrid diffusion-energy architecture.

Cross-platform support: Works on Windows, Linux, and macOS with graceful degradation
for optional dependencies.
"""

__version__ = "0.6.0"

# Basic platform info (always available)
import sys
PLATFORM = sys.platform
IS_WINDOWS = PLATFORM.startswith('win')
IS_LINUX = PLATFORM.startswith('linux')
IS_MACOS = PLATFORM.startswith('darwin')

# Lazy import for compatibility module
try:
    from . import compat
except ImportError:
    # Create basic compatibility class if full compat fails
    class BasicCompat:
        PLATFORM = PLATFORM
        IS_WINDOWS = IS_WINDOWS
        IS_LINUX = IS_LINUX
        IS_MACOS = IS_MACOS

        def get_installation_instructions(self):
            return "Please install PyTorch: pip install torch torchvision"

        def check_pytorch_cuda(self):
            return False, "PyTorch not available"

        def get_optimal_device(self):
            return None, "PyTorch not available"
    compat = BasicCompat()

# Core exports (backward compatible with graceful error handling)
try:
    from .core import (
        ModelConfig,
        MultimodalConfig,
        DiffusionTransformer,
        get_default_config,
    )
except Exception as e:
    import warnings
    warnings.warn(f"Core components not available: {e}. Please ensure PyTorch is properly installed.")
    # Provide dummy classes to prevent import errors
    ModelConfig = None
    MultimodalConfig = None
    DiffusionTransformer = None
    get_default_config = None

# Training exports (backward compatible)
try:
    from .training import (
        TrainingConfig,
        DistributedTrainer,
        DiffusionLoss,
        ContrastiveLoss,
    )
except Exception as e:
    import warnings
    warnings.warn(f"Training components not available: {e}. Some training features may be limited.")
    TrainingConfig = None
    DistributedTrainer = None
    DiffusionLoss = None
    ContrastiveLoss = None

# Inference exports (backward compatible)
try:
    from .inference import (
        InferenceConfig,
        GenerationConfig,
        ParallelGenerator,
        create_generator,
    )
except Exception as e:
    import warnings
    warnings.warn(f"Inference components not available: {e}. Inference features may be limited.")
    InferenceConfig = None
    GenerationConfig = None
    ParallelGenerator = None
    create_generator = None

# Utils exports
try:
    from .utils import (
        TextDataset,
        MultimodalDataset,
        PreTokenizedDataset,
    )
except Exception as e:
    import warnings
    warnings.warn(f"Utility components not available: {e}. Data loading features may be limited.")
    TextDataset = None
    MultimodalDataset = None
    PreTokenizedDataset = None

__all__ = [
    # Version
    '__version__',
    
    # Core
    'ModelConfig',
    'MultimodalConfig',
    'DiffusionTransformer',
    'get_default_config',
    
    # Training
    'TrainingConfig',
    'DistributedTrainer',
    'DiffusionLoss',
    'ContrastiveLoss',
    
    # Inference
    'InferenceConfig',
    'GenerationConfig',
    'ParallelGenerator',
    'create_generator',
    
    # Utils
    'TextDataset',
    'MultimodalDataset',
    'PreTokenizedDataset',
]
