"""
Cross-platform compatibility module for Parallel-LLM
Handles platform-specific imports and optional dependencies gracefully.
"""

import sys
import warnings
from typing import Optional, Any

# Suppress optional dependency warnings - they clutter output
# Users will see errors if they try to use features that need these
warnings.filterwarnings('ignore', category=UserWarning, module='parallel_llm.compat')

# Suppress Pydantic warnings from dependencies (transformers, etc.)
try:
    # Try to import PydanticUserWarning if available
    from pydantic import PydanticUserWarning
    warnings.filterwarnings('ignore', category=PydanticUserWarning)
except ImportError:
    pass
# Generic filter for the specific message if class import fails or for older versions
warnings.filterwarnings('ignore', message=".*UnsupportedFieldAttributeWarning.*")

# Platform detection
PLATFORM = sys.platform
IS_WINDOWS = PLATFORM.startswith('win')
IS_LINUX = PLATFORM.startswith('linux')
IS_MACOS = PLATFORM.startswith('darwin')

# Lowercase aliases for convenience
is_windows = IS_WINDOWS
is_linux = IS_LINUX
is_macos = IS_MACOS

def try_import(module_name: str, package_name: Optional[str] = None) -> Any:
    """
    Try to import a module, returning None if import fails.
    
    Args:
        module_name: The module to import
        package_name: Optional package name for better error messages
    
    Returns:
        The imported module or None if import fails
    """
    try:
        __import__(module_name)
        return sys.modules[module_name]
    except ImportError:
        # Silently return None - no warnings
        return None

# Core dependencies (lazy loading to avoid import errors)
torch = None  # Will be loaded on demand
numpy = None  # Will be loaded on demand

# Optional GPU dependencies (no warnings if not available)
triton = try_import('triton')
flash_attn = try_import('flash_attn')
xformers = try_import('xformers')

# Optional distributed training (no warnings)
deepspeed = try_import('deepspeed')

# Optional inference optimization (no warnings)
vllm = try_import('vllm')

# Optional multimodal (no warnings)
PIL = try_import('PIL')
requests = try_import('requests')
timm = try_import('timm')

# Optional logging (no warnings)
wandb = try_import('wandb')
tensorboard = try_import('tensorboard')

def check_pytorch_cuda():
    """Check if PyTorch CUDA is available."""
    global torch
    if torch is None:
        torch = try_import('torch')
    if torch is None:
        return False, "PyTorch not installed"

    if not torch.cuda.is_available():
        return False, "CUDA not available"

    return True, f"CUDA {torch.version.cuda} available"

def get_optimal_device():
    """Get the optimal device for computation."""
    global torch
    if torch is None:
        torch = try_import('torch')
    if torch is None:
        return None, "PyTorch not available"

    if torch.cuda.is_available():
        return torch.device('cuda'), "CUDA GPU"
    elif hasattr(torch, 'backends') and hasattr(torch.backends, 'mps') and torch.backends.mps.is_available():
        return torch.device('mps'), "Apple Silicon GPU"
    else:
        return torch.device('cpu'), "CPU"

def get_installation_instructions():
    """Get platform-specific installation instructions."""
    instructions = []

    # PyTorch installation
    if IS_WINDOWS:
        instructions.append("# For Windows:")
        instructions.append("pip install torch torchvision torchaudio --index-url https://download.pytorch.org/whl/cu121  # For CUDA")
        instructions.append("# OR")
        instructions.append("pip install torch torchvision torchaudio --index-url https://download.pytorch.org/whl/cpu    # For CPU only")
    elif IS_LINUX:
        instructions.append("# For Linux:")
        instructions.append("pip install torch torchvision torchaudio --index-url https://download.pytorch.org/whl/cu121  # For CUDA")
        instructions.append("# OR")
        instructions.append("pip install torch torchvision torchaudio --index-url https://download.pytorch.org/whl/cpu    # For CPU only")
    elif IS_MACOS:
        instructions.append("# For macOS:")
        instructions.append("pip install torch torchvision torchaudio  # CPU/MPS support included")

    # Optional dependencies
    instructions.append("\n# Optional GPU acceleration (may not be available on all platforms):")
    instructions.append("pip install parallel-llm[gpu]  # Installs triton, flash-attn, xformers")

    instructions.append("\n# Multimodal capabilities:")
    instructions.append("pip install parallel-llm[multimodal]  # Installs Pillow, timm, open-clip-torch")

    instructions.append("\n# Development tools:")
    instructions.append("pip install parallel-llm[dev]  # Installs pytest, black, mypy, etc.")

    instructions.append("\n# Everything:")
    instructions.append("pip install parallel-llm[all]  # Installs all optional dependencies")

    return "\n".join(instructions)

# Export compatibility info
__all__ = [
    'PLATFORM', 'IS_WINDOWS', 'IS_LINUX', 'IS_MACOS',
    'torch', 'numpy', 'triton', 'flash_attn', 'xformers',
    'deepspeed', 'vllm', 'PIL', 'requests', 'timm',
    'wandb', 'tensorboard',
    'check_pytorch_cuda', 'get_optimal_device', 'get_installation_instructions'
]
