"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('check that instantiation works', () => {
    // GIVEN
    const stack = testStack();
    const vpc = new ec2.Vpc(stack, 'VPC');
    // WHEN
    new lib_1.Cluster(stack, 'Redshift', {
        masterUser: {
            masterUsername: 'admin',
            masterPassword: cdk.SecretValue.plainText('tooshort'),
        },
        vpc,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Redshift::Cluster', {
        Properties: {
            AllowVersionUpgrade: true,
            MasterUsername: 'admin',
            MasterUserPassword: 'tooshort',
            ClusterType: 'multi-node',
            AutomatedSnapshotRetentionPeriod: 1,
            Encrypted: true,
            NumberOfNodes: 2,
            NodeType: 'dc2.large',
            DBName: 'default_db',
            PubliclyAccessible: false,
            ClusterSubnetGroupName: { Ref: 'RedshiftSubnetsDFE70E0A' },
            VpcSecurityGroupIds: [{ 'Fn::GetAtt': ['RedshiftSecurityGroup796D74A7', 'GroupId'] }],
        },
        DeletionPolicy: 'Retain',
        UpdateReplacePolicy: 'Retain',
    }, assert_1.ResourcePart.CompleteDefinition));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Redshift::ClusterSubnetGroup', {
        Properties: {
            Description: 'Subnets for Redshift Redshift cluster',
            SubnetIds: [
                { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                { Ref: 'VPCPrivateSubnet3Subnet3EDCD457' },
            ],
        },
        DeletionPolicy: 'Retain',
        UpdateReplacePolicy: 'Retain',
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('can create a cluster with imported vpc and security group', () => {
    // GIVEN
    const stack = testStack();
    const vpc = ec2.Vpc.fromLookup(stack, 'VPC', {
        vpcId: 'VPC12345',
    });
    const sg = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'SecurityGroupId12345');
    // WHEN
    new lib_1.Cluster(stack, 'Redshift', {
        masterUser: {
            masterUsername: 'admin',
            masterPassword: cdk.SecretValue.plainText('tooshort'),
        },
        vpc,
        securityGroups: [sg],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Redshift::Cluster', {
        ClusterSubnetGroupName: { Ref: 'RedshiftSubnetsDFE70E0A' },
        MasterUsername: 'admin',
        MasterUserPassword: 'tooshort',
        VpcSecurityGroupIds: ['SecurityGroupId12345'],
    }));
});
test('creates a secret when master credentials are not specified', () => {
    // GIVEN
    const stack = testStack();
    const vpc = new ec2.Vpc(stack, 'VPC');
    // WHEN
    new lib_1.Cluster(stack, 'Redshift', {
        masterUser: {
            masterUsername: 'admin',
        },
        vpc,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Redshift::Cluster', {
        MasterUsername: {
            'Fn::Join': [
                '',
                [
                    '{{resolve:secretsmanager:',
                    {
                        Ref: 'RedshiftSecretA08D42D6',
                    },
                    ':SecretString:username::}}',
                ],
            ],
        },
        MasterUserPassword: {
            'Fn::Join': [
                '',
                [
                    '{{resolve:secretsmanager:',
                    {
                        Ref: 'RedshiftSecretA08D42D6',
                    },
                    ':SecretString:password::}}',
                ],
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
        GenerateSecretString: {
            ExcludeCharacters: '"@/\\\ \'',
            GenerateStringKey: 'password',
            PasswordLength: 30,
            SecretStringTemplate: '{"username":"admin"}',
        },
    }));
});
test('SIngle Node CLusters spawn only single node', () => {
    // GIVEN
    const stack = testStack();
    const vpc = new ec2.Vpc(stack, 'VPC');
    // WHEN
    new lib_1.Cluster(stack, 'Redshift', {
        masterUser: {
            masterUsername: 'admin',
        },
        vpc,
        nodeType: lib_1.NodeType.DC1_8XLARGE,
        clusterType: lib_1.ClusterType.SINGLE_NODE,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Redshift::Cluster', {
        ClusterType: 'single-node',
        NodeType: 'dc1.8xlarge',
        NumberOfNodes: 1,
    }));
});
test('create an encrypted cluster with custom KMS key', () => {
    // GIVEN
    const stack = testStack();
    const vpc = new ec2.Vpc(stack, 'VPC');
    // WHEN
    new lib_1.Cluster(stack, 'Redshift', {
        masterUser: {
            masterUsername: 'admin',
        },
        encryptionKey: new kms.Key(stack, 'Key'),
        vpc,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Redshift::Cluster', {
        KmsKeyId: {
            'Fn::GetAtt': [
                'Key961B73FD',
                'Arn',
            ],
        },
    }));
});
test('cluster with parameter group', () => {
    // GIVEN
    const stack = testStack();
    const vpc = new ec2.Vpc(stack, 'VPC');
    // WHEN
    const group = new lib_1.ClusterParameterGroup(stack, 'Params', {
        description: 'bye',
        parameters: {
            param: 'value',
        },
    });
    new lib_1.Cluster(stack, 'Redshift', {
        masterUser: {
            masterUsername: 'admin',
        },
        vpc,
        parameterGroup: group,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Redshift::Cluster', {
        ClusterParameterGroupName: { Ref: 'ParamsA8366201' },
    }));
});
test('imported cluster with imported security group honors allowAllOutbound', () => {
    // GIVEN
    const stack = testStack();
    const cluster = lib_1.Cluster.fromClusterAttributes(stack, 'Database', {
        clusterEndpointAddress: 'addr',
        clusterName: 'identifier',
        clusterEndpointPort: 3306,
        securityGroups: [
            ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        ],
    });
    // WHEN
    cluster.connections.allowToAnyIpv4(ec2.Port.tcp(443));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
        GroupId: 'sg-123456789',
    }));
});
test('can create a cluster with logging enabled', () => {
    // GIVEN
    const stack = testStack();
    const vpc = new ec2.Vpc(stack, 'VPC');
    const bucket = s3.Bucket.fromBucketName(stack, 'bucket', 'logging-bucket');
    // WHEN
    new lib_1.Cluster(stack, 'Redshift', {
        masterUser: {
            masterUsername: 'admin',
        },
        vpc,
        loggingBucket: bucket,
        loggingKeyPrefix: 'prefix',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Redshift::Cluster', {
        LoggingProperties: {
            BucketName: 'logging-bucket',
            S3KeyPrefix: 'prefix',
        },
    }));
});
test('throws when trying to add rotation to a cluster without secret', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'VPC');
    // WHEN
    const cluster = new lib_1.Cluster(stack, 'Redshift', {
        masterUser: {
            masterUsername: 'admin',
            masterPassword: cdk.SecretValue.plainText('tooshort'),
        },
        vpc,
    });
    // THEN
    expect(() => {
        cluster.addRotationSingleUser();
    }).toThrowError();
});
test('throws validation error when trying to set encryptionKey without enabling encryption', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'VPC');
    const key = new kms.Key(stack, 'kms-key');
    // WHEN
    const props = {
        encrypted: false,
        encryptionKey: key,
        masterUser: {
            masterUsername: 'admin',
        },
        vpc,
    };
    // THEN
    expect(() => {
        new lib_1.Cluster(stack, 'Redshift', props);
    }).toThrowError();
});
test('throws when trying to add single user rotation multiple times', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const vpc = new ec2.Vpc(stack, 'VPC');
    const cluster = new lib_1.Cluster(stack, 'Redshift', {
        masterUser: {
            masterUsername: 'admin',
        },
        vpc,
    });
    // WHEN
    cluster.addRotationSingleUser();
    // THEN
    expect(() => {
        cluster.addRotationSingleUser();
    }).toThrowError();
});
function testStack() {
    const stack = new cdk.Stack(undefined, undefined, { env: { account: '12345', region: 'us-test-1' } });
    stack.node.setContext('availability-zones:12345:us-test-1', ['us-test-1a', 'us-test-1b']);
    return stack;
}
//# sourceMappingURL=data:application/json;base64,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