"""
Profiling script for sync check performance analysis.

This script profiles the test_sync_checks_have_no_async_overhead test case
to identify performance bottlenecks.
"""

import cProfile
import pstats
import io
from flex_evals import TestCase, Output, Check, evaluate


def run_sync_check_test() -> None:
    """Run the sync check test with 200 test cases."""
    # Create 200 test cases with sync checks (same as the test)
    num_test_cases = 200

    test_cases = [
        TestCase(id=str(i), input={"value": i})
        for i in range(num_test_cases)
    ]
    outputs = [
        Output(value={"result": i})
        for i in range(num_test_cases)
    ]

    # Sync check only
    checks = [
        Check(type="exact_match", arguments={
            "actual": "$.output.value.result",
            "expected": "$.test_case.input.value",
        }),
    ]

    # Run evaluation
    result = evaluate(test_cases, outputs, checks)

    # Verify it worked
    assert result.status == "completed"
    assert result.summary.total_test_cases == num_test_cases
    print(f"\nEvaluation completed successfully: {num_test_cases} test cases")


def profile_and_report() -> None:
    """Profile the sync check test and generate detailed reports."""
    print("=" * 80)
    print("PROFILING SYNC CHECK PERFORMANCE (200 test cases)")
    print("=" * 80)

    # Create profiler
    profiler = cProfile.Profile()

    # Run with profiling
    print("\nRunning evaluation with profiling enabled...")
    profiler.enable()
    run_sync_check_test()
    profiler.disable()

    # Create stats object
    stats = pstats.Stats(profiler)

    # Report 1: Top functions by cumulative time
    print("\n" + "=" * 80)
    print("TOP 30 FUNCTIONS BY CUMULATIVE TIME")
    print("=" * 80)
    stats.sort_stats('cumulative')
    stats.print_stats(30)

    # Report 2: Top functions by time spent in function itself (not including calls)
    print("\n" + "=" * 80)
    print("TOP 30 FUNCTIONS BY TIME SPENT (excluding subcalls)")
    print("=" * 80)
    stats.sort_stats('time')
    stats.print_stats(30)

    # Report 3: Focus on specific modules
    print("\n" + "=" * 80)
    print("TIME IN flex_evals.engine MODULE")
    print("=" * 80)
    stats.sort_stats('cumulative')
    stats.print_stats('flex_evals/engine')

    print("\n" + "=" * 80)
    print("TIME IN flex_evals.checks.base MODULE")
    print("=" * 80)
    stats.sort_stats('cumulative')
    stats.print_stats('flex_evals/checks/base')

    print("\n" + "=" * 80)
    print("TIME IN flex_evals.utils.jsonpath_resolver MODULE")
    print("=" * 80)
    stats.sort_stats('cumulative')
    stats.print_stats('flex_evals/utils/jsonpath_resolver')

    # Report 4: Callers of expensive functions
    print("\n" + "=" * 80)
    print("CALLERS OF resolve_fields (if it appears)")
    print("=" * 80)
    stats.print_callers('resolve_fields')

    print("\n" + "=" * 80)
    print("CALLERS OF resolve_argument (if it appears)")
    print("=" * 80)
    stats.print_callers('resolve_argument')

    # Generate detailed stats file for further analysis
    output_file = "profile_stats.txt"
    with open(output_file, 'w') as f:
        ps = pstats.Stats(profiler, stream=f)
        ps.sort_stats('cumulative')
        ps.print_stats()

    print(f"\n\nDetailed profiling stats saved to: {output_file}")
    print("\nYou can analyze this further with:")
    print("  import pstats")
    print("  p = pstats.Stats('profile_stats.txt')")
    print("  p.sort_stats('cumulative').print_stats(50)")


if __name__ == "__main__":
    profile_and_report()
