"""
Build Flow - Orchestrates the complete build process
"""

import os
import time
import asyncio
import aiohttp
from typing import List, Optional, Set
from dataclasses import dataclass, asdict

from .types import (
    Step,
    StepType,
    BuildOptions,
    BuildResult,
    CreateVMOptions,
    VM,
    UploadLinkResponse,
    BuildResponse,
    BuildStatusResponse,
)
from .file_hasher import FileHasher
from .tar_creator import TarCreator


DEFAULT_BASE_URL = "https://api.your-domain.com"


async def build_template(template, options: BuildOptions) -> BuildResult:
    """
    Build a template
    
    Args:
        template: Template instance
        options: Build options
        
    Returns:
        BuildResult with template ID and helpers
    """
    base_url = options.base_url or DEFAULT_BASE_URL
    context_path = options.context_path or os.getcwd()
    
    # Step 1: Calculate file hashes for COPY steps
    steps_with_hashes = await calculate_step_hashes(
        template.get_steps(), 
        context_path, 
        options
    )
    
    # Step 2: Upload files for COPY steps
    await upload_files(steps_with_hashes, context_path, base_url, options)
    
    # Step 3: Trigger build
    build_response = await trigger_build(
        steps_with_hashes,
        template.get_start_cmd(),
        template.get_ready_check(),
        base_url,
        options,
    )
    
    # Step 4: Stream logs (if callback provided)
    if options.on_log or options.on_progress:
        await stream_logs(build_response.build_id, base_url, options)
    
    # Step 5: Poll status until complete
    final_status = await poll_status(build_response.build_id, base_url, options)
    
    if final_status.status != "success":
        raise Exception(f"Build failed: {final_status.error or 'Unknown error'}")
    
    # Calculate duration
    duration = int(time.time() * 1000) - int(
        time.mktime(time.strptime(final_status.started_at, "%Y-%m-%dT%H:%M:%SZ")) * 1000
    )
    
    # Create VM helper function
    async def create_vm_helper(vm_options: CreateVMOptions = None) -> VM:
        return await create_vm_from_template(
            final_status.template_id, 
            base_url, 
            options, 
            vm_options or CreateVMOptions()
        )
    
    # Return result
    return BuildResult(
        build_id=build_response.build_id,
        template_id=final_status.template_id,
        duration=duration,
        _create_vm_func=create_vm_helper,
    )


async def calculate_step_hashes(
    steps: List[Step], 
    context_path: str, 
    options: BuildOptions
) -> List[Step]:
    """Calculate file hashes for COPY steps"""
    hasher = FileHasher()
    result = []
    
    for step in steps:
        if step.type == StepType.COPY:
            src, dest = step.args[0], step.args[1]
            sources = src.split(',')
            
            # Calculate hash for all sources
            hash_value = await hasher.calculate_multi_hash(
                [(s, dest) for s in sources],
                context_path
            )
            
            # Create new step with hash
            new_step = Step(
                type=step.type,
                args=step.args,
                files_hash=hash_value,
                skip_cache=step.skip_cache,
            )
            result.append(new_step)
        else:
            result.append(step)
    
    return result


async def upload_files(
    steps: List[Step],
    context_path: str,
    base_url: str,
    options: BuildOptions,
) -> None:
    """Upload files for COPY steps"""
    tar_creator = TarCreator()
    uploaded_hashes: Set[str] = set()
    
    async with aiohttp.ClientSession() as session:
        for step in steps:
            if step.type == StepType.COPY and step.files_hash:
                # Skip if already uploaded
                if step.files_hash in uploaded_hashes:
                    continue
                
                # Get sources
                src = step.args[0]
                sources = src.split(',')
                
                # Create tar.gz
                tar_result = await tar_creator.create_multi_tar_gz(sources, context_path)
                
                try:
                    # Request upload link
                    upload_link = await get_upload_link(
                        step.files_hash,
                        tar_result.size,
                        base_url,
                        options.api_key,
                        session,
                    )
                    
                    # Upload if not already present
                    if not upload_link.present and upload_link.upload_url:
                        await upload_file(upload_link.upload_url, tar_result, session)
                    
                    uploaded_hashes.add(step.files_hash)
                finally:
                    # Cleanup temporary file
                    tar_result.cleanup()


async def get_upload_link(
    files_hash: str,
    content_length: int,
    base_url: str,
    api_key: str,
    session: aiohttp.ClientSession,
) -> UploadLinkResponse:
    """Get presigned upload URL"""
    async with session.post(
        f"{base_url}/v1/templates/files/upload-link",
        headers={
            "Authorization": f"Bearer {api_key}",
            "Content-Type": "application/json",
        },
        json={
            "files_hash": files_hash,
            "content_length": content_length,
        },
    ) as response:
        if not response.ok:
            raise Exception(f"Failed to get upload link: {response.status}")
        
        data = await response.json()
        return UploadLinkResponse(**data)


async def upload_file(
    upload_url: str,
    tar_result,
    session: aiohttp.ClientSession,
) -> None:
    """Upload file to R2"""
    with tar_result.open('rb') as f:
        file_content = f.read()
    
    async with session.put(
        upload_url,
        headers={
            "Content-Type": "application/gzip",
            "Content-Length": str(tar_result.size),
        },
        data=file_content,
    ) as response:
        if not response.ok:
            raise Exception(f"Upload failed: {response.status}")


async def trigger_build(
    steps: List[Step],
    start_cmd: Optional[str],
    ready_cmd: Optional[dict],
    base_url: str,
    options: BuildOptions,
) -> BuildResponse:
    """Trigger build"""
    # Convert steps to dict
    steps_dict = []
    for step in steps:
        step_dict = {
            "type": step.type.value,
            "args": step.args,
        }
        if step.files_hash:
            step_dict["filesHash"] = step.files_hash
        if step.skip_cache:
            step_dict["skipCache"] = True
        steps_dict.append(step_dict)
    
    # Convert ready check to dict
    ready_cmd_dict = None
    if ready_cmd:
        ready_cmd_dict = {
            "type": ready_cmd.type.value,
            "timeout": ready_cmd.timeout,
            "interval": ready_cmd.interval,
        }
        if ready_cmd.port:
            ready_cmd_dict["port"] = ready_cmd.port
        if ready_cmd.url:
            ready_cmd_dict["url"] = ready_cmd.url
        if ready_cmd.path:
            ready_cmd_dict["path"] = ready_cmd.path
        if ready_cmd.process_name:
            ready_cmd_dict["processName"] = ready_cmd.process_name
        if ready_cmd.command:
            ready_cmd_dict["command"] = ready_cmd.command
    
    async with aiohttp.ClientSession() as session:
        async with session.post(
            f"{base_url}/v1/templates/build",
            headers={
                "Authorization": f"Bearer {options.api_key}",
                "Content-Type": "application/json",
            },
            json={
                "alias": options.alias,
                "steps": steps_dict,
                "startCmd": start_cmd,
                "readyCmd": ready_cmd_dict,
                "cpu": options.cpu,
                "memory": options.memory,
                "diskGB": options.disk_gb,
                "skipCache": options.skip_cache,
            },
        ) as response:
            if not response.ok:
                raise Exception(f"Build trigger failed: {response.status}")
            
            data = await response.json()
            return BuildResponse(**data)


async def stream_logs(
    build_id: str,
    base_url: str,
    options: BuildOptions,
) -> None:
    """Stream logs via SSE"""
    try:
        async with aiohttp.ClientSession() as session:
            async with session.get(
                f"{base_url}/v1/templates/build/{build_id}/logs",
                headers={
                    "Authorization": f"Bearer {options.api_key}",
                    "Accept": "text/event-stream",
                },
            ) as response:
                if not response.ok:
                    return  # Fallback to polling
                
                async for line in response.content:
                    line = line.decode('utf-8').strip()
                    
                    if line.startswith('event: '):
                        event = line[7:]
                        
                        if event == 'complete':
                            return
    except Exception as e:
        # Fallback to polling if SSE fails
        print(f"SSE streaming failed, falling back to polling: {e}")


async def poll_status(
    build_id: str,
    base_url: str,
    options: BuildOptions,
    interval_ms: int = 2000,
) -> BuildStatusResponse:
    """Poll build status"""
    async with aiohttp.ClientSession() as session:
        while True:
            async with session.get(
                f"{base_url}/v1/templates/build/{build_id}/status",
                headers={
                    "Authorization": f"Bearer {options.api_key}",
                },
            ) as response:
                if not response.ok:
                    raise Exception(f"Status check failed: {response.status}")
                
                data = await response.json()
                status = BuildStatusResponse(**data)
                
                if status.status in ["success", "failed"]:
                    return status
                
                # Wait before next poll
                await asyncio.sleep(interval_ms / 1000)


async def get_logs(
    build_id: str,
    api_key: str,
    offset: int = 0,
    base_url: str = None,
) -> "LogsResponse":
    """
    Get build logs with offset-based polling
    
    Args:
        build_id: Build ID
        api_key: API key
        offset: Starting offset (default: 0)
        base_url: Base URL (default: https://api.hopx.dev)
        
    Returns:
        LogsResponse with logs, offset, status, complete
        
    Example:
        ```python
        from bunnyshell.template import get_logs
        
        # Get logs from beginning
        response = await get_logs("123", "api_key")
        print(response.logs)
        
        # Get new logs from last offset
        response = await get_logs("123", "api_key", offset=response.offset)
        ```
    """
    from .types import LogsResponse
    
    if base_url is None:
        base_url = DEFAULT_BASE_URL
    
    async with aiohttp.ClientSession() as session:
        async with session.get(
            f"{base_url}/v1/templates/build/{build_id}/logs",
            params={"offset": offset},
            headers={
                "Authorization": f"Bearer {api_key}",
            },
        ) as response:
            if not response.ok:
                raise Exception(f"Get logs failed: {response.status}")
            
            data = await response.json()
            return LogsResponse(
                logs=data.get("logs", ""),
                offset=data.get("offset", 0),
                status=data.get("status", "unknown"),
                complete=data.get("complete", False),
                request_id=data.get("request_id"),
            )


async def create_vm_from_template(
    template_id: str,
    base_url: str,
    build_options: BuildOptions,
    vm_options: CreateVMOptions,
) -> VM:
    """Create VM from template"""
    async with aiohttp.ClientSession() as session:
        async with session.post(
            f"{base_url}/v1/vms/create",
            headers={
                "Authorization": f"Bearer {build_options.api_key}",
                "Content-Type": "application/json",
            },
            json={
                "templateID": template_id,
                "alias": vm_options.alias,
                "cpu": vm_options.cpu,
                "memory": vm_options.memory,
                "diskGB": vm_options.disk_gb,
                "envVars": vm_options.env_vars,
            },
        ) as response:
            if not response.ok:
                raise Exception(f"VM creation failed: {response.status}")
            
            vm_data = await response.json()
            
            # Create delete function
            async def delete_func():
                await delete_vm(vm_data["vmID"], base_url, build_options.api_key)
            
            return VM(
                vm_id=vm_data["vmID"],
                template_id=vm_data["templateID"],
                status=vm_data["status"],
                ip=vm_data["ip"],
                agent_url=vm_data["agentUrl"],
                started_at=vm_data["startedAt"],
                _delete_func=delete_func,
            )


async def delete_vm(vm_id: str, base_url: str, api_key: str) -> None:
    """Delete VM"""
    async with aiohttp.ClientSession() as session:
        async with session.delete(
            f"{base_url}/v1/vms/{vm_id}",
            headers={
                "Authorization": f"Bearer {api_key}",
            },
        ) as response:
            if not response.ok:
                raise Exception(f"VM deletion failed: {response.status}")

