# Analytics Dashboard API Example

A comprehensive analytics and business intelligence API built with FraiseQL, demonstrating time-series data handling, complex aggregations, and real-time analytics capabilities.

## Features

- **Time-series Analytics**: High-performance time-based data analysis
- **User Behavior Tracking**: Sessions, page views, and user journeys
- **Conversion Funnels**: Multi-step conversion tracking and analysis
- **A/B Testing**: Experiment management and statistical analysis
- **Performance Monitoring**: Application performance metrics and alerts
- **Revenue Analytics**: Financial metrics and revenue attribution
- **Real-time Dashboards**: Live updating analytics dashboards
- **Custom Events**: Flexible event tracking system
- **Error Tracking**: Application error monitoring and analysis
- **Cohort Analysis**: User retention and engagement analysis

## Architecture

This example demonstrates FraiseQL's analytical capabilities:

- **Time-series Optimization**: TimescaleDB integration for time-series data
- **Materialized Views**: Pre-computed aggregations for performance
- **Window Functions**: Advanced SQL analytics with PostgreSQL
- **Data Warehousing**: OLAP-style queries for business intelligence
- **Real-time Processing**: Live data ingestion and analysis

## Key Components

### 1. Data Collection
- User session tracking
- Page view analytics
- Custom event tracking
- Performance metrics
- Error monitoring

### 2. Analytics Engine
- Real-time aggregations
- Historical trend analysis
- Cohort analysis
- Funnel analysis
- A/B test results

### 3. Visualization
- GraphQL APIs for dashboards
- Time-series data export
- Custom report generation
- Real-time metric streaming

## Setup

### 1. Database Setup

```bash
# Create database with TimescaleDB (optional)
createdb analytics_db

# Install TimescaleDB extension (optional but recommended)
psql -d analytics_db -c "CREATE EXTENSION IF NOT EXISTS timescaledb CASCADE;"

# Run migrations
psql -d analytics_db -f db/migrations/001_analytics_schema.sql
psql -d analytics_db -f db/views/analytics_views.sql
psql -d analytics_db -f db/functions/analytics_functions.sql
```

### 2. Application Setup

```bash
# Install dependencies
pip install fraiseql fastapi uvicorn asyncpg pandas numpy

# Set environment
export DATABASE_URL="postgresql://user:password@localhost:5432/analytics_db"

# Run server
uvicorn app:app --reload
```

## Usage Examples

### Track Page View

```graphql
mutation TrackPageView {
  trackPageView(
    applicationId: "app123"
    sessionId: "session456"
    pageUrl: "https://example.com/product/123"
    pageTitle: "Product Details"
    loadTimeMs: 250
  ) {
    success
    message
  }
}
```

### Get Traffic Analytics

```graphql
query GetTrafficAnalytics($appId: UUID!, $timeRange: DateRange!) {
  trafficAnalytics(
    where: {
      applicationId: { _eq: $appId }
      timestamp: { _gte: $timeRange.start, _lte: $timeRange.end }
    }
  ) {
    date
    pageViews
    uniqueVisitors
    bounceRate
    avgSessionDuration
    topPages {
      url
      views
      uniqueViews
    }
  }
}
```

### Conversion Funnel Analysis

```graphql
query GetConversionFunnel($appId: UUID!) {
  conversionFunnel(
    where: { applicationId: { _eq: $appId } }
  ) {
    step
    stepName
    users
    conversionRate
    dropoffRate
  }
}
```

## Performance Features

### 1. Time-series Optimization
- TimescaleDB hypertables for automatic partitioning
- Efficient time-based queries
- Automatic data compression
- Retention policies for old data

### 2. Pre-computed Aggregations
- Materialized views for common metrics
- Incremental view updates
- Background refresh jobs
- Query result caching

### 3. Analytical SQL
- Window functions for trend analysis
- CTEs for complex calculations
- Lateral joins for correlated data
- Advanced aggregation functions

## Next Steps

1. Add machine learning insights
2. Implement real-time alerting
3. Build interactive dashboards
4. Add data export capabilities
5. Implement data privacy controls

This example showcases FraiseQL's power for building sophisticated analytics platforms with PostgreSQL's advanced analytical capabilities.
