# Copyright (c) 2024 Snowflake Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from __future__ import annotations

from dataclasses import dataclass
from typing import Callable, List

from snowflake.cli._app.commands_registration.command_plugins_loader import (
    load_builtin_and_external_command_plugins,
    load_only_builtin_command_plugins,
)
from snowflake.cli._app.commands_registration.typer_registration import (
    register_commands_from_plugins,
)
from snowflake.cli.api.plugins.plugin_config import PluginConfigProvider


@dataclass
class CommandRegistrationConfig:
    enable_external_command_plugins: bool


class CommandsRegistrationWithCallbacks:
    def __init__(self):
        self._plugin_config_manager = PluginConfigProvider()
        self._callbacks_after_registration: List[Callable[[], None]] = []
        self._commands_registration_config: CommandRegistrationConfig = (
            CommandRegistrationConfig(enable_external_command_plugins=True)
        )
        self._commands_already_registered: bool = False

    def register_commands_from_plugins(self) -> None:
        if self._commands_registration_config.enable_external_command_plugins:
            self._register_builtin_and_enabled_external_plugin_commands()
        else:
            self._register_only_builtin_plugin_commands()

        self._commands_already_registered = True
        for callback in self._callbacks_after_registration:
            callback()

    @staticmethod
    def _register_only_builtin_plugin_commands() -> None:
        loaded_command_plugins = load_only_builtin_command_plugins()
        register_commands_from_plugins(loaded_command_plugins)

    def _register_builtin_and_enabled_external_plugin_commands(self):
        enabled_external_plugins = (
            self._plugin_config_manager.get_enabled_plugin_names()
        )
        loaded_command_plugins = load_builtin_and_external_command_plugins(
            enabled_external_plugins
        )
        register_commands_from_plugins(loaded_command_plugins)

    def disable_external_command_plugins(self):
        self._commands_registration_config.enable_external_command_plugins = False

    def after(self, callback):
        def delayed_callback(value):
            self._callbacks_after_registration.append(lambda: callback(value))

        return delayed_callback

    def reset_running_instance_registration_state(self):
        self._callbacks_after_registration.clear()
        self._commands_registration_config.enable_external_command_plugins = True
