# Global State

`ethpwn` maintains a variety of global state across runs to allow the interaction scripts to focus only on the logic of the interaction, and not on the boilerplate of setting up the environment.

Specifically, `ethpwn` maintains the following concepts in a global state: global parameters (e.g., API keys, wallets), settings, and the [contract registry](#contractregistry).

## 📝 Configuration

| ❗️ Note                                                              |
|----------------------------------------------------------------------|
| To start out, if you haven't created your global config before, you should run `ethpwn config create` to generate your initial configuration file. This command will interactively prompt you for the most important settings, including the Ethereum node URL to use and wallets you want to use. |

The configuration for `ethpwn` is located at `~/.config/ethpwn/config.json`. This file allows you to configure a variety of settings globally, which are then used by the various modules to simplify the interaction process.

You can use the `ethpwn config` and `ethpwn wallet` commands to modify the most common configuration settings, or you can edit the file directly.

The most important settings are the following:

1. `default_network`: the default network to use for all interactions (currently supported: `mainnet` and `sepolia`)
2. `default_node_urls`: per-network default Ethereum node URLs to retrieve blockchain information state
3. `wallets`: Ethereum wallets to interact with the different blockchain networks
4. `etherscan_api_key`: API key for Etherscan to retrieve verified source code for contracts
5. `debugger configuration`: settings for `ethdbg`. See [ethdbg](/ethpwn/ethdbg/usage/).

The configuration is structured as follows and is shared between `ethdbg` and `ethpwn`:

```json
{
  "default_network": "mainnet",
  "default_node_urls": {
    "mainnet": "<YOUR_ETHEREUM_NODE_URL>",
  },
  "credentials": {
    "etherscan": "<OPTIONAL_ETHERSCAN_API>"
  },
  "dbg": {
  }
}
```

The other configuration file is `~/.config/ethpwn/wallets.json` and contains the Ethereum wallet configuration to use with `ethpwn`.

An example of such a configuration is the following:

```json
[
  {
    "address": "0x1a5984F43dAD95a5121b1b30B9190d619d84d21C",
    "private_key": "0x2838aa1e473a046941d3ee4481396b9c54c944a6b6321e489b654554125f374b",
    "network": "mainnet",
    "name": "my-test-wallet",
    "description": "Default wallet generated by ethpwn on 2023-07-09 13:12:05.184774"
  }
]
```

| 🛑 Warning                               |
|------------------------------------------|
| ***!!!!! DO NOT use accounts and private keys that hold valuable assets in this config file! This file is NOT protected in any way. Putting sensitive private keys here might lead to exposing them in the clear and can cause the loss of funds on the related account if someone can steal them. ALWAYS use test accounts!***|

## 📚 ContractRegistry
The contract registry is your personal library of smart contract metadata.
At a high-level, the contract-registry is simply a mapping from contract addresses to their corresponding metadata.

Contracts are automatically added to the contract registry by `ethpwn` in the following cases:

1. You deploy a smart contract via `ContractMetadata.deploy()`
2. You interact with a verified contract on-chain (if you have a valid Etherscan API configured)
3. You manually register contract metadata with a contract instance via `ContractMetadata.get_contract_at(<address>)` (or with the [CLI](/ethpwn/ethpwn/cli/))

`ethpwn` then stores this association in its global `ContractRegistry` object, which can be accessed via `contract_registry()`.
The contract registry is stored locally on your machine in `~/.config/ethpwn/contract_registry/` by default and will be loaded every time you use `ethpwn`.

The example in the [tutorial](#tutorial) below illustrates this by retrieving a contract instance for the UniswapRouter contract from the contract registry, and then using it to interact with the contract without having to specify the address, ABI, storage layout, or source code of the contract.

### ✅ Etherscan Verified Source Code
If you have configured `ethpwn` with an Etherscan API key, `ethpwn` can fetch source code for contracts from Etherscan's verified contracts library.
This allows you to transparently retrieve the metadata for these contracts without needing to explicitly compile them yourself. The target contract is automatically compiled and added to the contract registry for you.

To use this feature, set the `ETHERSCAN_API_KEY` environment variable to your etherscan API key, or add it to your `ethpwn` configuration file.

Then you can use the following command to fetch the verified source code for, e.g., the Uniswap Router contract:
```bash
ethpwn contract fetch_verified_contract 0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D
```

Or, in a Python script, you can do the following:
```python
from ethpwn import *

# either specify the API key in the environment or the configuration file
instance = fetch_verified_contract(0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D)

# or specify it explicitly
instance = fetch_verified_contract(0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D, api_key='<YOUR_API_KEY>')
```

## 🗃 ContractMetadata
Whenever you compile contracts within a Python script, `ethpwn` manages the compiled contract metadata in a *temporary*  `ContractMetadata` object that can be accessed via the `CONTRACT_METADATA` variable.

Note that, this object does NOT persist after the termination of your script.
Instead, we store the corresponding `ContractMetadata` object only when you record a deployed contract in the `contract_registry` (in this case the contract being deployed is unambiguous).

The contract metadata contains, among other things, the following information:

- `metadata.compiler` - the compiler used to compile the contract
- `metadata.sources` - the sources used to compile the contract
- `metadata.source_file` - the source file the current contract is found in
- `metadata.contract_name` - the name of the contract
- `metadata.bin_runtime` - the bytecode of the contract deployed on the blockchain
- `metadata.srcmap_runtime` - the source map of the deployed contract
- `metadata.abi` - the ABI of the contract, used to interact with the contract
- `metadata.storage_layout` - the storage layout of the contract, used to display and retrieve storage variables in `ethdbg`

It also provides various helper functions to manipulate or analyze instances of this contract, e.g.

```python
  def source_info_for_pc(self, pc, fork='paris') -> InstructionSourceInfo:
    '''
    Returns the source info for the instruction at the given program counter in the deployed bytecode.
    '''

  def deploy(self, *constructor_args, **tx_extras) -> Tuple[HexBytes, Contract]:
    '''
    Deploys an instance of this contract to the blockchain and registers it with the contract registry.
    '''

  def get_contract_at(self, addr) -> Contract:
    '''
    Returns a web3 contract instance for the contract at the given address. This will
    automatically register this contract instance with the contract registry.
    '''

  def decode_function_input(self, data):
    '''
    Decodes the function input data for a contract of this class. Returns a tuple of the
    function name and a dictionary of the arguments.
    '''
```

In the current session you can retrieve contract metadata by contract name, e.g., if you have previously compiled the code of the uniswap router contract, you can use `CONTRACT_METADATA['UniswapV2Router02']` to retrieve the `ContractMetadata` for this contract.

You can also retrieve the `ContractMetadata` for a contract that is stored in the `contract_registry` by using `contract_registry().get(<contract_address>).metadata`.

## 🐥 Tutorial

Here is an example of how you can leverage the global states used and exported by `ethpwn` in your scripts.

The following example relies on the following setup using the `ethpwn` CLI:

```bash
##########

# set up labels for the contracts we want to use for easy access
ethpwn label add 0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D uniswap-router
ethpwn label add 0xC02aaA39b223FE8D0A0e5C4F27eAD9083C756Cc2 WETH-token
ethpwn label add 0x6B175474E89094C44Da98b954EedeAC495271d0F DAI-token

# fetch the verified source code for the uniswap router contract from etherscan to access its metadata and ABI
ethpwn contract fetch_verified_source 0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D
```

### Direct Interaction in `ethpwn` Scripts

Then, we can use the uniswap router contract in our scripts to interact with it.
```python
from ethpwn import *

# output to default wallet (can also specify wallet name or address here)
OUT_ADDR = get_wallet(None).address

# deadline: at most 30 minutes
deadline = int(time.time()) + 30 * 60

swap_path = [
    contract_by_label('WETH-token'), # swap first to WETH
    contract_by_label('DAI-token')   # then to DAI
]

# fetch the Contract instance for the uniswap router contract
# this automatically retrieves the ABI, source code,
# and storage layout for the contract
uniswap_router = contract_registry().get(contract_by_label('uniswap-router'))

transact(
    # this uses the automatic abi to encode the function call
    uniswap_router.w3().swapExactETHForTokens(
        100,        # amountOutMin: we want at least 100 DAI
        swap_path,  # path
        OUT_ADDR,   # to
        deadline,   # deadline
    ),
    value=1 * ETHER
)
```

### Integration in contract deployment

Instead of performing this action manually, we can also deploy a solidity contract to perform this action for us.

```python
from ethpwn import *

solidity_source = """
pragma solidity ^0.6.0;

interface IUniswapV2Router02 {
    function swapExactETHForTokens(uint amountOutMin, address[] calldata path, address to, uint deadline)
        external
        payable
        returns (uint[] memory amounts);
}

contract Swapper {
    IUniswapV2Router02 uniswap_router = IUniswapV2Router02(0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D);
    address WETH_addr = 0xC02aaA39b223FE8D0A0e5C4F27eAD9083C756Cc2;
    address DAI_addr = 0x6B175474E89094C44Da98b954EedeAC495271d0F;
    address my_addr = 0x1234567890123456789012345678901234567890;
    uint deadline = block.timestamp + 30 * 60; // 30 minutes at most

    function getDAI(uint amountOutMin) public payable {
        uniswap_router.swapExactETHForTokens{value: msg.value}(
            amountOutMin,
            [WETH_addr, DAI_addr],
            my_addr,
            deadline
        );
    }
}
"""

# compile the contract
contract_metadata = CONTRACT_METADATA.compile_solidity_source(solidity_source, 'Swapper.sol')

# deploy the contract
contract_instance = contract_metadata.deploy()

# call the contract's `getDAI` function
txid, *_ = transact(contract_instance.w3.getDAI(100), value=1 * ETHER)
print(f"Transaction ID: {txid.hex()}")
```

### Integration with `ethdbg`

Additionally, we can use `ethdbg` to debug any transactions interacting with the contract instance we just deployed.
Thanks to the contract registry, `ethdbg` will automatically have the contract metadata available during the debug sessions
and display both the source code information and the storage layout for the contract.

If we registered the UniswapV2Router02 contract with the registry as the previous example did, `ethdbg` will display
the source code and storage information for that contract as well.
```bash
# ensure the debug information for UniswapV2Router02 is available
ethpwn contract fetch_verified_source 0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D

# launch ethdbg to debug the transaction, this should display the source code and storage layout for both contracts
ethdbg --txid <id of the transaction we just attempted>
```