import json
import os
import datetime
import sys

from typing import Any, Dict

from hexbytes import HexBytes

from ..utils import get_chain_name, normalize_contract_address
from .misc import get_default_network

class Wallet:
    def __init__(self, address=None, private_key=None, name=None, description=None, network=None):
        self.address = normalize_contract_address(address)
        self.private_key = private_key
        self.name = name
        self.description = description
        self.network = network

    def __repr__(self) -> str:
        return f"Wallet(address={self.address!r}, private_key=<blinded>, name={self.name!r}, description={self.description!r}, network={self.network!r})"

    def balance(self):
        from ..global_context import context
        return context.w3.eth.get_balance(self.address)

    def to_string_repr(self) -> str:
        return f"{self.address}:{self.private_key}:{self.name}:{self.description}:{self.network}"

    def __str__(self) -> str:
        return self.__repr__()

    @staticmethod
    def from_serializable(value):
        return Wallet(**value)

    def to_serializable(self) -> Dict[str, str]:
        return {
            'address': self.address,
            'private_key': self.private_key,
            'name': self.name,
            'description': self.description,
            'network': self.network,
        }

    @staticmethod
    def from_string_repr(repr, default_name=None, default_description=None):
        if default_name is None:
            default_name = lambda address, priv_key: f"Account {address}"
        if default_description is None:
            default_description = lambda address, priv_key: f"Account {address}"
        split = repr.split(":")
        if len(split) == 2:
            split.append(default_name(*split))
        if len(split) == 3:
            split.append(default_description(*split[:2]))
        if len(split) == 4:
            split.append("sepolia")
        address, private_key, name, description, network = split
        assert network == "sepolia", "You should only ever use testnet accounts for the love of god."
        return Wallet(address, private_key, name, description)


def create_new_default_wallet(wallets_config_path, w3=None):
    from ..global_context import context
    from coolname import generate_slug

    if w3 is None:
        if context.w3 is not None and context.w3.is_connected():
            w3 = context.w3

    new_user = w3.eth.account.create()
    wallet = dict()
    wallet['address'] = new_user.address
    wallet['private_key'] = new_user.key.hex()
    wallet['network'] = get_chain_name(w3.eth.chain_id)
    wallet['name'] = generate_slug()
    wallet['description'] = "Default wallet generated by ethpwn on {}".format(datetime.datetime.now())

    with open(wallets_config_path, 'w') as f:
        json.dump([wallet], f, indent=2)

    return Wallet.from_serializable(wallet)

def load_default_wallets():
    wallets_config_path = get_default_wallet_path()
    os.makedirs(os.path.dirname(wallets_config_path), exist_ok=True)
    if not os.path.exists(wallets_config_path):
        return {}

    # Check if the file is empty
    if os.stat(wallets_config_path).st_size == 0:
        return {}

    with open(wallets_config_path, 'r') as f:
        result = json.load(f)
    result = {d['address']: Wallet.from_serializable(d) for d in result}

    if os.environ.get('ETHPWN_ACCOUNTS', None) is not None:
        for account in os.environ['ETHPWN_ACCOUNTS'].split(';'):
            result.add(Wallet.from_string_repr(account))

    if os.environ.get("ETHADDRESS", None) and os.environ.get("ETHPRIVATEKEY", None):
        addr, priv_key = os.environ['ETHADDRESS'], os.environ['ETHPRIVATEKEY']
        result.add(Wallet.from_string_repr(f"{addr}:{priv_key}"))
        result[os.environ['ETHADDRESS']] = os.environ['ETHPRIVATEKEY']

    return result

def save_default_wallets(wallets: Dict[str, Wallet]):
    wallets_config_path = get_default_wallet_path()
    os.makedirs(os.path.dirname(wallets_config_path), exist_ok=True)
    with open(wallets_config_path, 'w') as f:
        json.dump([wallet.to_serializable() for wallet in wallets.values()], f, indent=2)

def add_wallet(wallet: Wallet):
    from . import GLOBAL_CONFIG

    assert wallet.address not in GLOBAL_CONFIG['wallets'] or GLOBAL_CONFIG['wallets'].get(wallet.address, None) == wallet, f"Account for address {wallet.address} already exists, likely a mistake."
    GLOBAL_CONFIG['wallets'][wallet.address] = wallet

def get_wallet_by_address(address) -> Wallet:
    from . import GLOBAL_CONFIG
    address = normalize_contract_address(address)
    return GLOBAL_CONFIG['wallets'].get(address, None)

def get_wallet_by_name(name) -> Wallet:
    from . import GLOBAL_CONFIG
    for address, wallet in GLOBAL_CONFIG['wallets'].items():
        if wallet.name == name:
            return wallet
    return None

def all_wallets() -> Dict[str, Wallet]:
    from . import GLOBAL_CONFIG
    return GLOBAL_CONFIG['wallets'].copy()

def get_wallet(address_or_name, network=None) -> Wallet:
    from . import GLOBAL_CONFIG

    if network is None:
        network = os.environ.get('ETHPWN_NETWORK', None)
    if network is None:
        network = get_default_network()

    if os.environ.get('ETHPWN_WALLET', None) is not None:
        address_or_name = os.environ['ETHPWN_WALLET']

    if address_or_name is None:
        if len(GLOBAL_CONFIG['wallets'].values()) == 0:
            return create_new_default_wallet(get_default_wallet_path())
        else:
            wallets = list(GLOBAL_CONFIG['wallets'].values())
            suitable = [w for w in wallets if network is None or w.network == network]
            if len(suitable) == 0:
                suitable = wallets
            return suitable[0] if len(suitable) > 0 else None
    if wallet := get_wallet_by_name(address_or_name):
        return wallet
    if wallet := get_wallet_by_address(address_or_name):
        return wallet


def delete_wallet(address):
    from . import GLOBAL_CONFIG
    if address in GLOBAL_CONFIG['wallets']:
        del GLOBAL_CONFIG['wallets'][address]

from . import get_default_wallet_path