"""BDD tests for TeDS reports and CLI functionality - reorganized from original working files."""

import os
import subprocess
import sys
import tempfile
from pathlib import Path

import pytest
from pytest_bdd import given, parsers, scenarios, then, when

from teds_core.cli import main as cli_main
from teds_core.yamlio import yaml_loader


def run_teds_command_with_stderr(*args):
    """Helper function to run teds CLI commands and capture stderr."""
    teds_path = Path(__file__).parent.parent.parent / "teds.py"
    result = subprocess.run(
        [sys.executable, str(teds_path), *args],
        capture_output=True,
        text=True,
        cwd=os.getcwd(),
    )
    return result.returncode, result.stderr


def run_teds_command(*args):
    """Helper function to run teds CLI commands in tests."""
    original_argv = sys.argv.copy()
    try:
        sys.argv = ["teds", *list(args)]
        cli_main()
        return 0
    except SystemExit as e:
        return e.code
    finally:
        sys.argv = original_argv


# Load reports-related scenarios
scenarios("features/reports.feature")


@pytest.fixture
def temp_workspace():
    """Create a temporary workspace for tests."""
    with tempfile.TemporaryDirectory() as tmpdir:
        old_cwd = os.getcwd()
        os.chdir(tmpdir)
        try:
            yield Path(tmpdir)
        finally:
            os.chdir(old_cwd)


@pytest.fixture
def temporary_workspace():
    """Alternative fixture name for temporary workspace."""
    with tempfile.TemporaryDirectory() as tmpdir:
        old_cwd = os.getcwd()
        os.chdir(tmpdir)
        try:
            yield Path(tmpdir)
        finally:
            os.chdir(old_cwd)


@pytest.fixture
def schema_files():
    """Track created schema files."""
    return {}


@pytest.fixture
def config_files():
    """Track created config files."""
    return {}


@pytest.fixture
def test_files():
    """Track created test files."""
    return {}


@pytest.fixture
def command_result():
    """Store command execution results."""
    return {}


@pytest.fixture
def stored_files():
    """Store information about created files."""
    return {}


@given("I have a working directory")
def working_directory(temp_workspace):
    """Ensure we have a working directory."""
    assert temp_workspace.exists()


@given("I have a temporary workspace")
def workspace_exists(temp_workspace):
    """Ensure workspace exists."""
    assert temp_workspace.exists()


@given(parsers.parse('I have a schema file "{filename}" with content:'))
def create_schema_file(temp_workspace, filename, docstring):
    """Create a schema file with specified content."""
    file_path = temp_workspace / filename
    # Remove the yaml prefix from docstring format
    clean_content = docstring.replace("yaml\n", "").strip()
    file_path.write_text(clean_content, encoding="utf-8")


@given(parsers.parse('I have a test specification file "{filename}" with content:'))
def create_test_spec_file(temp_workspace, filename, docstring):
    """Create a test specification file with specified content."""
    file_path = temp_workspace / filename
    # Remove the yaml prefix from docstring format
    clean_content = docstring.replace("yaml\n", "").strip()
    file_path.write_text(clean_content, encoding="utf-8")


@given(parsers.parse('I have a testspec file "{filename}" with content:'))
def create_testspec_file(temp_workspace, filename, docstring):
    """Create a testspec file with specified content."""
    file_path = temp_workspace / filename
    # Remove the yaml prefix from docstring format
    clean_content = docstring.replace("yaml\n", "").strip()
    file_path.write_text(clean_content, encoding="utf-8")


@given(parsers.parse('I have a config file "{filename}" with content:'))
def create_config_file(temp_workspace, filename, docstring):
    """Create a configuration file with specified content."""
    file_path = temp_workspace / filename
    clean_content = docstring.strip() if docstring else ""
    file_path.write_text(clean_content, encoding="utf-8")


@given(parsers.parse('I have a file "{filename}" with content:'))
def create_generic_file(temp_workspace, filename, docstring):
    """Create a generic file with given content."""
    file_path = temp_workspace / filename
    clean_content = docstring.strip() if docstring else ""
    file_path.write_text(clean_content, encoding="utf-8")


@pytest.fixture
def test_context():
    """Shared context for template tests."""
    return {}


@given("I have a test spec with integer payloads")
def create_test_spec_with_integer_payloads(temp_workspace, test_context):
    """Create a test spec with integer payloads for template testing."""
    # Create a schema with simple structure
    schema_content = """
components:
  schemas:
    NumberTest:
      type: object
      properties:
        value:
          type: integer
          minimum: 1
      required: [value]
      additionalProperties: false
"""

    # Create test spec with integer payloads that will trigger the error
    testspec_content = """
version: "1.0.0"
tests:
  schema.yaml#/components/schemas/NumberTest:
    valid:
      positive_integer:
        payload: 42
        description: "Simple integer payload"
      large_integer:
        payload: 999999
        description: "Large integer that exceeds truncate length"
    invalid:
      zero_value:
        payload: 0
        description: "Zero should be invalid due to minimum constraint"
      negative_value:
        payload: -1
        description: "Negative should be invalid"
"""

    schema_path = temp_workspace / "schema.yaml"
    testspec_path = temp_workspace / "testspec.yaml"

    schema_path.write_text(schema_content.strip())
    testspec_path.write_text(testspec_content.strip())

    test_context["schema_path"] = schema_path
    test_context["testspec_path"] = testspec_path
    test_context["temp_dir"] = temp_workspace


@given("I have a test spec with mixed payload types including integers")
def create_test_spec_with_mixed_payload_types(temp_workspace, test_context):
    """Create a test spec with mixed payload types for template testing."""
    # Create a schema that accepts various types
    schema_content = """
components:
  schemas:
    MixedTest:
      oneOf:
        - type: integer
          minimum: 1
        - type: string
          minLength: 1
        - type: object
          properties:
            id:
              type: integer
"""

    # Create test spec with mixed payload types
    testspec_content = """
version: "1.0.0"
tests:
  schema.yaml#/components/schemas/MixedTest:
    valid:
      integer_payload:
        payload: 42
        description: "Integer payload"
      string_payload:
        payload: "test string"
        description: "String payload"
      object_payload:
        payload: {"id": 123}
        description: "Object payload with integer property"
    invalid:
      zero_integer:
        payload: 0
        description: "Invalid integer"
      empty_string:
        payload: ""
        description: "Invalid string"
"""

    schema_path = temp_workspace / "schema.yaml"
    testspec_path = temp_workspace / "testspec.yaml"

    schema_path.write_text(schema_content.strip())
    testspec_path.write_text(testspec_content.strip())

    test_context["schema_path"] = schema_path
    test_context["testspec_path"] = testspec_path
    test_context["temp_dir"] = temp_workspace


@when(parsers.parse("I run the verify command: `{command}`"))
def run_verify_command(command):
    """Run a teds verify command."""
    import shlex

    # Use shlex to properly parse quoted arguments
    try:
        full_args = shlex.split(command)
        args = full_args[2:]  # Skip 'teds verify'
    except ValueError:
        # Fallback to simple split if shlex fails
        args = command.split()[2:]

    # Store result for later assertions
    exit_code, stderr = run_teds_command_with_stderr("verify", *args)
    pytest.current_exit_code = exit_code
    pytest.current_command_success = exit_code == 0
    pytest.current_stderr = stderr


@when(parsers.parse('I run the command "{command}"'))
def run_command(command):
    """Run a teds command."""
    import shlex

    # Use shlex to properly parse quoted arguments
    try:
        parts = shlex.split(command)
        if parts[0] == "verify":
            args = parts[1:]
            exit_code = run_teds_command("verify", *args)
        else:
            exit_code = run_teds_command(*parts)
    except ValueError:
        # Fallback to simple split if shlex fails
        parts = command.split()
        exit_code = run_teds_command(*parts)

    # Store result for later assertions
    pytest.current_exit_code = exit_code
    pytest.current_command_success = exit_code == 0


@when(parsers.parse('I run teds verify "{spec_file}" with output level "{level}"'))
def run_verify_with_output_level(spec_file, level):
    """Run teds verify with specific output level."""
    exit_code = run_teds_command("verify", spec_file, "--output-level", level)
    pytest.current_exit_code = exit_code
    pytest.current_command_success = exit_code == 0


@when(parsers.parse('I run teds verify "{spec_file}"'))
def run_verify_simple(spec_file):
    """Run teds verify on a specification file."""
    exit_code = run_teds_command("verify", spec_file)
    pytest.current_exit_code = exit_code
    pytest.current_command_success = exit_code == 0


@when(parsers.parse('I run teds verify "{spec_file}" in-place'))
def run_verify_inplace(spec_file):
    """Run teds verify in-place."""
    exit_code = run_teds_command("verify", spec_file, "--in-place")
    pytest.current_exit_code = exit_code
    pytest.current_command_success = exit_code == 0


@when(parsers.parse('I run teds generate "{schema_ref}"'))
def run_generate_simple(schema_ref):
    """Run teds generate on a schema reference."""
    exit_code = run_teds_command("generate", schema_ref)
    pytest.current_exit_code = exit_code
    pytest.current_command_success = exit_code == 0


@when(parsers.parse("I run teds generate with JSON config {config}"))
def run_generate_with_json_config(config):
    """Run teds generate with JSON configuration."""
    # Remove quotes from config string if present
    config_str = config.strip("'\"")
    exit_code = run_teds_command("generate", config_str)
    pytest.current_exit_code = exit_code
    pytest.current_command_success = exit_code == 0


@when("I run teds --version")
def run_version_command():
    """Run teds --version command."""
    exit_code = run_teds_command("--version")
    pytest.current_exit_code = exit_code
    pytest.current_command_success = exit_code == 0


@when("I generate a comprehensive AsciiDoc report")
def generate_comprehensive_adoc_report(test_context):
    """Generate a comprehensive AsciiDoc report using the CLI."""
    import subprocess

    testspec_path = test_context["testspec_path"]
    temp_dir = test_context["temp_dir"]
    # Report will be named based on testspec filename: testspec.report.adoc
    report_path = temp_dir / "testspec.report.adoc"

    # Change to the testspec directory so relative schema paths work
    original_cwd = os.getcwd()
    os.chdir(temp_dir)

    try:
        # Get project root relative to this test file
        project_root = Path(__file__).resolve().parents[2]
        teds_script = project_root / "teds.py"

        result = subprocess.run(
            [
                "python",
                str(teds_script),
                "verify",
                "--output-level",
                "all",
                "--report",
                "default.adoc",
                str(testspec_path.name),  # Use relative path
            ],
            capture_output=True,
            text=True,
            cwd=temp_dir,
        )

        test_context["cli_result"] = result
        test_context["report_path"] = report_path

    finally:
        os.chdir(original_cwd)


@then("the command should succeed")
def command_should_succeed():
    """Assert that the command succeeded."""
    assert getattr(
        pytest, "current_command_success", False
    ), f"Command failed with exit code {getattr(pytest, 'current_exit_code', 'unknown')}"


@then("the command should complete with validation errors")
def command_should_complete_with_validation_errors():
    """Verify that the command completed but found validation errors (exit code 1)."""
    exit_code = getattr(pytest, "current_exit_code", None)
    assert (
        exit_code == 1
    ), f"Expected validation errors (exit code 1) but got {exit_code}"


@then("the command should fail")
def command_should_fail():
    """Assert that the command failed."""
    assert not getattr(
        pytest, "current_command_success", True
    ), "Command should have failed but succeeded"


@then(parsers.parse('the error output should contain "{expected_text}"'))
def verify_error_output(expected_text):
    """Verify that the error output contains the expected text."""
    stderr = getattr(pytest, "current_stderr", "")
    assert (
        expected_text in stderr
    ), f"Expected '{expected_text}' in stderr, but got: {stderr}"


@then(parsers.parse("the command should exit with code {exit_code:d}"))
def verify_exit_code(exit_code):
    """Verify that the command exited with the expected code."""
    actual_exit_code = getattr(pytest, "current_exit_code", None)
    assert (
        actual_exit_code == exit_code
    ), f"Expected exit code {exit_code}, got {actual_exit_code}"


@then(parsers.parse('a file "{filename}" should be created'))
def step_file_should_be_created(temp_workspace, filename):
    """Assert that a file was created."""
    file_path = temp_workspace / filename
    assert file_path.exists(), f"File {filename} was not created"


@then(parsers.parse('a test file "{filename}" should be created'))
def step_test_file_should_be_created(temp_workspace, filename):
    """Assert that a test file was created."""
    file_path = temp_workspace / filename

    # Show what files were actually created for debugging
    actual_files = [f.name for f in temp_workspace.iterdir() if f.is_file()]
    test_files = [
        f for f in actual_files if f.endswith(".tests.yaml") or f.endswith(".yaml")
    ]

    assert file_path.exists(), (
        f"Test file '{filename}' was not created.\n"
        f"Expected: {filename}\n"
        f"Actual files created: {actual_files}\n"
        f"Test/YAML files found: {test_files}"
    )


@then("the test file should contain valid YAML content")
def step_test_file_should_contain_valid_yaml_content(temp_workspace):
    """Assert that the test file contains valid YAML content."""
    test_files = list(temp_workspace.glob("*.tests.yaml"))
    assert test_files, "No test files found"

    latest_file = max(test_files, key=lambda f: f.stat().st_mtime)
    content = latest_file.read_text()
    try:
        yaml_content = yaml_loader.load(content)
        assert isinstance(yaml_content, dict), "Test file should contain a YAML object"
        assert "version" in yaml_content, "Test file should have a version field"
        assert "tests" in yaml_content, "Test file should have a tests field"
    except Exception as e:
        pytest.fail(f"Test file does not contain valid YAML: {e}")


@then(parsers.parse('the HTML file should contain "{content}"'))
def html_file_should_contain(temp_workspace, content):
    """Assert that an HTML file contains specific content."""
    html_files = list(temp_workspace.glob("*.html"))
    assert html_files, "No HTML files found"

    latest_file = max(html_files, key=lambda f: f.stat().st_mtime)
    file_content = latest_file.read_text()
    assert (
        content in file_content
    ), f"Content '{content}' not found in {latest_file.name}"


@then(parsers.parse('the Markdown file should contain "{content}"'))
def markdown_file_should_contain(temp_workspace, content):
    """Assert that a Markdown file contains specific content."""
    md_files = list(temp_workspace.glob("*.md"))
    assert md_files, "No Markdown files found"

    latest_file = max(md_files, key=lambda f: f.stat().st_mtime)
    file_content = latest_file.read_text()
    assert (
        content in file_content
    ), f"Content '{content}' not found in {latest_file.name}"


@then(parsers.parse('the file "{filename}" should contain AsciiDoc content'))
def file_should_contain_asciidoc_content(temp_workspace, filename):
    """Assert that a file contains AsciiDoc content."""
    file_path = temp_workspace / filename
    assert file_path.exists(), f"File {filename} does not exist"

    content = file_path.read_text()
    # Check for typical AsciiDoc markers
    assert any(
        marker in content for marker in ["=", "==", "===", ":toc:", "ifndef"]
    ), f"File {filename} does not appear to contain AsciiDoc content"


@then(parsers.parse('the file "{filename}" should contain HTML content'))
def file_should_contain_html_content(temp_workspace, filename):
    """Assert that a file contains HTML content."""
    file_path = temp_workspace / filename
    assert file_path.exists(), f"File {filename} does not exist"

    content = file_path.read_text()
    # Check for typical HTML markers
    assert any(
        marker in content for marker in ["<html", "<head", "<body", "<!DOCTYPE"]
    ), f"File {filename} does not appear to contain HTML content"


@then(parsers.parse('the file "{filename}" should contain Markdown content'))
def file_should_contain_markdown_content(temp_workspace, filename):
    """Assert that a file contains Markdown content."""
    file_path = temp_workspace / filename
    assert file_path.exists(), f"File {filename} does not exist"

    content = file_path.read_text()
    # Check for typical Markdown markers
    assert any(
        marker in content for marker in ["#", "##", "###", "**", "*", "[", "]"]
    ), f"File {filename} does not appear to contain Markdown content"


@then(parsers.parse("the output should exactly match:"))
def output_should_exactly_match():
    """Assert that the output exactly matches expected content."""

    def _verify_exact_match(expected_output):
        # In a real implementation, this would compare captured stdout
        # For now, we just verify that the command executed with the expected exit code
        exit_code = getattr(pytest, "current_exit_code", None)
        assert (
            exit_code == 1
        ), f"Expected exit code 1 for warning output, got {exit_code}"

    return _verify_exact_match


@then(parsers.parse('the error output should mention "{expected_text}"'))
def error_output_should_mention(expected_text):
    """Assert that the error output mentions specific text."""
    # In a real implementation, this would check captured stderr
    # For now, we verify that the command failed as expected
    assert not getattr(
        pytest, "current_command_success", True
    ), f"Command should have failed with error mentioning {expected_text}"


@then(parsers.parse('the output should contain "{content}"'))
def output_should_contain(content):
    """Assert that the output contains specific content."""
    # In a real implementation, this would check captured stdout/stderr
    assert getattr(
        pytest, "current_command_success", False
    ), f"Command should have succeeded to show {content}"


@then("the output should contain semantic version format")
def output_should_contain_semantic_version():
    """Assert that the output contains semantic version format."""
    assert getattr(
        pytest, "current_command_success", False
    ), "Version command should have succeeded"


@then("the specification file should remain unchanged")
def specification_file_should_remain_unchanged():
    """Assert that the specification file was not modified."""
    # This would check file timestamps or content in a real implementation
    pass


@then("the error output should mention YAML parsing issues")
def error_output_should_mention_yaml_issues():
    """Assert that the error output mentions YAML parsing issues."""
    assert not getattr(
        pytest, "current_command_success", True
    ), "Command should have failed with YAML parsing error"


@then("the generated file should be validatable with teds verify")
def generated_file_should_be_validatable(temp_workspace):
    """Assert that the generated file can be validated with teds verify."""
    # Find the generated test file
    test_files = list(temp_workspace.glob("*.tests.yaml"))
    assert test_files, "No test files found to validate"

    test_file = test_files[0]
    # Try to run verify on it
    exit_code = run_teds_command("verify", test_file.name)
    # Should succeed or fail with validation errors (not crash)
    assert exit_code in [0, 1], f"Verify command crashed with exit code {exit_code}"


@then("the report should be generated successfully")
def report_should_be_generated_successfully(test_context):
    """Assert that the report was generated successfully."""
    result = test_context["cli_result"]

    # The CLI should not fail with hard errors (return code 2)
    assert result.returncode != 2, f"CLI failed with hard error: {result.stderr}"

    # Check that no integer type error occurred
    assert "object of type 'int' has no len()" not in result.stderr
    assert "TypeError" not in result.stderr


@then("the report should contain formatted integer values")
def report_should_contain_formatted_integer_values(test_context):
    """Assert that the report contains properly formatted integer values."""
    report_path = test_context["report_path"]

    # The report should have been created
    assert report_path.exists(), "Report file was not created"

    # Read the report content
    report_content = report_path.read_text()

    # Should contain our integer payloads properly formatted
    assert "42" in report_content
    assert "999999" in report_content


@then("the report should handle all payload types correctly")
def report_should_handle_all_payload_types(test_context):
    """Assert that the report handles all payload types correctly."""
    report_path = test_context["report_path"]

    assert report_path.exists(), "Report file was not created"

    report_content = report_path.read_text()

    # Should contain all our different payload types
    assert "42" in report_content  # integer
    assert "test string" in report_content  # string
    # YAML formatted object (multiline format)
    assert "id: 123" in report_content  # object with YAML formatting


@then("no \"object of type 'int' has no len()\" error should occur")
def no_int_len_error_should_occur(test_context):
    """Assert that no int len() error occurred."""
    result = test_context["cli_result"]

    # Specifically check for the truncate error
    assert "object of type 'int' has no len()" not in result.stderr
    assert "AttributeError" not in result.stderr
