"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeUniqueId = void 0;
// tslint:disable-next-line:no-var-requires
const crypto = require("crypto");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = "Resource";
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = "Default";
const PATH_SEP = "/";
const UNIQUE_SEP = "_";
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
const MAX_ID_LEN = 255;
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components, allowSepChars) {
    components = components.filter((x) => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error("Unable to calculate a unique id for an empty set of components");
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        // we filter out non-alpha characters but that is actually a bad idea
        // because it could create conflicts ("A-B" and "AB" will render the same
        // logical ID). sadly, changing it in the 1.x version line is impossible
        // because it will be a breaking change. we should consider for v2.0.
        // https://github.com/aws/aws-cdk/issues/6421
        const candidate = removeDisallowedCharacters(components[0], allowSepChars);
        // if our candidate is short enough, use it as is. otherwise, fall back to
        // the normal mode.
        if (candidate.length <= MAX_ID_LEN) {
            return candidate;
        }
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter((x) => x !== HIDDEN_FROM_HUMAN_ID)
        .map((s) => removeDisallowedCharacters(s, allowSepChars))
        .join(UNIQUE_SEP)
        .slice(0, MAX_HUMAN_LEN);
    return human + UNIQUE_SEP + hash;
}
exports.makeUniqueId = makeUniqueId;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto
        .createHash("md5")
        .update(path.join(PATH_SEP))
        .digest("hex");
    return md5.slice(0, HASH_LEN).toUpperCase();
}
function removeDisallowedCharacters(s, allowSepChars) {
    if (allowSepChars) {
        return removeNonAlphanumericSep(s);
    }
    else {
        return removeNonAlphanumeric(s);
    }
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, "");
}
function removeNonAlphanumericSep(s) {
    return s.replace(/[^A-Za-z0-9_-]/g, "");
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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