"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCustomSynthesis = exports.invokeAspects = exports.StackSynthesizer = void 0;
const path = require("path");
const fs = require("fs");
const stringify = require("json-stable-stringify");
const annotations_1 = require("../annotations");
const constructs_1 = require("constructs");
const aspect_1 = require("../aspect");
class StackSynthesizer {
    /**
     * @param stack the stack to synthesize
     * @param continueOnErrorAnnotations if set to true, `synthesize()` will not throw an error
     * if it encounters an error Annotation. However, it will still add all errors to the manifest
     * file. The cdktf-cli uses this flag to print the errors itself.
     */
    constructor(stack, continueOnErrorAnnotations = false) {
        this.stack = stack;
        this.continueOnErrorAnnotations = continueOnErrorAnnotations;
    }
    synthesize(session) {
        invokeAspects(this.stack);
        if (!session.skipValidation) {
            const errors = this.stack.node
                .findAll()
                .map((node) => node.node
                .validate()
                .map((error) => ({ message: error, source: node })))
                .reduce((prev, curr) => [...prev, ...curr], []);
            if (errors.length > 0) {
                const errorList = errors
                    .map((e) => `[${e.source.node.path}] ${e.message}`)
                    .join("\n  ");
                throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
            }
        }
        const manifest = session.manifest;
        const stackManifest = manifest.forStack(this.stack);
        const workingDirectory = path.join(session.outdir, stackManifest.workingDirectory);
        if (!fs.existsSync(workingDirectory))
            fs.mkdirSync(workingDirectory);
        // call custom synthesis on child nodes (leafs first)
        this.stack.node
            .findAll(constructs_1.ConstructOrder.POSTORDER)
            .forEach((node) => { var _a; return (_a = getCustomSynthesis(node)) === null || _a === void 0 ? void 0 : _a.onSynthesize(session); });
        // collect Annotations into Manifest
        const annotations = this.stack.node
            .findAll()
            .map((node) => ({
            node,
            metadatas: node.node.metadata.filter(isAnnotationMetadata),
        }))
            .map(({ node, metadatas }) => metadatas.map((metadata) => ({
            constructPath: node.node.path,
            level: metadata.type,
            message: metadata.data,
            stacktrace: metadata.trace,
        })))
            .reduce((list, metadatas) => [...list, ...metadatas], []); // Array.flat()
        // it is readonly but this is the place where we are allowed to write to it
        stackManifest.annotations = annotations;
        // abort if one or more error annotations have been encountered
        if (!this.continueOnErrorAnnotations &&
            annotations.some(isErrorAnnotation)) {
            throw new Error(`Encountered Annotations with level "ERROR":\n${annotations
                .filter(isErrorAnnotation)
                .map((a) => `[${a.constructPath}] ${a.message}`)
                .join("\n")}`);
        }
        const tfConfig = this.stack.toTerraform();
        fs.writeFileSync(path.join(session.outdir, stackManifest.synthesizedStackPath), stringify(tfConfig, { space: 2 }));
    }
}
exports.StackSynthesizer = StackSynthesizer;
/**
 * Invoke aspects on the given construct tree.
 *
 * originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/@aws-cdk/core/lib/private/synthesis.ts#L99-L137
 */
function invokeAspects(root) {
    const invokedByPath = {};
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...(inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : []), ...aspects.all];
        const nodeAspectsCount = aspects.all.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.all.length) {
                annotations_1.Annotations.of(construct).addWarning("We detected an Aspect was added via another Aspect, and will not be applied");
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            recurse(child, allAspectsHere);
        }
    }
}
exports.invokeAspects = invokeAspects;
const annotationMetadataEntryTypes = [
    annotations_1.AnnotationMetadataEntryType.INFO,
    annotations_1.AnnotationMetadataEntryType.WARN,
    annotations_1.AnnotationMetadataEntryType.ERROR,
];
function isAnnotationMetadata(metadata) {
    return annotationMetadataEntryTypes.includes(metadata.type);
}
function isErrorAnnotation(annotation) {
    return annotation.level === annotations_1.AnnotationMetadataEntryType.ERROR;
}
// originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/%40aws-cdk/core/lib/private/synthesis.ts#L52
const CUSTOM_SYNTHESIS_SYM = Symbol.for("cdktf/customSynthesis");
function addCustomSynthesis(construct, synthesis) {
    Object.defineProperty(construct, CUSTOM_SYNTHESIS_SYM, {
        value: synthesis,
        enumerable: false,
    });
}
exports.addCustomSynthesis = addCustomSynthesis;
function getCustomSynthesis(construct) {
    return construct[CUSTOM_SYNTHESIS_SYM];
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3ludGhlc2l6ZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzeW50aGVzaXplci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IseUJBQXlCO0FBQ3pCLG1EQUFvRDtBQUdwRCxnREFBMEU7QUFDMUUsMkNBQXVFO0FBQ3ZFLHNDQUE2QztBQUc3QyxNQUFhLGdCQUFnQjtJQUMzQjs7Ozs7T0FLRztJQUNILFlBQ1ksS0FBcUIsRUFDdkIsNkJBQTZCLEtBQUs7UUFEaEMsVUFBSyxHQUFMLEtBQUssQ0FBZ0I7UUFDdkIsK0JBQTBCLEdBQTFCLDBCQUEwQixDQUFRO0lBQ3pDLENBQUM7SUFFSixVQUFVLENBQUMsT0FBMEI7UUFDbkMsYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUxQixJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRTtZQUMzQixNQUFNLE1BQU0sR0FBOEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJO2lCQUN0RSxPQUFPLEVBQUU7aUJBQ1QsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FDWixJQUFJLENBQUMsSUFBSTtpQkFDTixRQUFRLEVBQUU7aUJBQ1YsR0FBRyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUN0RDtpQkFDQSxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDLEdBQUcsSUFBSSxFQUFFLEdBQUcsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDbEQsSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtnQkFDckIsTUFBTSxTQUFTLEdBQUcsTUFBTTtxQkFDckIsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7cUJBQ2xELElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztnQkFDaEIsTUFBTSxJQUFJLEtBQUssQ0FDYixtREFBbUQsU0FBUyxFQUFFLENBQy9ELENBQUM7YUFDSDtTQUNGO1FBRUQsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQztRQUNsQyxNQUFNLGFBQWEsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUVwRCxNQUFNLGdCQUFnQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQ2hDLE9BQU8sQ0FBQyxNQUFNLEVBQ2QsYUFBYSxDQUFDLGdCQUFnQixDQUMvQixDQUFDO1FBQ0YsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUM7WUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFFckUscURBQXFEO1FBQ3JELElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSTthQUNaLE9BQU8sQ0FBQywyQkFBYyxDQUFDLFNBQVMsQ0FBQzthQUNqQyxPQUFPLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSx3QkFBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsMENBQUUsWUFBWSxDQUFDLE9BQU8sSUFBQyxDQUFDLENBQUM7UUFFdEUsb0NBQW9DO1FBQ3BDLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSTthQUNoQyxPQUFPLEVBQUU7YUFDVCxHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDZCxJQUFJO1lBQ0osU0FBUyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQztTQUMzRCxDQUFDLENBQUM7YUFDRixHQUFHLENBQW9CLENBQUMsRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLEVBQUUsRUFBRSxDQUM5QyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQzNCLGFBQWEsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUk7WUFDN0IsS0FBSyxFQUFFLFFBQVEsQ0FBQyxJQUFtQztZQUNuRCxPQUFPLEVBQUUsUUFBUSxDQUFDLElBQUk7WUFDdEIsVUFBVSxFQUFFLFFBQVEsQ0FBQyxLQUFLO1NBQzNCLENBQUMsQ0FBQyxDQUNKO2FBQ0EsTUFBTSxDQUFDLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxHQUFHLElBQUksRUFBRSxHQUFHLFNBQVMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsZUFBZTtRQUU1RSwyRUFBMkU7UUFDMUUsYUFBYSxDQUFDLFdBQW1CLEdBQUcsV0FBVyxDQUFDO1FBRWpELCtEQUErRDtRQUMvRCxJQUNFLENBQUMsSUFBSSxDQUFDLDBCQUEwQjtZQUNoQyxXQUFXLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEVBQ25DO1lBQ0EsTUFBTSxJQUFJLEtBQUssQ0FDYixnREFBZ0QsV0FBVztpQkFDeEQsTUFBTSxDQUFDLGlCQUFpQixDQUFDO2lCQUN6QixHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLGFBQWEsS0FBSyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7aUJBQy9DLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUNoQixDQUFDO1NBQ0g7UUFFRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBRTFDLEVBQUUsQ0FBQyxhQUFhLENBQ2QsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLGFBQWEsQ0FBQyxvQkFBb0IsQ0FBQyxFQUM3RCxTQUFTLENBQUMsUUFBUSxFQUFFLEVBQUUsS0FBSyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQ2xDLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUF4RkQsNENBd0ZDO0FBRUQ7Ozs7R0FJRztBQUNILFNBQWdCLGFBQWEsQ0FBQyxJQUFnQjtJQUM1QyxNQUFNLGFBQWEsR0FBc0MsRUFBRSxDQUFDO0lBRTVELElBQUksbUJBQW1CLEdBQUcsS0FBSyxDQUFDO0lBQ2hDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFbEIsU0FBUyxPQUFPLENBQUMsU0FBcUIsRUFBRSxnQkFBMkI7UUFDakUsTUFBTSxJQUFJLEdBQUcsU0FBUyxDQUFDLElBQUksQ0FBQztRQUM1QixNQUFNLE9BQU8sR0FBRyxnQkFBTyxDQUFDLEVBQUUsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN0QyxNQUFNLGNBQWMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsYUFBaEIsZ0JBQWdCLGNBQWhCLGdCQUFnQixHQUFJLEVBQUUsQ0FBQyxFQUFFLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3JFLE1BQU0sZ0JBQWdCLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUM7UUFDNUMsS0FBSyxNQUFNLE1BQU0sSUFBSSxjQUFjLEVBQUU7WUFDbkMsSUFBSSxPQUFPLEdBQUcsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN2QyxJQUFJLENBQUMsT0FBTyxFQUFFO2dCQUNaLE9BQU8sR0FBRyxhQUFhLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQzthQUN6QztZQUVELElBQUksT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDNUIsU0FBUzthQUNWO1lBRUQsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUV4QiwwR0FBMEc7WUFDMUcsbUdBQW1HO1lBQ25HLElBQUksQ0FBQyxtQkFBbUIsSUFBSSxnQkFBZ0IsS0FBSyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRTtnQkFDbkUseUJBQVcsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUMsVUFBVSxDQUNsQyw2RUFBNkUsQ0FDOUUsQ0FBQztnQkFDRixtQkFBbUIsR0FBRyxJQUFJLENBQUM7YUFDNUI7WUFFRCxnQ0FBZ0M7WUFDaEMsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUN0QjtRQUVELEtBQUssTUFBTSxLQUFLLElBQUksU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUU7WUFDM0MsT0FBTyxDQUFDLEtBQUssRUFBRSxjQUFjLENBQUMsQ0FBQztTQUNoQztJQUNILENBQUM7QUFDSCxDQUFDO0FBeENELHNDQXdDQztBQUVELE1BQU0sNEJBQTRCLEdBQUc7SUFDbkMseUNBQTJCLENBQUMsSUFBSTtJQUNoQyx5Q0FBMkIsQ0FBQyxJQUFJO0lBQ2hDLHlDQUEyQixDQUFDLEtBQUs7Q0FDdEIsQ0FBQztBQUVkLFNBQVMsb0JBQW9CLENBQUMsUUFBdUI7SUFDbkQsT0FBTyw0QkFBNEIsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO0FBQzlELENBQUM7QUFFRCxTQUFTLGlCQUFpQixDQUFDLFVBQTJCO0lBQ3BELE9BQU8sVUFBVSxDQUFDLEtBQUssS0FBSyx5Q0FBMkIsQ0FBQyxLQUFLLENBQUM7QUFDaEUsQ0FBQztBQUVELHFKQUFxSjtBQUNySixNQUFNLG9CQUFvQixHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztBQWVqRSxTQUFnQixrQkFBa0IsQ0FDaEMsU0FBcUIsRUFDckIsU0FBMkI7SUFFM0IsTUFBTSxDQUFDLGNBQWMsQ0FBQyxTQUFTLEVBQUUsb0JBQW9CLEVBQUU7UUFDckQsS0FBSyxFQUFFLFNBQVM7UUFDaEIsVUFBVSxFQUFFLEtBQUs7S0FDbEIsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQVJELGdEQVFDO0FBRUQsU0FBUyxrQkFBa0IsQ0FDekIsU0FBcUI7SUFFckIsT0FBUSxTQUFpQixDQUFDLG9CQUFvQixDQUFDLENBQUM7QUFDbEQsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCAqIGFzIGZzIGZyb20gXCJmc1wiO1xuaW1wb3J0IHN0cmluZ2lmeSA9IHJlcXVpcmUoXCJqc29uLXN0YWJsZS1zdHJpbmdpZnlcIik7XG5pbXBvcnQgeyBUZXJyYWZvcm1TdGFjayB9IGZyb20gXCIuLi90ZXJyYWZvcm0tc3RhY2tcIjtcbmltcG9ydCB7IElTdGFja1N5bnRoZXNpemVyLCBJU3ludGhlc2lzU2Vzc2lvbiB9IGZyb20gXCIuL3R5cGVzXCI7XG5pbXBvcnQgeyBBbm5vdGF0aW9uTWV0YWRhdGFFbnRyeVR5cGUsIEFubm90YXRpb25zIH0gZnJvbSBcIi4uL2Fubm90YXRpb25zXCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3RPcmRlciwgSUNvbnN0cnVjdCwgTWV0YWRhdGFFbnRyeSB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQgeyBBc3BlY3RzLCBJQXNwZWN0IH0gZnJvbSBcIi4uL2FzcGVjdFwiO1xuaW1wb3J0IHsgU3RhY2tBbm5vdGF0aW9uIH0gZnJvbSBcIi4uL21hbmlmZXN0XCI7XG5cbmV4cG9ydCBjbGFzcyBTdGFja1N5bnRoZXNpemVyIGltcGxlbWVudHMgSVN0YWNrU3ludGhlc2l6ZXIge1xuICAvKipcbiAgICogQHBhcmFtIHN0YWNrIHRoZSBzdGFjayB0byBzeW50aGVzaXplXG4gICAqIEBwYXJhbSBjb250aW51ZU9uRXJyb3JBbm5vdGF0aW9ucyBpZiBzZXQgdG8gdHJ1ZSwgYHN5bnRoZXNpemUoKWAgd2lsbCBub3QgdGhyb3cgYW4gZXJyb3JcbiAgICogaWYgaXQgZW5jb3VudGVycyBhbiBlcnJvciBBbm5vdGF0aW9uLiBIb3dldmVyLCBpdCB3aWxsIHN0aWxsIGFkZCBhbGwgZXJyb3JzIHRvIHRoZSBtYW5pZmVzdFxuICAgKiBmaWxlLiBUaGUgY2RrdGYtY2xpIHVzZXMgdGhpcyBmbGFnIHRvIHByaW50IHRoZSBlcnJvcnMgaXRzZWxmLlxuICAgKi9cbiAgY29uc3RydWN0b3IoXG4gICAgcHJvdGVjdGVkIHN0YWNrOiBUZXJyYWZvcm1TdGFjayxcbiAgICBwcml2YXRlIGNvbnRpbnVlT25FcnJvckFubm90YXRpb25zID0gZmFsc2VcbiAgKSB7fVxuXG4gIHN5bnRoZXNpemUoc2Vzc2lvbjogSVN5bnRoZXNpc1Nlc3Npb24pIHtcbiAgICBpbnZva2VBc3BlY3RzKHRoaXMuc3RhY2spO1xuXG4gICAgaWYgKCFzZXNzaW9uLnNraXBWYWxpZGF0aW9uKSB7XG4gICAgICBjb25zdCBlcnJvcnM6IHsgbWVzc2FnZTogc3RyaW5nOyBzb3VyY2U6IElDb25zdHJ1Y3QgfVtdID0gdGhpcy5zdGFjay5ub2RlXG4gICAgICAgIC5maW5kQWxsKClcbiAgICAgICAgLm1hcCgobm9kZSkgPT5cbiAgICAgICAgICBub2RlLm5vZGVcbiAgICAgICAgICAgIC52YWxpZGF0ZSgpXG4gICAgICAgICAgICAubWFwKChlcnJvcikgPT4gKHsgbWVzc2FnZTogZXJyb3IsIHNvdXJjZTogbm9kZSB9KSlcbiAgICAgICAgKVxuICAgICAgICAucmVkdWNlKChwcmV2LCBjdXJyKSA9PiBbLi4ucHJldiwgLi4uY3Vycl0sIFtdKTtcbiAgICAgIGlmIChlcnJvcnMubGVuZ3RoID4gMCkge1xuICAgICAgICBjb25zdCBlcnJvckxpc3QgPSBlcnJvcnNcbiAgICAgICAgICAubWFwKChlKSA9PiBgWyR7ZS5zb3VyY2Uubm9kZS5wYXRofV0gJHtlLm1lc3NhZ2V9YClcbiAgICAgICAgICAuam9pbihcIlxcbiAgXCIpO1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgYFZhbGlkYXRpb24gZmFpbGVkIHdpdGggdGhlIGZvbGxvd2luZyBlcnJvcnM6XFxuICAke2Vycm9yTGlzdH1gXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgbWFuaWZlc3QgPSBzZXNzaW9uLm1hbmlmZXN0O1xuICAgIGNvbnN0IHN0YWNrTWFuaWZlc3QgPSBtYW5pZmVzdC5mb3JTdGFjayh0aGlzLnN0YWNrKTtcblxuICAgIGNvbnN0IHdvcmtpbmdEaXJlY3RvcnkgPSBwYXRoLmpvaW4oXG4gICAgICBzZXNzaW9uLm91dGRpcixcbiAgICAgIHN0YWNrTWFuaWZlc3Qud29ya2luZ0RpcmVjdG9yeVxuICAgICk7XG4gICAgaWYgKCFmcy5leGlzdHNTeW5jKHdvcmtpbmdEaXJlY3RvcnkpKSBmcy5ta2RpclN5bmMod29ya2luZ0RpcmVjdG9yeSk7XG5cbiAgICAvLyBjYWxsIGN1c3RvbSBzeW50aGVzaXMgb24gY2hpbGQgbm9kZXMgKGxlYWZzIGZpcnN0KVxuICAgIHRoaXMuc3RhY2subm9kZVxuICAgICAgLmZpbmRBbGwoQ29uc3RydWN0T3JkZXIuUE9TVE9SREVSKVxuICAgICAgLmZvckVhY2goKG5vZGUpID0+IGdldEN1c3RvbVN5bnRoZXNpcyhub2RlKT8ub25TeW50aGVzaXplKHNlc3Npb24pKTtcblxuICAgIC8vIGNvbGxlY3QgQW5ub3RhdGlvbnMgaW50byBNYW5pZmVzdFxuICAgIGNvbnN0IGFubm90YXRpb25zID0gdGhpcy5zdGFjay5ub2RlXG4gICAgICAuZmluZEFsbCgpXG4gICAgICAubWFwKChub2RlKSA9PiAoe1xuICAgICAgICBub2RlLFxuICAgICAgICBtZXRhZGF0YXM6IG5vZGUubm9kZS5tZXRhZGF0YS5maWx0ZXIoaXNBbm5vdGF0aW9uTWV0YWRhdGEpLFxuICAgICAgfSkpXG4gICAgICAubWFwPFN0YWNrQW5ub3RhdGlvbltdPigoeyBub2RlLCBtZXRhZGF0YXMgfSkgPT5cbiAgICAgICAgbWV0YWRhdGFzLm1hcCgobWV0YWRhdGEpID0+ICh7XG4gICAgICAgICAgY29uc3RydWN0UGF0aDogbm9kZS5ub2RlLnBhdGgsXG4gICAgICAgICAgbGV2ZWw6IG1ldGFkYXRhLnR5cGUgYXMgQW5ub3RhdGlvbk1ldGFkYXRhRW50cnlUeXBlLFxuICAgICAgICAgIG1lc3NhZ2U6IG1ldGFkYXRhLmRhdGEsXG4gICAgICAgICAgc3RhY2t0cmFjZTogbWV0YWRhdGEudHJhY2UsXG4gICAgICAgIH0pKVxuICAgICAgKVxuICAgICAgLnJlZHVjZSgobGlzdCwgbWV0YWRhdGFzKSA9PiBbLi4ubGlzdCwgLi4ubWV0YWRhdGFzXSwgW10pOyAvLyBBcnJheS5mbGF0KClcblxuICAgIC8vIGl0IGlzIHJlYWRvbmx5IGJ1dCB0aGlzIGlzIHRoZSBwbGFjZSB3aGVyZSB3ZSBhcmUgYWxsb3dlZCB0byB3cml0ZSB0byBpdFxuICAgIChzdGFja01hbmlmZXN0LmFubm90YXRpb25zIGFzIGFueSkgPSBhbm5vdGF0aW9ucztcblxuICAgIC8vIGFib3J0IGlmIG9uZSBvciBtb3JlIGVycm9yIGFubm90YXRpb25zIGhhdmUgYmVlbiBlbmNvdW50ZXJlZFxuICAgIGlmIChcbiAgICAgICF0aGlzLmNvbnRpbnVlT25FcnJvckFubm90YXRpb25zICYmXG4gICAgICBhbm5vdGF0aW9ucy5zb21lKGlzRXJyb3JBbm5vdGF0aW9uKVxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgRW5jb3VudGVyZWQgQW5ub3RhdGlvbnMgd2l0aCBsZXZlbCBcIkVSUk9SXCI6XFxuJHthbm5vdGF0aW9uc1xuICAgICAgICAgIC5maWx0ZXIoaXNFcnJvckFubm90YXRpb24pXG4gICAgICAgICAgLm1hcCgoYSkgPT4gYFske2EuY29uc3RydWN0UGF0aH1dICR7YS5tZXNzYWdlfWApXG4gICAgICAgICAgLmpvaW4oXCJcXG5cIil9YFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCB0ZkNvbmZpZyA9IHRoaXMuc3RhY2sudG9UZXJyYWZvcm0oKTtcblxuICAgIGZzLndyaXRlRmlsZVN5bmMoXG4gICAgICBwYXRoLmpvaW4oc2Vzc2lvbi5vdXRkaXIsIHN0YWNrTWFuaWZlc3Quc3ludGhlc2l6ZWRTdGFja1BhdGgpLFxuICAgICAgc3RyaW5naWZ5KHRmQ29uZmlnLCB7IHNwYWNlOiAyIH0pXG4gICAgKTtcbiAgfVxufVxuXG4vKipcbiAqIEludm9rZSBhc3BlY3RzIG9uIHRoZSBnaXZlbiBjb25zdHJ1Y3QgdHJlZS5cbiAqXG4gKiBvcmlnaW5hbGx5IGZyb20gaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2Jsb2IvZGNhZTNlZWFkMGRiZjlhY2IxZWQ4MGJhOTViYjEwNGM2NGNiMWJkNy9wYWNrYWdlcy9AYXdzLWNkay9jb3JlL2xpYi9wcml2YXRlL3N5bnRoZXNpcy50cyNMOTktTDEzN1xuICovXG5leHBvcnQgZnVuY3Rpb24gaW52b2tlQXNwZWN0cyhyb290OiBJQ29uc3RydWN0KSB7XG4gIGNvbnN0IGludm9rZWRCeVBhdGg6IHsgW25vZGVQYXRoOiBzdHJpbmddOiBJQXNwZWN0W10gfSA9IHt9O1xuXG4gIGxldCBuZXN0ZWRBc3BlY3RXYXJuaW5nID0gZmFsc2U7XG4gIHJlY3Vyc2Uocm9vdCwgW10pO1xuXG4gIGZ1bmN0aW9uIHJlY3Vyc2UoY29uc3RydWN0OiBJQ29uc3RydWN0LCBpbmhlcml0ZWRBc3BlY3RzOiBJQXNwZWN0W10pIHtcbiAgICBjb25zdCBub2RlID0gY29uc3RydWN0Lm5vZGU7XG4gICAgY29uc3QgYXNwZWN0cyA9IEFzcGVjdHMub2YoY29uc3RydWN0KTtcbiAgICBjb25zdCBhbGxBc3BlY3RzSGVyZSA9IFsuLi4oaW5oZXJpdGVkQXNwZWN0cyA/PyBbXSksIC4uLmFzcGVjdHMuYWxsXTtcbiAgICBjb25zdCBub2RlQXNwZWN0c0NvdW50ID0gYXNwZWN0cy5hbGwubGVuZ3RoO1xuICAgIGZvciAoY29uc3QgYXNwZWN0IG9mIGFsbEFzcGVjdHNIZXJlKSB7XG4gICAgICBsZXQgaW52b2tlZCA9IGludm9rZWRCeVBhdGhbbm9kZS5wYXRoXTtcbiAgICAgIGlmICghaW52b2tlZCkge1xuICAgICAgICBpbnZva2VkID0gaW52b2tlZEJ5UGF0aFtub2RlLnBhdGhdID0gW107XG4gICAgICB9XG5cbiAgICAgIGlmIChpbnZva2VkLmluY2x1ZGVzKGFzcGVjdCkpIHtcbiAgICAgICAgY29udGludWU7XG4gICAgICB9XG5cbiAgICAgIGFzcGVjdC52aXNpdChjb25zdHJ1Y3QpO1xuXG4gICAgICAvLyBpZiBhbiBhc3BlY3Qgd2FzIGFkZGVkIHRvIHRoZSBub2RlIHdoaWxlIGludm9raW5nIGFub3RoZXIgYXNwZWN0IGl0IHdpbGwgbm90IGJlIGludm9rZWQsIGVtaXQgYSB3YXJuaW5nXG4gICAgICAvLyB0aGUgYG5lc3RlZEFzcGVjdFdhcm5pbmdgIGZsYWcgaXMgdXNlZCB0byBwcmV2ZW50IHRoZSB3YXJuaW5nIGZyb20gYmVpbmcgZW1pdHRlZCBmb3IgZXZlcnkgY2hpbGRcbiAgICAgIGlmICghbmVzdGVkQXNwZWN0V2FybmluZyAmJiBub2RlQXNwZWN0c0NvdW50ICE9PSBhc3BlY3RzLmFsbC5sZW5ndGgpIHtcbiAgICAgICAgQW5ub3RhdGlvbnMub2YoY29uc3RydWN0KS5hZGRXYXJuaW5nKFxuICAgICAgICAgIFwiV2UgZGV0ZWN0ZWQgYW4gQXNwZWN0IHdhcyBhZGRlZCB2aWEgYW5vdGhlciBBc3BlY3QsIGFuZCB3aWxsIG5vdCBiZSBhcHBsaWVkXCJcbiAgICAgICAgKTtcbiAgICAgICAgbmVzdGVkQXNwZWN0V2FybmluZyA9IHRydWU7XG4gICAgICB9XG5cbiAgICAgIC8vIG1hcmsgYXMgaW52b2tlZCBmb3IgdGhpcyBub2RlXG4gICAgICBpbnZva2VkLnB1c2goYXNwZWN0KTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IGNoaWxkIG9mIGNvbnN0cnVjdC5ub2RlLmNoaWxkcmVuKSB7XG4gICAgICByZWN1cnNlKGNoaWxkLCBhbGxBc3BlY3RzSGVyZSk7XG4gICAgfVxuICB9XG59XG5cbmNvbnN0IGFubm90YXRpb25NZXRhZGF0YUVudHJ5VHlwZXMgPSBbXG4gIEFubm90YXRpb25NZXRhZGF0YUVudHJ5VHlwZS5JTkZPLFxuICBBbm5vdGF0aW9uTWV0YWRhdGFFbnRyeVR5cGUuV0FSTixcbiAgQW5ub3RhdGlvbk1ldGFkYXRhRW50cnlUeXBlLkVSUk9SLFxuXSBhcyBzdHJpbmdbXTtcblxuZnVuY3Rpb24gaXNBbm5vdGF0aW9uTWV0YWRhdGEobWV0YWRhdGE6IE1ldGFkYXRhRW50cnkpOiBib29sZWFuIHtcbiAgcmV0dXJuIGFubm90YXRpb25NZXRhZGF0YUVudHJ5VHlwZXMuaW5jbHVkZXMobWV0YWRhdGEudHlwZSk7XG59XG5cbmZ1bmN0aW9uIGlzRXJyb3JBbm5vdGF0aW9uKGFubm90YXRpb246IFN0YWNrQW5ub3RhdGlvbik6IGJvb2xlYW4ge1xuICByZXR1cm4gYW5ub3RhdGlvbi5sZXZlbCA9PT0gQW5ub3RhdGlvbk1ldGFkYXRhRW50cnlUeXBlLkVSUk9SO1xufVxuXG4vLyBvcmlnaW5hbGx5IGZyb20gaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2Jsb2IvZGNhZTNlZWFkMGRiZjlhY2IxZWQ4MGJhOTViYjEwNGM2NGNiMWJkNy9wYWNrYWdlcy8lNDBhd3MtY2RrL2NvcmUvbGliL3ByaXZhdGUvc3ludGhlc2lzLnRzI0w1MlxuY29uc3QgQ1VTVE9NX1NZTlRIRVNJU19TWU0gPSBTeW1ib2wuZm9yKFwiY2RrdGYvY3VzdG9tU3ludGhlc2lzXCIpO1xuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgY29uc3RydWN0cyB0aGF0IHdhbnQgdG8gZG8gc29tZXRoaW5nIGN1c3RvbSBkdXJpbmcgc3ludGhlc2lzXG4gKlxuICogVGhpcyBmZWF0dXJlIGlzIGludGVuZGVkIGZvciB1c2UgYnkgdGhlIENES1RGIG9ubHk7IDNyZCBwYXJ0eVxuICogbGlicmFyeSBhdXRob3JzIGFuZCBDREsgdXNlcnMgc2hvdWxkIG5vdCB1c2UgdGhpcyBmdW5jdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQ3VzdG9tU3ludGhlc2lzIHtcbiAgLyoqXG4gICAqIENhbGxlZCB3aGVuIHRoZSBjb25zdHJ1Y3QgaXMgc3ludGhlc2l6ZWRcbiAgICovXG4gIG9uU3ludGhlc2l6ZShzZXNzaW9uOiBJU3ludGhlc2lzU2Vzc2lvbik6IHZvaWQ7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhZGRDdXN0b21TeW50aGVzaXMoXG4gIGNvbnN0cnVjdDogSUNvbnN0cnVjdCxcbiAgc3ludGhlc2lzOiBJQ3VzdG9tU3ludGhlc2lzXG4pOiB2b2lkIHtcbiAgT2JqZWN0LmRlZmluZVByb3BlcnR5KGNvbnN0cnVjdCwgQ1VTVE9NX1NZTlRIRVNJU19TWU0sIHtcbiAgICB2YWx1ZTogc3ludGhlc2lzLFxuICAgIGVudW1lcmFibGU6IGZhbHNlLFxuICB9KTtcbn1cblxuZnVuY3Rpb24gZ2V0Q3VzdG9tU3ludGhlc2lzKFxuICBjb25zdHJ1Y3Q6IElDb25zdHJ1Y3Rcbik6IElDdXN0b21TeW50aGVzaXMgfCB1bmRlZmluZWQge1xuICByZXR1cm4gKGNvbnN0cnVjdCBhcyBhbnkpW0NVU1RPTV9TWU5USEVTSVNfU1lNXTtcbn1cbiJdfQ==