"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformAsset = exports.AssetType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const fs_1 = require("./private/fs");
const resource_1 = require("./resource");
const synthesizer_1 = require("./synthesize/synthesizer");
var AssetType;
(function (AssetType) {
    AssetType[AssetType["FILE"] = 0] = "FILE";
    AssetType[AssetType["DIRECTORY"] = 1] = "DIRECTORY";
    AssetType[AssetType["ARCHIVE"] = 2] = "ARCHIVE";
})(AssetType = exports.AssetType || (exports.AssetType = {}));
const ARCHIVE_NAME = "archive.zip";
const ASSETS_DIRECTORY = "assets";
class TerraformAsset extends resource_1.Resource {
    /**
     * A Terraform Asset takes a file or directory outside of the CDK for Terraform context and moves it into it.
     * Assets copy referenced files into the stacks context for further usage in other resources.
     * @param scope
     * @param id
     * @param config
     */
    constructor(scope, id, config) {
        var _b, _c;
        super(scope, id);
        if (path.isAbsolute(config.path)) {
            this.sourcePath = config.path;
        }
        else {
            const cdktfJsonPath = (_b = scope.node.tryGetContext("cdktfJsonPath")) !== null && _b !== void 0 ? _b : fs_1.findFileAboveCwd("cdktf.json");
            if (cdktfJsonPath) {
                // Relative paths are always considered to be relative to cdktf.json, but operations are performed relative to process.cwd
                const absolutePath = path.resolve(path.dirname(cdktfJsonPath), config.path);
                this.sourcePath = path.relative(process.cwd(), absolutePath);
            }
            else {
                throw new Error(`TerraformAsset ${id} was created with a relative path '${config.path}', but no cdktf.json file was found to base it on.`);
            }
        }
        const stat = fs.statSync(this.sourcePath);
        const inferredType = stat.isFile() ? AssetType.FILE : AssetType.DIRECTORY;
        this.type = (_c = config.type) !== null && _c !== void 0 ? _c : inferredType;
        this.assetHash = config.assetHash || fs_1.hashPath(this.sourcePath);
        if (stat.isFile() && this.type !== AssetType.FILE) {
            throw new Error(`TerraformAsset ${id} expects path to be a directory, a file was passed: '${config.path}'`);
        }
        if (!stat.isFile() && this.type === AssetType.FILE) {
            throw new Error(`TerraformAsset ${id} expects path to be a file, a directory was passed: '${config.path}'`);
        }
        synthesizer_1.addCustomSynthesis(this, {
            onSynthesize: this._onSynthesize.bind(this),
        });
    }
    get namedFolder() {
        return path.posix.join(ASSETS_DIRECTORY, this.stack.getLogicalId(this.node));
    }
    /**
     * The path relative to the root of the terraform directory in posix format
     * Use this property to reference the asset
     */
    get path() {
        return path.posix.join(this.namedFolder, // readable name
        this.assetHash, // hash depending on content so that path changes if content changes
        this.type === AssetType.DIRECTORY ? "" : this.fileName);
    }
    /**
     * Name of the asset
     */
    get fileName() {
        switch (this.type) {
            case AssetType.ARCHIVE:
                return ARCHIVE_NAME;
            default:
                return path.basename(this.sourcePath);
        }
    }
    _onSynthesize(session) {
        const stackManifest = session.manifest.forStack(this.stack);
        const basePath = path.join(session.manifest.outdir, stackManifest.synthesizedStackPath, "..");
        // Cleanup existing assets
        const previousVersionsFolder = path.join(basePath, this.namedFolder);
        if (fs.existsSync(previousVersionsFolder)) {
            fs.rmSync(previousVersionsFolder, { recursive: true });
        }
        const targetPath = path.join(basePath, this.path);
        if (this.type === AssetType.DIRECTORY) {
            fs.mkdirSync(targetPath, { recursive: true });
        }
        else {
            fs.mkdirSync(path.dirname(targetPath), { recursive: true });
        }
        switch (this.type) {
            case AssetType.FILE:
                fs.copyFileSync(this.sourcePath, targetPath);
                break;
            case AssetType.DIRECTORY:
                fs_1.copySync(this.sourcePath, targetPath);
                break;
            case AssetType.ARCHIVE:
                fs_1.archiveSync(this.sourcePath, targetPath);
                break;
            default:
                throw new Error(`Asset type ${this.type} is not implemented`);
        }
    }
}
exports.TerraformAsset = TerraformAsset;
_a = JSII_RTTI_SYMBOL_1;
TerraformAsset[_a] = { fqn: "cdktf.TerraformAsset", version: "0.11.0-pre.4" };
//# sourceMappingURL=data:application/json;base64,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