"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fn = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const token_1 = require("./tokens/token");
const tfExpression_1 = require("./tfExpression");
const token_map_1 = require("./tokens/private/token-map");
const encoding_1 = require("./tokens/private/encoding");
const _1 = require(".");
function hasUnescapedDoubleQuotes(str) {
    return /(^|[^\\])([\\]{2})*"/.test(str);
}
// Validators
function anyValue(value) {
    return value;
}
function mapValue(value) {
    return value;
}
function stringValue(value) {
    if (typeof value !== "string" && !token_1.Tokenization.isResolvable(value)) {
        throw new Error(`'${value}' is not a valid string nor a token`);
    }
    if (typeof value === "string" && hasUnescapedDoubleQuotes(value)) {
        throw new Error(`'${value}' can not be used as value directly since it has unescaped double quotes in it. To safely use the value please use Fn.rawString on your string.`);
    }
    return value;
}
function numericValue(value) {
    if (typeof value !== "number" && !token_1.Tokenization.isResolvable(value)) {
        throw new Error(`${value} is not a valid number nor a token`);
    }
    return value;
}
function listOf(type) {
    return (value) => {
        if (token_1.Tokenization.isResolvable(value)) {
            return value;
        }
        if (!Array.isArray(value)) {
            //   throw new Error(`${value} is not a valid list`);
            return value;
        }
        return value
            .filter((item) => item !== undefined && item !== null)
            .map((item, i) => {
            if (token_1.Tokenization.isResolvable(item)) {
                return item;
            }
            if (encoding_1.TokenString.forListToken(item).test()) {
                return item;
            }
            if (encoding_1.extractTokenDouble(item, true) !== undefined) {
                return item;
            }
            if (encoding_1.TokenString.forMapToken(item).test()) {
                return item;
            }
            if (typeof item === "string") {
                const tokenList = token_1.Tokenization.reverseString(item);
                const numberOfTokens = tokenList.tokens.length + tokenList.intrinsic.length;
                if (numberOfTokens === 1 && tokenList.literals.length === 0) {
                    return item;
                }
            }
            try {
                type(item);
                return typeof item === "string" ? `"${item}"` : item;
            }
            catch (error) {
                throw new Error(`Element in list ${value} at position ${i} is not of the right type: ${error}`);
            }
        });
    };
}
// Tokenization
function asString(value) {
    return token_map_1.TokenMap.instance().registerString(value);
}
function asNumber(value) {
    return token_map_1.TokenMap.instance().registerNumber(value);
}
function asList(value) {
    return token_map_1.TokenMap.instance().registerList(value);
}
function asBoolean(value) {
    return value; // Booleans can not be represented as a token
}
function asAny(value) {
    // Ordinarily casting to any can cause issues, but
    // in this case it makes using functions a bit easier in TS
    // and doesn't really harm other languages.
    // Jsii has issues when returning the value directly,
    // so wrap as a string.
    return asString(value);
}
function terraformFunction(name, argValidators) {
    return function (...args) {
        if (Array.isArray(argValidators)) {
            if (args.length !== argValidators.length) {
                throw new Error(`${name} takes ${argValidators.length} arguments, but ${args.length} were provided`);
            }
            return tfExpression_1.call(name, args.map((arg, i) => argValidators[i](arg)));
        }
        else {
            return tfExpression_1.call(name, argValidators(args));
        }
    };
}
class Fn {
    /**
     * {@link https://www.terraform.io/docs/language/functions/alltrue.html alltrue} returns true if all elements in a given collection are true or "true".
     * @param {Array} values
     */
    static alltrue(values) {
        return asBoolean(terraformFunction("alltrue", [listOf(anyValue)])(values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/anytrue.html anytrue} returns true if any element in a given collection is true or "true"
     * @param {Array} value
     */
    static anytrue(value) {
        return asBoolean(terraformFunction("anytrue", [listOf(anyValue)])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/chunklist.html chunklist} splits a single list into fixed-size chunks, returning a list of lists.
     * @param {Array} value
     * @param {number} chunkSize
     */
    static chunklist(value, chunkSize) {
        return asList(terraformFunction("chunklist", [listOf(anyValue), numericValue])(value, chunkSize));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/coalesce.html coalesce} takes any number of arguments and returns the first one that isn't null or an empty string.
     * @param {Array} value - Arguments are passed in an array
     */
    static coalesce(value) {
        return terraformFunction("coalesce", listOf(anyValue))(...value);
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/coalescelist.html coalescelist} takes any number of list arguments and returns the first one that isn't empty.
     * @param Array} value - Arguments are passed in an array
     */
    static coalescelist(value) {
        return asList(terraformFunction("coalescelist", listOf(anyValue))(...value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/compact.html compact} takes a list of strings and returns a new list with any empty string elements removed.
     * @param {Array} value
     */
    static compact(value) {
        return asList(terraformFunction("compact", [listOf(anyValue)])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/concat.html concat} takes two or more lists and combines them into a single list.
     * @param {Array} value
     */
    static concat(value) {
        return asList(terraformFunction("concat", listOf(anyValue))(...value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/contains.html contains} determines whether a given list or set contains a given single value as one of its elements.
     * @param {Array} list
     * @param {any} value
     */
    static contains(list, value) {
        return asBoolean(terraformFunction("contains", [listOf(anyValue), anyValue])(list, value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/distinct.html distinct} takes a list and returns a new list with any duplicate elements removed.
     * @param {Array} list
     */
    static distinct(list) {
        return asList(terraformFunction("distinct", [listOf(anyValue)])(list));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/element.html element} retrieves a single element from a list.
     * @param {Array} list
     * @param {number} index
     */
    static element(list, index) {
        return asAny(terraformFunction("element", [listOf(anyValue), numericValue])(list, index));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/flatten.html flatten} takes a list and replaces any elements that are lists with a flattened sequence of the list contents.
     * @param {Array} list
     */
    static flatten(list) {
        return asList(terraformFunction("flatten", [listOf(anyValue)])(list));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/index.html index} finds the element index for a given value in a list.
     * @param {Array} list
     * @param {any} value
     */
    static index(list, value) {
        return asNumber(terraformFunction("index", [listOf(anyValue), anyValue])(list, value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/keys.html keys} takes a map and returns a list containing the keys from that map.
     * @param {Object} map
     */
    static keys(map) {
        return asList(terraformFunction("keys", [mapValue])(map));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/length.html length} determines the length of a given list, map, or string.
     * @param {Object|Array|string} value
     */
    static lengthOf(value) {
        return asNumber(terraformFunction("length", [anyValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/lookup.html lookup} retrieves the value of a single element from a map, given its key. If the given key does not exist, the given default value is returned instead.
     * @param {Object} value
     * @param {any} key
     * @param {any} defaultValue
     */
    static lookup(value, key, defaultValue) {
        return asAny(terraformFunction("lookup", [anyValue, anyValue, anyValue])(value, key, defaultValue));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/matchkeys.html matchkeys} constructs a new list by taking a subset of elements from one list whose indexes match the corresponding indexes of values in another list.
     * @param {Array} valuesList
     * @param {Array} keysList
     * @param {Array} searchSet
     */
    static matchkeys(valuesList, keysList, searchSet) {
        return asList(terraformFunction("matchkeys", [anyValue, anyValue, anyValue])(valuesList, keysList, searchSet));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/merge.html merge} takes an arbitrary number of maps or objects, and returns a single map or object that contains a merged set of elements from all arguments.
     * @param {Array)} values - Arguments are passed in an array
     */
    static merge(values) {
        return asList(terraformFunction("merge", listOf(anyValue))(...values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/one.html one} takes a list, set, or tuple value with either zero or one elements.
     * @param {Array} list
     */
    static one(list) {
        return asAny(terraformFunction("one", [listOf(anyValue)])(list));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/range.html range} generates a list of numbers using a start value, a limit value, and a step value.
     * @param {number} start
     * @param {number} limit
     * @param {number=1} step
     */
    static range(start, limit, step = 1) {
        return asList(terraformFunction("range", listOf(anyValue))(start, limit, step));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/reverse.html reverse} takes a sequence and produces a new sequence of the same length with all of the same elements as the given sequence but in reverse order.
     * @param {Array} values
     */
    static reverse(values) {
        return asList(terraformFunction("reverse", [listOf(anyValue)])(values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/setintersection.html setintersection} function takes multiple sets and produces a single set containing only the elements that all of the given sets have in common
     * @param {Array} values
     */
    static setintersection(values) {
        return asList(terraformFunction("setintersection", listOf(anyValue))(...values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/setproduct.html setproduct} function finds all of the possible combinations of elements from all of the given sets by computing the Cartesian product.
     * @param {Array} values
     */
    static setproduct(values) {
        return asList(terraformFunction("setproduct", listOf(anyValue))(...values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/slice.html setsubtract} function returns a new set containing the elements from the first set that are not present in the second set.
     * @param {Array} minuend
     * @param {Array} subtrahend
     */
    static setsubtract(minuend, subtrahend) {
        return asList(terraformFunction("setsubtract", [listOf(anyValue), listOf(anyValue)])(minuend, subtrahend));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/setunion.html setunion} function takes multiple sets and produces a single set containing the elements from all of the given sets.
     * @param {Array} values - Arguments are passed in an array
     */
    static setunion(values) {
        return asList(terraformFunction("setunion", listOf(anyValue))(...values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/slice.html slice} extracts some consecutive elements from within a list.
     * @param {Array} list
     * @param {number} startindex
     * @param {number} endindex
     */
    static slice(list, startindex, endindex) {
        return asList(terraformFunction("slice", [
            listOf(anyValue),
            numericValue,
            numericValue,
        ])(list, startindex, endindex));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/sort.html sort} takes a list of strings and returns a new list with those strings sorted lexicographically.
     * @param {Array} list
     */
    static sort(list) {
        return asList(terraformFunction("sort", [listOf(anyValue)])(list));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/sum.html sum} takes a list or set of numbers and returns the sum of those numbers.
     * @param {Array} list
     */
    static sum(list) {
        return asNumber(terraformFunction("sum", [listOf(anyValue)])(list));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/transpose.html transpose} takes a map of lists of strings and swaps the keys and values to produce a new map of lists of strings.
     * @param {Object} value
     */
    static transpose(value) {
        return asAny(terraformFunction("transpose", [mapValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/values.html values} takes a map and returns a list containing the values of the elements in that map.
     * @param {Object} value
     */
    static values(value) {
        return asList(terraformFunction("values", [mapValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/zipmap.html zipmap} constructs a map from a list of keys and a corresponding list of values.
     * @param {Array} keyslist
     * @param {Array} valueslist
     */
    static zipmap(keyslist, valueslist) {
        return asAny(terraformFunction("zipmap", [mapValue])(keyslist, valueslist));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/base64sha256.html base64sha256} computes the SHA256 hash of a given string and encodes it with Base64.
     * @param {string} value
     */
    static base64sha256(value) {
        return asString(terraformFunction("base64sha256", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/base64sha512.html base64sha512} computes the SHA512 hash of a given string and encodes it with Base64.
     * @param {string} value
     */
    static base64sha512(value) {
        return asString(terraformFunction("base64sha512", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/bcrypt.html bcrypt} computes a hash of the given string using the Blowfish cipher, returning a string in the Modular Crypt Format usually expected in the shadow password file on many Unix systems.
     * @param {string} value
     * @param {number=10} cost
     */
    static bcrypt(value, cost) {
        return asString(terraformFunction("bcrypt", listOf(anyValue))(value, cost));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/filebase64sha256.html filebase64sha256} is a variant of base64sha256 that hashes the contents of a given file rather than a literal string.
     * @param {string} value
     */
    static filebase64sha256(value) {
        return asString(terraformFunction("filebase64sha256", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/filebase64sha512.html filebase64sha512} is a variant of base64sha512 that hashes the contents of a given file rather than a literal string.
     * @param {string} value
     */
    static filebase64sha512(value) {
        return asString(terraformFunction("filebase64sha512", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/filemd5.html filemd5} is a variant of md5 that hashes the contents of a given file rather than a literal string.
     * @param {string} value
     */
    static filemd5(value) {
        return asString(terraformFunction("filemd5", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/filesha1.html filesha1} is a variant of sha1 that hashes the contents of a given file rather than a literal string.
     * @param {string} value
     */
    static filesha1(value) {
        return asString(terraformFunction("filesha1", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/filesha256.html filesha256} is a variant of sha256 that hashes the contents of a given file rather than a literal string.
     * @param {string} value
     */
    static filesha256(value) {
        return asString(terraformFunction("filesha256", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/filesha512.html filesha512} is a variant of sha512 that hashes the contents of a given file rather than a literal string.
     * @param {string} value
     */
    static filesha512(value) {
        return asString(terraformFunction("filesha512", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/md5.html md5} computes the MD5 hash of a given string and encodes it with hexadecimal digits.
     * @param {string} value
     */
    static md5(value) {
        return asString(terraformFunction("md5", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/rsadecrypt.html rsadecrypt} decrypts an RSA-encrypted ciphertext, returning the corresponding cleartext.
     * @param {string} ciphertext
     * @param {string} privatekey
     */
    static rsadecrypt(ciphertext, privatekey) {
        return asString(terraformFunction("rsadecrypt", [stringValue, stringValue])(ciphertext, privatekey));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/sha1.html sha1} computes the SHA1 hash of a given string and encodes it with hexadecimal digits.
     * @param {string} value
     */
    static sha1(value) {
        return asString(terraformFunction("sha1", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/sha256.html sha256} computes the SHA256 hash of a given string and encodes it with hexadecimal digits.
     * @param {string} value
     */
    static sha256(value) {
        return asString(terraformFunction("sha256", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/sha512.html sha512} computes the SHA512 hash of a given string and encodes it with hexadecimal digits.
     * @param {string} value
     */
    static sha512(value) {
        return asString(terraformFunction("sha512", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/uuid.html uuid} generates a unique identifier string.
     */
    static uuid() {
        return asString(terraformFunction("uuid", [])());
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/uuidv5.html uuidv5} generates a unique identifier string.
     * @param {string} namespace
     * @param {string} name
     */
    static uuidv5(namespace, name) {
        return asString(terraformFunction("uuidv5", [stringValue, stringValue])(namespace, name));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/formatdate.html formatdate} converts a timestamp into a different time format.
     * @param {string} spec
     * @param {string} timestamp
     */
    static formatdate(spec, timestamp) {
        return asString(terraformFunction("formatdate", [stringValue, stringValue])(spec, timestamp));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/timeadd.html timeadd} adds a duration to a timestamp, returning a new timestamp.
     * @param {string} timestamp
     * @param {string} duration
     */
    static timeadd(timestamp, duration) {
        return asString(terraformFunction("timeadd", [stringValue, stringValue])(timestamp, duration));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/timestamp.html timestamp} returns a UTC timestamp string in RFC 3339 format.
     */
    static timestamp() {
        return asString(terraformFunction("timestamp", [])());
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/base64decode.html base64decode} takes a string containing a Base64 character sequence and returns the original string.
     * @param {string} value
     */
    static base64decode(value) {
        return asString(terraformFunction("base64decode", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/base64encode.html base64encode} takes a string containing a Base64 character sequence and returns the original string.
     * @param {string} value
     */
    static base64encode(value) {
        return asString(terraformFunction("base64encode", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/base64gzip.html base64gzip} compresses a string with gzip and then encodes the result in Base64 encoding.
     * @param {string} value
     */
    static base64gzip(value) {
        return asString(terraformFunction("base64gzip", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/csvdecode.html csvdecode} decodes a string containing CSV-formatted data and produces a list of maps representing that data.
     * @param {string} value
     */
    static csvdecode(value) {
        return asList(terraformFunction("csvdecode", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/jsondecode.html jsondecode} interprets a given string as JSON, returning a representation of the result of decoding that string.
     * @param {string} value
     */
    static jsondecode(value) {
        return asAny(terraformFunction("jsondecode", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/jsonencode.html jsonencode} encodes a given value to a string using JSON syntax.
     * @param {any} value
     */
    static jsonencode(value) {
        return asString(terraformFunction("jsonencode", [anyValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/textdecodebase64.html textdecodebase64} function decodes a string that was previously Base64-encoded, and then interprets the result as characters in a specified character encoding.
     * @param {string} value
     * @param {string} encodingName
     */
    static textdecodebase64(value, encodingName) {
        return asString(terraformFunction("textdecodebase64", [stringValue, stringValue])(value, encodingName));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/textencodebase64.html textencodebase64}  encodes the unicode characters in a given string using a specified character encoding, returning the result base64 encoded because Terraform language strings are always sequences of unicode characters.
     * @param {string} value
     * @param {string} encodingName
     */
    static textencodebase64(value, encodingName) {
        return asString(terraformFunction("textencodebase64", [stringValue, stringValue])(value, encodingName));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/urlencode.html urlencode} applies URL encoding to a given string.
     * @param {string} value
     */
    static urlencode(value) {
        return asString(terraformFunction("urlencode", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/yamldecode.html yamldecode} parses a string as a subset of YAML, and produces a representation of its value.
     * @param {string} value
     */
    static yamldecode(value) {
        return asAny(terraformFunction("yamldecode", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/yamlencode.html yamlencode} encodes a given value to a string using JSON syntax.
     * @param {any} value
     */
    static yamlencode(value) {
        return asString(terraformFunction("yamlencode", [anyValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/abspath.html abspath} takes a string containing a filesystem path and converts it to an absolute path.
     * @param {string} value
     */
    static abspath(value) {
        return asString(terraformFunction("abspath", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/dirname.html dirname} takes a string containing a filesystem path and removes the last portion from it.
     * @param {string} value
     */
    static dirname(value) {
        return asString(terraformFunction("dirname", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/pathexpand.html pathexpand} takes a string containing a filesystem path and removes the last portion from it.
     * @param {string} value
     */
    static pathexpand(value) {
        return asString(terraformFunction("pathexpand", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/basename.html basename} takes a string containing a filesystem path and removes all except the last portion from it.
     * @param {string} value
     */
    static basename(value) {
        return asString(terraformFunction("basename", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/file.html file} takes a string containing a filesystem path and removes all except the last portion from it.
     * @param {string} value
     */
    static file(value) {
        return asString(terraformFunction("file", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/fileexists.html fileexists} determines whether a file exists at a given path.
     * @param {string} value
     */
    static fileexists(value) {
        return asBoolean(terraformFunction("fileexists", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/fileset.html fileset} enumerates a set of regular file names given a path and pattern.
     * @param {string} path
     * @param {string} pattern
     */
    static fileset(path, pattern) {
        return asList(terraformFunction("fileset", [stringValue, stringValue])(path, pattern));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/filebase64.html filebase64} reads the contents of a file at the given path and returns them as a base64-encoded string.
     * @param {string} value
     */
    static filebase64(value) {
        return asString(terraformFunction("filebase64", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/templatefile.html templatefile} reads the file at the given path and renders its content as a template using a supplied set of template variables.
     * @param {string} path
     * @param {Object} vars
     */
    static templatefile(path, vars) {
        return asString(terraformFunction("templatefile", [stringValue, mapValue])(path, vars));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/cidrhost.html cidrhost} calculates a full host IP address for a given host number within a given IP network address prefix.
     * @param {string} prefix
     * @param {number} hostnum
     */
    static cidrhost(prefix, hostnum) {
        return asString(terraformFunction("cidrhost", [stringValue, numericValue])(prefix, hostnum));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/cidrnetmask.html cidrnetmask} converts an IPv4 address prefix given in CIDR notation into a subnet mask address.
     * @param {string} prefix
     */
    static cidrnetmask(prefix) {
        return asString(terraformFunction("cidrnetmask", [stringValue])(prefix));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/cidrsubnet.html cidrsubnet} calculates a subnet address within given IP network address prefix.
     * @param {string} prefix
     * @param {number} newbits
     * @param {number} netnum
     */
    static cidrsubnet(prefix, newbits, netnum) {
        return asString(terraformFunction("cidrsubnet", [
            stringValue,
            numericValue,
            numericValue,
        ])(prefix, newbits, netnum));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/cidrsubnets.html cidrsubnets} calculates a sequence of consecutive IP address ranges within a particular CIDR prefix.
     * @param {string} prefix
     * @param {...number} newbits
     */
    static cidrsubnets(prefix, newbits) {
        return asList(terraformFunction("cidrsubnets", listOf(anyValue))(prefix, ...newbits));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/abs.html abs} returns the absolute value of the given number
     * @param {number} value
     */
    static abs(value) {
        return asNumber(terraformFunction("abs", [numericValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/ceil.html ceil} returns the closest whole number that is greater than or equal to the given value, which may be a fraction.
     * @param {number} value
     */
    static ceil(value) {
        return asNumber(terraformFunction("ceil", [numericValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/floor.html floor} returns the closest whole number that is less than or equal to the given value, which may be a fraction
     * @param {number} value
     */
    static floor(value) {
        return asNumber(terraformFunction("floor", [numericValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/log.html log} returns the logarithm of a given number in a given base.
     * @param {number} value
     * @param {number} base
     */
    static log(value, base) {
        return asNumber(terraformFunction("log", [numericValue, numericValue])(value, base));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/max.html max} takes one or more numbers and returns the greatest number from the set.
     * @param {Array<number>} values
     */
    static max(values) {
        return asNumber(terraformFunction("max", listOf(numericValue))(...values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/min.html min} takes one or more numbers and returns the smallest number from the set.
     * @param {Array<number>} values
     */
    static min(values) {
        return asNumber(terraformFunction("min", listOf(numericValue))(...values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/parseint.html parseInt} parses the given string as a representation of an integer in the specified base and returns the resulting number. The base must be between 2 and 62 inclusive.
     * @param {string} value
     * @param {number} base
     */
    static parseInt(value, base) {
        return asNumber(terraformFunction("parseint", [stringValue, numericValue])(value, base));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/pow.html pow} calculates an exponent, by raising its first argument to the power of the second argument.
     * @param {number} value
     * @param {number} power
     */
    static pow(value, power) {
        return asNumber(terraformFunction("pow", [numericValue, numericValue])(value, power));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/signum.html signum} determines the sign of a number, returning a number between -1 and 1 to represent the sign.
     * @param {number} value
     */
    static signum(value) {
        return asNumber(terraformFunction("signum", [numericValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/chomp.html chomp} removes newline characters at the end of a string.
     * @param {string} value
     */
    static chomp(value) {
        return asString(terraformFunction("chomp", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/format.html format} produces a string by formatting a number of other values according to a specification string
     * @param {string} spec
     * @param {Array} values
     */
    static format(spec, values) {
        return asString(terraformFunction("format", listOf(anyValue))(spec, ...values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/formatlist.html formatlist} produces a list of strings by formatting a number of other values according to a specification string.
     * @param {string} spec
     * @param {Array<string>} values
     */
    static formatlist(spec, values) {
        return asList(terraformFunction("formatlist", listOf(anyValue))(spec, ...values));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/indent.html indent} adds a given number of spaces to the beginnings of all but the first line in a given multi-line string.
     * @param {number} indentation
     * @param {string} value
     */
    static indent(indentation, value) {
        return asString(terraformFunction("indent", [numericValue, stringValue])(indentation, value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/join.html join} produces a string by concatenating together all elements of a given list of strings with the given delimiter.
     * @param {string} separator
     * @param {Array} value
     */
    static join(separator, value) {
        return asString(terraformFunction("join", [stringValue, listOf(anyValue)])(separator, value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/lower.html lower} converts all cased letters in the given string to lowercase.
     * @param {string} value
     */
    static lower(value) {
        return asString(terraformFunction("lower", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/regexall.html regexall} applies a regular expression to a string and returns a list of all matches.
     * @param {string} pattern
     * @param {string} value
     */
    static regexall(pattern, value) {
        return asList(terraformFunction("regexall", [stringValue, stringValue])(pattern, value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/regex.html regex} applies a regular expression to a string and returns the matching substrings in pattern.
     * @param {string} pattern
     * @param {string} value
     */
    static regex(pattern, value) {
        return asString(terraformFunction("regex", [stringValue, stringValue])(pattern, value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/replace.html replace} searches a given string for another given substring, and replaces each occurrence with a given replacement string.
     * @param {string} value
     * @param {string} substring
     * @param {string} replacement
     */
    static replace(value, substring, replacement) {
        return asString(terraformFunction("replace", [stringValue, stringValue, stringValue])(value, substring, replacement));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/split.html split} produces a list by dividing a given string at all occurrences of a given separator.
     * @param {string} seperator
     * @param {string} value
     */
    static split(seperator, value) {
        return asList(terraformFunction("split", [stringValue, stringValue])(seperator, value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/strrev.html strrev} reverses the characters in a string.
     * @param {string} value
     */
    static strrev(value) {
        return asString(terraformFunction("strrev", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/substr.html substr} extracts a substring from a given string by offset and length.
     * @param {string} value
     * @param {number} offset
     * @param {number} length
     */
    static substr(value, offset, length) {
        return asString(terraformFunction("substr", [stringValue, numericValue, numericValue])(value, offset, length));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/title.html title} converts the first letter of each word in the given string to uppercase.
     * @param {string} value
     */
    static title(value) {
        return asString(terraformFunction("title", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/trim.html trim} removes the specified characters from the start and end of the given string.
     * @param {string} value
     * @param {string} replacement
     */
    static trim(value, replacement) {
        return asString(terraformFunction("trim", [stringValue, stringValue])(value, replacement));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/trimprefix.html trimprefix} removes the specified prefix from the start of the given string.
     * @param {string} value
     * @param {string} prefix
     */
    static trimprefix(value, prefix) {
        return asString(terraformFunction("trimprefix", [stringValue, stringValue])(value, prefix));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/trimsuffix.html trimsuffix} removes the specified suffix from the end of the given string.
     * @param {string} value
     * @param {string} suffix
     */
    static trimsuffix(value, suffix) {
        return asString(terraformFunction("trimsuffix", [stringValue, stringValue])(value, suffix));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/trimspace.html trimspace} removes any space characters from the start and end of the given string.
     * @param {string} value
     */
    static trimspace(value) {
        return asString(terraformFunction("trimspace", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/upper.html upper} converts all cased letters in the given string to uppercase.
     * @param {string} value
     */
    static upper(value) {
        return asString(terraformFunction("upper", [stringValue])(value));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/can.html can} evaluates the given expression and returns a boolean value indicating whether the expression produced a result without any errors.
     * @param {any} expression
     */
    static can(expression) {
        return asBoolean(terraformFunction("can", [anyValue])(expression));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/nonsensitive.html nonsensitive} takes a sensitive value and returns a copy of that value with the sensitive marking removed, thereby exposing the sensitive value.
     * @param {any} expression
     */
    static nonsensitive(expression) {
        return asAny(terraformFunction("nonsensitive", [anyValue])(expression));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/sensitive.html sensitive} takes any value and returns a copy of it marked so that Terraform will treat it as sensitive, with the same meaning and behavior as for sensitive input variables.
     * @param {any} expression
     */
    static sensitive(expression) {
        return asAny(terraformFunction("sensitive", [anyValue])(expression));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/tobool.html tobool} converts its argument to a boolean value.
     * @param {any} expression
     */
    static tobool(expression) {
        return asBoolean(terraformFunction("tobool", [anyValue])(expression));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/tolist.html tolist} converts its argument to a list value.
     * @param {any} expression
     */
    static tolist(expression) {
        return asList(terraformFunction("tolist", [anyValue])(expression));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/tomap.html tomap} converts its argument to a map value.
     * @param {any} expression
     */
    static tomap(expression) {
        return asAny(terraformFunction("tomap", [anyValue])(expression));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/tonumber.html tonumber} converts its argument to a number value.
     * @param {any} expression
     */
    static tonumber(expression) {
        return asNumber(terraformFunction("tonumber", [anyValue])(expression));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/toset.html toset} converts its argument to a set value.
     * @param {any} expression
     */
    static toset(expression) {
        return asAny(terraformFunction("toset", [anyValue])(expression));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/tostring.html tostring} converts its argument to a string value.
     * @param {any} expression
     */
    static tostring(expression) {
        return asString(terraformFunction("tostring", [anyValue])(expression));
    }
    /**
     * {@link https://www.terraform.io/docs/language/functions/try.html try} evaluates all of its argument expressions in turn and returns the result of the first one that does not produce any errors.
     * @param {Array} expression
     */
    static try(expression) {
        return asAny(terraformFunction("try", listOf(anyValue))(...expression));
    }
    /**
     * Use this function to wrap a string and escape it properly for the use in Terraform
     * This is only needed in certain scenarios (e.g., if you have unescaped double quotes in the string)
     * @param {String} str
     */
    static rawString(str) {
        return asString(_1.rawString(str));
    }
}
exports.Fn = Fn;
_a = JSII_RTTI_SYMBOL_1;
Fn[_a] = { fqn: "cdktf.Fn", version: "0.11.0-pre.4" };
//# sourceMappingURL=data:application/json;base64,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