"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const index_1 = require("../index");
const resource_1 = require("../../../test/helper/resource");
const matchers_1 = require("../matchers");
const data_source_1 = require("../../../test/helper/data-source");
const terraform_stack_1 = require("../../terraform-stack");
const provider_1 = require("../../../test/helper/provider");
const fs = require("fs");
const path = require("path");
function corruptSynthesizedStack(stackPath) {
    const manifest = JSON.parse(fs.readFileSync(path.resolve(stackPath, "manifest.json"), "utf8"));
    const { synthesizedStackPath } = Object.values(manifest.stacks)[0];
    const absoluteStackPath = path.resolve(stackPath, synthesizedStackPath);
    fs.writeFileSync(absoluteStackPath, fs.readFileSync(absoluteStackPath, "utf8") +
        "Weird text that makes the JSON file invalid", "utf8");
}
describe("deepEqualIgnoringObjectCasing", () => {
    it("should compare simple items", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(1, 1)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing("1", "1")).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(true, true)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(false, false)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(null, null)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(undefined, undefined)).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing([1, 2, 3], [1, 2, 3])).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: 1, b: 2 }, { a: 1, b: 2 })).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing(1, 2)).toBe(false);
    });
    it("should compare arrays", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing([1, 2, 3], [1, 2, 3])).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing([1, 2, 3], [1, 2, 4])).toBe(false);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing([1, 2, 3], [1, 2, 3, 4])).toBe(false);
    });
    it("should compare objects", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: 1, b: 2 }, { a: 1, b: 2 })).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: { c: 3 }, b: 2 }, { a: { c: 3 }, b: 2 })).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: 3, b: 2 }, { a: { c: 3 }, b: 2 })).toBe(false);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: { c: 3 }, b: 2 }, { a: 3, b: 2 })).toBe(false);
    });
    it("should ignore case when comparing object keys", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: 1, fooBar: 2 }, { a: 1, foo_bar: 2 })).toBe(true);
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: { fooBar: 2 } }, { a: { foo_bar: 2 } })).toBe(true);
    });
    it("should ignore keys not present in expectation", () => {
        expect(matchers_1.asymetricDeepEqualIgnoringObjectCasing({ a: { fooBar: 2 } }, { a: { foo_bar: 2 }, b: 3 })).toBe(true);
    });
});
describe("matchers", () => {
    const toHaveResourceWithProperties = matchers_1.getToHaveResourceWithProperties();
    describe("toHaveResourceWithProperties", () => {
        let synthesizedStack;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new resource_1.TestResource(scope, "test", { name: "test", tags: { foo: "bar" } });
            });
        });
        it("should pass with no properties", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeTruthy();
            expect(res.message()).toMatchInlineSnapshot(`
        "Expected no test_resource with properties {} to be present in synthesised stack.
        Found 1 test_resource resources instead:
        [
          {
            \\"name\\": \\"test\\",
            \\"tags\\": {
              \\"foo\\": \\"bar\\"
            }
          }
        ]"
      `);
        });
        it("should fail with wrong resouce", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, data_source_1.TestDataSource);
            expect(res.pass).toBeFalsy();
            expect(res.message()).toMatchInlineSnapshot(`
        "Expected test_data_source with properties {} to be present in synthesised stack.
        Found no test_data_source resources instead"
      `);
        });
        it("should work on nested elements", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.TestResource, {
                tags: { foo: "bar" },
            });
            expect(res.pass).toBeTruthy();
        });
        it("should fail if no item is found as asserted", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.DockerImage);
            expect(res.pass).toBeFalsy();
        });
    });
    describe("toHaveDataSourceWithProperties", () => {
        const toHaveDataSourceWithProperties = matchers_1.getToHaveDataSourceWithProperties();
        let synthesizedStack;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new resource_1.TestResource(scope, "test", { name: "test" });
                new data_source_1.TestDataSource(scope, "test-data-source", { name: "data" });
            });
        });
        it("should not find resources", () => {
            const res = toHaveDataSourceWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeFalsy();
        });
        it("should find data sources", () => {
            const res = toHaveDataSourceWithProperties(synthesizedStack, data_source_1.TestDataSource, { name: "data" });
            expect(res.pass).toBeTruthy();
        });
    });
    describe("toBeValidTerraform", () => {
        it("fails if anything but a path is passed", () => {
            const res = matchers_1.toBeValidTerraform("not a path");
            expect(res.pass).toBeFalsy();
            expect(res.message()).toMatchInlineSnapshot(`"Expected subject to be a terraform directory: Error: ENOENT: no such file or directory, stat 'not a path'"`);
        });
        it("succeeds if the terraform config passed is valid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const res = matchers_1.toBeValidTerraform(index_1.Testing.fullSynth(stack));
            expect(res.pass).toBeTruthy();
            expect(res.message()).toMatchInlineSnapshot(`"Expected subject not to be a valid terraform stack"`);
        });
        it("fails if the terraform config passed is invalid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const result = index_1.Testing.fullSynth(stack);
            corruptSynthesizedStack(result);
            const res = matchers_1.toBeValidTerraform(result);
            expect(res.pass).toBeFalsy();
            expect(res.message()).toEqual(expect.stringContaining("Expected subject to be a valid terraform stack"));
        });
    });
    describe("toPlanSuccessfully", () => {
        it("fails if anything but a path is passed", () => {
            const res = matchers_1.toPlanSuccessfully("not a path");
            expect(res.pass).toBeFalsy();
            expect(res.message()).toMatchInlineSnapshot(`"Expected subject to be a terraform directory: Error: ENOENT: no such file or directory, stat 'not a path'"`);
        });
        it("succeeds if the terraform config passed is valid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const res = matchers_1.toPlanSuccessfully(index_1.Testing.fullSynth(stack));
            expect(res.pass).toBeTruthy();
            expect(res.message()).toMatchInlineSnapshot(`"Expected subject not to plan successfully"`);
        });
        it("fails if the terraform config passed is invalid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const result = index_1.Testing.fullSynth(stack);
            corruptSynthesizedStack(result);
            const res = matchers_1.toPlanSuccessfully(result);
            expect(res.pass).toBeFalsy();
            expect(res.message()).toEqual(expect.stringContaining("Expected subject to plan successfully"));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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