"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Testing = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const os = require("os");
const stringify = require("json-stable-stringify");
const lib_1 = require("../../lib");
const terraform_stack_1 = require("../terraform-stack");
const manifest_1 = require("../manifest");
const features_1 = require("../features");
const jest_1 = require("./adapters/jest");
const synthesizer_1 = require("../synthesize/synthesizer");
const DefaultTestingAppOptions = {
    stackTraces: false,
    stubVersion: true,
    enableFutureFlags: true,
    fakeCdktfJsonPath: false,
};
/**
 * Testing utilities for cdktf applications.
 */
class Testing {
    /* istanbul ignore next */
    constructor() {
        return;
    }
    /**
     * Returns an app for testing with the following properties:
     * - Output directory is a temp dir.
     */
    static app(options = {}) {
        const appOptions = { ...DefaultTestingAppOptions, ...options };
        if (!appOptions.outdir) {
            appOptions.outdir = fs.mkdtempSync(path.join(os.tmpdir(), "cdktf.outdir."));
        }
        const app = new lib_1.App({
            outdir: appOptions.outdir,
            stackTraces: appOptions.stackTraces,
        });
        if (appOptions.stubVersion) {
            this.stubVersion(app);
        }
        if (appOptions.enableFutureFlags) {
            this.enableFutureFlags(app);
        }
        if (appOptions.fakeCdktfJsonPath) {
            this.fakeCdktfJsonPath(app);
        }
        return app;
    }
    static stubVersion(app) {
        app.node.setContext("cdktfVersion", "stubbed");
        app.manifest.version = "stubbed";
        return app;
    }
    static fakeCdktfJsonPath(app) {
        app.node.setContext("cdktfJsonPath", `${process.cwd()}/cdktf.json`);
        return app;
    }
    static enableFutureFlags(app) {
        const node = app.node;
        Object.entries(features_1.FUTURE_FLAGS).forEach(([key, value]) => node.setContext(key, value));
        return app;
    }
    static synthScope(fn) {
        const stack = new terraform_stack_1.TerraformStack(Testing.app(), "stack");
        fn(stack);
        return Testing.synth(stack);
    }
    /**
     * Returns the Terraform synthesized JSON.
     */
    static synth(stack) {
        synthesizer_1.invokeAspects(stack);
        const tfConfig = stack.toTerraform();
        function removeMetadata(item) {
            if (item !== null && typeof item === "object") {
                if (Array.isArray(item)) {
                    return item.map(removeMetadata);
                }
                const cleanedItem = Object.entries(item)
                    // order alphabetically
                    .sort(([a], [b]) => a.localeCompare(b))
                    .reduce((acc, [key, value]) => ({ ...acc, [key]: removeMetadata(value) }), {});
                // Remove metadata
                delete cleanedItem["//"];
                return cleanedItem;
            }
            return item;
        }
        const cleaned = removeMetadata(tfConfig);
        return stringify(cleaned, { space: 2 });
    }
    static fullSynth(stack) {
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), "cdktf.outdir."));
        const manifest = new manifest_1.Manifest("stubbed", outdir);
        stack.synthesizer.synthesize({
            outdir,
            manifest,
        });
        manifest.writeToFile();
        return outdir;
    }
    static renderConstructTree(construct) {
        return render(construct, 0, false);
        function render(construct, level, isLast) {
            let prefix = "";
            if (level > 0) {
                const spaces = " ".repeat((level - 1) * 4);
                const symbol = isLast ? "└" : "├";
                prefix = `${spaces}${symbol}── `;
            }
            const name = construct instanceof lib_1.App
                ? "App"
                : `${construct.node.id} (${construct.constructor.name})`;
            return `${prefix}${name}\n${construct.node.children
                .map((child, idx, arr) => {
                const isLast = idx === arr.length - 1;
                return render(child, level + 1, isLast);
            })
                .join("")}`;
        }
    }
    static setupJest() {
        jest_1.setupJest();
    }
}
exports.Testing = Testing;
_a = JSII_RTTI_SYMBOL_1;
Testing[_a] = { fqn: "cdktf.Testing", version: "0.11.0-pre.4" };
//# sourceMappingURL=data:application/json;base64,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