"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.toPlanSuccessfully = exports.toBeValidTerraform = exports.getToHaveResourceWithProperties = exports.getToHaveDataSourceWithProperties = exports.asymetricDeepEqualIgnoringObjectCasing = void 0;
const fs = require("fs");
const path = require("path");
const child_process_1 = require("child_process");
const util_1 = require("../util");
const terraformBinaryName = process.env.TERRAFORM_BINARY_NAME || "terraform";
// All expected properties are matched and considered equal if
// There can be more properties in the received object than in the expected object while still returning true
function asymetricDeepEqualIgnoringObjectCasing(expected, received) {
    switch (typeof expected) {
        case "object":
            if (Array.isArray(expected)) {
                return (Array.isArray(received) &&
                    expected.length === received.length &&
                    expected.every((item, index) => asymetricDeepEqualIgnoringObjectCasing(item, received[index]) // recursively compare arrays
                    ));
            }
            if (expected === null && received === null) {
                return true;
            }
            if (expected === undefined && received === undefined) {
                return true;
            }
            if (expected === null || received === null) {
                return false;
            }
            // recursively compare objects and allow snake case as well as camel case
            return Object.keys(expected).every((key) => {
                if (received[key] !== undefined) {
                    return asymetricDeepEqualIgnoringObjectCasing(expected[key], received[key]);
                }
                if (received[util_1.snakeCase(key)] !== undefined) {
                    return asymetricDeepEqualIgnoringObjectCasing(expected[key], received[util_1.snakeCase(key)]);
                }
                return false;
            });
        default:
            return expected === received;
    }
}
exports.asymetricDeepEqualIgnoringObjectCasing = asymetricDeepEqualIgnoringObjectCasing;
const defaultPassEvaluation = (items, assertedProperties) => {
    return Object.values(items).some((item) => asymetricDeepEqualIgnoringObjectCasing(assertedProperties, item));
};
function isAsymmetric(obj) {
    return !!obj && typeof obj === "object" && "asymmetricMatch" in obj;
}
// You can use expect.Anything(), expect.ObjectContaining, etc in jest, this makes it nicer to read
// when we print error mesages
function jestAsymetricMatcherStringifyReplacer(_key, value) {
    return isAsymmetric(value) ? `expect.${value.toString()}` : value;
}
function getAssertElementWithProperties(
// We have the evaluation function configurable so we can make use of the specific testing frameworks capabilities
// This makes the resulting tests more native to the testing framework
customPassEvaluation) {
    const passEvaluation = customPassEvaluation || defaultPassEvaluation;
    return function getAssertElementWithProperties(type, received, itemType, properties = {}) {
        var _a;
        let stack;
        try {
            stack = JSON.parse(received);
        }
        catch (e) {
            throw new Error(`invalid JSON string passed: ${received}`);
        }
        const items = Object.values(((_a = Object.entries(stack[type] || {}) // for all data/resource entries
            .find(
        // find the object with a matching name
        ([type, _values]) => type === itemType.tfResourceType)) === null || _a === void 0 ? void 0 : _a[1]) || {} // get all items of that type (encoded as a record of name -> config)
        ) || []; // get a list of all configs of that type
        const pass = passEvaluation(items, properties);
        if (pass) {
            return {
                pass,
                message: () => `Expected no ${itemType.tfResourceType} with properties ${JSON.stringify(properties, jestAsymetricMatcherStringifyReplacer)} to be present in synthesised stack.
Found ${items.length === 0 ? "no" : items.length} ${itemType.tfResourceType} resources instead${items.length > 0 ? ":\n" + JSON.stringify(items, null, 2) : ""}`,
            };
        }
        else {
            return {
                message: () => `Expected ${itemType.tfResourceType} with properties ${JSON.stringify(properties, jestAsymetricMatcherStringifyReplacer)} to be present in synthesised stack.
Found ${items.length === 0 ? "no" : items.length} ${itemType.tfResourceType} resources instead${items.length > 0 ? ":\n" + JSON.stringify(items, null, 2) : ""}`,
                pass,
            };
        }
    };
}
function getToHaveDataSourceWithProperties(customPassEvaluation) {
    return function toHaveDataSourceWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties(customPassEvaluation)("data", received, resourceType, properties);
    };
}
exports.getToHaveDataSourceWithProperties = getToHaveDataSourceWithProperties;
function getToHaveResourceWithProperties(customPassEvaluation) {
    return function toHaveResourceWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties(customPassEvaluation)("resource", received, resourceType, properties);
    };
}
exports.getToHaveResourceWithProperties = getToHaveResourceWithProperties;
function toBeValidTerraform(received) {
    try {
        if (!fs.statSync(received).isDirectory()) {
            throw new Error("Path is not a directory");
        }
    }
    catch (e) {
        return {
            message: () => `Expected subject to be a terraform directory: ${e}`,
            pass: false,
        };
    }
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(received, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        stacks.forEach(([name, stack]) => {
            const opts = {
                cwd: path.resolve(received, stack.workingDirectory),
                env: process.env,
                stdio: "pipe",
            };
            child_process_1.execSync(`${terraformBinaryName} init`, opts);
            const out = child_process_1.execSync(`${terraformBinaryName} validate -json`, opts);
            const result = JSON.parse(out.toString());
            if (!result.valid) {
                throw new Error(`Found ${result.error_count} Errors in stack ${name}: ${result.diagnostics.join("\n")}`);
            }
        });
        return {
            pass: true,
            message: () => `Expected subject not to be a valid terraform stack`,
        };
    }
    catch (e) {
        return {
            pass: false,
            message: () => `Expected subject to be a valid terraform stack: ${e}`,
        };
    }
}
exports.toBeValidTerraform = toBeValidTerraform;
function toPlanSuccessfully(received) {
    try {
        if (!fs.statSync(received).isDirectory()) {
            throw new Error("Path is not a directory");
        }
    }
    catch (e) {
        return {
            message: () => `Expected subject to be a terraform directory: ${e}`,
            pass: false,
        };
    }
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(received, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        stacks.forEach(([, stack]) => {
            const opts = {
                cwd: path.resolve(received, stack.workingDirectory),
                env: process.env,
                stdio: "ignore",
            };
            child_process_1.execSync(`${terraformBinaryName} init`, opts);
            // Throws on a non-zero exit code
            child_process_1.execSync(`${terraformBinaryName} plan -input=false `, opts);
        });
        return {
            pass: true,
            message: () => `Expected subject not to plan successfully`,
        };
    }
    catch (e) {
        return {
            pass: false,
            message: () => `Expected subject to plan successfully: ${e}`,
        };
    }
}
exports.toPlanSuccessfully = toPlanSuccessfully;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWF0Y2hlcnMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJtYXRjaGVycy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLGlEQUF5QztBQUN6QyxrQ0FBb0M7QUFFcEMsTUFBTSxtQkFBbUIsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLHFCQUFxQixJQUFJLFdBQVcsQ0FBQztBQVk3RSw4REFBOEQ7QUFDOUQsNkdBQTZHO0FBQzdHLFNBQWdCLHNDQUFzQyxDQUNwRCxRQUFpQixFQUNqQixRQUFpQjtJQUVqQixRQUFRLE9BQU8sUUFBUSxFQUFFO1FBQ3ZCLEtBQUssUUFBUTtZQUNYLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRTtnQkFDM0IsT0FBTyxDQUNMLEtBQUssQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO29CQUN2QixRQUFRLENBQUMsTUFBTSxLQUFLLFFBQVEsQ0FBQyxNQUFNO29CQUNuQyxRQUFRLENBQUMsS0FBSyxDQUNaLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxFQUFFLENBQ2Qsc0NBQXNDLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLDZCQUE2QjtxQkFDOUYsQ0FDRixDQUFDO2FBQ0g7WUFDRCxJQUFJLFFBQVEsS0FBSyxJQUFJLElBQUksUUFBUSxLQUFLLElBQUksRUFBRTtnQkFDMUMsT0FBTyxJQUFJLENBQUM7YUFDYjtZQUNELElBQUksUUFBUSxLQUFLLFNBQVMsSUFBSSxRQUFRLEtBQUssU0FBUyxFQUFFO2dCQUNwRCxPQUFPLElBQUksQ0FBQzthQUNiO1lBQ0QsSUFBSSxRQUFRLEtBQUssSUFBSSxJQUFJLFFBQVEsS0FBSyxJQUFJLEVBQUU7Z0JBQzFDLE9BQU8sS0FBSyxDQUFDO2FBQ2Q7WUFFRCx5RUFBeUU7WUFDekUsT0FBTyxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQW1DLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtnQkFDcEUsSUFBSyxRQUFnQixDQUFDLEdBQUcsQ0FBQyxLQUFLLFNBQVMsRUFBRTtvQkFDeEMsT0FBTyxzQ0FBc0MsQ0FDMUMsUUFBZ0IsQ0FBQyxHQUFHLENBQUMsRUFDckIsUUFBZ0IsQ0FBQyxHQUFHLENBQUMsQ0FDdkIsQ0FBQztpQkFDSDtnQkFFRCxJQUFLLFFBQWdCLENBQUMsZ0JBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLFNBQVMsRUFBRTtvQkFDbkQsT0FBTyxzQ0FBc0MsQ0FDMUMsUUFBZ0IsQ0FBQyxHQUFHLENBQUMsRUFDckIsUUFBZ0IsQ0FBQyxnQkFBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQ2xDLENBQUM7aUJBQ0g7Z0JBRUQsT0FBTyxLQUFLLENBQUM7WUFDZixDQUFDLENBQUMsQ0FBQztRQUNMO1lBQ0UsT0FBTyxRQUFRLEtBQUssUUFBUSxDQUFDO0tBQ2hDO0FBQ0gsQ0FBQztBQS9DRCx3RkErQ0M7QUFDRCxNQUFNLHFCQUFxQixHQUFHLENBQzVCLEtBQVUsRUFDVixrQkFBdUMsRUFDdkMsRUFBRTtJQUNGLE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFTLEVBQUUsRUFBRSxDQUM3QyxzQ0FBc0MsQ0FBQyxrQkFBa0IsRUFBRSxJQUFJLENBQUMsQ0FDakUsQ0FBQztBQUNKLENBQUMsQ0FBQztBQUVGLFNBQVMsWUFBWSxDQUFDLEdBQVE7SUFDNUIsT0FBTyxDQUFDLENBQUMsR0FBRyxJQUFJLE9BQU8sR0FBRyxLQUFLLFFBQVEsSUFBSSxpQkFBaUIsSUFBSSxHQUFHLENBQUM7QUFDdEUsQ0FBQztBQUNELG1HQUFtRztBQUNuRyw4QkFBOEI7QUFDOUIsU0FBUyxxQ0FBcUMsQ0FBQyxJQUFZLEVBQUUsS0FBVTtJQUNyRSxPQUFPLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxLQUFLLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO0FBQ3BFLENBQUM7QUFDRCxTQUFTLDhCQUE4QjtBQUNyQyxrSEFBa0g7QUFDbEgsc0VBQXNFO0FBQ3RFLG9CQUdZO0lBRVosTUFBTSxjQUFjLEdBQUcsb0JBQW9CLElBQUkscUJBQXFCLENBQUM7SUFDckUsT0FBTyxTQUFTLDhCQUE4QixDQUM1QyxJQUE0QixFQUM1QixRQUFnQixFQUNoQixRQUE4QixFQUM5QixhQUFrQyxFQUFFOztRQUVwQyxJQUFJLEtBQXVCLENBQUM7UUFDNUIsSUFBSTtZQUNGLEtBQUssR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBcUIsQ0FBQztTQUNsRDtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQywrQkFBK0IsUUFBUSxFQUFFLENBQUMsQ0FBQztTQUM1RDtRQUVELE1BQU0sS0FBSyxHQUNULE1BQU0sQ0FBQyxNQUFNLENBQ1gsT0FBQSxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxnQ0FBZ0M7YUFDL0QsSUFBSTtRQUNILHVDQUF1QztRQUN2QyxDQUFDLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxJQUFJLEtBQUssUUFBUSxDQUFDLGNBQWMsQ0FDdEQsMENBQUcsQ0FBQyxNQUFLLEVBQUUsQ0FBQyxxRUFBcUU7U0FDckYsSUFBSSxFQUFFLENBQUMsQ0FBQyx5Q0FBeUM7UUFDcEQsTUFBTSxJQUFJLEdBQUcsY0FBYyxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztRQUUvQyxJQUFJLElBQUksRUFBRTtZQUNSLE9BQU87Z0JBQ0wsSUFBSTtnQkFDSixPQUFPLEVBQUUsR0FBRyxFQUFFLENBQ1osZUFDRSxRQUFRLENBQUMsY0FDWCxvQkFBb0IsSUFBSSxDQUFDLFNBQVMsQ0FDaEMsVUFBVSxFQUNWLHFDQUFxQyxDQUN0QztRQUNILEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLElBQ3BDLFFBQVEsQ0FBQyxjQUNYLHFCQUNFLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUM5RCxFQUFFO2FBQ0wsQ0FBQztTQUNIO2FBQU07WUFDTCxPQUFPO2dCQUNMLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FDWixZQUFZLFFBQVEsQ0FBQyxjQUFjLG9CQUFvQixJQUFJLENBQUMsU0FBUyxDQUNuRSxVQUFVLEVBQ1YscUNBQXFDLENBQ3RDO1FBQ0gsS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sSUFDcEMsUUFBUSxDQUFDLGNBQ1gscUJBQ0UsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQzlELEVBQUU7Z0JBQ0osSUFBSTthQUNMLENBQUM7U0FDSDtJQUNILENBQUMsQ0FBQztBQUNKLENBQUM7QUFFRCxTQUFnQixpQ0FBaUMsQ0FDL0Msb0JBR1k7SUFFWixPQUFPLFNBQVMsOEJBQThCLENBQzVDLFFBQWdCLEVBQ2hCLFlBQWtDLEVBQ2xDLGFBQWtDLEVBQUU7UUFFcEMsT0FBTyw4QkFBOEIsQ0FBQyxvQkFBb0IsQ0FBQyxDQUN6RCxNQUFNLEVBQ04sUUFBUSxFQUNSLFlBQVksRUFDWixVQUFVLENBQ1gsQ0FBQztJQUNKLENBQUMsQ0FBQztBQUNKLENBQUM7QUFsQkQsOEVBa0JDO0FBRUQsU0FBZ0IsK0JBQStCLENBQzdDLG9CQUdZO0lBRVosT0FBTyxTQUFTLDRCQUE0QixDQUMxQyxRQUFnQixFQUNoQixZQUFrQyxFQUNsQyxhQUFrQyxFQUFFO1FBRXBDLE9BQU8sOEJBQThCLENBQUMsb0JBQW9CLENBQUMsQ0FDekQsVUFBVSxFQUNWLFFBQVEsRUFDUixZQUFZLEVBQ1osVUFBVSxDQUNYLENBQUM7SUFDSixDQUFDLENBQUM7QUFDSixDQUFDO0FBbEJELDBFQWtCQztBQUNELFNBQWdCLGtCQUFrQixDQUFDLFFBQWdCO0lBQ2pELElBQUk7UUFDRixJQUFJLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRTtZQUN4QyxNQUFNLElBQUksS0FBSyxDQUFDLHlCQUF5QixDQUFDLENBQUM7U0FDNUM7S0FDRjtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsT0FBTztZQUNMLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxpREFBaUQsQ0FBQyxFQUFFO1lBQ25FLElBQUksRUFBRSxLQUFLO1NBQ1osQ0FBQztLQUNIO0lBRUQsSUFBSTtRQUNGLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQ3pCLEVBQUUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsZUFBZSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQ2pFLENBQUM7UUFFRixNQUFNLE1BQU0sR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUUvQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRTtZQUMvQixNQUFNLElBQUksR0FBRztnQkFDWCxHQUFHLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUcsS0FBYSxDQUFDLGdCQUFnQixDQUFDO2dCQUM1RCxHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUc7Z0JBQ2hCLEtBQUssRUFBRSxNQUFNO2FBQ1AsQ0FBQztZQUNULHdCQUFRLENBQUMsR0FBRyxtQkFBbUIsT0FBTyxFQUFFLElBQUksQ0FBQyxDQUFDO1lBQzlDLE1BQU0sR0FBRyxHQUFHLHdCQUFRLENBQUMsR0FBRyxtQkFBbUIsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFFcEUsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztZQUMxQyxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRTtnQkFDakIsTUFBTSxJQUFJLEtBQUssQ0FDYixTQUNFLE1BQU0sQ0FBQyxXQUNULG9CQUFvQixJQUFJLEtBQUssTUFBTSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FDN0QsQ0FBQzthQUNIO1FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPO1lBQ0wsSUFBSSxFQUFFLElBQUk7WUFDVixPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsb0RBQW9EO1NBQ3BFLENBQUM7S0FDSDtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsT0FBTztZQUNMLElBQUksRUFBRSxLQUFLO1lBQ1gsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLG1EQUFtRCxDQUFDLEVBQUU7U0FDdEUsQ0FBQztLQUNIO0FBQ0gsQ0FBQztBQWhERCxnREFnREM7QUFFRCxTQUFnQixrQkFBa0IsQ0FBQyxRQUFnQjtJQUNqRCxJQUFJO1FBQ0YsSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDeEMsTUFBTSxJQUFJLEtBQUssQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1NBQzVDO0tBQ0Y7SUFBQyxPQUFPLENBQUMsRUFBRTtRQUNWLE9BQU87WUFDTCxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsaURBQWlELENBQUMsRUFBRTtZQUNuRSxJQUFJLEVBQUUsS0FBSztTQUNaLENBQUM7S0FDSDtJQUVELElBQUk7UUFDRixNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUN6QixFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLGVBQWUsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUNqRSxDQUFDO1FBRUYsTUFBTSxNQUFNLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFL0MsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxLQUFLLENBQUMsRUFBRSxFQUFFO1lBQzNCLE1BQU0sSUFBSSxHQUFHO2dCQUNYLEdBQUcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRyxLQUFhLENBQUMsZ0JBQWdCLENBQUM7Z0JBQzVELEdBQUcsRUFBRSxPQUFPLENBQUMsR0FBRztnQkFDaEIsS0FBSyxFQUFFLFFBQVE7YUFDVCxDQUFDO1lBQ1Qsd0JBQVEsQ0FBQyxHQUFHLG1CQUFtQixPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFFOUMsaUNBQWlDO1lBQ2pDLHdCQUFRLENBQUMsR0FBRyxtQkFBbUIscUJBQXFCLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDOUQsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPO1lBQ0wsSUFBSSxFQUFFLElBQUk7WUFDVixPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsMkNBQTJDO1NBQzNELENBQUM7S0FDSDtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsT0FBTztZQUNMLElBQUksRUFBRSxLQUFLO1lBQ1gsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLDBDQUEwQyxDQUFDLEVBQUU7U0FDN0QsQ0FBQztLQUNIO0FBQ0gsQ0FBQztBQXpDRCxnREF5Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBmcyBmcm9tIFwiZnNcIjtcbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IGV4ZWNTeW5jIH0gZnJvbSBcImNoaWxkX3Byb2Nlc3NcIjtcbmltcG9ydCB7IHNuYWtlQ2FzZSB9IGZyb20gXCIuLi91dGlsXCI7XG5cbmNvbnN0IHRlcnJhZm9ybUJpbmFyeU5hbWUgPSBwcm9jZXNzLmVudi5URVJSQUZPUk1fQklOQVJZX05BTUUgfHwgXCJ0ZXJyYWZvcm1cIjtcblxuZXhwb3J0IGludGVyZmFjZSBUZXJyYWZvcm1Db25zdHJ1Y3RvciB7XG4gIG5ldyAoLi4uYXJnczogYW55W10pOiBhbnk7XG4gIHJlYWRvbmx5IHRmUmVzb3VyY2VUeXBlOiBzdHJpbmc7XG59XG5leHBvcnQgdHlwZSBTeW50aGVzaXplZFN0YWNrID0ge1xuICByZXNvdXJjZTogUmVjb3JkPHN0cmluZywgYW55PjtcbiAgZGF0YTogUmVjb3JkPHN0cmluZywgYW55Pjtcbn07XG5leHBvcnQgdHlwZSBNYXRjaGVyUmV0dXJuID0geyBtZXNzYWdlOiAoKSA9PiBzdHJpbmc7IHBhc3M6IGJvb2xlYW4gfTtcblxuLy8gQWxsIGV4cGVjdGVkIHByb3BlcnRpZXMgYXJlIG1hdGNoZWQgYW5kIGNvbnNpZGVyZWQgZXF1YWwgaWZcbi8vIFRoZXJlIGNhbiBiZSBtb3JlIHByb3BlcnRpZXMgaW4gdGhlIHJlY2VpdmVkIG9iamVjdCB0aGFuIGluIHRoZSBleHBlY3RlZCBvYmplY3Qgd2hpbGUgc3RpbGwgcmV0dXJuaW5nIHRydWVcbmV4cG9ydCBmdW5jdGlvbiBhc3ltZXRyaWNEZWVwRXF1YWxJZ25vcmluZ09iamVjdENhc2luZyhcbiAgZXhwZWN0ZWQ6IHVua25vd24sXG4gIHJlY2VpdmVkOiB1bmtub3duXG4pOiBib29sZWFuIHtcbiAgc3dpdGNoICh0eXBlb2YgZXhwZWN0ZWQpIHtcbiAgICBjYXNlIFwib2JqZWN0XCI6XG4gICAgICBpZiAoQXJyYXkuaXNBcnJheShleHBlY3RlZCkpIHtcbiAgICAgICAgcmV0dXJuIChcbiAgICAgICAgICBBcnJheS5pc0FycmF5KHJlY2VpdmVkKSAmJlxuICAgICAgICAgIGV4cGVjdGVkLmxlbmd0aCA9PT0gcmVjZWl2ZWQubGVuZ3RoICYmXG4gICAgICAgICAgZXhwZWN0ZWQuZXZlcnkoXG4gICAgICAgICAgICAoaXRlbSwgaW5kZXgpID0+XG4gICAgICAgICAgICAgIGFzeW1ldHJpY0RlZXBFcXVhbElnbm9yaW5nT2JqZWN0Q2FzaW5nKGl0ZW0sIHJlY2VpdmVkW2luZGV4XSkgLy8gcmVjdXJzaXZlbHkgY29tcGFyZSBhcnJheXNcbiAgICAgICAgICApXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgICBpZiAoZXhwZWN0ZWQgPT09IG51bGwgJiYgcmVjZWl2ZWQgPT09IG51bGwpIHtcbiAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICB9XG4gICAgICBpZiAoZXhwZWN0ZWQgPT09IHVuZGVmaW5lZCAmJiByZWNlaXZlZCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgfVxuICAgICAgaWYgKGV4cGVjdGVkID09PSBudWxsIHx8IHJlY2VpdmVkID09PSBudWxsKSB7XG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgIH1cblxuICAgICAgLy8gcmVjdXJzaXZlbHkgY29tcGFyZSBvYmplY3RzIGFuZCBhbGxvdyBzbmFrZSBjYXNlIGFzIHdlbGwgYXMgY2FtZWwgY2FzZVxuICAgICAgcmV0dXJuIE9iamVjdC5rZXlzKGV4cGVjdGVkIGFzIFJlY29yZDxzdHJpbmcsIHVua25vd24+KS5ldmVyeSgoa2V5KSA9PiB7XG4gICAgICAgIGlmICgocmVjZWl2ZWQgYXMgYW55KVtrZXldICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICByZXR1cm4gYXN5bWV0cmljRGVlcEVxdWFsSWdub3JpbmdPYmplY3RDYXNpbmcoXG4gICAgICAgICAgICAoZXhwZWN0ZWQgYXMgYW55KVtrZXldLFxuICAgICAgICAgICAgKHJlY2VpdmVkIGFzIGFueSlba2V5XVxuICAgICAgICAgICk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoKHJlY2VpdmVkIGFzIGFueSlbc25ha2VDYXNlKGtleSldICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICByZXR1cm4gYXN5bWV0cmljRGVlcEVxdWFsSWdub3JpbmdPYmplY3RDYXNpbmcoXG4gICAgICAgICAgICAoZXhwZWN0ZWQgYXMgYW55KVtrZXldLFxuICAgICAgICAgICAgKHJlY2VpdmVkIGFzIGFueSlbc25ha2VDYXNlKGtleSldXG4gICAgICAgICAgKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgIH0pO1xuICAgIGRlZmF1bHQ6XG4gICAgICByZXR1cm4gZXhwZWN0ZWQgPT09IHJlY2VpdmVkO1xuICB9XG59XG5jb25zdCBkZWZhdWx0UGFzc0V2YWx1YXRpb24gPSAoXG4gIGl0ZW1zOiBhbnksXG4gIGFzc2VydGVkUHJvcGVydGllczogUmVjb3JkPHN0cmluZywgYW55PlxuKSA9PiB7XG4gIHJldHVybiBPYmplY3QudmFsdWVzKGl0ZW1zKS5zb21lKChpdGVtOiBhbnkpID0+XG4gICAgYXN5bWV0cmljRGVlcEVxdWFsSWdub3JpbmdPYmplY3RDYXNpbmcoYXNzZXJ0ZWRQcm9wZXJ0aWVzLCBpdGVtKVxuICApO1xufTtcblxuZnVuY3Rpb24gaXNBc3ltbWV0cmljKG9iajogYW55KSB7XG4gIHJldHVybiAhIW9iaiAmJiB0eXBlb2Ygb2JqID09PSBcIm9iamVjdFwiICYmIFwiYXN5bW1ldHJpY01hdGNoXCIgaW4gb2JqO1xufVxuLy8gWW91IGNhbiB1c2UgZXhwZWN0LkFueXRoaW5nKCksIGV4cGVjdC5PYmplY3RDb250YWluaW5nLCBldGMgaW4gamVzdCwgdGhpcyBtYWtlcyBpdCBuaWNlciB0byByZWFkXG4vLyB3aGVuIHdlIHByaW50IGVycm9yIG1lc2FnZXNcbmZ1bmN0aW9uIGplc3RBc3ltZXRyaWNNYXRjaGVyU3RyaW5naWZ5UmVwbGFjZXIoX2tleTogc3RyaW5nLCB2YWx1ZTogYW55KSB7XG4gIHJldHVybiBpc0FzeW1tZXRyaWModmFsdWUpID8gYGV4cGVjdC4ke3ZhbHVlLnRvU3RyaW5nKCl9YCA6IHZhbHVlO1xufVxuZnVuY3Rpb24gZ2V0QXNzZXJ0RWxlbWVudFdpdGhQcm9wZXJ0aWVzKFxuICAvLyBXZSBoYXZlIHRoZSBldmFsdWF0aW9uIGZ1bmN0aW9uIGNvbmZpZ3VyYWJsZSBzbyB3ZSBjYW4gbWFrZSB1c2Ugb2YgdGhlIHNwZWNpZmljIHRlc3RpbmcgZnJhbWV3b3JrcyBjYXBhYmlsaXRpZXNcbiAgLy8gVGhpcyBtYWtlcyB0aGUgcmVzdWx0aW5nIHRlc3RzIG1vcmUgbmF0aXZlIHRvIHRoZSB0ZXN0aW5nIGZyYW1ld29ya1xuICBjdXN0b21QYXNzRXZhbHVhdGlvbj86IChcbiAgICBpdGVtczogYW55W10sIC8vIGNvbmZpZ3VyYXRpb25zIG9mIHRoZSByZXF1ZXN0ZWQgdHlwZVxuICAgIGFzc2VydGVkUHJvcGVydGllczogUmVjb3JkPHN0cmluZywgYW55PlxuICApID0+IGJvb2xlYW5cbikge1xuICBjb25zdCBwYXNzRXZhbHVhdGlvbiA9IGN1c3RvbVBhc3NFdmFsdWF0aW9uIHx8IGRlZmF1bHRQYXNzRXZhbHVhdGlvbjtcbiAgcmV0dXJuIGZ1bmN0aW9uIGdldEFzc2VydEVsZW1lbnRXaXRoUHJvcGVydGllcyhcbiAgICB0eXBlOiBrZXlvZiBTeW50aGVzaXplZFN0YWNrLFxuICAgIHJlY2VpdmVkOiBzdHJpbmcsXG4gICAgaXRlbVR5cGU6IFRlcnJhZm9ybUNvbnN0cnVjdG9yLFxuICAgIHByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT4gPSB7fVxuICApIHtcbiAgICBsZXQgc3RhY2s6IFN5bnRoZXNpemVkU3RhY2s7XG4gICAgdHJ5IHtcbiAgICAgIHN0YWNrID0gSlNPTi5wYXJzZShyZWNlaXZlZCkgYXMgU3ludGhlc2l6ZWRTdGFjaztcbiAgICB9IGNhdGNoIChlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGludmFsaWQgSlNPTiBzdHJpbmcgcGFzc2VkOiAke3JlY2VpdmVkfWApO1xuICAgIH1cblxuICAgIGNvbnN0IGl0ZW1zID1cbiAgICAgIE9iamVjdC52YWx1ZXMoXG4gICAgICAgIE9iamVjdC5lbnRyaWVzKHN0YWNrW3R5cGVdIHx8IHt9KSAvLyBmb3IgYWxsIGRhdGEvcmVzb3VyY2UgZW50cmllc1xuICAgICAgICAgIC5maW5kKFxuICAgICAgICAgICAgLy8gZmluZCB0aGUgb2JqZWN0IHdpdGggYSBtYXRjaGluZyBuYW1lXG4gICAgICAgICAgICAoW3R5cGUsIF92YWx1ZXNdKSA9PiB0eXBlID09PSBpdGVtVHlwZS50ZlJlc291cmNlVHlwZVxuICAgICAgICAgICk/LlsxXSB8fCB7fSAvLyBnZXQgYWxsIGl0ZW1zIG9mIHRoYXQgdHlwZSAoZW5jb2RlZCBhcyBhIHJlY29yZCBvZiBuYW1lIC0+IGNvbmZpZylcbiAgICAgICkgfHwgW107IC8vIGdldCBhIGxpc3Qgb2YgYWxsIGNvbmZpZ3Mgb2YgdGhhdCB0eXBlXG4gICAgY29uc3QgcGFzcyA9IHBhc3NFdmFsdWF0aW9uKGl0ZW1zLCBwcm9wZXJ0aWVzKTtcblxuICAgIGlmIChwYXNzKSB7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBwYXNzLFxuICAgICAgICBtZXNzYWdlOiAoKSA9PlxuICAgICAgICAgIGBFeHBlY3RlZCBubyAke1xuICAgICAgICAgICAgaXRlbVR5cGUudGZSZXNvdXJjZVR5cGVcbiAgICAgICAgICB9IHdpdGggcHJvcGVydGllcyAke0pTT04uc3RyaW5naWZ5KFxuICAgICAgICAgICAgcHJvcGVydGllcyxcbiAgICAgICAgICAgIGplc3RBc3ltZXRyaWNNYXRjaGVyU3RyaW5naWZ5UmVwbGFjZXJcbiAgICAgICAgICApfSB0byBiZSBwcmVzZW50IGluIHN5bnRoZXNpc2VkIHN0YWNrLlxuRm91bmQgJHtpdGVtcy5sZW5ndGggPT09IDAgPyBcIm5vXCIgOiBpdGVtcy5sZW5ndGh9ICR7XG4gICAgICAgICAgICBpdGVtVHlwZS50ZlJlc291cmNlVHlwZVxuICAgICAgICAgIH0gcmVzb3VyY2VzIGluc3RlYWQke1xuICAgICAgICAgICAgaXRlbXMubGVuZ3RoID4gMCA/IFwiOlxcblwiICsgSlNPTi5zdHJpbmdpZnkoaXRlbXMsIG51bGwsIDIpIDogXCJcIlxuICAgICAgICAgIH1gLFxuICAgICAgfTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgbWVzc2FnZTogKCkgPT5cbiAgICAgICAgICBgRXhwZWN0ZWQgJHtpdGVtVHlwZS50ZlJlc291cmNlVHlwZX0gd2l0aCBwcm9wZXJ0aWVzICR7SlNPTi5zdHJpbmdpZnkoXG4gICAgICAgICAgICBwcm9wZXJ0aWVzLFxuICAgICAgICAgICAgamVzdEFzeW1ldHJpY01hdGNoZXJTdHJpbmdpZnlSZXBsYWNlclxuICAgICAgICAgICl9IHRvIGJlIHByZXNlbnQgaW4gc3ludGhlc2lzZWQgc3RhY2suXG5Gb3VuZCAke2l0ZW1zLmxlbmd0aCA9PT0gMCA/IFwibm9cIiA6IGl0ZW1zLmxlbmd0aH0gJHtcbiAgICAgICAgICAgIGl0ZW1UeXBlLnRmUmVzb3VyY2VUeXBlXG4gICAgICAgICAgfSByZXNvdXJjZXMgaW5zdGVhZCR7XG4gICAgICAgICAgICBpdGVtcy5sZW5ndGggPiAwID8gXCI6XFxuXCIgKyBKU09OLnN0cmluZ2lmeShpdGVtcywgbnVsbCwgMikgOiBcIlwiXG4gICAgICAgICAgfWAsXG4gICAgICAgIHBhc3MsXG4gICAgICB9O1xuICAgIH1cbiAgfTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldFRvSGF2ZURhdGFTb3VyY2VXaXRoUHJvcGVydGllcyhcbiAgY3VzdG9tUGFzc0V2YWx1YXRpb24/OiAoXG4gICAgaXRlbXM6IGFueSxcbiAgICBhc3NlcnRlZFByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT5cbiAgKSA9PiBib29sZWFuXG4pIHtcbiAgcmV0dXJuIGZ1bmN0aW9uIHRvSGF2ZURhdGFTb3VyY2VXaXRoUHJvcGVydGllcyhcbiAgICByZWNlaXZlZDogc3RyaW5nLFxuICAgIHJlc291cmNlVHlwZTogVGVycmFmb3JtQ29uc3RydWN0b3IsXG4gICAgcHJvcGVydGllczogUmVjb3JkPHN0cmluZywgYW55PiA9IHt9XG4gICk6IE1hdGNoZXJSZXR1cm4ge1xuICAgIHJldHVybiBnZXRBc3NlcnRFbGVtZW50V2l0aFByb3BlcnRpZXMoY3VzdG9tUGFzc0V2YWx1YXRpb24pKFxuICAgICAgXCJkYXRhXCIsXG4gICAgICByZWNlaXZlZCxcbiAgICAgIHJlc291cmNlVHlwZSxcbiAgICAgIHByb3BlcnRpZXNcbiAgICApO1xuICB9O1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZ2V0VG9IYXZlUmVzb3VyY2VXaXRoUHJvcGVydGllcyhcbiAgY3VzdG9tUGFzc0V2YWx1YXRpb24/OiAoXG4gICAgaXRlbXM6IGFueSxcbiAgICBhc3NlcnRlZFByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT5cbiAgKSA9PiBib29sZWFuXG4pIHtcbiAgcmV0dXJuIGZ1bmN0aW9uIHRvSGF2ZVJlc291cmNlV2l0aFByb3BlcnRpZXMoXG4gICAgcmVjZWl2ZWQ6IHN0cmluZyxcbiAgICByZXNvdXJjZVR5cGU6IFRlcnJhZm9ybUNvbnN0cnVjdG9yLFxuICAgIHByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT4gPSB7fVxuICApOiBNYXRjaGVyUmV0dXJuIHtcbiAgICByZXR1cm4gZ2V0QXNzZXJ0RWxlbWVudFdpdGhQcm9wZXJ0aWVzKGN1c3RvbVBhc3NFdmFsdWF0aW9uKShcbiAgICAgIFwicmVzb3VyY2VcIixcbiAgICAgIHJlY2VpdmVkLFxuICAgICAgcmVzb3VyY2VUeXBlLFxuICAgICAgcHJvcGVydGllc1xuICAgICk7XG4gIH07XG59XG5leHBvcnQgZnVuY3Rpb24gdG9CZVZhbGlkVGVycmFmb3JtKHJlY2VpdmVkOiBzdHJpbmcpOiBNYXRjaGVyUmV0dXJuIHtcbiAgdHJ5IHtcbiAgICBpZiAoIWZzLnN0YXRTeW5jKHJlY2VpdmVkKS5pc0RpcmVjdG9yeSgpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJQYXRoIGlzIG5vdCBhIGRpcmVjdG9yeVwiKTtcbiAgICB9XG4gIH0gY2F0Y2ggKGUpIHtcbiAgICByZXR1cm4ge1xuICAgICAgbWVzc2FnZTogKCkgPT4gYEV4cGVjdGVkIHN1YmplY3QgdG8gYmUgYSB0ZXJyYWZvcm0gZGlyZWN0b3J5OiAke2V9YCxcbiAgICAgIHBhc3M6IGZhbHNlLFxuICAgIH07XG4gIH1cblxuICB0cnkge1xuICAgIGNvbnN0IG1hbmlmZXN0ID0gSlNPTi5wYXJzZShcbiAgICAgIGZzLnJlYWRGaWxlU3luYyhwYXRoLnJlc29sdmUocmVjZWl2ZWQsIFwibWFuaWZlc3QuanNvblwiKSwgXCJ1dGY4XCIpXG4gICAgKTtcblxuICAgIGNvbnN0IHN0YWNrcyA9IE9iamVjdC5lbnRyaWVzKG1hbmlmZXN0LnN0YWNrcyk7XG5cbiAgICBzdGFja3MuZm9yRWFjaCgoW25hbWUsIHN0YWNrXSkgPT4ge1xuICAgICAgY29uc3Qgb3B0cyA9IHtcbiAgICAgICAgY3dkOiBwYXRoLnJlc29sdmUocmVjZWl2ZWQsIChzdGFjayBhcyBhbnkpLndvcmtpbmdEaXJlY3RvcnkpLFxuICAgICAgICBlbnY6IHByb2Nlc3MuZW52LFxuICAgICAgICBzdGRpbzogXCJwaXBlXCIsXG4gICAgICB9IGFzIGFueTtcbiAgICAgIGV4ZWNTeW5jKGAke3RlcnJhZm9ybUJpbmFyeU5hbWV9IGluaXRgLCBvcHRzKTtcbiAgICAgIGNvbnN0IG91dCA9IGV4ZWNTeW5jKGAke3RlcnJhZm9ybUJpbmFyeU5hbWV9IHZhbGlkYXRlIC1qc29uYCwgb3B0cyk7XG5cbiAgICAgIGNvbnN0IHJlc3VsdCA9IEpTT04ucGFyc2Uob3V0LnRvU3RyaW5nKCkpO1xuICAgICAgaWYgKCFyZXN1bHQudmFsaWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIGBGb3VuZCAke1xuICAgICAgICAgICAgcmVzdWx0LmVycm9yX2NvdW50XG4gICAgICAgICAgfSBFcnJvcnMgaW4gc3RhY2sgJHtuYW1lfTogJHtyZXN1bHQuZGlhZ25vc3RpY3Muam9pbihcIlxcblwiKX1gXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICByZXR1cm4ge1xuICAgICAgcGFzczogdHJ1ZSxcbiAgICAgIG1lc3NhZ2U6ICgpID0+IGBFeHBlY3RlZCBzdWJqZWN0IG5vdCB0byBiZSBhIHZhbGlkIHRlcnJhZm9ybSBzdGFja2AsXG4gICAgfTtcbiAgfSBjYXRjaCAoZSkge1xuICAgIHJldHVybiB7XG4gICAgICBwYXNzOiBmYWxzZSxcbiAgICAgIG1lc3NhZ2U6ICgpID0+IGBFeHBlY3RlZCBzdWJqZWN0IHRvIGJlIGEgdmFsaWQgdGVycmFmb3JtIHN0YWNrOiAke2V9YCxcbiAgICB9O1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiB0b1BsYW5TdWNjZXNzZnVsbHkocmVjZWl2ZWQ6IHN0cmluZyk6IE1hdGNoZXJSZXR1cm4ge1xuICB0cnkge1xuICAgIGlmICghZnMuc3RhdFN5bmMocmVjZWl2ZWQpLmlzRGlyZWN0b3J5KCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIlBhdGggaXMgbm90IGEgZGlyZWN0b3J5XCIpO1xuICAgIH1cbiAgfSBjYXRjaCAoZSkge1xuICAgIHJldHVybiB7XG4gICAgICBtZXNzYWdlOiAoKSA9PiBgRXhwZWN0ZWQgc3ViamVjdCB0byBiZSBhIHRlcnJhZm9ybSBkaXJlY3Rvcnk6ICR7ZX1gLFxuICAgICAgcGFzczogZmFsc2UsXG4gICAgfTtcbiAgfVxuXG4gIHRyeSB7XG4gICAgY29uc3QgbWFuaWZlc3QgPSBKU09OLnBhcnNlKFxuICAgICAgZnMucmVhZEZpbGVTeW5jKHBhdGgucmVzb2x2ZShyZWNlaXZlZCwgXCJtYW5pZmVzdC5qc29uXCIpLCBcInV0ZjhcIilcbiAgICApO1xuXG4gICAgY29uc3Qgc3RhY2tzID0gT2JqZWN0LmVudHJpZXMobWFuaWZlc3Quc3RhY2tzKTtcblxuICAgIHN0YWNrcy5mb3JFYWNoKChbLCBzdGFja10pID0+IHtcbiAgICAgIGNvbnN0IG9wdHMgPSB7XG4gICAgICAgIGN3ZDogcGF0aC5yZXNvbHZlKHJlY2VpdmVkLCAoc3RhY2sgYXMgYW55KS53b3JraW5nRGlyZWN0b3J5KSxcbiAgICAgICAgZW52OiBwcm9jZXNzLmVudixcbiAgICAgICAgc3RkaW86IFwiaWdub3JlXCIsXG4gICAgICB9IGFzIGFueTtcbiAgICAgIGV4ZWNTeW5jKGAke3RlcnJhZm9ybUJpbmFyeU5hbWV9IGluaXRgLCBvcHRzKTtcblxuICAgICAgLy8gVGhyb3dzIG9uIGEgbm9uLXplcm8gZXhpdCBjb2RlXG4gICAgICBleGVjU3luYyhgJHt0ZXJyYWZvcm1CaW5hcnlOYW1lfSBwbGFuIC1pbnB1dD1mYWxzZSBgLCBvcHRzKTtcbiAgICB9KTtcblxuICAgIHJldHVybiB7XG4gICAgICBwYXNzOiB0cnVlLFxuICAgICAgbWVzc2FnZTogKCkgPT4gYEV4cGVjdGVkIHN1YmplY3Qgbm90IHRvIHBsYW4gc3VjY2Vzc2Z1bGx5YCxcbiAgICB9O1xuICB9IGNhdGNoIChlKSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHBhc3M6IGZhbHNlLFxuICAgICAgbWVzc2FnZTogKCkgPT4gYEV4cGVjdGVkIHN1YmplY3QgdG8gcGxhbiBzdWNjZXNzZnVsbHk6ICR7ZX1gLFxuICAgIH07XG4gIH1cbn1cbiJdfQ==