"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.call = exports.orOperation = exports.andOperation = exports.neqOperation = exports.eqOperation = exports.lteOperation = exports.ltOperation = exports.gteOperation = exports.gtOperation = exports.subOperation = exports.addOperation = exports.modOperation = exports.divOperation = exports.mulOperation = exports.negateOperation = exports.notOperation = exports.conditional = exports.propertyAccess = exports.insideTfExpression = exports.ref = exports.rawString = void 0;
const intrinsic_1 = require("./tokens/private/intrinsic");
const token_1 = require("./tokens/token");
const lazy_1 = require("./tokens/lazy");
const app_1 = require("./app");
const terraform_stack_1 = require("./terraform-stack");
class TFExpression extends intrinsic_1.Intrinsic {
    constructor() {
        super(...arguments);
        this.isInnerTerraformExpression = false;
    }
    resolveArg(context, arg) {
        const resolvedArg = context.resolve(arg);
        if (token_1.Tokenization.isResolvable(arg)) {
            return resolvedArg;
        }
        if (typeof arg === "string") {
            return this.resolveString(arg, resolvedArg);
        }
        if (Array.isArray(resolvedArg)) {
            return `[${resolvedArg
                .map((_, index) => this.resolveArg(context, arg[index]))
                .join(", ")}]`;
        }
        if (typeof resolvedArg === "object" && resolvedArg !== null) {
            return `{${Object.keys(resolvedArg)
                .map((key) => `${key} = ${this.resolveArg(context, arg[key])}`)
                .join(", ")}}`;
        }
        return resolvedArg;
    }
    /**
     * Escape string removes characters from the string that are not allowed in Terraform or JSON
     * It must only be used on non-token values
     */
    escapeString(str) {
        return str
            .replace(/\n/g, "\\n") // escape newlines
            .replace(/\${/g, "$$${"); // escape ${ to $${
    }
    resolveString(str, resolvedArg) {
        const tokenList = token_1.Tokenization.reverseString(str);
        const numberOfTokens = tokenList.tokens.length + tokenList.intrinsic.length;
        // String literal
        if (numberOfTokens === 0) {
            return resolvedArg !== `"` &&
                resolvedArg.startsWith('"') &&
                resolvedArg.endsWith('"')
                ? this.escapeString(resolvedArg)
                : `"${this.escapeString(resolvedArg)}"`;
        }
        // Only a token reference
        if (tokenList.literals.length === 0 && numberOfTokens === 1) {
            return resolvedArg;
        }
        // String literal + token reference combination
        return `"${tokenList.join({
            join: (left, right) => {
                const leftTokens = token_1.Tokenization.reverse(left);
                const leftTokenList = token_1.Tokenization.reverseString(left);
                const rightTokens = token_1.Tokenization.reverse(right);
                const leftTokenCount = leftTokenList.intrinsic.length + leftTokenList.tokens.length;
                // if left is mixed, needs to be left alone (because it's a result of a previous join iteration)
                let leftValue = left;
                // if left is a string literal, then we need to escape it
                if (leftTokenList.literals.length === 1 && leftTokenCount === 0) {
                    leftValue = this.escapeString(left);
                }
                // if left is only a token, needs to be wrapped as terraform expression
                if (leftTokenList.literals.length === 0 && leftTokenCount === 1) {
                    leftValue = `\${${leftTokens[0]}}`;
                }
                const rightValue = rightTokens.length === 0
                    ? this.escapeString(right)
                    : `\${${rightTokens[0]}}`;
                return `${leftValue}${rightValue}`;
            },
        })}"`;
    }
}
// A string that represents an input value to be escaped
class RawString extends TFExpression {
    constructor(str) {
        super(str);
        this.str = str;
    }
    resolve() {
        const qts = this.isInnerTerraformExpression ? `"` : ``;
        return `${qts}${this.escapeString(this.str).replace(/\"/g, '\\"')}${qts}`; // eslint-disable-line no-useless-escape
    }
    toString() {
        return this.str;
    }
}
function rawString(str) {
    return new RawString(str);
}
exports.rawString = rawString;
class Reference extends TFExpression {
    constructor(identifier, originStack) {
        super(identifier);
        this.identifier = identifier;
        this.originStack = originStack;
        /**
         * A single reference could be used in multiple stacks,
         * e.g. if we expose the ref directly or as token on the stack.
         * We need to store the identifier for each stack,
         * so that the resolved identifier string matches the stack it's resolved in.
         */
        this.crossStackIdentifier = {};
    }
    resolve(context) {
        // We check for cross stack references on preparation, setting a new identifier
        const resolutionStack = terraform_stack_1.TerraformStack.of(context.scope);
        const stackName = resolutionStack.toString();
        if (context.preparing) {
            // Cross stack reference
            if (this.originStack && this.originStack !== resolutionStack) {
                const app = app_1.App.of(this.originStack);
                const csr = app.crossStackReference(this.originStack, resolutionStack, this.identifier);
                if (this.isInnerTerraformExpression) {
                    markAsInner(csr);
                }
                this.crossStackIdentifier[stackName] = csr;
            }
        }
        // If this is a cross stack reference we will resolve to a reference within this stack.
        if (this.crossStackIdentifier[stackName]) {
            return this.crossStackIdentifier[stackName];
        }
        return this.isInnerTerraformExpression
            ? this.identifier
            : `\${${this.identifier}}`;
    }
}
function ref(identifier, stack) {
    return new Reference(identifier, stack);
}
exports.ref = ref;
function markAsInner(arg) {
    if (arg instanceof TFExpression) {
        arg.isInnerTerraformExpression = true;
        return;
    }
    // reverese tokens here and set inner tf expression flag
    token_1.Tokenization.reverse(arg).map((resolvable) => {
        if (resolvable instanceof TFExpression) {
            resolvable.isInnerTerraformExpression = true;
        }
        else if (resolvable instanceof lazy_1.LazyBase) {
            resolvable.addPostProcessor({
                postProcess: (value) => {
                    markAsInner(value);
                    return value;
                },
            });
        }
    });
    if (typeof arg === "object" && arg !== null) {
        if (Array.isArray(arg)) {
            arg.forEach(markAsInner);
        }
        else {
            Object.keys(arg).forEach((key) => markAsInner(arg[key]));
        }
    }
}
/**
 * marks the argument as being used in a terraform expression
 */
function insideTfExpression(arg) {
    markAsInner(arg);
    return arg;
}
exports.insideTfExpression = insideTfExpression;
class PropertyAccess extends TFExpression {
    constructor(target, args) {
        super({ target, args });
        this.target = target;
        this.args = args;
    }
    resolve(context) {
        markAsInner(this.target);
        this.args.forEach(markAsInner);
        const serializedArgs = this.args
            .map((arg) => this.resolveArg(context, arg))
            .map((a) => `[${a}]`) // property access
            .join("");
        const expr = `${this.resolveArg(context, this.target)}${serializedArgs}`;
        return this.isInnerTerraformExpression ? expr : `\${${expr}}`;
    }
}
function propertyAccess(target, args) {
    return new PropertyAccess(target, args);
}
exports.propertyAccess = propertyAccess;
class ConditionalExpression extends TFExpression {
    constructor(condition, trueValue, falseValue) {
        super({ condition, trueValue, falseValue });
        this.condition = condition;
        this.trueValue = trueValue;
        this.falseValue = falseValue;
    }
    resolve(context) {
        markAsInner(this.condition);
        markAsInner(this.trueValue);
        markAsInner(this.falseValue);
        const condition = this.resolveArg(context, this.condition);
        const trueValue = this.resolveArg(context, this.trueValue);
        const falseValue = this.resolveArg(context, this.falseValue);
        const expr = `${condition} ? ${trueValue} : ${falseValue}`;
        return this.isInnerTerraformExpression ? expr : `\${${expr}}`;
    }
}
/**
 * creates an expression like `true ? 1 : 0`
 */
function conditional(condition, trueValue, falseValue) {
    return new ConditionalExpression(condition, trueValue, falseValue);
}
exports.conditional = conditional;
class OperatorExpression extends TFExpression {
    constructor(operator, left, right // optional for ! and - operator
    ) {
        super({ operator, leftValue: left, rightValue: right });
        this.operator = operator;
        this.left = left;
        this.right = right;
    }
    resolve(context) {
        markAsInner(this.left);
        if (this.right)
            markAsInner(this.right);
        const left = this.resolveArg(context, this.left);
        const right = this.right ? this.resolveArg(context, this.right) : undefined;
        let expr = "";
        switch (this.operator) {
            case "!": {
                expr = `!${left}`; // no braces needed as ! has highest precedence
                break;
            }
            case "-": {
                if (right) {
                    // subtraction
                    expr = `(${left} - ${right})`;
                }
                else {
                    // negation
                    expr = `-${left}`; // no braces needed as - has highest precedence
                }
                break;
            }
            default: {
                expr = `(${left} ${this.operator} ${right})`;
            }
        }
        return this.isInnerTerraformExpression ? expr : `\${${expr}}`;
    }
}
function notOperation(expression) {
    return new OperatorExpression("!", expression);
}
exports.notOperation = notOperation;
function negateOperation(expression) {
    return new OperatorExpression("-", expression);
}
exports.negateOperation = negateOperation;
function mulOperation(left, right) {
    return new OperatorExpression("*", left, right);
}
exports.mulOperation = mulOperation;
function divOperation(left, right) {
    return new OperatorExpression("/", left, right);
}
exports.divOperation = divOperation;
function modOperation(left, right) {
    return new OperatorExpression("%", left, right);
}
exports.modOperation = modOperation;
function addOperation(left, right) {
    return new OperatorExpression("+", left, right);
}
exports.addOperation = addOperation;
function subOperation(left, right) {
    return new OperatorExpression("-", left, right);
}
exports.subOperation = subOperation;
function gtOperation(left, right) {
    return new OperatorExpression(">", left, right);
}
exports.gtOperation = gtOperation;
function gteOperation(left, right) {
    return new OperatorExpression(">=", left, right);
}
exports.gteOperation = gteOperation;
function ltOperation(left, right) {
    return new OperatorExpression("<", left, right);
}
exports.ltOperation = ltOperation;
function lteOperation(left, right) {
    return new OperatorExpression("<=", left, right);
}
exports.lteOperation = lteOperation;
function eqOperation(left, right) {
    return new OperatorExpression("==", left, right);
}
exports.eqOperation = eqOperation;
function neqOperation(left, right) {
    return new OperatorExpression("!=", left, right);
}
exports.neqOperation = neqOperation;
function andOperation(left, right) {
    return new OperatorExpression("&&", left, right);
}
exports.andOperation = andOperation;
function orOperation(left, right) {
    return new OperatorExpression("||", left, right);
}
exports.orOperation = orOperation;
class FunctionCall extends TFExpression {
    constructor(name, args) {
        super({ name, args });
        this.name = name;
        this.args = args;
    }
    resolve(context) {
        this.args.forEach(markAsInner);
        const serializedArgs = this.args
            .map((arg) => this.resolveArg(context, arg))
            .join(", ");
        const expr = `${this.name}(${serializedArgs})`;
        return this.isInnerTerraformExpression ? expr : `\${${expr}}`;
    }
}
function call(name, args) {
    return new FunctionCall(name, args);
}
exports.call = call;
//# sourceMappingURL=data:application/json;base64,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