"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractTokenDouble = exports.createTokenDouble = exports.unresolved = exports.containsComplexElement = exports.isComplexElement = exports.containsMapToken = exports.containsNumberListTokenElement = exports.containsStringListTokenElement = exports.NullConcat = exports.regexQuote = exports.TokenString = exports.VALID_KEY_CHARS = exports.END_TOKEN_MARKER = exports.BEGIN_MAP_TOKEN_MARKER = exports.BEGIN_LIST_TOKEN_MARKER = exports.BEGIN_STRING_TOKEN_MARKER = void 0;
const string_fragments_1 = require("../string-fragments");
const token_1 = require("../token");
// Details for encoding and decoding Tokens into native types; should not be exported
exports.BEGIN_STRING_TOKEN_MARKER = "${TfToken[";
exports.BEGIN_LIST_TOKEN_MARKER = "#{TfToken[";
exports.BEGIN_MAP_TOKEN_MARKER = "&{TfToken[";
exports.END_TOKEN_MARKER = "]}";
exports.VALID_KEY_CHARS = "a-zA-Z0-9:._-";
const QUOTED_BEGIN_STRING_TOKEN_MARKER = regexQuote(exports.BEGIN_STRING_TOKEN_MARKER);
const QUOTED_BEGIN_LIST_TOKEN_MARKER = regexQuote(exports.BEGIN_LIST_TOKEN_MARKER);
const QUOTED_BEGIN_MAP_TOKEN_MARKER = regexQuote(exports.BEGIN_MAP_TOKEN_MARKER);
const QUOTED_END_TOKEN_MARKER = regexQuote(exports.END_TOKEN_MARKER);
const STRING_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_STRING_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const LIST_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_LIST_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const MAP_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_MAP_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const NUMBER_TOKEN_REGEX = new RegExp("[-|\\d|.\\+]+(e[-|\\d|.|e|E|\\+]+)", "g");
/**
 * A string with markers in it that can be resolved to external values
 */
class TokenString {
    constructor(str, re, regexMatchIndex = 1) {
        this.str = str;
        this.re = re;
        this.regexMatchIndex = regexMatchIndex;
    }
    /**
     * Returns a `TokenString` for this string.
     */
    static forString(s) {
        return new TokenString(s, STRING_TOKEN_REGEX);
    }
    /**
     * Returns a `TokenString` for this string (must be the first string element of the list)
     */
    static forListToken(s) {
        return new TokenString(s, LIST_TOKEN_REGEX);
    }
    /**
     * Returns a `TokenString` for this string that handles encoded numbers
     */
    static forNumbers(s) {
        return new TokenString(s, NUMBER_TOKEN_REGEX, 0);
    }
    /**
     * Returns a `TokenString` for this string that handles encoded maps
     */
    static forMapToken(s) {
        return new TokenString(s, MAP_TOKEN_REGEX);
    }
    /**
     * Split string on markers, substituting markers with Tokens
     */
    split(lookup) {
        const ret = new string_fragments_1.TokenizedStringFragments();
        let rest = 0;
        this.re.lastIndex = 0; // Reset
        let m = this.re.exec(this.str);
        while (m) {
            if (m.index > rest) {
                ret.addLiteral(this.str.substring(rest, m.index));
            }
            const token = lookup(m[this.regexMatchIndex]);
            if (token) {
                ret.addToken(token);
            }
            else {
                ret.addLiteral(this.str.substring(m.index, this.re.lastIndex));
            }
            rest = this.re.lastIndex;
            m = this.re.exec(this.str);
        }
        if (rest < this.str.length) {
            ret.addLiteral(this.str.substring(rest));
        }
        return ret;
    }
    /**
     * Indicates if this string includes tokens.
     */
    test() {
        this.re.lastIndex = 0; // Reset
        return this.re.test(this.str);
    }
}
exports.TokenString = TokenString;
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, "\\$&");
}
exports.regexQuote = regexQuote;
/**
 * Concatenator that disregards the input
 *
 * Can be used when traversing the tokens is important, but the
 * result isn't.
 */
class NullConcat {
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    join(_left, _right) {
        return undefined;
    }
}
exports.NullConcat = NullConcat;
function containsStringListTokenElement(xs) {
    return xs.some((x) => typeof x === "string" && TokenString.forListToken(x).test());
}
exports.containsStringListTokenElement = containsStringListTokenElement;
function containsNumberListTokenElement(xs) {
    return xs.some((x) => typeof x === "number" && extractTokenDouble(x, true) !== undefined);
}
exports.containsNumberListTokenElement = containsNumberListTokenElement;
function containsMapToken(xs) {
    return Object.keys(xs).some((x) => typeof x === "string" && TokenString.forMapToken(x).test());
}
exports.containsMapToken = containsMapToken;
function isComplexElement(xs) {
    return (typeof xs === "object" &&
        xs !== null &&
        "withinArray" in xs &&
        "interpolationAsList" in xs);
}
exports.isComplexElement = isComplexElement;
function containsComplexElement(xs) {
    return xs.length > 0 && isComplexElement(xs[0]);
}
exports.containsComplexElement = containsComplexElement;
/**
 * Returns true if obj is a token (i.e. has the resolve() method or is a string
 * that includes token markers), or it's a listifictaion of a Token string.
 *
 * @param obj The object to test.
 */
function unresolved(obj) {
    if (typeof obj === "string") {
        return TokenString.forString(obj).test();
    }
    else if (typeof obj === "number") {
        return extractTokenDouble(obj, false) !== undefined;
    }
    else if (Array.isArray(obj) && obj.length === 1) {
        return ((typeof obj[0] === "string" && TokenString.forListToken(obj[0]).test()) ||
            (typeof obj[0] === "number" &&
                extractTokenDouble(obj[0], true) !== undefined));
    }
    else if (obj != null && containsMapToken(obj)) {
        return true;
    }
    else {
        return token_1.Tokenization.isResolvable(obj);
    }
}
exports.unresolved = unresolved;
/**
 * Bit pattern in the top 16 bits of a double to indicate a Token
 *
 * An IEEE double in LE memory order looks like this (grouped
 * into octets, then grouped into 32-bit words):
 *
 * mmmmmmmm.mmmmmmmm.mmmmmmmm.mmmmmmmm | mmmmmmmm.mmmmmmmm.EEEEmmmm.sEEEEEEE
 *
 * - m: mantissa (52 bits)
 * - E: exponent (11 bits)
 * - s: sign (1 bit)
 *
 * We put the following marker into the top 16 bits (exponent and sign), and
 * use the mantissa part to encode the token index. To save some bit twiddling
 * we use all top 16 bits for the tag. That loses us 4 mantissa bits to store
 * information in but we still have 48, which is going to be plenty for any
 * number of tokens to be created during the lifetime of any CDK application.
 *
 * Can't have all bits set because that makes a NaN, so unset the least
 * significant exponent bit.
 *
 * Currently not supporting BE architectures.
 */
// tslint:disable-next-line:no-bitwise
const DOUBLE_TOKEN_MARKER_BITS = 0xfdff << 16;
// tslint:disable-next-line:no-bitwise
const DOUBLE_LIST_TOKEN_MARKER_BITS = 0xfbff << 16;
/**
 * Highest encodable number
 */
const MAX_ENCODABLE_INTEGER = Math.pow(2, 48) - 1;
/**
 * Get 2^32 as a number, so we can do multiplication and div instead of bit shifting
 *
 * Necessary because in JavaScript, bit operations implicitly convert
 * to int32 and we need them to work on "int64"s.
 *
 * So instead of x >> 32, we do Math.floor(x / 2^32), and vice versa.
 */
const BITS32 = Math.pow(2, 32);
/**
 * Return a special Double value that encodes the given nonnegative integer
 *
 * We use this to encode Token ordinals.
 */
function createTokenDouble(x, list) {
    if (Math.floor(x) !== x || x < 0) {
        throw new Error("Can only encode positive integers");
    }
    if (x > MAX_ENCODABLE_INTEGER) {
        throw new Error(`Got an index too large to encode: ${x}`);
    }
    const buf = new ArrayBuffer(8);
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    ints[0] = x & 0x0000ffffffff; // Bottom 32 bits of number
    // This needs an "x >> 32" but that will make it a 32-bit number instead
    // of a 64-bit number.
    ints[1] =
        (shr32(x) & 0xffff) |
            (list ? DOUBLE_LIST_TOKEN_MARKER_BITS : DOUBLE_TOKEN_MARKER_BITS); // Top 16 bits of number and the mask
    // tslint:enable:no-bitwise
    return new Float64Array(buf)[0];
}
exports.createTokenDouble = createTokenDouble;
/**
 * Shift a 64-bit int right 32 bits
 */
function shr32(x) {
    return Math.floor(x / BITS32);
}
/**
 * Shift a 64-bit left 32 bits
 */
function shl32(x) {
    return x * BITS32;
}
/**
 * Extract the encoded integer out of the special Double value
 *
 * Returns undefined if the float is a not an encoded token.
 */
function extractTokenDouble(encoded, list) {
    const buf = new ArrayBuffer(8);
    new Float64Array(buf)[0] = encoded;
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    if ((ints[1] & 0xffff0000) !==
        (list ? DOUBLE_LIST_TOKEN_MARKER_BITS : DOUBLE_TOKEN_MARKER_BITS)) {
        return undefined;
    }
    // Must use + instead of | here (bitwise operations
    // will force 32-bits integer arithmetic, + will not).
    return ints[0] + shl32(ints[1] & 0xffff);
    // tslint:enable:no-bitwise
}
exports.extractTokenDouble = extractTokenDouble;
//# sourceMappingURL=data:application/json;base64,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