"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tokenization = exports.Token = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const lazy_1 = require("./lazy");
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    /**
     * Returns true if obj represents an unresolved value
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     */
    static isUnresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    static asString(value, options = {}) {
        if (typeof value === "string") {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token
     */
    static asNumber(value) {
        if (typeof value === "number") {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every((x) => typeof x === "string")) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible list representation of this token
     */
    static asNumberList(value) {
        if (Array.isArray(value) && value.every((x) => typeof x === "number")) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumberList(Token.asAny(value));
    }
    /**
     * Return a reversible map representation of this token
     */
    static asMap(value, mapValue, options = {}) {
        // since the return value is basically an object, just encode always
        return token_map_1.TokenMap.instance().registerMap(Token.asAny(value), mapValue, options.displayHint);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asStringMap(value, options = {}) {
        return this.asMap(value, Token.STRING_MAP_TOKEN_VALUE, options);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asNumberMap(value, options = {}) {
        return this.asMap(value, Token.NUMBER_MAP_TOKEN_VALUE, options);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asBooleanMap(value, options = {}) {
        return this.asMap(value, true, options);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asAnyMap(value, options = {}) {
        return this.asMap(value, Token.ANY_MAP_TOKEN_VALUE, options);
    }
    /**
     * Return a resolvable representation of the given value
     */
    static asAny(value) {
        return Tokenization.isResolvable(value) ? value : new intrinsic_1.Intrinsic(value);
    }
}
exports.Token = Token;
_a = JSII_RTTI_SYMBOL_1;
Token[_a] = { fqn: "cdktf.Token", version: "0.11.0-pre.4" };
Token.STRING_MAP_TOKEN_VALUE = "String Map Token Value";
Token.NUMBER_MAP_TOKEN_VALUE = -123456789;
Token.ANY_MAP_TOKEN_VALUE = "Any Map Token Value";
/**
 * Less oft-needed functions to manipulate Tokens
 */
class Tokenization {
    /**
     * Reverse any value into Resolvables, if possible
     */
    static reverse(x) {
        if (Tokenization.isResolvable(x)) {
            return [x];
        }
        if (typeof x === "string") {
            const reversedString = Tokenization.reverseString(x);
            return [...reversedString.tokens, ...reversedString.intrinsic];
        }
        if (Array.isArray(x)) {
            const reversedList = Tokenization.reverseList(x);
            if (reversedList) {
                return [reversedList];
            }
            const reversedNumberList = Tokenization.reverseNumberList(x);
            return reversedNumberList ? [reversedNumberList] : [];
        }
        if (typeof x === "number") {
            const reversedNumber = Tokenization.reverseNumber(x);
            return reversedNumber ? [reversedNumber] : [];
        }
        if (typeof x === "object" && x !== null) {
            const reversedMap = Tokenization.reverseMap(x);
            return reversedMap ? [reversedMap] : [];
        }
        return []; // null or undefined cannot be reversed
    }
    /**
     * Un-encode a string potentially containing encoded tokens
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a Tokenized value from a number
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseNumberList(l) {
        return token_map_1.TokenMap.instance().lookupNumberList(l);
    }
    /**
     * Un-encode a Tokenized value from a map
     */
    static reverseMap(m) {
        return token_map_1.TokenMap.instance().lookupMap(m);
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     */
    static resolve(obj, options) {
        return resolve_1.resolve(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: options.preparing !== undefined ? options.preparing : false,
        });
    }
    /**
     * Return whether the given object is an IResolvable object
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     */
    static isResolvable(obj) {
        return (typeof obj === "object" &&
            obj !== null &&
            typeof obj.resolve === "function");
    }
    /**
     * Stringify a number directly or lazily if it's a Token. If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.stringValue({
                produce: (context) => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== "number" ? resolved : `${resolved}`;
                },
            });
        }
        else {
            return typeof x !== "number" ? x : `${x}`;
        }
    }
}
exports.Tokenization = Tokenization;
_b = JSII_RTTI_SYMBOL_1;
Tokenization[_b] = { fqn: "cdktf.Tokenization", version: "0.11.0-pre.4" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9rZW4uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0b2tlbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUVBLGlDQUE4QjtBQUM5QixpREFBZ0Q7QUFDaEQsbURBQWdEO0FBQ2hELCtDQUE0QztBQUM1QyxtREFBK0M7QUFJL0M7Ozs7Ozs7OztHQVNHO0FBQ0gsTUFBYSxLQUFLO0lBQ2hCOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLEdBQVE7UUFDakMsT0FBTyxxQkFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFVLEVBQUUsVUFBMkIsRUFBRTtRQUM5RCxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsRUFBRTtZQUM3QixPQUFPLEtBQUssQ0FBQztTQUNkO1FBRUQsT0FBTyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGNBQWMsQ0FDdkMsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsRUFDbEIsT0FBTyxDQUFDLFdBQVcsQ0FDcEIsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBVTtRQUMvQixJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsRUFBRTtZQUM3QixPQUFPLEtBQUssQ0FBQztTQUNkO1FBQ0QsT0FBTyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDaEUsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFVLEVBQUUsVUFBMkIsRUFBRTtRQUM1RCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLEtBQUssUUFBUSxDQUFDLEVBQUU7WUFDckUsT0FBTyxLQUFLLENBQUM7U0FDZDtRQUNELE9BQU8sb0JBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxZQUFZLENBQ3JDLEtBQUssQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEVBQ2xCLE9BQU8sQ0FBQyxXQUFXLENBQ3BCLENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLEtBQVU7UUFDbkMsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxLQUFLLFFBQVEsQ0FBQyxFQUFFO1lBQ3JFLE9BQU8sS0FBSyxDQUFDO1NBQ2Q7UUFDRCxPQUFPLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBQ3BFLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQ2pCLEtBQVUsRUFDVixRQUFhLEVBQ2IsVUFBMkIsRUFBRTtRQUU3QixvRUFBb0U7UUFDcEUsT0FBTyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLFdBQVcsQ0FDcEMsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsRUFDbEIsUUFBUSxFQUNSLE9BQU8sQ0FBQyxXQUFXLENBQ3BCLENBQUM7SUFDSixDQUFDO0lBSUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsV0FBVyxDQUN2QixLQUFVLEVBQ1YsVUFBMkIsRUFBRTtRQUU3QixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxzQkFBc0IsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBSUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsV0FBVyxDQUN2QixLQUFVLEVBQ1YsVUFBMkIsRUFBRTtRQUU3QixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxzQkFBc0IsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUN4QixLQUFVLEVBQ1YsVUFBMkIsRUFBRTtRQUU3QixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBSUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUNwQixLQUFVLEVBQ1YsVUFBMkIsRUFBRTtRQUU3QixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsS0FBSyxDQUFDLEtBQVU7UUFDNUIsT0FBTyxZQUFZLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUkscUJBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN6RSxDQUFDOztBQTdJSCxzQkE4SUM7OztBQXBEd0IsNEJBQXNCLEdBQUcsd0JBQXdCLENBQUM7QUFZbEQsNEJBQXNCLEdBQUcsQ0FBQyxTQUFTLENBQUM7QUFzQnBDLHlCQUFtQixHQUFHLHFCQUFxQixDQUFDO0FBb0JyRTs7R0FFRztBQUNILE1BQWEsWUFBWTtJQUN2Qjs7T0FFRztJQUNJLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBTTtRQUMxQixJQUFJLFlBQVksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDaEMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ1o7UUFDRCxJQUFJLE9BQU8sQ0FBQyxLQUFLLFFBQVEsRUFBRTtZQUN6QixNQUFNLGNBQWMsR0FBRyxZQUFZLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3JELE9BQU8sQ0FBQyxHQUFHLGNBQWMsQ0FBQyxNQUFNLEVBQUUsR0FBRyxjQUFjLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDaEU7UUFDRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDcEIsTUFBTSxZQUFZLEdBQUcsWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNqRCxJQUFJLFlBQVksRUFBRTtnQkFDaEIsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDO2FBQ3ZCO1lBRUQsTUFBTSxrQkFBa0IsR0FBRyxZQUFZLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDN0QsT0FBTyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7U0FDdkQ7UUFDRCxJQUFJLE9BQU8sQ0FBQyxLQUFLLFFBQVEsRUFBRTtZQUN6QixNQUFNLGNBQWMsR0FBRyxZQUFZLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3JELE9BQU8sY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7U0FDL0M7UUFDRCxJQUFJLE9BQU8sQ0FBQyxLQUFLLFFBQVEsSUFBSSxDQUFDLEtBQUssSUFBSSxFQUFFO1lBQ3ZDLE1BQU0sV0FBVyxHQUFHLFlBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDL0MsT0FBTyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztTQUN6QztRQUVELE9BQU8sRUFBRSxDQUFDLENBQUMsdUNBQXVDO0lBQ3BELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBUztRQUNuQyxPQUFPLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBUztRQUNuQyxPQUFPLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFXO1FBQ25DLE9BQU8sb0JBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLGlCQUFpQixDQUFDLENBQVc7UUFDekMsT0FBTyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBeUI7UUFDaEQsT0FBTyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFRLEVBQUUsT0FBdUI7UUFDckQsT0FBTyxpQkFBTyxDQUFDLEdBQUcsRUFBRTtZQUNsQixLQUFLLEVBQUUsT0FBTyxDQUFDLEtBQUs7WUFDcEIsUUFBUSxFQUFFLE9BQU8sQ0FBQyxRQUFRO1lBQzFCLFNBQVMsRUFBRSxPQUFPLENBQUMsU0FBUyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSztTQUN2RSxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxHQUFRO1FBQ2pDLE9BQU8sQ0FDTCxPQUFPLEdBQUcsS0FBSyxRQUFRO1lBQ3ZCLEdBQUcsS0FBSyxJQUFJO1lBQ1osT0FBTyxHQUFHLENBQUMsT0FBTyxLQUFLLFVBQVUsQ0FDbEMsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxlQUFlLENBQUMsQ0FBUztRQUNyQywwSEFBMEg7UUFFMUgsSUFBSSxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ3pCLE9BQU8sV0FBSSxDQUFDLFdBQVcsQ0FBQztnQkFDdEIsT0FBTyxFQUFFLENBQUMsT0FBTyxFQUFFLEVBQUU7b0JBQ25CLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7b0JBQ3BDLE9BQU8sT0FBTyxRQUFRLEtBQUssUUFBUSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLEdBQUcsUUFBUSxFQUFFLENBQUM7Z0JBQ2pFLENBQUM7YUFDRixDQUFDLENBQUM7U0FDSjthQUFNO1lBQ0wsT0FBTyxPQUFPLENBQUMsS0FBSyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztTQUMzQztJQUNILENBQUM7O0FBbEhILG9DQW1IQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIGNvcGllZCBmcm9tIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvY29uc3RydWN0cy9ibG9iL2UwMWU0N2Y3OGVmMWU5YjYwMGVmY2QyM2ZmNzcwNWFhOGQzODQwMTcvbGliL3Rva2VuLnRzXG5pbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSBcImNvbnN0cnVjdHNcIjtcbmltcG9ydCB7IExhenkgfSBmcm9tIFwiLi9sYXp5XCI7XG5pbXBvcnQgeyB1bnJlc29sdmVkIH0gZnJvbSBcIi4vcHJpdmF0ZS9lbmNvZGluZ1wiO1xuaW1wb3J0IHsgSW50cmluc2ljIH0gZnJvbSBcIi4vcHJpdmF0ZS9pbnRyaW5zaWNcIjtcbmltcG9ydCB7IHJlc29sdmUgfSBmcm9tIFwiLi9wcml2YXRlL3Jlc29sdmVcIjtcbmltcG9ydCB7IFRva2VuTWFwIH0gZnJvbSBcIi4vcHJpdmF0ZS90b2tlbi1tYXBcIjtcbmltcG9ydCB7IElSZXNvbHZhYmxlLCBJVG9rZW5SZXNvbHZlciB9IGZyb20gXCIuL3Jlc29sdmFibGVcIjtcbmltcG9ydCB7IFRva2VuaXplZFN0cmluZ0ZyYWdtZW50cyB9IGZyb20gXCIuL3N0cmluZy1mcmFnbWVudHNcIjtcblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgc3BlY2lhbCBvciBsYXppbHktZXZhbHVhdGVkIHZhbHVlLlxuICpcbiAqIENhbiBiZSB1c2VkIHRvIGRlbGF5IGV2YWx1YXRpb24gb2YgYSBjZXJ0YWluIHZhbHVlIGluIGNhc2UsIGZvciBleGFtcGxlLFxuICogdGhhdCBpdCByZXF1aXJlcyBzb21lIGNvbnRleHQgb3IgbGF0ZS1ib3VuZCBkYXRhLiBDYW4gYWxzbyBiZSB1c2VkIHRvXG4gKiBtYXJrIHZhbHVlcyB0aGF0IG5lZWQgc3BlY2lhbCBwcm9jZXNzaW5nIGF0IGRvY3VtZW50IHJlbmRlcmluZyB0aW1lLlxuICpcbiAqIFRva2VucyBjYW4gYmUgZW1iZWRkZWQgaW50byBzdHJpbmdzIHdoaWxlIHJldGFpbmluZyB0aGVpciBvcmlnaW5hbFxuICogc2VtYW50aWNzLlxuICovXG5leHBvcnQgY2xhc3MgVG9rZW4ge1xuICAvKipcbiAgICogUmV0dXJucyB0cnVlIGlmIG9iaiByZXByZXNlbnRzIGFuIHVucmVzb2x2ZWQgdmFsdWVcbiAgICpcbiAgICogT25lIG9mIHRoZXNlIG11c3QgYmUgdHJ1ZTpcbiAgICpcbiAgICogLSBgb2JqYCBpcyBhbiBJUmVzb2x2YWJsZVxuICAgKiAtIGBvYmpgIGlzIGEgc3RyaW5nIGNvbnRhaW5pbmcgYXQgbGVhc3Qgb25lIGVuY29kZWQgYElSZXNvbHZhYmxlYFxuICAgKiAtIGBvYmpgIGlzIGVpdGhlciBhbiBlbmNvZGVkIG51bWJlciBvciBsaXN0XG4gICAqXG4gICAqIFRoaXMgZG9lcyBOT1QgcmVjdXJzZSBpbnRvIGxpc3RzIG9yIG9iamVjdHMgdG8gc2VlIGlmIHRoZXlcbiAgICogY29udGFpbmluZyByZXNvbHZhYmxlcy5cbiAgICpcbiAgICogQHBhcmFtIG9iaiBUaGUgb2JqZWN0IHRvIHRlc3QuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGlzVW5yZXNvbHZlZChvYmo6IGFueSk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB1bnJlc29sdmVkKG9iaik7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIGEgcmV2ZXJzaWJsZSBzdHJpbmcgcmVwcmVzZW50YXRpb24gb2YgdGhpcyB0b2tlblxuICAgKlxuICAgKiBJZiB0aGUgVG9rZW4gaXMgaW5pdGlhbGl6ZWQgd2l0aCBhIGxpdGVyYWwsIHRoZSBzdHJpbmdpZmllZCB2YWx1ZSBvZiB0aGVcbiAgICogbGl0ZXJhbCBpcyByZXR1cm5lZC4gT3RoZXJ3aXNlLCBhIHNwZWNpYWwgcXVvdGVkIHN0cmluZyByZXByZXNlbnRhdGlvblxuICAgKiBvZiB0aGUgVG9rZW4gaXMgcmV0dXJuZWQgdGhhdCBjYW4gYmUgZW1iZWRkZWQgaW50byBvdGhlciBzdHJpbmdzLlxuICAgKlxuICAgKiBTdHJpbmdzIHdpdGggcXVvdGVkIFRva2VucyBpbiB0aGVtIGNhbiBiZSByZXN0b3JlZCBiYWNrIGludG9cbiAgICogY29tcGxleCB2YWx1ZXMgd2l0aCB0aGUgVG9rZW5zIHJlc3RvcmVkIGJ5IGNhbGxpbmcgYHJlc29sdmUoKWBcbiAgICogb24gdGhlIHN0cmluZy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYXNTdHJpbmcodmFsdWU6IGFueSwgb3B0aW9uczogRW5jb2RpbmdPcHRpb25zID0ge30pOiBzdHJpbmcge1xuICAgIGlmICh0eXBlb2YgdmFsdWUgPT09IFwic3RyaW5nXCIpIHtcbiAgICAgIHJldHVybiB2YWx1ZTtcbiAgICB9XG5cbiAgICByZXR1cm4gVG9rZW5NYXAuaW5zdGFuY2UoKS5yZWdpc3RlclN0cmluZyhcbiAgICAgIFRva2VuLmFzQW55KHZhbHVlKSxcbiAgICAgIG9wdGlvbnMuZGlzcGxheUhpbnRcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBhIHJldmVyc2libGUgbnVtYmVyIHJlcHJlc2VudGF0aW9uIG9mIHRoaXMgdG9rZW5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYXNOdW1iZXIodmFsdWU6IGFueSk6IG51bWJlciB7XG4gICAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gXCJudW1iZXJcIikge1xuICAgICAgcmV0dXJuIHZhbHVlO1xuICAgIH1cbiAgICByZXR1cm4gVG9rZW5NYXAuaW5zdGFuY2UoKS5yZWdpc3Rlck51bWJlcihUb2tlbi5hc0FueSh2YWx1ZSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBhIHJldmVyc2libGUgbGlzdCByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHRva2VuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFzTGlzdCh2YWx1ZTogYW55LCBvcHRpb25zOiBFbmNvZGluZ09wdGlvbnMgPSB7fSk6IHN0cmluZ1tdIHtcbiAgICBpZiAoQXJyYXkuaXNBcnJheSh2YWx1ZSkgJiYgdmFsdWUuZXZlcnkoKHgpID0+IHR5cGVvZiB4ID09PSBcInN0cmluZ1wiKSkge1xuICAgICAgcmV0dXJuIHZhbHVlO1xuICAgIH1cbiAgICByZXR1cm4gVG9rZW5NYXAuaW5zdGFuY2UoKS5yZWdpc3Rlckxpc3QoXG4gICAgICBUb2tlbi5hc0FueSh2YWx1ZSksXG4gICAgICBvcHRpb25zLmRpc3BsYXlIaW50XG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSByZXZlcnNpYmxlIGxpc3QgcmVwcmVzZW50YXRpb24gb2YgdGhpcyB0b2tlblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhc051bWJlckxpc3QodmFsdWU6IGFueSk6IG51bWJlcltdIHtcbiAgICBpZiAoQXJyYXkuaXNBcnJheSh2YWx1ZSkgJiYgdmFsdWUuZXZlcnkoKHgpID0+IHR5cGVvZiB4ID09PSBcIm51bWJlclwiKSkge1xuICAgICAgcmV0dXJuIHZhbHVlO1xuICAgIH1cbiAgICByZXR1cm4gVG9rZW5NYXAuaW5zdGFuY2UoKS5yZWdpc3Rlck51bWJlckxpc3QoVG9rZW4uYXNBbnkodmFsdWUpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSByZXZlcnNpYmxlIG1hcCByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHRva2VuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFzTWFwKFxuICAgIHZhbHVlOiBhbnksXG4gICAgbWFwVmFsdWU6IGFueSxcbiAgICBvcHRpb25zOiBFbmNvZGluZ09wdGlvbnMgPSB7fVxuICApOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IHtcbiAgICAvLyBzaW5jZSB0aGUgcmV0dXJuIHZhbHVlIGlzIGJhc2ljYWxseSBhbiBvYmplY3QsIGp1c3QgZW5jb2RlIGFsd2F5c1xuICAgIHJldHVybiBUb2tlbk1hcC5pbnN0YW5jZSgpLnJlZ2lzdGVyTWFwKFxuICAgICAgVG9rZW4uYXNBbnkodmFsdWUpLFxuICAgICAgbWFwVmFsdWUsXG4gICAgICBvcHRpb25zLmRpc3BsYXlIaW50XG4gICAgKTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgU1RSSU5HX01BUF9UT0tFTl9WQUxVRSA9IFwiU3RyaW5nIE1hcCBUb2tlbiBWYWx1ZVwiO1xuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSByZXZlcnNpYmxlIG1hcCByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHRva2VuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFzU3RyaW5nTWFwKFxuICAgIHZhbHVlOiBhbnksXG4gICAgb3B0aW9uczogRW5jb2RpbmdPcHRpb25zID0ge31cbiAgKTogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSB7XG4gICAgcmV0dXJuIHRoaXMuYXNNYXAodmFsdWUsIFRva2VuLlNUUklOR19NQVBfVE9LRU5fVkFMVUUsIG9wdGlvbnMpO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBOVU1CRVJfTUFQX1RPS0VOX1ZBTFVFID0gLTEyMzQ1Njc4OTtcblxuICAvKipcbiAgICogUmV0dXJuIGEgcmV2ZXJzaWJsZSBtYXAgcmVwcmVzZW50YXRpb24gb2YgdGhpcyB0b2tlblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhc051bWJlck1hcChcbiAgICB2YWx1ZTogYW55LFxuICAgIG9wdGlvbnM6IEVuY29kaW5nT3B0aW9ucyA9IHt9XG4gICk6IHsgW2tleTogc3RyaW5nXTogbnVtYmVyIH0ge1xuICAgIHJldHVybiB0aGlzLmFzTWFwKHZhbHVlLCBUb2tlbi5OVU1CRVJfTUFQX1RPS0VOX1ZBTFVFLCBvcHRpb25zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSByZXZlcnNpYmxlIG1hcCByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHRva2VuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFzQm9vbGVhbk1hcChcbiAgICB2YWx1ZTogYW55LFxuICAgIG9wdGlvbnM6IEVuY29kaW5nT3B0aW9ucyA9IHt9XG4gICk6IHsgW2tleTogc3RyaW5nXTogYm9vbGVhbiB9IHtcbiAgICByZXR1cm4gdGhpcy5hc01hcCh2YWx1ZSwgdHJ1ZSwgb3B0aW9ucyk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEFOWV9NQVBfVE9LRU5fVkFMVUUgPSBcIkFueSBNYXAgVG9rZW4gVmFsdWVcIjtcblxuICAvKipcbiAgICogUmV0dXJuIGEgcmV2ZXJzaWJsZSBtYXAgcmVwcmVzZW50YXRpb24gb2YgdGhpcyB0b2tlblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhc0FueU1hcChcbiAgICB2YWx1ZTogYW55LFxuICAgIG9wdGlvbnM6IEVuY29kaW5nT3B0aW9ucyA9IHt9XG4gICk6IHsgW2tleTogc3RyaW5nXTogYW55IH0ge1xuICAgIHJldHVybiB0aGlzLmFzTWFwKHZhbHVlLCBUb2tlbi5BTllfTUFQX1RPS0VOX1ZBTFVFLCBvcHRpb25zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSByZXNvbHZhYmxlIHJlcHJlc2VudGF0aW9uIG9mIHRoZSBnaXZlbiB2YWx1ZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhc0FueSh2YWx1ZTogYW55KTogSVJlc29sdmFibGUge1xuICAgIHJldHVybiBUb2tlbml6YXRpb24uaXNSZXNvbHZhYmxlKHZhbHVlKSA/IHZhbHVlIDogbmV3IEludHJpbnNpYyh2YWx1ZSk7XG4gIH1cbn1cblxuLyoqXG4gKiBMZXNzIG9mdC1uZWVkZWQgZnVuY3Rpb25zIHRvIG1hbmlwdWxhdGUgVG9rZW5zXG4gKi9cbmV4cG9ydCBjbGFzcyBUb2tlbml6YXRpb24ge1xuICAvKipcbiAgICogUmV2ZXJzZSBhbnkgdmFsdWUgaW50byBSZXNvbHZhYmxlcywgaWYgcG9zc2libGVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmV2ZXJzZSh4OiBhbnkpOiBJUmVzb2x2YWJsZVtdIHtcbiAgICBpZiAoVG9rZW5pemF0aW9uLmlzUmVzb2x2YWJsZSh4KSkge1xuICAgICAgcmV0dXJuIFt4XTtcbiAgICB9XG4gICAgaWYgKHR5cGVvZiB4ID09PSBcInN0cmluZ1wiKSB7XG4gICAgICBjb25zdCByZXZlcnNlZFN0cmluZyA9IFRva2VuaXphdGlvbi5yZXZlcnNlU3RyaW5nKHgpO1xuICAgICAgcmV0dXJuIFsuLi5yZXZlcnNlZFN0cmluZy50b2tlbnMsIC4uLnJldmVyc2VkU3RyaW5nLmludHJpbnNpY107XG4gICAgfVxuICAgIGlmIChBcnJheS5pc0FycmF5KHgpKSB7XG4gICAgICBjb25zdCByZXZlcnNlZExpc3QgPSBUb2tlbml6YXRpb24ucmV2ZXJzZUxpc3QoeCk7XG4gICAgICBpZiAocmV2ZXJzZWRMaXN0KSB7XG4gICAgICAgIHJldHVybiBbcmV2ZXJzZWRMaXN0XTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgcmV2ZXJzZWROdW1iZXJMaXN0ID0gVG9rZW5pemF0aW9uLnJldmVyc2VOdW1iZXJMaXN0KHgpO1xuICAgICAgcmV0dXJuIHJldmVyc2VkTnVtYmVyTGlzdCA/IFtyZXZlcnNlZE51bWJlckxpc3RdIDogW107XG4gICAgfVxuICAgIGlmICh0eXBlb2YgeCA9PT0gXCJudW1iZXJcIikge1xuICAgICAgY29uc3QgcmV2ZXJzZWROdW1iZXIgPSBUb2tlbml6YXRpb24ucmV2ZXJzZU51bWJlcih4KTtcbiAgICAgIHJldHVybiByZXZlcnNlZE51bWJlciA/IFtyZXZlcnNlZE51bWJlcl0gOiBbXTtcbiAgICB9XG4gICAgaWYgKHR5cGVvZiB4ID09PSBcIm9iamVjdFwiICYmIHggIT09IG51bGwpIHtcbiAgICAgIGNvbnN0IHJldmVyc2VkTWFwID0gVG9rZW5pemF0aW9uLnJldmVyc2VNYXAoeCk7XG4gICAgICByZXR1cm4gcmV2ZXJzZWRNYXAgPyBbcmV2ZXJzZWRNYXBdIDogW107XG4gICAgfVxuXG4gICAgcmV0dXJuIFtdOyAvLyBudWxsIG9yIHVuZGVmaW5lZCBjYW5ub3QgYmUgcmV2ZXJzZWRcbiAgfVxuXG4gIC8qKlxuICAgKiBVbi1lbmNvZGUgYSBzdHJpbmcgcG90ZW50aWFsbHkgY29udGFpbmluZyBlbmNvZGVkIHRva2Vuc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZXZlcnNlU3RyaW5nKHM6IHN0cmluZyk6IFRva2VuaXplZFN0cmluZ0ZyYWdtZW50cyB7XG4gICAgcmV0dXJuIFRva2VuTWFwLmluc3RhbmNlKCkuc3BsaXRTdHJpbmcocyk7XG4gIH1cblxuICAvKipcbiAgICogVW4tZW5jb2RlIGEgVG9rZW5pemVkIHZhbHVlIGZyb20gYSBudW1iZXJcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmV2ZXJzZU51bWJlcihuOiBudW1iZXIpOiBJUmVzb2x2YWJsZSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIFRva2VuTWFwLmluc3RhbmNlKCkubG9va3VwTnVtYmVyVG9rZW4obik7XG4gIH1cblxuICAvKipcbiAgICogVW4tZW5jb2RlIGEgVG9rZW5pemVkIHZhbHVlIGZyb20gYSBsaXN0XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJldmVyc2VMaXN0KGw6IHN0cmluZ1tdKTogSVJlc29sdmFibGUgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiBUb2tlbk1hcC5pbnN0YW5jZSgpLmxvb2t1cExpc3QobCk7XG4gIH1cblxuICAvKipcbiAgICogVW4tZW5jb2RlIGEgVG9rZW5pemVkIHZhbHVlIGZyb20gYSBsaXN0XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJldmVyc2VOdW1iZXJMaXN0KGw6IG51bWJlcltdKTogSVJlc29sdmFibGUgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiBUb2tlbk1hcC5pbnN0YW5jZSgpLmxvb2t1cE51bWJlckxpc3QobCk7XG4gIH1cblxuICAvKipcbiAgICogVW4tZW5jb2RlIGEgVG9rZW5pemVkIHZhbHVlIGZyb20gYSBtYXBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmV2ZXJzZU1hcChtOiB7IFtrZXk6IHN0cmluZ106IGFueSB9KTogSVJlc29sdmFibGUgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiBUb2tlbk1hcC5pbnN0YW5jZSgpLmxvb2t1cE1hcChtKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXNvbHZlcyBhbiBvYmplY3QgYnkgZXZhbHVhdGluZyBhbGwgdG9rZW5zIGFuZCByZW1vdmluZyBhbnkgdW5kZWZpbmVkIG9yIGVtcHR5IG9iamVjdHMgb3IgYXJyYXlzLlxuICAgKiBWYWx1ZXMgY2FuIG9ubHkgYmUgcHJpbWl0aXZlcywgYXJyYXlzIG9yIHRva2Vucy4gT3RoZXIgb2JqZWN0cyAoaS5lLiB3aXRoIG1ldGhvZHMpIHdpbGwgYmUgcmVqZWN0ZWQuXG4gICAqXG4gICAqIEBwYXJhbSBvYmogVGhlIG9iamVjdCB0byByZXNvbHZlLlxuICAgKiBAcGFyYW0gb3B0aW9ucyBQcmVmaXgga2V5IHBhdGggY29tcG9uZW50cyBmb3IgZGlhZ25vc3RpY3MuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlc29sdmUob2JqOiBhbnksIG9wdGlvbnM6IFJlc29sdmVPcHRpb25zKTogYW55IHtcbiAgICByZXR1cm4gcmVzb2x2ZShvYmosIHtcbiAgICAgIHNjb3BlOiBvcHRpb25zLnNjb3BlLFxuICAgICAgcmVzb2x2ZXI6IG9wdGlvbnMucmVzb2x2ZXIsXG4gICAgICBwcmVwYXJpbmc6IG9wdGlvbnMucHJlcGFyaW5nICE9PSB1bmRlZmluZWQgPyBvcHRpb25zLnByZXBhcmluZyA6IGZhbHNlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB3aGV0aGVyIHRoZSBnaXZlbiBvYmplY3QgaXMgYW4gSVJlc29sdmFibGUgb2JqZWN0XG4gICAqXG4gICAqIFRoaXMgaXMgZGlmZmVyZW50IGZyb20gVG9rZW4uaXNVbnJlc29sdmVkKCkgd2hpY2ggd2lsbCBhbHNvIGNoZWNrIGZvclxuICAgKiBlbmNvZGVkIFRva2Vucywgd2hlcmVhcyB0aGlzIG1ldGhvZCB3aWxsIG9ubHkgZG8gYSB0eXBlIGNoZWNrIG9uIHRoZSBnaXZlblxuICAgKiBvYmplY3QuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGlzUmVzb2x2YWJsZShvYmo6IGFueSk6IG9iaiBpcyBJUmVzb2x2YWJsZSB7XG4gICAgcmV0dXJuIChcbiAgICAgIHR5cGVvZiBvYmogPT09IFwib2JqZWN0XCIgJiZcbiAgICAgIG9iaiAhPT0gbnVsbCAmJlxuICAgICAgdHlwZW9mIG9iai5yZXNvbHZlID09PSBcImZ1bmN0aW9uXCJcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0cmluZ2lmeSBhIG51bWJlciBkaXJlY3RseSBvciBsYXppbHkgaWYgaXQncyBhIFRva2VuLiBJZiBpdCBpcyBhbiBvYmplY3QgKGkuZS4sIHsgUmVmOiAnU29tZUxvZ2ljYWxJZCcgfSksIHJldHVybiBpdCBhcy1pcy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgc3RyaW5naWZ5TnVtYmVyKHg6IG51bWJlcikge1xuICAgIC8vIG9ubHkgY29udmVydCBudW1iZXJzIHRvIHN0cmluZ3Mgc28gdGhhdCBSZWZzLCBjb25kaXRpb25zLCBhbmQgb3RoZXIgdGhpbmdzIGRvbid0IGVuZCB1cCBzeW50aGVzaXppbmcgYXMgW29iamVjdCBvYmplY3RdXG5cbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKHgpKSB7XG4gICAgICByZXR1cm4gTGF6eS5zdHJpbmdWYWx1ZSh7XG4gICAgICAgIHByb2R1Y2U6IChjb250ZXh0KSA9PiB7XG4gICAgICAgICAgY29uc3QgcmVzb2x2ZWQgPSBjb250ZXh0LnJlc29sdmUoeCk7XG4gICAgICAgICAgcmV0dXJuIHR5cGVvZiByZXNvbHZlZCAhPT0gXCJudW1iZXJcIiA/IHJlc29sdmVkIDogYCR7cmVzb2x2ZWR9YDtcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gdHlwZW9mIHggIT09IFwibnVtYmVyXCIgPyB4IDogYCR7eH1gO1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gdGhlIHJlc29sdmUoKSBvcGVyYXRpb25cbiAqXG4gKiBOT1QgdGhlIHNhbWUgYXMgdGhlIFJlc29sdmVDb250ZXh0OyBSZXNvbHZlQ29udGV4dCBpcyBleHBvc2VkIHRvIFRva2VuXG4gKiBpbXBsZW1lbnRvcnMgYW5kIHJlc29sdXRpb24gaG9va3MsIHdoZXJlYXMgdGhpcyBzdHJ1Y3QgaXMganVzdCB0byBidW5kbGVcbiAqIGEgbnVtYmVyIG9mIHRoaW5ncyB0aGF0IHdvdWxkIG90aGVyd2lzZSBiZSBhcmd1bWVudHMgdG8gcmVzb2x2ZSgpIGluIGFcbiAqIHJlYWRhYmxlIHdheS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZXNvbHZlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgc2NvcGUgZnJvbSB3aGljaCByZXNvbHV0aW9uIGlzIHBlcmZvcm1lZFxuICAgKi9cbiAgcmVhZG9ubHkgc2NvcGU6IElDb25zdHJ1Y3Q7XG5cbiAgLyoqXG4gICAqIFRoZSByZXNvbHZlciB0byBhcHBseSB0byBhbnkgcmVzb2x2YWJsZSB0b2tlbnMgZm91bmRcbiAgICovXG4gIHJlYWRvbmx5IHJlc29sdmVyOiBJVG9rZW5SZXNvbHZlcjtcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgcmVzb2x1dGlvbiBpcyBiZWluZyBleGVjdXRlZCBkdXJpbmcgdGhlIHByZXBhcmUgcGhhc2Ugb3Igbm90LlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJlcGFyaW5nPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRvIHN0cmluZyBlbmNvZGluZ3NcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbmNvZGluZ09wdGlvbnMge1xuICAvKipcbiAgICogQSBoaW50IGZvciB0aGUgVG9rZW4ncyBwdXJwb3NlIHdoZW4gc3RyaW5naWZ5aW5nIGl0XG4gICAqIEBkZWZhdWx0IC0gbm8gZGlzcGxheSBoaW50XG4gICAqL1xuICByZWFkb25seSBkaXNwbGF5SGludD86IHN0cmluZztcbn1cbiJdfQ==