"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidateProviderPresence = void 0;
const constructs_1 = require("constructs");
const terraform_provider_1 = require("../terraform-provider");
const terraform_resource_1 = require("../terraform-resource");
const terraform_data_source_1 = require("../terraform-data-source");
class ValidateProviderPresence {
    constructor(host) {
        this.host = host;
        this.providerNames = new Set();
        this.foundProviders = [];
    }
    check(node) {
        if (node instanceof terraform_resource_1.TerraformResource ||
            node instanceof terraform_data_source_1.TerraformDataSource) {
            if (node.terraformGeneratorMetadata) {
                this.providerNames.add(node.terraformGeneratorMetadata.providerName);
            }
        }
        if (node instanceof terraform_provider_1.TerraformProvider) {
            this.foundProviders.push(node);
        }
        for (const child of constructs_1.Node.of(node).children) {
            this.check(child);
        }
    }
    validate() {
        this.check(this.host);
        const missingProviders = Array.from(this.providerNames).filter((name) => {
            return !this.foundProviders.some((p) => { var _a; return ((_a = p.terraformGeneratorMetadata) === null || _a === void 0 ? void 0 : _a.providerName) === name; });
        });
        if (missingProviders.length === 0) {
            return [];
        }
        else {
            return [
                `Found resources without a matching provider. Please make sure to add the following providers to your stack: ${missingProviders.join(", ")}`,
            ];
        }
    }
}
exports.ValidateProviderPresence = ValidateProviderPresence;
//# sourceMappingURL=data:application/json;base64,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