import logging
import os
import subprocess
import sys

from setuptools import Command, find_packages, setup

# Dependencies are now specified in pyproject.toml


# Read the contents of your README file
this_directory = os.path.abspath(os.path.dirname(__file__))
with open(os.path.join(this_directory, "README.md"), encoding="utf-8") as f:
    long_description = f.read()

VERSION = "1.2.0"  # Define version in a variable for reuse

# Get git commit hash
git_hash = "unknown"

# First check if GitLab CI provides the commit SHA
if os.environ.get("CI_COMMIT_SHA"):
    git_hash = os.environ.get("CI_COMMIT_SHA")
# Otherwise try to get it from git directly
else:
    try:
        git_hash = subprocess.check_output(["git", "rev-parse", "HEAD"]).decode("ascii").strip()
    except (subprocess.SubprocessError, FileNotFoundError):
        # Keep default "unknown"
        pass

# Generate version.py from template
template_path = os.path.join(this_directory, "openssl_encrypt/version.py.template")
version_path = os.path.join(this_directory, "openssl_encrypt/version.py")

if os.path.exists(template_path):
    with open(template_path, "r") as template_file:
        template_content = template_file.read()

    # Replace placeholders
    content = template_content.replace("${VERSION}", VERSION).replace("${GIT_COMMIT}", git_hash)

    with open(version_path, "w") as version_file:
        version_file.write(content)
else:
    # Fallback if template doesn't exist
    with open(version_path, "w") as f:
        f.write(
            f"""# This file is automatically generated by setup.py
__version__ = "{VERSION}"
__git_commit__ = "{git_hash}"
"""
        )


# Custom install command that runs setup_whirlpool after installation
class PostInstallCommand(Command):
    description = "Command to run post install tasks"
    user_options = []

    def initialize_options(self):
        pass

    def finalize_options(self):
        pass

    def run(self):
        try:
            # Run the post-install script
            import subprocess

            subprocess.check_call([sys.executable, "-m", "openssl_encrypt.post_install"])
        except Exception as e:
            print(f"Warning: Failed to run post-install setup: {e}")
            print("You may need to manually install Whirlpool: pip install whirlpool-py311")


setup(
    cmdclass={
        "post_install": PostInstallCommand,
    },
    name="openssl_encrypt",
    version=VERSION,
    # Read requirements from requirements-prod.txt
    install_requires=[
        line.strip()
        for line in open("requirements-prod.txt")
        if line.strip() and not line.startswith("#") and not line.startswith("-")
    ],
    entry_points={
        "console_scripts": [
            "openssl-encrypt=openssl_encrypt.cli:main",
            "whirlpool-setup=openssl_encrypt.modules.setup_whirlpool:setup_whirlpool",
        ],
    },
    # Read dev requirements from requirements-dev.txt
    extras_require={
        "dev": [
            line.strip()
            for line in open("requirements-dev.txt")
            if line.strip()
            and not line.startswith("#")
            and not line.startswith("-")
            and line.strip()
            not in [
                l.strip()
                for l in open("requirements-prod.txt")
                if l.strip() and not l.startswith("#") and not l.startswith("-")
            ]
        ],
    },
    project_urls={
        "Bug Tracker": "https://gitlab.rm-rf.ch/world/openssl_encrypt/-/issues",
        "Documentation": "https://gitlab.rm-rf.ch/world/openssl_encrypt/-/tree/main/openssl_encrypt/docs",
        "Source Code": "https://gitlab.rm-rf.ch/world/openssl_encrypt/-/tree/main/openssl_encrypt",
    },
    packages=find_packages(),
    include_package_data=True,
    package_data={"": ["README.md"]},
    author="Tobi",
    author_email="jahlives@gmx.ch",
    description="A package for secure file encryption and decryption based on modern ciphers using heavy-compute-load chaining of hashing and KDF to generate strong encryption password based on users provided password to ensure secure encryption of files",
    long_description=long_description,
    long_description_content_type="text/markdown",
    keywords="encryption, decryption, random-password, secure shredding, security",
    url="https://gitlab.rm-rf.ch/world/openssl_encrypt",
    classifiers=[
        "Development Status :: 5 - Production/Stable",
        "Intended Audience :: Developers",
        "Intended Audience :: End Users/Desktop",
        "Intended Audience :: Information Technology",
        "Intended Audience :: System Administrators",
        "Programming Language :: Python :: 3",
        "Programming Language :: Python :: 3.9",
        "Programming Language :: Python :: 3.10",
        "Programming Language :: Python :: 3.11",
        "Programming Language :: Python :: 3.12",
        "Programming Language :: Python :: 3.13",
        "License :: OSI Approved :: MIT License",
        "Topic :: Security",
        "Topic :: Security :: Cryptography",
        "Topic :: Utilities",
        "Operating System :: OS Independent",
    ],
    python_requires=">=3.9",
)
