#!/usr/bin/env python3
"""
Web viewer for Drupal News Aggregator results.
Renders parsed.md and summary.md in a browser interface.
"""
import os
import sys
import re
from pathlib import Path
from flask import Flask, render_template_string, jsonify, request
import markdown

app = Flask(__name__, static_folder='static', static_url_path='/static')

# Configuration
RUN_ROOT = Path(__file__).parent / "runs"


def get_available_runs():
    """Get list of available run directories."""
    if not RUN_ROOT.exists():
        return []
    runs = [d.name for d in RUN_ROOT.iterdir() if d.is_dir()]
    return sorted(runs, reverse=True)


def read_markdown_file(run_date, filename):
    """Read markdown file and convert to HTML."""
    file_path = RUN_ROOT / run_date / filename
    if not file_path.exists():
        return None

    with open(file_path, 'r', encoding='utf-8') as f:
        md_content = f.read()

    html_content = markdown.markdown(md_content, extensions=['tables', 'fenced_code'])

    # Convert plain URLs in table cells to links
    def linkify_url(match):
        url = match.group(1)
        return f'<td><a href="{url}" target="_blank" rel="noopener noreferrer">{url}</a></td>'

    html_content = re.sub(
        r'<td>(https?://[^<]+)</td>',
        linkify_url,
        html_content
    )

    # Add target="_blank" to all existing links
    html_content = re.sub(
        r'<a\s+([^>]*?)href=',
        r'<a \1target="_blank" rel="noopener noreferrer" href=',
        html_content
    )

    return html_content


def read_json_file(run_date, filename):
    """Read JSON file."""
    file_path = RUN_ROOT / run_date / filename
    if not file_path.exists():
        return None

    with open(file_path, 'r', encoding='utf-8') as f:
        return f.read()


HTML_TEMPLATE = """
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Drupal News Viewer - {{ run_date }}</title>
    <link crossorigin="anonymous" media="all" rel="stylesheet" href="/static/css/viewer.css" />
</head>
<body>
    <header>
        <h1>💧 Drupal News Viewer</h1>
    </header>

    <div class="container">
        <div class="controls">
            <div>
                <label for="run-select">Select date: </label>
                <select id="run-select" onchange="changeRun(this.value)">
                    {% for run in runs %}
                    <option value="{{ run }}" {% if run == run_date %}selected{% endif %}>{{ run }}</option>
                    {% endfor %}
                </select>
            </div>
        </div>

        <div id="tabs-view" class="active">
            <div class="tabs">
                <button class="tab active" onclick="showTab('summary')">Summary</button>
                <button class="tab" onclick="showTab('parsed')">Parsed Data</button>
                <button class="tab" onclick="showTab('metrics')">Metrics</button>
                <button class="tab" onclick="showTab('log')">Log</button>
            </div>

            <div class="content markdown-content">
                <div id="summary-content" class="tab-content active">
                    {% if summary %}
                        {{ summary|safe }}
                    {% else %}
                        <div class="no-data">Summary not available</div>
                    {% endif %}
                </div>

                <div id="parsed-content" class="tab-content">
                    {% if parsed %}
                        {{ parsed|safe }}
                    {% else %}
                        <div class="no-data">Parsed data not available</div>
                    {% endif %}
                </div>

                <div id="metrics-content" class="tab-content">
                    {% if metrics %}
                        <h2>Run Metrics</h2>
                        <pre>{{ metrics }}</pre>
                    {% else %}
                        <div class="no-data">Metrics not available</div>
                    {% endif %}
                </div>

                <div id="log-content" class="tab-content">
                    {% if log %}
                        <h2>Run Log</h2>
                        <pre>{{ log }}</pre>
                    {% else %}
                        <div class="no-data">Log not available</div>
                    {% endif %}
                </div>
            </div>
        </div>
    </div>

    <script>
        function showTab(tabName) {
            // Hide all tab contents
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });

            // Remove active class from all tabs
            document.querySelectorAll('.tab').forEach(tab => {
                tab.classList.remove('active');
            });

            // Show selected tab content
            document.getElementById(tabName + '-content').classList.add('active');

            // Add active class to clicked tab
            event.target.classList.add('active');
        }

        function changeRun(runDate) {
            window.location.href = '/run/' + runDate;
        }
    </script>
</body>
</html>
"""


@app.route('/')
def index():
    """Show latest run."""
    runs = get_available_runs()
    if not runs:
        return "<h1>No runs available</h1><p>Run the aggregator first to generate data.</p>"

    return view_run(runs[0])


@app.route('/run/<run_date>')
def view_run(run_date):
    """View specific run."""
    runs = get_available_runs()
    if run_date not in runs:
        return f"<h1>Run not found: {run_date}</h1>", 404

    summary_html = read_markdown_file(run_date, 'summary.md')
    parsed_html = read_markdown_file(run_date, 'parsed.md')
    metrics_json = read_json_file(run_date, 'metrics.json')
    log_text = read_json_file(run_date, 'run.log')

    return render_template_string(
        HTML_TEMPLATE,
        run_date=run_date,
        runs=runs,
        summary=summary_html,
        parsed=parsed_html,
        metrics=metrics_json,
        log=log_text
    )


@app.route('/api/runs')
def api_runs():
    """API endpoint to list all runs."""
    runs = get_available_runs()
    return jsonify(runs)


if __name__ == '__main__':
    print("=" * 60)
    print("Drupal News Viewer")
    print("=" * 60)
    print(f"Run root: {RUN_ROOT}")
    print(f"Available runs: {len(get_available_runs())}")
    print("\nStarting server on http://localhost:5000")
    print("Press Ctrl+C to stop")
    print("=" * 60)

    app.run(debug=True, host='0.0.0.0', port=5000)
