from __future__ import annotations
import sys
import os
import logging
from http import HTTPStatus
from typing import List, Dict, Callable, Type, TYPE_CHECKING

from halo_app.classes import AbsBaseClass
from halo_app.app.event import AbsHaloEvent
from halo_app.const import LOC, OPType
from halo_app.app.context import HaloContext, InitCtxFactory
from halo_app.app.request import HaloEventRequest, HaloCommandRequest, AbsHaloRequest, HaloQueryRequest
from halo_app.app.query import HaloQuery
from .app.cmd_assembler import CmdAssemblerFactory
from .app.exceptions import FailException
from .app.notification import ValidError
from .entrypoints.client_type import ClientType
from .error import Error
from .logs import log_json
from .settingsx import settingsx
settings = settingsx()

logger = logging.getLogger(__name__)

BUS = None

class SysUtil(AbsBaseClass):

    @staticmethod
    def get_stage():
        """

        :return:
        """
        if 'HALO_STAGE' in os.environ:
            return os.environ['HALO_STAGE']
        return LOC

    @staticmethod
    def instance_full_name(o):
        # o.__module__ + "." + o.__class__.__qualname__ is an example in
        # this context of H.L. Mencken's "neat, plausible, and wrong."
        # Python makes no guarantees as to whether the __module__ special
        # attribute is defined, so we take a more circumspect approach.
        # Alas, the module name is explicitly excluded from __qualname__
        # in Python 3.

        module = o.__class__.__module__
        if module is None or module == str.__class__.__module__:
            return o.__class__.__name__  # Avoid reporting __builtin__
        else:
            return module + '.' + o.__class__.__name__

    @staticmethod
    def create_command_request(halo_context: HaloContext, usecase_id: str, vars: Dict,
                               security=False, roles=None) -> AbsHaloRequest:
        #halo_command = HaloCommand(method_id, vars)
        cmd_assembler = CmdAssemblerFactory.get_assembler_by_method_id(usecase_id)
        halo_command = cmd_assembler.write_cmd_for_method(usecase_id,vars)
        return HaloCommandRequest(halo_context,halo_command, security, roles)

    @staticmethod
    def create_event_request(halo_context:HaloContext,halo_event: AbsHaloEvent,
                               security=None, roles=None) -> AbsHaloRequest:
        return HaloEventRequest(halo_context,halo_event, security, roles)

    @staticmethod
    def create_query_request(halo_context:HaloContext,halo_query: HaloQuery,
                               security=None, roles=None) -> AbsHaloRequest:
        return HaloQueryRequest(halo_context,halo_query, security, roles)

    @staticmethod
    def get_bus():
        global BUS
        if BUS:
            return BUS
        from halo_app import bootstrap
        import importlib
        for usecase_id in settings.HANDLER_MAP:
            clazz_type = settings.HANDLER_MAP[usecase_id]
            clazz = clazz_type["class"]
            type = clazz_type["type"]
            try:
                module_name, class_name = clazz.rsplit(".", 1)
                x = getattr(importlib.import_module(module_name), class_name)
                if type == OPType.COMMAND.value:  # command
                    bootstrap.COMMAND_HANDLERS[usecase_id] = x.run_command_class
                if type == OPType.QUERY.value:  # query
                    bootstrap.QUERY_HANDLERS[usecase_id] = x.run_query_class
                if type == OPType.EVENT.value:  # event
                    bootstrap.EVENT_HANDLERS[usecase_id] = x.run_event_class
            except Exception as e:
                logger.error("config for handler missing: "+str(clazz) +" - "+str(e))
                raise e
        BUS = bootstrap.bootstrap()
        return BUS

    @staticmethod
    def process_api_ok1(halo_response, method):
        if halo_response:
            if halo_response.request.context.get(HaloContext.client_type) == ClientType.api:
                if halo_response.success:
                    if settings.ASYNC_MODE:
                        success = HTTPStatus.ACCEPTED
                    else:
                        success = HTTPStatus.OK
                    if halo_response.request:
                        if halo_response.request.context:
                            halo_response.code = success
                            if method == 'GET':
                                halo_response.code = success
                            if method == 'POST':
                                if success == HTTPStatus.ACCEPTED:
                                    halo_response.code = HTTPStatus.ACCEPTED
                                else:
                                    halo_response.code = HTTPStatus.CREATED
                            if method == 'PUT':
                                halo_response.code = HTTPStatus.ACCEPTED
                            if method == 'PATCH':
                                halo_response.code = HTTPStatus.ACCEPTED
                            if method == 'DELETE':
                                halo_response.code = success
                            logger.info('process_service_operation : ' + halo_response.request.usecase_id,
                                        extra=log_json(halo_response.request.context, {"return": "success"}))
                            return halo_response
                else:
                    halo_response.code = HTTPStatus.INTERNAL_SERVER_ERROR
                    if halo_response.payload:  # result-error,notification-errors,empty
                        if isinstance(halo_response.payload, [ValidError]):
                            halo_response.code = HTTPStatus.BAD_REQUEST
                    return halo_response
        raise FailException(halo_response)

    @staticmethod
    def process_response_for_client(halo_response):
        if halo_response:
            if halo_response.request.context.get(HaloContext.client_type) == ClientType.api:
                if halo_response.request.context.get(HaloContext.method):
                    method = halo_response.request.context.get(HaloContext.method)
                    if halo_response.success:
                        if settings.ASYNC_MODE:
                            success = HTTPStatus.ACCEPTED
                        else:
                            success = HTTPStatus.OK
                        if halo_response.request:
                            if halo_response.request.context:
                                halo_response.code = success
                                if method == 'GET':
                                    halo_response.code = success
                                if method == 'POST':
                                    if success == HTTPStatus.ACCEPTED:
                                        halo_response.code = HTTPStatus.ACCEPTED
                                    else:
                                        halo_response.code = HTTPStatus.CREATED
                                if method == 'PUT':
                                    halo_response.code = HTTPStatus.ACCEPTED
                                if method == 'PATCH':
                                    halo_response.code = HTTPStatus.ACCEPTED
                                if method == 'DELETE':
                                    halo_response.code = success
                                logger.info('process_service_operation : ' + halo_response.request.usecase_id,
                                            extra=log_json(halo_response.request.context, {"return": "success"}))
                                return halo_response
                    else:
                        halo_response.code = HTTPStatus.INTERNAL_SERVER_ERROR
                        if halo_response.error:  # result-error,notification-errors,exception-error
                            from halo_app.app.utilx import Util
                            if isinstance(halo_response.error,list):
                                halo_response.code = HTTPStatus.BAD_REQUEST
                                halo_response.error = Util.json_notification_response(halo_response.request.context,halo_response.error)
                                halo_response.error["status_code"]=halo_response.code
                                halo_response.error["path"] = halo_response.request.context.get(HaloContext.path)
                            else:
                                if isinstance(halo_response.error, Error):
                                    halo_response.error = Util.json_error_response(halo_response.request.context,halo_response.request.usecase_id,settings.ERR_MSG_CLASS, halo_response.error)
                                    halo_response.error["status_code"] = halo_response.code
                        return halo_response
            if halo_response.request.context.get(HaloContext.client_type) == ClientType.cli:
                if halo_response.success:
                    if halo_response.request:
                        if halo_response.request.context:
                            logger.info('process_service_operation : ' + halo_response.request.usecase_id,
                                        extra=log_json(halo_response.request.context, {"return": "success"}))
                            return halo_response
                else:
                    if halo_response.error:  # result-error,notification-errors,exception-error
                        from halo_app.app.utilx import Util
                        if isinstance(halo_response.error,list):
                            halo_response.error = Util.json_notification_response(halo_response.request.context,halo_response.error)
                        else:
                            if isinstance(halo_response.error, Error):
                                halo_response.error = Util.json_error_response(halo_response.request.context,halo_response.request.usecase_id,settings.ERR_MSG_CLASS, halo_response.error)
                    return halo_response
        raise FailException(halo_response)


