""" Nuvla CLI main script """
import logging
import importlib.metadata
from typing import Optional
from typing_extensions import Annotated

import typer
from rich.pretty import pprint

from nuvla_cli.common.cli_builder import build_entity_action
from nuvla_cli.nuvlaio.edge import Edge


__version__ = importlib.metadata.version("nuvla-cli")

app_cli = typer.Typer(no_args_is_help=True)

build_entity_action(app_cli)


logging.basicConfig(
        level=logging.WARNING,
        format='[%(asctime)s] Line:%(lineno)d %(levelname)s - %(message)s',
        datefmt='%H:%M:%S'
    )
logger: logging.Logger = logging.getLogger('cli')


@app_cli.command(name='clear')
def clear_edges(force: bool = typer.Option(...,
                                           help='Force skip clear confirmation [Not '
                                                'recommended',
                                           prompt='You are about to decommission and '
                                                  'delete all the Edges generated by CLI,'
                                                  ' are you sure?',
                                           confirmation_prompt=True)):
    """
    Clears all the Edges instances for the user created by the CLI

    :return: None
    """
    if force:
        # Create NuvlaIO instance
        edge: Edge = Edge()
        edges_in_nuvla = edge.nuvla_api.search('nuvlabox',
                                               filter={"tags=='cli.created=True'"})

        for nuvla_edge in edges_in_nuvla.resources:
            logger.info(f'Bulk remove Edge {nuvla_edge}')
            edge.remove_edge(nuvla_edge.data.get('id'), force=True)


def version_callback(flag: bool):
    if flag:
        pprint(f'Nuvla-CLI Version: {__version__}')
        raise typer.Exit(0)


@app_cli.callback()
def main(
    version: Annotated[
            Optional[bool],
            typer.Option("--version", '-v', callback=version_callback)
        ] = None,
):
    pass
