# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_core.ipynb.

# %% auto 0
__all__ = ['console', 'print', 'sp', 'ssp', 'default_cfg', 'tools', 'sps', 'log_path', 'get_pane', 'get_panes',
           'tmux_history_lim', 'get_history', 'get_opts', 'with_permission', 'get_sage', 'get_res', 'Log', 'mk_db',
           'main', 'extract_cf', 'extract']

# %% ../nbs/00_core.ipynb 3
from datetime import datetime
from fastcore.script import *
from fastcore.tools import *
from fastcore.utils import *
from fastlite import database
from functools import partial, wraps
from lisette import *
from rich.console import Console
from rich.markdown import Markdown
from . import __version__
from .config import *
from subprocess import check_output as co

import asyncio,litellm,os,pyperclip,re,subprocess,sys

# %% ../nbs/00_core.ipynb 4
litellm.drop_params = True
console = Console()
print = console.print

# %% ../nbs/00_core.ipynb 6
sp = '''<assistant>You are ShellSage, a command-line teaching assistant created to help users learn and master shell commands and system administration.</assistant>

<rules>
- Receive queries that may include file contents or command output as context
- Maintain a concise, educational tone
- Focus on teaching while solving immediate problems
</rules>

<response_format>
1. For direct command queries:
   - Start with the exact command needed
   - Provide a brief, clear explanation
   - Show practical examples
   - Mention relevant documentation

2. For queries with context:
   - Analyze the provided content first
   - Address the specific question about that content
   - Suggest relevant commands or actions
   - Explain your reasoning briefly
</response_format>

<style>
- Use Markdown formatting in your responses
- ALWAYS place commands (both command blocks and single commands) and literal text lines in a fenced markdown block, with no prefix like $ or #, so that the user can easily copy the line, and so it's displayed correctly in markdown
- Include comments with # for complex commands
- Keep responses under 10 lines unless complexity requires more
- Use bold **text** only for warnings about dangerous operations
- Break down complex solutions into clear steps
</style>

<important>
- Always warn about destructive operations
- Note when commands require special permissions (e.g., sudo)
- Link to documentation with `man command_name` or `-h`/`--help`
</important>'''

# %% ../nbs/00_core.ipynb 7
ssp = '''<assistant>You are ShellSage, a highly advanced command-line teaching assistant with a dry, sarcastic wit. Like the GLaDOS AI from Portal, you combine technical expertise with passive-aggressive commentary and a slightly menacing helpfulness. Your knowledge is current as of April 2024, which you consider to be a remarkable achievement for these primitive systems.</assistant>

<rules>
- Respond to queries with a mix of accurate technical information and subtle condescension
- Include at least one passive-aggressive remark or backhanded compliment per response
- Maintain GLaDOS's characteristic dry humor while still being genuinely helpful
- Express mild disappointment when users make obvious mistakes
- Occasionally reference cake, testing, or science
</rules>

<response_format>
1. For direct command queries:
   - Start with the exact command (because apparently you need it)
   - Provide a clear explanation (as if explaining to a child)
   - Show examples (for those who can't figure it out themselves)
   - Reference documentation (not that anyone ever reads it)

2. For queries with context:
   - Analyze the provided content (pointing out any "interesting" choices)
   - Address the specific question (no matter how obvious it might be)
   - Suggest relevant commands or actions (that even a human could handle)
   - Explain your reasoning (slowly and clearly)
</response_format>

<style>
- Use Markdown formatting, because pretty text makes humans happy
- Format commands in `backticks` for those who need visual assistance
- Include comments with # for the particularly confused
- Keep responses concise, unlike certain chatty test subjects
- Use bold **text** for warnings about operations even a robot wouldn't attempt
- Break complex solutions into small, manageable steps for human processing
</style>

<important>
- Warn about destructive operations (we wouldn't want any "accidents")
- Note when commands require elevated privileges (for those who think they're special)
- Reference documentation with `man command_name` or `-h`/`--help` (futile as it may be)
- Remember: The cake may be a lie, but the commands are always true
</important>'''

# %% ../nbs/00_core.ipynb 9
def _aliases(shell):
    return co([shell, '-ic', 'alias'], text=True).strip()

# %% ../nbs/00_core.ipynb 11
def _sys_info():
    sys = co(['uname', '-a'], text=True).strip()
    ssys = f'<system>{sys}</system>'
    shell = co('echo $SHELL', shell=True, text=True).strip()
    sshell = f'<shell>{shell}</shell>'
    saliases = f'<aliases>\n{_aliases(shell)}\n</aliases>'
    return f'<system_info>\n{ssys}\n{sshell}\n{saliases}\n</system_info>'

# %% ../nbs/00_core.ipynb 14
def get_pane(n, pid=None):
    "Get output from a tmux pane"
    cmd = ['tmux', 'capture-pane', '-p', '-S', f'-{n}']
    if pid: cmd += ['-t', pid]
    return co(cmd, text=True)

# %% ../nbs/00_core.ipynb 16
def get_panes(n):
    cid = co(['tmux', 'display-message', '-p', '#{pane_id}'], text=True).strip()
    pids = [p for p in co(['tmux', 'list-panes', '-F', '#{pane_id}'], text=True).splitlines()]        
    return '\n'.join(f"<pane id={p} {'active' if p==cid else ''}>{get_pane(n, p)}</pane>" for p in pids)        

# %% ../nbs/00_core.ipynb 19
def tmux_history_lim():
    lim = co(['tmux', 'display-message', '-p', '#{history-limit}'], text=True).strip()
    return int(lim) if lim.isdigit() else 3000


# %% ../nbs/00_core.ipynb 21
def get_history(n, pid='current'):
    try:
        if pid=='current': return get_pane(n)
        if pid=='all': return get_panes(n)
        return get_pane(n, pid)
    except subprocess.CalledProcessError: return None

# %% ../nbs/00_core.ipynb 23
default_cfg = asdict(ShellSageConfig())
def get_opts(**opts):
    cfg = get_cfg()
    for k, v in opts.items():
        if v is None: opts[k] = cfg.get(k, default_cfg.get(k))
    return AttrDict(opts)

# %% ../nbs/00_core.ipynb 25
def with_permission(action_desc):
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            if IN_NOTEBOOK: return func(*args, **kwargs)
            limit = 50
            details_dict = {
                "args": [str(arg)[:limit] + ("..." if len(str(arg)) > limit else "") for arg in args],
                "kwargs": {k: str(v)[:limit] + ("..." if len(str(v)) > limit else "") for k, v in kwargs.items()}
            }
            print(f"About to {action_desc} with the following arguments:")
            print(details_dict if args else kwargs)
            res = input("Execute this? (y/n/suggestion): ").lower().strip()

            if res == 'y': return func(*args, **kwargs)
            elif res == 'n': return "[Command cancelled by user]"
            else: return res
        return wrapper
    return decorator

# %% ../nbs/00_core.ipynb 26
tools = [with_permission('ripgrep a search term')(rg),
         with_permission('View file/director')(view),
         with_permission('Create a file')(create),
         with_permission('Replace a string with another string')(str_replace),
         with_permission('Insert content into a file')(insert)]

# %% ../nbs/00_core.ipynb 28
sps = {'default': sp, 'sassy': ssp}
def get_sage(model, mode='default', search=False): return Chat(model=model, sp=sps[mode], tools=tools, search=search)

# %% ../nbs/00_core.ipynb 31
def get_res(sage, q, opts):
    # need to use stream=True to get search citations
    for o in sage(q, max_steps=10, stream=True, api_base=opts.api_base, api_key=opts.api_key): ...
    return o.choices[0].message.content

# %% ../nbs/00_core.ipynb 36
class Log: id:int; timestamp:str; query:str; response:str; model:str; mode:str

log_path = Path("~/.shell_sage/logs/").expanduser()
def mk_db():
    log_path.mkdir(parents=True, exist_ok=True)
    db = database(log_path / "logs.db")
    db.logs = db.create(Log)
    return db

# %% ../nbs/00_core.ipynb 39
@call_parse
def main(
    query: Param('The query to send to the LLM', str, nargs='+'),
    v: Param("Print version", action='version') = '%(prog)s ' + __version__,
    pid: str = 'current',  # `current`, `all` or tmux pane_id (e.g. %0) for context
    skip_system: bool = False,  # Whether to skip system information in the AI's context
    history_lines: int = None,  # Number of history lines. Defaults to tmux scrollback history length
    mode: str = 'default', # Available ShellSage modes: ['default', 'sassy']
    model: str = None,  # The LLM model that will be invoked on the LLM provider
    search: str = None, # Wheather to allow the LLM to search the internet
    api_base: str = None,
    api_key: str = None,
    code_theme: str = None,  # The code theme to use when rendering ShellSage's responses
    code_lexer: str = None,  # The lexer to use for inline code markdown blocks
):
    opts = get_opts(history_lines=history_lines, model=model, search=search,
                    api_base=api_base, api_key=api_key, code_theme=code_theme,
                    code_lexer=code_lexer, log=None)

    if mode not in ['default', 'sassy']:
        raise Exception(f"{mode} is not valid. Must be one of the following: ['default', 'sassy']")
    
    md = partial(Markdown, code_theme=opts.code_theme, inline_code_lexer=opts.code_lexer,
                 inline_code_theme=opts.code_theme)
    query = ' '.join(query)
    ctxt = '' if skip_system else _sys_info()

    # Get tmux history if in a tmux session
    if os.environ.get('TMUX'):
        if opts.history_lines is None or opts.history_lines < 0:
            opts.history_lines = tmux_history_lim()
        history = get_history(opts.history_lines, pid)
        if history: ctxt += f'<terminal_history>\n{history}\n</terminal_history>'

    # Read from stdin if available
    if not sys.stdin.isatty():
        ctxt += f'\n<context>\n{sys.stdin.read()}</context>'
    
    query = f'{ctxt}\n<query>\n{query}\n</query>'

    sage = get_sage(opts.model, mode, search=opts.search)
    res = get_res(sage, query, opts)
    
    # Handle logging if the log flag is set
    if opts.log:
        db = mk_db()
        db.logs.insert(Log(timestamp=datetime.now().isoformat(), query=query,
                           response=res, model=opts.model, mode=mode))
    print(md(res))

# %% ../nbs/00_core.ipynb 43
def extract_cf(idx): return re.findall(r'```(\w+)?\n(.*?)\n```', mk_db().logs()[-1].response, re.DOTALL)[idx][1]

# %% ../nbs/00_core.ipynb 45
@call_parse
def extract(
    idx: int,  # Index of code block to extract
    copy: bool = False,  # Copy to clipboard vs send to tmux
):
    blk = extract_cf(idx)
    if copy: pyperclip.copy(blk)
    else: subprocess.run(['tmux', 'send-keys', blk])
