import unittest

def any_string_contains(s, l):
    return any(s in e for e in l)

def get_string_contains(s, l):
    return [el for el in l if s in el]

def programmer_is_usbasp(status):
    return get_programmer_from_commandline(status["commandline"]) in ["usbasp", "usbasp-clone"]

def programmer_is_avrispmk2(status):
    return get_programmer_from_commandline(status["commandline"]) in ["avrisp2", "avrispmkII"]

def usbasp_not_found(status):
    if not is_error_status(status):
        return False
    if not programmer_is_usbasp(status):
        return False
    usbasp_error = "could not find USB device with vid=0x16c0 pid=0x5dc vendor='www.fischl.de' product='USBasp'"
    return any_string_contains(usbasp_error, status["errors"])

def is_error_status(status):
    return status["result"] == "failed"

def avrispmk2_not_found(status):
    if not is_error_status(status):
        return False
    if not programmer_is_avrispmk2(status):
        return False
    output_lines = list(filter(None, status["full_output"].split('\n')))
    return len(output_lines) == 1 and output_lines[0] == "avrdude done.  Thank you."

def no_chip_found(status):
    if programmer_is_avrispmk2(status):
        avrisp2_cnf_marker_1 = "bad AVRISPmkII connection status"
        avrisp2_cnf_marker_2 = "Target not detected"
        if any_string_contains(avrisp2_cnf_marker_1, status["output"]) \
        and any_string_contains(avrisp2_cnf_marker_2, status["output"]):
            return True
    return any_string_contains("target doesn't answer", status["errors"])

def invalid_input_file(status):
    read_input_file = any_string_contains("reading input file", status["output"])
    read_from_file_strings = get_string_contains("read from file", status["output"])
    read_failed = any(["failed" in s for s in read_from_file_strings])
    invalid_format = any_string_contains("auto detected as invalid format", status["output"])
    return read_input_file and (read_failed or invalid_format)

def zero_length_input_file(status):
    read_input_file = any_string_contains("reading input file", status["output"])
    contains_zero_bytes = any_string_contains("contains 0 bytes", status["output"])
    return read_input_file and contains_zero_bytes

def wrong_signature(status):
    expected_signature_wrong = any_string_contains("Expected signature for", status["output"])
    double_check_chip = any_string_contains("Double check chip", status["other"])
    return expected_signature_wrong and double_check_chip

def unreliable_fuses_fail(status):
    unreliable_reading_fuses = any_string_contains("Sorry, reading back fuses was unreliable.", status["output"])
    return unreliable_reading_fuses

def fuse_verify_fail(status):
    safemode_fuse_changed = any_string_contains("fuse changed! Was ", status["output"])
    return safemode_fuse_changed

def get_fuses_from_success_status(status):
    fuse_marker = "safemode: Fuses OK ("
    fuse_strings = get_string_contains(fuse_marker, status["output"])
    # ['safemode: Fuses OK (E:05, H:DA, L:FF)']
    fuse_strings = [s[len(fuse_marker):].rstrip(')') for s in fuse_strings if s.startswith(fuse_marker)]
    # ['E:05, H:DA, L:FF']
    fuse_dict = {}
    for fuse_s in fuse_strings:
        fuses = fuse_s.split(', ')
        for fuse in fuses:
            name, value_str = fuse.split(':')
            fuse_dict[name] = int(value_str, 16)
    return fuse_dict

def get_found_expected_signatures(status):
    found_signature = get_found_signature(status)
    expected_marking = "Expected signature for "
    # Looking for what was expected
    expected_strings = get_string_contains(expected_marking, status["output"])
    # ['Expected signature for ATmega328P is 1E 95 0F']
    expected_signatures = [s[len(expected_marking):] for s in expected_strings if s.startswith(expected_marking)]
    # ['ATmega328P is 1E 95 0F']
    is_marker = " is "
    expected_signatures = [s.split(is_marker, 1)[1] for s in expected_signatures]
    # ['1E 95 0F']
    expected_signatures = ["0x"+("".join(s.split(' '))).lower() for s in expected_signatures]
    # ['0x1e950f']
    expected_signature = ",".join(expected_signatures)
    # '0x1e950f'
    return found_signature, expected_signature

def chip_is_found(status):
    success = status["exitcode"] == 0
    ready_to_accept_instr = any_string_contains("device initialized and ready to accept instructions", status["output"])
    has_fuses = get_fuses_from_success_status(status)
    return success and ready_to_accept_instr #and has_fuses

def get_found_signature(status):
    found_marking = "Device signature = "
    found_strings = get_string_contains(found_marking, status["output"])
    # ['Device signature = 0x1e9307']
    found_signatures = [s[len(found_marking):] for s in found_strings if s.startswith(found_marking)]
    # ['0x1e9307']
    found_signatures = [s for s in found_signatures if s.startswith('0x')]
    # ['0x1e9307'] (sanity checked)
    return ",".join(found_signatures)

def get_programmer_from_commandline(commandline):
    for i, el in enumerate(commandline):
        if el == "-c":
            break
    return commandline[i+1]

def get_human_readable_status(status):
    if status["result"] == "success":
        return ["Success"]
    else:
        if usbasp_not_found(status):
            return ["Failure", "USBASP not found!"]
        if avrispmk2_not_found(status):
            return ["Failure", "ISPMK2 not found!"]
        elif invalid_input_file(status):
            return ["Failure", "Invalid input file!"]
        elif zero_length_input_file(status):
            return ["Failure", "Zero length input!"]
        elif no_chip_found(status):
            return ["Failure", "No chip found"]
        elif wrong_signature(status):
            found_signature = get_found_signature(status)
            return ["Failure", "Wrong chip found", "Found: {}".format(found_signature)]
        elif unreliable_fuses_fail(status):
            return ["Failure", "Unreliable fuses!"]
        elif fuse_verify_fail(status):
            return ["Failure", "Can't verify fuses"]
        else:
            print(status)
            return ["Failure", "Unknown error"]

class TestAvrdude(unittest.TestCase):

    def test_cnf(self):
        """'Chip not found' tests"""
    # "Chip not found" error
        cnf_status = {'errors': ["programm enable: target doesn't answer. 1"], 'full_output': "\navrdude: error: programm enable: target doesn't answer. 1 \navrdude: initialization failed, rc=-1\n         Double check connections and try again, or use -F to override\n         this check.\n\n\navrdude done.  Thank you.\n\n", 'commandline': ['avrdude', '-p', 'm328p', '-c', 'usbasp'], 'other': ['Double check connections and try again, or use -F to override', 'this check.', 'avrdude done. Thank you.'], 'result': 'failed', 'output': ['initialization failed, rc=-1'], 'exitcode': 1}
        assert (get_human_readable_status(cnf_status) == ['Failure', 'No chip found']), "Chip not found status not recognized!"
        # "Chip not found" error with AVRISPMKII
        avrispmk2_cnf_status = {'errors': [], 'full_output': '\navrdude: stk500v2_command(): command failed\navrdude: stk500v2_program_enable(): bad AVRISPmkII connection status: Target not detected\navrdude: initialization failed, rc=-1\n         Double check connections and try again, or use -F to override\n         this check.\n\n\navrdude done.  Thank you.\n\n', 'commandline': ['avrdude', '-p', u'm328p', '-c', u'avrispmkII', '-B', '1'], 'other': ['Double check connections and try again, or use -F to override', 'this check.', 'avrdude done.  Thank you.'], 'result': 'failed', 'output': ['stk500v2_command(): command failed', 'stk500v2_program_enable(): bad AVRISPmkII connection status: Target not detected', 'initialization failed, rc=-1'], 'exitcode': 1}
        assert (get_human_readable_status(avrispmk2_cnf_status) == ['Failure', 'No chip found']), "Chip not found status for AVRISPMKII not recognized!"
        # "Chip not found" error with AVRISPMKII, v2
        avrispmk2_cnf_status_2 = {'errors': [], 'full_output': '\navrdude: stk500v2_command(): command failed\navrdude: stk500v2_program_enable(): bad AVRISPmkII connection status: RST fail, SCK fail, Target not detected\navrdude: initialization failed, rc=-1\n         Double check connections and try again, or use -F to override\n         this check.\n\n\navrdude done.  Thank you.\n\n', 'commandline': ['avrdude', '-p', u'm328p', '-c', u'avrispmkII', '-B', '1'], 'other': ['Double check connections and try again, or use -F to override', 'this check.', 'avrdude done.  Thank you.'], 'result': 'failed', 'output': ['stk500v2_command(): command failed', 'stk500v2_program_enable(): bad AVRISPmkII connection status: RST fail, SCK fail, Target not detected', 'initialization failed, rc=-1'], 'exitcode': 1}
        assert (get_human_readable_status(avrispmk2_cnf_status_2) == ['Failure', 'No chip found']), "Chip not found status for AVRISPMKII not recognized!"

    def test_pnf(self):
        """'Programmer not found' tests"""
        # 'USBASP not found' error
        uanf_status = {'errors': ["could not find USB device with vid=0x16c0 pid=0x5dc vendor='www.fischl.de' product='USBasp'"], 'full_output': "avrdude: error: could not find USB device with vid=0x16c0 pid=0x5dc vendor='www.fischl.de' product='USBasp'\n\navrdude done.  Thank you.\n\n", 'commandline':['avrdude', '-p', 'm328p', '-c', 'usbasp'], 'other': ['avrdude done.  Thank you.'], 'result': 'failed', 'output': [], 'exitcode': 1}
        assert (get_human_readable_status(uanf_status) == ['Failure', 'USBASP not found!']), "'USBASP not found' status not recognized!"

    def test_errors(self):
        """Error processing"""
        # 'Wrong signature' error
        ws_status = {'errors': [], 'full_output': '\navrdude: AVR device initialized and ready to accept instructions\n\nReading | ################################################## | 100% 0.01s\n\navrdude: Device signature = 0x1e9307\navrdude: Expected signature for ATmega328P is 1E 95 0F\n         Double check chip, or use -F to override this check.\n\navrdude done.  Thank you.\n\n', 'commandline': ['avrdude', '-p', 'm328p', '-c', 'usbasp'], 'other': ['Reading | ################################################## | 100% 0.01s', 'Double check chip, or use -F to override this check.', 'avrdude done.  Thank you.'], 'result': 'failed', 'output': ['AVR device initialized and ready to accept instructions', 'Device signature = 0x1e9307', 'Expected signature for ATmega328P is 1E 95 0F'], 'exitcode': 1}
        assert (get_human_readable_status(ws_status) == ["Failure", "Wrong chip found", "Found: 0x1e9307"]), "'Wrong chip' status not recognized!"
        assert (get_found_expected_signatures(ws_status) == ('0x1e9307', '0x1e950f')), "Found&expected signatures in 'wrong chip' status are not recognized!"
        # 'Unreliable fuses' error
        ufe_status = {'errors': [], 'full_output': '\r\navrdude: set SCK frequency to 187500 Hz\r\navrdude: AVR device initialized and ready to accept instructions\r\n\r\n\rReading |                                                    | 0% 0.00s\rReading | #################                                  | 33% 0.00s\rReading | #################################                  | 66% 0.00s\rReading | ################################################## | 100% 0.00s\r\n\r\navrdude: Device signature = 0x1e950f\r\navrdude: reading flash memory:\r\n\r\n\rReading |                                                    | 0% 0.00s\rReading | #                                                  | 1% 0.34s\rReading | #                                                  | 2% 0.68s\rReading | ##                                                 | 3% 0.90s\rReading | ##                                                 | 4% 1.24s\rReading | ###                                                | 5% 1.47s\rReading | ###                                                | 6% 1.81s\rReading | ####                                               | 7% 2.03s\rReading | ####                                               | 8% 2.37s\rReading | #####                                              | 9% 2.71s\rReading | #####                                              | 10% 2.94s\rReading | ######                                             | 11% 3.28s\rReading | ######                                             | 12% 3.50s\rReading | #######                                            | 13% 3.84s\rReading | #######                                            | 14% 4.07s\rReading | ########                                           | 15% 4.41s\rReading | ########                                           | 16% 4.63s\rReading | #########                                          | 17% 4.97s\rReading | #########                                          | 18% 5.31s\rReading | ##########                                         | 19% 5.54s\rReading | ##########                                         | 20% 5.87s\rReading | ###########                                        | 21% 6.10s\rReading | ###########                                        | 22% 6.44s\rReading | ############                                       | 23% 6.67s\rReading | ############                                       | 24% 7.00s\rReading | #############                                      | 25% 7.23s\rReading | #############                                      | 26% 7.57s\rReading | ##############                                     | 27% 7.91s\rReading | ##############                                     | 28% 8.13s\rReading | ###############                                    | 29% 8.47s\rReading | ###############                                    | 30% 8.70s\rReading | ################                                   | 31% 9.04s\rReading | ################                                   | 32% 9.26s\rReading | #################                                  | 33% 9.60s\rReading | #################                                  | 34% 9.94s\rReading | ##################                                 | 35% 10.17s\rReading | ##################                                 | 36% 10.51s\rReading | ###################                                | 37% 10.73s\rReading | ###################                                | 38% 11.07s\rReading | ####################                               | 39% 11.30s\rReading | ####################                               | 40% 11.64s\rReading | #####################                              | 41% 11.86s\rReading | #####################                              | 42% 12.20s\rReading | ######################                             | 43% 12.54s\rReading | ######################                             | 44% 12.77s\rReading | #######################                            | 45% 13.11s\rReading | #######################                            | 46% 13.33s\rReading | ########################                           | 47% 13.67s\rReading | ########################                           | 48% 13.90s\rReading | #########################                          | 49% 14.24s\rReading | #########################                          | 50% 14.46s\rReading | ##########################                         | 51% 14.80s\rReading | ##########################                         | 52% 15.14s\rReading | ###########################                        | 53% 15.37s\rReading | ###########################                        | 54% 15.71s\rReading | ############################                       | 55% 15.93s\rReading | ############################                       | 56% 16.27s\rReading | #############################                      | 57% 16.50s\rReading | #############################                      | 58% 16.84s\rReading | ##############################                     | 59% 17.17s\rReading | ##############################                     | 60% 17.40s\rReading | ###############################                    | 61% 17.74s\rReading | ###############################                    | 62% 17.97s\rReading | ################################                   | 63% 18.30s\rReading | ################################                   | 64% 18.53s\rReading | #################################                  | 65% 18.87s\rReading | #################################                  | 66% 19.10s\rReading | ##################################                 | 67% 19.43s\rReading | ##################################                 | 68% 19.77s\rReading | ###################################                | 69% 20.00s\rReading | ###################################                | 70% 20.34s\rReading | ####################################               | 71% 20.56s\rReading | ####################################               | 72% 20.90s\rReading | #####################################              | 73% 21.13s\rReading | #####################################              | 74% 21.47s\rReading | ######################################             | 75% 21.69s\rReading | ######################################             | 76% 22.03s\rReading | #######################################            | 77% 22.37s\rReading | #######################################            | 78% 22.60s\rReading | ########################################           | 79% 22.94s\rReading | ########################################           | 80% 23.16s\rReading | #########################################          | 81% 23.50s\rReading | #########################################          | 82% 23.73s\rReading | ##########################################         | 83% 24.07s\rReading | ##########################################         | 84% 24.41s\rReading | ###########################################        | 85% 24.63s\rReading | ###########################################        | 86% 24.97s\rReading | ############################################       | 87% 25.20s\rReading | ############################################       | 88% 25.54s\rReading | #############################################      | 89% 25.76s\rReading | #############################################      | 90% 26.10s\rReading | ##############################################     | 91% 26.33s\rReading | ##############################################     | 92% 26.67s\rReading | ###############################################    | 93% 27.01s\rReading | ###############################################    | 94% 27.23s\rReading | ################################################   | 95% 27.57s\rReading | ################################################   | 96% 27.80s\rReading | #################################################  | 97% 28.14s\rReading | #################################################  | 98% 28.36s\rReading | ################################################## | 99% 28.70s\rReading | ################################################## | 100% 28.93s\r\n\r\navrdude: writing output file "text.hex"\r\n\r\navrdude: safemode: Verify error - unable to read lfuse properly. Programmer may not be reliable.\r\navrdude: safemode: Verify error - unable to read lfuse properly. Programmer may not be reliable.\r\navrdude: safemode: Sorry, reading back fuses was unreliable. I have given up and exited programming mode\r\n\r\navrdude done.  Thank you.\r\n\r\n', 'commandline': ['avrdude', '-p', 'm328p', '-c', 'usbasp', '-B', '5', '-U', 'flash:r:text.hex:i'], 'other': ['Reading |                                                    | 0% 0.00s\rReading | #################                                  | 33% 0.00s\rReading | #################################                  | 66% 0.00s\rReading | ################################################## | 100% 0.00s', 'Reading |                                                    | 0% 0.00s\rReading | #                                                  | 1% 0.34s\rReading | #                                                  | 2% 0.68s\rReading | ##                                                 | 3% 0.90s\rReading | ##                                                 | 4% 1.24s\rReading | ###                                                | 5% 1.47s\rReading | ###                                                | 6% 1.81s\rReading | ####                                               | 7% 2.03s\rReading | ####                                               | 8% 2.37s\rReading | #####                                              | 9% 2.71s\rReading | #####                                              | 10% 2.94s\rReading | ######                                             | 11% 3.28s\rReading | ######                                             | 12% 3.50s\rReading | #######                                            | 13% 3.84s\rReading | #######                                            | 14% 4.07s\rReading | ########                                           | 15% 4.41s\rReading | ########                                           | 16% 4.63s\rReading | #########                                          | 17% 4.97s\rReading | #########                                          | 18% 5.31s\rReading | ##########                                         | 19% 5.54s\rReading | ##########                                         | 20% 5.87s\rReading | ###########                                        | 21% 6.10s\rReading | ###########                                        | 22% 6.44s\rReading | ############                                       | 23% 6.67s\rReading | ############                                       | 24% 7.00s\rReading | #############                                      | 25% 7.23s\rReading | #############                                      | 26% 7.57s\rReading | ##############                                     | 27% 7.91s\rReading | ##############                                     | 28% 8.13s\rReading | ###############                                    | 29% 8.47s\rReading | ###############                                    | 30% 8.70s\rReading | ################                                   | 31% 9.04s\rReading | ################                                   | 32% 9.26s\rReading | #################                                  | 33% 9.60s\rReading | #################                                  | 34% 9.94s\rReading | ##################                                 | 35% 10.17s\rReading | ##################                                 | 36% 10.51s\rReading | ###################                                | 37% 10.73s\rReading | ###################                                | 38% 11.07s\rReading | ####################                               | 39% 11.30s\rReading | ####################                               | 40% 11.64s\rReading | #####################                              | 41% 11.86s\rReading | #####################                              | 42% 12.20s\rReading | ######################                             | 43% 12.54s\rReading | ######################                             | 44% 12.77s\rReading | #######################                            | 45% 13.11s\rReading | #######################                            | 46% 13.33s\rReading | ########################                           | 47% 13.67s\rReading | ########################                           | 48% 13.90s\rReading | #########################                          | 49% 14.24s\rReading | #########################                          | 50% 14.46s\rReading | ##########################                         | 51% 14.80s\rReading | ##########################                         | 52% 15.14s\rReading | ###########################                        | 53% 15.37s\rReading | ###########################                        | 54% 15.71s\rReading | ############################                       | 55% 15.93s\rReading | ############################                       | 56% 16.27s\rReading | #############################                      | 57% 16.50s\rReading | #############################                      | 58% 16.84s\rReading | ##############################                     | 59% 17.17s\rReading | ##############################                     | 60% 17.40s\rReading | ###############################                    | 61% 17.74s\rReading | ###############################                    | 62% 17.97s\rReading | ################################                   | 63% 18.30s\rReading | ################################                   | 64% 18.53s\rReading | #################################                  | 65% 18.87s\rReading | #################################                  | 66% 19.10s\rReading | ##################################                 | 67% 19.43s\rReading | ##################################                 | 68% 19.77s\rReading | ###################################                | 69% 20.00s\rReading | ###################################                | 70% 20.34s\rReading | ####################################               | 71% 20.56s\rReading | ####################################               | 72% 20.90s\rReading | #####################################              | 73% 21.13s\rReading | #####################################              | 74% 21.47s\rReading | ######################################             | 75% 21.69s\rReading | ######################################             | 76% 22.03s\rReading | #######################################            | 77% 22.37s\rReading | #######################################            | 78% 22.60s\rReading | ########################################           | 79% 22.94s\rReading | ########################################           | 80% 23.16s\rReading | #########################################          | 81% 23.50s\rReading | #########################################          | 82% 23.73s\rReading | ##########################################         | 83% 24.07s\rReading | ##########################################         | 84% 24.41s\rReading | ###########################################        | 85% 24.63s\rReading | ###########################################        | 86% 24.97s\rReading | ############################################       | 87% 25.20s\rReading | ############################################       | 88% 25.54s\rReading | #############################################      | 89% 25.76s\rReading | #############################################      | 90% 26.10s\rReading | ##############################################     | 91% 26.33s\rReading | ##############################################     | 92% 26.67s\rReading | ###############################################    | 93% 27.01s\rReading | ###############################################    | 94% 27.23s\rReading | ################################################   | 95% 27.57s\rReading | ################################################   | 96% 27.80s\rReading | #################################################  | 97% 28.14s\rReading | #################################################  | 98% 28.36s\rReading | ################################################## | 99% 28.70s\rReading | ################################################## | 100% 28.93s', 'avrdude done.  Thank you.'], 'result': 'failed', 'output': ['set SCK frequency to 187500 Hz', 'AVR device initialized and ready to accept instructions', 'Device signature = 0x1e950f', 'reading flash memory:', 'writing output file "text.hex"', 'safemode: Verify error - unable to read lfuse properly. Programmer may not be reliable.', 'safemode: Verify error - unable to read lfuse properly. Programmer may not be reliable.', 'safemode: Sorry, reading back fuses was unreliable. I have given up and exited programming mode'], 'exitcode': 1}
        assert (get_human_readable_status(ufe_status) == ["Failure", "Unreliable fuses!"]), "'Unreliable fuses' status not recognized!"
        # 'Fuse verify' error
        verif_error = {'errors': [], 'full_output': '\r\navrdude: set SCK frequency to 187500 Hz\r\navrdude: AVR device initialized and ready to accept instructions\r\n\r\n\rReading |                                                    | 0% 0.00s\rReading | #################                                  | 33% 0.00s\rReading | #################################                  | 66% 0.00s\rReading | ################################################## | 100% 0.00s\r\n\r\navrdude: Device signature = 0x1e950f\r\navrdude: NOTE: "flash" memory has been specified, an erase cycle will be performed\r\n         To disable this feature, specify the -D option.\r\navrdude: erasing chip\r\navrdude: set SCK frequency to 187500 Hz\r\navrdude: reading input file "/opt/zpui/apps/hardware_apps/avrdude/bootloaders/ATmegaBOOT_atmega328.hex"\r\navrdude: writing flash (32670 bytes):\r\n\r\n\rWriting |                                                    | 0% 0.00s\rWriting | ###                                                | 6% 0.00s\rWriting | ######                                             | 12% 0.00s\rWriting | #########                                          | 18% 0.00s\rWriting | #############                                      | 25% 0.00s\rWriting | ################                                   | 31% 0.00s\rWriting | ###################                                | 37% 0.00s\rWriting | ######################                             | 43% 0.00s\rWriting | #########################                          | 50% 0.00s\rWriting | ############################                       | 56% 0.00s\rWriting | ###############################                    | 62% 0.00s\rWriting | ##################################                 | 68% 0.00s\rWriting | ######################################             | 75% 0.00s\rWriting | #########################################          | 81% 0.00s\rWriting | ############################################       | 87% 0.00s\rWriting | ###############################################    | 93% 0.00s\rWriting | ################################################## | 100% 0.00s\r\n\r\navrdude: 32670 bytes of flash written\r\navrdude: verifying flash memory against /opt/zpui/apps/hardware_apps/avrdude/bootloaders/ATmegaBOOT_atmega328.hex:\r\navrdude: load data flash data from input file /opt/zpui/apps/hardware_apps/avrdude/bootloaders/ATmegaBOOT_atmega328.hex:\r\navrdude: input file /opt/zpui/apps/hardware_apps/avrdude/bootloaders/ATmegaBOOT_atmega328.hex contains 32670 bytes\r\navrdude: reading on-chip flash data:\r\n\r\n\rReading |                                                    | 0% 0.00s\rReading | ###                                                | 6% 0.00s\rReading | ######                                             | 12% 0.00s\rReading | #########                                          | 18% 0.00s\rReading | #############                                      | 25% 0.00s\rReading | ################                                   | 31% 0.00s\rReading | ###################                                | 37% 0.00s\rReading | ######################                             | 43% 0.00s\rReading | #########################                          | 50% 0.00s\rReading | ############################                       | 56% 0.00s\rReading | ###############################                    | 62% 0.00s\rReading | ##################################                 | 68% 0.00s\rReading | ######################################             | 75% 0.00s\rReading | #########################################          | 81% 0.00s\rReading | ############################################       | 87% 0.00s\rReading | ###############################################    | 93% 0.00s\rReading | ################################################## | 100% 0.00s\r\n\r\navrdude: verifying ...\r\navrdude: 32670 bytes of flash verified\r\navrdude: reading input file "0xda"\r\navrdude: writing hfuse (1 bytes):\r\n\r\n\rWriting | | 0% 0.00s\rWriting | ################################################## | 100% 0.01s\r\n\r\navrdude: 1 bytes of hfuse written\r\navrdude: verifying hfuse memory against 0xda:\r\navrdude: load data hfuse data from input file 0xda:\r\navrdude: input file 0xda contains 1 bytes\r\navrdude: reading on-chip hfuse data:\r\n\r\n\rReading |                                                    | 0% 0.00s\rReading | ################################################## | 100% 0.01s\r\n\r\navrdude: verifying ...\r\navrdude: 1 bytes of hfuse verified\r\navrdude: reading input file "0xff"\r\navrdude: writing lfuse (1 bytes):\r\n\r\n\rWriting |                                                    | 0% 0.00s\rWriting | ################################################## | 100% 0.00s\r\n\r\navrdude: 1 bytes of lfuse written\r\navrdude: verifying lfuse memory against 0xff:\r\navrdude: load data lfuse data from input file 0xff:\r\navrdude: input file 0xff contains 1 bytes\r\navrdude: reading on-chip lfuse data:\r\n\r\n\rReading |                                                    | 0% 0.00s\rReading | ################################################## | 100% 0.00s\r\n\r\navrdude: verifying ...\r\navrdude: 1 bytes of lfuse verified\r\navrdude: reading input file "0xfd"\r\navrdude: writing efuse (1 bytes):\r\n\r\n\rWriting |                                                    | 0% 0.00s ***failed;  \r\n\rWriting | ################################################## | 100% 0.03s\r\n\r\navrdude: 1 bytes of efuse written\r\navrdude: verifying efuse memory against 0xfd:\r\navrdude: load data efuse data from input file 0xfd:\r\navrdude: input file 0xfd contains 1 bytes\r\navrdude: reading on-chip efuse data:\r\n\r\n\rReading |                                                    | 0% 0.00s\rReading | ################################################## | 100% 0.00s\r\n\r\navrdude: verifying ...\r\navrdude: verification error, first mismatch at byte 0x0000\r\n         0x05 != 0xfd\r\navrdude: verification error; content mismatch\r\n\r\navrdude: safemode: efuse changed! Was fd, and is now 5\r\nWould you like this fuse to be changed back? [y/n] ', 'commandline': ['avrdude', '-p', u'm328p', '-c', u'usbasp', '-B', '5', '-U', 'flash:w:/opt/zpui/apps/hardware_apps/avrdude/bootloaders/ATmegaBOOT_atmega328.hex:i', '-U', 'hfuse:w:0xda:m', '-U', 'lfuse:w:0xff:m', '-U', 'efuse:w:0xfd:m'], 'other': ['Reading |                                                    | 0% 0.00s\rReading | #################                                  | 33% 0.00s\rReading | #################################                  | 66% 0.00s\rReading | ################################################## | 100% 0.00s', 'To disable this feature, specify the -D option.', 'Writing |                                                    | 0% 0.00s\rWriting | ###                                                | 6% 0.00s\rWriting | ######                                             | 12% 0.00s\rWriting | #########                                          | 18% 0.00s\rWriting | #############                                      | 25% 0.00s\rWriting | ################                                   | 31% 0.00s\rWriting | ###################                                | 37% 0.00s\rWriting | ######################                             | 43% 0.00s\rWriting | #########################                          | 50% 0.00s\rWriting | ############################                       | 56% 0.00s\rWriting | ###############################                    | 62% 0.00s\rWriting | ##################################                 | 68% 0.00s\rWriting | ######################################             | 75% 0.00s\rWriting | #########################################          | 81% 0.00s\rWriting | ############################################       | 87% 0.00s\rWriting | ###############################################    | 93% 0.00s\rWriting | ################################################## | 100% 0.00s', 'Reading |                                                    | 0% 0.00s\rReading | ### | 6% 0.00s\rReading | ######                                             | 12% 0.00s\rReading | #########                                          | 18% 0.00s\rReading | #############                                      | 25% 0.00s\rReading | ################                                   | 31% 0.00s\rReading | ###################                                | 37% 0.00s\rReading | ######################                             | 43% 0.00s\rReading | #########################                          | 50% 0.00s\rReading | ############################                       | 56% 0.00s\rReading | ###############################                    | 62% 0.00s\rReading | ##################################                 | 68% 0.00s\rReading | ######################################             | 75% 0.00s\rReading | #########################################          | 81% 0.00s\rReading | ############################################       | 87% 0.00s\rReading | ###############################################    | 93% 0.00s\rReading | ################################################## | 100% 0.00s', 'Writing |                                                    | 0% 0.00s\rWriting | ################################################## | 100% 0.01s', 'Reading |                                                    | 0% 0.00s\rReading | ################################################## | 100% 0.01s', 'Writing |                                                    | 0% 0.00s\rWriting | ################################################## | 100% 0.00s', 'Reading |                                                    | 0% 0.00s\rReading | ################################################## | 100% 0.00s', 'Writing |                                                    | 0% 0.00s ***failed;', 'Writing | ################################################## | 100% 0.03s', 'Reading |                                                    | 0% 0.00s\rReading | ################################################## | 100% 0.00s', '0x05 != 0xfd', 'Would you like this fuse to be changed back? [y/n]'], 'result': 'failed', 'output': ['set SCK frequency to 187500 Hz', 'AVR device initialized and ready to accept instructions', 'Device signature = 0x1e950f', 'NOTE: "flash" memory has been specified, an erase cycle will be performed', 'erasing chip', 'set SCK frequency to 187500 Hz', 'reading input file "/opt/zpui/apps/hardware_apps/avrdude/bootloaders/ATmegaBOOT_atmega328.hex"', 'writing flash (32670 bytes):', '32670 bytes of flash written', 'verifying flash memory against /opt/zpui/apps/hardware_apps/avrdude/bootloaders/ATmegaBOOT_atmega328.hex:', 'load data flash data from input file /opt/zpui/apps/hardware_apps/avrdude/bootloaders/ATmegaBOOT_atmega328.hex:', 'input file /opt/zpui/apps/hardware_apps/avrdude/bootloaders/ATmegaBOOT_atmega328.hex contains 32670 bytes', 'reading on-chip flash data:', 'verifying ...', '32670 bytes of flash verified', 'reading input file "0xda"', 'writing hfuse (1 bytes):', '1 bytes of hfuse written', 'verifying hfuse memory against 0xda:', 'load data hfuse data from input file 0xda:', 'input file 0xda contains 1 bytes', 'reading on-chip hfuse data:', 'verifying ...', '1 bytes of hfuse verified', 'reading input file "0xff"', 'writing lfuse (1 bytes):', '1 bytes of lfuse written', 'verifying lfuse memory against 0xff:', 'load data lfuse data from input file 0xff:', 'input file 0xff contains 1 bytes', 'reading on-chip lfuse data:', 'verifying ...', '1 bytes of lfuse verified', 'reading input file "0xfd"', 'writing efuse (1 bytes):', '1 bytes of efuse written', 'verifying efuse memory against 0xfd:', 'load data efuse data from input file 0xfd:', 'input file 0xfd contains 1 bytes', 'reading on-chip efuse data:', 'verifying ...', 'verification error, first mismatch at byte 0x0000', 'verification error; content mismatch', 'safemode: efuse changed! Was fd, and is now 5'], 'exitcode': 1}
        assert (get_human_readable_status(verif_error) == ["Failure", "Can't verify fuses"]), "'Fuse failure' did not trigger on fuse verification error status!"
        # 'Invalid input file' error
        iif_status = {'errors': [], 'full_output': '\r\navrdude: set SCK frequency to 93750 Hz\r\navrdude: AVR device initialized and ready to accept instructions\r\n\r\n\rReading |                        | 0% 0.00s\rReading | #################   | 33% 0.00s\rReading | #################################                  | 66% 0.01s\rReading | ################################################## | 100% 0.01s\r\n\r\navrdude: Device signature = 0x1e950f (probably m328p)\r\navrdude: NOTE: "flash" memory has been specified, an erase cycle will be performed\r\n         To disable this feature, specify the -D option.\r\navrdude: erasing chip\r\navrdude: set SCK frequency to 93750 Hz\r\navrdude: reading input file "/root/Arduino/.gitignore"\r\navrdude: can\'t determine file format for /root/Arduino/.gitignore, specify explicitly\r\navrdude: read from file \'/root/Arduino/.gitignore\' failed\r\n\r\navrdude: safemode: Fuses OK (E:FD, H:DA, L:FF)\r\n\r\navrdude done.  Thank you.\r\n\r\n', 'commandline': ['avrdude', '-p', u'm328p', '-c', u'usbasp', '-B', '10', '-U', 'flash:w:/root/Arduino/.gitignore'], 'other': ['Reading |                                                    | 0% 0.00s\rReading | #################                                  | 33% 0.00s\rReading | #################################                  | 66% 0.01s\rReading | ################################################## | 100% 0.01s', 'To disable this feature, specify the -D option.', 'avrdude done.  Thank you.'], 'result': 'failed', 'output': ['set SCK frequency to 93750 Hz', 'AVR device initialized and ready to accept instructions', 'Device signature = 0x1e950f (probably m328p)', 'NOTE: "flash" memory has been specified, an erase cycle will be performed', 'erasing chip', 'set SCK frequency to 93750 Hz', 'reading input file "/root/Arduino/.gitignore"', "can't determine file format for /root/Arduino/.gitignore, specify explicitly", "read from file '/root/Arduino/.gitignore' failed", 'safemode: Fuses OK (E:FD, H:DA, L:FF)'], 'exitcode': 1}
        assert (get_human_readable_status(iif_status) == ["Failure", "Invalid input file!"]), "Invalid input file status not recognized!"

    def test_success(self):
        """Successful status catching tests"""
        # Successful status 1 - produced with 'subprocess.check_output'
        success_1 = {'errors': [], 'full_output': '\navrdude: AVR device initialized and ready to accept instructions\n\nReading | ################################################## | 100% 0.01s\n\navrdude: Device signature = 0x1e950f\n\navrdude: safemode: Fuses OK (E:05, H:DA, L:FF)\n\navrdude done.  Thank you.\n\n', 'commandline': ['avrdude', '-p', 'm328p', '-c', 'usbasp'], 'other': ['Reading | ################################################## | 100% 0.01s', 'avrdude done.  Thank you.'], 'result': 'success', 'output': ['AVR device initialized and ready to accept instructions', 'Device signature = 0x1e950f', 'safemode: Fuses OK (E:05, H:DA, L:FF)'], 'exitcode': 0}
        assert (get_human_readable_status(success_1) == ["Success"]), "Success status 1 not recognized!"
        assert (get_found_signature(success_1) == "0x1e950f"), "Signature in success status 1 not recognized!"
        assert (chip_is_found(success_1)), "'Chip is found' did not trigger on success status 1!"
        assert (get_fuses_from_success_status(success_1) == {"E":5, 'H':218, 'L':255}), "Fuses are not extracted correctly from successs status 1!"
        # Successful status 1 - produced with 'ProHelper'
        success_2 = {'errors': [], 'full_output': '\r\navrdude: set SCK frequency to 187500 Hz\r\navrdude: AVR device initialized and ready to accept instructions\r\n\r\n\rReading |                                                    | 0% 0.00s\rReading | #################                                  | 33% 0.00s\rReading | #################################                  | 66% 0.01s\rReading | ################################################## | 100% 0.01s\r\n\r\navrdude: Device signature = 0x1e950f\r\navrdude: reading flash memory:\r\n\r\n\rReading |                                                    | 0% 0.00s\rReading | #                                                  | 1% 0.34s\rReading | #                                                  | 2% 0.68s\rReading | ##                                                 | 3% 0.90s\rReading | ##                                                 | 4% 1.24s\rReading | ###                                                | 5% 1.47s\rReading | ###                                                | 6% 1.81s\rReading | ####                                               | 7% 2.03s\rReading | ####                                               | 8% 2.37s\rReading | #####                                              | 9% 2.71s\rReading | #####                                              | 10% 2.94s\rReading | ######                                             | 11% 3.28s\rReading | ######                                             | 12% 3.50s\rReading | #######                                            | 13% 3.84s\rReading | #######                                            | 14% 4.07s\rReading | ########                                           | 15% 4.41s\rReading | ########                                           | 16% 4.63s\rReading | #########                                          | 17% 4.97s\rReading | #########                                          | 18% 5.31s\rReading | ##########                                         | 19% 5.54s\rReading | ##########                                         | 20% 5.88s\rReading | ###########                                        | 21% 6.10s\rReading | ###########                                        | 22% 6.44s\rReading | ############                                       | 23% 6.67s\rReading | ############                                       | 24% 7.01s\rReading | #############                                      | 25% 7.23s\rReading | #############                                      | 26% 7.57s\rReading | ##############                                     | 27% 7.91s\rReading | ##############                                     | 28% 8.14s\rReading | ###############                                    | 29% 8.47s\rReading | ###############                                    | 30% 8.70s\rReading | ################                                   | 31% 9.04s\rReading | ################                                   | 32% 9.27s\rReading | #################                                  | 33% 9.60s\rReading | #################                                  | 34% 9.94s\rReading | ##################                                 | 35% 10.17s\rReading | ##################                                 | 36% 10.51s\rReading | ###################                                | 37% 10.73s\rReading | ###################                                | 38% 11.07s\rReading | ####################                               | 39% 11.30s\rReading | ####################                               | 40% 11.64s\rReading | #####################                              | 41% 11.86s\rReading | #####################                              | 42% 12.20s\rReading | ######################                             | 43% 12.54s\rReading | ######################                             | 44% 12.77s\rReading | #######################                            | 45% 13.11s\rReading | #######################                            | 46% 13.33s\rReading | ########################                           | 47% 13.67s\rReading | ########################                           | 48% 13.90s\rReading | #########################                          | 49% 14.24s\rReading | #########################                          | 50% 14.46s\rReading | ##########################                         | 51% 14.80s\rReading | ##########################                         | 52% 15.14s\rReading | ###########################                        | 53% 15.37s\rReading | ###########################                        | 54% 15.71s\rReading | ############################                       | 55% 15.93s\rReading | ############################                       | 56% 16.27s\rReading | #############################                      | 57% 16.50s\rReading | #############################                      | 58% 16.84s\rReading | ##############################                     | 59% 17.18s\rReading | ##############################                     | 60% 17.40s\rReading | ###############################                    | 61% 17.74s\rReading | ###############################                    | 62% 17.97s\rReading | ################################                   | 63% 18.31s\rReading | ################################                   | 64% 18.53s\rReading | #################################                  | 65% 18.87s\rReading | #################################                  | 66% 19.10s\rReading | ##################################                 | 67% 19.44s\rReading | ##################################                 | 68% 19.77s\rReading | ###################################                | 69% 20.00s\rReading | ###################################                | 70% 20.34s\rReading | ####################################               | 71% 20.57s\rReading | ####################################               | 72% 20.90s\rReading | #####################################              | 73% 21.13s\rReading | #####################################              | 74% 21.47s\rReading | ######################################             | 75% 21.70s\rReading | ######################################             | 76% 22.03s\rReading | #######################################            | 77% 22.37s\rReading | #######################################            | 78% 22.60s\rReading | ########################################           | 79% 22.94s\rReading | ########################################           | 80% 23.16s\rReading | #########################################          | 81% 23.50s\rReading | #########################################          | 82% 23.73s\rReading | ##########################################         | 83% 24.07s\rReading | ##########################################         | 84% 24.41s\rReading | ###########################################        | 85% 24.63s\rReading | ###########################################        | 86% 24.97s\rReading | ############################################       | 87% 25.20s\rReading | ############################################       | 88% 25.54s\rReading | #############################################      | 89% 25.76s\rReading | #############################################      | 90% 26.10s\rReading | ##############################################     | 91% 26.33s\rReading | ##############################################     | 92% 26.67s\rReading | ###############################################    | 93% 27.01s\rReading | ###############################################    | 94% 27.23s\rReading | ################################################   | 95% 27.57s\rReading | ################################################   | 96% 27.80s\rReading | #################################################  | 97% 28.14s\rReading | #################################################  | 98% 28.36s\rReading | ################################################## | 99% 28.70s\rReading | ################################################## | 100% 28.93s\r\n\r\navrdude: writing output file "text.hex"\r\n\r\navrdude: safemode: Fuses OK (E:05, H:DA, L:FF)\r\n\r\navrdude done.  Thank you.\r\n\r\n', 'commandline': ['avrdude', '-p', 'm328p', '-c', 'usbasp', '-B', '5', '-U', 'flash:r:text.hex:i'], 'other': ['Reading |                                                    | 0% 0.00s\rReading | #################                                  | 33% 0.00s\rReading | #################################                  | 66% 0.01s\rReading | ################################################## | 100% 0.01s', 'Reading |                                                    | 0% 0.00s\rReading | #                                                  | 1% 0.34s\rReading | #                                                  | 2% 0.68s\rReading | ##                                                 | 3% 0.90s\rReading | ##                                                 | 4% 1.24s\rReading | ###                                                | 5% 1.47s\rReading | ###                                                | 6% 1.81s\rReading | ####                                               | 7% 2.03s\rReading | ####                                               | 8% 2.37s\rReading | #####                                              | 9% 2.71s\rReading | #####                                              | 10% 2.94s\rReading | ######                                             | 11% 3.28s\rReading | ######                                             | 12% 3.50s\rReading | #######                                            | 13% 3.84s\rReading | #######                                            | 14% 4.07s\rReading | ########                                           | 15% 4.41s\rReading | ########                                           | 16% 4.63s\rReading | #########                                          | 17% 4.97s\rReading | #########                                          | 18% 5.31s\rReading | ##########                                         | 19% 5.54s\rReading | ##########                                         | 20% 5.88s\rReading | ###########                                        | 21% 6.10s\rReading | ###########                                        | 22% 6.44s\rReading | ############                                       | 23% 6.67s\rReading | ############                                       | 24% 7.01s\rReading | #############                                      | 25% 7.23s\rReading | #############                                      | 26% 7.57s\rReading | ##############                                     | 27% 7.91s\rReading | ##############                                     | 28% 8.14s\rReading | ###############                                    | 29% 8.47s\rReading | ###############                                    | 30% 8.70s\rReading | ################                                   | 31% 9.04s\rReading | ################                                   | 32% 9.27s\rReading | #################                                  | 33% 9.60s\rReading | #################                                  | 34% 9.94s\rReading | ##################                                 | 35% 10.17s\rReading | ##################                                 | 36% 10.51s\rReading | ###################                                | 37% 10.73s\rReading | ###################                                | 38% 11.07s\rReading | ####################                               | 39% 11.30s\rReading | ####################                               | 40% 11.64s\rReading | #####################                              | 41% 11.86s\rReading | #####################                              | 42% 12.20s\rReading | ######################                             | 43% 12.54s\rReading | ######################                             | 44% 12.77s\rReading | #######################                            | 45% 13.11s\rReading | #######################                            | 46% 13.33s\rReading | ########################                           | 47% 13.67s\rReading | ########################                           | 48% 13.90s\rReading | #########################                          | 49% 14.24s\rReading | #########################                          | 50% 14.46s\rReading | ##########################                         | 51% 14.80s\rReading | ##########################                         | 52% 15.14s\rReading | ###########################                        | 53% 15.37s\rReading | ###########################                        | 54% 15.71s\rReading | ############################                       | 55% 15.93s\rReading | ############################                       | 56% 16.27s\rReading | #############################                      | 57% 16.50s\rReading | #############################                      | 58% 16.84s\rReading | ##############################                     | 59% 17.18s\rReading | ##############################                     | 60% 17.40s\rReading | ###############################                    | 61% 17.74s\rReading | ###############################                    | 62% 17.97s\rReading | ################################                   | 63% 18.31s\rReading | ################################                   | 64% 18.53s\rReading | #################################                  | 65% 18.87s\rReading | #################################                  | 66% 19.10s\rReading | ##################################                 | 67% 19.44s\rReading | ##################################                 | 68% 19.77s\rReading | ###################################                | 69% 20.00s\rReading | ###################################                | 70% 20.34s\rReading | ####################################               | 71% 20.57s\rReading | ####################################               | 72% 20.90s\rReading | #####################################              | 73% 21.13s\rReading | #####################################              | 74% 21.47s\rReading | ######################################             | 75% 21.70s\rReading | ######################################             | 76% 22.03s\rReading | #######################################            | 77% 22.37s\rReading | #######################################            | 78% 22.60s\rReading | ########################################           | 79% 22.94s\rReading | ########################################           | 80% 23.16s\rReading | #########################################          | 81% 23.50s\rReading | #########################################          | 82% 23.73s\rReading | ##########################################         | 83% 24.07s\rReading | ##########################################         | 84% 24.41s\rReading | ###########################################        | 85% 24.63s\rReading | ###########################################        | 86% 24.97s\rReading | ############################################       | 87% 25.20s\rReading | ############################################       | 88% 25.54s\rReading | #############################################      | 89% 25.76s\rReading | #############################################      | 90% 26.10s\rReading | ##############################################     | 91% 26.33s\rReading | ##############################################     | 92% 26.67s\rReading | ###############################################    | 93% 27.01s\rReading | ###############################################    | 94% 27.23s\rReading | ################################################   | 95% 27.57s\rReading | ################################################   | 96% 27.80s\rReading | #################################################  | 97% 28.14s\rReading | #################################################  | 98% 28.36s\rReading | ################################################## | 99% 28.70s\rReading | ################################################## | 100% 28.93s', 'avrdude done.  Thank you.'], 'result': 'success', 'output': ['set SCK frequency to 187500 Hz', 'AVR device initialized and ready to accept instructions', 'Device signature = 0x1e950f', 'reading flash memory:', 'writing output file "text.hex"', 'safemode: Fuses OK (E:05, H:DA, L:FF)'], 'exitcode': 0}
        assert (get_human_readable_status(success_2) == ["Success"]), "Success status 2 not recognized!"
        assert (get_found_signature(success_2) == "0x1e950f"), "Signature in success status 2 not recognized!"
        assert (chip_is_found(success_2)), "'Chip is found' did not trigger on success status 2!"
        assert (get_fuses_from_success_status(success_2) == {"E":5, 'H':218, 'L':255}), "Fuses are not extracted correctly from success status 2!"
        avrispmk2_not_found_status = {'errors': [], 'full_output': '\navrdude done.  Thank you.\n\n', 'commandline': ['avrdude', '-p', u'm328p', '-c', u'avrispmkII', '-B', '1'], 'other': ['avrdude done.  Thank you.'], 'result': 'failed', 'output': [], 'exitcode': 1}
        assert (get_human_readable_status(avrispmk2_not_found_status) == ['Failure', 'ISPMK2 not found!']), "'AVR ISP MKII not found' status not recognized!"

if __name__ == "__main__":
    # Here be tests
    unittest.main()
