"""
Submodules for boto3 services.

Copyright 2025 Vlad Emelianov
"""

import importlib
import importlib.util
from typing import List


class Submodule:
    def __init__(
        self,
        module_name: str,
        import_name: str,
        boto3_name: str,
        class_name: str,
        pypi_name: str,
        has_resource: bool,
        has_waiter: bool,
        has_paginator: bool,
    ):
        self.module_name = module_name
        self.import_name = import_name
        self.boto3_name = boto3_name
        self.class_name = class_name
        self.pypi_name = pypi_name
        self.has_resource = has_resource
        self.has_waiter = has_waiter
        self.has_paginator = has_paginator
        self.is_installed = importlib.util.find_spec(module_name) is not None
        self.is_active = self.is_installed

    def get_all_names(self) -> List[str]:
        service_module = importlib.import_module(self.module_name)
        return getattr(service_module, "__all__", [])


SUBMODULES: List[Submodule] = [
    Submodule(
        module_name="mypy_boto3_accessanalyzer",
        import_name="accessanalyzer",
        boto3_name="accessanalyzer",
        class_name="AccessAnalyzer",
        pypi_name="mypy-boto3-accessanalyzer",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_account",
        import_name="account",
        boto3_name="account",
        class_name="Account",
        pypi_name="mypy-boto3-account",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_acm",
        import_name="acm",
        boto3_name="acm",
        class_name="ACM",
        pypi_name="mypy-boto3-acm",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_acm_pca",
        import_name="acm_pca",
        boto3_name="acm-pca",
        class_name="ACMPCA",
        pypi_name="mypy-boto3-acm-pca",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_aiops",
        import_name="aiops",
        boto3_name="aiops",
        class_name="AIOps",
        pypi_name="mypy-boto3-aiops",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_amp",
        import_name="amp",
        boto3_name="amp",
        class_name="PrometheusService",
        pypi_name="mypy-boto3-amp",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_amplify",
        import_name="amplify",
        boto3_name="amplify",
        class_name="Amplify",
        pypi_name="mypy-boto3-amplify",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_amplifybackend",
        import_name="amplifybackend",
        boto3_name="amplifybackend",
        class_name="AmplifyBackend",
        pypi_name="mypy-boto3-amplifybackend",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_amplifyuibuilder",
        import_name="amplifyuibuilder",
        boto3_name="amplifyuibuilder",
        class_name="AmplifyUIBuilder",
        pypi_name="mypy-boto3-amplifyuibuilder",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_apigateway",
        import_name="apigateway",
        boto3_name="apigateway",
        class_name="APIGateway",
        pypi_name="mypy-boto3-apigateway",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_apigatewaymanagementapi",
        import_name="apigatewaymanagementapi",
        boto3_name="apigatewaymanagementapi",
        class_name="ApiGatewayManagementApi",
        pypi_name="mypy-boto3-apigatewaymanagementapi",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_apigatewayv2",
        import_name="apigatewayv2",
        boto3_name="apigatewayv2",
        class_name="ApiGatewayV2",
        pypi_name="mypy-boto3-apigatewayv2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_appconfig",
        import_name="appconfig",
        boto3_name="appconfig",
        class_name="AppConfig",
        pypi_name="mypy-boto3-appconfig",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_appconfigdata",
        import_name="appconfigdata",
        boto3_name="appconfigdata",
        class_name="AppConfigData",
        pypi_name="mypy-boto3-appconfigdata",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_appfabric",
        import_name="appfabric",
        boto3_name="appfabric",
        class_name="AppFabric",
        pypi_name="mypy-boto3-appfabric",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_appflow",
        import_name="appflow",
        boto3_name="appflow",
        class_name="Appflow",
        pypi_name="mypy-boto3-appflow",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_appintegrations",
        import_name="appintegrations",
        boto3_name="appintegrations",
        class_name="AppIntegrationsService",
        pypi_name="mypy-boto3-appintegrations",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_application_autoscaling",
        import_name="application_autoscaling",
        boto3_name="application-autoscaling",
        class_name="ApplicationAutoScaling",
        pypi_name="mypy-boto3-application-autoscaling",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_application_insights",
        import_name="application_insights",
        boto3_name="application-insights",
        class_name="ApplicationInsights",
        pypi_name="mypy-boto3-application-insights",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_application_signals",
        import_name="application_signals",
        boto3_name="application-signals",
        class_name="CloudWatchApplicationSignals",
        pypi_name="mypy-boto3-application-signals",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_applicationcostprofiler",
        import_name="applicationcostprofiler",
        boto3_name="applicationcostprofiler",
        class_name="ApplicationCostProfiler",
        pypi_name="mypy-boto3-applicationcostprofiler",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_appmesh",
        import_name="appmesh",
        boto3_name="appmesh",
        class_name="AppMesh",
        pypi_name="mypy-boto3-appmesh",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_apprunner",
        import_name="apprunner",
        boto3_name="apprunner",
        class_name="AppRunner",
        pypi_name="mypy-boto3-apprunner",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_appstream",
        import_name="appstream",
        boto3_name="appstream",
        class_name="AppStream",
        pypi_name="mypy-boto3-appstream",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_appsync",
        import_name="appsync",
        boto3_name="appsync",
        class_name="AppSync",
        pypi_name="mypy-boto3-appsync",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_arc_region_switch",
        import_name="arc_region_switch",
        boto3_name="arc-region-switch",
        class_name="ARCRegionswitch",
        pypi_name="mypy-boto3-arc-region-switch",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_arc_zonal_shift",
        import_name="arc_zonal_shift",
        boto3_name="arc-zonal-shift",
        class_name="ARCZonalShift",
        pypi_name="mypy-boto3-arc-zonal-shift",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_artifact",
        import_name="artifact",
        boto3_name="artifact",
        class_name="Artifact",
        pypi_name="mypy-boto3-artifact",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_athena",
        import_name="athena",
        boto3_name="athena",
        class_name="Athena",
        pypi_name="mypy-boto3-athena",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_auditmanager",
        import_name="auditmanager",
        boto3_name="auditmanager",
        class_name="AuditManager",
        pypi_name="mypy-boto3-auditmanager",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_autoscaling",
        import_name="autoscaling",
        boto3_name="autoscaling",
        class_name="AutoScaling",
        pypi_name="mypy-boto3-autoscaling",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_autoscaling_plans",
        import_name="autoscaling_plans",
        boto3_name="autoscaling-plans",
        class_name="AutoScalingPlans",
        pypi_name="mypy-boto3-autoscaling-plans",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_b2bi",
        import_name="b2bi",
        boto3_name="b2bi",
        class_name="B2BI",
        pypi_name="mypy-boto3-b2bi",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_backup",
        import_name="backup",
        boto3_name="backup",
        class_name="Backup",
        pypi_name="mypy-boto3-backup",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_backup_gateway",
        import_name="backup_gateway",
        boto3_name="backup-gateway",
        class_name="BackupGateway",
        pypi_name="mypy-boto3-backup-gateway",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_backupsearch",
        import_name="backupsearch",
        boto3_name="backupsearch",
        class_name="BackupSearch",
        pypi_name="mypy-boto3-backupsearch",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_batch",
        import_name="batch",
        boto3_name="batch",
        class_name="Batch",
        pypi_name="mypy-boto3-batch",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bcm_dashboards",
        import_name="bcm_dashboards",
        boto3_name="bcm-dashboards",
        class_name="BillingandCostManagementDashboards",
        pypi_name="mypy-boto3-bcm-dashboards",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bcm_data_exports",
        import_name="bcm_data_exports",
        boto3_name="bcm-data-exports",
        class_name="BillingandCostManagementDataExports",
        pypi_name="mypy-boto3-bcm-data-exports",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bcm_pricing_calculator",
        import_name="bcm_pricing_calculator",
        boto3_name="bcm-pricing-calculator",
        class_name="BillingandCostManagementPricingCalculator",
        pypi_name="mypy-boto3-bcm-pricing-calculator",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bcm_recommended_actions",
        import_name="bcm_recommended_actions",
        boto3_name="bcm-recommended-actions",
        class_name="BillingandCostManagementRecommendedActions",
        pypi_name="mypy-boto3-bcm-recommended-actions",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bedrock",
        import_name="bedrock",
        boto3_name="bedrock",
        class_name="Bedrock",
        pypi_name="mypy-boto3-bedrock",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bedrock_agent",
        import_name="bedrock_agent",
        boto3_name="bedrock-agent",
        class_name="AgentsforBedrock",
        pypi_name="mypy-boto3-bedrock-agent",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bedrock_agent_runtime",
        import_name="bedrock_agent_runtime",
        boto3_name="bedrock-agent-runtime",
        class_name="AgentsforBedrockRuntime",
        pypi_name="mypy-boto3-bedrock-agent-runtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bedrock_agentcore",
        import_name="bedrock_agentcore",
        boto3_name="bedrock-agentcore",
        class_name="BedrockAgentCore",
        pypi_name="mypy-boto3-bedrock-agentcore",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bedrock_agentcore_control",
        import_name="bedrock_agentcore_control",
        boto3_name="bedrock-agentcore-control",
        class_name="BedrockAgentCoreControl",
        pypi_name="mypy-boto3-bedrock-agentcore-control",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bedrock_data_automation",
        import_name="bedrock_data_automation",
        boto3_name="bedrock-data-automation",
        class_name="DataAutomationforBedrock",
        pypi_name="mypy-boto3-bedrock-data-automation",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_bedrock_data_automation_runtime",
        import_name="bedrock_data_automation_runtime",
        boto3_name="bedrock-data-automation-runtime",
        class_name="RuntimeforBedrockDataAutomation",
        pypi_name="mypy-boto3-bedrock-data-automation-runtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_bedrock_runtime",
        import_name="bedrock_runtime",
        boto3_name="bedrock-runtime",
        class_name="BedrockRuntime",
        pypi_name="mypy-boto3-bedrock-runtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_billing",
        import_name="billing",
        boto3_name="billing",
        class_name="Billing",
        pypi_name="mypy-boto3-billing",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_billingconductor",
        import_name="billingconductor",
        boto3_name="billingconductor",
        class_name="BillingConductor",
        pypi_name="mypy-boto3-billingconductor",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_braket",
        import_name="braket",
        boto3_name="braket",
        class_name="Braket",
        pypi_name="mypy-boto3-braket",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_budgets",
        import_name="budgets",
        boto3_name="budgets",
        class_name="Budgets",
        pypi_name="mypy-boto3-budgets",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ce",
        import_name="ce",
        boto3_name="ce",
        class_name="CostExplorer",
        pypi_name="mypy-boto3-ce",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_chatbot",
        import_name="chatbot",
        boto3_name="chatbot",
        class_name="Chatbot",
        pypi_name="mypy-boto3-chatbot",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_chime",
        import_name="chime",
        boto3_name="chime",
        class_name="Chime",
        pypi_name="mypy-boto3-chime",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_chime_sdk_identity",
        import_name="chime_sdk_identity",
        boto3_name="chime-sdk-identity",
        class_name="ChimeSDKIdentity",
        pypi_name="mypy-boto3-chime-sdk-identity",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_chime_sdk_media_pipelines",
        import_name="chime_sdk_media_pipelines",
        boto3_name="chime-sdk-media-pipelines",
        class_name="ChimeSDKMediaPipelines",
        pypi_name="mypy-boto3-chime-sdk-media-pipelines",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_chime_sdk_meetings",
        import_name="chime_sdk_meetings",
        boto3_name="chime-sdk-meetings",
        class_name="ChimeSDKMeetings",
        pypi_name="mypy-boto3-chime-sdk-meetings",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_chime_sdk_messaging",
        import_name="chime_sdk_messaging",
        boto3_name="chime-sdk-messaging",
        class_name="ChimeSDKMessaging",
        pypi_name="mypy-boto3-chime-sdk-messaging",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_chime_sdk_voice",
        import_name="chime_sdk_voice",
        boto3_name="chime-sdk-voice",
        class_name="ChimeSDKVoice",
        pypi_name="mypy-boto3-chime-sdk-voice",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cleanrooms",
        import_name="cleanrooms",
        boto3_name="cleanrooms",
        class_name="CleanRoomsService",
        pypi_name="mypy-boto3-cleanrooms",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cleanroomsml",
        import_name="cleanroomsml",
        boto3_name="cleanroomsml",
        class_name="CleanRoomsML",
        pypi_name="mypy-boto3-cleanroomsml",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cloud9",
        import_name="cloud9",
        boto3_name="cloud9",
        class_name="Cloud9",
        pypi_name="mypy-boto3-cloud9",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cloudcontrol",
        import_name="cloudcontrol",
        boto3_name="cloudcontrol",
        class_name="CloudControlApi",
        pypi_name="mypy-boto3-cloudcontrol",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_clouddirectory",
        import_name="clouddirectory",
        boto3_name="clouddirectory",
        class_name="CloudDirectory",
        pypi_name="mypy-boto3-clouddirectory",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cloudformation",
        import_name="cloudformation",
        boto3_name="cloudformation",
        class_name="CloudFormation",
        pypi_name="mypy-boto3-cloudformation",
        has_resource=True,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cloudfront",
        import_name="cloudfront",
        boto3_name="cloudfront",
        class_name="CloudFront",
        pypi_name="mypy-boto3-cloudfront",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cloudfront_keyvaluestore",
        import_name="cloudfront_keyvaluestore",
        boto3_name="cloudfront-keyvaluestore",
        class_name="CloudFrontKeyValueStore",
        pypi_name="mypy-boto3-cloudfront-keyvaluestore",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cloudhsm",
        import_name="cloudhsm",
        boto3_name="cloudhsm",
        class_name="CloudHSM",
        pypi_name="mypy-boto3-cloudhsm",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cloudhsmv2",
        import_name="cloudhsmv2",
        boto3_name="cloudhsmv2",
        class_name="CloudHSMV2",
        pypi_name="mypy-boto3-cloudhsmv2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cloudsearch",
        import_name="cloudsearch",
        boto3_name="cloudsearch",
        class_name="CloudSearch",
        pypi_name="mypy-boto3-cloudsearch",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_cloudsearchdomain",
        import_name="cloudsearchdomain",
        boto3_name="cloudsearchdomain",
        class_name="CloudSearchDomain",
        pypi_name="mypy-boto3-cloudsearchdomain",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_cloudtrail",
        import_name="cloudtrail",
        boto3_name="cloudtrail",
        class_name="CloudTrail",
        pypi_name="mypy-boto3-cloudtrail",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cloudtrail_data",
        import_name="cloudtrail_data",
        boto3_name="cloudtrail-data",
        class_name="CloudTrailDataService",
        pypi_name="mypy-boto3-cloudtrail-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_cloudwatch",
        import_name="cloudwatch",
        boto3_name="cloudwatch",
        class_name="CloudWatch",
        pypi_name="mypy-boto3-cloudwatch",
        has_resource=True,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codeartifact",
        import_name="codeartifact",
        boto3_name="codeartifact",
        class_name="CodeArtifact",
        pypi_name="mypy-boto3-codeartifact",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codebuild",
        import_name="codebuild",
        boto3_name="codebuild",
        class_name="CodeBuild",
        pypi_name="mypy-boto3-codebuild",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codecatalyst",
        import_name="codecatalyst",
        boto3_name="codecatalyst",
        class_name="CodeCatalyst",
        pypi_name="mypy-boto3-codecatalyst",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codecommit",
        import_name="codecommit",
        boto3_name="codecommit",
        class_name="CodeCommit",
        pypi_name="mypy-boto3-codecommit",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codeconnections",
        import_name="codeconnections",
        boto3_name="codeconnections",
        class_name="CodeConnections",
        pypi_name="mypy-boto3-codeconnections",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_codedeploy",
        import_name="codedeploy",
        boto3_name="codedeploy",
        class_name="CodeDeploy",
        pypi_name="mypy-boto3-codedeploy",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codeguru_reviewer",
        import_name="codeguru_reviewer",
        boto3_name="codeguru-reviewer",
        class_name="CodeGuruReviewer",
        pypi_name="mypy-boto3-codeguru-reviewer",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codeguru_security",
        import_name="codeguru_security",
        boto3_name="codeguru-security",
        class_name="CodeGuruSecurity",
        pypi_name="mypy-boto3-codeguru-security",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codeguruprofiler",
        import_name="codeguruprofiler",
        boto3_name="codeguruprofiler",
        class_name="CodeGuruProfiler",
        pypi_name="mypy-boto3-codeguruprofiler",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codepipeline",
        import_name="codepipeline",
        boto3_name="codepipeline",
        class_name="CodePipeline",
        pypi_name="mypy-boto3-codepipeline",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_codestar_connections",
        import_name="codestar_connections",
        boto3_name="codestar-connections",
        class_name="CodeStarconnections",
        pypi_name="mypy-boto3-codestar-connections",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_codestar_notifications",
        import_name="codestar_notifications",
        boto3_name="codestar-notifications",
        class_name="CodeStarNotifications",
        pypi_name="mypy-boto3-codestar-notifications",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cognito_identity",
        import_name="cognito_identity",
        boto3_name="cognito-identity",
        class_name="CognitoIdentity",
        pypi_name="mypy-boto3-cognito-identity",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cognito_idp",
        import_name="cognito_idp",
        boto3_name="cognito-idp",
        class_name="CognitoIdentityProvider",
        pypi_name="mypy-boto3-cognito-idp",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cognito_sync",
        import_name="cognito_sync",
        boto3_name="cognito-sync",
        class_name="CognitoSync",
        pypi_name="mypy-boto3-cognito-sync",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_comprehend",
        import_name="comprehend",
        boto3_name="comprehend",
        class_name="Comprehend",
        pypi_name="mypy-boto3-comprehend",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_comprehendmedical",
        import_name="comprehendmedical",
        boto3_name="comprehendmedical",
        class_name="ComprehendMedical",
        pypi_name="mypy-boto3-comprehendmedical",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_compute_optimizer",
        import_name="compute_optimizer",
        boto3_name="compute-optimizer",
        class_name="ComputeOptimizer",
        pypi_name="mypy-boto3-compute-optimizer",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_config",
        import_name="config",
        boto3_name="config",
        class_name="ConfigService",
        pypi_name="mypy-boto3-config",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_connect",
        import_name="connect",
        boto3_name="connect",
        class_name="Connect",
        pypi_name="mypy-boto3-connect",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_connect_contact_lens",
        import_name="connect_contact_lens",
        boto3_name="connect-contact-lens",
        class_name="ConnectContactLens",
        pypi_name="mypy-boto3-connect-contact-lens",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_connectcampaigns",
        import_name="connectcampaigns",
        boto3_name="connectcampaigns",
        class_name="ConnectCampaignService",
        pypi_name="mypy-boto3-connectcampaigns",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_connectcampaignsv2",
        import_name="connectcampaignsv2",
        boto3_name="connectcampaignsv2",
        class_name="ConnectCampaignServiceV2",
        pypi_name="mypy-boto3-connectcampaignsv2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_connectcases",
        import_name="connectcases",
        boto3_name="connectcases",
        class_name="ConnectCases",
        pypi_name="mypy-boto3-connectcases",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_connectparticipant",
        import_name="connectparticipant",
        boto3_name="connectparticipant",
        class_name="ConnectParticipant",
        pypi_name="mypy-boto3-connectparticipant",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_controlcatalog",
        import_name="controlcatalog",
        boto3_name="controlcatalog",
        class_name="ControlCatalog",
        pypi_name="mypy-boto3-controlcatalog",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_controltower",
        import_name="controltower",
        boto3_name="controltower",
        class_name="ControlTower",
        pypi_name="mypy-boto3-controltower",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cost_optimization_hub",
        import_name="cost_optimization_hub",
        boto3_name="cost-optimization-hub",
        class_name="CostOptimizationHub",
        pypi_name="mypy-boto3-cost-optimization-hub",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_cur",
        import_name="cur",
        boto3_name="cur",
        class_name="CostandUsageReportService",
        pypi_name="mypy-boto3-cur",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_customer_profiles",
        import_name="customer_profiles",
        boto3_name="customer-profiles",
        class_name="CustomerProfiles",
        pypi_name="mypy-boto3-customer-profiles",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_databrew",
        import_name="databrew",
        boto3_name="databrew",
        class_name="GlueDataBrew",
        pypi_name="mypy-boto3-databrew",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_dataexchange",
        import_name="dataexchange",
        boto3_name="dataexchange",
        class_name="DataExchange",
        pypi_name="mypy-boto3-dataexchange",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_datapipeline",
        import_name="datapipeline",
        boto3_name="datapipeline",
        class_name="DataPipeline",
        pypi_name="mypy-boto3-datapipeline",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_datasync",
        import_name="datasync",
        boto3_name="datasync",
        class_name="DataSync",
        pypi_name="mypy-boto3-datasync",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_datazone",
        import_name="datazone",
        boto3_name="datazone",
        class_name="DataZone",
        pypi_name="mypy-boto3-datazone",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_dax",
        import_name="dax",
        boto3_name="dax",
        class_name="DAX",
        pypi_name="mypy-boto3-dax",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_deadline",
        import_name="deadline",
        boto3_name="deadline",
        class_name="DeadlineCloud",
        pypi_name="mypy-boto3-deadline",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_detective",
        import_name="detective",
        boto3_name="detective",
        class_name="Detective",
        pypi_name="mypy-boto3-detective",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_devicefarm",
        import_name="devicefarm",
        boto3_name="devicefarm",
        class_name="DeviceFarm",
        pypi_name="mypy-boto3-devicefarm",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_devops_guru",
        import_name="devops_guru",
        boto3_name="devops-guru",
        class_name="DevOpsGuru",
        pypi_name="mypy-boto3-devops-guru",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_directconnect",
        import_name="directconnect",
        boto3_name="directconnect",
        class_name="DirectConnect",
        pypi_name="mypy-boto3-directconnect",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_discovery",
        import_name="discovery",
        boto3_name="discovery",
        class_name="ApplicationDiscoveryService",
        pypi_name="mypy-boto3-discovery",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_dlm",
        import_name="dlm",
        boto3_name="dlm",
        class_name="DLM",
        pypi_name="mypy-boto3-dlm",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_dms",
        import_name="dms",
        boto3_name="dms",
        class_name="DatabaseMigrationService",
        pypi_name="mypy-boto3-dms",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_docdb",
        import_name="docdb",
        boto3_name="docdb",
        class_name="DocDB",
        pypi_name="mypy-boto3-docdb",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_docdb_elastic",
        import_name="docdb_elastic",
        boto3_name="docdb-elastic",
        class_name="DocDBElastic",
        pypi_name="mypy-boto3-docdb-elastic",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_drs",
        import_name="drs",
        boto3_name="drs",
        class_name="Drs",
        pypi_name="mypy-boto3-drs",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ds",
        import_name="ds",
        boto3_name="ds",
        class_name="DirectoryService",
        pypi_name="mypy-boto3-ds",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ds_data",
        import_name="ds_data",
        boto3_name="ds-data",
        class_name="DirectoryServiceData",
        pypi_name="mypy-boto3-ds-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_dsql",
        import_name="dsql",
        boto3_name="dsql",
        class_name="AuroraDSQL",
        pypi_name="mypy-boto3-dsql",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_dynamodb",
        import_name="dynamodb",
        boto3_name="dynamodb",
        class_name="DynamoDB",
        pypi_name="mypy-boto3-dynamodb",
        has_resource=True,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_dynamodbstreams",
        import_name="dynamodbstreams",
        boto3_name="dynamodbstreams",
        class_name="DynamoDBStreams",
        pypi_name="mypy-boto3-dynamodbstreams",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_ebs",
        import_name="ebs",
        boto3_name="ebs",
        class_name="EBS",
        pypi_name="mypy-boto3-ebs",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_ec2",
        import_name="ec2",
        boto3_name="ec2",
        class_name="EC2",
        pypi_name="mypy-boto3-ec2",
        has_resource=True,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ec2_instance_connect",
        import_name="ec2_instance_connect",
        boto3_name="ec2-instance-connect",
        class_name="EC2InstanceConnect",
        pypi_name="mypy-boto3-ec2-instance-connect",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_ecr",
        import_name="ecr",
        boto3_name="ecr",
        class_name="ECR",
        pypi_name="mypy-boto3-ecr",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ecr_public",
        import_name="ecr_public",
        boto3_name="ecr-public",
        class_name="ECRPublic",
        pypi_name="mypy-boto3-ecr-public",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ecs",
        import_name="ecs",
        boto3_name="ecs",
        class_name="ECS",
        pypi_name="mypy-boto3-ecs",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_efs",
        import_name="efs",
        boto3_name="efs",
        class_name="EFS",
        pypi_name="mypy-boto3-efs",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_eks",
        import_name="eks",
        boto3_name="eks",
        class_name="EKS",
        pypi_name="mypy-boto3-eks",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_eks_auth",
        import_name="eks_auth",
        boto3_name="eks-auth",
        class_name="EKSAuth",
        pypi_name="mypy-boto3-eks-auth",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_elasticache",
        import_name="elasticache",
        boto3_name="elasticache",
        class_name="ElastiCache",
        pypi_name="mypy-boto3-elasticache",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_elasticbeanstalk",
        import_name="elasticbeanstalk",
        boto3_name="elasticbeanstalk",
        class_name="ElasticBeanstalk",
        pypi_name="mypy-boto3-elasticbeanstalk",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_elastictranscoder",
        import_name="elastictranscoder",
        boto3_name="elastictranscoder",
        class_name="ElasticTranscoder",
        pypi_name="mypy-boto3-elastictranscoder",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_elb",
        import_name="elb",
        boto3_name="elb",
        class_name="ElasticLoadBalancing",
        pypi_name="mypy-boto3-elb",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_elbv2",
        import_name="elbv2",
        boto3_name="elbv2",
        class_name="ElasticLoadBalancingv2",
        pypi_name="mypy-boto3-elbv2",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_emr",
        import_name="emr",
        boto3_name="emr",
        class_name="EMR",
        pypi_name="mypy-boto3-emr",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_emr_containers",
        import_name="emr_containers",
        boto3_name="emr-containers",
        class_name="EMRContainers",
        pypi_name="mypy-boto3-emr-containers",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_emr_serverless",
        import_name="emr_serverless",
        boto3_name="emr-serverless",
        class_name="EMRServerless",
        pypi_name="mypy-boto3-emr-serverless",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_entityresolution",
        import_name="entityresolution",
        boto3_name="entityresolution",
        class_name="EntityResolution",
        pypi_name="mypy-boto3-entityresolution",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_es",
        import_name="es",
        boto3_name="es",
        class_name="ElasticsearchService",
        pypi_name="mypy-boto3-es",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_events",
        import_name="events",
        boto3_name="events",
        class_name="EventBridge",
        pypi_name="mypy-boto3-events",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_evidently",
        import_name="evidently",
        boto3_name="evidently",
        class_name="CloudWatchEvidently",
        pypi_name="mypy-boto3-evidently",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_evs",
        import_name="evs",
        boto3_name="evs",
        class_name="EVS",
        pypi_name="mypy-boto3-evs",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_finspace",
        import_name="finspace",
        boto3_name="finspace",
        class_name="Finspace",
        pypi_name="mypy-boto3-finspace",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_finspace_data",
        import_name="finspace_data",
        boto3_name="finspace-data",
        class_name="FinSpaceData",
        pypi_name="mypy-boto3-finspace-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_firehose",
        import_name="firehose",
        boto3_name="firehose",
        class_name="Firehose",
        pypi_name="mypy-boto3-firehose",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_fis",
        import_name="fis",
        boto3_name="fis",
        class_name="FIS",
        pypi_name="mypy-boto3-fis",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_fms",
        import_name="fms",
        boto3_name="fms",
        class_name="FMS",
        pypi_name="mypy-boto3-fms",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_forecast",
        import_name="forecast",
        boto3_name="forecast",
        class_name="ForecastService",
        pypi_name="mypy-boto3-forecast",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_forecastquery",
        import_name="forecastquery",
        boto3_name="forecastquery",
        class_name="ForecastQueryService",
        pypi_name="mypy-boto3-forecastquery",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_frauddetector",
        import_name="frauddetector",
        boto3_name="frauddetector",
        class_name="FraudDetector",
        pypi_name="mypy-boto3-frauddetector",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_freetier",
        import_name="freetier",
        boto3_name="freetier",
        class_name="FreeTier",
        pypi_name="mypy-boto3-freetier",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_fsx",
        import_name="fsx",
        boto3_name="fsx",
        class_name="FSx",
        pypi_name="mypy-boto3-fsx",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_gamelift",
        import_name="gamelift",
        boto3_name="gamelift",
        class_name="GameLift",
        pypi_name="mypy-boto3-gamelift",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_gameliftstreams",
        import_name="gameliftstreams",
        boto3_name="gameliftstreams",
        class_name="GameLiftStreams",
        pypi_name="mypy-boto3-gameliftstreams",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_geo_maps",
        import_name="geo_maps",
        boto3_name="geo-maps",
        class_name="LocationServiceMapsV2",
        pypi_name="mypy-boto3-geo-maps",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_geo_places",
        import_name="geo_places",
        boto3_name="geo-places",
        class_name="LocationServicePlacesV2",
        pypi_name="mypy-boto3-geo-places",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_geo_routes",
        import_name="geo_routes",
        boto3_name="geo-routes",
        class_name="LocationServiceRoutesV2",
        pypi_name="mypy-boto3-geo-routes",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_glacier",
        import_name="glacier",
        boto3_name="glacier",
        class_name="Glacier",
        pypi_name="mypy-boto3-glacier",
        has_resource=True,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_globalaccelerator",
        import_name="globalaccelerator",
        boto3_name="globalaccelerator",
        class_name="GlobalAccelerator",
        pypi_name="mypy-boto3-globalaccelerator",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_glue",
        import_name="glue",
        boto3_name="glue",
        class_name="Glue",
        pypi_name="mypy-boto3-glue",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_grafana",
        import_name="grafana",
        boto3_name="grafana",
        class_name="ManagedGrafana",
        pypi_name="mypy-boto3-grafana",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_greengrass",
        import_name="greengrass",
        boto3_name="greengrass",
        class_name="Greengrass",
        pypi_name="mypy-boto3-greengrass",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_greengrassv2",
        import_name="greengrassv2",
        boto3_name="greengrassv2",
        class_name="GreengrassV2",
        pypi_name="mypy-boto3-greengrassv2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_groundstation",
        import_name="groundstation",
        boto3_name="groundstation",
        class_name="GroundStation",
        pypi_name="mypy-boto3-groundstation",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_guardduty",
        import_name="guardduty",
        boto3_name="guardduty",
        class_name="GuardDuty",
        pypi_name="mypy-boto3-guardduty",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_health",
        import_name="health",
        boto3_name="health",
        class_name="Health",
        pypi_name="mypy-boto3-health",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_healthlake",
        import_name="healthlake",
        boto3_name="healthlake",
        class_name="HealthLake",
        pypi_name="mypy-boto3-healthlake",
        has_resource=False,
        has_waiter=True,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_iam",
        import_name="iam",
        boto3_name="iam",
        class_name="IAM",
        pypi_name="mypy-boto3-iam",
        has_resource=True,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_identitystore",
        import_name="identitystore",
        boto3_name="identitystore",
        class_name="IdentityStore",
        pypi_name="mypy-boto3-identitystore",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_imagebuilder",
        import_name="imagebuilder",
        boto3_name="imagebuilder",
        class_name="Imagebuilder",
        pypi_name="mypy-boto3-imagebuilder",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_importexport",
        import_name="importexport",
        boto3_name="importexport",
        class_name="ImportExport",
        pypi_name="mypy-boto3-importexport",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_inspector",
        import_name="inspector",
        boto3_name="inspector",
        class_name="Inspector",
        pypi_name="mypy-boto3-inspector",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_inspector_scan",
        import_name="inspector_scan",
        boto3_name="inspector-scan",
        class_name="Inspectorscan",
        pypi_name="mypy-boto3-inspector-scan",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_inspector2",
        import_name="inspector2",
        boto3_name="inspector2",
        class_name="Inspector2",
        pypi_name="mypy-boto3-inspector2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_internetmonitor",
        import_name="internetmonitor",
        boto3_name="internetmonitor",
        class_name="CloudWatchInternetMonitor",
        pypi_name="mypy-boto3-internetmonitor",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_invoicing",
        import_name="invoicing",
        boto3_name="invoicing",
        class_name="Invoicing",
        pypi_name="mypy-boto3-invoicing",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_iot",
        import_name="iot",
        boto3_name="iot",
        class_name="IoT",
        pypi_name="mypy-boto3-iot",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_iot_data",
        import_name="iot_data",
        boto3_name="iot-data",
        class_name="IoTDataPlane",
        pypi_name="mypy-boto3-iot-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_iot_jobs_data",
        import_name="iot_jobs_data",
        boto3_name="iot-jobs-data",
        class_name="IoTJobsDataPlane",
        pypi_name="mypy-boto3-iot-jobs-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_iot_managed_integrations",
        import_name="iot_managed_integrations",
        boto3_name="iot-managed-integrations",
        class_name="ManagedintegrationsforIoTDeviceManagement",
        pypi_name="mypy-boto3-iot-managed-integrations",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_iotanalytics",
        import_name="iotanalytics",
        boto3_name="iotanalytics",
        class_name="IoTAnalytics",
        pypi_name="mypy-boto3-iotanalytics",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_iotdeviceadvisor",
        import_name="iotdeviceadvisor",
        boto3_name="iotdeviceadvisor",
        class_name="IoTDeviceAdvisor",
        pypi_name="mypy-boto3-iotdeviceadvisor",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_iotevents",
        import_name="iotevents",
        boto3_name="iotevents",
        class_name="IoTEvents",
        pypi_name="mypy-boto3-iotevents",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_iotevents_data",
        import_name="iotevents_data",
        boto3_name="iotevents-data",
        class_name="IoTEventsData",
        pypi_name="mypy-boto3-iotevents-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_iotfleetwise",
        import_name="iotfleetwise",
        boto3_name="iotfleetwise",
        class_name="IoTFleetWise",
        pypi_name="mypy-boto3-iotfleetwise",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_iotsecuretunneling",
        import_name="iotsecuretunneling",
        boto3_name="iotsecuretunneling",
        class_name="IoTSecureTunneling",
        pypi_name="mypy-boto3-iotsecuretunneling",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_iotsitewise",
        import_name="iotsitewise",
        boto3_name="iotsitewise",
        class_name="IoTSiteWise",
        pypi_name="mypy-boto3-iotsitewise",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_iotthingsgraph",
        import_name="iotthingsgraph",
        boto3_name="iotthingsgraph",
        class_name="IoTThingsGraph",
        pypi_name="mypy-boto3-iotthingsgraph",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_iottwinmaker",
        import_name="iottwinmaker",
        boto3_name="iottwinmaker",
        class_name="IoTTwinMaker",
        pypi_name="mypy-boto3-iottwinmaker",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_iotwireless",
        import_name="iotwireless",
        boto3_name="iotwireless",
        class_name="IoTWireless",
        pypi_name="mypy-boto3-iotwireless",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_ivs",
        import_name="ivs",
        boto3_name="ivs",
        class_name="IVS",
        pypi_name="mypy-boto3-ivs",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ivs_realtime",
        import_name="ivs_realtime",
        boto3_name="ivs-realtime",
        class_name="Ivsrealtime",
        pypi_name="mypy-boto3-ivs-realtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ivschat",
        import_name="ivschat",
        boto3_name="ivschat",
        class_name="Ivschat",
        pypi_name="mypy-boto3-ivschat",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_kafka",
        import_name="kafka",
        boto3_name="kafka",
        class_name="Kafka",
        pypi_name="mypy-boto3-kafka",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_kafkaconnect",
        import_name="kafkaconnect",
        boto3_name="kafkaconnect",
        class_name="KafkaConnect",
        pypi_name="mypy-boto3-kafkaconnect",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_kendra",
        import_name="kendra",
        boto3_name="kendra",
        class_name="Kendra",
        pypi_name="mypy-boto3-kendra",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_kendra_ranking",
        import_name="kendra_ranking",
        boto3_name="kendra-ranking",
        class_name="KendraRanking",
        pypi_name="mypy-boto3-kendra-ranking",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_keyspaces",
        import_name="keyspaces",
        boto3_name="keyspaces",
        class_name="Keyspaces",
        pypi_name="mypy-boto3-keyspaces",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_keyspacesstreams",
        import_name="keyspacesstreams",
        boto3_name="keyspacesstreams",
        class_name="KeyspacesStreams",
        pypi_name="mypy-boto3-keyspacesstreams",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_kinesis",
        import_name="kinesis",
        boto3_name="kinesis",
        class_name="Kinesis",
        pypi_name="mypy-boto3-kinesis",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_kinesis_video_archived_media",
        import_name="kinesis_video_archived_media",
        boto3_name="kinesis-video-archived-media",
        class_name="KinesisVideoArchivedMedia",
        pypi_name="mypy-boto3-kinesis-video-archived-media",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_kinesis_video_media",
        import_name="kinesis_video_media",
        boto3_name="kinesis-video-media",
        class_name="KinesisVideoMedia",
        pypi_name="mypy-boto3-kinesis-video-media",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_kinesis_video_signaling",
        import_name="kinesis_video_signaling",
        boto3_name="kinesis-video-signaling",
        class_name="KinesisVideoSignalingChannels",
        pypi_name="mypy-boto3-kinesis-video-signaling",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_kinesis_video_webrtc_storage",
        import_name="kinesis_video_webrtc_storage",
        boto3_name="kinesis-video-webrtc-storage",
        class_name="KinesisVideoWebRTCStorage",
        pypi_name="mypy-boto3-kinesis-video-webrtc-storage",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_kinesisanalytics",
        import_name="kinesisanalytics",
        boto3_name="kinesisanalytics",
        class_name="KinesisAnalytics",
        pypi_name="mypy-boto3-kinesisanalytics",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_kinesisanalyticsv2",
        import_name="kinesisanalyticsv2",
        boto3_name="kinesisanalyticsv2",
        class_name="KinesisAnalyticsV2",
        pypi_name="mypy-boto3-kinesisanalyticsv2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_kinesisvideo",
        import_name="kinesisvideo",
        boto3_name="kinesisvideo",
        class_name="KinesisVideo",
        pypi_name="mypy-boto3-kinesisvideo",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_kms",
        import_name="kms",
        boto3_name="kms",
        class_name="KMS",
        pypi_name="mypy-boto3-kms",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_lakeformation",
        import_name="lakeformation",
        boto3_name="lakeformation",
        class_name="LakeFormation",
        pypi_name="mypy-boto3-lakeformation",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_lambda",
        import_name="lambda_",
        boto3_name="lambda",
        class_name="Lambda",
        pypi_name="mypy-boto3-lambda",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_launch_wizard",
        import_name="launch_wizard",
        boto3_name="launch-wizard",
        class_name="LaunchWizard",
        pypi_name="mypy-boto3-launch-wizard",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_lex_models",
        import_name="lex_models",
        boto3_name="lex-models",
        class_name="LexModelBuildingService",
        pypi_name="mypy-boto3-lex-models",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_lex_runtime",
        import_name="lex_runtime",
        boto3_name="lex-runtime",
        class_name="LexRuntimeService",
        pypi_name="mypy-boto3-lex-runtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_lexv2_models",
        import_name="lexv2_models",
        boto3_name="lexv2-models",
        class_name="LexModelsV2",
        pypi_name="mypy-boto3-lexv2-models",
        has_resource=False,
        has_waiter=True,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_lexv2_runtime",
        import_name="lexv2_runtime",
        boto3_name="lexv2-runtime",
        class_name="LexRuntimeV2",
        pypi_name="mypy-boto3-lexv2-runtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_license_manager",
        import_name="license_manager",
        boto3_name="license-manager",
        class_name="LicenseManager",
        pypi_name="mypy-boto3-license-manager",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_license_manager_linux_subscriptions",
        import_name="license_manager_linux_subscriptions",
        boto3_name="license-manager-linux-subscriptions",
        class_name="LicenseManagerLinuxSubscriptions",
        pypi_name="mypy-boto3-license-manager-linux-subscriptions",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_license_manager_user_subscriptions",
        import_name="license_manager_user_subscriptions",
        boto3_name="license-manager-user-subscriptions",
        class_name="LicenseManagerUserSubscriptions",
        pypi_name="mypy-boto3-license-manager-user-subscriptions",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_lightsail",
        import_name="lightsail",
        boto3_name="lightsail",
        class_name="Lightsail",
        pypi_name="mypy-boto3-lightsail",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_location",
        import_name="location",
        boto3_name="location",
        class_name="LocationService",
        pypi_name="mypy-boto3-location",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_logs",
        import_name="logs",
        boto3_name="logs",
        class_name="CloudWatchLogs",
        pypi_name="mypy-boto3-logs",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_lookoutequipment",
        import_name="lookoutequipment",
        boto3_name="lookoutequipment",
        class_name="LookoutEquipment",
        pypi_name="mypy-boto3-lookoutequipment",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_m2",
        import_name="m2",
        boto3_name="m2",
        class_name="MainframeModernization",
        pypi_name="mypy-boto3-m2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_machinelearning",
        import_name="machinelearning",
        boto3_name="machinelearning",
        class_name="MachineLearning",
        pypi_name="mypy-boto3-machinelearning",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_macie2",
        import_name="macie2",
        boto3_name="macie2",
        class_name="Macie2",
        pypi_name="mypy-boto3-macie2",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mailmanager",
        import_name="mailmanager",
        boto3_name="mailmanager",
        class_name="MailManager",
        pypi_name="mypy-boto3-mailmanager",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_managedblockchain",
        import_name="managedblockchain",
        boto3_name="managedblockchain",
        class_name="ManagedBlockchain",
        pypi_name="mypy-boto3-managedblockchain",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_managedblockchain_query",
        import_name="managedblockchain_query",
        boto3_name="managedblockchain-query",
        class_name="ManagedBlockchainQuery",
        pypi_name="mypy-boto3-managedblockchain-query",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_marketplace_agreement",
        import_name="marketplace_agreement",
        boto3_name="marketplace-agreement",
        class_name="AgreementService",
        pypi_name="mypy-boto3-marketplace-agreement",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_marketplace_catalog",
        import_name="marketplace_catalog",
        boto3_name="marketplace-catalog",
        class_name="MarketplaceCatalog",
        pypi_name="mypy-boto3-marketplace-catalog",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_marketplace_deployment",
        import_name="marketplace_deployment",
        boto3_name="marketplace-deployment",
        class_name="MarketplaceDeploymentService",
        pypi_name="mypy-boto3-marketplace-deployment",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_marketplace_entitlement",
        import_name="marketplace_entitlement",
        boto3_name="marketplace-entitlement",
        class_name="MarketplaceEntitlementService",
        pypi_name="mypy-boto3-marketplace-entitlement",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_marketplace_reporting",
        import_name="marketplace_reporting",
        boto3_name="marketplace-reporting",
        class_name="MarketplaceReportingService",
        pypi_name="mypy-boto3-marketplace-reporting",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_marketplacecommerceanalytics",
        import_name="marketplacecommerceanalytics",
        boto3_name="marketplacecommerceanalytics",
        class_name="MarketplaceCommerceAnalytics",
        pypi_name="mypy-boto3-marketplacecommerceanalytics",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_mediaconnect",
        import_name="mediaconnect",
        boto3_name="mediaconnect",
        class_name="MediaConnect",
        pypi_name="mypy-boto3-mediaconnect",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mediaconvert",
        import_name="mediaconvert",
        boto3_name="mediaconvert",
        class_name="MediaConvert",
        pypi_name="mypy-boto3-mediaconvert",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_medialive",
        import_name="medialive",
        boto3_name="medialive",
        class_name="MediaLive",
        pypi_name="mypy-boto3-medialive",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mediapackage",
        import_name="mediapackage",
        boto3_name="mediapackage",
        class_name="MediaPackage",
        pypi_name="mypy-boto3-mediapackage",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mediapackage_vod",
        import_name="mediapackage_vod",
        boto3_name="mediapackage-vod",
        class_name="MediaPackageVod",
        pypi_name="mypy-boto3-mediapackage-vod",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mediapackagev2",
        import_name="mediapackagev2",
        boto3_name="mediapackagev2",
        class_name="Mediapackagev2",
        pypi_name="mypy-boto3-mediapackagev2",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mediastore",
        import_name="mediastore",
        boto3_name="mediastore",
        class_name="MediaStore",
        pypi_name="mypy-boto3-mediastore",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mediastore_data",
        import_name="mediastore_data",
        boto3_name="mediastore-data",
        class_name="MediaStoreData",
        pypi_name="mypy-boto3-mediastore-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mediatailor",
        import_name="mediatailor",
        boto3_name="mediatailor",
        class_name="MediaTailor",
        pypi_name="mypy-boto3-mediatailor",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_medical_imaging",
        import_name="medical_imaging",
        boto3_name="medical-imaging",
        class_name="HealthImaging",
        pypi_name="mypy-boto3-medical-imaging",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_memorydb",
        import_name="memorydb",
        boto3_name="memorydb",
        class_name="MemoryDB",
        pypi_name="mypy-boto3-memorydb",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_meteringmarketplace",
        import_name="meteringmarketplace",
        boto3_name="meteringmarketplace",
        class_name="MarketplaceMetering",
        pypi_name="mypy-boto3-meteringmarketplace",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_mgh",
        import_name="mgh",
        boto3_name="mgh",
        class_name="MigrationHub",
        pypi_name="mypy-boto3-mgh",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mgn",
        import_name="mgn",
        boto3_name="mgn",
        class_name="Mgn",
        pypi_name="mypy-boto3-mgn",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_migration_hub_refactor_spaces",
        import_name="migration_hub_refactor_spaces",
        boto3_name="migration-hub-refactor-spaces",
        class_name="MigrationHubRefactorSpaces",
        pypi_name="mypy-boto3-migration-hub-refactor-spaces",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_migrationhub_config",
        import_name="migrationhub_config",
        boto3_name="migrationhub-config",
        class_name="MigrationHubConfig",
        pypi_name="mypy-boto3-migrationhub-config",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_migrationhuborchestrator",
        import_name="migrationhuborchestrator",
        boto3_name="migrationhuborchestrator",
        class_name="MigrationHubOrchestrator",
        pypi_name="mypy-boto3-migrationhuborchestrator",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_migrationhubstrategy",
        import_name="migrationhubstrategy",
        boto3_name="migrationhubstrategy",
        class_name="MigrationHubStrategyRecommendations",
        pypi_name="mypy-boto3-migrationhubstrategy",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mpa",
        import_name="mpa",
        boto3_name="mpa",
        class_name="MultipartyApproval",
        pypi_name="mypy-boto3-mpa",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mq",
        import_name="mq",
        boto3_name="mq",
        class_name="MQ",
        pypi_name="mypy-boto3-mq",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mturk",
        import_name="mturk",
        boto3_name="mturk",
        class_name="MTurk",
        pypi_name="mypy-boto3-mturk",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mwaa",
        import_name="mwaa",
        boto3_name="mwaa",
        class_name="MWAA",
        pypi_name="mypy-boto3-mwaa",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_mwaa_serverless",
        import_name="mwaa_serverless",
        boto3_name="mwaa-serverless",
        class_name="MWAAServerless",
        pypi_name="mypy-boto3-mwaa-serverless",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_neptune",
        import_name="neptune",
        boto3_name="neptune",
        class_name="Neptune",
        pypi_name="mypy-boto3-neptune",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_neptune_graph",
        import_name="neptune_graph",
        boto3_name="neptune-graph",
        class_name="NeptuneGraph",
        pypi_name="mypy-boto3-neptune-graph",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_neptunedata",
        import_name="neptunedata",
        boto3_name="neptunedata",
        class_name="NeptuneData",
        pypi_name="mypy-boto3-neptunedata",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_network_firewall",
        import_name="network_firewall",
        boto3_name="network-firewall",
        class_name="NetworkFirewall",
        pypi_name="mypy-boto3-network-firewall",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_networkflowmonitor",
        import_name="networkflowmonitor",
        boto3_name="networkflowmonitor",
        class_name="NetworkFlowMonitor",
        pypi_name="mypy-boto3-networkflowmonitor",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_networkmanager",
        import_name="networkmanager",
        boto3_name="networkmanager",
        class_name="NetworkManager",
        pypi_name="mypy-boto3-networkmanager",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_networkmonitor",
        import_name="networkmonitor",
        boto3_name="networkmonitor",
        class_name="CloudWatchNetworkMonitor",
        pypi_name="mypy-boto3-networkmonitor",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_notifications",
        import_name="notifications",
        boto3_name="notifications",
        class_name="UserNotifications",
        pypi_name="mypy-boto3-notifications",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_notificationscontacts",
        import_name="notificationscontacts",
        boto3_name="notificationscontacts",
        class_name="UserNotificationsContacts",
        pypi_name="mypy-boto3-notificationscontacts",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_oam",
        import_name="oam",
        boto3_name="oam",
        class_name="CloudWatchObservabilityAccessManager",
        pypi_name="mypy-boto3-oam",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_observabilityadmin",
        import_name="observabilityadmin",
        boto3_name="observabilityadmin",
        class_name="CloudWatchObservabilityAdminService",
        pypi_name="mypy-boto3-observabilityadmin",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_odb",
        import_name="odb",
        boto3_name="odb",
        class_name="Odb",
        pypi_name="mypy-boto3-odb",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_omics",
        import_name="omics",
        boto3_name="omics",
        class_name="Omics",
        pypi_name="mypy-boto3-omics",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_opensearch",
        import_name="opensearch",
        boto3_name="opensearch",
        class_name="OpenSearchService",
        pypi_name="mypy-boto3-opensearch",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_opensearchserverless",
        import_name="opensearchserverless",
        boto3_name="opensearchserverless",
        class_name="OpenSearchServiceServerless",
        pypi_name="mypy-boto3-opensearchserverless",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_organizations",
        import_name="organizations",
        boto3_name="organizations",
        class_name="Organizations",
        pypi_name="mypy-boto3-organizations",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_osis",
        import_name="osis",
        boto3_name="osis",
        class_name="OpenSearchIngestion",
        pypi_name="mypy-boto3-osis",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_outposts",
        import_name="outposts",
        boto3_name="outposts",
        class_name="Outposts",
        pypi_name="mypy-boto3-outposts",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_panorama",
        import_name="panorama",
        boto3_name="panorama",
        class_name="Panorama",
        pypi_name="mypy-boto3-panorama",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_partnercentral_channel",
        import_name="partnercentral_channel",
        boto3_name="partnercentral-channel",
        class_name="PartnerCentralChannelAPI",
        pypi_name="mypy-boto3-partnercentral-channel",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_partnercentral_selling",
        import_name="partnercentral_selling",
        boto3_name="partnercentral-selling",
        class_name="PartnerCentralSellingAPI",
        pypi_name="mypy-boto3-partnercentral-selling",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_payment_cryptography",
        import_name="payment_cryptography",
        boto3_name="payment-cryptography",
        class_name="PaymentCryptographyControlPlane",
        pypi_name="mypy-boto3-payment-cryptography",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_payment_cryptography_data",
        import_name="payment_cryptography_data",
        boto3_name="payment-cryptography-data",
        class_name="PaymentCryptographyDataPlane",
        pypi_name="mypy-boto3-payment-cryptography-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_pca_connector_ad",
        import_name="pca_connector_ad",
        boto3_name="pca-connector-ad",
        class_name="PcaConnectorAd",
        pypi_name="mypy-boto3-pca-connector-ad",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_pca_connector_scep",
        import_name="pca_connector_scep",
        boto3_name="pca-connector-scep",
        class_name="PrivateCAConnectorforSCEP",
        pypi_name="mypy-boto3-pca-connector-scep",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_pcs",
        import_name="pcs",
        boto3_name="pcs",
        class_name="ParallelComputingService",
        pypi_name="mypy-boto3-pcs",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_personalize",
        import_name="personalize",
        boto3_name="personalize",
        class_name="Personalize",
        pypi_name="mypy-boto3-personalize",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_personalize_events",
        import_name="personalize_events",
        boto3_name="personalize-events",
        class_name="PersonalizeEvents",
        pypi_name="mypy-boto3-personalize-events",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_personalize_runtime",
        import_name="personalize_runtime",
        boto3_name="personalize-runtime",
        class_name="PersonalizeRuntime",
        pypi_name="mypy-boto3-personalize-runtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_pi",
        import_name="pi",
        boto3_name="pi",
        class_name="PI",
        pypi_name="mypy-boto3-pi",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_pinpoint",
        import_name="pinpoint",
        boto3_name="pinpoint",
        class_name="Pinpoint",
        pypi_name="mypy-boto3-pinpoint",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_pinpoint_email",
        import_name="pinpoint_email",
        boto3_name="pinpoint-email",
        class_name="PinpointEmail",
        pypi_name="mypy-boto3-pinpoint-email",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_pinpoint_sms_voice",
        import_name="pinpoint_sms_voice",
        boto3_name="pinpoint-sms-voice",
        class_name="PinpointSMSVoice",
        pypi_name="mypy-boto3-pinpoint-sms-voice",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_pinpoint_sms_voice_v2",
        import_name="pinpoint_sms_voice_v2",
        boto3_name="pinpoint-sms-voice-v2",
        class_name="PinpointSMSVoiceV2",
        pypi_name="mypy-boto3-pinpoint-sms-voice-v2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_pipes",
        import_name="pipes",
        boto3_name="pipes",
        class_name="EventBridgePipes",
        pypi_name="mypy-boto3-pipes",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_polly",
        import_name="polly",
        boto3_name="polly",
        class_name="Polly",
        pypi_name="mypy-boto3-polly",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_pricing",
        import_name="pricing",
        boto3_name="pricing",
        class_name="Pricing",
        pypi_name="mypy-boto3-pricing",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_proton",
        import_name="proton",
        boto3_name="proton",
        class_name="Proton",
        pypi_name="mypy-boto3-proton",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_qapps",
        import_name="qapps",
        boto3_name="qapps",
        class_name="QApps",
        pypi_name="mypy-boto3-qapps",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_qbusiness",
        import_name="qbusiness",
        boto3_name="qbusiness",
        class_name="QBusiness",
        pypi_name="mypy-boto3-qbusiness",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_qconnect",
        import_name="qconnect",
        boto3_name="qconnect",
        class_name="QConnect",
        pypi_name="mypy-boto3-qconnect",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_quicksight",
        import_name="quicksight",
        boto3_name="quicksight",
        class_name="QuickSight",
        pypi_name="mypy-boto3-quicksight",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ram",
        import_name="ram",
        boto3_name="ram",
        class_name="RAM",
        pypi_name="mypy-boto3-ram",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_rbin",
        import_name="rbin",
        boto3_name="rbin",
        class_name="RecycleBin",
        pypi_name="mypy-boto3-rbin",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_rds",
        import_name="rds",
        boto3_name="rds",
        class_name="RDS",
        pypi_name="mypy-boto3-rds",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_rds_data",
        import_name="rds_data",
        boto3_name="rds-data",
        class_name="RDSDataService",
        pypi_name="mypy-boto3-rds-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_redshift",
        import_name="redshift",
        boto3_name="redshift",
        class_name="Redshift",
        pypi_name="mypy-boto3-redshift",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_redshift_data",
        import_name="redshift_data",
        boto3_name="redshift-data",
        class_name="RedshiftDataAPIService",
        pypi_name="mypy-boto3-redshift-data",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_redshift_serverless",
        import_name="redshift_serverless",
        boto3_name="redshift-serverless",
        class_name="RedshiftServerless",
        pypi_name="mypy-boto3-redshift-serverless",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_rekognition",
        import_name="rekognition",
        boto3_name="rekognition",
        class_name="Rekognition",
        pypi_name="mypy-boto3-rekognition",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_repostspace",
        import_name="repostspace",
        boto3_name="repostspace",
        class_name="RePostPrivate",
        pypi_name="mypy-boto3-repostspace",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_resiliencehub",
        import_name="resiliencehub",
        boto3_name="resiliencehub",
        class_name="ResilienceHub",
        pypi_name="mypy-boto3-resiliencehub",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_resource_explorer_2",
        import_name="resource_explorer_2",
        boto3_name="resource-explorer-2",
        class_name="ResourceExplorer",
        pypi_name="mypy-boto3-resource-explorer-2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_resource_groups",
        import_name="resource_groups",
        boto3_name="resource-groups",
        class_name="ResourceGroups",
        pypi_name="mypy-boto3-resource-groups",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_resourcegroupstaggingapi",
        import_name="resourcegroupstaggingapi",
        boto3_name="resourcegroupstaggingapi",
        class_name="ResourceGroupsTaggingAPI",
        pypi_name="mypy-boto3-resourcegroupstaggingapi",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_rolesanywhere",
        import_name="rolesanywhere",
        boto3_name="rolesanywhere",
        class_name="IAMRolesAnywhere",
        pypi_name="mypy-boto3-rolesanywhere",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_route53",
        import_name="route53",
        boto3_name="route53",
        class_name="Route53",
        pypi_name="mypy-boto3-route53",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_route53_recovery_cluster",
        import_name="route53_recovery_cluster",
        boto3_name="route53-recovery-cluster",
        class_name="Route53RecoveryCluster",
        pypi_name="mypy-boto3-route53-recovery-cluster",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_route53_recovery_control_config",
        import_name="route53_recovery_control_config",
        boto3_name="route53-recovery-control-config",
        class_name="Route53RecoveryControlConfig",
        pypi_name="mypy-boto3-route53-recovery-control-config",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_route53_recovery_readiness",
        import_name="route53_recovery_readiness",
        boto3_name="route53-recovery-readiness",
        class_name="Route53RecoveryReadiness",
        pypi_name="mypy-boto3-route53-recovery-readiness",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_route53domains",
        import_name="route53domains",
        boto3_name="route53domains",
        class_name="Route53Domains",
        pypi_name="mypy-boto3-route53domains",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_route53profiles",
        import_name="route53profiles",
        boto3_name="route53profiles",
        class_name="Route53Profiles",
        pypi_name="mypy-boto3-route53profiles",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_route53resolver",
        import_name="route53resolver",
        boto3_name="route53resolver",
        class_name="Route53Resolver",
        pypi_name="mypy-boto3-route53resolver",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_rtbfabric",
        import_name="rtbfabric",
        boto3_name="rtbfabric",
        class_name="RTBFabric",
        pypi_name="mypy-boto3-rtbfabric",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_rum",
        import_name="rum",
        boto3_name="rum",
        class_name="CloudWatchRUM",
        pypi_name="mypy-boto3-rum",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_s3",
        import_name="s3",
        boto3_name="s3",
        class_name="S3",
        pypi_name="mypy-boto3-s3",
        has_resource=True,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_s3control",
        import_name="s3control",
        boto3_name="s3control",
        class_name="S3Control",
        pypi_name="mypy-boto3-s3control",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_s3outposts",
        import_name="s3outposts",
        boto3_name="s3outposts",
        class_name="S3Outposts",
        pypi_name="mypy-boto3-s3outposts",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_s3tables",
        import_name="s3tables",
        boto3_name="s3tables",
        class_name="S3Tables",
        pypi_name="mypy-boto3-s3tables",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_s3vectors",
        import_name="s3vectors",
        boto3_name="s3vectors",
        class_name="S3Vectors",
        pypi_name="mypy-boto3-s3vectors",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sagemaker",
        import_name="sagemaker",
        boto3_name="sagemaker",
        class_name="SageMaker",
        pypi_name="mypy-boto3-sagemaker",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sagemaker_a2i_runtime",
        import_name="sagemaker_a2i_runtime",
        boto3_name="sagemaker-a2i-runtime",
        class_name="AugmentedAIRuntime",
        pypi_name="mypy-boto3-sagemaker-a2i-runtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sagemaker_edge",
        import_name="sagemaker_edge",
        boto3_name="sagemaker-edge",
        class_name="SagemakerEdgeManager",
        pypi_name="mypy-boto3-sagemaker-edge",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_sagemaker_featurestore_runtime",
        import_name="sagemaker_featurestore_runtime",
        boto3_name="sagemaker-featurestore-runtime",
        class_name="SageMakerFeatureStoreRuntime",
        pypi_name="mypy-boto3-sagemaker-featurestore-runtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_sagemaker_geospatial",
        import_name="sagemaker_geospatial",
        boto3_name="sagemaker-geospatial",
        class_name="SageMakergeospatialcapabilities",
        pypi_name="mypy-boto3-sagemaker-geospatial",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sagemaker_metrics",
        import_name="sagemaker_metrics",
        boto3_name="sagemaker-metrics",
        class_name="SageMakerMetrics",
        pypi_name="mypy-boto3-sagemaker-metrics",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_sagemaker_runtime",
        import_name="sagemaker_runtime",
        boto3_name="sagemaker-runtime",
        class_name="SageMakerRuntime",
        pypi_name="mypy-boto3-sagemaker-runtime",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_savingsplans",
        import_name="savingsplans",
        boto3_name="savingsplans",
        class_name="SavingsPlans",
        pypi_name="mypy-boto3-savingsplans",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_scheduler",
        import_name="scheduler",
        boto3_name="scheduler",
        class_name="EventBridgeScheduler",
        pypi_name="mypy-boto3-scheduler",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_schemas",
        import_name="schemas",
        boto3_name="schemas",
        class_name="Schemas",
        pypi_name="mypy-boto3-schemas",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sdb",
        import_name="sdb",
        boto3_name="sdb",
        class_name="SimpleDB",
        pypi_name="mypy-boto3-sdb",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_secretsmanager",
        import_name="secretsmanager",
        boto3_name="secretsmanager",
        class_name="SecretsManager",
        pypi_name="mypy-boto3-secretsmanager",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_security_ir",
        import_name="security_ir",
        boto3_name="security-ir",
        class_name="SecurityIncidentResponse",
        pypi_name="mypy-boto3-security-ir",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_securityhub",
        import_name="securityhub",
        boto3_name="securityhub",
        class_name="SecurityHub",
        pypi_name="mypy-boto3-securityhub",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_securitylake",
        import_name="securitylake",
        boto3_name="securitylake",
        class_name="SecurityLake",
        pypi_name="mypy-boto3-securitylake",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_serverlessrepo",
        import_name="serverlessrepo",
        boto3_name="serverlessrepo",
        class_name="ServerlessApplicationRepository",
        pypi_name="mypy-boto3-serverlessrepo",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_service_quotas",
        import_name="service_quotas",
        boto3_name="service-quotas",
        class_name="ServiceQuotas",
        pypi_name="mypy-boto3-service-quotas",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_servicecatalog",
        import_name="servicecatalog",
        boto3_name="servicecatalog",
        class_name="ServiceCatalog",
        pypi_name="mypy-boto3-servicecatalog",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_servicecatalog_appregistry",
        import_name="servicecatalog_appregistry",
        boto3_name="servicecatalog-appregistry",
        class_name="AppRegistry",
        pypi_name="mypy-boto3-servicecatalog-appregistry",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_servicediscovery",
        import_name="servicediscovery",
        boto3_name="servicediscovery",
        class_name="ServiceDiscovery",
        pypi_name="mypy-boto3-servicediscovery",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ses",
        import_name="ses",
        boto3_name="ses",
        class_name="SES",
        pypi_name="mypy-boto3-ses",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sesv2",
        import_name="sesv2",
        boto3_name="sesv2",
        class_name="SESV2",
        pypi_name="mypy-boto3-sesv2",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_shield",
        import_name="shield",
        boto3_name="shield",
        class_name="Shield",
        pypi_name="mypy-boto3-shield",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_signer",
        import_name="signer",
        boto3_name="signer",
        class_name="Signer",
        pypi_name="mypy-boto3-signer",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_signin",
        import_name="signin",
        boto3_name="signin",
        class_name="SignInService",
        pypi_name="mypy-boto3-signin",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_simspaceweaver",
        import_name="simspaceweaver",
        boto3_name="simspaceweaver",
        class_name="SimSpaceWeaver",
        pypi_name="mypy-boto3-simspaceweaver",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_snow_device_management",
        import_name="snow_device_management",
        boto3_name="snow-device-management",
        class_name="SnowDeviceManagement",
        pypi_name="mypy-boto3-snow-device-management",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_snowball",
        import_name="snowball",
        boto3_name="snowball",
        class_name="Snowball",
        pypi_name="mypy-boto3-snowball",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sns",
        import_name="sns",
        boto3_name="sns",
        class_name="SNS",
        pypi_name="mypy-boto3-sns",
        has_resource=True,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_socialmessaging",
        import_name="socialmessaging",
        boto3_name="socialmessaging",
        class_name="EndUserMessagingSocial",
        pypi_name="mypy-boto3-socialmessaging",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sqs",
        import_name="sqs",
        boto3_name="sqs",
        class_name="SQS",
        pypi_name="mypy-boto3-sqs",
        has_resource=True,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ssm",
        import_name="ssm",
        boto3_name="ssm",
        class_name="SSM",
        pypi_name="mypy-boto3-ssm",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ssm_contacts",
        import_name="ssm_contacts",
        boto3_name="ssm-contacts",
        class_name="SSMContacts",
        pypi_name="mypy-boto3-ssm-contacts",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ssm_guiconnect",
        import_name="ssm_guiconnect",
        boto3_name="ssm-guiconnect",
        class_name="SSMGUIConnect",
        pypi_name="mypy-boto3-ssm-guiconnect",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_ssm_incidents",
        import_name="ssm_incidents",
        boto3_name="ssm-incidents",
        class_name="SSMIncidents",
        pypi_name="mypy-boto3-ssm-incidents",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ssm_quicksetup",
        import_name="ssm_quicksetup",
        boto3_name="ssm-quicksetup",
        class_name="SystemsManagerQuickSetup",
        pypi_name="mypy-boto3-ssm-quicksetup",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_ssm_sap",
        import_name="ssm_sap",
        boto3_name="ssm-sap",
        class_name="SsmSap",
        pypi_name="mypy-boto3-ssm-sap",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sso",
        import_name="sso",
        boto3_name="sso",
        class_name="SSO",
        pypi_name="mypy-boto3-sso",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sso_admin",
        import_name="sso_admin",
        boto3_name="sso-admin",
        class_name="SSOAdmin",
        pypi_name="mypy-boto3-sso-admin",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sso_oidc",
        import_name="sso_oidc",
        boto3_name="sso-oidc",
        class_name="SSOOIDC",
        pypi_name="mypy-boto3-sso-oidc",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_stepfunctions",
        import_name="stepfunctions",
        boto3_name="stepfunctions",
        class_name="SFN",
        pypi_name="mypy-boto3-stepfunctions",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_storagegateway",
        import_name="storagegateway",
        boto3_name="storagegateway",
        class_name="StorageGateway",
        pypi_name="mypy-boto3-storagegateway",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_sts",
        import_name="sts",
        boto3_name="sts",
        class_name="STS",
        pypi_name="mypy-boto3-sts",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_supplychain",
        import_name="supplychain",
        boto3_name="supplychain",
        class_name="SupplyChain",
        pypi_name="mypy-boto3-supplychain",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_support",
        import_name="support",
        boto3_name="support",
        class_name="Support",
        pypi_name="mypy-boto3-support",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_support_app",
        import_name="support_app",
        boto3_name="support-app",
        class_name="SupportApp",
        pypi_name="mypy-boto3-support-app",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_swf",
        import_name="swf",
        boto3_name="swf",
        class_name="SWF",
        pypi_name="mypy-boto3-swf",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_synthetics",
        import_name="synthetics",
        boto3_name="synthetics",
        class_name="Synthetics",
        pypi_name="mypy-boto3-synthetics",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_taxsettings",
        import_name="taxsettings",
        boto3_name="taxsettings",
        class_name="TaxSettings",
        pypi_name="mypy-boto3-taxsettings",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_textract",
        import_name="textract",
        boto3_name="textract",
        class_name="Textract",
        pypi_name="mypy-boto3-textract",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_timestream_influxdb",
        import_name="timestream_influxdb",
        boto3_name="timestream-influxdb",
        class_name="TimestreamInfluxDB",
        pypi_name="mypy-boto3-timestream-influxdb",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_timestream_query",
        import_name="timestream_query",
        boto3_name="timestream-query",
        class_name="TimestreamQuery",
        pypi_name="mypy-boto3-timestream-query",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_timestream_write",
        import_name="timestream_write",
        boto3_name="timestream-write",
        class_name="TimestreamWrite",
        pypi_name="mypy-boto3-timestream-write",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_tnb",
        import_name="tnb",
        boto3_name="tnb",
        class_name="TelcoNetworkBuilder",
        pypi_name="mypy-boto3-tnb",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_transcribe",
        import_name="transcribe",
        boto3_name="transcribe",
        class_name="TranscribeService",
        pypi_name="mypy-boto3-transcribe",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_transfer",
        import_name="transfer",
        boto3_name="transfer",
        class_name="Transfer",
        pypi_name="mypy-boto3-transfer",
        has_resource=False,
        has_waiter=True,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_translate",
        import_name="translate",
        boto3_name="translate",
        class_name="Translate",
        pypi_name="mypy-boto3-translate",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_trustedadvisor",
        import_name="trustedadvisor",
        boto3_name="trustedadvisor",
        class_name="TrustedAdvisorPublicAPI",
        pypi_name="mypy-boto3-trustedadvisor",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_verifiedpermissions",
        import_name="verifiedpermissions",
        boto3_name="verifiedpermissions",
        class_name="VerifiedPermissions",
        pypi_name="mypy-boto3-verifiedpermissions",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_voice_id",
        import_name="voice_id",
        boto3_name="voice-id",
        class_name="VoiceID",
        pypi_name="mypy-boto3-voice-id",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_vpc_lattice",
        import_name="vpc_lattice",
        boto3_name="vpc-lattice",
        class_name="VPCLattice",
        pypi_name="mypy-boto3-vpc-lattice",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_waf",
        import_name="waf",
        boto3_name="waf",
        class_name="WAF",
        pypi_name="mypy-boto3-waf",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_waf_regional",
        import_name="waf_regional",
        boto3_name="waf-regional",
        class_name="WAFRegional",
        pypi_name="mypy-boto3-waf-regional",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_wafv2",
        import_name="wafv2",
        boto3_name="wafv2",
        class_name="WAFV2",
        pypi_name="mypy-boto3-wafv2",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_wellarchitected",
        import_name="wellarchitected",
        boto3_name="wellarchitected",
        class_name="WellArchitected",
        pypi_name="mypy-boto3-wellarchitected",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_wisdom",
        import_name="wisdom",
        boto3_name="wisdom",
        class_name="ConnectWisdomService",
        pypi_name="mypy-boto3-wisdom",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_workdocs",
        import_name="workdocs",
        boto3_name="workdocs",
        class_name="WorkDocs",
        pypi_name="mypy-boto3-workdocs",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_workmail",
        import_name="workmail",
        boto3_name="workmail",
        class_name="WorkMail",
        pypi_name="mypy-boto3-workmail",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_workmailmessageflow",
        import_name="workmailmessageflow",
        boto3_name="workmailmessageflow",
        class_name="WorkMailMessageFlow",
        pypi_name="mypy-boto3-workmailmessageflow",
        has_resource=False,
        has_waiter=False,
        has_paginator=False,
    ),
    Submodule(
        module_name="mypy_boto3_workspaces",
        import_name="workspaces",
        boto3_name="workspaces",
        class_name="WorkSpaces",
        pypi_name="mypy-boto3-workspaces",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_workspaces_instances",
        import_name="workspaces_instances",
        boto3_name="workspaces-instances",
        class_name="WorkspacesInstances",
        pypi_name="mypy-boto3-workspaces-instances",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_workspaces_thin_client",
        import_name="workspaces_thin_client",
        boto3_name="workspaces-thin-client",
        class_name="WorkSpacesThinClient",
        pypi_name="mypy-boto3-workspaces-thin-client",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_workspaces_web",
        import_name="workspaces_web",
        boto3_name="workspaces-web",
        class_name="WorkSpacesWeb",
        pypi_name="mypy-boto3-workspaces-web",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
    Submodule(
        module_name="mypy_boto3_xray",
        import_name="xray",
        boto3_name="xray",
        class_name="XRay",
        pypi_name="mypy-boto3-xray",
        has_resource=False,
        has_waiter=False,
        has_paginator=True,
    ),
]
