# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['awslambdalocal']

package_data = \
{'': ['*']}

install_requires = \
['python-dotenv>=0.19.2,<0.20.0']

setup_kwargs = {
    'name': 'awslambdalocal',
    'version': '1.0.3',
    'description': 'A tool to simulate running an AWS Lambda locally',
    'long_description': '# py-lambda-local\nawslambdalocal is a tool to simulate running an AWS Lambda locally, for lambda functions in Python.\n\n\n## Table of Contents\n\n* [Requirements](#requirements)\n* [Installation](#install)\n* [About: CLI](#about-cli)\n    * [Positional Arguments](#positional-arguments)\n    * [Optional Arguments](#optional-arguments)\n    * [CLI Examples](#cli-examples)\n* [Tutorials](#tutorials)\n    * [Debug Python in VSCode](#debug-python-in-vscode)\n\n\n## Requirements\n\n* Python => 3.8\n* Poetry => 1.1.12 or another package manager that supports direct git dependencies\n\n\n## Install\n\nTo install awslambdalocal, we recommend adding it to your pyproject.toml in the dev-dependencies section as shown in the example below.\n\n```bash\npip install awslambdalocal\n```\n**Obs.:** We recommend using Poetry. See https://python-poetry.org/docs/ \n\n\n## About: CLI\n\n### Positional Arguments:\n| Argument    | Description                                                 |\n|-------------|-------------------------------------------------------------|\n| lambda_path | Specify Lambda function file name.                          |\n| event_path  | Specify event data file name.                               |\n\n### Optional Arguments:\n| Argument    | Description                                                 |\n|-------------|-------------------------------------------------------------|\n| --help      | Show this help message and exit                             |\n| -h          | Lambda function handler name. Default is "lambda_handler"   |\n| -t          | Seconds until lambda function timeout. Default is 3 seconds |\n| -p          | Read the AWS profile of the file.                           |\n| -r          | Sets the AWS region, defaults to us-east-1.                 |\n\n\n### CLI Examples\n```sh\n# Simple usage\npyhton -m awslambdalocal main.py test-event.json\n\n# Input all arguments\npyhton -m awslambdalocal main.py test-event.json -p my_profile -r my_region -h lambda_handler -t 30\n```\n\n\n## Tutorials\n---\nThis session contains a collection of tutorials.\n\n### Debug Python in VSCode\nTo use vscode debug with awslambdalocal follow the steps below\n\n1. Click run and debug\n2. Click create a launch.json file\n\n    ![](data:image/png;base64,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)\n\n3. Choose Python\n    ![](data:image/png;base64,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)\n\n4. Choose Module\n    ![](data:image/png;base64,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)\n\n5. After this process, VSCode will create a file called launch.json in the .vscode folder located at the root of the project\n\n    ![](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAQEAAAAtCAYAAACu9d4EAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAcdvqGQAAASnSURBVHhe7Zy/axtJFMfvL3GXa+0iYGKIoiuCMViFuw3mSIQbHcLECNv4OC5CIZAE1o0PkuaKK1wFjjiYXOPCMdzhwn/Uu5nZ2Z03P9aRlPVE0nyLD2jfm9lZF+8zb0e2f7h370cCAKTB0tKSByQAQEJAAgAkDiTg0aFBPqBOMAfA4gEJeEACIC0alcDy6n1aDsQNK3R/dSUQnyUgAZAWDUrgCZ2cf6Gz450aEazQzsk/9OVTTlkwz1mj7jCnQYfF1ro0HHbpgfj84NmQ8jwv2OuYMZ2BibNCtsbnQ+qu6fGazl6Zk2s6EpDrVnMhB7B4NCgBwfIOvf8cEkEhgAshgJ1lHq9HFS4rcHk9fLZWFLqWAR9fCIAXfrf47MSLonYEwdYphFDmpRCYNOrWBmCOaVYCEk8EQgDHZxMJQMF2/qIz0MXoFHGJLF6rc1AEOgqBGesUucJ0Aq4g8KoAFpHmJSARIjg5vxAi+IV60whAwQpf7sDBtr8sSDa2ml8fr7qKYFE7EqheBUpC6wAwv9yNBCRKBP+pM4DJBVBQFmt4lxdU7Xl4x5+qE2CdhpEFywOwYNydBKbBbfXVK8GABgJ7t2Z5a9c248ozATfurqHOAOrOBNRY7PxgsZltCeidPPwq4LfmVvvODvCsuHV/iV5D571vB6z1BNYZAQDzz2xJAAAQHUgAgMSBBABIHEgAgMSBBABIHEgAgMQJSmB19SF9C1mWBeMAgNkDEgAgcSABABIHEgAgcSABABInqgQe91t0/Idgn8Wftujd+xa96LMYACAa8SSw36Krm5/o5qpNfzoSOP9XxEXulMeDbNNh/pJ6G6Hc3bLee0n5wXYwZ7HRp9F3ekYApiGeBN62RaG36a/nTlzyvBDEx7eBnMUcSACAOSOyBEShv3bikl8f0TUkAMB3IZ4E9OtAsNC1IE5/C+QsHAmo1tv8rf9hxuKjPq1X8zapNyrn6XtkbK5b3NmR+f8B+RFlIlZKIDso42w9jnqmYo685uNHvc1qHI+XaxS5MZ4PgAaJIwEhgPNL8d5/3aY3WyxestWij9cif9mmd7eeC9gSWO8dGSGowtXF9FUJiMKq8sV1VdD8PoL1Xt9IgI1T19YaGiaBuu7BmyvXdJ6n9vkAaJg4EtjVErh8RC9qJHAq8tcXk0nAz40rAfseclcud2n5OVRwXkE7O34wbhV3SehnGP/5AGiaOBKQ3Hb4V54XTHwmIK91y6yYTgKywIsi4+Ns/F1d3ucrEhCoed7O7s8z8rnt+UwMgKaIJwF9+PfhdyfOcpNJwC0WVlzfJIExOwG+HqemQzDz/fXHfz4TA6Ap4kmg6a8I3WKz3uXtQlIFWF3fXmTFWHNf60ygTgK87a+RAB+jDgW5pLwzAUgAxCOeBHb1Lwu5h39PW/RBnhdIQeyyeJBQcetXgYMjU5RObtTbnminte6ri1PFJpSAdR9HDEoEzhrmvpAAiEc8CQh+PmzR6d+OBLYe0qt8gX5t2HsVAWC2iSqBFPA7BgBmG0igIUzrHzgPAGCGgQQASBxIAIDEgQQASJygBEL/lhgAsJhAAgAkDiQAQOJAAgAkji+BJfofnO9DEy3n1gcAAAAASUVORK5CYII=)\n\n6. Copy and paste the json below into the launch.json file, this file aims to call the awslambdalocal module and passes the necessary and optional parameters as arguments\n\n    ```json\n    {\n        // Use o IntelliSense para saber mais sobre os atributos possíveis.\n        // Focalizar para exibir as descrições dos atributos existentes.\n        // Para obter mais informações, acesse: https://go.microsoft.com/fwlink/?linkid=830387\n        "version": "0.2.0",\n        "configurations": [\n            {\n                "name": "Lambda Local",\n                "type": "python",\n                "request": "launch",\n                "module": "awslambdalocal",\n                "args": [\n                    "file_python.py",\n                    "your_test_event.json",\n                    // Optional args ...\n                    "-h",\n                    "handler",\n                    "-t",\n                    "10",\n                    "-p",\n                    "your_profile",\n                    "-r",\n                    "us-east-1"\n                ]\n            }\n        ]\n    }\n    ```',
    'author': 'Miqueias BRS',
    'author_email': 'miqueias@capybaracode.com.br',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
