"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const run_task_1 = require("./run-task");
class IntegTesting {
    constructor() {
        const app = new aws_cdk_lib_1.App();
        const env = {
            account: process.env.CDK_DEFAULT_ACCOUNT || '123456789012',
            region: process.env.CDK_DEFAULT_REGION || 'us-east-1',
        };
        const stack = new aws_cdk_lib_1.Stack(app, 'run-task-demo-stack', { env });
        const task = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(stack, 'Task', { cpu: 256, memoryLimitMiB: 512 });
        task.addContainer('Ping', {
            image: aws_cdk_lib_1.aws_ecs.ContainerImage.fromRegistry('busybox'),
            command: [
                'sh', '-c',
                'ping -c 3 google.com',
            ],
            logging: new aws_cdk_lib_1.aws_ecs.AwsLogDriver({
                streamPrefix: 'Ping',
                logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(stack, 'LogGroup', {
                    retention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                }),
            }),
        });
        // deploy and run this task once
        const runTaskAtOnce = new run_task_1.RunTask(stack, 'RunDemoTaskOnce', {
            task,
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE_SPOT',
                    weight: 1,
                },
            ],
        });
        // or run it with schedule(every hour 0min)
        new run_task_1.RunTask(stack, 'RunDemoTaskEveryHour', {
            task,
            cluster: runTaskAtOnce.cluster,
            runAtOnce: false,
            schedule: aws_cdk_lib_1.aws_events.Schedule.cron({ minute: '0' }),
        });
        const vpc = getOrCreateVpc(stack);
        const existingCluster = aws_cdk_lib_1.aws_ecs.Cluster.fromClusterAttributes(stack, 'ExistingCluster', {
            clusterName: 'fargate',
            vpc,
            securityGroups: [new aws_cdk_lib_1.aws_ec2.SecurityGroup(stack, 'DummySG', { vpc })],
        });
        const externalTask = new aws_cdk_lib_1.aws_ecs.TaskDefinition(stack, 'ExternalTask', {
            cpu: '256',
            memoryMiB: '512',
            compatibility: aws_cdk_lib_1.aws_ecs.Compatibility.EXTERNAL,
        });
        externalTask.addContainer('ExternalPing', {
            image: aws_cdk_lib_1.aws_ecs.ContainerImage.fromRegistry('busybox'),
            command: [
                'sh', '-c',
                'ping -c 3 google.com',
            ],
            logging: new aws_cdk_lib_1.aws_ecs.AwsLogDriver({
                streamPrefix: 'Ping',
                logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(stack, 'ExternalLogGroup', {
                    retention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                }),
            }),
        });
        // run it once on external instance
        new run_task_1.RunTask(stack, 'RunDemoTaskFromExternal', {
            task: externalTask,
            cluster: existingCluster,
            launchType: run_task_1.LaunchType.EXTERNAL,
        });
        // run it by schedule  on external instance
        new run_task_1.RunTask(stack, 'RunDemoTaskFromExternalSchedule', {
            task: externalTask,
            cluster: existingCluster,
            launchType: run_task_1.LaunchType.EXTERNAL,
            runAtOnce: false,
            schedule: aws_cdk_lib_1.aws_events.Schedule.cron({ minute: '0' }),
        });
        app.synth();
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting;
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new aws_cdk_lib_1.aws_ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
//# sourceMappingURL=data:application/json;base64,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