"""Data models describing pipeline configuration and results."""

from __future__ import annotations

import os
import sys
import tempfile
from dataclasses import dataclass, field
from pathlib import Path
from typing import Optional


def default_temp_folder() -> Path:
    """Return an OS-appropriate default temporary workspace directory."""

    if sys.platform == "darwin":
        base = Path.home() / "Library" / "Application Support" / "talks-reducer"
    elif sys.platform == "win32":
        appdata = os.environ.get("LOCALAPPDATA") or os.environ.get("APPDATA")
        base = (
            Path(appdata)
            if appdata
            else Path.home() / "AppData" / "Local" / "talks-reducer"
        )
    else:
        xdg_runtime = os.environ.get("XDG_RUNTIME_DIR")
        if xdg_runtime:
            base = Path(xdg_runtime) / "talks-reducer"
        else:
            base = Path(tempfile.gettempdir()) / "talks-reducer"

    return base / "talks-reducer-temp"


@dataclass(frozen=True)
class ProcessingOptions:
    """Configuration values controlling how the talks reducer processes media.

    The fields map 1:1 to the command line options exposed by the CLI so the
    same configuration can be shared between user interfaces.
    """

    input_file: Path
    output_file: Optional[Path] = None
    frame_rate: float = 30.0
    sample_rate: int = 48000
    silent_threshold: float = 0.05
    silent_speed: float = 4.0
    sounded_speed: float = 1.0
    frame_spreadage: int = 2
    audio_fade_envelope_size: int = 400
    temp_folder: Path = field(default_factory=default_temp_folder)
    small: bool = False


@dataclass(frozen=True)
class ProcessingResult:
    """Summary of the metadata generated by a successful pipeline execution."""

    input_file: Path
    output_file: Path
    frame_rate: float
    original_duration: float
    output_duration: float
    chunk_count: int
    used_cuda: bool
    max_audio_volume: float
    time_ratio: Optional[float]
    size_ratio: Optional[float]
