'''
# Amazon ECS Construct Library

This package contains constructs for working with **Amazon Elastic Container
Service** (Amazon ECS).

Amazon Elastic Container Service (Amazon ECS) is a fully managed container orchestration service.

For further information on Amazon ECS,
see the [Amazon ECS documentation](https://docs.aws.amazon.com/ecs)

The following example creates an Amazon ECS cluster, adds capacity to it, and
runs a service on it:

```python
# vpc: ec2.Vpc


# Create an ECS cluster
cluster = ecs.Cluster(self, "Cluster", vpc=vpc)

# Add capacity to it
cluster.add_capacity("DefaultAutoScalingGroupCapacity",
    instance_type=ec2.InstanceType("t2.xlarge"),
    desired_capacity=3
)

task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")

task_definition.add_container("DefaultContainer",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=512
)

# Instantiate an Amazon ECS Service
ecs_service = ecs.Ec2Service(self, "Service",
    cluster=cluster,
    task_definition=task_definition
)
```

For a set of constructs defining common ECS architectural patterns, see the `@aws-cdk/aws-ecs-patterns` package.

## Launch Types: AWS Fargate vs Amazon EC2 vs AWS ECS Anywhere

There are three sets of constructs in this library:

* Use the `Ec2TaskDefinition` and `Ec2Service` constructs to run tasks on Amazon EC2 instances running in your account.
* Use the `FargateTaskDefinition` and `FargateService` constructs to run tasks on
  instances that are managed for you by AWS.
* Use the `ExternalTaskDefinition` and `ExternalService` constructs to run AWS ECS Anywhere tasks on self-managed infrastructure.

Here are the main differences:

* **Amazon EC2**: instances are under your control. Complete control of task to host
  allocation. Required to specify at least a memory reservation or limit for
  every container. Can use Host, Bridge and AwsVpc networking modes. Can attach
  Classic Load Balancer. Can share volumes between container and host.
* **AWS Fargate**: tasks run on AWS-managed instances, AWS manages task to host
  allocation for you. Requires specification of memory and cpu sizes at the
  taskdefinition level. Only supports AwsVpc networking modes and
  Application/Network Load Balancers. Only the AWS log driver is supported.
  Many host features are not supported such as adding kernel capabilities
  and mounting host devices/volumes inside the container.
* **AWS ECS Anywhere**: tasks are run and managed by AWS ECS Anywhere on infrastructure
  owned by the customer. Bridge, Host and None networking modes are supported. Does not
  support autoscaling, load balancing, cloudmap or attachment of volumes.

For more information on Amazon EC2 vs AWS Fargate, networking and ECS Anywhere see the AWS Documentation:
[AWS Fargate](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html),
[Task Networking](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html),
[ECS Anywhere](https://aws.amazon.com/ecs/anywhere/)

## Clusters

A `Cluster` defines the infrastructure to run your
tasks on. You can run many tasks on a single cluster.

The following code creates a cluster that can run AWS Fargate tasks:

```python
# vpc: ec2.Vpc


cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)
```

The following code imports an existing cluster using the ARN which can be used to
import an Amazon ECS service either EC2 or Fargate.

```python
cluster_arn = "arn:aws:ecs:us-east-1:012345678910:cluster/clusterName"

cluster = ecs.Cluster.from_cluster_arn(self, "Cluster", cluster_arn)
```

To use tasks with Amazon EC2 launch-type, you have to add capacity to
the cluster in order for tasks to be scheduled on your instances.  Typically,
you add an AutoScalingGroup with instances running the latest
Amazon ECS-optimized AMI to the cluster. There is a method to build and add such an
AutoScalingGroup automatically, or you can supply a customized AutoScalingGroup
that you construct yourself. It's possible to add multiple AutoScalingGroups
with various instance types.

The following example creates an Amazon ECS cluster and adds capacity to it:

```python
# vpc: ec2.Vpc


cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)

# Either add default capacity
cluster.add_capacity("DefaultAutoScalingGroupCapacity",
    instance_type=ec2.InstanceType("t2.xlarge"),
    desired_capacity=3
)

# Or add customized capacity. Be sure to start the Amazon ECS-optimized AMI.
auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=ec2.InstanceType("t2.xlarge"),
    machine_image=ecs.EcsOptimizedImage.amazon_linux(),
    # Or use Amazon ECS-Optimized Amazon Linux 2 AMI
    # machineImage: EcsOptimizedImage.amazonLinux2(),
    desired_capacity=3
)

capacity_provider = ecs.AsgCapacityProvider(self, "AsgCapacityProvider",
    auto_scaling_group=auto_scaling_group
)
cluster.add_asg_capacity_provider(capacity_provider)
```

If you omit the property `vpc`, the construct will create a new VPC with two AZs.

By default, all machine images will auto-update to the latest version
on each deployment, causing a replacement of the instances in your AutoScalingGroup
if the AMI has been updated since the last deployment.

If task draining is enabled, ECS will transparently reschedule tasks on to the new
instances before terminating your old instances. If you have disabled task draining,
the tasks will be terminated along with the instance. To prevent that, you
can pick a non-updating AMI by passing `cacheInContext: true`, but be sure
to periodically update to the latest AMI manually by using the [CDK CLI
context management commands](https://docs.aws.amazon.com/cdk/latest/guide/context.html):

```python
# vpc: ec2.Vpc

auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    machine_image=ecs.EcsOptimizedImage.amazon_linux(cached_in_context=True),
    vpc=vpc,
    instance_type=ec2.InstanceType("t2.micro")
)
```

To use `LaunchTemplate` with `AsgCapacityProvider`, make sure to specify the `userData` in the `LaunchTemplate`:

```python
# vpc: ec2.Vpc

launch_template = ec2.LaunchTemplate(self, "ASG-LaunchTemplate",
    instance_type=ec2.InstanceType("t3.medium"),
    machine_image=ecs.EcsOptimizedImage.amazon_linux2(),
    user_data=ec2.UserData.for_linux()
)

auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    mixed_instances_policy=autoscaling.MixedInstancesPolicy(
        instances_distribution=autoscaling.InstancesDistribution(
            on_demand_percentage_above_base_capacity=50
        ),
        launch_template=launch_template
    )
)

cluster = ecs.Cluster(self, "Cluster", vpc=vpc)

capacity_provider = ecs.AsgCapacityProvider(self, "AsgCapacityProvider",
    auto_scaling_group=auto_scaling_group,
    machine_image_type=ecs.MachineImageType.AMAZON_LINUX_2
)

cluster.add_asg_capacity_provider(capacity_provider)
```

### Bottlerocket

[Bottlerocket](https://aws.amazon.com/bottlerocket/) is a Linux-based open source operating system that is
purpose-built by AWS for running containers. You can launch Amazon ECS container instances with the Bottlerocket AMI.

The following example will create a capacity with self-managed Amazon EC2 capacity of 2 `c5.large` Linux instances running with `Bottlerocket` AMI.

The following example adds Bottlerocket capacity to the cluster:

```python
# cluster: ecs.Cluster


cluster.add_capacity("bottlerocket-asg",
    min_capacity=2,
    instance_type=ec2.InstanceType("c5.large"),
    machine_image=ecs.BottleRocketImage()
)
```

### ARM64 (Graviton) Instances

To launch instances with ARM64 hardware, you can use the Amazon ECS-optimized
Amazon Linux 2 (arm64) AMI. Based on Amazon Linux 2, this AMI is recommended
for use when launching your EC2 instances that are powered by Arm-based AWS
Graviton Processors.

```python
# cluster: ecs.Cluster


cluster.add_capacity("graviton-cluster",
    min_capacity=2,
    instance_type=ec2.InstanceType("c6g.large"),
    machine_image=ecs.EcsOptimizedImage.amazon_linux2(ecs.AmiHardwareType.ARM)
)
```

Bottlerocket is also supported:

```python
# cluster: ecs.Cluster


cluster.add_capacity("graviton-cluster",
    min_capacity=2,
    instance_type=ec2.InstanceType("c6g.large"),
    machine_image_type=ecs.MachineImageType.BOTTLEROCKET
)
```

### Spot Instances

To add spot instances into the cluster, you must specify the `spotPrice` in the `ecs.AddCapacityOptions` and optionally enable the `spotInstanceDraining` property.

```python
# cluster: ecs.Cluster


# Add an AutoScalingGroup with spot instances to the existing cluster
cluster.add_capacity("AsgSpot",
    max_capacity=2,
    min_capacity=2,
    desired_capacity=2,
    instance_type=ec2.InstanceType("c5.xlarge"),
    spot_price="0.0735",
    # Enable the Automated Spot Draining support for Amazon ECS
    spot_instance_draining=True
)
```

### SNS Topic Encryption

When the `ecs.AddCapacityOptions` that you provide has a non-zero `taskDrainTime` (the default) then an SNS topic and Lambda are created to ensure that the
cluster's instances have been properly drained of tasks before terminating. The SNS Topic is sent the instance-terminating lifecycle event from the AutoScalingGroup,
and the Lambda acts on that event. If you wish to engage [server-side encryption](https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html) for this SNS Topic
then you may do so by providing a KMS key for the `topicEncryptionKey` property of `ecs.AddCapacityOptions`.

```python
# Given
# cluster: ecs.Cluster
# key: kms.Key

# Then, use that key to encrypt the lifecycle-event SNS Topic.
cluster.add_capacity("ASGEncryptedSNS",
    instance_type=ec2.InstanceType("t2.xlarge"),
    desired_capacity=3,
    topic_encryption_key=key
)
```

## Task definitions

A task definition describes what a single copy of a **task** should look like.
A task definition has one or more containers; typically, it has one
main container (the *default container* is the first one that's added
to the task definition, and it is marked *essential*) and optionally
some supporting containers which are used to support the main container,
doings things like upload logs or metrics to monitoring services.

To run a task or service with Amazon EC2 launch type, use the `Ec2TaskDefinition`. For AWS Fargate tasks/services, use the
`FargateTaskDefinition`. For AWS ECS Anywhere use the `ExternalTaskDefinition`. These classes
provide simplified APIs that only contain properties relevant for each specific launch type.

For a `FargateTaskDefinition`, specify the task size (`memoryLimitMiB` and `cpu`):

```python
fargate_task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    memory_limit_mi_b=512,
    cpu=256
)
```

On Fargate Platform Version 1.4.0 or later, you may specify up to 200GiB of
[ephemeral storage](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-task-storage.html#fargate-task-storage-pv14):

```python
fargate_task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    memory_limit_mi_b=512,
    cpu=256,
    ephemeral_storage_gi_b=100
)
```

To add containers to a task definition, call `addContainer()`:

```python
fargate_task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    memory_limit_mi_b=512,
    cpu=256
)
container = fargate_task_definition.add_container("WebContainer",
    # Use an image from DockerHub
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample")
)
```

For an `Ec2TaskDefinition`:

```python
ec2_task_definition = ecs.Ec2TaskDefinition(self, "TaskDef",
    network_mode=ecs.NetworkMode.BRIDGE
)

container = ec2_task_definition.add_container("WebContainer",
    # Use an image from DockerHub
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024
)
```

For an `ExternalTaskDefinition`:

```python
external_task_definition = ecs.ExternalTaskDefinition(self, "TaskDef")

container = external_task_definition.add_container("WebContainer",
    # Use an image from DockerHub
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024
)
```

You can specify container properties when you add them to the task definition, or with various methods, e.g.:

To add a port mapping when adding a container to the task definition, specify the `portMappings` option:

```python
# task_definition: ecs.TaskDefinition


task_definition.add_container("WebContainer",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024,
    port_mappings=[ecs.PortMapping(container_port=3000)]
)
```

To add port mappings directly to a container definition, call `addPortMappings()`:

```python
# container: ecs.ContainerDefinition


container.add_port_mappings(
    container_port=3000
)
```

To add data volumes to a task definition, call `addVolume()`:

```python
fargate_task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    memory_limit_mi_b=512,
    cpu=256
)
volume = {
    # Use an Elastic FileSystem
    "name": "mydatavolume",
    "efs_volume_configuration": {
        "file_system_id": "EFS"
    }
}

container = fargate_task_definition.add_volume(volume)
```

> Note: ECS Anywhere doesn't support volume attachments in the task definition.

To use a TaskDefinition that can be used with either Amazon EC2 or
AWS Fargate launch types, use the `TaskDefinition` construct.

When creating a task definition you have to specify what kind of
tasks you intend to run: Amazon EC2, AWS Fargate, or both.
The following example uses both:

```python
task_definition = ecs.TaskDefinition(self, "TaskDef",
    memory_mi_b="512",
    cpu="256",
    network_mode=ecs.NetworkMode.AWS_VPC,
    compatibility=ecs.Compatibility.EC2_AND_FARGATE
)
```

To grant a principal permission to run your `TaskDefinition`, you can use the `TaskDefinition.grantRun()` method:

```python
# role: iam.IGrantable

task_def = ecs.TaskDefinition(self, "TaskDef",
    cpu="512",
    memory_mi_b="512",
    compatibility=ecs.Compatibility.EC2_AND_FARGATE
)

# Gives role required permissions to run taskDef
task_def.grant_run(role)
```

### Images

Images supply the software that runs inside the container. Images can be
obtained from either DockerHub or from ECR repositories, built directly from a local Dockerfile, or use an existing tarball.

* `ecs.ContainerImage.fromRegistry(imageName)`: use a public image.
* `ecs.ContainerImage.fromRegistry(imageName, { credentials: mySecret })`: use a private image that requires credentials.
* `ecs.ContainerImage.fromEcrRepository(repo, tagOrDigest)`: use the given ECR repository as the image
  to start. If no tag or digest is provided, "latest" is assumed.
* `ecs.ContainerImage.fromAsset('./image')`: build and upload an
  image directly from a `Dockerfile` in your source directory.
* `ecs.ContainerImage.fromDockerImageAsset(asset)`: uses an existing
  `@aws-cdk/aws-ecr-assets.DockerImageAsset` as a container image.
* `ecs.ContainerImage.fromTarball(file)`: use an existing tarball.
* `new ecs.TagParameterContainerImage(repository)`: use the given ECR repository as the image
  but a CloudFormation parameter as the tag.

### Environment variables

To pass environment variables to the container, you can use the `environment`, `environmentFiles`, and `secrets` props.

```python
# secret: secretsmanager.Secret
# db_secret: secretsmanager.Secret
# parameter: ssm.StringParameter
# task_definition: ecs.TaskDefinition
# s3_bucket: s3.Bucket


new_container = task_definition.add_container("container",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024,
    environment={ # clear text, not for sensitive data
        "STAGE": "prod"},
    environment_files=[ # list of environment files hosted either on local disk or S3
        ecs.EnvironmentFile.from_asset("./demo-env-file.env"),
        ecs.EnvironmentFile.from_bucket(s3_bucket, "assets/demo-env-file.env")],
    secrets={ # Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store at container start-up.
        "SECRET": ecs.Secret.from_secrets_manager(secret),
        "DB_PASSWORD": ecs.Secret.from_secrets_manager(db_secret, "password"),  # Reference a specific JSON field, (requires platform version 1.4.0 or later for Fargate tasks)
        "API_KEY": ecs.Secret.from_secrets_manager_version(secret, ecs.SecretVersionInfo(version_id="12345"), "apiKey"),  # Reference a specific version of the secret by its version id or version stage (requires platform version 1.4.0 or later for Fargate tasks)
        "PARAMETER": ecs.Secret.from_ssm_parameter(parameter)}
)
new_container.add_environment("QUEUE_NAME", "MyQueue")
new_container.add_secret("API_KEY", ecs.Secret.from_secrets_manager(secret))
new_container.add_secret("DB_PASSWORD", ecs.Secret.from_secrets_manager(secret, "password"))
```

The task execution role is automatically granted read permissions on the secrets/parameters. Support for environment
files is restricted to the EC2 launch type for files hosted on S3. Further details provided in the AWS documentation
about [specifying environment variables](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html).

### Linux parameters

To apply additional linux-specific options related to init process and memory management to the container, use the `linuxParameters` property:

```python
# task_definition: ecs.TaskDefinition


task_definition.add_container("container",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024,
    linux_parameters=ecs.LinuxParameters(self, "LinuxParameters",
        init_process_enabled=True,
        shared_memory_size=1024,
        max_swap=Size.mebibytes(5000),
        swappiness=90
    )
)
```

### System controls

To set system controls (kernel parameters) on the container, use the `systemControls` prop:

```python
# task_definition: ecs.TaskDefinition


task_definition.add_container("container",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024,
    system_controls=[ecs.SystemControl(
        namespace="net.ipv6.conf.all.default.disable_ipv6",
        value="1"
    )
    ]
)
```

### Using Windows containers on Fargate

AWS Fargate supports Amazon ECS Windows containers. For more details, please see this [blog post](https://aws.amazon.com/tw/blogs/containers/running-windows-containers-with-amazon-ecs-on-aws-fargate/)

```python
# Create a Task Definition for the Windows container to start
task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    runtime_platform=ecs.RuntimePlatform(
        operating_system_family=ecs.OperatingSystemFamily.WINDOWS_SERVER_2019_CORE,
        cpu_architecture=ecs.CpuArchitecture.X86_64
    ),
    cpu=1024,
    memory_limit_mi_b=2048
)

task_definition.add_container("windowsservercore",
    logging=ecs.LogDriver.aws_logs(stream_prefix="win-iis-on-fargate"),
    port_mappings=[ecs.PortMapping(container_port=80)],
    image=ecs.ContainerImage.from_registry("mcr.microsoft.com/windows/servercore/iis:windowsservercore-ltsc2019")
)
```

### Using Graviton2 with Fargate

AWS Graviton2 supports AWS Fargate. For more details, please see this [blog post](https://aws.amazon.com/blogs/aws/announcing-aws-graviton2-support-for-aws-fargate-get-up-to-40-better-price-performance-for-your-serverless-containers/)

```python
# Create a Task Definition for running container on Graviton Runtime.
task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    runtime_platform=ecs.RuntimePlatform(
        operating_system_family=ecs.OperatingSystemFamily.LINUX,
        cpu_architecture=ecs.CpuArchitecture.ARM64
    ),
    cpu=1024,
    memory_limit_mi_b=2048
)

task_definition.add_container("webarm64",
    logging=ecs.LogDriver.aws_logs(stream_prefix="graviton2-on-fargate"),
    port_mappings=[ecs.PortMapping(container_port=80)],
    image=ecs.ContainerImage.from_registry("public.ecr.aws/nginx/nginx:latest-arm64v8")
)
```

## Service

A `Service` instantiates a `TaskDefinition` on a `Cluster` a given number of
times, optionally associating them with a load balancer.
If a task fails,
Amazon ECS automatically restarts the task.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition


service = ecs.FargateService(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    desired_count=5
)
```

ECS Anywhere service definition looks like:

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition


service = ecs.ExternalService(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    desired_count=5
)
```

`Services` by default will create a security group if not provided.
If you'd like to specify which security groups to use you can override the `securityGroups` property.

### Deployment circuit breaker and rollback

Amazon ECS [deployment circuit breaker](https://aws.amazon.com/tw/blogs/containers/announcing-amazon-ecs-deployment-circuit-breaker/)
automatically rolls back unhealthy service deployments without the need for manual intervention. Use `circuitBreaker` to enable
deployment circuit breaker and optionally enable `rollback` for automatic rollback. See [Using the deployment circuit breaker](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html)
for more details.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition

service = ecs.FargateService(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    circuit_breaker=ecs.DeploymentCircuitBreaker(rollback=True)
)
```

> Note: ECS Anywhere doesn't support deployment circuit breakers and rollback.

### Deployment alarms

Amazon ECS [deployment alarms]
(https://aws.amazon.com/blogs/containers/automate-rollbacks-for-amazon-ecs-rolling-deployments-with-cloudwatch-alarms/)
allow monitoring and automatically reacting to changes during a rolling update
by using Amazon CloudWatch metric alarms.

Amazon ECS starts monitoring the configured deployment alarms as soon as one or
more tasks of the updated service are in a running state. The deployment process
continues until the primary deployment is healthy and has reached the desired
count and the active deployment has been scaled down to 0. Then, the deployment
remains in the IN_PROGRESS state for an additional "bake time." The length the
bake time is calculated based on the evaluation periods and period of the alarms.
After the bake time, if none of the alarms have been activated, then Amazon ECS
considers this to be a successful update and deletes the active deployment and
changes the status of the primary deployment to COMPLETED.

```python
# Example automatically generated from non-compiling source. May contain errors.
import aws_cdk.aws_cloudwatch as cw
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition
# elb_alarm: cloudwatch.Alarm

service = ecs.FargateService(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    deployment_alarms=ecs.DeploymentAlarmConfig(
        alarms=[elb_alarm.alarm_name],
        behavior=AlarmBehavior.ROLLBACK_ON_ALARM
    )
)

# Defining a deployment alarm after the service has been created
cpu_alarm_name = "MyCpuMetricAlarm"
cw.Alarm(self, "CPUAlarm",
    alarm_name=cpu_alarm_name,
    metric=service.metric_cpu_utilization(),
    evaluation_periods=2,
    threshold=80
)
service.enable_deployment_alarms([cpu_alarm_name], AlarmBehavior.FAIL_ON_ALARM)
```

> Note: Deployment alarms are only available when `deploymentController` is set
> to `DeploymentControllerType.ECS`, which is the default.

#### Troubleshooting circular dependencies

I saw this info message during synth time. What do I do?

```text
Deployment alarm ({"Ref":"MyAlarmABC1234"}) enabled on MyEcsService may cause a
circular dependency error when this stack deploys. The alarm name references the
alarm's logical id, or another resource. See the 'Deployment alarms' section in
the module README for more details.
```

If your app deploys successfully with this message, you can disregard it. But it
indicates that you could encounter a circular dependency error when you try to
deploy. If you want to alarm on metrics produced by the service, there will be a
circular dependency between the service and its deployment alarms. In this case,
there are two options to avoid the circular dependency.

1. Define the physical name for the alarm. Use a defined physical name that is
   unique within the deployment environment for the alarm name when creating the
   alarm, and re-use the defined name. This name could be a hardcoded string, a
   string generated based on the environment, or could reference another
   resource that does not depend on the service.
2. Define the physical name for the service. Then, don't use
   `metricCpuUtilization()` or similar methods. Create the metric object
   separately by referencing the service metrics using this name.

Option 1, defining a physical name for the alarm:

```python
# Example automatically generated from non-compiling source. May contain errors.
import aws_cdk.aws_cloudwatch as cw
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition

service = ecs.FargateService(self, "Service",
    cluster=cluster,
    task_definition=task_definition
)

cpu_alarm_name = "MyCpuMetricAlarm"
my_alarm = cw.Alarm(self, "CPUAlarm",
    alarm_name=cpu_alarm_name,
    metric=service.metric_cpu_utilization(),
    evaluation_periods=2,
    threshold=80
)

# Using `myAlarm.alarmName` here will cause a circular dependency
service.enable_deployment_alarms([cpu_alarm_name], AlarmBehavior.FAIL_ON_ALARM)
```

Option 2, defining a physical name for the service:

```python
# Example automatically generated from non-compiling source. May contain errors.
import aws_cdk as cdk
import aws_cdk.aws_cloudwatch as cw
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition

service_name = "MyFargateService"
service = ecs.FargateService(self, "Service",
    service_name=service_name,
    cluster=cluster,
    task_definition=task_definition
)

cpu_metric = cw.Metric(metric_name, "CPUUtilization", namespace, "AWS/ECS", period, cdk.Duration.minutes(5), statistic, "Average", dimensions_map, {
    "ClusterName": cluster.cluster_name,
    # Using `service.serviceName` here will cause a circular dependency
    "ServiceName": service_name
})
my_alarm = cw.Alarm(self, "CPUAlarm",
    alarm_name=cpu_alarm_name,
    metric=cpu_metric,
    evaluation_periods=2,
    threshold=80
)

service.enable_deployment_alarms([my_alarm.alarm_name], AlarmBehavior.FAIL_ON_ALARM)
```

This issue only applies if the metrics to alarm on are emitted by the service
itself. If the metrics are emitted by a different resource, that does not depend
on the service, there will be no restrictions on the alarm name.

### Include an application/network load balancer

`Services` are load balancing targets and can be added to a target group, which will be attached to an application/network load balancers:

```python
# vpc: ec2.Vpc
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition

service = ecs.FargateService(self, "Service", cluster=cluster, task_definition=task_definition)

lb = elbv2.ApplicationLoadBalancer(self, "LB", vpc=vpc, internet_facing=True)
listener = lb.add_listener("Listener", port=80)
target_group1 = listener.add_targets("ECS1",
    port=80,
    targets=[service]
)
target_group2 = listener.add_targets("ECS2",
    port=80,
    targets=[service.load_balancer_target(
        container_name="MyContainer",
        container_port=8080
    )]
)
```

> Note: ECS Anywhere doesn't support application/network load balancers.

Note that in the example above, the default `service` only allows you to register the first essential container or the first mapped port on the container as a target and add it to a new target group. To have more control over which container and port to register as targets, you can use `service.loadBalancerTarget()` to return a load balancing target for a specific container and port.

Alternatively, you can also create all load balancer targets to be registered in this service, add them to target groups, and attach target groups to listeners accordingly.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition
# vpc: ec2.Vpc

service = ecs.FargateService(self, "Service", cluster=cluster, task_definition=task_definition)

lb = elbv2.ApplicationLoadBalancer(self, "LB", vpc=vpc, internet_facing=True)
listener = lb.add_listener("Listener", port=80)
service.register_load_balancer_targets(
    container_name="web",
    container_port=80,
    new_target_group_id="ECS",
    listener=ecs.ListenerConfig.application_listener(listener,
        protocol=elbv2.ApplicationProtocol.HTTPS
    )
)
```

### Using a Load Balancer from a different Stack

If you want to put your Load Balancer and the Service it is load balancing to in
different stacks, you may not be able to use the convenience methods
`loadBalancer.addListener()` and `listener.addTargets()`.

The reason is that these methods will create resources in the same Stack as the
object they're called on, which may lead to cyclic references between stacks.
Instead, you will have to create an `ApplicationListener` in the service stack,
or an empty `TargetGroup` in the load balancer stack that you attach your
service to.

See the [ecs/cross-stack-load-balancer example](https://github.com/aws-samples/aws-cdk-examples/tree/master/typescript/ecs/cross-stack-load-balancer/)
for the alternatives.

### Include a classic load balancer

`Services` can also be directly attached to a classic load balancer as targets:

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition
# vpc: ec2.Vpc

service = ecs.Ec2Service(self, "Service", cluster=cluster, task_definition=task_definition)

lb = elb.LoadBalancer(self, "LB", vpc=vpc)
lb.add_listener(external_port=80)
lb.add_target(service)
```

Similarly, if you want to have more control over load balancer targeting:

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition
# vpc: ec2.Vpc

service = ecs.Ec2Service(self, "Service", cluster=cluster, task_definition=task_definition)

lb = elb.LoadBalancer(self, "LB", vpc=vpc)
lb.add_listener(external_port=80)
lb.add_target(service.load_balancer_target(
    container_name="MyContainer",
    container_port=80
))
```

There are two higher-level constructs available which include a load balancer for you that can be found in the aws-ecs-patterns module:

* `LoadBalancedFargateService`
* `LoadBalancedEc2Service`

### Import existing services

`Ec2Service` and `FargateService` provide methods to import existing EC2/Fargate services.
The ARN of the existing service has to be specified to import the service.

Since AWS has changed the [ARN format for ECS](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-account-settings.html#ecs-resource-ids),
feature flag `@aws-cdk/aws-ecs:arnFormatIncludesClusterName` must be enabled to use the new ARN format.
The feature flag changes behavior for the entire CDK project. Therefore it is not possible to mix the old and the new format in one CDK project.

```tss
declare const cluster: ecs.Cluster;

// Import service from EC2 service attributes
const service = ecs.Ec2Service.fromEc2ServiceAttributes(this, 'EcsService', {
  serviceArn: 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service',
  cluster,
});

// Import service from EC2 service ARN
const service = ecs.Ec2Service.fromEc2ServiceArn(this, 'EcsService', 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service');

// Import service from Fargate service attributes
const service = ecs.FargateService.fromFargateServiceAttributes(this, 'EcsService', {
  serviceArn: 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service',
  cluster,
});

// Import service from Fargate service ARN
const service = ecs.FargateService.fromFargateServiceArn(this, 'EcsService', 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service');
```

## Task Auto-Scaling

You can configure the task count of a service to match demand. Task auto-scaling is
configured by calling `autoScaleTaskCount()`:

```python
# target: elbv2.ApplicationTargetGroup
# service: ecs.BaseService

scaling = service.auto_scale_task_count(max_capacity=10)
scaling.scale_on_cpu_utilization("CpuScaling",
    target_utilization_percent=50
)

scaling.scale_on_request_count("RequestScaling",
    requests_per_target=10000,
    target_group=target
)
```

Task auto-scaling is powered by *Application Auto-Scaling*.
See that section for details.

## Integration with CloudWatch Events

To start an Amazon ECS task on an Amazon EC2-backed Cluster, instantiate an
`@aws-cdk/aws-events-targets.EcsTask` instead of an `Ec2Service`:

```python
# cluster: ecs.Cluster

# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_asset(path.resolve(__dirname, "..", "eventhandler-image")),
    memory_limit_mi_b=256,
    logging=ecs.AwsLogDriver(stream_prefix="EventDemo", mode=ecs.AwsLogDriverMode.NON_BLOCKING)
)

# An Rule that describes the event trigger (in this case a scheduled run)
rule = events.Rule(self, "Rule",
    schedule=events.Schedule.expression("rate(1 min)")
)

# Pass an environment variable to the container 'TheContainer' in the task
rule.add_target(targets.EcsTask(
    cluster=cluster,
    task_definition=task_definition,
    task_count=1,
    container_overrides=[targets.ContainerOverride(
        container_name="TheContainer",
        environment=[targets.TaskEnvironmentVariable(
            name="I_WAS_TRIGGERED",
            value="From CloudWatch Events"
        )]
    )]
))
```

## Log Drivers

Currently Supported Log Drivers:

* awslogs
* fluentd
* gelf
* journald
* json-file
* splunk
* syslog
* awsfirelens
* Generic

### awslogs Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.aws_logs(stream_prefix="EventDemo")
)
```

### fluentd Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.fluentd()
)
```

### gelf Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.gelf(address="my-gelf-address")
)
```

### journald Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.journald()
)
```

### json-file Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.json_file()
)
```

### splunk Log Driver

```python
# secret: ecs.Secret


# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.splunk(
        secret_token=secret,
        url="my-splunk-url"
    )
)
```

### syslog Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.syslog()
)
```

### firelens Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.firelens(
        options={
            "Name": "firehose",
            "region": "us-west-2",
            "delivery_stream": "my-stream"
        }
    )
)
```

To pass secrets to the log configuration, use the `secretOptions` property of the log configuration. The task execution role is automatically granted read permissions on the secrets/parameters.

```python
# secret: secretsmanager.Secret
# parameter: ssm.StringParameter


task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.firelens(
        options={},
        secret_options={ # Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store
            "apikey": ecs.Secret.from_secrets_manager(secret),
            "host": ecs.Secret.from_ssm_parameter(parameter)}
    )
)
```

### Generic Log Driver

A generic log driver object exists to provide a lower level abstraction of the log driver configuration.

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.GenericLogDriver(
        log_driver="fluentd",
        options={
            "tag": "example-tag"
        }
    )
)
```

## CloudMap Service Discovery

To register your ECS service with a CloudMap Service Registry, you may add the
`cloudMapOptions` property to your service:

```python
# task_definition: ecs.TaskDefinition
# cluster: ecs.Cluster


service = ecs.Ec2Service(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    cloud_map_options=ecs.CloudMapOptions(
        # Create A records - useful for AWSVPC network mode.
        dns_record_type=cloudmap.DnsRecordType.A
    )
)
```

With `bridge` or `host` network modes, only `SRV` DNS record types are supported.
By default, `SRV` DNS record types will target the default container and default
port. However, you may target a different container and port on the same ECS task:

```python
# task_definition: ecs.TaskDefinition
# cluster: ecs.Cluster


# Add a container to the task definition
specific_container = task_definition.add_container("Container",
    image=ecs.ContainerImage.from_registry("/aws/aws-example-app"),
    memory_limit_mi_b=2048
)

# Add a port mapping
specific_container.add_port_mappings(
    container_port=7600,
    protocol=ecs.Protocol.TCP
)

ecs.Ec2Service(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    cloud_map_options=ecs.CloudMapOptions(
        # Create SRV records - useful for bridge networking
        dns_record_type=cloudmap.DnsRecordType.SRV,
        # Targets port TCP port 7600 `specificContainer`
        container=specific_container,
        container_port=7600
    )
)
```

### Associate With a Specific CloudMap Service

You may associate an ECS service with a specific CloudMap service. To do
this, use the service's `associateCloudMapService` method:

```python
# cloud_map_service: cloudmap.Service
# ecs_service: ecs.FargateService


ecs_service.associate_cloud_map_service(
    service=cloud_map_service
)
```

## Capacity Providers

There are two major families of Capacity Providers: [AWS
Fargate](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-capacity-providers.html)
(including Fargate Spot) and EC2 [Auto Scaling
Group](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html)
Capacity Providers. Both are supported.

### Fargate Capacity Providers

To enable Fargate capacity providers, you can either set
`enableFargateCapacityProviders` to `true` when creating your cluster, or by
invoking the `enableFargateCapacityProviders()` method after creating your
cluster. This will add both `FARGATE` and `FARGATE_SPOT` as available capacity
providers on your cluster.

```python
# vpc: ec2.Vpc


cluster = ecs.Cluster(self, "FargateCPCluster",
    vpc=vpc,
    enable_fargate_capacity_providers=True
)

task_definition = ecs.FargateTaskDefinition(self, "TaskDef")

task_definition.add_container("web",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample")
)

ecs.FargateService(self, "FargateService",
    cluster=cluster,
    task_definition=task_definition,
    capacity_provider_strategies=[ecs.CapacityProviderStrategy(
        capacity_provider="FARGATE_SPOT",
        weight=2
    ), ecs.CapacityProviderStrategy(
        capacity_provider="FARGATE",
        weight=1
    )
    ]
)
```

### Auto Scaling Group Capacity Providers

To add an Auto Scaling Group Capacity Provider, first create an EC2 Auto Scaling
Group. Then, create an `AsgCapacityProvider` and pass the Auto Scaling Group to
it in the constructor. Then add the Capacity Provider to the cluster. Finally,
you can refer to the Provider by its name in your service's or task's Capacity
Provider strategy.

By default, Auto Scaling Group Capacity Providers will manage the scale-in and
scale-out behavior of the auto scaling group based on the load your tasks put on
the cluster, this is called [Managed Scaling](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html#asg-capacity-providers-managed-scaling). If you'd
rather manage scaling behavior yourself set `enableManagedScaling` to `false`.

Additionally [Managed Termination Protection](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-auto-scaling.html#managed-termination-protection) is enabled by default to
prevent scale-in behavior from terminating instances that have non-daemon tasks
running on them. This is ideal for tasks that should be ran to completion. If your
tasks are safe to interrupt then this protection can be disabled by setting
`enableManagedTerminationProtection` to `false`. Managed Scaling must be enabled for
Managed Termination Protection to work.

> Currently there is a known [CloudFormation issue](https://github.com/aws/containers-roadmap/issues/631)
> that prevents CloudFormation from automatically deleting Auto Scaling Groups that
> have Managed Termination Protection enabled. To work around this issue you could set
> `enableManagedTerminationProtection` to `false` on the Auto Scaling Group Capacity
> Provider. If you'd rather not disable Managed Termination Protection, you can [manually
> delete the Auto Scaling Group](https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-process-shutdown.html).
> For other workarounds, see [this GitHub issue](https://github.com/aws/aws-cdk/issues/18179).

```python
# vpc: ec2.Vpc


cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)

auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=ec2.InstanceType("t2.micro"),
    machine_image=ecs.EcsOptimizedImage.amazon_linux2(),
    min_capacity=0,
    max_capacity=100
)

capacity_provider = ecs.AsgCapacityProvider(self, "AsgCapacityProvider",
    auto_scaling_group=auto_scaling_group
)
cluster.add_asg_capacity_provider(capacity_provider)

task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")

task_definition.add_container("web",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_reservation_mi_b=256
)

ecs.Ec2Service(self, "EC2Service",
    cluster=cluster,
    task_definition=task_definition,
    capacity_provider_strategies=[ecs.CapacityProviderStrategy(
        capacity_provider=capacity_provider.capacity_provider_name,
        weight=1
    )
    ]
)
```

### Cluster Default Provider Strategy

A capacity provider strategy determines whether ECS tasks are launched on EC2 instances or Fargate/Fargate Spot. It can be specified at the cluster, service, or task level, and consists of one or more capacity providers. You can specify an optional base and weight value for finer control of how tasks are launched. The `base` specifies a minimum number of tasks on one capacity provider, and the `weight`s of each capacity provider determine how tasks are distributed after `base` is satisfied.

You can associate a default capacity provider strategy with an Amazon ECS cluster. After you do this, a default capacity provider strategy is used when creating a service or running a standalone task in the cluster and whenever a custom capacity provider strategy or a launch type isn't specified. We recommend that you define a default capacity provider strategy for each cluster.

For more information visit https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-capacity-providers.html

When the service does not have a capacity provider strategy, the cluster's default capacity provider strategy will be used. Default Capacity Provider Strategy can be added by using the method `addDefaultCapacityProviderStrategy`. A capacity provider strategy cannot contain a mix of EC2 Autoscaling Group capacity providers and Fargate providers.

```python
# capacity_provider: ecs.AsgCapacityProvider


cluster = ecs.Cluster(self, "EcsCluster",
    enable_fargate_capacity_providers=True
)
cluster.add_asg_capacity_provider(capacity_provider)

cluster.add_default_capacity_provider_strategy([capacity_provider="FARGATE", base=10, weight=50, capacity_provider="FARGATE_SPOT", weight=50
])
```

```python
# capacity_provider: ecs.AsgCapacityProvider


cluster = ecs.Cluster(self, "EcsCluster",
    enable_fargate_capacity_providers=True
)
cluster.add_asg_capacity_provider(capacity_provider)

cluster.add_default_capacity_provider_strategy([capacity_provider=capacity_provider.capacity_provider_name
])
```

## Elastic Inference Accelerators

Currently, this feature is only supported for services with EC2 launch types.

To add elastic inference accelerators to your EC2 instance, first add
`inferenceAccelerators` field to the Ec2TaskDefinition and set the `deviceName`
and `deviceType` properties.

```python
inference_accelerators = [{
    "device_name": "device1",
    "device_type": "eia2.medium"
}]

task_definition = ecs.Ec2TaskDefinition(self, "Ec2TaskDef",
    inference_accelerators=inference_accelerators
)
```

To enable using the inference accelerators in the containers, add `inferenceAcceleratorResources`
field and set it to a list of device names used for the inference accelerators. Each value in the
list should match a `DeviceName` for an `InferenceAccelerator` specified in the task definition.

```python
# task_definition: ecs.TaskDefinition

inference_accelerator_resources = ["device1"]

task_definition.add_container("cont",
    image=ecs.ContainerImage.from_registry("test"),
    memory_limit_mi_b=1024,
    inference_accelerator_resources=inference_accelerator_resources
)
```

## ECS Exec command

Please note, ECS Exec leverages AWS Systems Manager (SSM). So as a prerequisite for the exec command
to work, you need to have the SSM plugin for the AWS CLI installed locally. For more information, see
[Install Session Manager plugin for AWS CLI](https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-working-with-install-plugin.html).

To enable the ECS Exec feature for your containers, set the boolean flag `enableExecuteCommand` to `true` in
your `Ec2Service` or `FargateService`.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition


service = ecs.Ec2Service(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    enable_execute_command=True
)
```

### Enabling logging

You can enable sending logs of your execute session commands to a CloudWatch log group or S3 bucket by configuring
the `executeCommandConfiguration` property for your cluster. The default configuration will send the
logs to the CloudWatch Logs using the `awslogs` log driver that is configured in your task definition. Please note,
when using your own `logConfiguration` the log group or S3 Bucket specified must already be created.

To encrypt data using your own KMS Customer Key (CMK), you must create a CMK and provide the key in the `kmsKey` field
of the `executeCommandConfiguration`. To use this key for encrypting CloudWatch log data or S3 bucket, make sure to associate the key
to these resources on creation.

```python
# vpc: ec2.Vpc

kms_key = kms.Key(self, "KmsKey")

# Pass the KMS key in the `encryptionKey` field to associate the key to the log group
log_group = logs.LogGroup(self, "LogGroup",
    encryption_key=kms_key
)

# Pass the KMS key in the `encryptionKey` field to associate the key to the S3 bucket
exec_bucket = s3.Bucket(self, "EcsExecBucket",
    encryption_key=kms_key
)

cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc,
    execute_command_configuration=ecs.ExecuteCommandConfiguration(
        kms_key=kms_key,
        log_configuration=ecs.ExecuteCommandLogConfiguration(
            cloud_watch_log_group=log_group,
            cloud_watch_encryption_enabled=True,
            s3_bucket=exec_bucket,
            s3_encryption_enabled=True,
            s3_key_prefix="exec-command-output"
        ),
        logging=ecs.ExecuteCommandLogging.OVERRIDE
    )
)
```

## Amazon ECS Service Connect

Service Connect is a managed AWS mesh network offering. It simplifies DNS queries and inter-service communication for
ECS Services by allowing customers to set up simple DNS aliases for their services, which are accessible to all
services that have enabled Service Connect.

To enable Service Connect, you must have created a CloudMap namespace. The CDK can infer your cluster's default CloudMap namespace,
or you can specify a custom namespace. You must also have created a named port mapping on at least one container in your Task Definition.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition
# container_options: ecs.ContainerDefinitionOptions


container = task_definition.add_container("MyContainer", container_options)

container.add_port_mappings(
    name="api",
    container_port=8080
)

cluster.add_default_cloud_map_namespace(
    name="local"
)

service = ecs.FargateService(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    service_connect_configuration=ecs.ServiceConnectProps(
        services=[ecs.ServiceConnectService(
            port_mapping_name="api",
            dns_name="http-api",
            port=80
        )
        ]
    )
)
```

Service Connect-enabled services may now reach this service at `http-api:80`. Traffic to this endpoint will
be routed to the container's port 8080.

To opt a service into using service connect without advertising a port, simply call the 'enableServiceConnect' method on an initialized service.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition


service = ecs.FargateService(self, "Service",
    cluster=cluster,
    task_definition=task_definition
)
service.enable_service_connect()
```

Service Connect also allows custom logging, Service Discovery name, and configuration of the port where service connect traffic is received.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition


custom_service = ecs.FargateService(self, "CustomizedService",
    cluster=cluster,
    task_definition=task_definition,
    service_connect_configuration=ecs.ServiceConnectProps(
        log_driver=ecs.LogDrivers.aws_logs(
            stream_prefix="sc-traffic"
        ),
        services=[ecs.ServiceConnectService(
            port_mapping_name="api",
            dns_name="customized-api",
            port=80,
            ingress_port_override=20040,
            discovery_name="custom"
        )
        ]
    )
)
```

## Enable pseudo-terminal (TTY) allocation

You can allocate a pseudo-terminal (TTY) for a container passing `pseudoTerminal` option while adding the container
to the task definition.
This maps to Tty option in the ["Create a container section"](https://docs.docker.com/engine/api/v1.38/#operation/ContainerCreate)
of the [Docker Remote API](https://docs.docker.com/engine/api/v1.38/) and the --tty option to [`docker run`](https://docs.docker.com/engine/reference/commandline/run/).

```python
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    pseudo_terminal=True
)
```

## Specify a container ulimit

You can specify a container `ulimits` by specifying them in the `ulimits` option while adding the container
to the task definition.

```python
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    ulimits=[ecs.Ulimit(
        hard_limit=128,
        name=ecs.UlimitName.RSS,
        soft_limit=128
    )]
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    AssetHashType as _AssetHashType_05b67f2d,
    BundlingOptions as _BundlingOptions_588cc936,
    CfnResource as _CfnResource_9df397a6,
    CfnTag as _CfnTag_f6864754,
    Duration as _Duration_4839e8c3,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    IgnoreMode as _IgnoreMode_655a98e8,
    Resource as _Resource_45bc6135,
    Size as _Size_7b441c34,
    SymlinkFollowMode as _SymlinkFollowMode_047ec1f6,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_applicationautoscaling import (
    AdjustmentType as _AdjustmentType_f91bf569,
    BaseScalableAttribute as _BaseScalableAttribute_477f0393,
    BaseScalableAttributeProps as _BaseScalableAttributeProps_6e8d89f1,
    BaseTargetTrackingProps as _BaseTargetTrackingProps_540ba713,
    BasicStepScalingPolicyProps as _BasicStepScalingPolicyProps_be95b993,
    EnableScalingProps as _EnableScalingProps_45667ff9,
    MetricAggregationType as _MetricAggregationType_21e9eacb,
    ScalingInterval as _ScalingInterval_093a9434,
    ScalingSchedule as _ScalingSchedule_9604f271,
    Schedule as _Schedule_e93ba733,
    ServiceNamespace as _ServiceNamespace_c791119f,
)
from ..aws_autoscaling import (
    AutoScalingGroup as _AutoScalingGroup_c547a7b9,
    BlockDevice as _BlockDevice_0cfc0568,
    CommonAutoScalingGroupProps as _CommonAutoScalingGroupProps_808bbf2d,
    GroupMetrics as _GroupMetrics_7cdf729b,
    HealthCheck as _HealthCheck_03a4bd5a,
    IAutoScalingGroup as _IAutoScalingGroup_360f1cde,
    Monitoring as _Monitoring_50020f91,
    NotificationConfiguration as _NotificationConfiguration_d5911670,
    Signals as _Signals_69fbeb6e,
    TerminationPolicy as _TerminationPolicy_89633c56,
    UpdatePolicy as _UpdatePolicy_6dffc7ca,
)
from ..aws_cloudwatch import (
    IMetric as _IMetric_c7fd29de,
    Metric as _Metric_e396a4dc,
    MetricOptions as _MetricOptions_1788b62f,
    Unit as _Unit_61bc6f70,
)
from ..aws_ec2 import (
    Connections as _Connections_0f31fce8,
    IMachineImage as _IMachineImage_0e8bd50b,
    ISecurityGroup as _ISecurityGroup_acf8a799,
    IVpc as _IVpc_f30d5663,
    InstanceArchitecture as _InstanceArchitecture_7721cb36,
    InstanceType as _InstanceType_f64915b9,
    MachineImageConfig as _MachineImageConfig_187edaee,
    SubnetSelection as _SubnetSelection_e57d76df,
)
from ..aws_ecr import IRepository as _IRepository_e6004aa6
from ..aws_ecr_assets import (
    DockerCacheOption as _DockerCacheOption_58ef18ca,
    DockerImageAsset as _DockerImageAsset_c9210ac0,
    DockerImageAssetInvalidationOptions as _DockerImageAssetInvalidationOptions_4deb8d45,
    DockerImageAssetOptions as _DockerImageAssetOptions_9580cd76,
    NetworkMode as _NetworkMode_897e5081,
    Platform as _Platform_d16f3cf1,
)
from ..aws_elasticloadbalancing import (
    ILoadBalancerTarget as _ILoadBalancerTarget_2e052b5c,
    LoadBalancer as _LoadBalancer_a894d40e,
)
from ..aws_elasticloadbalancingv2 import (
    AddApplicationTargetsProps as _AddApplicationTargetsProps_76c7d190,
    AddNetworkTargetsProps as _AddNetworkTargetsProps_ce6bdf17,
    ApplicationListener as _ApplicationListener_e0620bf5,
    ApplicationProtocol as _ApplicationProtocol_aa5e9f29,
    ApplicationProtocolVersion as _ApplicationProtocolVersion_dddfe47b,
    ApplicationTargetGroup as _ApplicationTargetGroup_906fe365,
    HealthCheck as _HealthCheck_6f291880,
    IApplicationLoadBalancerTarget as _IApplicationLoadBalancerTarget_fabf9003,
    IApplicationTargetGroup as _IApplicationTargetGroup_57799827,
    INetworkLoadBalancerTarget as _INetworkLoadBalancerTarget_688b169f,
    INetworkTargetGroup as _INetworkTargetGroup_abca2df7,
    ListenerCondition as _ListenerCondition_e8416430,
    LoadBalancerTargetProps as _LoadBalancerTargetProps_4c30a73c,
    NetworkListener as _NetworkListener_539c17bf,
    Protocol as _Protocol_bc975832,
    TargetGroupLoadBalancingAlgorithmType as _TargetGroupLoadBalancingAlgorithmType_ed070d6d,
)
from ..aws_iam import (
    Grant as _Grant_a7ae64f8,
    IGrantable as _IGrantable_71c4f5de,
    IRole as _IRole_235f5d8e,
    PolicyStatement as _PolicyStatement_0fe33853,
)
from ..aws_kms import IKey as _IKey_5f11635f
from ..aws_logs import (
    ILogGroup as _ILogGroup_3c4fa718, RetentionDays as _RetentionDays_070f99f0
)
from ..aws_s3 import IBucket as _IBucket_42e086fd, Location as _Location_0948fa7f
from ..aws_s3_assets import AssetOptions as _AssetOptions_2aa69621
from ..aws_secretsmanager import ISecret as _ISecret_6e020e6a
from ..aws_servicediscovery import (
    DnsRecordType as _DnsRecordType_161ad276,
    INamespace as _INamespace_6b61e84f,
    IService as _IService_46860ae1,
    NamespaceType as _NamespaceType_5b8b96a7,
    Service as _Service_4cf2b7a6,
)
from ..aws_ssm import IParameter as _IParameter_509a0f80


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.AddAutoScalingGroupCapacityOptions",
    jsii_struct_bases=[],
    name_mapping={
        "can_containers_access_instance_role": "canContainersAccessInstanceRole",
        "machine_image_type": "machineImageType",
        "spot_instance_draining": "spotInstanceDraining",
        "topic_encryption_key": "topicEncryptionKey",
    },
)
class AddAutoScalingGroupCapacityOptions:
    def __init__(
        self,
        *,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        topic_encryption_key: typing.Optional[_IKey_5f11635f] = None,
    ) -> None:
        '''The properties for adding an AutoScalingGroup.

        :param can_containers_access_instance_role: Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: What type of machine image this is. Depending on the setting, different UserData will automatically be added to the ``AutoScalingGroup`` to configure it properly for use with ECS. If you create an ``AutoScalingGroup`` yourself and are adding it via ``addAutoScalingGroup()``, you must specify this value. If you are adding an ``autoScalingGroup`` via ``addCapacity``, this value will be determined from the ``machineImage`` you pass. Default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        :param spot_instance_draining: Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param topic_encryption_key: If ``AddAutoScalingGroupCapacityOptions.taskDrainTime`` is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            from aws_cdk import aws_kms as kms
            
            # key: kms.Key
            
            add_auto_scaling_group_capacity_options = ecs.AddAutoScalingGroupCapacityOptions(
                can_containers_access_instance_role=False,
                machine_image_type=ecs.MachineImageType.AMAZON_LINUX_2,
                spot_instance_draining=False,
                topic_encryption_key=key
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3bb1c2028b85a0ebcdd5cb2aa74890cddceee7c79afaa8e5e920a35f05c1dd6a)
            check_type(argname="argument can_containers_access_instance_role", value=can_containers_access_instance_role, expected_type=type_hints["can_containers_access_instance_role"])
            check_type(argname="argument machine_image_type", value=machine_image_type, expected_type=type_hints["machine_image_type"])
            check_type(argname="argument spot_instance_draining", value=spot_instance_draining, expected_type=type_hints["spot_instance_draining"])
            check_type(argname="argument topic_encryption_key", value=topic_encryption_key, expected_type=type_hints["topic_encryption_key"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if can_containers_access_instance_role is not None:
            self._values["can_containers_access_instance_role"] = can_containers_access_instance_role
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if spot_instance_draining is not None:
            self._values["spot_instance_draining"] = spot_instance_draining
        if topic_encryption_key is not None:
            self._values["topic_encryption_key"] = topic_encryption_key

    @builtins.property
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the containers can access the container instance role.

        :default: false
        '''
        result = self._values.get("can_containers_access_instance_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''What type of machine image this is.

        Depending on the setting, different UserData will automatically be added
        to the ``AutoScalingGroup`` to configure it properly for use with ECS.

        If you create an ``AutoScalingGroup`` yourself and are adding it via
        ``addAutoScalingGroup()``, you must specify this value. If you are adding an
        ``autoScalingGroup`` via ``addCapacity``, this value will be determined
        from the ``machineImage`` you pass.

        :default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def spot_instance_draining(self) -> typing.Optional[builtins.bool]:
        '''Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.

        For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_.

        :default: false
        '''
        result = self._values.get("spot_instance_draining")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def topic_encryption_key(self) -> typing.Optional[_IKey_5f11635f]:
        '''If ``AddAutoScalingGroupCapacityOptions.taskDrainTime`` is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information.

        :default: The SNS Topic will not be encrypted.
        '''
        result = self._values.get("topic_encryption_key")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddAutoScalingGroupCapacityOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.AddCapacityOptions",
    jsii_struct_bases=[
        AddAutoScalingGroupCapacityOptions, _CommonAutoScalingGroupProps_808bbf2d
    ],
    name_mapping={
        "can_containers_access_instance_role": "canContainersAccessInstanceRole",
        "machine_image_type": "machineImageType",
        "spot_instance_draining": "spotInstanceDraining",
        "topic_encryption_key": "topicEncryptionKey",
        "allow_all_outbound": "allowAllOutbound",
        "associate_public_ip_address": "associatePublicIpAddress",
        "auto_scaling_group_name": "autoScalingGroupName",
        "block_devices": "blockDevices",
        "capacity_rebalance": "capacityRebalance",
        "cooldown": "cooldown",
        "default_instance_warmup": "defaultInstanceWarmup",
        "desired_capacity": "desiredCapacity",
        "group_metrics": "groupMetrics",
        "health_check": "healthCheck",
        "ignore_unmodified_size_properties": "ignoreUnmodifiedSizeProperties",
        "instance_monitoring": "instanceMonitoring",
        "key_name": "keyName",
        "max_capacity": "maxCapacity",
        "max_instance_lifetime": "maxInstanceLifetime",
        "min_capacity": "minCapacity",
        "new_instances_protected_from_scale_in": "newInstancesProtectedFromScaleIn",
        "notifications": "notifications",
        "signals": "signals",
        "spot_price": "spotPrice",
        "ssm_session_permissions": "ssmSessionPermissions",
        "termination_policies": "terminationPolicies",
        "update_policy": "updatePolicy",
        "vpc_subnets": "vpcSubnets",
        "instance_type": "instanceType",
        "machine_image": "machineImage",
    },
)
class AddCapacityOptions(
    AddAutoScalingGroupCapacityOptions,
    _CommonAutoScalingGroupProps_808bbf2d,
):
    def __init__(
        self,
        *,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        topic_encryption_key: typing.Optional[_IKey_5f11635f] = None,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        associate_public_ip_address: typing.Optional[builtins.bool] = None,
        auto_scaling_group_name: typing.Optional[builtins.str] = None,
        block_devices: typing.Optional[typing.Sequence[typing.Union[_BlockDevice_0cfc0568, typing.Dict[builtins.str, typing.Any]]]] = None,
        capacity_rebalance: typing.Optional[builtins.bool] = None,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        default_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        desired_capacity: typing.Optional[jsii.Number] = None,
        group_metrics: typing.Optional[typing.Sequence[_GroupMetrics_7cdf729b]] = None,
        health_check: typing.Optional[_HealthCheck_03a4bd5a] = None,
        ignore_unmodified_size_properties: typing.Optional[builtins.bool] = None,
        instance_monitoring: typing.Optional[_Monitoring_50020f91] = None,
        key_name: typing.Optional[builtins.str] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        max_instance_lifetime: typing.Optional[_Duration_4839e8c3] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        new_instances_protected_from_scale_in: typing.Optional[builtins.bool] = None,
        notifications: typing.Optional[typing.Sequence[typing.Union[_NotificationConfiguration_d5911670, typing.Dict[builtins.str, typing.Any]]]] = None,
        signals: typing.Optional[_Signals_69fbeb6e] = None,
        spot_price: typing.Optional[builtins.str] = None,
        ssm_session_permissions: typing.Optional[builtins.bool] = None,
        termination_policies: typing.Optional[typing.Sequence[_TerminationPolicy_89633c56]] = None,
        update_policy: typing.Optional[_UpdatePolicy_6dffc7ca] = None,
        vpc_subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        instance_type: _InstanceType_f64915b9,
        machine_image: typing.Optional[_IMachineImage_0e8bd50b] = None,
    ) -> None:
        '''The properties for adding instance capacity to an AutoScalingGroup.

        :param can_containers_access_instance_role: Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: What type of machine image this is. Depending on the setting, different UserData will automatically be added to the ``AutoScalingGroup`` to configure it properly for use with ECS. If you create an ``AutoScalingGroup`` yourself and are adding it via ``addAutoScalingGroup()``, you must specify this value. If you are adding an ``autoScalingGroup`` via ``addCapacity``, this value will be determined from the ``machineImage`` you pass. Default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        :param spot_instance_draining: Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param topic_encryption_key: If ``AddAutoScalingGroupCapacityOptions.taskDrainTime`` is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.
        :param allow_all_outbound: Whether the instances can initiate connections to anywhere by default. Default: true
        :param associate_public_ip_address: Whether instances in the Auto Scaling Group should have public IP addresses associated with them. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Use subnet setting.
        :param auto_scaling_group_name: The name of the Auto Scaling group. This name must be unique per Region per account. Default: - Auto generated by CloudFormation
        :param block_devices: Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes. Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Uses the block device mapping of the AMI
        :param capacity_rebalance: Indicates whether Capacity Rebalancing is enabled. When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of interruption. After launching a new instance, it then terminates an old instance. Default: false
        :param cooldown: Default scaling cooldown for this AutoScalingGroup. Default: Duration.minutes(5)
        :param default_instance_warmup: The amount of time, in seconds, until a newly launched instance can contribute to the Amazon CloudWatch metrics. This delay lets an instance finish initializing before Amazon EC2 Auto Scaling aggregates instance metrics, resulting in more reliable usage data. Set this value equal to the amount of time that it takes for resource consumption to become stable after an instance reaches the InService state. To optimize the performance of scaling policies that scale continuously, such as target tracking and step scaling policies, we strongly recommend that you enable the default instance warmup, even if its value is set to 0 seconds Default instance warmup will not be added if no value is specified Default: None
        :param desired_capacity: Initial amount of instances in the fleet. If this is set to a number, every deployment will reset the amount of instances to this number. It is recommended to leave this value blank. Default: minCapacity, and leave unchanged during deployment
        :param group_metrics: Enable monitoring for group metrics, these metrics describe the group rather than any of its instances. To report all group metrics use ``GroupMetrics.all()`` Group metrics are reported in a granularity of 1 minute at no additional charge. Default: - no group metrics will be reported
        :param health_check: Configuration for health checks. Default: - HealthCheck.ec2 with no grace period
        :param ignore_unmodified_size_properties: If the ASG has scheduled actions, don't reset unchanged group sizes. Only used if the ASG has scheduled actions (which may scale your ASG up or down regardless of cdk deployments). If true, the size of the group will only be reset if it has been changed in the CDK app. If false, the sizes will always be changed back to what they were in the CDK app on deployment. Default: true
        :param instance_monitoring: Controls whether instances in this group are launched with detailed or basic monitoring. When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Monitoring.DETAILED
        :param key_name: Name of SSH keypair to grant access to instances. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - No SSH access will be possible.
        :param max_capacity: Maximum number of instances in the fleet. Default: desiredCapacity
        :param max_instance_lifetime: The maximum amount of time that an instance can be in service. The maximum duration applies to all current and future instances in the group. As an instance approaches its maximum duration, it is terminated and replaced, and cannot be used again. You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value, leave this property undefined. Default: none
        :param min_capacity: Minimum number of instances in the fleet. Default: 1
        :param new_instances_protected_from_scale_in: Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in. By default, Auto Scaling can terminate an instance at any time after launch when scaling in an Auto Scaling Group, subject to the group's termination policy. However, you may wish to protect newly-launched instances from being scaled in if they are going to run critical applications that should not be prematurely terminated. This flag must be enabled if the Auto Scaling Group will be associated with an ECS Capacity Provider with managed termination protection. Default: false
        :param notifications: Configure autoscaling group to send notifications about fleet changes to an SNS topic(s). Default: - No fleet change notifications will be sent.
        :param signals: Configure waiting for signals during deployment. Use this to pause the CloudFormation deployment to wait for the instances in the AutoScalingGroup to report successful startup during creation and updates. The UserData script needs to invoke ``cfn-signal`` with a success or failure code after it is done setting up the instance. Without waiting for signals, the CloudFormation deployment will proceed as soon as the AutoScalingGroup has been created or updated but before the instances in the group have been started. For example, to have instances wait for an Elastic Load Balancing health check before they signal success, add a health-check verification by using the cfn-init helper script. For an example, see the verify_instance_health command in the Auto Scaling rolling updates sample template: https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml Default: - Do not wait for signals
        :param spot_price: The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request. Spot Instances are launched when the price you specify exceeds the current Spot market price. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: none
        :param ssm_session_permissions: Add SSM session permissions to the instance role. Setting this to ``true`` adds the necessary permissions to connect to the instance using SSM Session Manager. You can do this from the AWS Console. NOTE: Setting this flag to ``true`` may not be enough by itself. You must also use an AMI that comes with the SSM Agent, or install the SSM Agent yourself. See `Working with SSM Agent <https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-agent.html>`_ in the SSM Developer Guide. Default: false
        :param termination_policies: A policy or a list of policies that are used to select the instances to terminate. The policies are executed in the order that you list them. Default: - ``TerminationPolicy.DEFAULT``
        :param update_policy: What to do when an AutoScalingGroup's instance configuration is changed. This is applied when any of the settings on the ASG are changed that affect how the instances should be created (VPC, instance type, startup scripts, etc.). It indicates how the existing instances should be replaced with new instances matching the new config. By default, nothing is done and only new instances are launched with the new config. Default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise
        :param vpc_subnets: Where to place instances within the VPC. Default: - All Private subnets.
        :param instance_type: The EC2 instance type to use when launching instances into the AutoScalingGroup.
        :param machine_image: The ECS-optimized AMI variant to use. The default is to use an ECS-optimized AMI of Amazon Linux 2 which is automatically updated to the latest version on every deployment. This will replace the instances in the AutoScalingGroup. Make sure you have not disabled task draining, to avoid downtime when the AMI updates. To use an image that does not update on every deployment, pass:: const machineImage = ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.STANDARD, { cachedInContext: true, }); For more information, see `Amazon ECS-optimized AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_. You must define either ``machineImage`` or ``machineImageType``, not both. Default: - Automatically updated, ECS-optimized Amazon Linux 2

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            
            
            # Create an ECS cluster
            cluster = ecs.Cluster(self, "Cluster", vpc=vpc)
            
            # Add capacity to it
            cluster.add_capacity("DefaultAutoScalingGroupCapacity",
                instance_type=ec2.InstanceType("t2.xlarge"),
                desired_capacity=3
            )
            
            task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
            
            task_definition.add_container("DefaultContainer",
                image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
                memory_limit_mi_b=512
            )
            
            # Instantiate an Amazon ECS Service
            ecs_service = ecs.Ec2Service(self, "Service",
                cluster=cluster,
                task_definition=task_definition
            )
        '''
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _SubnetSelection_e57d76df(**vpc_subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__64f2d9b3495e3be78346f77d5ad90928968c8ce230e670b6279dc67ad934bcd9)
            check_type(argname="argument can_containers_access_instance_role", value=can_containers_access_instance_role, expected_type=type_hints["can_containers_access_instance_role"])
            check_type(argname="argument machine_image_type", value=machine_image_type, expected_type=type_hints["machine_image_type"])
            check_type(argname="argument spot_instance_draining", value=spot_instance_draining, expected_type=type_hints["spot_instance_draining"])
            check_type(argname="argument topic_encryption_key", value=topic_encryption_key, expected_type=type_hints["topic_encryption_key"])
            check_type(argname="argument allow_all_outbound", value=allow_all_outbound, expected_type=type_hints["allow_all_outbound"])
            check_type(argname="argument associate_public_ip_address", value=associate_public_ip_address, expected_type=type_hints["associate_public_ip_address"])
            check_type(argname="argument auto_scaling_group_name", value=auto_scaling_group_name, expected_type=type_hints["auto_scaling_group_name"])
            check_type(argname="argument block_devices", value=block_devices, expected_type=type_hints["block_devices"])
            check_type(argname="argument capacity_rebalance", value=capacity_rebalance, expected_type=type_hints["capacity_rebalance"])
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument default_instance_warmup", value=default_instance_warmup, expected_type=type_hints["default_instance_warmup"])
            check_type(argname="argument desired_capacity", value=desired_capacity, expected_type=type_hints["desired_capacity"])
            check_type(argname="argument group_metrics", value=group_metrics, expected_type=type_hints["group_metrics"])
            check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
            check_type(argname="argument ignore_unmodified_size_properties", value=ignore_unmodified_size_properties, expected_type=type_hints["ignore_unmodified_size_properties"])
            check_type(argname="argument instance_monitoring", value=instance_monitoring, expected_type=type_hints["instance_monitoring"])
            check_type(argname="argument key_name", value=key_name, expected_type=type_hints["key_name"])
            check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
            check_type(argname="argument max_instance_lifetime", value=max_instance_lifetime, expected_type=type_hints["max_instance_lifetime"])
            check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
            check_type(argname="argument new_instances_protected_from_scale_in", value=new_instances_protected_from_scale_in, expected_type=type_hints["new_instances_protected_from_scale_in"])
            check_type(argname="argument notifications", value=notifications, expected_type=type_hints["notifications"])
            check_type(argname="argument signals", value=signals, expected_type=type_hints["signals"])
            check_type(argname="argument spot_price", value=spot_price, expected_type=type_hints["spot_price"])
            check_type(argname="argument ssm_session_permissions", value=ssm_session_permissions, expected_type=type_hints["ssm_session_permissions"])
            check_type(argname="argument termination_policies", value=termination_policies, expected_type=type_hints["termination_policies"])
            check_type(argname="argument update_policy", value=update_policy, expected_type=type_hints["update_policy"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument machine_image", value=machine_image, expected_type=type_hints["machine_image"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "instance_type": instance_type,
        }
        if can_containers_access_instance_role is not None:
            self._values["can_containers_access_instance_role"] = can_containers_access_instance_role
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if spot_instance_draining is not None:
            self._values["spot_instance_draining"] = spot_instance_draining
        if topic_encryption_key is not None:
            self._values["topic_encryption_key"] = topic_encryption_key
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if associate_public_ip_address is not None:
            self._values["associate_public_ip_address"] = associate_public_ip_address
        if auto_scaling_group_name is not None:
            self._values["auto_scaling_group_name"] = auto_scaling_group_name
        if block_devices is not None:
            self._values["block_devices"] = block_devices
        if capacity_rebalance is not None:
            self._values["capacity_rebalance"] = capacity_rebalance
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if default_instance_warmup is not None:
            self._values["default_instance_warmup"] = default_instance_warmup
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if group_metrics is not None:
            self._values["group_metrics"] = group_metrics
        if health_check is not None:
            self._values["health_check"] = health_check
        if ignore_unmodified_size_properties is not None:
            self._values["ignore_unmodified_size_properties"] = ignore_unmodified_size_properties
        if instance_monitoring is not None:
            self._values["instance_monitoring"] = instance_monitoring
        if key_name is not None:
            self._values["key_name"] = key_name
        if max_capacity is not None:
            self._values["max_capacity"] = max_capacity
        if max_instance_lifetime is not None:
            self._values["max_instance_lifetime"] = max_instance_lifetime
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity
        if new_instances_protected_from_scale_in is not None:
            self._values["new_instances_protected_from_scale_in"] = new_instances_protected_from_scale_in
        if notifications is not None:
            self._values["notifications"] = notifications
        if signals is not None:
            self._values["signals"] = signals
        if spot_price is not None:
            self._values["spot_price"] = spot_price
        if ssm_session_permissions is not None:
            self._values["ssm_session_permissions"] = ssm_session_permissions
        if termination_policies is not None:
            self._values["termination_policies"] = termination_policies
        if update_policy is not None:
            self._values["update_policy"] = update_policy
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets
        if machine_image is not None:
            self._values["machine_image"] = machine_image

    @builtins.property
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the containers can access the container instance role.

        :default: false
        '''
        result = self._values.get("can_containers_access_instance_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''What type of machine image this is.

        Depending on the setting, different UserData will automatically be added
        to the ``AutoScalingGroup`` to configure it properly for use with ECS.

        If you create an ``AutoScalingGroup`` yourself and are adding it via
        ``addAutoScalingGroup()``, you must specify this value. If you are adding an
        ``autoScalingGroup`` via ``addCapacity``, this value will be determined
        from the ``machineImage`` you pass.

        :default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def spot_instance_draining(self) -> typing.Optional[builtins.bool]:
        '''Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.

        For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_.

        :default: false
        '''
        result = self._values.get("spot_instance_draining")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def topic_encryption_key(self) -> typing.Optional[_IKey_5f11635f]:
        '''If ``AddAutoScalingGroupCapacityOptions.taskDrainTime`` is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information.

        :default: The SNS Topic will not be encrypted.
        '''
        result = self._values.get("topic_encryption_key")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether the instances can initiate connections to anywhere by default.

        :default: true
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def associate_public_ip_address(self) -> typing.Optional[builtins.bool]:
        '''Whether instances in the Auto Scaling Group should have public IP addresses associated with them.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - Use subnet setting.
        '''
        result = self._values.get("associate_public_ip_address")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def auto_scaling_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Auto Scaling group.

        This name must be unique per Region per account.

        :default: - Auto generated by CloudFormation
        '''
        result = self._values.get("auto_scaling_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def block_devices(self) -> typing.Optional[typing.List[_BlockDevice_0cfc0568]]:
        '''Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.

        Each instance that is launched has an associated root device volume,
        either an Amazon EBS volume or an instance store volume.
        You can use block device mappings to specify additional EBS volumes or
        instance store volumes to attach to an instance when it is launched.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - Uses the block device mapping of the AMI

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
        '''
        result = self._values.get("block_devices")
        return typing.cast(typing.Optional[typing.List[_BlockDevice_0cfc0568]], result)

    @builtins.property
    def capacity_rebalance(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether Capacity Rebalancing is enabled.

        When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling
        attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of
        interruption. After launching a new instance, it then terminates an old instance.

        :default: false

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-capacityrebalance
        '''
        result = self._values.get("capacity_rebalance")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Default scaling cooldown for this AutoScalingGroup.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def default_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The amount of time, in seconds, until a newly launched instance can contribute to the Amazon CloudWatch metrics.

        This delay lets an instance finish initializing before Amazon EC2 Auto Scaling aggregates instance metrics,
        resulting in more reliable usage data. Set this value equal to the amount of time that it takes for resource
        consumption to become stable after an instance reaches the InService state.

        To optimize the performance of scaling policies that scale continuously, such as target tracking and
        step scaling policies, we strongly recommend that you enable the default instance warmup, even if its value is set to 0 seconds

        Default instance warmup will not be added if no value is specified

        :default: None

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-default-instance-warmup.html
        '''
        result = self._values.get("default_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''Initial amount of instances in the fleet.

        If this is set to a number, every deployment will reset the amount of
        instances to this number. It is recommended to leave this value blank.

        :default: minCapacity, and leave unchanged during deployment

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacity
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def group_metrics(self) -> typing.Optional[typing.List[_GroupMetrics_7cdf729b]]:
        '''Enable monitoring for group metrics, these metrics describe the group rather than any of its instances.

        To report all group metrics use ``GroupMetrics.all()``
        Group metrics are reported in a granularity of 1 minute at no additional charge.

        :default: - no group metrics will be reported
        '''
        result = self._values.get("group_metrics")
        return typing.cast(typing.Optional[typing.List[_GroupMetrics_7cdf729b]], result)

    @builtins.property
    def health_check(self) -> typing.Optional[_HealthCheck_03a4bd5a]:
        '''Configuration for health checks.

        :default: - HealthCheck.ec2 with no grace period
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional[_HealthCheck_03a4bd5a], result)

    @builtins.property
    def ignore_unmodified_size_properties(self) -> typing.Optional[builtins.bool]:
        '''If the ASG has scheduled actions, don't reset unchanged group sizes.

        Only used if the ASG has scheduled actions (which may scale your ASG up
        or down regardless of cdk deployments). If true, the size of the group
        will only be reset if it has been changed in the CDK app. If false, the
        sizes will always be changed back to what they were in the CDK app
        on deployment.

        :default: true
        '''
        result = self._values.get("ignore_unmodified_size_properties")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def instance_monitoring(self) -> typing.Optional[_Monitoring_50020f91]:
        '''Controls whether instances in this group are launched with detailed or basic monitoring.

        When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account
        is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - Monitoring.DETAILED

        :see: https://docs.aws.amazon.com/autoscaling/latest/userguide/as-instance-monitoring.html#enable-as-instance-metrics
        '''
        result = self._values.get("instance_monitoring")
        return typing.cast(typing.Optional[_Monitoring_50020f91], result)

    @builtins.property
    def key_name(self) -> typing.Optional[builtins.str]:
        '''Name of SSH keypair to grant access to instances.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - No SSH access will be possible.
        '''
        result = self._values.get("key_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_capacity(self) -> typing.Optional[jsii.Number]:
        '''Maximum number of instances in the fleet.

        :default: desiredCapacity
        '''
        result = self._values.get("max_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_instance_lifetime(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum amount of time that an instance can be in service.

        The maximum duration applies
        to all current and future instances in the group. As an instance approaches its maximum duration,
        it is terminated and replaced, and cannot be used again.

        You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value,
        leave this property undefined.

        :default: none

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html
        '''
        result = self._values.get("max_instance_lifetime")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''Minimum number of instances in the fleet.

        :default: 1
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def new_instances_protected_from_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in.

        By default, Auto Scaling can terminate an instance at any time after launch
        when scaling in an Auto Scaling Group, subject to the group's termination
        policy. However, you may wish to protect newly-launched instances from
        being scaled in if they are going to run critical applications that should
        not be prematurely terminated.

        This flag must be enabled if the Auto Scaling Group will be associated with
        an ECS Capacity Provider with managed termination protection.

        :default: false
        '''
        result = self._values.get("new_instances_protected_from_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def notifications(
        self,
    ) -> typing.Optional[typing.List[_NotificationConfiguration_d5911670]]:
        '''Configure autoscaling group to send notifications about fleet changes to an SNS topic(s).

        :default: - No fleet change notifications will be sent.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-notificationconfigurations
        '''
        result = self._values.get("notifications")
        return typing.cast(typing.Optional[typing.List[_NotificationConfiguration_d5911670]], result)

    @builtins.property
    def signals(self) -> typing.Optional[_Signals_69fbeb6e]:
        '''Configure waiting for signals during deployment.

        Use this to pause the CloudFormation deployment to wait for the instances
        in the AutoScalingGroup to report successful startup during
        creation and updates. The UserData script needs to invoke ``cfn-signal``
        with a success or failure code after it is done setting up the instance.

        Without waiting for signals, the CloudFormation deployment will proceed as
        soon as the AutoScalingGroup has been created or updated but before the
        instances in the group have been started.

        For example, to have instances wait for an Elastic Load Balancing health check before
        they signal success, add a health-check verification by using the
        cfn-init helper script. For an example, see the verify_instance_health
        command in the Auto Scaling rolling updates sample template:

        https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml

        :default: - Do not wait for signals
        '''
        result = self._values.get("signals")
        return typing.cast(typing.Optional[_Signals_69fbeb6e], result)

    @builtins.property
    def spot_price(self) -> typing.Optional[builtins.str]:
        '''The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request.

        Spot Instances are
        launched when the price you specify exceeds the current Spot market price.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: none
        '''
        result = self._values.get("spot_price")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ssm_session_permissions(self) -> typing.Optional[builtins.bool]:
        '''Add SSM session permissions to the instance role.

        Setting this to ``true`` adds the necessary permissions to connect
        to the instance using SSM Session Manager. You can do this
        from the AWS Console.

        NOTE: Setting this flag to ``true`` may not be enough by itself.
        You must also use an AMI that comes with the SSM Agent, or install
        the SSM Agent yourself. See
        `Working with SSM Agent <https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-agent.html>`_
        in the SSM Developer Guide.

        :default: false
        '''
        result = self._values.get("ssm_session_permissions")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def termination_policies(
        self,
    ) -> typing.Optional[typing.List[_TerminationPolicy_89633c56]]:
        '''A policy or a list of policies that are used to select the instances to terminate.

        The policies are executed in the order that you list them.

        :default: - ``TerminationPolicy.DEFAULT``

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html
        '''
        result = self._values.get("termination_policies")
        return typing.cast(typing.Optional[typing.List[_TerminationPolicy_89633c56]], result)

    @builtins.property
    def update_policy(self) -> typing.Optional[_UpdatePolicy_6dffc7ca]:
        '''What to do when an AutoScalingGroup's instance configuration is changed.

        This is applied when any of the settings on the ASG are changed that
        affect how the instances should be created (VPC, instance type, startup
        scripts, etc.). It indicates how the existing instances should be
        replaced with new instances matching the new config. By default, nothing
        is done and only new instances are launched with the new config.

        :default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise
        '''
        result = self._values.get("update_policy")
        return typing.cast(typing.Optional[_UpdatePolicy_6dffc7ca], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''Where to place instances within the VPC.

        :default: - All Private subnets.
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    @builtins.property
    def instance_type(self) -> _InstanceType_f64915b9:
        '''The EC2 instance type to use when launching instances into the AutoScalingGroup.'''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(_InstanceType_f64915b9, result)

    @builtins.property
    def machine_image(self) -> typing.Optional[_IMachineImage_0e8bd50b]:
        '''The ECS-optimized AMI variant to use.

        The default is to use an ECS-optimized AMI of Amazon Linux 2 which is
        automatically updated to the latest version on every deployment. This will
        replace the instances in the AutoScalingGroup. Make sure you have not disabled
        task draining, to avoid downtime when the AMI updates.

        To use an image that does not update on every deployment, pass::

           machine_image = ecs.EcsOptimizedImage.amazon_linux2(ecs.AmiHardwareType.STANDARD,
               cached_in_context=True
           )

        For more information, see `Amazon ECS-optimized
        AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_.

        You must define either ``machineImage`` or ``machineImageType``, not both.

        :default: - Automatically updated, ECS-optimized Amazon Linux 2
        '''
        result = self._values.get("machine_image")
        return typing.cast(typing.Optional[_IMachineImage_0e8bd50b], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddCapacityOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_ecs.AlarmBehavior")
class AlarmBehavior(enum.Enum):
    '''Deployment behavior when an ECS Service Deployment Alarm is triggered.

    :exampleMetadata: infused

    Example::

        # Example automatically generated from non-compiling source. May contain errors.
        import aws_cdk.aws_cloudwatch as cw
        # cluster: ecs.Cluster
        # task_definition: ecs.TaskDefinition
        # elb_alarm: cloudwatch.Alarm
        
        service = ecs.FargateService(self, "Service",
            cluster=cluster,
            task_definition=task_definition,
            deployment_alarms=ecs.DeploymentAlarmConfig(
                alarms=[elb_alarm.alarm_name],
                behavior=AlarmBehavior.ROLLBACK_ON_ALARM
            )
        )
        
        # Defining a deployment alarm after the service has been created
        cpu_alarm_name = "MyCpuMetricAlarm"
        cw.Alarm(self, "CPUAlarm",
            alarm_name=cpu_alarm_name,
            metric=service.metric_cpu_utilization(),
            evaluation_periods=2,
            threshold=80
        )
        service.enable_deployment_alarms([cpu_alarm_name], AlarmBehavior.FAIL_ON_ALARM)
    '''

    ROLLBACK_ON_ALARM = "ROLLBACK_ON_ALARM"
    '''ROLLBACK_ON_ALARM causes the service to roll back to the previous deployment when any deployment alarm enters the 'Alarm' state.

    The Cloudformation stack
    will be rolled back and enter state "UPDATE_ROLLBACK_COMPLETE".
    '''
    FAIL_ON_ALARM = "FAIL_ON_ALARM"
    '''FAIL_ON_ALARM causes the deployment to fail immediately when any deployment alarm enters the 'Alarm' state.

    In order to restore functionality, you must
    roll the stack forward by pushing a new version of the ECS service.
    '''


@jsii.enum(jsii_type="aws-cdk-lib.aws_ecs.AmiHardwareType")
class AmiHardwareType(enum.Enum):
    '''The ECS-optimized AMI variant to use.

    For more information, see
    `Amazon ECS-optimized AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_.

    :exampleMetadata: infused

    Example::

        # cluster: ecs.Cluster
        
        
        cluster.add_capacity("graviton-cluster",
            min_capacity=2,
            instance_type=ec2.InstanceType("c6g.large"),
            machine_image=ecs.EcsOptimizedImage.amazon_linux2(ecs.AmiHardwareType.ARM)
        )
    '''

    STANDARD = "STANDARD"
    '''Use the standard Amazon ECS-optimized AMI.'''
    GPU = "GPU"
    '''Use the Amazon ECS GPU-optimized AMI.'''
    ARM = "ARM"
    '''Use the Amazon ECS-optimized Amazon Linux 2 (arm64) AMI.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.AppMeshProxyConfigurationConfigProps",
    jsii_struct_bases=[],
    name_mapping={"container_name": "containerName", "properties": "properties"},
)
class AppMeshProxyConfigurationConfigProps:
    def __init__(
        self,
        *,
        container_name: builtins.str,
        properties: typing.Union["AppMeshProxyConfigurationProps", typing.Dict[builtins.str, typing.Any]],
    ) -> None:
        '''The configuration to use when setting an App Mesh proxy configuration.

        :param container_name: The name of the container that will serve as the App Mesh proxy.
        :param properties: The set of network configuration parameters to provide the Container Network Interface (CNI) plugin.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            app_mesh_proxy_configuration_config_props = ecs.AppMeshProxyConfigurationConfigProps(
                container_name="containerName",
                properties=ecs.AppMeshProxyConfigurationProps(
                    app_ports=[123],
                    proxy_egress_port=123,
                    proxy_ingress_port=123,
            
                    # the properties below are optional
                    egress_ignored_iPs=["egressIgnoredIPs"],
                    egress_ignored_ports=[123],
                    ignored_gID=123,
                    ignored_uID=123
                )
            )
        '''
        if isinstance(properties, dict):
            properties = AppMeshProxyConfigurationProps(**properties)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0405fe235aadd4430faf2b963e8de52a23bd867724c74f1f76995c5f208aa3e6)
            check_type(argname="argument container_name", value=container_name, expected_type=type_hints["container_name"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "container_name": container_name,
            "properties": properties,
        }

    @builtins.property
    def container_name(self) -> builtins.str:
        '''The name of the container that will serve as the App Mesh proxy.'''
        result = self._values.get("container_name")
        assert result is not None, "Required property 'container_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def properties(self) -> "AppMeshProxyConfigurationProps":
        '''The set of network configuration parameters to provide the Container Network Interface (CNI) plugin.'''
        result = self._values.get("properties")
        assert result is not None, "Required property 'properties' is missing"
        return typing.cast("AppMeshProxyConfigurationProps", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AppMeshProxyConfigurationConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.AppMeshProxyConfigurationProps",
    jsii_struct_bases=[],
    name_mapping={
        "app_ports": "appPorts",
        "proxy_egress_port": "proxyEgressPort",
        "proxy_ingress_port": "proxyIngressPort",
        "egress_ignored_i_ps": "egressIgnoredIPs",
        "egress_ignored_ports": "egressIgnoredPorts",
        "ignored_gid": "ignoredGID",
        "ignored_uid": "ignoredUID",
    },
)
class AppMeshProxyConfigurationProps:
    def __init__(
        self,
        *,
        app_ports: typing.Sequence[jsii.Number],
        proxy_egress_port: jsii.Number,
        proxy_ingress_port: jsii.Number,
        egress_ignored_i_ps: typing.Optional[typing.Sequence[builtins.str]] = None,
        egress_ignored_ports: typing.Optional[typing.Sequence[jsii.Number]] = None,
        ignored_gid: typing.Optional[jsii.Number] = None,
        ignored_uid: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Interface for setting the properties of proxy configuration.

        :param app_ports: The list of ports that the application uses. Network traffic to these ports is forwarded to the ProxyIngressPort and ProxyEgressPort.
        :param proxy_egress_port: Specifies the port that outgoing traffic from the AppPorts is directed to.
        :param proxy_ingress_port: Specifies the port that incoming traffic to the AppPorts is directed to.
        :param egress_ignored_i_ps: The egress traffic going to these specified IP addresses is ignored and not redirected to the ProxyEgressPort. It can be an empty list.
        :param egress_ignored_ports: The egress traffic going to these specified ports is ignored and not redirected to the ProxyEgressPort. It can be an empty list.
        :param ignored_gid: The group ID (GID) of the proxy container as defined by the user parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If IgnoredUID is specified, this field can be empty.
        :param ignored_uid: The user ID (UID) of the proxy container as defined by the user parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If IgnoredGID is specified, this field can be empty.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            app_mesh_proxy_configuration_props = ecs.AppMeshProxyConfigurationProps(
                app_ports=[123],
                proxy_egress_port=123,
                proxy_ingress_port=123,
            
                # the properties below are optional
                egress_ignored_iPs=["egressIgnoredIPs"],
                egress_ignored_ports=[123],
                ignored_gID=123,
                ignored_uID=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4667ff085267890b2ad63b523cc4e0ef7a9142c48ba250be20487b7047375550)
            check_type(argname="argument app_ports", value=app_ports, expected_type=type_hints["app_ports"])
            check_type(argname="argument proxy_egress_port", value=proxy_egress_port, expected_type=type_hints["proxy_egress_port"])
            check_type(argname="argument proxy_ingress_port", value=proxy_ingress_port, expected_type=type_hints["proxy_ingress_port"])
            check_type(argname="argument egress_ignored_i_ps", value=egress_ignored_i_ps, expected_type=type_hints["egress_ignored_i_ps"])
            check_type(argname="argument egress_ignored_ports", value=egress_ignored_ports, expected_type=type_hints["egress_ignored_ports"])
            check_type(argname="argument ignored_gid", value=ignored_gid, expected_type=type_hints["ignored_gid"])
            check_type(argname="argument ignored_uid", value=ignored_uid, expected_type=type_hints["ignored_uid"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "app_ports": app_ports,
            "proxy_egress_port": proxy_egress_port,
            "proxy_ingress_port": proxy_ingress_port,
        }
        if egress_ignored_i_ps is not None:
            self._values["egress_ignored_i_ps"] = egress_ignored_i_ps
        if egress_ignored_ports is not None:
            self._values["egress_ignored_ports"] = egress_ignored_ports
        if ignored_gid is not None:
            self._values["ignored_gid"] = ignored_gid
        if ignored_uid is not None:
            self._values["ignored_uid"] = ignored_uid

    @builtins.property
    def app_ports(self) -> typing.List[jsii.Number]:
        '''The list of ports that the application uses.

        Network traffic to these ports is forwarded to the ProxyIngressPort and ProxyEgressPort.
        '''
        result = self._values.get("app_ports")
        assert result is not None, "Required property 'app_ports' is missing"
        return typing.cast(typing.List[jsii.Number], result)

    @builtins.property
    def proxy_egress_port(self) -> jsii.Number:
        '''Specifies the port that outgoing traffic from the AppPorts is directed to.'''
        result = self._values.get("proxy_egress_port")
        assert result is not None, "Required property 'proxy_egress_port' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def proxy_ingress_port(self) -> jsii.Number:
        '''Specifies the port that incoming traffic to the AppPorts is directed to.'''
        result = self._values.get("proxy_ingress_port")
        assert result is not None, "Required property 'proxy_ingress_port' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def egress_ignored_i_ps(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The egress traffic going to these specified IP addresses is ignored and not redirected to the ProxyEgressPort.

        It can be an empty list.
        '''
        result = self._values.get("egress_ignored_i_ps")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def egress_ignored_ports(self) -> typing.Optional[typing.List[jsii.Number]]:
        '''The egress traffic going to these specified ports is ignored and not redirected to the ProxyEgressPort.

        It can be an empty list.
        '''
        result = self._values.get("egress_ignored_ports")
        return typing.cast(typing.Optional[typing.List[jsii.Number]], result)

    @builtins.property
    def ignored_gid(self) -> typing.Optional[jsii.Number]:
        '''The group ID (GID) of the proxy container as defined by the user parameter in a container definition.

        This is used to ensure the proxy ignores its own traffic. If IgnoredUID is specified, this field can be empty.
        '''
        result = self._values.get("ignored_gid")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ignored_uid(self) -> typing.Optional[jsii.Number]:
        '''The user ID (UID) of the proxy container as defined by the user parameter in a container definition.

        This is used to ensure the proxy ignores its own traffic. If IgnoredGID is specified, this field can be empty.
        '''
        result = self._values.get("ignored_uid")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AppMeshProxyConfigurationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AppProtocol(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.aws_ecs.AppProtocol"):
    '''Service connect app protocol.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ecs as ecs
        
        app_protocol = ecs.AppProtocol.grpc
    '''

    def __init__(self, value: builtins.str) -> None:
        '''
        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa9f02c3b1858cd5337318a1c25418a91e4a16fcaa1ec5674736441ea9d77fe3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.create(self.__class__, self, [value])

    @jsii.python.classproperty
    @jsii.member(jsii_name="grpc")
    def grpc(cls) -> "AppProtocol":  # pyright: ignore [reportGeneralTypeIssues]
        '''GRPC app protocol.'''
        return typing.cast("AppProtocol", jsii.sget(cls, "grpc"))

    @grpc.setter # type: ignore[no-redef]
    def grpc(cls, value: "AppProtocol") -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__77b98455b2c3523f794df1e23970004d7b2edb7c5cdb877856bd0b746e4c1c50)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.sset(cls, "grpc", value)

    @jsii.python.classproperty
    @jsii.member(jsii_name="http")
    def http(cls) -> "AppProtocol":  # pyright: ignore [reportGeneralTypeIssues]
        '''HTTP app protocol.'''
        return typing.cast("AppProtocol", jsii.sget(cls, "http"))

    @http.setter # type: ignore[no-redef]
    def http(cls, value: "AppProtocol") -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0439c1eaca6689b173b1cd0312fdc06e461182b0be47279b1547bb0b68a90c4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.sset(cls, "http", value)

    @jsii.python.classproperty
    @jsii.member(jsii_name="http2")
    def http2(cls) -> "AppProtocol":  # pyright: ignore [reportGeneralTypeIssues]
        '''HTTP2 app protocol.'''
        return typing.cast("AppProtocol", jsii.sget(cls, "http2"))

    @http2.setter # type: ignore[no-redef]
    def http2(cls, value: "AppProtocol") -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__713cc19ddd985e6eacdf63b05272a22567b0bd13de0196a213d990f4d0116ca6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.sset(cls, "http2", value)

    @builtins.property
    @jsii.member(jsii_name="value")
    def value(self) -> builtins.str:
        '''Custom value.'''
        return typing.cast(builtins.str, jsii.get(self, "value"))


class AsgCapacityProvider(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ecs.AsgCapacityProvider",
):
    '''An Auto Scaling Group Capacity Provider.

    This allows an ECS cluster to target
    a specific EC2 Auto Scaling Group for the placement of tasks. Optionally (and
    recommended), ECS can manage the number of instances in the ASG to fit the
    tasks, and can ensure that instances are not prematurely terminated while
    there are still tasks running on them.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        
        
        cluster = ecs.Cluster(self, "Cluster",
            vpc=vpc
        )
        
        # Either add default capacity
        cluster.add_capacity("DefaultAutoScalingGroupCapacity",
            instance_type=ec2.InstanceType("t2.xlarge"),
            desired_capacity=3
        )
        
        # Or add customized capacity. Be sure to start the Amazon ECS-optimized AMI.
        auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
            vpc=vpc,
            instance_type=ec2.InstanceType("t2.xlarge"),
            machine_image=ecs.EcsOptimizedImage.amazon_linux(),
            # Or use Amazon ECS-Optimized Amazon Linux 2 AMI
            # machineImage: EcsOptimizedImage.amazonLinux2(),
            desired_capacity=3
        )
        
        capacity_provider = ecs.AsgCapacityProvider(self, "AsgCapacityProvider",
            auto_scaling_group=auto_scaling_group
        )
        cluster.add_asg_capacity_provider(capacity_provider)
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auto_scaling_group: _IAutoScalingGroup_360f1cde,
        capacity_provider_name: typing.Optional[builtins.str] = None,
        enable_managed_scaling: typing.Optional[builtins.bool] = None,
        enable_managed_termination_protection: typing.Optional[builtins.bool] = None,
        maximum_scaling_step_size: typing.Optional[jsii.Number] = None,
        minimum_scaling_step_size: typing.Optional[jsii.Number] = None,
        target_capacity_percent: typing.Optional[jsii.Number] = None,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        topic_encryption_key: typing.Optional[_IKey_5f11635f] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param auto_scaling_group: The autoscaling group to add as a Capacity Provider.
        :param capacity_provider_name: The name of the capacity provider. If a name is specified, it cannot start with ``aws``, ``ecs``, or ``fargate``. If no name is specified, a default name in the CFNStackName-CFNResourceName-RandomString format is used. Default: CloudFormation-generated name
        :param enable_managed_scaling: When enabled the scale-in and scale-out actions of the cluster's Auto Scaling Group will be managed for you. This means your cluster will automatically scale instances based on the load your tasks put on the cluster. For more information, see `Using Managed Scaling <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html#asg-capacity-providers-managed-scaling>`_ in the ECS Developer Guide. Default: true
        :param enable_managed_termination_protection: When enabled the Auto Scaling Group will only terminate EC2 instances that no longer have running non-daemon tasks. Scale-in protection will be automatically enabled on instances. When all non-daemon tasks are stopped on an instance, ECS initiates the scale-in process and turns off scale-in protection for the instance. The Auto Scaling Group can then terminate the instance. For more information see `Managed termination protection <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-auto-scaling.html#managed-termination-protection>`_ in the ECS Developer Guide. Managed scaling must also be enabled. Default: true
        :param maximum_scaling_step_size: Maximum scaling step size. In most cases this should be left alone. Default: 1000
        :param minimum_scaling_step_size: Minimum scaling step size. In most cases this should be left alone. Default: 1
        :param target_capacity_percent: Target capacity percent. In most cases this should be left alone. Default: 100
        :param can_containers_access_instance_role: Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: What type of machine image this is. Depending on the setting, different UserData will automatically be added to the ``AutoScalingGroup`` to configure it properly for use with ECS. If you create an ``AutoScalingGroup`` yourself and are adding it via ``addAutoScalingGroup()``, you must specify this value. If you are adding an ``autoScalingGroup`` via ``addCapacity``, this value will be determined from the ``machineImage`` you pass. Default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        :param spot_instance_draining: Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param topic_encryption_key: If ``AddAutoScalingGroupCapacityOptions.taskDrainTime`` is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aef5340f65d89265b7071155451a7bd520c9b5445fba23ce140055eba3d2fd36)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AsgCapacityProviderProps(
            auto_scaling_group=auto_scaling_group,
            capacity_provider_name=capacity_provider_name,
            enable_managed_scaling=enable_managed_scaling,
            enable_managed_termination_protection=enable_managed_termination_protection,
            maximum_scaling_step_size=maximum_scaling_step_size,
            minimum_scaling_step_size=minimum_scaling_step_size,
            target_capacity_percent=target_capacity_percent,
            can_containers_access_instance_role=can_containers_access_instance_role,
            machine_image_type=machine_image_type,
            spot_instance_draining=spot_instance_draining,
            topic_encryption_key=topic_encryption_key,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroup")
    def auto_scaling_group(self) -> _AutoScalingGroup_c547a7b9:
        '''Auto Scaling Group.'''
        return typing.cast(_AutoScalingGroup_c547a7b9, jsii.get(self, "autoScalingGroup"))

    @builtins.property
    @jsii.member(jsii_name="capacityProviderName")
    def capacity_provider_name(self) -> builtins.str:
        '''Capacity provider name.

        :default: Chosen by CloudFormation
        '''
        return typing.cast(builtins.str, jsii.get(self, "capacityProviderName"))

    @builtins.property
    @jsii.member(jsii_name="machineImageType")
    def machine_image_type(self) -> "MachineImageType":
        '''Auto Scaling Group machineImageType.'''
        return typing.cast("MachineImageType", jsii.get(self, "machineImageType"))

    @builtins.property
    @jsii.member(jsii_name="canContainersAccessInstanceRole")
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the containers can access the container instance role.

        :default: false
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "canContainersAccessInstanceRole"))

    @builtins.property
    @jsii.member(jsii_name="enableManagedTerminationProtection")
    def enable_managed_termination_protection(self) -> typing.Optional[builtins.bool]:
        '''Whether managed termination protection is enabled.'''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "enableManagedTerminationProtection"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.AsgCapacityProviderProps",
    jsii_struct_bases=[AddAutoScalingGroupCapacityOptions],
    name_mapping={
        "can_containers_access_instance_role": "canContainersAccessInstanceRole",
        "machine_image_type": "machineImageType",
        "spot_instance_draining": "spotInstanceDraining",
        "topic_encryption_key": "topicEncryptionKey",
        "auto_scaling_group": "autoScalingGroup",
        "capacity_provider_name": "capacityProviderName",
        "enable_managed_scaling": "enableManagedScaling",
        "enable_managed_termination_protection": "enableManagedTerminationProtection",
        "maximum_scaling_step_size": "maximumScalingStepSize",
        "minimum_scaling_step_size": "minimumScalingStepSize",
        "target_capacity_percent": "targetCapacityPercent",
    },
)
class AsgCapacityProviderProps(AddAutoScalingGroupCapacityOptions):
    def __init__(
        self,
        *,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        topic_encryption_key: typing.Optional[_IKey_5f11635f] = None,
        auto_scaling_group: _IAutoScalingGroup_360f1cde,
        capacity_provider_name: typing.Optional[builtins.str] = None,
        enable_managed_scaling: typing.Optional[builtins.bool] = None,
        enable_managed_termination_protection: typing.Optional[builtins.bool] = None,
        maximum_scaling_step_size: typing.Optional[jsii.Number] = None,
        minimum_scaling_step_size: typing.Optional[jsii.Number] = None,
        target_capacity_percent: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''The options for creating an Auto Scaling Group Capacity Provider.

        :param can_containers_access_instance_role: Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: What type of machine image this is. Depending on the setting, different UserData will automatically be added to the ``AutoScalingGroup`` to configure it properly for use with ECS. If you create an ``AutoScalingGroup`` yourself and are adding it via ``addAutoScalingGroup()``, you must specify this value. If you are adding an ``autoScalingGroup`` via ``addCapacity``, this value will be determined from the ``machineImage`` you pass. Default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        :param spot_instance_draining: Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param topic_encryption_key: If ``AddAutoScalingGroupCapacityOptions.taskDrainTime`` is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.
        :param auto_scaling_group: The autoscaling group to add as a Capacity Provider.
        :param capacity_provider_name: The name of the capacity provider. If a name is specified, it cannot start with ``aws``, ``ecs``, or ``fargate``. If no name is specified, a default name in the CFNStackName-CFNResourceName-RandomString format is used. Default: CloudFormation-generated name
        :param enable_managed_scaling: When enabled the scale-in and scale-out actions of the cluster's Auto Scaling Group will be managed for you. This means your cluster will automatically scale instances based on the load your tasks put on the cluster. For more information, see `Using Managed Scaling <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html#asg-capacity-providers-managed-scaling>`_ in the ECS Developer Guide. Default: true
        :param enable_managed_termination_protection: When enabled the Auto Scaling Group will only terminate EC2 instances that no longer have running non-daemon tasks. Scale-in protection will be automatically enabled on instances. When all non-daemon tasks are stopped on an instance, ECS initiates the scale-in process and turns off scale-in protection for the instance. The Auto Scaling Group can then terminate the instance. For more information see `Managed termination protection <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-auto-scaling.html#managed-termination-protection>`_ in the ECS Developer Guide. Managed scaling must also be enabled. Default: true
        :param maximum_scaling_step_size: Maximum scaling step size. In most cases this should be left alone. Default: 1000
        :param minimum_scaling_step_size: Minimum scaling step size. In most cases this should be left alone. Default: 1
        :param target_capacity_percent: Target capacity percent. In most cases this should be left alone. Default: 100

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            
            launch_template = ec2.LaunchTemplate(self, "ASG-LaunchTemplate",
                instance_type=ec2.InstanceType("t3.medium"),
                machine_image=ecs.EcsOptimizedImage.amazon_linux2(),
                user_data=ec2.UserData.for_linux()
            )
            
            auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
                vpc=vpc,
                mixed_instances_policy=autoscaling.MixedInstancesPolicy(
                    instances_distribution=autoscaling.InstancesDistribution(
                        on_demand_percentage_above_base_capacity=50
                    ),
                    launch_template=launch_template
                )
            )
            
            cluster = ecs.Cluster(self, "Cluster", vpc=vpc)
            
            capacity_provider = ecs.AsgCapacityProvider(self, "AsgCapacityProvider",
                auto_scaling_group=auto_scaling_group,
                machine_image_type=ecs.MachineImageType.AMAZON_LINUX_2
            )
            
            cluster.add_asg_capacity_provider(capacity_provider)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__357922855f121de04cb93122c31dd2cf23ade3be254ba5ae30e152fdb2411e5c)
            check_type(argname="argument can_containers_access_instance_role", value=can_containers_access_instance_role, expected_type=type_hints["can_containers_access_instance_role"])
            check_type(argname="argument machine_image_type", value=machine_image_type, expected_type=type_hints["machine_image_type"])
            check_type(argname="argument spot_instance_draining", value=spot_instance_draining, expected_type=type_hints["spot_instance_draining"])
            check_type(argname="argument topic_encryption_key", value=topic_encryption_key, expected_type=type_hints["topic_encryption_key"])
            check_type(argname="argument auto_scaling_group", value=auto_scaling_group, expected_type=type_hints["auto_scaling_group"])
            check_type(argname="argument capacity_provider_name", value=capacity_provider_name, expected_type=type_hints["capacity_provider_name"])
            check_type(argname="argument enable_managed_scaling", value=enable_managed_scaling, expected_type=type_hints["enable_managed_scaling"])
            check_type(argname="argument enable_managed_termination_protection", value=enable_managed_termination_protection, expected_type=type_hints["enable_managed_termination_protection"])
            check_type(argname="argument maximum_scaling_step_size", value=maximum_scaling_step_size, expected_type=type_hints["maximum_scaling_step_size"])
            check_type(argname="argument minimum_scaling_step_size", value=minimum_scaling_step_size, expected_type=type_hints["minimum_scaling_step_size"])
            check_type(argname="argument target_capacity_percent", value=target_capacity_percent, expected_type=type_hints["target_capacity_percent"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auto_scaling_group": auto_scaling_group,
        }
        if can_containers_access_instance_role is not None:
            self._values["can_containers_access_instance_role"] = can_containers_access_instance_role
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if spot_instance_draining is not None:
            self._values["spot_instance_draining"] = spot_instance_draining
        if topic_encryption_key is not None:
            self._values["topic_encryption_key"] = topic_encryption_key
        if capacity_provider_name is not None:
            self._values["capacity_provider_name"] = capacity_provider_name
        if enable_managed_scaling is not None:
            self._values["enable_managed_scaling"] = enable_managed_scaling
        if enable_managed_termination_protection is not None:
            self._values["enable_managed_termination_protection"] = enable_managed_termination_protection
        if maximum_scaling_step_size is not None:
            self._values["maximum_scaling_step_size"] = maximum_scaling_step_size
        if minimum_scaling_step_size is not None:
            self._values["minimum_scaling_step_size"] = minimum_scaling_step_size
        if target_capacity_percent is not None:
            self._values["target_capacity_percent"] = target_capacity_percent

    @builtins.property
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the containers can access the container instance role.

        :default: false
        '''
        result = self._values.get("can_containers_access_instance_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''What type of machine image this is.

        Depending on the setting, different UserData will automatically be added
        to the ``AutoScalingGroup`` to configure it properly for use with ECS.

        If you create an ``AutoScalingGroup`` yourself and are adding it via
        ``addAutoScalingGroup()``, you must specify this value. If you are adding an
        ``autoScalingGroup`` via ``addCapacity``, this value will be determined
        from the ``machineImage`` you pass.

        :default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def spot_instance_draining(self) -> typing.Optional[builtins.bool]:
        '''Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.

        For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_.

        :default: false
        '''
        result = self._values.get("spot_instance_draining")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def topic_encryption_key(self) -> typing.Optional[_IKey_5f11635f]:
        '''If ``AddAutoScalingGroupCapacityOptions.taskDrainTime`` is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information.

        :default: The SNS Topic will not be encrypted.
        '''
        result = self._values.get("topic_encryption_key")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    @builtins.property
    def auto_scaling_group(self) -> _IAutoScalingGroup_360f1cde:
        '''The autoscaling group to add as a Capacity Provider.'''
        result = self._values.get("auto_scaling_group")
        assert result is not None, "Required property 'auto_scaling_group' is missing"
        return typing.cast(_IAutoScalingGroup_360f1cde, result)

    @builtins.property
    def capacity_provider_name(self) -> typing.Optional[builtins.str]:
        '''The name of the capacity provider.

        If a name is specified,
        it cannot start with ``aws``, ``ecs``, or ``fargate``. If no name is specified,
        a default name in the CFNStackName-CFNResourceName-RandomString format is used.

        :default: CloudFormation-generated name
        '''
        result = self._values.get("capacity_provider_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_managed_scaling(self) -> typing.Optional[builtins.bool]:
        '''When enabled the scale-in and scale-out actions of the cluster's Auto Scaling Group will be managed for you.

        This means your cluster will automatically scale instances based on the load your tasks put on the cluster.
        For more information, see `Using Managed Scaling <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html#asg-capacity-providers-managed-scaling>`_ in the ECS Developer Guide.

        :default: true
        '''
        result = self._values.get("enable_managed_scaling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_managed_termination_protection(self) -> typing.Optional[builtins.bool]:
        '''When enabled the Auto Scaling Group will only terminate EC2 instances that no longer have running non-daemon tasks.

        Scale-in protection will be automatically enabled on instances. When all non-daemon tasks are
        stopped on an instance, ECS initiates the scale-in process and turns off scale-in protection for the
        instance. The Auto Scaling Group can then terminate the instance. For more information see `Managed termination
        protection <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-auto-scaling.html#managed-termination-protection>`_
        in the ECS Developer Guide.

        Managed scaling must also be enabled.

        :default: true
        '''
        result = self._values.get("enable_managed_termination_protection")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def maximum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
        '''Maximum scaling step size.

        In most cases this should be left alone.

        :default: 1000
        '''
        result = self._values.get("maximum_scaling_step_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def minimum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
        '''Minimum scaling step size.

        In most cases this should be left alone.

        :default: 1
        '''
        result = self._values.get("minimum_scaling_step_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def target_capacity_percent(self) -> typing.Optional[jsii.Number]:
        '''Target capacity percent.

        In most cases this should be left alone.

        :default: 100
        '''
        result = self._values.get("target_capacity_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AsgCapacityProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.AssetImageProps",
    jsii_struct_bases=[_DockerImageAssetOptions_9580cd76],
    name_mapping={
        "exclude": "exclude",
        "follow_symlinks": "followSymlinks",
        "ignore_mode": "ignoreMode",
        "extra_hash": "extraHash",
        "asset_name": "assetName",
        "build_args": "buildArgs",
        "build_secrets": "buildSecrets",
        "cache_from": "cacheFrom",
        "cache_to": "cacheTo",
        "file": "file",
        "invalidation": "invalidation",
        "network_mode": "networkMode",
        "outputs": "outputs",
        "platform": "platform",
        "target": "target",
    },
)
class AssetImageProps(_DockerImageAssetOptions_9580cd76):
    def __init__(
        self,
        *,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        asset_name: typing.Optional[builtins.str] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_secrets: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[typing.Union[_DockerImageAssetInvalidationOptions_4deb8d45, typing.Dict[builtins.str, typing.Any]]] = None,
        network_mode: typing.Optional[_NetworkMode_897e5081] = None,
        outputs: typing.Optional[typing.Sequence[builtins.str]] = None,
        platform: typing.Optional[_Platform_d16f3cf1] = None,
        target: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The properties for building an AssetImage.

        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param asset_name: Unique identifier of the docker image asset and its potential revisions. Required if using AppScopedStagingSynthesizer. Default: - no asset name
        :param build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param build_secrets: Build secrets. Docker BuildKit must be enabled to use build secrets. Default: - no build secrets
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from options are passed to the build command
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to options are passed to the build command
        :param file: Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param outputs: Outputs to pass to the ``docker build`` command. Default: - no outputs are passed to the build command (default outputs are used)
        :param platform: Platform to build for. *Requires Docker Buildx*. Default: - no platform specified (the current machine architecture will be used)
        :param target: Docker target to build to. Default: - no target

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_ecr_assets as ecr_assets
            from aws_cdk import aws_ecs as ecs
            
            # network_mode: ecr_assets.NetworkMode
            # platform: ecr_assets.Platform
            
            asset_image_props = ecs.AssetImageProps(
                asset_name="assetName",
                build_args={
                    "build_args_key": "buildArgs"
                },
                build_secrets={
                    "build_secrets_key": "buildSecrets"
                },
                cache_from=[ecr_assets.DockerCacheOption(
                    type="type",
            
                    # the properties below are optional
                    params={
                        "params_key": "params"
                    }
                )],
                cache_to=ecr_assets.DockerCacheOption(
                    type="type",
            
                    # the properties below are optional
                    params={
                        "params_key": "params"
                    }
                ),
                exclude=["exclude"],
                extra_hash="extraHash",
                file="file",
                follow_symlinks=cdk.SymlinkFollowMode.NEVER,
                ignore_mode=cdk.IgnoreMode.GLOB,
                invalidation=ecr_assets.DockerImageAssetInvalidationOptions(
                    build_args=False,
                    build_secrets=False,
                    extra_hash=False,
                    file=False,
                    network_mode=False,
                    outputs=False,
                    platform=False,
                    repository_name=False,
                    target=False
                ),
                network_mode=network_mode,
                outputs=["outputs"],
                platform=platform,
                target="target"
            )
        '''
        if isinstance(cache_to, dict):
            cache_to = _DockerCacheOption_58ef18ca(**cache_to)
        if isinstance(invalidation, dict):
            invalidation = _DockerImageAssetInvalidationOptions_4deb8d45(**invalidation)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e2e094929b9952c79f2d2f10cac8da6c1f6dad2f0e50e17e639ed0168684baa)
            check_type(argname="argument exclude", value=exclude, expected_type=type_hints["exclude"])
            check_type(argname="argument follow_symlinks", value=follow_symlinks, expected_type=type_hints["follow_symlinks"])
            check_type(argname="argument ignore_mode", value=ignore_mode, expected_type=type_hints["ignore_mode"])
            check_type(argname="argument extra_hash", value=extra_hash, expected_type=type_hints["extra_hash"])
            check_type(argname="argument asset_name", value=asset_name, expected_type=type_hints["asset_name"])
            check_type(argname="argument build_args", value=build_args, expected_type=type_hints["build_args"])
            check_type(argname="argument build_secrets", value=build_secrets, expected_type=type_hints["build_secrets"])
            check_type(argname="argument cache_from", value=cache_from, expected_type=type_hints["cache_from"])
            check_type(argname="argument cache_to", value=cache_to, expected_type=type_hints["cache_to"])
            check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            check_type(argname="argument invalidation", value=invalidation, expected_type=type_hints["invalidation"])
            check_type(argname="argument network_mode", value=network_mode, expected_type=type_hints["network_mode"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow_symlinks is not None:
            self._values["follow_symlinks"] = follow_symlinks
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode
        if extra_hash is not None:
            self._values["extra_hash"] = extra_hash
        if asset_name is not None:
            self._values["asset_name"] = asset_name
        if build_args is not None:
            self._values["build_args"] = build_args
        if build_secrets is not None:
            self._values["build_secrets"] = build_secrets
        if cache_from is not None:
            self._values["cache_from"] = cache_from
        if cache_to is not None:
            self._values["cache_to"] = cache_to
        if file is not None:
            self._values["file"] = file
        if invalidation is not None:
            self._values["invalidation"] = invalidation
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if outputs is not None:
            self._values["outputs"] = outputs
        if platform is not None:
            self._values["platform"] = platform
        if target is not None:
            self._values["target"] = target

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''File paths matching the patterns will be excluded.

        See ``ignoreMode`` to set the matching behavior.
        Has no effect on Assets bundled using the ``bundling`` property.

        :default: - nothing is excluded
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow_symlinks(self) -> typing.Optional[_SymlinkFollowMode_047ec1f6]:
        '''A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER
        '''
        result = self._values.get("follow_symlinks")
        return typing.cast(typing.Optional[_SymlinkFollowMode_047ec1f6], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional[_IgnoreMode_655a98e8]:
        '''The ignore behavior to use for ``exclude`` patterns.

        :default: IgnoreMode.GLOB
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional[_IgnoreMode_655a98e8], result)

    @builtins.property
    def extra_hash(self) -> typing.Optional[builtins.str]:
        '''Extra information to encode into the fingerprint (e.g. build instructions and other inputs).

        :default: - hash is only based on source content
        '''
        result = self._values.get("extra_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def asset_name(self) -> typing.Optional[builtins.str]:
        '''Unique identifier of the docker image asset and its potential revisions.

        Required if using AppScopedStagingSynthesizer.

        :default: - no asset name
        '''
        result = self._values.get("asset_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def build_args(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build args to pass to the ``docker build`` command.

        Since Docker build arguments are resolved before deployment, keys and
        values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or
        ``queue.queueUrl``).

        :default: - no build args are passed
        '''
        result = self._values.get("build_args")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def build_secrets(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build secrets.

        Docker BuildKit must be enabled to use build secrets.

        :default: - no build secrets

        :see: https://docs.docker.com/build/buildkit/

        Example::

            from aws_cdk import DockerBuildSecret
            
            
            build_secrets = {
                "MY_SECRET": DockerBuildSecret.from_src("file.txt")
            }
        '''
        result = self._values.get("build_secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def cache_from(self) -> typing.Optional[typing.List[_DockerCacheOption_58ef18ca]]:
        '''Cache from options to pass to the ``docker build`` command.

        :default: - no cache from options are passed to the build command

        :see: https://docs.docker.com/build/cache/backends/
        '''
        result = self._values.get("cache_from")
        return typing.cast(typing.Optional[typing.List[_DockerCacheOption_58ef18ca]], result)

    @builtins.property
    def cache_to(self) -> typing.Optional[_DockerCacheOption_58ef18ca]:
        '''Cache to options to pass to the ``docker build`` command.

        :default: - no cache to options are passed to the build command

        :see: https://docs.docker.com/build/cache/backends/
        '''
        result = self._values.get("cache_to")
        return typing.cast(typing.Optional[_DockerCacheOption_58ef18ca], result)

    @builtins.property
    def file(self) -> typing.Optional[builtins.str]:
        '''Path to the Dockerfile (relative to the directory).

        :default: 'Dockerfile'
        '''
        result = self._values.get("file")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def invalidation(
        self,
    ) -> typing.Optional[_DockerImageAssetInvalidationOptions_4deb8d45]:
        '''Options to control which parameters are used to invalidate the asset hash.

        :default: - hash all parameters
        '''
        result = self._values.get("invalidation")
        return typing.cast(typing.Optional[_DockerImageAssetInvalidationOptions_4deb8d45], result)

    @builtins.property
    def network_mode(self) -> typing.Optional[_NetworkMode_897e5081]:
        '''Networking mode for the RUN commands during build.

        Support docker API 1.25+.

        :default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional[_NetworkMode_897e5081], result)

    @builtins.property
    def outputs(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Outputs to pass to the ``docker build`` command.

        :default: - no outputs are passed to the build command (default outputs are used)

        :see: https://docs.docker.com/engine/reference/commandline/build/#custom-build-outputs
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def platform(self) -> typing.Optional[_Platform_d16f3cf1]:
        '''Platform to build for.

        *Requires Docker Buildx*.

        :default: - no platform specified (the current machine architecture will be used)
        '''
        result = self._values.get("platform")
        return typing.cast(typing.Optional[_Platform_d16f3cf1], result)

    @builtins.property
    def target(self) -> typing.Optional[builtins.str]:
        '''Docker target to build to.

        :default: - no target
        '''
        result = self._values.get("target")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssetImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.AssociateCloudMapServiceOptions",
    jsii_struct_bases=[],
    name_mapping={
        "service": "service",
        "container": "container",
        "container_port": "containerPort",
    },
)
class AssociateCloudMapServiceOptions:
    def __init__(
        self,
        *,
        service: _IService_46860ae1,
        container: typing.Optional["ContainerDefinition"] = None,
        container_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''The options for using a cloudmap service.

        :param service: The cloudmap service to register with.
        :param container: The container to point to for a SRV record. Default: - the task definition's default container
        :param container_port: The port to point to for a SRV record. Default: - the default port of the task definition's default container

        :exampleMetadata: infused

        Example::

            # cloud_map_service: cloudmap.Service
            # ecs_service: ecs.FargateService
            
            
            ecs_service.associate_cloud_map_service(
                service=cloud_map_service
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c46c3e1db22dfc4c1792ae79570c2913fd9d1e44e558cea83d8caa4ed6f55c60)
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument container", value=container, expected_type=type_hints["container"])
            check_type(argname="argument container_port", value=container_port, expected_type=type_hints["container_port"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "service": service,
        }
        if container is not None:
            self._values["container"] = container
        if container_port is not None:
            self._values["container_port"] = container_port

    @builtins.property
    def service(self) -> _IService_46860ae1:
        '''The cloudmap service to register with.'''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(_IService_46860ae1, result)

    @builtins.property
    def container(self) -> typing.Optional["ContainerDefinition"]:
        '''The container to point to for a SRV record.

        :default: - the task definition's default container
        '''
        result = self._values.get("container")
        return typing.cast(typing.Optional["ContainerDefinition"], result)

    @builtins.property
    def container_port(self) -> typing.Optional[jsii.Number]:
        '''The port to point to for a SRV record.

        :default: - the default port of the task definition's default container
        '''
        result = self._values.get("container_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssociateCloudMapServiceOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.AuthorizationConfig",
    jsii_struct_bases=[],
    name_mapping={"access_point_id": "accessPointId", "iam": "iam"},
)
class AuthorizationConfig:
    def __init__(
        self,
        *,
        access_point_id: typing.Optional[builtins.str] = None,
        iam: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The authorization configuration details for the Amazon EFS file system.

        :param access_point_id: The access point ID to use. If an access point is specified, the root directory value will be relative to the directory set for the access point. If specified, transit encryption must be enabled in the EFSVolumeConfiguration. Default: No id
        :param iam: Whether or not to use the Amazon ECS task IAM role defined in a task definition when mounting the Amazon EFS file system. If enabled, transit encryption must be enabled in the EFSVolumeConfiguration. Valid values: ENABLED | DISABLED Default: If this parameter is omitted, the default value of DISABLED is used.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            authorization_config = ecs.AuthorizationConfig(
                access_point_id="accessPointId",
                iam="iam"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__810895b0b0df12762130d2cdd05cb036ed79a7f35a4fce2de0c2bba0adb55dc0)
            check_type(argname="argument access_point_id", value=access_point_id, expected_type=type_hints["access_point_id"])
            check_type(argname="argument iam", value=iam, expected_type=type_hints["iam"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if access_point_id is not None:
            self._values["access_point_id"] = access_point_id
        if iam is not None:
            self._values["iam"] = iam

    @builtins.property
    def access_point_id(self) -> typing.Optional[builtins.str]:
        '''The access point ID to use.

        If an access point is specified, the root directory value will be
        relative to the directory set for the access point.
        If specified, transit encryption must be enabled in the EFSVolumeConfiguration.

        :default: No id
        '''
        result = self._values.get("access_point_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iam(self) -> typing.Optional[builtins.str]:
        '''Whether or not to use the Amazon ECS task IAM role defined in a task definition when mounting the Amazon EFS file system.

        If enabled, transit encryption must be enabled in the EFSVolumeConfiguration.

        Valid values: ENABLED | DISABLED

        :default: If this parameter is omitted, the default value of DISABLED is used.
        '''
        result = self._values.get("iam")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuthorizationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_ecs.AwsLogDriverMode")
class AwsLogDriverMode(enum.Enum):
    '''awslogs provides two modes for delivering messages from the container to the log driver.

    :exampleMetadata: infused

    Example::

        # cluster: ecs.Cluster
        
        # Create a Task Definition for the container to start
        task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
        task_definition.add_container("TheContainer",
            image=ecs.ContainerImage.from_asset(path.resolve(__dirname, "..", "eventhandler-image")),
            memory_limit_mi_b=256,
            logging=ecs.AwsLogDriver(stream_prefix="EventDemo", mode=ecs.AwsLogDriverMode.NON_BLOCKING)
        )
        
        # An Rule that describes the event trigger (in this case a scheduled run)
        rule = events.Rule(self, "Rule",
            schedule=events.Schedule.expression("rate(1 min)")
        )
        
        # Pass an environment variable to the container 'TheContainer' in the task
        rule.add_target(targets.EcsTask(
            cluster=cluster,
            task_definition=task_definition,
            task_count=1,
            container_overrides=[targets.ContainerOverride(
                container_name="TheContainer",
                environment=[targets.TaskEnvironmentVariable(
                    name="I_WAS_TRIGGERED",
                    value="From CloudWatch Events"
                )]
            )]
        ))
    '''

    BLOCKING = "BLOCKING"
    '''(default) direct, blocking delivery from container to driver.'''
    NON_BLOCKING = "NON_BLOCKING"
    '''The non-blocking message delivery mode prevents applications from blocking due to logging back pressure.

    Applications are likely to fail in unexpected ways when STDERR or STDOUT streams block.
    '''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.AwsLogDriverProps",
    jsii_struct_bases=[],
    name_mapping={
        "stream_prefix": "streamPrefix",
        "datetime_format": "datetimeFormat",
        "log_group": "logGroup",
        "log_retention": "logRetention",
        "mode": "mode",
        "multiline_pattern": "multilinePattern",
    },
)
class AwsLogDriverProps:
    def __init__(
        self,
        *,
        stream_prefix: builtins.str,
        datetime_format: typing.Optional[builtins.str] = None,
        log_group: typing.Optional[_ILogGroup_3c4fa718] = None,
        log_retention: typing.Optional[_RetentionDays_070f99f0] = None,
        mode: typing.Optional[AwsLogDriverMode] = None,
        multiline_pattern: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Specifies the awslogs log driver configuration options.

        :param stream_prefix: Prefix for the log streams. The awslogs-stream-prefix option allows you to associate a log stream with the specified prefix, the container name, and the ID of the Amazon ECS task to which the container belongs. If you specify a prefix with this option, then the log stream takes the following format:: prefix-name/container-name/ecs-task-id
        :param datetime_format: This option defines a multiline start pattern in Python strftime format. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. Default: - No multiline matching.
        :param log_group: The log group to log to. Default: - A log group is automatically created.
        :param log_retention: The number of days log events are kept in CloudWatch Logs when the log group is automatically created by this construct. Default: - Logs never expire.
        :param mode: The delivery mode of log messages from the container to awslogs. Default: - AwsLogDriverMode.BLOCKING
        :param multiline_pattern: This option defines a multiline start pattern using a regular expression. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. This option is ignored if datetimeFormat is also configured. Default: - No multiline matching.

        :exampleMetadata: infused

        Example::

            # Create a Task Definition for the Windows container to start
            task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
                runtime_platform=ecs.RuntimePlatform(
                    operating_system_family=ecs.OperatingSystemFamily.WINDOWS_SERVER_2019_CORE,
                    cpu_architecture=ecs.CpuArchitecture.X86_64
                ),
                cpu=1024,
                memory_limit_mi_b=2048
            )
            
            task_definition.add_container("windowsservercore",
                logging=ecs.LogDriver.aws_logs(stream_prefix="win-iis-on-fargate"),
                port_mappings=[ecs.PortMapping(container_port=80)],
                image=ecs.ContainerImage.from_registry("mcr.microsoft.com/windows/servercore/iis:windowsservercore-ltsc2019")
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__896fbbf9fdadbf59606c3a005d089549f308bf995f12c3ee79981c98b56fbd0d)
            check_type(argname="argument stream_prefix", value=stream_prefix, expected_type=type_hints["stream_prefix"])
            check_type(argname="argument datetime_format", value=datetime_format, expected_type=type_hints["datetime_format"])
            check_type(argname="argument log_group", value=log_group, expected_type=type_hints["log_group"])
            check_type(argname="argument log_retention", value=log_retention, expected_type=type_hints["log_retention"])
            check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
            check_type(argname="argument multiline_pattern", value=multiline_pattern, expected_type=type_hints["multiline_pattern"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "stream_prefix": stream_prefix,
        }
        if datetime_format is not None:
            self._values["datetime_format"] = datetime_format
        if log_group is not None:
            self._values["log_group"] = log_group
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if mode is not None:
            self._values["mode"] = mode
        if multiline_pattern is not None:
            self._values["multiline_pattern"] = multiline_pattern

    @builtins.property
    def stream_prefix(self) -> builtins.str:
        '''Prefix for the log streams.

        The awslogs-stream-prefix option allows you to associate a log stream
        with the specified prefix, the container name, and the ID of the Amazon
        ECS task to which the container belongs. If you specify a prefix with
        this option, then the log stream takes the following format::

           prefix-name/container-name/ecs-task-id
        '''
        result = self._values.get("stream_prefix")
        assert result is not None, "Required property 'stream_prefix' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def datetime_format(self) -> typing.Optional[builtins.str]:
        '''This option defines a multiline start pattern in Python strftime format.

        A log message consists of a line that matches the pattern and any
        following lines that don’t match the pattern. Thus the matched line is
        the delimiter between log messages.

        :default: - No multiline matching.
        '''
        result = self._values.get("datetime_format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_group(self) -> typing.Optional[_ILogGroup_3c4fa718]:
        '''The log group to log to.

        :default: - A log group is automatically created.
        '''
        result = self._values.get("log_group")
        return typing.cast(typing.Optional[_ILogGroup_3c4fa718], result)

    @builtins.property
    def log_retention(self) -> typing.Optional[_RetentionDays_070f99f0]:
        '''The number of days log events are kept in CloudWatch Logs when the log group is automatically created by this construct.

        :default: - Logs never expire.
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[_RetentionDays_070f99f0], result)

    @builtins.property
    def mode(self) -> typing.Optional[AwsLogDriverMode]:
        '''The delivery mode of log messages from the container to awslogs.

        :default: - AwsLogDriverMode.BLOCKING
        '''
        result = self._values.get("mode")
        return typing.cast(typing.Optional[AwsLogDriverMode], result)

    @builtins.property
    def multiline_pattern(self) -> typing.Optional[builtins.str]:
        '''This option defines a multiline start pattern using a regular expression.

        A log message consists of a line that matches the pattern and any
        following lines that don’t match the pattern. Thus the matched line is
        the delimiter between log messages.

        This option is ignored if datetimeFormat is also configured.

        :default: - No multiline matching.
        '''
        result = self._values.get("multiline_pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.BaseLogDriverProps",
    jsii_struct_bases=[],
    name_mapping={
        "env": "env",
        "env_regex": "envRegex",
        "labels": "labels",
        "tag": "tag",
    },
)
class BaseLogDriverProps:
    def __init__(
        self,
        *,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param env: The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            base_log_driver_props = ecs.BaseLogDriverProps(
                env=["env"],
                env_regex="envRegex",
                labels=["labels"],
                tag="tag"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__589c29654c9c7d30b6e91e3b5f5f3bd2302870b0e8f0f1008ecdf0d5017435c9)
            check_type(argname="argument env", value=env, expected_type=type_hints["env"])
            check_type(argname="argument env_regex", value=env_regex, expected_type=type_hints["env_regex"])
            check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
            check_type(argname="argument tag", value=tag, expected_type=type_hints["tag"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if env is not None:
            self._values["env"] = env
        if env_regex is not None:
            self._values["env_regex"] = env_regex
        if labels is not None:
            self._values["labels"] = labels
        if tag is not None:
            self._values["tag"] = tag

    @builtins.property
    def env(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The env option takes an array of keys.

        If there is collision between
        label and env keys, the value of the env takes precedence. Adds additional fields
        to the extra attributes of a logging message.

        :default: - No env
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def env_regex(self) -> typing.Optional[builtins.str]:
        '''The env-regex option is similar to and compatible with env.

        Its value is a regular
        expression to match logging-related environment variables. It is used for advanced
        log tag options.

        :default: - No envRegex
        '''
        result = self._values.get("env_regex")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The labels option takes an array of keys.

        If there is collision
        between label and env keys, the value of the env takes precedence. Adds additional
        fields to the extra attributes of a logging message.

        :default: - No labels
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''By default, Docker uses the first 12 characters of the container ID to tag log messages.

        Refer to the log tag option documentation for customizing the
        log tag format.

        :default: - The first 12 characters of the container ID
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.BaseServiceOptions",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "capacity_provider_strategies": "capacityProviderStrategies",
        "circuit_breaker": "circuitBreaker",
        "cloud_map_options": "cloudMapOptions",
        "deployment_alarms": "deploymentAlarms",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableECSManagedTags",
        "enable_execute_command": "enableExecuteCommand",
        "health_check_grace_period": "healthCheckGracePeriod",
        "max_healthy_percent": "maxHealthyPercent",
        "min_healthy_percent": "minHealthyPercent",
        "propagate_tags": "propagateTags",
        "service_connect_configuration": "serviceConnectConfiguration",
        "service_name": "serviceName",
    },
)
class BaseServiceOptions:
    def __init__(
        self,
        *,
        cluster: "ICluster",
        capacity_provider_strategies: typing.Optional[typing.Sequence[typing.Union["CapacityProviderStrategy", typing.Dict[builtins.str, typing.Any]]]] = None,
        circuit_breaker: typing.Optional[typing.Union["DeploymentCircuitBreaker", typing.Dict[builtins.str, typing.Any]]] = None,
        cloud_map_options: typing.Optional[typing.Union["CloudMapOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        deployment_alarms: typing.Optional[typing.Union["DeploymentAlarmConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        deployment_controller: typing.Optional[typing.Union["DeploymentController", typing.Dict[builtins.str, typing.Any]]] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[builtins.bool] = None,
        enable_execute_command: typing.Optional[builtins.bool] = None,
        health_check_grace_period: typing.Optional[_Duration_4839e8c3] = None,
        max_healthy_percent: typing.Optional[jsii.Number] = None,
        min_healthy_percent: typing.Optional[jsii.Number] = None,
        propagate_tags: typing.Optional["PropagatedTagSource"] = None,
        service_connect_configuration: typing.Optional[typing.Union["ServiceConnectProps", typing.Dict[builtins.str, typing.Any]]] = None,
        service_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The properties for the base Ec2Service or FargateService service.

        :param cluster: The name of the cluster that hosts the service.
        :param capacity_provider_strategies: A list of Capacity Provider strategies used to place a service. Default: - undefined
        :param circuit_breaker: Whether to enable the deployment circuit breaker. If this property is defined, circuit breaker will be implicitly enabled. Default: - disabled
        :param cloud_map_options: The options for configuring an Amazon ECS service to use service discovery. Default: - AWS Cloud Map service discovery is not enabled.
        :param deployment_alarms: The alarm(s) to monitor during deployment, and behavior to apply if at least one enters a state of alarm during the deployment or bake time. Default: - No alarms will be monitored during deployment.
        :param deployment_controller: Specifies which deployment controller to use for the service. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ Default: - Rolling update (ECS)
        :param desired_count: The desired number of instantiations of the task definition to keep running on the service. Default: - When creating the service, default is 1; when updating the service, default uses the current task number.
        :param enable_ecs_managed_tags: Specifies whether to enable Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ Default: false
        :param enable_execute_command: Whether to enable the ability to execute into a container. Default: - undefined
        :param health_check_grace_period: The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. Default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        :param max_healthy_percent: The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment. Default: - 100 if daemon, otherwise 200
        :param min_healthy_percent: The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment. Default: - 0 if daemon, otherwise 50
        :param propagate_tags: Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE Default: PropagatedTagSource.NONE
        :param service_connect_configuration: Configuration for Service Connect. Default: No ports are advertised via Service Connect on this service, and the service cannot make requests to other services via Service Connect.
        :param service_name: The name of the service. Default: - CloudFormation-generated name.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_ecs as ecs
            from aws_cdk import aws_servicediscovery as servicediscovery
            
            # cluster: ecs.Cluster
            # container_definition: ecs.ContainerDefinition
            # log_driver: ecs.LogDriver
            # namespace: servicediscovery.INamespace
            
            base_service_options = ecs.BaseServiceOptions(
                cluster=cluster,
            
                # the properties below are optional
                capacity_provider_strategies=[ecs.CapacityProviderStrategy(
                    capacity_provider="capacityProvider",
            
                    # the properties below are optional
                    base=123,
                    weight=123
                )],
                circuit_breaker=ecs.DeploymentCircuitBreaker(
                    rollback=False
                ),
                cloud_map_options=ecs.CloudMapOptions(
                    cloud_map_namespace=namespace,
                    container=container_definition,
                    container_port=123,
                    dns_record_type=servicediscovery.DnsRecordType.A,
                    dns_ttl=cdk.Duration.minutes(30),
                    failure_threshold=123,
                    name="name"
                ),
                deployment_alarms=ecs.DeploymentAlarmConfig(
                    alarm_names=["alarmNames"],
            
                    # the properties below are optional
                    behavior=ecs.AlarmBehavior.ROLLBACK_ON_ALARM
                ),
                deployment_controller=ecs.DeploymentController(
                    type=ecs.DeploymentControllerType.ECS
                ),
                desired_count=123,
                enable_eCSManaged_tags=False,
                enable_execute_command=False,
                health_check_grace_period=cdk.Duration.minutes(30),
                max_healthy_percent=123,
                min_healthy_percent=123,
                propagate_tags=ecs.PropagatedTagSource.SERVICE,
                service_connect_configuration=ecs.ServiceConnectProps(
                    log_driver=log_driver,
                    namespace="namespace",
                    services=[ecs.ServiceConnectService(
                        port_mapping_name="portMappingName",
            
                        # the properties below are optional
                        discovery_name="discoveryName",
                        dns_name="dnsName",
                        ingress_port_override=123,
                        port=123
                    )]
                ),
                service_name="serviceName"
            )
        '''
        if isinstance(circuit_breaker, dict):
            circuit_breaker = DeploymentCircuitBreaker(**circuit_breaker)
        if isinstance(cloud_map_options, dict):
            cloud_map_options = CloudMapOptions(**cloud_map_options)
        if isinstance(deployment_alarms, dict):
            deployment_alarms = DeploymentAlarmConfig(**deployment_alarms)
        if isinstance(deployment_controller, dict):
            deployment_controller = DeploymentController(**deployment_controller)
        if isinstance(service_connect_configuration, dict):
            service_connect_configuration = ServiceConnectProps(**service_connect_configuration)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c2e0ba28c74987301a54b0d197b791a6a94084b5f40d15304ffabf113b3f7daa)
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument capacity_provider_strategies", value=capacity_provider_strategies, expected_type=type_hints["capacity_provider_strategies"])
            check_type(argname="argument circuit_breaker", value=circuit_breaker, expected_type=type_hints["circuit_breaker"])
            check_type(argname="argument cloud_map_options", value=cloud_map_options, expected_type=type_hints["cloud_map_options"])
            check_type(argname="argument deployment_alarms", value=deployment_alarms, expected_type=type_hints["deployment_alarms"])
            check_type(argname="argument deployment_controller", value=deployment_controller, expected_type=type_hints["deployment_controller"])
            check_type(argname="argument desired_count", value=desired_count, expected_type=type_hints["desired_count"])
            check_type(argname="argument enable_ecs_managed_tags", value=enable_ecs_managed_tags, expected_type=type_hints["enable_ecs_managed_tags"])
            check_type(argname="argument enable_execute_command", value=enable_execute_command, expected_type=type_hints["enable_execute_command"])
            check_type(argname="argument health_check_grace_period", value=health_check_grace_period, expected_type=type_hints["health_check_grace_period"])
            check_type(argname="argument max_healthy_percent", value=max_healthy_percent, expected_type=type_hints["max_healthy_percent"])
            check_type(argname="argument min_healthy_percent", value=min_healthy_percent, expected_type=type_hints["min_healthy_percent"])
            check_type(argname="argument propagate_tags", value=propagate_tags, expected_type=type_hints["propagate_tags"])
            check_type(argname="argument service_connect_configuration", value=service_connect_configuration, expected_type=type_hints["service_connect_configuration"])
            check_type(argname="argument service_name", value=service_name, expected_type=type_hints["service_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
        }
        if capacity_provider_strategies is not None:
            self._values["capacity_provider_strategies"] = capacity_provider_strategies
        if circuit_breaker is not None:
            self._values["circuit_breaker"] = circuit_breaker
        if cloud_map_options is not None:
            self._values["cloud_map_options"] = cloud_map_options
        if deployment_alarms is not None:
            self._values["deployment_alarms"] = deployment_alarms
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if enable_execute_command is not None:
            self._values["enable_execute_command"] = enable_execute_command
        if health_check_grace_period is not None:
            self._values["health_check_grace_period"] = health_check_grace_period
        if max_healthy_percent is not None:
            self._values["max_healthy_percent"] = max_healthy_percent
        if min_healthy_percent is not None:
            self._values["min_healthy_percent"] = min_healthy_percent
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if service_connect_configuration is not None:
            self._values["service_connect_configuration"] = service_connect_configuration
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def cluster(self) -> "ICluster":
        '''The name of the cluster that hosts the service.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def capacity_provider_strategies(
        self,
    ) -> typing.Optional[typing.List["CapacityProviderStrategy"]]:
        '''A list of Capacity Provider strategies used to place a service.

        :default: - undefined
        '''
        result = self._values.get("capacity_provider_strategies")
        return typing.cast(typing.Optional[typing.List["CapacityProviderStrategy"]], result)

    @builtins.property
    def circuit_breaker(self) -> typing.Optional["DeploymentCircuitBreaker"]:
        '''Whether to enable the deployment circuit breaker.

        If this property is defined, circuit breaker will be implicitly
        enabled.

        :default: - disabled
        '''
        result = self._values.get("circuit_breaker")
        return typing.cast(typing.Optional["DeploymentCircuitBreaker"], result)

    @builtins.property
    def cloud_map_options(self) -> typing.Optional["CloudMapOptions"]:
        '''The options for configuring an Amazon ECS service to use service discovery.

        :default: - AWS Cloud Map service discovery is not enabled.
        '''
        result = self._values.get("cloud_map_options")
        return typing.cast(typing.Optional["CloudMapOptions"], result)

    @builtins.property
    def deployment_alarms(self) -> typing.Optional["DeploymentAlarmConfig"]:
        '''The alarm(s) to monitor during deployment, and behavior to apply if at least one enters a state of alarm during the deployment or bake time.

        :default: - No alarms will be monitored during deployment.
        '''
        result = self._values.get("deployment_alarms")
        return typing.cast(typing.Optional["DeploymentAlarmConfig"], result)

    @builtins.property
    def deployment_controller(self) -> typing.Optional["DeploymentController"]:
        '''Specifies which deployment controller to use for the service.

        For more information, see
        `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_

        :default: - Rolling update (ECS)
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional["DeploymentController"], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''The desired number of instantiations of the task definition to keep running on the service.

        :default:

        - When creating the service, default is 1; when updating the service, default uses
        the current task number.
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether to enable Amazon ECS managed tags for the tasks within the service.

        For more information, see
        `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_

        :default: false
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_execute_command(self) -> typing.Optional[builtins.bool]:
        '''Whether to enable the ability to execute into a container.

        :default: - undefined
        '''
        result = self._values.get("enable_execute_command")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def health_check_grace_period(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        :default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        '''
        result = self._values.get("health_check_grace_period")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def max_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment.

        :default: - 100 if daemon, otherwise 200
        '''
        result = self._values.get("max_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment.

        :default: - 0 if daemon, otherwise 50
        '''
        result = self._values.get("min_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional["PropagatedTagSource"]:
        '''Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE

        :default: PropagatedTagSource.NONE
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def service_connect_configuration(self) -> typing.Optional["ServiceConnectProps"]:
        '''Configuration for Service Connect.

        :default:

        No ports are advertised via Service Connect on this service, and the service
        cannot make requests to other services via Service Connect.
        '''
        result = self._values.get("service_connect_configuration")
        return typing.cast(typing.Optional["ServiceConnectProps"], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''The name of the service.

        :default: - CloudFormation-generated name.
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseServiceOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.BaseServiceProps",
    jsii_struct_bases=[BaseServiceOptions],
    name_mapping={
        "cluster": "cluster",
        "capacity_provider_strategies": "capacityProviderStrategies",
        "circuit_breaker": "circuitBreaker",
        "cloud_map_options": "cloudMapOptions",
        "deployment_alarms": "deploymentAlarms",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableECSManagedTags",
        "enable_execute_command": "enableExecuteCommand",
        "health_check_grace_period": "healthCheckGracePeriod",
        "max_healthy_percent": "maxHealthyPercent",
        "min_healthy_percent": "minHealthyPercent",
        "propagate_tags": "propagateTags",
        "service_connect_configuration": "serviceConnectConfiguration",
        "service_name": "serviceName",
        "launch_type": "launchType",
    },
)
class BaseServiceProps(BaseServiceOptions):
    def __init__(
        self,
        *,
        cluster: "ICluster",
        capacity_provider_strategies: typing.Optional[typing.Sequence[typing.Union["CapacityProviderStrategy", typing.Dict[builtins.str, typing.Any]]]] = None,
        circuit_breaker: typing.Optional[typing.Union["DeploymentCircuitBreaker", typing.Dict[builtins.str, typing.Any]]] = None,
        cloud_map_options: typing.Optional[typing.Union["CloudMapOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        deployment_alarms: typing.Optional[typing.Union["DeploymentAlarmConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        deployment_controller: typing.Optional[typing.Union["DeploymentController", typing.Dict[builtins.str, typing.Any]]] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[builtins.bool] = None,
        enable_execute_command: typing.Optional[builtins.bool] = None,
        health_check_grace_period: typing.Optional[_Duration_4839e8c3] = None,
        max_healthy_percent: typing.Optional[jsii.Number] = None,
        min_healthy_percent: typing.Optional[jsii.Number] = None,
        propagate_tags: typing.Optional["PropagatedTagSource"] = None,
        service_connect_configuration: typing.Optional[typing.Union["ServiceConnectProps", typing.Dict[builtins.str, typing.Any]]] = None,
        service_name: typing.Optional[builtins.str] = None,
        launch_type: "LaunchType",
    ) -> None:
        '''Complete base service properties that are required to be supplied by the implementation of the BaseService class.

        :param cluster: The name of the cluster that hosts the service.
        :param capacity_provider_strategies: A list of Capacity Provider strategies used to place a service. Default: - undefined
        :param circuit_breaker: Whether to enable the deployment circuit breaker. If this property is defined, circuit breaker will be implicitly enabled. Default: - disabled
        :param cloud_map_options: The options for configuring an Amazon ECS service to use service discovery. Default: - AWS Cloud Map service discovery is not enabled.
        :param deployment_alarms: The alarm(s) to monitor during deployment, and behavior to apply if at least one enters a state of alarm during the deployment or bake time. Default: - No alarms will be monitored during deployment.
        :param deployment_controller: Specifies which deployment controller to use for the service. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ Default: - Rolling update (ECS)
        :param desired_count: The desired number of instantiations of the task definition to keep running on the service. Default: - When creating the service, default is 1; when updating the service, default uses the current task number.
        :param enable_ecs_managed_tags: Specifies whether to enable Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ Default: false
        :param enable_execute_command: Whether to enable the ability to execute into a container. Default: - undefined
        :param health_check_grace_period: The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. Default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        :param max_healthy_percent: The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment. Default: - 100 if daemon, otherwise 200
        :param min_healthy_percent: The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment. Default: - 0 if daemon, otherwise 50
        :param propagate_tags: Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE Default: PropagatedTagSource.NONE
        :param service_connect_configuration: Configuration for Service Connect. Default: No ports are advertised via Service Connect on this service, and the service cannot make requests to other services via Service Connect.
        :param service_name: The name of the service. Default: - CloudFormation-generated name.
        :param launch_type: The launch type on which to run your service. LaunchType will be omitted if capacity provider strategies are specified on the service.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_ecs as ecs
            from aws_cdk import aws_servicediscovery as servicediscovery
            
            # cluster: ecs.Cluster
            # container_definition: ecs.ContainerDefinition
            # log_driver: ecs.LogDriver
            # namespace: servicediscovery.INamespace
            
            base_service_props = ecs.BaseServiceProps(
                cluster=cluster,
                launch_type=ecs.LaunchType.EC2,
            
                # the properties below are optional
                capacity_provider_strategies=[ecs.CapacityProviderStrategy(
                    capacity_provider="capacityProvider",
            
                    # the properties below are optional
                    base=123,
                    weight=123
                )],
                circuit_breaker=ecs.DeploymentCircuitBreaker(
                    rollback=False
                ),
                cloud_map_options=ecs.CloudMapOptions(
                    cloud_map_namespace=namespace,
                    container=container_definition,
                    container_port=123,
                    dns_record_type=servicediscovery.DnsRecordType.A,
                    dns_ttl=cdk.Duration.minutes(30),
                    failure_threshold=123,
                    name="name"
                ),
                deployment_alarms=ecs.DeploymentAlarmConfig(
                    alarm_names=["alarmNames"],
            
                    # the properties below are optional
                    behavior=ecs.AlarmBehavior.ROLLBACK_ON_ALARM
                ),
                deployment_controller=ecs.DeploymentController(
                    type=ecs.DeploymentControllerType.ECS
                ),
                desired_count=123,
                enable_eCSManaged_tags=False,
                enable_execute_command=False,
                health_check_grace_period=cdk.Duration.minutes(30),
                max_healthy_percent=123,
                min_healthy_percent=123,
                propagate_tags=ecs.PropagatedTagSource.SERVICE,
                service_connect_configuration=ecs.ServiceConnectProps(
                    log_driver=log_driver,
                    namespace="namespace",
                    services=[ecs.ServiceConnectService(
                        port_mapping_name="portMappingName",
            
                        # the properties below are optional
                        discovery_name="discoveryName",
                        dns_name="dnsName",
                        ingress_port_override=123,
                        port=123
                    )]
                ),
                service_name="serviceName"
            )
        '''
        if isinstance(circuit_breaker, dict):
            circuit_breaker = DeploymentCircuitBreaker(**circuit_breaker)
        if isinstance(cloud_map_options, dict):
            cloud_map_options = CloudMapOptions(**cloud_map_options)
        if isinstance(deployment_alarms, dict):
            deployment_alarms = DeploymentAlarmConfig(**deployment_alarms)
        if isinstance(deployment_controller, dict):
            deployment_controller = DeploymentController(**deployment_controller)
        if isinstance(service_connect_configuration, dict):
            service_connect_configuration = ServiceConnectProps(**service_connect_configuration)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ecfd95265b873c2042a9d5cb8465a48f9e325e2271c18461e2b266333563d84)
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument capacity_provider_strategies", value=capacity_provider_strategies, expected_type=type_hints["capacity_provider_strategies"])
            check_type(argname="argument circuit_breaker", value=circuit_breaker, expected_type=type_hints["circuit_breaker"])
            check_type(argname="argument cloud_map_options", value=cloud_map_options, expected_type=type_hints["cloud_map_options"])
            check_type(argname="argument deployment_alarms", value=deployment_alarms, expected_type=type_hints["deployment_alarms"])
            check_type(argname="argument deployment_controller", value=deployment_controller, expected_type=type_hints["deployment_controller"])
            check_type(argname="argument desired_count", value=desired_count, expected_type=type_hints["desired_count"])
            check_type(argname="argument enable_ecs_managed_tags", value=enable_ecs_managed_tags, expected_type=type_hints["enable_ecs_managed_tags"])
            check_type(argname="argument enable_execute_command", value=enable_execute_command, expected_type=type_hints["enable_execute_command"])
            check_type(argname="argument health_check_grace_period", value=health_check_grace_period, expected_type=type_hints["health_check_grace_period"])
            check_type(argname="argument max_healthy_percent", value=max_healthy_percent, expected_type=type_hints["max_healthy_percent"])
            check_type(argname="argument min_healthy_percent", value=min_healthy_percent, expected_type=type_hints["min_healthy_percent"])
            check_type(argname="argument propagate_tags", value=propagate_tags, expected_type=type_hints["propagate_tags"])
            check_type(argname="argument service_connect_configuration", value=service_connect_configuration, expected_type=type_hints["service_connect_configuration"])
            check_type(argname="argument service_name", value=service_name, expected_type=type_hints["service_name"])
            check_type(argname="argument launch_type", value=launch_type, expected_type=type_hints["launch_type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
            "launch_type": launch_type,
        }
        if capacity_provider_strategies is not None:
            self._values["capacity_provider_strategies"] = capacity_provider_strategies
        if circuit_breaker is not None:
            self._values["circuit_breaker"] = circuit_breaker
        if cloud_map_options is not None:
            self._values["cloud_map_options"] = cloud_map_options
        if deployment_alarms is not None:
            self._values["deployment_alarms"] = deployment_alarms
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if enable_execute_command is not None:
            self._values["enable_execute_command"] = enable_execute_command
        if health_check_grace_period is not None:
            self._values["health_check_grace_period"] = health_check_grace_period
        if max_healthy_percent is not None:
            self._values["max_healthy_percent"] = max_healthy_percent
        if min_healthy_percent is not None:
            self._values["min_healthy_percent"] = min_healthy_percent
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if service_connect_configuration is not None:
            self._values["service_connect_configuration"] = service_connect_configuration
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def cluster(self) -> "ICluster":
        '''The name of the cluster that hosts the service.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def capacity_provider_strategies(
        self,
    ) -> typing.Optional[typing.List["CapacityProviderStrategy"]]:
        '''A list of Capacity Provider strategies used to place a service.

        :default: - undefined
        '''
        result = self._values.get("capacity_provider_strategies")
        return typing.cast(typing.Optional[typing.List["CapacityProviderStrategy"]], result)

    @builtins.property
    def circuit_breaker(self) -> typing.Optional["DeploymentCircuitBreaker"]:
        '''Whether to enable the deployment circuit breaker.

        If this property is defined, circuit breaker will be implicitly
        enabled.

        :default: - disabled
        '''
        result = self._values.get("circuit_breaker")
        return typing.cast(typing.Optional["DeploymentCircuitBreaker"], result)

    @builtins.property
    def cloud_map_options(self) -> typing.Optional["CloudMapOptions"]:
        '''The options for configuring an Amazon ECS service to use service discovery.

        :default: - AWS Cloud Map service discovery is not enabled.
        '''
        result = self._values.get("cloud_map_options")
        return typing.cast(typing.Optional["CloudMapOptions"], result)

    @builtins.property
    def deployment_alarms(self) -> typing.Optional["DeploymentAlarmConfig"]:
        '''The alarm(s) to monitor during deployment, and behavior to apply if at least one enters a state of alarm during the deployment or bake time.

        :default: - No alarms will be monitored during deployment.
        '''
        result = self._values.get("deployment_alarms")
        return typing.cast(typing.Optional["DeploymentAlarmConfig"], result)

    @builtins.property
    def deployment_controller(self) -> typing.Optional["DeploymentController"]:
        '''Specifies which deployment controller to use for the service.

        For more information, see
        `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_

        :default: - Rolling update (ECS)
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional["DeploymentController"], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''The desired number of instantiations of the task definition to keep running on the service.

        :default:

        - When creating the service, default is 1; when updating the service, default uses
        the current task number.
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether to enable Amazon ECS managed tags for the tasks within the service.

        For more information, see
        `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_

        :default: false
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_execute_command(self) -> typing.Optional[builtins.bool]:
        '''Whether to enable the ability to execute into a container.

        :default: - undefined
        '''
        result = self._values.get("enable_execute_command")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def health_check_grace_period(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        :default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        '''
        result = self._values.get("health_check_grace_period")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def max_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment.

        :default: - 100 if daemon, otherwise 200
        '''
        result = self._values.get("max_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment.

        :default: - 0 if daemon, otherwise 50
        '''
        result = self._values.get("min_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional["PropagatedTagSource"]:
        '''Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE

        :default: PropagatedTagSource.NONE
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def service_connect_configuration(self) -> typing.Optional["ServiceConnectProps"]:
        '''Configuration for Service Connect.

        :default:

        No ports are advertised via Service Connect on this service, and the service
        cannot make requests to other services via Service Connect.
        '''
        result = self._values.get("service_connect_configuration")
        return typing.cast(typing.Optional["ServiceConnectProps"], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''The name of the service.

        :default: - CloudFormation-generated name.
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_type(self) -> "LaunchType":
        '''The launch type on which to run your service.

        LaunchType will be omitted if capacity provider strategies are specified on the service.

        :see:

        - https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-capacityproviderstrategy

        Valid values are: LaunchType.ECS or LaunchType.FARGATE or LaunchType.EXTERNAL
        '''
        result = self._values.get("launch_type")
        assert result is not None, "Required property 'launch_type' is missing"
        return typing.cast("LaunchType", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_ecs.BinPackResource")
class BinPackResource(enum.Enum):
    '''Instance resource used for bin packing.'''

    CPU = "CPU"
    '''Fill up hosts' CPU allocations first.'''
    MEMORY = "MEMORY"
    '''Fill up hosts' memory allocations first.'''


@jsii.implements(_IMachineImage_0e8bd50b)
class BottleRocketImage(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ecs.BottleRocketImage",
):
    '''Construct an Bottlerocket image from the latest AMI published in SSM.

    :exampleMetadata: infused

    Example::

        # cluster: ecs.Cluster
        
        
        cluster.add_capacity("bottlerocket-asg",
            min_capacity=2,
            instance_type=ec2.InstanceType("c5.large"),
            machine_image=ecs.BottleRocketImage()
        )
    '''

    def __init__(
        self,
        *,
        architecture: typing.Optional[_InstanceArchitecture_7721cb36] = None,
        cached_in_context: typing.Optional[builtins.bool] = None,
        variant: typing.Optional["BottlerocketEcsVariant"] = None,
    ) -> None:
        '''Constructs a new instance of the BottleRocketImage class.

        :param architecture: The CPU architecture. Default: - x86_64
        :param cached_in_context: Whether the AMI ID is cached to be stable between deployments. By default, the newest image is used on each deployment. This will cause instances to be replaced whenever a new version is released, and may cause downtime if there aren't enough running instances in the AutoScalingGroup to reschedule the tasks on. If set to true, the AMI ID will be cached in ``cdk.context.json`` and the same value will be used on future runs. Your instances will not be replaced but your AMI version will grow old over time. To refresh the AMI lookup, you will have to evict the value from the cache using the ``cdk context`` command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information. Can not be set to ``true`` in environment-agnostic stacks. Default: false
        :param variant: The Amazon ECS variant to use. Only ``aws-ecs-1`` is currently available Default: - BottlerocketEcsVariant.AWS_ECS_1
        '''
        props = BottleRocketImageProps(
            architecture=architecture,
            cached_in_context=cached_in_context,
            variant=variant,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="isBottleRocketImage")
    @builtins.classmethod
    def is_bottle_rocket_image(cls, x: typing.Any) -> builtins.bool:
        '''Return whether the given object is a BottleRocketImage.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d653cea8659e47876050d4fcb46a5798071ff2b2bae786385697e7f032f8d45e)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isBottleRocketImage", [x]))

    @jsii.member(jsii_name="getImage")
    def get_image(
        self,
        scope: _constructs_77d1e7e8.Construct,
    ) -> _MachineImageConfig_187edaee:
        '''Return the correct image.

        :param scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df65d4c374c4639217ff804e4aa42cc012373c9a3f00da061302e69d73c5f4eb)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast(_MachineImageConfig_187edaee, jsii.invoke(self, "getImage", [scope]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.BottleRocketImageProps",
    jsii_struct_bases=[],
    name_mapping={
        "architecture": "architecture",
        "cached_in_context": "cachedInContext",
        "variant": "variant",
    },
)
class BottleRocketImageProps:
    def __init__(
        self,
        *,
        architecture: typing.Optional[_InstanceArchitecture_7721cb36] = None,
        cached_in_context: typing.Optional[builtins.bool] = None,
        variant: typing.Optional["BottlerocketEcsVariant"] = None,
    ) -> None:
        '''Properties for BottleRocketImage.

        :param architecture: The CPU architecture. Default: - x86_64
        :param cached_in_context: Whether the AMI ID is cached to be stable between deployments. By default, the newest image is used on each deployment. This will cause instances to be replaced whenever a new version is released, and may cause downtime if there aren't enough running instances in the AutoScalingGroup to reschedule the tasks on. If set to true, the AMI ID will be cached in ``cdk.context.json`` and the same value will be used on future runs. Your instances will not be replaced but your AMI version will grow old over time. To refresh the AMI lookup, you will have to evict the value from the cache using the ``cdk context`` command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information. Can not be set to ``true`` in environment-agnostic stacks. Default: false
        :param variant: The Amazon ECS variant to use. Only ``aws-ecs-1`` is currently available Default: - BottlerocketEcsVariant.AWS_ECS_1

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_ecs as ecs
            
            bottle_rocket_image_props = ecs.BottleRocketImageProps(
                architecture=ec2.InstanceArchitecture.ARM_64,
                cached_in_context=False,
                variant=ecs.BottlerocketEcsVariant.AWS_ECS_1
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1888954247daafe3322b2fa144458be6e77f5b9d04b3406207497f8b279b2cf6)
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
            check_type(argname="argument cached_in_context", value=cached_in_context, expected_type=type_hints["cached_in_context"])
            check_type(argname="argument variant", value=variant, expected_type=type_hints["variant"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if architecture is not None:
            self._values["architecture"] = architecture
        if cached_in_context is not None:
            self._values["cached_in_context"] = cached_in_context
        if variant is not None:
            self._values["variant"] = variant

    @builtins.property
    def architecture(self) -> typing.Optional[_InstanceArchitecture_7721cb36]:
        '''The CPU architecture.

        :default: - x86_64
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[_InstanceArchitecture_7721cb36], result)

    @builtins.property
    def cached_in_context(self) -> typing.Optional[builtins.bool]:
        '''Whether the AMI ID is cached to be stable between deployments.

        By default, the newest image is used on each deployment. This will cause
        instances to be replaced whenever a new version is released, and may cause
        downtime if there aren't enough running instances in the AutoScalingGroup
        to reschedule the tasks on.

        If set to true, the AMI ID will be cached in ``cdk.context.json`` and the
        same value will be used on future runs. Your instances will not be replaced
        but your AMI version will grow old over time. To refresh the AMI lookup,
        you will have to evict the value from the cache using the ``cdk context``
        command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for
        more information.

        Can not be set to ``true`` in environment-agnostic stacks.

        :default: false
        '''
        result = self._values.get("cached_in_context")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def variant(self) -> typing.Optional["BottlerocketEcsVariant"]:
        '''The Amazon ECS variant to use.

        Only ``aws-ecs-1`` is currently available

        :default: - BottlerocketEcsVariant.AWS_ECS_1
        '''
        result = self._values.get("variant")
        return typing.cast(typing.Optional["BottlerocketEcsVariant"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BottleRocketImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_ecs.BottlerocketEcsVariant")
class BottlerocketEcsVariant(enum.Enum):
    '''Amazon ECS variant.'''

    AWS_ECS_1 = "AWS_ECS_1"
    '''aws-ecs-1 variant.'''


class BuiltInAttributes(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ecs.BuiltInAttributes",
):
    '''The built-in container instance attributes.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ecs as ecs
        
        built_in_attributes = ecs.BuiltInAttributes()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMI_ID")
    def AMI_ID(cls) -> builtins.str:
        '''The AMI id the instance is using.'''
        return typing.cast(builtins.str, jsii.sget(cls, "AMI_ID"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AVAILABILITY_ZONE")
    def AVAILABILITY_ZONE(cls) -> builtins.str:
        '''The AvailabilityZone where the instance is running in.'''
        return typing.cast(builtins.str, jsii.sget(cls, "AVAILABILITY_ZONE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="INSTANCE_ID")
    def INSTANCE_ID(cls) -> builtins.str:
        '''The id of the instance.'''
        return typing.cast(builtins.str, jsii.sget(cls, "INSTANCE_ID"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="INSTANCE_TYPE")
    def INSTANCE_TYPE(cls) -> builtins.str:
        '''The EC2 instance type.'''
        return typing.cast(builtins.str, jsii.sget(cls, "INSTANCE_TYPE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="OS_TYPE")
    def OS_TYPE(cls) -> builtins.str:
        '''The operating system of the instance.

        Either 'linux' or 'windows'.
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "OS_TYPE"))


@jsii.enum(jsii_type="aws-cdk-lib.aws_ecs.Capability")
class Capability(enum.Enum):
    '''A Linux capability.'''

    ALL = "ALL"
    AUDIT_CONTROL = "AUDIT_CONTROL"
    AUDIT_WRITE = "AUDIT_WRITE"
    BLOCK_SUSPEND = "BLOCK_SUSPEND"
    CHOWN = "CHOWN"
    DAC_OVERRIDE = "DAC_OVERRIDE"
    DAC_READ_SEARCH = "DAC_READ_SEARCH"
    FOWNER = "FOWNER"
    FSETID = "FSETID"
    IPC_LOCK = "IPC_LOCK"
    IPC_OWNER = "IPC_OWNER"
    KILL = "KILL"
    LEASE = "LEASE"
    LINUX_IMMUTABLE = "LINUX_IMMUTABLE"
    MAC_ADMIN = "MAC_ADMIN"
    MAC_OVERRIDE = "MAC_OVERRIDE"
    MKNOD = "MKNOD"
    NET_ADMIN = "NET_ADMIN"
    NET_BIND_SERVICE = "NET_BIND_SERVICE"
    NET_BROADCAST = "NET_BROADCAST"
    NET_RAW = "NET_RAW"
    SETFCAP = "SETFCAP"
    SETGID = "SETGID"
    SETPCAP = "SETPCAP"
    SETUID = "SETUID"
    SYS_ADMIN = "SYS_ADMIN"
    SYS_BOOT = "SYS_BOOT"
    SYS_CHROOT = "SYS_CHROOT"
    SYS_MODULE = "SYS_MODULE"
    SYS_NICE = "SYS_NICE"
    SYS_PACCT = "SYS_PACCT"
    SYS_PTRACE = "SYS_PTRACE"
    SYS_RAWIO = "SYS_RAWIO"
    SYS_RESOURCE = "SYS_RESOURCE"
    SYS_TIME = "SYS_TIME"
    SYS_TTY_CONFIG = "SYS_TTY_CONFIG"
    SYSLOG = "SYSLOG"
    WAKE_ALARM = "WAKE_ALARM"


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.CapacityProviderStrategy",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_provider": "capacityProvider",
        "base": "base",
        "weight": "weight",
    },
)
class CapacityProviderStrategy:
    def __init__(
        self,
        *,
        capacity_provider: builtins.str,
        base: typing.Optional[jsii.Number] = None,
        weight: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''A Capacity Provider strategy to use for the service.

        :param capacity_provider: The name of the capacity provider.
        :param base: The base value designates how many tasks, at a minimum, to run on the specified capacity provider. Only one capacity provider in a capacity provider strategy can have a base defined. If no value is specified, the default value of 0 is used. Default: - none
        :param weight: The weight value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider. The weight value is taken into consideration after the base value, if defined, is satisfied. Default: - 0

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            capacity_provider_strategy = ecs.CapacityProviderStrategy(
                capacity_provider="capacityProvider",
            
                # the properties below are optional
                base=123,
                weight=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__481a7064afb879135819febdd572a91fb6799e4bfadf66c711f640b51c07d027)
            check_type(argname="argument capacity_provider", value=capacity_provider, expected_type=type_hints["capacity_provider"])
            check_type(argname="argument base", value=base, expected_type=type_hints["base"])
            check_type(argname="argument weight", value=weight, expected_type=type_hints["weight"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "capacity_provider": capacity_provider,
        }
        if base is not None:
            self._values["base"] = base
        if weight is not None:
            self._values["weight"] = weight

    @builtins.property
    def capacity_provider(self) -> builtins.str:
        '''The name of the capacity provider.'''
        result = self._values.get("capacity_provider")
        assert result is not None, "Required property 'capacity_provider' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def base(self) -> typing.Optional[jsii.Number]:
        '''The base value designates how many tasks, at a minimum, to run on the specified capacity provider.

        Only one
        capacity provider in a capacity provider strategy can have a base defined. If no value is specified, the default
        value of 0 is used.

        :default: - none
        '''
        result = self._values.get("base")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def weight(self) -> typing.Optional[jsii.Number]:
        '''The weight value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.

        The weight value is taken into consideration after the base value, if defined, is satisfied.

        :default: - 0
        '''
        result = self._values.get("weight")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CapacityProviderStrategy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnCapacityProvider(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ecs.CfnCapacityProvider",
):
    '''A CloudFormation ``AWS::ECS::CapacityProvider``.

    Creates a new capacity provider. Capacity providers are associated with an Amazon ECS cluster and are used in capacity provider strategies to facilitate cluster auto scaling.

    Only capacity providers that use an Auto Scaling group can be created. Amazon ECS tasks on AWS Fargate use the ``FARGATE`` and ``FARGATE_SPOT`` capacity providers. These providers are available to all accounts in the AWS Regions that AWS Fargate supports.

    :cloudformationResource: AWS::ECS::CapacityProvider
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ecs as ecs
        
        cfn_capacity_provider = ecs.CfnCapacityProvider(self, "MyCfnCapacityProvider",
            auto_scaling_group_provider=ecs.CfnCapacityProvider.AutoScalingGroupProviderProperty(
                auto_scaling_group_arn="autoScalingGroupArn",
        
                # the properties below are optional
                managed_scaling=ecs.CfnCapacityProvider.ManagedScalingProperty(
                    instance_warmup_period=123,
                    maximum_scaling_step_size=123,
                    minimum_scaling_step_size=123,
                    status="status",
                    target_capacity=123
                ),
                managed_termination_protection="managedTerminationProtection"
            ),
        
            # the properties below are optional
            name="name",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auto_scaling_group_provider: typing.Union[_IResolvable_da3f097b, typing.Union["CfnCapacityProvider.AutoScalingGroupProviderProperty", typing.Dict[builtins.str, typing.Any]]],
        name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::CapacityProvider``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auto_scaling_group_provider: The Auto Scaling group settings for the capacity provider.
        :param name: The name of the capacity provider. If a name is specified, it cannot start with ``aws`` , ``ecs`` , or ``fargate`` . If no name is specified, a default name in the ``CFNStackName-CFNResourceName-RandomString`` format is used.
        :param tags: The metadata that you apply to the capacity provider to help you categorize and organize it. Each tag consists of a key and an optional value. You define both. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__59a913caee739f6d41600bf8ae89985db638913fbcb77a8abd5451cda3893b56)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCapacityProviderProps(
            auto_scaling_group_provider=auto_scaling_group_provider,
            name=name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__53c0f84221516e5285bbcd4c1565c1b5af2df9c65da6dfc915f76bb224443013)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8ee137a83ca18edf089a21cbcaba0156daf4f4612dbd8c0aeac582bcf968fdd5)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''The metadata that you apply to the capacity provider to help you categorize and organize it.

        Each tag consists of a key and an optional value. You define both.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupProvider")
    def auto_scaling_group_provider(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnCapacityProvider.AutoScalingGroupProviderProperty"]:
        '''The Auto Scaling group settings for the capacity provider.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider
        '''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnCapacityProvider.AutoScalingGroupProviderProperty"], jsii.get(self, "autoScalingGroupProvider"))

    @auto_scaling_group_provider.setter
    def auto_scaling_group_provider(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnCapacityProvider.AutoScalingGroupProviderProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5888da07adc4050987d977b4699983a6760a2abcd538f800018e65953cb2746e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoScalingGroupProvider", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the capacity provider.

        If a name is specified, it cannot start with ``aws`` , ``ecs`` , or ``fargate`` . If no name is specified, a default name in the ``CFNStackName-CFNResourceName-RandomString`` format is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-name
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6b097d35daad25c1c594150aba1acbb2fe2a7052d42d247db35563ec5fed6bd4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnCapacityProvider.AutoScalingGroupProviderProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auto_scaling_group_arn": "autoScalingGroupArn",
            "managed_scaling": "managedScaling",
            "managed_termination_protection": "managedTerminationProtection",
        },
    )
    class AutoScalingGroupProviderProperty:
        def __init__(
            self,
            *,
            auto_scaling_group_arn: builtins.str,
            managed_scaling: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCapacityProvider.ManagedScalingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            managed_termination_protection: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The details of the Auto Scaling group for the capacity provider.

            :param auto_scaling_group_arn: The Amazon Resource Name (ARN) that identifies the Auto Scaling group.
            :param managed_scaling: The managed scaling settings for the Auto Scaling group capacity provider.
            :param managed_termination_protection: The managed termination protection setting to use for the Auto Scaling group capacity provider. This determines whether the Auto Scaling group has managed termination protection. The default is off. .. epigraph:: When using managed termination protection, managed scaling must also be used otherwise managed termination protection doesn't work. When managed termination protection is on, Amazon ECS prevents the Amazon EC2 instances in an Auto Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling group and each instance in the Auto Scaling group must have instance protection from scale-in actions on as well. For more information, see `Instance Protection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection>`_ in the *AWS Auto Scaling User Guide* . When managed termination protection is off, your Amazon EC2 instances aren't protected from termination when the Auto Scaling group scales in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                auto_scaling_group_provider_property = ecs.CfnCapacityProvider.AutoScalingGroupProviderProperty(
                    auto_scaling_group_arn="autoScalingGroupArn",
                
                    # the properties below are optional
                    managed_scaling=ecs.CfnCapacityProvider.ManagedScalingProperty(
                        instance_warmup_period=123,
                        maximum_scaling_step_size=123,
                        minimum_scaling_step_size=123,
                        status="status",
                        target_capacity=123
                    ),
                    managed_termination_protection="managedTerminationProtection"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ca441075e92a847965e776db4f5ab9f545ce368b5efa4bc2476d58061dd0b742)
                check_type(argname="argument auto_scaling_group_arn", value=auto_scaling_group_arn, expected_type=type_hints["auto_scaling_group_arn"])
                check_type(argname="argument managed_scaling", value=managed_scaling, expected_type=type_hints["managed_scaling"])
                check_type(argname="argument managed_termination_protection", value=managed_termination_protection, expected_type=type_hints["managed_termination_protection"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "auto_scaling_group_arn": auto_scaling_group_arn,
            }
            if managed_scaling is not None:
                self._values["managed_scaling"] = managed_scaling
            if managed_termination_protection is not None:
                self._values["managed_termination_protection"] = managed_termination_protection

        @builtins.property
        def auto_scaling_group_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) that identifies the Auto Scaling group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider-autoscalinggrouparn
            '''
            result = self._values.get("auto_scaling_group_arn")
            assert result is not None, "Required property 'auto_scaling_group_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def managed_scaling(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCapacityProvider.ManagedScalingProperty"]]:
            '''The managed scaling settings for the Auto Scaling group capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider-managedscaling
            '''
            result = self._values.get("managed_scaling")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCapacityProvider.ManagedScalingProperty"]], result)

        @builtins.property
        def managed_termination_protection(self) -> typing.Optional[builtins.str]:
            '''The managed termination protection setting to use for the Auto Scaling group capacity provider.

            This determines whether the Auto Scaling group has managed termination protection. The default is off.
            .. epigraph::

               When using managed termination protection, managed scaling must also be used otherwise managed termination protection doesn't work.

            When managed termination protection is on, Amazon ECS prevents the Amazon EC2 instances in an Auto Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling group and each instance in the Auto Scaling group must have instance protection from scale-in actions on as well. For more information, see `Instance Protection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection>`_ in the *AWS Auto Scaling User Guide* .

            When managed termination protection is off, your Amazon EC2 instances aren't protected from termination when the Auto Scaling group scales in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider-managedterminationprotection
            '''
            result = self._values.get("managed_termination_protection")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AutoScalingGroupProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnCapacityProvider.ManagedScalingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_warmup_period": "instanceWarmupPeriod",
            "maximum_scaling_step_size": "maximumScalingStepSize",
            "minimum_scaling_step_size": "minimumScalingStepSize",
            "status": "status",
            "target_capacity": "targetCapacity",
        },
    )
    class ManagedScalingProperty:
        def __init__(
            self,
            *,
            instance_warmup_period: typing.Optional[jsii.Number] = None,
            maximum_scaling_step_size: typing.Optional[jsii.Number] = None,
            minimum_scaling_step_size: typing.Optional[jsii.Number] = None,
            status: typing.Optional[builtins.str] = None,
            target_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The managed scaling settings for the Auto Scaling group capacity provider.

            When managed scaling is turned on, Amazon ECS manages the scale-in and scale-out actions of the Auto Scaling group. Amazon ECS manages a target tracking scaling policy using an Amazon ECS managed CloudWatch metric with the specified ``targetCapacity`` value as the target value for the metric. For more information, see `Using managed scaling <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html#asg-capacity-providers-managed-scaling>`_ in the *Amazon Elastic Container Service Developer Guide* .

            If managed scaling is off, the user must manage the scaling of the Auto Scaling group.

            :param instance_warmup_period: The period of time, in seconds, after a newly launched Amazon EC2 instance can contribute to CloudWatch metrics for Auto Scaling group. If this parameter is omitted, the default value of ``300`` seconds is used.
            :param maximum_scaling_step_size: The maximum number of Amazon EC2 instances that Amazon ECS will scale out at one time. The scale in process is not affected by this parameter. If this parameter is omitted, the default value of ``1`` is used.
            :param minimum_scaling_step_size: The minimum number of Amazon EC2 instances that Amazon ECS will scale out at one time. The scale in process is not affected by this parameter If this parameter is omitted, the default value of ``1`` is used. When additional capacity is required, Amazon ECS will scale up the minimum scaling step size even if the actual demand is less than the minimum scaling step size. If you use a capacity provider with an Auto Scaling group configured with more than one Amazon EC2 instance type or Availability Zone, Amazon ECS will scale up by the exact minimum scaling step size value and will ignore both the maximum scaling step size as well as the capacity demand.
            :param status: Determines whether to use managed scaling for the capacity provider.
            :param target_capacity: The target capacity utilization as a percentage for the capacity provider. The specified value must be greater than ``0`` and less than or equal to ``100`` . For example, if you want the capacity provider to maintain 10% spare capacity, then that means the utilization is 90%, so use a ``targetCapacity`` of ``90`` . The default value of ``100`` percent results in the Amazon EC2 instances in your Auto Scaling group being completely used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                managed_scaling_property = ecs.CfnCapacityProvider.ManagedScalingProperty(
                    instance_warmup_period=123,
                    maximum_scaling_step_size=123,
                    minimum_scaling_step_size=123,
                    status="status",
                    target_capacity=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b6ba7d15ac2121ea6e10f20fe3e2441ae17d00f30897db384f8f72e7d6406865)
                check_type(argname="argument instance_warmup_period", value=instance_warmup_period, expected_type=type_hints["instance_warmup_period"])
                check_type(argname="argument maximum_scaling_step_size", value=maximum_scaling_step_size, expected_type=type_hints["maximum_scaling_step_size"])
                check_type(argname="argument minimum_scaling_step_size", value=minimum_scaling_step_size, expected_type=type_hints["minimum_scaling_step_size"])
                check_type(argname="argument status", value=status, expected_type=type_hints["status"])
                check_type(argname="argument target_capacity", value=target_capacity, expected_type=type_hints["target_capacity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if instance_warmup_period is not None:
                self._values["instance_warmup_period"] = instance_warmup_period
            if maximum_scaling_step_size is not None:
                self._values["maximum_scaling_step_size"] = maximum_scaling_step_size
            if minimum_scaling_step_size is not None:
                self._values["minimum_scaling_step_size"] = minimum_scaling_step_size
            if status is not None:
                self._values["status"] = status
            if target_capacity is not None:
                self._values["target_capacity"] = target_capacity

        @builtins.property
        def instance_warmup_period(self) -> typing.Optional[jsii.Number]:
            '''The period of time, in seconds, after a newly launched Amazon EC2 instance can contribute to CloudWatch metrics for Auto Scaling group.

            If this parameter is omitted, the default value of ``300`` seconds is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-instancewarmupperiod
            '''
            result = self._values.get("instance_warmup_period")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def maximum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of Amazon EC2 instances that Amazon ECS will scale out at one time.

            The scale in process is not affected by this parameter. If this parameter is omitted, the default value of ``1`` is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-maximumscalingstepsize
            '''
            result = self._values.get("maximum_scaling_step_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def minimum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of Amazon EC2 instances that Amazon ECS will scale out at one time.

            The scale in process is not affected by this parameter If this parameter is omitted, the default value of ``1`` is used.

            When additional capacity is required, Amazon ECS will scale up the minimum scaling step size even if the actual demand is less than the minimum scaling step size.

            If you use a capacity provider with an Auto Scaling group configured with more than one Amazon EC2 instance type or Availability Zone, Amazon ECS will scale up by the exact minimum scaling step size value and will ignore both the maximum scaling step size as well as the capacity demand.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-minimumscalingstepsize
            '''
            result = self._values.get("minimum_scaling_step_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def status(self) -> typing.Optional[builtins.str]:
            '''Determines whether to use managed scaling for the capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-status
            '''
            result = self._values.get("status")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The target capacity utilization as a percentage for the capacity provider.

            The specified value must be greater than ``0`` and less than or equal to ``100`` . For example, if you want the capacity provider to maintain 10% spare capacity, then that means the utilization is 90%, so use a ``targetCapacity`` of ``90`` . The default value of ``100`` percent results in the Amazon EC2 instances in your Auto Scaling group being completely used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-targetcapacity
            '''
            result = self._values.get("target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ManagedScalingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.CfnCapacityProviderProps",
    jsii_struct_bases=[],
    name_mapping={
        "auto_scaling_group_provider": "autoScalingGroupProvider",
        "name": "name",
        "tags": "tags",
    },
)
class CfnCapacityProviderProps:
    def __init__(
        self,
        *,
        auto_scaling_group_provider: typing.Union[_IResolvable_da3f097b, typing.Union[CfnCapacityProvider.AutoScalingGroupProviderProperty, typing.Dict[builtins.str, typing.Any]]],
        name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCapacityProvider``.

        :param auto_scaling_group_provider: The Auto Scaling group settings for the capacity provider.
        :param name: The name of the capacity provider. If a name is specified, it cannot start with ``aws`` , ``ecs`` , or ``fargate`` . If no name is specified, a default name in the ``CFNStackName-CFNResourceName-RandomString`` format is used.
        :param tags: The metadata that you apply to the capacity provider to help you categorize and organize it. Each tag consists of a key and an optional value. You define both. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            cfn_capacity_provider_props = ecs.CfnCapacityProviderProps(
                auto_scaling_group_provider=ecs.CfnCapacityProvider.AutoScalingGroupProviderProperty(
                    auto_scaling_group_arn="autoScalingGroupArn",
            
                    # the properties below are optional
                    managed_scaling=ecs.CfnCapacityProvider.ManagedScalingProperty(
                        instance_warmup_period=123,
                        maximum_scaling_step_size=123,
                        minimum_scaling_step_size=123,
                        status="status",
                        target_capacity=123
                    ),
                    managed_termination_protection="managedTerminationProtection"
                ),
            
                # the properties below are optional
                name="name",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__48080bdf05dc1c4ca9ab46c833774163f6afcd0d1551b378b8d59e67bc180c3f)
            check_type(argname="argument auto_scaling_group_provider", value=auto_scaling_group_provider, expected_type=type_hints["auto_scaling_group_provider"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auto_scaling_group_provider": auto_scaling_group_provider,
        }
        if name is not None:
            self._values["name"] = name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def auto_scaling_group_provider(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnCapacityProvider.AutoScalingGroupProviderProperty]:
        '''The Auto Scaling group settings for the capacity provider.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider
        '''
        result = self._values.get("auto_scaling_group_provider")
        assert result is not None, "Required property 'auto_scaling_group_provider' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnCapacityProvider.AutoScalingGroupProviderProperty], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the capacity provider.

        If a name is specified, it cannot start with ``aws`` , ``ecs`` , or ``fargate`` . If no name is specified, a default name in the ``CFNStackName-CFNResourceName-RandomString`` format is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The metadata that you apply to the capacity provider to help you categorize and organize it.

        Each tag consists of a key and an optional value. You define both.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCapacityProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnCluster(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ecs.CfnCluster",
):
    '''A CloudFormation ``AWS::ECS::Cluster``.

    The ``AWS::ECS::Cluster`` resource creates an Amazon Elastic Container Service (Amazon ECS) cluster.

    :cloudformationResource: AWS::ECS::Cluster
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ecs as ecs
        
        cfn_cluster = ecs.CfnCluster(self, "MyCfnCluster",
            capacity_providers=["capacityProviders"],
            cluster_name="clusterName",
            cluster_settings=[ecs.CfnCluster.ClusterSettingsProperty(
                name="name",
                value="value"
            )],
            configuration=ecs.CfnCluster.ClusterConfigurationProperty(
                execute_command_configuration=ecs.CfnCluster.ExecuteCommandConfigurationProperty(
                    kms_key_id="kmsKeyId",
                    log_configuration=ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                        cloud_watch_encryption_enabled=False,
                        cloud_watch_log_group_name="cloudWatchLogGroupName",
                        s3_bucket_name="s3BucketName",
                        s3_encryption_enabled=False,
                        s3_key_prefix="s3KeyPrefix"
                    ),
                    logging="logging"
                )
            ),
            default_capacity_provider_strategy=[ecs.CfnCluster.CapacityProviderStrategyItemProperty(
                base=123,
                capacity_provider="capacityProvider",
                weight=123
            )],
            service_connect_defaults=ecs.CfnCluster.ServiceConnectDefaultsProperty(
                namespace="namespace"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        capacity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        cluster_name: typing.Optional[builtins.str] = None,
        cluster_settings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ClusterSettingsProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ClusterConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        default_capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.CapacityProviderStrategyItemProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        service_connect_defaults: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ServiceConnectDefaultsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::Cluster``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param capacity_providers: The short name of one or more capacity providers to associate with the cluster. A capacity provider must be associated with a cluster before it can be included as part of the default capacity provider strategy of the cluster or used in a capacity provider strategy when calling the `CreateService <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html>`_ or `RunTask <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html>`_ actions. If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must be created but not associated with another cluster. New Auto Scaling group capacity providers can be created with the `CreateCapacityProvider <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateCapacityProvider.html>`_ API operation. To use a AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used. The `PutCapacityProvider <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutCapacityProvider.html>`_ API operation is used to update the list of available capacity providers for a cluster after the cluster is created.
        :param cluster_name: A user-generated string that you use to identify your cluster. If you don't specify a name, AWS CloudFormation generates a unique physical ID for the name.
        :param cluster_settings: The settings to use when creating a cluster. This parameter is used to turn on CloudWatch Container Insights for a cluster.
        :param configuration: The execute command configuration for the cluster.
        :param default_capacity_provider_strategy: The default capacity provider strategy for the cluster. When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.
        :param service_connect_defaults: Use this parameter to set a default Service Connect namespace. After you set a default Service Connect namespace, any new services with Service Connect turned on that are created in the cluster are added as client services in the namespace. This setting only applies to new services that set the ``enabled`` parameter to ``true`` in the ``ServiceConnectConfiguration`` . You can set the namespace of each service individually in the ``ServiceConnectConfiguration`` to override this default parameter. Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param tags: The metadata that you apply to the cluster to help you categorize and organize them. Each tag consists of a key and an optional value. You define both. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea27f9318b2a509011f1175119715629617e6b8d976d0782e37d54e45f1e94c8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnClusterProps(
            capacity_providers=capacity_providers,
            cluster_name=cluster_name,
            cluster_settings=cluster_settings,
            configuration=configuration,
            default_capacity_provider_strategy=default_capacity_provider_strategy,
            service_connect_defaults=service_connect_defaults,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c2cb7cb74fe2c923d0a807e5216751b2af96248eb8da823ea44f5c2902ba8882)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__937558a3311c7ac22ad24aaf6d722b5da6e120b9c20222d51118ea330da65b18)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the Amazon ECS cluster, such as ``arn:aws:ecs:us-east-2:123456789012:cluster/MyECSCluster`` .

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''The metadata that you apply to the cluster to help you categorize and organize them.

        Each tag consists of a key and an optional value. You define both.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="capacityProviders")
    def capacity_providers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The short name of one or more capacity providers to associate with the cluster.

        A capacity provider must be associated with a cluster before it can be included as part of the default capacity provider strategy of the cluster or used in a capacity provider strategy when calling the `CreateService <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html>`_ or `RunTask <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html>`_ actions.

        If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must be created but not associated with another cluster. New Auto Scaling group capacity providers can be created with the `CreateCapacityProvider <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateCapacityProvider.html>`_ API operation.

        To use a AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used.

        The `PutCapacityProvider <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutCapacityProvider.html>`_ API operation is used to update the list of available capacity providers for a cluster after the cluster is created.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-capacityproviders
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "capacityProviders"))

    @capacity_providers.setter
    def capacity_providers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f72bcd78ec28f8b6235a45c9a874d3074411135c04d1189f7b711f41fb19011f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "capacityProviders", value)

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> typing.Optional[builtins.str]:
        '''A user-generated string that you use to identify your cluster.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID for the name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "clusterName"))

    @cluster_name.setter
    def cluster_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a0a6687e9de573d67f4c2c57c08282746a1cc17e96eef26e0fac441e47050c61)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clusterName", value)

    @builtins.property
    @jsii.member(jsii_name="clusterSettings")
    def cluster_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.ClusterSettingsProperty"]]]]:
        '''The settings to use when creating a cluster.

        This parameter is used to turn on CloudWatch Container Insights for a cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustersettings
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.ClusterSettingsProperty"]]]], jsii.get(self, "clusterSettings"))

    @cluster_settings.setter
    def cluster_settings(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.ClusterSettingsProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7b904705b8bfde5e70548f3ba2688b1bc9b3237562659c56228f3d6f5f77860f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clusterSettings", value)

    @builtins.property
    @jsii.member(jsii_name="configuration")
    def configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ClusterConfigurationProperty"]]:
        '''The execute command configuration for the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-configuration
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ClusterConfigurationProperty"]], jsii.get(self, "configuration"))

    @configuration.setter
    def configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ClusterConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bc8e68ff9b3aeb40cad7c3029e01ab654eab91cf6e4423fbff6adb70d068b7ae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "configuration", value)

    @builtins.property
    @jsii.member(jsii_name="defaultCapacityProviderStrategy")
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.CapacityProviderStrategyItemProperty"]]]]:
        '''The default capacity provider strategy for the cluster.

        When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-defaultcapacityproviderstrategy
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.CapacityProviderStrategyItemProperty"]]]], jsii.get(self, "defaultCapacityProviderStrategy"))

    @default_capacity_provider_strategy.setter
    def default_capacity_provider_strategy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.CapacityProviderStrategyItemProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__73d775a9ee477c6e3311cfa033b6e7c2992a47b3a58187c38c6fb8e55ced8dda)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "defaultCapacityProviderStrategy", value)

    @builtins.property
    @jsii.member(jsii_name="serviceConnectDefaults")
    def service_connect_defaults(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ServiceConnectDefaultsProperty"]]:
        '''Use this parameter to set a default Service Connect namespace.

        After you set a default Service Connect namespace, any new services with Service Connect turned on that are created in the cluster are added as client services in the namespace. This setting only applies to new services that set the ``enabled`` parameter to ``true`` in the ``ServiceConnectConfiguration`` . You can set the namespace of each service individually in the ``ServiceConnectConfiguration`` to override this default parameter.

        Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-serviceconnectdefaults
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ServiceConnectDefaultsProperty"]], jsii.get(self, "serviceConnectDefaults"))

    @service_connect_defaults.setter
    def service_connect_defaults(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ServiceConnectDefaultsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4513757d6bf1dd7171c57a96c233575a7dfdc1797a261ca445a4dd29f575095c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serviceConnectDefaults", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnCluster.CapacityProviderStrategyItemProperty",
        jsii_struct_bases=[],
        name_mapping={
            "base": "base",
            "capacity_provider": "capacityProvider",
            "weight": "weight",
        },
    )
    class CapacityProviderStrategyItemProperty:
        def __init__(
            self,
            *,
            base: typing.Optional[jsii.Number] = None,
            capacity_provider: typing.Optional[builtins.str] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``CapacityProviderStrategyItem`` property specifies the details of the default capacity provider strategy for the cluster.

            When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.

            :param base: The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider. Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.
            :param capacity_provider: The short name of the capacity provider.
            :param weight: The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider. The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied. If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` can't be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail. An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that's run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                capacity_provider_strategy_item_property = ecs.CfnCluster.CapacityProviderStrategyItemProperty(
                    base=123,
                    capacity_provider="capacityProvider",
                    weight=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c0111db321c8b52d6283d3435e19f561034b11c2a7a4a80e235d2ee03de6dab4)
                check_type(argname="argument base", value=base, expected_type=type_hints["base"])
                check_type(argname="argument capacity_provider", value=capacity_provider, expected_type=type_hints["capacity_provider"])
                check_type(argname="argument weight", value=weight, expected_type=type_hints["weight"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if base is not None:
                self._values["base"] = base
            if capacity_provider is not None:
                self._values["capacity_provider"] = capacity_provider
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def base(self) -> typing.Optional[jsii.Number]:
            '''The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider.

            Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html#cfn-ecs-cluster-capacityproviderstrategyitem-base
            '''
            result = self._values.get("base")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def capacity_provider(self) -> typing.Optional[builtins.str]:
            '''The short name of the capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html#cfn-ecs-cluster-capacityproviderstrategyitem-capacityprovider
            '''
            result = self._values.get("capacity_provider")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.

            The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied.

            If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` can't be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail.

            An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that's run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html#cfn-ecs-cluster-capacityproviderstrategyitem-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityProviderStrategyItemProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnCluster.ClusterConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"execute_command_configuration": "executeCommandConfiguration"},
    )
    class ClusterConfigurationProperty:
        def __init__(
            self,
            *,
            execute_command_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ExecuteCommandConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The execute command configuration for the cluster.

            :param execute_command_configuration: The details of the execute command configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clusterconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                cluster_configuration_property = ecs.CfnCluster.ClusterConfigurationProperty(
                    execute_command_configuration=ecs.CfnCluster.ExecuteCommandConfigurationProperty(
                        kms_key_id="kmsKeyId",
                        log_configuration=ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                            cloud_watch_encryption_enabled=False,
                            cloud_watch_log_group_name="cloudWatchLogGroupName",
                            s3_bucket_name="s3BucketName",
                            s3_encryption_enabled=False,
                            s3_key_prefix="s3KeyPrefix"
                        ),
                        logging="logging"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d329e717aee636e9b40a663cb42030119882fdc5e5e743ac1d055ce0cbbadb3c)
                check_type(argname="argument execute_command_configuration", value=execute_command_configuration, expected_type=type_hints["execute_command_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if execute_command_configuration is not None:
                self._values["execute_command_configuration"] = execute_command_configuration

        @builtins.property
        def execute_command_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ExecuteCommandConfigurationProperty"]]:
            '''The details of the execute command configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clusterconfiguration.html#cfn-ecs-cluster-clusterconfiguration-executecommandconfiguration
            '''
            result = self._values.get("execute_command_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ExecuteCommandConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClusterConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnCluster.ClusterSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class ClusterSettingsProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The settings to use when creating a cluster.

            This parameter is used to turn on CloudWatch Container Insights for a cluster.

            :param name: The name of the cluster setting. The value is ``containerInsights`` .
            :param value: The value to set for the cluster setting. The supported values are ``enabled`` and ``disabled`` . If you set ``name`` to ``containerInsights`` and ``value`` to ``enabled`` , CloudWatch Container Insights will be on for the cluster, otherwise it will be off unless the ``containerInsights`` account setting is turned on. If a cluster value is specified, it will override the ``containerInsights`` value set with `PutAccountSetting <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html>`_ or `PutAccountSettingDefault <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                cluster_settings_property = ecs.CfnCluster.ClusterSettingsProperty(
                    name="name",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7e46ca2f184cd05213e6ee21bf8c7951f8cb2615617b125d090af778810914a0)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the cluster setting.

            The value is ``containerInsights`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html#cfn-ecs-cluster-clustersettings-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value to set for the cluster setting. The supported values are ``enabled`` and ``disabled`` .

            If you set ``name`` to ``containerInsights`` and ``value`` to ``enabled`` , CloudWatch Container Insights will be on for the cluster, otherwise it will be off unless the ``containerInsights`` account setting is turned on. If a cluster value is specified, it will override the ``containerInsights`` value set with `PutAccountSetting <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html>`_ or `PutAccountSettingDefault <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html#cfn-ecs-cluster-clustersettings-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClusterSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnCluster.ExecuteCommandConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "kms_key_id": "kmsKeyId",
            "log_configuration": "logConfiguration",
            "logging": "logging",
        },
    )
    class ExecuteCommandConfigurationProperty:
        def __init__(
            self,
            *,
            kms_key_id: typing.Optional[builtins.str] = None,
            log_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ExecuteCommandLogConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            logging: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The details of the execute command configuration.

            :param kms_key_id: Specify an AWS Key Management Service key ID to encrypt the data between the local client and the container.
            :param log_configuration: The log configuration for the results of the execute command actions. The logs can be sent to CloudWatch Logs or an Amazon S3 bucket. When ``logging=OVERRIDE`` is specified, a ``logConfiguration`` must be provided.
            :param logging: The log setting to use for redirecting logs for your execute command results. The following log settings are available. - ``NONE`` : The execute command session is not logged. - ``DEFAULT`` : The ``awslogs`` configuration in the task definition is used. If no logging parameter is specified, it defaults to this value. If no ``awslogs`` log driver is configured in the task definition, the output won't be logged. - ``OVERRIDE`` : Specify the logging details as a part of ``logConfiguration`` . If the ``OVERRIDE`` logging option is specified, the ``logConfiguration`` is required.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                execute_command_configuration_property = ecs.CfnCluster.ExecuteCommandConfigurationProperty(
                    kms_key_id="kmsKeyId",
                    log_configuration=ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                        cloud_watch_encryption_enabled=False,
                        cloud_watch_log_group_name="cloudWatchLogGroupName",
                        s3_bucket_name="s3BucketName",
                        s3_encryption_enabled=False,
                        s3_key_prefix="s3KeyPrefix"
                    ),
                    logging="logging"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9a7f784d4e942a95bfacace8f98a82cccf7b1feabecab9a3cd274e66168e4c75)
                check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
                check_type(argname="argument log_configuration", value=log_configuration, expected_type=type_hints["log_configuration"])
                check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if log_configuration is not None:
                self._values["log_configuration"] = log_configuration
            if logging is not None:
                self._values["logging"] = logging

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''Specify an AWS Key Management Service key ID to encrypt the data between the local client and the container.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def log_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ExecuteCommandLogConfigurationProperty"]]:
            '''The log configuration for the results of the execute command actions.

            The logs can be sent to CloudWatch Logs or an Amazon S3 bucket. When ``logging=OVERRIDE`` is specified, a ``logConfiguration`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-logconfiguration
            '''
            result = self._values.get("log_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ExecuteCommandLogConfigurationProperty"]], result)

        @builtins.property
        def logging(self) -> typing.Optional[builtins.str]:
            '''The log setting to use for redirecting logs for your execute command results. The following log settings are available.

            - ``NONE`` : The execute command session is not logged.
            - ``DEFAULT`` : The ``awslogs`` configuration in the task definition is used. If no logging parameter is specified, it defaults to this value. If no ``awslogs`` log driver is configured in the task definition, the output won't be logged.
            - ``OVERRIDE`` : Specify the logging details as a part of ``logConfiguration`` . If the ``OVERRIDE`` logging option is specified, the ``logConfiguration`` is required.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ExecuteCommandConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnCluster.ExecuteCommandLogConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cloud_watch_encryption_enabled": "cloudWatchEncryptionEnabled",
            "cloud_watch_log_group_name": "cloudWatchLogGroupName",
            "s3_bucket_name": "s3BucketName",
            "s3_encryption_enabled": "s3EncryptionEnabled",
            "s3_key_prefix": "s3KeyPrefix",
        },
    )
    class ExecuteCommandLogConfigurationProperty:
        def __init__(
            self,
            *,
            cloud_watch_encryption_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            cloud_watch_log_group_name: typing.Optional[builtins.str] = None,
            s3_bucket_name: typing.Optional[builtins.str] = None,
            s3_encryption_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            s3_key_prefix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The log configuration for the results of the execute command actions.

            The logs can be sent to CloudWatch Logs or an Amazon S3 bucket.

            :param cloud_watch_encryption_enabled: Determines whether to use encryption on the CloudWatch logs. If not specified, encryption will be off.
            :param cloud_watch_log_group_name: The name of the CloudWatch log group to send logs to. .. epigraph:: The CloudWatch log group must already be created.
            :param s3_bucket_name: The name of the S3 bucket to send logs to. .. epigraph:: The S3 bucket must already be created.
            :param s3_encryption_enabled: Determines whether to use encryption on the S3 logs. If not specified, encryption is not used.
            :param s3_key_prefix: An optional folder in the S3 bucket to place logs in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                execute_command_log_configuration_property = ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                    cloud_watch_encryption_enabled=False,
                    cloud_watch_log_group_name="cloudWatchLogGroupName",
                    s3_bucket_name="s3BucketName",
                    s3_encryption_enabled=False,
                    s3_key_prefix="s3KeyPrefix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c05052099f204719b48aff84daacc7380c4e383137e6a2a3c00551dee345324f)
                check_type(argname="argument cloud_watch_encryption_enabled", value=cloud_watch_encryption_enabled, expected_type=type_hints["cloud_watch_encryption_enabled"])
                check_type(argname="argument cloud_watch_log_group_name", value=cloud_watch_log_group_name, expected_type=type_hints["cloud_watch_log_group_name"])
                check_type(argname="argument s3_bucket_name", value=s3_bucket_name, expected_type=type_hints["s3_bucket_name"])
                check_type(argname="argument s3_encryption_enabled", value=s3_encryption_enabled, expected_type=type_hints["s3_encryption_enabled"])
                check_type(argname="argument s3_key_prefix", value=s3_key_prefix, expected_type=type_hints["s3_key_prefix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cloud_watch_encryption_enabled is not None:
                self._values["cloud_watch_encryption_enabled"] = cloud_watch_encryption_enabled
            if cloud_watch_log_group_name is not None:
                self._values["cloud_watch_log_group_name"] = cloud_watch_log_group_name
            if s3_bucket_name is not None:
                self._values["s3_bucket_name"] = s3_bucket_name
            if s3_encryption_enabled is not None:
                self._values["s3_encryption_enabled"] = s3_encryption_enabled
            if s3_key_prefix is not None:
                self._values["s3_key_prefix"] = s3_key_prefix

        @builtins.property
        def cloud_watch_encryption_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Determines whether to use encryption on the CloudWatch logs.

            If not specified, encryption will be off.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-cloudwatchencryptionenabled
            '''
            result = self._values.get("cloud_watch_encryption_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def cloud_watch_log_group_name(self) -> typing.Optional[builtins.str]:
            '''The name of the CloudWatch log group to send logs to.

            .. epigraph::

               The CloudWatch log group must already be created.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-cloudwatchloggroupname
            '''
            result = self._values.get("cloud_watch_log_group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_bucket_name(self) -> typing.Optional[builtins.str]:
            '''The name of the S3 bucket to send logs to.

            .. epigraph::

               The S3 bucket must already be created.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-s3bucketname
            '''
            result = self._values.get("s3_bucket_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_encryption_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Determines whether to use encryption on the S3 logs.

            If not specified, encryption is not used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-s3encryptionenabled
            '''
            result = self._values.get("s3_encryption_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def s3_key_prefix(self) -> typing.Optional[builtins.str]:
            '''An optional folder in the S3 bucket to place logs in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-s3keyprefix
            '''
            result = self._values.get("s3_key_prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ExecuteCommandLogConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnCluster.ServiceConnectDefaultsProperty",
        jsii_struct_bases=[],
        name_mapping={"namespace": "namespace"},
    )
    class ServiceConnectDefaultsProperty:
        def __init__(self, *, namespace: typing.Optional[builtins.str] = None) -> None:
            '''Use this parameter to set a default Service Connect namespace.

            After you set a default Service Connect namespace, any new services with Service Connect turned on that are created in the cluster are added as client services in the namespace. This setting only applies to new services that set the ``enabled`` parameter to ``true`` in the ``ServiceConnectConfiguration`` . You can set the namespace of each service individually in the ``ServiceConnectConfiguration`` to override this default parameter.

            Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param namespace: The namespace name or full Amazon Resource Name (ARN) of the AWS Cloud Map namespace that's used when you create a service and don't specify a Service Connect configuration. The namespace name can include up to 1024 characters. The name is case-sensitive. The name can't include hyphens (-), tilde (~), greater than (>), less than (<), or slash (/). If you enter an existing namespace name or ARN, then that namespace will be used. Any namespace type is supported. The namespace must be in this account and this AWS Region. If you enter a new name, a AWS Cloud Map namespace will be created. Amazon ECS creates a AWS Cloud Map namespace with the "API calls" method of instance discovery only. This instance discovery method is the "HTTP" namespace type in the AWS Command Line Interface . Other types of instance discovery aren't used by Service Connect. If you update the service with an empty string ``""`` for the namespace name, the cluster configuration for Service Connect is removed. Note that the namespace will remain in AWS Cloud Map and must be deleted separately. For more information about AWS Cloud Map , see `Working with Services <https://docs.aws.amazon.com/cloud-map/latest/dg/working-with-services.html>`_ in the *AWS Cloud Map Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-serviceconnectdefaults.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                service_connect_defaults_property = ecs.CfnCluster.ServiceConnectDefaultsProperty(
                    namespace="namespace"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__406d33f45440b3838cc9c39ff62fb2d038305a2814d49c398a3ffa7e3abbd9cd)
                check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if namespace is not None:
                self._values["namespace"] = namespace

        @builtins.property
        def namespace(self) -> typing.Optional[builtins.str]:
            '''The namespace name or full Amazon Resource Name (ARN) of the AWS Cloud Map namespace that's used when you create a service and don't specify a Service Connect configuration.

            The namespace name can include up to 1024 characters. The name is case-sensitive. The name can't include hyphens (-), tilde (~), greater than (>), less than (<), or slash (/).

            If you enter an existing namespace name or ARN, then that namespace will be used. Any namespace type is supported. The namespace must be in this account and this AWS Region.

            If you enter a new name, a AWS Cloud Map namespace will be created. Amazon ECS creates a AWS Cloud Map namespace with the "API calls" method of instance discovery only. This instance discovery method is the "HTTP" namespace type in the AWS Command Line Interface . Other types of instance discovery aren't used by Service Connect.

            If you update the service with an empty string ``""`` for the namespace name, the cluster configuration for Service Connect is removed. Note that the namespace will remain in AWS Cloud Map and must be deleted separately.

            For more information about AWS Cloud Map , see `Working with Services <https://docs.aws.amazon.com/cloud-map/latest/dg/working-with-services.html>`_ in the *AWS Cloud Map Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-serviceconnectdefaults.html#cfn-ecs-cluster-serviceconnectdefaults-namespace
            '''
            result = self._values.get("namespace")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceConnectDefaultsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnClusterCapacityProviderAssociations(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ecs.CfnClusterCapacityProviderAssociations",
):
    '''A CloudFormation ``AWS::ECS::ClusterCapacityProviderAssociations``.

    The ``AWS::ECS::ClusterCapacityProviderAssociations`` resource associates one or more capacity providers and a default capacity provider strategy with a cluster.

    :cloudformationResource: AWS::ECS::ClusterCapacityProviderAssociations
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ecs as ecs
        
        cfn_cluster_capacity_provider_associations = ecs.CfnClusterCapacityProviderAssociations(self, "MyCfnClusterCapacityProviderAssociations",
            capacity_providers=["capacityProviders"],
            cluster="cluster",
            default_capacity_provider_strategy=[ecs.CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty(
                capacity_provider="capacityProvider",
        
                # the properties below are optional
                base=123,
                weight=123
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        capacity_providers: typing.Sequence[builtins.str],
        cluster: builtins.str,
        default_capacity_provider_strategy: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty", typing.Dict[builtins.str, typing.Any]]]]],
    ) -> None:
        '''Create a new ``AWS::ECS::ClusterCapacityProviderAssociations``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param capacity_providers: The capacity providers to associate with the cluster.
        :param cluster: The cluster the capacity provider association is the target of.
        :param default_capacity_provider_strategy: The default capacity provider strategy to associate with the cluster.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b726b14d4a82695a68a7344f7ef1201a7390a69e77c604686bf3a3a16980041)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnClusterCapacityProviderAssociationsProps(
            capacity_providers=capacity_providers,
            cluster=cluster,
            default_capacity_provider_strategy=default_capacity_provider_strategy,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8dafa8668b2f8bb87dc45edfa02cb9d6bbf8954e459f23c8e6fb62be9d958c4)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__93896bd7ebbe840c8fa66a8f20ac3a94d52db05f416661f3eb272dbf9b1f204b)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="capacityProviders")
    def capacity_providers(self) -> typing.List[builtins.str]:
        '''The capacity providers to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-capacityproviders
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "capacityProviders"))

    @capacity_providers.setter
    def capacity_providers(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__179caeeed8cf1e3badcd9269f89a732efb1582d0dc5fe1a1431aee106eefe333)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "capacityProviders", value)

    @builtins.property
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> builtins.str:
        '''The cluster the capacity provider association is the target of.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-cluster
        '''
        return typing.cast(builtins.str, jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df256c18acb77e38e5110ec5f1e3b55621c49f4da43f861e9d55ef39c9207076)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cluster", value)

    @builtins.property
    @jsii.member(jsii_name="defaultCapacityProviderStrategy")
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty"]]]:
        '''The default capacity provider strategy to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-defaultcapacityproviderstrategy
        '''
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty"]]], jsii.get(self, "defaultCapacityProviderStrategy"))

    @default_capacity_provider_strategy.setter
    def default_capacity_provider_strategy(
        self,
        value: typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__adacafb6e01b8e49c325edb233c0e67268b6787a00fc50eea158771a202ed80a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "defaultCapacityProviderStrategy", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capacity_provider": "capacityProvider",
            "base": "base",
            "weight": "weight",
        },
    )
    class CapacityProviderStrategyProperty:
        def __init__(
            self,
            *,
            capacity_provider: builtins.str,
            base: typing.Optional[jsii.Number] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``CapacityProviderStrategy`` property specifies the details of the default capacity provider strategy for the cluster.

            When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.

            :param capacity_provider: The short name of the capacity provider.
            :param base: The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider. Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.
            :param weight: The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider. The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied. If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` will not be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail. An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that is run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                capacity_provider_strategy_property = ecs.CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty(
                    capacity_provider="capacityProvider",
                
                    # the properties below are optional
                    base=123,
                    weight=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__35e502a63db6f1c8f1c9225401bd8b2f0070d6dc6d8696b559629d5ccb289a80)
                check_type(argname="argument capacity_provider", value=capacity_provider, expected_type=type_hints["capacity_provider"])
                check_type(argname="argument base", value=base, expected_type=type_hints["base"])
                check_type(argname="argument weight", value=weight, expected_type=type_hints["weight"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "capacity_provider": capacity_provider,
            }
            if base is not None:
                self._values["base"] = base
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def capacity_provider(self) -> builtins.str:
            '''The short name of the capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html#cfn-ecs-clustercapacityproviderassociations-capacityproviderstrategy-capacityprovider
            '''
            result = self._values.get("capacity_provider")
            assert result is not None, "Required property 'capacity_provider' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def base(self) -> typing.Optional[jsii.Number]:
            '''The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider.

            Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html#cfn-ecs-clustercapacityproviderassociations-capacityproviderstrategy-base
            '''
            result = self._values.get("base")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.

            The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied.

            If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` will not be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail.

            An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that is run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html#cfn-ecs-clustercapacityproviderassociations-capacityproviderstrategy-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityProviderStrategyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.CfnClusterCapacityProviderAssociationsProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_providers": "capacityProviders",
        "cluster": "cluster",
        "default_capacity_provider_strategy": "defaultCapacityProviderStrategy",
    },
)
class CfnClusterCapacityProviderAssociationsProps:
    def __init__(
        self,
        *,
        capacity_providers: typing.Sequence[builtins.str],
        cluster: builtins.str,
        default_capacity_provider_strategy: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty, typing.Dict[builtins.str, typing.Any]]]]],
    ) -> None:
        '''Properties for defining a ``CfnClusterCapacityProviderAssociations``.

        :param capacity_providers: The capacity providers to associate with the cluster.
        :param cluster: The cluster the capacity provider association is the target of.
        :param default_capacity_provider_strategy: The default capacity provider strategy to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            cfn_cluster_capacity_provider_associations_props = ecs.CfnClusterCapacityProviderAssociationsProps(
                capacity_providers=["capacityProviders"],
                cluster="cluster",
                default_capacity_provider_strategy=[ecs.CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty(
                    capacity_provider="capacityProvider",
            
                    # the properties below are optional
                    base=123,
                    weight=123
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dec87cf6e858f074737c41c7a13a61e2e324b94deacf1f66ca9c0a48eb0b81b2)
            check_type(argname="argument capacity_providers", value=capacity_providers, expected_type=type_hints["capacity_providers"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument default_capacity_provider_strategy", value=default_capacity_provider_strategy, expected_type=type_hints["default_capacity_provider_strategy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "capacity_providers": capacity_providers,
            "cluster": cluster,
            "default_capacity_provider_strategy": default_capacity_provider_strategy,
        }

    @builtins.property
    def capacity_providers(self) -> typing.List[builtins.str]:
        '''The capacity providers to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-capacityproviders
        '''
        result = self._values.get("capacity_providers")
        assert result is not None, "Required property 'capacity_providers' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def cluster(self) -> builtins.str:
        '''The cluster the capacity provider association is the target of.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-cluster
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty]]]:
        '''The default capacity provider strategy to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-defaultcapacityproviderstrategy
        '''
        result = self._values.get("default_capacity_provider_strategy")
        assert result is not None, "Required property 'default_capacity_provider_strategy' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClusterCapacityProviderAssociationsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.CfnClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_providers": "capacityProviders",
        "cluster_name": "clusterName",
        "cluster_settings": "clusterSettings",
        "configuration": "configuration",
        "default_capacity_provider_strategy": "defaultCapacityProviderStrategy",
        "service_connect_defaults": "serviceConnectDefaults",
        "tags": "tags",
    },
)
class CfnClusterProps:
    def __init__(
        self,
        *,
        capacity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        cluster_name: typing.Optional[builtins.str] = None,
        cluster_settings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.ClusterSettingsProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.ClusterConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        default_capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.CapacityProviderStrategyItemProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        service_connect_defaults: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.ServiceConnectDefaultsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCluster``.

        :param capacity_providers: The short name of one or more capacity providers to associate with the cluster. A capacity provider must be associated with a cluster before it can be included as part of the default capacity provider strategy of the cluster or used in a capacity provider strategy when calling the `CreateService <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html>`_ or `RunTask <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html>`_ actions. If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must be created but not associated with another cluster. New Auto Scaling group capacity providers can be created with the `CreateCapacityProvider <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateCapacityProvider.html>`_ API operation. To use a AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used. The `PutCapacityProvider <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutCapacityProvider.html>`_ API operation is used to update the list of available capacity providers for a cluster after the cluster is created.
        :param cluster_name: A user-generated string that you use to identify your cluster. If you don't specify a name, AWS CloudFormation generates a unique physical ID for the name.
        :param cluster_settings: The settings to use when creating a cluster. This parameter is used to turn on CloudWatch Container Insights for a cluster.
        :param configuration: The execute command configuration for the cluster.
        :param default_capacity_provider_strategy: The default capacity provider strategy for the cluster. When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.
        :param service_connect_defaults: Use this parameter to set a default Service Connect namespace. After you set a default Service Connect namespace, any new services with Service Connect turned on that are created in the cluster are added as client services in the namespace. This setting only applies to new services that set the ``enabled`` parameter to ``true`` in the ``ServiceConnectConfiguration`` . You can set the namespace of each service individually in the ``ServiceConnectConfiguration`` to override this default parameter. Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param tags: The metadata that you apply to the cluster to help you categorize and organize them. Each tag consists of a key and an optional value. You define both. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            cfn_cluster_props = ecs.CfnClusterProps(
                capacity_providers=["capacityProviders"],
                cluster_name="clusterName",
                cluster_settings=[ecs.CfnCluster.ClusterSettingsProperty(
                    name="name",
                    value="value"
                )],
                configuration=ecs.CfnCluster.ClusterConfigurationProperty(
                    execute_command_configuration=ecs.CfnCluster.ExecuteCommandConfigurationProperty(
                        kms_key_id="kmsKeyId",
                        log_configuration=ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                            cloud_watch_encryption_enabled=False,
                            cloud_watch_log_group_name="cloudWatchLogGroupName",
                            s3_bucket_name="s3BucketName",
                            s3_encryption_enabled=False,
                            s3_key_prefix="s3KeyPrefix"
                        ),
                        logging="logging"
                    )
                ),
                default_capacity_provider_strategy=[ecs.CfnCluster.CapacityProviderStrategyItemProperty(
                    base=123,
                    capacity_provider="capacityProvider",
                    weight=123
                )],
                service_connect_defaults=ecs.CfnCluster.ServiceConnectDefaultsProperty(
                    namespace="namespace"
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7ff73a79103ae391f3b72b66851726ce38d98e8f5615c045aee9f3b899102a2)
            check_type(argname="argument capacity_providers", value=capacity_providers, expected_type=type_hints["capacity_providers"])
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
            check_type(argname="argument cluster_settings", value=cluster_settings, expected_type=type_hints["cluster_settings"])
            check_type(argname="argument configuration", value=configuration, expected_type=type_hints["configuration"])
            check_type(argname="argument default_capacity_provider_strategy", value=default_capacity_provider_strategy, expected_type=type_hints["default_capacity_provider_strategy"])
            check_type(argname="argument service_connect_defaults", value=service_connect_defaults, expected_type=type_hints["service_connect_defaults"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if capacity_providers is not None:
            self._values["capacity_providers"] = capacity_providers
        if cluster_name is not None:
            self._values["cluster_name"] = cluster_name
        if cluster_settings is not None:
            self._values["cluster_settings"] = cluster_settings
        if configuration is not None:
            self._values["configuration"] = configuration
        if default_capacity_provider_strategy is not None:
            self._values["default_capacity_provider_strategy"] = default_capacity_provider_strategy
        if service_connect_defaults is not None:
            self._values["service_connect_defaults"] = service_connect_defaults
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def capacity_providers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The short name of one or more capacity providers to associate with the cluster.

        A capacity provider must be associated with a cluster before it can be included as part of the default capacity provider strategy of the cluster or used in a capacity provider strategy when calling the `CreateService <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html>`_ or `RunTask <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html>`_ actions.

        If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must be created but not associated with another cluster. New Auto Scaling group capacity providers can be created with the `CreateCapacityProvider <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateCapacityProvider.html>`_ API operation.

        To use a AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used.

        The `PutCapacityProvider <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutCapacityProvider.html>`_ API operation is used to update the list of available capacity providers for a cluster after the cluster is created.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-capacityproviders
        '''
        result = self._values.get("capacity_providers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def cluster_name(self) -> typing.Optional[builtins.str]:
        '''A user-generated string that you use to identify your cluster.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID for the name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustername
        '''
        result = self._values.get("cluster_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cluster_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCluster.ClusterSettingsProperty]]]]:
        '''The settings to use when creating a cluster.

        This parameter is used to turn on CloudWatch Container Insights for a cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustersettings
        '''
        result = self._values.get("cluster_settings")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCluster.ClusterSettingsProperty]]]], result)

    @builtins.property
    def configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.ClusterConfigurationProperty]]:
        '''The execute command configuration for the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-configuration
        '''
        result = self._values.get("configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.ClusterConfigurationProperty]], result)

    @builtins.property
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCluster.CapacityProviderStrategyItemProperty]]]]:
        '''The default capacity provider strategy for the cluster.

        When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-defaultcapacityproviderstrategy
        '''
        result = self._values.get("default_capacity_provider_strategy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCluster.CapacityProviderStrategyItemProperty]]]], result)

    @builtins.property
    def service_connect_defaults(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.ServiceConnectDefaultsProperty]]:
        '''Use this parameter to set a default Service Connect namespace.

        After you set a default Service Connect namespace, any new services with Service Connect turned on that are created in the cluster are added as client services in the namespace. This setting only applies to new services that set the ``enabled`` parameter to ``true`` in the ``ServiceConnectConfiguration`` . You can set the namespace of each service individually in the ``ServiceConnectConfiguration`` to override this default parameter.

        Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-serviceconnectdefaults
        '''
        result = self._values.get("service_connect_defaults")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.ServiceConnectDefaultsProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The metadata that you apply to the cluster to help you categorize and organize them.

        Each tag consists of a key and an optional value. You define both.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnPrimaryTaskSet(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ecs.CfnPrimaryTaskSet",
):
    '''A CloudFormation ``AWS::ECS::PrimaryTaskSet``.

    Modifies which task set in a service is the primary task set. Any parameters that are updated on the primary task set in a service will transition to the service. This is used when a service uses the ``EXTERNAL`` deployment controller type. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

    :cloudformationResource: AWS::ECS::PrimaryTaskSet
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ecs as ecs
        
        cfn_primary_task_set = ecs.CfnPrimaryTaskSet(self, "MyCfnPrimaryTaskSet",
            cluster="cluster",
            service="service",
            task_set_id="taskSetId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_set_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::ECS::PrimaryTaskSet``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task set exists in.
        :param service: The short name or full Amazon Resource Name (ARN) of the service that the task set exists in.
        :param task_set_id: The short name or full Amazon Resource Name (ARN) of the task set to set as the primary task set in the deployment.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4124238541747ca453a0cbaeab47b4113ad0e26c4b5699058307d3befaef0e19)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnPrimaryTaskSetProps(
            cluster=cluster, service=service, task_set_id=task_set_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8af8c91ac7f99402228444cedfff4320e98144e15a69e334e73b7d6d7568a717)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e05e3107371a99154717669a7153aec288a2110f336b5dd7fa3d8bef5524749a)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task set exists in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-cluster
        '''
        return typing.cast(builtins.str, jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5fd19c81a21dcde3d0aa66f988ca71a954c06f012283264cd1c45c7ead4f6015)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cluster", value)

    @builtins.property
    @jsii.member(jsii_name="service")
    def service(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the service that the task set exists in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-service
        '''
        return typing.cast(builtins.str, jsii.get(self, "service"))

    @service.setter
    def service(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__09bbd50b9591c66bcf1ae74e4545273df95cc2f47a666659291c35d1e0a6d5df)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "service", value)

    @builtins.property
    @jsii.member(jsii_name="taskSetId")
    def task_set_id(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the task set to set as the primary task set in the deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-tasksetid
        '''
        return typing.cast(builtins.str, jsii.get(self, "taskSetId"))

    @task_set_id.setter
    def task_set_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1bd9f89b5b3447da3ac908b4cc96953c097c7d010d2ad281309f140d10f0ea13)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "taskSetId", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.CfnPrimaryTaskSetProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "service": "service",
        "task_set_id": "taskSetId",
    },
)
class CfnPrimaryTaskSetProps:
    def __init__(
        self,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_set_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnPrimaryTaskSet``.

        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task set exists in.
        :param service: The short name or full Amazon Resource Name (ARN) of the service that the task set exists in.
        :param task_set_id: The short name or full Amazon Resource Name (ARN) of the task set to set as the primary task set in the deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            cfn_primary_task_set_props = ecs.CfnPrimaryTaskSetProps(
                cluster="cluster",
                service="service",
                task_set_id="taskSetId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__280d7e3d1af2b5fa90d0679e5451a0349c8c4beed5bef49f2deac2e4eebc2589)
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument task_set_id", value=task_set_id, expected_type=type_hints["task_set_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
            "service": service,
            "task_set_id": task_set_id,
        }

    @builtins.property
    def cluster(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task set exists in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-cluster
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the service that the task set exists in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-service
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def task_set_id(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the task set to set as the primary task set in the deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-tasksetid
        '''
        result = self._values.get("task_set_id")
        assert result is not None, "Required property 'task_set_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPrimaryTaskSetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnService(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ecs.CfnService",
):
    '''A CloudFormation ``AWS::ECS::Service``.

    The ``AWS::ECS::Service`` resource creates an Amazon Elastic Container Service (Amazon ECS) service that runs and maintains the requested number of tasks and associated load balancers.
    .. epigraph::

       The stack update fails if you change any properties that require replacement and at least one Amazon ECS Service Connect ``ServiceConnectService`` is configured. This is because AWS CloudFormation creates the replacement service first, but each ``ServiceConnectService`` must have a name that is unique in the namespace. > Starting April 15, 2023, AWS ; will not onboard new customers to Amazon Elastic Inference (EI), and will help current customers migrate their workloads to options that offer better price and performance. After April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon SageMaker, Amazon ECS , or Amazon EC2 . However, customers who have used Amazon EI at least once during the past 30-day period are considered current customers and will be able to continue using the service.

    :cloudformationResource: AWS::ECS::Service
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ecs as ecs
        
        cfn_service = ecs.CfnService(self, "MyCfnService",
            capacity_provider_strategy=[ecs.CfnService.CapacityProviderStrategyItemProperty(
                base=123,
                capacity_provider="capacityProvider",
                weight=123
            )],
            cluster="cluster",
            deployment_configuration=ecs.CfnService.DeploymentConfigurationProperty(
                alarms=ecs.CfnService.DeploymentAlarmsProperty(
                    alarm_names=["alarmNames"],
                    enable=False,
                    rollback=False
                ),
                deployment_circuit_breaker=ecs.CfnService.DeploymentCircuitBreakerProperty(
                    enable=False,
                    rollback=False
                ),
                maximum_percent=123,
                minimum_healthy_percent=123
            ),
            deployment_controller=ecs.CfnService.DeploymentControllerProperty(
                type="type"
            ),
            desired_count=123,
            enable_ecs_managed_tags=False,
            enable_execute_command=False,
            health_check_grace_period_seconds=123,
            launch_type="launchType",
            load_balancers=[ecs.CfnService.LoadBalancerProperty(
                container_port=123,
        
                # the properties below are optional
                container_name="containerName",
                load_balancer_name="loadBalancerName",
                target_group_arn="targetGroupArn"
            )],
            network_configuration=ecs.CfnService.NetworkConfigurationProperty(
                awsvpc_configuration=ecs.CfnService.AwsVpcConfigurationProperty(
                    subnets=["subnets"],
        
                    # the properties below are optional
                    assign_public_ip="assignPublicIp",
                    security_groups=["securityGroups"]
                )
            ),
            placement_constraints=[ecs.CfnService.PlacementConstraintProperty(
                type="type",
        
                # the properties below are optional
                expression="expression"
            )],
            placement_strategies=[ecs.CfnService.PlacementStrategyProperty(
                type="type",
        
                # the properties below are optional
                field="field"
            )],
            platform_version="platformVersion",
            propagate_tags="propagateTags",
            role="role",
            scheduling_strategy="schedulingStrategy",
            service_connect_configuration=ecs.CfnService.ServiceConnectConfigurationProperty(
                enabled=False,
        
                # the properties below are optional
                log_configuration=ecs.CfnService.LogConfigurationProperty(
                    log_driver="logDriver",
                    options={
                        "options_key": "options"
                    },
                    secret_options=[ecs.CfnService.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )]
                ),
                namespace="namespace",
                services=[ecs.CfnService.ServiceConnectServiceProperty(
                    port_name="portName",
        
                    # the properties below are optional
                    client_aliases=[ecs.CfnService.ServiceConnectClientAliasProperty(
                        port=123,
        
                        # the properties below are optional
                        dns_name="dnsName"
                    )],
                    discovery_name="discoveryName",
                    ingress_port_override=123
                )]
            ),
            service_name="serviceName",
            service_registries=[ecs.CfnService.ServiceRegistryProperty(
                container_name="containerName",
                container_port=123,
                port=123,
                registry_arn="registryArn"
            )],
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            task_definition="taskDefinition"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.CapacityProviderStrategyItemProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        cluster: typing.Optional[builtins.str] = None,
        deployment_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.DeploymentConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        deployment_controller: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.DeploymentControllerProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        enable_execute_command: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        health_check_grace_period_seconds: typing.Optional[jsii.Number] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.LoadBalancerProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        network_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.NetworkConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.PlacementConstraintProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        placement_strategies: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.PlacementStrategyProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        propagate_tags: typing.Optional[builtins.str] = None,
        role: typing.Optional[builtins.str] = None,
        scheduling_strategy: typing.Optional[builtins.str] = None,
        service_connect_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.ServiceConnectConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        service_name: typing.Optional[builtins.str] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.ServiceRegistryProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        task_definition: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::Service``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param capacity_provider_strategy: The capacity provider strategy to use for the service. If a ``capacityProviderStrategy`` is specified, the ``launchType`` parameter must be omitted. If no ``capacityProviderStrategy`` or ``launchType`` is specified, the ``defaultCapacityProviderStrategy`` for the cluster is used. A capacity provider strategy may contain a maximum of 6 capacity providers.
        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that you run your service on. If you do not specify a cluster, the default cluster is assumed.
        :param deployment_configuration: Optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.
        :param deployment_controller: The deployment controller to use for the service. If no deployment controller is specified, the default value of ``ECS`` is used.
        :param desired_count: The number of instantiations of the specified task definition to place and keep running in your service. For new services, if a desired count is not specified, a default value of ``1`` is used. When using the ``DAEMON`` scheduling strategy, the desired count is not required. For existing services, if a desired count is not specified, it is omitted from the operation.
        :param enable_ecs_managed_tags: Specifies whether to turn on Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging your Amazon ECS resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ in the *Amazon Elastic Container Service Developer Guide* . When you use Amazon ECS managed tags, you need to set the ``propagateTags`` request parameter.
        :param enable_execute_command: Determines whether the execute command functionality is turned on for the service. If ``true`` , the execute command functionality is turned on for all containers in tasks as part of the service.
        :param health_check_grace_period_seconds: The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. This is only used when your service is configured to use a load balancer. If your service has a load balancer defined and you don't specify a health check grace period value, the default value of ``0`` is used. If you do not use an Elastic Load Balancing, we recommend that you use the ``startPeriod`` in the task definition health check parameters. For more information, see `Health check <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_HealthCheck.html>`_ . If your service's tasks take a while to start and respond to Elastic Load Balancing health checks, you can specify a health check grace period of up to 2,147,483,647 seconds (about 69 years). During that time, the Amazon ECS service scheduler ignores health check status. This grace period can prevent the service scheduler from marking tasks as unhealthy and stopping them before they have time to come up.
        :param launch_type: The launch type on which to run your service. For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param load_balancers: A list of load balancer objects to associate with the service. If you specify the ``Role`` property, ``LoadBalancers`` must be specified as well. For information about the number of load balancers that you can specify per service, see `Service Load Balancing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-load-balancing.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param network_configuration: The network configuration for the service. This parameter is required for task definitions that use the ``awsvpc`` network mode to receive their own elastic network interface, and it is not supported for other network modes. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param placement_constraints: An array of placement constraint objects to use for tasks in your service. You can specify a maximum of 10 constraints for each task. This limit includes constraints in the task definition and those specified at runtime.
        :param placement_strategies: The placement strategy objects to use for tasks in your service. You can specify a maximum of 5 strategy rules for each service.
        :param platform_version: The platform version that your tasks in the service are running on. A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param propagate_tags: Specifies whether to propagate the tags from the task definition to the task. If no value is specified, the tags aren't propagated. Tags can only be propagated to the task during task creation. To add tags to a task after task creation, use the `TagResource <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html>`_ API action. The default is ``NONE`` .
        :param role: The name or full Amazon Resource Name (ARN) of the IAM role that allows Amazon ECS to make calls to your load balancer on your behalf. This parameter is only permitted if you are using a load balancer with your service and your task definition doesn't use the ``awsvpc`` network mode. If you specify the ``role`` parameter, you must also specify a load balancer object with the ``loadBalancers`` parameter. .. epigraph:: If your account has already created the Amazon ECS service-linked role, that role is used for your service unless you specify a role here. The service-linked role is required if your task definition uses the ``awsvpc`` network mode or if the service is configured to use service discovery, an external deployment controller, multiple target groups, or Elastic Inference accelerators in which case you don't specify a role here. For more information, see `Using service-linked roles for Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If your specified role has a path other than ``/`` , then you must either specify the full role ARN (this is recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` then you would specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* .
        :param scheduling_strategy: The scheduling strategy to use for the service. For more information, see `Services <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html>`_ . There are two service scheduler strategies available: - ``REPLICA`` -The replica scheduling strategy places and maintains the desired number of tasks across your cluster. By default, the service scheduler spreads tasks across Availability Zones. You can use task placement strategies and constraints to customize task placement decisions. This scheduler strategy is required if the service uses the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types. - ``DAEMON`` -The daemon scheduling strategy deploys exactly one task on each active container instance that meets all of the task placement constraints that you specify in your cluster. The service scheduler also evaluates the task placement constraints for running tasks and will stop tasks that don't meet the placement constraints. When you're using this strategy, you don't need to specify a desired number of tasks, a task placement strategy, or use Service Auto Scaling policies. .. epigraph:: Tasks using the Fargate launch type or the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types don't support the ``DAEMON`` scheduling strategy.
        :param service_connect_configuration: The configuration for this service to discover and connect to services, and be discovered by, and connected from, other services within a namespace. Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param service_name: The name of your service. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. Service names must be unique within a cluster, but you can have similarly named services in multiple clusters within a Region or across multiple Regions. .. epigraph:: The stack update fails if you change any properties that require replacement and the ``ServiceName`` is configured. This is because AWS CloudFormation creates the replacement service first, but each ``ServiceName`` must be unique in the cluster.
        :param service_registries: The details of the service discovery registry to associate with this service. For more information, see `Service discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ . .. epigraph:: Each service may be associated with one service registry. Multiple service registries for each service isn't supported.
        :param tags: The metadata that you apply to the service to help you categorize and organize them. Each tag consists of a key and an optional value, both of which you define. When a service is deleted, the tags are deleted as well. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        :param task_definition: The ``family`` and ``revision`` ( ``family:revision`` ) or full ARN of the task definition to run in your service. If a ``revision`` isn't specified, the latest ``ACTIVE`` revision is used. A task definition must be specified if the service uses either the ``ECS`` or ``CODE_DEPLOY`` deployment controllers. For more information about deployment types, see `Amazon ECS deployment types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ec1192a1d20e03deef75c7fa1457b92ecf9506c5c5df97b5a4473fc3a9a714ef)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnServiceProps(
            capacity_provider_strategy=capacity_provider_strategy,
            cluster=cluster,
            deployment_configuration=deployment_configuration,
            deployment_controller=deployment_controller,
            desired_count=desired_count,
            enable_ecs_managed_tags=enable_ecs_managed_tags,
            enable_execute_command=enable_execute_command,
            health_check_grace_period_seconds=health_check_grace_period_seconds,
            launch_type=launch_type,
            load_balancers=load_balancers,
            network_configuration=network_configuration,
            placement_constraints=placement_constraints,
            placement_strategies=placement_strategies,
            platform_version=platform_version,
            propagate_tags=propagate_tags,
            role=role,
            scheduling_strategy=scheduling_strategy,
            service_connect_configuration=service_connect_configuration,
            service_name=service_name,
            service_registries=service_registries,
            tags=tags,
            task_definition=task_definition,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1badb735638b03a625f2b75f2b200e567f1ee535fd3bce64bc4428842f4cf245)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b2418aef5740852913fc8b9231af6ac6ece638d13d24b7a51e74b98f2093915c)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrName")
    def attr_name(self) -> builtins.str:
        '''The name of the Amazon ECS service, such as ``sample-webapp`` .

        :cloudformationAttribute: Name
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrName"))

    @builtins.property
    @jsii.member(jsii_name="attrServiceArn")
    def attr_service_arn(self) -> builtins.str:
        '''Not currently supported in AWS CloudFormation .

        :cloudformationAttribute: ServiceArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrServiceArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''The metadata that you apply to the service to help you categorize and organize them.

        Each tag consists of a key and an optional value, both of which you define. When a service is deleted, the tags are deleted as well.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="capacityProviderStrategy")
    def capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.CapacityProviderStrategyItemProperty"]]]]:
        '''The capacity provider strategy to use for the service.

        If a ``capacityProviderStrategy`` is specified, the ``launchType`` parameter must be omitted. If no ``capacityProviderStrategy`` or ``launchType`` is specified, the ``defaultCapacityProviderStrategy`` for the cluster is used.

        A capacity provider strategy may contain a maximum of 6 capacity providers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-capacityproviderstrategy
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.CapacityProviderStrategyItemProperty"]]]], jsii.get(self, "capacityProviderStrategy"))

    @capacity_provider_strategy.setter
    def capacity_provider_strategy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.CapacityProviderStrategyItemProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2cc9404cedce0aac2d2f2061fe9ad3f442cf33919c1a9f64eb934ded9311df13)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "capacityProviderStrategy", value)

    @builtins.property
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> typing.Optional[builtins.str]:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that you run your service on.

        If you do not specify a cluster, the default cluster is assumed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-cluster
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7e240f3ef87c9daa3aaaff4cc149c4fa59e0dff32cafc981254674ff0494c46)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cluster", value)

    @builtins.property
    @jsii.member(jsii_name="deploymentConfiguration")
    def deployment_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentConfigurationProperty"]]:
        '''Optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentConfigurationProperty"]], jsii.get(self, "deploymentConfiguration"))

    @deployment_configuration.setter
    def deployment_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4351ccf04150d6a51cbf4f07cf8bc2c688c395eae2adbee7112e0ec69c423200)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deploymentConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="deploymentController")
    def deployment_controller(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentControllerProperty"]]:
        '''The deployment controller to use for the service.

        If no deployment controller is specified, the default value of ``ECS`` is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentcontroller
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentControllerProperty"]], jsii.get(self, "deploymentController"))

    @deployment_controller.setter
    def deployment_controller(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentControllerProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c1106bf2bbd4b6b2f33facf01c316e1225c8a11ca790d69a763f51e4ec975df5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deploymentController", value)

    @builtins.property
    @jsii.member(jsii_name="desiredCount")
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''The number of instantiations of the specified task definition to place and keep running in your service.

        For new services, if a desired count is not specified, a default value of ``1`` is used. When using the ``DAEMON`` scheduling strategy, the desired count is not required.

        For existing services, if a desired count is not specified, it is omitted from the operation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-desiredcount
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "desiredCount"))

    @desired_count.setter
    def desired_count(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__51ebcdae4644237879f751e8bd294ce85b04be48f7d60fe368358307143c4872)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "desiredCount", value)

    @builtins.property
    @jsii.member(jsii_name="enableEcsManagedTags")
    def enable_ecs_managed_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to turn on Amazon ECS managed tags for the tasks within the service.

        For more information, see `Tagging your Amazon ECS resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        When you use Amazon ECS managed tags, you need to set the ``propagateTags`` request parameter.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableecsmanagedtags
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enableEcsManagedTags"))

    @enable_ecs_managed_tags.setter
    def enable_ecs_managed_tags(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0796264636a393525b9d077dfa82f77a09a8b352f87f00df117276bb342684f3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableEcsManagedTags", value)

    @builtins.property
    @jsii.member(jsii_name="enableExecuteCommand")
    def enable_execute_command(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Determines whether the execute command functionality is turned on for the service.

        If ``true`` , the execute command functionality is turned on for all containers in tasks as part of the service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableexecutecommand
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enableExecuteCommand"))

    @enable_execute_command.setter
    def enable_execute_command(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e2bc49b758cd7bd2d560e7ef349cc9fd90122266047080d15fab50f91a3db2e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableExecuteCommand", value)

    @builtins.property
    @jsii.member(jsii_name="healthCheckGracePeriodSeconds")
    def health_check_grace_period_seconds(self) -> typing.Optional[jsii.Number]:
        '''The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        This is only used when your service is configured to use a load balancer. If your service has a load balancer defined and you don't specify a health check grace period value, the default value of ``0`` is used.

        If you do not use an Elastic Load Balancing, we recommend that you use the ``startPeriod`` in the task definition health check parameters. For more information, see `Health check <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_HealthCheck.html>`_ .

        If your service's tasks take a while to start and respond to Elastic Load Balancing health checks, you can specify a health check grace period of up to 2,147,483,647 seconds (about 69 years). During that time, the Amazon ECS service scheduler ignores health check status. This grace period can prevent the service scheduler from marking tasks as unhealthy and stopping them before they have time to come up.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-healthcheckgraceperiodseconds
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "healthCheckGracePeriodSeconds"))

    @health_check_grace_period_seconds.setter
    def health_check_grace_period_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f711c1f6061fd4cde1ef9b3193fac0a0271ff054aad08ad6b79914e6f0f6db0f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "healthCheckGracePeriodSeconds", value)

    @builtins.property
    @jsii.member(jsii_name="launchType")
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''The launch type on which to run your service.

        For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-launchtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "launchType"))

    @launch_type.setter
    def launch_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ccfa1a8d0c5680335f975536f0056a9bfc4a669734ecfe273518232469da36d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "launchType", value)

    @builtins.property
    @jsii.member(jsii_name="loadBalancers")
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.LoadBalancerProperty"]]]]:
        '''A list of load balancer objects to associate with the service.

        If you specify the ``Role`` property, ``LoadBalancers`` must be specified as well. For information about the number of load balancers that you can specify per service, see `Service Load Balancing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-load-balancing.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-loadbalancers
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.LoadBalancerProperty"]]]], jsii.get(self, "loadBalancers"))

    @load_balancers.setter
    def load_balancers(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.LoadBalancerProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2c0697b43444ff5c04248d1f0d58b447432c7db5822c3044eafda33bbf88a1f9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "loadBalancers", value)

    @builtins.property
    @jsii.member(jsii_name="networkConfiguration")
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.NetworkConfigurationProperty"]]:
        '''The network configuration for the service.

        This parameter is required for task definitions that use the ``awsvpc`` network mode to receive their own elastic network interface, and it is not supported for other network modes. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-networkconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.NetworkConfigurationProperty"]], jsii.get(self, "networkConfiguration"))

    @network_configuration.setter
    def network_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.NetworkConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35091d03d59114e8b5f549279016726b6edc31d72aa07ca0bd9a1ca58fa54d4a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "networkConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="placementConstraints")
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.PlacementConstraintProperty"]]]]:
        '''An array of placement constraint objects to use for tasks in your service.

        You can specify a maximum of 10 constraints for each task. This limit includes constraints in the task definition and those specified at runtime.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementconstraints
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.PlacementConstraintProperty"]]]], jsii.get(self, "placementConstraints"))

    @placement_constraints.setter
    def placement_constraints(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.PlacementConstraintProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b9c4a420dfc576115223ce2e82d2b4d68c81e7692279f6b1d38c0d733ceaaf0d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "placementConstraints", value)

    @builtins.property
    @jsii.member(jsii_name="placementStrategies")
    def placement_strategies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.PlacementStrategyProperty"]]]]:
        '''The placement strategy objects to use for tasks in your service.

        You can specify a maximum of 5 strategy rules for each service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementstrategies
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.PlacementStrategyProperty"]]]], jsii.get(self, "placementStrategies"))

    @placement_strategies.setter
    def placement_strategies(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.PlacementStrategyProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8733fbcd708d31c0cac1ce1162fdb73412467c7dc418ef96de79816517a84d88)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "placementStrategies", value)

    @builtins.property
    @jsii.member(jsii_name="platformVersion")
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''The platform version that your tasks in the service are running on.

        A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-platformversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "platformVersion"))

    @platform_version.setter
    def platform_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__928f760123332244b95f1f404b0f7d7243c69ba98e6e26e575ec71d648a2ee42)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "platformVersion", value)

    @builtins.property
    @jsii.member(jsii_name="propagateTags")
    def propagate_tags(self) -> typing.Optional[builtins.str]:
        '''Specifies whether to propagate the tags from the task definition to the task.

        If no value is specified, the tags aren't propagated. Tags can only be propagated to the task during task creation. To add tags to a task after task creation, use the `TagResource <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html>`_ API action.

        The default is ``NONE`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-propagatetags
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "propagateTags"))

    @propagate_tags.setter
    def propagate_tags(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8498daf3b1adf30d5b499943ec44ec00c9b1944b2a6661792cc65ede76ab1f44)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "propagateTags", value)

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> typing.Optional[builtins.str]:
        '''The name or full Amazon Resource Name (ARN) of the IAM role that allows Amazon ECS to make calls to your load balancer on your behalf.

        This parameter is only permitted if you are using a load balancer with your service and your task definition doesn't use the ``awsvpc`` network mode. If you specify the ``role`` parameter, you must also specify a load balancer object with the ``loadBalancers`` parameter.
        .. epigraph::

           If your account has already created the Amazon ECS service-linked role, that role is used for your service unless you specify a role here. The service-linked role is required if your task definition uses the ``awsvpc`` network mode or if the service is configured to use service discovery, an external deployment controller, multiple target groups, or Elastic Inference accelerators in which case you don't specify a role here. For more information, see `Using service-linked roles for Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        If your specified role has a path other than ``/`` , then you must either specify the full role ARN (this is recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` then you would specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-role
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "role"))

    @role.setter
    def role(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f0d825970ce9ebc967d4246e01150a71610948c81bb17b5f2ae0e7dc81c2a92)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "role", value)

    @builtins.property
    @jsii.member(jsii_name="schedulingStrategy")
    def scheduling_strategy(self) -> typing.Optional[builtins.str]:
        '''The scheduling strategy to use for the service. For more information, see `Services <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html>`_ .

        There are two service scheduler strategies available:

        - ``REPLICA`` -The replica scheduling strategy places and maintains the desired number of tasks across your cluster. By default, the service scheduler spreads tasks across Availability Zones. You can use task placement strategies and constraints to customize task placement decisions. This scheduler strategy is required if the service uses the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types.
        - ``DAEMON`` -The daemon scheduling strategy deploys exactly one task on each active container instance that meets all of the task placement constraints that you specify in your cluster. The service scheduler also evaluates the task placement constraints for running tasks and will stop tasks that don't meet the placement constraints. When you're using this strategy, you don't need to specify a desired number of tasks, a task placement strategy, or use Service Auto Scaling policies.

        .. epigraph::

           Tasks using the Fargate launch type or the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types don't support the ``DAEMON`` scheduling strategy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-schedulingstrategy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "schedulingStrategy"))

    @scheduling_strategy.setter
    def scheduling_strategy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__18eaba97d3bc557a2d16a14c816fd02f6c372e3be105c5364bb15b0596b2dda0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "schedulingStrategy", value)

    @builtins.property
    @jsii.member(jsii_name="serviceConnectConfiguration")
    def service_connect_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceConnectConfigurationProperty"]]:
        '''The configuration for this service to discover and connect to services, and be discovered by, and connected from, other services within a namespace.

        Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-serviceconnectconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceConnectConfigurationProperty"]], jsii.get(self, "serviceConnectConfiguration"))

    @service_connect_configuration.setter
    def service_connect_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceConnectConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9b60b9aa5da511a557985bc3c729367eaca606c908b2b3826a2c387142433f65)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serviceConnectConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="serviceName")
    def service_name(self) -> typing.Optional[builtins.str]:
        '''The name of your service.

        Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. Service names must be unique within a cluster, but you can have similarly named services in multiple clusters within a Region or across multiple Regions.
        .. epigraph::

           The stack update fails if you change any properties that require replacement and the ``ServiceName`` is configured. This is because AWS CloudFormation creates the replacement service first, but each ``ServiceName`` must be unique in the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-servicename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "serviceName"))

    @service_name.setter
    def service_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__07a61aef5a3e67b762670027e581758f9be97a8133c5f3da0345d5667c237910)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serviceName", value)

    @builtins.property
    @jsii.member(jsii_name="serviceRegistries")
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceRegistryProperty"]]]]:
        '''The details of the service discovery registry to associate with this service. For more information, see `Service discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ .

        .. epigraph::

           Each service may be associated with one service registry. Multiple service registries for each service isn't supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-serviceregistries
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceRegistryProperty"]]]], jsii.get(self, "serviceRegistries"))

    @service_registries.setter
    def service_registries(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceRegistryProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a16af752d5bae77aefe7a7fa7946e7d908fe3c24ac325730d54f53c73e868d5d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serviceRegistries", value)

    @builtins.property
    @jsii.member(jsii_name="taskDefinition")
    def task_definition(self) -> typing.Optional[builtins.str]:
        '''The ``family`` and ``revision`` ( ``family:revision`` ) or full ARN of the task definition to run in your service.

        If a ``revision`` isn't specified, the latest ``ACTIVE`` revision is used.

        A task definition must be specified if the service uses either the ``ECS`` or ``CODE_DEPLOY`` deployment controllers.

        For more information about deployment types, see `Amazon ECS deployment types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-taskdefinition
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "taskDefinition"))

    @task_definition.setter
    def task_definition(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__622044881e70698cedaad575e114b69f1f2050c5c95de5f1d2fd013bbe56491b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "taskDefinition", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.AwsVpcConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "subnets": "subnets",
            "assign_public_ip": "assignPublicIp",
            "security_groups": "securityGroups",
        },
    )
    class AwsVpcConfigurationProperty:
        def __init__(
            self,
            *,
            subnets: typing.Sequence[builtins.str],
            assign_public_ip: typing.Optional[builtins.str] = None,
            security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object representing the networking details for a task or service.

            :param subnets: The IDs of the subnets associated with the task or service. There's a limit of 16 subnets that can be specified per ``AwsVpcConfiguration`` . .. epigraph:: All specified subnets must be from the same VPC.
            :param assign_public_ip: Whether the task's elastic network interface receives a public IP address. The default value is ``DISABLED`` .
            :param security_groups: The IDs of the security groups associated with the task or service. If you don't specify a security group, the default security group for the VPC is used. There's a limit of 5 security groups that can be specified per ``AwsVpcConfiguration`` . .. epigraph:: All specified security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                aws_vpc_configuration_property = ecs.CfnService.AwsVpcConfigurationProperty(
                    subnets=["subnets"],
                
                    # the properties below are optional
                    assign_public_ip="assignPublicIp",
                    security_groups=["securityGroups"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__40010db1fb42b63f942b5b11b7277545c2a1bffc42369636da9c14f813a744c5)
                check_type(argname="argument subnets", value=subnets, expected_type=type_hints["subnets"])
                check_type(argname="argument assign_public_ip", value=assign_public_ip, expected_type=type_hints["assign_public_ip"])
                check_type(argname="argument security_groups", value=security_groups, expected_type=type_hints["security_groups"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "subnets": subnets,
            }
            if assign_public_ip is not None:
                self._values["assign_public_ip"] = assign_public_ip
            if security_groups is not None:
                self._values["security_groups"] = security_groups

        @builtins.property
        def subnets(self) -> typing.List[builtins.str]:
            '''The IDs of the subnets associated with the task or service.

            There's a limit of 16 subnets that can be specified per ``AwsVpcConfiguration`` .
            .. epigraph::

               All specified subnets must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html#cfn-ecs-service-awsvpcconfiguration-subnets
            '''
            result = self._values.get("subnets")
            assert result is not None, "Required property 'subnets' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def assign_public_ip(self) -> typing.Optional[builtins.str]:
            '''Whether the task's elastic network interface receives a public IP address.

            The default value is ``DISABLED`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html#cfn-ecs-service-awsvpcconfiguration-assignpublicip
            '''
            result = self._values.get("assign_public_ip")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The IDs of the security groups associated with the task or service.

            If you don't specify a security group, the default security group for the VPC is used. There's a limit of 5 security groups that can be specified per ``AwsVpcConfiguration`` .
            .. epigraph::

               All specified security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html#cfn-ecs-service-awsvpcconfiguration-securitygroups
            '''
            result = self._values.get("security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsVpcConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.CapacityProviderStrategyItemProperty",
        jsii_struct_bases=[],
        name_mapping={
            "base": "base",
            "capacity_provider": "capacityProvider",
            "weight": "weight",
        },
    )
    class CapacityProviderStrategyItemProperty:
        def __init__(
            self,
            *,
            base: typing.Optional[jsii.Number] = None,
            capacity_provider: typing.Optional[builtins.str] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The details of a capacity provider strategy.

            A capacity provider strategy can be set when using the ``RunTask`` or ``CreateService`` APIs or as the default capacity provider strategy for a cluster with the ``CreateCluster`` API.

            Only capacity providers that are already associated with a cluster and have an ``ACTIVE`` or ``UPDATING`` status can be used in a capacity provider strategy. The ``PutClusterCapacityProviders`` API is used to associate a capacity provider with a cluster.

            If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created. New Auto Scaling group capacity providers can be created with the ``CreateCapacityProvider`` API operation.

            To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used in a capacity provider strategy.

            :param base: The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider. Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.
            :param capacity_provider: The short name of the capacity provider.
            :param weight: The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider. The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied. If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` can't be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail. An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that's run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                capacity_provider_strategy_item_property = ecs.CfnService.CapacityProviderStrategyItemProperty(
                    base=123,
                    capacity_provider="capacityProvider",
                    weight=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6035027887a35659d2eefa7fe47b352231895c442c86f6d2efb07eee0ee27a0c)
                check_type(argname="argument base", value=base, expected_type=type_hints["base"])
                check_type(argname="argument capacity_provider", value=capacity_provider, expected_type=type_hints["capacity_provider"])
                check_type(argname="argument weight", value=weight, expected_type=type_hints["weight"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if base is not None:
                self._values["base"] = base
            if capacity_provider is not None:
                self._values["capacity_provider"] = capacity_provider
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def base(self) -> typing.Optional[jsii.Number]:
            '''The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider.

            Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html#cfn-ecs-service-capacityproviderstrategyitem-base
            '''
            result = self._values.get("base")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def capacity_provider(self) -> typing.Optional[builtins.str]:
            '''The short name of the capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html#cfn-ecs-service-capacityproviderstrategyitem-capacityprovider
            '''
            result = self._values.get("capacity_provider")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.

            The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied.

            If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` can't be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail.

            An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that's run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html#cfn-ecs-service-capacityproviderstrategyitem-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityProviderStrategyItemProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.DeploymentAlarmsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "alarm_names": "alarmNames",
            "enable": "enable",
            "rollback": "rollback",
        },
    )
    class DeploymentAlarmsProperty:
        def __init__(
            self,
            *,
            alarm_names: typing.Sequence[builtins.str],
            enable: typing.Union[builtins.bool, _IResolvable_da3f097b],
            rollback: typing.Union[builtins.bool, _IResolvable_da3f097b],
        ) -> None:
            '''One of the methods which provide a way for you to quickly identify when a deployment has failed, and then to optionally roll back the failure to the last working deployment.

            When the alarms are generated, Amazon ECS sets the service deployment to failed. Set the rollback parameter to have Amazon ECS to roll back your service to the last completed deployment after a failure.

            You can only use the ``DeploymentAlarms`` method to detect failures when the ``DeploymentController`` is set to ``ECS`` (rolling update).

            For more information, see `Rolling update <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html>`_ in the **Amazon Elastic Container Service Developer Guide** .

            :param alarm_names: One or more CloudWatch alarm names. Use a "," to separate the alarms.
            :param enable: Determines whether to use the CloudWatch alarm option in the service deployment process.
            :param rollback: Determines whether to configure Amazon ECS to roll back the service if a service deployment fails. If rollback is used, when a service deployment fails, the service is rolled back to the last deployment that completed successfully.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentalarms.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                deployment_alarms_property = ecs.CfnService.DeploymentAlarmsProperty(
                    alarm_names=["alarmNames"],
                    enable=False,
                    rollback=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__08fef8ade8d5a7c824f3e4a36efcfb440d1fb353db12edb15b61d579dc5ec858)
                check_type(argname="argument alarm_names", value=alarm_names, expected_type=type_hints["alarm_names"])
                check_type(argname="argument enable", value=enable, expected_type=type_hints["enable"])
                check_type(argname="argument rollback", value=rollback, expected_type=type_hints["rollback"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "alarm_names": alarm_names,
                "enable": enable,
                "rollback": rollback,
            }

        @builtins.property
        def alarm_names(self) -> typing.List[builtins.str]:
            '''One or more CloudWatch alarm names.

            Use a "," to separate the alarms.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentalarms.html#cfn-ecs-service-deploymentalarms-alarmnames
            '''
            result = self._values.get("alarm_names")
            assert result is not None, "Required property 'alarm_names' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def enable(self) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Determines whether to use the CloudWatch alarm option in the service deployment process.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentalarms.html#cfn-ecs-service-deploymentalarms-enable
            '''
            result = self._values.get("enable")
            assert result is not None, "Required property 'enable' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        @builtins.property
        def rollback(self) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Determines whether to configure Amazon ECS to roll back the service if a service deployment fails.

            If rollback is used, when a service deployment fails, the service is rolled back to the last deployment that completed successfully.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentalarms.html#cfn-ecs-service-deploymentalarms-rollback
            '''
            result = self._values.get("rollback")
            assert result is not None, "Required property 'rollback' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentAlarmsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.DeploymentCircuitBreakerProperty",
        jsii_struct_bases=[],
        name_mapping={"enable": "enable", "rollback": "rollback"},
    )
    class DeploymentCircuitBreakerProperty:
        def __init__(
            self,
            *,
            enable: typing.Union[builtins.bool, _IResolvable_da3f097b],
            rollback: typing.Union[builtins.bool, _IResolvable_da3f097b],
        ) -> None:
            '''.. epigraph::

   The deployment circuit breaker can only be used for services using the rolling update ( ``ECS`` ) deployment type.

            The *deployment circuit breaker* determines whether a service deployment will fail if the service can't reach a steady state. If it is turned on, a service deployment will transition to a failed state and stop launching new tasks. You can also configure Amazon ECS to roll back your service to the last completed deployment after a failure. For more information, see `Rolling update <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param enable: Determines whether to use the deployment circuit breaker logic for the service.
            :param rollback: Determines whether to configure Amazon ECS to roll back the service if a service deployment fails. If rollback is on, when a service deployment fails, the service is rolled back to the last deployment that completed successfully.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcircuitbreaker.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                deployment_circuit_breaker_property = ecs.CfnService.DeploymentCircuitBreakerProperty(
                    enable=False,
                    rollback=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b2dbcb552902e6050bf0a575d5654d1d6ff74c24d93692dd2b9a98d506a89cc1)
                check_type(argname="argument enable", value=enable, expected_type=type_hints["enable"])
                check_type(argname="argument rollback", value=rollback, expected_type=type_hints["rollback"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "enable": enable,
                "rollback": rollback,
            }

        @builtins.property
        def enable(self) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Determines whether to use the deployment circuit breaker logic for the service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcircuitbreaker.html#cfn-ecs-service-deploymentcircuitbreaker-enable
            '''
            result = self._values.get("enable")
            assert result is not None, "Required property 'enable' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        @builtins.property
        def rollback(self) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Determines whether to configure Amazon ECS to roll back the service if a service deployment fails.

            If rollback is on, when a service deployment fails, the service is rolled back to the last deployment that completed successfully.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcircuitbreaker.html#cfn-ecs-service-deploymentcircuitbreaker-rollback
            '''
            result = self._values.get("rollback")
            assert result is not None, "Required property 'rollback' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentCircuitBreakerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.DeploymentConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "alarms": "alarms",
            "deployment_circuit_breaker": "deploymentCircuitBreaker",
            "maximum_percent": "maximumPercent",
            "minimum_healthy_percent": "minimumHealthyPercent",
        },
    )
    class DeploymentConfigurationProperty:
        def __init__(
            self,
            *,
            alarms: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.DeploymentAlarmsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            deployment_circuit_breaker: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.DeploymentCircuitBreakerProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            maximum_percent: typing.Optional[jsii.Number] = None,
            minimum_healthy_percent: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``DeploymentConfiguration`` property specifies optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.

            :param alarms: Information about the CloudWatch alarms.
            :param deployment_circuit_breaker: .. epigraph:: The deployment circuit breaker can only be used for services using the rolling update ( ``ECS`` ) deployment type. The *deployment circuit breaker* determines whether a service deployment will fail if the service can't reach a steady state. If you use the deployment circuit breaker, a service deployment will transition to a failed state and stop launching new tasks. If you use the rollback option, when a service deployment fails, the service is rolled back to the last deployment that completed successfully. For more information, see `Rolling update <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html>`_ in the *Amazon Elastic Container Service Developer Guide*
            :param maximum_percent: If a service is using the rolling update ( ``ECS`` ) deployment type, the ``maximumPercent`` parameter represents an upper limit on the number of your service's tasks that are allowed in the ``RUNNING`` or ``PENDING`` state during a deployment, as a percentage of the ``desiredCount`` (rounded down to the nearest integer). This parameter enables you to define the deployment batch size. For example, if your service is using the ``REPLICA`` service scheduler and has a ``desiredCount`` of four tasks and a ``maximumPercent`` value of 200%, the scheduler may start four new tasks before stopping the four older tasks (provided that the cluster resources required to do this are available). The default ``maximumPercent`` value for a service using the ``REPLICA`` service scheduler is 200%. If a service is using either the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and tasks that use the EC2 launch type, the *maximum percent* value is set to the default value and is used to define the upper limit on the number of the tasks in the service that remain in the ``RUNNING`` state while the container instances are in the ``DRAINING`` state. If the tasks in the service use the Fargate launch type, the maximum percent value is not used, although it is returned when describing your service.
            :param minimum_healthy_percent: If a service is using the rolling update ( ``ECS`` ) deployment type, the ``minimumHealthyPercent`` represents a lower limit on the number of your service's tasks that must remain in the ``RUNNING`` state during a deployment, as a percentage of the ``desiredCount`` (rounded up to the nearest integer). This parameter enables you to deploy without using additional cluster capacity. For example, if your service has a ``desiredCount`` of four tasks and a ``minimumHealthyPercent`` of 50%, the service scheduler may stop two existing tasks to free up cluster capacity before starting two new tasks. For services that *do not* use a load balancer, the following should be noted: - A service is considered healthy if all essential containers within the tasks in the service pass their health checks. - If a task has no essential containers with a health check defined, the service scheduler will wait for 40 seconds after a task reaches a ``RUNNING`` state before the task is counted towards the minimum healthy percent total. - If a task has one or more essential containers with a health check defined, the service scheduler will wait for the task to reach a healthy status before counting it towards the minimum healthy percent total. A task is considered healthy when all essential containers within the task have passed their health checks. The amount of time the service scheduler can wait for is determined by the container health check settings. For services are that *do* use a load balancer, the following should be noted: - If a task has no essential containers with a health check defined, the service scheduler will wait for the load balancer target group health check to return a healthy status before counting the task towards the minimum healthy percent total. - If a task has an essential container with a health check defined, the service scheduler will wait for both the task to reach a healthy status and the load balancer target group health check to return a healthy status before counting the task towards the minimum healthy percent total. If a service is using either the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and is running tasks that use the EC2 launch type, the *minimum healthy percent* value is set to the default value and is used to define the lower limit on the number of the tasks in the service that remain in the ``RUNNING`` state while the container instances are in the ``DRAINING`` state. If a service is using either the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and is running tasks that use the Fargate launch type, the minimum healthy percent value is not used, although it is returned when describing your service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                deployment_configuration_property = ecs.CfnService.DeploymentConfigurationProperty(
                    alarms=ecs.CfnService.DeploymentAlarmsProperty(
                        alarm_names=["alarmNames"],
                        enable=False,
                        rollback=False
                    ),
                    deployment_circuit_breaker=ecs.CfnService.DeploymentCircuitBreakerProperty(
                        enable=False,
                        rollback=False
                    ),
                    maximum_percent=123,
                    minimum_healthy_percent=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d809d14e704a11675cbface3b43579e5f8f2a29c9b48e27608c625a3f01cb3a6)
                check_type(argname="argument alarms", value=alarms, expected_type=type_hints["alarms"])
                check_type(argname="argument deployment_circuit_breaker", value=deployment_circuit_breaker, expected_type=type_hints["deployment_circuit_breaker"])
                check_type(argname="argument maximum_percent", value=maximum_percent, expected_type=type_hints["maximum_percent"])
                check_type(argname="argument minimum_healthy_percent", value=minimum_healthy_percent, expected_type=type_hints["minimum_healthy_percent"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if alarms is not None:
                self._values["alarms"] = alarms
            if deployment_circuit_breaker is not None:
                self._values["deployment_circuit_breaker"] = deployment_circuit_breaker
            if maximum_percent is not None:
                self._values["maximum_percent"] = maximum_percent
            if minimum_healthy_percent is not None:
                self._values["minimum_healthy_percent"] = minimum_healthy_percent

        @builtins.property
        def alarms(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentAlarmsProperty"]]:
            '''Information about the CloudWatch alarms.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-alarms
            '''
            result = self._values.get("alarms")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentAlarmsProperty"]], result)

        @builtins.property
        def deployment_circuit_breaker(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentCircuitBreakerProperty"]]:
            '''.. epigraph::

   The deployment circuit breaker can only be used for services using the rolling update ( ``ECS`` ) deployment type.

            The *deployment circuit breaker* determines whether a service deployment will fail if the service can't reach a steady state. If you use the deployment circuit breaker, a service deployment will transition to a failed state and stop launching new tasks. If you use the rollback option, when a service deployment fails, the service is rolled back to the last deployment that completed successfully. For more information, see `Rolling update <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html>`_ in the *Amazon Elastic Container Service Developer Guide*

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-deploymentcircuitbreaker
            '''
            result = self._values.get("deployment_circuit_breaker")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.DeploymentCircuitBreakerProperty"]], result)

        @builtins.property
        def maximum_percent(self) -> typing.Optional[jsii.Number]:
            '''If a service is using the rolling update ( ``ECS`` ) deployment type, the ``maximumPercent`` parameter represents an upper limit on the number of your service's tasks that are allowed in the ``RUNNING`` or ``PENDING`` state during a deployment, as a percentage of the ``desiredCount`` (rounded down to the nearest integer).

            This parameter enables you to define the deployment batch size. For example, if your service is using the ``REPLICA`` service scheduler and has a ``desiredCount`` of four tasks and a ``maximumPercent`` value of 200%, the scheduler may start four new tasks before stopping the four older tasks (provided that the cluster resources required to do this are available). The default ``maximumPercent`` value for a service using the ``REPLICA`` service scheduler is 200%.

            If a service is using either the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and tasks that use the EC2 launch type, the *maximum percent* value is set to the default value and is used to define the upper limit on the number of the tasks in the service that remain in the ``RUNNING`` state while the container instances are in the ``DRAINING`` state. If the tasks in the service use the Fargate launch type, the maximum percent value is not used, although it is returned when describing your service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-maximumpercent
            '''
            result = self._values.get("maximum_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def minimum_healthy_percent(self) -> typing.Optional[jsii.Number]:
            '''If a service is using the rolling update ( ``ECS`` ) deployment type, the ``minimumHealthyPercent`` represents a lower limit on the number of your service's tasks that must remain in the ``RUNNING`` state during a deployment, as a percentage of the ``desiredCount`` (rounded up to the nearest integer).

            This parameter enables you to deploy without using additional cluster capacity. For example, if your service has a ``desiredCount`` of four tasks and a ``minimumHealthyPercent`` of 50%, the service scheduler may stop two existing tasks to free up cluster capacity before starting two new tasks.

            For services that *do not* use a load balancer, the following should be noted:

            - A service is considered healthy if all essential containers within the tasks in the service pass their health checks.
            - If a task has no essential containers with a health check defined, the service scheduler will wait for 40 seconds after a task reaches a ``RUNNING`` state before the task is counted towards the minimum healthy percent total.
            - If a task has one or more essential containers with a health check defined, the service scheduler will wait for the task to reach a healthy status before counting it towards the minimum healthy percent total. A task is considered healthy when all essential containers within the task have passed their health checks. The amount of time the service scheduler can wait for is determined by the container health check settings.

            For services are that *do* use a load balancer, the following should be noted:

            - If a task has no essential containers with a health check defined, the service scheduler will wait for the load balancer target group health check to return a healthy status before counting the task towards the minimum healthy percent total.
            - If a task has an essential container with a health check defined, the service scheduler will wait for both the task to reach a healthy status and the load balancer target group health check to return a healthy status before counting the task towards the minimum healthy percent total.

            If a service is using either the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and is running tasks that use the EC2 launch type, the *minimum healthy percent* value is set to the default value and is used to define the lower limit on the number of the tasks in the service that remain in the ``RUNNING`` state while the container instances are in the ``DRAINING`` state. If a service is using either the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and is running tasks that use the Fargate launch type, the minimum healthy percent value is not used, although it is returned when describing your service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-minimumhealthypercent
            '''
            result = self._values.get("minimum_healthy_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.DeploymentControllerProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class DeploymentControllerProperty:
        def __init__(self, *, type: typing.Optional[builtins.str] = None) -> None:
            '''The deployment controller to use for the service.

            For more information, see `Amazon ECS deployment types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param type: The deployment controller type to use. There are three deployment controller types available:. - **ECS** - The rolling update ( ``ECS`` ) deployment type involves replacing the current running version of the container with the latest version. The number of containers Amazon ECS adds or removes from the service during a rolling update is controlled by adjusting the minimum and maximum number of healthy tasks allowed during a service deployment, as specified in the `DeploymentConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeploymentConfiguration.html>`_ . - **CODE_DEPLOY** - The blue/green ( ``CODE_DEPLOY`` ) deployment type uses the blue/green deployment model powered by AWS CodeDeploy , which allows you to verify a new deployment of a service before sending production traffic to it. - **EXTERNAL** - The external ( ``EXTERNAL`` ) deployment type enables you to use any third-party deployment controller for full control over the deployment process for an Amazon ECS service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcontroller.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                deployment_controller_property = ecs.CfnService.DeploymentControllerProperty(
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e9921934df7686d8808e649b9ab979d2747a6f8cba336af2424f597839d2103f)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The deployment controller type to use. There are three deployment controller types available:.

            - **ECS** - The rolling update ( ``ECS`` ) deployment type involves replacing the current running version of the container with the latest version. The number of containers Amazon ECS adds or removes from the service during a rolling update is controlled by adjusting the minimum and maximum number of healthy tasks allowed during a service deployment, as specified in the `DeploymentConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeploymentConfiguration.html>`_ .
            - **CODE_DEPLOY** - The blue/green ( ``CODE_DEPLOY`` ) deployment type uses the blue/green deployment model powered by AWS CodeDeploy , which allows you to verify a new deployment of a service before sending production traffic to it.
            - **EXTERNAL** - The external ( ``EXTERNAL`` ) deployment type enables you to use any third-party deployment controller for full control over the deployment process for an Amazon ECS service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcontroller.html#cfn-ecs-service-deploymentcontroller-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentControllerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.LoadBalancerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_port": "containerPort",
            "container_name": "containerName",
            "load_balancer_name": "loadBalancerName",
            "target_group_arn": "targetGroupArn",
        },
    )
    class LoadBalancerProperty:
        def __init__(
            self,
            *,
            container_port: jsii.Number,
            container_name: typing.Optional[builtins.str] = None,
            load_balancer_name: typing.Optional[builtins.str] = None,
            target_group_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``LoadBalancer`` property specifies details on a load balancer that is used with a service.

            If the service is using the ``CODE_DEPLOY`` deployment controller, the service is required to use either an Application Load Balancer or Network Load Balancer. When you are creating an AWS CodeDeploy deployment group, you specify two target groups (referred to as a ``targetGroupPair`` ). Each target group binds to a separate task set in the deployment. The load balancer can also have up to two listeners, a required listener for production traffic and an optional listener that allows you to test new revisions of the service before routing production traffic to it.

            Services with tasks that use the ``awsvpc`` network mode (for example, those with the Fargate launch type) only support Application Load Balancers and Network Load Balancers. Classic Load Balancers are not supported. Also, when you create any target groups for these services, you must choose ``ip`` as the target type, not ``instance`` . Tasks that use the ``awsvpc`` network mode are associated with an elastic network interface, not an Amazon EC2 instance.

            :param container_port: The port on the container to associate with the load balancer. This port must correspond to a ``containerPort`` in the task definition the tasks in the service are using. For tasks that use the EC2 launch type, the container instance they're launched on must allow ingress traffic on the ``hostPort`` of the port mapping.
            :param container_name: The name of the container (as it appears in a container definition) to associate with the load balancer.
            :param load_balancer_name: The name of the load balancer to associate with the Amazon ECS service or task set. A load balancer name is only specified when using a Classic Load Balancer. If you are using an Application Load Balancer or a Network Load Balancer the load balancer name parameter should be omitted.
            :param target_group_arn: The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated with a service or task set. A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer. If you're using a Classic Load Balancer, omit the target group ARN. For services using the ``ECS`` deployment controller, you can specify one or multiple target groups. For more information, see `Registering multiple target groups with a service <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For services using the ``CODE_DEPLOY`` deployment controller, you're required to define two target groups for the load balancer. For more information, see `Blue/green deployment with CodeDeploy <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: If your service's task definition uses the ``awsvpc`` network mode, you must choose ``ip`` as the target type, not ``instance`` . Do this when creating your target groups because tasks that use the ``awsvpc`` network mode are associated with an elastic network interface, not an Amazon EC2 instance. This network mode is required for the Fargate launch type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                load_balancer_property = ecs.CfnService.LoadBalancerProperty(
                    container_port=123,
                
                    # the properties below are optional
                    container_name="containerName",
                    load_balancer_name="loadBalancerName",
                    target_group_arn="targetGroupArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__251c3999c1586967f7c9091782e6a113831e05b5f853b910cad8c8e75f654def)
                check_type(argname="argument container_port", value=container_port, expected_type=type_hints["container_port"])
                check_type(argname="argument container_name", value=container_name, expected_type=type_hints["container_name"])
                check_type(argname="argument load_balancer_name", value=load_balancer_name, expected_type=type_hints["load_balancer_name"])
                check_type(argname="argument target_group_arn", value=target_group_arn, expected_type=type_hints["target_group_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "container_port": container_port,
            }
            if container_name is not None:
                self._values["container_name"] = container_name
            if load_balancer_name is not None:
                self._values["load_balancer_name"] = load_balancer_name
            if target_group_arn is not None:
                self._values["target_group_arn"] = target_group_arn

        @builtins.property
        def container_port(self) -> jsii.Number:
            '''The port on the container to associate with the load balancer.

            This port must correspond to a ``containerPort`` in the task definition the tasks in the service are using. For tasks that use the EC2 launch type, the container instance they're launched on must allow ingress traffic on the ``hostPort`` of the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-containerport
            '''
            result = self._values.get("container_port")
            assert result is not None, "Required property 'container_port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''The name of the container (as it appears in a container definition) to associate with the load balancer.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def load_balancer_name(self) -> typing.Optional[builtins.str]:
            '''The name of the load balancer to associate with the Amazon ECS service or task set.

            A load balancer name is only specified when using a Classic Load Balancer. If you are using an Application Load Balancer or a Network Load Balancer the load balancer name parameter should be omitted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-loadbalancername
            '''
            result = self._values.get("load_balancer_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_group_arn(self) -> typing.Optional[builtins.str]:
            '''The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated with a service or task set.

            A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer. If you're using a Classic Load Balancer, omit the target group ARN.

            For services using the ``ECS`` deployment controller, you can specify one or multiple target groups. For more information, see `Registering multiple target groups with a service <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For services using the ``CODE_DEPLOY`` deployment controller, you're required to define two target groups for the load balancer. For more information, see `Blue/green deployment with CodeDeploy <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               If your service's task definition uses the ``awsvpc`` network mode, you must choose ``ip`` as the target type, not ``instance`` . Do this when creating your target groups because tasks that use the ``awsvpc`` network mode are associated with an elastic network interface, not an Amazon EC2 instance. This network mode is required for the Fargate launch type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-targetgrouparn
            '''
            result = self._values.get("target_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoadBalancerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.LogConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "log_driver": "logDriver",
            "options": "options",
            "secret_options": "secretOptions",
        },
    )
    class LogConfigurationProperty:
        def __init__(
            self,
            *,
            log_driver: typing.Optional[builtins.str] = None,
            options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]] = None,
            secret_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.SecretProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The log configuration for the container.

            This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--log-driver`` option to ```docker run`` <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/run/>`_ .

            By default, containers use the same logging driver that the Docker daemon uses. However, the container might use a different logging driver than the Docker daemon by specifying a log driver configuration in the container definition. For more information about the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the Docker documentation.

            Understand the following when specifying a log configuration for your containers.

            - Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the valid values below). Additional log drivers may be available in future releases of the Amazon ECS container agent.
            - This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
            - For tasks that are hosted on Amazon EC2 instances, the Amazon ECS container agent must register the available logging drivers with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS container agent configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            - For tasks that are on AWS Fargate , because you don't have access to the underlying infrastructure your tasks are hosted on, any additional software needed must be installed outside of the task. For example, the Fluentd output aggregators or a remote host running Logstash to send Gelf logs to.

            :param log_driver: The log driver to use for the container. For tasks on AWS Fargate , the supported log drivers are ``awslogs`` , ``splunk`` , and ``awsfirelens`` . For tasks hosted on Amazon EC2 instances, the supported log drivers are ``awslogs`` , ``fluentd`` , ``gelf`` , ``json-file`` , ``journald`` , ``logentries`` , ``syslog`` , ``splunk`` , and ``awsfirelens`` . For more information about using the ``awslogs`` log driver, see `Using the awslogs log driver <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For more information about using the ``awsfirelens`` log driver, see `Custom log routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's `available on GitHub <https://docs.aws.amazon.com/https://github.com/aws/amazon-ecs-agent>`_ and customize it to work with that driver. We encourage you to submit pull requests for changes that you would like to have included. However, we don't currently provide support for running modified copies of this software.
            :param options: The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            :param secret_options: The secrets to pass to the log configuration. For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-logconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                log_configuration_property = ecs.CfnService.LogConfigurationProperty(
                    log_driver="logDriver",
                    options={
                        "options_key": "options"
                    },
                    secret_options=[ecs.CfnService.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dc71ad10c4d7e167753ff81a7c450165b3118a6f5f0f8c2ad282bc4bc057a60b)
                check_type(argname="argument log_driver", value=log_driver, expected_type=type_hints["log_driver"])
                check_type(argname="argument options", value=options, expected_type=type_hints["options"])
                check_type(argname="argument secret_options", value=secret_options, expected_type=type_hints["secret_options"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if log_driver is not None:
                self._values["log_driver"] = log_driver
            if options is not None:
                self._values["options"] = options
            if secret_options is not None:
                self._values["secret_options"] = secret_options

        @builtins.property
        def log_driver(self) -> typing.Optional[builtins.str]:
            '''The log driver to use for the container.

            For tasks on AWS Fargate , the supported log drivers are ``awslogs`` , ``splunk`` , and ``awsfirelens`` .

            For tasks hosted on Amazon EC2 instances, the supported log drivers are ``awslogs`` , ``fluentd`` , ``gelf`` , ``json-file`` , ``journald`` , ``logentries`` , ``syslog`` , ``splunk`` , and ``awsfirelens`` .

            For more information about using the ``awslogs`` log driver, see `Using the awslogs log driver <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For more information about using the ``awsfirelens`` log driver, see `Custom log routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's `available on GitHub <https://docs.aws.amazon.com/https://github.com/aws/amazon-ecs-agent>`_ and customize it to work with that driver. We encourage you to submit pull requests for changes that you would like to have included. However, we don't currently provide support for running modified copies of this software.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-logconfiguration.html#cfn-ecs-service-logconfiguration-logdriver
            '''
            result = self._values.get("log_driver")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]]:
            '''The configuration options to send to the log driver.

            This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-logconfiguration.html#cfn-ecs-service-logconfiguration-options
            '''
            result = self._values.get("options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def secret_options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.SecretProperty"]]]]:
            '''The secrets to pass to the log configuration.

            For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-logconfiguration.html#cfn-ecs-service-logconfiguration-secretoptions
            '''
            result = self._values.get("secret_options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.SecretProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LogConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.NetworkConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"awsvpc_configuration": "awsvpcConfiguration"},
    )
    class NetworkConfigurationProperty:
        def __init__(
            self,
            *,
            awsvpc_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.AwsVpcConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The ``NetworkConfiguration`` property specifies an object representing the network configuration for a task or service.

            :param awsvpc_configuration: The VPC subnets and security groups that are associated with a task. .. epigraph:: All specified subnets and security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-networkconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                network_configuration_property = ecs.CfnService.NetworkConfigurationProperty(
                    awsvpc_configuration=ecs.CfnService.AwsVpcConfigurationProperty(
                        subnets=["subnets"],
                
                        # the properties below are optional
                        assign_public_ip="assignPublicIp",
                        security_groups=["securityGroups"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__69d6c8f5c91c26fb4e5d380e50f88db48e07eae26511e843bc32888305e7bf13)
                check_type(argname="argument awsvpc_configuration", value=awsvpc_configuration, expected_type=type_hints["awsvpc_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if awsvpc_configuration is not None:
                self._values["awsvpc_configuration"] = awsvpc_configuration

        @builtins.property
        def awsvpc_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.AwsVpcConfigurationProperty"]]:
            '''The VPC subnets and security groups that are associated with a task.

            .. epigraph::

               All specified subnets and security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-networkconfiguration.html#cfn-ecs-service-networkconfiguration-awsvpcconfiguration
            '''
            result = self._values.get("awsvpc_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.AwsVpcConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.PlacementConstraintProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "expression": "expression"},
    )
    class PlacementConstraintProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            expression: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``PlacementConstraint`` property specifies an object representing a constraint on task placement in the task definition.

            For more information, see `Task Placement Constraints <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param type: The type of constraint. Use ``distinctInstance`` to ensure that each task in a particular group is running on a different container instance. Use ``memberOf`` to restrict the selection to a group of valid candidates.
            :param expression: A cluster query language expression to apply to the constraint. The expression can have a maximum length of 2000 characters. You can't specify an expression if the constraint type is ``distinctInstance`` . For more information, see `Cluster query language <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementconstraint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                placement_constraint_property = ecs.CfnService.PlacementConstraintProperty(
                    type="type",
                
                    # the properties below are optional
                    expression="expression"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__05a9067174cd06772ffa91d63e1f25e9b6d67537f58ed1fc04c0b7b880f9fb06)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if expression is not None:
                self._values["expression"] = expression

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of constraint.

            Use ``distinctInstance`` to ensure that each task in a particular group is running on a different container instance. Use ``memberOf`` to restrict the selection to a group of valid candidates.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementconstraint.html#cfn-ecs-service-placementconstraint-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def expression(self) -> typing.Optional[builtins.str]:
            '''A cluster query language expression to apply to the constraint.

            The expression can have a maximum length of 2000 characters. You can't specify an expression if the constraint type is ``distinctInstance`` . For more information, see `Cluster query language <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementconstraint.html#cfn-ecs-service-placementconstraint-expression
            '''
            result = self._values.get("expression")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementConstraintProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.PlacementStrategyProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "field": "field"},
    )
    class PlacementStrategyProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            field: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``PlacementStrategy`` property specifies the task placement strategy for a task or service.

            For more information, see `Task Placement Strategies <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param type: The type of placement strategy. The ``random`` placement strategy randomly places tasks on available candidates. The ``spread`` placement strategy spreads placement across available candidates evenly based on the ``field`` parameter. The ``binpack`` strategy places tasks on available candidates that have the least available amount of the resource that's specified with the ``field`` parameter. For example, if you binpack on memory, a task is placed on the instance with the least amount of remaining memory but still enough to run the task.
            :param field: The field to apply the placement strategy against. For the ``spread`` placement strategy, valid values are ``instanceId`` (or ``host`` , which has the same effect), or any platform or custom attribute that is applied to a container instance, such as ``attribute:ecs.availability-zone`` . For the ``binpack`` placement strategy, valid values are ``CPU`` and ``MEMORY`` . For the ``random`` placement strategy, this field is not used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementstrategy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                placement_strategy_property = ecs.CfnService.PlacementStrategyProperty(
                    type="type",
                
                    # the properties below are optional
                    field="field"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b1d06da52bf85084d08288d5061162ccbc213d5bfe42e66d4fdbf68354afb1fc)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument field", value=field, expected_type=type_hints["field"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if field is not None:
                self._values["field"] = field

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of placement strategy.

            The ``random`` placement strategy randomly places tasks on available candidates. The ``spread`` placement strategy spreads placement across available candidates evenly based on the ``field`` parameter. The ``binpack`` strategy places tasks on available candidates that have the least available amount of the resource that's specified with the ``field`` parameter. For example, if you binpack on memory, a task is placed on the instance with the least amount of remaining memory but still enough to run the task.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementstrategy.html#cfn-ecs-service-placementstrategy-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def field(self) -> typing.Optional[builtins.str]:
            '''The field to apply the placement strategy against.

            For the ``spread`` placement strategy, valid values are ``instanceId`` (or ``host`` , which has the same effect), or any platform or custom attribute that is applied to a container instance, such as ``attribute:ecs.availability-zone`` . For the ``binpack`` placement strategy, valid values are ``CPU`` and ``MEMORY`` . For the ``random`` placement strategy, this field is not used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementstrategy.html#cfn-ecs-service-placementstrategy-field
            '''
            result = self._values.get("field")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementStrategyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.SecretProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value_from": "valueFrom"},
    )
    class SecretProperty:
        def __init__(self, *, name: builtins.str, value_from: builtins.str) -> None:
            '''An object representing the secret to expose to your container.

            Secrets can be exposed to a container in the following ways:

            - To inject sensitive data into your containers as environment variables, use the ``secrets`` container definition parameter.
            - To reference sensitive information in the log configuration of a container, use the ``secretOptions`` container definition parameter.

            For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param name: The name of the secret.
            :param value_from: The secret to expose to the container. The supported values are either the full ARN of the AWS Secrets Manager secret or the full ARN of the parameter in the SSM Parameter Store. For information about the require AWS Identity and Access Management permissions, see `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-secrets.html#secrets-iam>`_ (for Secrets Manager) or `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-parameters.html>`_ (for Systems Manager Parameter store) in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: If the SSM Parameter Store parameter exists in the same Region as the task you're launching, then you can use either the full ARN or name of the parameter. If the parameter exists in a different Region, then the full ARN must be specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-secret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                secret_property = ecs.CfnService.SecretProperty(
                    name="name",
                    value_from="valueFrom"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f6ff8ad631f969096a18f72ba06774558b189607bf05e1f0d41fcf2e8db6db2c)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value_from", value=value_from, expected_type=type_hints["value_from"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "value_from": value_from,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-secret.html#cfn-ecs-service-secret-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value_from(self) -> builtins.str:
            '''The secret to expose to the container.

            The supported values are either the full ARN of the AWS Secrets Manager secret or the full ARN of the parameter in the SSM Parameter Store.

            For information about the require AWS Identity and Access Management permissions, see `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-secrets.html#secrets-iam>`_ (for Secrets Manager) or `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-parameters.html>`_ (for Systems Manager Parameter store) in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               If the SSM Parameter Store parameter exists in the same Region as the task you're launching, then you can use either the full ARN or name of the parameter. If the parameter exists in a different Region, then the full ARN must be specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-secret.html#cfn-ecs-service-secret-valuefrom
            '''
            result = self._values.get("value_from")
            assert result is not None, "Required property 'value_from' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.ServiceConnectClientAliasProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "dns_name": "dnsName"},
    )
    class ServiceConnectClientAliasProperty:
        def __init__(
            self,
            *,
            port: jsii.Number,
            dns_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Each alias ("endpoint") is a fully-qualified name and port number that other tasks ("clients") can use to connect to this service.

            Each name and port mapping must be unique within the namespace.

            Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param port: The listening port number for the Service Connect proxy. This port is available inside of all of the tasks within the same namespace. To avoid changing your applications in client Amazon ECS services, set this to the same port that the client application uses by default. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param dns_name: The ``dnsName`` is the name that you use in the applications of client tasks to connect to this service. The name must be a valid DNS name but doesn't need to be fully-qualified. The name can include up to 127 characters. The name can include lowercase letters, numbers, underscores (_), hyphens (-), and periods (.). The name can't start with a hyphen. If this parameter isn't specified, the default value of ``discoveryName.namespace`` is used. If the ``discoveryName`` isn't specified, the port mapping name from the task definition is used in ``portName.namespace`` . To avoid changing your applications in client Amazon ECS services, set this to the same name that the client application uses by default. For example, a few common names are ``database`` , ``db`` , or the lowercase name of a database, such as ``mysql`` or ``redis`` . For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectclientalias.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                service_connect_client_alias_property = ecs.CfnService.ServiceConnectClientAliasProperty(
                    port=123,
                
                    # the properties below are optional
                    dns_name="dnsName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7552e3d2b970cfb26552bec096b0680acf8bc8c7b59096bbf5a210dd3266fd35)
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument dns_name", value=dns_name, expected_type=type_hints["dns_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port": port,
            }
            if dns_name is not None:
                self._values["dns_name"] = dns_name

        @builtins.property
        def port(self) -> jsii.Number:
            '''The listening port number for the Service Connect proxy.

            This port is available inside of all of the tasks within the same namespace.

            To avoid changing your applications in client Amazon ECS services, set this to the same port that the client application uses by default. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectclientalias.html#cfn-ecs-service-serviceconnectclientalias-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def dns_name(self) -> typing.Optional[builtins.str]:
            '''The ``dnsName`` is the name that you use in the applications of client tasks to connect to this service.

            The name must be a valid DNS name but doesn't need to be fully-qualified. The name can include up to 127 characters. The name can include lowercase letters, numbers, underscores (_), hyphens (-), and periods (.). The name can't start with a hyphen.

            If this parameter isn't specified, the default value of ``discoveryName.namespace`` is used. If the ``discoveryName`` isn't specified, the port mapping name from the task definition is used in ``portName.namespace`` .

            To avoid changing your applications in client Amazon ECS services, set this to the same name that the client application uses by default. For example, a few common names are ``database`` , ``db`` , or the lowercase name of a database, such as ``mysql`` or ``redis`` . For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectclientalias.html#cfn-ecs-service-serviceconnectclientalias-dnsname
            '''
            result = self._values.get("dns_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceConnectClientAliasProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.ServiceConnectConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "log_configuration": "logConfiguration",
            "namespace": "namespace",
            "services": "services",
        },
    )
    class ServiceConnectConfigurationProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, _IResolvable_da3f097b],
            log_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.LogConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            namespace: typing.Optional[builtins.str] = None,
            services: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.ServiceConnectServiceProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The Service Connect configuration of your Amazon ECS service.

            The configuration for this service to discover and connect to services, and be discovered by, and connected from, other services within a namespace.

            Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param enabled: Specifies whether to use Service Connect with this service.
            :param log_configuration: The log configuration for the container. This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--log-driver`` option to ```docker run`` <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/run/>`_ . By default, containers use the same logging driver that the Docker daemon uses. However, the container might use a different logging driver than the Docker daemon by specifying a log driver configuration in the container definition. For more information about the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the Docker documentation. Understand the following when specifying a log configuration for your containers. - Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the valid values below). Additional log drivers may be available in future releases of the Amazon ECS container agent. - This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. - For tasks that are hosted on Amazon EC2 instances, the Amazon ECS container agent must register the available logging drivers with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS container agent configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* . - For tasks that are on AWS Fargate , because you don't have access to the underlying infrastructure your tasks are hosted on, any additional software needed must be installed outside of the task. For example, the Fluentd output aggregators or a remote host running Logstash to send Gelf logs to.
            :param namespace: The namespace name or full Amazon Resource Name (ARN) of the AWS Cloud Map namespace for use with Service Connect. The namespace must be in the same AWS Region as the Amazon ECS service and cluster. The type of namespace doesn't affect Service Connect. For more information about AWS Cloud Map , see `Working with Services <https://docs.aws.amazon.com/cloud-map/latest/dg/working-with-services.html>`_ in the *AWS Cloud Map Developer Guide* .
            :param services: The list of Service Connect service objects. These are names and aliases (also known as endpoints) that are used by other Amazon ECS services to connect to this service. This field is not required for a "client" Amazon ECS service that's a member of a namespace only to connect to other services within the namespace. An example of this would be a frontend application that accepts incoming requests from either a load balancer that's attached to the service or by other means. An object selects a port from the task definition, assigns a name for the AWS Cloud Map service, and a list of aliases (endpoints) and ports for client applications to refer to this service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                service_connect_configuration_property = ecs.CfnService.ServiceConnectConfigurationProperty(
                    enabled=False,
                
                    # the properties below are optional
                    log_configuration=ecs.CfnService.LogConfigurationProperty(
                        log_driver="logDriver",
                        options={
                            "options_key": "options"
                        },
                        secret_options=[ecs.CfnService.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )]
                    ),
                    namespace="namespace",
                    services=[ecs.CfnService.ServiceConnectServiceProperty(
                        port_name="portName",
                
                        # the properties below are optional
                        client_aliases=[ecs.CfnService.ServiceConnectClientAliasProperty(
                            port=123,
                
                            # the properties below are optional
                            dns_name="dnsName"
                        )],
                        discovery_name="discoveryName",
                        ingress_port_override=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8da75c12a30b7fe7fd27e23d778d5a095611ed616a9d92b56881eb771db7ab84)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument log_configuration", value=log_configuration, expected_type=type_hints["log_configuration"])
                check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
                check_type(argname="argument services", value=services, expected_type=type_hints["services"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "enabled": enabled,
            }
            if log_configuration is not None:
                self._values["log_configuration"] = log_configuration
            if namespace is not None:
                self._values["namespace"] = namespace
            if services is not None:
                self._values["services"] = services

        @builtins.property
        def enabled(self) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Specifies whether to use Service Connect with this service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectconfiguration.html#cfn-ecs-service-serviceconnectconfiguration-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        @builtins.property
        def log_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.LogConfigurationProperty"]]:
            '''The log configuration for the container.

            This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--log-driver`` option to ```docker run`` <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/run/>`_ .

            By default, containers use the same logging driver that the Docker daemon uses. However, the container might use a different logging driver than the Docker daemon by specifying a log driver configuration in the container definition. For more information about the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the Docker documentation.

            Understand the following when specifying a log configuration for your containers.

            - Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the valid values below). Additional log drivers may be available in future releases of the Amazon ECS container agent.
            - This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
            - For tasks that are hosted on Amazon EC2 instances, the Amazon ECS container agent must register the available logging drivers with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS container agent configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            - For tasks that are on AWS Fargate , because you don't have access to the underlying infrastructure your tasks are hosted on, any additional software needed must be installed outside of the task. For example, the Fluentd output aggregators or a remote host running Logstash to send Gelf logs to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectconfiguration.html#cfn-ecs-service-serviceconnectconfiguration-logconfiguration
            '''
            result = self._values.get("log_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnService.LogConfigurationProperty"]], result)

        @builtins.property
        def namespace(self) -> typing.Optional[builtins.str]:
            '''The namespace name or full Amazon Resource Name (ARN) of the AWS Cloud Map namespace for use with Service Connect.

            The namespace must be in the same AWS Region as the Amazon ECS service and cluster. The type of namespace doesn't affect Service Connect. For more information about AWS Cloud Map , see `Working with Services <https://docs.aws.amazon.com/cloud-map/latest/dg/working-with-services.html>`_ in the *AWS Cloud Map Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectconfiguration.html#cfn-ecs-service-serviceconnectconfiguration-namespace
            '''
            result = self._values.get("namespace")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def services(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceConnectServiceProperty"]]]]:
            '''The list of Service Connect service objects.

            These are names and aliases (also known as endpoints) that are used by other Amazon ECS services to connect to this service.

            This field is not required for a "client" Amazon ECS service that's a member of a namespace only to connect to other services within the namespace. An example of this would be a frontend application that accepts incoming requests from either a load balancer that's attached to the service or by other means.

            An object selects a port from the task definition, assigns a name for the AWS Cloud Map service, and a list of aliases (endpoints) and ports for client applications to refer to this service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectconfiguration.html#cfn-ecs-service-serviceconnectconfiguration-services
            '''
            result = self._values.get("services")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceConnectServiceProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceConnectConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.ServiceConnectServiceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "port_name": "portName",
            "client_aliases": "clientAliases",
            "discovery_name": "discoveryName",
            "ingress_port_override": "ingressPortOverride",
        },
    )
    class ServiceConnectServiceProperty:
        def __init__(
            self,
            *,
            port_name: builtins.str,
            client_aliases: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnService.ServiceConnectClientAliasProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            discovery_name: typing.Optional[builtins.str] = None,
            ingress_port_override: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The Service Connect service object configuration.

            For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param port_name: The ``portName`` must match the name of one of the ``portMappings`` from all the containers in the task definition of this Amazon ECS service.
            :param client_aliases: The list of client aliases for this Service Connect service. You use these to assign names that can be used by client applications. The maximum number of client aliases that you can have in this list is 1. Each alias ("endpoint") is a fully-qualified name and port number that other Amazon ECS tasks ("clients") can use to connect to this service. Each name and port mapping must be unique within the namespace. For each ``ServiceConnectService`` , you must provide at least one ``clientAlias`` with one ``port`` .
            :param discovery_name: The ``discoveryName`` is the name of the new AWS Cloud Map service that Amazon ECS creates for this Amazon ECS service. This must be unique within the AWS Cloud Map namespace. The name can contain up to 64 characters. The name can include lowercase letters, numbers, underscores (_), and hyphens (-). The name can't start with a hyphen. If the ``discoveryName`` isn't specified, the port mapping name from the task definition is used in ``portName.namespace`` .
            :param ingress_port_override: The port number for the Service Connect proxy to listen on. Use the value of this field to bypass the proxy for traffic on the port number specified in the named ``portMapping`` in the task definition of this application, and then use it in your VPC security groups to allow traffic into the proxy for this Amazon ECS service. In ``awsvpc`` mode and Fargate, the default value is the container port number. The container port number is in the ``portMapping`` in the task definition. In bridge mode, the default value is the ephemeral port of the Service Connect proxy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectservice.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                service_connect_service_property = ecs.CfnService.ServiceConnectServiceProperty(
                    port_name="portName",
                
                    # the properties below are optional
                    client_aliases=[ecs.CfnService.ServiceConnectClientAliasProperty(
                        port=123,
                
                        # the properties below are optional
                        dns_name="dnsName"
                    )],
                    discovery_name="discoveryName",
                    ingress_port_override=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__047411a3798b037ac5ad218c8ed5492e98c896db58ad86ac1acde082c63059e4)
                check_type(argname="argument port_name", value=port_name, expected_type=type_hints["port_name"])
                check_type(argname="argument client_aliases", value=client_aliases, expected_type=type_hints["client_aliases"])
                check_type(argname="argument discovery_name", value=discovery_name, expected_type=type_hints["discovery_name"])
                check_type(argname="argument ingress_port_override", value=ingress_port_override, expected_type=type_hints["ingress_port_override"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port_name": port_name,
            }
            if client_aliases is not None:
                self._values["client_aliases"] = client_aliases
            if discovery_name is not None:
                self._values["discovery_name"] = discovery_name
            if ingress_port_override is not None:
                self._values["ingress_port_override"] = ingress_port_override

        @builtins.property
        def port_name(self) -> builtins.str:
            '''The ``portName`` must match the name of one of the ``portMappings`` from all the containers in the task definition of this Amazon ECS service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectservice.html#cfn-ecs-service-serviceconnectservice-portname
            '''
            result = self._values.get("port_name")
            assert result is not None, "Required property 'port_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def client_aliases(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceConnectClientAliasProperty"]]]]:
            '''The list of client aliases for this Service Connect service.

            You use these to assign names that can be used by client applications. The maximum number of client aliases that you can have in this list is 1.

            Each alias ("endpoint") is a fully-qualified name and port number that other Amazon ECS tasks ("clients") can use to connect to this service.

            Each name and port mapping must be unique within the namespace.

            For each ``ServiceConnectService`` , you must provide at least one ``clientAlias`` with one ``port`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectservice.html#cfn-ecs-service-serviceconnectservice-clientaliases
            '''
            result = self._values.get("client_aliases")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnService.ServiceConnectClientAliasProperty"]]]], result)

        @builtins.property
        def discovery_name(self) -> typing.Optional[builtins.str]:
            '''The ``discoveryName`` is the name of the new AWS Cloud Map service that Amazon ECS creates for this Amazon ECS service.

            This must be unique within the AWS Cloud Map namespace. The name can contain up to 64 characters. The name can include lowercase letters, numbers, underscores (_), and hyphens (-). The name can't start with a hyphen.

            If the ``discoveryName`` isn't specified, the port mapping name from the task definition is used in ``portName.namespace`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectservice.html#cfn-ecs-service-serviceconnectservice-discoveryname
            '''
            result = self._values.get("discovery_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ingress_port_override(self) -> typing.Optional[jsii.Number]:
            '''The port number for the Service Connect proxy to listen on.

            Use the value of this field to bypass the proxy for traffic on the port number specified in the named ``portMapping`` in the task definition of this application, and then use it in your VPC security groups to allow traffic into the proxy for this Amazon ECS service.

            In ``awsvpc`` mode and Fargate, the default value is the container port number. The container port number is in the ``portMapping`` in the task definition. In bridge mode, the default value is the ephemeral port of the Service Connect proxy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceconnectservice.html#cfn-ecs-service-serviceconnectservice-ingressportoverride
            '''
            result = self._values.get("ingress_port_override")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceConnectServiceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnService.ServiceRegistryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "container_port": "containerPort",
            "port": "port",
            "registry_arn": "registryArn",
        },
    )
    class ServiceRegistryProperty:
        def __init__(
            self,
            *,
            container_name: typing.Optional[builtins.str] = None,
            container_port: typing.Optional[jsii.Number] = None,
            port: typing.Optional[jsii.Number] = None,
            registry_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ServiceRegistry`` property specifies details of the service registry.

            For more information, see `Service Discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param container_name: The container name value to be used for your service discovery service. It's already specified in the task definition. If the task definition that your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition that your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.
            :param container_port: The port value to be used for your service discovery service. It's already specified in the task definition. If the task definition your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.
            :param port: The port value used if your service discovery service specified an SRV record. This field might be used if both the ``awsvpc`` network mode and SRV records are used.
            :param registry_arn: The Amazon Resource Name (ARN) of the service registry. The currently supported service registry is AWS Cloud Map . For more information, see `CreateService <https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                service_registry_property = ecs.CfnService.ServiceRegistryProperty(
                    container_name="containerName",
                    container_port=123,
                    port=123,
                    registry_arn="registryArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7673ebf2b080281fae8bd32d25189b0105509e993f04b7511342452dfc1d681b)
                check_type(argname="argument container_name", value=container_name, expected_type=type_hints["container_name"])
                check_type(argname="argument container_port", value=container_port, expected_type=type_hints["container_port"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument registry_arn", value=registry_arn, expected_type=type_hints["registry_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if container_name is not None:
                self._values["container_name"] = container_name
            if container_port is not None:
                self._values["container_port"] = container_port
            if port is not None:
                self._values["port"] = port
            if registry_arn is not None:
                self._values["registry_arn"] = registry_arn

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''The container name value to be used for your service discovery service.

            It's already specified in the task definition. If the task definition that your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition that your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''The port value to be used for your service discovery service.

            It's already specified in the task definition. If the task definition your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port value used if your service discovery service specified an SRV record.

            This field might be used if both the ``awsvpc`` network mode and SRV records are used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def registry_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the service registry.

            The currently supported service registry is AWS Cloud Map . For more information, see `CreateService <https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-registryarn
            '''
            result = self._values.get("registry_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceRegistryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.CfnServiceProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_provider_strategy": "capacityProviderStrategy",
        "cluster": "cluster",
        "deployment_configuration": "deploymentConfiguration",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableEcsManagedTags",
        "enable_execute_command": "enableExecuteCommand",
        "health_check_grace_period_seconds": "healthCheckGracePeriodSeconds",
        "launch_type": "launchType",
        "load_balancers": "loadBalancers",
        "network_configuration": "networkConfiguration",
        "placement_constraints": "placementConstraints",
        "placement_strategies": "placementStrategies",
        "platform_version": "platformVersion",
        "propagate_tags": "propagateTags",
        "role": "role",
        "scheduling_strategy": "schedulingStrategy",
        "service_connect_configuration": "serviceConnectConfiguration",
        "service_name": "serviceName",
        "service_registries": "serviceRegistries",
        "tags": "tags",
        "task_definition": "taskDefinition",
    },
)
class CfnServiceProps:
    def __init__(
        self,
        *,
        capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnService.CapacityProviderStrategyItemProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        cluster: typing.Optional[builtins.str] = None,
        deployment_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnService.DeploymentConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        deployment_controller: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnService.DeploymentControllerProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        enable_execute_command: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        health_check_grace_period_seconds: typing.Optional[jsii.Number] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnService.LoadBalancerProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        network_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnService.NetworkConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnService.PlacementConstraintProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        placement_strategies: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnService.PlacementStrategyProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        propagate_tags: typing.Optional[builtins.str] = None,
        role: typing.Optional[builtins.str] = None,
        scheduling_strategy: typing.Optional[builtins.str] = None,
        service_connect_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnService.ServiceConnectConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        service_name: typing.Optional[builtins.str] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnService.ServiceRegistryProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        task_definition: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnService``.

        :param capacity_provider_strategy: The capacity provider strategy to use for the service. If a ``capacityProviderStrategy`` is specified, the ``launchType`` parameter must be omitted. If no ``capacityProviderStrategy`` or ``launchType`` is specified, the ``defaultCapacityProviderStrategy`` for the cluster is used. A capacity provider strategy may contain a maximum of 6 capacity providers.
        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that you run your service on. If you do not specify a cluster, the default cluster is assumed.
        :param deployment_configuration: Optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.
        :param deployment_controller: The deployment controller to use for the service. If no deployment controller is specified, the default value of ``ECS`` is used.
        :param desired_count: The number of instantiations of the specified task definition to place and keep running in your service. For new services, if a desired count is not specified, a default value of ``1`` is used. When using the ``DAEMON`` scheduling strategy, the desired count is not required. For existing services, if a desired count is not specified, it is omitted from the operation.
        :param enable_ecs_managed_tags: Specifies whether to turn on Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging your Amazon ECS resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ in the *Amazon Elastic Container Service Developer Guide* . When you use Amazon ECS managed tags, you need to set the ``propagateTags`` request parameter.
        :param enable_execute_command: Determines whether the execute command functionality is turned on for the service. If ``true`` , the execute command functionality is turned on for all containers in tasks as part of the service.
        :param health_check_grace_period_seconds: The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. This is only used when your service is configured to use a load balancer. If your service has a load balancer defined and you don't specify a health check grace period value, the default value of ``0`` is used. If you do not use an Elastic Load Balancing, we recommend that you use the ``startPeriod`` in the task definition health check parameters. For more information, see `Health check <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_HealthCheck.html>`_ . If your service's tasks take a while to start and respond to Elastic Load Balancing health checks, you can specify a health check grace period of up to 2,147,483,647 seconds (about 69 years). During that time, the Amazon ECS service scheduler ignores health check status. This grace period can prevent the service scheduler from marking tasks as unhealthy and stopping them before they have time to come up.
        :param launch_type: The launch type on which to run your service. For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param load_balancers: A list of load balancer objects to associate with the service. If you specify the ``Role`` property, ``LoadBalancers`` must be specified as well. For information about the number of load balancers that you can specify per service, see `Service Load Balancing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-load-balancing.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param network_configuration: The network configuration for the service. This parameter is required for task definitions that use the ``awsvpc`` network mode to receive their own elastic network interface, and it is not supported for other network modes. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param placement_constraints: An array of placement constraint objects to use for tasks in your service. You can specify a maximum of 10 constraints for each task. This limit includes constraints in the task definition and those specified at runtime.
        :param placement_strategies: The placement strategy objects to use for tasks in your service. You can specify a maximum of 5 strategy rules for each service.
        :param platform_version: The platform version that your tasks in the service are running on. A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param propagate_tags: Specifies whether to propagate the tags from the task definition to the task. If no value is specified, the tags aren't propagated. Tags can only be propagated to the task during task creation. To add tags to a task after task creation, use the `TagResource <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html>`_ API action. The default is ``NONE`` .
        :param role: The name or full Amazon Resource Name (ARN) of the IAM role that allows Amazon ECS to make calls to your load balancer on your behalf. This parameter is only permitted if you are using a load balancer with your service and your task definition doesn't use the ``awsvpc`` network mode. If you specify the ``role`` parameter, you must also specify a load balancer object with the ``loadBalancers`` parameter. .. epigraph:: If your account has already created the Amazon ECS service-linked role, that role is used for your service unless you specify a role here. The service-linked role is required if your task definition uses the ``awsvpc`` network mode or if the service is configured to use service discovery, an external deployment controller, multiple target groups, or Elastic Inference accelerators in which case you don't specify a role here. For more information, see `Using service-linked roles for Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If your specified role has a path other than ``/`` , then you must either specify the full role ARN (this is recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` then you would specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* .
        :param scheduling_strategy: The scheduling strategy to use for the service. For more information, see `Services <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html>`_ . There are two service scheduler strategies available: - ``REPLICA`` -The replica scheduling strategy places and maintains the desired number of tasks across your cluster. By default, the service scheduler spreads tasks across Availability Zones. You can use task placement strategies and constraints to customize task placement decisions. This scheduler strategy is required if the service uses the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types. - ``DAEMON`` -The daemon scheduling strategy deploys exactly one task on each active container instance that meets all of the task placement constraints that you specify in your cluster. The service scheduler also evaluates the task placement constraints for running tasks and will stop tasks that don't meet the placement constraints. When you're using this strategy, you don't need to specify a desired number of tasks, a task placement strategy, or use Service Auto Scaling policies. .. epigraph:: Tasks using the Fargate launch type or the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types don't support the ``DAEMON`` scheduling strategy.
        :param service_connect_configuration: The configuration for this service to discover and connect to services, and be discovered by, and connected from, other services within a namespace. Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param service_name: The name of your service. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. Service names must be unique within a cluster, but you can have similarly named services in multiple clusters within a Region or across multiple Regions. .. epigraph:: The stack update fails if you change any properties that require replacement and the ``ServiceName`` is configured. This is because AWS CloudFormation creates the replacement service first, but each ``ServiceName`` must be unique in the cluster.
        :param service_registries: The details of the service discovery registry to associate with this service. For more information, see `Service discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ . .. epigraph:: Each service may be associated with one service registry. Multiple service registries for each service isn't supported.
        :param tags: The metadata that you apply to the service to help you categorize and organize them. Each tag consists of a key and an optional value, both of which you define. When a service is deleted, the tags are deleted as well. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        :param task_definition: The ``family`` and ``revision`` ( ``family:revision`` ) or full ARN of the task definition to run in your service. If a ``revision`` isn't specified, the latest ``ACTIVE`` revision is used. A task definition must be specified if the service uses either the ``ECS`` or ``CODE_DEPLOY`` deployment controllers. For more information about deployment types, see `Amazon ECS deployment types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            
            cfn_service_props = ecs.CfnServiceProps(
                capacity_provider_strategy=[ecs.CfnService.CapacityProviderStrategyItemProperty(
                    base=123,
                    capacity_provider="capacityProvider",
                    weight=123
                )],
                cluster="cluster",
                deployment_configuration=ecs.CfnService.DeploymentConfigurationProperty(
                    alarms=ecs.CfnService.DeploymentAlarmsProperty(
                        alarm_names=["alarmNames"],
                        enable=False,
                        rollback=False
                    ),
                    deployment_circuit_breaker=ecs.CfnService.DeploymentCircuitBreakerProperty(
                        enable=False,
                        rollback=False
                    ),
                    maximum_percent=123,
                    minimum_healthy_percent=123
                ),
                deployment_controller=ecs.CfnService.DeploymentControllerProperty(
                    type="type"
                ),
                desired_count=123,
                enable_ecs_managed_tags=False,
                enable_execute_command=False,
                health_check_grace_period_seconds=123,
                launch_type="launchType",
                load_balancers=[ecs.CfnService.LoadBalancerProperty(
                    container_port=123,
            
                    # the properties below are optional
                    container_name="containerName",
                    load_balancer_name="loadBalancerName",
                    target_group_arn="targetGroupArn"
                )],
                network_configuration=ecs.CfnService.NetworkConfigurationProperty(
                    awsvpc_configuration=ecs.CfnService.AwsVpcConfigurationProperty(
                        subnets=["subnets"],
            
                        # the properties below are optional
                        assign_public_ip="assignPublicIp",
                        security_groups=["securityGroups"]
                    )
                ),
                placement_constraints=[ecs.CfnService.PlacementConstraintProperty(
                    type="type",
            
                    # the properties below are optional
                    expression="expression"
                )],
                placement_strategies=[ecs.CfnService.PlacementStrategyProperty(
                    type="type",
            
                    # the properties below are optional
                    field="field"
                )],
                platform_version="platformVersion",
                propagate_tags="propagateTags",
                role="role",
                scheduling_strategy="schedulingStrategy",
                service_connect_configuration=ecs.CfnService.ServiceConnectConfigurationProperty(
                    enabled=False,
            
                    # the properties below are optional
                    log_configuration=ecs.CfnService.LogConfigurationProperty(
                        log_driver="logDriver",
                        options={
                            "options_key": "options"
                        },
                        secret_options=[ecs.CfnService.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )]
                    ),
                    namespace="namespace",
                    services=[ecs.CfnService.ServiceConnectServiceProperty(
                        port_name="portName",
            
                        # the properties below are optional
                        client_aliases=[ecs.CfnService.ServiceConnectClientAliasProperty(
                            port=123,
            
                            # the properties below are optional
                            dns_name="dnsName"
                        )],
                        discovery_name="discoveryName",
                        ingress_port_override=123
                    )]
                ),
                service_name="serviceName",
                service_registries=[ecs.CfnService.ServiceRegistryProperty(
                    container_name="containerName",
                    container_port=123,
                    port=123,
                    registry_arn="registryArn"
                )],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                task_definition="taskDefinition"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7f93658ccdbf3f250d0d1ce12224e9eddaef71a8f664c6a279122f60df5b8e83)
            check_type(argname="argument capacity_provider_strategy", value=capacity_provider_strategy, expected_type=type_hints["capacity_provider_strategy"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument deployment_configuration", value=deployment_configuration, expected_type=type_hints["deployment_configuration"])
            check_type(argname="argument deployment_controller", value=deployment_controller, expected_type=type_hints["deployment_controller"])
            check_type(argname="argument desired_count", value=desired_count, expected_type=type_hints["desired_count"])
            check_type(argname="argument enable_ecs_managed_tags", value=enable_ecs_managed_tags, expected_type=type_hints["enable_ecs_managed_tags"])
            check_type(argname="argument enable_execute_command", value=enable_execute_command, expected_type=type_hints["enable_execute_command"])
            check_type(argname="argument health_check_grace_period_seconds", value=health_check_grace_period_seconds, expected_type=type_hints["health_check_grace_period_seconds"])
            check_type(argname="argument launch_type", value=launch_type, expected_type=type_hints["launch_type"])
            check_type(argname="argument load_balancers", value=load_balancers, expected_type=type_hints["load_balancers"])
            check_type(argname="argument network_configuration", value=network_configuration, expected_type=type_hints["network_configuration"])
            check_type(argname="argument placement_constraints", value=placement_constraints, expected_type=type_hints["placement_constraints"])
            check_type(argname="argument placement_strategies", value=placement_strategies, expected_type=type_hints["placement_strategies"])
            check_type(argname="argument platform_version", value=platform_version, expected_type=type_hints["platform_version"])
            check_type(argname="argument propagate_tags", value=propagate_tags, expected_type=type_hints["propagate_tags"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument scheduling_strategy", value=scheduling_strategy, expected_type=type_hints["scheduling_strategy"])
            check_type(argname="argument service_connect_configuration", value=service_connect_configuration, expected_type=type_hints["service_connect_configuration"])
            check_type(argname="argument service_name", value=service_name, expected_type=type_hints["service_name"])
            check_type(argname="argument service_registries", value=service_registries, expected_type=type_hints["service_registries"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument task_definition", value=task_definition, expected_type=type_hints["task_definition"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if capacity_provider_strategy is not None:
            self._values["capacity_provider_strategy"] = capacity_provider_strategy
        if cluster is not None:
            self._values["cluster"] = cluster
        if deployment_configuration is not None:
            self._values["deployment_configuration"] = deployment_configuration
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if enable_execute_command is not None:
            self._values["enable_execute_command"] = enable_execute_command
        if health_check_grace_period_seconds is not None:
            self._values["health_check_grace_period_seconds"] = health_check_grace_period_seconds
        if launch_type is not None:
            self._values["launch_type"] = launch_type
        if load_balancers is not None:
            self._values["load_balancers"] = load_balancers
        if network_configuration is not None:
            self._values["network_configuration"] = network_configuration
        if placement_constraints is not None:
            self._values["placement_constraints"] = placement_constraints
        if placement_strategies is not None:
            self._values["placement_strategies"] = placement_strategies
        if platform_version is not None:
            self._values["platform_version"] = platform_version
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if role is not None:
            self._values["role"] = role
        if scheduling_strategy is not None:
            self._values["scheduling_strategy"] = scheduling_strategy
        if service_connect_configuration is not None:
            self._values["service_connect_configuration"] = service_connect_configuration
        if service_name is not None:
            self._values["service_name"] = service_name
        if service_registries is not None:
            self._values["service_registries"] = service_registries
        if tags is not None:
            self._values["tags"] = tags
        if task_definition is not None:
            self._values["task_definition"] = task_definition

    @builtins.property
    def capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.CapacityProviderStrategyItemProperty]]]]:
        '''The capacity provider strategy to use for the service.

        If a ``capacityProviderStrategy`` is specified, the ``launchType`` parameter must be omitted. If no ``capacityProviderStrategy`` or ``launchType`` is specified, the ``defaultCapacityProviderStrategy`` for the cluster is used.

        A capacity provider strategy may contain a maximum of 6 capacity providers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-capacityproviderstrategy
        '''
        result = self._values.get("capacity_provider_strategy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.CapacityProviderStrategyItemProperty]]]], result)

    @builtins.property
    def cluster(self) -> typing.Optional[builtins.str]:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that you run your service on.

        If you do not specify a cluster, the default cluster is assumed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-cluster
        '''
        result = self._values.get("cluster")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deployment_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnService.DeploymentConfigurationProperty]]:
        '''Optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentconfiguration
        '''
        result = self._values.get("deployment_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnService.DeploymentConfigurationProperty]], result)

    @builtins.property
    def deployment_controller(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnService.DeploymentControllerProperty]]:
        '''The deployment controller to use for the service.

        If no deployment controller is specified, the default value of ``ECS`` is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentcontroller
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnService.DeploymentControllerProperty]], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''The number of instantiations of the specified task definition to place and keep running in your service.

        For new services, if a desired count is not specified, a default value of ``1`` is used. When using the ``DAEMON`` scheduling strategy, the desired count is not required.

        For existing services, if a desired count is not specified, it is omitted from the operation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-desiredcount
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to turn on Amazon ECS managed tags for the tasks within the service.

        For more information, see `Tagging your Amazon ECS resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        When you use Amazon ECS managed tags, you need to set the ``propagateTags`` request parameter.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableecsmanagedtags
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def enable_execute_command(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Determines whether the execute command functionality is turned on for the service.

        If ``true`` , the execute command functionality is turned on for all containers in tasks as part of the service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableexecutecommand
        '''
        result = self._values.get("enable_execute_command")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def health_check_grace_period_seconds(self) -> typing.Optional[jsii.Number]:
        '''The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        This is only used when your service is configured to use a load balancer. If your service has a load balancer defined and you don't specify a health check grace period value, the default value of ``0`` is used.

        If you do not use an Elastic Load Balancing, we recommend that you use the ``startPeriod`` in the task definition health check parameters. For more information, see `Health check <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_HealthCheck.html>`_ .

        If your service's tasks take a while to start and respond to Elastic Load Balancing health checks, you can specify a health check grace period of up to 2,147,483,647 seconds (about 69 years). During that time, the Amazon ECS service scheduler ignores health check status. This grace period can prevent the service scheduler from marking tasks as unhealthy and stopping them before they have time to come up.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-healthcheckgraceperiodseconds
        '''
        result = self._values.get("health_check_grace_period_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''The launch type on which to run your service.

        For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-launchtype
        '''
        result = self._values.get("launch_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.LoadBalancerProperty]]]]:
        '''A list of load balancer objects to associate with the service.

        If you specify the ``Role`` property, ``LoadBalancers`` must be specified as well. For information about the number of load balancers that you can specify per service, see `Service Load Balancing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-load-balancing.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-loadbalancers
        '''
        result = self._values.get("load_balancers")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.LoadBalancerProperty]]]], result)

    @builtins.property
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnService.NetworkConfigurationProperty]]:
        '''The network configuration for the service.

        This parameter is required for task definitions that use the ``awsvpc`` network mode to receive their own elastic network interface, and it is not supported for other network modes. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-networkconfiguration
        '''
        result = self._values.get("network_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnService.NetworkConfigurationProperty]], result)

    @builtins.property
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.PlacementConstraintProperty]]]]:
        '''An array of placement constraint objects to use for tasks in your service.

        You can specify a maximum of 10 constraints for each task. This limit includes constraints in the task definition and those specified at runtime.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementconstraints
        '''
        result = self._values.get("placement_constraints")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.PlacementConstraintProperty]]]], result)

    @builtins.property
    def placement_strategies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.PlacementStrategyProperty]]]]:
        '''The placement strategy objects to use for tasks in your service.

        You can specify a maximum of 5 strategy rules for each service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementstrategies
        '''
        result = self._values.get("placement_strategies")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.PlacementStrategyProperty]]]], result)

    @builtins.property
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''The platform version that your tasks in the service are running on.

        A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-platformversion
        '''
        result = self._values.get("platform_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional[builtins.str]:
        '''Specifies whether to propagate the tags from the task definition to the task.

        If no value is specified, the tags aren't propagated. Tags can only be propagated to the task during task creation. To add tags to a task after task creation, use the `TagResource <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html>`_ API action.

        The default is ``NONE`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-propagatetags
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def role(self) -> typing.Optional[builtins.str]:
        '''The name or full Amazon Resource Name (ARN) of the IAM role that allows Amazon ECS to make calls to your load balancer on your behalf.

        This parameter is only permitted if you are using a load balancer with your service and your task definition doesn't use the ``awsvpc`` network mode. If you specify the ``role`` parameter, you must also specify a load balancer object with the ``loadBalancers`` parameter.
        .. epigraph::

           If your account has already created the Amazon ECS service-linked role, that role is used for your service unless you specify a role here. The service-linked role is required if your task definition uses the ``awsvpc`` network mode or if the service is configured to use service discovery, an external deployment controller, multiple target groups, or Elastic Inference accelerators in which case you don't specify a role here. For more information, see `Using service-linked roles for Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        If your specified role has a path other than ``/`` , then you must either specify the full role ARN (this is recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` then you would specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-role
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scheduling_strategy(self) -> typing.Optional[builtins.str]:
        '''The scheduling strategy to use for the service. For more information, see `Services <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html>`_ .

        There are two service scheduler strategies available:

        - ``REPLICA`` -The replica scheduling strategy places and maintains the desired number of tasks across your cluster. By default, the service scheduler spreads tasks across Availability Zones. You can use task placement strategies and constraints to customize task placement decisions. This scheduler strategy is required if the service uses the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types.
        - ``DAEMON`` -The daemon scheduling strategy deploys exactly one task on each active container instance that meets all of the task placement constraints that you specify in your cluster. The service scheduler also evaluates the task placement constraints for running tasks and will stop tasks that don't meet the placement constraints. When you're using this strategy, you don't need to specify a desired number of tasks, a task placement strategy, or use Service Auto Scaling policies.

        .. epigraph::

           Tasks using the Fargate launch type or the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types don't support the ``DAEMON`` scheduling strategy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-schedulingstrategy
        '''
        result = self._values.get("scheduling_strategy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_connect_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnService.ServiceConnectConfigurationProperty]]:
        '''The configuration for this service to discover and connect to services, and be discovered by, and connected from, other services within a namespace.

        Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-serviceconnectconfiguration
        '''
        result = self._values.get("service_connect_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnService.ServiceConnectConfigurationProperty]], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''The name of your service.

        Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. Service names must be unique within a cluster, but you can have similarly named services in multiple clusters within a Region or across multiple Regions.
        .. epigraph::

           The stack update fails if you change any properties that require replacement and the ``ServiceName`` is configured. This is because AWS CloudFormation creates the replacement service first, but each ``ServiceName`` must be unique in the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-servicename
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.ServiceRegistryProperty]]]]:
        '''The details of the service discovery registry to associate with this service. For more information, see `Service discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ .

        .. epigraph::

           Each service may be associated with one service registry. Multiple service registries for each service isn't supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-serviceregistries
        '''
        result = self._values.get("service_registries")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnService.ServiceRegistryProperty]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The metadata that you apply to the service to help you categorize and organize them.

        Each tag consists of a key and an optional value, both of which you define. When a service is deleted, the tags are deleted as well.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def task_definition(self) -> typing.Optional[builtins.str]:
        '''The ``family`` and ``revision`` ( ``family:revision`` ) or full ARN of the task definition to run in your service.

        If a ``revision`` isn't specified, the latest ``ACTIVE`` revision is used.

        A task definition must be specified if the service uses either the ``ECS`` or ``CODE_DEPLOY`` deployment controllers.

        For more information about deployment types, see `Amazon ECS deployment types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-taskdefinition
        '''
        result = self._values.get("task_definition")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnTaskDefinition(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition",
):
    '''A CloudFormation ``AWS::ECS::TaskDefinition``.

    The details of a task definition which describes the container and volume definitions of an Amazon Elastic Container Service task. You can specify which Docker images to use, the required resources, and other configurations related to launching the task definition through an Amazon ECS service or task.

    :cloudformationResource: AWS::ECS::TaskDefinition
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ecs as ecs
        
        cfn_task_definition = ecs.CfnTaskDefinition(self, "MyCfnTaskDefinition",
            container_definitions=[ecs.CfnTaskDefinition.ContainerDefinitionProperty(
                image="image",
                name="name",
        
                # the properties below are optional
                command=["command"],
                cpu=123,
                depends_on=[ecs.CfnTaskDefinition.ContainerDependencyProperty(
                    condition="condition",
                    container_name="containerName"
                )],
                disable_networking=False,
                dns_search_domains=["dnsSearchDomains"],
                dns_servers=["dnsServers"],
                docker_labels={
                    "docker_labels_key": "dockerLabels"
                },
                docker_security_options=["dockerSecurityOptions"],
                entry_point=["entryPoint"],
                environment=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                    name="name",
                    value="value"
                )],
                environment_files=[ecs.CfnTaskDefinition.EnvironmentFileProperty(
                    type="type",
                    value="value"
                )],
                essential=False,
                extra_hosts=[ecs.CfnTaskDefinition.HostEntryProperty(
                    hostname="hostname",
                    ip_address="ipAddress"
                )],
                firelens_configuration=ecs.CfnTaskDefinition.FirelensConfigurationProperty(
                    options={
                        "options_key": "options"
                    },
                    type="type"
                ),
                health_check=ecs.CfnTaskDefinition.HealthCheckProperty(
                    command=["command"],
                    interval=123,
                    retries=123,
                    start_period=123,
                    timeout=123
                ),
                hostname="hostname",
                interactive=False,
                links=["links"],
                linux_parameters=ecs.CfnTaskDefinition.LinuxParametersProperty(
                    capabilities=ecs.CfnTaskDefinition.KernelCapabilitiesProperty(
                        add=["add"],
                        drop=["drop"]
                    ),
                    devices=[ecs.CfnTaskDefinition.DeviceProperty(
                        container_path="containerPath",
                        host_path="hostPath",
                        permissions=["permissions"]
                    )],
                    init_process_enabled=False,
                    max_swap=123,
                    shared_memory_size=123,
                    swappiness=123,
                    tmpfs=[ecs.CfnTaskDefinition.TmpfsProperty(
                        size=123,
        
                        # the properties below are optional
                        container_path="containerPath",
                        mount_options=["mountOptions"]
                    )]
                ),
                log_configuration=ecs.CfnTaskDefinition.LogConfigurationProperty(
                    log_driver="logDriver",
        
                    # the properties below are optional
                    options={
                        "options_key": "options"
                    },
                    secret_options=[ecs.CfnTaskDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )]
                ),
                memory=123,
                memory_reservation=123,
                mount_points=[ecs.CfnTaskDefinition.MountPointProperty(
                    container_path="containerPath",
                    read_only=False,
                    source_volume="sourceVolume"
                )],
                port_mappings=[ecs.CfnTaskDefinition.PortMappingProperty(
                    app_protocol="appProtocol",
                    container_port=123,
                    container_port_range="containerPortRange",
                    host_port=123,
                    name="name",
                    protocol="protocol"
                )],
                privileged=False,
                pseudo_terminal=False,
                readonly_root_filesystem=False,
                repository_credentials=ecs.CfnTaskDefinition.RepositoryCredentialsProperty(
                    credentials_parameter="credentialsParameter"
                ),
                resource_requirements=[ecs.CfnTaskDefinition.ResourceRequirementProperty(
                    type="type",
                    value="value"
                )],
                secrets=[ecs.CfnTaskDefinition.SecretProperty(
                    name="name",
                    value_from="valueFrom"
                )],
                start_timeout=123,
                stop_timeout=123,
                system_controls=[ecs.CfnTaskDefinition.SystemControlProperty(
                    namespace="namespace",
                    value="value"
                )],
                ulimits=[ecs.CfnTaskDefinition.UlimitProperty(
                    hard_limit=123,
                    name="name",
                    soft_limit=123
                )],
                user="user",
                volumes_from=[ecs.CfnTaskDefinition.VolumeFromProperty(
                    read_only=False,
                    source_container="sourceContainer"
                )],
                working_directory="workingDirectory"
            )],
            cpu="cpu",
            ephemeral_storage=ecs.CfnTaskDefinition.EphemeralStorageProperty(
                size_in_gi_b=123
            ),
            execution_role_arn="executionRoleArn",
            family="family",
            inference_accelerators=[ecs.CfnTaskDefinition.InferenceAcceleratorProperty(
                device_name="deviceName",
                device_type="deviceType"
            )],
            ipc_mode="ipcMode",
            memory="memory",
            network_mode="networkMode",
            pid_mode="pidMode",
            placement_constraints=[ecs.CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty(
                type="type",
        
                # the properties below are optional
                expression="expression"
            )],
            proxy_configuration=ecs.CfnTaskDefinition.ProxyConfigurationProperty(
                container_name="containerName",
        
                # the properties below are optional
                proxy_configuration_properties=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                    name="name",
                    value="value"
                )],
                type="type"
            ),
            requires_compatibilities=["requiresCompatibilities"],
            runtime_platform=ecs.CfnTaskDefinition.RuntimePlatformProperty(
                cpu_architecture="cpuArchitecture",
                operating_system_family="operatingSystemFamily"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            task_role_arn="taskRoleArn",
            volumes=[ecs.CfnTaskDefinition.VolumeProperty(
                docker_volume_configuration=ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty(
                    autoprovision=False,
                    driver="driver",
                    driver_opts={
                        "driver_opts_key": "driverOpts"
                    },
                    labels={
                        "labels_key": "labels"
                    },
                    scope="scope"
                ),
                efs_volume_configuration=ecs.CfnTaskDefinition.EFSVolumeConfigurationProperty(
                    filesystem_id="filesystemId",
        
                    # the properties below are optional
                    authorization_config=ecs.CfnTaskDefinition.AuthorizationConfigProperty(
                        access_point_id="accessPointId",
                        iam="iam"
                    ),
                    root_directory="rootDirectory",
                    transit_encryption="transitEncryption",
                    transit_encryption_port=123
                ),
                host=ecs.CfnTaskDefinition.HostVolumePropertiesProperty(
                    source_path="sourcePath"
                ),
                name="name"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        container_definitions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.ContainerDefinitionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        cpu: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.EphemeralStorageProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        execution_role_arn: typing.Optional[builtins.str] = None,
        family: typing.Optional[builtins.str] = None,
        inference_accelerators: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.InferenceAcceleratorProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ipc_mode: typing.Optional[builtins.str] = None,
        memory: typing.Optional[builtins.str] = None,
        network_mode: typing.Optional[builtins.str] = None,
        pid_mode: typing.Optional[builtins.str] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        proxy_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.ProxyConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        requires_compatibilities: typing.Optional[typing.Sequence[builtins.str]] = None,
        runtime_platform: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.RuntimePlatformProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        task_role_arn: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.VolumeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::TaskDefinition``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param container_definitions: A list of container definitions in JSON format that describe the different containers that make up your task. For more information about container definition parameters and defaults, see `Amazon ECS Task Definitions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param cpu: The number of ``cpu`` units used by the task. If you use the EC2 launch type, this field is optional. Any value can be used. If you use the Fargate launch type, this field is required. You must use one of the following values. The value that you choose determines your range of valid values for the ``memory`` parameter. The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate. - 256 (.25 vCPU) - Available ``memory`` values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - 512 (.5 vCPU) - Available ``memory`` values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - 1024 (1 vCPU) - Available ``memory`` values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - 2048 (2 vCPU) - Available ``memory`` values: 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - 4096 (4 vCPU) - Available ``memory`` values: 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - 8192 (8 vCPU) - Available ``memory`` values: 16 GB and 60 GB in 4 GB increments This option requires Linux platform ``1.4.0`` or later. - 16384 (16vCPU) - Available ``memory`` values: 32GB and 120 GB in 8 GB increments This option requires Linux platform ``1.4.0`` or later.
        :param ephemeral_storage: The ephemeral storage settings to use for tasks run with the task definition.
        :param execution_role_arn: The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent permission to make AWS API calls on your behalf. The task execution IAM role is required depending on the requirements of your task. For more information, see `Amazon ECS task execution IAM role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_execution_IAM_role.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param family: The name of a family that this task definition is registered to. Up to 255 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed. A family groups multiple versions of a task definition. Amazon ECS gives the first task definition that you registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers to each task definition that you add. .. epigraph:: To use revision numbers when you update a task definition, specify this property. If you don't specify a value, AWS CloudFormation generates a new task definition each time that you update it.
        :param inference_accelerators: The Elastic Inference accelerators to use for the containers in the task.
        :param ipc_mode: The IPC resource namespace to use for the containers in the task. The valid values are ``host`` , ``task`` , or ``none`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same IPC resources. If ``none`` is specified, then IPC resources within the containers of a task are private and not shared with other containers in a task or on the container instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the *Docker run reference* . If the ``host`` IPC mode is used, be aware that there is a heightened risk of undesired IPC namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ . If you are setting namespaced kernel parameters using ``systemControls`` for the containers in the task, the following will apply to your IPC resource namespace. For more information, see `System Controls <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html>`_ in the *Amazon Elastic Container Service Developer Guide* . - For tasks that use the ``host`` IPC mode, IPC namespace related ``systemControls`` are not supported. - For tasks that use the ``task`` IPC mode, IPC namespace related ``systemControls`` will apply to all containers within a task. .. epigraph:: This parameter is not supported for Windows containers or tasks run on AWS Fargate .
        :param memory: The amount (in MiB) of memory used by the task. If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a container-level memory value. This field is optional and any value can be used. If a task-level memory value is specified, the container-level memory value is optional. For more information regarding container-level memory and memory reservation, see `ContainerDefinition <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html>`_ . If your tasks runs on AWS Fargate , this field is required. You must use one of the following values. The value you choose determines your range of valid values for the ``cpu`` parameter. - 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available ``cpu`` values: 256 (.25 vCPU) - 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available ``cpu`` values: 512 (.5 vCPU) - 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available ``cpu`` values: 1024 (1 vCPU) - Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 2048 (2 vCPU) - Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 4096 (4 vCPU) - Between 16 GB and 60 GB in 4 GB increments - Available ``cpu`` values: 8192 (8 vCPU) This option requires Linux platform ``1.4.0`` or later. - Between 32GB and 120 GB in 8 GB increments - Available ``cpu`` values: 16384 (16 vCPU) This option requires Linux platform ``1.4.0`` or later.
        :param network_mode: The Docker networking mode to use for the containers in the task. The valid values are ``none`` , ``bridge`` , ``awsvpc`` , and ``host`` . If no network mode is specified, the default is ``bridge`` . For Amazon ECS tasks on Fargate, the ``awsvpc`` network mode is required. For Amazon ECS tasks on Amazon EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2 Windows instances, ``<default>`` or ``awsvpc`` can be used. If the network mode is set to ``none`` , you cannot specify port mappings in your container definitions, and the tasks containers do not have external connectivity. The ``host`` and ``awsvpc`` network modes offer the highest networking performance for containers because they use the EC2 network stack instead of the virtualized network stack provided by the ``bridge`` mode. With the ``host`` and ``awsvpc`` network modes, exposed container ports are mapped directly to the corresponding host port (for the ``host`` network mode) or the attached elastic network interface port (for the ``awsvpc`` network mode), so you cannot take advantage of dynamic host port mappings. .. epigraph:: When using the ``host`` network mode, you should not run containers using the root user (UID 0). It is considered best practice to use a non-root user. If the network mode is ``awsvpc`` , the task is allocated an elastic network interface, and you must specify a ``NetworkConfiguration`` value when you create a service or run a task with the task definition. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If the network mode is ``host`` , you cannot run multiple instantiations of the same task on a single container instance when port mappings are used. For more information, see `Network settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#network-settings>`_ in the *Docker run reference* .
        :param pid_mode: The process namespace to use for the containers in the task. The valid values are ``host`` or ``task`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` PID mode on the same container instance share the same process namespace with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same process namespace. If no value is specified, the default is a private namespace. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the *Docker run reference* . If the ``host`` PID mode is used, be aware that there is a heightened risk of undesired process namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ . .. epigraph:: This parameter is not supported for Windows containers or tasks run on AWS Fargate .
        :param placement_constraints: An array of placement constraint objects to use for tasks. .. epigraph:: This parameter isn't supported for tasks run on AWS Fargate .
        :param proxy_configuration: The configuration details for the App Mesh proxy. Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least version 1.26.0-1 of the ``ecs-init`` package to use a proxy configuration. If your container instances are launched from the Amazon ECS optimized AMI version ``20190301`` or later, they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param requires_compatibilities: The task launch types the task definition was validated against. For more information, see `Amazon ECS launch types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param runtime_platform: The operating system that your tasks definitions run on. A platform family is specified only for tasks using the Fargate launch type. When you specify a task definition in a service, this value must match the ``runtimePlatform`` value of the service.
        :param tags: The metadata that you apply to the task definition to help you categorize and organize them. Each tag consists of a key and an optional value. You define both of them. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        :param task_role_arn: The short name or full Amazon Resource Name (ARN) of the AWS Identity and Access Management role that grants containers in the task permission to call AWS APIs on your behalf. For more information, see `Amazon ECS Task Role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* . IAM roles for tasks on Windows require that the ``-EnableTaskIAMRole`` option is set when you launch the Amazon ECS-optimized Windows AMI. Your containers must also run some configuration code to use the feature. For more information, see `Windows IAM roles for tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows_task_IAM_roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param volumes: The list of data volume definitions for the task. For more information, see `Using data volumes in tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: The ``host`` and ``sourcePath`` parameters aren't supported for tasks run on AWS Fargate .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a77e92d9ff0a9ac5b9f5909726a2e91dafa1fae662c3fdef09e5f9c3fe674a95)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnTaskDefinitionProps(
            container_definitions=container_definitions,
            cpu=cpu,
            ephemeral_storage=ephemeral_storage,
            execution_role_arn=execution_role_arn,
            family=family,
            inference_accelerators=inference_accelerators,
            ipc_mode=ipc_mode,
            memory=memory,
            network_mode=network_mode,
            pid_mode=pid_mode,
            placement_constraints=placement_constraints,
            proxy_configuration=proxy_configuration,
            requires_compatibilities=requires_compatibilities,
            runtime_platform=runtime_platform,
            tags=tags,
            task_role_arn=task_role_arn,
            volumes=volumes,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ea38df420ceb9e6eaf9d5bd98fc65ace2c13e416a8da9c3acc7937bf5d6f242)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8c5210ed52c4036135861e17b58c979c262bc1d2c8f862dde03a7b5a73af09c0)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrTaskDefinitionArn")
    def attr_task_definition_arn(self) -> builtins.str:
        '''
        :cloudformationAttribute: TaskDefinitionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrTaskDefinitionArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''The metadata that you apply to the task definition to help you categorize and organize them.

        Each tag consists of a key and an optional value. You define both of them.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="containerDefinitions")
    def container_definitions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ContainerDefinitionProperty"]]]]:
        '''A list of container definitions in JSON format that describe the different containers that make up your task.

        For more information about container definition parameters and defaults, see `Amazon ECS Task Definitions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-containerdefinitions
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ContainerDefinitionProperty"]]]], jsii.get(self, "containerDefinitions"))

    @container_definitions.setter
    def container_definitions(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ContainerDefinitionProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__29800844d323164e3490b40a3de2221a272dcb4fda034c3b6c2735adba47d9e9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "containerDefinitions", value)

    @builtins.property
    @jsii.member(jsii_name="cpu")
    def cpu(self) -> typing.Optional[builtins.str]:
        '''The number of ``cpu`` units used by the task.

        If you use the EC2 launch type, this field is optional. Any value can be used. If you use the Fargate launch type, this field is required. You must use one of the following values. The value that you choose determines your range of valid values for the ``memory`` parameter.

        The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate.

        - 256 (.25 vCPU) - Available ``memory`` values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)
        - 512 (.5 vCPU) - Available ``memory`` values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB)
        - 1024 (1 vCPU) - Available ``memory`` values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB)
        - 2048 (2 vCPU) - Available ``memory`` values: 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB)
        - 4096 (4 vCPU) - Available ``memory`` values: 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB)
        - 8192 (8 vCPU) - Available ``memory`` values: 16 GB and 60 GB in 4 GB increments

        This option requires Linux platform ``1.4.0`` or later.

        - 16384 (16vCPU) - Available ``memory`` values: 32GB and 120 GB in 8 GB increments

        This option requires Linux platform ``1.4.0`` or later.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-cpu
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cpu"))

    @cpu.setter
    def cpu(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54ee0f0538c4fcd482362fac760fb6e813d1c01e85a1bb6e1da58c885b54325a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cpu", value)

    @builtins.property
    @jsii.member(jsii_name="ephemeralStorage")
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.EphemeralStorageProperty"]]:
        '''The ephemeral storage settings to use for tasks run with the task definition.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ephemeralstorage
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.EphemeralStorageProperty"]], jsii.get(self, "ephemeralStorage"))

    @ephemeral_storage.setter
    def ephemeral_storage(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.EphemeralStorageProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__501da7d8d052de8848e38b3ce458cd787c3006be2870679cd704efc8594577e3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ephemeralStorage", value)

    @builtins.property
    @jsii.member(jsii_name="executionRoleArn")
    def execution_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent permission to make AWS API calls on your behalf.

        The task execution IAM role is required depending on the requirements of your task. For more information, see `Amazon ECS task execution IAM role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_execution_IAM_role.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-executionrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "executionRoleArn"))

    @execution_role_arn.setter
    def execution_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__44f1cf99c07ba4c253e7667f369e0a9e2f3e26420ee732abbfb1211d0c3b71b8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "executionRoleArn", value)

    @builtins.property
    @jsii.member(jsii_name="family")
    def family(self) -> typing.Optional[builtins.str]:
        '''The name of a family that this task definition is registered to.

        Up to 255 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.

        A family groups multiple versions of a task definition. Amazon ECS gives the first task definition that you registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers to each task definition that you add.
        .. epigraph::

           To use revision numbers when you update a task definition, specify this property. If you don't specify a value, AWS CloudFormation generates a new task definition each time that you update it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-family
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "family"))

    @family.setter
    def family(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__023c08e98e6a3d7c301f4a3c8f957a42823b7070f0792318d812978fde2496a5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "family", value)

    @builtins.property
    @jsii.member(jsii_name="inferenceAccelerators")
    def inference_accelerators(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.InferenceAcceleratorProperty"]]]]:
        '''The Elastic Inference accelerators to use for the containers in the task.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-inferenceaccelerators
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.InferenceAcceleratorProperty"]]]], jsii.get(self, "inferenceAccelerators"))

    @inference_accelerators.setter
    def inference_accelerators(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.InferenceAcceleratorProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ede2dda4307f7ac6298b1b9b166ee4dd51bc0c2d23fdf524db491c115f153bb5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "inferenceAccelerators", value)

    @builtins.property
    @jsii.member(jsii_name="ipcMode")
    def ipc_mode(self) -> typing.Optional[builtins.str]:
        '''The IPC resource namespace to use for the containers in the task.

        The valid values are ``host`` , ``task`` , or ``none`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same IPC resources. If ``none`` is specified, then IPC resources within the containers of a task are private and not shared with other containers in a task or on the container instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the *Docker run reference* .

        If the ``host`` IPC mode is used, be aware that there is a heightened risk of undesired IPC namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ .

        If you are setting namespaced kernel parameters using ``systemControls`` for the containers in the task, the following will apply to your IPC resource namespace. For more information, see `System Controls <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        - For tasks that use the ``host`` IPC mode, IPC namespace related ``systemControls`` are not supported.
        - For tasks that use the ``task`` IPC mode, IPC namespace related ``systemControls`` will apply to all containers within a task.

        .. epigraph::

           This parameter is not supported for Windows containers or tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ipcmode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ipcMode"))

    @ipc_mode.setter
    def ipc_mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__17a4c1ac170c4b0b9de038d3c93319ece4b9c94e5a8f478f58e8187349fc446b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ipcMode", value)

    @builtins.property
    @jsii.member(jsii_name="memory")
    def memory(self) -> typing.Optional[builtins.str]:
        '''The amount (in MiB) of memory used by the task.

        If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a container-level memory value. This field is optional and any value can be used. If a task-level memory value is specified, the container-level memory value is optional. For more information regarding container-level memory and memory reservation, see `ContainerDefinition <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html>`_ .

        If your tasks runs on AWS Fargate , this field is required. You must use one of the following values. The value you choose determines your range of valid values for the ``cpu`` parameter.

        - 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available ``cpu`` values: 256 (.25 vCPU)
        - 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available ``cpu`` values: 512 (.5 vCPU)
        - 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available ``cpu`` values: 1024 (1 vCPU)
        - Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 2048 (2 vCPU)
        - Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 4096 (4 vCPU)
        - Between 16 GB and 60 GB in 4 GB increments - Available ``cpu`` values: 8192 (8 vCPU)

        This option requires Linux platform ``1.4.0`` or later.

        - Between 32GB and 120 GB in 8 GB increments - Available ``cpu`` values: 16384 (16 vCPU)

        This option requires Linux platform ``1.4.0`` or later.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-memory
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "memory"))

    @memory.setter
    def memory(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__90c650a2ff91b31f35b46c6d56ed671a7ad6c10fb98853123db145e5afaca79d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "memory", value)

    @builtins.property
    @jsii.member(jsii_name="networkMode")
    def network_mode(self) -> typing.Optional[builtins.str]:
        '''The Docker networking mode to use for the containers in the task.

        The valid values are ``none`` , ``bridge`` , ``awsvpc`` , and ``host`` . If no network mode is specified, the default is ``bridge`` .

        For Amazon ECS tasks on Fargate, the ``awsvpc`` network mode is required. For Amazon ECS tasks on Amazon EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2 Windows instances, ``<default>`` or ``awsvpc`` can be used. If the network mode is set to ``none`` , you cannot specify port mappings in your container definitions, and the tasks containers do not have external connectivity. The ``host`` and ``awsvpc`` network modes offer the highest networking performance for containers because they use the EC2 network stack instead of the virtualized network stack provided by the ``bridge`` mode.

        With the ``host`` and ``awsvpc`` network modes, exposed container ports are mapped directly to the corresponding host port (for the ``host`` network mode) or the attached elastic network interface port (for the ``awsvpc`` network mode), so you cannot take advantage of dynamic host port mappings.
        .. epigraph::

           When using the ``host`` network mode, you should not run containers using the root user (UID 0). It is considered best practice to use a non-root user.

        If the network mode is ``awsvpc`` , the task is allocated an elastic network interface, and you must specify a ``NetworkConfiguration`` value when you create a service or run a task with the task definition. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        If the network mode is ``host`` , you cannot run multiple instantiations of the same task on a single container instance when port mappings are used.

        For more information, see `Network settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#network-settings>`_ in the *Docker run reference* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-networkmode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkMode"))

    @network_mode.setter
    def network_mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e159792237fa557a00d6e83525878a4ca13d788b9a55460f624fa17db4d318f7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "networkMode", value)

    @builtins.property
    @jsii.member(jsii_name="pidMode")
    def pid_mode(self) -> typing.Optional[builtins.str]:
        '''The process namespace to use for the containers in the task.

        The valid values are ``host`` or ``task`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` PID mode on the same container instance share the same process namespace with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same process namespace. If no value is specified, the default is a private namespace. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the *Docker run reference* .

        If the ``host`` PID mode is used, be aware that there is a heightened risk of undesired process namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ .
        .. epigraph::

           This parameter is not supported for Windows containers or tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-pidmode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "pidMode"))

    @pid_mode.setter
    def pid_mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a010cfd9fe0814d7bf70aeb04b33220b4eda5505e0df81930142c33e32349d7c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "pidMode", value)

    @builtins.property
    @jsii.member(jsii_name="placementConstraints")
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty"]]]]:
        '''An array of placement constraint objects to use for tasks.

        .. epigraph::

           This parameter isn't supported for tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-placementconstraints
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty"]]]], jsii.get(self, "placementConstraints"))

    @placement_constraints.setter
    def placement_constraints(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__78ff7fa573165f9b46173519e514653ca91dd23771683ef4b6fef7f3ba019ca8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "placementConstraints", value)

    @builtins.property
    @jsii.member(jsii_name="proxyConfiguration")
    def proxy_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ProxyConfigurationProperty"]]:
        '''The configuration details for the App Mesh proxy.

        Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least version 1.26.0-1 of the ``ecs-init`` package to use a proxy configuration. If your container instances are launched from the Amazon ECS optimized AMI version ``20190301`` or later, they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-proxyconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ProxyConfigurationProperty"]], jsii.get(self, "proxyConfiguration"))

    @proxy_configuration.setter
    def proxy_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ProxyConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95b572af184339d30467e75236df050da2c8b1cbfcdaec2fb33043679721b36a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "proxyConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="requiresCompatibilities")
    def requires_compatibilities(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The task launch types the task definition was validated against.

        For more information, see `Amazon ECS launch types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-requirescompatibilities
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "requiresCompatibilities"))

    @requires_compatibilities.setter
    def requires_compatibilities(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d378452d1061955f2e06eba3695d7fd0d56599a7892066f2e3fee39393c6f2d7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requiresCompatibilities", value)

    @builtins.property
    @jsii.member(jsii_name="runtimePlatform")
    def runtime_platform(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.RuntimePlatformProperty"]]:
        '''The operating system that your tasks definitions run on.

        A platform family is specified only for tasks using the Fargate launch type.

        When you specify a task definition in a service, this value must match the ``runtimePlatform`` value of the service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-runtimeplatform
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.RuntimePlatformProperty"]], jsii.get(self, "runtimePlatform"))

    @runtime_platform.setter
    def runtime_platform(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.RuntimePlatformProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4078855b99b24a22f7ec944a3300120e776acdf5322a1827b9eec52b86791e13)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "runtimePlatform", value)

    @builtins.property
    @jsii.member(jsii_name="taskRoleArn")
    def task_role_arn(self) -> typing.Optional[builtins.str]:
        '''The short name or full Amazon Resource Name (ARN) of the AWS Identity and Access Management role that grants containers in the task permission to call AWS APIs on your behalf.

        For more information, see `Amazon ECS Task Role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        IAM roles for tasks on Windows require that the ``-EnableTaskIAMRole`` option is set when you launch the Amazon ECS-optimized Windows AMI. Your containers must also run some configuration code to use the feature. For more information, see `Windows IAM roles for tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows_task_IAM_roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-taskrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "taskRoleArn"))

    @task_role_arn.setter
    def task_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e690d52997a10244860f4acf1ba169c925dd19894fbdbda0d6faee5c979cd4c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "taskRoleArn", value)

    @builtins.property
    @jsii.member(jsii_name="volumes")
    def volumes(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.VolumeProperty"]]]]:
        '''The list of data volume definitions for the task.

        For more information, see `Using data volumes in tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        .. epigraph::

           The ``host`` and ``sourcePath`` parameters aren't supported for tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-volumes
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.VolumeProperty"]]]], jsii.get(self, "volumes"))

    @volumes.setter
    def volumes(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.VolumeProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fdfd0bcb877fbf1d248ac6f9103ba77ddf96ac5dfa28578b48267df2db46467d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "volumes", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.AuthorizationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"access_point_id": "accessPointId", "iam": "iam"},
    )
    class AuthorizationConfigProperty:
        def __init__(
            self,
            *,
            access_point_id: typing.Optional[builtins.str] = None,
            iam: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The authorization configuration details for the Amazon EFS file system.

            :param access_point_id: The Amazon EFS access point ID to use. If an access point is specified, the root directory value specified in the ``EFSVolumeConfiguration`` must either be omitted or set to ``/`` which will enforce the path set on the EFS access point. If an access point is used, transit encryption must be on in the ``EFSVolumeConfiguration`` . For more information, see `Working with Amazon EFS access points <https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html>`_ in the *Amazon Elastic File System User Guide* .
            :param iam: Determines whether to use the Amazon ECS task role defined in a task definition when mounting the Amazon EFS file system. If it is turned on, transit encryption must be turned on in the ``EFSVolumeConfiguration`` . If this parameter is omitted, the default value of ``DISABLED`` is used. For more information, see `Using Amazon EFS access points <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/efs-volumes.html#efs-volume-accesspoints>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-authorizationconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                authorization_config_property = ecs.CfnTaskDefinition.AuthorizationConfigProperty(
                    access_point_id="accessPointId",
                    iam="iam"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__24146df32f9b930beafb52c1cf654f0175d8a8df8e54a5d7c80e3905b5643a41)
                check_type(argname="argument access_point_id", value=access_point_id, expected_type=type_hints["access_point_id"])
                check_type(argname="argument iam", value=iam, expected_type=type_hints["iam"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if access_point_id is not None:
                self._values["access_point_id"] = access_point_id
            if iam is not None:
                self._values["iam"] = iam

        @builtins.property
        def access_point_id(self) -> typing.Optional[builtins.str]:
            '''The Amazon EFS access point ID to use.

            If an access point is specified, the root directory value specified in the ``EFSVolumeConfiguration`` must either be omitted or set to ``/`` which will enforce the path set on the EFS access point. If an access point is used, transit encryption must be on in the ``EFSVolumeConfiguration`` . For more information, see `Working with Amazon EFS access points <https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html>`_ in the *Amazon Elastic File System User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-authorizationconfig.html#cfn-ecs-taskdefinition-authorizationconfig-accesspointid
            '''
            result = self._values.get("access_point_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def iam(self) -> typing.Optional[builtins.str]:
            '''Determines whether to use the Amazon ECS task role defined in a task definition when mounting the Amazon EFS file system.

            If it is turned on, transit encryption must be turned on in the ``EFSVolumeConfiguration`` . If this parameter is omitted, the default value of ``DISABLED`` is used. For more information, see `Using Amazon EFS access points <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/efs-volumes.html#efs-volume-accesspoints>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-authorizationconfig.html#cfn-ecs-taskdefinition-authorizationconfig-iam
            '''
            result = self._values.get("iam")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AuthorizationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.ContainerDefinitionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "image": "image",
            "name": "name",
            "command": "command",
            "cpu": "cpu",
            "depends_on": "dependsOn",
            "disable_networking": "disableNetworking",
            "dns_search_domains": "dnsSearchDomains",
            "dns_servers": "dnsServers",
            "docker_labels": "dockerLabels",
            "docker_security_options": "dockerSecurityOptions",
            "entry_point": "entryPoint",
            "environment": "environment",
            "environment_files": "environmentFiles",
            "essential": "essential",
            "extra_hosts": "extraHosts",
            "firelens_configuration": "firelensConfiguration",
            "health_check": "healthCheck",
            "hostname": "hostname",
            "interactive": "interactive",
            "links": "links",
            "linux_parameters": "linuxParameters",
            "log_configuration": "logConfiguration",
            "memory": "memory",
            "memory_reservation": "memoryReservation",
            "mount_points": "mountPoints",
            "port_mappings": "portMappings",
            "privileged": "privileged",
            "pseudo_terminal": "pseudoTerminal",
            "readonly_root_filesystem": "readonlyRootFilesystem",
            "repository_credentials": "repositoryCredentials",
            "resource_requirements": "resourceRequirements",
            "secrets": "secrets",
            "start_timeout": "startTimeout",
            "stop_timeout": "stopTimeout",
            "system_controls": "systemControls",
            "ulimits": "ulimits",
            "user": "user",
            "volumes_from": "volumesFrom",
            "working_directory": "workingDirectory",
        },
    )
    class ContainerDefinitionProperty:
        def __init__(
            self,
            *,
            image: builtins.str,
            name: builtins.str,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            cpu: typing.Optional[jsii.Number] = None,
            depends_on: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.ContainerDependencyProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            disable_networking: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
            dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
            docker_labels: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]] = None,
            docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
            entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
            environment: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.KeyValuePairProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            environment_files: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.EnvironmentFileProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            essential: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            extra_hosts: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.HostEntryProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            firelens_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.FirelensConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            health_check: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.HealthCheckProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            hostname: typing.Optional[builtins.str] = None,
            interactive: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            links: typing.Optional[typing.Sequence[builtins.str]] = None,
            linux_parameters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.LinuxParametersProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            log_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.LogConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            memory: typing.Optional[jsii.Number] = None,
            memory_reservation: typing.Optional[jsii.Number] = None,
            mount_points: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.MountPointProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            port_mappings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.PortMappingProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            privileged: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            pseudo_terminal: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            readonly_root_filesystem: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            repository_credentials: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.RepositoryCredentialsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            resource_requirements: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.ResourceRequirementProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            secrets: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.SecretProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            start_timeout: typing.Optional[jsii.Number] = None,
            stop_timeout: typing.Optional[jsii.Number] = None,
            system_controls: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.SystemControlProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            ulimits: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.UlimitProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            user: typing.Optional[builtins.str] = None,
            volumes_from: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.VolumeFromProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            working_directory: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ContainerDefinition`` property specifies a container definition.

            Container definitions are used in task definitions to describe the different containers that are launched as part of a task.

            :param image: The image used to start a container. This string is passed directly to the Docker daemon. By default, images in the Docker Hub registry are available. Other repositories are specified with either ``*repository-url* / *image* : *tag*`` or ``*repository-url* / *image* @ *digest*`` . Up to 255 letters (uppercase and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are allowed. This parameter maps to ``Image`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``IMAGE`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . - When a new task starts, the Amazon ECS container agent pulls the latest version of the specified image and tag for the container to use. However, subsequent updates to a repository image aren't propagated to already running tasks. - Images in Amazon ECR repositories can be specified by either using the full ``registry/repository:tag`` or ``registry/repository@digest`` . For example, ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>:latest`` or ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE`` . - Images in official repositories on Docker Hub use a single name (for example, ``ubuntu`` or ``mongo`` ). - Images in other repositories on Docker Hub are qualified with an organization name (for example, ``amazon/amazon-ecs-agent`` ). - Images in other online repositories are qualified further by a domain name (for example, ``quay.io/assemblyline/ubuntu`` ).
            :param name: The name of a container. If you're linking multiple containers together in a task definition, the ``name`` of one container can be entered in the ``links`` of another container to connect the containers. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. This parameter maps to ``name`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--name`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param command: The command that's passed to the container. This parameter maps to ``Cmd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``COMMAND`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#cmd <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ . If there are multiple arguments, each argument is a separated string in the array.
            :param cpu: The number of ``cpu`` units reserved for the container. This parameter maps to ``CpuShares`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cpu-shares`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This field is optional for tasks using the Fargate launch type, and the only requirement is that the total amount of CPU reserved for all containers within a task be lower than the task-level ``cpu`` value. .. epigraph:: You can determine the number of CPU units that are available per EC2 instance type by multiplying the vCPUs listed for that instance type on the `Amazon EC2 Instances <https://docs.aws.amazon.com/ec2/instance-types/>`_ detail page by 1,024. Linux containers share unallocated CPU units with other containers on the container instance with the same ratio as their allocated amount. For example, if you run a single-container task on a single-core instance type with 512 CPU units specified for that container, and that's the only task running on the container instance, that container could use the full 1,024 CPU unit share at any given time. However, if you launched another copy of the same task on that container instance, each task is guaranteed a minimum of 512 CPU units when needed. Moreover, each container could float to higher CPU usage if the other container was not using it. If both tasks were 100% active all of the time, they would be limited to 512 CPU units. On Linux container instances, the Docker daemon on the container instance uses the CPU value to calculate the relative CPU share ratios for running containers. For more information, see `CPU share constraint <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#cpu-share-constraint>`_ in the Docker documentation. The minimum valid CPU share value that the Linux kernel allows is 2. However, the CPU parameter isn't required, and you can use CPU values below 2 in your container definitions. For CPU values below 2 (including null), the behavior varies based on your Amazon ECS container agent version: - *Agent versions less than or equal to 1.1.0:* Null and zero CPU values are passed to Docker as 0, which Docker then converts to 1,024 CPU shares. CPU values of 1 are passed to Docker as 1, which the Linux kernel converts to two CPU shares. - *Agent versions greater than or equal to 1.2.0:* Null, zero, and CPU values of 1 are passed to Docker as 2. On Windows container instances, the CPU limit is enforced as an absolute limit, or a quota. Windows containers only have access to the specified amount of CPU that's described in the task definition. A null or zero CPU value is passed to Docker as ``0`` , which Windows interprets as 1% of one CPU.
            :param depends_on: The dependencies defined for container startup and shutdown. A container can contain multiple dependencies. When a dependency is defined for container startup, for container shutdown it is reversed. For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container agent to turn on container dependencies. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For tasks using the Fargate launch type, the task or service requires the following platforms: - Linux platform version ``1.3.0`` or later. - Windows platform version ``1.0.0`` or later. If the task definition is used in a blue/green deployment that uses `AWS::CodeDeploy::DeploymentGroup BlueGreenDeploymentConfiguration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codedeploy-deploymentgroup-bluegreendeploymentconfiguration.html>`_ , the ``dependsOn`` parameter is not supported. For more information see `Issue #680 <https://docs.aws.amazon.com/https://github.com/aws-cloudformation/cloudformation-coverage-roadmap/issues/680>`_ on the on the GitHub website.
            :param disable_networking: When this parameter is true, networking is off within the container. This parameter maps to ``NetworkDisabled`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param dns_search_domains: A list of DNS search domains that are presented to the container. This parameter maps to ``DnsSearch`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--dns-search`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param dns_servers: A list of DNS servers that are presented to the container. This parameter maps to ``Dns`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--dns`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param docker_labels: A key/value map of labels to add to the container. This parameter maps to ``Labels`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--label`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            :param docker_security_options: A list of strings to provide custom configuration for multiple security systems. For more information about valid values, see `Docker Run Security Configuration <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This field isn't valid for containers in tasks using the Fargate launch type. For Linux tasks on EC2, this parameter can be used to reference custom labels for SELinux and AppArmor multi-level security systems. For any tasks on EC2, this parameter can be used to reference a credential spec file that configures a container for Active Directory authentication. For more information, see `Using gMSAs for Windows Containers <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html>`_ and `Using gMSAs for Linux Containers <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html>`_ in the *Amazon Elastic Container Service Developer Guide* . This parameter maps to ``SecurityOpt`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--security-opt`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: The Amazon ECS container agent running on a container instance must register with the ``ECS_SELINUX_CAPABLE=true`` or ``ECS_APPARMOR_CAPABLE=true`` environment variables before containers placed on that instance can use these security options. For more information, see `Amazon ECS Container Agent Configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For more information about valid values, see `Docker Run Security Configuration <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" | "credentialspec:CredentialSpecFilePath"
            :param entry_point: .. epigraph:: Early versions of the Amazon ECS container agent don't properly handle ``entryPoint`` parameters. If you have problems using ``entryPoint`` , update your container agent or enter your commands and arguments as ``command`` array items instead. The entry point that's passed to the container. This parameter maps to ``Entrypoint`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--entrypoint`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#entrypoint <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#entrypoint>`_ .
            :param environment: The environment variables to pass to a container. This parameter maps to ``Env`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--env`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: We don't recommend that you use plaintext environment variables for sensitive information, such as credential data.
            :param environment_files: A list of files containing the environment variables to pass to a container. This parameter maps to the ``--env-file`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . You can specify up to ten environment files. The file must have a ``.env`` file extension. Each line in an environment file contains an environment variable in ``VARIABLE=VALUE`` format. Lines beginning with ``#`` are treated as comments and are ignored. For more information about the environment variable file syntax, see `Declare default environment variables in file <https://docs.aws.amazon.com/https://docs.docker.com/compose/env-file/>`_ . If there are environment variables specified using the ``environment`` parameter in a container definition, they take precedence over the variables contained within an environment file. If multiple environment files are specified that contain the same variable, they're processed from the top down. We recommend that you use unique variable names. For more information, see `Specifying Environment Variables <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param essential: If the ``essential`` parameter of a container is marked as ``true`` , and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the ``essential`` parameter of a container is marked as ``false`` , its failure doesn't affect the rest of the containers in a task. If this parameter is omitted, a container is assumed to be essential. All tasks must have at least one essential container. If you have an application that's composed of multiple containers, group containers that are used for a common purpose into components, and separate the different components into multiple task definitions. For more information, see `Application Architecture <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param extra_hosts: A list of hostnames and IP address mappings to append to the ``/etc/hosts`` file on the container. This parameter maps to ``ExtraHosts`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--add-host`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter isn't supported for Windows containers or tasks that use the ``awsvpc`` network mode.
            :param firelens_configuration: The FireLens configuration for the container. This is used to specify and configure a log router for container logs. For more information, see `Custom Log Routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param health_check: The container health check command and associated configuration parameters for the container. This parameter maps to ``HealthCheck`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``HEALTHCHECK`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param hostname: The hostname to use for your container. This parameter maps to ``Hostname`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--hostname`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: The ``hostname`` parameter is not supported if you're using the ``awsvpc`` network mode.
            :param interactive: When this parameter is ``true`` , you can deploy containerized applications that require ``stdin`` or a ``tty`` to be allocated. This parameter maps to ``OpenStdin`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--interactive`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param links: The ``links`` parameter allows containers to communicate with each other without the need for port mappings. This parameter is only supported if the network mode of a task definition is ``bridge`` . The ``name:internalName`` construct is analogous to ``name:alias`` in Docker links. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. For more information about linking Docker containers, go to `Legacy container links <https://docs.aws.amazon.com/https://docs.docker.com/network/links/>`_ in the Docker documentation. This parameter maps to ``Links`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--link`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers. > Containers that are collocated on a single container instance may be able to communicate with each other without requiring links or host port mappings. Network isolation is achieved on the container instance using security groups and VPC settings.
            :param linux_parameters: Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param log_configuration: The log configuration specification for the container. This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--log-driver`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . By default, containers use the same logging driver that the Docker daemon uses. However, the container may use a different logging driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a different logging driver for a container, the log system must be configured properly on the container instance (or on a different log server for remote logging options). For more information on the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the Docker documentation. .. epigraph:: Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the `LogConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_LogConfiguration.html>`_ data type). Additional log drivers may be available in future releases of the Amazon ECS container agent. This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'`` .. epigraph:: The Amazon ECS container agent running on a container instance must register the logging drivers available on that instance with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS Container Agent Configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param memory: The amount (in MiB) of memory to present to the container. If your container attempts to exceed the memory specified here, the container is killed. The total amount of memory reserved for all containers within a task must be lower than the task ``memory`` value, if one is specified. This parameter maps to ``Memory`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--memory`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . If using the Fargate launch type, this parameter is optional. If using the EC2 launch type, you must specify either a task-level memory value or a container-level memory value. If you specify both a container-level ``memory`` and ``memoryReservation`` value, ``memory`` must be greater than ``memoryReservation`` . If you specify ``memoryReservation`` , then that value is subtracted from the available memory resources for the container instance where the container is placed. Otherwise, the value of ``memory`` is used. The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container, so you should not specify fewer than 6 MiB of memory for your containers. The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container, so you should not specify fewer than 4 MiB of memory for your containers.
            :param memory_reservation: The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the ``memory`` parameter (if applicable), or all of the available memory on the container instance, whichever comes first. This parameter maps to ``MemoryReservation`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--memory-reservation`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . If a task-level memory value is not specified, you must specify a non-zero integer for one or both of ``memory`` or ``memoryReservation`` in a container definition. If you specify both, ``memory`` must be greater than ``memoryReservation`` . If you specify ``memoryReservation`` , then that value is subtracted from the available memory resources for the container instance where the container is placed. Otherwise, the value of ``memory`` is used. For example, if your container normally uses 128 MiB of memory, but occasionally bursts to 256 MiB of memory for short periods of time, you can set a ``memoryReservation`` of 128 MiB, and a ``memory`` hard limit of 300 MiB. This configuration would allow the container to only reserve 128 MiB of memory from the remaining resources on the container instance, but also allow the container to consume more memory resources when needed. The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't specify less than 6 MiB of memory for your containers. The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So, don't specify less than 4 MiB of memory for your containers.
            :param mount_points: The mount points for data volumes in your container. This parameter maps to ``Volumes`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--volume`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives.
            :param port_mappings: The list of port mappings for the container. Port mappings allow containers to access ports on the host container instance to send or receive traffic. For task definitions that use the ``awsvpc`` network mode, you should only specify the ``containerPort`` . The ``hostPort`` can be left blank or it must be the same value as the ``containerPort`` . Port mappings on Windows use the ``NetNAT`` gateway address rather than ``localhost`` . There is no loopback for port mappings on Windows, so you cannot access a container's mapped port from the host itself. This parameter maps to ``PortBindings`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--publish`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . If the network mode of a task definition is set to ``none`` , then you can't specify port mappings. If the network mode of a task definition is set to ``host`` , then host ports must either be undefined or they must match the container port in the port mapping. .. epigraph:: After a task reaches the ``RUNNING`` status, manual and automatic host and container port assignments are visible in the *Network Bindings* section of a container description for a selected task in the Amazon ECS console. The assignments are also visible in the ``networkBindings`` section `DescribeTasks <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html>`_ responses.
            :param privileged: When this parameter is true, the container is given elevated privileges on the host container instance (similar to the ``root`` user). This parameter maps to ``Privileged`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--privileged`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers or tasks run on AWS Fargate .
            :param pseudo_terminal: When this parameter is ``true`` , a TTY is allocated. This parameter maps to ``Tty`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--tty`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param readonly_root_filesystem: When this parameter is true, the container is given read-only access to its root file system. This parameter maps to ``ReadonlyRootfs`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--read-only`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param repository_credentials: The private repository authentication credentials to use.
            :param resource_requirements: The type and amount of a resource to assign to a container. The only supported resource is a GPU.
            :param secrets: The secrets to pass to the container. For more information, see `Specifying Sensitive Data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param start_timeout: Time duration (in seconds) to wait before giving up on resolving dependencies for a container. For example, you specify two containers in a task definition with containerA having a dependency on containerB reaching a ``COMPLETE`` , ``SUCCESS`` , or ``HEALTHY`` status. If a ``startTimeout`` value is specified for containerB and it doesn't reach the desired status within that time then containerA gives up and not start. This results in the task transitioning to a ``STOPPED`` state. .. epigraph:: When the ``ECS_CONTAINER_START_TIMEOUT`` container agent configuration variable is used, it's enforced independently from this start timeout value. For tasks using the Fargate launch type, the task or service requires the following platforms: - Linux platform version ``1.3.0`` or later. - Windows platform version ``1.0.0`` or later. For tasks using the EC2 launch type, your container instances require at least version ``1.26.0`` of the container agent to use a container start timeout value. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version ``1.26.0-1`` of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param stop_timeout: Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. For tasks using the Fargate launch type, the task or service requires the following platforms: - Linux platform version ``1.3.0`` or later. - Windows platform version ``1.0.0`` or later. The max stop timeout value is 120 seconds and if the parameter is not specified, the default value of 30 seconds is used. For tasks that use the EC2 launch type, if the ``stopTimeout`` parameter isn't specified, the value set for the Amazon ECS container agent configuration variable ``ECS_CONTAINER_STOP_TIMEOUT`` is used. If neither the ``stopTimeout`` parameter or the ``ECS_CONTAINER_STOP_TIMEOUT`` agent configuration variable are set, then the default values of 30 seconds for Linux containers and 30 seconds on Windows containers are used. Your container instances require at least version 1.26.0 of the container agent to use a container stop timeout value. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param system_controls: A list of namespaced kernel parameters to set in the container. This parameter maps to ``Sysctls`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--sysctl`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: We don't recommended that you specify network-related ``systemControls`` parameters for multiple containers in a single task that also uses either the ``awsvpc`` or ``host`` network modes. For tasks that use the ``awsvpc`` network mode, the container that's started last determines which ``systemControls`` parameters take effect. For tasks that use the ``host`` network mode, it changes the container instance's namespaced kernel parameters as well as the containers.
            :param ulimits: A list of ``ulimits`` to set in the container. This parameter maps to ``Ulimits`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--ulimit`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . Valid naming values are displayed in the `Ulimit <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html>`_ data type. This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'`` .. epigraph:: This parameter is not supported for Windows containers.
            :param user: The user to use inside the container. This parameter maps to ``User`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--user`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: When running tasks using the ``host`` network mode, don't run containers using the root user (UID 0). We recommend using a non-root user for better security. You can specify the ``user`` using the following formats. If specifying a UID or GID, you must specify it as a positive integer. - ``user`` - ``user:group`` - ``uid`` - ``uid:gid`` - ``user:gid`` - ``uid:group`` .. epigraph:: This parameter is not supported for Windows containers.
            :param volumes_from: Data volumes to mount from another container. This parameter maps to ``VolumesFrom`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--volumes-from`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param working_directory: The working directory to run commands inside the container in. This parameter maps to ``WorkingDir`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--workdir`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                container_definition_property = ecs.CfnTaskDefinition.ContainerDefinitionProperty(
                    image="image",
                    name="name",
                
                    # the properties below are optional
                    command=["command"],
                    cpu=123,
                    depends_on=[ecs.CfnTaskDefinition.ContainerDependencyProperty(
                        condition="condition",
                        container_name="containerName"
                    )],
                    disable_networking=False,
                    dns_search_domains=["dnsSearchDomains"],
                    dns_servers=["dnsServers"],
                    docker_labels={
                        "docker_labels_key": "dockerLabels"
                    },
                    docker_security_options=["dockerSecurityOptions"],
                    entry_point=["entryPoint"],
                    environment=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                        name="name",
                        value="value"
                    )],
                    environment_files=[ecs.CfnTaskDefinition.EnvironmentFileProperty(
                        type="type",
                        value="value"
                    )],
                    essential=False,
                    extra_hosts=[ecs.CfnTaskDefinition.HostEntryProperty(
                        hostname="hostname",
                        ip_address="ipAddress"
                    )],
                    firelens_configuration=ecs.CfnTaskDefinition.FirelensConfigurationProperty(
                        options={
                            "options_key": "options"
                        },
                        type="type"
                    ),
                    health_check=ecs.CfnTaskDefinition.HealthCheckProperty(
                        command=["command"],
                        interval=123,
                        retries=123,
                        start_period=123,
                        timeout=123
                    ),
                    hostname="hostname",
                    interactive=False,
                    links=["links"],
                    linux_parameters=ecs.CfnTaskDefinition.LinuxParametersProperty(
                        capabilities=ecs.CfnTaskDefinition.KernelCapabilitiesProperty(
                            add=["add"],
                            drop=["drop"]
                        ),
                        devices=[ecs.CfnTaskDefinition.DeviceProperty(
                            container_path="containerPath",
                            host_path="hostPath",
                            permissions=["permissions"]
                        )],
                        init_process_enabled=False,
                        max_swap=123,
                        shared_memory_size=123,
                        swappiness=123,
                        tmpfs=[ecs.CfnTaskDefinition.TmpfsProperty(
                            size=123,
                
                            # the properties below are optional
                            container_path="containerPath",
                            mount_options=["mountOptions"]
                        )]
                    ),
                    log_configuration=ecs.CfnTaskDefinition.LogConfigurationProperty(
                        log_driver="logDriver",
                
                        # the properties below are optional
                        options={
                            "options_key": "options"
                        },
                        secret_options=[ecs.CfnTaskDefinition.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )]
                    ),
                    memory=123,
                    memory_reservation=123,
                    mount_points=[ecs.CfnTaskDefinition.MountPointProperty(
                        container_path="containerPath",
                        read_only=False,
                        source_volume="sourceVolume"
                    )],
                    port_mappings=[ecs.CfnTaskDefinition.PortMappingProperty(
                        app_protocol="appProtocol",
                        container_port=123,
                        container_port_range="containerPortRange",
                        host_port=123,
                        name="name",
                        protocol="protocol"
                    )],
                    privileged=False,
                    pseudo_terminal=False,
                    readonly_root_filesystem=False,
                    repository_credentials=ecs.CfnTaskDefinition.RepositoryCredentialsProperty(
                        credentials_parameter="credentialsParameter"
                    ),
                    resource_requirements=[ecs.CfnTaskDefinition.ResourceRequirementProperty(
                        type="type",
                        value="value"
                    )],
                    secrets=[ecs.CfnTaskDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )],
                    start_timeout=123,
                    stop_timeout=123,
                    system_controls=[ecs.CfnTaskDefinition.SystemControlProperty(
                        namespace="namespace",
                        value="value"
                    )],
                    ulimits=[ecs.CfnTaskDefinition.UlimitProperty(
                        hard_limit=123,
                        name="name",
                        soft_limit=123
                    )],
                    user="user",
                    volumes_from=[ecs.CfnTaskDefinition.VolumeFromProperty(
                        read_only=False,
                        source_container="sourceContainer"
                    )],
                    working_directory="workingDirectory"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d367f5be98d90056ca7f199c577c5744b20417ce5d1c8ad339824ec9d1a433f2)
                check_type(argname="argument image", value=image, expected_type=type_hints["image"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument command", value=command, expected_type=type_hints["command"])
                check_type(argname="argument cpu", value=cpu, expected_type=type_hints["cpu"])
                check_type(argname="argument depends_on", value=depends_on, expected_type=type_hints["depends_on"])
                check_type(argname="argument disable_networking", value=disable_networking, expected_type=type_hints["disable_networking"])
                check_type(argname="argument dns_search_domains", value=dns_search_domains, expected_type=type_hints["dns_search_domains"])
                check_type(argname="argument dns_servers", value=dns_servers, expected_type=type_hints["dns_servers"])
                check_type(argname="argument docker_labels", value=docker_labels, expected_type=type_hints["docker_labels"])
                check_type(argname="argument docker_security_options", value=docker_security_options, expected_type=type_hints["docker_security_options"])
                check_type(argname="argument entry_point", value=entry_point, expected_type=type_hints["entry_point"])
                check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
                check_type(argname="argument environment_files", value=environment_files, expected_type=type_hints["environment_files"])
                check_type(argname="argument essential", value=essential, expected_type=type_hints["essential"])
                check_type(argname="argument extra_hosts", value=extra_hosts, expected_type=type_hints["extra_hosts"])
                check_type(argname="argument firelens_configuration", value=firelens_configuration, expected_type=type_hints["firelens_configuration"])
                check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument interactive", value=interactive, expected_type=type_hints["interactive"])
                check_type(argname="argument links", value=links, expected_type=type_hints["links"])
                check_type(argname="argument linux_parameters", value=linux_parameters, expected_type=type_hints["linux_parameters"])
                check_type(argname="argument log_configuration", value=log_configuration, expected_type=type_hints["log_configuration"])
                check_type(argname="argument memory", value=memory, expected_type=type_hints["memory"])
                check_type(argname="argument memory_reservation", value=memory_reservation, expected_type=type_hints["memory_reservation"])
                check_type(argname="argument mount_points", value=mount_points, expected_type=type_hints["mount_points"])
                check_type(argname="argument port_mappings", value=port_mappings, expected_type=type_hints["port_mappings"])
                check_type(argname="argument privileged", value=privileged, expected_type=type_hints["privileged"])
                check_type(argname="argument pseudo_terminal", value=pseudo_terminal, expected_type=type_hints["pseudo_terminal"])
                check_type(argname="argument readonly_root_filesystem", value=readonly_root_filesystem, expected_type=type_hints["readonly_root_filesystem"])
                check_type(argname="argument repository_credentials", value=repository_credentials, expected_type=type_hints["repository_credentials"])
                check_type(argname="argument resource_requirements", value=resource_requirements, expected_type=type_hints["resource_requirements"])
                check_type(argname="argument secrets", value=secrets, expected_type=type_hints["secrets"])
                check_type(argname="argument start_timeout", value=start_timeout, expected_type=type_hints["start_timeout"])
                check_type(argname="argument stop_timeout", value=stop_timeout, expected_type=type_hints["stop_timeout"])
                check_type(argname="argument system_controls", value=system_controls, expected_type=type_hints["system_controls"])
                check_type(argname="argument ulimits", value=ulimits, expected_type=type_hints["ulimits"])
                check_type(argname="argument user", value=user, expected_type=type_hints["user"])
                check_type(argname="argument volumes_from", value=volumes_from, expected_type=type_hints["volumes_from"])
                check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "image": image,
                "name": name,
            }
            if command is not None:
                self._values["command"] = command
            if cpu is not None:
                self._values["cpu"] = cpu
            if depends_on is not None:
                self._values["depends_on"] = depends_on
            if disable_networking is not None:
                self._values["disable_networking"] = disable_networking
            if dns_search_domains is not None:
                self._values["dns_search_domains"] = dns_search_domains
            if dns_servers is not None:
                self._values["dns_servers"] = dns_servers
            if docker_labels is not None:
                self._values["docker_labels"] = docker_labels
            if docker_security_options is not None:
                self._values["docker_security_options"] = docker_security_options
            if entry_point is not None:
                self._values["entry_point"] = entry_point
            if environment is not None:
                self._values["environment"] = environment
            if environment_files is not None:
                self._values["environment_files"] = environment_files
            if essential is not None:
                self._values["essential"] = essential
            if extra_hosts is not None:
                self._values["extra_hosts"] = extra_hosts
            if firelens_configuration is not None:
                self._values["firelens_configuration"] = firelens_configuration
            if health_check is not None:
                self._values["health_check"] = health_check
            if hostname is not None:
                self._values["hostname"] = hostname
            if interactive is not None:
                self._values["interactive"] = interactive
            if links is not None:
                self._values["links"] = links
            if linux_parameters is not None:
                self._values["linux_parameters"] = linux_parameters
            if log_configuration is not None:
                self._values["log_configuration"] = log_configuration
            if memory is not None:
                self._values["memory"] = memory
            if memory_reservation is not None:
                self._values["memory_reservation"] = memory_reservation
            if mount_points is not None:
                self._values["mount_points"] = mount_points
            if port_mappings is not None:
                self._values["port_mappings"] = port_mappings
            if privileged is not None:
                self._values["privileged"] = privileged
            if pseudo_terminal is not None:
                self._values["pseudo_terminal"] = pseudo_terminal
            if readonly_root_filesystem is not None:
                self._values["readonly_root_filesystem"] = readonly_root_filesystem
            if repository_credentials is not None:
                self._values["repository_credentials"] = repository_credentials
            if resource_requirements is not None:
                self._values["resource_requirements"] = resource_requirements
            if secrets is not None:
                self._values["secrets"] = secrets
            if start_timeout is not None:
                self._values["start_timeout"] = start_timeout
            if stop_timeout is not None:
                self._values["stop_timeout"] = stop_timeout
            if system_controls is not None:
                self._values["system_controls"] = system_controls
            if ulimits is not None:
                self._values["ulimits"] = ulimits
            if user is not None:
                self._values["user"] = user
            if volumes_from is not None:
                self._values["volumes_from"] = volumes_from
            if working_directory is not None:
                self._values["working_directory"] = working_directory

        @builtins.property
        def image(self) -> builtins.str:
            '''The image used to start a container.

            This string is passed directly to the Docker daemon. By default, images in the Docker Hub registry are available. Other repositories are specified with either ``*repository-url* / *image* : *tag*`` or ``*repository-url* / *image* @ *digest*`` . Up to 255 letters (uppercase and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are allowed. This parameter maps to ``Image`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``IMAGE`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            - When a new task starts, the Amazon ECS container agent pulls the latest version of the specified image and tag for the container to use. However, subsequent updates to a repository image aren't propagated to already running tasks.
            - Images in Amazon ECR repositories can be specified by either using the full ``registry/repository:tag`` or ``registry/repository@digest`` . For example, ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>:latest`` or ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE`` .
            - Images in official repositories on Docker Hub use a single name (for example, ``ubuntu`` or ``mongo`` ).
            - Images in other repositories on Docker Hub are qualified with an organization name (for example, ``amazon/amazon-ecs-agent`` ).
            - Images in other online repositories are qualified further by a domain name (for example, ``quay.io/assemblyline/ubuntu`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-image
            '''
            result = self._values.get("image")
            assert result is not None, "Required property 'image' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of a container.

            If you're linking multiple containers together in a task definition, the ``name`` of one container can be entered in the ``links`` of another container to connect the containers. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. This parameter maps to ``name`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--name`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The command that's passed to the container.

            This parameter maps to ``Cmd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``COMMAND`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#cmd <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ . If there are multiple arguments, each argument is a separated string in the array.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cpu(self) -> typing.Optional[jsii.Number]:
            '''The number of ``cpu`` units reserved for the container.

            This parameter maps to ``CpuShares`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cpu-shares`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            This field is optional for tasks using the Fargate launch type, and the only requirement is that the total amount of CPU reserved for all containers within a task be lower than the task-level ``cpu`` value.
            .. epigraph::

               You can determine the number of CPU units that are available per EC2 instance type by multiplying the vCPUs listed for that instance type on the `Amazon EC2 Instances <https://docs.aws.amazon.com/ec2/instance-types/>`_ detail page by 1,024.

            Linux containers share unallocated CPU units with other containers on the container instance with the same ratio as their allocated amount. For example, if you run a single-container task on a single-core instance type with 512 CPU units specified for that container, and that's the only task running on the container instance, that container could use the full 1,024 CPU unit share at any given time. However, if you launched another copy of the same task on that container instance, each task is guaranteed a minimum of 512 CPU units when needed. Moreover, each container could float to higher CPU usage if the other container was not using it. If both tasks were 100% active all of the time, they would be limited to 512 CPU units.

            On Linux container instances, the Docker daemon on the container instance uses the CPU value to calculate the relative CPU share ratios for running containers. For more information, see `CPU share constraint <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#cpu-share-constraint>`_ in the Docker documentation. The minimum valid CPU share value that the Linux kernel allows is 2. However, the CPU parameter isn't required, and you can use CPU values below 2 in your container definitions. For CPU values below 2 (including null), the behavior varies based on your Amazon ECS container agent version:

            - *Agent versions less than or equal to 1.1.0:* Null and zero CPU values are passed to Docker as 0, which Docker then converts to 1,024 CPU shares. CPU values of 1 are passed to Docker as 1, which the Linux kernel converts to two CPU shares.
            - *Agent versions greater than or equal to 1.2.0:* Null, zero, and CPU values of 1 are passed to Docker as 2.

            On Windows container instances, the CPU limit is enforced as an absolute limit, or a quota. Windows containers only have access to the specified amount of CPU that's described in the task definition. A null or zero CPU value is passed to Docker as ``0`` , which Windows interprets as 1% of one CPU.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-cpu
            '''
            result = self._values.get("cpu")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def depends_on(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ContainerDependencyProperty"]]]]:
            '''The dependencies defined for container startup and shutdown.

            A container can contain multiple dependencies. When a dependency is defined for container startup, for container shutdown it is reversed.

            For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container agent to turn on container dependencies. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For tasks using the Fargate launch type, the task or service requires the following platforms:

            - Linux platform version ``1.3.0`` or later.
            - Windows platform version ``1.0.0`` or later.

            If the task definition is used in a blue/green deployment that uses `AWS::CodeDeploy::DeploymentGroup BlueGreenDeploymentConfiguration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codedeploy-deploymentgroup-bluegreendeploymentconfiguration.html>`_ , the ``dependsOn`` parameter is not supported. For more information see `Issue #680 <https://docs.aws.amazon.com/https://github.com/aws-cloudformation/cloudformation-coverage-roadmap/issues/680>`_ on the on the GitHub website.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-dependson
            '''
            result = self._values.get("depends_on")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ContainerDependencyProperty"]]]], result)

        @builtins.property
        def disable_networking(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When this parameter is true, networking is off within the container.

            This parameter maps to ``NetworkDisabled`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-disablenetworking
            '''
            result = self._values.get("disable_networking")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def dns_search_domains(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of DNS search domains that are presented to the container.

            This parameter maps to ``DnsSearch`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--dns-search`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-dnssearchdomains
            '''
            result = self._values.get("dns_search_domains")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of DNS servers that are presented to the container.

            This parameter maps to ``Dns`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--dns`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-dnsservers
            '''
            result = self._values.get("dns_servers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def docker_labels(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]]:
            '''A key/value map of labels to add to the container.

            This parameter maps to ``Labels`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--label`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-dockerlabels
            '''
            result = self._values.get("docker_labels")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def docker_security_options(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of strings to provide custom configuration for multiple security systems.

            For more information about valid values, see `Docker Run Security Configuration <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This field isn't valid for containers in tasks using the Fargate launch type.

            For Linux tasks on EC2, this parameter can be used to reference custom labels for SELinux and AppArmor multi-level security systems.

            For any tasks on EC2, this parameter can be used to reference a credential spec file that configures a container for Active Directory authentication. For more information, see `Using gMSAs for Windows Containers <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html>`_ and `Using gMSAs for Linux Containers <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            This parameter maps to ``SecurityOpt`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--security-opt`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               The Amazon ECS container agent running on a container instance must register with the ``ECS_SELINUX_CAPABLE=true`` or ``ECS_APPARMOR_CAPABLE=true`` environment variables before containers placed on that instance can use these security options. For more information, see `Amazon ECS Container Agent Configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For more information about valid values, see `Docker Run Security Configuration <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" | "credentialspec:CredentialSpecFilePath"

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-dockersecurityoptions
            '''
            result = self._values.get("docker_security_options")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
            '''.. epigraph::

   Early versions of the Amazon ECS container agent don't properly handle ``entryPoint`` parameters.

            If you have problems using ``entryPoint`` , update your container agent or enter your commands and arguments as ``command`` array items instead.

            The entry point that's passed to the container. This parameter maps to ``Entrypoint`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--entrypoint`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#entrypoint <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#entrypoint>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-entrypoint
            '''
            result = self._values.get("entry_point")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def environment(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.KeyValuePairProperty"]]]]:
            '''The environment variables to pass to a container.

            This parameter maps to ``Env`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--env`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               We don't recommend that you use plaintext environment variables for sensitive information, such as credential data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-environment
            '''
            result = self._values.get("environment")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.KeyValuePairProperty"]]]], result)

        @builtins.property
        def environment_files(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.EnvironmentFileProperty"]]]]:
            '''A list of files containing the environment variables to pass to a container.

            This parameter maps to the ``--env-file`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            You can specify up to ten environment files. The file must have a ``.env`` file extension. Each line in an environment file contains an environment variable in ``VARIABLE=VALUE`` format. Lines beginning with ``#`` are treated as comments and are ignored. For more information about the environment variable file syntax, see `Declare default environment variables in file <https://docs.aws.amazon.com/https://docs.docker.com/compose/env-file/>`_ .

            If there are environment variables specified using the ``environment`` parameter in a container definition, they take precedence over the variables contained within an environment file. If multiple environment files are specified that contain the same variable, they're processed from the top down. We recommend that you use unique variable names. For more information, see `Specifying Environment Variables <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-environmentfiles
            '''
            result = self._values.get("environment_files")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.EnvironmentFileProperty"]]]], result)

        @builtins.property
        def essential(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If the ``essential`` parameter of a container is marked as ``true`` , and that container fails or stops for any reason, all other containers that are part of the task are stopped.

            If the ``essential`` parameter of a container is marked as ``false`` , its failure doesn't affect the rest of the containers in a task. If this parameter is omitted, a container is assumed to be essential.

            All tasks must have at least one essential container. If you have an application that's composed of multiple containers, group containers that are used for a common purpose into components, and separate the different components into multiple task definitions. For more information, see `Application Architecture <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-essential
            '''
            result = self._values.get("essential")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def extra_hosts(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.HostEntryProperty"]]]]:
            '''A list of hostnames and IP address mappings to append to the ``/etc/hosts`` file on the container.

            This parameter maps to ``ExtraHosts`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--add-host`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter isn't supported for Windows containers or tasks that use the ``awsvpc`` network mode.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-extrahosts
            '''
            result = self._values.get("extra_hosts")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.HostEntryProperty"]]]], result)

        @builtins.property
        def firelens_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.FirelensConfigurationProperty"]]:
            '''The FireLens configuration for the container.

            This is used to specify and configure a log router for container logs. For more information, see `Custom Log Routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-firelensconfiguration
            '''
            result = self._values.get("firelens_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.FirelensConfigurationProperty"]], result)

        @builtins.property
        def health_check(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.HealthCheckProperty"]]:
            '''The container health check command and associated configuration parameters for the container.

            This parameter maps to ``HealthCheck`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``HEALTHCHECK`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-healthcheck
            '''
            result = self._values.get("health_check")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.HealthCheckProperty"]], result)

        @builtins.property
        def hostname(self) -> typing.Optional[builtins.str]:
            '''The hostname to use for your container.

            This parameter maps to ``Hostname`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--hostname`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               The ``hostname`` parameter is not supported if you're using the ``awsvpc`` network mode.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def interactive(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When this parameter is ``true`` , you can deploy containerized applications that require ``stdin`` or a ``tty`` to be allocated.

            This parameter maps to ``OpenStdin`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--interactive`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-interactive
            '''
            result = self._values.get("interactive")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def links(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The ``links`` parameter allows containers to communicate with each other without the need for port mappings.

            This parameter is only supported if the network mode of a task definition is ``bridge`` . The ``name:internalName`` construct is analogous to ``name:alias`` in Docker links. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. For more information about linking Docker containers, go to `Legacy container links <https://docs.aws.amazon.com/https://docs.docker.com/network/links/>`_ in the Docker documentation. This parameter maps to ``Links`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--link`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers. > Containers that are collocated on a single container instance may be able to communicate with each other without requiring links or host port mappings. Network isolation is achieved on the container instance using security groups and VPC settings.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-links
            '''
            result = self._values.get("links")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def linux_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.LinuxParametersProperty"]]:
            '''Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_ .

            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-linuxparameters
            '''
            result = self._values.get("linux_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.LinuxParametersProperty"]], result)

        @builtins.property
        def log_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.LogConfigurationProperty"]]:
            '''The log configuration specification for the container.

            This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--log-driver`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . By default, containers use the same logging driver that the Docker daemon uses. However, the container may use a different logging driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a different logging driver for a container, the log system must be configured properly on the container instance (or on a different log server for remote logging options). For more information on the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the Docker documentation.
            .. epigraph::

               Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the `LogConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_LogConfiguration.html>`_ data type). Additional log drivers may be available in future releases of the Amazon ECS container agent.

            This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            .. epigraph::

               The Amazon ECS container agent running on a container instance must register the logging drivers available on that instance with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS Container Agent Configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-logconfiguration
            '''
            result = self._values.get("log_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.LogConfigurationProperty"]], result)

        @builtins.property
        def memory(self) -> typing.Optional[jsii.Number]:
            '''The amount (in MiB) of memory to present to the container.

            If your container attempts to exceed the memory specified here, the container is killed. The total amount of memory reserved for all containers within a task must be lower than the task ``memory`` value, if one is specified. This parameter maps to ``Memory`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--memory`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            If using the Fargate launch type, this parameter is optional.

            If using the EC2 launch type, you must specify either a task-level memory value or a container-level memory value. If you specify both a container-level ``memory`` and ``memoryReservation`` value, ``memory`` must be greater than ``memoryReservation`` . If you specify ``memoryReservation`` , then that value is subtracted from the available memory resources for the container instance where the container is placed. Otherwise, the value of ``memory`` is used.

            The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container, so you should not specify fewer than 6 MiB of memory for your containers.

            The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container, so you should not specify fewer than 4 MiB of memory for your containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-memory
            '''
            result = self._values.get("memory")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def memory_reservation(self) -> typing.Optional[jsii.Number]:
            '''The soft limit (in MiB) of memory to reserve for the container.

            When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the ``memory`` parameter (if applicable), or all of the available memory on the container instance, whichever comes first. This parameter maps to ``MemoryReservation`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--memory-reservation`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            If a task-level memory value is not specified, you must specify a non-zero integer for one or both of ``memory`` or ``memoryReservation`` in a container definition. If you specify both, ``memory`` must be greater than ``memoryReservation`` . If you specify ``memoryReservation`` , then that value is subtracted from the available memory resources for the container instance where the container is placed. Otherwise, the value of ``memory`` is used.

            For example, if your container normally uses 128 MiB of memory, but occasionally bursts to 256 MiB of memory for short periods of time, you can set a ``memoryReservation`` of 128 MiB, and a ``memory`` hard limit of 300 MiB. This configuration would allow the container to only reserve 128 MiB of memory from the remaining resources on the container instance, but also allow the container to consume more memory resources when needed.

            The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't specify less than 6 MiB of memory for your containers.

            The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So, don't specify less than 4 MiB of memory for your containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-memoryreservation
            '''
            result = self._values.get("memory_reservation")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def mount_points(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.MountPointProperty"]]]]:
            '''The mount points for data volumes in your container.

            This parameter maps to ``Volumes`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--volume`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-mountpoints
            '''
            result = self._values.get("mount_points")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.MountPointProperty"]]]], result)

        @builtins.property
        def port_mappings(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.PortMappingProperty"]]]]:
            '''The list of port mappings for the container.

            Port mappings allow containers to access ports on the host container instance to send or receive traffic.

            For task definitions that use the ``awsvpc`` network mode, you should only specify the ``containerPort`` . The ``hostPort`` can be left blank or it must be the same value as the ``containerPort`` .

            Port mappings on Windows use the ``NetNAT`` gateway address rather than ``localhost`` . There is no loopback for port mappings on Windows, so you cannot access a container's mapped port from the host itself.

            This parameter maps to ``PortBindings`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--publish`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . If the network mode of a task definition is set to ``none`` , then you can't specify port mappings. If the network mode of a task definition is set to ``host`` , then host ports must either be undefined or they must match the container port in the port mapping.
            .. epigraph::

               After a task reaches the ``RUNNING`` status, manual and automatic host and container port assignments are visible in the *Network Bindings* section of a container description for a selected task in the Amazon ECS console. The assignments are also visible in the ``networkBindings`` section `DescribeTasks <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html>`_ responses.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-portmappings
            '''
            result = self._values.get("port_mappings")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.PortMappingProperty"]]]], result)

        @builtins.property
        def privileged(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When this parameter is true, the container is given elevated privileges on the host container instance (similar to the ``root`` user).

            This parameter maps to ``Privileged`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--privileged`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers or tasks run on AWS Fargate .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-privileged
            '''
            result = self._values.get("privileged")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def pseudo_terminal(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When this parameter is ``true`` , a TTY is allocated.

            This parameter maps to ``Tty`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--tty`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-pseudoterminal
            '''
            result = self._values.get("pseudo_terminal")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def readonly_root_filesystem(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When this parameter is true, the container is given read-only access to its root file system.

            This parameter maps to ``ReadonlyRootfs`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--read-only`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-readonlyrootfilesystem
            '''
            result = self._values.get("readonly_root_filesystem")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def repository_credentials(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.RepositoryCredentialsProperty"]]:
            '''The private repository authentication credentials to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-repositorycredentials
            '''
            result = self._values.get("repository_credentials")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.RepositoryCredentialsProperty"]], result)

        @builtins.property
        def resource_requirements(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ResourceRequirementProperty"]]]]:
            '''The type and amount of a resource to assign to a container.

            The only supported resource is a GPU.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-resourcerequirements
            '''
            result = self._values.get("resource_requirements")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.ResourceRequirementProperty"]]]], result)

        @builtins.property
        def secrets(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.SecretProperty"]]]]:
            '''The secrets to pass to the container.

            For more information, see `Specifying Sensitive Data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-secrets
            '''
            result = self._values.get("secrets")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.SecretProperty"]]]], result)

        @builtins.property
        def start_timeout(self) -> typing.Optional[jsii.Number]:
            '''Time duration (in seconds) to wait before giving up on resolving dependencies for a container.

            For example, you specify two containers in a task definition with containerA having a dependency on containerB reaching a ``COMPLETE`` , ``SUCCESS`` , or ``HEALTHY`` status. If a ``startTimeout`` value is specified for containerB and it doesn't reach the desired status within that time then containerA gives up and not start. This results in the task transitioning to a ``STOPPED`` state.
            .. epigraph::

               When the ``ECS_CONTAINER_START_TIMEOUT`` container agent configuration variable is used, it's enforced independently from this start timeout value.

            For tasks using the Fargate launch type, the task or service requires the following platforms:

            - Linux platform version ``1.3.0`` or later.
            - Windows platform version ``1.0.0`` or later.

            For tasks using the EC2 launch type, your container instances require at least version ``1.26.0`` of the container agent to use a container start timeout value. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version ``1.26.0-1`` of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-starttimeout
            '''
            result = self._values.get("start_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stop_timeout(self) -> typing.Optional[jsii.Number]:
            '''Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own.

            For tasks using the Fargate launch type, the task or service requires the following platforms:

            - Linux platform version ``1.3.0`` or later.
            - Windows platform version ``1.0.0`` or later.

            The max stop timeout value is 120 seconds and if the parameter is not specified, the default value of 30 seconds is used.

            For tasks that use the EC2 launch type, if the ``stopTimeout`` parameter isn't specified, the value set for the Amazon ECS container agent configuration variable ``ECS_CONTAINER_STOP_TIMEOUT`` is used. If neither the ``stopTimeout`` parameter or the ``ECS_CONTAINER_STOP_TIMEOUT`` agent configuration variable are set, then the default values of 30 seconds for Linux containers and 30 seconds on Windows containers are used. Your container instances require at least version 1.26.0 of the container agent to use a container stop timeout value. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-stoptimeout
            '''
            result = self._values.get("stop_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def system_controls(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.SystemControlProperty"]]]]:
            '''A list of namespaced kernel parameters to set in the container.

            This parameter maps to ``Sysctls`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--sysctl`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               We don't recommended that you specify network-related ``systemControls`` parameters for multiple containers in a single task that also uses either the ``awsvpc`` or ``host`` network modes. For tasks that use the ``awsvpc`` network mode, the container that's started last determines which ``systemControls`` parameters take effect. For tasks that use the ``host`` network mode, it changes the container instance's namespaced kernel parameters as well as the containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-systemcontrols
            '''
            result = self._values.get("system_controls")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.SystemControlProperty"]]]], result)

        @builtins.property
        def ulimits(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.UlimitProperty"]]]]:
            '''A list of ``ulimits`` to set in the container.

            This parameter maps to ``Ulimits`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--ulimit`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . Valid naming values are displayed in the `Ulimit <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html>`_ data type. This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-ulimits
            '''
            result = self._values.get("ulimits")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.UlimitProperty"]]]], result)

        @builtins.property
        def user(self) -> typing.Optional[builtins.str]:
            '''The user to use inside the container.

            This parameter maps to ``User`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--user`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               When running tasks using the ``host`` network mode, don't run containers using the root user (UID 0). We recommend using a non-root user for better security.

            You can specify the ``user`` using the following formats. If specifying a UID or GID, you must specify it as a positive integer.

            - ``user``
            - ``user:group``
            - ``uid``
            - ``uid:gid``
            - ``user:gid``
            - ``uid:group``

            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-user
            '''
            result = self._values.get("user")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def volumes_from(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.VolumeFromProperty"]]]]:
            '''Data volumes to mount from another container.

            This parameter maps to ``VolumesFrom`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--volumes-from`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-volumesfrom
            '''
            result = self._values.get("volumes_from")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.VolumeFromProperty"]]]], result)

        @builtins.property
        def working_directory(self) -> typing.Optional[builtins.str]:
            '''The working directory to run commands inside the container in.

            This parameter maps to ``WorkingDir`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--workdir`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinition.html#cfn-ecs-taskdefinition-containerdefinition-workingdirectory
            '''
            result = self._values.get("working_directory")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContainerDefinitionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.ContainerDependencyProperty",
        jsii_struct_bases=[],
        name_mapping={"condition": "condition", "container_name": "containerName"},
    )
    class ContainerDependencyProperty:
        def __init__(
            self,
            *,
            condition: typing.Optional[builtins.str] = None,
            container_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ContainerDependency`` property specifies the dependencies defined for container startup and shutdown.

            A container can contain multiple dependencies. When a dependency is defined for container startup, for container shutdown it is reversed.

            Your Amazon ECS container instances require at least version 1.26.0 of the container agent to enable container dependencies. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you are using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               For tasks using the Fargate launch type, this parameter requires that the task or service uses platform version 1.3.0 or later.

            :param condition: The dependency condition of the container. The following are the available conditions and their behavior:. - ``START`` - This condition emulates the behavior of links and volumes today. It validates that a dependent container is started before permitting other containers to start. - ``COMPLETE`` - This condition validates that a dependent container runs to completion (exits) before permitting other containers to start. This can be useful for nonessential containers that run a script and then exit. This condition can't be set on an essential container. - ``SUCCESS`` - This condition is the same as ``COMPLETE`` , but it also requires that the container exits with a ``zero`` status. This condition can't be set on an essential container. - ``HEALTHY`` - This condition validates that the dependent container passes its Docker health check before permitting other containers to start. This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.
            :param container_name: The name of a container.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdependency.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                container_dependency_property = ecs.CfnTaskDefinition.ContainerDependencyProperty(
                    condition="condition",
                    container_name="containerName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1a4b3aeb3a959441694c28ccdeedcc2d1bcf44aa689db8f4b214a84afd414a43)
                check_type(argname="argument condition", value=condition, expected_type=type_hints["condition"])
                check_type(argname="argument container_name", value=container_name, expected_type=type_hints["container_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if condition is not None:
                self._values["condition"] = condition
            if container_name is not None:
                self._values["container_name"] = container_name

        @builtins.property
        def condition(self) -> typing.Optional[builtins.str]:
            '''The dependency condition of the container. The following are the available conditions and their behavior:.

            - ``START`` - This condition emulates the behavior of links and volumes today. It validates that a dependent container is started before permitting other containers to start.
            - ``COMPLETE`` - This condition validates that a dependent container runs to completion (exits) before permitting other containers to start. This can be useful for nonessential containers that run a script and then exit. This condition can't be set on an essential container.
            - ``SUCCESS`` - This condition is the same as ``COMPLETE`` , but it also requires that the container exits with a ``zero`` status. This condition can't be set on an essential container.
            - ``HEALTHY`` - This condition validates that the dependent container passes its Docker health check before permitting other containers to start. This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdependency.html#cfn-ecs-taskdefinition-containerdependency-condition
            '''
            result = self._values.get("condition")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''The name of a container.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdependency.html#cfn-ecs-taskdefinition-containerdependency-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContainerDependencyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.DeviceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "host_path": "hostPath",
            "permissions": "permissions",
        },
    )
    class DeviceProperty:
        def __init__(
            self,
            *,
            container_path: typing.Optional[builtins.str] = None,
            host_path: typing.Optional[builtins.str] = None,
            permissions: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``Device`` property specifies an object representing a container instance host device.

            :param container_path: The path inside the container at which to expose the host device.
            :param host_path: The path for the device on the host container instance.
            :param permissions: The explicit permissions to provide to the container for the device. By default, the container has permissions for ``read`` , ``write`` , and ``mknod`` for the device.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                device_property = ecs.CfnTaskDefinition.DeviceProperty(
                    container_path="containerPath",
                    host_path="hostPath",
                    permissions=["permissions"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2e035af2a0e5b0a2bc30657f34497d08308b62947e2a10177938a13809ca414f)
                check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
                check_type(argname="argument host_path", value=host_path, expected_type=type_hints["host_path"])
                check_type(argname="argument permissions", value=permissions, expected_type=type_hints["permissions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if container_path is not None:
                self._values["container_path"] = container_path
            if host_path is not None:
                self._values["host_path"] = host_path
            if permissions is not None:
                self._values["permissions"] = permissions

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''The path inside the container at which to expose the host device.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html#cfn-ecs-taskdefinition-device-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_path(self) -> typing.Optional[builtins.str]:
            '''The path for the device on the host container instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html#cfn-ecs-taskdefinition-device-hostpath
            '''
            result = self._values.get("host_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def permissions(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The explicit permissions to provide to the container for the device.

            By default, the container has permissions for ``read`` , ``write`` , and ``mknod`` for the device.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html#cfn-ecs-taskdefinition-device-permissions
            '''
            result = self._values.get("permissions")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeviceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "autoprovision": "autoprovision",
            "driver": "driver",
            "driver_opts": "driverOpts",
            "labels": "labels",
            "scope": "scope",
        },
    )
    class DockerVolumeConfigurationProperty:
        def __init__(
            self,
            *,
            autoprovision: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            driver: typing.Optional[builtins.str] = None,
            driver_opts: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]] = None,
            labels: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]] = None,
            scope: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``DockerVolumeConfiguration`` property specifies a Docker volume configuration and is used when you use Docker volumes.

            Docker volumes are only supported when you are using the EC2 launch type. Windows containers only support the use of the ``local`` driver. To use bind mounts, specify a ``host`` instead.

            :param autoprovision: If this value is ``true`` , the Docker volume is created if it doesn't already exist. .. epigraph:: This field is only used if the ``scope`` is ``shared`` .
            :param driver: The Docker volume driver to use. The driver value must match the driver name provided by Docker because it is used for task placement. If the driver was installed using the Docker plugin CLI, use ``docker plugin ls`` to retrieve the driver name from your container instance. If the driver was installed using another method, use Docker plugin discovery to retrieve the driver name. For more information, see `Docker plugin discovery <https://docs.aws.amazon.com/https://docs.docker.com/engine/extend/plugin_api/#plugin-discovery>`_ . This parameter maps to ``Driver`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxdriver`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .
            :param driver_opts: A map of Docker driver-specific options passed through. This parameter maps to ``DriverOpts`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxopt`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .
            :param labels: Custom metadata to add to your Docker volume. This parameter maps to ``Labels`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxlabel`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .
            :param scope: The scope for the Docker volume that determines its lifecycle. Docker volumes that are scoped to a ``task`` are automatically provisioned when the task starts and destroyed when the task stops. Docker volumes that are scoped as ``shared`` persist after the task stops.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                docker_volume_configuration_property = ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty(
                    autoprovision=False,
                    driver="driver",
                    driver_opts={
                        "driver_opts_key": "driverOpts"
                    },
                    labels={
                        "labels_key": "labels"
                    },
                    scope="scope"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fa8da109bc6888a0534920dcd1b365d2de33c75ccc0bbb15b5246ab7536a166f)
                check_type(argname="argument autoprovision", value=autoprovision, expected_type=type_hints["autoprovision"])
                check_type(argname="argument driver", value=driver, expected_type=type_hints["driver"])
                check_type(argname="argument driver_opts", value=driver_opts, expected_type=type_hints["driver_opts"])
                check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
                check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if autoprovision is not None:
                self._values["autoprovision"] = autoprovision
            if driver is not None:
                self._values["driver"] = driver
            if driver_opts is not None:
                self._values["driver_opts"] = driver_opts
            if labels is not None:
                self._values["labels"] = labels
            if scope is not None:
                self._values["scope"] = scope

        @builtins.property
        def autoprovision(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If this value is ``true`` , the Docker volume is created if it doesn't already exist.

            .. epigraph::

               This field is only used if the ``scope`` is ``shared`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-autoprovision
            '''
            result = self._values.get("autoprovision")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def driver(self) -> typing.Optional[builtins.str]:
            '''The Docker volume driver to use.

            The driver value must match the driver name provided by Docker because it is used for task placement. If the driver was installed using the Docker plugin CLI, use ``docker plugin ls`` to retrieve the driver name from your container instance. If the driver was installed using another method, use Docker plugin discovery to retrieve the driver name. For more information, see `Docker plugin discovery <https://docs.aws.amazon.com/https://docs.docker.com/engine/extend/plugin_api/#plugin-discovery>`_ . This parameter maps to ``Driver`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxdriver`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-driver
            '''
            result = self._values.get("driver")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def driver_opts(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]]:
            '''A map of Docker driver-specific options passed through.

            This parameter maps to ``DriverOpts`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxopt`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-driveropts
            '''
            result = self._values.get("driver_opts")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def labels(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]]:
            '''Custom metadata to add to your Docker volume.

            This parameter maps to ``Labels`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxlabel`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-labels
            '''
            result = self._values.get("labels")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def scope(self) -> typing.Optional[builtins.str]:
            '''The scope for the Docker volume that determines its lifecycle.

            Docker volumes that are scoped to a ``task`` are automatically provisioned when the task starts and destroyed when the task stops. Docker volumes that are scoped as ``shared`` persist after the task stops.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-scope
            '''
            result = self._values.get("scope")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DockerVolumeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.EFSVolumeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "filesystem_id": "filesystemId",
            "authorization_config": "authorizationConfig",
            "root_directory": "rootDirectory",
            "transit_encryption": "transitEncryption",
            "transit_encryption_port": "transitEncryptionPort",
        },
    )
    class EFSVolumeConfigurationProperty:
        def __init__(
            self,
            *,
            filesystem_id: builtins.str,
            authorization_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.AuthorizationConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            root_directory: typing.Optional[builtins.str] = None,
            transit_encryption: typing.Optional[builtins.str] = None,
            transit_encryption_port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''This parameter is specified when you're using an Amazon Elastic File System file system for task storage.

            For more information, see `Amazon EFS volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/efs-volumes.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param filesystem_id: The Amazon EFS file system ID to use.
            :param authorization_config: The authorization configuration details for the Amazon EFS file system.
            :param root_directory: The directory within the Amazon EFS file system to mount as the root directory inside the host. If this parameter is omitted, the root of the Amazon EFS volume will be used. Specifying ``/`` will have the same effect as omitting this parameter. .. epigraph:: If an EFS access point is specified in the ``authorizationConfig`` , the root directory parameter must either be omitted or set to ``/`` which will enforce the path set on the EFS access point.
            :param transit_encryption: Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server. Transit encryption must be turned on if Amazon EFS IAM authorization is used. If this parameter is omitted, the default value of ``DISABLED`` is used. For more information, see `Encrypting data in transit <https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html>`_ in the *Amazon Elastic File System User Guide* .
            :param transit_encryption_port: The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server. If you do not specify a transit encryption port, it will use the port selection strategy that the Amazon EFS mount helper uses. For more information, see `EFS mount helper <https://docs.aws.amazon.com/efs/latest/ug/efs-mount-helper.html>`_ in the *Amazon Elastic File System User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                e_fSVolume_configuration_property = ecs.CfnTaskDefinition.EFSVolumeConfigurationProperty(
                    filesystem_id="filesystemId",
                
                    # the properties below are optional
                    authorization_config=ecs.CfnTaskDefinition.AuthorizationConfigProperty(
                        access_point_id="accessPointId",
                        iam="iam"
                    ),
                    root_directory="rootDirectory",
                    transit_encryption="transitEncryption",
                    transit_encryption_port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__589f5a30129158655f4ca2eb0312f25c77f9d0890b6066733d26235074123665)
                check_type(argname="argument filesystem_id", value=filesystem_id, expected_type=type_hints["filesystem_id"])
                check_type(argname="argument authorization_config", value=authorization_config, expected_type=type_hints["authorization_config"])
                check_type(argname="argument root_directory", value=root_directory, expected_type=type_hints["root_directory"])
                check_type(argname="argument transit_encryption", value=transit_encryption, expected_type=type_hints["transit_encryption"])
                check_type(argname="argument transit_encryption_port", value=transit_encryption_port, expected_type=type_hints["transit_encryption_port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "filesystem_id": filesystem_id,
            }
            if authorization_config is not None:
                self._values["authorization_config"] = authorization_config
            if root_directory is not None:
                self._values["root_directory"] = root_directory
            if transit_encryption is not None:
                self._values["transit_encryption"] = transit_encryption
            if transit_encryption_port is not None:
                self._values["transit_encryption_port"] = transit_encryption_port

        @builtins.property
        def filesystem_id(self) -> builtins.str:
            '''The Amazon EFS file system ID to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-filesystemid
            '''
            result = self._values.get("filesystem_id")
            assert result is not None, "Required property 'filesystem_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def authorization_config(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.AuthorizationConfigProperty"]]:
            '''The authorization configuration details for the Amazon EFS file system.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-authorizationconfig
            '''
            result = self._values.get("authorization_config")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.AuthorizationConfigProperty"]], result)

        @builtins.property
        def root_directory(self) -> typing.Optional[builtins.str]:
            '''The directory within the Amazon EFS file system to mount as the root directory inside the host.

            If this parameter is omitted, the root of the Amazon EFS volume will be used. Specifying ``/`` will have the same effect as omitting this parameter.
            .. epigraph::

               If an EFS access point is specified in the ``authorizationConfig`` , the root directory parameter must either be omitted or set to ``/`` which will enforce the path set on the EFS access point.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-rootdirectory
            '''
            result = self._values.get("root_directory")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def transit_encryption(self) -> typing.Optional[builtins.str]:
            '''Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server.

            Transit encryption must be turned on if Amazon EFS IAM authorization is used. If this parameter is omitted, the default value of ``DISABLED`` is used. For more information, see `Encrypting data in transit <https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html>`_ in the *Amazon Elastic File System User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-transitencryption
            '''
            result = self._values.get("transit_encryption")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def transit_encryption_port(self) -> typing.Optional[jsii.Number]:
            '''The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server.

            If you do not specify a transit encryption port, it will use the port selection strategy that the Amazon EFS mount helper uses. For more information, see `EFS mount helper <https://docs.aws.amazon.com/efs/latest/ug/efs-mount-helper.html>`_ in the *Amazon Elastic File System User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-transitencryptionport
            '''
            result = self._values.get("transit_encryption_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EFSVolumeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.EnvironmentFileProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "value": "value"},
    )
    class EnvironmentFileProperty:
        def __init__(
            self,
            *,
            type: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A list of files containing the environment variables to pass to a container.

            You can specify up to ten environment files. The file must have a ``.env`` file extension. Each line in an environment file should contain an environment variable in ``VARIABLE=VALUE`` format. Lines beginning with ``#`` are treated as comments and are ignored. For more information about the environment variable file syntax, see `Declare default environment variables in file <https://docs.aws.amazon.com/https://docs.docker.com/compose/env-file/>`_ .

            If there are environment variables specified using the ``environment`` parameter in a container definition, they take precedence over the variables contained within an environment file. If multiple environment files are specified that contain the same variable, they're processed from the top down. We recommend that you use unique variable names. For more information, see `Specifying environment variables <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            This parameter is only supported for tasks hosted on Fargate using the following platform versions:

            - Linux platform version ``1.4.0`` or later.
            - Windows platform version ``1.0.0`` or later.

            :param type: The file type to use. The only supported value is ``s3`` .
            :param value: The Amazon Resource Name (ARN) of the Amazon S3 object containing the environment variable file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-environmentfile.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                environment_file_property = ecs.CfnTaskDefinition.EnvironmentFileProperty(
                    type="type",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__08857275a4e4136ad48cda0357d7696b4810f7b25fe754aa697c4235aa941fe9)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The file type to use.

            The only supported value is ``s3`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-environmentfile.html#cfn-ecs-taskdefinition-environmentfile-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the Amazon S3 object containing the environment variable file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-environmentfile.html#cfn-ecs-taskdefinition-environmentfile-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnvironmentFileProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.EphemeralStorageProperty",
        jsii_struct_bases=[],
        name_mapping={"size_in_gib": "sizeInGiB"},
    )
    class EphemeralStorageProperty:
        def __init__(self, *, size_in_gib: typing.Optional[jsii.Number] = None) -> None:
            '''The amount of ephemeral storage to allocate for the task.

            This parameter is used to expand the total amount of ephemeral storage available, beyond the default amount, for tasks hosted on AWS Fargate . For more information, see `Fargate task storage <https://docs.aws.amazon.com/AmazonECS/latest/userguide/using_data_volumes.html>`_ in the *Amazon ECS User Guide for AWS Fargate* .
            .. epigraph::

               For tasks using the Fargate launch type, the task requires the following platforms:

               - Linux platform version ``1.4.0`` or later.
               - Windows platform version ``1.0.0`` or later.

            :param size_in_gib: The total amount, in GiB, of ephemeral storage to set for the task. The minimum supported value is ``21`` GiB and the maximum supported value is ``200`` GiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ephemeralstorage.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                ephemeral_storage_property = ecs.CfnTaskDefinition.EphemeralStorageProperty(
                    size_in_gi_b=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fdddb0d48923d3583475a8b34d0a6079606e57e6de57c6ca22978691d87763db)
                check_type(argname="argument size_in_gib", value=size_in_gib, expected_type=type_hints["size_in_gib"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if size_in_gib is not None:
                self._values["size_in_gib"] = size_in_gib

        @builtins.property
        def size_in_gib(self) -> typing.Optional[jsii.Number]:
            '''The total amount, in GiB, of ephemeral storage to set for the task.

            The minimum supported value is ``21`` GiB and the maximum supported value is ``200`` GiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ephemeralstorage.html#cfn-ecs-taskdefinition-ephemeralstorage-sizeingib
            '''
            result = self._values.get("size_in_gib")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EphemeralStorageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.FirelensConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"options": "options", "type": "type"},
    )
    class FirelensConfigurationProperty:
        def __init__(
            self,
            *,
            options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]] = None,
            type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The FireLens configuration for the container.

            This is used to specify and configure a log router for container logs. For more information, see `Custom log routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param options: The options to use when configuring the log router. This field is optional and can be used to add additional metadata, such as the task, task definition, cluster, and container instance details to the log event. If specified, valid option keys are: - ``enable-ecs-log-metadata`` , which can be ``true`` or ``false`` - ``config-file-type`` , which can be ``s3`` or ``file`` - ``config-file-value`` , which is either an S3 ARN or a file path
            :param type: The log router to use. The valid values are ``fluentd`` or ``fluentbit`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-firelensconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                firelens_configuration_property = ecs.CfnTaskDefinition.FirelensConfigurationProperty(
                    options={
                        "options_key": "options"
                    },
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6a1cd84d62e34083a37bc4c25d849ac547dacc5fe8a4db04170cf83e3cd1e9d3)
                check_type(argname="argument options", value=options, expected_type=type_hints["options"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if options is not None:
                self._values["options"] = options
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]]:
            '''The options to use when configuring the log router.

            This field is optional and can be used to add additional metadata, such as the task, task definition, cluster, and container instance details to the log event.

            If specified, valid option keys are:

            - ``enable-ecs-log-metadata`` , which can be ``true`` or ``false``
            - ``config-file-type`` , which can be ``s3`` or ``file``
            - ``config-file-value`` , which is either an S3 ARN or a file path

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-firelensconfiguration.html#cfn-ecs-taskdefinition-firelensconfiguration-options
            '''
            result = self._values.get("options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The log router to use.

            The valid values are ``fluentd`` or ``fluentbit`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-firelensconfiguration.html#cfn-ecs-taskdefinition-firelensconfiguration-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FirelensConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.HealthCheckProperty",
        jsii_struct_bases=[],
        name_mapping={
            "command": "command",
            "interval": "interval",
            "retries": "retries",
            "start_period": "startPeriod",
            "timeout": "timeout",
        },
    )
    class HealthCheckProperty:
        def __init__(
            self,
            *,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            interval: typing.Optional[jsii.Number] = None,
            retries: typing.Optional[jsii.Number] = None,
            start_period: typing.Optional[jsii.Number] = None,
            timeout: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``HealthCheck`` property specifies an object representing a container health check.

            Health check parameters that are specified in a container definition override any Docker health checks that exist in the container image (such as those specified in a parent image or from the image's Dockerfile). This configuration maps to the ``HEALTHCHECK`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            .. epigraph::

               The Amazon ECS container agent only monitors and reports on the health checks specified in the task definition. Amazon ECS does not monitor Docker health checks that are embedded in a container image and not specified in the container definition. Health check parameters that are specified in a container definition override any Docker health checks that exist in the container image.

            If a task is run manually, and not as part of a service, the task will continue its lifecycle regardless of its health status. For tasks that are part of a service, if the task reports as unhealthy then the task will be stopped and the service scheduler will replace it.

            The following are notes about container health check support:

            - Container health checks require version 1.17.0 or greater of the Amazon ECS container agent. For more information, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ .
            - Container health checks are supported for Fargate tasks if you are using platform version 1.1.0 or greater. For more information, see `AWS Fargate Platform Versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ .
            - Container health checks are not supported for tasks that are part of a service that is configured to use a Classic Load Balancer.

            :param command: A string array representing the command that the container runs to determine if it is healthy. The string array must start with ``CMD`` to run the command arguments directly, or ``CMD-SHELL`` to run the command with the container's default shell. When you use the AWS Management Console JSON panel, the AWS Command Line Interface , or the APIs, enclose the list of commands in double quotes and brackets. ``[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]`` You don't include the double quotes and brackets when you use the AWS Management Console. ``CMD-SHELL, curl -f http://localhost/ || exit 1`` An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information, see ``HealthCheck`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ .
            :param interval: The time period in seconds between each health check execution. You may specify between 5 and 300 seconds. The default value is 30 seconds.
            :param retries: The number of times to retry a failed health check before the container is considered unhealthy. You may specify between 1 and 10 retries. The default value is 3.
            :param start_period: The optional grace period to provide containers time to bootstrap before failed health checks count towards the maximum number of retries. You can specify between 0 and 300 seconds. By default, the ``startPeriod`` is off. .. epigraph:: If a health check succeeds within the ``startPeriod`` , then the container is considered healthy and any subsequent failures count toward the maximum number of retries.
            :param timeout: The time period in seconds to wait for a health check to succeed before it is considered a failure. You may specify between 2 and 60 seconds. The default value is 5.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                health_check_property = ecs.CfnTaskDefinition.HealthCheckProperty(
                    command=["command"],
                    interval=123,
                    retries=123,
                    start_period=123,
                    timeout=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b0ac524a8ee2939bf180c2808bb9c5b40e45ffab3bc7731208ea137d731e360a)
                check_type(argname="argument command", value=command, expected_type=type_hints["command"])
                check_type(argname="argument interval", value=interval, expected_type=type_hints["interval"])
                check_type(argname="argument retries", value=retries, expected_type=type_hints["retries"])
                check_type(argname="argument start_period", value=start_period, expected_type=type_hints["start_period"])
                check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if command is not None:
                self._values["command"] = command
            if interval is not None:
                self._values["interval"] = interval
            if retries is not None:
                self._values["retries"] = retries
            if start_period is not None:
                self._values["start_period"] = start_period
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A string array representing the command that the container runs to determine if it is healthy.

            The string array must start with ``CMD`` to run the command arguments directly, or ``CMD-SHELL`` to run the command with the container's default shell.

            When you use the AWS Management Console JSON panel, the AWS Command Line Interface , or the APIs, enclose the list of commands in double quotes and brackets.

            ``[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]``

            You don't include the double quotes and brackets when you use the AWS Management Console.

            ``CMD-SHELL, curl -f http://localhost/ || exit 1``

            An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information, see ``HealthCheck`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def interval(self) -> typing.Optional[jsii.Number]:
            '''The time period in seconds between each health check execution.

            You may specify between 5 and 300 seconds. The default value is 30 seconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-interval
            '''
            result = self._values.get("interval")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def retries(self) -> typing.Optional[jsii.Number]:
            '''The number of times to retry a failed health check before the container is considered unhealthy.

            You may specify between 1 and 10 retries. The default value is 3.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-retries
            '''
            result = self._values.get("retries")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def start_period(self) -> typing.Optional[jsii.Number]:
            '''The optional grace period to provide containers time to bootstrap before failed health checks count towards the maximum number of retries.

            You can specify between 0 and 300 seconds. By default, the ``startPeriod`` is off.
            .. epigraph::

               If a health check succeeds within the ``startPeriod`` , then the container is considered healthy and any subsequent failures count toward the maximum number of retries.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-startperiod
            '''
            result = self._values.get("start_period")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def timeout(self) -> typing.Optional[jsii.Number]:
            '''The time period in seconds to wait for a health check to succeed before it is considered a failure.

            You may specify between 2 and 60 seconds. The default value is 5.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HealthCheckProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.HostEntryProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname", "ip_address": "ipAddress"},
    )
    class HostEntryProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[builtins.str] = None,
            ip_address: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``HostEntry`` property specifies a hostname and an IP address that are added to the ``/etc/hosts`` file of a container through the ``extraHosts`` parameter of its ``ContainerDefinition`` resource.

            :param hostname: The hostname to use in the ``/etc/hosts`` entry.
            :param ip_address: The IP address to use in the ``/etc/hosts`` entry.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-hostentry.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                host_entry_property = ecs.CfnTaskDefinition.HostEntryProperty(
                    hostname="hostname",
                    ip_address="ipAddress"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c1efa75aea9d3290329110f0d93f050804a9ccc956e68b107355da56b0897a7e)
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument ip_address", value=ip_address, expected_type=type_hints["ip_address"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname
            if ip_address is not None:
                self._values["ip_address"] = ip_address

        @builtins.property
        def hostname(self) -> typing.Optional[builtins.str]:
            '''The hostname to use in the ``/etc/hosts`` entry.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-hostentry.html#cfn-ecs-taskdefinition-hostentry-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ip_address(self) -> typing.Optional[builtins.str]:
            '''The IP address to use in the ``/etc/hosts`` entry.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-hostentry.html#cfn-ecs-taskdefinition-hostentry-ipaddress
            '''
            result = self._values.get("ip_address")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HostEntryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.HostVolumePropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={"source_path": "sourcePath"},
    )
    class HostVolumePropertiesProperty:
        def __init__(
            self,
            *,
            source_path: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``HostVolumeProperties`` property specifies details on a container instance bind mount host volume.

            :param source_path: When the ``host`` parameter is used, specify a ``sourcePath`` to declare the path on the host container instance that's presented to the container. If this parameter is empty, then the Docker daemon has assigned a host path for you. If the ``host`` parameter contains a ``sourcePath`` file location, then the data volume persists at the specified location on the host container instance until you delete it manually. If the ``sourcePath`` value doesn't exist on the host container instance, the Docker daemon creates it. If the location does exist, the contents of the source path folder are exported. If you're using the Fargate launch type, the ``sourcePath`` parameter is not supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-hostvolumeproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                host_volume_properties_property = ecs.CfnTaskDefinition.HostVolumePropertiesProperty(
                    source_path="sourcePath"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__304f060f3334436a780d67b06c8e7190a9553b22db3abb1bec9c47779bd0ea06)
                check_type(argname="argument source_path", value=source_path, expected_type=type_hints["source_path"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if source_path is not None:
                self._values["source_path"] = source_path

        @builtins.property
        def source_path(self) -> typing.Optional[builtins.str]:
            '''When the ``host`` parameter is used, specify a ``sourcePath`` to declare the path on the host container instance that's presented to the container.

            If this parameter is empty, then the Docker daemon has assigned a host path for you. If the ``host`` parameter contains a ``sourcePath`` file location, then the data volume persists at the specified location on the host container instance until you delete it manually. If the ``sourcePath`` value doesn't exist on the host container instance, the Docker daemon creates it. If the location does exist, the contents of the source path folder are exported.

            If you're using the Fargate launch type, the ``sourcePath`` parameter is not supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-hostvolumeproperties.html#cfn-ecs-taskdefinition-hostvolumeproperties-sourcepath
            '''
            result = self._values.get("source_path")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HostVolumePropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.InferenceAcceleratorProperty",
        jsii_struct_bases=[],
        name_mapping={"device_name": "deviceName", "device_type": "deviceType"},
    )
    class InferenceAcceleratorProperty:
        def __init__(
            self,
            *,
            device_name: typing.Optional[builtins.str] = None,
            device_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Details on an Elastic Inference accelerator.

            For more information, see `Working with Amazon Elastic Inference on Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-inference.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param device_name: The Elastic Inference accelerator device name. The ``deviceName`` must also be referenced in a container definition as a `ResourceRequirement <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ResourceRequirement.html>`_ .
            :param device_type: The Elastic Inference accelerator type to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                inference_accelerator_property = ecs.CfnTaskDefinition.InferenceAcceleratorProperty(
                    device_name="deviceName",
                    device_type="deviceType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a71edc4ee74174d76209f4507eeccf8fda36392af756a49f7f2309c16d318376)
                check_type(argname="argument device_name", value=device_name, expected_type=type_hints["device_name"])
                check_type(argname="argument device_type", value=device_type, expected_type=type_hints["device_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if device_name is not None:
                self._values["device_name"] = device_name
            if device_type is not None:
                self._values["device_type"] = device_type

        @builtins.property
        def device_name(self) -> typing.Optional[builtins.str]:
            '''The Elastic Inference accelerator device name.

            The ``deviceName`` must also be referenced in a container definition as a `ResourceRequirement <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ResourceRequirement.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html#cfn-ecs-taskdefinition-inferenceaccelerator-devicename
            '''
            result = self._values.get("device_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def device_type(self) -> typing.Optional[builtins.str]:
            '''The Elastic Inference accelerator type to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html#cfn-ecs-taskdefinition-inferenceaccelerator-devicetype
            '''
            result = self._values.get("device_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InferenceAcceleratorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.KernelCapabilitiesProperty",
        jsii_struct_bases=[],
        name_mapping={"add": "add", "drop": "drop"},
    )
    class KernelCapabilitiesProperty:
        def __init__(
            self,
            *,
            add: typing.Optional[typing.Sequence[builtins.str]] = None,
            drop: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``KernelCapabilities`` property specifies the Linux capabilities for the container that are added to or dropped from the default configuration that is provided by Docker.

            For more information on the default capabilities and the non-default available capabilities, see `Runtime privilege and Linux capabilities <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#runtime-privilege-and-linux-capabilities>`_ in the *Docker run reference* . For more detailed information on these Linux capabilities, see the `capabilities(7) <https://docs.aws.amazon.com/http://man7.org/linux/man-pages/man7/capabilities.7.html>`_ Linux manual page.

            :param add: The Linux capabilities for the container that have been added to the default configuration provided by Docker. This parameter maps to ``CapAdd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cap-add`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: Tasks launched on AWS Fargate only support adding the ``SYS_PTRACE`` kernel capability. Valid values: ``"ALL" | "AUDIT_CONTROL" | "AUDIT_WRITE" | "BLOCK_SUSPEND" | "CHOWN" | "DAC_OVERRIDE" | "DAC_READ_SEARCH" | "FOWNER" | "FSETID" | "IPC_LOCK" | "IPC_OWNER" | "KILL" | "LEASE" | "LINUX_IMMUTABLE" | "MAC_ADMIN" | "MAC_OVERRIDE" | "MKNOD" | "NET_ADMIN" | "NET_BIND_SERVICE" | "NET_BROADCAST" | "NET_RAW" | "SETFCAP" | "SETGID" | "SETPCAP" | "SETUID" | "SYS_ADMIN" | "SYS_BOOT" | "SYS_CHROOT" | "SYS_MODULE" | "SYS_NICE" | "SYS_PACCT" | "SYS_PTRACE" | "SYS_RAWIO" | "SYS_RESOURCE" | "SYS_TIME" | "SYS_TTY_CONFIG" | "SYSLOG" | "WAKE_ALARM"``
            :param drop: The Linux capabilities for the container that have been removed from the default configuration provided by Docker. This parameter maps to ``CapDrop`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cap-drop`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . Valid values: ``"ALL" | "AUDIT_CONTROL" | "AUDIT_WRITE" | "BLOCK_SUSPEND" | "CHOWN" | "DAC_OVERRIDE" | "DAC_READ_SEARCH" | "FOWNER" | "FSETID" | "IPC_LOCK" | "IPC_OWNER" | "KILL" | "LEASE" | "LINUX_IMMUTABLE" | "MAC_ADMIN" | "MAC_OVERRIDE" | "MKNOD" | "NET_ADMIN" | "NET_BIND_SERVICE" | "NET_BROADCAST" | "NET_RAW" | "SETFCAP" | "SETGID" | "SETPCAP" | "SETUID" | "SYS_ADMIN" | "SYS_BOOT" | "SYS_CHROOT" | "SYS_MODULE" | "SYS_NICE" | "SYS_PACCT" | "SYS_PTRACE" | "SYS_RAWIO" | "SYS_RESOURCE" | "SYS_TIME" | "SYS_TTY_CONFIG" | "SYSLOG" | "WAKE_ALARM"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-kernelcapabilities.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                kernel_capabilities_property = ecs.CfnTaskDefinition.KernelCapabilitiesProperty(
                    add=["add"],
                    drop=["drop"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b5411138e9f4fb38508f93bb338b04b3532cecdf3da3d1de00c1aec958ceacc8)
                check_type(argname="argument add", value=add, expected_type=type_hints["add"])
                check_type(argname="argument drop", value=drop, expected_type=type_hints["drop"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if add is not None:
                self._values["add"] = add
            if drop is not None:
                self._values["drop"] = drop

        @builtins.property
        def add(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The Linux capabilities for the container that have been added to the default configuration provided by Docker.

            This parameter maps to ``CapAdd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cap-add`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               Tasks launched on AWS Fargate only support adding the ``SYS_PTRACE`` kernel capability.

            Valid values: ``"ALL" | "AUDIT_CONTROL" | "AUDIT_WRITE" | "BLOCK_SUSPEND" | "CHOWN" | "DAC_OVERRIDE" | "DAC_READ_SEARCH" | "FOWNER" | "FSETID" | "IPC_LOCK" | "IPC_OWNER" | "KILL" | "LEASE" | "LINUX_IMMUTABLE" | "MAC_ADMIN" | "MAC_OVERRIDE" | "MKNOD" | "NET_ADMIN" | "NET_BIND_SERVICE" | "NET_BROADCAST" | "NET_RAW" | "SETFCAP" | "SETGID" | "SETPCAP" | "SETUID" | "SYS_ADMIN" | "SYS_BOOT" | "SYS_CHROOT" | "SYS_MODULE" | "SYS_NICE" | "SYS_PACCT" | "SYS_PTRACE" | "SYS_RAWIO" | "SYS_RESOURCE" | "SYS_TIME" | "SYS_TTY_CONFIG" | "SYSLOG" | "WAKE_ALARM"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-kernelcapabilities.html#cfn-ecs-taskdefinition-kernelcapabilities-add
            '''
            result = self._values.get("add")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def drop(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The Linux capabilities for the container that have been removed from the default configuration provided by Docker.

            This parameter maps to ``CapDrop`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cap-drop`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            Valid values: ``"ALL" | "AUDIT_CONTROL" | "AUDIT_WRITE" | "BLOCK_SUSPEND" | "CHOWN" | "DAC_OVERRIDE" | "DAC_READ_SEARCH" | "FOWNER" | "FSETID" | "IPC_LOCK" | "IPC_OWNER" | "KILL" | "LEASE" | "LINUX_IMMUTABLE" | "MAC_ADMIN" | "MAC_OVERRIDE" | "MKNOD" | "NET_ADMIN" | "NET_BIND_SERVICE" | "NET_BROADCAST" | "NET_RAW" | "SETFCAP" | "SETGID" | "SETPCAP" | "SETUID" | "SYS_ADMIN" | "SYS_BOOT" | "SYS_CHROOT" | "SYS_MODULE" | "SYS_NICE" | "SYS_PACCT" | "SYS_PTRACE" | "SYS_RAWIO" | "SYS_RESOURCE" | "SYS_TIME" | "SYS_TTY_CONFIG" | "SYSLOG" | "WAKE_ALARM"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-kernelcapabilities.html#cfn-ecs-taskdefinition-kernelcapabilities-drop
            '''
            result = self._values.get("drop")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KernelCapabilitiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.KeyValuePairProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class KeyValuePairProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A key-value pair object.

            :param name: The name of the key-value pair. For environment variables, this is the name of the environment variable.
            :param value: The value of the key-value pair. For environment variables, this is the value of the environment variable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-keyvaluepair.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                key_value_pair_property = ecs.CfnTaskDefinition.KeyValuePairProperty(
                    name="name",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__91632c1e5ee5d123f3e7f3b7abb922b5d9fe0af91e9f434c7a677e8882d15aeb)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the key-value pair.

            For environment variables, this is the name of the environment variable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-keyvaluepair.html#cfn-ecs-taskdefinition-keyvaluepair-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value of the key-value pair.

            For environment variables, this is the value of the environment variable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-keyvaluepair.html#cfn-ecs-taskdefinition-keyvaluepair-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KeyValuePairProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.LinuxParametersProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capabilities": "capabilities",
            "devices": "devices",
            "init_process_enabled": "initProcessEnabled",
            "max_swap": "maxSwap",
            "shared_memory_size": "sharedMemorySize",
            "swappiness": "swappiness",
            "tmpfs": "tmpfs",
        },
    )
    class LinuxParametersProperty:
        def __init__(
            self,
            *,
            capabilities: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.KernelCapabilitiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            devices: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.DeviceProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            init_process_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            max_swap: typing.Optional[jsii.Number] = None,
            shared_memory_size: typing.Optional[jsii.Number] = None,
            swappiness: typing.Optional[jsii.Number] = None,
            tmpfs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.TmpfsProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The Linux-specific options that are applied to the container, such as Linux `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_ .

            :param capabilities: The Linux capabilities for the container that are added to or dropped from the default configuration provided by Docker. .. epigraph:: For tasks that use the Fargate launch type, ``capabilities`` is supported for all platform versions but the ``add`` parameter is only supported if using platform version 1.4.0 or later.
            :param devices: Any host devices to expose to the container. This parameter maps to ``Devices`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--device`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: If you're using tasks that use the Fargate launch type, the ``devices`` parameter isn't supported.
            :param init_process_enabled: Run an ``init`` process inside the container that forwards signals and reaps processes. This parameter maps to the ``--init`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This parameter requires version 1.25 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            :param max_swap: The total amount of swap memory (in MiB) a container can use. This parameter will be translated to the ``--memory-swap`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ where the value would be the sum of the container memory plus the ``maxSwap`` value. If a ``maxSwap`` value of ``0`` is specified, the container will not use swap. Accepted values are ``0`` or any positive integer. If the ``maxSwap`` parameter is omitted, the container will use the swap configuration for the container instance it is running on. A ``maxSwap`` value must be set for the ``swappiness`` parameter to be used. .. epigraph:: If you're using tasks that use the Fargate launch type, the ``maxSwap`` parameter isn't supported. If you're using tasks on Amazon Linux 2023 the ``swappiness`` parameter isn't supported.
            :param shared_memory_size: The value for the size (in MiB) of the ``/dev/shm`` volume. This parameter maps to the ``--shm-size`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: If you are using tasks that use the Fargate launch type, the ``sharedMemorySize`` parameter is not supported.
            :param swappiness: This allows you to tune a container's memory swappiness behavior. A ``swappiness`` value of ``0`` will cause swapping to not happen unless absolutely necessary. A ``swappiness`` value of ``100`` will cause pages to be swapped very aggressively. Accepted values are whole numbers between ``0`` and ``100`` . If the ``swappiness`` parameter is not specified, a default value of ``60`` is used. If a value is not specified for ``maxSwap`` then this parameter is ignored. This parameter maps to the ``--memory-swappiness`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: If you're using tasks that use the Fargate launch type, the ``swappiness`` parameter isn't supported. If you're using tasks on Amazon Linux 2023 the ``swappiness`` parameter isn't supported.
            :param tmpfs: The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the ``--tmpfs`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: If you're using tasks that use the Fargate launch type, the ``tmpfs`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                linux_parameters_property = ecs.CfnTaskDefinition.LinuxParametersProperty(
                    capabilities=ecs.CfnTaskDefinition.KernelCapabilitiesProperty(
                        add=["add"],
                        drop=["drop"]
                    ),
                    devices=[ecs.CfnTaskDefinition.DeviceProperty(
                        container_path="containerPath",
                        host_path="hostPath",
                        permissions=["permissions"]
                    )],
                    init_process_enabled=False,
                    max_swap=123,
                    shared_memory_size=123,
                    swappiness=123,
                    tmpfs=[ecs.CfnTaskDefinition.TmpfsProperty(
                        size=123,
                
                        # the properties below are optional
                        container_path="containerPath",
                        mount_options=["mountOptions"]
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c0aaf73788be7f1d342c8a05599189d471843835d7b8d191218db671721e0e7a)
                check_type(argname="argument capabilities", value=capabilities, expected_type=type_hints["capabilities"])
                check_type(argname="argument devices", value=devices, expected_type=type_hints["devices"])
                check_type(argname="argument init_process_enabled", value=init_process_enabled, expected_type=type_hints["init_process_enabled"])
                check_type(argname="argument max_swap", value=max_swap, expected_type=type_hints["max_swap"])
                check_type(argname="argument shared_memory_size", value=shared_memory_size, expected_type=type_hints["shared_memory_size"])
                check_type(argname="argument swappiness", value=swappiness, expected_type=type_hints["swappiness"])
                check_type(argname="argument tmpfs", value=tmpfs, expected_type=type_hints["tmpfs"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if capabilities is not None:
                self._values["capabilities"] = capabilities
            if devices is not None:
                self._values["devices"] = devices
            if init_process_enabled is not None:
                self._values["init_process_enabled"] = init_process_enabled
            if max_swap is not None:
                self._values["max_swap"] = max_swap
            if shared_memory_size is not None:
                self._values["shared_memory_size"] = shared_memory_size
            if swappiness is not None:
                self._values["swappiness"] = swappiness
            if tmpfs is not None:
                self._values["tmpfs"] = tmpfs

        @builtins.property
        def capabilities(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.KernelCapabilitiesProperty"]]:
            '''The Linux capabilities for the container that are added to or dropped from the default configuration provided by Docker.

            .. epigraph::

               For tasks that use the Fargate launch type, ``capabilities`` is supported for all platform versions but the ``add`` parameter is only supported if using platform version 1.4.0 or later.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-capabilities
            '''
            result = self._values.get("capabilities")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.KernelCapabilitiesProperty"]], result)

        @builtins.property
        def devices(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.DeviceProperty"]]]]:
            '''Any host devices to expose to the container.

            This parameter maps to ``Devices`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--device`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               If you're using tasks that use the Fargate launch type, the ``devices`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-devices
            '''
            result = self._values.get("devices")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.DeviceProperty"]]]], result)

        @builtins.property
        def init_process_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Run an ``init`` process inside the container that forwards signals and reaps processes.

            This parameter maps to the ``--init`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This parameter requires version 1.25 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-initprocessenabled
            '''
            result = self._values.get("init_process_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def max_swap(self) -> typing.Optional[jsii.Number]:
            '''The total amount of swap memory (in MiB) a container can use.

            This parameter will be translated to the ``--memory-swap`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ where the value would be the sum of the container memory plus the ``maxSwap`` value.

            If a ``maxSwap`` value of ``0`` is specified, the container will not use swap. Accepted values are ``0`` or any positive integer. If the ``maxSwap`` parameter is omitted, the container will use the swap configuration for the container instance it is running on. A ``maxSwap`` value must be set for the ``swappiness`` parameter to be used.
            .. epigraph::

               If you're using tasks that use the Fargate launch type, the ``maxSwap`` parameter isn't supported.

               If you're using tasks on Amazon Linux 2023 the ``swappiness`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-maxswap
            '''
            result = self._values.get("max_swap")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def shared_memory_size(self) -> typing.Optional[jsii.Number]:
            '''The value for the size (in MiB) of the ``/dev/shm`` volume.

            This parameter maps to the ``--shm-size`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               If you are using tasks that use the Fargate launch type, the ``sharedMemorySize`` parameter is not supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-sharedmemorysize
            '''
            result = self._values.get("shared_memory_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def swappiness(self) -> typing.Optional[jsii.Number]:
            '''This allows you to tune a container's memory swappiness behavior.

            A ``swappiness`` value of ``0`` will cause swapping to not happen unless absolutely necessary. A ``swappiness`` value of ``100`` will cause pages to be swapped very aggressively. Accepted values are whole numbers between ``0`` and ``100`` . If the ``swappiness`` parameter is not specified, a default value of ``60`` is used. If a value is not specified for ``maxSwap`` then this parameter is ignored. This parameter maps to the ``--memory-swappiness`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               If you're using tasks that use the Fargate launch type, the ``swappiness`` parameter isn't supported.

               If you're using tasks on Amazon Linux 2023 the ``swappiness`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-swappiness
            '''
            result = self._values.get("swappiness")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def tmpfs(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.TmpfsProperty"]]]]:
            '''The container path, mount options, and size (in MiB) of the tmpfs mount.

            This parameter maps to the ``--tmpfs`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               If you're using tasks that use the Fargate launch type, the ``tmpfs`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-tmpfs
            '''
            result = self._values.get("tmpfs")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.TmpfsProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LinuxParametersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.LogConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "log_driver": "logDriver",
            "options": "options",
            "secret_options": "secretOptions",
        },
    )
    class LogConfigurationProperty:
        def __init__(
            self,
            *,
            log_driver: builtins.str,
            options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]] = None,
            secret_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.SecretProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The ``LogConfiguration`` property specifies log configuration options to send to a custom log driver for the container.

            :param log_driver: The log driver to use for the container. For tasks on AWS Fargate , the supported log drivers are ``awslogs`` , ``splunk`` , and ``awsfirelens`` . For tasks hosted on Amazon EC2 instances, the supported log drivers are ``awslogs`` , ``fluentd`` , ``gelf`` , ``json-file`` , ``journald`` , ``logentries`` , ``syslog`` , ``splunk`` , and ``awsfirelens`` . For more information about using the ``awslogs`` log driver, see `Using the awslogs log driver <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For more information about using the ``awsfirelens`` log driver, see `Custom log routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's `available on GitHub <https://docs.aws.amazon.com/https://github.com/aws/amazon-ecs-agent>`_ and customize it to work with that driver. We encourage you to submit pull requests for changes that you would like to have included. However, we don't currently provide support for running modified copies of this software.
            :param options: The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            :param secret_options: The secrets to pass to the log configuration. For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-logconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                log_configuration_property = ecs.CfnTaskDefinition.LogConfigurationProperty(
                    log_driver="logDriver",
                
                    # the properties below are optional
                    options={
                        "options_key": "options"
                    },
                    secret_options=[ecs.CfnTaskDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__21877ca71c2d5ead7d497c9d9a61ebef73ab6b2fcae2f5452db9d1dec56c00aa)
                check_type(argname="argument log_driver", value=log_driver, expected_type=type_hints["log_driver"])
                check_type(argname="argument options", value=options, expected_type=type_hints["options"])
                check_type(argname="argument secret_options", value=secret_options, expected_type=type_hints["secret_options"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "log_driver": log_driver,
            }
            if options is not None:
                self._values["options"] = options
            if secret_options is not None:
                self._values["secret_options"] = secret_options

        @builtins.property
        def log_driver(self) -> builtins.str:
            '''The log driver to use for the container.

            For tasks on AWS Fargate , the supported log drivers are ``awslogs`` , ``splunk`` , and ``awsfirelens`` .

            For tasks hosted on Amazon EC2 instances, the supported log drivers are ``awslogs`` , ``fluentd`` , ``gelf`` , ``json-file`` , ``journald`` , ``logentries`` , ``syslog`` , ``splunk`` , and ``awsfirelens`` .

            For more information about using the ``awslogs`` log driver, see `Using the awslogs log driver <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For more information about using the ``awsfirelens`` log driver, see `Custom log routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's `available on GitHub <https://docs.aws.amazon.com/https://github.com/aws/amazon-ecs-agent>`_ and customize it to work with that driver. We encourage you to submit pull requests for changes that you would like to have included. However, we don't currently provide support for running modified copies of this software.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-logconfiguration.html#cfn-ecs-taskdefinition-logconfiguration-logdriver
            '''
            result = self._values.get("log_driver")
            assert result is not None, "Required property 'log_driver' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]]:
            '''The configuration options to send to the log driver.

            This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-logconfiguration.html#cfn-ecs-taskdefinition-containerdefinition-logconfiguration-options
            '''
            result = self._values.get("options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def secret_options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.SecretProperty"]]]]:
            '''The secrets to pass to the log configuration.

            For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-logconfiguration.html#cfn-ecs-taskdefinition-logconfiguration-secretoptions
            '''
            result = self._values.get("secret_options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.SecretProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LogConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.MountPointProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "read_only": "readOnly",
            "source_volume": "sourceVolume",
        },
    )
    class MountPointProperty:
        def __init__(
            self,
            *,
            container_path: typing.Optional[builtins.str] = None,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            source_volume: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The details for a volume mount point that's used in a container definition.

            :param container_path: The path on the container to mount the host volume at.
            :param read_only: If this value is ``true`` , the container has read-only access to the volume. If this value is ``false`` , then the container can write to the volume. The default value is ``false`` .
            :param source_volume: The name of the volume to mount. Must be a volume name referenced in the ``name`` parameter of task definition ``volume`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-mountpoint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                mount_point_property = ecs.CfnTaskDefinition.MountPointProperty(
                    container_path="containerPath",
                    read_only=False,
                    source_volume="sourceVolume"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7ca068c461cda36d8574700ee3ea4cda89a56e1f92d6195de0c44fe7912f6a40)
                check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
                check_type(argname="argument read_only", value=read_only, expected_type=type_hints["read_only"])
                check_type(argname="argument source_volume", value=source_volume, expected_type=type_hints["source_volume"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if container_path is not None:
                self._values["container_path"] = container_path
            if read_only is not None:
                self._values["read_only"] = read_only
            if source_volume is not None:
                self._values["source_volume"] = source_volume

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''The path on the container to mount the host volume at.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-mountpoint.html#cfn-ecs-taskdefinition-mountpoint-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If this value is ``true`` , the container has read-only access to the volume.

            If this value is ``false`` , then the container can write to the volume. The default value is ``false`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-mountpoint.html#cfn-ecs-taskdefinition-mountpoint-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def source_volume(self) -> typing.Optional[builtins.str]:
            '''The name of the volume to mount.

            Must be a volume name referenced in the ``name`` parameter of task definition ``volume`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-mountpoint.html#cfn-ecs-taskdefinition-mountpoint-sourcevolume
            '''
            result = self._values.get("source_volume")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MountPointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.PortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "app_protocol": "appProtocol",
            "container_port": "containerPort",
            "container_port_range": "containerPortRange",
            "host_port": "hostPort",
            "name": "name",
            "protocol": "protocol",
        },
    )
    class PortMappingProperty:
        def __init__(
            self,
            *,
            app_protocol: typing.Optional[builtins.str] = None,
            container_port: typing.Optional[jsii.Number] = None,
            container_port_range: typing.Optional[builtins.str] = None,
            host_port: typing.Optional[jsii.Number] = None,
            name: typing.Optional[builtins.str] = None,
            protocol: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``PortMapping`` property specifies a port mapping.

            Port mappings allow containers to access ports on the host container instance to send or receive traffic. Port mappings are specified as part of the container definition.

            If you are using containers in a task with the ``awsvpc`` or ``host`` network mode, exposed ports should be specified using ``containerPort`` . The ``hostPort`` can be left blank or it must be the same value as the ``containerPort`` .

            After a task reaches the ``RUNNING`` status, manual and automatic host and container port assignments are visible in the ``networkBindings`` section of `DescribeTasks <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html>`_ API responses.

            :param app_protocol: The application protocol that's used for the port mapping. This parameter only applies to Service Connect. We recommend that you set this parameter to be consistent with the protocol that your application uses. If you set this parameter, Amazon ECS adds protocol-specific connection handling to the Service Connect proxy. If you set this parameter, Amazon ECS adds protocol-specific telemetry in the Amazon ECS console and CloudWatch. If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add protocol-specific telemetry for TCP. Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param container_port: The port number on the container that's bound to the user-specified or automatically assigned host port. If you use containers in a task with the ``awsvpc`` or ``host`` network mode, specify the exposed ports using ``containerPort`` . If you use containers in a task with the ``bridge`` network mode and you specify a container port and not a host port, your container automatically receives a host port in the ephemeral port range. For more information, see ``hostPort`` . Port mappings that are automatically assigned in this way do not count toward the 100 reserved ports limit of a container instance.
            :param container_port_range: The port number range on the container that's bound to the dynamically mapped host port range. The following rules apply when you specify a ``containerPortRange`` : - You must use either the ``bridge`` network mode or the ``awsvpc`` network mode. - This parameter is available for both the EC2 and AWS Fargate launch types. - This parameter is available for both the Linux and Windows operating systems. - The container instance must have at least version 1.67.0 of the container agent and at least version 1.67.0-1 of the ``ecs-init`` package - You can specify a maximum of 100 port ranges per container. - You do not specify a ``hostPortRange`` . The value of the ``hostPortRange`` is set as follows: - For containers in a task with the ``awsvpc`` network mode, the ``hostPort`` is set to the same value as the ``containerPort`` . This is a static mapping strategy. - For containers in a task with the ``bridge`` network mode, the Amazon ECS agent finds open host ports from the default ephemeral range and passes it to docker to bind them to the container ports. - The ``containerPortRange`` valid values are between 1 and 65535. - A port can only be included in one port mapping per container. - You cannot specify overlapping port ranges. - The first port in the range must be less than last port in the range. - Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a large number of ports. For more information, see `Issue #11185 <https://docs.aws.amazon.com/https://github.com/moby/moby/issues/11185>`_ on the Github website. For information about how to turn off the docker-proxy in the Docker daemon config file, see `Docker daemon <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon>`_ in the *Amazon ECS Developer Guide* . You can call ```DescribeTasks`` <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html>`_ to view the ``hostPortRange`` which are the host ports that are bound to the container ports.
            :param host_port: The port number on the container instance to reserve for your container. If you specify a ``containerPortRange`` , leave this field empty and the value of the ``hostPort`` is set as follows: - For containers in a task with the ``awsvpc`` network mode, the ``hostPort`` is set to the same value as the ``containerPort`` . This is a static mapping strategy. - For containers in a task with the ``bridge`` network mode, the Amazon ECS agent finds open ports on the host and automatically binds them to the container ports. This is a dynamic mapping strategy. If you use containers in a task with the ``awsvpc`` or ``host`` network mode, the ``hostPort`` can either be left blank or set to the same value as the ``containerPort`` . If you use containers in a task with the ``bridge`` network mode, you can specify a non-reserved host port for your container port mapping, or you can omit the ``hostPort`` (or set it to ``0`` ) while specifying a ``containerPort`` and your container automatically receives a port in the ephemeral port range for your container instance operating system and Docker version. The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under ``/proc/sys/net/ipv4/ip_local_port_range`` . If this kernel parameter is unavailable, the default ephemeral port range from 49153 through 65535 is used. Do not attempt to specify a host port in the ephemeral port range as these are reserved for automatic assignment. In general, ports below 32768 are outside of the ephemeral port range. The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container agent ports 51678-51680. Any host port that was previously specified in a running task is also reserved while the task is running. That is, after a task stops, the host port is released. The current reserved ports are displayed in the ``remainingResources`` of `DescribeContainerInstances <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html>`_ output. A container instance can have up to 100 reserved ports at a time. This number includes the default reserved ports. Automatically assigned ports aren't included in the 100 reserved ports quota.
            :param name: The name that's used for the port mapping. This parameter only applies to Service Connect. This parameter is the name that you use in the ``serviceConnectConfiguration`` of a service. The name can include up to 64 characters. The characters can include lowercase letters, numbers, underscores (_), and hyphens (-). The name can't start with a hyphen. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param protocol: The protocol used for the port mapping. Valid values are ``tcp`` and ``udp`` . The default is ``tcp`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-portmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                port_mapping_property = ecs.CfnTaskDefinition.PortMappingProperty(
                    app_protocol="appProtocol",
                    container_port=123,
                    container_port_range="containerPortRange",
                    host_port=123,
                    name="name",
                    protocol="protocol"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__446c7352216030258b4d70f5c5937f6375d59f2aa071552346ada383837db06a)
                check_type(argname="argument app_protocol", value=app_protocol, expected_type=type_hints["app_protocol"])
                check_type(argname="argument container_port", value=container_port, expected_type=type_hints["container_port"])
                check_type(argname="argument container_port_range", value=container_port_range, expected_type=type_hints["container_port_range"])
                check_type(argname="argument host_port", value=host_port, expected_type=type_hints["host_port"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if app_protocol is not None:
                self._values["app_protocol"] = app_protocol
            if container_port is not None:
                self._values["container_port"] = container_port
            if container_port_range is not None:
                self._values["container_port_range"] = container_port_range
            if host_port is not None:
                self._values["host_port"] = host_port
            if name is not None:
                self._values["name"] = name
            if protocol is not None:
                self._values["protocol"] = protocol

        @builtins.property
        def app_protocol(self) -> typing.Optional[builtins.str]:
            '''The application protocol that's used for the port mapping.

            This parameter only applies to Service Connect. We recommend that you set this parameter to be consistent with the protocol that your application uses. If you set this parameter, Amazon ECS adds protocol-specific connection handling to the Service Connect proxy. If you set this parameter, Amazon ECS adds protocol-specific telemetry in the Amazon ECS console and CloudWatch.

            If you don't set a value for this parameter, then TCP is used. However, Amazon ECS doesn't add protocol-specific telemetry for TCP.

            Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service Connect. For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-portmapping.html#cfn-ecs-taskdefinition-portmapping-appprotocol
            '''
            result = self._values.get("app_protocol")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''The port number on the container that's bound to the user-specified or automatically assigned host port.

            If you use containers in a task with the ``awsvpc`` or ``host`` network mode, specify the exposed ports using ``containerPort`` .

            If you use containers in a task with the ``bridge`` network mode and you specify a container port and not a host port, your container automatically receives a host port in the ephemeral port range. For more information, see ``hostPort`` . Port mappings that are automatically assigned in this way do not count toward the 100 reserved ports limit of a container instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-portmapping.html#cfn-ecs-taskdefinition-portmapping-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def container_port_range(self) -> typing.Optional[builtins.str]:
            '''The port number range on the container that's bound to the dynamically mapped host port range.

            The following rules apply when you specify a ``containerPortRange`` :

            - You must use either the ``bridge`` network mode or the ``awsvpc`` network mode.
            - This parameter is available for both the EC2 and AWS Fargate launch types.
            - This parameter is available for both the Linux and Windows operating systems.
            - The container instance must have at least version 1.67.0 of the container agent and at least version 1.67.0-1 of the ``ecs-init`` package
            - You can specify a maximum of 100 port ranges per container.
            - You do not specify a ``hostPortRange`` . The value of the ``hostPortRange`` is set as follows:
            - For containers in a task with the ``awsvpc`` network mode, the ``hostPort`` is set to the same value as the ``containerPort`` . This is a static mapping strategy.
            - For containers in a task with the ``bridge`` network mode, the Amazon ECS agent finds open host ports from the default ephemeral range and passes it to docker to bind them to the container ports.
            - The ``containerPortRange`` valid values are between 1 and 65535.
            - A port can only be included in one port mapping per container.
            - You cannot specify overlapping port ranges.
            - The first port in the range must be less than last port in the range.
            - Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a large number of ports.

            For more information, see `Issue #11185 <https://docs.aws.amazon.com/https://github.com/moby/moby/issues/11185>`_ on the Github website.

            For information about how to turn off the docker-proxy in the Docker daemon config file, see `Docker daemon <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon>`_ in the *Amazon ECS Developer Guide* .

            You can call ```DescribeTasks`` <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html>`_ to view the ``hostPortRange`` which are the host ports that are bound to the container ports.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-portmapping.html#cfn-ecs-taskdefinition-portmapping-containerportrange
            '''
            result = self._values.get("container_port_range")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_port(self) -> typing.Optional[jsii.Number]:
            '''The port number on the container instance to reserve for your container.

            If you specify a ``containerPortRange`` , leave this field empty and the value of the ``hostPort`` is set as follows:

            - For containers in a task with the ``awsvpc`` network mode, the ``hostPort`` is set to the same value as the ``containerPort`` . This is a static mapping strategy.
            - For containers in a task with the ``bridge`` network mode, the Amazon ECS agent finds open ports on the host and automatically binds them to the container ports. This is a dynamic mapping strategy.

            If you use containers in a task with the ``awsvpc`` or ``host`` network mode, the ``hostPort`` can either be left blank or set to the same value as the ``containerPort`` .

            If you use containers in a task with the ``bridge`` network mode, you can specify a non-reserved host port for your container port mapping, or you can omit the ``hostPort`` (or set it to ``0`` ) while specifying a ``containerPort`` and your container automatically receives a port in the ephemeral port range for your container instance operating system and Docker version.

            The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under ``/proc/sys/net/ipv4/ip_local_port_range`` . If this kernel parameter is unavailable, the default ephemeral port range from 49153 through 65535 is used. Do not attempt to specify a host port in the ephemeral port range as these are reserved for automatic assignment. In general, ports below 32768 are outside of the ephemeral port range.

            The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container agent ports 51678-51680. Any host port that was previously specified in a running task is also reserved while the task is running. That is, after a task stops, the host port is released. The current reserved ports are displayed in the ``remainingResources`` of `DescribeContainerInstances <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html>`_ output. A container instance can have up to 100 reserved ports at a time. This number includes the default reserved ports. Automatically assigned ports aren't included in the 100 reserved ports quota.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-portmapping.html#cfn-ecs-taskdefinition-portmapping-hostport
            '''
            result = self._values.get("host_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name that's used for the port mapping.

            This parameter only applies to Service Connect. This parameter is the name that you use in the ``serviceConnectConfiguration`` of a service. The name can include up to 64 characters. The characters can include lowercase letters, numbers, underscores (_), and hyphens (-). The name can't start with a hyphen.

            For more information, see `Service Connect <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-portmapping.html#cfn-ecs-taskdefinition-portmapping-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def protocol(self) -> typing.Optional[builtins.str]:
            '''The protocol used for the port mapping.

            Valid values are ``tcp`` and ``udp`` . The default is ``tcp`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-portmapping.html#cfn-ecs-taskdefinition-portmapping-protocol
            '''
            result = self._values.get("protocol")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.ProxyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "proxy_configuration_properties": "proxyConfigurationProperties",
            "type": "type",
        },
    )
    class ProxyConfigurationProperty:
        def __init__(
            self,
            *,
            container_name: builtins.str,
            proxy_configuration_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.KeyValuePairProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The configuration details for the App Mesh proxy.

            For tasks that use the EC2 launch type, the container instances require at least version 1.26.0 of the container agent and at least version 1.26.0-1 of the ``ecs-init`` package to use a proxy configuration. If your container instances are launched from the Amazon ECS optimized AMI version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_

            :param container_name: The name of the container that will serve as the App Mesh proxy.
            :param proxy_configuration_properties: The set of network configuration parameters to provide the Container Network Interface (CNI) plugin, specified as key-value pairs. - ``IgnoredUID`` - (Required) The user ID (UID) of the proxy container as defined by the ``user`` parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If ``IgnoredGID`` is specified, this field can be empty. - ``IgnoredGID`` - (Required) The group ID (GID) of the proxy container as defined by the ``user`` parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If ``IgnoredUID`` is specified, this field can be empty. - ``AppPorts`` - (Required) The list of ports that the application uses. Network traffic to these ports is forwarded to the ``ProxyIngressPort`` and ``ProxyEgressPort`` . - ``ProxyIngressPort`` - (Required) Specifies the port that incoming traffic to the ``AppPorts`` is directed to. - ``ProxyEgressPort`` - (Required) Specifies the port that outgoing traffic from the ``AppPorts`` is directed to. - ``EgressIgnoredPorts`` - (Required) The egress traffic going to the specified ports is ignored and not redirected to the ``ProxyEgressPort`` . It can be an empty list. - ``EgressIgnoredIPs`` - (Required) The egress traffic going to the specified IP addresses is ignored and not redirected to the ``ProxyEgressPort`` . It can be an empty list.
            :param type: The proxy type. The only supported value is ``APPMESH`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                proxy_configuration_property = ecs.CfnTaskDefinition.ProxyConfigurationProperty(
                    container_name="containerName",
                
                    # the properties below are optional
                    proxy_configuration_properties=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                        name="name",
                        value="value"
                    )],
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b9c318252373ee4bc0ab9dcb49bdd3512e46eaf2d9b2363fa1dd17bf87005a92)
                check_type(argname="argument container_name", value=container_name, expected_type=type_hints["container_name"])
                check_type(argname="argument proxy_configuration_properties", value=proxy_configuration_properties, expected_type=type_hints["proxy_configuration_properties"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "container_name": container_name,
            }
            if proxy_configuration_properties is not None:
                self._values["proxy_configuration_properties"] = proxy_configuration_properties
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def container_name(self) -> builtins.str:
            '''The name of the container that will serve as the App Mesh proxy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html#cfn-ecs-taskdefinition-proxyconfiguration-containername
            '''
            result = self._values.get("container_name")
            assert result is not None, "Required property 'container_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def proxy_configuration_properties(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.KeyValuePairProperty"]]]]:
            '''The set of network configuration parameters to provide the Container Network Interface (CNI) plugin, specified as key-value pairs.

            - ``IgnoredUID`` - (Required) The user ID (UID) of the proxy container as defined by the ``user`` parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If ``IgnoredGID`` is specified, this field can be empty.
            - ``IgnoredGID`` - (Required) The group ID (GID) of the proxy container as defined by the ``user`` parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If ``IgnoredUID`` is specified, this field can be empty.
            - ``AppPorts`` - (Required) The list of ports that the application uses. Network traffic to these ports is forwarded to the ``ProxyIngressPort`` and ``ProxyEgressPort`` .
            - ``ProxyIngressPort`` - (Required) Specifies the port that incoming traffic to the ``AppPorts`` is directed to.
            - ``ProxyEgressPort`` - (Required) Specifies the port that outgoing traffic from the ``AppPorts`` is directed to.
            - ``EgressIgnoredPorts`` - (Required) The egress traffic going to the specified ports is ignored and not redirected to the ``ProxyEgressPort`` . It can be an empty list.
            - ``EgressIgnoredIPs`` - (Required) The egress traffic going to the specified IP addresses is ignored and not redirected to the ``ProxyEgressPort`` . It can be an empty list.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html#cfn-ecs-taskdefinition-proxyconfiguration-proxyconfigurationproperties
            '''
            result = self._values.get("proxy_configuration_properties")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.KeyValuePairProperty"]]]], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The proxy type.

            The only supported value is ``APPMESH`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html#cfn-ecs-taskdefinition-proxyconfiguration-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProxyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.RepositoryCredentialsProperty",
        jsii_struct_bases=[],
        name_mapping={"credentials_parameter": "credentialsParameter"},
    )
    class RepositoryCredentialsProperty:
        def __init__(
            self,
            *,
            credentials_parameter: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The repository credentials for private registry authentication.

            :param credentials_parameter: The Amazon Resource Name (ARN) of the secret containing the private repository credentials. .. epigraph:: When you use the Amazon ECS API, AWS CLI , or AWS SDK, if the secret exists in the same Region as the task that you're launching then you can use either the full ARN or the name of the secret. When you use the AWS Management Console, you must specify the full ARN of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-repositorycredentials.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                repository_credentials_property = ecs.CfnTaskDefinition.RepositoryCredentialsProperty(
                    credentials_parameter="credentialsParameter"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__399418444e0587ba8546ff3739e9e32a136259706817def7df6a20ba3b9afec6)
                check_type(argname="argument credentials_parameter", value=credentials_parameter, expected_type=type_hints["credentials_parameter"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if credentials_parameter is not None:
                self._values["credentials_parameter"] = credentials_parameter

        @builtins.property
        def credentials_parameter(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the secret containing the private repository credentials.

            .. epigraph::

               When you use the Amazon ECS API, AWS CLI , or AWS SDK, if the secret exists in the same Region as the task that you're launching then you can use either the full ARN or the name of the secret. When you use the AWS Management Console, you must specify the full ARN of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-repositorycredentials.html#cfn-ecs-taskdefinition-repositorycredentials-credentialsparameter
            '''
            result = self._values.get("credentials_parameter")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RepositoryCredentialsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.ResourceRequirementProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "value": "value"},
    )
    class ResourceRequirementProperty:
        def __init__(self, *, type: builtins.str, value: builtins.str) -> None:
            '''The type and amount of a resource to assign to a container.

            The supported resource types are GPUs and Elastic Inference accelerators. For more information, see `Working with GPUs on Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-gpu.html>`_ or `Working with Amazon Elastic Inference on Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-inference.html>`_ in the *Amazon Elastic Container Service Developer Guide*

            :param type: The type of resource to assign to a container. The supported values are ``GPU`` or ``InferenceAccelerator`` .
            :param value: The value for the specified resource type. If the ``GPU`` type is used, the value is the number of physical ``GPUs`` the Amazon ECS container agent reserves for the container. The number of GPUs that's reserved for all containers in a task can't exceed the number of available GPUs on the container instance that the task is launched on. If the ``InferenceAccelerator`` type is used, the ``value`` matches the ``deviceName`` for an `InferenceAccelerator <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_InferenceAccelerator.html>`_ specified in a task definition.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                resource_requirement_property = ecs.CfnTaskDefinition.ResourceRequirementProperty(
                    type="type",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9f0e59715edb889cdb53a1111c692d2fcadb558bf86158ff87dec568d5ffa94e)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
                "value": value,
            }

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of resource to assign to a container.

            The supported values are ``GPU`` or ``InferenceAccelerator`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html#cfn-ecs-taskdefinition-resourcerequirement-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The value for the specified resource type.

            If the ``GPU`` type is used, the value is the number of physical ``GPUs`` the Amazon ECS container agent reserves for the container. The number of GPUs that's reserved for all containers in a task can't exceed the number of available GPUs on the container instance that the task is launched on.

            If the ``InferenceAccelerator`` type is used, the ``value`` matches the ``deviceName`` for an `InferenceAccelerator <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_InferenceAccelerator.html>`_ specified in a task definition.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html#cfn-ecs-taskdefinition-resourcerequirement-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ResourceRequirementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.RuntimePlatformProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cpu_architecture": "cpuArchitecture",
            "operating_system_family": "operatingSystemFamily",
        },
    )
    class RuntimePlatformProperty:
        def __init__(
            self,
            *,
            cpu_architecture: typing.Optional[builtins.str] = None,
            operating_system_family: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Information about the platform for the Amazon ECS service or task.

            For more information about ``RuntimePlatform`` , see `RuntimePlatform <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#runtime-platform>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param cpu_architecture: The CPU architecture. You can run your Linux tasks on an ARM-based platform by setting the value to ``ARM64`` . This option is available for tasks that run on Linux Amazon EC2 instance or Linux containers on Fargate.
            :param operating_system_family: The operating system.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-runtimeplatform.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                runtime_platform_property = ecs.CfnTaskDefinition.RuntimePlatformProperty(
                    cpu_architecture="cpuArchitecture",
                    operating_system_family="operatingSystemFamily"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__13b8fed83205e8f8c805885892fec108a4cc1b8e74b0e647c14e5b5235ba8e7c)
                check_type(argname="argument cpu_architecture", value=cpu_architecture, expected_type=type_hints["cpu_architecture"])
                check_type(argname="argument operating_system_family", value=operating_system_family, expected_type=type_hints["operating_system_family"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cpu_architecture is not None:
                self._values["cpu_architecture"] = cpu_architecture
            if operating_system_family is not None:
                self._values["operating_system_family"] = operating_system_family

        @builtins.property
        def cpu_architecture(self) -> typing.Optional[builtins.str]:
            '''The CPU architecture.

            You can run your Linux tasks on an ARM-based platform by setting the value to ``ARM64`` . This option is available for tasks that run on Linux Amazon EC2 instance or Linux containers on Fargate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-runtimeplatform.html#cfn-ecs-taskdefinition-runtimeplatform-cpuarchitecture
            '''
            result = self._values.get("cpu_architecture")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def operating_system_family(self) -> typing.Optional[builtins.str]:
            '''The operating system.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-runtimeplatform.html#cfn-ecs-taskdefinition-runtimeplatform-operatingsystemfamily
            '''
            result = self._values.get("operating_system_family")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RuntimePlatformProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.SecretProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value_from": "valueFrom"},
    )
    class SecretProperty:
        def __init__(self, *, name: builtins.str, value_from: builtins.str) -> None:
            '''An object representing the secret to expose to your container.

            Secrets can be exposed to a container in the following ways:

            - To inject sensitive data into your containers as environment variables, use the ``secrets`` container definition parameter.
            - To reference sensitive information in the log configuration of a container, use the ``secretOptions`` container definition parameter.

            For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param name: The name of the secret.
            :param value_from: The secret to expose to the container. The supported values are either the full ARN of the AWS Secrets Manager secret or the full ARN of the parameter in the SSM Parameter Store. For information about the require AWS Identity and Access Management permissions, see `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-secrets.html#secrets-iam>`_ (for Secrets Manager) or `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-parameters.html>`_ (for Systems Manager Parameter store) in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: If the SSM Parameter Store parameter exists in the same Region as the task you're launching, then you can use either the full ARN or name of the parameter. If the parameter exists in a different Region, then the full ARN must be specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-secret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                secret_property = ecs.CfnTaskDefinition.SecretProperty(
                    name="name",
                    value_from="valueFrom"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e3ca5a3f91fcd5bfb7247a9348affa35a1e4aa755ce41b312515338a898948df)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value_from", value=value_from, expected_type=type_hints["value_from"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "value_from": value_from,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-secret.html#cfn-ecs-taskdefinition-secret-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value_from(self) -> builtins.str:
            '''The secret to expose to the container.

            The supported values are either the full ARN of the AWS Secrets Manager secret or the full ARN of the parameter in the SSM Parameter Store.

            For information about the require AWS Identity and Access Management permissions, see `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-secrets.html#secrets-iam>`_ (for Secrets Manager) or `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-parameters.html>`_ (for Systems Manager Parameter store) in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               If the SSM Parameter Store parameter exists in the same Region as the task you're launching, then you can use either the full ARN or name of the parameter. If the parameter exists in a different Region, then the full ARN must be specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-secret.html#cfn-ecs-taskdefinition-secret-valuefrom
            '''
            result = self._values.get("value_from")
            assert result is not None, "Required property 'value_from' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.SystemControlProperty",
        jsii_struct_bases=[],
        name_mapping={"namespace": "namespace", "value": "value"},
    )
    class SystemControlProperty:
        def __init__(
            self,
            *,
            namespace: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A list of namespaced kernel parameters to set in the container.

            This parameter maps to ``Sysctls`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--sysctl`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            We don't recommend that you specify network-related ``systemControls`` parameters for multiple containers in a single task. This task also uses either the ``awsvpc`` or ``host`` network mode. It does it for the following reasons.

            - For tasks that use the ``awsvpc`` network mode, if you set ``systemControls`` for any container, it applies to all containers in the task. If you set different ``systemControls`` for multiple containers in a single task, the container that's started last determines which ``systemControls`` take effect.
            - For tasks that use the ``host`` network mode, the ``systemControls`` parameter applies to the container instance's kernel parameter and that of all containers of any tasks running on that container instance.

            :param namespace: The namespaced kernel parameter to set a ``value`` for.
            :param value: The value for the namespaced kernel parameter that's specified in ``namespace`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-systemcontrol.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                system_control_property = ecs.CfnTaskDefinition.SystemControlProperty(
                    namespace="namespace",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__305ae5fb011957fbe49f54d1e9f9e507b694c055ef25ee0b32d191cbb8a33c96)
                check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if namespace is not None:
                self._values["namespace"] = namespace
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def namespace(self) -> typing.Optional[builtins.str]:
            '''The namespaced kernel parameter to set a ``value`` for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-systemcontrol.html#cfn-ecs-taskdefinition-systemcontrol-namespace
            '''
            result = self._values.get("namespace")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value for the namespaced kernel parameter that's specified in ``namespace`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-systemcontrol.html#cfn-ecs-taskdefinition-systemcontrol-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SystemControlProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "expression": "expression"},
    )
    class TaskDefinitionPlacementConstraintProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            expression: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The constraint on task placement in the task definition.

            For more information, see `Task placement constraints <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               Task placement constraints aren't supported for tasks run on AWS Fargate .

            :param type: The type of constraint. The ``MemberOf`` constraint restricts selection to be from a group of valid candidates.
            :param expression: A cluster query language expression to apply to the constraint. For more information, see `Cluster query language <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-taskdefinitionplacementconstraint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                task_definition_placement_constraint_property = ecs.CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty(
                    type="type",
                
                    # the properties below are optional
                    expression="expression"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d5e542ed23a783cdf71e481e057352c685654e02157299c4e3e6be8e33ec00f1)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if expression is not None:
                self._values["expression"] = expression

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of constraint.

            The ``MemberOf`` constraint restricts selection to be from a group of valid candidates.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-taskdefinitionplacementconstraint.html#cfn-ecs-taskdefinition-taskdefinitionplacementconstraint-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def expression(self) -> typing.Optional[builtins.str]:
            '''A cluster query language expression to apply to the constraint.

            For more information, see `Cluster query language <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-taskdefinitionplacementconstraint.html#cfn-ecs-taskdefinition-taskdefinitionplacementconstraint-expression
            '''
            result = self._values.get("expression")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TaskDefinitionPlacementConstraintProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.TmpfsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "size": "size",
            "container_path": "containerPath",
            "mount_options": "mountOptions",
        },
    )
    class TmpfsProperty:
        def __init__(
            self,
            *,
            size: jsii.Number,
            container_path: typing.Optional[builtins.str] = None,
            mount_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The container path, mount options, and size of the tmpfs mount.

            :param size: The maximum size (in MiB) of the tmpfs volume.
            :param container_path: The absolute file path where the tmpfs volume is to be mounted.
            :param mount_options: The list of tmpfs volume mount options. Valid values: ``"defaults" | "ro" | "rw" | "suid" | "nosuid" | "dev" | "nodev" | "exec" | "noexec" | "sync" | "async" | "dirsync" | "remount" | "mand" | "nomand" | "atime" | "noatime" | "diratime" | "nodiratime" | "bind" | "rbind" | "unbindable" | "runbindable" | "private" | "rprivate" | "shared" | "rshared" | "slave" | "rslave" | "relatime" | "norelatime" | "strictatime" | "nostrictatime" | "mode" | "uid" | "gid" | "nr_inodes" | "nr_blocks" | "mpol"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                tmpfs_property = ecs.CfnTaskDefinition.TmpfsProperty(
                    size=123,
                
                    # the properties below are optional
                    container_path="containerPath",
                    mount_options=["mountOptions"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3fb1a407faa8f21c50c1ffcc36460e9539fd4a2ef2a3ef159f6d20b80de0e1a1)
                check_type(argname="argument size", value=size, expected_type=type_hints["size"])
                check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
                check_type(argname="argument mount_options", value=mount_options, expected_type=type_hints["mount_options"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "size": size,
            }
            if container_path is not None:
                self._values["container_path"] = container_path
            if mount_options is not None:
                self._values["mount_options"] = mount_options

        @builtins.property
        def size(self) -> jsii.Number:
            '''The maximum size (in MiB) of the tmpfs volume.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html#cfn-ecs-taskdefinition-tmpfs-size
            '''
            result = self._values.get("size")
            assert result is not None, "Required property 'size' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''The absolute file path where the tmpfs volume is to be mounted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html#cfn-ecs-taskdefinition-tmpfs-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def mount_options(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The list of tmpfs volume mount options.

            Valid values: ``"defaults" | "ro" | "rw" | "suid" | "nosuid" | "dev" | "nodev" | "exec" | "noexec" | "sync" | "async" | "dirsync" | "remount" | "mand" | "nomand" | "atime" | "noatime" | "diratime" | "nodiratime" | "bind" | "rbind" | "unbindable" | "runbindable" | "private" | "rprivate" | "shared" | "rshared" | "slave" | "rslave" | "relatime" | "norelatime" | "strictatime" | "nostrictatime" | "mode" | "uid" | "gid" | "nr_inodes" | "nr_blocks" | "mpol"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html#cfn-ecs-taskdefinition-tmpfs-mountoptions
            '''
            result = self._values.get("mount_options")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TmpfsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.UlimitProperty",
        jsii_struct_bases=[],
        name_mapping={
            "hard_limit": "hardLimit",
            "name": "name",
            "soft_limit": "softLimit",
        },
    )
    class UlimitProperty:
        def __init__(
            self,
            *,
            hard_limit: jsii.Number,
            name: builtins.str,
            soft_limit: jsii.Number,
        ) -> None:
            '''The ``ulimit`` settings to pass to the container.

            Amazon ECS tasks hosted on AWS Fargate use the default resource limit values set by the operating system with the exception of the ``nofile`` resource limit parameter which AWS Fargate overrides. The ``nofile`` resource limit sets a restriction on the number of open files that a container can use. The default ``nofile`` soft limit is ``1024`` and the default hard limit is ``4096`` .

            You can specify the ``ulimit`` settings for a container in a task definition.

            :param hard_limit: The hard limit for the ``ulimit`` type.
            :param name: The ``type`` of the ``ulimit`` .
            :param soft_limit: The soft limit for the ``ulimit`` type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ulimit.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                ulimit_property = ecs.CfnTaskDefinition.UlimitProperty(
                    hard_limit=123,
                    name="name",
                    soft_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0adfefae3543396f8f72f2d545bf100dabd5bca10ce84fb035deeec38c16a8b1)
                check_type(argname="argument hard_limit", value=hard_limit, expected_type=type_hints["hard_limit"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument soft_limit", value=soft_limit, expected_type=type_hints["soft_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "hard_limit": hard_limit,
                "name": name,
                "soft_limit": soft_limit,
            }

        @builtins.property
        def hard_limit(self) -> jsii.Number:
            '''The hard limit for the ``ulimit`` type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ulimit.html#cfn-ecs-taskdefinition-ulimit-hardlimit
            '''
            result = self._values.get("hard_limit")
            assert result is not None, "Required property 'hard_limit' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def name(self) -> builtins.str:
            '''The ``type`` of the ``ulimit`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ulimit.html#cfn-ecs-taskdefinition-ulimit-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def soft_limit(self) -> jsii.Number:
            '''The soft limit for the ``ulimit`` type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ulimit.html#cfn-ecs-taskdefinition-ulimit-softlimit
            '''
            result = self._values.get("soft_limit")
            assert result is not None, "Required property 'soft_limit' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UlimitProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.VolumeFromProperty",
        jsii_struct_bases=[],
        name_mapping={"read_only": "readOnly", "source_container": "sourceContainer"},
    )
    class VolumeFromProperty:
        def __init__(
            self,
            *,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            source_container: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Details on a data volume from another container in the same task definition.

            :param read_only: If this value is ``true`` , the container has read-only access to the volume. If this value is ``false`` , then the container can write to the volume. The default value is ``false`` .
            :param source_container: The name of another container within the same task definition to mount volumes from.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumefrom.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                volume_from_property = ecs.CfnTaskDefinition.VolumeFromProperty(
                    read_only=False,
                    source_container="sourceContainer"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d439c801995bea63d48c7a955f7150f2c0b89e63e1cdda4f23240a715c9af735)
                check_type(argname="argument read_only", value=read_only, expected_type=type_hints["read_only"])
                check_type(argname="argument source_container", value=source_container, expected_type=type_hints["source_container"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if read_only is not None:
                self._values["read_only"] = read_only
            if source_container is not None:
                self._values["source_container"] = source_container

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If this value is ``true`` , the container has read-only access to the volume.

            If this value is ``false`` , then the container can write to the volume. The default value is ``false`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumefrom.html#cfn-ecs-taskdefinition-volumefrom-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def source_container(self) -> typing.Optional[builtins.str]:
            '''The name of another container within the same task definition to mount volumes from.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumefrom.html#cfn-ecs-taskdefinition-volumefrom-sourcecontainer
            '''
            result = self._values.get("source_container")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeFromProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinition.VolumeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "docker_volume_configuration": "dockerVolumeConfiguration",
            "efs_volume_configuration": "efsVolumeConfiguration",
            "host": "host",
            "name": "name",
        },
    )
    class VolumeProperty:
        def __init__(
            self,
            *,
            docker_volume_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.DockerVolumeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            efs_volume_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.EFSVolumeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            host: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnTaskDefinition.HostVolumePropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``Volume`` property specifies a data volume used in a task definition.

            For tasks that use a Docker volume, specify a ``DockerVolumeConfiguration`` . For tasks that use a bind mount host volume, specify a ``host`` and optional ``sourcePath`` . For more information about ``host`` and optional ``sourcePath`` , see `Volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#volumes>`_ and `Using Data Volumes in Tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html>`_ .

            :param docker_volume_configuration: This parameter is specified when you use Docker volumes. Windows containers only support the use of the ``local`` driver. To use bind mounts, specify the ``host`` parameter instead. .. epigraph:: Docker volumes aren't supported by tasks run on AWS Fargate .
            :param efs_volume_configuration: This parameter is specified when you use an Amazon Elastic File System file system for task storage.
            :param host: This parameter is specified when you use bind mount host volumes. The contents of the ``host`` parameter determine whether your bind mount host volume persists on the host container instance and where it's stored. If the ``host`` parameter is empty, then the Docker daemon assigns a host path for your data volume. However, the data isn't guaranteed to persist after the containers that are associated with it stop running. Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives. For example, you can mount ``C:\\my\\path:C:\\my\\path`` and ``D:\\:D:\\`` , but not ``D:\\my\\path:C:\\my\\path`` or ``D:\\:C:\\my\\path`` .
            :param name: The name of the volume. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. This name is referenced in the ``sourceVolume`` parameter of container definition ``mountPoints`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volume.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ecs as ecs
                
                volume_property = ecs.CfnTaskDefinition.VolumeProperty(
                    docker_volume_configuration=ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty(
                        autoprovision=False,
                        driver="driver",
                        driver_opts={
                            "driver_opts_key": "driverOpts"
                        },
                        labels={
                            "labels_key": "labels"
                        },
                        scope="scope"
                    ),
                    efs_volume_configuration=ecs.CfnTaskDefinition.EFSVolumeConfigurationProperty(
                        filesystem_id="filesystemId",
                
                        # the properties below are optional
                        authorization_config=ecs.CfnTaskDefinition.AuthorizationConfigProperty(
                            access_point_id="accessPointId",
                            iam="iam"
                        ),
                        root_directory="rootDirectory",
                        transit_encryption="transitEncryption",
                        transit_encryption_port=123
                    ),
                    host=ecs.CfnTaskDefinition.HostVolumePropertiesProperty(
                        source_path="sourcePath"
                    ),
                    name="name"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0b8eeaf561ff6334fbacca7f706104a9a5e10a366526af78d2a6b7173ee46302)
                check_type(argname="argument docker_volume_configuration", value=docker_volume_configuration, expected_type=type_hints["docker_volume_configuration"])
                check_type(argname="argument efs_volume_configuration", value=efs_volume_configuration, expected_type=type_hints["efs_volume_configuration"])
                check_type(argname="argument host", value=host, expected_type=type_hints["host"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if docker_volume_configuration is not None:
                self._values["docker_volume_configuration"] = docker_volume_configuration
            if efs_volume_configuration is not None:
                self._values["efs_volume_configuration"] = efs_volume_configuration
            if host is not None:
                self._values["host"] = host
            if name is not None:
                self._values["name"] = name

        @builtins.property
        def docker_volume_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.DockerVolumeConfigurationProperty"]]:
            '''This parameter is specified when you use Docker volumes.

            Windows containers only support the use of the ``local`` driver. To use bind mounts, specify the ``host`` parameter instead.
            .. epigraph::

               Docker volumes aren't supported by tasks run on AWS Fargate .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volume.html#cfn-ecs-taskdefinition-volume-dockervolumeconfiguration
            '''
            result = self._values.get("docker_volume_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.DockerVolumeConfigurationProperty"]], result)

        @builtins.property
        def efs_volume_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.EFSVolumeConfigurationProperty"]]:
            '''This parameter is specified when you use an Amazon Elastic File System file system for task storage.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volume.html#cfn-ecs-taskdefinition-volume-efsvolumeconfiguration
            '''
            result = self._values.get("efs_volume_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.EFSVolumeConfigurationProperty"]], result)

        @builtins.property
        def host(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.HostVolumePropertiesProperty"]]:
            '''This parameter is specified when you use bind mount host volumes.

            The contents of the ``host`` parameter determine whether your bind mount host volume persists on the host container instance and where it's stored. If the ``host`` parameter is empty, then the Docker daemon assigns a host path for your data volume. However, the data isn't guaranteed to persist after the containers that are associated with it stop running.

            Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives. For example, you can mount ``C:\\my\\path:C:\\my\\path`` and ``D:\\:D:\\`` , but not ``D:\\my\\path:C:\\my\\path`` or ``D:\\:C:\\my\\path`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volume.html#cfn-ecs-taskdefinition-volume-host
            '''
            result = self._values.get("host")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnTaskDefinition.HostVolumePropertiesProperty"]], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the volume.

            Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. This name is referenced in the ``sourceVolume`` parameter of container definition ``mountPoints`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volume.html#cfn-ecs-taskdefinition-volume-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ecs.CfnTaskDefinitionProps",
    jsii_struct_bases=[],
    name_mapping={
        "container_definitions": "containerDefinitions",
        "cpu": "cpu",
        "ephemeral_storage": "ephemeralStorage",
        "execution_role_arn": "executionRoleArn",
        "family": "family",
        "inference_accelerators": "inferenceAccelerators",
        "ipc_mode": "ipcMode",
        "memory": "memory",
        "network_mode": "networkMode",
        "pid_mode": "pidMode",
        "placement_constraints": "placementConstraints",
        "proxy_configuration": "proxyConfiguration",
        "requires_compatibilities": "requiresCompatibilities",
        "runtime_platform": "runtimePlatform",
        "tags": "tags",
        "task_role_arn": "taskRoleArn",
        "volumes": "volumes",
    },
)
class CfnTaskDefinitionProps:
    def __init__(
        self,
        *,
        container_definitions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnTaskDefinition.ContainerDefinitionProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        cpu: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnTaskDefinition.EphemeralStorageProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        execution_role_arn: typing.Optional[builtins.str] = None,
        family: typing.Optional[builtins.str] = None,
        inference_accelerators: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnTaskDefinition.InferenceAcceleratorProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ipc_mode: typing.Optional[builtins.str] = None,
        memory: typing.Optional[builtins.str] = None,
        network_mode: typing.Optional[builtins.str] = None,
        pid_mode: typing.Optional[builtins.str] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        proxy_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnTaskDefinition.ProxyConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        requires_compatibilities: typing.Optional[typing.Sequence[builtins.str]] = None,
        runtime_platform: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnTaskDefinition.RuntimePlatformProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        task_role_arn: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnTaskDefinition.VolumeProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnTaskDefinition``.

        :param container_definitions: A list of container definitions in JSON format that describe the different containers that make up your task. For more information about container definition parameters and defaults, see `Amazon ECS Task Definitions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param cpu: The number of ``cpu`` units used by the task. If you use the EC2 launch type, this field is optional. Any value can be used. If you use the Fargate launch type, this field is required. You must use one of the following values. The value that you choose determines your range of valid values for the ``memory`` parameter. The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate. - 256 (.25 vCPU) - Available ``memory`` values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - 512 (.5 vCPU) - Available ``memory`` values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - 1024 (1 vCPU) - Available ``memory`` values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - 2048 (2 vCPU) - Available ``memory`` values: 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - 4096 (4 vCPU) - Available ``memory`` values: 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - 8192 (8 vCPU) - Available ``memory`` values: 16 GB and 60 GB in 4 GB increments This option requires Linux platform ``1.4.0`` or later. - 16384 (16vCPU) - Available ``memory`` values: 32GB and 120 GB in 8 GB increments This option requires Linux platform ``1.4.0`` or later.
        :param ephemeral_storage: The ephemeral storage settings to use for tasks run with the task definition.
        :param execution_role_arn: The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent permission to make AWS API calls on your behalf. The task execution IAM role is required depending on the requirements of your task. For more information, see `Amazon ECS task execution IAM role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_execution_IAM_role.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param family: The name of a family that this task 