'''
# Amazon Relational Database Service Construct Library

```python
import aws_cdk.aws_rds as rds
```

## Starting a clustered database

To set up a clustered database (like Aurora), define a `DatabaseCluster`. You must
always launch a database in a VPC. Use the `vpcSubnets` attribute to control whether
your instances will be launched privately or publicly:

You must specify the instance to use as the writer, along with an optional list
of readers (up to 15).

```python
# Example automatically generated from non-compiling source. May contain errors.
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_2_08_1),
    credentials=rds.Credentials.from_generated_secret("clusteradmin"),  # Optional - will default to 'admin' username and generated password
    writer=rds.ClusterInstance.provisioned("writer",
        readers=[
            rds.ClusterInstance.provisioned("reader1", promotion_tier=1),
            rds.ClusterInstance.serverless_v2("reader2")
        ],
        vpc_subnets={
            "subnet_type": ec2.SubnetType.PRIVATE_WITH_EGRESS
        },
        vpc=vpc
    )
```

To adopt Aurora I/O-Optimized. Speicify `DBClusterStorageType.AURORA_IOPT1` on the `storageType` property.

```python
# Example automatically generated from non-compiling source. May contain errors.
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_postgres(version=rds.AuroraPostgresEngineVersion.VER_15_2),
    credentials=rds.Credentials.from_username("adminuser", password=cdk.SecretValue.unsafe_plain_text("7959866cacc02c2d243ecfe177464fe6")),
    instance_props=rds.InstanceProps(
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.X2G, ec2.InstanceSize.XLARGE),
        vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC),
        vpc=vpc
    ),
    storage_type=rds.DBClusterStorageType.AURORA_IOPT1
)
```

If there isn't a constant for the exact version you want to use,
all of the `Version` classes have a static `of` method that can be used to create an arbitrary version.

```python
custom_engine_version = rds.AuroraMysqlEngineVersion.of("5.7.mysql_aurora.2.08.1")
```

By default, the master password will be generated and stored in AWS Secrets Manager with auto-generated description.

Your cluster will be empty by default. To add a default database upon construction, specify the
`defaultDatabaseName` attribute.

To use dual-stack mode, specify `NetworkType.DUAL` on the `networkType` property:

```python
# vpc: ec2.Vpc
# VPC and subnets must have IPv6 CIDR blocks
cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_3_02_1),
    writer=rds.ClusterInstance.provisioned("writer",
        publicly_accessible=False
    ),
    vpc=vpc,
    network_type=rds.NetworkType.DUAL
)
```

For more information about dual-stack mode, see [Working with a DB cluster in a VPC](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html).

Use `DatabaseClusterFromSnapshot` to create a cluster from a snapshot:

```python
# vpc: ec2.Vpc

rds.DatabaseClusterFromSnapshot(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora(version=rds.AuroraEngineVersion.VER_1_22_2),
    writer=rds.ClusterInstance.provisioned("writer"),
    vpc=vpc,
    snapshot_identifier="mySnapshot"
)
```

### Updating the database instances in a cluster

Database cluster instances may be updated in bulk or on a rolling basis.

An update to all instances in a cluster may cause significant downtime. To reduce the downtime, set the `instanceUpdateBehavior` property in `DatabaseClusterBaseProps` to `InstanceUpdateBehavior.ROLLING`. This adds a dependency between each instance so the update is performed on only one instance at a time.

Use `InstanceUpdateBehavior.BULK` to update all instances at once.

```python
# Example automatically generated from non-compiling source. May contain errors.
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_3_01_0),
    writer=rds.ClusterInstance.provisioned({
        "instance_type": ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL)
    }),
    readers=readers, rds.ClusterInstance.provisioned("reader")=,
    instance_update_behaviour=rds.InstanceUpdateBehaviour.ROLLING,  # Optional - defaults to rds.InstanceUpdateBehaviour.BULK
    vpc=vpc
)
```

### Serverless V2 instances in a Cluster

It is possible to create an RDS cluster with *both* serverlessV2 and provisioned
instances. For example, this will create a cluster with a provisioned writer and
a serverless v2 reader.

> *Note* Before getting starting with this type of cluster it is
> highly recommended that you read through the [Developer Guide](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html)
> which goes into much more detail on the things you need to take into
> consideration.

```python
# Example automatically generated from non-compiling source. May contain errors.
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_3_01_0),
    writer=rds.ClusterInstance.provisioned("writer"),
    readers=[
        rds.ClusterInstance.serverless_v2("reader")
    ],
    vpc=vpc
)
```

### Monitoring

There are some CloudWatch metrics that are [important for Aurora Serverless
v2](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html#aurora-serverless-v2.viewing.monitoring).

* `ServerlessDatabaseCapacity`: An instance-level metric that can also be
  evaluated at the cluster level. At the cluster-level it represents the average
  capacity of all the instances in the cluster.
* `ACUUtilization`: Value of the `ServerlessDatabaseCapacity`/ max ACU of the
  cluster.

```python
# Example automatically generated from non-compiling source. May contain errors.
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_3_01_0),
    writer=rds.ClusterInstance.provisioned("writer"),
    readers=[
        rds.ClusterInstance.serverless_v2("reader")
    ],
    vpc=vpc
)

cluster.metric_serverless_database_capacity(
    period=Duration.minutes(10)
).create_alarm(self, "capacity",
    threshold=1.5,
    evaluation_periods=3
)
cluster.metric_aCUUtilization(
    period=Duration.minutes(10)
).create_alarm(self, "alarm",
    evaluation_periods=3,
    threshold=90
)
```

#### Capacity & Scaling

There are some things to take into consideration with Aurora Serverless v2.

To create a cluster that can support serverless v2 instances you configure a
minimum and maximum capacity range on the cluster. This is an example showing
the default values:

```python
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_3_01_0),
    writer=rds.ClusterInstance.serverless_v2("writer"),
    serverless_v2_min_capacity=0.5,
    serverless_v2_max_capacity=2,
    vpc=vpc
)
```

The capacity is defined as a number of Aurora capacity units (ACUs). You can
specify in half-step increments (40, 40.5, 41, etc). Each serverless instance in
the cluster inherits the capacity that is defined on the cluster. It is not
possible to configure separate capacity at the instance level.

The maximum capacity is mainly used for budget control since it allows you to
set a cap on how high your instance can scale.

The minimum capacity is a little more involved. This controls a couple different
things.

* The scale-up rate is proportional to the current capacity (larger instances
  scale up faster)

  * Adjust the minimum capacity to obtain a suitable scaling rate
* Network throughput is proportional to capacity

> *Info* More complete details can be found [in the docs](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html#aurora-serverless-v2-examples-setting-capacity-range-for-cluster)

Another way that you control the capacity/scaling of your serverless v2 reader
instances is based on the [promotion tier](https://aws.amazon.com/blogs/aws/additional-failover-control-for-amazon-aurora/)
which can be between 0-15. Any serverless v2 instance in the 0-1 tiers will scale alongside the
writer even if the current read load does not require the capacity. This is
because instances in the 0-1 tier are first priority for failover and Aurora
wants to ensure that in the event of a failover the reader that gets promoted is
scaled to handle the write load.

```python
# Example automatically generated from non-compiling source. May contain errors.
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_2_08_1),
    writer=rds.ClusterInstance.serverless_v2("writer"),
    readers=[
        # will be put in promotion tier 1 and will scale with the writer
        rds.ClusterInstance.serverless_v2("reader1", scale_with_writer=True),
        # will be put in promotion tier 2 and will not scale with the writer
        rds.ClusterInstance.serverless_v2("reader2")
    ],
    vpc=vpc
)
```

* When the writer scales up, any readers in tier 0-1 will scale up to match
* Scaling for tier 2-15 is independent of what is happening on the writer
* Readers in tier 2-15 scale up based on read load against the individual reader

When configuring your cluster it is important to take this into consideration
and ensure that in the event of a failover there is an instance that is scaled
up to take over.

### Mixing Serverless v2 and Provisioned instances

You are able to create a cluster that has both provisioned and serverless
instances. [This blog post](https://aws.amazon.com/blogs/database/evaluate-amazon-aurora-serverless-v2-for-your-provisioned-aurora-clusters/)
has an excellent guide on choosing between serverless and provisioned instances
based on use case.

There are a couple of high level differences:

* Engine Version (serverless only supports MySQL 8+ & PostgreSQL 13+)
* Memory up to 256GB can be supported by serverless

#### Provisioned writer

With a provisioned writer and serverless v2 readers, some of the serverless
readers will need to be configured to scale with the writer so they can act as
failover targets. You will need to determine the correct capacity based on the
provisioned instance type and it's utilization.

As an example, if the CPU utilization for a db.r6g.4xlarge (128 GB) instance
stays at 10% most times, then the minimum ACUs may be set at 6.5 ACUs
(10% of 128 GB) and maximum may be set at 64 ACUs (64x2GB=128GB). Keep in mind
that the speed at which the serverless instance can scale up is determined by
the minimum capacity so if your cluster has spiky workloads you may need to set
a higher minimum capacity.

```python
# Example automatically generated from non-compiling source. May contain errors.
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_2_08_1),
    writer=rds.ClusterInstance.provisioned("writer",
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.R6G, ec2.InstanceSize.XLARGE4)
    ),
    serverless_v2_min_capacity=6.5,
    serverless_v2_max_capacity=64,
    readers=[
        # will be put in promotion tier 1 and will scale with the writer
        rds.ClusterInstance.serverless_v2("reader1", scale_with_writer=True),
        # will be put in promotion tier 2 and will not scale with the writer
        rds.ClusterInstance.serverless_v2("reader2")
    ],
    vpc=vpc
)
```

In the above example `reader1` will scale with the writer based on the writer's
utilization. So if the writer were to go to `50%` utilization then `reader1`
would scale up to use `32` ACUs. If the read load stayed consistent then
`reader2` may remain at `6.5` since it is not configured to scale with the
writer.

If one of your Aurora Serverless v2 DB instances consistently reaches the
limit of its maximum capacity, Aurora indicates this condition by setting the
DB instance to a status of `incompatible-parameters`. While the DB instance has
the incompatible-parameters status, some operations are blocked. For example,
you can't upgrade the engine version.

### Migrating from instanceProps

Creating instances in a `DatabaseCluster` using `instanceProps` & `instances` is
deprecated. To migrate to the new properties you can provide the
`isFromLegacyInstanceProps` property.

For example, in order to migrate from this deprecated config:

```python
# Example automatically generated from non-compiling source. May contain errors.
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(stack, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(
        version=rds.AuroraMysqlEngineVersion.VER_3_03_0
    ),
    instances=2,
    instance_props=rds.InstanceProps(
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL),
        vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC),
        vpc=vpc
    )
)
```

You would need to migrate to this. The old method of providing `instanceProps`
and `instances` will create the number of `instances` that you provide. The
first instance will be the writer and the rest will be the readers. It's
important that the `id` that you provide is `Instance{NUMBER}`. The writer
should always be `Instance1` and the readers will increment from there.

Make sure to run a `cdk diff` before deploying to make sure that all changes are
expected. **Always test the migration in a non-production environment first.**

```python
# Example automatically generated from non-compiling source. May contain errors.
# vpc: ec2.Vpc
instance_props = {
    "instance_type": ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL),
    "is_from_legacy_instance_props": True
}
cluster = rds.DatabaseCluster(stack, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(
        version=rds.AuroraMysqlEngineVersion.VER_3_03_0
    ),
    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC),
    vpc=vpc,
    writer=ClusterInstance.provisioned("Instance1",
        (SpreadAssignment ...instanceProps
          instance_props)
    ),
    readers=[
        ClusterInstance.provisioned("Instance2",
            (SpreadAssignment ...instanceProps
              instance_props)
        )
    ]
)
```

## Starting an instance database

To set up an instance database, define a `DatabaseInstance`. You must
always launch a database in a VPC. Use the `vpcSubnets` attribute to control whether
your instances will be launched privately or publicly:

```python
# vpc: ec2.Vpc

instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    # optional, defaults to m5.large
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL),
    credentials=rds.Credentials.from_generated_secret("syscdk"),  # Optional - will default to 'admin' username and generated password
    vpc=vpc,
    vpc_subnets=ec2.SubnetSelection(
        subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS
    )
)
```

If there isn't a constant for the exact engine version you want to use,
all of the `Version` classes have a static `of` method that can be used to create an arbitrary version.

```python
custom_engine_version = rds.OracleEngineVersion.of("19.0.0.0.ru-2020-04.rur-2020-04.r1", "19")
```

By default, the master password will be generated and stored in AWS Secrets Manager.

To use the storage auto scaling option of RDS you can specify the maximum allocated storage.
This is the upper limit to which RDS can automatically scale the storage. More info can be found
[here](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling)
Example for max storage configuration:

```python
# vpc: ec2.Vpc

instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_15_2),
    # optional, defaults to m5.large
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
    vpc=vpc,
    max_allocated_storage=200
)
```

To use dual-stack mode, specify `NetworkType.DUAL` on the `networkType` property:

```python
# vpc: ec2.Vpc
# VPC and subnets must have IPv6 CIDR blocks
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_15_2),
    vpc=vpc,
    network_type=rds.NetworkType.DUAL,
    publicly_accessible=False
)
```

For more information about dual-stack mode, see [Working with a DB instance in a VPC](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html).

Use `DatabaseInstanceFromSnapshot` and `DatabaseInstanceReadReplica` to create an instance from snapshot or
a source database respectively:

```python
# vpc: ec2.Vpc

# source_instance: rds.DatabaseInstance

rds.DatabaseInstanceFromSnapshot(self, "Instance",
    snapshot_identifier="my-snapshot",
    engine=rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_15_2),
    # optional, defaults to m5.large
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
    vpc=vpc
)
rds.DatabaseInstanceReadReplica(self, "ReadReplica",
    source_database_instance=source_instance,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
    vpc=vpc
)
```

Automatic backups of read replica instances are only supported for MySQL and MariaDB. By default,
automatic backups are disabled for read replicas and can only be enabled (using `backupRetention`)
if also enabled on the source instance.

Creating a "production" Oracle database instance with option and parameter groups:

```python
# Set open cursors with parameter group
parameter_group = rds.ParameterGroup(self, "ParameterGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    parameters={
        "open_cursors": "2500"
    }
)

option_group = rds.OptionGroup(self, "OptionGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    configurations=[cdk.aws_rds.OptionConfiguration(
        name="LOCATOR"
    ), cdk.aws_rds.OptionConfiguration(
        name="OEM",
        port=1158,
        vpc=vpc
    )
    ]
)

# Allow connections to OEM
option_group.option_connections.OEM.connections.allow_default_port_from_any_ipv4()

# Database instance with production values
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    license_model=rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
    multi_az=True,
    storage_type=rds.StorageType.IO1,
    credentials=rds.Credentials.from_username("syscdk"),
    vpc=vpc,
    database_name="ORCL",
    storage_encrypted=True,
    backup_retention=cdk.Duration.days(7),
    monitoring_interval=cdk.Duration.seconds(60),
    enable_performance_insights=True,
    cloudwatch_logs_exports=["trace", "audit", "alert", "listener"
    ],
    cloudwatch_logs_retention=logs.RetentionDays.ONE_MONTH,
    auto_minor_version_upgrade=True,  # required to be true if LOCATOR is used in the option group
    option_group=option_group,
    parameter_group=parameter_group,
    removal_policy=RemovalPolicy.DESTROY
)

# Allow connections on default port from any IPV4
instance.connections.allow_default_port_from_any_ipv4()

# Rotate the master user password every 30 days
instance.add_rotation_single_user()

# Add alarm for high CPU
cloudwatch.Alarm(self, "HighCPU",
    metric=instance.metric_cPUUtilization(),
    threshold=90,
    evaluation_periods=1
)

# Trigger Lambda function on instance availability events
fn = lambda_.Function(self, "Function",
    code=lambda_.Code.from_inline("exports.handler = (event) => console.log(event);"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_14_X
)

availability_rule = instance.on_event("Availability", target=targets.LambdaFunction(fn))
availability_rule.add_event_pattern(
    detail={
        "EventCategories": ["availability"
        ]
    }
)
```

Add XMLDB and OEM with option group

```python
# Set open cursors with parameter group
parameter_group = rds.ParameterGroup(self, "ParameterGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    parameters={
        "open_cursors": "2500"
    }
)

option_group = rds.OptionGroup(self, "OptionGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    configurations=[cdk.aws_rds.OptionConfiguration(
        name="LOCATOR"
    ), cdk.aws_rds.OptionConfiguration(
        name="OEM",
        port=1158,
        vpc=vpc
    )
    ]
)

# Allow connections to OEM
option_group.option_connections.OEM.connections.allow_default_port_from_any_ipv4()

# Database instance with production values
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    license_model=rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
    multi_az=True,
    storage_type=rds.StorageType.IO1,
    credentials=rds.Credentials.from_username("syscdk"),
    vpc=vpc,
    database_name="ORCL",
    storage_encrypted=True,
    backup_retention=cdk.Duration.days(7),
    monitoring_interval=cdk.Duration.seconds(60),
    enable_performance_insights=True,
    cloudwatch_logs_exports=["trace", "audit", "alert", "listener"
    ],
    cloudwatch_logs_retention=logs.RetentionDays.ONE_MONTH,
    auto_minor_version_upgrade=True,  # required to be true if LOCATOR is used in the option group
    option_group=option_group,
    parameter_group=parameter_group,
    removal_policy=RemovalPolicy.DESTROY
)

# Allow connections on default port from any IPV4
instance.connections.allow_default_port_from_any_ipv4()

# Rotate the master user password every 30 days
instance.add_rotation_single_user()

# Add alarm for high CPU
cloudwatch.Alarm(self, "HighCPU",
    metric=instance.metric_cPUUtilization(),
    threshold=90,
    evaluation_periods=1
)

# Trigger Lambda function on instance availability events
fn = lambda_.Function(self, "Function",
    code=lambda_.Code.from_inline("exports.handler = (event) => console.log(event);"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_14_X
)

availability_rule = instance.on_event("Availability", target=targets.LambdaFunction(fn))
availability_rule.add_event_pattern(
    detail={
        "EventCategories": ["availability"
        ]
    }
)
```

Use the `storageType` property to specify the [type of storage](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html)
to use for the instance:

```python
# vpc: ec2.Vpc


iops_instance = rds.DatabaseInstance(self, "IopsInstance",
    engine=rds.DatabaseInstanceEngine.mysql(version=rds.MysqlEngineVersion.VER_8_0_30),
    vpc=vpc,
    storage_type=rds.StorageType.IO1,
    iops=5000
)

gp3_instance = rds.DatabaseInstance(self, "Gp3Instance",
    engine=rds.DatabaseInstanceEngine.mysql(version=rds.MysqlEngineVersion.VER_8_0_30),
    vpc=vpc,
    allocated_storage=500,
    storage_type=rds.StorageType.GP3,
    storage_throughput=500
)
```

## Setting Public Accessibility

You can set public accessibility for the database instance or cluster using the `publiclyAccessible` property.
If you specify `true`, it creates an instance with a publicly resolvable DNS name, which resolves to a public IP address.
If you specify `false`, it creates an internal instance with a DNS name that resolves to a private IP address.
The default value depends on `vpcSubnets`.
It will be `true` if `vpcSubnets` is `subnetType: SubnetType.PUBLIC`, `false` otherwise.

```python
# vpc: ec2.Vpc

# Setting public accessibility for DB instance
rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.mysql(
        version=rds.MysqlEngineVersion.VER_8_0_19
    ),
    vpc=vpc,
    vpc_subnets=ec2.SubnetSelection(
        subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS
    ),
    publicly_accessible=True
)

# Setting public accessibility for DB cluster
rds.DatabaseCluster(self, "DatabaseCluster",
    engine=rds.DatabaseClusterEngine.aurora_mysql(
        version=rds.AuroraMysqlEngineVersion.VER_3_03_0
    ),
    instance_props=rds.InstanceProps(
        vpc=vpc,
        vpc_subnets=ec2.SubnetSelection(
            subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS
        ),
        publicly_accessible=True
    )
)
```

## Instance events

To define Amazon CloudWatch event rules for database instances, use the `onEvent`
method:

```python
# instance: rds.DatabaseInstance
# fn: lambda.Function

rule = instance.on_event("InstanceEvent", target=targets.LambdaFunction(fn))
```

## Login credentials

By default, database instances and clusters (with the exception of `DatabaseInstanceFromSnapshot` and `ServerlessClusterFromSnapshot`) will have `admin` user with an auto-generated password.
An alternative username (and password) may be specified for the admin user instead of the default.

The following examples use a `DatabaseInstance`, but the same usage is applicable to `DatabaseCluster`.

```python
# vpc: ec2.Vpc

engine = rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_15_2)
rds.DatabaseInstance(self, "InstanceWithUsername",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_generated_secret("postgres")
)

rds.DatabaseInstance(self, "InstanceWithUsernameAndPassword",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_password("postgres", SecretValue.ssm_secure("/dbPassword", "1"))
)

my_secret = secretsmanager.Secret.from_secret_name(self, "DBSecret", "myDBLoginInfo")
rds.DatabaseInstance(self, "InstanceWithSecretLogin",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_secret(my_secret)
)
```

Secrets generated by `fromGeneratedSecret()` can be customized:

```python
# vpc: ec2.Vpc

engine = rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_15_2)
my_key = kms.Key(self, "MyKey")

rds.DatabaseInstance(self, "InstanceWithCustomizedSecret",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_generated_secret("postgres",
        secret_name="my-cool-name",
        encryption_key=my_key,
        exclude_characters="!&*^#@()",
        replica_regions=[secretsmanager.ReplicaRegion(region="eu-west-1"), secretsmanager.ReplicaRegion(region="eu-west-2")]
    )
)
```

### Snapshot credentials

As noted above, Databases created with `DatabaseInstanceFromSnapshot` or `ServerlessClusterFromSnapshot` will not create user and auto-generated password by default because it's not possible to change the master username for a snapshot. Instead, they will use the existing username and password from the snapshot. You can still generate a new password - to generate a secret similarly to the other constructs, pass in credentials with `fromGeneratedSecret()` or `fromGeneratedPassword()`.

```python
# vpc: ec2.Vpc

engine = rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_15_2)
my_key = kms.Key(self, "MyKey")

rds.DatabaseInstanceFromSnapshot(self, "InstanceFromSnapshotWithCustomizedSecret",
    engine=engine,
    vpc=vpc,
    snapshot_identifier="mySnapshot",
    credentials=rds.SnapshotCredentials.from_generated_secret("username",
        encryption_key=my_key,
        exclude_characters="!&*^#@()",
        replica_regions=[secretsmanager.ReplicaRegion(region="eu-west-1"), secretsmanager.ReplicaRegion(region="eu-west-2")]
    )
)
```

## Connecting

To control who can access the cluster or instance, use the `.connections` attribute. RDS databases have
a default port, so you don't need to specify the port:

```python
# cluster: rds.DatabaseCluster

cluster.connections.allow_from_any_ipv4(ec2.Port.all_traffic(), "Open to the world")
```

The endpoints to access your database cluster will be available as the `.clusterEndpoint` and `.readerEndpoint`
attributes:

```python
# cluster: rds.DatabaseCluster

write_address = cluster.cluster_endpoint.socket_address
```

For an instance database:

```python
# instance: rds.DatabaseInstance

address = instance.instance_endpoint.socket_address
```

## Rotating credentials

When the master password is generated and stored in AWS Secrets Manager, it can be rotated automatically:

```python
import aws_cdk as cdk

# instance: rds.DatabaseInstance
# my_security_group: ec2.SecurityGroup

instance.add_rotation_single_user(
    automatically_after=Duration.days(7),  # defaults to 30 days
    exclude_characters="!@#$%^&*",  # defaults to the set " %+~`#/// here*()|[]{}:;<>?!'/@\"\\"
    security_group=my_security_group
)
```

```python
cluster = rds.DatabaseCluster(stack, "Database",
    engine=rds.DatabaseClusterEngine.AURORA,
    instance_props=cdk.aws_rds.InstanceProps(
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL),
        vpc=vpc
    )
)

cluster.add_rotation_single_user()

cluster_with_custom_rotation_options = rds.DatabaseCluster(stack, "CustomRotationOptions",
    engine=rds.DatabaseClusterEngine.AURORA,
    instance_props=cdk.aws_rds.InstanceProps(
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL),
        vpc=vpc
    )
)
cluster_with_custom_rotation_options.add_rotation_single_user(
    automatically_after=cdk.Duration.days(7),
    exclude_characters="!@#$%^&*",
    security_group=security_group,
    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS),
    endpoint=endpoint
)
```

The multi user rotation scheme is also available:

```python
# instance: rds.DatabaseInstance
# my_imported_secret: rds.DatabaseSecret

instance.add_rotation_multi_user("MyUser",
    secret=my_imported_secret
)
```

It's also possible to create user credentials together with the instance/cluster and add rotation:

```python
# instance: rds.DatabaseInstance

my_user_secret = rds.DatabaseSecret(self, "MyUserSecret",
    username="myuser",
    secret_name="my-user-secret",  # optional, defaults to a CloudFormation-generated name
    dbname="mydb",  # optional, defaults to the main database of the RDS cluster this secret gets attached to
    master_secret=instance.secret,
    exclude_characters="{}[]()'\"/\\"
)
my_user_secret_attached = my_user_secret.attach(instance) # Adds DB connections information in the secret

instance.add_rotation_multi_user("MyUser",  # Add rotation using the multi user scheme
    secret=my_user_secret_attached)
```

**Note**: This user must be created manually in the database using the master credentials.
The rotation will start as soon as this user exists.

Access to the Secrets Manager API is required for the secret rotation. This can be achieved either with
internet connectivity (through NAT) or with a VPC interface endpoint. By default, the rotation Lambda function
is deployed in the same subnets as the instance/cluster. If access to the Secrets Manager API is not possible from
those subnets or using the default API endpoint, use the `vpcSubnets` and/or `endpoint` options:

```python
# instance: rds.DatabaseInstance
# my_endpoint: ec2.InterfaceVpcEndpoint


instance.add_rotation_single_user(
    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS),  # Place rotation Lambda in private subnets
    endpoint=my_endpoint
)
```

See also [@aws-cdk/aws-secretsmanager](https://github.com/aws/aws-cdk/blob/main/packages/%40aws-cdk/aws-secretsmanager/README.md) for credentials rotation of existing clusters/instances.

## IAM Authentication

You can also authenticate to a database instance using AWS Identity and Access Management (IAM) database authentication;
See [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html) for more information
and a list of supported versions and limitations.

**Note**: `grantConnect()` does not currently work - see [this GitHub issue](https://github.com/aws/aws-cdk/issues/11851).

The following example shows enabling IAM authentication for a database instance and granting connection access to an IAM role.

```python
# vpc: ec2.Vpc

instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.mysql(version=rds.MysqlEngineVersion.VER_8_0_19),
    vpc=vpc,
    iam_authentication=True
)
role = iam.Role(self, "DBRole", assumed_by=iam.AccountPrincipal(self.account))
instance.grant_connect(role)
```

The following example shows granting connection access for RDS Proxy to an IAM role.

```python
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(
        version=rds.AuroraMysqlEngineVersion.VER_3_03_0
    ),
    writer=rds.ClusterInstance.provisioned("writer"),
    vpc=vpc
)

proxy = rds.DatabaseProxy(self, "Proxy",
    proxy_target=rds.ProxyTarget.from_cluster(cluster),
    secrets=[cluster.secret],
    vpc=vpc
)

role = iam.Role(self, "DBProxyRole", assumed_by=iam.AccountPrincipal(self.account))
proxy.grant_connect(role, "admin")
```

**Note**: In addition to the setup above, a database user will need to be created to support IAM auth.
See [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html) for setup instructions.

## Kerberos Authentication

You can also authenticate using Kerberos to a database instance using AWS Managed Microsoft AD for authentication;
See [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html) for more information
and a list of supported versions and limitations.

The following example shows enabling domain support for a database instance and creating an IAM role to access
Directory Services.

```python
# vpc: ec2.Vpc

role = iam.Role(self, "RDSDirectoryServicesRole",
    assumed_by=iam.ServicePrincipal("rds.amazonaws.com"),
    managed_policies=[
        iam.ManagedPolicy.from_aws_managed_policy_name("service-role/AmazonRDSDirectoryServiceAccess")
    ]
)
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.mysql(version=rds.MysqlEngineVersion.VER_8_0_19),
    vpc=vpc,
    domain="d-????????",  # The ID of the domain for the instance to join.
    domain_role=role
)
```

**Note**: In addition to the setup above, you need to make sure that the database instance has network connectivity
to the domain controllers. This includes enabling cross-VPC traffic if in a different VPC and setting up the
appropriate security groups/network ACL to allow traffic between the database instance and domain controllers.
Once configured, see [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html) for details
on configuring users for each available database engine.

## Metrics

Database instances and clusters both expose metrics (`cloudwatch.Metric`):

```python
# The number of database connections in use (average over 5 minutes)
# instance: rds.DatabaseInstance

# Average CPU utilization over 5 minutes
# cluster: rds.DatabaseCluster

db_connections = instance.metric_database_connections()
cpu_utilization = cluster.metric_cPUUtilization()

# The average amount of time taken per disk I/O operation (average over 1 minute)
read_latency = instance.metric("ReadLatency", statistic="Average", period=Duration.seconds(60))
```

## Enabling S3 integration

Data in S3 buckets can be imported to and exported from certain database engines using SQL queries. To enable this
functionality, set the `s3ImportBuckets` and `s3ExportBuckets` properties for import and export respectively. When
configured, the CDK automatically creates and configures IAM roles as required.
Additionally, the `s3ImportRole` and `s3ExportRole` properties can be used to set this role directly.

You can read more about loading data to (or from) S3 here:

* Aurora MySQL - [import](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.LoadFromS3.html)
  and [export](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.SaveIntoS3.html).
* Aurora PostgreSQL - [import](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html#USER_PostgreSQL.S3Import)
  and [export](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html).
* Microsoft SQL Server - [import and export](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/SQLServer.Procedural.Importing.html)
* PostgreSQL - [import](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html)
  and [export](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/postgresql-s3-export.html)
* Oracle - [import and export](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html)

The following snippet sets up a database cluster with different S3 buckets where the data is imported and exported -

```python
import aws_cdk.aws_s3 as s3

# vpc: ec2.Vpc

import_bucket = s3.Bucket(self, "importbucket")
export_bucket = s3.Bucket(self, "exportbucket")
rds.DatabaseCluster(self, "dbcluster",
    engine=rds.DatabaseClusterEngine.aurora_mysql(
        version=rds.AuroraMysqlEngineVersion.VER_3_03_0
    ),
    writer=rds.ClusterInstance.provisioned("writer"),
    vpc=vpc,
    s3_import_buckets=[import_bucket],
    s3_export_buckets=[export_bucket]
)
```

## Creating a Database Proxy

Amazon RDS Proxy sits between your application and your relational database to efficiently manage
connections to the database and improve scalability of the application. Learn more about at [Amazon RDS Proxy](https://aws.amazon.com/rds/proxy/)

The following code configures an RDS Proxy for a `DatabaseInstance`.

```python
# vpc: ec2.Vpc
# security_group: ec2.SecurityGroup
# secrets: List[secretsmanager.Secret[]]
# db_instance: rds.DatabaseInstance


proxy = db_instance.add_proxy("proxy",
    borrow_timeout=Duration.seconds(30),
    max_connections_percent=50,
    secrets=secrets,
    vpc=vpc
)
```

## Exporting Logs

You can publish database logs to Amazon CloudWatch Logs. With CloudWatch Logs, you can perform real-time analysis of the log data,
store the data in highly durable storage, and manage the data with the CloudWatch Logs Agent. This is available for both database
instances and clusters; the types of logs available depend on the database type and engine being used.

```python
import aws_cdk.aws_logs as logs
# my_logs_publishing_role: iam.Role
# vpc: ec2.Vpc


# Exporting logs from a cluster
cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora(
        version=rds.AuroraEngineVersion.VER_1_17_9
    ),
    writer=rds.ClusterInstance.provisioned("writer"),
    vpc=vpc,
    cloudwatch_logs_exports=["error", "general", "slowquery", "audit"],  # Export all available MySQL-based logs
    cloudwatch_logs_retention=logs.RetentionDays.THREE_MONTHS,  # Optional - default is to never expire logs
    cloudwatch_logs_retention_role=my_logs_publishing_role
)

# Exporting logs from an instance
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.postgres(
        version=rds.PostgresEngineVersion.VER_15_2
    ),
    vpc=vpc,
    cloudwatch_logs_exports=["postgresql"]
)
```

## Option Groups

Some DB engines offer additional features that make it easier to manage data and databases, and to provide additional security for your database.
Amazon RDS uses option groups to enable and configure these features. An option group can specify features, called options,
that are available for a particular Amazon RDS DB instance.

```python
# vpc: ec2.Vpc
# security_group: ec2.SecurityGroup


rds.OptionGroup(self, "Options",
    engine=rds.DatabaseInstanceEngine.oracle_se2(
        version=rds.OracleEngineVersion.VER_19
    ),
    configurations=[rds.OptionConfiguration(
        name="OEM",
        port=5500,
        vpc=vpc,
        security_groups=[security_group]
    )
    ]
)
```

## Parameter Groups

Database parameters specify how the database is configured.
For example, database parameters can specify the amount of resources, such as memory, to allocate to a database.
You manage your database configuration by associating your DB instances with parameter groups.
Amazon RDS defines parameter groups with default settings.

You can create your own parameter group for your cluster or instance and associate it with your database:

```python
# vpc: ec2.Vpc


parameter_group = rds.ParameterGroup(self, "ParameterGroup",
    engine=rds.DatabaseInstanceEngine.sql_server_ee(
        version=rds.SqlServerEngineVersion.VER_11
    ),
    parameters={
        "locks": "100"
    }
)

rds.DatabaseInstance(self, "Database",
    engine=rds.DatabaseInstanceEngine.SQL_SERVER_EE,
    vpc=vpc,
    parameter_group=parameter_group
)
```

Another way to specify parameters is to use the inline field `parameters` that creates an RDS parameter group for you.
You can use this if you do not want to reuse the parameter group instance for different instances:

```python
# vpc: ec2.Vpc


rds.DatabaseInstance(self, "Database",
    engine=rds.DatabaseInstanceEngine.sql_server_ee(version=rds.SqlServerEngineVersion.VER_11),
    vpc=vpc,
    parameters={
        "locks": "100"
    }
)
```

You cannot specify a parameter map and a parameter group at the same time.

## Serverless

[Amazon Aurora Serverless](https://aws.amazon.com/rds/aurora/serverless/) is an on-demand, auto-scaling configuration for Amazon
Aurora. The database will automatically start up, shut down, and scale capacity
up or down based on your application's needs. It enables you to run your database
in the cloud without managing any database instances.

The following example initializes an Aurora Serverless PostgreSql cluster.
Aurora Serverless clusters can specify scaling properties which will be used to
automatically scale the database cluster seamlessly based on the workload.

```python
# vpc: ec2.Vpc


cluster = rds.ServerlessCluster(self, "AnotherCluster",
    engine=rds.DatabaseClusterEngine.AURORA_POSTGRESQL,
    copy_tags_to_snapshot=True,  # whether to save the cluster tags when creating the snapshot. Default is 'true'
    parameter_group=rds.ParameterGroup.from_parameter_group_name(self, "ParameterGroup", "default.aurora-postgresql10"),
    vpc=vpc,
    scaling=rds.ServerlessScalingOptions(
        auto_pause=Duration.minutes(10),  # default is to pause after 5 minutes of idle time
        min_capacity=rds.AuroraCapacityUnit.ACU_8,  # default is 2 Aurora capacity units (ACUs)
        max_capacity=rds.AuroraCapacityUnit.ACU_32
    )
)
```

Aurora Serverless Clusters do not support the following features:

* Loading data from an Amazon S3 bucket
* Saving data to an Amazon S3 bucket
* Invoking an AWS Lambda function with an Aurora MySQL native function
* Aurora replicas
* Backtracking
* Multi-master clusters
* Database cloning
* IAM database cloning
* IAM database authentication
* Restoring a snapshot from MySQL DB instance
* Performance Insights
* RDS Proxy

Read more about the [limitations of Aurora Serverless](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations)

Learn more about using Amazon Aurora Serverless by reading the [documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html)

Use `ServerlessClusterFromSnapshot` to create a serverless cluster from a snapshot:

```python
# vpc: ec2.Vpc

rds.ServerlessClusterFromSnapshot(self, "Cluster",
    engine=rds.DatabaseClusterEngine.AURORA_MYSQL,
    vpc=vpc,
    snapshot_identifier="mySnapshot"
)
```

### Data API

You can access your Aurora Serverless DB cluster using the built-in Data API. The Data API doesn't require a persistent connection to the DB cluster. Instead, it provides a secure HTTP endpoint and integration with AWS SDKs.

The following example shows granting Data API access to a Lamba function.

```python
# vpc: ec2.Vpc

# code: lambda.Code


cluster = rds.ServerlessCluster(self, "AnotherCluster",
    engine=rds.DatabaseClusterEngine.AURORA_MYSQL,
    vpc=vpc,  # this parameter is optional for serverless Clusters
    enable_data_api=True
)
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_14_X,
    handler="index.handler",
    code=code,
    environment={
        "CLUSTER_ARN": cluster.cluster_arn,
        "SECRET_ARN": cluster.secret.secret_arn
    }
)
cluster.grant_data_api_access(fn)
```

**Note**: To invoke the Data API, the resource will need to read the secret associated with the cluster.

To learn more about using the Data API, see the [documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html).

### Default VPC

The `vpc` parameter is optional.

If not provided, the cluster will be created in the default VPC of the account and region.
As this VPC is not deployed with AWS CDK, you can't configure the `vpcSubnets`, `subnetGroup` or `securityGroups` of the Aurora Serverless Cluster.
If you want to provide one of `vpcSubnets`, `subnetGroup` or `securityGroups` parameter, please provide a `vpc`.
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    CfnResource as _CfnResource_9df397a6,
    CfnTag as _CfnTag_f6864754,
    Duration as _Duration_4839e8c3,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    RemovalPolicy as _RemovalPolicy_9f93c814,
    Resource as _Resource_45bc6135,
    ResourceProps as _ResourceProps_15a65b4e,
    SecretValue as _SecretValue_3dd0ddae,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_cloudwatch import (
    Metric as _Metric_e396a4dc,
    MetricOptions as _MetricOptions_1788b62f,
    Unit as _Unit_61bc6f70,
)
from ..aws_ec2 import (
    Connections as _Connections_0f31fce8,
    IConnectable as _IConnectable_10015a05,
    IInterfaceVpcEndpoint as _IInterfaceVpcEndpoint_7481aea1,
    ISecurityGroup as _ISecurityGroup_acf8a799,
    IVpc as _IVpc_f30d5663,
    InstanceType as _InstanceType_f64915b9,
    SubnetSelection as _SubnetSelection_e57d76df,
)
from ..aws_events import (
    EventPattern as _EventPattern_fe557901,
    IRuleTarget as _IRuleTarget_7a91f454,
    OnEventOptions as _OnEventOptions_8711b8b3,
    Rule as _Rule_334ed2b5,
)
from ..aws_iam import (
    Grant as _Grant_a7ae64f8,
    IGrantable as _IGrantable_71c4f5de,
    IRole as _IRole_235f5d8e,
)
from ..aws_kms import IKey as _IKey_5f11635f
from ..aws_logs import RetentionDays as _RetentionDays_070f99f0
from ..aws_s3 import IBucket as _IBucket_42e086fd
from ..aws_secretsmanager import (
    ISecret as _ISecret_6e020e6a,
    ISecretAttachmentTarget as _ISecretAttachmentTarget_123e2df9,
    ReplicaRegion as _ReplicaRegion_bccfea79,
    Secret as _Secret_50778576,
    SecretAttachmentTargetProps as _SecretAttachmentTargetProps_9ec7949d,
    SecretRotation as _SecretRotation_38c354d9,
    SecretRotationApplication as _SecretRotationApplication_bfbcd452,
)


@jsii.enum(jsii_type="aws-cdk-lib.aws_rds.AuroraCapacityUnit")
class AuroraCapacityUnit(enum.Enum):
    '''Aurora capacity units (ACUs).

    Each ACU is a combination of processing and memory capacity.

    :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.architecture
    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        
        
        cluster = rds.ServerlessCluster(self, "AnotherCluster",
            engine=rds.DatabaseClusterEngine.AURORA_POSTGRESQL,
            copy_tags_to_snapshot=True,  # whether to save the cluster tags when creating the snapshot. Default is 'true'
            parameter_group=rds.ParameterGroup.from_parameter_group_name(self, "ParameterGroup", "default.aurora-postgresql10"),
            vpc=vpc,
            scaling=rds.ServerlessScalingOptions(
                auto_pause=Duration.minutes(10),  # default is to pause after 5 minutes of idle time
                min_capacity=rds.AuroraCapacityUnit.ACU_8,  # default is 2 Aurora capacity units (ACUs)
                max_capacity=rds.AuroraCapacityUnit.ACU_32
            )
        )
    '''

    ACU_1 = "ACU_1"
    '''1 Aurora Capacity Unit.'''
    ACU_2 = "ACU_2"
    '''2 Aurora Capacity Units.'''
    ACU_4 = "ACU_4"
    '''4 Aurora Capacity Units.'''
    ACU_8 = "ACU_8"
    '''8 Aurora Capacity Units.'''
    ACU_16 = "ACU_16"
    '''16 Aurora Capacity Units.'''
    ACU_32 = "ACU_32"
    '''32 Aurora Capacity Units.'''
    ACU_64 = "ACU_64"
    '''64 Aurora Capacity Units.'''
    ACU_128 = "ACU_128"
    '''128 Aurora Capacity Units.'''
    ACU_192 = "ACU_192"
    '''192 Aurora Capacity Units.'''
    ACU_256 = "ACU_256"
    '''256 Aurora Capacity Units.'''
    ACU_384 = "ACU_384"
    '''384 Aurora Capacity Units.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.AuroraClusterEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class AuroraClusterEngineProps:
    def __init__(self, *, version: "AuroraEngineVersion") -> None:
        '''Creation properties of the plain Aurora database cluster engine.

        Used in ``DatabaseClusterEngine.aurora``.

        :param version: The version of the Aurora cluster engine.

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            
            rds.DatabaseClusterFromSnapshot(self, "Database",
                engine=rds.DatabaseClusterEngine.aurora(version=rds.AuroraEngineVersion.VER_1_22_2),
                writer=rds.ClusterInstance.provisioned("writer"),
                vpc=vpc,
                snapshot_identifier="mySnapshot"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__50b0faf7b0e7a058936423fc1e2256df92f1017249cb37013902487759075d06)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "AuroraEngineVersion":
        '''The version of the Aurora cluster engine.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast("AuroraEngineVersion", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraClusterEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuroraEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.AuroraEngineVersion",
):
    '''The versions for the Aurora cluster engine (those returned by ``DatabaseClusterEngine.aurora``).

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        
        rds.DatabaseClusterFromSnapshot(self, "Database",
            engine=rds.DatabaseClusterEngine.aurora(version=rds.AuroraEngineVersion.VER_1_22_2),
            writer=rds.ClusterInstance.provisioned("writer"),
            vpc=vpc,
            snapshot_identifier="mySnapshot"
        )
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        aurora_full_version: builtins.str,
        aurora_major_version: typing.Optional[builtins.str] = None,
    ) -> "AuroraEngineVersion":
        '''Create a new AuroraEngineVersion with an arbitrary version.

        :param aurora_full_version: the full version string, for example "5.6.mysql_aurora.1.78.3.6".
        :param aurora_major_version: the major version of the engine, defaults to "5.6".
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df17b87254cb31dd21f497140057c7bf82f0bc2436472cda0fb6ff7a099ecbde)
            check_type(argname="argument aurora_full_version", value=aurora_full_version, expected_type=type_hints["aurora_full_version"])
            check_type(argname="argument aurora_major_version", value=aurora_major_version, expected_type=type_hints["aurora_major_version"])
        return typing.cast("AuroraEngineVersion", jsii.sinvoke(cls, "of", [aurora_full_version, aurora_major_version]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_17_9")
    def VER_1_17_9(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.17.9".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_17_9"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_0")
    def VER_1_19_0(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.0".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_1")
    def VER_1_19_1(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.1".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_2")
    def VER_1_19_2(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.2".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_5")
    def VER_1_19_5(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.5".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_6")
    def VER_1_19_6(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.6".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_20_0")
    def VER_1_20_0(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.20.0".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_20_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_20_1")
    def VER_1_20_1(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.20.1".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_20_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_21_0")
    def VER_1_21_0(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.21.0".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_21_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_0")
    def VER_1_22_0(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.0".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_1")
    def VER_1_22_1(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.1".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_1_3")
    def VER_1_22_1_3(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.1.3".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_2")
    def VER_1_22_2(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.2".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_3")
    def VER_1_22_3(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.3".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_4")
    def VER_1_22_4(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.4".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_5")
    def VER_1_22_5(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.5".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_23_0")
    def VER_1_23_0(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.23.0".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_23_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_23_1")
    def VER_1_23_1(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.23.1".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_23_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_23_2")
    def VER_1_23_2(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.23.2".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_23_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_23_3")
    def VER_1_23_3(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.23.3".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_23_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_23_4")
    def VER_1_23_4(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.23.4".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_23_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10A")
    def VER_10_A(cls) -> "AuroraEngineVersion":
        '''Version "5.6.10a".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_10A"))

    @builtins.property
    @jsii.member(jsii_name="auroraFullVersion")
    def aurora_full_version(self) -> builtins.str:
        '''The full version string, for example, "5.6.mysql_aurora.1.78.3.6".'''
        return typing.cast(builtins.str, jsii.get(self, "auroraFullVersion"))

    @builtins.property
    @jsii.member(jsii_name="auroraMajorVersion")
    def aurora_major_version(self) -> builtins.str:
        '''The major version of the engine.

        Currently, it's always "5.6".
        '''
        return typing.cast(builtins.str, jsii.get(self, "auroraMajorVersion"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.AuroraMysqlClusterEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class AuroraMysqlClusterEngineProps:
    def __init__(self, *, version: "AuroraMysqlEngineVersion") -> None:
        '''Creation properties of the Aurora MySQL database cluster engine.

        Used in ``DatabaseClusterEngine.auroraMysql``.

        :param version: The version of the Aurora MySQL cluster engine.

        :exampleMetadata: infused

        Example::

            # Example automatically generated from non-compiling source. May contain errors.
            # vpc: ec2.Vpc
            
            cluster = rds.DatabaseCluster(self, "Database",
                engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_2_08_1),
                writer=rds.ClusterInstance.provisioned("writer",
                    instance_type=ec2.InstanceType.of(ec2.InstanceClass.R6G, ec2.InstanceSize.XLARGE4)
                ),
                serverless_v2_min_capacity=6.5,
                serverless_v2_max_capacity=64,
                readers=[
                    # will be put in promotion tier 1 and will scale with the writer
                    rds.ClusterInstance.serverless_v2("reader1", scale_with_writer=True),
                    # will be put in promotion tier 2 and will not scale with the writer
                    rds.ClusterInstance.serverless_v2("reader2")
                ],
                vpc=vpc
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f31d66e13f7e025b9cde216e94ec567ffde2b01479e83ccb6adc53afc0afa2ba)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "AuroraMysqlEngineVersion":
        '''The version of the Aurora MySQL cluster engine.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast("AuroraMysqlEngineVersion", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraMysqlClusterEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuroraMysqlEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.AuroraMysqlEngineVersion",
):
    '''The versions for the Aurora MySQL cluster engine (those returned by ``DatabaseClusterEngine.auroraMysql``).

    https://docs.aws.amazon.com/AmazonRDS/latest/AuroraMySQLReleaseNotes/Welcome.html

    :exampleMetadata: infused

    Example::

        # Example automatically generated from non-compiling source. May contain errors.
        # vpc: ec2.Vpc
        
        cluster = rds.DatabaseCluster(self, "Database",
            engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_2_08_1),
            credentials=rds.Credentials.from_generated_secret("clusteradmin"),  # Optional - will default to 'admin' username and generated password
            writer=rds.ClusterInstance.provisioned("writer",
                readers=[
                    rds.ClusterInstance.provisioned("reader1", promotion_tier=1),
                    rds.ClusterInstance.serverless_v2("reader2")
                ],
                vpc_subnets={
                    "subnet_type": ec2.SubnetType.PRIVATE_WITH_EGRESS
                },
                vpc=vpc
            )
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        aurora_mysql_full_version: builtins.str,
        aurora_mysql_major_version: typing.Optional[builtins.str] = None,
    ) -> "AuroraMysqlEngineVersion":
        '''Create a new AuroraMysqlEngineVersion with an arbitrary version.

        :param aurora_mysql_full_version: the full version string, for example "5.7.mysql_aurora.2.78.3.6".
        :param aurora_mysql_major_version: the major version of the engine, defaults to "5.7".
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4eb357ecf1e7185cb40f90b43951c381cee0f14f6e88296f265edcd155828370)
            check_type(argname="argument aurora_mysql_full_version", value=aurora_mysql_full_version, expected_type=type_hints["aurora_mysql_full_version"])
            check_type(argname="argument aurora_mysql_major_version", value=aurora_mysql_major_version, expected_type=type_hints["aurora_mysql_major_version"])
        return typing.cast("AuroraMysqlEngineVersion", jsii.sinvoke(cls, "of", [aurora_mysql_full_version, aurora_mysql_major_version]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_03_2")
    def VER_2_03_2(cls) -> "AuroraMysqlEngineVersion":
        '''(deprecated) Version "5.7.mysql_aurora.2.03.2".

        :deprecated: Version 5.7.mysql_aurora.2.03.2 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_03_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_03_3")
    def VER_2_03_3(cls) -> "AuroraMysqlEngineVersion":
        '''(deprecated) Version "5.7.mysql_aurora.2.03.3".

        :deprecated: Version 5.7.mysql_aurora.2.03.3 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_03_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_03_4")
    def VER_2_03_4(cls) -> "AuroraMysqlEngineVersion":
        '''(deprecated) Version "5.7.mysql_aurora.2.03.4".

        :deprecated: Version 5.7.mysql_aurora.2.03.4 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_03_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_0")
    def VER_2_04_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_1")
    def VER_2_04_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_2")
    def VER_2_04_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_3")
    def VER_2_04_3(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.3".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_4")
    def VER_2_04_4(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.4".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_5")
    def VER_2_04_5(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.5".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_6")
    def VER_2_04_6(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.6".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_7")
    def VER_2_04_7(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.7".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_8")
    def VER_2_04_8(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.8".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_05_0")
    def VER_2_05_0(cls) -> "AuroraMysqlEngineVersion":
        '''(deprecated) Version "5.7.mysql_aurora.2.05.0".

        :deprecated: Version 5.7.mysql_aurora.2.05.0 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_05_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_06_0")
    def VER_2_06_0(cls) -> "AuroraMysqlEngineVersion":
        '''(deprecated) Version "5.7.mysql_aurora.2.06.0".

        :deprecated: Version 5.7.mysql_aurora.2.06.0 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_06_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_0")
    def VER_2_07_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_1")
    def VER_2_07_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_2")
    def VER_2_07_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_3")
    def VER_2_07_3(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.3".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_4")
    def VER_2_07_4(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.4".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_5")
    def VER_2_07_5(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.5".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_6")
    def VER_2_07_6(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.6".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_7")
    def VER_2_07_7(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.7".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_8")
    def VER_2_07_8(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.8".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_08_0")
    def VER_2_08_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.08.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_08_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_08_1")
    def VER_2_08_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.08.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_08_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_08_2")
    def VER_2_08_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.08.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_08_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_08_3")
    def VER_2_08_3(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.08.3".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_08_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_08_4")
    def VER_2_08_4(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.08.4".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_08_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_09_0")
    def VER_2_09_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.09.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_09_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_09_1")
    def VER_2_09_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.09.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_09_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_09_2")
    def VER_2_09_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.09.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_09_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_09_3")
    def VER_2_09_3(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.09.3".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_09_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_10_0")
    def VER_2_10_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.10.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_10_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_10_1")
    def VER_2_10_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.10.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_10_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_10_2")
    def VER_2_10_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.10.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_10_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_10_3")
    def VER_2_10_3(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.10.3".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_10_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_11_0")
    def VER_2_11_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.11.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_11_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_11_1")
    def VER_2_11_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.11.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_11_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_11_2")
    def VER_2_11_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.11.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_11_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_3_01_0")
    def VER_3_01_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "8.0.mysql_aurora.3.01.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_3_01_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_3_01_1")
    def VER_3_01_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "8.0.mysql_aurora.3.01.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_3_01_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_3_02_0")
    def VER_3_02_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "8.0.mysql_aurora.3.02.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_3_02_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_3_02_1")
    def VER_3_02_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "8.0.mysql_aurora.3.02.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_3_02_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_3_02_2")
    def VER_3_02_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "8.0.mysql_aurora.3.02.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_3_02_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_3_03_0")
    def VER_3_03_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "8.0.mysql_aurora.3.03.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_3_03_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_5_7_12")
    def VER_5_7_12(cls) -> "AuroraMysqlEngineVersion":
        '''(deprecated) Version "5.7.12".

        :deprecated: Version 5.7.12 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_5_7_12"))

    @builtins.property
    @jsii.member(jsii_name="auroraMysqlFullVersion")
    def aurora_mysql_full_version(self) -> builtins.str:
        '''The full version string, for example, "5.7.mysql_aurora.1.78.3.6".'''
        return typing.cast(builtins.str, jsii.get(self, "auroraMysqlFullVersion"))

    @builtins.property
    @jsii.member(jsii_name="auroraMysqlMajorVersion")
    def aurora_mysql_major_version(self) -> builtins.str:
        '''The major version of the engine.

        Currently, it's either "5.7", or "8.0".
        '''
        return typing.cast(builtins.str, jsii.get(self, "auroraMysqlMajorVersion"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.AuroraPostgresClusterEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class AuroraPostgresClusterEngineProps:
    def __init__(self, *, version: "AuroraPostgresEngineVersion") -> None:
        '''Creation properties of the Aurora PostgreSQL database cluster engine.

        Used in ``DatabaseClusterEngine.auroraPostgres``.

        :param version: The version of the Aurora PostgreSQL cluster engine.

        :exampleMetadata: infused

        Example::

            # Example automatically generated from non-compiling source. May contain errors.
            # vpc: ec2.Vpc
            
            cluster = rds.DatabaseCluster(self, "Database",
                engine=rds.DatabaseClusterEngine.aurora_postgres(version=rds.AuroraPostgresEngineVersion.VER_15_2),
                credentials=rds.Credentials.from_username("adminuser", password=cdk.SecretValue.unsafe_plain_text("7959866cacc02c2d243ecfe177464fe6")),
                instance_props=rds.InstanceProps(
                    instance_type=ec2.InstanceType.of(ec2.InstanceClass.X2G, ec2.InstanceSize.XLARGE),
                    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC),
                    vpc=vpc
                ),
                storage_type=rds.DBClusterStorageType.AURORA_IOPT1
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7190dd768e4e8ab1f1c599c51dcffc20f53b372ff5f3d2fee7ab27d37cde126d)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "AuroraPostgresEngineVersion":
        '''The version of the Aurora PostgreSQL cluster engine.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast("AuroraPostgresEngineVersion", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraPostgresClusterEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.AuroraPostgresEngineFeatures",
    jsii_struct_bases=[],
    name_mapping={"s3_export": "s3Export", "s3_import": "s3Import"},
)
class AuroraPostgresEngineFeatures:
    def __init__(
        self,
        *,
        s3_export: typing.Optional[builtins.bool] = None,
        s3_import: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Features supported by this version of the Aurora Postgres cluster engine.

        :param s3_export: Whether this version of the Aurora Postgres cluster engine supports the S3 data export feature. Default: false
        :param s3_import: Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature. Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            aurora_postgres_engine_features = rds.AuroraPostgresEngineFeatures(
                s3_export=False,
                s3_import=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b814dd981cca1ca2c5c82455e0e07de3c617842b40c164fd2d5e0c0ceea436fe)
            check_type(argname="argument s3_export", value=s3_export, expected_type=type_hints["s3_export"])
            check_type(argname="argument s3_import", value=s3_import, expected_type=type_hints["s3_import"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if s3_export is not None:
            self._values["s3_export"] = s3_export
        if s3_import is not None:
            self._values["s3_import"] = s3_import

    @builtins.property
    def s3_export(self) -> typing.Optional[builtins.bool]:
        '''Whether this version of the Aurora Postgres cluster engine supports the S3 data export feature.

        :default: false
        '''
        result = self._values.get("s3_export")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def s3_import(self) -> typing.Optional[builtins.bool]:
        '''Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature.

        :default: false
        '''
        result = self._values.get("s3_import")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraPostgresEngineFeatures(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuroraPostgresEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.AuroraPostgresEngineVersion",
):
    '''The versions for the Aurora PostgreSQL cluster engine (those returned by ``DatabaseClusterEngine.auroraPostgres``).

    https://docs.aws.amazon.com/AmazonRDS/latest/AuroraPostgreSQLReleaseNotes/AuroraPostgreSQL.Updates.html

    :exampleMetadata: infused

    Example::

        # Example automatically generated from non-compiling source. May contain errors.
        # vpc: ec2.Vpc
        
        cluster = rds.DatabaseCluster(self, "Database",
            engine=rds.DatabaseClusterEngine.aurora_postgres(version=rds.AuroraPostgresEngineVersion.VER_15_2),
            credentials=rds.Credentials.from_username("adminuser", password=cdk.SecretValue.unsafe_plain_text("7959866cacc02c2d243ecfe177464fe6")),
            instance_props=rds.InstanceProps(
                instance_type=ec2.InstanceType.of(ec2.InstanceClass.X2G, ec2.InstanceSize.XLARGE),
                vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC),
                vpc=vpc
            ),
            storage_type=rds.DBClusterStorageType.AURORA_IOPT1
        )
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        aurora_postgres_full_version: builtins.str,
        aurora_postgres_major_version: builtins.str,
        *,
        s3_export: typing.Optional[builtins.bool] = None,
        s3_import: typing.Optional[builtins.bool] = None,
    ) -> "AuroraPostgresEngineVersion":
        '''Create a new AuroraPostgresEngineVersion with an arbitrary version.

        :param aurora_postgres_full_version: the full version string, for example "9.6.25.1".
        :param aurora_postgres_major_version: the major version of the engine, for example "9.6".
        :param s3_export: Whether this version of the Aurora Postgres cluster engine supports the S3 data export feature. Default: false
        :param s3_import: Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature. Default: false
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__98ade0032a588940a6a5692a85f772441fd63ea4236e4018d1aa1b1ef7177eae)
            check_type(argname="argument aurora_postgres_full_version", value=aurora_postgres_full_version, expected_type=type_hints["aurora_postgres_full_version"])
            check_type(argname="argument aurora_postgres_major_version", value=aurora_postgres_major_version, expected_type=type_hints["aurora_postgres_major_version"])
        aurora_postgres_features = AuroraPostgresEngineFeatures(
            s3_export=s3_export, s3_import=s3_import
        )

        return typing.cast("AuroraPostgresEngineVersion", jsii.sinvoke(cls, "of", [aurora_postgres_full_version, aurora_postgres_major_version, aurora_postgres_features]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_11")
    def VER_10_11(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "10.11".

        :deprecated: Version 10.11 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_11"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_12")
    def VER_10_12(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "10.12".

        :deprecated: Version 10.12 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_12"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_13")
    def VER_10_13(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "10.13".

        :deprecated: Version 10.13 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_13"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_14")
    def VER_10_14(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "10.14".

        :deprecated: Version 10.14 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_14"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_16")
    def VER_10_16(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.16".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_16"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_17")
    def VER_10_17(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.17".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_17"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_18")
    def VER_10_18(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.18".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_18"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_19")
    def VER_10_19(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.19".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_19"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_20")
    def VER_10_20(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.20".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_20"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_21")
    def VER_10_21(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.21".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_21"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_4")
    def VER_10_4(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "10.4".

        :deprecated: Version 10.4 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_5")
    def VER_10_5(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "10.5".

        :deprecated: Version 10.5 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_6")
    def VER_10_6(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "10.6".

        :deprecated: Version 10.6 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_7")
    def VER_10_7(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "10.7".

        :deprecated: Version 10.7 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_11")
    def VER_11_11(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.11".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_11"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_12")
    def VER_11_12(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.12".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_12"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_13")
    def VER_11_13(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.13".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_13"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_14")
    def VER_11_14(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.14".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_14"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_15")
    def VER_11_15(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.15".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_15"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_16")
    def VER_11_16(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.16".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_16"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_17")
    def VER_11_17(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.17".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_17"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_18")
    def VER_11_18(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.18".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_18"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_19")
    def VER_11_19(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.19".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_19"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_4")
    def VER_11_4(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "11.4".

        :deprecated: Version 11.4 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_6")
    def VER_11_6(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "11.6".

        :deprecated: Version 11.6 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_7")
    def VER_11_7(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "11.7".

        :deprecated: Version 11.7 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_8")
    def VER_11_8(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "11.8".

        :deprecated: Version 11.8 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_9")
    def VER_11_9(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.9".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_9"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_10")
    def VER_12_10(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.10".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_10"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_11")
    def VER_12_11(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.11".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_11"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_12")
    def VER_12_12(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.12".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_12"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_13")
    def VER_12_13(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.13".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_13"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_14")
    def VER_12_14(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.14".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_14"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_4")
    def VER_12_4(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.4".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_6")
    def VER_12_6(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.6".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_7")
    def VER_12_7(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.7".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_8")
    def VER_12_8(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.8".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_9")
    def VER_12_9(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.9".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_9"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_10")
    def VER_13_10(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.10".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_10"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_3")
    def VER_13_3(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.3".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_4")
    def VER_13_4(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.4".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_5")
    def VER_13_5(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.5".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_6")
    def VER_13_6(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.6".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_7")
    def VER_13_7(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.7".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_8")
    def VER_13_8(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.8".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_9")
    def VER_13_9(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.9".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_9"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_14_3")
    def VER_14_3(cls) -> "AuroraPostgresEngineVersion":
        '''Version "14.3".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_14_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_14_4")
    def VER_14_4(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "14.4".

        :deprecated: Version 14.4 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_14_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_14_5")
    def VER_14_5(cls) -> "AuroraPostgresEngineVersion":
        '''Version "14.5".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_14_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_14_6")
    def VER_14_6(cls) -> "AuroraPostgresEngineVersion":
        '''Version "14.6".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_14_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_14_7")
    def VER_14_7(cls) -> "AuroraPostgresEngineVersion":
        '''Version "14.7".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_14_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_15_2")
    def VER_15_2(cls) -> "AuroraPostgresEngineVersion":
        '''Version "15.2".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_15_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_11")
    def VER_9_6_11(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "9.6.11".

        :deprecated: Version 9.6.11 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_11"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_12")
    def VER_9_6_12(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "9.6.12".

        :deprecated: Version 9.6.12 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_12"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_16")
    def VER_9_6_16(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "9.6.16".

        :deprecated: Version 9.6.16 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_16"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_17")
    def VER_9_6_17(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "9.6.17".

        :deprecated: Version 9.6.17 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_17"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_18")
    def VER_9_6_18(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "9.6.18".

        :deprecated: Version 9.6.18 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_18"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_19")
    def VER_9_6_19(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "9.6.19".

        :deprecated: Version 9.6.19 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_19"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_8")
    def VER_9_6_8(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "9.6.8".

        :deprecated: Version 9.6.8 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_9")
    def VER_9_6_9(cls) -> "AuroraPostgresEngineVersion":
        '''(deprecated) Version "9.6.9".

        :deprecated: Version 9.6.9 is no longer supported by Amazon RDS.

        :stability: deprecated
        '''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_9"))

    @builtins.property
    @jsii.member(jsii_name="auroraPostgresFullVersion")
    def aurora_postgres_full_version(self) -> builtins.str:
        '''The full version string, for example, "9.6.25.1".'''
        return typing.cast(builtins.str, jsii.get(self, "auroraPostgresFullVersion"))

    @builtins.property
    @jsii.member(jsii_name="auroraPostgresMajorVersion")
    def aurora_postgres_major_version(self) -> builtins.str:
        '''The major version of the engine, for example, "9.6".'''
        return typing.cast(builtins.str, jsii.get(self, "auroraPostgresMajorVersion"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.BackupProps",
    jsii_struct_bases=[],
    name_mapping={"retention": "retention", "preferred_window": "preferredWindow"},
)
class BackupProps:
    def __init__(
        self,
        *,
        retention: _Duration_4839e8c3,
        preferred_window: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Backup configuration for RDS databases.

        :param retention: How many days to retain the backup.
        :param preferred_window: A daily time range in 24-hours UTC format in which backups preferably execute. Must be at least 30 minutes long. Example: '01:00-02:00' Default: - a 30-minute window selected at random from an 8-hour block of time for each AWS Region. To see the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow

        :default:

        - The retention period for automated backups is 1 day.
        The preferred backup window will be a 30-minute window selected at random
        from an 8-hour block of time for each AWS Region.

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_rds as rds
            
            backup_props = rds.BackupProps(
                retention=cdk.Duration.minutes(30),
            
                # the properties below are optional
                preferred_window="preferredWindow"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4b1b9d82383453f3cc9869e51ee8303d65c2e0eadfa6d1cf8782f7ae80438970)
            check_type(argname="argument retention", value=retention, expected_type=type_hints["retention"])
            check_type(argname="argument preferred_window", value=preferred_window, expected_type=type_hints["preferred_window"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "retention": retention,
        }
        if preferred_window is not None:
            self._values["preferred_window"] = preferred_window

    @builtins.property
    def retention(self) -> _Duration_4839e8c3:
        '''How many days to retain the backup.'''
        result = self._values.get("retention")
        assert result is not None, "Required property 'retention' is missing"
        return typing.cast(_Duration_4839e8c3, result)

    @builtins.property
    def preferred_window(self) -> typing.Optional[builtins.str]:
        '''A daily time range in 24-hours UTC format in which backups preferably execute.

        Must be at least 30 minutes long.

        Example: '01:00-02:00'

        :default:

        - a 30-minute window selected at random from an 8-hour block of
        time for each AWS Region. To see the time blocks available, see
        https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        '''
        result = self._values.get("preferred_window")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BackupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnCustomDBEngineVersion(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnCustomDBEngineVersion",
):
    '''A CloudFormation ``AWS::RDS::CustomDBEngineVersion``.

    Creates a custom DB engine version (CEV).

    :cloudformationResource: AWS::RDS::CustomDBEngineVersion
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_custom_dBEngine_version = rds.CfnCustomDBEngineVersion(self, "MyCfnCustomDBEngineVersion",
            database_installation_files_s3_bucket_name="databaseInstallationFilesS3BucketName",
            engine="engine",
            engine_version="engineVersion",
        
            # the properties below are optional
            database_installation_files_s3_prefix="databaseInstallationFilesS3Prefix",
            description="description",
            kms_key_id="kmsKeyId",
            manifest="manifest",
            status="status",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        database_installation_files_s3_bucket_name: builtins.str,
        engine: builtins.str,
        engine_version: builtins.str,
        database_installation_files_s3_prefix: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        manifest: typing.Optional[builtins.str] = None,
        status: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::CustomDBEngineVersion``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param database_installation_files_s3_bucket_name: The name of an Amazon S3 bucket that contains database installation files for your CEV. For example, a valid bucket name is ``my-custom-installation-files`` .
        :param engine: The database engine to use for your custom engine version (CEV). Valid values: - ``custom-oracle-ee`` - ``custom-oracle-ee-cdb``
        :param engine_version: The name of your CEV. The name format is ``major version.customized_string`` . For example, a valid CEV name is ``19.my_cev1`` . This setting is required for RDS Custom for Oracle, but optional for Amazon RDS. The combination of ``Engine`` and ``EngineVersion`` is unique per customer per Region. *Constraints:* Minimum length is 1. Maximum length is 60. *Pattern:* ``^[a-z0-9_.-]{1,60$`` }
        :param database_installation_files_s3_prefix: The Amazon S3 directory that contains the database installation files for your CEV. For example, a valid bucket name is ``123456789012/cev1`` . If this setting isn't specified, no prefix is assumed.
        :param description: An optional description of your CEV.
        :param kms_key_id: The AWS KMS key identifier for an encrypted CEV. A symmetric encryption KMS key is required for RDS Custom, but optional for Amazon RDS. If you have an existing symmetric encryption KMS key in your account, you can use it with RDS Custom. No further action is necessary. If you don't already have a symmetric encryption KMS key in your account, follow the instructions in `Creating a symmetric encryption KMS key <https://docs.aws.amazon.com/kms/latest/developerguide/create-keys.html#create-symmetric-cmk>`_ in the *AWS Key Management Service Developer Guide* . You can choose the same symmetric encryption key when you create a CEV and a DB instance, or choose different keys.
        :param manifest: The CEV manifest, which is a JSON document that describes the installation .zip files stored in Amazon S3. Specify the name/value pairs in a file or a quoted string. RDS Custom applies the patches in the order in which they are listed. The following JSON fields are valid: - **MediaImportTemplateVersion** - Version of the CEV manifest. The date is in the format ``YYYY-MM-DD`` . - **databaseInstallationFileNames** - Ordered list of installation files for the CEV. - **opatchFileNames** - Ordered list of OPatch installers used for the Oracle DB engine. - **psuRuPatchFileNames** - The PSU and RU patches for this CEV. - **OtherPatchFileNames** - The patches that are not in the list of PSU and RU patches. Amazon RDS applies these patches after applying the PSU and RU patches. For more information, see `Creating the CEV manifest <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-cev.html#custom-cev.preparing.manifest>`_ in the *Amazon RDS User Guide* .
        :param status: A value that indicates the status of a custom engine version (CEV).
        :param tags: A list of tags. For more information, see `Tagging Amazon RDS Resources <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html>`_ in the *Amazon RDS User Guide.*
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dccc4ebf781452ad945a67eeaa182293b79bd42fee7658aaea830a59e6546b0b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCustomDBEngineVersionProps(
            database_installation_files_s3_bucket_name=database_installation_files_s3_bucket_name,
            engine=engine,
            engine_version=engine_version,
            database_installation_files_s3_prefix=database_installation_files_s3_prefix,
            description=description,
            kms_key_id=kms_key_id,
            manifest=manifest,
            status=status,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__74719765ef5959695edd3d77886d4364474437f2e536dbd8c1ffd94e5067ae8d)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3e83f468f7dd18d22c9b612734116e5d82023ddbd1b241f1dae9c33e1cc350d8)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDbEngineVersionArn")
    def attr_db_engine_version_arn(self) -> builtins.str:
        '''The ARN of the custom engine version.

        :cloudformationAttribute: DBEngineVersionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbEngineVersionArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''A list of tags.

        For more information, see `Tagging Amazon RDS Resources <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="databaseInstallationFilesS3BucketName")
    def database_installation_files_s3_bucket_name(self) -> builtins.str:
        '''The name of an Amazon S3 bucket that contains database installation files for your CEV.

        For example, a valid bucket name is ``my-custom-installation-files`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-databaseinstallationfiless3bucketname
        '''
        return typing.cast(builtins.str, jsii.get(self, "databaseInstallationFilesS3BucketName"))

    @database_installation_files_s3_bucket_name.setter
    def database_installation_files_s3_bucket_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e8dcd4aecfb92c6617a76401829247508a5196d1219bdf3c496c1eef202e0c37)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "databaseInstallationFilesS3BucketName", value)

    @builtins.property
    @jsii.member(jsii_name="engine")
    def engine(self) -> builtins.str:
        '''The database engine to use for your custom engine version (CEV).

        Valid values:

        - ``custom-oracle-ee``
        - ``custom-oracle-ee-cdb``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-engine
        '''
        return typing.cast(builtins.str, jsii.get(self, "engine"))

    @engine.setter
    def engine(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__57b06d6a61a7aa40468b988c47f903780541aacb379dfa0bf146f37e0c779729)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engine", value)

    @builtins.property
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> builtins.str:
        '''The name of your CEV.

        The name format is ``major version.customized_string`` . For example, a valid CEV name is ``19.my_cev1`` . This setting is required for RDS Custom for Oracle, but optional for Amazon RDS. The combination of ``Engine`` and ``EngineVersion`` is unique per customer per Region.

        *Constraints:* Minimum length is 1. Maximum length is 60.

        *Pattern:* ``^[a-z0-9_.-]{1,60$`` }

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-engineversion
        '''
        return typing.cast(builtins.str, jsii.get(self, "engineVersion"))

    @engine_version.setter
    def engine_version(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__09530b6c217cbc4998aaea3601330c702807beb29fc30fa9af5ceab12214408d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineVersion", value)

    @builtins.property
    @jsii.member(jsii_name="databaseInstallationFilesS3Prefix")
    def database_installation_files_s3_prefix(self) -> typing.Optional[builtins.str]:
        '''The Amazon S3 directory that contains the database installation files for your CEV.

        For example, a valid bucket name is ``123456789012/cev1`` . If this setting isn't specified, no prefix is assumed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-databaseinstallationfiless3prefix
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "databaseInstallationFilesS3Prefix"))

    @database_installation_files_s3_prefix.setter
    def database_installation_files_s3_prefix(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eed0c37a026ac98aca414552d356dfb6e8f65704640034a73702c850f465b41a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "databaseInstallationFilesS3Prefix", value)

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''An optional description of your CEV.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4aaac4d275f054bf611e124a013cfc149ce4ce49db255d6b8bdbf993794d3044)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for an encrypted CEV.

        A symmetric encryption KMS key is required for RDS Custom, but optional for Amazon RDS.

        If you have an existing symmetric encryption KMS key in your account, you can use it with RDS Custom. No further action is necessary. If you don't already have a symmetric encryption KMS key in your account, follow the instructions in `Creating a symmetric encryption KMS key <https://docs.aws.amazon.com/kms/latest/developerguide/create-keys.html#create-symmetric-cmk>`_ in the *AWS Key Management Service Developer Guide* .

        You can choose the same symmetric encryption key when you create a CEV and a DB instance, or choose different keys.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-kmskeyid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyId"))

    @kms_key_id.setter
    def kms_key_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1a0cd73871adb0d339f703041dbaa1e90b6c3ad3be57fe4b1f55f2e61af11dca)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyId", value)

    @builtins.property
    @jsii.member(jsii_name="manifest")
    def manifest(self) -> typing.Optional[builtins.str]:
        '''The CEV manifest, which is a JSON document that describes the installation .zip files stored in Amazon S3. Specify the name/value pairs in a file or a quoted string. RDS Custom applies the patches in the order in which they are listed.

        The following JSON fields are valid:

        - **MediaImportTemplateVersion** - Version of the CEV manifest. The date is in the format ``YYYY-MM-DD`` .
        - **databaseInstallationFileNames** - Ordered list of installation files for the CEV.
        - **opatchFileNames** - Ordered list of OPatch installers used for the Oracle DB engine.
        - **psuRuPatchFileNames** - The PSU and RU patches for this CEV.
        - **OtherPatchFileNames** - The patches that are not in the list of PSU and RU patches. Amazon RDS applies these patches after applying the PSU and RU patches.

        For more information, see `Creating the CEV manifest <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-cev.html#custom-cev.preparing.manifest>`_ in the *Amazon RDS User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-manifest
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "manifest"))

    @manifest.setter
    def manifest(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__463fc7056b1df9d389ba18cfed697b7ea579834be61dc7627fe4a6bba7a0ce5b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "manifest", value)

    @builtins.property
    @jsii.member(jsii_name="status")
    def status(self) -> typing.Optional[builtins.str]:
        '''A value that indicates the status of a custom engine version (CEV).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-status
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "status"))

    @status.setter
    def status(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ec128ecf80f4b2e6d4529f80259fe9fb957476b36db57ec7b870982375842eac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "status", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnCustomDBEngineVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "database_installation_files_s3_bucket_name": "databaseInstallationFilesS3BucketName",
        "engine": "engine",
        "engine_version": "engineVersion",
        "database_installation_files_s3_prefix": "databaseInstallationFilesS3Prefix",
        "description": "description",
        "kms_key_id": "kmsKeyId",
        "manifest": "manifest",
        "status": "status",
        "tags": "tags",
    },
)
class CfnCustomDBEngineVersionProps:
    def __init__(
        self,
        *,
        database_installation_files_s3_bucket_name: builtins.str,
        engine: builtins.str,
        engine_version: builtins.str,
        database_installation_files_s3_prefix: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        manifest: typing.Optional[builtins.str] = None,
        status: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCustomDBEngineVersion``.

        :param database_installation_files_s3_bucket_name: The name of an Amazon S3 bucket that contains database installation files for your CEV. For example, a valid bucket name is ``my-custom-installation-files`` .
        :param engine: The database engine to use for your custom engine version (CEV). Valid values: - ``custom-oracle-ee`` - ``custom-oracle-ee-cdb``
        :param engine_version: The name of your CEV. The name format is ``major version.customized_string`` . For example, a valid CEV name is ``19.my_cev1`` . This setting is required for RDS Custom for Oracle, but optional for Amazon RDS. The combination of ``Engine`` and ``EngineVersion`` is unique per customer per Region. *Constraints:* Minimum length is 1. Maximum length is 60. *Pattern:* ``^[a-z0-9_.-]{1,60$`` }
        :param database_installation_files_s3_prefix: The Amazon S3 directory that contains the database installation files for your CEV. For example, a valid bucket name is ``123456789012/cev1`` . If this setting isn't specified, no prefix is assumed.
        :param description: An optional description of your CEV.
        :param kms_key_id: The AWS KMS key identifier for an encrypted CEV. A symmetric encryption KMS key is required for RDS Custom, but optional for Amazon RDS. If you have an existing symmetric encryption KMS key in your account, you can use it with RDS Custom. No further action is necessary. If you don't already have a symmetric encryption KMS key in your account, follow the instructions in `Creating a symmetric encryption KMS key <https://docs.aws.amazon.com/kms/latest/developerguide/create-keys.html#create-symmetric-cmk>`_ in the *AWS Key Management Service Developer Guide* . You can choose the same symmetric encryption key when you create a CEV and a DB instance, or choose different keys.
        :param manifest: The CEV manifest, which is a JSON document that describes the installation .zip files stored in Amazon S3. Specify the name/value pairs in a file or a quoted string. RDS Custom applies the patches in the order in which they are listed. The following JSON fields are valid: - **MediaImportTemplateVersion** - Version of the CEV manifest. The date is in the format ``YYYY-MM-DD`` . - **databaseInstallationFileNames** - Ordered list of installation files for the CEV. - **opatchFileNames** - Ordered list of OPatch installers used for the Oracle DB engine. - **psuRuPatchFileNames** - The PSU and RU patches for this CEV. - **OtherPatchFileNames** - The patches that are not in the list of PSU and RU patches. Amazon RDS applies these patches after applying the PSU and RU patches. For more information, see `Creating the CEV manifest <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-cev.html#custom-cev.preparing.manifest>`_ in the *Amazon RDS User Guide* .
        :param status: A value that indicates the status of a custom engine version (CEV).
        :param tags: A list of tags. For more information, see `Tagging Amazon RDS Resources <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_custom_dBEngine_version_props = rds.CfnCustomDBEngineVersionProps(
                database_installation_files_s3_bucket_name="databaseInstallationFilesS3BucketName",
                engine="engine",
                engine_version="engineVersion",
            
                # the properties below are optional
                database_installation_files_s3_prefix="databaseInstallationFilesS3Prefix",
                description="description",
                kms_key_id="kmsKeyId",
                manifest="manifest",
                status="status",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__70c4d3797ab8093b0278a7f532fcb1c42c50e74a1c04dfe5259dbb57e7fe2191)
            check_type(argname="argument database_installation_files_s3_bucket_name", value=database_installation_files_s3_bucket_name, expected_type=type_hints["database_installation_files_s3_bucket_name"])
            check_type(argname="argument engine", value=engine, expected_type=type_hints["engine"])
            check_type(argname="argument engine_version", value=engine_version, expected_type=type_hints["engine_version"])
            check_type(argname="argument database_installation_files_s3_prefix", value=database_installation_files_s3_prefix, expected_type=type_hints["database_installation_files_s3_prefix"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
            check_type(argname="argument manifest", value=manifest, expected_type=type_hints["manifest"])
            check_type(argname="argument status", value=status, expected_type=type_hints["status"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "database_installation_files_s3_bucket_name": database_installation_files_s3_bucket_name,
            "engine": engine,
            "engine_version": engine_version,
        }
        if database_installation_files_s3_prefix is not None:
            self._values["database_installation_files_s3_prefix"] = database_installation_files_s3_prefix
        if description is not None:
            self._values["description"] = description
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id
        if manifest is not None:
            self._values["manifest"] = manifest
        if status is not None:
            self._values["status"] = status
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def database_installation_files_s3_bucket_name(self) -> builtins.str:
        '''The name of an Amazon S3 bucket that contains database installation files for your CEV.

        For example, a valid bucket name is ``my-custom-installation-files`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-databaseinstallationfiless3bucketname
        '''
        result = self._values.get("database_installation_files_s3_bucket_name")
        assert result is not None, "Required property 'database_installation_files_s3_bucket_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def engine(self) -> builtins.str:
        '''The database engine to use for your custom engine version (CEV).

        Valid values:

        - ``custom-oracle-ee``
        - ``custom-oracle-ee-cdb``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-engine
        '''
        result = self._values.get("engine")
        assert result is not None, "Required property 'engine' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def engine_version(self) -> builtins.str:
        '''The name of your CEV.

        The name format is ``major version.customized_string`` . For example, a valid CEV name is ``19.my_cev1`` . This setting is required for RDS Custom for Oracle, but optional for Amazon RDS. The combination of ``Engine`` and ``EngineVersion`` is unique per customer per Region.

        *Constraints:* Minimum length is 1. Maximum length is 60.

        *Pattern:* ``^[a-z0-9_.-]{1,60$`` }

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-engineversion
        '''
        result = self._values.get("engine_version")
        assert result is not None, "Required property 'engine_version' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def database_installation_files_s3_prefix(self) -> typing.Optional[builtins.str]:
        '''The Amazon S3 directory that contains the database installation files for your CEV.

        For example, a valid bucket name is ``123456789012/cev1`` . If this setting isn't specified, no prefix is assumed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-databaseinstallationfiless3prefix
        '''
        result = self._values.get("database_installation_files_s3_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''An optional description of your CEV.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for an encrypted CEV.

        A symmetric encryption KMS key is required for RDS Custom, but optional for Amazon RDS.

        If you have an existing symmetric encryption KMS key in your account, you can use it with RDS Custom. No further action is necessary. If you don't already have a symmetric encryption KMS key in your account, follow the instructions in `Creating a symmetric encryption KMS key <https://docs.aws.amazon.com/kms/latest/developerguide/create-keys.html#create-symmetric-cmk>`_ in the *AWS Key Management Service Developer Guide* .

        You can choose the same symmetric encryption key when you create a CEV and a DB instance, or choose different keys.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-kmskeyid
        '''
        result = self._values.get("kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def manifest(self) -> typing.Optional[builtins.str]:
        '''The CEV manifest, which is a JSON document that describes the installation .zip files stored in Amazon S3. Specify the name/value pairs in a file or a quoted string. RDS Custom applies the patches in the order in which they are listed.

        The following JSON fields are valid:

        - **MediaImportTemplateVersion** - Version of the CEV manifest. The date is in the format ``YYYY-MM-DD`` .
        - **databaseInstallationFileNames** - Ordered list of installation files for the CEV.
        - **opatchFileNames** - Ordered list of OPatch installers used for the Oracle DB engine.
        - **psuRuPatchFileNames** - The PSU and RU patches for this CEV.
        - **OtherPatchFileNames** - The patches that are not in the list of PSU and RU patches. Amazon RDS applies these patches after applying the PSU and RU patches.

        For more information, see `Creating the CEV manifest <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-cev.html#custom-cev.preparing.manifest>`_ in the *Amazon RDS User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-manifest
        '''
        result = self._values.get("manifest")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def status(self) -> typing.Optional[builtins.str]:
        '''A value that indicates the status of a custom engine version (CEV).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-status
        '''
        result = self._values.get("status")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''A list of tags.

        For more information, see `Tagging Amazon RDS Resources <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-customdbengineversion.html#cfn-rds-customdbengineversion-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCustomDBEngineVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDBCluster(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBCluster",
):
    '''A CloudFormation ``AWS::RDS::DBCluster``.

    The ``AWS::RDS::DBCluster`` resource creates an Amazon Aurora DB cluster or Multi-AZ DB cluster.

    For more information about creating an Aurora DB cluster, see `Creating an Amazon Aurora DB cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.CreateInstance.html>`_ in the *Amazon Aurora User Guide* .

    For more information about creating a Multi-AZ DB cluster, see `Creating a Multi-AZ DB cluster <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/create-multi-az-db-cluster.html>`_ in the *Amazon RDS User Guide* .
    .. epigraph::

       You can only create this resource in AWS Regions where Amazon Aurora or Multi-AZ DB clusters are supported.

    *Updating DB clusters*

    When properties labeled " *Update requires:* `Replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ " are updated, AWS CloudFormation first creates a replacement DB cluster, then changes references from other dependent resources to point to the replacement DB cluster, and finally deletes the old DB cluster.
    .. epigraph::

       We highly recommend that you take a snapshot of the database before updating the stack. If you don't, you lose the data when AWS CloudFormation replaces your DB cluster. To preserve your data, perform the following procedure:

       - Deactivate any applications that are using the DB cluster so that there's no activity on the DB instance.
       - Create a snapshot of the DB cluster. For more information, see `Creating a DB Cluster Snapshot <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_CreateSnapshotCluster.html>`_ .
       - If you want to restore your DB cluster using a DB cluster snapshot, modify the updated template with your DB cluster changes and add the ``SnapshotIdentifier`` property with the ID of the DB cluster snapshot that you want to use.

       After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the DB cluster snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted.

       - Update the stack.

    Currently, when you are updating the stack for an Aurora Serverless DB cluster, you can't include changes to any other properties when you specify one of the following properties: ``PreferredBackupWindow`` , ``PreferredMaintenanceWindow`` , and ``Port`` . This limitation doesn't apply to provisioned DB clusters.

    For more information about updating other properties of this resource, see ``[ModifyDBCluster](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_ModifyDBCluster.html)`` . For more information about updating stacks, see `AWS CloudFormation Stacks Updates <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks.html>`_ .

    *Deleting DB clusters*

    The default ``DeletionPolicy`` for ``AWS::RDS::DBCluster`` resources is ``Snapshot`` . For more information about how AWS CloudFormation deletes resources, see `DeletionPolicy Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html>`_ .

    :cloudformationResource: AWS::RDS::DBCluster
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_dBCluster = rds.CfnDBCluster(self, "MyCfnDBCluster",
            allocated_storage=123,
            associated_roles=[rds.CfnDBCluster.DBClusterRoleProperty(
                role_arn="roleArn",
        
                # the properties below are optional
                feature_name="featureName"
            )],
            auto_minor_version_upgrade=False,
            availability_zones=["availabilityZones"],
            backtrack_window=123,
            backup_retention_period=123,
            copy_tags_to_snapshot=False,
            database_name="databaseName",
            db_cluster_identifier="dbClusterIdentifier",
            db_cluster_instance_class="dbClusterInstanceClass",
            db_cluster_parameter_group_name="dbClusterParameterGroupName",
            db_instance_parameter_group_name="dbInstanceParameterGroupName",
            db_subnet_group_name="dbSubnetGroupName",
            db_system_id="dbSystemId",
            deletion_protection=False,
            domain="domain",
            domain_iam_role_name="domainIamRoleName",
            enable_cloudwatch_logs_exports=["enableCloudwatchLogsExports"],
            enable_http_endpoint=False,
            enable_iam_database_authentication=False,
            engine="engine",
            engine_mode="engineMode",
            engine_version="engineVersion",
            global_cluster_identifier="globalClusterIdentifier",
            iops=123,
            kms_key_id="kmsKeyId",
            manage_master_user_password=False,
            master_username="masterUsername",
            master_user_password="masterUserPassword",
            master_user_secret=rds.CfnDBCluster.MasterUserSecretProperty(
                kms_key_id="kmsKeyId",
                secret_arn="secretArn"
            ),
            monitoring_interval=123,
            monitoring_role_arn="monitoringRoleArn",
            network_type="networkType",
            performance_insights_enabled=False,
            performance_insights_kms_key_id="performanceInsightsKmsKeyId",
            performance_insights_retention_period=123,
            port=123,
            preferred_backup_window="preferredBackupWindow",
            preferred_maintenance_window="preferredMaintenanceWindow",
            publicly_accessible=False,
            replication_source_identifier="replicationSourceIdentifier",
            restore_to_time="restoreToTime",
            restore_type="restoreType",
            scaling_configuration=rds.CfnDBCluster.ScalingConfigurationProperty(
                auto_pause=False,
                max_capacity=123,
                min_capacity=123,
                seconds_before_timeout=123,
                seconds_until_auto_pause=123,
                timeout_action="timeoutAction"
            ),
            serverless_v2_scaling_configuration=rds.CfnDBCluster.ServerlessV2ScalingConfigurationProperty(
                max_capacity=123,
                min_capacity=123
            ),
            snapshot_identifier="snapshotIdentifier",
            source_db_cluster_identifier="sourceDbClusterIdentifier",
            source_region="sourceRegion",
            storage_encrypted=False,
            storage_type="storageType",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            use_latest_restorable_time=False,
            vpc_security_group_ids=["vpcSecurityGroupIds"]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        allocated_storage: typing.Optional[jsii.Number] = None,
        associated_roles: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBCluster.DBClusterRoleProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        availability_zones: typing.Optional[typing.Sequence[builtins.str]] = None,
        backtrack_window: typing.Optional[jsii.Number] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        database_name: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_instance_class: typing.Optional[builtins.str] = None,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        db_instance_parameter_group_name: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        db_system_id: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.Sequence[builtins.str]] = None,
        enable_http_endpoint: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_mode: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        manage_master_user_password: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        master_user_secret: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBCluster.MasterUserSecretProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        network_type: typing.Optional[builtins.str] = None,
        performance_insights_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        replication_source_identifier: typing.Optional[builtins.str] = None,
        restore_to_time: typing.Optional[builtins.str] = None,
        restore_type: typing.Optional[builtins.str] = None,
        scaling_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBCluster.ScalingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        serverless_v2_scaling_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBCluster.ServerlessV2ScalingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        snapshot_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBCluster``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param allocated_storage: The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster. Valid for Cluster Type: Multi-AZ DB clusters only This setting is required to create a Multi-AZ DB cluster.
        :param associated_roles: Provides a list of the AWS Identity and Access Management (IAM) roles that are associated with the DB cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other Amazon Web Services on your behalf. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param auto_minor_version_upgrade: Specifies whether minor engine upgrades are applied automatically to the DB cluster during the maintenance window. By default, minor engine upgrades are applied automatically. Valid for Cluster Type: Multi-AZ DB clusters only
        :param availability_zones: A list of Availability Zones (AZs) where instances in the DB cluster can be created. For information on AWS Regions and Availability Zones, see `Choosing the Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param backtrack_window: The target backtrack window, in seconds. To disable backtracking, set this value to 0. .. epigraph:: Currently, Backtrack is only supported for Aurora MySQL DB clusters. Default: 0 Constraints: - If specified, this value must be set to a number from 0 to 259,200 (72 hours). Valid for: Aurora MySQL DB clusters only
        :param backup_retention_period: The number of days for which automated backups are retained. Default: 1 Constraints: - Must be a value from 1 to 35 Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param copy_tags_to_snapshot: A value that indicates whether to copy all tags from the DB cluster to snapshots of the DB cluster. The default is not to copy them. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param database_name: The name of your database. If you don't provide a name, then Amazon RDS won't create a database in this DB cluster. For naming constraints, see `Naming Constraints <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_cluster_identifier: The DB cluster identifier. This parameter is stored as a lowercase string. Constraints: - Must contain from 1 to 63 letters, numbers, or hyphens. - First character must be a letter. - Can't end with a hyphen or contain two consecutive hyphens. Example: ``my-cluster1`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_cluster_instance_class: The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example ``db.m6gd.xlarge`` . Not all DB instance classes are available in all AWS Regions , or for all database engines. For the full list of DB instance classes and availability for your engine, see `DB instance class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide* . This setting is required to create a Multi-AZ DB cluster. Valid for Cluster Type: Multi-AZ DB clusters only
        :param db_cluster_parameter_group_name: The name of the DB cluster parameter group to associate with this DB cluster. .. epigraph:: If you apply a parameter group to an existing DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting. If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started. To list all of the available DB cluster parameter group names, use the following command: ``aws rds describe-db-cluster-parameter-groups --query "DBClusterParameterGroups[].DBClusterParameterGroupName" --output text`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_instance_parameter_group_name: The name of the DB parameter group to apply to all instances of the DB cluster. .. epigraph:: When you apply a parameter group using the ``DBInstanceParameterGroupName`` parameter, the DB cluster isn't rebooted automatically. Also, parameter changes are applied immediately rather than during the next maintenance window. Default: The existing name setting Constraints: - The DB parameter group must be in the same DB parameter group family as this DB cluster.
        :param db_subnet_group_name: A DB subnet group that you want to associate with this DB cluster. If you are restoring a DB cluster to a point in time with ``RestoreType`` set to ``copy-on-write`` , and don't specify a DB subnet group name, then the DB cluster is restored with a default DB subnet group. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_system_id: Reserved for future use.
        :param deletion_protection: A value that indicates whether the DB cluster has deletion protection enabled. The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param domain: Indicates the directory ID of the Active Directory to create the DB cluster. For Amazon Aurora DB clusters, Amazon RDS can use Kerberos authentication to authenticate users that connect to the DB cluster. For more information, see `Kerberos authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/kerberos-authentication.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param domain_iam_role_name: Specifies the name of the IAM role to use when making API calls to the Directory Service. Valid for: Aurora DB clusters only
        :param enable_cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Aurora User Guide* . *Aurora MySQL* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Aurora PostgreSQL* Valid values: ``postgresql`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param enable_http_endpoint: A value that indicates whether to enable the HTTP endpoint for an Aurora Serverless DB cluster. By default, the HTTP endpoint is disabled. When enabled, the HTTP endpoint provides a connectionless web service API for running SQL queries on the Aurora Serverless DB cluster. You can also query your database from inside the RDS console with the query editor. For more information, see `Using the Data API for Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param enable_iam_database_authentication: A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. By default, mapping is disabled. For more information, see `IAM Database Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon Aurora User Guide.* Valid for: Aurora DB clusters only
        :param engine: The name of the database engine to be used for this DB cluster. Valid Values: - ``aurora-mysql`` - ``aurora-postgresql`` - ``mysql`` - ``postgres`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param engine_mode: The DB engine mode of the DB cluster, either ``provisioned`` or ``serverless`` . The ``serverless`` engine mode only supports Aurora Serverless v1. Limitations and requirements apply to some DB engine modes. For more information, see the following sections in the *Amazon Aurora User Guide* : - `Limitations of Aurora Serverless v1 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations>`_ - `Requirements for Aurora Serverless v2 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.requirements.html>`_ - `Limitations of parallel query <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-parallel-query.html#aurora-mysql-parallel-query-limitations>`_ - `Limitations of Aurora global databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html#aurora-global-database.limitations>`_ Valid for: Aurora DB clusters only
        :param engine_version: The version number of the database engine to use. To list all of the available engine versions for Aurora MySQL version 2 (5.7-compatible) and version 3 (8.0-compatible), use the following command: ``aws rds describe-db-engine-versions --engine aurora-mysql --query "DBEngineVersions[].EngineVersion"`` You can supply either ``5.7`` or ``8.0`` to use the default engine version for Aurora MySQL version 2 or version 3, respectively. To list all of the available engine versions for Aurora PostgreSQL, use the following command: ``aws rds describe-db-engine-versions --engine aurora-postgresql --query "DBEngineVersions[].EngineVersion"`` To list all of the available engine versions for RDS for MySQL, use the following command: ``aws rds describe-db-engine-versions --engine mysql --query "DBEngineVersions[].EngineVersion"`` To list all of the available engine versions for RDS for PostgreSQL, use the following command: ``aws rds describe-db-engine-versions --engine postgres --query "DBEngineVersions[].EngineVersion"`` *Aurora MySQL* For information, see `Database engine updates for Amazon Aurora MySQL <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Updates.html>`_ in the *Amazon Aurora User Guide* . *Aurora PostgreSQL* For information, see `Amazon Aurora PostgreSQL releases and engine versions <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.20180305.html>`_ in the *Amazon Aurora User Guide* . *MySQL* For information, see `Amazon RDS for MySQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide* . *PostgreSQL* For information, see `Amazon RDS for PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts>`_ in the *Amazon RDS User Guide* . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param global_cluster_identifier: If you are configuring an Aurora global database cluster and want your Aurora DB cluster to be a secondary member in the global database cluster, specify the global cluster ID of the global database cluster. To define the primary database cluster of the global cluster, use the `AWS::RDS::GlobalCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html>`_ resource. If you aren't configuring a global database cluster, don't specify this property. .. epigraph:: To remove the DB cluster from a global database cluster, specify an empty value for the ``GlobalClusterIdentifier`` property. For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param iops: The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid IOPS values, see `Provisioned IOPS storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* . This setting is required to create a Multi-AZ DB cluster. Valid for Cluster Type: Multi-AZ DB clusters only Constraints: - Must be a multiple between .5 and 50 of the storage amount for the DB cluster.
        :param kms_key_id: The Amazon Resource Name (ARN) of the AWS KMS key that is used to encrypt the database instances in the DB cluster, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` . If you enable the ``StorageEncrypted`` property but don't specify this property, the default KMS key is used. If you specify this property, you must set the ``StorageEncrypted`` property to ``true`` . If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param manage_master_user_password: Specifies whether to manage the master user password with AWS Secrets Manager. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.* Valid for Cluster Type: Aurora DB clusters and Multi-AZ DB clusters Constraints: - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.
        :param master_username: The name of the master user for the DB cluster. .. epigraph:: If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_user_password: The master password for the DB instance. .. epigraph:: If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_user_secret: The secret managed by RDS in AWS Secrets Manager for the master user password. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*
        :param monitoring_interval: The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster. To turn off collecting Enhanced Monitoring metrics, specify ``0`` . If ``MonitoringRoleArn`` is specified, also set ``MonitoringInterval`` to a value other than ``0`` . Valid for Cluster Type: Multi-AZ DB clusters only Valid Values: ``0 | 1 | 5 | 10 | 15 | 30 | 60`` Default: ``0``
        :param monitoring_role_arn: The Amazon Resource Name (ARN) for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs. An example is ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting up and enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* . If ``MonitoringInterval`` is set to a value other than ``0`` , supply a ``MonitoringRoleArn`` value. Valid for Cluster Type: Multi-AZ DB clusters only
        :param network_type: The network type of the DB cluster. Valid values: - ``IPV4`` - ``DUAL`` The network type is determined by the ``DBSubnetGroup`` specified for the DB cluster. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ). For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon Aurora User Guide.* Valid for: Aurora DB clusters only
        :param performance_insights_enabled: Specifies whether to turn on Performance Insights for the DB cluster. For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* . Valid for Cluster Type: Multi-AZ DB clusters only
        :param performance_insights_kms_key_id: The AWS KMS key identifier for encryption of Performance Insights data. The AWS KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. If you don't specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account . Your AWS account has a different default KMS key for each AWS Region . Valid for Cluster Type: Multi-AZ DB clusters only
        :param performance_insights_retention_period: The number of days to retain Performance Insights data. Valid for Cluster Type: Multi-AZ DB clusters only Valid Values: - ``7`` - *month* * 31, where *month* is a number of months from 1-23. Examples: ``93`` (3 months * 31), ``341`` (11 months * 31), ``589`` (19 months * 31) - ``731`` Default: ``7`` days If you specify a retention period that isn't valid, such as ``94`` , Amazon RDS issues an error.
        :param port: The port number on which the DB instances in the DB cluster accept connections. Default: - When ``EngineMode`` is ``provisioned`` , ``3306`` (for both Aurora MySQL and Aurora PostgreSQL) - When ``EngineMode`` is ``serverless`` : - ``3306`` when ``Engine`` is ``aurora`` or ``aurora-mysql`` - ``5432`` when ``Engine`` is ``aurora-postgresql`` .. epigraph:: The ``No interruption`` on update behavior only applies to DB clusters. If you are updating a DB instance, see `Port <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port>`_ for the AWS::RDS::DBInstance resource. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param preferred_backup_window: The daily time range during which automated backups are created. For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.Backups.BackupWindow>`_ in the *Amazon Aurora User Guide.* Constraints: - Must be in the format ``hh24:mi-hh24:mi`` . - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param preferred_maintenance_window: The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC). Format: ``ddd:hh24:mi-ddd:hh24:mi`` The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Cluster Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow.Aurora>`_ in the *Amazon Aurora User Guide.* Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun. Constraints: Minimum 30-minute window. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param publicly_accessible: Specifies whether the DB cluster is publicly accessible. When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it uses. That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it. When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to a private IP address. Valid for Cluster Type: Multi-AZ DB clusters only Default: The default behavior varies depending on whether ``DBSubnetGroupName`` is specified. If ``DBSubnetGroupName`` isn't specified, and ``PubliclyAccessible`` isn't specified, the following applies: - If the default VPC in the target Region doesn’t have an internet gateway attached to it, the DB cluster is private. - If the default VPC in the target Region has an internet gateway attached to it, the DB cluster is public. If ``DBSubnetGroupName`` is specified, and ``PubliclyAccessible`` isn't specified, the following applies: - If the subnets are part of a VPC that doesn’t have an internet gateway attached to it, the DB cluster is private. - If the subnets are part of a VPC that has an internet gateway attached to it, the DB cluster is public.
        :param replication_source_identifier: The Amazon Resource Name (ARN) of the source DB instance or DB cluster if this DB cluster is created as a read replica. Valid for: Aurora DB clusters only
        :param restore_to_time: The date and time to restore the DB cluster to. Valid Values: Value must be a time in Universal Coordinated Time (UTC) format Constraints: - Must be before the latest restorable time for the DB instance - Must be specified if ``UseLatestRestorableTime`` parameter isn't provided - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled - Can't be specified if the ``RestoreType`` parameter is ``copy-on-write`` Example: ``2015-03-07T23:45:00Z`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param restore_type: The type of restore to be performed. You can specify one of the following values:. - ``full-copy`` - The new DB cluster is restored as a full copy of the source DB cluster. - ``copy-on-write`` - The new DB cluster is restored as a clone of the source DB cluster. If you don't specify a ``RestoreType`` value, then the new DB cluster is restored as a full copy of the source DB cluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param scaling_configuration: The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster. This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property. Valid for: Aurora DB clusters only
        :param serverless_v2_scaling_configuration: The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster. This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property. Valid for: Aurora DB clusters only
        :param snapshot_identifier: The identifier for the DB snapshot or DB cluster snapshot to restore from. You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot. After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted. If you specify the ``SnapshotIdentifier`` property to restore a DB cluster (as opposed to specifying it for DB cluster updates), then don't specify the following properties: - ``GlobalClusterIdentifier`` - ``MasterUsername`` - ``MasterUserPassword`` - ``ReplicationSourceIdentifier`` - ``RestoreType`` - ``SourceDBClusterIdentifier`` - ``SourceRegion`` - ``StorageEncrypted`` (for an encrypted snapshot) - ``UseLatestRestorableTime`` Constraints: - Must match the identifier of an existing Snapshot. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param source_db_cluster_identifier: When restoring a DB cluster to a point in time, the identifier of the source DB cluster from which to restore. Constraints: - Must match the identifier of an existing DBCluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param source_region: The AWS Region which contains the source DB cluster when replicating a DB cluster. For example, ``us-east-1`` . Valid for: Aurora DB clusters only
        :param storage_encrypted: Indicates whether the DB cluster is encrypted. If you specify the ``KmsKeyId`` property, then you must enable encryption. If you specify the ``SourceDBClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB cluster is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB cluster to be encrypted, then don't set this property or set it to ``false`` . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param storage_type: The storage type to associate with the DB cluster. For information on storage types for Aurora DB clusters, see `Storage configurations for Amazon Aurora DB clusters <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Overview.StorageReliability.html#aurora-storage-type>`_ . For information on storage types for Multi-AZ DB clusters, see `Settings for creating Multi-AZ DB clusters <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/create-multi-az-db-cluster.html#create-multi-az-db-cluster-settings>`_ . This setting is required to create a Multi-AZ DB cluster. When specified for a Multi-AZ DB cluster, a value for the ``Iops`` parameter is required. Valid for Cluster Type: Aurora DB clusters and Multi-AZ DB clusters Valid Values: - Aurora DB clusters - ``aurora | aurora-iopt1`` - Multi-AZ DB clusters - ``io1`` Default: - Aurora DB clusters - ``aurora`` - Multi-AZ DB clusters - ``io1``
        :param tags: An optional array of key-value pairs to apply to this DB cluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param use_latest_restorable_time: A value that indicates whether to restore the DB cluster to the latest restorable backup time. By default, the DB cluster is not restored to the latest restorable backup time. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param vpc_security_group_ids: A list of EC2 VPC security groups to associate with this DB cluster. If you plan to update the resource, don't specify VPC security groups in a shared VPC. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1eb14b9dcc306eabcc2963c7b6ef9b87bf8d616bb5691dbc6656242be7c6bf28)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBClusterProps(
            allocated_storage=allocated_storage,
            associated_roles=associated_roles,
            auto_minor_version_upgrade=auto_minor_version_upgrade,
            availability_zones=availability_zones,
            backtrack_window=backtrack_window,
            backup_retention_period=backup_retention_period,
            copy_tags_to_snapshot=copy_tags_to_snapshot,
            database_name=database_name,
            db_cluster_identifier=db_cluster_identifier,
            db_cluster_instance_class=db_cluster_instance_class,
            db_cluster_parameter_group_name=db_cluster_parameter_group_name,
            db_instance_parameter_group_name=db_instance_parameter_group_name,
            db_subnet_group_name=db_subnet_group_name,
            db_system_id=db_system_id,
            deletion_protection=deletion_protection,
            domain=domain,
            domain_iam_role_name=domain_iam_role_name,
            enable_cloudwatch_logs_exports=enable_cloudwatch_logs_exports,
            enable_http_endpoint=enable_http_endpoint,
            enable_iam_database_authentication=enable_iam_database_authentication,
            engine=engine,
            engine_mode=engine_mode,
            engine_version=engine_version,
            global_cluster_identifier=global_cluster_identifier,
            iops=iops,
            kms_key_id=kms_key_id,
            manage_master_user_password=manage_master_user_password,
            master_username=master_username,
            master_user_password=master_user_password,
            master_user_secret=master_user_secret,
            monitoring_interval=monitoring_interval,
            monitoring_role_arn=monitoring_role_arn,
            network_type=network_type,
            performance_insights_enabled=performance_insights_enabled,
            performance_insights_kms_key_id=performance_insights_kms_key_id,
            performance_insights_retention_period=performance_insights_retention_period,
            port=port,
            preferred_backup_window=preferred_backup_window,
            preferred_maintenance_window=preferred_maintenance_window,
            publicly_accessible=publicly_accessible,
            replication_source_identifier=replication_source_identifier,
            restore_to_time=restore_to_time,
            restore_type=restore_type,
            scaling_configuration=scaling_configuration,
            serverless_v2_scaling_configuration=serverless_v2_scaling_configuration,
            snapshot_identifier=snapshot_identifier,
            source_db_cluster_identifier=source_db_cluster_identifier,
            source_region=source_region,
            storage_encrypted=storage_encrypted,
            storage_type=storage_type,
            tags=tags,
            use_latest_restorable_time=use_latest_restorable_time,
            vpc_security_group_ids=vpc_security_group_ids,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a900d9d9f5a1fec7e38e864e3ae07baa523eea91420e0ddee0d0a81b28e642df)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__66110d91f9570f1e38af512c1d77e760bc5605b8ea96fc492bd380c3a7a05049)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDbClusterArn")
    def attr_db_cluster_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) for the DB cluster.

        :cloudformationAttribute: DBClusterArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbClusterArn"))

    @builtins.property
    @jsii.member(jsii_name="attrDbClusterResourceId")
    def attr_db_cluster_resource_id(self) -> builtins.str:
        '''The AWS Region -unique, immutable identifier for the DB cluster.

        This identifier is found in AWS CloudTrail log entries whenever the KMS key for the DB cluster is accessed.

        :cloudformationAttribute: DBClusterResourceId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbClusterResourceId"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointAddress")
    def attr_endpoint_address(self) -> builtins.str:
        '''The connection endpoint for the DB cluster.

        For example: ``mystack-mydbcluster-123456789012.us-east-2.rds.amazonaws.com``

        :cloudformationAttribute: Endpoint.Address
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointAddress"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointPort")
    def attr_endpoint_port(self) -> builtins.str:
        '''The port number that will accept connections on this DB cluster.

        For example: ``3306``

        :cloudformationAttribute: Endpoint.Port
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointPort"))

    @builtins.property
    @jsii.member(jsii_name="attrMasterUserSecretSecretArn")
    def attr_master_user_secret_secret_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the secret.

        :cloudformationAttribute: MasterUserSecret.SecretArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMasterUserSecretSecretArn"))

    @builtins.property
    @jsii.member(jsii_name="attrReadEndpointAddress")
    def attr_read_endpoint_address(self) -> builtins.str:
        '''The reader endpoint for the DB cluster.

        For example: ``mystack-mydbcluster-ro-123456789012.us-east-2.rds.amazonaws.com``

        :cloudformationAttribute: ReadEndpoint.Address
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrReadEndpointAddress"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''An optional array of key-value pairs to apply to this DB cluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="allocatedStorage")
    def allocated_storage(self) -> typing.Optional[jsii.Number]:
        '''The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.

        Valid for Cluster Type: Multi-AZ DB clusters only

        This setting is required to create a Multi-AZ DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-allocatedstorage
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "allocatedStorage"))

    @allocated_storage.setter
    def allocated_storage(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b26bf63f703c49cf31086603eee69e2db43fbf32c1853bc110ebaf79f1dab39a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocatedStorage", value)

    @builtins.property
    @jsii.member(jsii_name="associatedRoles")
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.DBClusterRoleProperty"]]]]:
        '''Provides a list of the AWS Identity and Access Management (IAM) roles that are associated with the DB cluster.

        IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other Amazon Web Services on your behalf.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-associatedroles
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.DBClusterRoleProperty"]]]], jsii.get(self, "associatedRoles"))

    @associated_roles.setter
    def associated_roles(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.DBClusterRoleProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__de474bccf1889322c51196d8d4ba255be11c15b5fc0aedc1544af2907fa6839c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "associatedRoles", value)

    @builtins.property
    @jsii.member(jsii_name="autoMinorVersionUpgrade")
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether minor engine upgrades are applied automatically to the DB cluster during the maintenance window.

        By default, minor engine upgrades are applied automatically.

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-autominorversionupgrade
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "autoMinorVersionUpgrade"))

    @auto_minor_version_upgrade.setter
    def auto_minor_version_upgrade(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b57470a09145bed53fc4a1308635f9b075d8f9123010ef7c1dc3b85c549e681)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoMinorVersionUpgrade", value)

    @builtins.property
    @jsii.member(jsii_name="availabilityZones")
    def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of Availability Zones (AZs) where instances in the DB cluster can be created.

        For information on AWS Regions and Availability Zones, see `Choosing the Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-availabilityzones
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "availabilityZones"))

    @availability_zones.setter
    def availability_zones(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b64e0486b077769f0d7b122ad62924be65560c00ad9f13d222f1ab55fd1051ba)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "availabilityZones", value)

    @builtins.property
    @jsii.member(jsii_name="backtrackWindow")
    def backtrack_window(self) -> typing.Optional[jsii.Number]:
        '''The target backtrack window, in seconds. To disable backtracking, set this value to 0.

        .. epigraph::

           Currently, Backtrack is only supported for Aurora MySQL DB clusters.

        Default: 0

        Constraints:

        - If specified, this value must be set to a number from 0 to 259,200 (72 hours).

        Valid for: Aurora MySQL DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backtrackwindow
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "backtrackWindow"))

    @backtrack_window.setter
    def backtrack_window(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d3f12b5a2273631891f7cf85219c21cc48606155594350de01bdcdd7a4cf403b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "backtrackWindow", value)

    @builtins.property
    @jsii.member(jsii_name="backupRetentionPeriod")
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days for which automated backups are retained.

        Default: 1

        Constraints:

        - Must be a value from 1 to 35

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backupretentionperiod
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "backupRetentionPeriod"))

    @backup_retention_period.setter
    def backup_retention_period(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8dbb89a95c41c7cbbc3c4804ae23a1ef19f4b4b61e97878b54d704b3dc955eb7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "backupRetentionPeriod", value)

    @builtins.property
    @jsii.member(jsii_name="copyTagsToSnapshot")
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to copy all tags from the DB cluster to snapshots of the DB cluster.

        The default is not to copy them.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-copytagstosnapshot
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "copyTagsToSnapshot"))

    @copy_tags_to_snapshot.setter
    def copy_tags_to_snapshot(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7aaebe942bef486fa4f2def6b116a47a3513d6909fa29d9fdd20eb31f9d9dc03)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "copyTagsToSnapshot", value)

    @builtins.property
    @jsii.member(jsii_name="databaseName")
    def database_name(self) -> typing.Optional[builtins.str]:
        '''The name of your database.

        If you don't provide a name, then Amazon RDS won't create a database in this DB cluster. For naming constraints, see `Naming Constraints <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-databasename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "databaseName"))

    @database_name.setter
    def database_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__133622b74470290a34fd20f1677f713d047e14ddfc8138b0693817c347f0904f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "databaseName", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterIdentifier")
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The DB cluster identifier. This parameter is stored as a lowercase string.

        Constraints:

        - Must contain from 1 to 63 letters, numbers, or hyphens.
        - First character must be a letter.
        - Can't end with a hyphen or contain two consecutive hyphens.

        Example: ``my-cluster1``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterIdentifier"))

    @db_cluster_identifier.setter
    def db_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__55617043df14666e3665f3947be2c7d08d730bd5dafad5d1b2b7394f6af7e64a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterInstanceClass")
    def db_cluster_instance_class(self) -> typing.Optional[builtins.str]:
        '''The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example ``db.m6gd.xlarge`` . Not all DB instance classes are available in all AWS Regions , or for all database engines.

        For the full list of DB instance classes and availability for your engine, see `DB instance class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide* .

        This setting is required to create a Multi-AZ DB cluster.

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterinstanceclass
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterInstanceClass"))

    @db_cluster_instance_class.setter
    def db_cluster_instance_class(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4698294b15845f23b998066d033e773c1f919d48f4dd8d3547ae9465ca4c8623)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterInstanceClass", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterParameterGroupName")
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB cluster parameter group to associate with this DB cluster.

        .. epigraph::

           If you apply a parameter group to an existing DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting.

           If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started.

        To list all of the available DB cluster parameter group names, use the following command:

        ``aws rds describe-db-cluster-parameter-groups --query "DBClusterParameterGroups[].DBClusterParameterGroupName" --output text``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterParameterGroupName"))

    @db_cluster_parameter_group_name.setter
    def db_cluster_parameter_group_name(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a8ce71543e697fea0216ebbb847f7ea81a4524c91c47b22e9025e7907c8d944e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterParameterGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="dbInstanceParameterGroupName")
    def db_instance_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB parameter group to apply to all instances of the DB cluster.

        .. epigraph::

           When you apply a parameter group using the ``DBInstanceParameterGroupName`` parameter, the DB cluster isn't rebooted automatically. Also, parameter changes are applied immediately rather than during the next maintenance window.

        Default: The existing name setting

        Constraints:

        - The DB parameter group must be in the same DB parameter group family as this DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbinstanceparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbInstanceParameterGroupName"))

    @db_instance_parameter_group_name.setter
    def db_instance_parameter_group_name(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__030bf42bf8de41396ecd68ad44900e73fba0d7d7b572e13c4693f5c399f4ece6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbInstanceParameterGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="dbSubnetGroupName")
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''A DB subnet group that you want to associate with this DB cluster.

        If you are restoring a DB cluster to a point in time with ``RestoreType`` set to ``copy-on-write`` , and don't specify a DB subnet group name, then the DB cluster is restored with a default DB subnet group.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsubnetgroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSubnetGroupName"))

    @db_subnet_group_name.setter
    def db_subnet_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2659ab1026eaa69df2aeac64c6195777b6de1b6c75de8c797bfe2b7420168e99)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSubnetGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="dbSystemId")
    def db_system_id(self) -> typing.Optional[builtins.str]:
        '''Reserved for future use.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsystemid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSystemId"))

    @db_system_id.setter
    def db_system_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__41f2773ed2c93659b8bf06ee2830b9675bbe867fb30ad2e89f77f72a45a3b0e0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSystemId", value)

    @builtins.property
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether the DB cluster has deletion protection enabled.

        The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-deletionprotection
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "deletionProtection"))

    @deletion_protection.setter
    def deletion_protection(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a96b132db813926d0ef31c223b961804a782255c93a76d8ab91de8c2ca7b0f35)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deletionProtection", value)

    @builtins.property
    @jsii.member(jsii_name="domain")
    def domain(self) -> typing.Optional[builtins.str]:
        '''Indicates the directory ID of the Active Directory to create the DB cluster.

        For Amazon Aurora DB clusters, Amazon RDS can use Kerberos authentication to authenticate users that connect to the DB cluster.

        For more information, see `Kerberos authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/kerberos-authentication.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-domain
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domain"))

    @domain.setter
    def domain(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4c2ea957168641e8393a37255bd4ba2d434b8e535b1d5a9ab7fa27f77a5170ce)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domain", value)

    @builtins.property
    @jsii.member(jsii_name="domainIamRoleName")
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        '''Specifies the name of the IAM role to use when making API calls to the Directory Service.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-domainiamrolename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainIamRoleName"))

    @domain_iam_role_name.setter
    def domain_iam_role_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c5de45b0b0fd43e3ef9a298cddfe7e10c64f9f20a7d58aa4ee83b083b4846a61)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainIamRoleName", value)

    @builtins.property
    @jsii.member(jsii_name="enableCloudwatchLogsExports")
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of log types that need to be enabled for exporting to CloudWatch Logs.

        The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Aurora User Guide* .

        *Aurora MySQL*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Aurora PostgreSQL*

        Valid values: ``postgresql``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablecloudwatchlogsexports
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "enableCloudwatchLogsExports"))

    @enable_cloudwatch_logs_exports.setter
    def enable_cloudwatch_logs_exports(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f36bc814efe003a4b1230628ace46a0c376dd67efae9edd14e7b2455cddae236)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableCloudwatchLogsExports", value)

    @builtins.property
    @jsii.member(jsii_name="enableHttpEndpoint")
    def enable_http_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to enable the HTTP endpoint for an Aurora Serverless DB cluster.

        By default, the HTTP endpoint is disabled.

        When enabled, the HTTP endpoint provides a connectionless web service API for running SQL queries on the Aurora Serverless DB cluster. You can also query your database from inside the RDS console with the query editor.

        For more information, see `Using the Data API for Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablehttpendpoint
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enableHttpEndpoint"))

    @enable_http_endpoint.setter
    def enable_http_endpoint(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a15b1a5e04501a5e471c4c72ca3df92d9893d051244504b2eb71d63cca41b49)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableHttpEndpoint", value)

    @builtins.property
    @jsii.member(jsii_name="enableIamDatabaseAuthentication")
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        By default, mapping is disabled.

        For more information, see `IAM Database Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon Aurora User Guide.*

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enableiamdatabaseauthentication
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enableIamDatabaseAuthentication"))

    @enable_iam_database_authentication.setter
    def enable_iam_database_authentication(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7946d487f5118b449bf2e0669e304a130a394dee094f86279a80a19ef45afaf6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableIamDatabaseAuthentication", value)

    @builtins.property
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine to be used for this DB cluster.

        Valid Values:

        - ``aurora-mysql``
        - ``aurora-postgresql``
        - ``mysql``
        - ``postgres``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engine
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engine"))

    @engine.setter
    def engine(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b719515b7b61b9703874b8014129227da07665daf27a6c705c2777e43612e46)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engine", value)

    @builtins.property
    @jsii.member(jsii_name="engineMode")
    def engine_mode(self) -> typing.Optional[builtins.str]:
        '''The DB engine mode of the DB cluster, either ``provisioned`` or ``serverless`` .

        The ``serverless`` engine mode only supports Aurora Serverless v1.

        Limitations and requirements apply to some DB engine modes. For more information, see the following sections in the *Amazon Aurora User Guide* :

        - `Limitations of Aurora Serverless v1 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations>`_
        - `Requirements for Aurora Serverless v2 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.requirements.html>`_
        - `Limitations of parallel query <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-parallel-query.html#aurora-mysql-parallel-query-limitations>`_
        - `Limitations of Aurora global databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html#aurora-global-database.limitations>`_

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enginemode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engineMode"))

    @engine_mode.setter
    def engine_mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1720afb1a1233807859444f6991dbfb4334fef08adc5e3a2104ed03f4269a766)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineMode", value)

    @builtins.property
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The version number of the database engine to use.

        To list all of the available engine versions for Aurora MySQL version 2 (5.7-compatible) and version 3 (8.0-compatible), use the following command:

        ``aws rds describe-db-engine-versions --engine aurora-mysql --query "DBEngineVersions[].EngineVersion"``

        You can supply either ``5.7`` or ``8.0`` to use the default engine version for Aurora MySQL version 2 or version 3, respectively.

        To list all of the available engine versions for Aurora PostgreSQL, use the following command:

        ``aws rds describe-db-engine-versions --engine aurora-postgresql --query "DBEngineVersions[].EngineVersion"``

        To list all of the available engine versions for RDS for MySQL, use the following command:

        ``aws rds describe-db-engine-versions --engine mysql --query "DBEngineVersions[].EngineVersion"``

        To list all of the available engine versions for RDS for PostgreSQL, use the following command:

        ``aws rds describe-db-engine-versions --engine postgres --query "DBEngineVersions[].EngineVersion"``

        *Aurora MySQL*

        For information, see `Database engine updates for Amazon Aurora MySQL <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Updates.html>`_ in the *Amazon Aurora User Guide* .

        *Aurora PostgreSQL*

        For information, see `Amazon Aurora PostgreSQL releases and engine versions <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.20180305.html>`_ in the *Amazon Aurora User Guide* .

        *MySQL*

        For information, see `Amazon RDS for MySQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide* .

        *PostgreSQL*

        For information, see `Amazon RDS for PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts>`_ in the *Amazon RDS User Guide* .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engineVersion"))

    @engine_version.setter
    def engine_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6fb241407bb17623e1e6f2f9310a1b3ebac9240cdbdcf5b152e7df5cc0a21488)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineVersion", value)

    @builtins.property
    @jsii.member(jsii_name="globalClusterIdentifier")
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''If you are configuring an Aurora global database cluster and want your Aurora DB cluster to be a secondary member in the global database cluster, specify the global cluster ID of the global database cluster.

        To define the primary database cluster of the global cluster, use the `AWS::RDS::GlobalCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html>`_ resource.

        If you aren't configuring a global database cluster, don't specify this property.
        .. epigraph::

           To remove the DB cluster from a global database cluster, specify an empty value for the ``GlobalClusterIdentifier`` property.

        For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-globalclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "globalClusterIdentifier"))

    @global_cluster_identifier.setter
    def global_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__32506437b25c6239bdf71df465671ddee1a1e4067ca0e6d79b7399d81b6b7f45)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "globalClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="iops")
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster.

        For information about valid IOPS values, see `Provisioned IOPS storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* .

        This setting is required to create a Multi-AZ DB cluster.

        Valid for Cluster Type: Multi-AZ DB clusters only

        Constraints:

        - Must be a multiple between .5 and 50 of the storage amount for the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-iops
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "iops"))

    @iops.setter
    def iops(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d385d85f593ce39d43b397540e5fb9f90e7feb955d365ea85b413633bb27d03d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "iops", value)

    @builtins.property
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the AWS KMS key that is used to encrypt the database instances in the DB cluster, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` .

        If you enable the ``StorageEncrypted`` property but don't specify this property, the default KMS key is used. If you specify this property, you must set the ``StorageEncrypted`` property to ``true`` .

        If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-kmskeyid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyId"))

    @kms_key_id.setter
    def kms_key_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8d0fdd0070aafd30ad6bcb3a42b259df4088aada78fe5a4669ad688f585b7f4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyId", value)

    @builtins.property
    @jsii.member(jsii_name="manageMasterUserPassword")
    def manage_master_user_password(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to manage the master user password with AWS Secrets Manager.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

        Valid for Cluster Type: Aurora DB clusters and Multi-AZ DB clusters

        Constraints:

        - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-managemasteruserpassword
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "manageMasterUserPassword"))

    @manage_master_user_password.setter
    def manage_master_user_password(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ca876954b5556832b0c941ce0f6ef94a61eb0cd960bb6e8a0106369ccc13a12)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "manageMasterUserPassword", value)

    @builtins.property
    @jsii.member(jsii_name="masterUsername")
    def master_username(self) -> typing.Optional[builtins.str]:
        '''The name of the master user for the DB cluster.

        .. epigraph::

           If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "masterUsername"))

    @master_username.setter
    def master_username(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0dd3cba91c2c8db72b7eb5eeec9e8fbd7f06d277b876e3e89834d03071aaf6d8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUsername", value)

    @builtins.property
    @jsii.member(jsii_name="masterUserPassword")
    def master_user_password(self) -> typing.Optional[builtins.str]:
        '''The master password for the DB instance.

        .. epigraph::

           If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masteruserpassword
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "masterUserPassword"))

    @master_user_password.setter
    def master_user_password(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__226485ff852dd8e2c2c05d1e84d6d461e5fa073f85f9c00a6f3481b72ef96495)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUserPassword", value)

    @builtins.property
    @jsii.member(jsii_name="masterUserSecret")
    def master_user_secret(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.MasterUserSecretProperty"]]:
        '''The secret managed by RDS in AWS Secrets Manager for the master user password.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusersecret
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.MasterUserSecretProperty"]], jsii.get(self, "masterUserSecret"))

    @master_user_secret.setter
    def master_user_secret(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.MasterUserSecretProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca6746fea0781a1a399dcc243d901ddf185eaeeaa1c195a09f668c467ff7485f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUserSecret", value)

    @builtins.property
    @jsii.member(jsii_name="monitoringInterval")
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        '''The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster.

        To turn off collecting Enhanced Monitoring metrics, specify ``0`` .

        If ``MonitoringRoleArn`` is specified, also set ``MonitoringInterval`` to a value other than ``0`` .

        Valid for Cluster Type: Multi-AZ DB clusters only

        Valid Values: ``0 | 1 | 5 | 10 | 15 | 30 | 60``

        Default: ``0``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-monitoringinterval
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "monitoringInterval"))

    @monitoring_interval.setter
    def monitoring_interval(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca22a124f153ad06adcc2d086a7b548cb1efcf6716b8a7b1f4f3241a11ab9538)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "monitoringInterval", value)

    @builtins.property
    @jsii.member(jsii_name="monitoringRoleArn")
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs.

        An example is ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting up and enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* .

        If ``MonitoringInterval`` is set to a value other than ``0`` , supply a ``MonitoringRoleArn`` value.

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-monitoringrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "monitoringRoleArn"))

    @monitoring_role_arn.setter
    def monitoring_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a0ed903ae6c409f8db9f60a46656ecd36777befccd23abacc27700b6b56ac0f8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "monitoringRoleArn", value)

    @builtins.property
    @jsii.member(jsii_name="networkType")
    def network_type(self) -> typing.Optional[builtins.str]:
        '''The network type of the DB cluster.

        Valid values:

        - ``IPV4``
        - ``DUAL``

        The network type is determined by the ``DBSubnetGroup`` specified for the DB cluster. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ).

        For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon Aurora User Guide.*

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-networktype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkType"))

    @network_type.setter
    def network_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d91088eeabeff13b3d044e9d5767593b2142a6f71342b829d605562d87e4dab)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "networkType", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsEnabled")
    def performance_insights_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to turn on Performance Insights for the DB cluster.

        For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* .

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightsenabled
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "performanceInsightsEnabled"))

    @performance_insights_enabled.setter
    def performance_insights_enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__66cd4e8195f60ef500150681ff7782ad12c06bfe45fa29747fcfabdccca5ba66)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsEnabled", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsKmsKeyId")
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for encryption of Performance Insights data.

        The AWS KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.

        If you don't specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account . Your AWS account has a different default KMS key for each AWS Region .

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightskmskeyid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "performanceInsightsKmsKeyId"))

    @performance_insights_kms_key_id.setter
    def performance_insights_kms_key_id(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf157608d0a77dbe9fb6c7ea88b4e939cfbeafdeede50421bf1570b1de2e8aa2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsKmsKeyId", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsRetentionPeriod")
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days to retain Performance Insights data.

        Valid for Cluster Type: Multi-AZ DB clusters only

        Valid Values:

        - ``7``
        - *month* * 31, where *month* is a number of months from 1-23. Examples: ``93`` (3 months * 31), ``341`` (11 months * 31), ``589`` (19 months * 31)
        - ``731``

        Default: ``7`` days

        If you specify a retention period that isn't valid, such as ``94`` , Amazon RDS issues an error.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightsretentionperiod
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "performanceInsightsRetentionPeriod"))

    @performance_insights_retention_period.setter
    def performance_insights_retention_period(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd68a178f4f64cdaf56c7b6d5478f034a1449271243c8949641c0f1d1278e6e7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsRetentionPeriod", value)

    @builtins.property
    @jsii.member(jsii_name="port")
    def port(self) -> typing.Optional[jsii.Number]:
        '''The port number on which the DB instances in the DB cluster accept connections.

        Default:

        - When ``EngineMode`` is ``provisioned`` , ``3306`` (for both Aurora MySQL and Aurora PostgreSQL)
        - When ``EngineMode`` is ``serverless`` :
        - ``3306`` when ``Engine`` is ``aurora`` or ``aurora-mysql``
        - ``5432`` when ``Engine`` is ``aurora-postgresql``

        .. epigraph::

           The ``No interruption`` on update behavior only applies to DB clusters. If you are updating a DB instance, see `Port <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port>`_ for the AWS::RDS::DBInstance resource.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-port
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "port"))

    @port.setter
    def port(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__07fd869af6e08eb8d08ffcb2930cfc8084e9af885f80caa554c1017a14533953)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "port", value)

    @builtins.property
    @jsii.member(jsii_name="preferredBackupWindow")
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        '''The daily time range during which automated backups are created.

        For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.Backups.BackupWindow>`_ in the *Amazon Aurora User Guide.*

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi`` .
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredbackupwindow
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "preferredBackupWindow"))

    @preferred_backup_window.setter
    def preferred_backup_window(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__46ed70953175e13a4a0e3215c1f628258e678b4b33374b7bcab3055aa26f43b5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preferredBackupWindow", value)

    @builtins.property
    @jsii.member(jsii_name="preferredMaintenanceWindow")
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        '''The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).

        Format: ``ddd:hh24:mi-ddd:hh24:mi``

        The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Cluster Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow.Aurora>`_ in the *Amazon Aurora User Guide.*

        Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun.

        Constraints: Minimum 30-minute window.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredmaintenancewindow
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "preferredMaintenanceWindow"))

    @preferred_maintenance_window.setter
    def preferred_maintenance_window(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__99907fd311daffb4e77200a4de40cf06114de840559ce190f327fae84815447d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preferredMaintenanceWindow", value)

    @builtins.property
    @jsii.member(jsii_name="publiclyAccessible")
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the DB cluster is publicly accessible.

        When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it uses. That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it.

        When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to a private IP address.

        Valid for Cluster Type: Multi-AZ DB clusters only

        Default: The default behavior varies depending on whether ``DBSubnetGroupName`` is specified.

        If ``DBSubnetGroupName`` isn't specified, and ``PubliclyAccessible`` isn't specified, the following applies:

        - If the default VPC in the target Region doesn’t have an internet gateway attached to it, the DB cluster is private.
        - If the default VPC in the target Region has an internet gateway attached to it, the DB cluster is public.

        If ``DBSubnetGroupName`` is specified, and ``PubliclyAccessible`` isn't specified, the following applies:

        - If the subnets are part of a VPC that doesn’t have an internet gateway attached to it, the DB cluster is private.
        - If the subnets are part of a VPC that has an internet gateway attached to it, the DB cluster is public.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-publiclyaccessible
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "publiclyAccessible"))

    @publicly_accessible.setter
    def publicly_accessible(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3a0b6d6f1bc188271336aaf17873ab87aacd8647ebe3cf297378988a529d0cc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "publiclyAccessible", value)

    @builtins.property
    @jsii.member(jsii_name="replicationSourceIdentifier")
    def replication_source_identifier(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the source DB instance or DB cluster if this DB cluster is created as a read replica.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-replicationsourceidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "replicationSourceIdentifier"))

    @replication_source_identifier.setter
    def replication_source_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__56b1a90f3bce5fc91bac762e68220d7408521d009b606dc505619b0c7ebd3c64)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "replicationSourceIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="restoreToTime")
    def restore_to_time(self) -> typing.Optional[builtins.str]:
        '''The date and time to restore the DB cluster to.

        Valid Values: Value must be a time in Universal Coordinated Time (UTC) format

        Constraints:

        - Must be before the latest restorable time for the DB instance
        - Must be specified if ``UseLatestRestorableTime`` parameter isn't provided
        - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled
        - Can't be specified if the ``RestoreType`` parameter is ``copy-on-write``

        Example: ``2015-03-07T23:45:00Z``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-restoretotime
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "restoreToTime"))

    @restore_to_time.setter
    def restore_to_time(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__968c8eea9022fda98e791ba1ed74ac7968f89dc147e76f278478ff3ed225bb72)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restoreToTime", value)

    @builtins.property
    @jsii.member(jsii_name="restoreType")
    def restore_type(self) -> typing.Optional[builtins.str]:
        '''The type of restore to be performed. You can specify one of the following values:.

        - ``full-copy`` - The new DB cluster is restored as a full copy of the source DB cluster.
        - ``copy-on-write`` - The new DB cluster is restored as a clone of the source DB cluster.

        If you don't specify a ``RestoreType`` value, then the new DB cluster is restored as a full copy of the source DB cluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-restoretype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "restoreType"))

    @restore_type.setter
    def restore_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a357bfe90277fc3992eb9dae141ee8d090d0466086def5838c34cce565f37fa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restoreType", value)

    @builtins.property
    @jsii.member(jsii_name="scalingConfiguration")
    def scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.ScalingConfigurationProperty"]]:
        '''The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster.

        This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-scalingconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.ScalingConfigurationProperty"]], jsii.get(self, "scalingConfiguration"))

    @scaling_configuration.setter
    def scaling_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.ScalingConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8471c5722f16288bc2a5c637b511c81be4cd4b09f3f7899592b7fd02808dba4c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "scalingConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="serverlessV2ScalingConfiguration")
    def serverless_v2_scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.ServerlessV2ScalingConfigurationProperty"]]:
        '''The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster.

        This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-serverlessv2scalingconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.ServerlessV2ScalingConfigurationProperty"]], jsii.get(self, "serverlessV2ScalingConfiguration"))

    @serverless_v2_scaling_configuration.setter
    def serverless_v2_scaling_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBCluster.ServerlessV2ScalingConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0876352acbd9ca1b842c3ece417cf0c5ed155f8c160597474dd24e11ac318ad)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serverlessV2ScalingConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="snapshotIdentifier")
    def snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier for the DB snapshot or DB cluster snapshot to restore from.

        You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot.

        After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted.

        If you specify the ``SnapshotIdentifier`` property to restore a DB cluster (as opposed to specifying it for DB cluster updates), then don't specify the following properties:

        - ``GlobalClusterIdentifier``
        - ``MasterUsername``
        - ``MasterUserPassword``
        - ``ReplicationSourceIdentifier``
        - ``RestoreType``
        - ``SourceDBClusterIdentifier``
        - ``SourceRegion``
        - ``StorageEncrypted`` (for an encrypted snapshot)
        - ``UseLatestRestorableTime``

        Constraints:

        - Must match the identifier of an existing Snapshot.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-snapshotidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "snapshotIdentifier"))

    @snapshot_identifier.setter
    def snapshot_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__25045b6189370a8d94ebe994abba5de432cd9df49345a99151f42b12416f3025)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "snapshotIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbClusterIdentifier")
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''When restoring a DB cluster to a point in time, the identifier of the source DB cluster from which to restore.

        Constraints:

        - Must match the identifier of an existing DBCluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourcedbclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbClusterIdentifier"))

    @source_db_cluster_identifier.setter
    def source_db_cluster_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df647406eec5ba2e24bb07694e286def32cd947e8728ac6ec3727fd145515f2f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="sourceRegion")
    def source_region(self) -> typing.Optional[builtins.str]:
        '''The AWS Region which contains the source DB cluster when replicating a DB cluster. For example, ``us-east-1`` .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourceregion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceRegion"))

    @source_region.setter
    def source_region(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2ee6860d8d1d897c5c878ca6693329e97bdddac0cb37f822ceb26aabdb5b78fb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceRegion", value)

    @builtins.property
    @jsii.member(jsii_name="storageEncrypted")
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether the DB cluster is encrypted.

        If you specify the ``KmsKeyId`` property, then you must enable encryption.

        If you specify the ``SourceDBClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB cluster is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB cluster to be encrypted, then don't set this property or set it to ``false`` .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storageencrypted
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "storageEncrypted"))

    @storage_encrypted.setter
    def storage_encrypted(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__723ea3f9c61b8480429b9258611e755a66af6ea0e34bcf9fb1b6bcb4a5d1fcfd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageEncrypted", value)

    @builtins.property
    @jsii.member(jsii_name="storageType")
    def storage_type(self) -> typing.Optional[builtins.str]:
        '''The storage type to associate with the DB cluster.

        For information on storage types for Aurora DB clusters, see `Storage configurations for Amazon Aurora DB clusters <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Overview.StorageReliability.html#aurora-storage-type>`_ . For information on storage types for Multi-AZ DB clusters, see `Settings for creating Multi-AZ DB clusters <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/create-multi-az-db-cluster.html#create-multi-az-db-cluster-settings>`_ .

        This setting is required to create a Multi-AZ DB cluster.

        When specified for a Multi-AZ DB cluster, a value for the ``Iops`` parameter is required.

        Valid for Cluster Type: Aurora DB clusters and Multi-AZ DB clusters

        Valid Values:

        - Aurora DB clusters - ``aurora | aurora-iopt1``
        - Multi-AZ DB clusters - ``io1``

        Default:

        - Aurora DB clusters - ``aurora``
        - Multi-AZ DB clusters - ``io1``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storagetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "storageType"))

    @storage_type.setter
    def storage_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6ea7aa6db917d8cfeff95d9bf2e01ba5d459e1c34f195572dd025eb91defd280)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageType", value)

    @builtins.property
    @jsii.member(jsii_name="useLatestRestorableTime")
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to restore the DB cluster to the latest restorable backup time.

        By default, the DB cluster is not restored to the latest restorable backup time.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-uselatestrestorabletime
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "useLatestRestorableTime"))

    @use_latest_restorable_time.setter
    def use_latest_restorable_time(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2233cd63167d19c1f7b632ba4a2ecd1142eba944359f46221ba7e9394e0501c6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "useLatestRestorableTime", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSecurityGroupIds")
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of EC2 VPC security groups to associate with this DB cluster.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-vpcsecuritygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "vpcSecurityGroupIds"))

    @vpc_security_group_ids.setter
    def vpc_security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9157d76fdc416fb2667fc663c3e08fe740d3b43b5c2fb5bc0da881dc4fd3d1c3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSecurityGroupIds", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBCluster.DBClusterRoleProperty",
        jsii_struct_bases=[],
        name_mapping={"role_arn": "roleArn", "feature_name": "featureName"},
    )
    class DBClusterRoleProperty:
        def __init__(
            self,
            *,
            role_arn: builtins.str,
            feature_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Describes an AWS Identity and Access Management (IAM) role that is associated with a DB cluster.

            :param role_arn: The Amazon Resource Name (ARN) of the IAM role that is associated with the DB cluster.
            :param feature_name: The name of the feature associated with the AWS Identity and Access Management (IAM) role. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other AWS services on your behalf. For the list of supported feature names, see the ``SupportedFeatureNames`` description in `DBEngineVersion <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html>`_ in the *Amazon RDS API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-dbclusterrole.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                d_bCluster_role_property = rds.CfnDBCluster.DBClusterRoleProperty(
                    role_arn="roleArn",
                
                    # the properties below are optional
                    feature_name="featureName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f433da378490e4c47bb4dc2c638ebcc057288a2e3fe24ac2927d3a7e6953fdcc)
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
                check_type(argname="argument feature_name", value=feature_name, expected_type=type_hints["feature_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "role_arn": role_arn,
            }
            if feature_name is not None:
                self._values["feature_name"] = feature_name

        @builtins.property
        def role_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the IAM role that is associated with the DB cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-dbclusterrole.html#cfn-rds-dbcluster-dbclusterrole-rolearn
            '''
            result = self._values.get("role_arn")
            assert result is not None, "Required property 'role_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def feature_name(self) -> typing.Optional[builtins.str]:
            '''The name of the feature associated with the AWS Identity and Access Management (IAM) role.

            IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other AWS services on your behalf. For the list of supported feature names, see the ``SupportedFeatureNames`` description in `DBEngineVersion <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html>`_ in the *Amazon RDS API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-dbclusterrole.html#cfn-rds-dbcluster-dbclusterrole-featurename
            '''
            result = self._values.get("feature_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DBClusterRoleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBCluster.EndpointProperty",
        jsii_struct_bases=[],
        name_mapping={"address": "address", "port": "port"},
    )
    class EndpointProperty:
        def __init__(
            self,
            *,
            address: typing.Optional[builtins.str] = None,
            port: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``Endpoint`` return value specifies the connection endpoint for the primary instance of the DB cluster.

            :param address: Specifies the connection endpoint for the primary instance of the DB cluster.
            :param port: Specifies the port that the database engine is listening on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-endpoint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                endpoint_property = rds.CfnDBCluster.EndpointProperty(
                    address="address",
                    port="port"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d1fc6079fba3f65792cab567c2accd401fe3294c8372c1f74a72fd997a355436)
                check_type(argname="argument address", value=address, expected_type=type_hints["address"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if address is not None:
                self._values["address"] = address
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def address(self) -> typing.Optional[builtins.str]:
            '''Specifies the connection endpoint for the primary instance of the DB cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-endpoint.html#cfn-rds-dbcluster-endpoint-address
            '''
            result = self._values.get("address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[builtins.str]:
            '''Specifies the port that the database engine is listening on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-endpoint.html#cfn-rds-dbcluster-endpoint-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBCluster.MasterUserSecretProperty",
        jsii_struct_bases=[],
        name_mapping={"kms_key_id": "kmsKeyId", "secret_arn": "secretArn"},
    )
    class MasterUserSecretProperty:
        def __init__(
            self,
            *,
            kms_key_id: typing.Optional[builtins.str] = None,
            secret_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``MasterUserSecret`` return value specifies the secret managed by RDS in AWS Secrets Manager for the master user password.

            For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

            :param kms_key_id: The AWS KMS key identifier that is used to encrypt the secret.
            :param secret_arn: The Amazon Resource Name (ARN) of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-masterusersecret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                master_user_secret_property = rds.CfnDBCluster.MasterUserSecretProperty(
                    kms_key_id="kmsKeyId",
                    secret_arn="secretArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__68c4ee6dfa3d05f95cf6f2d1c8afa7072dfd1b240604e0bcdd23c34045ca2b10)
                check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
                check_type(argname="argument secret_arn", value=secret_arn, expected_type=type_hints["secret_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if secret_arn is not None:
                self._values["secret_arn"] = secret_arn

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''The AWS KMS key identifier that is used to encrypt the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-masterusersecret.html#cfn-rds-dbcluster-masterusersecret-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def secret_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-masterusersecret.html#cfn-rds-dbcluster-masterusersecret-secretarn
            '''
            result = self._values.get("secret_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MasterUserSecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBCluster.ReadEndpointProperty",
        jsii_struct_bases=[],
        name_mapping={"address": "address"},
    )
    class ReadEndpointProperty:
        def __init__(self, *, address: typing.Optional[builtins.str] = None) -> None:
            '''The ``ReadEndpoint`` return value specifies the reader endpoint for the DB cluster.

            The reader endpoint for a DB cluster load-balances connections across the Aurora Replicas that are available in a DB cluster. As clients request new connections to the reader endpoint, Aurora distributes the connection requests among the Aurora Replicas in the DB cluster. This functionality can help balance your read workload across multiple Aurora Replicas in your DB cluster.

            If a failover occurs, and the Aurora Replica that you are connected to is promoted to be the primary instance, your connection is dropped. To continue sending your read workload to other Aurora Replicas in the cluster, you can then reconnect to the reader endpoint.

            For more information about Aurora endpoints, see `Amazon Aurora connection management <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Overview.Endpoints.html>`_ in the *Amazon Aurora User Guide* .

            :param address: The host address of the reader endpoint.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-readendpoint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                read_endpoint_property = rds.CfnDBCluster.ReadEndpointProperty(
                    address="address"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6daa6513ed544869f5247ab992c8a52165bd4722cffcea9a6f7d2280d251b4cf)
                check_type(argname="argument address", value=address, expected_type=type_hints["address"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if address is not None:
                self._values["address"] = address

        @builtins.property
        def address(self) -> typing.Optional[builtins.str]:
            '''The host address of the reader endpoint.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-readendpoint.html#cfn-rds-dbcluster-readendpoint-address
            '''
            result = self._values.get("address")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ReadEndpointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBCluster.ScalingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auto_pause": "autoPause",
            "max_capacity": "maxCapacity",
            "min_capacity": "minCapacity",
            "seconds_before_timeout": "secondsBeforeTimeout",
            "seconds_until_auto_pause": "secondsUntilAutoPause",
            "timeout_action": "timeoutAction",
        },
    )
    class ScalingConfigurationProperty:
        def __init__(
            self,
            *,
            auto_pause: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            max_capacity: typing.Optional[jsii.Number] = None,
            min_capacity: typing.Optional[jsii.Number] = None,
            seconds_before_timeout: typing.Optional[jsii.Number] = None,
            seconds_until_auto_pause: typing.Optional[jsii.Number] = None,
            timeout_action: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster.

            For more information, see `Using Amazon Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html>`_ in the *Amazon Aurora User Guide* .

            This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property.

            Valid for: Aurora DB clusters only

            :param auto_pause: A value that indicates whether to allow or disallow automatic pause for an Aurora DB cluster in ``serverless`` DB engine mode. A DB cluster can be paused only when it's idle (it has no connections). .. epigraph:: If a DB cluster is paused for more than seven days, the DB cluster might be backed up with a snapshot. In this case, the DB cluster is restored when there is a request to connect to it.
            :param max_capacity: The maximum capacity for an Aurora DB cluster in ``serverless`` DB engine mode. For Aurora MySQL, valid capacity values are ``1`` , ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``128`` , and ``256`` . For Aurora PostgreSQL, valid capacity values are ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``192`` , and ``384`` . The maximum capacity must be greater than or equal to the minimum capacity.
            :param min_capacity: The minimum capacity for an Aurora DB cluster in ``serverless`` DB engine mode. For Aurora MySQL, valid capacity values are ``1`` , ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``128`` , and ``256`` . For Aurora PostgreSQL, valid capacity values are ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``192`` , and ``384`` . The minimum capacity must be less than or equal to the maximum capacity.
            :param seconds_before_timeout: The amount of time, in seconds, that Aurora Serverless v1 tries to find a scaling point to perform seamless scaling before enforcing the timeout action. The default is 300. Specify a value between 60 and 600 seconds.
            :param seconds_until_auto_pause: The time, in seconds, before an Aurora DB cluster in ``serverless`` mode is paused. Specify a value between 300 and 86,400 seconds.
            :param timeout_action: The action to take when the timeout is reached, either ``ForceApplyCapacityChange`` or ``RollbackCapacityChange`` . ``ForceApplyCapacityChange`` sets the capacity to the specified value as soon as possible. ``RollbackCapacityChange`` , the default, ignores the capacity change if a scaling point isn't found in the timeout period. .. epigraph:: If you specify ``ForceApplyCapacityChange`` , connections that prevent Aurora Serverless v1 from finding a scaling point might be dropped. For more information, see `Autoscaling for Aurora Serverless v1 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.how-it-works.auto-scaling>`_ in the *Amazon Aurora User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                scaling_configuration_property = rds.CfnDBCluster.ScalingConfigurationProperty(
                    auto_pause=False,
                    max_capacity=123,
                    min_capacity=123,
                    seconds_before_timeout=123,
                    seconds_until_auto_pause=123,
                    timeout_action="timeoutAction"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ab7fc36d999cf62c15dd859594658064019d8f605047c4ff5522885f3cf57d62)
                check_type(argname="argument auto_pause", value=auto_pause, expected_type=type_hints["auto_pause"])
                check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
                check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
                check_type(argname="argument seconds_before_timeout", value=seconds_before_timeout, expected_type=type_hints["seconds_before_timeout"])
                check_type(argname="argument seconds_until_auto_pause", value=seconds_until_auto_pause, expected_type=type_hints["seconds_until_auto_pause"])
                check_type(argname="argument timeout_action", value=timeout_action, expected_type=type_hints["timeout_action"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if auto_pause is not None:
                self._values["auto_pause"] = auto_pause
            if max_capacity is not None:
                self._values["max_capacity"] = max_capacity
            if min_capacity is not None:
                self._values["min_capacity"] = min_capacity
            if seconds_before_timeout is not None:
                self._values["seconds_before_timeout"] = seconds_before_timeout
            if seconds_until_auto_pause is not None:
                self._values["seconds_until_auto_pause"] = seconds_until_auto_pause
            if timeout_action is not None:
                self._values["timeout_action"] = timeout_action

        @builtins.property
        def auto_pause(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''A value that indicates whether to allow or disallow automatic pause for an Aurora DB cluster in ``serverless`` DB engine mode.

            A DB cluster can be paused only when it's idle (it has no connections).
            .. epigraph::

               If a DB cluster is paused for more than seven days, the DB cluster might be backed up with a snapshot. In this case, the DB cluster is restored when there is a request to connect to it.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-autopause
            '''
            result = self._values.get("auto_pause")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def max_capacity(self) -> typing.Optional[jsii.Number]:
            '''The maximum capacity for an Aurora DB cluster in ``serverless`` DB engine mode.

            For Aurora MySQL, valid capacity values are ``1`` , ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``128`` , and ``256`` .

            For Aurora PostgreSQL, valid capacity values are ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``192`` , and ``384`` .

            The maximum capacity must be greater than or equal to the minimum capacity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-maxcapacity
            '''
            result = self._values.get("max_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min_capacity(self) -> typing.Optional[jsii.Number]:
            '''The minimum capacity for an Aurora DB cluster in ``serverless`` DB engine mode.

            For Aurora MySQL, valid capacity values are ``1`` , ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``128`` , and ``256`` .

            For Aurora PostgreSQL, valid capacity values are ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``192`` , and ``384`` .

            The minimum capacity must be less than or equal to the maximum capacity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-mincapacity
            '''
            result = self._values.get("min_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def seconds_before_timeout(self) -> typing.Optional[jsii.Number]:
            '''The amount of time, in seconds, that Aurora Serverless v1 tries to find a scaling point to perform seamless scaling before enforcing the timeout action.

            The default is 300.

            Specify a value between 60 and 600 seconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-secondsbeforetimeout
            '''
            result = self._values.get("seconds_before_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def seconds_until_auto_pause(self) -> typing.Optional[jsii.Number]:
            '''The time, in seconds, before an Aurora DB cluster in ``serverless`` mode is paused.

            Specify a value between 300 and 86,400 seconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-secondsuntilautopause
            '''
            result = self._values.get("seconds_until_auto_pause")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def timeout_action(self) -> typing.Optional[builtins.str]:
            '''The action to take when the timeout is reached, either ``ForceApplyCapacityChange`` or ``RollbackCapacityChange`` .

            ``ForceApplyCapacityChange`` sets the capacity to the specified value as soon as possible.

            ``RollbackCapacityChange`` , the default, ignores the capacity change if a scaling point isn't found in the timeout period.
            .. epigraph::

               If you specify ``ForceApplyCapacityChange`` , connections that prevent Aurora Serverless v1 from finding a scaling point might be dropped.

            For more information, see `Autoscaling for Aurora Serverless v1 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.how-it-works.auto-scaling>`_ in the *Amazon Aurora User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-timeoutaction
            '''
            result = self._values.get("timeout_action")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBCluster.ServerlessV2ScalingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"max_capacity": "maxCapacity", "min_capacity": "minCapacity"},
    )
    class ServerlessV2ScalingConfigurationProperty:
        def __init__(
            self,
            *,
            max_capacity: typing.Optional[jsii.Number] = None,
            min_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster.

            For more information, see `Using Amazon Aurora Serverless v2 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.html>`_ in the *Amazon Aurora User Guide* .

            If you have an Aurora cluster, you must set the ``ScalingConfigurationInfo`` attribute before you add a DB instance that uses the ``db.serverless`` DB instance class. For more information, see `Clusters that use Aurora Serverless v2 must have a capacity range specified <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.requirements.html#aurora-serverless-v2.requirements.capacity-range>`_ in the *Amazon Aurora User Guide* .

            This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property.

            :param max_capacity: The maximum number of Aurora capacity units (ACUs) for a DB instance in an Aurora Serverless v2 cluster. You can specify ACU values in half-step increments, such as 40, 40.5, 41, and so on. The largest value that you can use is 128. The maximum capacity must be higher than 0.5 ACUs. For more information, see `Choosing the maximum Aurora Serverless v2 capacity setting for a cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html#aurora-serverless-v2.max_capacity_considerations>`_ in the *Amazon Aurora User Guide* .
            :param min_capacity: The minimum number of Aurora capacity units (ACUs) for a DB instance in an Aurora Serverless v2 cluster. You can specify ACU values in half-step increments, such as 8, 8.5, 9, and so on. The smallest value that you can use is 0.5.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-serverlessv2scalingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                serverless_v2_scaling_configuration_property = rds.CfnDBCluster.ServerlessV2ScalingConfigurationProperty(
                    max_capacity=123,
                    min_capacity=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5a6717af18f7b2b042b74ca423a540eb39f5ae817292b338733c6a3cee2b4284)
                check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
                check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max_capacity is not None:
                self._values["max_capacity"] = max_capacity
            if min_capacity is not None:
                self._values["min_capacity"] = min_capacity

        @builtins.property
        def max_capacity(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of Aurora capacity units (ACUs) for a DB instance in an Aurora Serverless v2 cluster.

            You can specify ACU values in half-step increments, such as 40, 40.5, 41, and so on. The largest value that you can use is 128.

            The maximum capacity must be higher than 0.5 ACUs. For more information, see `Choosing the maximum Aurora Serverless v2 capacity setting for a cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html#aurora-serverless-v2.max_capacity_considerations>`_ in the *Amazon Aurora User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-serverlessv2scalingconfiguration.html#cfn-rds-dbcluster-serverlessv2scalingconfiguration-maxcapacity
            '''
            result = self._values.get("max_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min_capacity(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of Aurora capacity units (ACUs) for a DB instance in an Aurora Serverless v2 cluster.

            You can specify ACU values in half-step increments, such as 8, 8.5, 9, and so on. The smallest value that you can use is 0.5.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-serverlessv2scalingconfiguration.html#cfn-rds-dbcluster-serverlessv2scalingconfiguration-mincapacity
            '''
            result = self._values.get("min_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServerlessV2ScalingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnDBClusterParameterGroup(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBClusterParameterGroup",
):
    '''A CloudFormation ``AWS::RDS::DBClusterParameterGroup``.

    The ``AWS::RDS::DBClusterParameterGroup`` resource creates a new Amazon RDS DB cluster parameter group.

    For information about configuring parameters for Amazon Aurora DB clusters, see `Working with parameter groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* .
    .. epigraph::

       If you apply a parameter group to a DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting.

       If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started.

    :cloudformationResource: AWS::RDS::DBClusterParameterGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        # parameters: Any
        
        cfn_dBCluster_parameter_group = rds.CfnDBClusterParameterGroup(self, "MyCfnDBClusterParameterGroup",
            description="description",
            family="family",
            parameters=parameters,
        
            # the properties below are optional
            db_cluster_parameter_group_name="dbClusterParameterGroupName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        description: builtins.str,
        family: builtins.str,
        parameters: typing.Any,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBClusterParameterGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param description: A friendly description for this DB cluster parameter group.
        :param family: The DB cluster parameter group family name. A DB cluster parameter group can be associated with one and only one DB cluster parameter group family, and can be applied only to a DB cluster running a DB engine and engine version compatible with that DB cluster parameter group family. .. epigraph:: The DB cluster parameter group family can't be changed when updating a DB cluster parameter group. To list all of the available parameter group families, use the following command: ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"`` The output contains duplicates. For more information, see ``[CreateDBClusterParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html)`` .
        :param parameters: Provides a list of parameters for the DB cluster parameter group.
        :param db_cluster_parameter_group_name: The name of the DB cluster parameter group. Constraints: - Must not match the name of an existing DB cluster parameter group. If you don't specify a value for ``DBClusterParameterGroupName`` property, a name is automatically created for the DB cluster parameter group. .. epigraph:: This value is stored as a lowercase string.
        :param tags: An optional array of key-value pairs to apply to this DB cluster parameter group.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d66d67784eaba0a01483098e06526f4c025b7ff96275fcc6e41e7d979aa520c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBClusterParameterGroupProps(
            description=description,
            family=family,
            parameters=parameters,
            db_cluster_parameter_group_name=db_cluster_parameter_group_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__91620936135acc2eba770911b4583bbfc6133f6cca4a768a0ac283dcc9589927)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__390c4bb1994cb3d59be0c98e412ad992b1425ccaaf921f8bb19168f81ed5609c)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''An optional array of key-value pairs to apply to this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> builtins.str:
        '''A friendly description for this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-description
        '''
        return typing.cast(builtins.str, jsii.get(self, "description"))

    @description.setter
    def description(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a385d094303e0167ab68da37077fab8909f20f87d11b139fd5ef69090e76f8fb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="family")
    def family(self) -> builtins.str:
        '''The DB cluster parameter group family name.

        A DB cluster parameter group can be associated with one and only one DB cluster parameter group family, and can be applied only to a DB cluster running a DB engine and engine version compatible with that DB cluster parameter group family.
        .. epigraph::

           The DB cluster parameter group family can't be changed when updating a DB cluster parameter group.

        To list all of the available parameter group families, use the following command:

        ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"``

        The output contains duplicates.

        For more information, see ``[CreateDBClusterParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-family
        '''
        return typing.cast(builtins.str, jsii.get(self, "family"))

    @family.setter
    def family(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dbdfcaf985524b33f444a02535bb68b9ba3263c90ef5e0985151f104aeeb260b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "family", value)

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Any:
        '''Provides a list of parameters for the DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-parameters
        '''
        return typing.cast(typing.Any, jsii.get(self, "parameters"))

    @parameters.setter
    def parameters(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c77a7367d69d8d946eb31a37bfbabde86b3fd025985be9f6006fb54af746b2a8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parameters", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterParameterGroupName")
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB cluster parameter group.

        Constraints:

        - Must not match the name of an existing DB cluster parameter group.

        If you don't specify a value for ``DBClusterParameterGroupName`` property, a name is automatically created for the DB cluster parameter group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-dbclusterparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterParameterGroupName"))

    @db_cluster_parameter_group_name.setter
    def db_cluster_parameter_group_name(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0f4022845bd024f0aff7d6277fa4309ef0b25b2e9f601bc06782d5194fdea18a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterParameterGroupName", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBClusterParameterGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "family": "family",
        "parameters": "parameters",
        "db_cluster_parameter_group_name": "dbClusterParameterGroupName",
        "tags": "tags",
    },
)
class CfnDBClusterParameterGroupProps:
    def __init__(
        self,
        *,
        description: builtins.str,
        family: builtins.str,
        parameters: typing.Any,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBClusterParameterGroup``.

        :param description: A friendly description for this DB cluster parameter group.
        :param family: The DB cluster parameter group family name. A DB cluster parameter group can be associated with one and only one DB cluster parameter group family, and can be applied only to a DB cluster running a DB engine and engine version compatible with that DB cluster parameter group family. .. epigraph:: The DB cluster parameter group family can't be changed when updating a DB cluster parameter group. To list all of the available parameter group families, use the following command: ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"`` The output contains duplicates. For more information, see ``[CreateDBClusterParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html)`` .
        :param parameters: Provides a list of parameters for the DB cluster parameter group.
        :param db_cluster_parameter_group_name: The name of the DB cluster parameter group. Constraints: - Must not match the name of an existing DB cluster parameter group. If you don't specify a value for ``DBClusterParameterGroupName`` property, a name is automatically created for the DB cluster parameter group. .. epigraph:: This value is stored as a lowercase string.
        :param tags: An optional array of key-value pairs to apply to this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            # parameters: Any
            
            cfn_dBCluster_parameter_group_props = rds.CfnDBClusterParameterGroupProps(
                description="description",
                family="family",
                parameters=parameters,
            
                # the properties below are optional
                db_cluster_parameter_group_name="dbClusterParameterGroupName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e37c06a29695a900fc99ca3ea97a86c72d21d474561ffc72ce13829934195382)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument family", value=family, expected_type=type_hints["family"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument db_cluster_parameter_group_name", value=db_cluster_parameter_group_name, expected_type=type_hints["db_cluster_parameter_group_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "description": description,
            "family": family,
            "parameters": parameters,
        }
        if db_cluster_parameter_group_name is not None:
            self._values["db_cluster_parameter_group_name"] = db_cluster_parameter_group_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def description(self) -> builtins.str:
        '''A friendly description for this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-description
        '''
        result = self._values.get("description")
        assert result is not None, "Required property 'description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def family(self) -> builtins.str:
        '''The DB cluster parameter group family name.

        A DB cluster parameter group can be associated with one and only one DB cluster parameter group family, and can be applied only to a DB cluster running a DB engine and engine version compatible with that DB cluster parameter group family.
        .. epigraph::

           The DB cluster parameter group family can't be changed when updating a DB cluster parameter group.

        To list all of the available parameter group families, use the following command:

        ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"``

        The output contains duplicates.

        For more information, see ``[CreateDBClusterParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-family
        '''
        result = self._values.get("family")
        assert result is not None, "Required property 'family' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def parameters(self) -> typing.Any:
        '''Provides a list of parameters for the DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-parameters
        '''
        result = self._values.get("parameters")
        assert result is not None, "Required property 'parameters' is missing"
        return typing.cast(typing.Any, result)

    @builtins.property
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB cluster parameter group.

        Constraints:

        - Must not match the name of an existing DB cluster parameter group.

        If you don't specify a value for ``DBClusterParameterGroupName`` property, a name is automatically created for the DB cluster parameter group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-dbclusterparametergroupname
        '''
        result = self._values.get("db_cluster_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''An optional array of key-value pairs to apply to this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBClusterParameterGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "allocated_storage": "allocatedStorage",
        "associated_roles": "associatedRoles",
        "auto_minor_version_upgrade": "autoMinorVersionUpgrade",
        "availability_zones": "availabilityZones",
        "backtrack_window": "backtrackWindow",
        "backup_retention_period": "backupRetentionPeriod",
        "copy_tags_to_snapshot": "copyTagsToSnapshot",
        "database_name": "databaseName",
        "db_cluster_identifier": "dbClusterIdentifier",
        "db_cluster_instance_class": "dbClusterInstanceClass",
        "db_cluster_parameter_group_name": "dbClusterParameterGroupName",
        "db_instance_parameter_group_name": "dbInstanceParameterGroupName",
        "db_subnet_group_name": "dbSubnetGroupName",
        "db_system_id": "dbSystemId",
        "deletion_protection": "deletionProtection",
        "domain": "domain",
        "domain_iam_role_name": "domainIamRoleName",
        "enable_cloudwatch_logs_exports": "enableCloudwatchLogsExports",
        "enable_http_endpoint": "enableHttpEndpoint",
        "enable_iam_database_authentication": "enableIamDatabaseAuthentication",
        "engine": "engine",
        "engine_mode": "engineMode",
        "engine_version": "engineVersion",
        "global_cluster_identifier": "globalClusterIdentifier",
        "iops": "iops",
        "kms_key_id": "kmsKeyId",
        "manage_master_user_password": "manageMasterUserPassword",
        "master_username": "masterUsername",
        "master_user_password": "masterUserPassword",
        "master_user_secret": "masterUserSecret",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role_arn": "monitoringRoleArn",
        "network_type": "networkType",
        "performance_insights_enabled": "performanceInsightsEnabled",
        "performance_insights_kms_key_id": "performanceInsightsKmsKeyId",
        "performance_insights_retention_period": "performanceInsightsRetentionPeriod",
        "port": "port",
        "preferred_backup_window": "preferredBackupWindow",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "publicly_accessible": "publiclyAccessible",
        "replication_source_identifier": "replicationSourceIdentifier",
        "restore_to_time": "restoreToTime",
        "restore_type": "restoreType",
        "scaling_configuration": "scalingConfiguration",
        "serverless_v2_scaling_configuration": "serverlessV2ScalingConfiguration",
        "snapshot_identifier": "snapshotIdentifier",
        "source_db_cluster_identifier": "sourceDbClusterIdentifier",
        "source_region": "sourceRegion",
        "storage_encrypted": "storageEncrypted",
        "storage_type": "storageType",
        "tags": "tags",
        "use_latest_restorable_time": "useLatestRestorableTime",
        "vpc_security_group_ids": "vpcSecurityGroupIds",
    },
)
class CfnDBClusterProps:
    def __init__(
        self,
        *,
        allocated_storage: typing.Optional[jsii.Number] = None,
        associated_roles: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBCluster.DBClusterRoleProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        availability_zones: typing.Optional[typing.Sequence[builtins.str]] = None,
        backtrack_window: typing.Optional[jsii.Number] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        database_name: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_instance_class: typing.Optional[builtins.str] = None,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        db_instance_parameter_group_name: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        db_system_id: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.Sequence[builtins.str]] = None,
        enable_http_endpoint: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_mode: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        manage_master_user_password: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        master_user_secret: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBCluster.MasterUserSecretProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        network_type: typing.Optional[builtins.str] = None,
        performance_insights_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        replication_source_identifier: typing.Optional[builtins.str] = None,
        restore_to_time: typing.Optional[builtins.str] = None,
        restore_type: typing.Optional[builtins.str] = None,
        scaling_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBCluster.ScalingConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        serverless_v2_scaling_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBCluster.ServerlessV2ScalingConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        snapshot_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBCluster``.

        :param allocated_storage: The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster. Valid for Cluster Type: Multi-AZ DB clusters only This setting is required to create a Multi-AZ DB cluster.
        :param associated_roles: Provides a list of the AWS Identity and Access Management (IAM) roles that are associated with the DB cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other Amazon Web Services on your behalf. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param auto_minor_version_upgrade: Specifies whether minor engine upgrades are applied automatically to the DB cluster during the maintenance window. By default, minor engine upgrades are applied automatically. Valid for Cluster Type: Multi-AZ DB clusters only
        :param availability_zones: A list of Availability Zones (AZs) where instances in the DB cluster can be created. For information on AWS Regions and Availability Zones, see `Choosing the Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param backtrack_window: The target backtrack window, in seconds. To disable backtracking, set this value to 0. .. epigraph:: Currently, Backtrack is only supported for Aurora MySQL DB clusters. Default: 0 Constraints: - If specified, this value must be set to a number from 0 to 259,200 (72 hours). Valid for: Aurora MySQL DB clusters only
        :param backup_retention_period: The number of days for which automated backups are retained. Default: 1 Constraints: - Must be a value from 1 to 35 Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param copy_tags_to_snapshot: A value that indicates whether to copy all tags from the DB cluster to snapshots of the DB cluster. The default is not to copy them. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param database_name: The name of your database. If you don't provide a name, then Amazon RDS won't create a database in this DB cluster. For naming constraints, see `Naming Constraints <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_cluster_identifier: The DB cluster identifier. This parameter is stored as a lowercase string. Constraints: - Must contain from 1 to 63 letters, numbers, or hyphens. - First character must be a letter. - Can't end with a hyphen or contain two consecutive hyphens. Example: ``my-cluster1`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_cluster_instance_class: The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example ``db.m6gd.xlarge`` . Not all DB instance classes are available in all AWS Regions , or for all database engines. For the full list of DB instance classes and availability for your engine, see `DB instance class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide* . This setting is required to create a Multi-AZ DB cluster. Valid for Cluster Type: Multi-AZ DB clusters only
        :param db_cluster_parameter_group_name: The name of the DB cluster parameter group to associate with this DB cluster. .. epigraph:: If you apply a parameter group to an existing DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting. If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started. To list all of the available DB cluster parameter group names, use the following command: ``aws rds describe-db-cluster-parameter-groups --query "DBClusterParameterGroups[].DBClusterParameterGroupName" --output text`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_instance_parameter_group_name: The name of the DB parameter group to apply to all instances of the DB cluster. .. epigraph:: When you apply a parameter group using the ``DBInstanceParameterGroupName`` parameter, the DB cluster isn't rebooted automatically. Also, parameter changes are applied immediately rather than during the next maintenance window. Default: The existing name setting Constraints: - The DB parameter group must be in the same DB parameter group family as this DB cluster.
        :param db_subnet_group_name: A DB subnet group that you want to associate with this DB cluster. If you are restoring a DB cluster to a point in time with ``RestoreType`` set to ``copy-on-write`` , and don't specify a DB subnet group name, then the DB cluster is restored with a default DB subnet group. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_system_id: Reserved for future use.
        :param deletion_protection: A value that indicates whether the DB cluster has deletion protection enabled. The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param domain: Indicates the directory ID of the Active Directory to create the DB cluster. For Amazon Aurora DB clusters, Amazon RDS can use Kerberos authentication to authenticate users that connect to the DB cluster. For more information, see `Kerberos authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/kerberos-authentication.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param domain_iam_role_name: Specifies the name of the IAM role to use when making API calls to the Directory Service. Valid for: Aurora DB clusters only
        :param enable_cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Aurora User Guide* . *Aurora MySQL* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Aurora PostgreSQL* Valid values: ``postgresql`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param enable_http_endpoint: A value that indicates whether to enable the HTTP endpoint for an Aurora Serverless DB cluster. By default, the HTTP endpoint is disabled. When enabled, the HTTP endpoint provides a connectionless web service API for running SQL queries on the Aurora Serverless DB cluster. You can also query your database from inside the RDS console with the query editor. For more information, see `Using the Data API for Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param enable_iam_database_authentication: A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. By default, mapping is disabled. For more information, see `IAM Database Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon Aurora User Guide.* Valid for: Aurora DB clusters only
        :param engine: The name of the database engine to be used for this DB cluster. Valid Values: - ``aurora-mysql`` - ``aurora-postgresql`` - ``mysql`` - ``postgres`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param engine_mode: The DB engine mode of the DB cluster, either ``provisioned`` or ``serverless`` . The ``serverless`` engine mode only supports Aurora Serverless v1. Limitations and requirements apply to some DB engine modes. For more information, see the following sections in the *Amazon Aurora User Guide* : - `Limitations of Aurora Serverless v1 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations>`_ - `Requirements for Aurora Serverless v2 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.requirements.html>`_ - `Limitations of parallel query <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-parallel-query.html#aurora-mysql-parallel-query-limitations>`_ - `Limitations of Aurora global databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html#aurora-global-database.limitations>`_ Valid for: Aurora DB clusters only
        :param engine_version: The version number of the database engine to use. To list all of the available engine versions for Aurora MySQL version 2 (5.7-compatible) and version 3 (8.0-compatible), use the following command: ``aws rds describe-db-engine-versions --engine aurora-mysql --query "DBEngineVersions[].EngineVersion"`` You can supply either ``5.7`` or ``8.0`` to use the default engine version for Aurora MySQL version 2 or version 3, respectively. To list all of the available engine versions for Aurora PostgreSQL, use the following command: ``aws rds describe-db-engine-versions --engine aurora-postgresql --query "DBEngineVersions[].EngineVersion"`` To list all of the available engine versions for RDS for MySQL, use the following command: ``aws rds describe-db-engine-versions --engine mysql --query "DBEngineVersions[].EngineVersion"`` To list all of the available engine versions for RDS for PostgreSQL, use the following command: ``aws rds describe-db-engine-versions --engine postgres --query "DBEngineVersions[].EngineVersion"`` *Aurora MySQL* For information, see `Database engine updates for Amazon Aurora MySQL <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Updates.html>`_ in the *Amazon Aurora User Guide* . *Aurora PostgreSQL* For information, see `Amazon Aurora PostgreSQL releases and engine versions <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.20180305.html>`_ in the *Amazon Aurora User Guide* . *MySQL* For information, see `Amazon RDS for MySQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide* . *PostgreSQL* For information, see `Amazon RDS for PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts>`_ in the *Amazon RDS User Guide* . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param global_cluster_identifier: If you are configuring an Aurora global database cluster and want your Aurora DB cluster to be a secondary member in the global database cluster, specify the global cluster ID of the global database cluster. To define the primary database cluster of the global cluster, use the `AWS::RDS::GlobalCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html>`_ resource. If you aren't configuring a global database cluster, don't specify this property. .. epigraph:: To remove the DB cluster from a global database cluster, specify an empty value for the ``GlobalClusterIdentifier`` property. For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param iops: The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid IOPS values, see `Provisioned IOPS storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* . This setting is required to create a Multi-AZ DB cluster. Valid for Cluster Type: Multi-AZ DB clusters only Constraints: - Must be a multiple between .5 and 50 of the storage amount for the DB cluster.
        :param kms_key_id: The Amazon Resource Name (ARN) of the AWS KMS key that is used to encrypt the database instances in the DB cluster, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` . If you enable the ``StorageEncrypted`` property but don't specify this property, the default KMS key is used. If you specify this property, you must set the ``StorageEncrypted`` property to ``true`` . If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param manage_master_user_password: Specifies whether to manage the master user password with AWS Secrets Manager. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.* Valid for Cluster Type: Aurora DB clusters and Multi-AZ DB clusters Constraints: - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.
        :param master_username: The name of the master user for the DB cluster. .. epigraph:: If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_user_password: The master password for the DB instance. .. epigraph:: If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_user_secret: The secret managed by RDS in AWS Secrets Manager for the master user password. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*
        :param monitoring_interval: The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster. To turn off collecting Enhanced Monitoring metrics, specify ``0`` . If ``MonitoringRoleArn`` is specified, also set ``MonitoringInterval`` to a value other than ``0`` . Valid for Cluster Type: Multi-AZ DB clusters only Valid Values: ``0 | 1 | 5 | 10 | 15 | 30 | 60`` Default: ``0``
        :param monitoring_role_arn: The Amazon Resource Name (ARN) for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs. An example is ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting up and enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* . If ``MonitoringInterval`` is set to a value other than ``0`` , supply a ``MonitoringRoleArn`` value. Valid for Cluster Type: Multi-AZ DB clusters only
        :param network_type: The network type of the DB cluster. Valid values: - ``IPV4`` - ``DUAL`` The network type is determined by the ``DBSubnetGroup`` specified for the DB cluster. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ). For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon Aurora User Guide.* Valid for: Aurora DB clusters only
        :param performance_insights_enabled: Specifies whether to turn on Performance Insights for the DB cluster. For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* . Valid for Cluster Type: Multi-AZ DB clusters only
        :param performance_insights_kms_key_id: The AWS KMS key identifier for encryption of Performance Insights data. The AWS KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. If you don't specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account . Your AWS account has a different default KMS key for each AWS Region . Valid for Cluster Type: Multi-AZ DB clusters only
        :param performance_insights_retention_period: The number of days to retain Performance Insights data. Valid for Cluster Type: Multi-AZ DB clusters only Valid Values: - ``7`` - *month* * 31, where *month* is a number of months from 1-23. Examples: ``93`` (3 months * 31), ``341`` (11 months * 31), ``589`` (19 months * 31) - ``731`` Default: ``7`` days If you specify a retention period that isn't valid, such as ``94`` , Amazon RDS issues an error.
        :param port: The port number on which the DB instances in the DB cluster accept connections. Default: - When ``EngineMode`` is ``provisioned`` , ``3306`` (for both Aurora MySQL and Aurora PostgreSQL) - When ``EngineMode`` is ``serverless`` : - ``3306`` when ``Engine`` is ``aurora`` or ``aurora-mysql`` - ``5432`` when ``Engine`` is ``aurora-postgresql`` .. epigraph:: The ``No interruption`` on update behavior only applies to DB clusters. If you are updating a DB instance, see `Port <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port>`_ for the AWS::RDS::DBInstance resource. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param preferred_backup_window: The daily time range during which automated backups are created. For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.Backups.BackupWindow>`_ in the *Amazon Aurora User Guide.* Constraints: - Must be in the format ``hh24:mi-hh24:mi`` . - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param preferred_maintenance_window: The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC). Format: ``ddd:hh24:mi-ddd:hh24:mi`` The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Cluster Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow.Aurora>`_ in the *Amazon Aurora User Guide.* Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun. Constraints: Minimum 30-minute window. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param publicly_accessible: Specifies whether the DB cluster is publicly accessible. When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it uses. That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it. When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to a private IP address. Valid for Cluster Type: Multi-AZ DB clusters only Default: The default behavior varies depending on whether ``DBSubnetGroupName`` is specified. If ``DBSubnetGroupName`` isn't specified, and ``PubliclyAccessible`` isn't specified, the following applies: - If the default VPC in the target Region doesn’t have an internet gateway attached to it, the DB cluster is private. - If the default VPC in the target Region has an internet gateway attached to it, the DB cluster is public. If ``DBSubnetGroupName`` is specified, and ``PubliclyAccessible`` isn't specified, the following applies: - If the subnets are part of a VPC that doesn’t have an internet gateway attached to it, the DB cluster is private. - If the subnets are part of a VPC that has an internet gateway attached to it, the DB cluster is public.
        :param replication_source_identifier: The Amazon Resource Name (ARN) of the source DB instance or DB cluster if this DB cluster is created as a read replica. Valid for: Aurora DB clusters only
        :param restore_to_time: The date and time to restore the DB cluster to. Valid Values: Value must be a time in Universal Coordinated Time (UTC) format Constraints: - Must be before the latest restorable time for the DB instance - Must be specified if ``UseLatestRestorableTime`` parameter isn't provided - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled - Can't be specified if the ``RestoreType`` parameter is ``copy-on-write`` Example: ``2015-03-07T23:45:00Z`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param restore_type: The type of restore to be performed. You can specify one of the following values:. - ``full-copy`` - The new DB cluster is restored as a full copy of the source DB cluster. - ``copy-on-write`` - The new DB cluster is restored as a clone of the source DB cluster. If you don't specify a ``RestoreType`` value, then the new DB cluster is restored as a full copy of the source DB cluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param scaling_configuration: The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster. This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property. Valid for: Aurora DB clusters only
        :param serverless_v2_scaling_configuration: The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster. This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property. Valid for: Aurora DB clusters only
        :param snapshot_identifier: The identifier for the DB snapshot or DB cluster snapshot to restore from. You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot. After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted. If you specify the ``SnapshotIdentifier`` property to restore a DB cluster (as opposed to specifying it for DB cluster updates), then don't specify the following properties: - ``GlobalClusterIdentifier`` - ``MasterUsername`` - ``MasterUserPassword`` - ``ReplicationSourceIdentifier`` - ``RestoreType`` - ``SourceDBClusterIdentifier`` - ``SourceRegion`` - ``StorageEncrypted`` (for an encrypted snapshot) - ``UseLatestRestorableTime`` Constraints: - Must match the identifier of an existing Snapshot. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param source_db_cluster_identifier: When restoring a DB cluster to a point in time, the identifier of the source DB cluster from which to restore. Constraints: - Must match the identifier of an existing DBCluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param source_region: The AWS Region which contains the source DB cluster when replicating a DB cluster. For example, ``us-east-1`` . Valid for: Aurora DB clusters only
        :param storage_encrypted: Indicates whether the DB cluster is encrypted. If you specify the ``KmsKeyId`` property, then you must enable encryption. If you specify the ``SourceDBClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB cluster is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB cluster to be encrypted, then don't set this property or set it to ``false`` . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param storage_type: The storage type to associate with the DB cluster. For information on storage types for Aurora DB clusters, see `Storage configurations for Amazon Aurora DB clusters <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Overview.StorageReliability.html#aurora-storage-type>`_ . For information on storage types for Multi-AZ DB clusters, see `Settings for creating Multi-AZ DB clusters <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/create-multi-az-db-cluster.html#create-multi-az-db-cluster-settings>`_ . This setting is required to create a Multi-AZ DB cluster. When specified for a Multi-AZ DB cluster, a value for the ``Iops`` parameter is required. Valid for Cluster Type: Aurora DB clusters and Multi-AZ DB clusters Valid Values: - Aurora DB clusters - ``aurora | aurora-iopt1`` - Multi-AZ DB clusters - ``io1`` Default: - Aurora DB clusters - ``aurora`` - Multi-AZ DB clusters - ``io1``
        :param tags: An optional array of key-value pairs to apply to this DB cluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param use_latest_restorable_time: A value that indicates whether to restore the DB cluster to the latest restorable backup time. By default, the DB cluster is not restored to the latest restorable backup time. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param vpc_security_group_ids: A list of EC2 VPC security groups to associate with this DB cluster. If you plan to update the resource, don't specify VPC security groups in a shared VPC. Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_dBCluster_props = rds.CfnDBClusterProps(
                allocated_storage=123,
                associated_roles=[rds.CfnDBCluster.DBClusterRoleProperty(
                    role_arn="roleArn",
            
                    # the properties below are optional
                    feature_name="featureName"
                )],
                auto_minor_version_upgrade=False,
                availability_zones=["availabilityZones"],
                backtrack_window=123,
                backup_retention_period=123,
                copy_tags_to_snapshot=False,
                database_name="databaseName",
                db_cluster_identifier="dbClusterIdentifier",
                db_cluster_instance_class="dbClusterInstanceClass",
                db_cluster_parameter_group_name="dbClusterParameterGroupName",
                db_instance_parameter_group_name="dbInstanceParameterGroupName",
                db_subnet_group_name="dbSubnetGroupName",
                db_system_id="dbSystemId",
                deletion_protection=False,
                domain="domain",
                domain_iam_role_name="domainIamRoleName",
                enable_cloudwatch_logs_exports=["enableCloudwatchLogsExports"],
                enable_http_endpoint=False,
                enable_iam_database_authentication=False,
                engine="engine",
                engine_mode="engineMode",
                engine_version="engineVersion",
                global_cluster_identifier="globalClusterIdentifier",
                iops=123,
                kms_key_id="kmsKeyId",
                manage_master_user_password=False,
                master_username="masterUsername",
                master_user_password="masterUserPassword",
                master_user_secret=rds.CfnDBCluster.MasterUserSecretProperty(
                    kms_key_id="kmsKeyId",
                    secret_arn="secretArn"
                ),
                monitoring_interval=123,
                monitoring_role_arn="monitoringRoleArn",
                network_type="networkType",
                performance_insights_enabled=False,
                performance_insights_kms_key_id="performanceInsightsKmsKeyId",
                performance_insights_retention_period=123,
                port=123,
                preferred_backup_window="preferredBackupWindow",
                preferred_maintenance_window="preferredMaintenanceWindow",
                publicly_accessible=False,
                replication_source_identifier="replicationSourceIdentifier",
                restore_to_time="restoreToTime",
                restore_type="restoreType",
                scaling_configuration=rds.CfnDBCluster.ScalingConfigurationProperty(
                    auto_pause=False,
                    max_capacity=123,
                    min_capacity=123,
                    seconds_before_timeout=123,
                    seconds_until_auto_pause=123,
                    timeout_action="timeoutAction"
                ),
                serverless_v2_scaling_configuration=rds.CfnDBCluster.ServerlessV2ScalingConfigurationProperty(
                    max_capacity=123,
                    min_capacity=123
                ),
                snapshot_identifier="snapshotIdentifier",
                source_db_cluster_identifier="sourceDbClusterIdentifier",
                source_region="sourceRegion",
                storage_encrypted=False,
                storage_type="storageType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                use_latest_restorable_time=False,
                vpc_security_group_ids=["vpcSecurityGroupIds"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aaf089104646bb0ea95e48cd2107d642585c3eb3785a21112fc029b15f7867e7)
            check_type(argname="argument allocated_storage", value=allocated_storage, expected_type=type_hints["allocated_storage"])
            check_type(argname="argument associated_roles", value=associated_roles, expected_type=type_hints["associated_roles"])
            check_type(argname="argument auto_minor_version_upgrade", value=auto_minor_version_upgrade, expected_type=type_hints["auto_minor_version_upgrade"])
            check_type(argname="argument availability_zones", value=availability_zones, expected_type=type_hints["availability_zones"])
            check_type(argname="argument backtrack_window", value=backtrack_window, expected_type=type_hints["backtrack_window"])
            check_type(argname="argument backup_retention_period", value=backup_retention_period, expected_type=type_hints["backup_retention_period"])
            check_type(argname="argument copy_tags_to_snapshot", value=copy_tags_to_snapshot, expected_type=type_hints["copy_tags_to_snapshot"])
            check_type(argname="argument database_name", value=database_name, expected_type=type_hints["database_name"])
            check_type(argname="argument db_cluster_identifier", value=db_cluster_identifier, expected_type=type_hints["db_cluster_identifier"])
            check_type(argname="argument db_cluster_instance_class", value=db_cluster_instance_class, expected_type=type_hints["db_cluster_instance_class"])
            check_type(argname="argument db_cluster_parameter_group_name", value=db_cluster_parameter_group_name, expected_type=type_hints["db_cluster_parameter_group_name"])
            check_type(argname="argument db_instance_parameter_group_name", value=db_instance_parameter_group_name, expected_type=type_hints["db_instance_parameter_group_name"])
            check_type(argname="argument db_subnet_group_name", value=db_subnet_group_name, expected_type=type_hints["db_subnet_group_name"])
            check_type(argname="argument db_system_id", value=db_system_id, expected_type=type_hints["db_system_id"])
            check_type(argname="argument deletion_protection", value=deletion_protection, expected_type=type_hints["deletion_protection"])
            check_type(argname="argument domain", value=domain, expected_type=type_hints["domain"])
            check_type(argname="argument domain_iam_role_name", value=domain_iam_role_name, expected_type=type_hints["domain_iam_role_name"])
            check_type(argname="argument enable_cloudwatch_logs_exports", value=enable_cloudwatch_logs_exports, expected_type=type_hints["enable_cloudwatch_logs_exports"])
            check_type(argname="argument enable_http_endpoint", value=enable_http_endpoint, expected_type=type_hints["enable_http_endpoint"])
            check_type(argname="argument enable_iam_database_authentication", value=enable_iam_database_authentication, expected_type=type_hints["enable_iam_database_authentication"])
            check_type(argname="argument engine", value=engine, expected_type=type_hints["engine"])
            check_type(argname="argument engine_mode", value=engine_mode, expected_type=type_hints["engine_mode"])
            check_type(argname="argument engine_version", value=engine_version, expected_type=type_hints["engine_version"])
            check_type(argname="argument global_cluster_identifier", value=global_cluster_identifier, expected_type=type_hints["global_cluster_identifier"])
            check_type(argname="argument iops", value=iops, expected_type=type_hints["iops"])
            check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
            check_type(argname="argument manage_master_user_password", value=manage_master_user_password, expected_type=type_hints["manage_master_user_password"])
            check_type(argname="argument master_username", value=master_username, expected_type=type_hints["master_username"])
            check_type(argname="argument master_user_password", value=master_user_password, expected_type=type_hints["master_user_password"])
            check_type(argname="argument master_user_secret", value=master_user_secret, expected_type=type_hints["master_user_secret"])
            check_type(argname="argument monitoring_interval", value=monitoring_interval, expected_type=type_hints["monitoring_interval"])
            check_type(argname="argument monitoring_role_arn", value=monitoring_role_arn, expected_type=type_hints["monitoring_role_arn"])
            check_type(argname="argument network_type", value=network_type, expected_type=type_hints["network_type"])
            check_type(argname="argument performance_insights_enabled", value=performance_insights_enabled, expected_type=type_hints["performance_insights_enabled"])
            check_type(argname="argument performance_insights_kms_key_id", value=performance_insights_kms_key_id, expected_type=type_hints["performance_insights_kms_key_id"])
            check_type(argname="argument performance_insights_retention_period", value=performance_insights_retention_period, expected_type=type_hints["performance_insights_retention_period"])
            check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            check_type(argname="argument preferred_backup_window", value=preferred_backup_window, expected_type=type_hints["preferred_backup_window"])
            check_type(argname="argument preferred_maintenance_window", value=preferred_maintenance_window, expected_type=type_hints["preferred_maintenance_window"])
            check_type(argname="argument publicly_accessible", value=publicly_accessible, expected_type=type_hints["publicly_accessible"])
            check_type(argname="argument replication_source_identifier", value=replication_source_identifier, expected_type=type_hints["replication_source_identifier"])
            check_type(argname="argument restore_to_time", value=restore_to_time, expected_type=type_hints["restore_to_time"])
            check_type(argname="argument restore_type", value=restore_type, expected_type=type_hints["restore_type"])
            check_type(argname="argument scaling_configuration", value=scaling_configuration, expected_type=type_hints["scaling_configuration"])
            check_type(argname="argument serverless_v2_scaling_configuration", value=serverless_v2_scaling_configuration, expected_type=type_hints["serverless_v2_scaling_configuration"])
            check_type(argname="argument snapshot_identifier", value=snapshot_identifier, expected_type=type_hints["snapshot_identifier"])
            check_type(argname="argument source_db_cluster_identifier", value=source_db_cluster_identifier, expected_type=type_hints["source_db_cluster_identifier"])
            check_type(argname="argument source_region", value=source_region, expected_type=type_hints["source_region"])
            check_type(argname="argument storage_encrypted", value=storage_encrypted, expected_type=type_hints["storage_encrypted"])
            check_type(argname="argument storage_type", value=storage_type, expected_type=type_hints["storage_type"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument use_latest_restorable_time", value=use_latest_restorable_time, expected_type=type_hints["use_latest_restorable_time"])
            check_type(argname="argument vpc_security_group_ids", value=vpc_security_group_ids, expected_type=type_hints["vpc_security_group_ids"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allocated_storage is not None:
            self._values["allocated_storage"] = allocated_storage
        if associated_roles is not None:
            self._values["associated_roles"] = associated_roles
        if auto_minor_version_upgrade is not None:
            self._values["auto_minor_version_upgrade"] = auto_minor_version_upgrade
        if availability_zones is not None:
            self._values["availability_zones"] = availability_zones
        if backtrack_window is not None:
            self._values["backtrack_window"] = backtrack_window
        if backup_retention_period is not None:
            self._values["backup_retention_period"] = backup_retention_period
        if copy_tags_to_snapshot is not None:
            self._values["copy_tags_to_snapshot"] = copy_tags_to_snapshot
        if database_name is not None:
            self._values["database_name"] = database_name
        if db_cluster_identifier is not None:
            self._values["db_cluster_identifier"] = db_cluster_identifier
        if db_cluster_instance_class is not None:
            self._values["db_cluster_instance_class"] = db_cluster_instance_class
        if db_cluster_parameter_group_name is not None:
            self._values["db_cluster_parameter_group_name"] = db_cluster_parameter_group_name
        if db_instance_parameter_group_name is not None:
            self._values["db_instance_parameter_group_name"] = db_instance_parameter_group_name
        if db_subnet_group_name is not None:
            self._values["db_subnet_group_name"] = db_subnet_group_name
        if db_system_id is not None:
            self._values["db_system_id"] = db_system_id
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if domain is not None:
            self._values["domain"] = domain
        if domain_iam_role_name is not None:
            self._values["domain_iam_role_name"] = domain_iam_role_name
        if enable_cloudwatch_logs_exports is not None:
            self._values["enable_cloudwatch_logs_exports"] = enable_cloudwatch_logs_exports
        if enable_http_endpoint is not None:
            self._values["enable_http_endpoint"] = enable_http_endpoint
        if enable_iam_database_authentication is not None:
            self._values["enable_iam_database_authentication"] = enable_iam_database_authentication
        if engine is not None:
            self._values["engine"] = engine
        if engine_mode is not None:
            self._values["engine_mode"] = engine_mode
        if engine_version is not None:
            self._values["engine_version"] = engine_version
        if global_cluster_identifier is not None:
            self._values["global_cluster_identifier"] = global_cluster_identifier
        if iops is not None:
            self._values["iops"] = iops
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id
        if manage_master_user_password is not None:
            self._values["manage_master_user_password"] = manage_master_user_password
        if master_username is not None:
            self._values["master_username"] = master_username
        if master_user_password is not None:
            self._values["master_user_password"] = master_user_password
        if master_user_secret is not None:
            self._values["master_user_secret"] = master_user_secret
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role_arn is not None:
            self._values["monitoring_role_arn"] = monitoring_role_arn
        if network_type is not None:
            self._values["network_type"] = network_type
        if performance_insights_enabled is not None:
            self._values["performance_insights_enabled"] = performance_insights_enabled
        if performance_insights_kms_key_id is not None:
            self._values["performance_insights_kms_key_id"] = performance_insights_kms_key_id
        if performance_insights_retention_period is not None:
            self._values["performance_insights_retention_period"] = performance_insights_retention_period
        if port is not None:
            self._values["port"] = port
        if preferred_backup_window is not None:
            self._values["preferred_backup_window"] = preferred_backup_window
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if publicly_accessible is not None:
            self._values["publicly_accessible"] = publicly_accessible
        if replication_source_identifier is not None:
            self._values["replication_source_identifier"] = replication_source_identifier
        if restore_to_time is not None:
            self._values["restore_to_time"] = restore_to_time
        if restore_type is not None:
            self._values["restore_type"] = restore_type
        if scaling_configuration is not None:
            self._values["scaling_configuration"] = scaling_configuration
        if serverless_v2_scaling_configuration is not None:
            self._values["serverless_v2_scaling_configuration"] = serverless_v2_scaling_configuration
        if snapshot_identifier is not None:
            self._values["snapshot_identifier"] = snapshot_identifier
        if source_db_cluster_identifier is not None:
            self._values["source_db_cluster_identifier"] = source_db_cluster_identifier
        if source_region is not None:
            self._values["source_region"] = source_region
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted
        if storage_type is not None:
            self._values["storage_type"] = storage_type
        if tags is not None:
            self._values["tags"] = tags
        if use_latest_restorable_time is not None:
            self._values["use_latest_restorable_time"] = use_latest_restorable_time
        if vpc_security_group_ids is not None:
            self._values["vpc_security_group_ids"] = vpc_security_group_ids

    @builtins.property
    def allocated_storage(self) -> typing.Optional[jsii.Number]:
        '''The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.

        Valid for Cluster Type: Multi-AZ DB clusters only

        This setting is required to create a Multi-AZ DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-allocatedstorage
        '''
        result = self._values.get("allocated_storage")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBCluster.DBClusterRoleProperty]]]]:
        '''Provides a list of the AWS Identity and Access Management (IAM) roles that are associated with the DB cluster.

        IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other Amazon Web Services on your behalf.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-associatedroles
        '''
        result = self._values.get("associated_roles")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBCluster.DBClusterRoleProperty]]]], result)

    @builtins.property
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether minor engine upgrades are applied automatically to the DB cluster during the maintenance window.

        By default, minor engine upgrades are applied automatically.

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-autominorversionupgrade
        '''
        result = self._values.get("auto_minor_version_upgrade")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of Availability Zones (AZs) where instances in the DB cluster can be created.

        For information on AWS Regions and Availability Zones, see `Choosing the Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-availabilityzones
        '''
        result = self._values.get("availability_zones")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def backtrack_window(self) -> typing.Optional[jsii.Number]:
        '''The target backtrack window, in seconds. To disable backtracking, set this value to 0.

        .. epigraph::

           Currently, Backtrack is only supported for Aurora MySQL DB clusters.

        Default: 0

        Constraints:

        - If specified, this value must be set to a number from 0 to 259,200 (72 hours).

        Valid for: Aurora MySQL DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backtrackwindow
        '''
        result = self._values.get("backtrack_window")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days for which automated backups are retained.

        Default: 1

        Constraints:

        - Must be a value from 1 to 35

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backupretentionperiod
        '''
        result = self._values.get("backup_retention_period")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to copy all tags from the DB cluster to snapshots of the DB cluster.

        The default is not to copy them.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-copytagstosnapshot
        '''
        result = self._values.get("copy_tags_to_snapshot")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def database_name(self) -> typing.Optional[builtins.str]:
        '''The name of your database.

        If you don't provide a name, then Amazon RDS won't create a database in this DB cluster. For naming constraints, see `Naming Constraints <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-databasename
        '''
        result = self._values.get("database_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The DB cluster identifier. This parameter is stored as a lowercase string.

        Constraints:

        - Must contain from 1 to 63 letters, numbers, or hyphens.
        - First character must be a letter.
        - Can't end with a hyphen or contain two consecutive hyphens.

        Example: ``my-cluster1``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusteridentifier
        '''
        result = self._values.get("db_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_instance_class(self) -> typing.Optional[builtins.str]:
        '''The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example ``db.m6gd.xlarge`` . Not all DB instance classes are available in all AWS Regions , or for all database engines.

        For the full list of DB instance classes and availability for your engine, see `DB instance class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide* .

        This setting is required to create a Multi-AZ DB cluster.

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterinstanceclass
        '''
        result = self._values.get("db_cluster_instance_class")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB cluster parameter group to associate with this DB cluster.

        .. epigraph::

           If you apply a parameter group to an existing DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting.

           If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started.

        To list all of the available DB cluster parameter group names, use the following command:

        ``aws rds describe-db-cluster-parameter-groups --query "DBClusterParameterGroups[].DBClusterParameterGroupName" --output text``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterparametergroupname
        '''
        result = self._values.get("db_cluster_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_instance_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB parameter group to apply to all instances of the DB cluster.

        .. epigraph::

           When you apply a parameter group using the ``DBInstanceParameterGroupName`` parameter, the DB cluster isn't rebooted automatically. Also, parameter changes are applied immediately rather than during the next maintenance window.

        Default: The existing name setting

        Constraints:

        - The DB parameter group must be in the same DB parameter group family as this DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbinstanceparametergroupname
        '''
        result = self._values.get("db_instance_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''A DB subnet group that you want to associate with this DB cluster.

        If you are restoring a DB cluster to a point in time with ``RestoreType`` set to ``copy-on-write`` , and don't specify a DB subnet group name, then the DB cluster is restored with a default DB subnet group.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsubnetgroupname
        '''
        result = self._values.get("db_subnet_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_system_id(self) -> typing.Optional[builtins.str]:
        '''Reserved for future use.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsystemid
        '''
        result = self._values.get("db_system_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether the DB cluster has deletion protection enabled.

        The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-deletionprotection
        '''
        result = self._values.get("deletion_protection")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        '''Indicates the directory ID of the Active Directory to create the DB cluster.

        For Amazon Aurora DB clusters, Amazon RDS can use Kerberos authentication to authenticate users that connect to the DB cluster.

        For more information, see `Kerberos authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/kerberos-authentication.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-domain
        '''
        result = self._values.get("domain")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        '''Specifies the name of the IAM role to use when making API calls to the Directory Service.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-domainiamrolename
        '''
        result = self._values.get("domain_iam_role_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of log types that need to be enabled for exporting to CloudWatch Logs.

        The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Aurora User Guide* .

        *Aurora MySQL*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Aurora PostgreSQL*

        Valid values: ``postgresql``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablecloudwatchlogsexports
        '''
        result = self._values.get("enable_cloudwatch_logs_exports")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def enable_http_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to enable the HTTP endpoint for an Aurora Serverless DB cluster.

        By default, the HTTP endpoint is disabled.

        When enabled, the HTTP endpoint provides a connectionless web service API for running SQL queries on the Aurora Serverless DB cluster. You can also query your database from inside the RDS console with the query editor.

        For more information, see `Using the Data API for Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablehttpendpoint
        '''
        result = self._values.get("enable_http_endpoint")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        By default, mapping is disabled.

        For more information, see `IAM Database Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon Aurora User Guide.*

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enableiamdatabaseauthentication
        '''
        result = self._values.get("enable_iam_database_authentication")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine to be used for this DB cluster.

        Valid Values:

        - ``aurora-mysql``
        - ``aurora-postgresql``
        - ``mysql``
        - ``postgres``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engine
        '''
        result = self._values.get("engine")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def engine_mode(self) -> typing.Optional[builtins.str]:
        '''The DB engine mode of the DB cluster, either ``provisioned`` or ``serverless`` .

        The ``serverless`` engine mode only supports Aurora Serverless v1.

        Limitations and requirements apply to some DB engine modes. For more information, see the following sections in the *Amazon Aurora User Guide* :

        - `Limitations of Aurora Serverless v1 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations>`_
        - `Requirements for Aurora Serverless v2 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.requirements.html>`_
        - `Limitations of parallel query <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-parallel-query.html#aurora-mysql-parallel-query-limitations>`_
        - `Limitations of Aurora global databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html#aurora-global-database.limitations>`_

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enginemode
        '''
        result = self._values.get("engine_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The version number of the database engine to use.

        To list all of the available engine versions for Aurora MySQL version 2 (5.7-compatible) and version 3 (8.0-compatible), use the following command:

        ``aws rds describe-db-engine-versions --engine aurora-mysql --query "DBEngineVersions[].EngineVersion"``

        You can supply either ``5.7`` or ``8.0`` to use the default engine version for Aurora MySQL version 2 or version 3, respectively.

        To list all of the available engine versions for Aurora PostgreSQL, use the following command:

        ``aws rds describe-db-engine-versions --engine aurora-postgresql --query "DBEngineVersions[].EngineVersion"``

        To list all of the available engine versions for RDS for MySQL, use the following command:

        ``aws rds describe-db-engine-versions --engine mysql --query "DBEngineVersions[].EngineVersion"``

        To list all of the available engine versions for RDS for PostgreSQL, use the following command:

        ``aws rds describe-db-engine-versions --engine postgres --query "DBEngineVersions[].EngineVersion"``

        *Aurora MySQL*

        For information, see `Database engine updates for Amazon Aurora MySQL <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Updates.html>`_ in the *Amazon Aurora User Guide* .

        *Aurora PostgreSQL*

        For information, see `Amazon Aurora PostgreSQL releases and engine versions <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Updates.20180305.html>`_ in the *Amazon Aurora User Guide* .

        *MySQL*

        For information, see `Amazon RDS for MySQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide* .

        *PostgreSQL*

        For information, see `Amazon RDS for PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts>`_ in the *Amazon RDS User Guide* .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion
        '''
        result = self._values.get("engine_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''If you are configuring an Aurora global database cluster and want your Aurora DB cluster to be a secondary member in the global database cluster, specify the global cluster ID of the global database cluster.

        To define the primary database cluster of the global cluster, use the `AWS::RDS::GlobalCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html>`_ resource.

        If you aren't configuring a global database cluster, don't specify this property.
        .. epigraph::

           To remove the DB cluster from a global database cluster, specify an empty value for the ``GlobalClusterIdentifier`` property.

        For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-globalclusteridentifier
        '''
        result = self._values.get("global_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster.

        For information about valid IOPS values, see `Provisioned IOPS storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* .

        This setting is required to create a Multi-AZ DB cluster.

        Valid for Cluster Type: Multi-AZ DB clusters only

        Constraints:

        - Must be a multiple between .5 and 50 of the storage amount for the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-iops
        '''
        result = self._values.get("iops")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the AWS KMS key that is used to encrypt the database instances in the DB cluster, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` .

        If you enable the ``StorageEncrypted`` property but don't specify this property, the default KMS key is used. If you specify this property, you must set the ``StorageEncrypted`` property to ``true`` .

        If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-kmskeyid
        '''
        result = self._values.get("kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def manage_master_user_password(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to manage the master user password with AWS Secrets Manager.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

        Valid for Cluster Type: Aurora DB clusters and Multi-AZ DB clusters

        Constraints:

        - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-managemasteruserpassword
        '''
        result = self._values.get("manage_master_user_password")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def master_username(self) -> typing.Optional[builtins.str]:
        '''The name of the master user for the DB cluster.

        .. epigraph::

           If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusername
        '''
        result = self._values.get("master_username")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def master_user_password(self) -> typing.Optional[builtins.str]:
        '''The master password for the DB instance.

        .. epigraph::

           If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masteruserpassword
        '''
        result = self._values.get("master_user_password")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def master_user_secret(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBCluster.MasterUserSecretProperty]]:
        '''The secret managed by RDS in AWS Secrets Manager for the master user password.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusersecret
        '''
        result = self._values.get("master_user_secret")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBCluster.MasterUserSecretProperty]], result)

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        '''The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster.

        To turn off collecting Enhanced Monitoring metrics, specify ``0`` .

        If ``MonitoringRoleArn`` is specified, also set ``MonitoringInterval`` to a value other than ``0`` .

        Valid for Cluster Type: Multi-AZ DB clusters only

        Valid Values: ``0 | 1 | 5 | 10 | 15 | 30 | 60``

        Default: ``0``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-monitoringinterval
        '''
        result = self._values.get("monitoring_interval")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs.

        An example is ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting up and enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* .

        If ``MonitoringInterval`` is set to a value other than ``0`` , supply a ``MonitoringRoleArn`` value.

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-monitoringrolearn
        '''
        result = self._values.get("monitoring_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_type(self) -> typing.Optional[builtins.str]:
        '''The network type of the DB cluster.

        Valid values:

        - ``IPV4``
        - ``DUAL``

        The network type is determined by the ``DBSubnetGroup`` specified for the DB cluster. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ).

        For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon Aurora User Guide.*

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-networktype
        '''
        result = self._values.get("network_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def performance_insights_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to turn on Performance Insights for the DB cluster.

        For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* .

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightsenabled
        '''
        result = self._values.get("performance_insights_enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for encryption of Performance Insights data.

        The AWS KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.

        If you don't specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account . Your AWS account has a different default KMS key for each AWS Region .

        Valid for Cluster Type: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightskmskeyid
        '''
        result = self._values.get("performance_insights_kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days to retain Performance Insights data.

        Valid for Cluster Type: Multi-AZ DB clusters only

        Valid Values:

        - ``7``
        - *month* * 31, where *month* is a number of months from 1-23. Examples: ``93`` (3 months * 31), ``341`` (11 months * 31), ``589`` (19 months * 31)
        - ``731``

        Default: ``7`` days

        If you specify a retention period that isn't valid, such as ``94`` , Amazon RDS issues an error.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightsretentionperiod
        '''
        result = self._values.get("performance_insights_retention_period")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''The port number on which the DB instances in the DB cluster accept connections.

        Default:

        - When ``EngineMode`` is ``provisioned`` , ``3306`` (for both Aurora MySQL and Aurora PostgreSQL)
        - When ``EngineMode`` is ``serverless`` :
        - ``3306`` when ``Engine`` is ``aurora`` or ``aurora-mysql``
        - ``5432`` when ``Engine`` is ``aurora-postgresql``

        .. epigraph::

           The ``No interruption`` on update behavior only applies to DB clusters. If you are updating a DB instance, see `Port <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port>`_ for the AWS::RDS::DBInstance resource.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-port
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        '''The daily time range during which automated backups are created.

        For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.Backups.BackupWindow>`_ in the *Amazon Aurora User Guide.*

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi`` .
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredbackupwindow
        '''
        result = self._values.get("preferred_backup_window")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        '''The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).

        Format: ``ddd:hh24:mi-ddd:hh24:mi``

        The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Cluster Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow.Aurora>`_ in the *Amazon Aurora User Guide.*

        Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun.

        Constraints: Minimum 30-minute window.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredmaintenancewindow
        '''
        result = self._values.get("preferred_maintenance_window")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the DB cluster is publicly accessible.

        When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it uses. That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it.

        When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to a private IP address.

        Valid for Cluster Type: Multi-AZ DB clusters only

        Default: The default behavior varies depending on whether ``DBSubnetGroupName`` is specified.

        If ``DBSubnetGroupName`` isn't specified, and ``PubliclyAccessible`` isn't specified, the following applies:

        - If the default VPC in the target Region doesn’t have an internet gateway attached to it, the DB cluster is private.
        - If the default VPC in the target Region has an internet gateway attached to it, the DB cluster is public.

        If ``DBSubnetGroupName`` is specified, and ``PubliclyAccessible`` isn't specified, the following applies:

        - If the subnets are part of a VPC that doesn’t have an internet gateway attached to it, the DB cluster is private.
        - If the subnets are part of a VPC that has an internet gateway attached to it, the DB cluster is public.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-publiclyaccessible
        '''
        result = self._values.get("publicly_accessible")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def replication_source_identifier(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the source DB instance or DB cluster if this DB cluster is created as a read replica.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-replicationsourceidentifier
        '''
        result = self._values.get("replication_source_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def restore_to_time(self) -> typing.Optional[builtins.str]:
        '''The date and time to restore the DB cluster to.

        Valid Values: Value must be a time in Universal Coordinated Time (UTC) format

        Constraints:

        - Must be before the latest restorable time for the DB instance
        - Must be specified if ``UseLatestRestorableTime`` parameter isn't provided
        - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled
        - Can't be specified if the ``RestoreType`` parameter is ``copy-on-write``

        Example: ``2015-03-07T23:45:00Z``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-restoretotime
        '''
        result = self._values.get("restore_to_time")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def restore_type(self) -> typing.Optional[builtins.str]:
        '''The type of restore to be performed. You can specify one of the following values:.

        - ``full-copy`` - The new DB cluster is restored as a full copy of the source DB cluster.
        - ``copy-on-write`` - The new DB cluster is restored as a clone of the source DB cluster.

        If you don't specify a ``RestoreType`` value, then the new DB cluster is restored as a full copy of the source DB cluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-restoretype
        '''
        result = self._values.get("restore_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBCluster.ScalingConfigurationProperty]]:
        '''The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster.

        This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-scalingconfiguration
        '''
        result = self._values.get("scaling_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBCluster.ScalingConfigurationProperty]], result)

    @builtins.property
    def serverless_v2_scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBCluster.ServerlessV2ScalingConfigurationProperty]]:
        '''The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster.

        This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-serverlessv2scalingconfiguration
        '''
        result = self._values.get("serverless_v2_scaling_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBCluster.ServerlessV2ScalingConfigurationProperty]], result)

    @builtins.property
    def snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier for the DB snapshot or DB cluster snapshot to restore from.

        You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot.

        After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted.

        If you specify the ``SnapshotIdentifier`` property to restore a DB cluster (as opposed to specifying it for DB cluster updates), then don't specify the following properties:

        - ``GlobalClusterIdentifier``
        - ``MasterUsername``
        - ``MasterUserPassword``
        - ``ReplicationSourceIdentifier``
        - ``RestoreType``
        - ``SourceDBClusterIdentifier``
        - ``SourceRegion``
        - ``StorageEncrypted`` (for an encrypted snapshot)
        - ``UseLatestRestorableTime``

        Constraints:

        - Must match the identifier of an existing Snapshot.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-snapshotidentifier
        '''
        result = self._values.get("snapshot_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''When restoring a DB cluster to a point in time, the identifier of the source DB cluster from which to restore.

        Constraints:

        - Must match the identifier of an existing DBCluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourcedbclusteridentifier
        '''
        result = self._values.get("source_db_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_region(self) -> typing.Optional[builtins.str]:
        '''The AWS Region which contains the source DB cluster when replicating a DB cluster. For example, ``us-east-1`` .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourceregion
        '''
        result = self._values.get("source_region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether the DB cluster is encrypted.

        If you specify the ``KmsKeyId`` property, then you must enable encryption.

        If you specify the ``SourceDBClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB cluster is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB cluster to be encrypted, then don't set this property or set it to ``false`` .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storageencrypted
        '''
        result = self._values.get("storage_encrypted")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def storage_type(self) -> typing.Optional[builtins.str]:
        '''The storage type to associate with the DB cluster.

        For information on storage types for Aurora DB clusters, see `Storage configurations for Amazon Aurora DB clusters <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Overview.StorageReliability.html#aurora-storage-type>`_ . For information on storage types for Multi-AZ DB clusters, see `Settings for creating Multi-AZ DB clusters <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/create-multi-az-db-cluster.html#create-multi-az-db-cluster-settings>`_ .

        This setting is required to create a Multi-AZ DB cluster.

        When specified for a Multi-AZ DB cluster, a value for the ``Iops`` parameter is required.

        Valid for Cluster Type: Aurora DB clusters and Multi-AZ DB clusters

        Valid Values:

        - Aurora DB clusters - ``aurora | aurora-iopt1``
        - Multi-AZ DB clusters - ``io1``

        Default:

        - Aurora DB clusters - ``aurora``
        - Multi-AZ DB clusters - ``io1``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storagetype
        '''
        result = self._values.get("storage_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''An optional array of key-value pairs to apply to this DB cluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to restore the DB cluster to the latest restorable backup time.

        By default, the DB cluster is not restored to the latest restorable backup time.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-uselatestrestorabletime
        '''
        result = self._values.get("use_latest_restorable_time")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of EC2 VPC security groups to associate with this DB cluster.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-vpcsecuritygroupids
        '''
        result = self._values.get("vpc_security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDBInstance(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBInstance",
):
    '''A CloudFormation ``AWS::RDS::DBInstance``.

    The ``AWS::RDS::DBInstance`` resource creates an Amazon DB instance. The new DB instance can be an RDS DB instance, or it can be a DB instance in an Aurora DB cluster.

    For more information about creating an RDS DB instance, see `Creating an Amazon RDS DB instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_CreateDBInstance.html>`_ in the *Amazon RDS User Guide* .

    For more information about creating a DB instance in an Aurora DB cluster, see `Creating an Amazon Aurora DB cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.CreateInstance.html>`_ in the *Amazon Aurora User Guide* .

    If you import an existing DB instance, and the template configuration doesn't match the actual configuration of the DB instance, AWS CloudFormation applies the changes in the template during the import operation.
    .. epigraph::

       If a DB instance is deleted or replaced during an update, AWS CloudFormation deletes all automated snapshots. However, it retains manual DB snapshots. During an update that requires replacement, you can apply a stack policy to prevent DB instances from being replaced. For more information, see `Prevent Updates to Stack Resources <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/protect-stack-resources.html>`_ .

    *Updating DB instances*

    When properties labeled " *Update requires:* `Replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ " are updated, AWS CloudFormation first creates a replacement DB instance, then changes references from other dependent resources to point to the replacement DB instance, and finally deletes the old DB instance.
    .. epigraph::

       We highly recommend that you take a snapshot of the database before updating the stack. If you don't, you lose the data when AWS CloudFormation replaces your DB instance. To preserve your data, perform the following procedure:

       - Deactivate any applications that are using the DB instance so that there's no activity on the DB instance.
       - Create a snapshot of the DB instance. For more information, see `Creating a DB Snapshot <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_CreateSnapshot.html>`_ .
       - If you want to restore your instance using a DB snapshot, modify the updated template with your DB instance changes and add the ``DBSnapshotIdentifier`` property with the ID of the DB snapshot that you want to use.

       After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted.

       - Update the stack.

    For more information about updating other properties of this resource, see ``[ModifyDBInstance](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_ModifyDBInstance.html)`` . For more information about updating stacks, see `AWS CloudFormation Stacks Updates <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks.html>`_ .

    *Deleting DB instances*

    For DB instances that are part of an Aurora DB cluster, you can set a deletion policy for your DB instance to control how AWS CloudFormation handles the DB instance when the stack is deleted. For Amazon RDS DB instances, you can choose to *retain* the DB instance, to *delete* the DB instance, or to *create a snapshot* of the DB instance. The default AWS CloudFormation behavior depends on the ``DBClusterIdentifier`` property:

    - For ``AWS::RDS::DBInstance`` resources that don't specify the ``DBClusterIdentifier`` property, AWS CloudFormation saves a snapshot of the DB instance.
    - For ``AWS::RDS::DBInstance`` resources that do specify the ``DBClusterIdentifier`` property, AWS CloudFormation deletes the DB instance.

    For more information, see `DeletionPolicy Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html>`_ .

    :cloudformationResource: AWS::RDS::DBInstance
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_dBInstance = rds.CfnDBInstance(self, "MyCfnDBInstance",
            allocated_storage="allocatedStorage",
            allow_major_version_upgrade=False,
            associated_roles=[rds.CfnDBInstance.DBInstanceRoleProperty(
                feature_name="featureName",
                role_arn="roleArn"
            )],
            auto_minor_version_upgrade=False,
            availability_zone="availabilityZone",
            backup_retention_period=123,
            ca_certificate_identifier="caCertificateIdentifier",
            certificate_details=rds.CfnDBInstance.CertificateDetailsProperty(
                ca_identifier="caIdentifier",
                valid_till="validTill"
            ),
            certificate_rotation_restart=False,
            character_set_name="characterSetName",
            copy_tags_to_snapshot=False,
            custom_iam_instance_profile="customIamInstanceProfile",
            db_cluster_identifier="dbClusterIdentifier",
            db_cluster_snapshot_identifier="dbClusterSnapshotIdentifier",
            db_instance_class="dbInstanceClass",
            db_instance_identifier="dbInstanceIdentifier",
            db_name="dbName",
            db_parameter_group_name="dbParameterGroupName",
            db_security_groups=["dbSecurityGroups"],
            db_snapshot_identifier="dbSnapshotIdentifier",
            db_subnet_group_name="dbSubnetGroupName",
            delete_automated_backups=False,
            deletion_protection=False,
            domain="domain",
            domain_iam_role_name="domainIamRoleName",
            enable_cloudwatch_logs_exports=["enableCloudwatchLogsExports"],
            enable_iam_database_authentication=False,
            enable_performance_insights=False,
            endpoint=rds.CfnDBInstance.EndpointProperty(
                address="address",
                hosted_zone_id="hostedZoneId",
                port="port"
            ),
            engine="engine",
            engine_version="engineVersion",
            iops=123,
            kms_key_id="kmsKeyId",
            license_model="licenseModel",
            manage_master_user_password=False,
            master_username="masterUsername",
            master_user_password="masterUserPassword",
            master_user_secret=rds.CfnDBInstance.MasterUserSecretProperty(
                kms_key_id="kmsKeyId",
                secret_arn="secretArn"
            ),
            max_allocated_storage=123,
            monitoring_interval=123,
            monitoring_role_arn="monitoringRoleArn",
            multi_az=False,
            nchar_character_set_name="ncharCharacterSetName",
            network_type="networkType",
            option_group_name="optionGroupName",
            performance_insights_kms_key_id="performanceInsightsKmsKeyId",
            performance_insights_retention_period=123,
            port="port",
            preferred_backup_window="preferredBackupWindow",
            preferred_maintenance_window="preferredMaintenanceWindow",
            processor_features=[rds.CfnDBInstance.ProcessorFeatureProperty(
                name="name",
                value="value"
            )],
            promotion_tier=123,
            publicly_accessible=False,
            replica_mode="replicaMode",
            restore_time="restoreTime",
            source_db_cluster_identifier="sourceDbClusterIdentifier",
            source_db_instance_automated_backups_arn="sourceDbInstanceAutomatedBackupsArn",
            source_db_instance_identifier="sourceDbInstanceIdentifier",
            source_dbi_resource_id="sourceDbiResourceId",
            source_region="sourceRegion",
            storage_encrypted=False,
            storage_throughput=123,
            storage_type="storageType",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            timezone="timezone",
            use_default_processor_features=False,
            use_latest_restorable_time=False,
            vpc_security_groups=["vpcSecurityGroups"]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        allocated_storage: typing.Optional[builtins.str] = None,
        allow_major_version_upgrade: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        associated_roles: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBInstance.DBInstanceRoleProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        ca_certificate_identifier: typing.Optional[builtins.str] = None,
        certificate_details: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBInstance.CertificateDetailsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        certificate_rotation_restart: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        character_set_name: typing.Optional[builtins.str] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        custom_iam_instance_profile: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_instance_class: typing.Optional[builtins.str] = None,
        db_instance_identifier: typing.Optional[builtins.str] = None,
        db_name: typing.Optional[builtins.str] = None,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        db_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        db_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        delete_automated_backups: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.Sequence[builtins.str]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        enable_performance_insights: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        endpoint: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBInstance.EndpointProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        license_model: typing.Optional[builtins.str] = None,
        manage_master_user_password: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        master_user_secret: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBInstance.MasterUserSecretProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        max_allocated_storage: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        multi_az: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        nchar_character_set_name: typing.Optional[builtins.str] = None,
        network_type: typing.Optional[builtins.str] = None,
        option_group_name: typing.Optional[builtins.str] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[builtins.str] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        processor_features: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBInstance.ProcessorFeatureProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        promotion_tier: typing.Optional[jsii.Number] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        replica_mode: typing.Optional[builtins.str] = None,
        restore_time: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        source_db_instance_automated_backups_arn: typing.Optional[builtins.str] = None,
        source_db_instance_identifier: typing.Optional[builtins.str] = None,
        source_dbi_resource_id: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        storage_throughput: typing.Optional[jsii.Number] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        timezone: typing.Optional[builtins.str] = None,
        use_default_processor_features: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        vpc_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBInstance``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param allocated_storage: The amount of storage in gibibytes (GiB) to be initially allocated for the database instance. .. epigraph:: If any value is set in the ``Iops`` parameter, ``AllocatedStorage`` must be at least 100 GiB, which corresponds to the minimum Iops value of 1,000. If you increase the ``Iops`` value (in 1,000 IOPS increments), then you must also increase the ``AllocatedStorage`` value (in 100-GiB increments). *Amazon Aurora* Not applicable. Aurora cluster volumes automatically grow as the amount of data in your database increases, though you are only charged for the space that you use in an Aurora cluster volume. *MySQL* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *MariaDB* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *PostgreSQL* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *Oracle* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 10 to 3072. *SQL Server* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): - Enterprise and Standard editions: Must be an integer from 20 to 16384. - Web and Express editions: Must be an integer from 20 to 16384. - Provisioned IOPS storage (io1): - Enterprise and Standard editions: Must be an integer from 20 to 16384. - Web and Express editions: Must be an integer from 20 to 16384. - Magnetic storage (standard): - Enterprise and Standard editions: Must be an integer from 20 to 1024. - Web and Express editions: Must be an integer from 20 to 1024.
        :param allow_major_version_upgrade: A value that indicates whether major version upgrades are allowed. Changing this parameter doesn't result in an outage and the change is asynchronously applied as soon as possible. Constraints: Major version upgrades must be allowed when specifying a value for the ``EngineVersion`` parameter that is a different major version than the DB instance's current version.
        :param associated_roles: The AWS Identity and Access Management (IAM) roles associated with the DB instance. *Amazon Aurora* Not applicable. The associated roles are managed by the DB cluster.
        :param auto_minor_version_upgrade: A value that indicates whether minor engine upgrades are applied automatically to the DB instance during the maintenance window. By default, minor engine upgrades are applied automatically.
        :param availability_zone: The Availability Zone (AZ) where the database will be created. For information on AWS Regions and Availability Zones, see `Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ . For Amazon Aurora, each Aurora DB cluster hosts copies of its storage in three separate Availability Zones. Specify one of these Availability Zones. Aurora automatically chooses an appropriate Availability Zone if you don't specify one. Default: A random, system-chosen Availability Zone in the endpoint's AWS Region . Constraints: - The ``AvailabilityZone`` parameter can't be specified if the DB instance is a Multi-AZ deployment. - The specified Availability Zone must be in the same AWS Region as the current endpoint. Example: ``us-east-1d``
        :param backup_retention_period: The number of days for which automated backups are retained. Setting this parameter to a positive number enables backups. Setting this parameter to 0 disables automated backups. *Amazon Aurora* Not applicable. The retention period for automated backups is managed by the DB cluster. Default: 1 Constraints: - Must be a value from 0 to 35 - Can't be set to 0 if the DB instance is a source to read replicas
        :param ca_certificate_identifier: The identifier of the CA certificate for this DB instance. .. epigraph:: Specifying or updating this property triggers a reboot. For more information about CA certificate identifiers for RDS DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide* . For more information about CA certificate identifiers for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .
        :param certificate_details: The details of the DB instance's server certificate.
        :param certificate_rotation_restart: Specifies whether the DB instance is restarted when you rotate your SSL/TLS certificate. By default, the DB instance is restarted when you rotate your SSL/TLS certificate. The certificate is not updated until the DB instance is restarted. .. epigraph:: Set this parameter only if you are *not* using SSL/TLS to connect to the DB instance. If you are using SSL/TLS to connect to the DB instance, follow the appropriate instructions for your DB engine to rotate your SSL/TLS certificate: - For more information about rotating your SSL/TLS certificate for RDS DB engines, see `Rotating Your SSL/TLS Certificate. <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide.* - For more information about rotating your SSL/TLS certificate for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* . This setting doesn't apply to RDS Custom DB instances.
        :param character_set_name: For supported engines, indicates that the DB instance should be associated with the specified character set. *Amazon Aurora* Not applicable. The character set is managed by the DB cluster. For more information, see `AWS::RDS::DBCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html>`_ .
        :param copy_tags_to_snapshot: Spcifies whether to copy tags from the DB instance to snapshots of the DB instance. By default, tags are not copied. This setting doesn't apply to Amazon Aurora DB instances. Copying tags to snapshots is managed by the DB cluster. Setting this value for an Aurora DB instance has no effect on the DB cluster setting.
        :param custom_iam_instance_profile: The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance. This setting is required for RDS Custom. Constraints: - The profile must exist in your account. - The profile must have an IAM role that Amazon EC2 has permissions to assume. - The instance profile name and the associated IAM role name must start with the prefix ``AWSRDSCustom`` . For the list of permissions required for the IAM role, see `Configure IAM and your VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-setup-orcl.html#custom-setup-orcl.iam-vpc>`_ in the *Amazon RDS User Guide* .
        :param db_cluster_identifier: The identifier of the DB cluster that the instance will belong to.
        :param db_cluster_snapshot_identifier: The identifier for the RDS for MySQL Multi-AZ DB cluster snapshot to restore from. For more information on Multi-AZ DB clusters, see `Multi-AZ DB cluster deployments <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html>`_ in the *Amazon RDS User Guide* . Constraints: - Must match the identifier of an existing Multi-AZ DB cluster snapshot. - Can't be specified when ``DBSnapshotIdentifier`` is specified. - Must be specified when ``DBSnapshotIdentifier`` isn't specified. - If you are restoring from a shared manual Multi-AZ DB cluster snapshot, the ``DBClusterSnapshotIdentifier`` must be the ARN of the shared snapshot. - Can't be the identifier of an Aurora DB cluster snapshot. - Can't be the identifier of an RDS for PostgreSQL Multi-AZ DB cluster snapshot.
        :param db_instance_class: The compute and memory capacity of the DB instance, for example, ``db.m4.large`` . Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes, and availability for your engine, see `DB Instance Class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide.* For more information about DB instance class pricing and AWS Region support for DB instance classes, see `Amazon RDS Pricing <https://docs.aws.amazon.com/rds/pricing/>`_ .
        :param db_instance_identifier: A name for the DB instance. If you specify a name, AWS CloudFormation converts it to lowercase. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the DB instance. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* . .. epigraph:: If you specify a name, you can't perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param db_name: The meaning of this parameter differs according to the database engine you use. .. epigraph:: If you specify the ``[DBSnapshotIdentifier](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier)`` property, this property only applies to RDS for Oracle. *Amazon Aurora* Not applicable. The database name is managed by the DB cluster. *MySQL* The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Constraints: - Must contain 1 to 64 letters or numbers. - Can't be a word reserved by the specified database engine *MariaDB* The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Constraints: - Must contain 1 to 64 letters or numbers. - Can't be a word reserved by the specified database engine *PostgreSQL* The name of the database to create when the DB instance is created. If this parameter is not specified, the default ``postgres`` database is created in the DB instance. Constraints: - Must begin with a letter. Subsequent characters can be letters, underscores, or digits (0-9). - Must contain 1 to 63 characters. - Can't be a word reserved by the specified database engine *Oracle* The Oracle System ID (SID) of the created DB instance. If you specify ``null`` , the default value ``ORCL`` is used. You can't specify the string NULL, or any other reserved word, for ``DBName`` . Default: ``ORCL`` Constraints: - Can't be longer than 8 characters *SQL Server* Not applicable. Must be null.
        :param db_parameter_group_name: The name of an existing DB parameter group or a reference to an `AWS::RDS::DBParameterGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html>`_ resource created in the template. To list all of the available DB parameter group names, use the following command: ``aws rds describe-db-parameter-groups --query "DBParameterGroups[].DBParameterGroupName" --output text`` .. epigraph:: If any of the data members of the referenced parameter group are changed during an update, the DB instance might need to be restarted, which causes some interruption. If the parameter group contains static parameters, whether they were changed or not, an update triggers a reboot. If you don't specify a value for ``DBParameterGroupName`` property, the default DB parameter group for the specified engine and engine version is used.
        :param db_security_groups: A list of the DB security groups to assign to the DB instance. The list can include both the name of existing DB security groups or references to AWS::RDS::DBSecurityGroup resources created in the template. If you set DBSecurityGroups, you must not set VPCSecurityGroups, and vice versa. Also, note that the DBSecurityGroups property exists only for backwards compatibility with older regions and is no longer recommended for providing security information to an RDS DB instance. Instead, use VPCSecurityGroups. .. epigraph:: If you specify this property, AWS CloudFormation sends only the following properties (if specified) to Amazon RDS during create operations: - ``AllocatedStorage`` - ``AutoMinorVersionUpgrade`` - ``AvailabilityZone`` - ``BackupRetentionPeriod`` - ``CharacterSetName`` - ``DBInstanceClass`` - ``DBName`` - ``DBParameterGroupName`` - ``DBSecurityGroups`` - ``DBSubnetGroupName`` - ``Engine`` - ``EngineVersion`` - ``Iops`` - ``LicenseModel`` - ``MasterUsername`` - ``MasterUserPassword`` - ``MultiAZ`` - ``OptionGroupName`` - ``PreferredBackupWindow`` - ``PreferredMaintenanceWindow`` All other properties are ignored. Specify a virtual private cloud (VPC) security group if you want to submit other properties, such as ``StorageType`` , ``StorageEncrypted`` , or ``KmsKeyId`` . If you're already using the ``DBSecurityGroups`` property, you can't use these other properties by updating your DB instance to use a VPC security group. You must recreate the DB instance.
        :param db_snapshot_identifier: The name or Amazon Resource Name (ARN) of the DB snapshot that's used to restore the DB instance. If you're restoring from a shared manual DB snapshot, you must specify the ARN of the snapshot. By specifying this property, you can create a DB instance from the specified DB snapshot. If the ``DBSnapshotIdentifier`` property is an empty string or the ``AWS::RDS::DBInstance`` declaration has no ``DBSnapshotIdentifier`` property, AWS CloudFormation creates a new database. If the property contains a value (other than an empty string), AWS CloudFormation creates a database from the specified snapshot. If a snapshot with the specified name doesn't exist, AWS CloudFormation can't create the database and it rolls back the stack. Some DB instance properties aren't valid when you restore from a snapshot, such as the ``MasterUsername`` and ``MasterUserPassword`` properties. For information about the properties that you can specify, see the ``RestoreDBInstanceFromDBSnapshot`` action in the *Amazon RDS API Reference* . After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted. If you specify the ``DBSnapshotIdentifier`` property to restore a DB instance (as opposed to specifying it for DB instance updates), then don't specify the following properties: - ``CharacterSetName`` - ``DBClusterIdentifier`` - ``DBName`` - ``DeleteAutomatedBackups`` - ``EnablePerformanceInsights`` - ``KmsKeyId`` - ``MasterUsername`` - ``MasterUserPassword`` - ``PerformanceInsightsKMSKeyId`` - ``PerformanceInsightsRetentionPeriod`` - ``PromotionTier`` - ``SourceDBInstanceIdentifier`` - ``SourceRegion`` - ``StorageEncrypted`` (for an encrypted snapshot) - ``Timezone`` *Amazon Aurora* Not applicable. Snapshot restore is managed by the DB cluster.
        :param db_subnet_group_name: A DB subnet group to associate with the DB instance. If you update this value, the new subnet group must be a subnet group in a new VPC. If there's no DB subnet group, then the DB instance isn't a VPC DB instance. For more information about using Amazon RDS in a VPC, see `Using Amazon RDS with Amazon Virtual Private Cloud (VPC) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. The DB subnet group is managed by the DB cluster. If specified, the setting must match the DB cluster setting.
        :param delete_automated_backups: A value that indicates whether to remove automated backups immediately after the DB instance is deleted. This parameter isn't case-sensitive. The default is to remove automated backups immediately after the DB instance is deleted. *Amazon Aurora* Not applicable. When you delete a DB cluster, all automated backups for that DB cluster are deleted and can't be recovered. Manual DB cluster snapshots of the DB cluster are not deleted.
        :param deletion_protection: A value that indicates whether the DB instance has deletion protection enabled. The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. For more information, see `Deleting a DB Instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_DeleteInstance.html>`_ . *Amazon Aurora* Not applicable. You can enable or disable deletion protection for the DB cluster. For more information, see ``CreateDBCluster`` . DB instances in a DB cluster can be deleted even when deletion protection is enabled for the DB cluster.
        :param domain: The Active Directory directory ID to create the DB instance in. Currently, only Microsoft SQL Server, Oracle, and PostgreSQL DB instances can be created in an Active Directory Domain. For more information, see `Kerberos Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html>`_ in the *Amazon RDS User Guide* .
        :param domain_iam_role_name: The name of the IAM role to use when making API calls to the Directory Service. This setting doesn't apply to the following DB instances: - Amazon Aurora (The domain is managed by the DB cluster.) - RDS Custom
        :param enable_cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Relational Database Service User Guide* . *Amazon Aurora* Not applicable. CloudWatch Logs exports are managed by the DB cluster. *MariaDB* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Microsoft SQL Server* Valid values: ``agent`` , ``error`` *MySQL* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Oracle* Valid values: ``alert`` , ``audit`` , ``listener`` , ``trace`` *PostgreSQL* Valid values: ``postgresql`` , ``upgrade``
        :param enable_iam_database_authentication: A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. By default, mapping is disabled. This property is supported for RDS for MariaDB, RDS for MySQL, and RDS for PostgreSQL. For more information, see `IAM Database Authentication for MariaDB, MySQL, and PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon RDS User Guide.* *Amazon Aurora* Not applicable. Mapping AWS IAM accounts to database accounts is managed by the DB cluster.
        :param enable_performance_insights: Specifies whether to enable Performance Insights for the DB instance. For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* . This setting doesn't apply to RDS Custom DB instances.
        :param endpoint: The connection endpoint for the DB instance. .. epigraph:: The endpoint might not be shown for instances with the status of ``creating`` .
        :param engine: The name of the database engine that you want to use for this DB instance. .. epigraph:: When you are creating a DB instance, the ``Engine`` property is required. Valid Values: - ``aurora-mysql`` (for Aurora MySQL DB instances) - ``aurora-postgresql`` (for Aurora PostgreSQL DB instances) - ``custom-oracle-ee`` (for RDS Custom for Oracle DB instances) - ``custom-oracle-ee-cdb`` (for RDS Custom for Oracle DB instances) - ``custom-sqlserver-ee`` (for RDS Custom for SQL Server DB instances) - ``custom-sqlserver-se`` (for RDS Custom for SQL Server DB instances) - ``custom-sqlserver-web`` (for RDS Custom for SQL Server DB instances) - ``mariadb`` - ``mysql`` - ``oracle-ee`` - ``oracle-ee-cdb`` - ``oracle-se2`` - ``oracle-se2-cdb`` - ``postgres`` - ``sqlserver-ee`` - ``sqlserver-se`` - ``sqlserver-ex`` - ``sqlserver-web``
        :param engine_version: The version number of the database engine to use. For a list of valid engine versions, use the ``DescribeDBEngineVersions`` action. The following are the database engines and links to information about the major and minor versions that are available with Amazon RDS. Not every database engine is available for every AWS Region. *Amazon Aurora* Not applicable. The version number of the database engine to be used by the DB instance is managed by the DB cluster. *MariaDB* See `MariaDB on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MariaDB.html#MariaDB.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.* *Microsoft SQL Server* See `Microsoft SQL Server Versions on Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.VersionSupport>`_ in the *Amazon RDS User Guide.* *MySQL* See `MySQL on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.* *Oracle* See `Oracle Database Engine Release Notes <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.Oracle.PatchComposition.html>`_ in the *Amazon RDS User Guide.* *PostgreSQL* See `Supported PostgreSQL Database Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts.General.DBVersions>`_ in the *Amazon RDS User Guide.*
        :param iops: The number of I/O operations per second (IOPS) that the database provisions. The value must be equal to or greater than 1000. If you specify this property, you must follow the range of allowed ratios of your requested IOPS rate to the amount of storage that you allocate (IOPS to allocated storage). For example, you can provision an Oracle database instance with 1000 IOPS and 200 GiB of storage (a ratio of 5:1), or specify 2000 IOPS with 200 GiB of storage (a ratio of 10:1). For more information, see `Amazon RDS Provisioned IOPS Storage to Improve Performance <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* . .. epigraph:: If you specify ``io1`` for the ``StorageType`` property, then you must also specify the ``Iops`` property.
        :param kms_key_id: The ARN of the AWS KMS key that's used to encrypt the DB instance, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` . If you enable the StorageEncrypted property but don't specify this property, AWS CloudFormation uses the default KMS key. If you specify this property, you must set the StorageEncrypted property to true. If you specify the ``SourceDBInstanceIdentifier`` property, the value is inherited from the source DB instance if the read replica is created in the same region. If you create an encrypted read replica in a different AWS Region, then you must specify a KMS key for the destination AWS Region. KMS encryption keys are specific to the region that they're created in, and you can't use encryption keys from one region in another region. If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used. If you specify ``DBSecurityGroups`` , AWS CloudFormation ignores this property. To specify both a security group and this property, you must use a VPC security group. For more information about Amazon RDS and VPC, see `Using Amazon RDS with Amazon VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. The KMS key identifier is managed by the DB cluster.
        :param license_model: License model information for this DB instance. Valid values: - Aurora MySQL - ``general-public-license`` - Aurora PostgreSQL - ``postgresql-license`` - MariaDB - ``general-public-license`` - Microsoft SQL Server - ``license-included`` - MySQL - ``general-public-license`` - Oracle - ``bring-your-own-license`` or ``license-included`` - PostgreSQL - ``postgresql-license`` .. epigraph:: If you've specified ``DBSecurityGroups`` and then you update the license model, AWS CloudFormation replaces the underlying DB instance. This will incur some interruptions to database availability.
        :param manage_master_user_password: Specifies whether to manage the master user password with AWS Secrets Manager. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.* Constraints: - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.
        :param master_username: The master user name for the DB instance. .. epigraph:: If you specify the ``SourceDBInstanceIdentifier`` or ``DBSnapshotIdentifier`` property, don't specify this property. The value is inherited from the source DB instance or snapshot. *Amazon Aurora* Not applicable. The name for the master user is managed by the DB cluster. *MariaDB* Constraints: - Required for MariaDB. - Must be 1 to 16 letters or numbers. - Can't be a reserved word for the chosen database engine. *Microsoft SQL Server* Constraints: - Required for SQL Server. - Must be 1 to 128 letters or numbers. - The first character must be a letter. - Can't be a reserved word for the chosen database engine. *MySQL* Constraints: - Required for MySQL. - Must be 1 to 16 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine. *Oracle* Constraints: - Required for Oracle. - Must be 1 to 30 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine. *PostgreSQL* Constraints: - Required for PostgreSQL. - Must be 1 to 63 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine.
        :param master_user_password: The password for the master user. The password can include any printable ASCII character except "/", """, or "@". *Amazon Aurora* Not applicable. The password for the master user is managed by the DB cluster. *MariaDB* Constraints: Must contain from 8 to 41 characters. *Microsoft SQL Server* Constraints: Must contain from 8 to 128 characters. *MySQL* Constraints: Must contain from 8 to 41 characters. *Oracle* Constraints: Must contain from 8 to 30 characters. *PostgreSQL* Constraints: Must contain from 8 to 128 characters.
        :param master_user_secret: The secret managed by RDS in AWS Secrets Manager for the master user password. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*
        :param max_allocated_storage: The upper limit in gibibytes (GiB) to which Amazon RDS can automatically scale the storage of the DB instance. For more information about this setting, including limitations that apply to it, see `Managing capacity automatically with Amazon RDS storage autoscaling <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling>`_ in the *Amazon RDS User Guide* . This setting doesn't apply to the following DB instances: - Amazon Aurora (Storage is managed by the DB cluster.) - RDS Custom
        :param monitoring_interval: The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance. To disable collection of Enhanced Monitoring metrics, specify 0. The default is 0. If ``MonitoringRoleArn`` is specified, then you must set ``MonitoringInterval`` to a value other than 0. This setting doesn't apply to RDS Custom. Valid Values: ``0, 1, 5, 10, 15, 30, 60``
        :param monitoring_role_arn: The ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs. For example, ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting Up and Enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* . If ``MonitoringInterval`` is set to a value other than ``0`` , then you must supply a ``MonitoringRoleArn`` value. This setting doesn't apply to RDS Custom DB instances.
        :param multi_az: Specifies whether the database instance is a Multi-AZ DB instance deployment. You can't set the ``AvailabilityZone`` parameter if the ``MultiAZ`` parameter is set to true. For more information, see `Multi-AZ deployments for high availability <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.MultiAZ.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. Amazon Aurora storage is replicated across all of the Availability Zones and doesn't require the ``MultiAZ`` option to be set.
        :param nchar_character_set_name: The name of the NCHAR character set for the Oracle DB instance. This setting doesn't apply to RDS Custom DB instances.
        :param network_type: The network type of the DB instance. Valid values: - ``IPV4`` - ``DUAL`` The network type is determined by the ``DBSubnetGroup`` specified for the DB instance. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ). For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon RDS User Guide.*
        :param option_group_name: Indicates that the DB instance should be associated with the specified option group. Permanent options, such as the TDE option for Oracle Advanced Security TDE, can't be removed from an option group. Also, that option group can't be removed from a DB instance once it is associated with a DB instance.
        :param performance_insights_kms_key_id: The AWS KMS key identifier for encryption of Performance Insights data. The KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. If you do not specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account. Your AWS account has a different default KMS key for each AWS Region. For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .
        :param performance_insights_retention_period: The number of days to retain Performance Insights data. This setting doesn't apply to RDS Custom DB instances. Valid Values: - ``7`` - *month* * 31, where *month* is a number of months from 1-23. Examples: ``93`` (3 months * 31), ``341`` (11 months * 31), ``589`` (19 months * 31) - ``731`` Default: ``7`` days If you specify a retention period that isn't valid, such as ``94`` , Amazon RDS returns an error.
        :param port: The port number on which the database accepts connections. *Amazon Aurora* Not applicable. The port number is managed by the DB cluster.
        :param preferred_backup_window: The daily time range during which automated backups are created if automated backups are enabled, using the ``BackupRetentionPeriod`` parameter. For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow>`_ in the *Amazon RDS User Guide.* Constraints: - Must be in the format ``hh24:mi-hh24:mi`` . - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. *Amazon Aurora* Not applicable. The daily time range for creating automated backups is managed by the DB cluster.
        :param preferred_maintenance_window: The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC). Format: ``ddd:hh24:mi-ddd:hh24:mi`` The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Instance Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow>`_ in the *Amazon RDS User Guide.* .. epigraph:: This property applies when AWS CloudFormation initially creates the DB instance. If you use AWS CloudFormation to update the DB instance, those updates are applied immediately. Constraints: Minimum 30-minute window.
        :param processor_features: The number of CPU cores and the number of threads per core for the DB instance class of the DB instance. This setting doesn't apply to Amazon Aurora or RDS Custom DB instances.
        :param promotion_tier: The order of priority in which an Aurora Replica is promoted to the primary instance after a failure of the existing primary instance. For more information, see `Fault Tolerance for an Aurora DB Cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.AuroraHighAvailability.html#Aurora.Managing.FaultTolerance>`_ in the *Amazon Aurora User Guide* . This setting doesn't apply to RDS Custom DB instances. Default: ``1`` Valid Values: ``0 - 15``
        :param publicly_accessible: Indicates whether the DB instance is an internet-facing instance. If you specify true, AWS CloudFormation creates an instance with a publicly resolvable DNS name, which resolves to a public IP address. If you specify false, AWS CloudFormation creates an internal instance with a DNS name that resolves to a private IP address. The default behavior value depends on your VPC setup and the database subnet group. For more information, see the ``PubliclyAccessible`` parameter in the `CreateDBInstance <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html>`_ in the *Amazon RDS API Reference* .
        :param replica_mode: The open mode of an Oracle read replica. For more information, see `Working with Oracle Read Replicas for Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html>`_ in the *Amazon RDS User Guide* . This setting is only supported in RDS for Oracle. Default: ``open-read-only`` Valid Values: ``open-read-only`` or ``mounted``
        :param restore_time: The date and time to restore from. Valid Values: Value must be a time in Universal Coordinated Time (UTC) format Constraints: - Must be before the latest restorable time for the DB instance - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled Example: ``2009-09-07T23:45:00Z``
        :param source_db_cluster_identifier: The identifier of the Multi-AZ DB cluster that will act as the source for the read replica. Each DB cluster can have up to 15 read replicas. Constraints: - Must be the identifier of an existing Multi-AZ DB cluster. - Can't be specified if the ``SourceDBInstanceIdentifier`` parameter is also specified. - The specified DB cluster must have automatic backups enabled, that is, its backup retention period must be greater than 0. - The source DB cluster must be in the same AWS Region as the read replica. Cross-Region replication isn't supported.
        :param source_db_instance_automated_backups_arn: The Amazon Resource Name (ARN) of the replicated automated backups from which to restore, for example, ``arn:aws:rds:useast-1:123456789012:auto-backup:ab-L2IJCEXJP7XQ7HOJ4SIEXAMPLE`` . This setting doesn't apply to RDS Custom.
        :param source_db_instance_identifier: If you want to create a read replica DB instance, specify the ID of the source DB instance. Each DB instance can have a limited number of read replicas. For more information, see `Working with Read Replicas <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html>`_ in the *Amazon RDS User Guide* . For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* . The ``SourceDBInstanceIdentifier`` property determines whether a DB instance is a read replica. If you remove the ``SourceDBInstanceIdentifier`` property from your template and then update your stack, AWS CloudFormation promotes the Read Replica to a standalone DB instance. .. epigraph:: - If you specify a source DB instance that uses VPC security groups, we recommend that you specify the ``VPCSecurityGroups`` property. If you don't specify the property, the read replica inherits the value of the ``VPCSecurityGroups`` property from the source DB when you create the replica. However, if you update the stack, AWS CloudFormation reverts the replica's ``VPCSecurityGroups`` property to the default value because it's not defined in the stack's template. This change might cause unexpected issues. - Read replicas don't support deletion policies. AWS CloudFormation ignores any deletion policy that's associated with a read replica. - If you specify ``SourceDBInstanceIdentifier`` , don't specify the ``DBSnapshotIdentifier`` property. You can't create a read replica from a snapshot. - Don't set the ``BackupRetentionPeriod`` , ``DBName`` , ``MasterUsername`` , ``MasterUserPassword`` , and ``PreferredBackupWindow`` properties. The database attributes are inherited from the source DB instance, and backups are disabled for read replicas. - If the source DB instance is in a different region than the read replica, specify the source region in ``SourceRegion`` , and specify an ARN for a valid DB instance in ``SourceDBInstanceIdentifier`` . For more information, see `Constructing a Amazon RDS Amazon Resource Name (ARN) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html#USER_Tagging.ARN>`_ in the *Amazon RDS User Guide* . - For DB instances in Amazon Aurora clusters, don't specify this property. Amazon RDS automatically assigns writer and reader DB instances.
        :param source_dbi_resource_id: The resource ID of the source DB instance from which to restore.
        :param source_region: The ID of the region that contains the source DB instance for the read replica.
        :param storage_encrypted: A value that indicates whether the DB instance is encrypted. By default, it isn't encrypted. If you specify the ``KmsKeyId`` property, then you must enable encryption. If you specify the ``SourceDBInstanceIdentifier`` property, don't specify this property. The value is inherited from the source DB instance, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB instance is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB instance to be encrypted, then don't set this property or set it to ``false`` . *Amazon Aurora* Not applicable. The encryption for DB instances is managed by the DB cluster.
        :param storage_throughput: Specifies the storage throughput value for the DB instance. This setting applies only to the ``gp3`` storage type. This setting doesn't apply to RDS Custom or Amazon Aurora.
        :param storage_type: Specifies the storage type to be associated with the DB instance. Valid values: ``gp2 | gp3 | io1 | standard`` The ``standard`` value is also known as magnetic. If you specify ``io1`` or ``gp3`` , you must also include a value for the ``Iops`` parameter. Default: ``io1`` if the ``Iops`` parameter is specified, otherwise ``gp2`` For more information, see `Amazon RDS DB Instance Storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. Aurora data is stored in the cluster volume, which is a single, virtual volume that uses solid state drives (SSDs).
        :param tags: An optional array of key-value pairs to apply to this DB instance.
        :param timezone: The time zone of the DB instance. The time zone parameter is currently supported only by `Microsoft SQL Server <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone>`_ .
        :param use_default_processor_features: Specifies whether the DB instance class of the DB instance uses its default processor features. This setting doesn't apply to RDS Custom DB instances.
        :param use_latest_restorable_time: A value that indicates whether the DB instance is restored from the latest backup time. By default, the DB instance isn't restored from the latest backup time. Constraints: Can't be specified if the ``RestoreTime`` parameter is provided.
        :param vpc_security_groups: A list of the VPC security group IDs to assign to the DB instance. The list can include both the physical IDs of existing VPC security groups and references to `AWS::EC2::SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template. If you plan to update the resource, don't specify VPC security groups in a shared VPC. If you set ``VPCSecurityGroups`` , you must not set ```DBSecurityGroups`` <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ , and vice versa. .. epigraph:: You can migrate a DB instance in your stack from an RDS DB security group to a VPC security group, but keep the following in mind: - You can't revert to using an RDS security group after you establish a VPC security group membership. - When you migrate your DB instance to VPC security groups, if your stack update rolls back because the DB instance update fails or because an update fails in another AWS CloudFormation resource, the rollback fails because it can't revert to an RDS security group. - To use the properties that are available when you use a VPC security group, you must recreate the DB instance. If you don't, AWS CloudFormation submits only the property values that are listed in the ```DBSecurityGroups`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ property. To avoid this situation, migrate your DB instance to using VPC security groups only when that is the only change in your stack template. *Amazon Aurora* Not applicable. The associated list of EC2 VPC security groups is managed by the DB cluster. If specified, the setting must match the DB cluster setting.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__255b0779ca741853674876540bf77279f6293bea05de2cd18724d2b920ac5202)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBInstanceProps(
            allocated_storage=allocated_storage,
            allow_major_version_upgrade=allow_major_version_upgrade,
            associated_roles=associated_roles,
            auto_minor_version_upgrade=auto_minor_version_upgrade,
            availability_zone=availability_zone,
            backup_retention_period=backup_retention_period,
            ca_certificate_identifier=ca_certificate_identifier,
            certificate_details=certificate_details,
            certificate_rotation_restart=certificate_rotation_restart,
            character_set_name=character_set_name,
            copy_tags_to_snapshot=copy_tags_to_snapshot,
            custom_iam_instance_profile=custom_iam_instance_profile,
            db_cluster_identifier=db_cluster_identifier,
            db_cluster_snapshot_identifier=db_cluster_snapshot_identifier,
            db_instance_class=db_instance_class,
            db_instance_identifier=db_instance_identifier,
            db_name=db_name,
            db_parameter_group_name=db_parameter_group_name,
            db_security_groups=db_security_groups,
            db_snapshot_identifier=db_snapshot_identifier,
            db_subnet_group_name=db_subnet_group_name,
            delete_automated_backups=delete_automated_backups,
            deletion_protection=deletion_protection,
            domain=domain,
            domain_iam_role_name=domain_iam_role_name,
            enable_cloudwatch_logs_exports=enable_cloudwatch_logs_exports,
            enable_iam_database_authentication=enable_iam_database_authentication,
            enable_performance_insights=enable_performance_insights,
            endpoint=endpoint,
            engine=engine,
            engine_version=engine_version,
            iops=iops,
            kms_key_id=kms_key_id,
            license_model=license_model,
            manage_master_user_password=manage_master_user_password,
            master_username=master_username,
            master_user_password=master_user_password,
            master_user_secret=master_user_secret,
            max_allocated_storage=max_allocated_storage,
            monitoring_interval=monitoring_interval,
            monitoring_role_arn=monitoring_role_arn,
            multi_az=multi_az,
            nchar_character_set_name=nchar_character_set_name,
            network_type=network_type,
            option_group_name=option_group_name,
            performance_insights_kms_key_id=performance_insights_kms_key_id,
            performance_insights_retention_period=performance_insights_retention_period,
            port=port,
            preferred_backup_window=preferred_backup_window,
            preferred_maintenance_window=preferred_maintenance_window,
            processor_features=processor_features,
            promotion_tier=promotion_tier,
            publicly_accessible=publicly_accessible,
            replica_mode=replica_mode,
            restore_time=restore_time,
            source_db_cluster_identifier=source_db_cluster_identifier,
            source_db_instance_automated_backups_arn=source_db_instance_automated_backups_arn,
            source_db_instance_identifier=source_db_instance_identifier,
            source_dbi_resource_id=source_dbi_resource_id,
            source_region=source_region,
            storage_encrypted=storage_encrypted,
            storage_throughput=storage_throughput,
            storage_type=storage_type,
            tags=tags,
            timezone=timezone,
            use_default_processor_features=use_default_processor_features,
            use_latest_restorable_time=use_latest_restorable_time,
            vpc_security_groups=vpc_security_groups,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ae03ec324e0070f7cc8c2f2e303d0f6e50d95aaceae332584867e610809fabd)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__21c13f3ff42101254dd0145b4f37f31574e3048d1f2f05b59ca91ffabc98da29)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCertificateDetailsCaIdentifier")
    def attr_certificate_details_ca_identifier(self) -> builtins.str:
        '''The CA identifier of the CA certificate used for the DB instance's server certificate.

        :cloudformationAttribute: CertificateDetails.CAIdentifier
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateDetailsCaIdentifier"))

    @builtins.property
    @jsii.member(jsii_name="attrCertificateDetailsValidTill")
    def attr_certificate_details_valid_till(self) -> builtins.str:
        '''The expiration date of the DB instance’s server certificate.

        :cloudformationAttribute: CertificateDetails.ValidTill
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateDetailsValidTill"))

    @builtins.property
    @jsii.member(jsii_name="attrDbInstanceArn")
    def attr_db_instance_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) for the DB instance.

        :cloudformationAttribute: DBInstanceArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbInstanceArn"))

    @builtins.property
    @jsii.member(jsii_name="attrDbiResourceId")
    def attr_dbi_resource_id(self) -> builtins.str:
        '''The AWS Region-unique, immutable identifier for the DB instance.

        This identifier is found in AWS CloudTrail log entries whenever the AWS KMS key for the DB instance is accessed.

        :cloudformationAttribute: DbiResourceId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbiResourceId"))

    @builtins.property
    @jsii.member(jsii_name="attrDbSystemId")
    def attr_db_system_id(self) -> builtins.str:
        '''The Oracle system ID (Oracle SID) for a container database (CDB).

        The Oracle SID is also the name of the CDB.

        This setting is valid for RDS Custom only.

        :cloudformationAttribute: DBSystemId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbSystemId"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointAddress")
    def attr_endpoint_address(self) -> builtins.str:
        '''The connection endpoint for the database. For example: ``mystack-mydb-1apw1j4phylrk.cg034hpkmmjt.us-east-2.rds.amazonaws.com``.

        For Aurora Serverless DB clusters, the connection endpoint only applies to the DB cluster.

        :cloudformationAttribute: Endpoint.Address
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointAddress"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointHostedZoneId")
    def attr_endpoint_hosted_zone_id(self) -> builtins.str:
        '''The ID that Amazon Route 53 assigns when you create a hosted zone.

        :cloudformationAttribute: Endpoint.HostedZoneId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointHostedZoneId"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointPort")
    def attr_endpoint_port(self) -> builtins.str:
        '''The port number on which the database accepts connections.

        For example: ``3306``

        :cloudformationAttribute: Endpoint.Port
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointPort"))

    @builtins.property
    @jsii.member(jsii_name="attrMasterUserSecretSecretArn")
    def attr_master_user_secret_secret_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the secret.

        :cloudformationAttribute: MasterUserSecret.SecretArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMasterUserSecretSecretArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''An optional array of key-value pairs to apply to this DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="allocatedStorage")
    def allocated_storage(self) -> typing.Optional[builtins.str]:
        '''The amount of storage in gibibytes (GiB) to be initially allocated for the database instance.

        .. epigraph::

           If any value is set in the ``Iops`` parameter, ``AllocatedStorage`` must be at least 100 GiB, which corresponds to the minimum Iops value of 1,000. If you increase the ``Iops`` value (in 1,000 IOPS increments), then you must also increase the ``AllocatedStorage`` value (in 100-GiB increments).

        *Amazon Aurora*

        Not applicable. Aurora cluster volumes automatically grow as the amount of data in your database increases, though you are only charged for the space that you use in an Aurora cluster volume.

        *MySQL*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *MariaDB*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *PostgreSQL*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *Oracle*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 10 to 3072.

        *SQL Server*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2):
        - Enterprise and Standard editions: Must be an integer from 20 to 16384.
        - Web and Express editions: Must be an integer from 20 to 16384.
        - Provisioned IOPS storage (io1):
        - Enterprise and Standard editions: Must be an integer from 20 to 16384.
        - Web and Express editions: Must be an integer from 20 to 16384.
        - Magnetic storage (standard):
        - Enterprise and Standard editions: Must be an integer from 20 to 1024.
        - Web and Express editions: Must be an integer from 20 to 1024.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-allocatedstorage
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allocatedStorage"))

    @allocated_storage.setter
    def allocated_storage(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__59d3152c4d5e7cecbdcd76480fe92201176e0eb1c7199001a91e36c3d8409682)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocatedStorage", value)

    @builtins.property
    @jsii.member(jsii_name="allowMajorVersionUpgrade")
    def allow_major_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether major version upgrades are allowed.

        Changing this parameter doesn't result in an outage and the change is asynchronously applied as soon as possible.

        Constraints: Major version upgrades must be allowed when specifying a value for the ``EngineVersion`` parameter that is a different major version than the DB instance's current version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-allowmajorversionupgrade
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "allowMajorVersionUpgrade"))

    @allow_major_version_upgrade.setter
    def allow_major_version_upgrade(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__57716fbcac28344145554895630d9e7d3ed16ff833d836025eba0e8559dae931)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowMajorVersionUpgrade", value)

    @builtins.property
    @jsii.member(jsii_name="associatedRoles")
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.DBInstanceRoleProperty"]]]]:
        '''The AWS Identity and Access Management (IAM) roles associated with the DB instance.

        *Amazon Aurora*

        Not applicable. The associated roles are managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-associatedroles
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.DBInstanceRoleProperty"]]]], jsii.get(self, "associatedRoles"))

    @associated_roles.setter
    def associated_roles(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.DBInstanceRoleProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ce8e79d061f4d460ac08196091e00285779b468080063faf9f5f8f7466a94f4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "associatedRoles", value)

    @builtins.property
    @jsii.member(jsii_name="autoMinorVersionUpgrade")
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether minor engine upgrades are applied automatically to the DB instance during the maintenance window.

        By default, minor engine upgrades are applied automatically.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-autominorversionupgrade
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "autoMinorVersionUpgrade"))

    @auto_minor_version_upgrade.setter
    def auto_minor_version_upgrade(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e378bc6005eb441ad86673219aa0312b39d55a509b73b385123b215a0aeff4c6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoMinorVersionUpgrade", value)

    @builtins.property
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone (AZ) where the database will be created.

        For information on AWS Regions and Availability Zones, see `Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ .

        For Amazon Aurora, each Aurora DB cluster hosts copies of its storage in three separate Availability Zones. Specify one of these Availability Zones. Aurora automatically chooses an appropriate Availability Zone if you don't specify one.

        Default: A random, system-chosen Availability Zone in the endpoint's AWS Region .

        Constraints:

        - The ``AvailabilityZone`` parameter can't be specified if the DB instance is a Multi-AZ deployment.
        - The specified Availability Zone must be in the same AWS Region as the current endpoint.

        Example: ``us-east-1d``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-availabilityzone
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2e158ac3f914684014bad2a96a64f29bf9a9be0671712af053302bafb993e770)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "availabilityZone", value)

    @builtins.property
    @jsii.member(jsii_name="backupRetentionPeriod")
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days for which automated backups are retained.

        Setting this parameter to a positive number enables backups. Setting this parameter to 0 disables automated backups.

        *Amazon Aurora*

        Not applicable. The retention period for automated backups is managed by the DB cluster.

        Default: 1

        Constraints:

        - Must be a value from 0 to 35
        - Can't be set to 0 if the DB instance is a source to read replicas

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-backupretentionperiod
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "backupRetentionPeriod"))

    @backup_retention_period.setter
    def backup_retention_period(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fd3e9f688b875610637e10e82f5815bcffe370acaf10d2173ec8c9703912f28b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "backupRetentionPeriod", value)

    @builtins.property
    @jsii.member(jsii_name="caCertificateIdentifier")
    def ca_certificate_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the CA certificate for this DB instance.

        .. epigraph::

           Specifying or updating this property triggers a reboot.

        For more information about CA certificate identifiers for RDS DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide* .

        For more information about CA certificate identifiers for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-cacertificateidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "caCertificateIdentifier"))

    @ca_certificate_identifier.setter
    def ca_certificate_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__62ccd61fdb1bb7842d321fcd1476ed686baf586bf907671076cafd6ba9e2c776)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "caCertificateIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="certificateDetails")
    def certificate_details(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.CertificateDetailsProperty"]]:
        '''The details of the DB instance's server certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-certificatedetails
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.CertificateDetailsProperty"]], jsii.get(self, "certificateDetails"))

    @certificate_details.setter
    def certificate_details(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.CertificateDetailsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8fa6bdf9b58edb0597b51a45bb01b0942f96853a6387af364c3ca19f8ebd5009)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "certificateDetails", value)

    @builtins.property
    @jsii.member(jsii_name="certificateRotationRestart")
    def certificate_rotation_restart(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the DB instance is restarted when you rotate your SSL/TLS certificate.

        By default, the DB instance is restarted when you rotate your SSL/TLS certificate. The certificate is not updated until the DB instance is restarted.
        .. epigraph::

           Set this parameter only if you are *not* using SSL/TLS to connect to the DB instance.

        If you are using SSL/TLS to connect to the DB instance, follow the appropriate instructions for your DB engine to rotate your SSL/TLS certificate:

        - For more information about rotating your SSL/TLS certificate for RDS DB engines, see `Rotating Your SSL/TLS Certificate. <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide.*
        - For more information about rotating your SSL/TLS certificate for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-certificaterotationrestart
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "certificateRotationRestart"))

    @certificate_rotation_restart.setter
    def certificate_rotation_restart(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8300a7358b8f3721e212076b4d43c16dce7eaecbef2865d4c267632cacd1e540)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "certificateRotationRestart", value)

    @builtins.property
    @jsii.member(jsii_name="characterSetName")
    def character_set_name(self) -> typing.Optional[builtins.str]:
        '''For supported engines, indicates that the DB instance should be associated with the specified character set.

        *Amazon Aurora*

        Not applicable. The character set is managed by the DB cluster. For more information, see `AWS::RDS::DBCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-charactersetname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "characterSetName"))

    @character_set_name.setter
    def character_set_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7b1c408e443b2c694ab495abfa8baccc012dc2d461177a5fa00b9476bc4dc78a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "characterSetName", value)

    @builtins.property
    @jsii.member(jsii_name="copyTagsToSnapshot")
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Spcifies whether to copy tags from the DB instance to snapshots of the DB instance.

        By default, tags are not copied.

        This setting doesn't apply to Amazon Aurora DB instances. Copying tags to snapshots is managed by the DB cluster. Setting this value for an Aurora DB instance has no effect on the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-copytagstosnapshot
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "copyTagsToSnapshot"))

    @copy_tags_to_snapshot.setter
    def copy_tags_to_snapshot(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cefd61afb4df8c491b91bfe227e165b190c98273315fd7ee9c3fb459c756ed53)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "copyTagsToSnapshot", value)

    @builtins.property
    @jsii.member(jsii_name="customIamInstanceProfile")
    def custom_iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.

        This setting is required for RDS Custom.

        Constraints:

        - The profile must exist in your account.
        - The profile must have an IAM role that Amazon EC2 has permissions to assume.
        - The instance profile name and the associated IAM role name must start with the prefix ``AWSRDSCustom`` .

        For the list of permissions required for the IAM role, see `Configure IAM and your VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-setup-orcl.html#custom-setup-orcl.iam-vpc>`_ in the *Amazon RDS User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-customiaminstanceprofile
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "customIamInstanceProfile"))

    @custom_iam_instance_profile.setter
    def custom_iam_instance_profile(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__226e820a1b504a7c3743da9e7c7340401e28197ad7bd4b0cb3f5526c4eee15e0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "customIamInstanceProfile", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterIdentifier")
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the DB cluster that the instance will belong to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterIdentifier"))

    @db_cluster_identifier.setter
    def db_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__47e201d1e2bcfacba9b2fdb9245c97a0850bff8695fda8e45cb8e1df70fbde73)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterSnapshotIdentifier")
    def db_cluster_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier for the RDS for MySQL Multi-AZ DB cluster snapshot to restore from.

        For more information on Multi-AZ DB clusters, see `Multi-AZ DB cluster deployments <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html>`_ in the *Amazon RDS User Guide* .

        Constraints:

        - Must match the identifier of an existing Multi-AZ DB cluster snapshot.
        - Can't be specified when ``DBSnapshotIdentifier`` is specified.
        - Must be specified when ``DBSnapshotIdentifier`` isn't specified.
        - If you are restoring from a shared manual Multi-AZ DB cluster snapshot, the ``DBClusterSnapshotIdentifier`` must be the ARN of the shared snapshot.
        - Can't be the identifier of an Aurora DB cluster snapshot.
        - Can't be the identifier of an RDS for PostgreSQL Multi-AZ DB cluster snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbclustersnapshotidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterSnapshotIdentifier"))

    @db_cluster_snapshot_identifier.setter
    def db_cluster_snapshot_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6ac99ea519540338b63e73e1edb5ebcecb0684d4ae715eddd3fc36e8cce5268)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterSnapshotIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbInstanceClass")
    def db_instance_class(self) -> typing.Optional[builtins.str]:
        '''The compute and memory capacity of the DB instance, for example, ``db.m4.large`` . Not all DB instance classes are available in all AWS Regions, or for all database engines.

        For the full list of DB instance classes, and availability for your engine, see `DB Instance Class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide.* For more information about DB instance class pricing and AWS Region support for DB instance classes, see `Amazon RDS Pricing <https://docs.aws.amazon.com/rds/pricing/>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbinstanceclass
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbInstanceClass"))

    @db_instance_class.setter
    def db_instance_class(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e7a5d9ef2683da37e4ade6581c3ea4b7e1dd46b2492278a1c7d1aadacedc8f7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbInstanceClass", value)

    @builtins.property
    @jsii.member(jsii_name="dbInstanceIdentifier")
    def db_instance_identifier(self) -> typing.Optional[builtins.str]:
        '''A name for the DB instance.

        If you specify a name, AWS CloudFormation converts it to lowercase. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the DB instance. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .

        For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* .
        .. epigraph::

           If you specify a name, you can't perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbinstanceidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbInstanceIdentifier"))

    @db_instance_identifier.setter
    def db_instance_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d43e0be95e8cce4fb01016f4a0386ec7484b400a153c752c5dfc73e4b1b71899)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbInstanceIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbName")
    def db_name(self) -> typing.Optional[builtins.str]:
        '''The meaning of this parameter differs according to the database engine you use.

        .. epigraph::

           If you specify the ``[DBSnapshotIdentifier](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier)`` property, this property only applies to RDS for Oracle.

        *Amazon Aurora*

        Not applicable. The database name is managed by the DB cluster.

        *MySQL*

        The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance.

        Constraints:

        - Must contain 1 to 64 letters or numbers.
        - Can't be a word reserved by the specified database engine

        *MariaDB*

        The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance.

        Constraints:

        - Must contain 1 to 64 letters or numbers.
        - Can't be a word reserved by the specified database engine

        *PostgreSQL*

        The name of the database to create when the DB instance is created. If this parameter is not specified, the default ``postgres`` database is created in the DB instance.

        Constraints:

        - Must begin with a letter. Subsequent characters can be letters, underscores, or digits (0-9).
        - Must contain 1 to 63 characters.
        - Can't be a word reserved by the specified database engine

        *Oracle*

        The Oracle System ID (SID) of the created DB instance. If you specify ``null`` , the default value ``ORCL`` is used. You can't specify the string NULL, or any other reserved word, for ``DBName`` .

        Default: ``ORCL``

        Constraints:

        - Can't be longer than 8 characters

        *SQL Server*

        Not applicable. Must be null.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbName"))

    @db_name.setter
    def db_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d256e48a4652f318e0f5c5f55fdbb24a3bbb22c363551d47299bad26b3e97a4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbName", value)

    @builtins.property
    @jsii.member(jsii_name="dbParameterGroupName")
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of an existing DB parameter group or a reference to an `AWS::RDS::DBParameterGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html>`_ resource created in the template.

        To list all of the available DB parameter group names, use the following command:

        ``aws rds describe-db-parameter-groups --query "DBParameterGroups[].DBParameterGroupName" --output text``
        .. epigraph::

           If any of the data members of the referenced parameter group are changed during an update, the DB instance might need to be restarted, which causes some interruption. If the parameter group contains static parameters, whether they were changed or not, an update triggers a reboot.

        If you don't specify a value for ``DBParameterGroupName`` property, the default DB parameter group for the specified engine and engine version is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbParameterGroupName"))

    @db_parameter_group_name.setter
    def db_parameter_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3a82f6fc74079b4a8567e6ca4252240887a8500538096be62863c72b24cec5a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbParameterGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="dbSecurityGroups")
    def db_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the DB security groups to assign to the DB instance.

        The list can include both the name of existing DB security groups or references to AWS::RDS::DBSecurityGroup resources created in the template.

        If you set DBSecurityGroups, you must not set VPCSecurityGroups, and vice versa. Also, note that the DBSecurityGroups property exists only for backwards compatibility with older regions and is no longer recommended for providing security information to an RDS DB instance. Instead, use VPCSecurityGroups.
        .. epigraph::

           If you specify this property, AWS CloudFormation sends only the following properties (if specified) to Amazon RDS during create operations:

           - ``AllocatedStorage``
           - ``AutoMinorVersionUpgrade``
           - ``AvailabilityZone``
           - ``BackupRetentionPeriod``
           - ``CharacterSetName``
           - ``DBInstanceClass``
           - ``DBName``
           - ``DBParameterGroupName``
           - ``DBSecurityGroups``
           - ``DBSubnetGroupName``
           - ``Engine``
           - ``EngineVersion``
           - ``Iops``
           - ``LicenseModel``
           - ``MasterUsername``
           - ``MasterUserPassword``
           - ``MultiAZ``
           - ``OptionGroupName``
           - ``PreferredBackupWindow``
           - ``PreferredMaintenanceWindow``

           All other properties are ignored. Specify a virtual private cloud (VPC) security group if you want to submit other properties, such as ``StorageType`` , ``StorageEncrypted`` , or ``KmsKeyId`` . If you're already using the ``DBSecurityGroups`` property, you can't use these other properties by updating your DB instance to use a VPC security group. You must recreate the DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsecuritygroups
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "dbSecurityGroups"))

    @db_security_groups.setter
    def db_security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e071dd53356b085b74267fe26c1ebcd9289d04cd1ed93b2939cc4e361678c22)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSecurityGroups", value)

    @builtins.property
    @jsii.member(jsii_name="dbSnapshotIdentifier")
    def db_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The name or Amazon Resource Name (ARN) of the DB snapshot that's used to restore the DB instance.

        If you're restoring from a shared manual DB snapshot, you must specify the ARN of the snapshot.

        By specifying this property, you can create a DB instance from the specified DB snapshot. If the ``DBSnapshotIdentifier`` property is an empty string or the ``AWS::RDS::DBInstance`` declaration has no ``DBSnapshotIdentifier`` property, AWS CloudFormation creates a new database. If the property contains a value (other than an empty string), AWS CloudFormation creates a database from the specified snapshot. If a snapshot with the specified name doesn't exist, AWS CloudFormation can't create the database and it rolls back the stack.

        Some DB instance properties aren't valid when you restore from a snapshot, such as the ``MasterUsername`` and ``MasterUserPassword`` properties. For information about the properties that you can specify, see the ``RestoreDBInstanceFromDBSnapshot`` action in the *Amazon RDS API Reference* .

        After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted.

        If you specify the ``DBSnapshotIdentifier`` property to restore a DB instance (as opposed to specifying it for DB instance updates), then don't specify the following properties:

        - ``CharacterSetName``
        - ``DBClusterIdentifier``
        - ``DBName``
        - ``DeleteAutomatedBackups``
        - ``EnablePerformanceInsights``
        - ``KmsKeyId``
        - ``MasterUsername``
        - ``MasterUserPassword``
        - ``PerformanceInsightsKMSKeyId``
        - ``PerformanceInsightsRetentionPeriod``
        - ``PromotionTier``
        - ``SourceDBInstanceIdentifier``
        - ``SourceRegion``
        - ``StorageEncrypted`` (for an encrypted snapshot)
        - ``Timezone``

        *Amazon Aurora*

        Not applicable. Snapshot restore is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsnapshotidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSnapshotIdentifier"))

    @db_snapshot_identifier.setter
    def db_snapshot_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__320452dcf7b927bfc9aec68b39b5030136baa93bd57004318389808127706212)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSnapshotIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbSubnetGroupName")
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''A DB subnet group to associate with the DB instance.

        If you update this value, the new subnet group must be a subnet group in a new VPC.

        If there's no DB subnet group, then the DB instance isn't a VPC DB instance.

        For more information about using Amazon RDS in a VPC, see `Using Amazon RDS with Amazon Virtual Private Cloud (VPC) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. The DB subnet group is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsubnetgroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSubnetGroupName"))

    @db_subnet_group_name.setter
    def db_subnet_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ee9b5bb70dbab22ab4481a77c04dc86a6cae16b4c7d30ccb17a77b55dfce2817)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSubnetGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="deleteAutomatedBackups")
    def delete_automated_backups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to remove automated backups immediately after the DB instance is deleted.

        This parameter isn't case-sensitive. The default is to remove automated backups immediately after the DB instance is deleted.

        *Amazon Aurora*

        Not applicable. When you delete a DB cluster, all automated backups for that DB cluster are deleted and can't be recovered. Manual DB cluster snapshots of the DB cluster are not deleted.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-deleteautomatedbackups
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "deleteAutomatedBackups"))

    @delete_automated_backups.setter
    def delete_automated_backups(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c979622824ea89b2c82e75127adcccb34f01b788bee2b54f7640626087bf93fe)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deleteAutomatedBackups", value)

    @builtins.property
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether the DB instance has deletion protection enabled.

        The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. For more information, see `Deleting a DB Instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_DeleteInstance.html>`_ .

        *Amazon Aurora*

        Not applicable. You can enable or disable deletion protection for the DB cluster. For more information, see ``CreateDBCluster`` . DB instances in a DB cluster can be deleted even when deletion protection is enabled for the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-deletionprotection
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "deletionProtection"))

    @deletion_protection.setter
    def deletion_protection(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b43d542832e60e979f75dde2ef2c09d4c4186b15c1ab0b0111aa86d3d9f6241c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deletionProtection", value)

    @builtins.property
    @jsii.member(jsii_name="domain")
    def domain(self) -> typing.Optional[builtins.str]:
        '''The Active Directory directory ID to create the DB instance in.

        Currently, only Microsoft SQL Server, Oracle, and PostgreSQL DB instances can be created in an Active Directory Domain.

        For more information, see `Kerberos Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html>`_ in the *Amazon RDS User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-domain
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domain"))

    @domain.setter
    def domain(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__983187a72c004fd527e146e600a3ba6e403cc37a21153eb4891e215e54ee58df)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domain", value)

    @builtins.property
    @jsii.member(jsii_name="domainIamRoleName")
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        '''The name of the IAM role to use when making API calls to the Directory Service.

        This setting doesn't apply to the following DB instances:

        - Amazon Aurora (The domain is managed by the DB cluster.)
        - RDS Custom

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-domainiamrolename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainIamRoleName"))

    @domain_iam_role_name.setter
    def domain_iam_role_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7fd9be68a3f0ada770758f20ae678e93c881b405fc762613862a364717217184)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainIamRoleName", value)

    @builtins.property
    @jsii.member(jsii_name="enableCloudwatchLogsExports")
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of log types that need to be enabled for exporting to CloudWatch Logs.

        The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Relational Database Service User Guide* .

        *Amazon Aurora*

        Not applicable. CloudWatch Logs exports are managed by the DB cluster.

        *MariaDB*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Microsoft SQL Server*

        Valid values: ``agent`` , ``error``

        *MySQL*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Oracle*

        Valid values: ``alert`` , ``audit`` , ``listener`` , ``trace``

        *PostgreSQL*

        Valid values: ``postgresql`` , ``upgrade``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enablecloudwatchlogsexports
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "enableCloudwatchLogsExports"))

    @enable_cloudwatch_logs_exports.setter
    def enable_cloudwatch_logs_exports(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7c0c6818a7c294ab57309fd6f8297e3dcbd9fe8f30e3c917aa472918a93f5ab3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableCloudwatchLogsExports", value)

    @builtins.property
    @jsii.member(jsii_name="enableIamDatabaseAuthentication")
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        By default, mapping is disabled.

        This property is supported for RDS for MariaDB, RDS for MySQL, and RDS for PostgreSQL. For more information, see `IAM Database Authentication for MariaDB, MySQL, and PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon RDS User Guide.*

        *Amazon Aurora*

        Not applicable. Mapping AWS IAM accounts to database accounts is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enableiamdatabaseauthentication
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enableIamDatabaseAuthentication"))

    @enable_iam_database_authentication.setter
    def enable_iam_database_authentication(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__02b9754bcfc32fe9a220fb4db94bd394cdd93ece54fea3e49ae5a0855157e259)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableIamDatabaseAuthentication", value)

    @builtins.property
    @jsii.member(jsii_name="enablePerformanceInsights")
    def enable_performance_insights(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to enable Performance Insights for the DB instance.

        For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* .

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enableperformanceinsights
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enablePerformanceInsights"))

    @enable_performance_insights.setter
    def enable_performance_insights(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__55acaa45776cf6b5432f2a97177653a9807b3c5b475626ccb7e031f2c6fba8b0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enablePerformanceInsights", value)

    @builtins.property
    @jsii.member(jsii_name="endpoint")
    def endpoint(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.EndpointProperty"]]:
        '''The connection endpoint for the DB instance.

        .. epigraph::

           The endpoint might not be shown for instances with the status of ``creating`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-endpoint
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.EndpointProperty"]], jsii.get(self, "endpoint"))

    @endpoint.setter
    def endpoint(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.EndpointProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__33fac2f64e551e516fef566fbc6d4ca2de02e0f84b5ef369d43a164e7683d854)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endpoint", value)

    @builtins.property
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine that you want to use for this DB instance.

        .. epigraph::

           When you are creating a DB instance, the ``Engine`` property is required.

        Valid Values:

        - ``aurora-mysql`` (for Aurora MySQL DB instances)
        - ``aurora-postgresql`` (for Aurora PostgreSQL DB instances)
        - ``custom-oracle-ee`` (for RDS Custom for Oracle DB instances)
        - ``custom-oracle-ee-cdb`` (for RDS Custom for Oracle DB instances)
        - ``custom-sqlserver-ee`` (for RDS Custom for SQL Server DB instances)
        - ``custom-sqlserver-se`` (for RDS Custom for SQL Server DB instances)
        - ``custom-sqlserver-web`` (for RDS Custom for SQL Server DB instances)
        - ``mariadb``
        - ``mysql``
        - ``oracle-ee``
        - ``oracle-ee-cdb``
        - ``oracle-se2``
        - ``oracle-se2-cdb``
        - ``postgres``
        - ``sqlserver-ee``
        - ``sqlserver-se``
        - ``sqlserver-ex``
        - ``sqlserver-web``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-engine
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engine"))

    @engine.setter
    def engine(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c1390f99110e4c5a9ba1f2e9dd4d5ccd134a1dd6c4610d4f96e29fca66cf7a9b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engine", value)

    @builtins.property
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The version number of the database engine to use.

        For a list of valid engine versions, use the ``DescribeDBEngineVersions`` action.

        The following are the database engines and links to information about the major and minor versions that are available with Amazon RDS. Not every database engine is available for every AWS Region.

        *Amazon Aurora*

        Not applicable. The version number of the database engine to be used by the DB instance is managed by the DB cluster.

        *MariaDB*

        See `MariaDB on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MariaDB.html#MariaDB.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.*

        *Microsoft SQL Server*

        See `Microsoft SQL Server Versions on Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.VersionSupport>`_ in the *Amazon RDS User Guide.*

        *MySQL*

        See `MySQL on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.*

        *Oracle*

        See `Oracle Database Engine Release Notes <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.Oracle.PatchComposition.html>`_ in the *Amazon RDS User Guide.*

        *PostgreSQL*

        See `Supported PostgreSQL Database Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts.General.DBVersions>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-engineversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engineVersion"))

    @engine_version.setter
    def engine_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8701105551fd11936573565b33d18a229c8e49f33df0332b41d26d0e2844e469)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineVersion", value)

    @builtins.property
    @jsii.member(jsii_name="iops")
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The number of I/O operations per second (IOPS) that the database provisions.

        The value must be equal to or greater than 1000.

        If you specify this property, you must follow the range of allowed ratios of your requested IOPS rate to the amount of storage that you allocate (IOPS to allocated storage). For example, you can provision an Oracle database instance with 1000 IOPS and 200 GiB of storage (a ratio of 5:1), or specify 2000 IOPS with 200 GiB of storage (a ratio of 10:1). For more information, see `Amazon RDS Provisioned IOPS Storage to Improve Performance <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* .
        .. epigraph::

           If you specify ``io1`` for the ``StorageType`` property, then you must also specify the ``Iops`` property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-iops
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "iops"))

    @iops.setter
    def iops(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4dbde1a697cf13178cdc952ba20d96e49376a7d1a06d1a66526a976aed755024)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "iops", value)

    @builtins.property
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS KMS key that's used to encrypt the DB instance, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` .

        If you enable the StorageEncrypted property but don't specify this property, AWS CloudFormation uses the default KMS key. If you specify this property, you must set the StorageEncrypted property to true.

        If you specify the ``SourceDBInstanceIdentifier`` property, the value is inherited from the source DB instance if the read replica is created in the same region.

        If you create an encrypted read replica in a different AWS Region, then you must specify a KMS key for the destination AWS Region. KMS encryption keys are specific to the region that they're created in, and you can't use encryption keys from one region in another region.

        If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify ``DBSecurityGroups`` , AWS CloudFormation ignores this property. To specify both a security group and this property, you must use a VPC security group. For more information about Amazon RDS and VPC, see `Using Amazon RDS with Amazon VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. The KMS key identifier is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-kmskeyid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyId"))

    @kms_key_id.setter
    def kms_key_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__de6372899c9340294ca3177a2377c0d7daa930a95e1af32be42f420531b1b38e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyId", value)

    @builtins.property
    @jsii.member(jsii_name="licenseModel")
    def license_model(self) -> typing.Optional[builtins.str]:
        '''License model information for this DB instance.

        Valid values:

        - Aurora MySQL - ``general-public-license``
        - Aurora PostgreSQL - ``postgresql-license``
        - MariaDB - ``general-public-license``
        - Microsoft SQL Server - ``license-included``
        - MySQL - ``general-public-license``
        - Oracle - ``bring-your-own-license`` or ``license-included``
        - PostgreSQL - ``postgresql-license``

        .. epigraph::

           If you've specified ``DBSecurityGroups`` and then you update the license model, AWS CloudFormation replaces the underlying DB instance. This will incur some interruptions to database availability.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-licensemodel
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "licenseModel"))

    @license_model.setter
    def license_model(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c41787d61e55405af610e638cfb248119ac823a7f335c64acea8a348fe45c03)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "licenseModel", value)

    @builtins.property
    @jsii.member(jsii_name="manageMasterUserPassword")
    def manage_master_user_password(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to manage the master user password with AWS Secrets Manager.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*

        Constraints:

        - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-managemasteruserpassword
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "manageMasterUserPassword"))

    @manage_master_user_password.setter
    def manage_master_user_password(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8ecef20f6921c76abf800f6b708fec7f9adf740c5601480addc60f0e306f478d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "manageMasterUserPassword", value)

    @builtins.property
    @jsii.member(jsii_name="masterUsername")
    def master_username(self) -> typing.Optional[builtins.str]:
        '''The master user name for the DB instance.

        .. epigraph::

           If you specify the ``SourceDBInstanceIdentifier`` or ``DBSnapshotIdentifier`` property, don't specify this property. The value is inherited from the source DB instance or snapshot.

        *Amazon Aurora*

        Not applicable. The name for the master user is managed by the DB cluster.

        *MariaDB*

        Constraints:

        - Required for MariaDB.
        - Must be 1 to 16 letters or numbers.
        - Can't be a reserved word for the chosen database engine.

        *Microsoft SQL Server*

        Constraints:

        - Required for SQL Server.
        - Must be 1 to 128 letters or numbers.
        - The first character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *MySQL*

        Constraints:

        - Required for MySQL.
        - Must be 1 to 16 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *Oracle*

        Constraints:

        - Required for Oracle.
        - Must be 1 to 30 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *PostgreSQL*

        Constraints:

        - Required for PostgreSQL.
        - Must be 1 to 63 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masterusername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "masterUsername"))

    @master_username.setter
    def master_username(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e4f4f227c9a762f25ad0222448fd600edf7e310cc003e4a2fed9244a343499d9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUsername", value)

    @builtins.property
    @jsii.member(jsii_name="masterUserPassword")
    def master_user_password(self) -> typing.Optional[builtins.str]:
        '''The password for the master user. The password can include any printable ASCII character except "/", """, or "@".

        *Amazon Aurora*

        Not applicable. The password for the master user is managed by the DB cluster.

        *MariaDB*

        Constraints: Must contain from 8 to 41 characters.

        *Microsoft SQL Server*

        Constraints: Must contain from 8 to 128 characters.

        *MySQL*

        Constraints: Must contain from 8 to 41 characters.

        *Oracle*

        Constraints: Must contain from 8 to 30 characters.

        *PostgreSQL*

        Constraints: Must contain from 8 to 128 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masteruserpassword
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "masterUserPassword"))

    @master_user_password.setter
    def master_user_password(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a65967234a8bc97dd1c51d3614691b923323d4f925dea11d8efcd922f9c90774)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUserPassword", value)

    @builtins.property
    @jsii.member(jsii_name="masterUserSecret")
    def master_user_secret(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.MasterUserSecretProperty"]]:
        '''The secret managed by RDS in AWS Secrets Manager for the master user password.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masterusersecret
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.MasterUserSecretProperty"]], jsii.get(self, "masterUserSecret"))

    @master_user_secret.setter
    def master_user_secret(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.MasterUserSecretProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b1517e3589a0cd23b097cdb03130a65dc936f20953fc163bd77838fa521130dd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUserSecret", value)

    @builtins.property
    @jsii.member(jsii_name="maxAllocatedStorage")
    def max_allocated_storage(self) -> typing.Optional[jsii.Number]:
        '''The upper limit in gibibytes (GiB) to which Amazon RDS can automatically scale the storage of the DB instance.

        For more information about this setting, including limitations that apply to it, see `Managing capacity automatically with Amazon RDS storage autoscaling <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling>`_ in the *Amazon RDS User Guide* .

        This setting doesn't apply to the following DB instances:

        - Amazon Aurora (Storage is managed by the DB cluster.)
        - RDS Custom

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-maxallocatedstorage
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxAllocatedStorage"))

    @max_allocated_storage.setter
    def max_allocated_storage(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb250fb7de4268a630027fa9d484face4c61a0e4b77adb24be3cd520b206064e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maxAllocatedStorage", value)

    @builtins.property
    @jsii.member(jsii_name="monitoringInterval")
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        '''The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance.

        To disable collection of Enhanced Monitoring metrics, specify 0. The default is 0.

        If ``MonitoringRoleArn`` is specified, then you must set ``MonitoringInterval`` to a value other than 0.

        This setting doesn't apply to RDS Custom.

        Valid Values: ``0, 1, 5, 10, 15, 30, 60``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-monitoringinterval
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "monitoringInterval"))

    @monitoring_interval.setter
    def monitoring_interval(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b528d960661e1022702318dc03705b042bc858142a319cc10ec73e9b1e97c946)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "monitoringInterval", value)

    @builtins.property
    @jsii.member(jsii_name="monitoringRoleArn")
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs.

        For example, ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting Up and Enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* .

        If ``MonitoringInterval`` is set to a value other than ``0`` , then you must supply a ``MonitoringRoleArn`` value.

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-monitoringrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "monitoringRoleArn"))

    @monitoring_role_arn.setter
    def monitoring_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7bedeace820a42b41d81c9462da797c3dc857df410e818904dbf41f51f08b8ac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "monitoringRoleArn", value)

    @builtins.property
    @jsii.member(jsii_name="multiAz")
    def multi_az(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the database instance is a Multi-AZ DB instance deployment.

        You can't set the ``AvailabilityZone`` parameter if the ``MultiAZ`` parameter is set to true.

        For more information, see `Multi-AZ deployments for high availability <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.MultiAZ.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. Amazon Aurora storage is replicated across all of the Availability Zones and doesn't require the ``MultiAZ`` option to be set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-multiaz
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "multiAz"))

    @multi_az.setter
    def multi_az(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3327a234013f3d0335a19b3511368b3abb4012a785056954d57fd348f2fa5338)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "multiAz", value)

    @builtins.property
    @jsii.member(jsii_name="ncharCharacterSetName")
    def nchar_character_set_name(self) -> typing.Optional[builtins.str]:
        '''The name of the NCHAR character set for the Oracle DB instance.

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-ncharcharactersetname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ncharCharacterSetName"))

    @nchar_character_set_name.setter
    def nchar_character_set_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__892bb93f0162e32f045bd286f21569851a2ad0901f846b0f97f5f799b92c5c12)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ncharCharacterSetName", value)

    @builtins.property
    @jsii.member(jsii_name="networkType")
    def network_type(self) -> typing.Optional[builtins.str]:
        '''The network type of the DB instance.

        Valid values:

        - ``IPV4``
        - ``DUAL``

        The network type is determined by the ``DBSubnetGroup`` specified for the DB instance. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ).

        For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-networktype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkType"))

    @network_type.setter
    def network_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3e81a4acfb4dd0912b4cfd138d002111d620357759e7ea378643291526acd06)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "networkType", value)

    @builtins.property
    @jsii.member(jsii_name="optionGroupName")
    def option_group_name(self) -> typing.Optional[builtins.str]:
        '''Indicates that the DB instance should be associated with the specified option group.

        Permanent options, such as the TDE option for Oracle Advanced Security TDE, can't be removed from an option group. Also, that option group can't be removed from a DB instance once it is associated with a DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-optiongroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "optionGroupName"))

    @option_group_name.setter
    def option_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c2c9f5e382f496654dae761c27630aa8347daa36712b25ae0ab1d5510c8a482)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "optionGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsKmsKeyId")
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for encryption of Performance Insights data.

        The KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.

        If you do not specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account. Your AWS account has a different default KMS key for each AWS Region.

        For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-performanceinsightskmskeyid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "performanceInsightsKmsKeyId"))

    @performance_insights_kms_key_id.setter
    def performance_insights_kms_key_id(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0b2c4ed20f0c5bd0071d155c8fb9f35b9f449062dc47146657cb0b594689ce7c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsKmsKeyId", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsRetentionPeriod")
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days to retain Performance Insights data.

        This setting doesn't apply to RDS Custom DB instances.

        Valid Values:

        - ``7``
        - *month* * 31, where *month* is a number of months from 1-23. Examples: ``93`` (3 months * 31), ``341`` (11 months * 31), ``589`` (19 months * 31)
        - ``731``

        Default: ``7`` days

        If you specify a retention period that isn't valid, such as ``94`` , Amazon RDS returns an error.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-performanceinsightsretentionperiod
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "performanceInsightsRetentionPeriod"))

    @performance_insights_retention_period.setter
    def performance_insights_retention_period(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6fb914e791795f1a4e550f9cee3704def112b60601f4bf9edb1a56e6eaccd14a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsRetentionPeriod", value)

    @builtins.property
    @jsii.member(jsii_name="port")
    def port(self) -> typing.Optional[builtins.str]:
        '''The port number on which the database accepts connections.

        *Amazon Aurora*

        Not applicable. The port number is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-port
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "port"))

    @port.setter
    def port(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__50adadf77381c2728bec99c8b734bfb963b3e4a2c290d2556fdee15d6691af74)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "port", value)

    @builtins.property
    @jsii.member(jsii_name="preferredBackupWindow")
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        '''The daily time range during which automated backups are created if automated backups are enabled, using the ``BackupRetentionPeriod`` parameter.

        For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow>`_ in the *Amazon RDS User Guide.*

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi`` .
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        *Amazon Aurora*

        Not applicable. The daily time range for creating automated backups is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-preferredbackupwindow
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "preferredBackupWindow"))

    @preferred_backup_window.setter
    def preferred_backup_window(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa4b3fb0a080432fe19a9cc48251373285d2ef78d3f0997ec15a613e49eafc46)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preferredBackupWindow", value)

    @builtins.property
    @jsii.member(jsii_name="preferredMaintenanceWindow")
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        '''The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).

        Format: ``ddd:hh24:mi-ddd:hh24:mi``

        The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Instance Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow>`_ in the *Amazon RDS User Guide.*
        .. epigraph::

           This property applies when AWS CloudFormation initially creates the DB instance. If you use AWS CloudFormation to update the DB instance, those updates are applied immediately.

        Constraints: Minimum 30-minute window.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-preferredmaintenancewindow
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "preferredMaintenanceWindow"))

    @preferred_maintenance_window.setter
    def preferred_maintenance_window(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__17724379e33a6330192b417d9aed05a4de6e5dd002684e29f403d3dd93c845ad)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preferredMaintenanceWindow", value)

    @builtins.property
    @jsii.member(jsii_name="processorFeatures")
    def processor_features(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.ProcessorFeatureProperty"]]]]:
        '''The number of CPU cores and the number of threads per core for the DB instance class of the DB instance.

        This setting doesn't apply to Amazon Aurora or RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-processorfeatures
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.ProcessorFeatureProperty"]]]], jsii.get(self, "processorFeatures"))

    @processor_features.setter
    def processor_features(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBInstance.ProcessorFeatureProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a8243c3c3cfc99dee0c95cbbece0bf27fc12383f043ab0dfbe786d7d666807b3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "processorFeatures", value)

    @builtins.property
    @jsii.member(jsii_name="promotionTier")
    def promotion_tier(self) -> typing.Optional[jsii.Number]:
        '''The order of priority in which an Aurora Replica is promoted to the primary instance after a failure of the existing primary instance.

        For more information, see `Fault Tolerance for an Aurora DB Cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.AuroraHighAvailability.html#Aurora.Managing.FaultTolerance>`_ in the *Amazon Aurora User Guide* .

        This setting doesn't apply to RDS Custom DB instances.

        Default: ``1``

        Valid Values: ``0 - 15``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-promotiontier
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "promotionTier"))

    @promotion_tier.setter
    def promotion_tier(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__33e74d15d4d5a91207335c5c9d0dd5a8206f8a5538c08e40663b0d0c1dccd362)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "promotionTier", value)

    @builtins.property
    @jsii.member(jsii_name="publiclyAccessible")
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether the DB instance is an internet-facing instance.

        If you specify true, AWS CloudFormation creates an instance with a publicly resolvable DNS name, which resolves to a public IP address. If you specify false, AWS CloudFormation creates an internal instance with a DNS name that resolves to a private IP address.

        The default behavior value depends on your VPC setup and the database subnet group. For more information, see the ``PubliclyAccessible`` parameter in the `CreateDBInstance <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html>`_ in the *Amazon RDS API Reference* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-publiclyaccessible
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "publiclyAccessible"))

    @publicly_accessible.setter
    def publicly_accessible(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a8c13e6d86c8f02d54e695d3280055c4d496df3feac4a13a1b2068a580c019e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "publiclyAccessible", value)

    @builtins.property
    @jsii.member(jsii_name="replicaMode")
    def replica_mode(self) -> typing.Optional[builtins.str]:
        '''The open mode of an Oracle read replica.

        For more information, see `Working with Oracle Read Replicas for Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html>`_ in the *Amazon RDS User Guide* .

        This setting is only supported in RDS for Oracle.

        Default: ``open-read-only``

        Valid Values: ``open-read-only`` or ``mounted``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-replicamode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "replicaMode"))

    @replica_mode.setter
    def replica_mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3591a1aca4c267b7cff5154e9b62939375b9a03cceff31d5ee1c7b3a9abd5c5f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "replicaMode", value)

    @builtins.property
    @jsii.member(jsii_name="restoreTime")
    def restore_time(self) -> typing.Optional[builtins.str]:
        '''The date and time to restore from.

        Valid Values: Value must be a time in Universal Coordinated Time (UTC) format

        Constraints:

        - Must be before the latest restorable time for the DB instance
        - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled

        Example: ``2009-09-07T23:45:00Z``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-restoretime
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "restoreTime"))

    @restore_time.setter
    def restore_time(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__263fd1ac2359aa08c71bfffc7bfa9a50b76cd283ff791db92e88940527896ed0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restoreTime", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbClusterIdentifier")
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the Multi-AZ DB cluster that will act as the source for the read replica.

        Each DB cluster can have up to 15 read replicas.

        Constraints:

        - Must be the identifier of an existing Multi-AZ DB cluster.
        - Can't be specified if the ``SourceDBInstanceIdentifier`` parameter is also specified.
        - The specified DB cluster must have automatic backups enabled, that is, its backup retention period must be greater than 0.
        - The source DB cluster must be in the same AWS Region as the read replica. Cross-Region replication isn't supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbClusterIdentifier"))

    @source_db_cluster_identifier.setter
    def source_db_cluster_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c25e998ba1faa1810a989309027441507208bcd60b801d55051b0646ace6beb8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbInstanceAutomatedBackupsArn")
    def source_db_instance_automated_backups_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the replicated automated backups from which to restore, for example, ``arn:aws:rds:useast-1:123456789012:auto-backup:ab-L2IJCEXJP7XQ7HOJ4SIEXAMPLE`` .

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbinstanceautomatedbackupsarn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbInstanceAutomatedBackupsArn"))

    @source_db_instance_automated_backups_arn.setter
    def source_db_instance_automated_backups_arn(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__af18d610cff75c65be584ce8e423768f378e5c61ac363a8cbaf9d3b96beb20b8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbInstanceAutomatedBackupsArn", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbInstanceIdentifier")
    def source_db_instance_identifier(self) -> typing.Optional[builtins.str]:
        '''If you want to create a read replica DB instance, specify the ID of the source DB instance.

        Each DB instance can have a limited number of read replicas. For more information, see `Working with Read Replicas <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html>`_ in the *Amazon RDS User Guide* .

        For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* .

        The ``SourceDBInstanceIdentifier`` property determines whether a DB instance is a read replica. If you remove the ``SourceDBInstanceIdentifier`` property from your template and then update your stack, AWS CloudFormation promotes the Read Replica to a standalone DB instance.
        .. epigraph::

           - If you specify a source DB instance that uses VPC security groups, we recommend that you specify the ``VPCSecurityGroups`` property. If you don't specify the property, the read replica inherits the value of the ``VPCSecurityGroups`` property from the source DB when you create the replica. However, if you update the stack, AWS CloudFormation reverts the replica's ``VPCSecurityGroups`` property to the default value because it's not defined in the stack's template. This change might cause unexpected issues.
           - Read replicas don't support deletion policies. AWS CloudFormation ignores any deletion policy that's associated with a read replica.
           - If you specify ``SourceDBInstanceIdentifier`` , don't specify the ``DBSnapshotIdentifier`` property. You can't create a read replica from a snapshot.
           - Don't set the ``BackupRetentionPeriod`` , ``DBName`` , ``MasterUsername`` , ``MasterUserPassword`` , and ``PreferredBackupWindow`` properties. The database attributes are inherited from the source DB instance, and backups are disabled for read replicas.
           - If the source DB instance is in a different region than the read replica, specify the source region in ``SourceRegion`` , and specify an ARN for a valid DB instance in ``SourceDBInstanceIdentifier`` . For more information, see `Constructing a Amazon RDS Amazon Resource Name (ARN) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html#USER_Tagging.ARN>`_ in the *Amazon RDS User Guide* .
           - For DB instances in Amazon Aurora clusters, don't specify this property. Amazon RDS automatically assigns writer and reader DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbinstanceidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbInstanceIdentifier"))

    @source_db_instance_identifier.setter
    def source_db_instance_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1da20396eef426f835570b10e92c00cf53d2a42dd847d680daf9832f45ca5d76)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbInstanceIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbiResourceId")
    def source_dbi_resource_id(self) -> typing.Optional[builtins.str]:
        '''The resource ID of the source DB instance from which to restore.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbiresourceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbiResourceId"))

    @source_dbi_resource_id.setter
    def source_dbi_resource_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8d4ea3cac4a190f60e6b97aa5cf291a71db11554bc2d7074974c73bd6d712ce9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbiResourceId", value)

    @builtins.property
    @jsii.member(jsii_name="sourceRegion")
    def source_region(self) -> typing.Optional[builtins.str]:
        '''The ID of the region that contains the source DB instance for the read replica.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourceregion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceRegion"))

    @source_region.setter
    def source_region(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54e6abd38ac0878b58009488af0c37180d8361eafe2d4e20fd21251d299b85ce)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceRegion", value)

    @builtins.property
    @jsii.member(jsii_name="storageEncrypted")
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether the DB instance is encrypted. By default, it isn't encrypted.

        If you specify the ``KmsKeyId`` property, then you must enable encryption.

        If you specify the ``SourceDBInstanceIdentifier`` property, don't specify this property. The value is inherited from the source DB instance, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB instance is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB instance to be encrypted, then don't set this property or set it to ``false`` .

        *Amazon Aurora*

        Not applicable. The encryption for DB instances is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storageencrypted
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "storageEncrypted"))

    @storage_encrypted.setter
    def storage_encrypted(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e45dbb04a9f91d51cce34e9ad406fb8b0bf3ca45ebf5191d2c3fe3ef60c0f73a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageEncrypted", value)

    @builtins.property
    @jsii.member(jsii_name="storageThroughput")
    def storage_throughput(self) -> typing.Optional[jsii.Number]:
        '''Specifies the storage throughput value for the DB instance. This setting applies only to the ``gp3`` storage type.

        This setting doesn't apply to RDS Custom or Amazon Aurora.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storagethroughput
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "storageThroughput"))

    @storage_throughput.setter
    def storage_throughput(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6a2e75b4699a0265165309d411411ba379fe29e69969b094d7db7f269836aee7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageThroughput", value)

    @builtins.property
    @jsii.member(jsii_name="storageType")
    def storage_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the storage type to be associated with the DB instance.

        Valid values: ``gp2 | gp3 | io1 | standard``

        The ``standard`` value is also known as magnetic.

        If you specify ``io1`` or ``gp3`` , you must also include a value for the ``Iops`` parameter.

        Default: ``io1`` if the ``Iops`` parameter is specified, otherwise ``gp2``

        For more information, see `Amazon RDS DB Instance Storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. Aurora data is stored in the cluster volume, which is a single, virtual volume that uses solid state drives (SSDs).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storagetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "storageType"))

    @storage_type.setter
    def storage_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__877e0a83255a41fdac41927a9ffb31f450bea764beca34bd0e59e3090b315b5a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageType", value)

    @builtins.property
    @jsii.member(jsii_name="timezone")
    def timezone(self) -> typing.Optional[builtins.str]:
        '''The time zone of the DB instance.

        The time zone parameter is currently supported only by `Microsoft SQL Server <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-timezone
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "timezone"))

    @timezone.setter
    def timezone(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e8a3ddd2e63e665f0964d94694ca633ce56b8b4a4257c33430373cb04ea8633)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "timezone", value)

    @builtins.property
    @jsii.member(jsii_name="useDefaultProcessorFeatures")
    def use_default_processor_features(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the DB instance class of the DB instance uses its default processor features.

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-usedefaultprocessorfeatures
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "useDefaultProcessorFeatures"))

    @use_default_processor_features.setter
    def use_default_processor_features(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea18e3dd0128a4bd7998126d71646de66d7aaf15a75a3ef90a69134b69dac881)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "useDefaultProcessorFeatures", value)

    @builtins.property
    @jsii.member(jsii_name="useLatestRestorableTime")
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether the DB instance is restored from the latest backup time.

        By default, the DB instance isn't restored from the latest backup time.

        Constraints: Can't be specified if the ``RestoreTime`` parameter is provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-uselatestrestorabletime
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "useLatestRestorableTime"))

    @use_latest_restorable_time.setter
    def use_latest_restorable_time(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7795f768216594eb6974dbf42046d2cc7bad24f83c78e5c2389203a763ca5edf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "useLatestRestorableTime", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSecurityGroups")
    def vpc_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the VPC security group IDs to assign to the DB instance.

        The list can include both the physical IDs of existing VPC security groups and references to `AWS::EC2::SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        If you set ``VPCSecurityGroups`` , you must not set ```DBSecurityGroups`` <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ , and vice versa.
        .. epigraph::

           You can migrate a DB instance in your stack from an RDS DB security group to a VPC security group, but keep the following in mind:

           - You can't revert to using an RDS security group after you establish a VPC security group membership.
           - When you migrate your DB instance to VPC security groups, if your stack update rolls back because the DB instance update fails or because an update fails in another AWS CloudFormation resource, the rollback fails because it can't revert to an RDS security group.
           - To use the properties that are available when you use a VPC security group, you must recreate the DB instance. If you don't, AWS CloudFormation submits only the property values that are listed in the ```DBSecurityGroups`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ property.

        To avoid this situation, migrate your DB instance to using VPC security groups only when that is the only change in your stack template.

        *Amazon Aurora*

        Not applicable. The associated list of EC2 VPC security groups is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-vpcsecuritygroups
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "vpcSecurityGroups"))

    @vpc_security_groups.setter
    def vpc_security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__347bbde8fcaff5958e8257d0b86319fb49d7cad9dd1f919f9270a1de11ca907c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSecurityGroups", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBInstance.CertificateDetailsProperty",
        jsii_struct_bases=[],
        name_mapping={"ca_identifier": "caIdentifier", "valid_till": "validTill"},
    )
    class CertificateDetailsProperty:
        def __init__(
            self,
            *,
            ca_identifier: typing.Optional[builtins.str] = None,
            valid_till: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Returns the details of the DB instance’s server certificate.

            For more information, see `Using SSL/TLS to encrypt a connection to a DB instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL.html>`_ in the *Amazon RDS User Guide* and `Using SSL/TLS to encrypt a connection to a DB cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL.html>`_ in the *Amazon Aurora User Guide* .

            :param ca_identifier: The CA identifier of the CA certificate used for the DB instance's server certificate.
            :param valid_till: The expiration date of the DB instance’s server certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-certificatedetails.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                certificate_details_property = rds.CfnDBInstance.CertificateDetailsProperty(
                    ca_identifier="caIdentifier",
                    valid_till="validTill"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2aa54c462c4c6282e2c878f84c54d452ef39c981551f5dcf7cdbf90ebff1ccac)
                check_type(argname="argument ca_identifier", value=ca_identifier, expected_type=type_hints["ca_identifier"])
                check_type(argname="argument valid_till", value=valid_till, expected_type=type_hints["valid_till"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if ca_identifier is not None:
                self._values["ca_identifier"] = ca_identifier
            if valid_till is not None:
                self._values["valid_till"] = valid_till

        @builtins.property
        def ca_identifier(self) -> typing.Optional[builtins.str]:
            '''The CA identifier of the CA certificate used for the DB instance's server certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-certificatedetails.html#cfn-rds-dbinstance-certificatedetails-caidentifier
            '''
            result = self._values.get("ca_identifier")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def valid_till(self) -> typing.Optional[builtins.str]:
            '''The expiration date of the DB instance’s server certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-certificatedetails.html#cfn-rds-dbinstance-certificatedetails-validtill
            '''
            result = self._values.get("valid_till")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CertificateDetailsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBInstance.DBInstanceRoleProperty",
        jsii_struct_bases=[],
        name_mapping={"feature_name": "featureName", "role_arn": "roleArn"},
    )
    class DBInstanceRoleProperty:
        def __init__(
            self,
            *,
            feature_name: builtins.str,
            role_arn: builtins.str,
        ) -> None:
            '''Describes an AWS Identity and Access Management (IAM) role that is associated with a DB instance.

            :param feature_name: The name of the feature associated with the AWS Identity and Access Management (IAM) role. IAM roles that are associated with a DB instance grant permission for the DB instance to access other AWS services on your behalf. For the list of supported feature names, see the ``SupportedFeatureNames`` description in `DBEngineVersion <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html>`_ in the *Amazon RDS API Reference* .
            :param role_arn: The Amazon Resource Name (ARN) of the IAM role that is associated with the DB instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-dbinstancerole.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                d_bInstance_role_property = rds.CfnDBInstance.DBInstanceRoleProperty(
                    feature_name="featureName",
                    role_arn="roleArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f9f3eeec177271fb0e6eb23b5e89d9aa215910a445a7f378d200e7c8defa62c2)
                check_type(argname="argument feature_name", value=feature_name, expected_type=type_hints["feature_name"])
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "feature_name": feature_name,
                "role_arn": role_arn,
            }

        @builtins.property
        def feature_name(self) -> builtins.str:
            '''The name of the feature associated with the AWS Identity and Access Management (IAM) role.

            IAM roles that are associated with a DB instance grant permission for the DB instance to access other AWS services on your behalf. For the list of supported feature names, see the ``SupportedFeatureNames`` description in `DBEngineVersion <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html>`_ in the *Amazon RDS API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-dbinstancerole.html#cfn-rds-dbinstance-dbinstancerole-featurename
            '''
            result = self._values.get("feature_name")
            assert result is not None, "Required property 'feature_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def role_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the IAM role that is associated with the DB instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-dbinstancerole.html#cfn-rds-dbinstance-dbinstancerole-rolearn
            '''
            result = self._values.get("role_arn")
            assert result is not None, "Required property 'role_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DBInstanceRoleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBInstance.EndpointProperty",
        jsii_struct_bases=[],
        name_mapping={
            "address": "address",
            "hosted_zone_id": "hostedZoneId",
            "port": "port",
        },
    )
    class EndpointProperty:
        def __init__(
            self,
            *,
            address: typing.Optional[builtins.str] = None,
            hosted_zone_id: typing.Optional[builtins.str] = None,
            port: typing.Optional[builtins.str] = None,
        ) -> None:
            '''This data type represents the information you need to connect to an Amazon RDS DB instance.

            This data type is used as a response element in the following actions:

            - ``CreateDBInstance``
            - ``DescribeDBInstances``
            - ``DeleteDBInstance``

            For the data structure that represents Amazon Aurora DB cluster endpoints, see ``DBClusterEndpoint`` .

            :param address: Specifies the DNS address of the DB instance.
            :param hosted_zone_id: Specifies the ID that Amazon Route 53 assigns when you create a hosted zone.
            :param port: Specifies the port that the database engine is listening on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-endpoint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                endpoint_property = rds.CfnDBInstance.EndpointProperty(
                    address="address",
                    hosted_zone_id="hostedZoneId",
                    port="port"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0a8162d3b83ddccf54f9146dc897a65d9b54102ddb2bd5d0e5d32f10565b91ab)
                check_type(argname="argument address", value=address, expected_type=type_hints["address"])
                check_type(argname="argument hosted_zone_id", value=hosted_zone_id, expected_type=type_hints["hosted_zone_id"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if address is not None:
                self._values["address"] = address
            if hosted_zone_id is not None:
                self._values["hosted_zone_id"] = hosted_zone_id
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def address(self) -> typing.Optional[builtins.str]:
            '''Specifies the DNS address of the DB instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-endpoint.html#cfn-rds-dbinstance-endpoint-address
            '''
            result = self._values.get("address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def hosted_zone_id(self) -> typing.Optional[builtins.str]:
            '''Specifies the ID that Amazon Route 53 assigns when you create a hosted zone.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-endpoint.html#cfn-rds-dbinstance-endpoint-hostedzoneid
            '''
            result = self._values.get("hosted_zone_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[builtins.str]:
            '''Specifies the port that the database engine is listening on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-endpoint.html#cfn-rds-dbinstance-endpoint-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBInstance.MasterUserSecretProperty",
        jsii_struct_bases=[],
        name_mapping={"kms_key_id": "kmsKeyId", "secret_arn": "secretArn"},
    )
    class MasterUserSecretProperty:
        def __init__(
            self,
            *,
            kms_key_id: typing.Optional[builtins.str] = None,
            secret_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``MasterUserSecret`` return value specifies the secret managed by RDS in AWS Secrets Manager for the master user password.

            For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

            :param kms_key_id: The AWS KMS key identifier that is used to encrypt the secret.
            :param secret_arn: The Amazon Resource Name (ARN) of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-masterusersecret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                master_user_secret_property = rds.CfnDBInstance.MasterUserSecretProperty(
                    kms_key_id="kmsKeyId",
                    secret_arn="secretArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9f77f6c1dc708c27c932ca984fd4c7d457c959472f08829b2fbb538583765161)
                check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
                check_type(argname="argument secret_arn", value=secret_arn, expected_type=type_hints["secret_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if secret_arn is not None:
                self._values["secret_arn"] = secret_arn

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''The AWS KMS key identifier that is used to encrypt the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-masterusersecret.html#cfn-rds-dbinstance-masterusersecret-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def secret_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-masterusersecret.html#cfn-rds-dbinstance-masterusersecret-secretarn
            '''
            result = self._values.get("secret_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MasterUserSecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBInstance.ProcessorFeatureProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class ProcessorFeatureProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ProcessorFeature`` property type specifies the processor features of a DB instance class status.

            :param name: The name of the processor feature. Valid names are ``coreCount`` and ``threadsPerCore`` .
            :param value: The value of a processor feature name.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-processorfeature.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                processor_feature_property = rds.CfnDBInstance.ProcessorFeatureProperty(
                    name="name",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__af956ca567992aaf20ac35fd3f87f41dced1cfcec90d57a985ec068e0132596c)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the processor feature.

            Valid names are ``coreCount`` and ``threadsPerCore`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-processorfeature.html#cfn-rds-dbinstance-processorfeature-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value of a processor feature name.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-processorfeature.html#cfn-rds-dbinstance-processorfeature-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProcessorFeatureProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBInstanceProps",
    jsii_struct_bases=[],
    name_mapping={
        "allocated_storage": "allocatedStorage",
        "allow_major_version_upgrade": "allowMajorVersionUpgrade",
        "associated_roles": "associatedRoles",
        "auto_minor_version_upgrade": "autoMinorVersionUpgrade",
        "availability_zone": "availabilityZone",
        "backup_retention_period": "backupRetentionPeriod",
        "ca_certificate_identifier": "caCertificateIdentifier",
        "certificate_details": "certificateDetails",
        "certificate_rotation_restart": "certificateRotationRestart",
        "character_set_name": "characterSetName",
        "copy_tags_to_snapshot": "copyTagsToSnapshot",
        "custom_iam_instance_profile": "customIamInstanceProfile",
        "db_cluster_identifier": "dbClusterIdentifier",
        "db_cluster_snapshot_identifier": "dbClusterSnapshotIdentifier",
        "db_instance_class": "dbInstanceClass",
        "db_instance_identifier": "dbInstanceIdentifier",
        "db_name": "dbName",
        "db_parameter_group_name": "dbParameterGroupName",
        "db_security_groups": "dbSecurityGroups",
        "db_snapshot_identifier": "dbSnapshotIdentifier",
        "db_subnet_group_name": "dbSubnetGroupName",
        "delete_automated_backups": "deleteAutomatedBackups",
        "deletion_protection": "deletionProtection",
        "domain": "domain",
        "domain_iam_role_name": "domainIamRoleName",
        "enable_cloudwatch_logs_exports": "enableCloudwatchLogsExports",
        "enable_iam_database_authentication": "enableIamDatabaseAuthentication",
        "enable_performance_insights": "enablePerformanceInsights",
        "endpoint": "endpoint",
        "engine": "engine",
        "engine_version": "engineVersion",
        "iops": "iops",
        "kms_key_id": "kmsKeyId",
        "license_model": "licenseModel",
        "manage_master_user_password": "manageMasterUserPassword",
        "master_username": "masterUsername",
        "master_user_password": "masterUserPassword",
        "master_user_secret": "masterUserSecret",
        "max_allocated_storage": "maxAllocatedStorage",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role_arn": "monitoringRoleArn",
        "multi_az": "multiAz",
        "nchar_character_set_name": "ncharCharacterSetName",
        "network_type": "networkType",
        "option_group_name": "optionGroupName",
        "performance_insights_kms_key_id": "performanceInsightsKmsKeyId",
        "performance_insights_retention_period": "performanceInsightsRetentionPeriod",
        "port": "port",
        "preferred_backup_window": "preferredBackupWindow",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "processor_features": "processorFeatures",
        "promotion_tier": "promotionTier",
        "publicly_accessible": "publiclyAccessible",
        "replica_mode": "replicaMode",
        "restore_time": "restoreTime",
        "source_db_cluster_identifier": "sourceDbClusterIdentifier",
        "source_db_instance_automated_backups_arn": "sourceDbInstanceAutomatedBackupsArn",
        "source_db_instance_identifier": "sourceDbInstanceIdentifier",
        "source_dbi_resource_id": "sourceDbiResourceId",
        "source_region": "sourceRegion",
        "storage_encrypted": "storageEncrypted",
        "storage_throughput": "storageThroughput",
        "storage_type": "storageType",
        "tags": "tags",
        "timezone": "timezone",
        "use_default_processor_features": "useDefaultProcessorFeatures",
        "use_latest_restorable_time": "useLatestRestorableTime",
        "vpc_security_groups": "vpcSecurityGroups",
    },
)
class CfnDBInstanceProps:
    def __init__(
        self,
        *,
        allocated_storage: typing.Optional[builtins.str] = None,
        allow_major_version_upgrade: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        associated_roles: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBInstance.DBInstanceRoleProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        ca_certificate_identifier: typing.Optional[builtins.str] = None,
        certificate_details: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBInstance.CertificateDetailsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        certificate_rotation_restart: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        character_set_name: typing.Optional[builtins.str] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        custom_iam_instance_profile: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_instance_class: typing.Optional[builtins.str] = None,
        db_instance_identifier: typing.Optional[builtins.str] = None,
        db_name: typing.Optional[builtins.str] = None,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        db_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        db_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        delete_automated_backups: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.Sequence[builtins.str]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        enable_performance_insights: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        endpoint: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBInstance.EndpointProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        license_model: typing.Optional[builtins.str] = None,
        manage_master_user_password: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        master_user_secret: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBInstance.MasterUserSecretProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        max_allocated_storage: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        multi_az: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        nchar_character_set_name: typing.Optional[builtins.str] = None,
        network_type: typing.Optional[builtins.str] = None,
        option_group_name: typing.Optional[builtins.str] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[builtins.str] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        processor_features: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBInstance.ProcessorFeatureProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        promotion_tier: typing.Optional[jsii.Number] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        replica_mode: typing.Optional[builtins.str] = None,
        restore_time: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        source_db_instance_automated_backups_arn: typing.Optional[builtins.str] = None,
        source_db_instance_identifier: typing.Optional[builtins.str] = None,
        source_dbi_resource_id: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        storage_throughput: typing.Optional[jsii.Number] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        timezone: typing.Optional[builtins.str] = None,
        use_default_processor_features: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        vpc_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBInstance``.

        :param allocated_storage: The amount of storage in gibibytes (GiB) to be initially allocated for the database instance. .. epigraph:: If any value is set in the ``Iops`` parameter, ``AllocatedStorage`` must be at least 100 GiB, which corresponds to the minimum Iops value of 1,000. If you increase the ``Iops`` value (in 1,000 IOPS increments), then you must also increase the ``AllocatedStorage`` value (in 100-GiB increments). *Amazon Aurora* Not applicable. Aurora cluster volumes automatically grow as the amount of data in your database increases, though you are only charged for the space that you use in an Aurora cluster volume. *MySQL* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *MariaDB* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *PostgreSQL* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *Oracle* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 10 to 3072. *SQL Server* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): - Enterprise and Standard editions: Must be an integer from 20 to 16384. - Web and Express editions: Must be an integer from 20 to 16384. - Provisioned IOPS storage (io1): - Enterprise and Standard editions: Must be an integer from 20 to 16384. - Web and Express editions: Must be an integer from 20 to 16384. - Magnetic storage (standard): - Enterprise and Standard editions: Must be an integer from 20 to 1024. - Web and Express editions: Must be an integer from 20 to 1024.
        :param allow_major_version_upgrade: A value that indicates whether major version upgrades are allowed. Changing this parameter doesn't result in an outage and the change is asynchronously applied as soon as possible. Constraints: Major version upgrades must be allowed when specifying a value for the ``EngineVersion`` parameter that is a different major version than the DB instance's current version.
        :param associated_roles: The AWS Identity and Access Management (IAM) roles associated with the DB instance. *Amazon Aurora* Not applicable. The associated roles are managed by the DB cluster.
        :param auto_minor_version_upgrade: A value that indicates whether minor engine upgrades are applied automatically to the DB instance during the maintenance window. By default, minor engine upgrades are applied automatically.
        :param availability_zone: The Availability Zone (AZ) where the database will be created. For information on AWS Regions and Availability Zones, see `Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ . For Amazon Aurora, each Aurora DB cluster hosts copies of its storage in three separate Availability Zones. Specify one of these Availability Zones. Aurora automatically chooses an appropriate Availability Zone if you don't specify one. Default: A random, system-chosen Availability Zone in the endpoint's AWS Region . Constraints: - The ``AvailabilityZone`` parameter can't be specified if the DB instance is a Multi-AZ deployment. - The specified Availability Zone must be in the same AWS Region as the current endpoint. Example: ``us-east-1d``
        :param backup_retention_period: The number of days for which automated backups are retained. Setting this parameter to a positive number enables backups. Setting this parameter to 0 disables automated backups. *Amazon Aurora* Not applicable. The retention period for automated backups is managed by the DB cluster. Default: 1 Constraints: - Must be a value from 0 to 35 - Can't be set to 0 if the DB instance is a source to read replicas
        :param ca_certificate_identifier: The identifier of the CA certificate for this DB instance. .. epigraph:: Specifying or updating this property triggers a reboot. For more information about CA certificate identifiers for RDS DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide* . For more information about CA certificate identifiers for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .
        :param certificate_details: The details of the DB instance's server certificate.
        :param certificate_rotation_restart: Specifies whether the DB instance is restarted when you rotate your SSL/TLS certificate. By default, the DB instance is restarted when you rotate your SSL/TLS certificate. The certificate is not updated until the DB instance is restarted. .. epigraph:: Set this parameter only if you are *not* using SSL/TLS to connect to the DB instance. If you are using SSL/TLS to connect to the DB instance, follow the appropriate instructions for your DB engine to rotate your SSL/TLS certificate: - For more information about rotating your SSL/TLS certificate for RDS DB engines, see `Rotating Your SSL/TLS Certificate. <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide.* - For more information about rotating your SSL/TLS certificate for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* . This setting doesn't apply to RDS Custom DB instances.
        :param character_set_name: For supported engines, indicates that the DB instance should be associated with the specified character set. *Amazon Aurora* Not applicable. The character set is managed by the DB cluster. For more information, see `AWS::RDS::DBCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html>`_ .
        :param copy_tags_to_snapshot: Spcifies whether to copy tags from the DB instance to snapshots of the DB instance. By default, tags are not copied. This setting doesn't apply to Amazon Aurora DB instances. Copying tags to snapshots is managed by the DB cluster. Setting this value for an Aurora DB instance has no effect on the DB cluster setting.
        :param custom_iam_instance_profile: The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance. This setting is required for RDS Custom. Constraints: - The profile must exist in your account. - The profile must have an IAM role that Amazon EC2 has permissions to assume. - The instance profile name and the associated IAM role name must start with the prefix ``AWSRDSCustom`` . For the list of permissions required for the IAM role, see `Configure IAM and your VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-setup-orcl.html#custom-setup-orcl.iam-vpc>`_ in the *Amazon RDS User Guide* .
        :param db_cluster_identifier: The identifier of the DB cluster that the instance will belong to.
        :param db_cluster_snapshot_identifier: The identifier for the RDS for MySQL Multi-AZ DB cluster snapshot to restore from. For more information on Multi-AZ DB clusters, see `Multi-AZ DB cluster deployments <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html>`_ in the *Amazon RDS User Guide* . Constraints: - Must match the identifier of an existing Multi-AZ DB cluster snapshot. - Can't be specified when ``DBSnapshotIdentifier`` is specified. - Must be specified when ``DBSnapshotIdentifier`` isn't specified. - If you are restoring from a shared manual Multi-AZ DB cluster snapshot, the ``DBClusterSnapshotIdentifier`` must be the ARN of the shared snapshot. - Can't be the identifier of an Aurora DB cluster snapshot. - Can't be the identifier of an RDS for PostgreSQL Multi-AZ DB cluster snapshot.
        :param db_instance_class: The compute and memory capacity of the DB instance, for example, ``db.m4.large`` . Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes, and availability for your engine, see `DB Instance Class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide.* For more information about DB instance class pricing and AWS Region support for DB instance classes, see `Amazon RDS Pricing <https://docs.aws.amazon.com/rds/pricing/>`_ .
        :param db_instance_identifier: A name for the DB instance. If you specify a name, AWS CloudFormation converts it to lowercase. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the DB instance. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* . .. epigraph:: If you specify a name, you can't perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param db_name: The meaning of this parameter differs according to the database engine you use. .. epigraph:: If you specify the ``[DBSnapshotIdentifier](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier)`` property, this property only applies to RDS for Oracle. *Amazon Aurora* Not applicable. The database name is managed by the DB cluster. *MySQL* The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Constraints: - Must contain 1 to 64 letters or numbers. - Can't be a word reserved by the specified database engine *MariaDB* The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Constraints: - Must contain 1 to 64 letters or numbers. - Can't be a word reserved by the specified database engine *PostgreSQL* The name of the database to create when the DB instance is created. If this parameter is not specified, the default ``postgres`` database is created in the DB instance. Constraints: - Must begin with a letter. Subsequent characters can be letters, underscores, or digits (0-9). - Must contain 1 to 63 characters. - Can't be a word reserved by the specified database engine *Oracle* The Oracle System ID (SID) of the created DB instance. If you specify ``null`` , the default value ``ORCL`` is used. You can't specify the string NULL, or any other reserved word, for ``DBName`` . Default: ``ORCL`` Constraints: - Can't be longer than 8 characters *SQL Server* Not applicable. Must be null.
        :param db_parameter_group_name: The name of an existing DB parameter group or a reference to an `AWS::RDS::DBParameterGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html>`_ resource created in the template. To list all of the available DB parameter group names, use the following command: ``aws rds describe-db-parameter-groups --query "DBParameterGroups[].DBParameterGroupName" --output text`` .. epigraph:: If any of the data members of the referenced parameter group are changed during an update, the DB instance might need to be restarted, which causes some interruption. If the parameter group contains static parameters, whether they were changed or not, an update triggers a reboot. If you don't specify a value for ``DBParameterGroupName`` property, the default DB parameter group for the specified engine and engine version is used.
        :param db_security_groups: A list of the DB security groups to assign to the DB instance. The list can include both the name of existing DB security groups or references to AWS::RDS::DBSecurityGroup resources created in the template. If you set DBSecurityGroups, you must not set VPCSecurityGroups, and vice versa. Also, note that the DBSecurityGroups property exists only for backwards compatibility with older regions and is no longer recommended for providing security information to an RDS DB instance. Instead, use VPCSecurityGroups. .. epigraph:: If you specify this property, AWS CloudFormation sends only the following properties (if specified) to Amazon RDS during create operations: - ``AllocatedStorage`` - ``AutoMinorVersionUpgrade`` - ``AvailabilityZone`` - ``BackupRetentionPeriod`` - ``CharacterSetName`` - ``DBInstanceClass`` - ``DBName`` - ``DBParameterGroupName`` - ``DBSecurityGroups`` - ``DBSubnetGroupName`` - ``Engine`` - ``EngineVersion`` - ``Iops`` - ``LicenseModel`` - ``MasterUsername`` - ``MasterUserPassword`` - ``MultiAZ`` - ``OptionGroupName`` - ``PreferredBackupWindow`` - ``PreferredMaintenanceWindow`` All other properties are ignored. Specify a virtual private cloud (VPC) security group if you want to submit other properties, such as ``StorageType`` , ``StorageEncrypted`` , or ``KmsKeyId`` . If you're already using the ``DBSecurityGroups`` property, you can't use these other properties by updating your DB instance to use a VPC security group. You must recreate the DB instance.
        :param db_snapshot_identifier: The name or Amazon Resource Name (ARN) of the DB snapshot that's used to restore the DB instance. If you're restoring from a shared manual DB snapshot, you must specify the ARN of the snapshot. By specifying this property, you can create a DB instance from the specified DB snapshot. If the ``DBSnapshotIdentifier`` property is an empty string or the ``AWS::RDS::DBInstance`` declaration has no ``DBSnapshotIdentifier`` property, AWS CloudFormation creates a new database. If the property contains a value (other than an empty string), AWS CloudFormation creates a database from the specified snapshot. If a snapshot with the specified name doesn't exist, AWS CloudFormation can't create the database and it rolls back the stack. Some DB instance properties aren't valid when you restore from a snapshot, such as the ``MasterUsername`` and ``MasterUserPassword`` properties. For information about the properties that you can specify, see the ``RestoreDBInstanceFromDBSnapshot`` action in the *Amazon RDS API Reference* . After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted. If you specify the ``DBSnapshotIdentifier`` property to restore a DB instance (as opposed to specifying it for DB instance updates), then don't specify the following properties: - ``CharacterSetName`` - ``DBClusterIdentifier`` - ``DBName`` - ``DeleteAutomatedBackups`` - ``EnablePerformanceInsights`` - ``KmsKeyId`` - ``MasterUsername`` - ``MasterUserPassword`` - ``PerformanceInsightsKMSKeyId`` - ``PerformanceInsightsRetentionPeriod`` - ``PromotionTier`` - ``SourceDBInstanceIdentifier`` - ``SourceRegion`` - ``StorageEncrypted`` (for an encrypted snapshot) - ``Timezone`` *Amazon Aurora* Not applicable. Snapshot restore is managed by the DB cluster.
        :param db_subnet_group_name: A DB subnet group to associate with the DB instance. If you update this value, the new subnet group must be a subnet group in a new VPC. If there's no DB subnet group, then the DB instance isn't a VPC DB instance. For more information about using Amazon RDS in a VPC, see `Using Amazon RDS with Amazon Virtual Private Cloud (VPC) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. The DB subnet group is managed by the DB cluster. If specified, the setting must match the DB cluster setting.
        :param delete_automated_backups: A value that indicates whether to remove automated backups immediately after the DB instance is deleted. This parameter isn't case-sensitive. The default is to remove automated backups immediately after the DB instance is deleted. *Amazon Aurora* Not applicable. When you delete a DB cluster, all automated backups for that DB cluster are deleted and can't be recovered. Manual DB cluster snapshots of the DB cluster are not deleted.
        :param deletion_protection: A value that indicates whether the DB instance has deletion protection enabled. The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. For more information, see `Deleting a DB Instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_DeleteInstance.html>`_ . *Amazon Aurora* Not applicable. You can enable or disable deletion protection for the DB cluster. For more information, see ``CreateDBCluster`` . DB instances in a DB cluster can be deleted even when deletion protection is enabled for the DB cluster.
        :param domain: The Active Directory directory ID to create the DB instance in. Currently, only Microsoft SQL Server, Oracle, and PostgreSQL DB instances can be created in an Active Directory Domain. For more information, see `Kerberos Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html>`_ in the *Amazon RDS User Guide* .
        :param domain_iam_role_name: The name of the IAM role to use when making API calls to the Directory Service. This setting doesn't apply to the following DB instances: - Amazon Aurora (The domain is managed by the DB cluster.) - RDS Custom
        :param enable_cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Relational Database Service User Guide* . *Amazon Aurora* Not applicable. CloudWatch Logs exports are managed by the DB cluster. *MariaDB* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Microsoft SQL Server* Valid values: ``agent`` , ``error`` *MySQL* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Oracle* Valid values: ``alert`` , ``audit`` , ``listener`` , ``trace`` *PostgreSQL* Valid values: ``postgresql`` , ``upgrade``
        :param enable_iam_database_authentication: A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. By default, mapping is disabled. This property is supported for RDS for MariaDB, RDS for MySQL, and RDS for PostgreSQL. For more information, see `IAM Database Authentication for MariaDB, MySQL, and PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon RDS User Guide.* *Amazon Aurora* Not applicable. Mapping AWS IAM accounts to database accounts is managed by the DB cluster.
        :param enable_performance_insights: Specifies whether to enable Performance Insights for the DB instance. For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* . This setting doesn't apply to RDS Custom DB instances.
        :param endpoint: The connection endpoint for the DB instance. .. epigraph:: The endpoint might not be shown for instances with the status of ``creating`` .
        :param engine: The name of the database engine that you want to use for this DB instance. .. epigraph:: When you are creating a DB instance, the ``Engine`` property is required. Valid Values: - ``aurora-mysql`` (for Aurora MySQL DB instances) - ``aurora-postgresql`` (for Aurora PostgreSQL DB instances) - ``custom-oracle-ee`` (for RDS Custom for Oracle DB instances) - ``custom-oracle-ee-cdb`` (for RDS Custom for Oracle DB instances) - ``custom-sqlserver-ee`` (for RDS Custom for SQL Server DB instances) - ``custom-sqlserver-se`` (for RDS Custom for SQL Server DB instances) - ``custom-sqlserver-web`` (for RDS Custom for SQL Server DB instances) - ``mariadb`` - ``mysql`` - ``oracle-ee`` - ``oracle-ee-cdb`` - ``oracle-se2`` - ``oracle-se2-cdb`` - ``postgres`` - ``sqlserver-ee`` - ``sqlserver-se`` - ``sqlserver-ex`` - ``sqlserver-web``
        :param engine_version: The version number of the database engine to use. For a list of valid engine versions, use the ``DescribeDBEngineVersions`` action. The following are the database engines and links to information about the major and minor versions that are available with Amazon RDS. Not every database engine is available for every AWS Region. *Amazon Aurora* Not applicable. The version number of the database engine to be used by the DB instance is managed by the DB cluster. *MariaDB* See `MariaDB on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MariaDB.html#MariaDB.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.* *Microsoft SQL Server* See `Microsoft SQL Server Versions on Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.VersionSupport>`_ in the *Amazon RDS User Guide.* *MySQL* See `MySQL on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.* *Oracle* See `Oracle Database Engine Release Notes <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.Oracle.PatchComposition.html>`_ in the *Amazon RDS User Guide.* *PostgreSQL* See `Supported PostgreSQL Database Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts.General.DBVersions>`_ in the *Amazon RDS User Guide.*
        :param iops: The number of I/O operations per second (IOPS) that the database provisions. The value must be equal to or greater than 1000. If you specify this property, you must follow the range of allowed ratios of your requested IOPS rate to the amount of storage that you allocate (IOPS to allocated storage). For example, you can provision an Oracle database instance with 1000 IOPS and 200 GiB of storage (a ratio of 5:1), or specify 2000 IOPS with 200 GiB of storage (a ratio of 10:1). For more information, see `Amazon RDS Provisioned IOPS Storage to Improve Performance <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* . .. epigraph:: If you specify ``io1`` for the ``StorageType`` property, then you must also specify the ``Iops`` property.
        :param kms_key_id: The ARN of the AWS KMS key that's used to encrypt the DB instance, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` . If you enable the StorageEncrypted property but don't specify this property, AWS CloudFormation uses the default KMS key. If you specify this property, you must set the StorageEncrypted property to true. If you specify the ``SourceDBInstanceIdentifier`` property, the value is inherited from the source DB instance if the read replica is created in the same region. If you create an encrypted read replica in a different AWS Region, then you must specify a KMS key for the destination AWS Region. KMS encryption keys are specific to the region that they're created in, and you can't use encryption keys from one region in another region. If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used. If you specify ``DBSecurityGroups`` , AWS CloudFormation ignores this property. To specify both a security group and this property, you must use a VPC security group. For more information about Amazon RDS and VPC, see `Using Amazon RDS with Amazon VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. The KMS key identifier is managed by the DB cluster.
        :param license_model: License model information for this DB instance. Valid values: - Aurora MySQL - ``general-public-license`` - Aurora PostgreSQL - ``postgresql-license`` - MariaDB - ``general-public-license`` - Microsoft SQL Server - ``license-included`` - MySQL - ``general-public-license`` - Oracle - ``bring-your-own-license`` or ``license-included`` - PostgreSQL - ``postgresql-license`` .. epigraph:: If you've specified ``DBSecurityGroups`` and then you update the license model, AWS CloudFormation replaces the underlying DB instance. This will incur some interruptions to database availability.
        :param manage_master_user_password: Specifies whether to manage the master user password with AWS Secrets Manager. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.* Constraints: - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.
        :param master_username: The master user name for the DB instance. .. epigraph:: If you specify the ``SourceDBInstanceIdentifier`` or ``DBSnapshotIdentifier`` property, don't specify this property. The value is inherited from the source DB instance or snapshot. *Amazon Aurora* Not applicable. The name for the master user is managed by the DB cluster. *MariaDB* Constraints: - Required for MariaDB. - Must be 1 to 16 letters or numbers. - Can't be a reserved word for the chosen database engine. *Microsoft SQL Server* Constraints: - Required for SQL Server. - Must be 1 to 128 letters or numbers. - The first character must be a letter. - Can't be a reserved word for the chosen database engine. *MySQL* Constraints: - Required for MySQL. - Must be 1 to 16 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine. *Oracle* Constraints: - Required for Oracle. - Must be 1 to 30 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine. *PostgreSQL* Constraints: - Required for PostgreSQL. - Must be 1 to 63 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine.
        :param master_user_password: The password for the master user. The password can include any printable ASCII character except "/", """, or "@". *Amazon Aurora* Not applicable. The password for the master user is managed by the DB cluster. *MariaDB* Constraints: Must contain from 8 to 41 characters. *Microsoft SQL Server* Constraints: Must contain from 8 to 128 characters. *MySQL* Constraints: Must contain from 8 to 41 characters. *Oracle* Constraints: Must contain from 8 to 30 characters. *PostgreSQL* Constraints: Must contain from 8 to 128 characters.
        :param master_user_secret: The secret managed by RDS in AWS Secrets Manager for the master user password. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*
        :param max_allocated_storage: The upper limit in gibibytes (GiB) to which Amazon RDS can automatically scale the storage of the DB instance. For more information about this setting, including limitations that apply to it, see `Managing capacity automatically with Amazon RDS storage autoscaling <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling>`_ in the *Amazon RDS User Guide* . This setting doesn't apply to the following DB instances: - Amazon Aurora (Storage is managed by the DB cluster.) - RDS Custom
        :param monitoring_interval: The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance. To disable collection of Enhanced Monitoring metrics, specify 0. The default is 0. If ``MonitoringRoleArn`` is specified, then you must set ``MonitoringInterval`` to a value other than 0. This setting doesn't apply to RDS Custom. Valid Values: ``0, 1, 5, 10, 15, 30, 60``
        :param monitoring_role_arn: The ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs. For example, ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting Up and Enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* . If ``MonitoringInterval`` is set to a value other than ``0`` , then you must supply a ``MonitoringRoleArn`` value. This setting doesn't apply to RDS Custom DB instances.
        :param multi_az: Specifies whether the database instance is a Multi-AZ DB instance deployment. You can't set the ``AvailabilityZone`` parameter if the ``MultiAZ`` parameter is set to true. For more information, see `Multi-AZ deployments for high availability <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.MultiAZ.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. Amazon Aurora storage is replicated across all of the Availability Zones and doesn't require the ``MultiAZ`` option to be set.
        :param nchar_character_set_name: The name of the NCHAR character set for the Oracle DB instance. This setting doesn't apply to RDS Custom DB instances.
        :param network_type: The network type of the DB instance. Valid values: - ``IPV4`` - ``DUAL`` The network type is determined by the ``DBSubnetGroup`` specified for the DB instance. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ). For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon RDS User Guide.*
        :param option_group_name: Indicates that the DB instance should be associated with the specified option group. Permanent options, such as the TDE option for Oracle Advanced Security TDE, can't be removed from an option group. Also, that option group can't be removed from a DB instance once it is associated with a DB instance.
        :param performance_insights_kms_key_id: The AWS KMS key identifier for encryption of Performance Insights data. The KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. If you do not specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account. Your AWS account has a different default KMS key for each AWS Region. For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .
        :param performance_insights_retention_period: The number of days to retain Performance Insights data. This setting doesn't apply to RDS Custom DB instances. Valid Values: - ``7`` - *month* * 31, where *month* is a number of months from 1-23. Examples: ``93`` (3 months * 31), ``341`` (11 months * 31), ``589`` (19 months * 31) - ``731`` Default: ``7`` days If you specify a retention period that isn't valid, such as ``94`` , Amazon RDS returns an error.
        :param port: The port number on which the database accepts connections. *Amazon Aurora* Not applicable. The port number is managed by the DB cluster.
        :param preferred_backup_window: The daily time range during which automated backups are created if automated backups are enabled, using the ``BackupRetentionPeriod`` parameter. For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow>`_ in the *Amazon RDS User Guide.* Constraints: - Must be in the format ``hh24:mi-hh24:mi`` . - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. *Amazon Aurora* Not applicable. The daily time range for creating automated backups is managed by the DB cluster.
        :param preferred_maintenance_window: The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC). Format: ``ddd:hh24:mi-ddd:hh24:mi`` The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Instance Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow>`_ in the *Amazon RDS User Guide.* .. epigraph:: This property applies when AWS CloudFormation initially creates the DB instance. If you use AWS CloudFormation to update the DB instance, those updates are applied immediately. Constraints: Minimum 30-minute window.
        :param processor_features: The number of CPU cores and the number of threads per core for the DB instance class of the DB instance. This setting doesn't apply to Amazon Aurora or RDS Custom DB instances.
        :param promotion_tier: The order of priority in which an Aurora Replica is promoted to the primary instance after a failure of the existing primary instance. For more information, see `Fault Tolerance for an Aurora DB Cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.AuroraHighAvailability.html#Aurora.Managing.FaultTolerance>`_ in the *Amazon Aurora User Guide* . This setting doesn't apply to RDS Custom DB instances. Default: ``1`` Valid Values: ``0 - 15``
        :param publicly_accessible: Indicates whether the DB instance is an internet-facing instance. If you specify true, AWS CloudFormation creates an instance with a publicly resolvable DNS name, which resolves to a public IP address. If you specify false, AWS CloudFormation creates an internal instance with a DNS name that resolves to a private IP address. The default behavior value depends on your VPC setup and the database subnet group. For more information, see the ``PubliclyAccessible`` parameter in the `CreateDBInstance <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html>`_ in the *Amazon RDS API Reference* .
        :param replica_mode: The open mode of an Oracle read replica. For more information, see `Working with Oracle Read Replicas for Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html>`_ in the *Amazon RDS User Guide* . This setting is only supported in RDS for Oracle. Default: ``open-read-only`` Valid Values: ``open-read-only`` or ``mounted``
        :param restore_time: The date and time to restore from. Valid Values: Value must be a time in Universal Coordinated Time (UTC) format Constraints: - Must be before the latest restorable time for the DB instance - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled Example: ``2009-09-07T23:45:00Z``
        :param source_db_cluster_identifier: The identifier of the Multi-AZ DB cluster that will act as the source for the read replica. Each DB cluster can have up to 15 read replicas. Constraints: - Must be the identifier of an existing Multi-AZ DB cluster. - Can't be specified if the ``SourceDBInstanceIdentifier`` parameter is also specified. - The specified DB cluster must have automatic backups enabled, that is, its backup retention period must be greater than 0. - The source DB cluster must be in the same AWS Region as the read replica. Cross-Region replication isn't supported.
        :param source_db_instance_automated_backups_arn: The Amazon Resource Name (ARN) of the replicated automated backups from which to restore, for example, ``arn:aws:rds:useast-1:123456789012:auto-backup:ab-L2IJCEXJP7XQ7HOJ4SIEXAMPLE`` . This setting doesn't apply to RDS Custom.
        :param source_db_instance_identifier: If you want to create a read replica DB instance, specify the ID of the source DB instance. Each DB instance can have a limited number of read replicas. For more information, see `Working with Read Replicas <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html>`_ in the *Amazon RDS User Guide* . For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* . The ``SourceDBInstanceIdentifier`` property determines whether a DB instance is a read replica. If you remove the ``SourceDBInstanceIdentifier`` property from your template and then update your stack, AWS CloudFormation promotes the Read Replica to a standalone DB instance. .. epigraph:: - If you specify a source DB instance that uses VPC security groups, we recommend that you specify the ``VPCSecurityGroups`` property. If you don't specify the property, the read replica inherits the value of the ``VPCSecurityGroups`` property from the source DB when you create the replica. However, if you update the stack, AWS CloudFormation reverts the replica's ``VPCSecurityGroups`` property to the default value because it's not defined in the stack's template. This change might cause unexpected issues. - Read replicas don't support deletion policies. AWS CloudFormation ignores any deletion policy that's associated with a read replica. - If you specify ``SourceDBInstanceIdentifier`` , don't specify the ``DBSnapshotIdentifier`` property. You can't create a read replica from a snapshot. - Don't set the ``BackupRetentionPeriod`` , ``DBName`` , ``MasterUsername`` , ``MasterUserPassword`` , and ``PreferredBackupWindow`` properties. The database attributes are inherited from the source DB instance, and backups are disabled for read replicas. - If the source DB instance is in a different region than the read replica, specify the source region in ``SourceRegion`` , and specify an ARN for a valid DB instance in ``SourceDBInstanceIdentifier`` . For more information, see `Constructing a Amazon RDS Amazon Resource Name (ARN) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html#USER_Tagging.ARN>`_ in the *Amazon RDS User Guide* . - For DB instances in Amazon Aurora clusters, don't specify this property. Amazon RDS automatically assigns writer and reader DB instances.
        :param source_dbi_resource_id: The resource ID of the source DB instance from which to restore.
        :param source_region: The ID of the region that contains the source DB instance for the read replica.
        :param storage_encrypted: A value that indicates whether the DB instance is encrypted. By default, it isn't encrypted. If you specify the ``KmsKeyId`` property, then you must enable encryption. If you specify the ``SourceDBInstanceIdentifier`` property, don't specify this property. The value is inherited from the source DB instance, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB instance is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB instance to be encrypted, then don't set this property or set it to ``false`` . *Amazon Aurora* Not applicable. The encryption for DB instances is managed by the DB cluster.
        :param storage_throughput: Specifies the storage throughput value for the DB instance. This setting applies only to the ``gp3`` storage type. This setting doesn't apply to RDS Custom or Amazon Aurora.
        :param storage_type: Specifies the storage type to be associated with the DB instance. Valid values: ``gp2 | gp3 | io1 | standard`` The ``standard`` value is also known as magnetic. If you specify ``io1`` or ``gp3`` , you must also include a value for the ``Iops`` parameter. Default: ``io1`` if the ``Iops`` parameter is specified, otherwise ``gp2`` For more information, see `Amazon RDS DB Instance Storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. Aurora data is stored in the cluster volume, which is a single, virtual volume that uses solid state drives (SSDs).
        :param tags: An optional array of key-value pairs to apply to this DB instance.
        :param timezone: The time zone of the DB instance. The time zone parameter is currently supported only by `Microsoft SQL Server <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone>`_ .
        :param use_default_processor_features: Specifies whether the DB instance class of the DB instance uses its default processor features. This setting doesn't apply to RDS Custom DB instances.
        :param use_latest_restorable_time: A value that indicates whether the DB instance is restored from the latest backup time. By default, the DB instance isn't restored from the latest backup time. Constraints: Can't be specified if the ``RestoreTime`` parameter is provided.
        :param vpc_security_groups: A list of the VPC security group IDs to assign to the DB instance. The list can include both the physical IDs of existing VPC security groups and references to `AWS::EC2::SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template. If you plan to update the resource, don't specify VPC security groups in a shared VPC. If you set ``VPCSecurityGroups`` , you must not set ```DBSecurityGroups`` <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ , and vice versa. .. epigraph:: You can migrate a DB instance in your stack from an RDS DB security group to a VPC security group, but keep the following in mind: - You can't revert to using an RDS security group after you establish a VPC security group membership. - When you migrate your DB instance to VPC security groups, if your stack update rolls back because the DB instance update fails or because an update fails in another AWS CloudFormation resource, the rollback fails because it can't revert to an RDS security group. - To use the properties that are available when you use a VPC security group, you must recreate the DB instance. If you don't, AWS CloudFormation submits only the property values that are listed in the ```DBSecurityGroups`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ property. To avoid this situation, migrate your DB instance to using VPC security groups only when that is the only change in your stack template. *Amazon Aurora* Not applicable. The associated list of EC2 VPC security groups is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_dBInstance_props = rds.CfnDBInstanceProps(
                allocated_storage="allocatedStorage",
                allow_major_version_upgrade=False,
                associated_roles=[rds.CfnDBInstance.DBInstanceRoleProperty(
                    feature_name="featureName",
                    role_arn="roleArn"
                )],
                auto_minor_version_upgrade=False,
                availability_zone="availabilityZone",
                backup_retention_period=123,
                ca_certificate_identifier="caCertificateIdentifier",
                certificate_details=rds.CfnDBInstance.CertificateDetailsProperty(
                    ca_identifier="caIdentifier",
                    valid_till="validTill"
                ),
                certificate_rotation_restart=False,
                character_set_name="characterSetName",
                copy_tags_to_snapshot=False,
                custom_iam_instance_profile="customIamInstanceProfile",
                db_cluster_identifier="dbClusterIdentifier",
                db_cluster_snapshot_identifier="dbClusterSnapshotIdentifier",
                db_instance_class="dbInstanceClass",
                db_instance_identifier="dbInstanceIdentifier",
                db_name="dbName",
                db_parameter_group_name="dbParameterGroupName",
                db_security_groups=["dbSecurityGroups"],
                db_snapshot_identifier="dbSnapshotIdentifier",
                db_subnet_group_name="dbSubnetGroupName",
                delete_automated_backups=False,
                deletion_protection=False,
                domain="domain",
                domain_iam_role_name="domainIamRoleName",
                enable_cloudwatch_logs_exports=["enableCloudwatchLogsExports"],
                enable_iam_database_authentication=False,
                enable_performance_insights=False,
                endpoint=rds.CfnDBInstance.EndpointProperty(
                    address="address",
                    hosted_zone_id="hostedZoneId",
                    port="port"
                ),
                engine="engine",
                engine_version="engineVersion",
                iops=123,
                kms_key_id="kmsKeyId",
                license_model="licenseModel",
                manage_master_user_password=False,
                master_username="masterUsername",
                master_user_password="masterUserPassword",
                master_user_secret=rds.CfnDBInstance.MasterUserSecretProperty(
                    kms_key_id="kmsKeyId",
                    secret_arn="secretArn"
                ),
                max_allocated_storage=123,
                monitoring_interval=123,
                monitoring_role_arn="monitoringRoleArn",
                multi_az=False,
                nchar_character_set_name="ncharCharacterSetName",
                network_type="networkType",
                option_group_name="optionGroupName",
                performance_insights_kms_key_id="performanceInsightsKmsKeyId",
                performance_insights_retention_period=123,
                port="port",
                preferred_backup_window="preferredBackupWindow",
                preferred_maintenance_window="preferredMaintenanceWindow",
                processor_features=[rds.CfnDBInstance.ProcessorFeatureProperty(
                    name="name",
                    value="value"
                )],
                promotion_tier=123,
                publicly_accessible=False,
                replica_mode="replicaMode",
                restore_time="restoreTime",
                source_db_cluster_identifier="sourceDbClusterIdentifier",
                source_db_instance_automated_backups_arn="sourceDbInstanceAutomatedBackupsArn",
                source_db_instance_identifier="sourceDbInstanceIdentifier",
                source_dbi_resource_id="sourceDbiResourceId",
                source_region="sourceRegion",
                storage_encrypted=False,
                storage_throughput=123,
                storage_type="storageType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                timezone="timezone",
                use_default_processor_features=False,
                use_latest_restorable_time=False,
                vpc_security_groups=["vpcSecurityGroups"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3bddb1be0bd1f1699e3a084c5859d94d8879ff15011f2f2eaac29ec16f6eaebc)
            check_type(argname="argument allocated_storage", value=allocated_storage, expected_type=type_hints["allocated_storage"])
            check_type(argname="argument allow_major_version_upgrade", value=allow_major_version_upgrade, expected_type=type_hints["allow_major_version_upgrade"])
            check_type(argname="argument associated_roles", value=associated_roles, expected_type=type_hints["associated_roles"])
            check_type(argname="argument auto_minor_version_upgrade", value=auto_minor_version_upgrade, expected_type=type_hints["auto_minor_version_upgrade"])
            check_type(argname="argument availability_zone", value=availability_zone, expected_type=type_hints["availability_zone"])
            check_type(argname="argument backup_retention_period", value=backup_retention_period, expected_type=type_hints["backup_retention_period"])
            check_type(argname="argument ca_certificate_identifier", value=ca_certificate_identifier, expected_type=type_hints["ca_certificate_identifier"])
            check_type(argname="argument certificate_details", value=certificate_details, expected_type=type_hints["certificate_details"])
            check_type(argname="argument certificate_rotation_restart", value=certificate_rotation_restart, expected_type=type_hints["certificate_rotation_restart"])
            check_type(argname="argument character_set_name", value=character_set_name, expected_type=type_hints["character_set_name"])
            check_type(argname="argument copy_tags_to_snapshot", value=copy_tags_to_snapshot, expected_type=type_hints["copy_tags_to_snapshot"])
            check_type(argname="argument custom_iam_instance_profile", value=custom_iam_instance_profile, expected_type=type_hints["custom_iam_instance_profile"])
            check_type(argname="argument db_cluster_identifier", value=db_cluster_identifier, expected_type=type_hints["db_cluster_identifier"])
            check_type(argname="argument db_cluster_snapshot_identifier", value=db_cluster_snapshot_identifier, expected_type=type_hints["db_cluster_snapshot_identifier"])
            check_type(argname="argument db_instance_class", value=db_instance_class, expected_type=type_hints["db_instance_class"])
            check_type(argname="argument db_instance_identifier", value=db_instance_identifier, expected_type=type_hints["db_instance_identifier"])
            check_type(argname="argument db_name", value=db_name, expected_type=type_hints["db_name"])
            check_type(argname="argument db_parameter_group_name", value=db_parameter_group_name, expected_type=type_hints["db_parameter_group_name"])
            check_type(argname="argument db_security_groups", value=db_security_groups, expected_type=type_hints["db_security_groups"])
            check_type(argname="argument db_snapshot_identifier", value=db_snapshot_identifier, expected_type=type_hints["db_snapshot_identifier"])
            check_type(argname="argument db_subnet_group_name", value=db_subnet_group_name, expected_type=type_hints["db_subnet_group_name"])
            check_type(argname="argument delete_automated_backups", value=delete_automated_backups, expected_type=type_hints["delete_automated_backups"])
            check_type(argname="argument deletion_protection", value=deletion_protection, expected_type=type_hints["deletion_protection"])
            check_type(argname="argument domain", value=domain, expected_type=type_hints["domain"])
            check_type(argname="argument domain_iam_role_name", value=domain_iam_role_name, expected_type=type_hints["domain_iam_role_name"])
            check_type(argname="argument enable_cloudwatch_logs_exports", value=enable_cloudwatch_logs_exports, expected_type=type_hints["enable_cloudwatch_logs_exports"])
            check_type(argname="argument enable_iam_database_authentication", value=enable_iam_database_authentication, expected_type=type_hints["enable_iam_database_authentication"])
            check_type(argname="argument enable_performance_insights", value=enable_performance_insights, expected_type=type_hints["enable_performance_insights"])
            check_type(argname="argument endpoint", value=endpoint, expected_type=type_hints["endpoint"])
            check_type(argname="argument engine", value=engine, expected_type=type_hints["engine"])
            check_type(argname="argument engine_version", value=engine_version, expected_type=type_hints["engine_version"])
            check_type(argname="argument iops", value=iops, expected_type=type_hints["iops"])
            check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
            check_type(argname="argument license_model", value=license_model, expected_type=type_hints["license_model"])
            check_type(argname="argument manage_master_user_password", value=manage_master_user_password, expected_type=type_hints["manage_master_user_password"])
            check_type(argname="argument master_username", value=master_username, expected_type=type_hints["master_username"])
            check_type(argname="argument master_user_password", value=master_user_password, expected_type=type_hints["master_user_password"])
            check_type(argname="argument master_user_secret", value=master_user_secret, expected_type=type_hints["master_user_secret"])
            check_type(argname="argument max_allocated_storage", value=max_allocated_storage, expected_type=type_hints["max_allocated_storage"])
            check_type(argname="argument monitoring_interval", value=monitoring_interval, expected_type=type_hints["monitoring_interval"])
            check_type(argname="argument monitoring_role_arn", value=monitoring_role_arn, expected_type=type_hints["monitoring_role_arn"])
            check_type(argname="argument multi_az", value=multi_az, expected_type=type_hints["multi_az"])
            check_type(argname="argument nchar_character_set_name", value=nchar_character_set_name, expected_type=type_hints["nchar_character_set_name"])
            check_type(argname="argument network_type", value=network_type, expected_type=type_hints["network_type"])
            check_type(argname="argument option_group_name", value=option_group_name, expected_type=type_hints["option_group_name"])
            check_type(argname="argument performance_insights_kms_key_id", value=performance_insights_kms_key_id, expected_type=type_hints["performance_insights_kms_key_id"])
            check_type(argname="argument performance_insights_retention_period", value=performance_insights_retention_period, expected_type=type_hints["performance_insights_retention_period"])
            check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            check_type(argname="argument preferred_backup_window", value=preferred_backup_window, expected_type=type_hints["preferred_backup_window"])
            check_type(argname="argument preferred_maintenance_window", value=preferred_maintenance_window, expected_type=type_hints["preferred_maintenance_window"])
            check_type(argname="argument processor_features", value=processor_features, expected_type=type_hints["processor_features"])
            check_type(argname="argument promotion_tier", value=promotion_tier, expected_type=type_hints["promotion_tier"])
            check_type(argname="argument publicly_accessible", value=publicly_accessible, expected_type=type_hints["publicly_accessible"])
            check_type(argname="argument replica_mode", value=replica_mode, expected_type=type_hints["replica_mode"])
            check_type(argname="argument restore_time", value=restore_time, expected_type=type_hints["restore_time"])
            check_type(argname="argument source_db_cluster_identifier", value=source_db_cluster_identifier, expected_type=type_hints["source_db_cluster_identifier"])
            check_type(argname="argument source_db_instance_automated_backups_arn", value=source_db_instance_automated_backups_arn, expected_type=type_hints["source_db_instance_automated_backups_arn"])
            check_type(argname="argument source_db_instance_identifier", value=source_db_instance_identifier, expected_type=type_hints["source_db_instance_identifier"])
            check_type(argname="argument source_dbi_resource_id", value=source_dbi_resource_id, expected_type=type_hints["source_dbi_resource_id"])
            check_type(argname="argument source_region", value=source_region, expected_type=type_hints["source_region"])
            check_type(argname="argument storage_encrypted", value=storage_encrypted, expected_type=type_hints["storage_encrypted"])
            check_type(argname="argument storage_throughput", value=storage_throughput, expected_type=type_hints["storage_throughput"])
            check_type(argname="argument storage_type", value=storage_type, expected_type=type_hints["storage_type"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument timezone", value=timezone, expected_type=type_hints["timezone"])
            check_type(argname="argument use_default_processor_features", value=use_default_processor_features, expected_type=type_hints["use_default_processor_features"])
            check_type(argname="argument use_latest_restorable_time", value=use_latest_restorable_time, expected_type=type_hints["use_latest_restorable_time"])
            check_type(argname="argument vpc_security_groups", value=vpc_security_groups, expected_type=type_hints["vpc_security_groups"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allocated_storage is not None:
            self._values["allocated_storage"] = allocated_storage
        if allow_major_version_upgrade is not None:
            self._values["allow_major_version_upgrade"] = allow_major_version_upgrade
        if associated_roles is not None:
            self._values["associated_roles"] = associated_roles
        if auto_minor_version_upgrade is not None:
            self._values["auto_minor_version_upgrade"] = auto_minor_version_upgrade
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if backup_retention_period is not None:
            self._values["backup_retention_period"] = backup_retention_period
        if ca_certificate_identifier is not None:
            self._values["ca_certificate_identifier"] = ca_certificate_identifier
        if certificate_details is not None:
            self._values["certificate_details"] = certificate_details
        if certificate_rotation_restart is not None:
            self._values["certificate_rotation_restart"] = certificate_rotation_restart
        if character_set_name is not None:
            self._values["character_set_name"] = character_set_name
        if copy_tags_to_snapshot is not None:
            self._values["copy_tags_to_snapshot"] = copy_tags_to_snapshot
        if custom_iam_instance_profile is not None:
            self._values["custom_iam_instance_profile"] = custom_iam_instance_profile
        if db_cluster_identifier is not None:
            self._values["db_cluster_identifier"] = db_cluster_identifier
        if db_cluster_snapshot_identifier is not None:
            self._values["db_cluster_snapshot_identifier"] = db_cluster_snapshot_identifier
        if db_instance_class is not None:
            self._values["db_instance_class"] = db_instance_class
        if db_instance_identifier is not None:
            self._values["db_instance_identifier"] = db_instance_identifier
        if db_name is not None:
            self._values["db_name"] = db_name
        if db_parameter_group_name is not None:
            self._values["db_parameter_group_name"] = db_parameter_group_name
        if db_security_groups is not None:
            self._values["db_security_groups"] = db_security_groups
        if db_snapshot_identifier is not None:
            self._values["db_snapshot_identifier"] = db_snapshot_identifier
        if db_subnet_group_name is not None:
            self._values["db_subnet_group_name"] = db_subnet_group_name
        if delete_automated_backups is not None:
            self._values["delete_automated_backups"] = delete_automated_backups
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if domain is not None:
            self._values["domain"] = domain
        if domain_iam_role_name is not None:
            self._values["domain_iam_role_name"] = domain_iam_role_name
        if enable_cloudwatch_logs_exports is not None:
            self._values["enable_cloudwatch_logs_exports"] = enable_cloudwatch_logs_exports
        if enable_iam_database_authentication is not None:
            self._values["enable_iam_database_authentication"] = enable_iam_database_authentication
        if enable_performance_insights is not None:
            self._values["enable_performance_insights"] = enable_performance_insights
        if endpoint is not None:
            self._values["endpoint"] = endpoint
        if engine is not None:
            self._values["engine"] = engine
        if engine_version is not None:
            self._values["engine_version"] = engine_version
        if iops is not None:
            self._values["iops"] = iops
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id
        if license_model is not None:
            self._values["license_model"] = license_model
        if manage_master_user_password is not None:
            self._values["manage_master_user_password"] = manage_master_user_password
        if master_username is not None:
            self._values["master_username"] = master_username
        if master_user_password is not None:
            self._values["master_user_password"] = master_user_password
        if master_user_secret is not None:
            self._values["master_user_secret"] = master_user_secret
        if max_allocated_storage is not None:
            self._values["max_allocated_storage"] = max_allocated_storage
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role_arn is not None:
            self._values["monitoring_role_arn"] = monitoring_role_arn
        if multi_az is not None:
            self._values["multi_az"] = multi_az
        if nchar_character_set_name is not None:
            self._values["nchar_character_set_name"] = nchar_character_set_name
        if network_type is not None:
            self._values["network_type"] = network_type
        if option_group_name is not None:
            self._values["option_group_name"] = option_group_name
        if performance_insights_kms_key_id is not None:
            self._values["performance_insights_kms_key_id"] = performance_insights_kms_key_id
        if performance_insights_retention_period is not None:
            self._values["performance_insights_retention_period"] = performance_insights_retention_period
        if port is not None:
            self._values["port"] = port
        if preferred_backup_window is not None:
            self._values["preferred_backup_window"] = preferred_backup_window
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if processor_features is not None:
            self._values["processor_features"] = processor_features
        if promotion_tier is not None:
            self._values["promotion_tier"] = promotion_tier
        if publicly_accessible is not None:
            self._values["publicly_accessible"] = publicly_accessible
        if replica_mode is not None:
            self._values["replica_mode"] = replica_mode
        if restore_time is not None:
            self._values["restore_time"] = restore_time
        if source_db_cluster_identifier is not None:
            self._values["source_db_cluster_identifier"] = source_db_cluster_identifier
        if source_db_instance_automated_backups_arn is not None:
            self._values["source_db_instance_automated_backups_arn"] = source_db_instance_automated_backups_arn
        if source_db_instance_identifier is not None:
            self._values["source_db_instance_identifier"] = source_db_instance_identifier
        if source_dbi_resource_id is not None:
            self._values["source_dbi_resource_id"] = source_dbi_resource_id
        if source_region is not None:
            self._values["source_region"] = source_region
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted
        if storage_throughput is not None:
            self._values["storage_throughput"] = storage_throughput
        if storage_type is not None:
            self._values["storage_type"] = storage_type
        if tags is not None:
            self._values["tags"] = tags
        if timezone is not None:
            self._values["timezone"] = timezone
        if use_default_processor_features is not None:
            self._values["use_default_processor_features"] = use_default_processor_features
        if use_latest_restorable_time is not None:
            self._values["use_latest_restorable_time"] = use_latest_restorable_time
        if vpc_security_groups is not None:
            self._values["vpc_security_groups"] = vpc_security_groups

    @builtins.property
    def allocated_storage(self) -> typing.Optional[builtins.str]:
        '''The amount of storage in gibibytes (GiB) to be initially allocated for the database instance.

        .. epigraph::

           If any value is set in the ``Iops`` parameter, ``AllocatedStorage`` must be at least 100 GiB, which corresponds to the minimum Iops value of 1,000. If you increase the ``Iops`` value (in 1,000 IOPS increments), then you must also increase the ``AllocatedStorage`` value (in 100-GiB increments).

        *Amazon Aurora*

        Not applicable. Aurora cluster volumes automatically grow as the amount of data in your database increases, though you are only charged for the space that you use in an Aurora cluster volume.

        *MySQL*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *MariaDB*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *PostgreSQL*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *Oracle*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 10 to 3072.

        *SQL Server*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2):
        - Enterprise and Standard editions: Must be an integer from 20 to 16384.
        - Web and Express editions: Must be an integer from 20 to 16384.
        - Provisioned IOPS storage (io1):
        - Enterprise and Standard editions: Must be an integer from 20 to 16384.
        - Web and Express editions: Must be an integer from 20 to 16384.
        - Magnetic storage (standard):
        - Enterprise and Standard editions: Must be an integer from 20 to 1024.
        - Web and Express editions: Must be an integer from 20 to 1024.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-allocatedstorage
        '''
        result = self._values.get("allocated_storage")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def allow_major_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether major version upgrades are allowed.

        Changing this parameter doesn't result in an outage and the change is asynchronously applied as soon as possible.

        Constraints: Major version upgrades must be allowed when specifying a value for the ``EngineVersion`` parameter that is a different major version than the DB instance's current version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-allowmajorversionupgrade
        '''
        result = self._values.get("allow_major_version_upgrade")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBInstance.DBInstanceRoleProperty]]]]:
        '''The AWS Identity and Access Management (IAM) roles associated with the DB instance.

        *Amazon Aurora*

        Not applicable. The associated roles are managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-associatedroles
        '''
        result = self._values.get("associated_roles")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBInstance.DBInstanceRoleProperty]]]], result)

    @builtins.property
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether minor engine upgrades are applied automatically to the DB instance during the maintenance window.

        By default, minor engine upgrades are applied automatically.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-autominorversionupgrade
        '''
        result = self._values.get("auto_minor_version_upgrade")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone (AZ) where the database will be created.

        For information on AWS Regions and Availability Zones, see `Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ .

        For Amazon Aurora, each Aurora DB cluster hosts copies of its storage in three separate Availability Zones. Specify one of these Availability Zones. Aurora automatically chooses an appropriate Availability Zone if you don't specify one.

        Default: A random, system-chosen Availability Zone in the endpoint's AWS Region .

        Constraints:

        - The ``AvailabilityZone`` parameter can't be specified if the DB instance is a Multi-AZ deployment.
        - The specified Availability Zone must be in the same AWS Region as the current endpoint.

        Example: ``us-east-1d``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-availabilityzone
        '''
        result = self._values.get("availability_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days for which automated backups are retained.

        Setting this parameter to a positive number enables backups. Setting this parameter to 0 disables automated backups.

        *Amazon Aurora*

        Not applicable. The retention period for automated backups is managed by the DB cluster.

        Default: 1

        Constraints:

        - Must be a value from 0 to 35
        - Can't be set to 0 if the DB instance is a source to read replicas

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-backupretentionperiod
        '''
        result = self._values.get("backup_retention_period")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ca_certificate_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the CA certificate for this DB instance.

        .. epigraph::

           Specifying or updating this property triggers a reboot.

        For more information about CA certificate identifiers for RDS DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide* .

        For more information about CA certificate identifiers for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-cacertificateidentifier
        '''
        result = self._values.get("ca_certificate_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def certificate_details(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBInstance.CertificateDetailsProperty]]:
        '''The details of the DB instance's server certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-certificatedetails
        '''
        result = self._values.get("certificate_details")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBInstance.CertificateDetailsProperty]], result)

    @builtins.property
    def certificate_rotation_restart(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the DB instance is restarted when you rotate your SSL/TLS certificate.

        By default, the DB instance is restarted when you rotate your SSL/TLS certificate. The certificate is not updated until the DB instance is restarted.
        .. epigraph::

           Set this parameter only if you are *not* using SSL/TLS to connect to the DB instance.

        If you are using SSL/TLS to connect to the DB instance, follow the appropriate instructions for your DB engine to rotate your SSL/TLS certificate:

        - For more information about rotating your SSL/TLS certificate for RDS DB engines, see `Rotating Your SSL/TLS Certificate. <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide.*
        - For more information about rotating your SSL/TLS certificate for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-certificaterotationrestart
        '''
        result = self._values.get("certificate_rotation_restart")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def character_set_name(self) -> typing.Optional[builtins.str]:
        '''For supported engines, indicates that the DB instance should be associated with the specified character set.

        *Amazon Aurora*

        Not applicable. The character set is managed by the DB cluster. For more information, see `AWS::RDS::DBCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-charactersetname
        '''
        result = self._values.get("character_set_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Spcifies whether to copy tags from the DB instance to snapshots of the DB instance.

        By default, tags are not copied.

        This setting doesn't apply to Amazon Aurora DB instances. Copying tags to snapshots is managed by the DB cluster. Setting this value for an Aurora DB instance has no effect on the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-copytagstosnapshot
        '''
        result = self._values.get("copy_tags_to_snapshot")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def custom_iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.

        This setting is required for RDS Custom.

        Constraints:

        - The profile must exist in your account.
        - The profile must have an IAM role that Amazon EC2 has permissions to assume.
        - The instance profile name and the associated IAM role name must start with the prefix ``AWSRDSCustom`` .

        For the list of permissions required for the IAM role, see `Configure IAM and your VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-setup-orcl.html#custom-setup-orcl.iam-vpc>`_ in the *Amazon RDS User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-customiaminstanceprofile
        '''
        result = self._values.get("custom_iam_instance_profile")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the DB cluster that the instance will belong to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbclusteridentifier
        '''
        result = self._values.get("db_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier for the RDS for MySQL Multi-AZ DB cluster snapshot to restore from.

        For more information on Multi-AZ DB clusters, see `Multi-AZ DB cluster deployments <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html>`_ in the *Amazon RDS User Guide* .

        Constraints:

        - Must match the identifier of an existing Multi-AZ DB cluster snapshot.
        - Can't be specified when ``DBSnapshotIdentifier`` is specified.
        - Must be specified when ``DBSnapshotIdentifier`` isn't specified.
        - If you are restoring from a shared manual Multi-AZ DB cluster snapshot, the ``DBClusterSnapshotIdentifier`` must be the ARN of the shared snapshot.
        - Can't be the identifier of an Aurora DB cluster snapshot.
        - Can't be the identifier of an RDS for PostgreSQL Multi-AZ DB cluster snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbclustersnapshotidentifier
        '''
        result = self._values.get("db_cluster_snapshot_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_instance_class(self) -> typing.Optional[builtins.str]:
        '''The compute and memory capacity of the DB instance, for example, ``db.m4.large`` . Not all DB instance classes are available in all AWS Regions, or for all database engines.

        For the full list of DB instance classes, and availability for your engine, see `DB Instance Class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide.* For more information about DB instance class pricing and AWS Region support for DB instance classes, see `Amazon RDS Pricing <https://docs.aws.amazon.com/rds/pricing/>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbinstanceclass
        '''
        result = self._values.get("db_instance_class")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_instance_identifier(self) -> typing.Optional[builtins.str]:
        '''A name for the DB instance.

        If you specify a name, AWS CloudFormation converts it to lowercase. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the DB instance. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .

        For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* .
        .. epigraph::

           If you specify a name, you can't perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbinstanceidentifier
        '''
        result = self._values.get("db_instance_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_name(self) -> typing.Optional[builtins.str]:
        '''The meaning of this parameter differs according to the database engine you use.

        .. epigraph::

           If you specify the ``[DBSnapshotIdentifier](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier)`` property, this property only applies to RDS for Oracle.

        *Amazon Aurora*

        Not applicable. The database name is managed by the DB cluster.

        *MySQL*

        The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance.

        Constraints:

        - Must contain 1 to 64 letters or numbers.
        - Can't be a word reserved by the specified database engine

        *MariaDB*

        The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance.

        Constraints:

        - Must contain 1 to 64 letters or numbers.
        - Can't be a word reserved by the specified database engine

        *PostgreSQL*

        The name of the database to create when the DB instance is created. If this parameter is not specified, the default ``postgres`` database is created in the DB instance.

        Constraints:

        - Must begin with a letter. Subsequent characters can be letters, underscores, or digits (0-9).
        - Must contain 1 to 63 characters.
        - Can't be a word reserved by the specified database engine

        *Oracle*

        The Oracle System ID (SID) of the created DB instance. If you specify ``null`` , the default value ``ORCL`` is used. You can't specify the string NULL, or any other reserved word, for ``DBName`` .

        Default: ``ORCL``

        Constraints:

        - Can't be longer than 8 characters

        *SQL Server*

        Not applicable. Must be null.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbname
        '''
        result = self._values.get("db_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of an existing DB parameter group or a reference to an `AWS::RDS::DBParameterGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html>`_ resource created in the template.

        To list all of the available DB parameter group names, use the following command:

        ``aws rds describe-db-parameter-groups --query "DBParameterGroups[].DBParameterGroupName" --output text``
        .. epigraph::

           If any of the data members of the referenced parameter group are changed during an update, the DB instance might need to be restarted, which causes some interruption. If the parameter group contains static parameters, whether they were changed or not, an update triggers a reboot.

        If you don't specify a value for ``DBParameterGroupName`` property, the default DB parameter group for the specified engine and engine version is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbparametergroupname
        '''
        result = self._values.get("db_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the DB security groups to assign to the DB instance.

        The list can include both the name of existing DB security groups or references to AWS::RDS::DBSecurityGroup resources created in the template.

        If you set DBSecurityGroups, you must not set VPCSecurityGroups, and vice versa. Also, note that the DBSecurityGroups property exists only for backwards compatibility with older regions and is no longer recommended for providing security information to an RDS DB instance. Instead, use VPCSecurityGroups.
        .. epigraph::

           If you specify this property, AWS CloudFormation sends only the following properties (if specified) to Amazon RDS during create operations:

           - ``AllocatedStorage``
           - ``AutoMinorVersionUpgrade``
           - ``AvailabilityZone``
           - ``BackupRetentionPeriod``
           - ``CharacterSetName``
           - ``DBInstanceClass``
           - ``DBName``
           - ``DBParameterGroupName``
           - ``DBSecurityGroups``
           - ``DBSubnetGroupName``
           - ``Engine``
           - ``EngineVersion``
           - ``Iops``
           - ``LicenseModel``
           - ``MasterUsername``
           - ``MasterUserPassword``
           - ``MultiAZ``
           - ``OptionGroupName``
           - ``PreferredBackupWindow``
           - ``PreferredMaintenanceWindow``

           All other properties are ignored. Specify a virtual private cloud (VPC) security group if you want to submit other properties, such as ``StorageType`` , ``StorageEncrypted`` , or ``KmsKeyId`` . If you're already using the ``DBSecurityGroups`` property, you can't use these other properties by updating your DB instance to use a VPC security group. You must recreate the DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsecuritygroups
        '''
        result = self._values.get("db_security_groups")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def db_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The name or Amazon Resource Name (ARN) of the DB snapshot that's used to restore the DB instance.

        If you're restoring from a shared manual DB snapshot, you must specify the ARN of the snapshot.

        By specifying this property, you can create a DB instance from the specified DB snapshot. If the ``DBSnapshotIdentifier`` property is an empty string or the ``AWS::RDS::DBInstance`` declaration has no ``DBSnapshotIdentifier`` property, AWS CloudFormation creates a new database. If the property contains a value (other than an empty string), AWS CloudFormation creates a database from the specified snapshot. If a snapshot with the specified name doesn't exist, AWS CloudFormation can't create the database and it rolls back the stack.

        Some DB instance properties aren't valid when you restore from a snapshot, such as the ``MasterUsername`` and ``MasterUserPassword`` properties. For information about the properties that you can specify, see the ``RestoreDBInstanceFromDBSnapshot`` action in the *Amazon RDS API Reference* .

        After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted.

        If you specify the ``DBSnapshotIdentifier`` property to restore a DB instance (as opposed to specifying it for DB instance updates), then don't specify the following properties:

        - ``CharacterSetName``
        - ``DBClusterIdentifier``
        - ``DBName``
        - ``DeleteAutomatedBackups``
        - ``EnablePerformanceInsights``
        - ``KmsKeyId``
        - ``MasterUsername``
        - ``MasterUserPassword``
        - ``PerformanceInsightsKMSKeyId``
        - ``PerformanceInsightsRetentionPeriod``
        - ``PromotionTier``
        - ``SourceDBInstanceIdentifier``
        - ``SourceRegion``
        - ``StorageEncrypted`` (for an encrypted snapshot)
        - ``Timezone``

        *Amazon Aurora*

        Not applicable. Snapshot restore is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsnapshotidentifier
        '''
        result = self._values.get("db_snapshot_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''A DB subnet group to associate with the DB instance.

        If you update this value, the new subnet group must be a subnet group in a new VPC.

        If there's no DB subnet group, then the DB instance isn't a VPC DB instance.

        For more information about using Amazon RDS in a VPC, see `Using Amazon RDS with Amazon Virtual Private Cloud (VPC) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. The DB subnet group is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsubnetgroupname
        '''
        result = self._values.get("db_subnet_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def delete_automated_backups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to remove automated backups immediately after the DB instance is deleted.

        This parameter isn't case-sensitive. The default is to remove automated backups immediately after the DB instance is deleted.

        *Amazon Aurora*

        Not applicable. When you delete a DB cluster, all automated backups for that DB cluster are deleted and can't be recovered. Manual DB cluster snapshots of the DB cluster are not deleted.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-deleteautomatedbackups
        '''
        result = self._values.get("delete_automated_backups")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether the DB instance has deletion protection enabled.

        The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. For more information, see `Deleting a DB Instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_DeleteInstance.html>`_ .

        *Amazon Aurora*

        Not applicable. You can enable or disable deletion protection for the DB cluster. For more information, see ``CreateDBCluster`` . DB instances in a DB cluster can be deleted even when deletion protection is enabled for the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-deletionprotection
        '''
        result = self._values.get("deletion_protection")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        '''The Active Directory directory ID to create the DB instance in.

        Currently, only Microsoft SQL Server, Oracle, and PostgreSQL DB instances can be created in an Active Directory Domain.

        For more information, see `Kerberos Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html>`_ in the *Amazon RDS User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-domain
        '''
        result = self._values.get("domain")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        '''The name of the IAM role to use when making API calls to the Directory Service.

        This setting doesn't apply to the following DB instances:

        - Amazon Aurora (The domain is managed by the DB cluster.)
        - RDS Custom

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-domainiamrolename
        '''
        result = self._values.get("domain_iam_role_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of log types that need to be enabled for exporting to CloudWatch Logs.

        The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Relational Database Service User Guide* .

        *Amazon Aurora*

        Not applicable. CloudWatch Logs exports are managed by the DB cluster.

        *MariaDB*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Microsoft SQL Server*

        Valid values: ``agent`` , ``error``

        *MySQL*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Oracle*

        Valid values: ``alert`` , ``audit`` , ``listener`` , ``trace``

        *PostgreSQL*

        Valid values: ``postgresql`` , ``upgrade``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enablecloudwatchlogsexports
        '''
        result = self._values.get("enable_cloudwatch_logs_exports")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        By default, mapping is disabled.

        This property is supported for RDS for MariaDB, RDS for MySQL, and RDS for PostgreSQL. For more information, see `IAM Database Authentication for MariaDB, MySQL, and PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon RDS User Guide.*

        *Amazon Aurora*

        Not applicable. Mapping AWS IAM accounts to database accounts is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enableiamdatabaseauthentication
        '''
        result = self._values.get("enable_iam_database_authentication")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def enable_performance_insights(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to enable Performance Insights for the DB instance.

        For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* .

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enableperformanceinsights
        '''
        result = self._values.get("enable_performance_insights")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def endpoint(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBInstance.EndpointProperty]]:
        '''The connection endpoint for the DB instance.

        .. epigraph::

           The endpoint might not be shown for instances with the status of ``creating`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-endpoint
        '''
        result = self._values.get("endpoint")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBInstance.EndpointProperty]], result)

    @builtins.property
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine that you want to use for this DB instance.

        .. epigraph::

           When you are creating a DB instance, the ``Engine`` property is required.

        Valid Values:

        - ``aurora-mysql`` (for Aurora MySQL DB instances)
        - ``aurora-postgresql`` (for Aurora PostgreSQL DB instances)
        - ``custom-oracle-ee`` (for RDS Custom for Oracle DB instances)
        - ``custom-oracle-ee-cdb`` (for RDS Custom for Oracle DB instances)
        - ``custom-sqlserver-ee`` (for RDS Custom for SQL Server DB instances)
        - ``custom-sqlserver-se`` (for RDS Custom for SQL Server DB instances)
        - ``custom-sqlserver-web`` (for RDS Custom for SQL Server DB instances)
        - ``mariadb``
        - ``mysql``
        - ``oracle-ee``
        - ``oracle-ee-cdb``
        - ``oracle-se2``
        - ``oracle-se2-cdb``
        - ``postgres``
        - ``sqlserver-ee``
        - ``sqlserver-se``
        - ``sqlserver-ex``
        - ``sqlserver-web``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-engine
        '''
        result = self._values.get("engine")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The version number of the database engine to use.

        For a list of valid engine versions, use the ``DescribeDBEngineVersions`` action.

        The following are the database engines and links to information about the major and minor versions that are available with Amazon RDS. Not every database engine is available for every AWS Region.

        *Amazon Aurora*

        Not applicable. The version number of the database engine to be used by the DB instance is managed by the DB cluster.

        *MariaDB*

        See `MariaDB on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MariaDB.html#MariaDB.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.*

        *Microsoft SQL Server*

        See `Microsoft SQL Server Versions on Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.VersionSupport>`_ in the *Amazon RDS User Guide.*

        *MySQL*

        See `MySQL on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.*

        *Oracle*

        See `Oracle Database Engine Release Notes <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.Oracle.PatchComposition.html>`_ in the *Amazon RDS User Guide.*

        *PostgreSQL*

        See `Supported PostgreSQL Database Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts.General.DBVersions>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-engineversion
        '''
        result = self._values.get("engine_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The number of I/O operations per second (IOPS) that the database provisions.

        The value must be equal to or greater than 1000.

        If you specify this property, you must follow the range of allowed ratios of your requested IOPS rate to the amount of storage that you allocate (IOPS to allocated storage). For example, you can provision an Oracle database instance with 1000 IOPS and 200 GiB of storage (a ratio of 5:1), or specify 2000 IOPS with 200 GiB of storage (a ratio of 10:1). For more information, see `Amazon RDS Provisioned IOPS Storage to Improve Performance <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* .
        .. epigraph::

           If you specify ``io1`` for the ``StorageType`` property, then you must also specify the ``Iops`` property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-iops
        '''
        result = self._values.get("iops")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS KMS key that's used to encrypt the DB instance, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` .

        If you enable the StorageEncrypted property but don't specify this property, AWS CloudFormation uses the default KMS key. If you specify this property, you must set the StorageEncrypted property to true.

        If you specify the ``SourceDBInstanceIdentifier`` property, the value is inherited from the source DB instance if the read replica is created in the same region.

        If you create an encrypted read replica in a different AWS Region, then you must specify a KMS key for the destination AWS Region. KMS encryption keys are specific to the region that they're created in, and you can't use encryption keys from one region in another region.

        If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify ``DBSecurityGroups`` , AWS CloudFormation ignores this property. To specify both a security group and this property, you must use a VPC security group. For more information about Amazon RDS and VPC, see `Using Amazon RDS with Amazon VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. The KMS key identifier is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-kmskeyid
        '''
        result = self._values.get("kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def license_model(self) -> typing.Optional[builtins.str]:
        '''License model information for this DB instance.

        Valid values:

        - Aurora MySQL - ``general-public-license``
        - Aurora PostgreSQL - ``postgresql-license``
        - MariaDB - ``general-public-license``
        - Microsoft SQL Server - ``license-included``
        - MySQL - ``general-public-license``
        - Oracle - ``bring-your-own-license`` or ``license-included``
        - PostgreSQL - ``postgresql-license``

        .. epigraph::

           If you've specified ``DBSecurityGroups`` and then you update the license model, AWS CloudFormation replaces the underlying DB instance. This will incur some interruptions to database availability.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-licensemodel
        '''
        result = self._values.get("license_model")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def manage_master_user_password(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to manage the master user password with AWS Secrets Manager.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*

        Constraints:

        - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-managemasteruserpassword
        '''
        result = self._values.get("manage_master_user_password")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def master_username(self) -> typing.Optional[builtins.str]:
        '''The master user name for the DB instance.

        .. epigraph::

           If you specify the ``SourceDBInstanceIdentifier`` or ``DBSnapshotIdentifier`` property, don't specify this property. The value is inherited from the source DB instance or snapshot.

        *Amazon Aurora*

        Not applicable. The name for the master user is managed by the DB cluster.

        *MariaDB*

        Constraints:

        - Required for MariaDB.
        - Must be 1 to 16 letters or numbers.
        - Can't be a reserved word for the chosen database engine.

        *Microsoft SQL Server*

        Constraints:

        - Required for SQL Server.
        - Must be 1 to 128 letters or numbers.
        - The first character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *MySQL*

        Constraints:

        - Required for MySQL.
        - Must be 1 to 16 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *Oracle*

        Constraints:

        - Required for Oracle.
        - Must be 1 to 30 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *PostgreSQL*

        Constraints:

        - Required for PostgreSQL.
        - Must be 1 to 63 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masterusername
        '''
        result = self._values.get("master_username")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def master_user_password(self) -> typing.Optional[builtins.str]:
        '''The password for the master user. The password can include any printable ASCII character except "/", """, or "@".

        *Amazon Aurora*

        Not applicable. The password for the master user is managed by the DB cluster.

        *MariaDB*

        Constraints: Must contain from 8 to 41 characters.

        *Microsoft SQL Server*

        Constraints: Must contain from 8 to 128 characters.

        *MySQL*

        Constraints: Must contain from 8 to 41 characters.

        *Oracle*

        Constraints: Must contain from 8 to 30 characters.

        *PostgreSQL*

        Constraints: Must contain from 8 to 128 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masteruserpassword
        '''
        result = self._values.get("master_user_password")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def master_user_secret(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBInstance.MasterUserSecretProperty]]:
        '''The secret managed by RDS in AWS Secrets Manager for the master user password.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masterusersecret
        '''
        result = self._values.get("master_user_secret")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBInstance.MasterUserSecretProperty]], result)

    @builtins.property
    def max_allocated_storage(self) -> typing.Optional[jsii.Number]:
        '''The upper limit in gibibytes (GiB) to which Amazon RDS can automatically scale the storage of the DB instance.

        For more information about this setting, including limitations that apply to it, see `Managing capacity automatically with Amazon RDS storage autoscaling <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling>`_ in the *Amazon RDS User Guide* .

        This setting doesn't apply to the following DB instances:

        - Amazon Aurora (Storage is managed by the DB cluster.)
        - RDS Custom

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-maxallocatedstorage
        '''
        result = self._values.get("max_allocated_storage")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        '''The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance.

        To disable collection of Enhanced Monitoring metrics, specify 0. The default is 0.

        If ``MonitoringRoleArn`` is specified, then you must set ``MonitoringInterval`` to a value other than 0.

        This setting doesn't apply to RDS Custom.

        Valid Values: ``0, 1, 5, 10, 15, 30, 60``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-monitoringinterval
        '''
        result = self._values.get("monitoring_interval")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs.

        For example, ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting Up and Enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* .

        If ``MonitoringInterval`` is set to a value other than ``0`` , then you must supply a ``MonitoringRoleArn`` value.

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-monitoringrolearn
        '''
        result = self._values.get("monitoring_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def multi_az(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the database instance is a Multi-AZ DB instance deployment.

        You can't set the ``AvailabilityZone`` parameter if the ``MultiAZ`` parameter is set to true.

        For more information, see `Multi-AZ deployments for high availability <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.MultiAZ.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. Amazon Aurora storage is replicated across all of the Availability Zones and doesn't require the ``MultiAZ`` option to be set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-multiaz
        '''
        result = self._values.get("multi_az")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def nchar_character_set_name(self) -> typing.Optional[builtins.str]:
        '''The name of the NCHAR character set for the Oracle DB instance.

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-ncharcharactersetname
        '''
        result = self._values.get("nchar_character_set_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_type(self) -> typing.Optional[builtins.str]:
        '''The network type of the DB instance.

        Valid values:

        - ``IPV4``
        - ``DUAL``

        The network type is determined by the ``DBSubnetGroup`` specified for the DB instance. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ).

        For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-networktype
        '''
        result = self._values.get("network_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def option_group_name(self) -> typing.Optional[builtins.str]:
        '''Indicates that the DB instance should be associated with the specified option group.

        Permanent options, such as the TDE option for Oracle Advanced Security TDE, can't be removed from an option group. Also, that option group can't be removed from a DB instance once it is associated with a DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-optiongroupname
        '''
        result = self._values.get("option_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for encryption of Performance Insights data.

        The KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.

        If you do not specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account. Your AWS account has a different default KMS key for each AWS Region.

        For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-performanceinsightskmskeyid
        '''
        result = self._values.get("performance_insights_kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days to retain Performance Insights data.

        This setting doesn't apply to RDS Custom DB instances.

        Valid Values:

        - ``7``
        - *month* * 31, where *month* is a number of months from 1-23. Examples: ``93`` (3 months * 31), ``341`` (11 months * 31), ``589`` (19 months * 31)
        - ``731``

        Default: ``7`` days

        If you specify a retention period that isn't valid, such as ``94`` , Amazon RDS returns an error.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-performanceinsightsretentionperiod
        '''
        result = self._values.get("performance_insights_retention_period")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def port(self) -> typing.Optional[builtins.str]:
        '''The port number on which the database accepts connections.

        *Amazon Aurora*

        Not applicable. The port number is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-port
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        '''The daily time range during which automated backups are created if automated backups are enabled, using the ``BackupRetentionPeriod`` parameter.

        For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow>`_ in the *Amazon RDS User Guide.*

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi`` .
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        *Amazon Aurora*

        Not applicable. The daily time range for creating automated backups is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-preferredbackupwindow
        '''
        result = self._values.get("preferred_backup_window")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        '''The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).

        Format: ``ddd:hh24:mi-ddd:hh24:mi``

        The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Instance Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow>`_ in the *Amazon RDS User Guide.*
        .. epigraph::

           This property applies when AWS CloudFormation initially creates the DB instance. If you use AWS CloudFormation to update the DB instance, those updates are applied immediately.

        Constraints: Minimum 30-minute window.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-preferredmaintenancewindow
        '''
        result = self._values.get("preferred_maintenance_window")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def processor_features(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBInstance.ProcessorFeatureProperty]]]]:
        '''The number of CPU cores and the number of threads per core for the DB instance class of the DB instance.

        This setting doesn't apply to Amazon Aurora or RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-processorfeatures
        '''
        result = self._values.get("processor_features")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBInstance.ProcessorFeatureProperty]]]], result)

    @builtins.property
    def promotion_tier(self) -> typing.Optional[jsii.Number]:
        '''The order of priority in which an Aurora Replica is promoted to the primary instance after a failure of the existing primary instance.

        For more information, see `Fault Tolerance for an Aurora DB Cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.AuroraHighAvailability.html#Aurora.Managing.FaultTolerance>`_ in the *Amazon Aurora User Guide* .

        This setting doesn't apply to RDS Custom DB instances.

        Default: ``1``

        Valid Values: ``0 - 15``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-promotiontier
        '''
        result = self._values.get("promotion_tier")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether the DB instance is an internet-facing instance.

        If you specify true, AWS CloudFormation creates an instance with a publicly resolvable DNS name, which resolves to a public IP address. If you specify false, AWS CloudFormation creates an internal instance with a DNS name that resolves to a private IP address.

        The default behavior value depends on your VPC setup and the database subnet group. For more information, see the ``PubliclyAccessible`` parameter in the `CreateDBInstance <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html>`_ in the *Amazon RDS API Reference* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-publiclyaccessible
        '''
        result = self._values.get("publicly_accessible")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def replica_mode(self) -> typing.Optional[builtins.str]:
        '''The open mode of an Oracle read replica.

        For more information, see `Working with Oracle Read Replicas for Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html>`_ in the *Amazon RDS User Guide* .

        This setting is only supported in RDS for Oracle.

        Default: ``open-read-only``

        Valid Values: ``open-read-only`` or ``mounted``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-replicamode
        '''
        result = self._values.get("replica_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def restore_time(self) -> typing.Optional[builtins.str]:
        '''The date and time to restore from.

        Valid Values: Value must be a time in Universal Coordinated Time (UTC) format

        Constraints:

        - Must be before the latest restorable time for the DB instance
        - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled

        Example: ``2009-09-07T23:45:00Z``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-restoretime
        '''
        result = self._values.get("restore_time")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the Multi-AZ DB cluster that will act as the source for the read replica.

        Each DB cluster can have up to 15 read replicas.

        Constraints:

        - Must be the identifier of an existing Multi-AZ DB cluster.
        - Can't be specified if the ``SourceDBInstanceIdentifier`` parameter is also specified.
        - The specified DB cluster must have automatic backups enabled, that is, its backup retention period must be greater than 0.
        - The source DB cluster must be in the same AWS Region as the read replica. Cross-Region replication isn't supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbclusteridentifier
        '''
        result = self._values.get("source_db_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_db_instance_automated_backups_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the replicated automated backups from which to restore, for example, ``arn:aws:rds:useast-1:123456789012:auto-backup:ab-L2IJCEXJP7XQ7HOJ4SIEXAMPLE`` .

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbinstanceautomatedbackupsarn
        '''
        result = self._values.get("source_db_instance_automated_backups_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_db_instance_identifier(self) -> typing.Optional[builtins.str]:
        '''If you want to create a read replica DB instance, specify the ID of the source DB instance.

        Each DB instance can have a limited number of read replicas. For more information, see `Working with Read Replicas <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html>`_ in the *Amazon RDS User Guide* .

        For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* .

        The ``SourceDBInstanceIdentifier`` property determines whether a DB instance is a read replica. If you remove the ``SourceDBInstanceIdentifier`` property from your template and then update your stack, AWS CloudFormation promotes the Read Replica to a standalone DB instance.
        .. epigraph::

           - If you specify a source DB instance that uses VPC security groups, we recommend that you specify the ``VPCSecurityGroups`` property. If you don't specify the property, the read replica inherits the value of the ``VPCSecurityGroups`` property from the source DB when you create the replica. However, if you update the stack, AWS CloudFormation reverts the replica's ``VPCSecurityGroups`` property to the default value because it's not defined in the stack's template. This change might cause unexpected issues.
           - Read replicas don't support deletion policies. AWS CloudFormation ignores any deletion policy that's associated with a read replica.
           - If you specify ``SourceDBInstanceIdentifier`` , don't specify the ``DBSnapshotIdentifier`` property. You can't create a read replica from a snapshot.
           - Don't set the ``BackupRetentionPeriod`` , ``DBName`` , ``MasterUsername`` , ``MasterUserPassword`` , and ``PreferredBackupWindow`` properties. The database attributes are inherited from the source DB instance, and backups are disabled for read replicas.
           - If the source DB instance is in a different region than the read replica, specify the source region in ``SourceRegion`` , and specify an ARN for a valid DB instance in ``SourceDBInstanceIdentifier`` . For more information, see `Constructing a Amazon RDS Amazon Resource Name (ARN) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html#USER_Tagging.ARN>`_ in the *Amazon RDS User Guide* .
           - For DB instances in Amazon Aurora clusters, don't specify this property. Amazon RDS automatically assigns writer and reader DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbinstanceidentifier
        '''
        result = self._values.get("source_db_instance_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_dbi_resource_id(self) -> typing.Optional[builtins.str]:
        '''The resource ID of the source DB instance from which to restore.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbiresourceid
        '''
        result = self._values.get("source_dbi_resource_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_region(self) -> typing.Optional[builtins.str]:
        '''The ID of the region that contains the source DB instance for the read replica.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourceregion
        '''
        result = self._values.get("source_region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether the DB instance is encrypted. By default, it isn't encrypted.

        If you specify the ``KmsKeyId`` property, then you must enable encryption.

        If you specify the ``SourceDBInstanceIdentifier`` property, don't specify this property. The value is inherited from the source DB instance, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB instance is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB instance to be encrypted, then don't set this property or set it to ``false`` .

        *Amazon Aurora*

        Not applicable. The encryption for DB instances is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storageencrypted
        '''
        result = self._values.get("storage_encrypted")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def storage_throughput(self) -> typing.Optional[jsii.Number]:
        '''Specifies the storage throughput value for the DB instance. This setting applies only to the ``gp3`` storage type.

        This setting doesn't apply to RDS Custom or Amazon Aurora.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storagethroughput
        '''
        result = self._values.get("storage_throughput")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def storage_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the storage type to be associated with the DB instance.

        Valid values: ``gp2 | gp3 | io1 | standard``

        The ``standard`` value is also known as magnetic.

        If you specify ``io1`` or ``gp3`` , you must also include a value for the ``Iops`` parameter.

        Default: ``io1`` if the ``Iops`` parameter is specified, otherwise ``gp2``

        For more information, see `Amazon RDS DB Instance Storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. Aurora data is stored in the cluster volume, which is a single, virtual volume that uses solid state drives (SSDs).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storagetype
        '''
        result = self._values.get("storage_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''An optional array of key-value pairs to apply to this DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def timezone(self) -> typing.Optional[builtins.str]:
        '''The time zone of the DB instance.

        The time zone parameter is currently supported only by `Microsoft SQL Server <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-timezone
        '''
        result = self._values.get("timezone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def use_default_processor_features(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the DB instance class of the DB instance uses its default processor features.

        This setting doesn't apply to RDS Custom DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-usedefaultprocessorfeatures
        '''
        result = self._values.get("use_default_processor_features")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether the DB instance is restored from the latest backup time.

        By default, the DB instance isn't restored from the latest backup time.

        Constraints: Can't be specified if the ``RestoreTime`` parameter is provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-uselatestrestorabletime
        '''
        result = self._values.get("use_latest_restorable_time")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def vpc_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the VPC security group IDs to assign to the DB instance.

        The list can include both the physical IDs of existing VPC security groups and references to `AWS::EC2::SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        If you set ``VPCSecurityGroups`` , you must not set ```DBSecurityGroups`` <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ , and vice versa.
        .. epigraph::

           You can migrate a DB instance in your stack from an RDS DB security group to a VPC security group, but keep the following in mind:

           - You can't revert to using an RDS security group after you establish a VPC security group membership.
           - When you migrate your DB instance to VPC security groups, if your stack update rolls back because the DB instance update fails or because an update fails in another AWS CloudFormation resource, the rollback fails because it can't revert to an RDS security group.
           - To use the properties that are available when you use a VPC security group, you must recreate the DB instance. If you don't, AWS CloudFormation submits only the property values that are listed in the ```DBSecurityGroups`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ property.

        To avoid this situation, migrate your DB instance to using VPC security groups only when that is the only change in your stack template.

        *Amazon Aurora*

        Not applicable. The associated list of EC2 VPC security groups is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-vpcsecuritygroups
        '''
        result = self._values.get("vpc_security_groups")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBInstanceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDBParameterGroup(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBParameterGroup",
):
    '''A CloudFormation ``AWS::RDS::DBParameterGroup``.

    The ``AWS::RDS::DBParameterGroup`` resource creates a custom parameter group for an RDS database family.

    This type can be declared in a template and referenced in the ``DBParameterGroupName`` property of an ``[AWS::RDS::DBInstance](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html)`` resource.

    For information about configuring parameters for Amazon RDS DB instances, see `Working with parameter groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* .

    For information about configuring parameters for Amazon Aurora DB instances, see `Working with parameter groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* .
    .. epigraph::

       Applying a parameter group to a DB instance may require the DB instance to reboot, resulting in a database outage for the duration of the reboot.

    :cloudformationResource: AWS::RDS::DBParameterGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        # parameters: Any
        
        cfn_dBParameter_group = rds.CfnDBParameterGroup(self, "MyCfnDBParameterGroup",
            description="description",
            family="family",
        
            # the properties below are optional
            db_parameter_group_name="dbParameterGroupName",
            parameters=parameters,
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        description: builtins.str,
        family: builtins.str,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        parameters: typing.Any = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBParameterGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param description: Provides the customer-specified description for this DB parameter group.
        :param family: The DB parameter group family name. A DB parameter group can be associated with one and only one DB parameter group family, and can be applied only to a DB instance running a DB engine and engine version compatible with that DB parameter group family. .. epigraph:: The DB parameter group family can't be changed when updating a DB parameter group. To list all of the available parameter group families, use the following command: ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"`` The output contains duplicates. For more information, see ``[CreateDBParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html)`` .
        :param db_parameter_group_name: The name of the DB parameter group. Constraints: - Must be 1 to 255 letters, numbers, or hyphens. - First character must be a letter - Can't end with a hyphen or contain two consecutive hyphens If you don't specify a value for ``DBParameterGroupName`` property, a name is automatically created for the DB parameter group. .. epigraph:: This value is stored as a lowercase string.
        :param parameters: An array of parameter names and values for the parameter update. At least one parameter name and value must be supplied. Subsequent arguments are optional. For more information about DB parameters and DB parameter groups for Amazon RDS DB engines, see `Working with DB Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* . For more information about DB cluster and DB instance parameters and parameter groups for Amazon Aurora DB engines, see `Working with DB Parameter Groups and DB Cluster Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* . .. epigraph:: AWS CloudFormation doesn't support specifying an apply method for each individual parameter. The default apply method for each parameter is used.
        :param tags: An optional array of key-value pairs to apply to this DB parameter group. .. epigraph:: Currently, this is the only property that supports drift detection.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c10aa78132a0eae6199258b5e55e45968fdce240e932138b48918eafb4c5da7b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBParameterGroupProps(
            description=description,
            family=family,
            db_parameter_group_name=db_parameter_group_name,
            parameters=parameters,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__efc4b731e12f3eba06751176aaa9c9fa8973f098809ed57b6bcd046116fd2dce)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__50c1ca61271e7c27de7f85a3bdb1255030dcb1b10d47cc10db4b1d7a38965369)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDbParameterGroupName")
    def attr_db_parameter_group_name(self) -> builtins.str:
        '''The name of the DB parameter group.

        :cloudformationAttribute: DBParameterGroupName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbParameterGroupName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''An optional array of key-value pairs to apply to this DB parameter group.

        .. epigraph::

           Currently, this is the only property that supports drift detection.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> builtins.str:
        '''Provides the customer-specified description for this DB parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-description
        '''
        return typing.cast(builtins.str, jsii.get(self, "description"))

    @description.setter
    def description(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b1e742192c12a8df7282b8be0119d1f389986eece1c470f22704a55b6114379)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="family")
    def family(self) -> builtins.str:
        '''The DB parameter group family name.

        A DB parameter group can be associated with one and only one DB parameter group family, and can be applied only to a DB instance running a DB engine and engine version compatible with that DB parameter group family.
        .. epigraph::

           The DB parameter group family can't be changed when updating a DB parameter group.

        To list all of the available parameter group families, use the following command:

        ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"``

        The output contains duplicates.

        For more information, see ``[CreateDBParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-family
        '''
        return typing.cast(builtins.str, jsii.get(self, "family"))

    @family.setter
    def family(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9e21a919c5d348b9b21e85ad530ebd9d61da372e1b23e938e25fddde0c0ce7d2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "family", value)

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Any:
        '''An array of parameter names and values for the parameter update.

        At least one parameter name and value must be supplied. Subsequent arguments are optional.

        For more information about DB parameters and DB parameter groups for Amazon RDS DB engines, see `Working with DB Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* .

        For more information about DB cluster and DB instance parameters and parameter groups for Amazon Aurora DB engines, see `Working with DB Parameter Groups and DB Cluster Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* .
        .. epigraph::

           AWS CloudFormation doesn't support specifying an apply method for each individual parameter. The default apply method for each parameter is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-parameters
        '''
        return typing.cast(typing.Any, jsii.get(self, "parameters"))

    @parameters.setter
    def parameters(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__987f1082fdf4a5da0b26fcc782aacbe415062f22410b31c0cf4ce6fdbfcf67ab)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parameters", value)

    @builtins.property
    @jsii.member(jsii_name="dbParameterGroupName")
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB parameter group.

        Constraints:

        - Must be 1 to 255 letters, numbers, or hyphens.
        - First character must be a letter
        - Can't end with a hyphen or contain two consecutive hyphens

        If you don't specify a value for ``DBParameterGroupName`` property, a name is automatically created for the DB parameter group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-dbparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbParameterGroupName"))

    @db_parameter_group_name.setter
    def db_parameter_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a22c7ecd784517cdedb05f2be1ead50de4ec8f05506992db84a909159e60e408)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbParameterGroupName", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBParameterGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "family": "family",
        "db_parameter_group_name": "dbParameterGroupName",
        "parameters": "parameters",
        "tags": "tags",
    },
)
class CfnDBParameterGroupProps:
    def __init__(
        self,
        *,
        description: builtins.str,
        family: builtins.str,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        parameters: typing.Any = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBParameterGroup``.

        :param description: Provides the customer-specified description for this DB parameter group.
        :param family: The DB parameter group family name. A DB parameter group can be associated with one and only one DB parameter group family, and can be applied only to a DB instance running a DB engine and engine version compatible with that DB parameter group family. .. epigraph:: The DB parameter group family can't be changed when updating a DB parameter group. To list all of the available parameter group families, use the following command: ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"`` The output contains duplicates. For more information, see ``[CreateDBParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html)`` .
        :param db_parameter_group_name: The name of the DB parameter group. Constraints: - Must be 1 to 255 letters, numbers, or hyphens. - First character must be a letter - Can't end with a hyphen or contain two consecutive hyphens If you don't specify a value for ``DBParameterGroupName`` property, a name is automatically created for the DB parameter group. .. epigraph:: This value is stored as a lowercase string.
        :param parameters: An array of parameter names and values for the parameter update. At least one parameter name and value must be supplied. Subsequent arguments are optional. For more information about DB parameters and DB parameter groups for Amazon RDS DB engines, see `Working with DB Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* . For more information about DB cluster and DB instance parameters and parameter groups for Amazon Aurora DB engines, see `Working with DB Parameter Groups and DB Cluster Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* . .. epigraph:: AWS CloudFormation doesn't support specifying an apply method for each individual parameter. The default apply method for each parameter is used.
        :param tags: An optional array of key-value pairs to apply to this DB parameter group. .. epigraph:: Currently, this is the only property that supports drift detection.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            # parameters: Any
            
            cfn_dBParameter_group_props = rds.CfnDBParameterGroupProps(
                description="description",
                family="family",
            
                # the properties below are optional
                db_parameter_group_name="dbParameterGroupName",
                parameters=parameters,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__587f9d7bfe1a369750fa782e04fc584380cc72ad65e3aaab22a3765433bb2450)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument family", value=family, expected_type=type_hints["family"])
            check_type(argname="argument db_parameter_group_name", value=db_parameter_group_name, expected_type=type_hints["db_parameter_group_name"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "description": description,
            "family": family,
        }
        if db_parameter_group_name is not None:
            self._values["db_parameter_group_name"] = db_parameter_group_name
        if parameters is not None:
            self._values["parameters"] = parameters
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def description(self) -> builtins.str:
        '''Provides the customer-specified description for this DB parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-description
        '''
        result = self._values.get("description")
        assert result is not None, "Required property 'description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def family(self) -> builtins.str:
        '''The DB parameter group family name.

        A DB parameter group can be associated with one and only one DB parameter group family, and can be applied only to a DB instance running a DB engine and engine version compatible with that DB parameter group family.
        .. epigraph::

           The DB parameter group family can't be changed when updating a DB parameter group.

        To list all of the available parameter group families, use the following command:

        ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"``

        The output contains duplicates.

        For more information, see ``[CreateDBParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-family
        '''
        result = self._values.get("family")
        assert result is not None, "Required property 'family' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB parameter group.

        Constraints:

        - Must be 1 to 255 letters, numbers, or hyphens.
        - First character must be a letter
        - Can't end with a hyphen or contain two consecutive hyphens

        If you don't specify a value for ``DBParameterGroupName`` property, a name is automatically created for the DB parameter group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-dbparametergroupname
        '''
        result = self._values.get("db_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Any:
        '''An array of parameter names and values for the parameter update.

        At least one parameter name and value must be supplied. Subsequent arguments are optional.

        For more information about DB parameters and DB parameter groups for Amazon RDS DB engines, see `Working with DB Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* .

        For more information about DB cluster and DB instance parameters and parameter groups for Amazon Aurora DB engines, see `Working with DB Parameter Groups and DB Cluster Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* .
        .. epigraph::

           AWS CloudFormation doesn't support specifying an apply method for each individual parameter. The default apply method for each parameter is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Any, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''An optional array of key-value pairs to apply to this DB parameter group.

        .. epigraph::

           Currently, this is the only property that supports drift detection.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBParameterGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDBProxy(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBProxy",
):
    '''A CloudFormation ``AWS::RDS::DBProxy``.

    The ``AWS::RDS::DBProxy`` resource creates or updates a DB proxy.

    For information about RDS Proxy for Amazon RDS, see `Managing Connections with Amazon RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html>`_ in the *Amazon RDS User Guide* .

    For information about RDS Proxy for Amazon Aurora, see `Managing Connections with Amazon RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html>`_ in the *Amazon Aurora User Guide* .
    .. epigraph::

       Limitations apply to RDS Proxy, including DB engine version limitations and AWS Region limitations.

       For information about limitations that apply to RDS Proxy for Amazon RDS, see `Limitations for RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy.limitations>`_ in the *Amazon RDS User Guide* .

       For information about that apply to RDS Proxy for Amazon Aurora, see `Limitations for RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html#rds-proxy.limitations>`_ in the *Amazon Aurora User Guide* .

    :cloudformationResource: AWS::RDS::DBProxy
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_dBProxy = rds.CfnDBProxy(self, "MyCfnDBProxy",
            auth=[rds.CfnDBProxy.AuthFormatProperty(
                auth_scheme="authScheme",
                client_password_auth_type="clientPasswordAuthType",
                description="description",
                iam_auth="iamAuth",
                secret_arn="secretArn"
            )],
            db_proxy_name="dbProxyName",
            engine_family="engineFamily",
            role_arn="roleArn",
            vpc_subnet_ids=["vpcSubnetIds"],
        
            # the properties below are optional
            debug_logging=False,
            idle_client_timeout=123,
            require_tls=False,
            tags=[rds.CfnDBProxy.TagFormatProperty(
                key="key",
                value="value"
            )],
            vpc_security_group_ids=["vpcSecurityGroupIds"]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auth: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBProxy.AuthFormatProperty", typing.Dict[builtins.str, typing.Any]]]]],
        db_proxy_name: builtins.str,
        engine_family: builtins.str,
        role_arn: builtins.str,
        vpc_subnet_ids: typing.Sequence[builtins.str],
        debug_logging: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        idle_client_timeout: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union["CfnDBProxy.TagFormatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBProxy``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auth: The authorization mechanism that the proxy uses.
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region . An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.
        :param engine_family: The kinds of databases that the proxy can connect to. This value determines which database network protocol the proxy recognizes when it interprets network traffic to and from the database. For Aurora MySQL, RDS for MariaDB, and RDS for MySQL databases, specify ``MYSQL`` . For Aurora PostgreSQL and RDS for PostgreSQL databases, specify ``POSTGRESQL`` . For RDS for Microsoft SQL Server, specify ``SQLSERVER`` . *Valid values* : ``MYSQL`` | ``POSTGRESQL`` | ``SQLSERVER``
        :param role_arn: The Amazon Resource Name (ARN) of the IAM role that the proxy uses to access secrets in AWS Secrets Manager.
        :param vpc_subnet_ids: One or more VPC subnet IDs to associate with the new proxy.
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs.
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database.
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy.
        :param tags: An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.
        :param vpc_security_group_ids: One or more VPC security group IDs to associate with the new proxy. If you plan to update the resource, don't specify VPC security groups in a shared VPC.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d94f4b32c99ad047b3a29dd2b83cad728335675f9397ef40b6355d8a4f79f2a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBProxyProps(
            auth=auth,
            db_proxy_name=db_proxy_name,
            engine_family=engine_family,
            role_arn=role_arn,
            vpc_subnet_ids=vpc_subnet_ids,
            debug_logging=debug_logging,
            idle_client_timeout=idle_client_timeout,
            require_tls=require_tls,
            tags=tags,
            vpc_security_group_ids=vpc_security_group_ids,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__351cc29b05d72e928a5b2d1751feed3437ab7086670c86649b38919b452760b2)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__069f114e3593f89959ab0180ffc2c341744b96a557ca447e885c4c7a4931659e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDbProxyArn")
    def attr_db_proxy_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) representing the target group.

        :cloudformationAttribute: DBProxyArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbProxyArn"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpoint")
    def attr_endpoint(self) -> builtins.str:
        '''The writer endpoint for the RDS DB instance or Aurora DB cluster.

        :cloudformationAttribute: Endpoint
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="attrVpcId")
    def attr_vpc_id(self) -> builtins.str:
        '''The VPC ID to associate with the DB proxy.

        :cloudformationAttribute: VpcId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVpcId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="auth")
    def auth(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBProxy.AuthFormatProperty"]]]:
        '''The authorization mechanism that the proxy uses.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-auth
        '''
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBProxy.AuthFormatProperty"]]], jsii.get(self, "auth"))

    @auth.setter
    def auth(
        self,
        value: typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBProxy.AuthFormatProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__869ef0e6807a044fbcf1b07244e827164f63faefbf4f14d0b5f86f685fd1c949)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "auth", value)

    @builtins.property
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        '''The identifier for the proxy.

        This name must be unique for all proxies owned by your AWS account in the specified AWS Region . An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-dbproxyname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbProxyName"))

    @db_proxy_name.setter
    def db_proxy_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3a9e7095215a0a9bdb8ab44b83362bbd101299c63fc799b2a6ff66bca4e33e1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbProxyName", value)

    @builtins.property
    @jsii.member(jsii_name="engineFamily")
    def engine_family(self) -> builtins.str:
        '''The kinds of databases that the proxy can connect to.

        This value determines which database network protocol the proxy recognizes when it interprets network traffic to and from the database. For Aurora MySQL, RDS for MariaDB, and RDS for MySQL databases, specify ``MYSQL`` . For Aurora PostgreSQL and RDS for PostgreSQL databases, specify ``POSTGRESQL`` . For RDS for Microsoft SQL Server, specify ``SQLSERVER`` .

        *Valid values* : ``MYSQL`` | ``POSTGRESQL`` | ``SQLSERVER``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-enginefamily
        '''
        return typing.cast(builtins.str, jsii.get(self, "engineFamily"))

    @engine_family.setter
    def engine_family(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__33274cc615f5efe896fb2bb53359c7a72868f554a4c622c60251562ecf4513b6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineFamily", value)

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role that the proxy uses to access secrets in AWS Secrets Manager.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-rolearn
        '''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @role_arn.setter
    def role_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__26bbab9272e0e8cc38f3c322aa24d461c5a03ae7e9f59e8ca108b48fb8b1f3a5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "roleArn", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSubnetIds")
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        '''One or more VPC subnet IDs to associate with the new proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsubnetids
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "vpcSubnetIds"))

    @vpc_subnet_ids.setter
    def vpc_subnet_ids(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__41d92774518120032be7c505a800af682ea1962c565406683dc31aa332e238e9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSubnetIds", value)

    @builtins.property
    @jsii.member(jsii_name="debugLogging")
    def debug_logging(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Whether the proxy includes detailed information about SQL statements in its logs.

        This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-debuglogging
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "debugLogging"))

    @debug_logging.setter
    def debug_logging(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__74bf8fd129aa7db6aa8cbe4c31cc1e5c4cc49092cdf5e1df037dc0169913f027)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "debugLogging", value)

    @builtins.property
    @jsii.member(jsii_name="idleClientTimeout")
    def idle_client_timeout(self) -> typing.Optional[jsii.Number]:
        '''The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it.

        You can set this value higher or lower than the connection timeout limit for the associated database.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-idleclienttimeout
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "idleClientTimeout"))

    @idle_client_timeout.setter
    def idle_client_timeout(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__20d24f3922bd4d273f50d9ca90e005b2b8fc4e91024a96337ec2377ca188b833)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "idleClientTimeout", value)

    @builtins.property
    @jsii.member(jsii_name="requireTls")
    def require_tls(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy.

        By enabling this setting, you can enforce encrypted TLS connections to the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-requiretls
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "requireTls"))

    @require_tls.setter
    def require_tls(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cafb97ac157af82b7b8950ec8723a47d1feed5e51aae3131c9f0eae5c7471530)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requireTls", value)

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List["CfnDBProxy.TagFormatProperty"]]:
        '''An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-tags
        '''
        return typing.cast(typing.Optional[typing.List["CfnDBProxy.TagFormatProperty"]], jsii.get(self, "tags"))

    @tags.setter
    def tags(
        self,
        value: typing.Optional[typing.List["CfnDBProxy.TagFormatProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__40e4176c733c58b29a8f2c935c044c4a11574a0150e5b0cfa2bfd50ca6c8e729)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSecurityGroupIds")
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more VPC security group IDs to associate with the new proxy.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsecuritygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "vpcSecurityGroupIds"))

    @vpc_security_group_ids.setter
    def vpc_security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0e3b5046d5f0ce58c561c6efda61644df88a6d63ab0ce20e5ce3314ed37dba62)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSecurityGroupIds", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBProxy.AuthFormatProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auth_scheme": "authScheme",
            "client_password_auth_type": "clientPasswordAuthType",
            "description": "description",
            "iam_auth": "iamAuth",
            "secret_arn": "secretArn",
        },
    )
    class AuthFormatProperty:
        def __init__(
            self,
            *,
            auth_scheme: typing.Optional[builtins.str] = None,
            client_password_auth_type: typing.Optional[builtins.str] = None,
            description: typing.Optional[builtins.str] = None,
            iam_auth: typing.Optional[builtins.str] = None,
            secret_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies the details of authentication used by a proxy to log in as a specific database user.

            :param auth_scheme: The type of authentication that the proxy uses for connections from the proxy to the underlying database. Valid Values: ``SECRETS``
            :param client_password_auth_type: Specifies the details of authentication used by a proxy to log in as a specific database user.
            :param description: A user-specified description about the authentication used by a proxy to log in as a specific database user.
            :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. The ``ENABLED`` value is valid only for proxies with RDS for Microsoft SQL Server. Valid Values: ``ENABLED | DISABLED | REQUIRED``
            :param secret_arn: The Amazon Resource Name (ARN) representing the secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                auth_format_property = rds.CfnDBProxy.AuthFormatProperty(
                    auth_scheme="authScheme",
                    client_password_auth_type="clientPasswordAuthType",
                    description="description",
                    iam_auth="iamAuth",
                    secret_arn="secretArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a8e9d7d6e63c918ff7e579edbd1d99d6ead2d526324d49c0b381f90050235c51)
                check_type(argname="argument auth_scheme", value=auth_scheme, expected_type=type_hints["auth_scheme"])
                check_type(argname="argument client_password_auth_type", value=client_password_auth_type, expected_type=type_hints["client_password_auth_type"])
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
                check_type(argname="argument iam_auth", value=iam_auth, expected_type=type_hints["iam_auth"])
                check_type(argname="argument secret_arn", value=secret_arn, expected_type=type_hints["secret_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if auth_scheme is not None:
                self._values["auth_scheme"] = auth_scheme
            if client_password_auth_type is not None:
                self._values["client_password_auth_type"] = client_password_auth_type
            if description is not None:
                self._values["description"] = description
            if iam_auth is not None:
                self._values["iam_auth"] = iam_auth
            if secret_arn is not None:
                self._values["secret_arn"] = secret_arn

        @builtins.property
        def auth_scheme(self) -> typing.Optional[builtins.str]:
            '''The type of authentication that the proxy uses for connections from the proxy to the underlying database.

            Valid Values: ``SECRETS``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-authscheme
            '''
            result = self._values.get("auth_scheme")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def client_password_auth_type(self) -> typing.Optional[builtins.str]:
            '''Specifies the details of authentication used by a proxy to log in as a specific database user.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-clientpasswordauthtype
            '''
            result = self._values.get("client_password_auth_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''A user-specified description about the authentication used by a proxy to log in as a specific database user.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def iam_auth(self) -> typing.Optional[builtins.str]:
            '''Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy.

            The ``ENABLED`` value is valid only for proxies with RDS for Microsoft SQL Server.

            Valid Values: ``ENABLED | DISABLED | REQUIRED``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-iamauth
            '''
            result = self._values.get("iam_auth")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def secret_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) representing the secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster.

            These secrets are stored within Amazon Secrets Manager.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-secretarn
            '''
            result = self._values.get("secret_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AuthFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBProxy.TagFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class TagFormatProperty:
        def __init__(
            self,
            *,
            key: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Metadata assigned to a DB proxy consisting of a key-value pair.

            :param key: A key is the required name of the tag. The string value can be 1-128 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+\\-]*)$").
            :param value: A value is the optional value of the tag. The string value can be 1-256 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+\\-]*)$").

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-tagformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                tag_format_property = rds.CfnDBProxy.TagFormatProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c2a838d51de0d587c5831bee2e69a94c00091ee455b057a2493d4bae9b295818)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if key is not None:
                self._values["key"] = key
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def key(self) -> typing.Optional[builtins.str]:
            '''A key is the required name of the tag.

            The string value can be 1-128 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+-]*)$").

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-tagformat.html#cfn-rds-dbproxy-tagformat-key
            '''
            result = self._values.get("key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''A value is the optional value of the tag.

            The string value can be 1-256 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+-]*)$").

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-tagformat.html#cfn-rds-dbproxy-tagformat-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnDBProxyEndpoint(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBProxyEndpoint",
):
    '''A CloudFormation ``AWS::RDS::DBProxyEndpoint``.

    The ``AWS::RDS::DBProxyEndpoint`` resource creates or updates a DB proxy endpoint. You can use custom proxy endpoints to access a proxy through a different VPC than the proxy's default VPC.

    For more information about RDS Proxy, see `AWS::RDS::DBProxy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html>`_ .

    :cloudformationResource: AWS::RDS::DBProxyEndpoint
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_dBProxy_endpoint = rds.CfnDBProxyEndpoint(self, "MyCfnDBProxyEndpoint",
            db_proxy_endpoint_name="dbProxyEndpointName",
            db_proxy_name="dbProxyName",
            vpc_subnet_ids=["vpcSubnetIds"],
        
            # the properties below are optional
            tags=[rds.CfnDBProxyEndpoint.TagFormatProperty(
                key="key",
                value="value"
            )],
            target_role="targetRole",
            vpc_security_group_ids=["vpcSecurityGroupIds"]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        db_proxy_endpoint_name: builtins.str,
        db_proxy_name: builtins.str,
        vpc_subnet_ids: typing.Sequence[builtins.str],
        tags: typing.Optional[typing.Sequence[typing.Union["CfnDBProxyEndpoint.TagFormatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        target_role: typing.Optional[builtins.str] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBProxyEndpoint``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_proxy_endpoint_name: The name of the DB proxy endpoint to create.
        :param db_proxy_name: The name of the DB proxy associated with the DB proxy endpoint that you create.
        :param vpc_subnet_ids: The VPC subnet IDs for the DB proxy endpoint that you create. You can specify a different set of subnet IDs than for the original DB proxy.
        :param tags: An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.
        :param target_role: A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations. Valid Values: ``READ_WRITE | READ_ONLY``
        :param vpc_security_group_ids: The VPC security group IDs for the DB proxy endpoint that you create. You can specify a different set of security group IDs than for the original DB proxy. The default is the default security group for the VPC.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54411dbda583caaf4997925cbfd0ea949cf7e685dcde3cdc9e9e301dc04c681b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBProxyEndpointProps(
            db_proxy_endpoint_name=db_proxy_endpoint_name,
            db_proxy_name=db_proxy_name,
            vpc_subnet_ids=vpc_subnet_ids,
            tags=tags,
            target_role=target_role,
            vpc_security_group_ids=vpc_security_group_ids,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8cd814b1e081a7a013f22a10ad34249f6767094840e35f3af5c35a47574ee155)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f0fc4a41a4b0304369bfd2211b2c230e0149c46e488fc431093f26121debf2b5)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDbProxyEndpointArn")
    def attr_db_proxy_endpoint_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) representing the DB proxy endpoint.

        :cloudformationAttribute: DBProxyEndpointArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbProxyEndpointArn"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpoint")
    def attr_endpoint(self) -> builtins.str:
        '''The custom endpoint for the RDS DB instance or Aurora DB cluster.

        :cloudformationAttribute: Endpoint
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="attrIsDefault")
    def attr_is_default(self) -> _IResolvable_da3f097b:
        '''A value that indicates whether this endpoint is the default endpoint for the associated DB proxy.

        Default DB proxy endpoints always have read/write capability. Other endpoints that you associate with the DB proxy can be either read/write or read-only.

        :cloudformationAttribute: IsDefault
        '''
        return typing.cast(_IResolvable_da3f097b, jsii.get(self, "attrIsDefault"))

    @builtins.property
    @jsii.member(jsii_name="attrVpcId")
    def attr_vpc_id(self) -> builtins.str:
        '''The VPC ID of the DB proxy endpoint.

        :cloudformationAttribute: VpcId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVpcId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="dbProxyEndpointName")
    def db_proxy_endpoint_name(self) -> builtins.str:
        '''The name of the DB proxy endpoint to create.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-dbproxyendpointname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbProxyEndpointName"))

    @db_proxy_endpoint_name.setter
    def db_proxy_endpoint_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4fb93bcffa32328d910e09228d3ab5aa421446dc563897ed2166fb90de6ef4b3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbProxyEndpointName", value)

    @builtins.property
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        '''The name of the DB proxy associated with the DB proxy endpoint that you create.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-dbproxyname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbProxyName"))

    @db_proxy_name.setter
    def db_proxy_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3095e42d7cd1f0e74a79a5ee9fc57b041d819bd79c12932b424bf150aa93eea3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbProxyName", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSubnetIds")
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        '''The VPC subnet IDs for the DB proxy endpoint that you create.

        You can specify a different set of subnet IDs than for the original DB proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-vpcsubnetids
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "vpcSubnetIds"))

    @vpc_subnet_ids.setter
    def vpc_subnet_ids(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__46c2bdfcac6804daeebeae78f3431ef98a8d5656ee2fedd5f51ebf7343ca46d8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSubnetIds", value)

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(
        self,
    ) -> typing.Optional[typing.List["CfnDBProxyEndpoint.TagFormatProperty"]]:
        '''An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-tags
        '''
        return typing.cast(typing.Optional[typing.List["CfnDBProxyEndpoint.TagFormatProperty"]], jsii.get(self, "tags"))

    @tags.setter
    def tags(
        self,
        value: typing.Optional[typing.List["CfnDBProxyEndpoint.TagFormatProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6a5dc208e5f863d586ec1e35a3b9362d553db4adbb0ff7cc43308284678c640)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value)

    @builtins.property
    @jsii.member(jsii_name="targetRole")
    def target_role(self) -> typing.Optional[builtins.str]:
        '''A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.

        Valid Values: ``READ_WRITE | READ_ONLY``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-targetrole
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "targetRole"))

    @target_role.setter
    def target_role(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aee92c413e62fd2a5a22a00876726fa8421a5c4d7295829abae273dcf808ab6a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetRole", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSecurityGroupIds")
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The VPC security group IDs for the DB proxy endpoint that you create.

        You can specify a different set of security group IDs than for the original DB proxy. The default is the default security group for the VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-vpcsecuritygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "vpcSecurityGroupIds"))

    @vpc_security_group_ids.setter
    def vpc_security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3cc8627aac139cb2d52a0c04399764a62f3f25267f6089132138b2c719830039)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSecurityGroupIds", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBProxyEndpoint.TagFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class TagFormatProperty:
        def __init__(
            self,
            *,
            key: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Metadata assigned to a DB proxy endpoint consisting of a key-value pair.

            :param key: A value is the optional value of the tag. The string value can be 1-256 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+\\-]*)$").
            :param value: Metadata assigned to a DB instance consisting of a key-value pair.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxyendpoint-tagformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                tag_format_property = rds.CfnDBProxyEndpoint.TagFormatProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5a43cebdeeee4c38fc29f9f6004b44e7c42752d002f8ff4f934c2ca809ca74cd)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if key is not None:
                self._values["key"] = key
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def key(self) -> typing.Optional[builtins.str]:
            '''A value is the optional value of the tag.

            The string value can be 1-256 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+-]*)$").

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxyendpoint-tagformat.html#cfn-rds-dbproxyendpoint-tagformat-key
            '''
            result = self._values.get("key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''Metadata assigned to a DB instance consisting of a key-value pair.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxyendpoint-tagformat.html#cfn-rds-dbproxyendpoint-tagformat-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBProxyEndpointProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_proxy_endpoint_name": "dbProxyEndpointName",
        "db_proxy_name": "dbProxyName",
        "vpc_subnet_ids": "vpcSubnetIds",
        "tags": "tags",
        "target_role": "targetRole",
        "vpc_security_group_ids": "vpcSecurityGroupIds",
    },
)
class CfnDBProxyEndpointProps:
    def __init__(
        self,
        *,
        db_proxy_endpoint_name: builtins.str,
        db_proxy_name: builtins.str,
        vpc_subnet_ids: typing.Sequence[builtins.str],
        tags: typing.Optional[typing.Sequence[typing.Union[CfnDBProxyEndpoint.TagFormatProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        target_role: typing.Optional[builtins.str] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBProxyEndpoint``.

        :param db_proxy_endpoint_name: The name of the DB proxy endpoint to create.
        :param db_proxy_name: The name of the DB proxy associated with the DB proxy endpoint that you create.
        :param vpc_subnet_ids: The VPC subnet IDs for the DB proxy endpoint that you create. You can specify a different set of subnet IDs than for the original DB proxy.
        :param tags: An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.
        :param target_role: A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations. Valid Values: ``READ_WRITE | READ_ONLY``
        :param vpc_security_group_ids: The VPC security group IDs for the DB proxy endpoint that you create. You can specify a different set of security group IDs than for the original DB proxy. The default is the default security group for the VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_dBProxy_endpoint_props = rds.CfnDBProxyEndpointProps(
                db_proxy_endpoint_name="dbProxyEndpointName",
                db_proxy_name="dbProxyName",
                vpc_subnet_ids=["vpcSubnetIds"],
            
                # the properties below are optional
                tags=[rds.CfnDBProxyEndpoint.TagFormatProperty(
                    key="key",
                    value="value"
                )],
                target_role="targetRole",
                vpc_security_group_ids=["vpcSecurityGroupIds"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3a9b3c0e1e94a5b7b563dceb2d99b8e9a70c4025e99ff97121a4d04a018e085a)
            check_type(argname="argument db_proxy_endpoint_name", value=db_proxy_endpoint_name, expected_type=type_hints["db_proxy_endpoint_name"])
            check_type(argname="argument db_proxy_name", value=db_proxy_name, expected_type=type_hints["db_proxy_name"])
            check_type(argname="argument vpc_subnet_ids", value=vpc_subnet_ids, expected_type=type_hints["vpc_subnet_ids"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target_role", value=target_role, expected_type=type_hints["target_role"])
            check_type(argname="argument vpc_security_group_ids", value=vpc_security_group_ids, expected_type=type_hints["vpc_security_group_ids"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_proxy_endpoint_name": db_proxy_endpoint_name,
            "db_proxy_name": db_proxy_name,
            "vpc_subnet_ids": vpc_subnet_ids,
        }
        if tags is not None:
            self._values["tags"] = tags
        if target_role is not None:
            self._values["target_role"] = target_role
        if vpc_security_group_ids is not None:
            self._values["vpc_security_group_ids"] = vpc_security_group_ids

    @builtins.property
    def db_proxy_endpoint_name(self) -> builtins.str:
        '''The name of the DB proxy endpoint to create.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-dbproxyendpointname
        '''
        result = self._values.get("db_proxy_endpoint_name")
        assert result is not None, "Required property 'db_proxy_endpoint_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def db_proxy_name(self) -> builtins.str:
        '''The name of the DB proxy associated with the DB proxy endpoint that you create.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-dbproxyname
        '''
        result = self._values.get("db_proxy_name")
        assert result is not None, "Required property 'db_proxy_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        '''The VPC subnet IDs for the DB proxy endpoint that you create.

        You can specify a different set of subnet IDs than for the original DB proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-vpcsubnetids
        '''
        result = self._values.get("vpc_subnet_ids")
        assert result is not None, "Required property 'vpc_subnet_ids' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def tags(
        self,
    ) -> typing.Optional[typing.List[CfnDBProxyEndpoint.TagFormatProperty]]:
        '''An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[CfnDBProxyEndpoint.TagFormatProperty]], result)

    @builtins.property
    def target_role(self) -> typing.Optional[builtins.str]:
        '''A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.

        Valid Values: ``READ_WRITE | READ_ONLY``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-targetrole
        '''
        result = self._values.get("target_role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The VPC security group IDs for the DB proxy endpoint that you create.

        You can specify a different set of security group IDs than for the original DB proxy. The default is the default security group for the VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-vpcsecuritygroupids
        '''
        result = self._values.get("vpc_security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBProxyEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBProxyProps",
    jsii_struct_bases=[],
    name_mapping={
        "auth": "auth",
        "db_proxy_name": "dbProxyName",
        "engine_family": "engineFamily",
        "role_arn": "roleArn",
        "vpc_subnet_ids": "vpcSubnetIds",
        "debug_logging": "debugLogging",
        "idle_client_timeout": "idleClientTimeout",
        "require_tls": "requireTls",
        "tags": "tags",
        "vpc_security_group_ids": "vpcSecurityGroupIds",
    },
)
class CfnDBProxyProps:
    def __init__(
        self,
        *,
        auth: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBProxy.AuthFormatProperty, typing.Dict[builtins.str, typing.Any]]]]],
        db_proxy_name: builtins.str,
        engine_family: builtins.str,
        role_arn: builtins.str,
        vpc_subnet_ids: typing.Sequence[builtins.str],
        debug_logging: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        idle_client_timeout: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[CfnDBProxy.TagFormatProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBProxy``.

        :param auth: The authorization mechanism that the proxy uses.
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region . An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.
        :param engine_family: The kinds of databases that the proxy can connect to. This value determines which database network protocol the proxy recognizes when it interprets network traffic to and from the database. For Aurora MySQL, RDS for MariaDB, and RDS for MySQL databases, specify ``MYSQL`` . For Aurora PostgreSQL and RDS for PostgreSQL databases, specify ``POSTGRESQL`` . For RDS for Microsoft SQL Server, specify ``SQLSERVER`` . *Valid values* : ``MYSQL`` | ``POSTGRESQL`` | ``SQLSERVER``
        :param role_arn: The Amazon Resource Name (ARN) of the IAM role that the proxy uses to access secrets in AWS Secrets Manager.
        :param vpc_subnet_ids: One or more VPC subnet IDs to associate with the new proxy.
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs.
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database.
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy.
        :param tags: An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.
        :param vpc_security_group_ids: One or more VPC security group IDs to associate with the new proxy. If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_dBProxy_props = rds.CfnDBProxyProps(
                auth=[rds.CfnDBProxy.AuthFormatProperty(
                    auth_scheme="authScheme",
                    client_password_auth_type="clientPasswordAuthType",
                    description="description",
                    iam_auth="iamAuth",
                    secret_arn="secretArn"
                )],
                db_proxy_name="dbProxyName",
                engine_family="engineFamily",
                role_arn="roleArn",
                vpc_subnet_ids=["vpcSubnetIds"],
            
                # the properties below are optional
                debug_logging=False,
                idle_client_timeout=123,
                require_tls=False,
                tags=[rds.CfnDBProxy.TagFormatProperty(
                    key="key",
                    value="value"
                )],
                vpc_security_group_ids=["vpcSecurityGroupIds"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__331ec6568969c058d2fc6c0482e8c31b8cc3af661d01d569ed27059f5a683e34)
            check_type(argname="argument auth", value=auth, expected_type=type_hints["auth"])
            check_type(argname="argument db_proxy_name", value=db_proxy_name, expected_type=type_hints["db_proxy_name"])
            check_type(argname="argument engine_family", value=engine_family, expected_type=type_hints["engine_family"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            check_type(argname="argument vpc_subnet_ids", value=vpc_subnet_ids, expected_type=type_hints["vpc_subnet_ids"])
            check_type(argname="argument debug_logging", value=debug_logging, expected_type=type_hints["debug_logging"])
            check_type(argname="argument idle_client_timeout", value=idle_client_timeout, expected_type=type_hints["idle_client_timeout"])
            check_type(argname="argument require_tls", value=require_tls, expected_type=type_hints["require_tls"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument vpc_security_group_ids", value=vpc_security_group_ids, expected_type=type_hints["vpc_security_group_ids"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auth": auth,
            "db_proxy_name": db_proxy_name,
            "engine_family": engine_family,
            "role_arn": role_arn,
            "vpc_subnet_ids": vpc_subnet_ids,
        }
        if debug_logging is not None:
            self._values["debug_logging"] = debug_logging
        if idle_client_timeout is not None:
            self._values["idle_client_timeout"] = idle_client_timeout
        if require_tls is not None:
            self._values["require_tls"] = require_tls
        if tags is not None:
            self._values["tags"] = tags
        if vpc_security_group_ids is not None:
            self._values["vpc_security_group_ids"] = vpc_security_group_ids

    @builtins.property
    def auth(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBProxy.AuthFormatProperty]]]:
        '''The authorization mechanism that the proxy uses.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-auth
        '''
        result = self._values.get("auth")
        assert result is not None, "Required property 'auth' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBProxy.AuthFormatProperty]]], result)

    @builtins.property
    def db_proxy_name(self) -> builtins.str:
        '''The identifier for the proxy.

        This name must be unique for all proxies owned by your AWS account in the specified AWS Region . An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-dbproxyname
        '''
        result = self._values.get("db_proxy_name")
        assert result is not None, "Required property 'db_proxy_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def engine_family(self) -> builtins.str:
        '''The kinds of databases that the proxy can connect to.

        This value determines which database network protocol the proxy recognizes when it interprets network traffic to and from the database. For Aurora MySQL, RDS for MariaDB, and RDS for MySQL databases, specify ``MYSQL`` . For Aurora PostgreSQL and RDS for PostgreSQL databases, specify ``POSTGRESQL`` . For RDS for Microsoft SQL Server, specify ``SQLSERVER`` .

        *Valid values* : ``MYSQL`` | ``POSTGRESQL`` | ``SQLSERVER``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-enginefamily
        '''
        result = self._values.get("engine_family")
        assert result is not None, "Required property 'engine_family' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role that the proxy uses to access secrets in AWS Secrets Manager.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-rolearn
        '''
        result = self._values.get("role_arn")
        assert result is not None, "Required property 'role_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        '''One or more VPC subnet IDs to associate with the new proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsubnetids
        '''
        result = self._values.get("vpc_subnet_ids")
        assert result is not None, "Required property 'vpc_subnet_ids' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def debug_logging(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Whether the proxy includes detailed information about SQL statements in its logs.

        This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-debuglogging
        '''
        result = self._values.get("debug_logging")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def idle_client_timeout(self) -> typing.Optional[jsii.Number]:
        '''The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it.

        You can set this value higher or lower than the connection timeout limit for the associated database.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-idleclienttimeout
        '''
        result = self._values.get("idle_client_timeout")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def require_tls(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy.

        By enabling this setting, you can enforce encrypted TLS connections to the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-requiretls
        '''
        result = self._values.get("require_tls")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[CfnDBProxy.TagFormatProperty]]:
        '''An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[CfnDBProxy.TagFormatProperty]], result)

    @builtins.property
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more VPC security group IDs to associate with the new proxy.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsecuritygroupids
        '''
        result = self._values.get("vpc_security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBProxyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDBProxyTargetGroup(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBProxyTargetGroup",
):
    '''A CloudFormation ``AWS::RDS::DBProxyTargetGroup``.

    The ``AWS::RDS::DBProxyTargetGroup`` resource represents a set of RDS DB instances, Aurora DB clusters, or both that a proxy can connect to. Currently, each target group is associated with exactly one RDS DB instance or Aurora DB cluster.

    This data type is used as a response element in the ``DescribeDBProxyTargetGroups`` action.

    For information about RDS Proxy for Amazon RDS, see `Managing Connections with Amazon RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html>`_ in the *Amazon RDS User Guide* .

    For information about RDS Proxy for Amazon Aurora, see `Managing Connections with Amazon RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html>`_ in the *Amazon Aurora User Guide* .

    For a sample template that creates a DB proxy and registers a DB instance, see `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#aws-resource-rds-dbproxy--examples>`_ in AWS::RDS::DBProxy.
    .. epigraph::

       Limitations apply to RDS Proxy, including DB engine version limitations and AWS Region limitations.

       For information about limitations that apply to RDS Proxy for Amazon RDS, see `Limitations for RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy.limitations>`_ in the *Amazon RDS User Guide* .

       For information about that apply to RDS Proxy for Amazon Aurora, see `Limitations for RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html#rds-proxy.limitations>`_ in the *Amazon Aurora User Guide* .

    :cloudformationResource: AWS::RDS::DBProxyTargetGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_dBProxy_target_group = rds.CfnDBProxyTargetGroup(self, "MyCfnDBProxyTargetGroup",
            db_proxy_name="dbProxyName",
            target_group_name="targetGroupName",
        
            # the properties below are optional
            connection_pool_configuration_info=rds.CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty(
                connection_borrow_timeout=123,
                init_query="initQuery",
                max_connections_percent=123,
                max_idle_connections_percent=123,
                session_pinning_filters=["sessionPinningFilters"]
            ),
            db_cluster_identifiers=["dbClusterIdentifiers"],
            db_instance_identifiers=["dbInstanceIdentifiers"]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        db_proxy_name: builtins.str,
        target_group_name: builtins.str,
        connection_pool_configuration_info: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        db_cluster_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
        db_instance_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBProxyTargetGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_proxy_name: The identifier of the ``DBProxy`` that is associated with the ``DBProxyTargetGroup`` .
        :param target_group_name: The identifier for the target group. .. epigraph:: Currently, this property must be set to ``default`` .
        :param connection_pool_configuration_info: Settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .
        :param db_cluster_identifiers: One or more DB cluster identifiers.
        :param db_instance_identifiers: One or more DB instance identifiers.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__112a32157ab76a5f2f76f32ed941524642f6f3ce76f61a0d05694fdaaa109acf)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBProxyTargetGroupProps(
            db_proxy_name=db_proxy_name,
            target_group_name=target_group_name,
            connection_pool_configuration_info=connection_pool_configuration_info,
            db_cluster_identifiers=db_cluster_identifiers,
            db_instance_identifiers=db_instance_identifiers,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7e835973d8c39d41ae824b1338ef6baf42edf5fab602fe2d196dd30106560c45)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3691564fe7c245b1fbaa4510b5f75376cba6af00fc43e34e607ba350fea4b67b)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrTargetGroupArn")
    def attr_target_group_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) representing the target group.

        :cloudformationAttribute: TargetGroupArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrTargetGroupArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        '''The identifier of the ``DBProxy`` that is associated with the ``DBProxyTargetGroup`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbproxyname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbProxyName"))

    @db_proxy_name.setter
    def db_proxy_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c8ca62159383c226618750b550b9e5bec499f33b3efc4ee5837df640c5decb73)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbProxyName", value)

    @builtins.property
    @jsii.member(jsii_name="targetGroupName")
    def target_group_name(self) -> builtins.str:
        '''The identifier for the target group.

        .. epigraph::

           Currently, this property must be set to ``default`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-targetgroupname
        '''
        return typing.cast(builtins.str, jsii.get(self, "targetGroupName"))

    @target_group_name.setter
    def target_group_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ea68affa8cf2a2cfc1debf82e87971b7038f848912829aff3c950dded41eae3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="connectionPoolConfigurationInfo")
    def connection_pool_configuration_info(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty"]]:
        '''Settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfo
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty"]], jsii.get(self, "connectionPoolConfigurationInfo"))

    @connection_pool_configuration_info.setter
    def connection_pool_configuration_info(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__520851595c02429c55b9e77ddbc1a10539d75d1f1b3c4ee51d2f11794c3c6a8e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "connectionPoolConfigurationInfo", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterIdentifiers")
    def db_cluster_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more DB cluster identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbclusteridentifiers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "dbClusterIdentifiers"))

    @db_cluster_identifiers.setter
    def db_cluster_identifiers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__785e2a5c6a85bae9af5725efc3ca3a314f0cf13fb07115a23485e5f65478ef8d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterIdentifiers", value)

    @builtins.property
    @jsii.member(jsii_name="dbInstanceIdentifiers")
    def db_instance_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more DB instance identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbinstanceidentifiers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "dbInstanceIdentifiers"))

    @db_instance_identifiers.setter
    def db_instance_identifiers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35bfb6397667331b569deebc6de4d2a7bb40d612d7d70e07f9191c69e9d56e7e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbInstanceIdentifiers", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty",
        jsii_struct_bases=[],
        name_mapping={
            "connection_borrow_timeout": "connectionBorrowTimeout",
            "init_query": "initQuery",
            "max_connections_percent": "maxConnectionsPercent",
            "max_idle_connections_percent": "maxIdleConnectionsPercent",
            "session_pinning_filters": "sessionPinningFilters",
        },
    )
    class ConnectionPoolConfigurationInfoFormatProperty:
        def __init__(
            self,
            *,
            connection_borrow_timeout: typing.Optional[jsii.Number] = None,
            init_query: typing.Optional[builtins.str] = None,
            max_connections_percent: typing.Optional[jsii.Number] = None,
            max_idle_connections_percent: typing.Optional[jsii.Number] = None,
            session_pinning_filters: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Specifies the settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .

            :param connection_borrow_timeout: The number of seconds for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Default: 120 Constraints: between 1 and 3600, or 0 representing unlimited
            :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with ``SET`` statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single ``SET`` statement, such as ``SET x=1, y=2`` . Default: no initialization query
            :param max_connections_percent: The maximum size of the connection pool for each target in a target group. The value is expressed as a percentage of the ``max_connections`` setting for the RDS DB instance or Aurora DB cluster used by the target group. If you specify ``MaxIdleConnectionsPercent`` , then you must also include a value for this parameter. Default: 10 for RDS for Microsoft SQL Server, and 100 for all other engines Constraints: Must be between 1 and 100.
            :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. The value is expressed as a percentage of the ``max_connections`` setting for the RDS DB instance or Aurora DB cluster used by the target group. With a high value, the proxy leaves a high percentage of idle database connections open. A low value causes the proxy to close more idle connections and return them to the database. If you specify this parameter, then you must also include a value for ``MaxConnectionsPercent`` . Default: The default value is half of the value of ``MaxConnectionsPercent`` . For example, if ``MaxConnectionsPercent`` is 80, then the default value of ``MaxIdleConnectionsPercent`` is 40. If the value of ``MaxConnectionsPercent`` isn't specified, then for SQL Server, ``MaxIdleConnectionsPercent`` is 5, and for all other engines, the default is 50. Constraints: Must be between 0 and the value of ``MaxConnectionsPercent`` .
            :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: no session pinning filters

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                connection_pool_configuration_info_format_property = rds.CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty(
                    connection_borrow_timeout=123,
                    init_query="initQuery",
                    max_connections_percent=123,
                    max_idle_connections_percent=123,
                    session_pinning_filters=["sessionPinningFilters"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__42d639daffefef5729c8051d75bbd49babf647363ede6bca87c067f42ab78ca7)
                check_type(argname="argument connection_borrow_timeout", value=connection_borrow_timeout, expected_type=type_hints["connection_borrow_timeout"])
                check_type(argname="argument init_query", value=init_query, expected_type=type_hints["init_query"])
                check_type(argname="argument max_connections_percent", value=max_connections_percent, expected_type=type_hints["max_connections_percent"])
                check_type(argname="argument max_idle_connections_percent", value=max_idle_connections_percent, expected_type=type_hints["max_idle_connections_percent"])
                check_type(argname="argument session_pinning_filters", value=session_pinning_filters, expected_type=type_hints["session_pinning_filters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if connection_borrow_timeout is not None:
                self._values["connection_borrow_timeout"] = connection_borrow_timeout
            if init_query is not None:
                self._values["init_query"] = init_query
            if max_connections_percent is not None:
                self._values["max_connections_percent"] = max_connections_percent
            if max_idle_connections_percent is not None:
                self._values["max_idle_connections_percent"] = max_idle_connections_percent
            if session_pinning_filters is not None:
                self._values["session_pinning_filters"] = session_pinning_filters

        @builtins.property
        def connection_borrow_timeout(self) -> typing.Optional[jsii.Number]:
            '''The number of seconds for a proxy to wait for a connection to become available in the connection pool.

            Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions.

            Default: 120

            Constraints: between 1 and 3600, or 0 representing unlimited

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-connectionborrowtimeout
            '''
            result = self._values.get("connection_borrow_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def init_query(self) -> typing.Optional[builtins.str]:
            '''One or more SQL statements for the proxy to run when opening each new database connection.

            Typically used with ``SET`` statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single ``SET`` statement, such as ``SET x=1, y=2`` .

            Default: no initialization query

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-initquery
            '''
            result = self._values.get("init_query")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def max_connections_percent(self) -> typing.Optional[jsii.Number]:
            '''The maximum size of the connection pool for each target in a target group.

            The value is expressed as a percentage of the ``max_connections`` setting for the RDS DB instance or Aurora DB cluster used by the target group.

            If you specify ``MaxIdleConnectionsPercent`` , then you must also include a value for this parameter.

            Default: 10 for RDS for Microsoft SQL Server, and 100 for all other engines

            Constraints: Must be between 1 and 100.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-maxconnectionspercent
            '''
            result = self._values.get("max_connections_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def max_idle_connections_percent(self) -> typing.Optional[jsii.Number]:
            '''Controls how actively the proxy closes idle database connections in the connection pool.

            The value is expressed as a percentage of the ``max_connections`` setting for the RDS DB instance or Aurora DB cluster used by the target group. With a high value, the proxy leaves a high percentage of idle database connections open. A low value causes the proxy to close more idle connections and return them to the database.

            If you specify this parameter, then you must also include a value for ``MaxConnectionsPercent`` .

            Default: The default value is half of the value of ``MaxConnectionsPercent`` . For example, if ``MaxConnectionsPercent`` is 80, then the default value of ``MaxIdleConnectionsPercent`` is 40. If the value of ``MaxConnectionsPercent`` isn't specified, then for SQL Server, ``MaxIdleConnectionsPercent`` is 5, and for all other engines, the default is 50.

            Constraints: Must be between 0 and the value of ``MaxConnectionsPercent`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-maxidleconnectionspercent
            '''
            result = self._values.get("max_idle_connections_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def session_pinning_filters(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection.

            Including an item in the list exempts that class of SQL operations from the pinning behavior.

            Default: no session pinning filters

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-sessionpinningfilters
            '''
            result = self._values.get("session_pinning_filters")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConnectionPoolConfigurationInfoFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBProxyTargetGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_proxy_name": "dbProxyName",
        "target_group_name": "targetGroupName",
        "connection_pool_configuration_info": "connectionPoolConfigurationInfo",
        "db_cluster_identifiers": "dbClusterIdentifiers",
        "db_instance_identifiers": "dbInstanceIdentifiers",
    },
)
class CfnDBProxyTargetGroupProps:
    def __init__(
        self,
        *,
        db_proxy_name: builtins.str,
        target_group_name: builtins.str,
        connection_pool_configuration_info: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        db_cluster_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
        db_instance_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBProxyTargetGroup``.

        :param db_proxy_name: The identifier of the ``DBProxy`` that is associated with the ``DBProxyTargetGroup`` .
        :param target_group_name: The identifier for the target group. .. epigraph:: Currently, this property must be set to ``default`` .
        :param connection_pool_configuration_info: Settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .
        :param db_cluster_identifiers: One or more DB cluster identifiers.
        :param db_instance_identifiers: One or more DB instance identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_dBProxy_target_group_props = rds.CfnDBProxyTargetGroupProps(
                db_proxy_name="dbProxyName",
                target_group_name="targetGroupName",
            
                # the properties below are optional
                connection_pool_configuration_info=rds.CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty(
                    connection_borrow_timeout=123,
                    init_query="initQuery",
                    max_connections_percent=123,
                    max_idle_connections_percent=123,
                    session_pinning_filters=["sessionPinningFilters"]
                ),
                db_cluster_identifiers=["dbClusterIdentifiers"],
                db_instance_identifiers=["dbInstanceIdentifiers"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9da17c344959c8c8c8f82cc65de990bc4d662a9e9ebf37719fbb1019f5e30e6d)
            check_type(argname="argument db_proxy_name", value=db_proxy_name, expected_type=type_hints["db_proxy_name"])
            check_type(argname="argument target_group_name", value=target_group_name, expected_type=type_hints["target_group_name"])
            check_type(argname="argument connection_pool_configuration_info", value=connection_pool_configuration_info, expected_type=type_hints["connection_pool_configuration_info"])
            check_type(argname="argument db_cluster_identifiers", value=db_cluster_identifiers, expected_type=type_hints["db_cluster_identifiers"])
            check_type(argname="argument db_instance_identifiers", value=db_instance_identifiers, expected_type=type_hints["db_instance_identifiers"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_proxy_name": db_proxy_name,
            "target_group_name": target_group_name,
        }
        if connection_pool_configuration_info is not None:
            self._values["connection_pool_configuration_info"] = connection_pool_configuration_info
        if db_cluster_identifiers is not None:
            self._values["db_cluster_identifiers"] = db_cluster_identifiers
        if db_instance_identifiers is not None:
            self._values["db_instance_identifiers"] = db_instance_identifiers

    @builtins.property
    def db_proxy_name(self) -> builtins.str:
        '''The identifier of the ``DBProxy`` that is associated with the ``DBProxyTargetGroup`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbproxyname
        '''
        result = self._values.get("db_proxy_name")
        assert result is not None, "Required property 'db_proxy_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_group_name(self) -> builtins.str:
        '''The identifier for the target group.

        .. epigraph::

           Currently, this property must be set to ``default`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-targetgroupname
        '''
        result = self._values.get("target_group_name")
        assert result is not None, "Required property 'target_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def connection_pool_configuration_info(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty]]:
        '''Settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfo
        '''
        result = self._values.get("connection_pool_configuration_info")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty]], result)

    @builtins.property
    def db_cluster_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more DB cluster identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbclusteridentifiers
        '''
        result = self._values.get("db_cluster_identifiers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def db_instance_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more DB instance identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbinstanceidentifiers
        '''
        result = self._values.get("db_instance_identifiers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBProxyTargetGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDBSecurityGroup(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBSecurityGroup",
):
    '''A CloudFormation ``AWS::RDS::DBSecurityGroup``.

    The ``AWS::RDS::DBSecurityGroup`` resource creates or updates an Amazon RDS DB security group.
    .. epigraph::

       EC2-Classic was retired on August 15, 2022. If you haven't migrated from EC2-Classic to a VPC, we recommend that you migrate as soon as possible. For more information, see `Migrate from EC2-Classic to a VPC <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/vpc-migrate.html>`_ in the *Amazon EC2 User Guide* , the blog `EC2-Classic Networking is Retiring – Here’s How to Prepare <https://docs.aws.amazon.com/aws/ec2-classic-is-retiring-heres-how-to-prepare/>`_ , and `Moving a DB instance not in a VPC into a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Non-VPC2VPC.html>`_ in the *Amazon RDS User Guide* .

    :cloudformationResource: AWS::RDS::DBSecurityGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_dBSecurity_group = rds.CfnDBSecurityGroup(self, "MyCfnDBSecurityGroup",
            db_security_group_ingress=[rds.CfnDBSecurityGroup.IngressProperty(
                cidrip="cidrip",
                ec2_security_group_id="ec2SecurityGroupId",
                ec2_security_group_name="ec2SecurityGroupName",
                ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
            )],
            group_description="groupDescription",
        
            # the properties below are optional
            ec2_vpc_id="ec2VpcId",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        db_security_group_ingress: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDBSecurityGroup.IngressProperty", typing.Dict[builtins.str, typing.Any]]]]],
        group_description: builtins.str,
        ec2_vpc_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBSecurityGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_security_group_ingress: Ingress rules to be applied to the DB security group.
        :param group_description: Provides the description of the DB security group.
        :param ec2_vpc_id: The identifier of an Amazon VPC. This property indicates the VPC that this DB security group belongs to. .. epigraph:: The ``EC2VpcId`` property is for backward compatibility with older regions, and is no longer recommended for providing security information to an RDS DB instance.
        :param tags: An optional array of key-value pairs to apply to this DB security group.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b692ab00b4d5e8d1153b5156b9ec79e26f77bbcf930e94b43eb9adff11e07a12)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBSecurityGroupProps(
            db_security_group_ingress=db_security_group_ingress,
            group_description=group_description,
            ec2_vpc_id=ec2_vpc_id,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a639c8441a9b53f60f1bd979b80117dcf0844be7cc3f85e11bb21a0ef4e9f703)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1f773eb26f64d926a0c1f611df3adcd003c0430d775ce7af328d290dfa0b75ec)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''An optional array of key-value pairs to apply to this DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="dbSecurityGroupIngress")
    def db_security_group_ingress(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBSecurityGroup.IngressProperty"]]]:
        '''Ingress rules to be applied to the DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-dbsecuritygroupingress
        '''
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBSecurityGroup.IngressProperty"]]], jsii.get(self, "dbSecurityGroupIngress"))

    @db_security_group_ingress.setter
    def db_security_group_ingress(
        self,
        value: typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDBSecurityGroup.IngressProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f0a2828efed62f85e7a4227c409bcfcba0cc543df215aea11c2fbcb718b43eac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSecurityGroupIngress", value)

    @builtins.property
    @jsii.member(jsii_name="groupDescription")
    def group_description(self) -> builtins.str:
        '''Provides the description of the DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-groupdescription
        '''
        return typing.cast(builtins.str, jsii.get(self, "groupDescription"))

    @group_description.setter
    def group_description(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f62ca646c854acb28107210bd01d8eb5aba231c9aed0b63a809dbbee3ce39ec1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "groupDescription", value)

    @builtins.property
    @jsii.member(jsii_name="ec2VpcId")
    def ec2_vpc_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of an Amazon VPC. This property indicates the VPC that this DB security group belongs to.

        .. epigraph::

           The ``EC2VpcId`` property is for backward compatibility with older regions, and is no longer recommended for providing security information to an RDS DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-ec2vpcid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ec2VpcId"))

    @ec2_vpc_id.setter
    def ec2_vpc_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2dfd06adbf58bcb2beaf48e685c038d8a5fcd5eee11160cb7bbe777a59419f27)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ec2VpcId", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_rds.CfnDBSecurityGroup.IngressProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cidrip": "cidrip",
            "ec2_security_group_id": "ec2SecurityGroupId",
            "ec2_security_group_name": "ec2SecurityGroupName",
            "ec2_security_group_owner_id": "ec2SecurityGroupOwnerId",
        },
    )
    class IngressProperty:
        def __init__(
            self,
            *,
            cidrip: typing.Optional[builtins.str] = None,
            ec2_security_group_id: typing.Optional[builtins.str] = None,
            ec2_security_group_name: typing.Optional[builtins.str] = None,
            ec2_security_group_owner_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``Ingress`` property type specifies an individual ingress rule within an ``AWS::RDS::DBSecurityGroup`` resource.

            .. epigraph::

               EC2-Classic was retired on August 15, 2022. If you haven't migrated from EC2-Classic to a VPC, we recommend that you migrate as soon as possible. For more information, see `Migrate from EC2-Classic to a VPC <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/vpc-migrate.html>`_ in the *Amazon EC2 User Guide* , the blog `EC2-Classic Networking is Retiring – Here’s How to Prepare <https://docs.aws.amazon.com/aws/ec2-classic-is-retiring-heres-how-to-prepare/>`_ , and `Moving a DB instance not in a VPC into a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Non-VPC2VPC.html>`_ in the *Amazon RDS User Guide* .

            :param cidrip: The IP range to authorize.
            :param ec2_security_group_id: Id of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
            :param ec2_security_group_name: Name of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
            :param ec2_security_group_owner_id: AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter. The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_rds as rds
                
                ingress_property = rds.CfnDBSecurityGroup.IngressProperty(
                    cidrip="cidrip",
                    ec2_security_group_id="ec2SecurityGroupId",
                    ec2_security_group_name="ec2SecurityGroupName",
                    ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a48c4109d9d4d43f9460f70213a72a2e648c37730c6814118975426ce0619e53)
                check_type(argname="argument cidrip", value=cidrip, expected_type=type_hints["cidrip"])
                check_type(argname="argument ec2_security_group_id", value=ec2_security_group_id, expected_type=type_hints["ec2_security_group_id"])
                check_type(argname="argument ec2_security_group_name", value=ec2_security_group_name, expected_type=type_hints["ec2_security_group_name"])
                check_type(argname="argument ec2_security_group_owner_id", value=ec2_security_group_owner_id, expected_type=type_hints["ec2_security_group_owner_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cidrip is not None:
                self._values["cidrip"] = cidrip
            if ec2_security_group_id is not None:
                self._values["ec2_security_group_id"] = ec2_security_group_id
            if ec2_security_group_name is not None:
                self._values["ec2_security_group_name"] = ec2_security_group_name
            if ec2_security_group_owner_id is not None:
                self._values["ec2_security_group_owner_id"] = ec2_security_group_owner_id

        @builtins.property
        def cidrip(self) -> typing.Optional[builtins.str]:
            '''The IP range to authorize.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-cidrip
            '''
            result = self._values.get("cidrip")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_security_group_id(self) -> typing.Optional[builtins.str]:
            '''Id of the EC2 security group to authorize.

            For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-ec2securitygroupid
            '''
            result = self._values.get("ec2_security_group_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_security_group_name(self) -> typing.Optional[builtins.str]:
            '''Name of the EC2 security group to authorize.

            For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-ec2securitygroupname
            '''
            result = self._values.get("ec2_security_group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_security_group_owner_id(self) -> typing.Optional[builtins.str]:
            '''AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter.

            The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-ec2securitygroupownerid
            '''
            result = self._values.get("ec2_security_group_owner_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IngressProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnDBSecurityGroupIngress(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBSecurityGroupIngress",
):
    '''A CloudFormation ``AWS::RDS::DBSecurityGroupIngress``.

    The ``AWS::RDS::DBSecurityGroupIngress`` resource enables ingress to a DB security group using one of two forms of authorization. First, you can add EC2 or VPC security groups to the DB security group if the application using the database is running on EC2 or VPC instances. Second, IP ranges are available if the application accessing your database is running on the Internet.

    This type supports updates. For more information about updating stacks, see `AWS CloudFormation Stacks Updates <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks.html>`_ .

    For details about the settings for DB security group ingress, see `AuthorizeDBSecurityGroupIngress <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AuthorizeDBSecurityGroupIngress.html>`_ .
    .. epigraph::

       EC2-Classic was retired on August 15, 2022. If you haven't migrated from EC2-Classic to a VPC, we recommend that you migrate as soon as possible. For more information, see `Migrate from EC2-Classic to a VPC <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/vpc-migrate.html>`_ in the *Amazon EC2 User Guide* , the blog `EC2-Classic Networking is Retiring – Here’s How to Prepare <https://docs.aws.amazon.com/aws/ec2-classic-is-retiring-heres-how-to-prepare/>`_ , and `Moving a DB instance not in a VPC into a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Non-VPC2VPC.html>`_ in the *Amazon RDS User Guide* .

    :cloudformationResource: AWS::RDS::DBSecurityGroupIngress
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_dBSecurity_group_ingress = rds.CfnDBSecurityGroupIngress(self, "MyCfnDBSecurityGroupIngress",
            db_security_group_name="dbSecurityGroupName",
        
            # the properties below are optional
            cidrip="cidrip",
            ec2_security_group_id="ec2SecurityGroupId",
            ec2_security_group_name="ec2SecurityGroupName",
            ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        db_security_group_name: builtins.str,
        cidrip: typing.Optional[builtins.str] = None,
        ec2_security_group_id: typing.Optional[builtins.str] = None,
        ec2_security_group_name: typing.Optional[builtins.str] = None,
        ec2_security_group_owner_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBSecurityGroupIngress``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_security_group_name: The name of the DB security group to add authorization to.
        :param cidrip: The IP range to authorize.
        :param ec2_security_group_id: Id of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        :param ec2_security_group_name: Name of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        :param ec2_security_group_owner_id: AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter. The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ef332a54da387f7a97590f1d1d7f7363a9384151eafa12aae1f46abc72fc5a51)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBSecurityGroupIngressProps(
            db_security_group_name=db_security_group_name,
            cidrip=cidrip,
            ec2_security_group_id=ec2_security_group_id,
            ec2_security_group_name=ec2_security_group_name,
            ec2_security_group_owner_id=ec2_security_group_owner_id,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aec4efd08c20faa51bcb56008900f6329f0147d05ca47f0e96356e55f00496f6)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9fd46e18e460df4c798576e896cb82c23b772b47545518751efdd0b82b650bbd)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="dbSecurityGroupName")
    def db_security_group_name(self) -> builtins.str:
        '''The name of the DB security group to add authorization to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-dbsecuritygroupname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbSecurityGroupName"))

    @db_security_group_name.setter
    def db_security_group_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__33cb0b467b4b5a2127be68dfeee1b484e3f625bdde8c73d7f2f5861004c0f361)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSecurityGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="cidrip")
    def cidrip(self) -> typing.Optional[builtins.str]:
        '''The IP range to authorize.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-cidrip
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cidrip"))

    @cidrip.setter
    def cidrip(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b822e8578847e0e12edd4bd77727d367bfdb2efdab5adb3a74ecbea2b11e891)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cidrip", value)

    @builtins.property
    @jsii.member(jsii_name="ec2SecurityGroupId")
    def ec2_security_group_id(self) -> typing.Optional[builtins.str]:
        '''Id of the EC2 security group to authorize.

        For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ec2SecurityGroupId"))

    @ec2_security_group_id.setter
    def ec2_security_group_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bc4d69fafb31a7c2c2d8154ceba60a149b275d471fe03bc14976992875633538)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ec2SecurityGroupId", value)

    @builtins.property
    @jsii.member(jsii_name="ec2SecurityGroupName")
    def ec2_security_group_name(self) -> typing.Optional[builtins.str]:
        '''Name of the EC2 security group to authorize.

        For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ec2SecurityGroupName"))

    @ec2_security_group_name.setter
    def ec2_security_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2349d21856d92d874a8cb171c952c917d8223483bc87d0e3d511d3ef6e07157)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ec2SecurityGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="ec2SecurityGroupOwnerId")
    def ec2_security_group_owner_id(self) -> typing.Optional[builtins.str]:
        '''AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter.

        The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupownerid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ec2SecurityGroupOwnerId"))

    @ec2_security_group_owner_id.setter
    def ec2_security_group_owner_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__77347fbce3871e8527d4550ee6c73c1ce3513c725367d951e52c076697239be5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ec2SecurityGroupOwnerId", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBSecurityGroupIngressProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_security_group_name": "dbSecurityGroupName",
        "cidrip": "cidrip",
        "ec2_security_group_id": "ec2SecurityGroupId",
        "ec2_security_group_name": "ec2SecurityGroupName",
        "ec2_security_group_owner_id": "ec2SecurityGroupOwnerId",
    },
)
class CfnDBSecurityGroupIngressProps:
    def __init__(
        self,
        *,
        db_security_group_name: builtins.str,
        cidrip: typing.Optional[builtins.str] = None,
        ec2_security_group_id: typing.Optional[builtins.str] = None,
        ec2_security_group_name: typing.Optional[builtins.str] = None,
        ec2_security_group_owner_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBSecurityGroupIngress``.

        :param db_security_group_name: The name of the DB security group to add authorization to.
        :param cidrip: The IP range to authorize.
        :param ec2_security_group_id: Id of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        :param ec2_security_group_name: Name of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        :param ec2_security_group_owner_id: AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter. The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_dBSecurity_group_ingress_props = rds.CfnDBSecurityGroupIngressProps(
                db_security_group_name="dbSecurityGroupName",
            
                # the properties below are optional
                cidrip="cidrip",
                ec2_security_group_id="ec2SecurityGroupId",
                ec2_security_group_name="ec2SecurityGroupName",
                ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__25ad01ac698ececdbbf7de0b069793b746bef7fa348afbcd9f93610af8e16e57)
            check_type(argname="argument db_security_group_name", value=db_security_group_name, expected_type=type_hints["db_security_group_name"])
            check_type(argname="argument cidrip", value=cidrip, expected_type=type_hints["cidrip"])
            check_type(argname="argument ec2_security_group_id", value=ec2_security_group_id, expected_type=type_hints["ec2_security_group_id"])
            check_type(argname="argument ec2_security_group_name", value=ec2_security_group_name, expected_type=type_hints["ec2_security_group_name"])
            check_type(argname="argument ec2_security_group_owner_id", value=ec2_security_group_owner_id, expected_type=type_hints["ec2_security_group_owner_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_security_group_name": db_security_group_name,
        }
        if cidrip is not None:
            self._values["cidrip"] = cidrip
        if ec2_security_group_id is not None:
            self._values["ec2_security_group_id"] = ec2_security_group_id
        if ec2_security_group_name is not None:
            self._values["ec2_security_group_name"] = ec2_security_group_name
        if ec2_security_group_owner_id is not None:
            self._values["ec2_security_group_owner_id"] = ec2_security_group_owner_id

    @builtins.property
    def db_security_group_name(self) -> builtins.str:
        '''The name of the DB security group to add authorization to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-dbsecuritygroupname
        '''
        result = self._values.get("db_security_group_name")
        assert result is not None, "Required property 'db_security_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cidrip(self) -> typing.Optional[builtins.str]:
        '''The IP range to authorize.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-cidrip
        '''
        result = self._values.get("cidrip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ec2_security_group_id(self) -> typing.Optional[builtins.str]:
        '''Id of the EC2 security group to authorize.

        For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupid
        '''
        result = self._values.get("ec2_security_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ec2_security_group_name(self) -> typing.Optional[builtins.str]:
        '''Name of the EC2 security group to authorize.

        For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupname
        '''
        result = self._values.get("ec2_security_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ec2_security_group_owner_id(self) -> typing.Optional[builtins.str]:
        '''AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter.

        The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupownerid
        '''
        result = self._values.get("ec2_security_group_owner_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBSecurityGroupIngressProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBSecurityGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_security_group_ingress": "dbSecurityGroupIngress",
        "group_description": "groupDescription",
        "ec2_vpc_id": "ec2VpcId",
        "tags": "tags",
    },
)
class CfnDBSecurityGroupProps:
    def __init__(
        self,
        *,
        db_security_group_ingress: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDBSecurityGroup.IngressProperty, typing.Dict[builtins.str, typing.Any]]]]],
        group_description: builtins.str,
        ec2_vpc_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBSecurityGroup``.

        :param db_security_group_ingress: Ingress rules to be applied to the DB security group.
        :param group_description: Provides the description of the DB security group.
        :param ec2_vpc_id: The identifier of an Amazon VPC. This property indicates the VPC that this DB security group belongs to. .. epigraph:: The ``EC2VpcId`` property is for backward compatibility with older regions, and is no longer recommended for providing security information to an RDS DB instance.
        :param tags: An optional array of key-value pairs to apply to this DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_dBSecurity_group_props = rds.CfnDBSecurityGroupProps(
                db_security_group_ingress=[rds.CfnDBSecurityGroup.IngressProperty(
                    cidrip="cidrip",
                    ec2_security_group_id="ec2SecurityGroupId",
                    ec2_security_group_name="ec2SecurityGroupName",
                    ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
                )],
                group_description="groupDescription",
            
                # the properties below are optional
                ec2_vpc_id="ec2VpcId",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__353ab04c42f26a9f8baa07d83a7623ee82f726e2e95aed3dc34a610717fd30be)
            check_type(argname="argument db_security_group_ingress", value=db_security_group_ingress, expected_type=type_hints["db_security_group_ingress"])
            check_type(argname="argument group_description", value=group_description, expected_type=type_hints["group_description"])
            check_type(argname="argument ec2_vpc_id", value=ec2_vpc_id, expected_type=type_hints["ec2_vpc_id"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_security_group_ingress": db_security_group_ingress,
            "group_description": group_description,
        }
        if ec2_vpc_id is not None:
            self._values["ec2_vpc_id"] = ec2_vpc_id
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def db_security_group_ingress(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBSecurityGroup.IngressProperty]]]:
        '''Ingress rules to be applied to the DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-dbsecuritygroupingress
        '''
        result = self._values.get("db_security_group_ingress")
        assert result is not None, "Required property 'db_security_group_ingress' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnDBSecurityGroup.IngressProperty]]], result)

    @builtins.property
    def group_description(self) -> builtins.str:
        '''Provides the description of the DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-groupdescription
        '''
        result = self._values.get("group_description")
        assert result is not None, "Required property 'group_description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def ec2_vpc_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of an Amazon VPC. This property indicates the VPC that this DB security group belongs to.

        .. epigraph::

           The ``EC2VpcId`` property is for backward compatibility with older regions, and is no longer recommended for providing security information to an RDS DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-ec2vpcid
        '''
        result = self._values.get("ec2_vpc_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''An optional array of key-value pairs to apply to this DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBSecurityGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDBSubnetGroup(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnDBSubnetGroup",
):
    '''A CloudFormation ``AWS::RDS::DBSubnetGroup``.

    The ``AWS::RDS::DBSubnetGroup`` resource creates a database subnet group. Subnet groups must contain at least two subnets in two different Availability Zones in the same region.

    For more information, see `Working with DB subnet groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html#USER_VPC.Subnets>`_ in the *Amazon RDS User Guide* .

    :cloudformationResource: AWS::RDS::DBSubnetGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_dBSubnet_group = rds.CfnDBSubnetGroup(self, "MyCfnDBSubnetGroup",
            db_subnet_group_description="dbSubnetGroupDescription",
            subnet_ids=["subnetIds"],
        
            # the properties below are optional
            db_subnet_group_name="dbSubnetGroupName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        db_subnet_group_description: builtins.str,
        subnet_ids: typing.Sequence[builtins.str],
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBSubnetGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_subnet_group_description: The description for the DB subnet group.
        :param subnet_ids: The EC2 Subnet IDs for the DB subnet group.
        :param db_subnet_group_name: The name for the DB subnet group. This value is stored as a lowercase string. Constraints: Must contain no more than 255 lowercase alphanumeric characters or hyphens. Must not be "Default". Example: ``mysubnetgroup``
        :param tags: An optional array of key-value pairs to apply to this DB subnet group.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3437b6f3359034732d6fbef14006ac020c94fa5b18aa95cb4c8bb7332cc58dc7)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBSubnetGroupProps(
            db_subnet_group_description=db_subnet_group_description,
            subnet_ids=subnet_ids,
            db_subnet_group_name=db_subnet_group_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0c57b3aebd9b5ec4ca30322b2d669752510e9ab264e911c81dbcc6ba86736df)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b80096e1e9e87186913f37281dd86910ef95217ead8ec5697e0635c9858af16f)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''An optional array of key-value pairs to apply to this DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="dbSubnetGroupDescription")
    def db_subnet_group_description(self) -> builtins.str:
        '''The description for the DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-dbsubnetgroupdescription
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbSubnetGroupDescription"))

    @db_subnet_group_description.setter
    def db_subnet_group_description(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f8fc42270bedfbabbcd6ae88d4f87f1ddad75d82484bd7bd257c9456db2629b5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSubnetGroupDescription", value)

    @builtins.property
    @jsii.member(jsii_name="subnetIds")
    def subnet_ids(self) -> typing.List[builtins.str]:
        '''The EC2 Subnet IDs for the DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-subnetids
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "subnetIds"))

    @subnet_ids.setter
    def subnet_ids(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__80adf03e2774c29f8199884349ab971da0c13541b8a93956d440e6cef2f35a33)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "subnetIds", value)

    @builtins.property
    @jsii.member(jsii_name="dbSubnetGroupName")
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''The name for the DB subnet group. This value is stored as a lowercase string.

        Constraints: Must contain no more than 255 lowercase alphanumeric characters or hyphens. Must not be "Default".

        Example: ``mysubnetgroup``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-dbsubnetgroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSubnetGroupName"))

    @db_subnet_group_name.setter
    def db_subnet_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f106bcbb8c63de9fd9ad65eff1a8a73d3e0cdb022528b57cd97e00d93c8e504e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSubnetGroupName", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnDBSubnetGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_subnet_group_description": "dbSubnetGroupDescription",
        "subnet_ids": "subnetIds",
        "db_subnet_group_name": "dbSubnetGroupName",
        "tags": "tags",
    },
)
class CfnDBSubnetGroupProps:
    def __init__(
        self,
        *,
        db_subnet_group_description: builtins.str,
        subnet_ids: typing.Sequence[builtins.str],
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBSubnetGroup``.

        :param db_subnet_group_description: The description for the DB subnet group.
        :param subnet_ids: The EC2 Subnet IDs for the DB subnet group.
        :param db_subnet_group_name: The name for the DB subnet group. This value is stored as a lowercase string. Constraints: Must contain no more than 255 lowercase alphanumeric characters or hyphens. Must not be "Default". Example: ``mysubnetgroup``
        :param tags: An optional array of key-value pairs to apply to this DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_dBSubnet_group_props = rds.CfnDBSubnetGroupProps(
                db_subnet_group_description="dbSubnetGroupDescription",
                subnet_ids=["subnetIds"],
            
                # the properties below are optional
                db_subnet_group_name="dbSubnetGroupName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ee6eb06c99be08b951a99e2e5521e7dcf9b3c7213a342f750abfc3bb56351b21)
            check_type(argname="argument db_subnet_group_description", value=db_subnet_group_description, expected_type=type_hints["db_subnet_group_description"])
            check_type(argname="argument subnet_ids", value=subnet_ids, expected_type=type_hints["subnet_ids"])
            check_type(argname="argument db_subnet_group_name", value=db_subnet_group_name, expected_type=type_hints["db_subnet_group_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_subnet_group_description": db_subnet_group_description,
            "subnet_ids": subnet_ids,
        }
        if db_subnet_group_name is not None:
            self._values["db_subnet_group_name"] = db_subnet_group_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def db_subnet_group_description(self) -> builtins.str:
        '''The description for the DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-dbsubnetgroupdescription
        '''
        result = self._values.get("db_subnet_group_description")
        assert result is not None, "Required property 'db_subnet_group_description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def subnet_ids(self) -> typing.List[builtins.str]:
        '''The EC2 Subnet IDs for the DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-subnetids
        '''
        result = self._values.get("subnet_ids")
        assert result is not None, "Required property 'subnet_ids' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''The name for the DB subnet group. This value is stored as a lowercase string.

        Constraints: Must contain no more than 255 lowercase alphanumeric characters or hyphens. Must not be "Default".

        Example: ``mysubnetgroup``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-dbsubnetgroupname
        '''
        result = self._values.get("db_subnet_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''An optional array of key-value pairs to apply to this DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBSubnetGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnEventSubscription(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnEventSubscription",
):
    '''A CloudFormation ``AWS::RDS::EventSubscription``.

    The ``AWS::RDS::EventSubscription`` resource allows you to receive notifications for Amazon Relational Database Service events through the Amazon Simple Notification Service (Amazon SNS). For more information, see `Using Amazon RDS Event Notification <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.html>`_ in the *Amazon RDS User Guide* .

    :cloudformationResource: AWS::RDS::EventSubscription
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_event_subscription = rds.CfnEventSubscription(self, "MyCfnEventSubscription",
            sns_topic_arn="snsTopicArn",
        
            # the properties below are optional
            enabled=False,
            event_categories=["eventCategories"],
            source_ids=["sourceIds"],
            source_type="sourceType",
            subscription_name="subscriptionName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        sns_topic_arn: builtins.str,
        enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        event_categories: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_type: typing.Optional[builtins.str] = None,
        subscription_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::EventSubscription``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param sns_topic_arn: The Amazon Resource Name (ARN) of the SNS topic created for event notification. The ARN is created by Amazon SNS when you create a topic and subscribe to it.
        :param enabled: A value that indicates whether to activate the subscription. If the event notification subscription isn't activated, the subscription is created but not active.
        :param event_categories: A list of event categories for a particular source type ( ``SourceType`` ) that you want to subscribe to. You can see a list of the categories for a given source type in the "Amazon RDS event categories and event messages" section of the `*Amazon RDS User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.Messages.html>`_ or the `*Amazon Aurora User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_Events.Messages.html>`_ . You can also see this list by using the ``DescribeEventCategories`` operation.
        :param source_ids: The list of identifiers of the event sources for which events are returned. If not specified, then all sources are included in the response. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens. It can't end with a hyphen or contain two consecutive hyphens. Constraints: - If a ``SourceIds`` value is supplied, ``SourceType`` must also be provided. - If the source type is a DB instance, a ``DBInstanceIdentifier`` value must be supplied. - If the source type is a DB cluster, a ``DBClusterIdentifier`` value must be supplied. - If the source type is a DB parameter group, a ``DBParameterGroupName`` value must be supplied. - If the source type is a DB security group, a ``DBSecurityGroupName`` value must be supplied. - If the source type is a DB snapshot, a ``DBSnapshotIdentifier`` value must be supplied. - If the source type is a DB cluster snapshot, a ``DBClusterSnapshotIdentifier`` value must be supplied.
        :param source_type: The type of source that is generating the events. For example, if you want to be notified of events generated by a DB instance, set this parameter to ``db-instance`` . If this value isn't specified, all events are returned. Valid values: ``db-instance`` | ``db-cluster`` | ``db-parameter-group`` | ``db-security-group`` | ``db-snapshot`` | ``db-cluster-snapshot``
        :param subscription_name: The name of the subscription. Constraints: The name must be less than 255 characters.
        :param tags: An optional array of key-value pairs to apply to this subscription.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c5db068cdeb685ae34885c6eea1acaa01b83ca0af2c63429f3a30e0af5ab7db)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEventSubscriptionProps(
            sns_topic_arn=sns_topic_arn,
            enabled=enabled,
            event_categories=event_categories,
            source_ids=source_ids,
            source_type=source_type,
            subscription_name=subscription_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__31f076db464d46bf940b81eae58c6d887f3b50913e8d41f1e9eaf45eecb21f4a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a20ae04fb6b0603709ecf11c6ea17741d5849f8054b8b6c662dc225586cca4e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''An optional array of key-value pairs to apply to this subscription.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="snsTopicArn")
    def sns_topic_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the SNS topic created for event notification.

        The ARN is created by Amazon SNS when you create a topic and subscribe to it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-snstopicarn
        '''
        return typing.cast(builtins.str, jsii.get(self, "snsTopicArn"))

    @sns_topic_arn.setter
    def sns_topic_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__725eef3acf6a3d45ddad2bbb8033eecd25b8265a0113b5a4bde635eccc922e26)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "snsTopicArn", value)

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to activate the subscription.

        If the event notification subscription isn't activated, the subscription is created but not active.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-enabled
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enabled"))

    @enabled.setter
    def enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d27f61978b38a92532fbb55b533f68090bb9edde6df3cf1aae5ca81d6fd745cb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enabled", value)

    @builtins.property
    @jsii.member(jsii_name="eventCategories")
    def event_categories(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of event categories for a particular source type ( ``SourceType`` ) that you want to subscribe to.

        You can see a list of the categories for a given source type in the "Amazon RDS event categories and event messages" section of the `*Amazon RDS User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.Messages.html>`_ or the `*Amazon Aurora User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_Events.Messages.html>`_ . You can also see this list by using the ``DescribeEventCategories`` operation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-eventcategories
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "eventCategories"))

    @event_categories.setter
    def event_categories(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b451c3a9b4f0ecc0f0d2566ff7304db2ebdbb202d3898d49b840a814d78832f5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "eventCategories", value)

    @builtins.property
    @jsii.member(jsii_name="sourceIds")
    def source_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of identifiers of the event sources for which events are returned.

        If not specified, then all sources are included in the response. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens. It can't end with a hyphen or contain two consecutive hyphens.

        Constraints:

        - If a ``SourceIds`` value is supplied, ``SourceType`` must also be provided.
        - If the source type is a DB instance, a ``DBInstanceIdentifier`` value must be supplied.
        - If the source type is a DB cluster, a ``DBClusterIdentifier`` value must be supplied.
        - If the source type is a DB parameter group, a ``DBParameterGroupName`` value must be supplied.
        - If the source type is a DB security group, a ``DBSecurityGroupName`` value must be supplied.
        - If the source type is a DB snapshot, a ``DBSnapshotIdentifier`` value must be supplied.
        - If the source type is a DB cluster snapshot, a ``DBClusterSnapshotIdentifier`` value must be supplied.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourceids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "sourceIds"))

    @source_ids.setter
    def source_ids(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__117c27a2bfa4fdddd71e0cf520f0a488e608552d4e40fc438d80956908d02094)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceIds", value)

    @builtins.property
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> typing.Optional[builtins.str]:
        '''The type of source that is generating the events.

        For example, if you want to be notified of events generated by a DB instance, set this parameter to ``db-instance`` . If this value isn't specified, all events are returned.

        Valid values: ``db-instance`` | ``db-cluster`` | ``db-parameter-group`` | ``db-security-group`` | ``db-snapshot`` | ``db-cluster-snapshot``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourcetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceType"))

    @source_type.setter
    def source_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f291135ba489a08dc82c67edf431fcab65c4ee57d2fe99484d61a7278397a932)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceType", value)

    @builtins.property
    @jsii.member(jsii_name="subscriptionName")
    def subscription_name(self) -> typing.Optional[builtins.str]:
        '''The name of the subscription.

        Constraints: The name must be less than 255 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-subscriptionname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "subscriptionName"))

    @subscription_name.setter
    def subscription_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d411ca6aa9fb044fefb897f96abcf36ea0e4876e110a7e51d644679861e68fd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "subscriptionName", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnEventSubscriptionProps",
    jsii_struct_bases=[],
    name_mapping={
        "sns_topic_arn": "snsTopicArn",
        "enabled": "enabled",
        "event_categories": "eventCategories",
        "source_ids": "sourceIds",
        "source_type": "sourceType",
        "subscription_name": "subscriptionName",
        "tags": "tags",
    },
)
class CfnEventSubscriptionProps:
    def __init__(
        self,
        *,
        sns_topic_arn: builtins.str,
        enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        event_categories: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_type: typing.Optional[builtins.str] = None,
        subscription_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnEventSubscription``.

        :param sns_topic_arn: The Amazon Resource Name (ARN) of the SNS topic created for event notification. The ARN is created by Amazon SNS when you create a topic and subscribe to it.
        :param enabled: A value that indicates whether to activate the subscription. If the event notification subscription isn't activated, the subscription is created but not active.
        :param event_categories: A list of event categories for a particular source type ( ``SourceType`` ) that you want to subscribe to. You can see a list of the categories for a given source type in the "Amazon RDS event categories and event messages" section of the `*Amazon RDS User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.Messages.html>`_ or the `*Amazon Aurora User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_Events.Messages.html>`_ . You can also see this list by using the ``DescribeEventCategories`` operation.
        :param source_ids: The list of identifiers of the event sources for which events are returned. If not specified, then all sources are included in the response. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens. It can't end with a hyphen or contain two consecutive hyphens. Constraints: - If a ``SourceIds`` value is supplied, ``SourceType`` must also be provided. - If the source type is a DB instance, a ``DBInstanceIdentifier`` value must be supplied. - If the source type is a DB cluster, a ``DBClusterIdentifier`` value must be supplied. - If the source type is a DB parameter group, a ``DBParameterGroupName`` value must be supplied. - If the source type is a DB security group, a ``DBSecurityGroupName`` value must be supplied. - If the source type is a DB snapshot, a ``DBSnapshotIdentifier`` value must be supplied. - If the source type is a DB cluster snapshot, a ``DBClusterSnapshotIdentifier`` value must be supplied.
        :param source_type: The type of source that is generating the events. For example, if you want to be notified of events generated by a DB instance, set this parameter to ``db-instance`` . If this value isn't specified, all events are returned. Valid values: ``db-instance`` | ``db-cluster`` | ``db-parameter-group`` | ``db-security-group`` | ``db-snapshot`` | ``db-cluster-snapshot``
        :param subscription_name: The name of the subscription. Constraints: The name must be less than 255 characters.
        :param tags: An optional array of key-value pairs to apply to this subscription.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_event_subscription_props = rds.CfnEventSubscriptionProps(
                sns_topic_arn="snsTopicArn",
            
                # the properties below are optional
                enabled=False,
                event_categories=["eventCategories"],
                source_ids=["sourceIds"],
                source_type="sourceType",
                subscription_name="subscriptionName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f8812900c3de7bd4f6e3f5180fd49f8280b522418a0ba50cefedcb2d6cca8979)
            check_type(argname="argument sns_topic_arn", value=sns_topic_arn, expected_type=type_hints["sns_topic_arn"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument event_categories", value=event_categories, expected_type=type_hints["event_categories"])
            check_type(argname="argument source_ids", value=source_ids, expected_type=type_hints["source_ids"])
            check_type(argname="argument source_type", value=source_type, expected_type=type_hints["source_type"])
            check_type(argname="argument subscription_name", value=subscription_name, expected_type=type_hints["subscription_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "sns_topic_arn": sns_topic_arn,
        }
        if enabled is not None:
            self._values["enabled"] = enabled
        if event_categories is not None:
            self._values["event_categories"] = event_categories
        if source_ids is not None:
            self._values["source_ids"] = source_ids
        if source_type is not None:
            self._values["source_type"] = source_type
        if subscription_name is not None:
            self._values["subscription_name"] = subscription_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def sns_topic_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the SNS topic created for event notification.

        The ARN is created by Amazon SNS when you create a topic and subscribe to it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-snstopicarn
        '''
        result = self._values.get("sns_topic_arn")
        assert result is not None, "Required property 'sns_topic_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A value that indicates whether to activate the subscription.

        If the event notification subscription isn't activated, the subscription is created but not active.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-enabled
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def event_categories(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of event categories for a particular source type ( ``SourceType`` ) that you want to subscribe to.

        You can see a list of the categories for a given source type in the "Amazon RDS event categories and event messages" section of the `*Amazon RDS User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.Messages.html>`_ or the `*Amazon Aurora User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_Events.Messages.html>`_ . You can also see this list by using the ``DescribeEventCategories`` operation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-eventcategories
        '''
        result = self._values.get("event_categories")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def source_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of identifiers of the event sources for which events are returned.

        If not specified, then all sources are included in the response. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens. It can't end with a hyphen or contain two consecutive hyphens.

        Constraints:

        - If a ``SourceIds`` value is supplied, ``SourceType`` must also be provided.
        - If the source type is a DB instance, a ``DBInstanceIdentifier`` value must be supplied.
        - If the source type is a DB cluster, a ``DBClusterIdentifier`` value must be supplied.
        - If the source type is a DB parameter group, a ``DBParameterGroupName`` value must be supplied.
        - If the source type is a DB security group, a ``DBSecurityGroupName`` value must be supplied.
        - If the source type is a DB snapshot, a ``DBSnapshotIdentifier`` value must be supplied.
        - If the source type is a DB cluster snapshot, a ``DBClusterSnapshotIdentifier`` value must be supplied.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourceids
        '''
        result = self._values.get("source_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def source_type(self) -> typing.Optional[builtins.str]:
        '''The type of source that is generating the events.

        For example, if you want to be notified of events generated by a DB instance, set this parameter to ``db-instance`` . If this value isn't specified, all events are returned.

        Valid values: ``db-instance`` | ``db-cluster`` | ``db-parameter-group`` | ``db-security-group`` | ``db-snapshot`` | ``db-cluster-snapshot``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourcetype
        '''
        result = self._values.get("source_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def subscription_name(self) -> typing.Optional[builtins.str]:
        '''The name of the subscription.

        Constraints: The name must be less than 255 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-subscriptionname
        '''
        result = self._values.get("subscription_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''An optional array of key-value pairs to apply to this subscription.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEventSubscriptionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnGlobalCluster(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnGlobalCluster",
):
    '''A CloudFormation ``AWS::RDS::GlobalCluster``.

    The ``AWS::RDS::GlobalCluster`` resource creates or updates an Amazon Aurora global database spread across multiple AWS Regions.

    The global database contains a single primary cluster with read-write capability, and a read-only secondary cluster that receives data from the primary cluster through high-speed replication performed by the Aurora storage subsystem.

    You can create a global database that is initially empty, and then add a primary cluster and a secondary cluster to it.

    For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* .

    :cloudformationResource: AWS::RDS::GlobalCluster
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_rds as rds
        
        cfn_global_cluster = rds.CfnGlobalCluster(self, "MyCfnGlobalCluster",
            deletion_protection=False,
            engine="engine",
            engine_version="engineVersion",
            global_cluster_identifier="globalClusterIdentifier",
            source_db_cluster_identifier="sourceDbClusterIdentifier",
            storage_encrypted=False
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::GlobalCluster``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param deletion_protection: The deletion protection setting for the new global database. The global database can't be deleted when deletion protection is enabled.
        :param engine: The name of the database engine to be used for this DB cluster. If this property isn't specified, the database engine is derived from the source DB cluster specified by the ``SourceDBClusterIdentifier`` property. .. epigraph:: If the ``SourceDBClusterIdentifier`` property isn't specified, this property is required. If the ``SourceDBClusterIdentifier`` property is specified, make sure this property isn't specified.
        :param engine_version: The engine version of the Aurora global database.
        :param global_cluster_identifier: The cluster identifier of the global database cluster.
        :param source_db_cluster_identifier: The DB cluster identifier or Amazon Resource Name (ARN) to use as the primary cluster of the global database. .. epigraph:: If the ``Engine`` property isn't specified, this property is required. If the ``Engine`` property is specified, make sure this property isn't specified.
        :param storage_encrypted: The storage encryption setting for the global database cluster.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1611fa62b935d4f304c9fd8befd7c639fa3cc4898c7c6d9f86feb2d669b72e80)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnGlobalClusterProps(
            deletion_protection=deletion_protection,
            engine=engine,
            engine_version=engine_version,
            global_cluster_identifier=global_cluster_identifier,
            source_db_cluster_identifier=source_db_cluster_identifier,
            storage_encrypted=storage_encrypted,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4e2d3cad6f02076a09cf1f547dcac015f24cce853ab75aa973cb2d9f1b21077c)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35fd13f4b6b6c4b977b13a46e1fc25368c1277f30d60113a195c02e0a23f6223)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''The deletion protection setting for the new global database.

        The global database can't be deleted when deletion protection is enabled.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-deletionprotection
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "deletionProtection"))

    @deletion_protection.setter
    def deletion_protection(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3cfbe56a53cd6d95cfdd32ab00cf3bfe312f7c16be1b0f4f4e34aeb398290763)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deletionProtection", value)

    @builtins.property
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine to be used for this DB cluster.

        If this property isn't specified, the database engine is derived from the source DB cluster specified by the ``SourceDBClusterIdentifier`` property.
        .. epigraph::

           If the ``SourceDBClusterIdentifier`` property isn't specified, this property is required. If the ``SourceDBClusterIdentifier`` property is specified, make sure this property isn't specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engine
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engine"))

    @engine.setter
    def engine(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__377a45f027efcee5d8bac171dee917b896fe5f052401019cb110b569cded8f8d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engine", value)

    @builtins.property
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The engine version of the Aurora global database.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engineversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engineVersion"))

    @engine_version.setter
    def engine_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__484590244ee41b9290f1f270ea1960cd87e41b189211e6d5978a3239b94469b9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineVersion", value)

    @builtins.property
    @jsii.member(jsii_name="globalClusterIdentifier")
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The cluster identifier of the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-globalclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "globalClusterIdentifier"))

    @global_cluster_identifier.setter
    def global_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8c0878c8feb5b0667b16ebe0d996cc33dadb0920b96e5594e9cf1b8cc8e82a5d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "globalClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbClusterIdentifier")
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The DB cluster identifier or Amazon Resource Name (ARN) to use as the primary cluster of the global database.

        .. epigraph::

           If the ``Engine`` property isn't specified, this property is required. If the ``Engine`` property is specified, make sure this property isn't specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-sourcedbclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbClusterIdentifier"))

    @source_db_cluster_identifier.setter
    def source_db_cluster_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2b8b99331248625e609d8c61fdd2aeead968468cd83f1da2306ce25de200e221)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="storageEncrypted")
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''The storage encryption setting for the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-storageencrypted
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "storageEncrypted"))

    @storage_encrypted.setter
    def storage_encrypted(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f760cdb237d4844bc219ed58856db0d37bc81d9e590f5413f4b7b4f0cdc9a17d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageEncrypted", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_rds.CfnGlobalClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "deletion_protection": "deletionProtection",
        "engine": "engine",
        "engine_version": "engineVersion",
        "global_cluster_identifier": "globalClusterIdentifier",
        "source_db_cluster_identifier": "sourceDbClusterIdentifier",
        "storage_encrypted": "storageEncrypted",
    },
)
class CfnGlobalClusterProps:
    def __init__(
        self,
        *,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
    ) -> None:
        '''Properties for defining a ``CfnGlobalCluster``.

        :param deletion_protection: The deletion protection setting for the new global database. The global database can't be deleted when deletion protection is enabled.
        :param engine: The name of the database engine to be used for this DB cluster. If this property isn't specified, the database engine is derived from the source DB cluster specified by the ``SourceDBClusterIdentifier`` property. .. epigraph:: If the ``SourceDBClusterIdentifier`` property isn't specified, this property is required. If the ``SourceDBClusterIdentifier`` property is specified, make sure this property isn't specified.
        :param engine_version: The engine version of the Aurora global database.
        :param global_cluster_identifier: The cluster identifier of the global database cluster.
        :param source_db_cluster_identifier: The DB cluster identifier or Amazon Resource Name (ARN) to use as the primary cluster of the global database. .. epigraph:: If the ``Engine`` property isn't specified, this property is required. If the ``Engine`` property is specified, make sure this property isn't specified.
        :param storage_encrypted: The storage encryption setting for the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_rds as rds
            
            cfn_global_cluster_props = rds.CfnGlobalClusterProps(
                deletion_protection=False,
                engine="engine",
                engine_version="engineVersion",
                global_cluster_identifier="globalClusterIdentifier",
                source_db_cluster_identifier="sourceDbClusterIdentifier",
                storage_encrypted=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ef2e57f0cb9427badb90bc7e1248f0f26bc8de21a104bb924da9733667030430)
            check_type(argname="argument deletion_protection", value=deletion_protection, expected_type=type_hints["deletion_protection"])
            check_type(argname="argument engine", value=engine, expected_type=type_hints["engine"])
            check_type(argname="argument engine_version", value=engine_version, expected_type=type_hints["engine_version"])
            check_type(argname="argument global_cluster_identifier", value=global_cluster_identifier, expected_type=type_hints["global_cluster_identifier"])
            check_type(argname="argument source_db_cluster_identifier", value=source_db_cluster_identifier, expected_type=type_hints["source_db_cluster_identifier"])
            check_type(argname="argument storage_encrypted", value=storage_encrypted, expected_type=type_hints["storage_encrypted"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if engine is not None:
            self._values["engine"] = engine
        if engine_version is not None:
            self._values["engine_version"] = engine_version
        if global_cluster_identifier is not None:
            self._values["global_cluster_identifier"] = global_cluster_identifier
        if source_db_cluster_identifier is not None:
            self._values["source_db_cluster_identifier"] = source_db_cluster_identifier
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted

    @builtins.property
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''The deletion protection setting for the new global database.

        The global database can't be deleted when deletion protection is enabled.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-deletionprotection
        '''
        result = self._values.get("deletion_protection")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine to be used for this DB cluster.

        If this property isn't specified, the database engine is derived from the source DB cluster specified by the ``SourceDBClusterIdentifier`` property.
        .. epigraph::

           If the ``SourceDBClusterIdentifier`` property isn't specified, this property is required. If the ``SourceDBClusterIdentifier`` property is specified, make sure this property isn't specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engine
        '''
        result = self._values.get("engine")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The engine version of the Aurora global database.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engineversion
        '''
        result = self._values.get("engine_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The cluster identifier of the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-globalclusteridentifier
        '''
        result = self._values.get("global_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The DB cluster identifier or Amazon Resource Name (ARN) to use as the primary cluster of the global database.

        .. epigraph::

           If the ``Engine`` property isn't specified, this property is required. If the ``Engine`` property is specified, make sure this property isn't specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-sourcedbclusteridentifier
        '''
        result = self._values.get("source_db_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''The storage encryption setting for the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-storageencrypted
        '''
        result = self._values.get("storage_encrypted")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGlobalClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnOptionGroup(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_rds.CfnOptionGroup",
):
    '''A CloudFormation ``AWS::RDS::OptionGroup``.

    The ``AWS::RDS::OptionGroup`` resource creates or updates an option group, to enable and configure features that are specific to a particular DB engine.

    :cloudformationResource: AWS::RDS::OptionGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk im