import { Construct } from 'constructs';
import { IIdentity } from './identity-base';
import { IManagedPolicy } from './managed-policy';
import { Policy } from './policy';
import { PolicyStatement } from './policy-statement';
import { AddToPrincipalPolicyResult, IPrincipal, PrincipalPolicyFragment } from './principals';
import { IUser } from './user';
import { Resource } from '../../core';
/**
 * Represents an IAM Group.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_groups.html
 */
export interface IGroup extends IIdentity {
    /**
     * Returns the IAM Group Name
     *
     * @attribute
     */
    readonly groupName: string;
    /**
     * Returns the IAM Group ARN
     *
     * @attribute
     */
    readonly groupArn: string;
}
/**
 * Properties for defining an IAM group
 */
export interface GroupProps {
    /**
     * A name for the IAM group. For valid values, see the GroupName parameter
     * for the CreateGroup action in the IAM API Reference. If you don't specify
     * a name, AWS CloudFormation generates a unique physical ID and uses that
     * ID for the group name.
     *
     * If you specify a name, you must specify the CAPABILITY_NAMED_IAM value to
     * acknowledge your template's capabilities. For more information, see
     * Acknowledging IAM Resources in AWS CloudFormation Templates.
     *
     * @default Generated by CloudFormation (recommended)
     */
    readonly groupName?: string;
    /**
     * A list of managed policies associated with this role.
     *
     * You can add managed policies later using
     * `addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName(policyName))`.
     *
     * @default - No managed policies.
     */
    readonly managedPolicies?: IManagedPolicy[];
    /**
     * The path to the group. For more information about paths, see [IAM
     * Identifiers](http://docs.aws.amazon.com/IAM/latest/UserGuide/index.html?Using_Identifiers.html)
     * in the IAM User Guide.
     *
     * @default /
     */
    readonly path?: string;
}
declare abstract class GroupBase extends Resource implements IGroup {
    abstract readonly groupName: string;
    abstract readonly groupArn: string;
    readonly grantPrincipal: IPrincipal;
    readonly principalAccount: string | undefined;
    readonly assumeRoleAction: string;
    private readonly attachedPolicies;
    private defaultPolicy?;
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Attaches a policy to this group.
     * @param policy The policy to attach.
     */
    attachInlinePolicy(policy: Policy): void;
    addManagedPolicy(_policy: IManagedPolicy): void;
    /**
     * Adds a user to this group.
     */
    addUser(user: IUser): void;
    /**
     * Adds an IAM statement to the default policy.
     */
    addToPrincipalPolicy(statement: PolicyStatement): AddToPrincipalPolicyResult;
    addToPolicy(statement: PolicyStatement): boolean;
}
/**
 * An IAM Group (collection of IAM users) lets you specify permissions for
 * multiple users, which can make it easier to manage permissions for those users.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_groups.html
 */
export declare class Group extends GroupBase {
    /**
     * Import an external group by ARN.
     *
     * If the imported Group ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * group has a `path` (like `arn:...:group/AdminGroup/NetworkAdmin`), the
     * `groupName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Group ARN should be supplied without the
     * `path` in order to resolve the correct group resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param groupArn the ARN of the group to import (e.g. `arn:aws:iam::account-id:group/group-name`)
     */
    static fromGroupArn(scope: Construct, id: string, groupArn: string): IGroup;
    /**
     * Import an existing group by given name (with path).
     * This method has same caveats of `fromGroupArn`
     *
     * @param scope construct scope
     * @param id construct id
     * @param groupName the groupName (path included) of the existing group to import
     */
    static fromGroupName(scope: Construct, id: string, groupName: string): IGroup;
    readonly groupName: string;
    readonly groupArn: string;
    private readonly managedPolicies;
    constructor(scope: Construct, id: string, props?: GroupProps);
    /**
     * Attaches a managed policy to this group. See [IAM and AWS STS quotas, name requirements, and character limits]
     * (https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-quotas.html#reference_iam-quotas-entities)
     * for quota of managed policies attached to an IAM group.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy: IManagedPolicy): void;
    private managedPoliciesExceededWarning;
}
export {};
