"""Output schema implementation for FEP."""

import dataclasses
from dataclasses import dataclass
from typing import Any


@dataclass
class Output:
    """
    Contains the output generated by the system being evaluated.

    The output generation process is treated as a black box by the protocol.
    The output, combined with the test case, forms the complete evaluation context.

    Required Fields:
    - value: The actual output from the system being evaluated

    Optional Fields:
    - id: Unique identifier for this output instance
    - metadata: System-specific information about the output generation
    """

    value: Any | None
    id: str | None = None
    metadata: dict[str, Any] | None = None

    def to_dict(self) -> dict[str, Any]:
        """
        Convert Output to dictionary, handling Pydantic models and dataclasses.

        Returns:
            Dictionary representation of the Output instance.
        """
        # Handle different serialization patterns
        if hasattr(self.value, 'model_dump'):
            # Pydantic v2
            value_dict = self.value.model_dump()
        elif hasattr(self.value, 'dict'):
            # Pydantic v1 or custom .dict() method
            value_dict = self.value.dict()
        elif hasattr(self.value, 'to_dict'):
            # Custom .to_dict() method
            value_dict = self.value.to_dict()
        elif dataclasses.is_dataclass(self.value) and not isinstance(self.value, type):
            # Dataclass instance (not the class itself)
            value_dict = dataclasses.asdict(self.value)
        else:
            # Primitive types, dicts, None, or other objects
            value_dict = self.value

        return {
            'value': value_dict,
            'id': self.id,
            'metadata': self.metadata,
        }
